/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "RxDcOffsetClbrHndlr.h"
#include "MT_Math.h"
#include "CalibrationHandlerUtils.h"
#include "stringLibApi.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "Afe_API.h"
#include "loggerAPI.h"
#include "MT_Math.h"
#include "mt_sysdefs.h"
#include "mt_sysrst.h"
#include "RegAccess_Api.h"

#include "LmHdk_API.h"
#include "ErrorHandler_Api.h"
#include "mt_sysdefs.h"
#include "CalibrationHandler.h"
#include "Shram_ClbrDataBuffer.h"
#include "mt_sysrst.h"
#include "AFEDefines.h"
#include "PSD.h"
#include "lib_rf_4x4_api.h"
#include "Pac_Api.h"
#include "RficDriver_API.h"


#define LOG_LOCAL_GID	GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 16


/********************************************************************************
CalAlgo_DC:

Description:
------------
Calibrate RFIC DC
1. Calibrate DAC2
	*Short PGC1
	*Set PGC3 gain to max value
	*Find DAC value which minimize RFIC DC (Lion in the desert algo needs 6 measurements to find min) 
2. Calibrate DAC0
3. Calibrate DAC1
	*Undo DAC1 short 
	*Set PGC3 gain the same as in DAC2 calibration
	*Set PGC2 gain to 6dB
	*Find DAC value which minimize RFIC DC
4. For each DAC1 calibration, calibrate residual DAC2

  Note - the calibration can work with inband as high as -35dBm when T/R is in Tx mode - Avima 7/6/07

Output:
-------
	DC tables with calibrated values	
********************************************************************************/

/******************************************************************************/
/***						Constant Defines								***/
/******************************************************************************/
typedef struct _RxDC_Config
{
	uint32	minReference;
	uint32	numOfSamplesDac1[RXDC_MAX_NUM_OF_RETRIES];//8192
	uint32	numOfSamplesDac2;//8192
	uint32  numOfCorrelatorSamples;//10000
	uint8	numRetriesDac2Chip;//
	uint8	numRetriesDac1Chip;
	uint8	numRetriesDac2Sw;
	uint8	numRetriesDac1Sw;
	uint32	param_STM_DAC2_config[RXDC_MAX_NUM_OF_RETRIES];//13
	uint32	param_STM_DAC1_config;//13
	uint32	offlineTimeLimit;
	uint32	onlineTimeLimit;
	int32	dac2failTH;
	uint8	calibrateAllBw;
} RxDC_Config_t;

typedef struct _RxDC_Params
{
	uint8	currentAnt;
	uint8	currentBW;
	uint8	curCalibType;
	int32	timeLimit;
	int32	minRef[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	//uint32	GainConfiguration[RXDC_MAX_GAIN_PERMUTATIONS][2];
	//int32	resultsDB[MAX_NUM_OF_ANTENNAS][MAX_NUM_OF_BW][RXDC_MAX_GAIN_PERMUTATIONS];
} RxDC_Params_t;

#define DC_TABLE1_NUMBER_OF_ENTRIES					(96)

#define RXDC_ONLINE_MAX_STEPS		2
#define RXDC_ONLINE_INITIAL_STEP	1

#define RXDC_OFFLINE_CAL_TIME_LIMIT	(0xFFFFFFFF)
#define RXDC_ONLINE_CAL_TIME_LIMIT	(2000)
#define RXDC_ONLINE_CAL_TIME_LIMIT_CANCELLATION (2500)

#define RXDC_NUM_SPECIAL_GAIN_TABLE1	10
#define RXDC_MAX_NUM_SPECIAL_GAINS		22

uint8 rxdcCalMode;
#define ONLINE_MAX_POWER_NUM_SAMPLES	13
#ifdef RFIC_WRX300
uint8 PGC2GainsCalMode[RXDC_NUM_PGC2_GAINS] = 
{
	RXDC_CONFIG_CALIBRATION_MODE_SW,
	RXDC_CONFIG_CALIBRATION_MODE_SW,
	RXDC_CONFIG_CALIBRATION_MODE_SW,
	RXDC_CONFIG_CALIBRATION_MODE_SW,
	RXDC_CONFIG_CALIBRATION_MODE_SW,
	RXDC_CONFIG_CALIBRATION_MODE_SW,
};

uint32 LnaPgcGainsConfig[RXDC_MAX_GAIN_PERMUTATIONS][2] =
{
	0x02fa0c0c, 0x01110064,
	0x01fa0c0c, 0x01110064,
	0x00fa0c0c, 0x01110064,
	0x02000c0c, 0x01110064,
	0x01000c0c, 0x01110064,
	0x00000c0c, 0x01110064,
	0x02060c0c, 0x01110064,
	0x01060c0c, 0x01110064,
	0x00060c0c, 0x01110064,
	0x020c0c0c, 0x01110064,
	0x010c0c0c, 0x01110064,
	0x000c0c0c, 0x01110064,
	0x02120c0c, 0x01110064,
	0x01120c0c, 0x01110064,
	0x00120c0c, 0x01110064,
	0x02180c0c, 0x01110064,
	0x01180c0c, 0x01110064,
	0x00180c0c, 0x01110064,
};	
#else //WRX500
uint8 PGC2GainsCalMode[RXDC_NUM_PGC2_GAINS] = 
{
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_NONE,
	RXDC_CONFIG_CALIBRATION_MODE_CHIP,
	RXDC_CONFIG_CALIBRATION_MODE_CHIP,
};

uint32 LnaPgcGainsConfig[RXDC_MAX_GAIN_PERMUTATIONS][2] =
{
	0x04120c00, 0x00080032,
	0x03120c00, 0x00080032,
	0x02120c00, 0x00080032,
	0x01120c00, 0x00080032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
	0x00120c00, 0x00040032,
};	
#endif //ifdef RFIC_WRX300
uint8 gainPermutationsLUT[RXDC_MAX_NUM_OF_GAIN_PERMUTATIONS];
uint8 gainVecSize = RXDC_MAX_GAIN_PERMUTATIONS;

uint32 rxdcTimeLimit = 0;
uint32 rxdcStartTime;
uint32 rxdcEndTime;
uint8 nextPgc2GainIndex;
uint8 nextGainIndex;
uint8 nextDigitalGain;
uint8 pgc1index;

RxDC_Config_t	RxDCconfiguration;
RxDC_Params_t	RxDCparams;


/******************************************************************************/
/***						MACROs											***/
/******************************************************************************/
#define GET_GAINS_TABLE_LNA_VAL(x)					(HdkLnaIndex_e)((x & (0x30)) >> 4)
#define GET_GAINS_TABLE_PGC1_VAL(x)					(x &(0xF))


extern Calibration_params_otf_t Calibration_params_otf;	

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/ 
static void RxDcOffset_StoreResults(void);
void 	RxDC_copySpecialGainFromPSD(uint8 index);
void	RxDC_InitCalibration(void);
void	RxDC_InitAnt(uint8 ant);
void	RxDC_RestoreAnt(void);
void	RxDC_CalibrateAntennas(void);
void	RxDC_LoopPGC2gain(void);
void	RxDC_LoopLNAandPGC1Gains(void);
void	RxDC_ReadNextPGC2gain(uint8 index, int32* pgc2gain, uint32* configWord);
bool	RxDC_CalibrateDAC2(uint8 Pgc2gain, int8 dac2val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX], bool saveRef);
void	RxDC_ReadNextGainPermutation(uint8 index, uint32* gainWord, uint32* configWord);
void	RxDC_CalibrateDAC0(RficRxGains_t* rxGains, int8 dac0val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX]);
void	RxDC_WriteDigitalCorrection(void);
void	RxDC_WriteSpecialGain(uint8 index, uint8 lnaIndex, uint8 pgc1index, uint8 pgc2index, uint8 sw_chip);
void	RxDC_Restore(void);
void	RxDC_ExtractDAC2CalVector(uint32 gains, uint32 configWord);
void	RxDC_ExtractDAC0CalVector(uint32 gains, uint32 configWord);
void	RxDC_FindBestDac(uint8 searchedDac,uint8 maxSteps,uint8 initialStep, int8 startingDacs[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX],int32 minRef [MAX_NUM_OF_ANTENNAS][ RFIC_PATH_TYPE_MAX ],int8 bestDacs [MAX_NUM_OF_ANTENNAS][ RFIC_PATH_TYPE_MAX ], HdkLnaIndex_e lnaIndex);
bool	RxDC_VerifyDAC0calibration(RficRxGains_t* rxGains, int8 dac0vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX], int8 dac2vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX], int32 failTH, uint32 numSamples);
void	RxDC_SetCalMode(uint8 newCalMode);
bool    RxDC_shouldDoSWCal(int lnaIndex_toCal, int32 pgc1Index_toCal, int32 pgc2Index_toCal);
uint8   RxDC_getSelRxDC(uint32 bw);
void 	setCalibraitionToRF(void);
void 	getCalibraitionFromRF(void);


/******************************************************************************/
/***						Functions Definitions							***/
/******************************************************************************/

void RxDC_copySpecialGainFromPSD(uint8 index)
{
	uint8 lnaIndex;
	uint8 pgc1Index;
	uint8 pgc1Start;
	uint8 pgc1End;
	uint8 pgc2Index;
	uint8 gainIndex;
	uint32 gainPermutation;
	uint32 dcCancellationword;
	uint32 dcMode;
	for (gainIndex = 0,lnaIndex=0; gainIndex < RXDC_NUM_SPECIAL_GAIN_TABLE1; gainIndex++,lnaIndex++)
	{
		gainPermutation = pShramAddress[index];
		lnaIndex = (gainPermutation & RXDC_GAIN_WORD_LNA_MASK) >> RXDC_GAIN_WORD_LNA_SHIFT;
		pgc1Index = (gainPermutation & RXDC_GAIN_WORD_PGC1_MASK) >> RXDC_GAIN_WORD_PGC1_SHIFT;
		pgc2Index = (gainPermutation & RXDC_GAIN_WORD_PGC2_MASK) >> RXDC_GAIN_WORD_PGC2_SHIFT;
		RficDriver_SetDCspecialGainTo1stTable(gainIndex, lnaIndex, pgc1Index , pgc2Index);
		index += 2;
	}
	//ASSERT(0);
	lnaIndex = pShramAddress[index];
	pgc2Index = pShramAddress[index+2];
	pgc1Start = pShramAddress[index+4];
	pgc1End = pShramAddress[index+6];
	dcMode = pShramAddress[index+8];
	dcCancellationword = pShramAddress[index+10];
	
	RficDriver_SetDCspecialGainTo2ndTable(lnaIndex, pgc2Index, pgc1Start, pgc1End);
	RficDriver_writeDigitalCorrectionToPhy(dcCancellationword);
	RficDriver_writeDcModeToPhy(dcMode);
}


/*
writeDigitalCorrectionToPhy(uint32 dcCancellationword)

typedef enum DcMode
{
    CHIP_MODE = 0,
    SW_MODE,
    MIX_MODE,
} DcMode_e;

writeDcModeToPhy(DcMode_e dcMode)
*/
void RxDC_ReadPSD()
{
	uint8 numRows;
	uint8 rowNum;
	uint8 index;
	uint32 gainPermutation;
	uint32 calConfigWord;
	uint16 dcLevel;
	for (index = 0; index < RXDC_NUM_PGC2_GAINS; index++)
	{
		PGC2GainsCalMode[index] = RXDC_CONFIG_CALIBRATION_MODE_NONE;
	}

	for (index = 0; index < RXDC_MAX_NUM_OF_GAIN_PERMUTATIONS; index++)
	{
		gainPermutationsLUT[index] = RXDC_NON_EXIST_GAIN;
	}

	gainVecSize = 0;
	numRows = pShramAddress[PSD_NUM_OF_ROWS_INDEX];
	index = PSD_TABLE_START_INDEX;
	for (rowNum = 0; rowNum < numRows; rowNum++)
	{
		gainPermutation = pShramAddress[index];
		calConfigWord = pShramAddress[index + 1];
		dcLevel = (calConfigWord & RXDC_CONFIG_MAX_DC_LEVEL_MASK) >> RXDC_CONFIG_MAX_DC_LEVEL_SHIFT;
		if(dcLevel)
		{	
			RxDC_ExtractDAC2CalVector(gainPermutation, calConfigWord);
			RxDC_ExtractDAC0CalVector(gainPermutation, calConfigWord);
		}
		else//identifier to start of special gains table
		{
			RxDC_copySpecialGainFromPSD(index);
			rowNum=numRows;
		}
		index += 2;
	}
	
}

void RxDC_ExtractDAC0CalVector(uint32 gains,uint32 configWord)
{
	uint8 lnaGain;
	int8 pgc1gain;
	int8 pgc1index;
	uint8 vecIndex;
	int8 lutIndex;
	uint16 dcLevel;
	uint16 curDcLevel;
	uint8 calMode;
	uint8 curCalMode;
	uint8 writeToRfic;
	uint8 resBypassMode;
	uint8 curResBypassMode;
	uint32 newConfigWord;
	uint32 curConfigWord;
	uint8 curNumSamples;

	lnaGain = (gains & RXDC_GAIN_WORD_LNA_MASK) >> RXDC_GAIN_WORD_LNA_SHIFT;
	pgc1gain = (gains & RXDC_GAIN_WORD_PGC1_MASK) >> RXDC_GAIN_WORD_PGC1_SHIFT;
	pgc1index = (pgc1gain + PGC1_INDEX_TO_GAIN_BIAS) / PGC1_INDEX_TO_GAIN_FACTOR;
	lutIndex = pgc1index + (lnaGain * RXDC_NUM_PGC1_GAINS);

	if (gainPermutationsLUT[lutIndex] == RXDC_NON_EXIST_GAIN)
	{
		ASSERT(gainVecSize < RXDC_MAX_GAIN_PERMUTATIONS);
		newConfigWord = configWord;
		vecIndex = gainVecSize;
		gainPermutationsLUT[lutIndex] = vecIndex;
		gainVecSize++;
	}
	else //gain permutation already exists in table
	{
		vecIndex = gainPermutationsLUT[lutIndex];
		curConfigWord = LnaPgcGainsConfig[vecIndex][1];
		curDcLevel = (curConfigWord & RXDC_CONFIG_MAX_DC_LEVEL_MASK) >> RXDC_CONFIG_MAX_DC_LEVEL_SHIFT;
		curCalMode = (curConfigWord & RXDC_CONFIG_CALIBRATION_MODE_MASK) >> RXDC_CONFIG_CALIBRATION_MODE_SHIFT;
		curResBypassMode = (curConfigWord & RXDC_CONFIG_RESIDUAL_BYPASS_MASK) >> RXDC_CONFIG_RESIDUAL_BYPASS_SHIFT;
		dcLevel = (configWord & RXDC_CONFIG_MAX_DC_LEVEL_MASK) >> RXDC_CONFIG_MAX_DC_LEVEL_SHIFT;
		calMode = (configWord & RXDC_CONFIG_CALIBRATION_MODE_MASK) >> RXDC_CONFIG_CALIBRATION_MODE_SHIFT;
		writeToRfic = (configWord & RXDC_CONFIG_WRITE_TO_RFIC_MASK) >> RXDC_CONFIG_WRITE_TO_RFIC_SHIFT;
		resBypassMode = (configWord & RXDC_CONFIG_RESIDUAL_BYPASS_MASK) >> RXDC_CONFIG_RESIDUAL_BYPASS_SHIFT;
		curNumSamples = (configWord & RXDC_CONFIG_POWER_NUM_SAMPLES_MASK) >> RXDC_CONFIG_POWER_NUM_SAMPLES_SHIFT;

		if (calMode == RXDC_CONFIG_CALIBRATION_MODE_CHIP) //chip calibration mode
		{
			if (curCalMode == RXDC_CONFIG_CALIBRATION_MODE_SW)
			{
				calMode = curCalMode;
			}
		}
		else if (calMode == RXDC_CONFIG_CALIBRATION_MODE_NONE)
		{
			calMode = curCalMode;
		}

		if (resBypassMode == TRUE)
		{
			resBypassMode = curResBypassMode;
		}

		if (dcLevel > curDcLevel)
		{
			dcLevel = curDcLevel;
		}

		newConfigWord = (dcLevel << RXDC_CONFIG_MAX_DC_LEVEL_SHIFT) |
						(calMode << RXDC_CONFIG_CALIBRATION_MODE_SHIFT) |
						(writeToRfic << RXDC_CONFIG_WRITE_TO_RFIC_SHIFT) |
						(resBypassMode << RXDC_CONFIG_RESIDUAL_BYPASS_SHIFT) |
						(curNumSamples << RXDC_CONFIG_POWER_NUM_SAMPLES_SHIFT);
	}

	LnaPgcGainsConfig[vecIndex][0] = gains;
	LnaPgcGainsConfig[vecIndex][1] = newConfigWord;
}

void RxDC_ExtractDAC2CalVector(uint32 gains,uint32 configWord)
{
	int8 pgc2gain;
	uint8 calMode;
	int8 pgc2index;

	pgc2gain = (gains & RXDC_GAIN_WORD_PGC2_MASK) >> RXDC_GAIN_WORD_PGC2_SHIFT;
	calMode = (configWord & RXDC_CONFIG_CALIBRATION_MODE_MASK) >> RXDC_CONFIG_CALIBRATION_MODE_SHIFT;
	if (pgc2gain == PGC2_HIGH_GAIN_DB)
	{
		pgc2index = RXDC_NUM_PGC2_GAINS - 1;
	}
	else
	{
		pgc2index = (pgc2gain + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR;
	}

	if (calMode == RXDC_CONFIG_CALIBRATION_MODE_CHIP)
	{
		if (PGC2GainsCalMode[pgc2index] == RXDC_CONFIG_CALIBRATION_MODE_NONE)
		{
			PGC2GainsCalMode[pgc2index] = RXDC_CONFIG_CALIBRATION_MODE_CHIP;
		}
	}
	else
	{
		PGC2GainsCalMode[pgc2index] = RXDC_CONFIG_CALIBRATION_MODE_SW;
	}
}

void RxDC_CalibrateAntennas(void)
{
	uint8 bw;
	uint8 ant;
	uint8 antMask = Hdk_GetRxAntMask();
	bw = HDK_getChannelWidth();
	RxDCparams.currentBW = bw;
	
	RxDC_LoopPGC2gain();
	
	RxDC_LoopLNAandPGC1Gains();

	RxDC_SetCalMode(RXDC_CAL_MODE_NONE);
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			if (RxDcOffset_GetStatus(ant, RFIC_PATH_TYPE_I) == CAL_STATUS_IN_PROCESS)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_PASS);
			}
			if (RxDcOffset_GetStatus(ant, RFIC_PATH_TYPE_Q) == CAL_STATUS_IN_PROCESS)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_PASS);
			}
		}
	}
}

void RxDC_LoopPGC2gain()
{
	int32 pgc2gain;
	int8 dac2val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint32 configWord;
	RficRxGains_t rxGains;
	bool calResult;
	
	memset(&rxGains, 0, sizeof(RficRxGains_t));
	rxGains.pgc1Gain = RXDC_PGC1_GAIN_FOR_DAC2_CAL;
	rxGains.pgc3Gain = RXDC_PGC3_GAIN_FOR_DAC2_CAL;
	rxGains.lnaIndex = (HdkLnaIndex_e)RXDC_LNA_GAIN_FOR_DAC2_CAL;
	PhyDrv_GetLnaActiveValue(rxGains.lnaIndex, &rxGains.lnaActiveValue);

	rxdcEndTime = Pac_TimGetTsfLow() - rxdcStartTime;
	while ((rxdcTimeLimit > rxdcEndTime) && (nextPgc2GainIndex < RXDC_NUM_PGC2_GAINS))
	{
		RxDC_ReadNextPGC2gain(nextPgc2GainIndex, &pgc2gain, &configWord);
		rxGains.pgc2Gain = pgc2gain;
		RficDriver_SetRxGains((Antenna_e)RxDCparams.currentAnt, &rxGains);
		if ((configWord == RXDC_CONFIG_CALIBRATION_MODE_SW) || (pgc2gain == RXDC_REF_PGC2_GAIN))
		{
			RxDC_SetCalMode(RXDC_CAL_MODE_SW);
			RficDriver_SetRxPgcInputPowerState(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_OFF); //short PGC1
			calResult = RxDC_CalibrateDAC2(nextPgc2GainIndex, dac2val, TRUE);
			RficDriver_SetRxPgcInputPowerState(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_ON); //short PGC1
			if (calResult == TRUE)
			{
				//PhyCalDrv_DCTable2Set(RxDCparams.currentAnt, RxDCparams.currentBW, nextPgc2GainIndex, dac2val);
				PhyCalDrv_DCTable2SetAll(nextPgc2GainIndex,dac2val);
			}
		}
		
		if (configWord == RXDC_CONFIG_CALIBRATION_MODE_CHIP)
		{
			RxDC_SetCalMode(RXDC_CAL_MODE_CHIP);
			RficDriver_RxDcActivateDac2ChipCalAll(pgc2gain,
												RxDCconfiguration.param_STM_DAC2_config[0],
												RxDCconfiguration.numRetriesDac2Chip,
												RxDCconfiguration.dac2failTH,
												RxDCconfiguration.numOfCorrelatorSamples);
		}

		nextPgc2GainIndex++;
		rxdcEndTime = Pac_TimGetTsfLow() - rxdcStartTime;
	}
}

void RxDC_LoopLNAandPGC1Gains()
{
	int32 pgc2gain;
	int8 dac2val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 dac0val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 dac2res[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 dac2prev[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint32 configWord;
	uint32 gainsWord;
	RficRxGains_t rxGains;
	uint8 flag;
	uint8 tries;
	int32 calThreshold;
	bool calRes;
	uint32 numSamples;
	uint8 powerNumSamples;
	uint8 dcMode;
	uint8 writeToRfic;
	bool mixModeAndSW;
	uint8 pgc1Index;
	uint8 pgc2Index;
	uint8 ant;

	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);
	
	mixModeAndSW = FALSE;
	numSamples = 0;

	memset(&rxGains, 0, sizeof(RficRxGains_t));
	dcMode = readDcModeFromPhy();//for wave 400 or wrx300 return MODE_SW
	RficDriver_RxDcInitBW(HDK_getChannelWidth());
	rxdcEndTime = Pac_TimGetTsfLow() - rxdcStartTime;
	while ((rxdcTimeLimit > rxdcEndTime) && (nextGainIndex < gainVecSize))
	{
		RxDC_ReadNextGainPermutation(nextGainIndex, &gainsWord, &configWord);
		rxGains.lnaIndex = (HdkLnaIndex_e)((gainsWord & RXDC_GAIN_WORD_LNA_MASK) >> RXDC_GAIN_WORD_LNA_SHIFT);
		rxGains.pgc1Gain = (int32)(int8)((gainsWord & RXDC_GAIN_WORD_PGC1_MASK) >> RXDC_GAIN_WORD_PGC1_SHIFT);
		rxGains.pgc2Gain = (int32)(int8)((gainsWord & RXDC_GAIN_WORD_PGC2_MASK) >> RXDC_GAIN_WORD_PGC2_SHIFT);
		rxGains.pgc3Gain = (int32)(int8)((gainsWord & RXDC_GAIN_WORD_PGC3_MASK) >> RXDC_GAIN_WORD_PGC3_SHIFT);
		PhyDrv_GetLnaActiveValue(rxGains.lnaIndex, &rxGains.lnaActiveValue);
		SetRxGains((AntennaBitmaps_e)antMask,rxGains.lnaIndex,rxGains.pgc1Gain,rxGains.pgc2Gain,rxGains.pgc3Gain);
		calThreshold = (configWord & RXDC_CONFIG_MAX_DC_LEVEL_MASK) >> RXDC_CONFIG_MAX_DC_LEVEL_SHIFT;
		//flag = (configWord & RXDC_CONFIG_CALIBRATION_MODE_MASK) >> RXDC_CONFIG_CALIBRATION_MODE_SHIFT;
		powerNumSamples = (configWord & RXDC_CONFIG_POWER_NUM_SAMPLES_MASK) >> RXDC_CONFIG_POWER_NUM_SAMPLES_SHIFT;


		pgc1Index = rxGains.pgc1Gain;
		if (rxGains.pgc2Gain == PGC2_HIGH_GAIN_DB)
		{
			pgc2Index = RXDC_NUM_PGC2_GAINS - 1;
		}
		else
		{
			pgc2Index = (rxGains.pgc2Gain + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR;
		}

		
		if (MIX_MODE == dcMode)
		{
			mixModeAndSW = RxDC_shouldDoSWCal(rxGains.lnaIndex, pgc1Index, pgc2Index);	
		}
		if ((SW_MODE == dcMode) || (TRUE == mixModeAndSW))
		{
			tries = 0;
			calRes = FALSE;
			if (numSamples != (1 << powerNumSamples))
			{
				if ((RxDCparams.curCalibType == CLBR_TYPE_ONLINE) && (powerNumSamples > ONLINE_MAX_POWER_NUM_SAMPLES))
				{
					powerNumSamples = ONLINE_MAX_POWER_NUM_SAMPLES;
				}
				numSamples = 1 << powerNumSamples;
				PhyCalDrv_CorrInit(DC_MODE, antMask, numSamples, 0, 0);
			}
			RxDC_SetCalMode(RXDC_CAL_MODE_SW);
			while ((calRes == FALSE) && (tries < RxDCconfiguration.numRetriesDac1Chip))
			{
				RxDC_CalibrateDAC0(&rxGains, dac0val);
				flag = (configWord & RXDC_CONFIG_RESIDUAL_BYPASS_MASK) >> RXDC_CONFIG_RESIDUAL_BYPASS_SHIFT;
				if (flag == FALSE)
				{
					pgc2gain = (rxGains.pgc2Gain + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR;
					RxDC_CalibrateDAC2(pgc2gain, dac2val, FALSE);
					PhyCalDrv_DCTable2GetAll(pgc2gain, dac2prev);
					
					for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
					{
						if (((1 << ant) & antMask) != FALSE)
						{
							dac2res[ant][RFIC_PATH_TYPE_I] = dac2val[ant][RFIC_PATH_TYPE_I] - dac2prev[ant][RFIC_PATH_TYPE_I];
							dac2res[ant][RFIC_PATH_TYPE_Q] = dac2val[ant][RFIC_PATH_TYPE_Q] - dac2prev[ant][RFIC_PATH_TYPE_Q];
						}
					}
				}
				else
				{
					for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
					{
						if (((1 << ant) & antMask) != FALSE)
						{
							dac2res[ant][RFIC_PATH_TYPE_I] = 0;
							dac2res[ant][RFIC_PATH_TYPE_Q] = 0;
						}
					}
				}
				calRes = RxDC_VerifyDAC0calibration(&rxGains, dac0val, dac2val, calThreshold, numSamples);///////////////////
				tries++;
			}

			if (calRes == FALSE)
			{
				for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
				{
					if (((1 << ant) & antMask) != FALSE)
					{
						dac2res[ant][RFIC_PATH_TYPE_I] = dac2val[ant][RFIC_PATH_TYPE_I] - dac2prev[ant][RFIC_PATH_TYPE_I];
						dac2res[ant][RFIC_PATH_TYPE_Q] = dac2val[ant][RFIC_PATH_TYPE_Q] - dac2prev[ant][RFIC_PATH_TYPE_Q];
					}
				}
			}
			writeToRfic = (configWord & RXDC_CONFIG_WRITE_TO_RFIC_MASK) >> RXDC_CONFIG_WRITE_TO_RFIC_SHIFT;
			if (writeToRfic)
			{
				RficDriver_writeSWCalToRfic(rxGains.lnaIndex, dac0val);//
			}
			PhyCalDrv_DCTable1SetAll(rxGains.pgc1Gain, rxGains.lnaIndex, dac0val, dac2res);
			
		}
		else if ((RXDC_CONFIG_CALIBRATION_MODE_CHIP == dcMode) || (FALSE == mixModeAndSW))//i can write here only "else"
		{
			RxDC_SetCalMode(RXDC_CAL_MODE_CHIP);
			RficDriver_RxDcActivateDac0ChipCalAll(&rxGains,
												RxDCconfiguration.param_STM_DAC1_config,
												calThreshold,
												RxDCconfiguration.numRetriesDac1Chip,
												RxDCconfiguration.numOfCorrelatorSamples);
		}
		nextGainIndex++;
		rxdcEndTime = Pac_TimGetTsfLow() - rxdcStartTime;
	}
}

void RxDC_ReadNextPGC2gain(uint8 index,int32 * pgc2gain,uint32 * configWord)
{
	DEBUG_ASSERT(index < RXDC_NUM_PGC2_GAINS);
	if (index == (RXDC_NUM_PGC2_GAINS - 1))
	{
		*pgc2gain = PGC2_HIGH_GAIN_DB;
	}
	else
	{
		*pgc2gain = (index * PGC2_INDEX_TO_GAIN_FACTOR) - PGC2_INDEX_TO_GAIN_BIAS;
	}
	*configWord = PGC2GainsCalMode[index];
}

void RxDC_ReadNextGainPermutation(uint8 index,uint32* gainWord,uint32 * configWord)
{
	DEBUG_ASSERT(index < RXDC_MAX_GAIN_PERMUTATIONS);
	*gainWord = LnaPgcGainsConfig[index][0];
	*configWord = LnaPgcGainsConfig[index][1];
}

//************************************
// Method:    finishCalibration
// Purpose:   TBD
// Parameter: void
// Returns:   void
//************************************
static void finishCalibration(void)
{
//temporary block for WRX500 till init & restore calibrations are fixed
#ifdef RFIC_WRX300
	//RficDriver_LoadRegDB();
#endif	
}

void RxDC_Restore()
{
	RficDriver_RxDcStopDcCal();
	finishCalibration();
}

void RxDC_FindBestDac(uint8 searchedDac,uint8 maxSteps,uint8 initialStep, int8 startingDacs[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX],int32 minRef[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX],int8 bestDacs [MAX_NUM_OF_ANTENNAS][ RFIC_PATH_TYPE_MAX ], HdkLnaIndex_e lnaIndex)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	int32 bestResult[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint8 step;
	uint8 stepSize[MAX_NUM_OF_ANTENNAS];
	int8 nextStep;
	int32 dcDiff[RFIC_PATH_TYPE_MAX];
	int8 curDacs[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint8 iqIndex;
	uint8 ant;
	int8 DacMaxValue;
	int8 DacMinValue;
	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	if (RFIC_DAC1 == searchedDac)
	{
		DacMaxValue = RXDC_DAC1_MAX_VALUE;
		DacMinValue = RXDC_DAC1_MIN_VALUE;
	}
	else if (RFIC_DAC2 == searchedDac)
	{
		DacMaxValue = RXDC_DAC2_MAX_VALUE;
		DacMinValue = RXDC_DAC2_MIN_VALUE;
	}
	
	for (ant=0; ant<MAX_NUM_OF_ANTENNAS; ant++)
	{	
		if (((1 << ant) & antMask) != FALSE)
		{
			bestDacs[ant][RFIC_PATH_TYPE_I] = startingDacs[ant][RFIC_PATH_TYPE_I];
			bestDacs[ant][RFIC_PATH_TYPE_Q] = startingDacs[ant][RFIC_PATH_TYPE_Q];
			curDacs[ant][RFIC_PATH_TYPE_I] = startingDacs[ant][RFIC_PATH_TYPE_I];
			curDacs[ant][RFIC_PATH_TYPE_Q] = startingDacs[ant][RFIC_PATH_TYPE_Q];
			bestResult[ant][RFIC_PATH_TYPE_I] = 0x7FFFFFFF;
			bestResult[ant][RFIC_PATH_TYPE_Q] = 0x7FFFFFFF;
			stepSize[ant] = initialStep;
		}
	}

	for (step = 0; step < maxSteps; step++)
	{
		RficDriver_SetRxDACs((AntennaBitmaps_e)(antMask), RFIC_PATH_TYPE_I_AND_Q, (RficDacIndex_e)searchedDac, curDacs, (HdkLnaIndex_e)lnaIndex);
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		for (ant=0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != FALSE)
			{
				dcDiff[RFIC_PATH_TYPE_I] = results[ant].II - minRef[ant][RFIC_PATH_TYPE_I];
				dcDiff[RFIC_PATH_TYPE_Q] = results[ant].QQ - minRef[ant][RFIC_PATH_TYPE_Q];
				for (iqIndex = RFIC_PATH_TYPE_I; iqIndex < RFIC_PATH_TYPE_MAX; iqIndex++)
				{
					if (Abs(dcDiff[iqIndex]) < Abs(bestResult[ant][iqIndex]))
					{
						bestResult[ant][iqIndex] = dcDiff[iqIndex];
						bestDacs[ant][iqIndex] = curDacs[ant][iqIndex];
					}
		
					if (dcDiff[iqIndex] < 0)
					{
						nextStep = stepSize[ant];
					}
					else
					{
						nextStep = -stepSize[ant];
					}
					curDacs[ant][iqIndex] += nextStep;
					
					if (curDacs[ant][iqIndex] > DacMaxValue)//verify in online calibration that dac is in the range
					{
						curDacs[ant][iqIndex] = DacMaxValue;
					} 
					else if(curDacs[ant][iqIndex] < DacMinValue)
					{
						curDacs[ant][iqIndex] = DacMinValue;
					}
					
				}
				if (stepSize[ant] > 1)
				{
					stepSize[ant] >>= 1;
				}
			}
		}
		
	}

	for (ant=0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			curDacs[ant][RFIC_PATH_TYPE_I] = bestDacs[ant][RFIC_PATH_TYPE_I];
			curDacs[ant][RFIC_PATH_TYPE_Q] = bestDacs[ant][RFIC_PATH_TYPE_Q];
			RficDriver_SetRxDACs((AntennaBitmaps_e)(1 << ant), RFIC_PATH_TYPE_I_AND_Q, (RficDacIndex_e)searchedDac, curDacs, (HdkLnaIndex_e)lnaIndex);
			minRef[ant][RFIC_PATH_TYPE_I] = bestResult[ant][RFIC_PATH_TYPE_I];
			minRef[ant][RFIC_PATH_TYPE_Q] = bestResult[ant][RFIC_PATH_TYPE_Q];
		}
	}
}

//************************************
// Method:    run
// Purpose:   Internal master function to execute single algorithm session
// Parameter: IN ClbrType_e calibType - OFFLINE/ONLINE
// Parameter: IN DcElements_t * pElements
// Parameter: IN uint32 scanType
// Returns:   RetVal_e
//************************************
bool RxDC_VerifyDAC0calibration(RficRxGains_t * rxGains,int8 dac0vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX],int8 dac2vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX],int32 failTH, uint32 numSamples)
{
	//uint8 antMask = PSD_getDataField(PSD_FIELD_RX_ANTS_MASK);
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelI[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelQ[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelDefault;
	int32 pgc2index;
	int8 dacsArray[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 defaultDac1[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 defaultDac2[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 defaultDac2base[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 tmpVal;
	uint8 ant;
	uint8 dcRes_counter;
	bool dcRes;
	
	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	dcRes_counter=0;

	//ant = RxDCparams.currentAnt;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			dacsArray[ant][RFIC_PATH_TYPE_I] = dac0vals[ant][RFIC_PATH_TYPE_I];
			dacsArray[ant][RFIC_PATH_TYPE_Q] = dac0vals[ant][RFIC_PATH_TYPE_Q];
		}
	}
	RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC1, dacsArray, rxGains->lnaIndex);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			dacsArray[ant][RFIC_PATH_TYPE_I] = dac2vals[ant][RFIC_PATH_TYPE_I];
			dacsArray[ant][RFIC_PATH_TYPE_Q] = dac2vals[ant][RFIC_PATH_TYPE_Q];
		}
	}
	RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC2, dacsArray, rxGains->lnaIndex);

	PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			dcLevelI[ant] = Abs(results[ant].II) / numSamples;
			dcLevelI[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			dcLevelQ[ant] = Abs(results[ant].QQ) / numSamples;
			dcLevelQ[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			if ((dcLevelI[ant] < failTH) && (dcLevelQ[ant] < failTH))
			{
				dcRes_counter++;
			}
		}
	}
	if (maxAnts == dcRes_counter)
	{
		dcRes = TRUE;
	}
	else
	{
		dcRes = FALSE;
		PhyCalDrv_DCTable1GetAll(rxGains->pgc1Gain, rxGains->lnaIndex, defaultDac1, defaultDac2);
		pgc2index = (rxGains->pgc2Gain + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR;
		PhyCalDrv_DCTable2GetAll(pgc2index, defaultDac2base);

		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != FALSE)
			{
				dacsArray[ant][RFIC_PATH_TYPE_I] = defaultDac1[ant][RFIC_PATH_TYPE_I];
				dacsArray[ant][RFIC_PATH_TYPE_Q] = defaultDac1[ant][RFIC_PATH_TYPE_Q];
			}
		}
		RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC1, dacsArray, rxGains->lnaIndex);
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != FALSE)
			{
				tmpVal = defaultDac2[ant][RFIC_PATH_TYPE_I] + defaultDac2base[ant][RFIC_PATH_TYPE_I];
				if ((tmpVal <= RXDC_DAC2_MAX_VALUE) && (tmpVal >= RXDC_DAC2_MIN_VALUE))
				{
					dacsArray[ant][RFIC_PATH_TYPE_I] = tmpVal;
				}
				else
				{
					dacsArray[ant][RFIC_PATH_TYPE_I] = defaultDac2base[ant][RFIC_PATH_TYPE_I];
				}
				tmpVal = defaultDac2[ant][RFIC_PATH_TYPE_Q] + defaultDac2base[ant][RFIC_PATH_TYPE_Q];
				if ((tmpVal <= RXDC_DAC2_MAX_VALUE) && (tmpVal >= RXDC_DAC2_MIN_VALUE))
				{
					dacsArray[ant][RFIC_PATH_TYPE_Q] = tmpVal;
				}
				else
				{
					dacsArray[ant][RFIC_PATH_TYPE_Q] = defaultDac2base[ant][RFIC_PATH_TYPE_I];
				}
			}
		}
		RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC2, dacsArray, rxGains->lnaIndex);
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != FALSE)
			{
				if (dcLevelI[ant] >= failTH)
				{
					RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
					dcLevelDefault = Abs(results[ant].II) / numSamples;
					dcLevelDefault >>= RXDC_CORRELATOR_SHIFT_FACTOR;
					if (dcLevelDefault < dcLevelI[ant])
					{
						dac0vals[ant][RFIC_PATH_TYPE_I] = defaultDac1[ant][RFIC_PATH_TYPE_I];
						dac2vals[ant][RFIC_PATH_TYPE_I] = dacsArray[ant][RFIC_PATH_TYPE_I];
					}
				}
				if (dcLevelQ[ant] >= failTH)
				{
					RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
					dcLevelDefault = Abs(results[ant].QQ) / numSamples;
					dcLevelDefault >>= RXDC_CORRELATOR_SHIFT_FACTOR;
					if (dcLevelDefault < dcLevelQ[ant])
					{
						dac0vals[ant][RFIC_PATH_TYPE_Q] = defaultDac1[ant][RFIC_PATH_TYPE_Q];
						dac2vals[ant][RFIC_PATH_TYPE_Q] = dacsArray[ant][RFIC_PATH_TYPE_Q];
					}
				}
			}
		}
	}
	return dcRes;
}

//************************************
// Method:    run
// Purpose:   Internal master function to execute single algorithm session
// Parameter: IN ClbrType_e calibType - OFFLINE/ONLINE
// Parameter: IN DcElements_t * pElements
// Parameter: IN uint32 scanType
// Returns:   RetVal_e
//************************************
RetVal_e run( IN ClbrType_e calibType, IN DcElements_t* pElements)
{
	uint8 antMask = Hdk_GetRxAntMask();
	RetVal_e retVal;
	uint8 ant;
	uint32 rxdcEndTimeCancelation;
	uint32 rxdcTimeLimitCancelation;
	CalDataState_e calStoreState;	 

	rxdcStartTime = Pac_TimGetTsfLow();
	RxDCparams.curCalibType = calibType;
	
	
	if (calibType == CLBR_TYPE_OFFLINE)
	{
		rxdcTimeLimit = RXDC_OFFLINE_CAL_TIME_LIMIT;
		rxdcTimeLimitCancelation = RXDC_OFFLINE_CAL_TIME_LIMIT;
		nextPgc2GainIndex = 0;
		nextGainIndex = 0;
		nextDigitalGain = 0;
	}
	else //online
	{
		rxdcTimeLimit = RXDC_ONLINE_CAL_TIME_LIMIT;
		rxdcTimeLimitCancelation = RXDC_ONLINE_CAL_TIME_LIMIT_CANCELLATION;
	}

	RxDC_InitCalibration();
	AFE_RestoreFromLoopBack();
	AFE_SetLoopBack((Antenna_e)0, (Antenna_e)1, AFE_MODE_RX, 0);
	rxdcEndTime = 0;
	
	for (ant=0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_IN_PROCESS);
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_IN_PROCESS);
			RxDC_InitAnt(ant);
		}
	}
	RxDC_CalibrateAntennas();
	rxdcEndTimeCancelation = Pac_TimGetTsfLow() - rxdcStartTime;
	retVal = RET_VAL_FRAGMENTED;
	//check if done
	if ((nextGainIndex == gainVecSize) && (rxdcEndTimeCancelation < rxdcTimeLimitCancelation))//didnt pass more the 2.5ms so remain 2.5ms
	{
		if (RficDriver_readDigitalCorrectionfromPhy() == TRUE)
		{
			RxDC_WriteDigitalCorrection();
		}
		else
		{
			nextDigitalGain = RXDC_MAX_NUM_SPECIAL_GAINS;
		}
		if (nextDigitalGain == RXDC_MAX_NUM_SPECIAL_GAINS)
		{
			nextDigitalGain = 0;
			nextGainIndex = 0;
			nextPgc2GainIndex = 0;
			retVal = RET_VAL_SUCCESS;
		}
	}

	RxDC_Restore();
	RficDriver_StoreRegDB();

    /* Store the calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if(calStoreState == CAL_DATA_STATE_STORE)
	{
		RxDcOffset_StoreResults();
	}    

	return retVal;
}

	//do digcancelation

bool RxDC_CalibrateDAC2(uint8 Pgc2gain,int8 dac2val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX], bool saveRef)
{
	//uint8 antMask = PSD_getDataField(PSD_FIELD_RX_ANTS_MASK);
	uint32 dcLevelI[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelQ[MAX_NUM_OF_ANTENNAS];
	int32 minRef[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 prevDac[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint8 maxSteps;
	uint8 initialStep;
	uint8 tryIndex;
	int success_counter;
	int ant;
	bool success;

	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);
	
	tryIndex = 0;
	success_counter= 0;
	success=FALSE;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			if (saveRef == TRUE)
			{
				minRef[ant][RFIC_PATH_TYPE_I] = 0;
				minRef[ant][RFIC_PATH_TYPE_Q] = 0;
			}
			else
			{
				minRef[ant][RFIC_PATH_TYPE_I] = RxDCparams.minRef[ant][RFIC_PATH_TYPE_I];
				minRef[ant][RFIC_PATH_TYPE_Q] = RxDCparams.minRef[ant][RFIC_PATH_TYPE_Q];
			}
		}
	}
	while ((success == FALSE) && (tryIndex < RxDCconfiguration.numRetriesDac2Sw))//to update numRetriesDac2Swto be higher because move on all antennas.
	{
		//get DAC values from table (for online)
		if (RxDCparams.curCalibType == CLBR_TYPE_ONLINE)
		{
			//PhyCalDrv_DCTable2Get(RxDCparams.currentAnt, RxDCparams.currentBW, Pgc2gain, prevDac);//prevDac--problem because maybe should give it prev dac before any antenna caibration
			PhyCalDrv_DCTable2GetAll(Pgc2gain, prevDac);
			maxSteps = RXDC_ONLINE_MAX_STEPS;
			initialStep = RXDC_ONLINE_INITIAL_STEP;
		}
		else
		{
			for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
			{
				if (((1 << ant) & antMask) != FALSE)
				{
					prevDac[ant][RFIC_PATH_TYPE_I] = 0;
					prevDac[ant][RFIC_PATH_TYPE_Q] = 0;
				}
			}
			maxSteps = RXDC_DAC2_OFFLINE_MAX_STEPS;
			initialStep = RXDC_DAC2_OFFLINE_INITIAL_STEP;
		}
		
		//min search
		RxDC_FindBestDac(RFIC_DAC2, maxSteps, initialStep, prevDac, minRef, dac2val, (HdkLnaIndex_e)0);
		success_counter = 0;
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
		if (((1 << ant) & antMask) != FALSE)
			{
				dcLevelI[ant] = Abs(minRef[ant][RFIC_PATH_TYPE_I]) / RxDCconfiguration.numOfCorrelatorSamples;
				dcLevelI[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
				dcLevelQ[ant] = Abs(minRef[ant][RFIC_PATH_TYPE_Q]) / RxDCconfiguration.numOfCorrelatorSamples;
				dcLevelQ[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
				if ((dcLevelI[ant] < RxDCconfiguration.dac2failTH) && (dcLevelQ[ant] < RxDCconfiguration.dac2failTH))
				{
					success_counter++;
				}
			}
		}
		if (maxAnts == success_counter)
		{
			success = TRUE;
		}
		tryIndex++;
	}
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			if (dcLevelI[ant] >= RxDCconfiguration.dac2failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
			}
			if (dcLevelQ[ant] >= RxDCconfiguration.dac2failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
			}
			if ((saveRef == TRUE) && (Pgc2gain == RXDC_REF_PGC2_INDEX))
			{
				RxDCparams.minRef[ant][RFIC_PATH_TYPE_I] = minRef[ant][RFIC_PATH_TYPE_I];
				RxDCparams.minRef[ant][RFIC_PATH_TYPE_Q] = minRef[ant][RFIC_PATH_TYPE_Q];
			}
		}
	}
	return success;
}

void RxDC_CalibrateDAC0(RficRxGains_t* rxGains, int8 dac0val[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX])
{
	int32 minRef[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 prevDac0[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 prevDac2[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8 prevDac2Res[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint8 maxSteps;
	uint8 initialStep;
	int8 pgc2index;
	uint8 ant;
	uint8 iqIndex;

	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	pgc2index = (rxGains->pgc2Gain + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR;
	//get DAC values from table (for online)
	if (RxDCparams.curCalibType == CLBR_TYPE_ONLINE)
	{
		PhyCalDrv_DCTable1GetAll(rxGains->pgc1Gain, rxGains->lnaIndex, prevDac0, prevDac2Res);
		maxSteps = RXDC_ONLINE_MAX_STEPS;
		initialStep = RXDC_ONLINE_INITIAL_STEP;
	}
	else
	{
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != FALSE)
			{
				prevDac0[ant][RFIC_PATH_TYPE_I] = 0;
				prevDac0[ant][RFIC_PATH_TYPE_Q] = 0;
				prevDac2Res[ant][RFIC_PATH_TYPE_I] = 0;
				prevDac2Res[ant][RFIC_PATH_TYPE_Q] = 0;
			}
		}
		maxSteps = RXDC_DAC1_OFFLINE_MAX_STEPS;
		initialStep = RXDC_DAC1_OFFLINE_INITIAL_STEP;
	}
	PhyCalDrv_DCTable2GetAll(pgc2index, prevDac2);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			for (iqIndex = RFIC_PATH_TYPE_I; iqIndex < RFIC_PATH_TYPE_MAX; iqIndex++)
			{
				prevDac2[ant][iqIndex] += prevDac2Res[ant][iqIndex];
				if (prevDac2[ant][iqIndex] > RXDC_DAC2_MAX_VALUE)//verify in online calibration that dac is in the range
				{
					prevDac2[ant][iqIndex] = RXDC_DAC2_MAX_VALUE;
				} 
				else if(prevDac2[ant][iqIndex] < RXDC_DAC2_MIN_VALUE)
				{
					prevDac2[ant][iqIndex] = RXDC_DAC2_MIN_VALUE;
				}
			}
		}
	}
//	ANTENNA_BITMAP_ALL_ANTENNAS
	RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC2, prevDac2, HDK_LNA_INDEX_HIGH_GAIN);
	//min search
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			minRef[ant][RFIC_PATH_TYPE_I] = RxDCparams.minRef[ant][RFIC_PATH_TYPE_I];
			minRef[ant][RFIC_PATH_TYPE_Q] = RxDCparams.minRef[ant][RFIC_PATH_TYPE_Q];
		}
	}
	RxDC_FindBestDac(RFIC_DAC1, maxSteps, initialStep, prevDac0, minRef, dac0val, rxGains->lnaIndex);
}

bool RxDC_shouldDoSWCal(int lnaIndex_toCal, int32 pgc1index_toCal, int32 pgc2index_toCal){
	uint8 gainIndex;
	uint8 lnaIndex;
	uint8 pgc1index;
	uint8 pgc2index;
	uint8 pgc1start;
	uint8 pgc1end;
	bool  gainsExist;

	for (gainIndex = 0; gainIndex < RXDC_NUM_SPECIAL_GAIN_TABLE1; gainIndex++)
	{
		gainsExist = RficDriver_GetDCspecialGainFrom1stTable(gainIndex, &lnaIndex, &pgc1index, &pgc2index);
		if ((gainsExist == TRUE) && (lnaIndex_toCal == lnaIndex) && (pgc1index_toCal == pgc1index) && (pgc2index_toCal == pgc2index))
		{
			return TRUE;
		}
	}
	
	gainsExist = RficDriver_GetDCspecialGainFrom2ndTable(&lnaIndex, &pgc2index, &pgc1start, &pgc1end);
	if (gainsExist == TRUE)
	{
		pgc1index = pgc1start;
		while (pgc1index <= pgc1end)
		{
			if((lnaIndex_toCal == lnaIndex) && (pgc1index_toCal == pgc1index) && (pgc2index_toCal == pgc2index))
			{
				return TRUE;
			}
			pgc1index++;
		}
	}
	return FALSE;

}




void RxDC_WriteDigitalCorrection()
{
	uint32 rxdcEndTimeCancelation;
	uint8 lnaIndex;
	uint8 pgc2index;
	uint8 pgc1start;
	uint8 pgc1end;
	bool  gainsExist;
	uint8 dcMode;
	uint8 sw_chip;
	bool thereIsTime;
	uint8 pgc1index_1;
	
	thereIsTime=TRUE;
	///////////to send SetCalMode function by what register in Fhy says. if chip ->chip, otherwise, SW
	dcMode = readDcModeFromPhy();
	if ((MIX_MODE == dcMode) || (SW_MODE == dcMode))
	{
		sw_chip = RXDC_CAL_MODE_SW;
	}
	else//CHIP_MODE==dcMode
	{
		sw_chip = RXDC_CAL_MODE_CHIP;
	}
	RxDC_SetCalMode(sw_chip);
	while ((nextDigitalGain < RXDC_NUM_SPECIAL_GAIN_TABLE1) && (thereIsTime == TRUE))
	{
		gainsExist = RficDriver_GetDCspecialGainFrom1stTable(nextDigitalGain, &lnaIndex, &pgc1index_1, &pgc2index);
		rxdcEndTimeCancelation = Pac_TimGetTsfLow() - rxdcStartTime;
		if (rxdcEndTimeCancelation < rxdcTimeLimit)
		{
			if (gainsExist == TRUE)
			{
				RxDC_WriteSpecialGain(nextDigitalGain, lnaIndex, pgc1index_1, pgc2index, sw_chip);
			}
			nextDigitalGain++;
		}
		else 
		{
			thereIsTime = FALSE;
		}
	}
	gainsExist = RficDriver_GetDCspecialGainFrom2ndTable(&lnaIndex, &pgc2index, &pgc1start, &pgc1end);
	if (gainsExist == TRUE)
	{
		if (nextDigitalGain == RXDC_NUM_SPECIAL_GAIN_TABLE1)
		{
			pgc1index = pgc1start;
		}
		while ((pgc1index <= pgc1end) && (thereIsTime == TRUE))
		{
			rxdcEndTimeCancelation = Pac_TimGetTsfLow() - rxdcStartTime;
			if (rxdcEndTimeCancelation < rxdcTimeLimit)
			{
				RxDC_WriteSpecialGain(nextDigitalGain, lnaIndex, pgc1index, pgc2index, sw_chip);
				pgc1index++;
				nextDigitalGain++;
			}
			else 
			{
				thereIsTime = FALSE;
			}
		}
	}
	else
	{
		if (nextDigitalGain == RXDC_NUM_SPECIAL_GAIN_TABLE1)
		{
			nextDigitalGain = RXDC_MAX_NUM_SPECIAL_GAINS;
		}
	}
	if (pgc1index > pgc1end)
	{
		nextDigitalGain = RXDC_MAX_NUM_SPECIAL_GAINS;
	}
	RxDC_SetCalMode(RXDC_CAL_MODE_NONE);
}

void RxDC_WriteSpecialGain(uint8 index, uint8 lnaIndex,uint8 pgc1index,uint8 pgc2index, uint8 sw_chip)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	RficRxGains_t rxGains;
	int8  dac0vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8  dac2vals[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	int8  dac2res[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX];
	uint8 ant;

	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	rxGains.lnaIndex = (HdkLnaIndex_e)lnaIndex;
	rxGains.pgc1Gain = pgc1index;
	rxGains.pgc2Gain = pgc2index;
	PhyDrv_GetLnaActiveValue(rxGains.lnaIndex, &rxGains.lnaActiveValue);
	PhyCalDrv_DCTable2GetAll(pgc2index, dac2vals);
	PhyCalDrv_DCTable1GetAll(pgc1index,(HdkLnaIndex_e)lnaIndex, dac0vals, dac2res);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			dac2vals[ant][RFIC_PATH_TYPE_I] += dac2res[ant][RFIC_PATH_TYPE_I];
			dac2vals[ant][RFIC_PATH_TYPE_Q] += dac2res[ant][RFIC_PATH_TYPE_Q];
			RficDriver_SetRxGains((Antenna_e)ant, &rxGains);
		}
	}
	//to do only when SW to ask amir 
	if (sw_chip)
	{
		RficDriver_SetRxDACs((AntennaBitmaps_e)antMask, RFIC_PATH_TYPE_I_AND_Q, RFIC_DAC1, dac0vals,(HdkLnaIndex_e)lnaIndex);
	}
	PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != FALSE)
		{
			results[ant].II /= (int32)RxDCconfiguration.numOfCorrelatorSamples;
			results[ant].QQ /= (int32)RxDCconfiguration.numOfCorrelatorSamples;
		}
	}
	RficDriver_DCwriteSpecialGain(results, index, RxDCparams.currentBW);
}

void RxDC_InitCalibration()
{
	rxdcCalMode = RXDC_CAL_MODE_NONE;
	if (RxDCparams.curCalibType == CLBR_TYPE_OFFLINE)
	{
		PhyCalDriver_ResetDcTables();
		PhyCalDriver_ResetDcDigTable(HDK_getChannelWidth());
		RxDCconfiguration.param_STM_DAC1_config = 13;
		RxDCconfiguration.param_STM_DAC2_config[0] = 13;
	}
	else
	{
		RxDCconfiguration.param_STM_DAC1_config = 11;
		RxDCconfiguration.param_STM_DAC2_config[0] = 11;
	}

	RxDCconfiguration.numOfSamplesDac1[0] = 8192;
	RxDCconfiguration.numOfSamplesDac2 = 8192;
	RxDCconfiguration.numRetriesDac1Chip = 3;
	RxDCconfiguration.numRetriesDac1Sw = 3;
	RxDCconfiguration.numRetriesDac2Chip = 3;
	RxDCconfiguration.numRetriesDac2Sw = 3;
	RxDCconfiguration.dac2failTH = 100;
	RxDCconfiguration.numOfCorrelatorSamples = 8192;
}

void RxDC_InitAnt(uint8 ant)
{
	PhyCalDrv_CorrInit(DC_MODE, (1 << ant), RxDCconfiguration.numOfCorrelatorSamples, 0, 0);
	if (RxDCparams.curCalibType == CLBR_TYPE_OFFLINE)
	{
		RxDCparams.minRef[ant][RFIC_PATH_TYPE_I] = 0;
		RxDCparams.minRef[ant][RFIC_PATH_TYPE_Q] = 0;
	}
}

void RxDC_RestoreAnt()
{
}

void RxDC_SetCalMode(uint8 newCalMode)
{
	uint8 maxAnts;
	uint8 antMask;
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	if (rxdcCalMode != newCalMode)
	{
		if (rxdcCalMode != RXDC_CAL_MODE_NONE)
		{
			if (newCalMode == RXDC_CAL_MODE_SW)
			{
				AFE_RxDCrestoreAntAll();
				RficDriver_RxDcStopDcCal();
			}
			else if (newCalMode == RXDC_CAL_MODE_CHIP)
			{
				AFE_RxDCrestoreAntAll();
				PhyCalDrv_RestoreOperationalTx();
				AFE_RestoreFromLoopBack();
			}
			else //return to none
			{
				if (rxdcCalMode == RXDC_CAL_MODE_SW)
				{
					PhyCalDrv_RestoreOperationalTx();
				}
				AFE_RxDCrestoreAntAll();
				AFE_RestoreFromLoopBack();
				AFE_RxDCrestore();
			}
		}

		if (newCalMode == RXDC_CAL_MODE_SW)
		{
			AFE_RxDCinit();
			RficDriver_ActivateRxAntennas ((AntennaBitmaps_e)antMask); // Enable all receive antennas.
			RficDriver_setRffeMode((CalRFFEState_e)RX_MODE,(AntennaBitmaps_e)antMask);//Rffe mode
			PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_RX_DC_OFFSET, ANTENNA_0);
			AFE_SetLoopBack(ANTENNA_0, ANTENNA_0, AFE_MODE_RX, 0);
		}
		else if (newCalMode == RXDC_CAL_MODE_CHIP)
		{
			AFE_RxDCinit();
			AFE_SetLoopBack(ANTENNA_0, ANTENNA_0, AFE_MODE_RX, 0);
			RficDriver_RxDcInitAntAll();
			RficDriver_RxDcInitBW(HDK_getChannelWidth());
		}
	}
	RficDriver_SetDCcalType(newCalMode);
	rxdcCalMode = newCalMode;
}

void RxDcOffset_LoadResults(void)
{
	uint8 antMask = Hdk_GetRxAntMask();
	Antenna_e ant;
	int8 pgc1, pgc1Ind, pgc2Ind;
	HdkLnaIndex_e lnaIndex;
	for(ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ant++)
	{
		for(lnaIndex = HDK_LNA_INDEX_MIN_INDEX; lnaIndex < HDK_LNA_INDEX_MAX_INDEX; lnaIndex++)
		{
			for(pgc1Ind = 0; pgc1Ind < RXDC_NUM_PGC1_GAINS; pgc1Ind++)
			{
				pgc1 = (pgc1Ind*PGC1_INDEX_TO_GAIN_FACTOR)-PGC1_INDEX_TO_GAIN_BIAS;
				PhyCalDrv_DCTable1Set(ant, HDK_getChannelWidth(), pgc1, lnaIndex, ShramClbrDataBufferStoreChanData.rxDcData.dcTbl1Entry[ant][lnaIndex][pgc1Ind],ShramClbrDataBufferStoreChanData.rxDcData.dcTbl1CorrectionEntry[ant][lnaIndex][pgc1Ind]);
			}
		}
		for(pgc2Ind = 0; pgc2Ind < RXDC_NUM_PGC2_GAINS; pgc2Ind++)
		{
			PhyCalDrv_DCTable2Set(ant, HDK_getChannelWidth(), pgc2Ind, ShramClbrDataBufferStoreChanData.rxDcData.dcTbl2Entry[ant][pgc2Ind]);
		}
		if (((1 << ant) & antMask) != FALSE)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, ShramClbrDataBufferStoreChanData.rxDcData.calStatus[ant][HDK_PATH_TYPE_I]);
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, ShramClbrDataBufferStoreChanData.rxDcData.calStatus[ant][HDK_PATH_TYPE_Q]);
		}
		PhyCalDriver_LoadDcDigTable(ant,HDK_getChannelWidth(),ShramClbrDataBufferStoreChanData.rxDcData.DcDigCorrectionTable[ant]);
	}
	setCalibraitionToRF();//for chip mode
} 

//************************************
// Method:     RxDcOffset_StoreResults
// Purpose:    Stores DC calibration data to a buffer
// Parameter: None
// Returns:     None
//************************************
static void RxDcOffset_StoreResults(void)
{
	Antenna_e ant;
	bool CalState = TRUE;
	CalStatus_e antStatusI;
	CalStatus_e antStatusQ;
	int8 pgc1, pgc1Ind, pgc2Ind;
	HdkLnaIndex_e lnaIndex;
	for(ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ant++)
	{
		antStatusI = RxDcOffset_GetStatus(ant, RFIC_PATH_TYPE_I);
		antStatusQ = RxDcOffset_GetStatus(ant, RFIC_PATH_TYPE_Q);
		ShramClbrDataBufferStoreChanData.rxDcData.calStatus[ant][HDK_PATH_TYPE_I] = antStatusI;
		ShramClbrDataBufferStoreChanData.rxDcData.calStatus[ant][HDK_PATH_TYPE_Q] = antStatusQ;
		if ((antStatusI == CAL_STATUS_FAIL) || (antStatusI == CAL_STATUS_IN_PROCESS) || (antStatusQ == CAL_STATUS_FAIL) || (antStatusQ == CAL_STATUS_IN_PROCESS))
		{
			CalState = FALSE;
		}
		for(lnaIndex = HDK_LNA_INDEX_MIN_INDEX; lnaIndex < HDK_LNA_INDEX_MAX_INDEX; lnaIndex++)
		{
			for(pgc1Ind = 0; pgc1Ind < RXDC_NUM_PGC1_GAINS; pgc1Ind++)
			{
				pgc1 = (pgc1Ind*PGC1_INDEX_TO_GAIN_FACTOR)-PGC1_INDEX_TO_GAIN_BIAS;
				PhyCalDrv_DCTable1Get(ant, HDK_getChannelWidth(), pgc1, lnaIndex, ShramClbrDataBufferStoreChanData.rxDcData.dcTbl1Entry[ant][lnaIndex][pgc1Ind],ShramClbrDataBufferStoreChanData.rxDcData.dcTbl1CorrectionEntry[ant][lnaIndex][pgc1Ind]);
			}
		}
		for(pgc2Ind = 0; pgc2Ind < RXDC_NUM_PGC2_GAINS; pgc2Ind++)
		{
			PhyCalDrv_DCTable2Get(ant, HDK_getChannelWidth(), pgc2Ind, ShramClbrDataBufferStoreChanData.rxDcData.dcTbl2Entry[ant][pgc2Ind]);
		}
		PhyCalDriver_StoreDcDigTable(ant,HDK_getChannelWidth(),ShramClbrDataBufferStoreChanData.rxDcData.DcDigCorrectionTable[ant]);
	}
	getCalibraitionFromRF();//for chip mode
    ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_RX_DC_OFFSET, CalState);
} 

void RxDcOffset_Reset(void)
{
	Calibration_params_otf.RxDCparams.params.calState = FIRST_STATE;
}

RetVal_e RxDcOffset_Calibrate( IN ClbrType_e calibType )
{
	return run(calibType, &Calibration_params_otf.RxDCparams);
}

void RxDcOffset_SetStatus( IN uint32 antenna, IN RficPathTypeIndex_e path, IN CalStatus_e status )
{
	Calibration_params_otf.RxDCparams.results.calStatus[antenna][path] = status;
}

CalStatus_e RxDcOffset_GetStatus( IN uint32 antenna, IN RficPathTypeIndex_e path)
{
	return Calibration_params_otf.RxDCparams.results.calStatus[antenna][path];
}

uint32 RxDcOffset_GetLastRunTime(void)
{
	return Calibration_params_otf.RxDCparams.results.lastRunTime;
}

uint8 RxDC_getSelRxDC(uint32 bw)
{
	if(bw == BANDWIDTH_TWENTY)
		return 2;
	else if(bw == BANDWIDTH_FOURTY)
		return 1;
	else if(bw == BANDWIDTH_EIGHTY)
		return 0;
	return bw;
}
uint32 RxDC_getAddressPerBW(void)
{
	uint32 bw = 0;

	if(HDK_IsFrequencyJumpEnabled() == TRUE)
	{
		bw = HDK_getChannelWidth();
		return (GAIN0_CH_MODE0 + RxDC_getSelRxDC(bw));
	}
	else
	{
		return GAIN0_CH_MODE0;
	}
}

 /********************************************************************************

Method:	uint32* RxDc_GetElement(void)

Description:
------------
    
    Get m_RxDcElements address for BroadBand statistic  
  
Input:
-----
    
			
Output:
-------
	return - pointer to m_RxDcElements 

********************************************************************************/
 uint32* RxDc_GetElement(void)
 {
	 return (uint32*)&Calibration_params_otf.RxDCparams;
 }
 
 void setCalibraitionToRF(void)
 {
	 int reg_address, gain;
	 reg_address = RxDC_getAddressPerBW();
 
	 for (gain=0;gain<NUM_OF_GAINS_FOR_DCTBlCHIPMODE;gain++)
	 {
		 RficDriver_WriteRegister(reg_address, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.gain[gain]);
		 reg_address+=3;
	 }
	 RficDriver_WriteRegister(pgc2rxdc0_radar, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.pgc2rxdc0_radar);
	 RficDriver_WriteRegister(pgc2rxdc1, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.pgc2rxdc1);
 }
 
 void getCalibraitionFromRF(void)
 {
	 int reg_address, gain;
	 reg_address = RxDC_getAddressPerBW();
	 for (gain=0;gain<NUM_OF_GAINS_FOR_DCTBlCHIPMODE;gain++)
	 {
		 RficDriver_ReadRegister(reg_address, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.gain[gain]);
		 reg_address+=3;
	 }
	 RficDriver_ReadRegister(pgc2rxdc0_radar, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.pgc2rxdc0_radar);
	 RficDriver_ReadRegister(pgc2rxdc1, ShramClbrDataBufferStoreChanData.rxDcData.dcTblChipMode.pgc2rxdc1);
 }
