
/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "System_GlobalDefinitions.h"
#include "OSAL_Api.h"
#include <protocol.h>
#include <enet_pas.h>
#include <System_Configuration.h>
#include <PhyDriver_API.h>

#include <lmi.h>
#include <lm.h>
#include "RegAccess_Api.h"
#include "MT_Math.h"
#include <CalibrationHandler.h>
#include "CalibrationHandlerUtils.h"
#include "CalibrationHandlerStatistics.h"
#include "RxIQMismatchClbrHndlr.h"
#include "mt_sysrst.h"
#include "RficDriver_API.h"
#include "Afe_API.h"
#include "PhyCalDriver_API.h"
#include "Shram_ClbrDataBuffer.h"
#include "mhi_dut.h"
#include "lib_rf_4x4_api.h"
#include "CoC_Api.h"
#include "PhyTxRegsRegs.h"
#include "PhyRxTdRegs.h"
#include "PhyRegsIncluder.h"

#include "PhyRxTdIfRiscPage0TdRegs.h"

#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 17

/******************************************************************************/
/***						Constant Definitions										  ***/
/******************************************************************************/


/******************************************************************************/
/***						Type Definitions											  ***/
/******************************************************************************/	
extern Calibration_params_otf_t Calibration_params_otf;


/******************************************************************************/
/***						Macros													  ***/
/******************************************************************************/
#define ALLOWED_W1_ERROR		((outElements_p->params.errorPercentageW)*(W1_ERROR_VAL))
#define ALLOWED_W2_ERROR		((outElements_p->params.errorPercentageW)*(W2_ERROR_VAL))
#define FIRST_LOOP_BACK   0
#define SECOND_LOOP_BACK  1


									
/******************************************************************************/
/***					Private Function Declarations									  ***/
/******************************************************************************/   

 static RetVal_e run(RxIQRF_elements_t* outElements_p);
 static void initCalibration(uint8 inRxAntMask, RxIQRF_elements_t* inElements_p);
 static void finishCalibration(uint8 rxAntMask);
 static void  calcWs(int32 inII,int32 inQQ,int32 inIQ,int16 *outW1_p,int16 *outW2_p);
 static void findW1(int32 inII,int32 inIQ,int32 inQQ,int32 inW2, int32 *outW1_p);
 static RetVal_e verifyCalibrationResults(int8 inRxAntMask, RxIQRF_elements_t* outElements_p);
 static void RxIQ_StoreResults(void);
 static void  RxIQ_ResetOfflineStatus(void); 
 static void RxIQ_StoreValues(RxIQRF_elements_t* outElements_p, int16 W1_bckp[MAX_NUM_OF_ANTENNAS],int16 W2_bckp[MAX_NUM_OF_ANTENNAS]);


/********************************************************************************
Method:	run
Description: - Execute RX IQ calibration using air noise (Open rx path in RFIC)
---------
Input:
-----
Output: - RxIQRF_elements_t* outElements_p 
------
Returns: - RetVal_e - 		RET_VAL_SUCCESS - verify success
-------		                   RET_VAL_FAIL - verify fail
*******************************************************************************/
//uint32 enableFdlCalibration =1;
uint32 numberOfSamplesToCalibrate;
uint32 numOfCalibrations = 0;
uint32 numberOfSamplesToVerify;
ClbrIteration_e iterationType;
CorrRes_t resultsForOnlineCalibration[CLBR_ITERATION_MAX_ONLINE][MAX_NUM_OF_ANTENNAS];
static RetVal_e run(RxIQRF_elements_t* outElements_p)
{
    CorrRes_t Results[MAX_NUM_OF_ANTENNAS];
	int16 W1,W2,W1_bckp[MAX_NUM_OF_ANTENNAS],W2_bckp[MAX_NUM_OF_ANTENNAS];
	RetVal_e ret = RET_VAL_SUCCESS;
    uint8 antNum;  
    int32 startTime= TIME_STAMP(START_TIME,0);
    CalDataState_e calStoreState;
	int8 Delay_us;
	uint8 bw;
	uint8 rxAntMask, numOfAnts;
	
	bw = HDK_getChannelWidth();
	
	/*in online we save the last calibration results, for the case of failure*/ 																																			
	RxIQ_StoreValues(outElements_p,W1_bckp,W2_bckp);
	
#if LM_CALIBRATION_LOGS_ON
	ILOG2_V("Rx IQ: Executing Calibration");
#endif

	RxIQ_ResetOfflineStatus();

    // Get the enabled RX antennas
    
	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &rxAntMask);

	initCalibration(rxAntMask, outElements_p);
    
	Delay_us = DELAY_AFTER_PHY_RFIC_CHANGE - (DELAY_CHANGE_PER_BW * bw);
	
	outElements_p->params.iir_mu = RX_IQ_IIR_MU-bw;


	RxIQ_adaptiveGain(Results, outElements_p, rxAntMask, Delay_us);

	PhyCalDrv_CorrInit(CORR_MODE,rxAntMask,numberOfSamplesToCalibrate,RX_IQ_ROUND,outElements_p->params.iir_mu);

	PhyCalDrv_setAccumTo80Mh();//only to wave500
	
	PhyCalDrv_CorrMeasureAll(Results,Delay_us,PHY_CAL_DRV_IGNORE_IQ_SWAP);

	// Run algorithm: 

	// Calculate W's coefficients for each antenna and update phy registers 
	
	for (antNum = 0; antNum < MAX_NUM_OF_ANTENNAS; antNum++)
	{
		if (RX_ANTENNA_ENABLED(rxAntMask,antNum)) /* masking so we will run the calibration only on active antennas */
		{
			
			resultsForOnlineCalibration[iterationType][antNum].II = Results[antNum].II;
			resultsForOnlineCalibration[iterationType][antNum].QQ = Results[antNum].QQ;
			resultsForOnlineCalibration[iterationType][antNum].IQ = Results[antNum].IQ;
			if(iterationType == CLBR_ITERATION_SECOND_ONLINE)
			{
				Results[antNum].II += resultsForOnlineCalibration[CLBR_ITERATION_FIRST_ONLINE][antNum].II;
				Results[antNum].QQ += resultsForOnlineCalibration[CLBR_ITERATION_FIRST_ONLINE][antNum].QQ;
				Results[antNum].IQ += resultsForOnlineCalibration[CLBR_ITERATION_FIRST_ONLINE][antNum].IQ;
			}
			calcWs(Results[antNum].II,Results[antNum].QQ,Results[antNum].IQ,&W1,&W2);

			PhyCalDrv_RxIQSetWCoeffs(bw,antNum,W1,W2,1);

			outElements_p->results.w1[antNum] = W1;
			outElements_p->results.w2[antNum] = W2;
		}
	}
	// Check that the coefficients are correct

	ret = verifyCalibrationResults(rxAntMask, outElements_p);
	if((ret == RET_VAL_FAIL) || (iterationType == CLBR_ITERATION_FIRST_ONLINE))
	{			
		
		for (antNum = 0; antNum < MAX_NUM_OF_ANTENNAS; antNum++)
		{
			if (RX_ANTENNA_ENABLED(rxAntMask,antNum)) /* masking so we will run the calibration only on active antennas */
			{
				if((outElements_p->results.status[antNum] == CAL_STATUS_FAIL) || (iterationType == CLBR_ITERATION_FIRST_ONLINE))
				{
					PhyCalDrv_RxIQSetWCoeffs(bw,antNum,W1_bckp[antNum],W2_bckp[antNum],2);
				}
			}
		}
		
	}
	
	finishCalibration(rxAntMask);

    outElements_p->results.lastRunTime = TIME_STAMP(END_TIME,startTime);

#if LM_CALIBRATION_LOGS_ON
	ILOG9_DD("RxIQ ALGO:Finished Calibration. Ret Code = %d, Run Time = %d", ret, outElements_p->results.lastRunTime);
#endif

	/* Store calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);

	if((calStoreState == CAL_DATA_STATE_STORE) && (CLBR_TYPE_OFFLINE == ClbrHndlr_GetClbrType()))
	{
		RxIQ_StoreResults();
	}
    
	return(ret);
    
}

//************************************
// Method:    finishCalibration
// FullName:  finishCalibration
// Access:    public static 
// Returns:   void
// Qualifier:
//************************************
 static void finishCalibration(uint8 rxAntMask)
{
#if LM_CALIBRATION_LOGS_ON
	int32 restoreStartTime= TIME_STAMP(START_TIME,0);
#endif

	// Restore operation for all Antennas
	RficCalDrv_RestoreOperational(rxAntMask);//1) Rfic rx off.//2) disable LnaNoise
	PhyCalDrv_RestoreOperationalRx();
	AFE_RestoreFromLoopBack();
	AFE_Restore_Dither_Tone_Frequency();
	RficDriver_LoadRegDB();

	/* Restore spurs - restore original frequencies after calibration, in case of spurs (platform dependent) */
	RficDriver_RxIqFreqSpursCancellingReset();

#if LM_CALIBRATION_LOGS_ON
	ILOG9_D("Rx IQ: Restore Done. Duration = %d", TIME_STAMP(END_TIME, restoreStartTime));
#endif
}

/********************************************************************************

Method: initCalibration 

Description:
------------

  
Input:
-----
	inRxAntMask
	inElements_p
			
Output:
-------
			

********************************************************************************/
static void initCalibration(uint8 inRxAntMask, RxIQRF_elements_t* inElements_p)
{
#if LM_CALIBRATION_LOGS_ON
	int32 initStartTime= TIME_STAMP(START_TIME,0);
#endif
	uint32 pgc2Value;
	pgc2Value = inElements_p->params.pgc2;


	if(HDK_getChannelWidth() == BANDWIDTH_TWENTY)
	{
#if  defined(RFIC_WRX500)
		pgc2Value = 18;
#endif
	}

    //1.Initialize RF: Set RF to receive and set gain values 
    RficDriver_ActivateRxAntennas ((AntennaBitmaps_e)inRxAntMask); // Activate all receive enabled antennas.
 
	SetRxGains((AntennaBitmaps_e)inRxAntMask ,inElements_p->params.lnaIndex,inElements_p->params.pgc1,pgc2Value,inElements_p->params.pgc3);

	//Rffe mode T/R switch + LNA off
	
	SetRffeMode((uint8)inRxAntMask);

	//Only for wave500,	
	RficDriver_EnableLnaNoise(inRxAntMask);

    PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_RX_IQ_MISMATCH,0);
	
	AFE_SetLoopBack((Antenna_e)0, (Antenna_e)1, AFE_MODE_RX, 0);

	AFE_Set_Dither_Tone_Frequency(DOUBLED_FREQUENCY);

#if LM_CALIBRATION_LOGS_ON
	ILOG9_D("Rx IQ: Init Done. Duration = %d", TIME_STAMP(END_TIME, initStartTime));
#endif

	return;
}

/********************************************************************************

Method: calcWs

Description:
------------
Calculate W1 and W2
  
Input:
-----
	inII
	inQQ
	inIQ
			
Output:
-------
	outW1
	outW2

********************************************************************************/
static void calcWs(int32 inII,int32 inQQ,int32 inIQ,int16 *outW1,int16 *outW2)
{ 	
	int32 W2,W1;

	MATH_LongDivision(inIQ,inQQ,W2_NUM_RES_BITS,&W2);

	findW1( inII, inIQ, inQQ, W2, &W1);
	*outW1 = (int16)W1;
	*outW2 = (int16)W2;
	return;
}

/********************************************************************************
Method: findW1

Input:
	1. inII		: the auto-correlation of I (in-phase)				
	2. inIQ		: the cross-correlation between I,Q 	
	3. inQQ		: the auto-correlation of Q (quadrature)				
	4. inW2		: the coefficient that makes I,Q orthogonal			

Output:
	1. outW1_p - the normalized amplitude factor between I,Q 

Description:
	implementation of an equation, by binary search of the parameter (Result), who is square root
	of several other parameters (all according to the equation)

*******************************************************************************/
 static void findW1(int32 inII,int32 inIQ,int32 inQQ,int32 inW2, int32 *outW1_p)
{
	int16 i,Delta;
	int64 a,b,denominator,Result;
	int32 numerator,prev_step,curr_step,next_step;
	int64 W2_64b,expected_IQ_64b;
	int32 num_of_frac_bits,num_res_bits;

	num_of_frac_bits = W1_NUM_FRAC_BITS;
	num_res_bits = W2_NUM_RES_BITS;

	numerator = inQQ;
	W2_64b = inW2;				   	// extension from 32bit -> 64 bit.
	expected_IQ_64b = inIQ; 		// extension from 32bit -> 64 bit.

	denominator = inII - ((W2_64b*expected_IQ_64b+(1<<(num_res_bits-1)))>>num_res_bits); // must add the normalized factor !!!
										//,and before the shifting,need to add half from the division resolution(round)
	Result = 1<<(num_of_frac_bits);		// requested value is between 0-2,and in fixed point (12bit) its between 0-8192,so we will start in the middle 2^12(4096)
	Delta   = 1<<(num_of_frac_bits-1);	//2048 is the first step that we need if we start from 4096 and our range is 8192 
	i = 0;	

	b=numerator;
	b<<=(2*num_of_frac_bits); 			// this is the right side of the equation
							  			//must split this action in two ,because "numerator" is just 32 bit and "b" is 64.

	while (i < num_of_frac_bits)
	{   
		a=Result*Result*denominator;
				
		if (a<b) 
		{
			Result += Delta; 			//'b' bigger than 'a', need to add a value to 'a'
		} 
		else
		{
			Result -= Delta;
		}
		Delta >>= 1;
		++i;  
	}
	
	// check if the best value is one step near	
	prev_step = Abs((int32)((Result-1)*(Result-1)*denominator-b));
	curr_step = Abs((int32)(Result*Result*denominator-b));
	next_step = Abs((int32)((Result+1)*(Result+1)*denominator-b));

	if (curr_step > next_step)
	{
		Result+=1;
	} 
	else if (curr_step > prev_step)
	{
		Result-=1;
	}
	
	*outW1_p =  (int32)Result;

	return;
}

/*************************************************************************************

Method: verifyCalibrationResults

Input:
	inRxAntMask

Output: 
	outElements_p

Return:
	RetVal_e - 
		success(RET_VAL_SUCCESS) - if the the W1,W2 coeff are good (the deflection si less than errorPercentageW)
		fail(RET_VAL_FAIL) - if coefficients are not good

Description:	this function compares the W1,W2 after cal (supposed to be W1 =1 ,W2 = 0)
				in deflection of errorPercentageW and returns status respectively .

*************************************************************************************/
static RetVal_e verifyCalibrationResults(int8 inRxAntMask, RxIQRF_elements_t* outElements_p)
{
	
	uint8 antNum;  
	RetVal_e ret = RET_VAL_SUCCESS;
	CorrRes_t Results[MAX_NUM_OF_ANTENNAS];
	int16 W1,W2;

	// Check that the coefficients are correct
	PhyCalDrv_CorrInit(CORR_MODE,inRxAntMask,numberOfSamplesToVerify,RX_IQ_ROUND,outElements_p->params.iir_mu);
	PhyCalDrv_setAccumTo80Mh();
	PhyCalDrv_CorrMeasureAll(Results,MINIMUM_DELAY,PHY_CAL_DRV_IGNORE_IQ_SWAP);
	
	for (antNum = 0; antNum < MAX_NUM_OF_ANTENNAS; antNum++)
	{
		if (RX_ANTENNA_ENABLED(inRxAntMask,antNum)) /* run the verification only on active antennas */
		{
			calcWs(Results[antNum].II,Results[antNum].QQ,Results[antNum].IQ,&W1,&W2);

#if LM_CALIBRATION_LOGS_ON
			ILOG9_DDD("RxIQ error verification for ANTENNA %d  w1 = %d w2 = %d",antNum,W1,W2);
			ILOG9_DDD("RxIQ error verification II = %d QQ = %d IQ = %d",Results[antNum].II,Results[antNum].QQ,Results[antNum].IQ);
#endif          

			outElements_p->results.w1_err[antNum] = (W1 - INITIAL_W1_VALUE);
			outElements_p->results.w2_err[antNum] = (W2 - INITIAL_W2_VALUE);

			if ( (Abs(outElements_p->results.w1_err[antNum]) > ALLOWED_W1_ERROR) || (Abs(outElements_p->results.w2_err[antNum]) > ALLOWED_W2_ERROR) )
			{
				outElements_p->results.status[antNum] = CAL_STATUS_FAIL; 
				ret = RET_VAL_FAIL;
			}	
			else
			{
				outElements_p->results.status[antNum] = CAL_STATUS_PASS;
			}

#if LM_CALIBRATION_LOGS_ON
			ILOG9_DDD("RxIQ verifyCalibrationResults: error verification Status = %d w1_Error= %d w2_Error= %d",outElements_p->results.status[antNum],outElements_p->results.w1_err[antNum],outElements_p->results.w2_err[antNum]);
#endif			
		}
	}

	return (ret);
}

/********************************************************************************

Method:	RxIQ_Calibrate

Description:
------------
    
    Calibrate RxIQ mismatch - main calibration function
  
Input:
-----
    ClbrType_e inCalibType - OFFLINE or ONLINE (not used)	
			
Output:
-------
	return status - Success(RET_VAL_SUCCESS) Fail(RET_VAL_FAIL)

********************************************************************************/
RetVal_e RxIQ_Calibrate(IN ClbrType_e inCalibType)
{
	RetVal_e tempReturnVal;

	if (CoC_getNumActiveTxAnts() > 1)
	{
		if(inCalibType == CLBR_TYPE_ONLINE)
		{
			if(iterationType == CLBR_ITERATION_OFFLINE)
			{
				iterationType = CLBR_ITERATION_FIRST_ONLINE;
				numberOfSamplesToCalibrate = RX_IQ_NUM_SAMPLES_ONLINE_FIRST_STAGE;
				numberOfSamplesToVerify = RX_IQ_NUM_SAMPLES_ONLINE_VERIFY_STAGE;
				run(&(Calibration_params_otf.RxIQparams));
				return RET_VAL_FRAGMENTED;
			}
			if(iterationType == CLBR_ITERATION_FIRST_ONLINE)
			{
				numberOfSamplesToCalibrate = RX_IQ_NUM_SAMPLES_ONLINE_SECOND_STAGE;
				numberOfSamplesToVerify = RX_IQ_NUM_SAMPLES_ONLINE_VERIFY_STAGE;
				iterationType = CLBR_ITERATION_SECOND_ONLINE;
				
				tempReturnVal = run(&(Calibration_params_otf.RxIQparams));
				iterationType = CLBR_ITERATION_OFFLINE;
				return tempReturnVal;
			}
			
		}
		else
		{
			iterationType = CLBR_ITERATION_OFFLINE;
			numberOfSamplesToCalibrate = RX_IQ_NUM_SAMPLES_OFFLINE;
			numberOfSamplesToVerify = RX_IQ_NUM_SAMPLES_OFFLINE;
			
		}
		return run(&(Calibration_params_otf.RxIQparams));
	}
	else
	{
		return RET_VAL_SUCCESS;
	}
}

/********************************************************************************

Method:	RxIQ_SetStatus

Description:
------------
    
    Set calibration status of given antenna
  
Input:
-----
			
Output:
-------

********************************************************************************/
 void RxIQ_SetStatus(IN uint32 inAntenna, IN CalStatus_e inStatus)
{
	Calibration_params_otf.RxIQparams.results.status[inAntenna] = inStatus;
}

/********************************************************************************

Method:	RxIQ_GetStatus

Description:
------------
    
    Get calibration status of given antenna
  
Input:
-----
			
Output:
-------
	status - CAL_STATUS_FAIL or CAL_STATUS_PASS
	

********************************************************************************/
 CalStatus_e RxIQ_GetStatus( IN uint32 inAntenna )
{
	return Calibration_params_otf.RxIQparams.results.status[inAntenna];
}

/********************************************************************************

Method:	RxIQ_GetLastRunTime

Description:
------------
    
    Get last RXIQ mismatch calibration run time duration
  
Input:
-----
    
			
Output:
-------
	return - run time duration	

********************************************************************************/
 uint32 RxIQ_GetLastRunTime(void)
{
	return Calibration_params_otf.RxIQparams.results.lastRunTime;
}

/********************************************************************************

Method:	RxIQ_LoadResults

Description:
------------
    
    Loads RXIQ calibration data from a buffer
  
Input:
-----
    param None	
			
Output:
-------
	return None	

********************************************************************************/
void RxIQ_LoadResults(void)
{

int8 iAntenna,BandNum;

	for(iAntenna = ANTENNA_0; iAntenna < MAX_NUM_OF_ANTENNAS; iAntenna++)
	{
		BandNum = HDK_getChannelWidth();
		PhyCalDrv_RxIQSetWCoeffs(BandNum, iAntenna, ShramClbrDataBufferStoreChanData.rxIqData.w1[iAntenna], ShramClbrDataBufferStoreChanData.rxIqData.w2[iAntenna],3);

		/* Update internal variables with loaded values */
		Calibration_params_otf.RxIQparams.results.w1[iAntenna] = ShramClbrDataBufferStoreChanData.rxIqData.w1[iAntenna];
		Calibration_params_otf.RxIQparams.results.w2[iAntenna] = ShramClbrDataBufferStoreChanData.rxIqData.w2[iAntenna];    
		RxIQ_SetStatus(iAntenna, ShramClbrDataBufferStoreChanData.rxIqData.calStatus[iAntenna]);
	}
}

void RxIQ_NLMS_LoadResults(void)
{

}
void RxIQ_Xtalk_LoadResults(void)
{

}

/********************************************************************************
Method:	        RxIQ_StoreResults
Description:     Stores RXIQ calibration data to a buffer
Input:              param None		
Output:            return None
********************************************************************************/
static void RxIQ_StoreResults(void)
{
    Antenna_e ant;
	uint8 BandNum;
    bool CalState = TRUE;
    CalStatus_e antStatus;

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
			BandNum = HDK_getChannelWidth();
        	antStatus = RxIQ_GetStatus(ant);
        	ShramClbrDataBufferStoreChanData.rxIqData.calStatus[ant] = antStatus;

        	PhyCalDrv_RxIQGetWCoeffs(BandNum, ant, &(ShramClbrDataBufferStoreChanData.rxIqData.w1[ant]), &(ShramClbrDataBufferStoreChanData.rxIqData.w2[ant]));
        
        	if ((antStatus == CAL_STATUS_FAIL) || (antStatus == CAL_STATUS_IN_PROCESS))
        	{
            	CalState = FALSE;
        	}
		
    	ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_RX_IQ_MISMATCH, CalState);
	}
}

 //************************************
  // Method:		 void RxIQ_StoreValues
  // Description:	 Store W1 & W2 with last success calibratio or to default values
  // Returns:		  void
  // Qualifier:
  //************************************ 

 static void RxIQ_StoreValues(RxIQRF_elements_t* outElements_p, int16 W1_bckp[MAX_NUM_OF_ANTENNAS], int16 W2_bckp[MAX_NUM_OF_ANTENNAS])
 { 
	 int8 ant;
	 
	 for(ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)		 
	 {
		 	if(CLBR_TYPE_ONLINE == ClbrHndlr_GetClbrType())
		 	{
			 	W1_bckp[ant] = outElements_p->results.w1[ant];
			 	W2_bckp[ant] = outElements_p->results.w2[ant]; 		   
		 	}
		 	else
		 	{
			 	W1_bckp[ant] =  INITIAL_W1_VALUE;
			 	W2_bckp[ant] =  INITIAL_W2_VALUE;
		 	}
	 }
 }

 //************************************
 // Method:    RxIQ_ResetOfflineStatus
 // Description:    Reset calibration status - start from scratch
 // Returns:   void
 // Qualifier:
 //************************************

 static void  RxIQ_ResetOfflineStatus(void)
 {
	uint8 i=0;

	for(i=0;i< MAX_NUM_OF_ANTENNAS ;i++)
	{
		RxIQ_SetStatus(i, CAL_STATUS_INIT);
	}
}

 uint32* RxIQ_GetElement(void)
 {
	 return (uint32*)&Calibration_params_otf.RxIQparams;
 }
 

