/******************************************************************************/
/***					      Include Files									***/
/******************************************************************************/
#include "System_MainApi.h"
#include "TpcClbrHndlr.h"
#include "CalibrationsDefines.h"
#include "HwMemoryMap.h"
#include "RegAccess_Api.h"
#include "PhyTxRegsRegs.h"
#include "PhyRxTdRegs.h"
#include "MT_Math.h"
#include "RficDriver_API.h"
#include "loggerAPI.h"
#include "PSD.h"
#include "stringLibApi.h"
#include "ShramTpc.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 18

/******************************************************************************/
/***					      Definitions													***/
/******************************************************************************/
#define TPC_FREQ_LEN_1S2D_NO_CAL	7
#define TPC_FREQ_LEN_2S2D_NO_CAL	11
#define TPC_FREQ_LEN_3S2D_NO_CAL	15
#define TPC_FREQ_LEN_1S2D			8
#define TPC_FREQ_LEN_2S2D			14
#define TPC_FREQ_LEN_3S2D			19

#define MAX_NUM_OF_TPC_FREQ (3)
/******************************************************************************/
/***					      Static Variables												***/
/******************************************************************************/

// Overlay with initialization section
uint8 (*gPowerOut)[MAX_NUM_OF_BW][TPC_POWER_OUT_MAX_ROWS];

extern uint16 digGainWords[TPC_NUM_DIGITAL_WORDS];
extern uint8 preDrvGainWords[TPC_NUM_PRE_DRV_WORDS];
extern uint8 pgcGainWords[TPC_NUM_PGC_WORDS];

#define PGC_WORD_0_EXT_PA	2
#define PGC_WORD_1_EXT_PA	3
#define EXT_PA_NUM_DIFF_REGS	15



/******************************************************************************/
/***					      Public Functions											***/
/******************************************************************************/
#if !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif

void TPC_fillDefines(void)
{
	uint8 index1, index2;
    memset(&tpcData, 0, sizeof(tpc_params_t));
    memset(&tpcConfiguration, 0, sizeof(tpcConfig_t));
    memset(tpcAntParamsArray, 0, sizeof(tpcFreqList_t)*MAX_NUM_OF_ANTENNAS);
	
	tpcData.digGainMax = DIG_MAX_GAIN_DEFAULT;
	tpcData.digGainMin = DIG_LOWER_LIMIT_DEFAULT;
	tpcData.digGainStep = DIG_STEP_DEFAULT;
	tpcData.digGainUpperLimit = DIG_UPPER_LIMIT_DEFAULT;
	tpcData.digitalMinGain = DIG_MIN_GAIN_DEFAULT;
	tpcData.digWordOffset = DIG_WORD_OFFSET_DEFAULT;
	tpcData.digWordShift = DIG_WORD_SHIFT_DEFAULT;
	tpcData.numberOfTssiCutPts = NUM_TSSI_CUT_POINTS_DEFAULT;
	tpcData.pgc2lowerLimit = PGC_LOWER_LIMIT_DEFAULT;
	tpcData.pgc2step = PGC_STEP_DEFAULT;
	tpcData.pgc2upperLimit = PGC_UPPER_LIMIT_DEFAULT;
	tpcData.pgcWordOffset = PGC_WORD_OFFSET_DEFAULT;
	tpcData.pgcWordShift = PGC_WORD_SHIFT_DEFAULT;
	tpcData.powerVectorLen = POWER_VECTOR_LEN_DEFAULT;
	tpcData.powerVectorStep = POWER_VEC_STEP_DEFAULT;
	tpcData.preDrvMinGain = PRE_DRV_MIN_GAIN_DEFAULT;
	tpcData.preDrvStep = PRE_DRV_STEP_DEFAULT;
	tpcData.preDrvUpperLimit = PRE_DRV_UPPER_LIMIT_DEFAULT;
	tpcData.preDrvWordOffset = PRE_DRV_WORD_OFFSET_DEFAULT;
	tpcData.preDrvWordShift = PRE_DRV_WORD_SHIFT_DEFAULT;
	tpcData.tssiCharacteriztionType = TPC_CHARACTERIZATION_DEFAULT;
	tpcData.gainStepsInPowerStep = tpcData.powerVectorStep / tpcData.digGainStep;
	tpcData.fixedGain = NO_FIXED_GAIN;
	tpcData.powerLimitOffset = 0;

	for (index1 = 0; index1 < MAX_NUM_OF_ANTENNAS; index1++)
	{
		for (index2 = 0; index2 < MAX_NUM_OF_BW; index2++)
		{
			tpcData.minPowerGain[index1][index2] = MIN_POWER_INDEX_DEFAULT;
			tpcData.minPower[index1][index2] = MIN_POWER_DEFAULT;
		}
	}
}
#if !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

void TPC_checkExtPA()
{
}

void TPC_init(uint8 tpcType)
{
	TPC_checkExtPA();
	//TPC_buildGainTable();
//	TPC_InitPout();
}

#if !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
void TPC_PostInit()
{
    
    
	System_MainAllocInitializationMemory((uint8 **)&gPowerOut, (MAX_NUM_OF_ANTENNAS*MAX_NUM_OF_BW*TPC_POWER_OUT_MAX_ROWS));
}
#pragma ghs section text=default
#endif

void TPC_loadChannelData(uint16 chID,uint8 bandwidth)
{
    uint8 ant;
   	uint32 antMask;
    
	antMask = PSD_getDataField(PSD_FIELD_TX_ANTS_MASK);

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (((1 << ant) & antMask) != 0)
		{
			TPC_extractTpcFreqParams(ant, chID, bandwidth);
		}
	}

	TPC_calcAntennaBoosts();

	if (tpcData.fixedGain == NO_FIXED_GAIN)
	{
		TPC_buildPout();
	}
}


void TPC_getPowerToIndexLUT(LA_SET_TPC_DATA_t* tpcInfo)
{
	tpcInfo->indexOffset= 0;
	tpcInfo->tpcPowerVectorLen = tpcData.powerVectorLen;
	tpcInfo->tpcpPwerVectorStep = tpcData.powerVectorStep;
}



void TPC_extractTpcFreqParams(uint8 ant, uint16 channel, uint8 bandwidth)
{
	TPC_FREQ				tpcFreqParams[MAX_NUM_OF_TPC_FREQ];
	uint8					tpcList[MAX_NUM_OF_BW][2] = {0};
	TPC_FREQ_1S2D_NO_CALIB*	tpcFreqPtr;
	uint8					numElements[MAX_NUM_OF_BW] = {0};
	uint8					index;
	uint8					bw;
	uint8					tpcBwBitmap;

	//1. sort elements by bandwidth
	for (index = 0; index < TPC_FREQ_STR_NUM; index++)
	{
		tpcFreqPtr = (TPC_FREQ_1S2D_NO_CALIB*)(&tpcAntParamsArray[ant].tpcFreq[index]);
		if (tpcFreqPtr->chID != TPC_INVALID_CHID)
		{
			bw = tpcFreqPtr->uevm_bw.uevm_bw_fields.BW;
            ASSERT(bw < MAX_NUM_OF_BW);
            ASSERT(numElements[bw ] < 2);
			tpcList[bw][numElements[bw]] = index;
			numElements[bw]++;
		}
	}
	//2. for each bandwidth:
	//	2.1. if only one element - take this element
	//	2.2. if 2 elements - interpolate
	tpcBwBitmap = 0;
	for (index = 0; index < MAX_NUM_OF_BW; index++)
	{
		if (numElements[index] == 1)
		{
			memcpy(&tpcFreqParams[index], &tpcAntParamsArray[ant].tpcFreq[tpcList[index][0]], sizeof(TPC_FREQ));
			tpcBwBitmap |= (1 << index);
		}
		else if (numElements[index] == 2)
		{
			tpcFreqParams[index].tpcFreq1s2dNoCal.chID = channel;
			TPC_interpolateFrequencies(&tpcFreqParams[index],&tpcAntParamsArray[ant].tpcFreq[tpcList[index][0]], &tpcAntParamsArray[ant].tpcFreq[tpcList[index][1]]);
			tpcBwBitmap |= (1 << index);
		}
	}

	//3. check if some bandwidths are missing and interpolate from other bandwidth
	switch (tpcBwBitmap)
	{
		case 1: //only 20MHz exist
			memcpy(&tpcFreqParams[1], &tpcFreqParams[0], sizeof(TPC_FREQ)); //40 = 20
			memcpy(&tpcFreqParams[2], &tpcFreqParams[0], sizeof(TPC_FREQ)); //80 = 20
			break;
		case 2:	//only 40MHz exist
			memcpy(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ));	//20 = 40
			memcpy(&tpcFreqParams[2], &tpcFreqParams[1], sizeof(TPC_FREQ)); //80 = 40
			break;
		case 3:	//only 20 & 40 MHz exist
			memcpy(&tpcFreqParams[2], &tpcFreqParams[1], sizeof(TPC_FREQ)); //80 = 40
			break;
		case 4: //only 80MHz exist
			memcpy(&tpcFreqParams[0], &tpcFreqParams[2], sizeof(TPC_FREQ)); //20 = 80
			memcpy(&tpcFreqParams[1], &tpcFreqParams[2], sizeof(TPC_FREQ)); //20 = 40
			break;
		case 5: //only 20 & 80 MHz exist -40 = (20 + 80)/2
			tpcFreqParams[1].tpcFreq1s2dNoCal.chID = tpcFreqParams[0].tpcFreq1s2dNoCal.chID;
			TPC_interpolateBandwidths(&tpcFreqParams[1], &tpcFreqParams[0], &tpcFreqParams[2]);
			break;
		case 6: //only 40 & 80 MHz exist
			memcpy(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ)); //20 = 40
			break;
		default: //must be that all bandwidths exist, otherwise - error (non exist)
			//ASSERT(tpcBwBitmap);
			break;			
	}

	//3.1 restore bandwidth after interpolation
	for (index = 0; index < MAX_NUM_OF_BW; index++)
	{
		tpcFreqParams[index].tpcFreq1s2dNoCal.uevm_bw.uevm_bw_fields.BW = index;
	}

	//4. build TPC list for current channel and all bandwidths
	TPC_buildTpcLimitsFromTpcFreq(ant, tpcFreqParams);

	TPC_loadTSSIparams(ant, tpcFreqParams);

}

void TPC_calcAntennaBoosts()
{
	uint8	bw;
	uint8	ant;
	uint8 	minPower;
	uint8 	boost;
	uint8 	minPowerAnt;
	uint32	txAntMask = Hdk_GetTxAntMask();

	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		//first - find antenna with minimal maximum EVM power
		minPower = tpcData.powerVectorLen;
		minPowerAnt = 0;
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & txAntMask) != 0)
			{
				if (tpcData.maxEvmLimit[ant][bw] < minPower)
				{
					minPower = tpcData.maxEvmLimit[ant][bw];
					minPowerAnt = ant;
				}
			}
		}

		//now boost of each antenna is maximum EVM of antenna minus minimal maximum EVM power
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & txAntMask) != 0)
			{
				tpcData.minBoostAntIndex[bw] = minPowerAnt;
				boost = tpcData.maxEvmLimit[ant][bw] - minPower;
				 //round down to 1dB resolution and limit by 3 dB
				tpcData.antPowerBoost[ant][bw] = (boost * tpcData.powerVectorStep) >> GAIN_RESOLUTION;
				if (tpcData.antPowerBoost[ant][bw] > TPC_MAX_BOOST)
				{
					tpcData.antPowerBoost[ant][bw] = TPC_MAX_BOOST;
				}
			}
		}
	}
}

void TPC_interpolateFrequencies(TPC_FREQ * outTpcFreq,TPC_FREQ * tpcFreq1,TPC_FREQ * tpcFreq2)
{
	switch (tpcData.tpcFreqLen)
	{
		case TPC_FREQ_LEN_1S2D_NO_CAL:
			TPC_FreqInterpolation_1s2d_no_cal((TPC_FREQ_1S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_1S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_1S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_1S2D:
			TPC_FreqInterpolation_1s2d((TPC_FREQ_1S2D*)outTpcFreq, (TPC_FREQ_1S2D*)tpcFreq1, (TPC_FREQ_1S2D*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_2S2D_NO_CAL:
			TPC_FreqInterpolation_2s2d_no_cal((TPC_FREQ_2S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_2S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_2S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_2S2D:
			TPC_FreqInterpolation_2s2d((TPC_FREQ_2S2D*)outTpcFreq, (TPC_FREQ_2S2D*)tpcFreq1, (TPC_FREQ_2S2D*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_3S2D_NO_CAL:
			TPC_FreqInterpolation_3s2d_no_cal((TPC_FREQ_3S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_3S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_3S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_3S2D:
			TPC_FreqInterpolation_3s2d((TPC_FREQ_3S2D*)outTpcFreq, (TPC_FREQ_3S2D*)tpcFreq1, (TPC_FREQ_3S2D*)tpcFreq2);
			break;
	}
}

void TPC_interpolateBandwidths(TPC_FREQ * outTpcFreq,TPC_FREQ * tpcFreq1,TPC_FREQ * tpcFreq2)
{
	outTpcFreq->tpcFreq1s2dNoCal.chID = tpcFreq1->tpcFreq1s2dNoCal.chID;
	
	switch (tpcData.tpcFreqLen)
	{
		case TPC_FREQ_LEN_1S2D_NO_CAL:
			TPC_bwInterpolation_1s2d_no_cal((TPC_FREQ_1S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_1S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_1S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_1S2D:
			TPC_bwInterpolation_1s2d((TPC_FREQ_1S2D*)outTpcFreq, (TPC_FREQ_1S2D*)tpcFreq1, (TPC_FREQ_1S2D*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_2S2D_NO_CAL:
			TPC_bwInterpolation_2s2d_no_cal((TPC_FREQ_2S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_2S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_2S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_2S2D:
			TPC_bwInterpolation_2s2d((TPC_FREQ_2S2D*)outTpcFreq, (TPC_FREQ_2S2D*)tpcFreq1, (TPC_FREQ_2S2D*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_3S2D_NO_CAL:
			TPC_bwInterpolation_3s2d_no_cal((TPC_FREQ_3S2D_NO_CALIB*)outTpcFreq, (TPC_FREQ_3S2D_NO_CALIB*)tpcFreq1, (TPC_FREQ_3S2D_NO_CALIB*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_3S2D:
			TPC_bwInterpolation_3s2d((TPC_FREQ_3S2D*)outTpcFreq, (TPC_FREQ_3S2D*)tpcFreq1, (TPC_FREQ_3S2D*)tpcFreq2);
			break;
	}
}

void TPC_buildTpcLimitsFromTpcFreq(uint8 ant, TPC_FREQ* tpcList)
{
	TPC_FREQ* tpcFreqPtr;
	uint8 bw;
	
	// copy parameters
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		tpcFreqPtr = (TPC_FREQ*)(&tpcList[bw]);
		tpcData.maxEvmLimit[ant][bw] = tpcFreqPtr->tpcFreq1s2dNoCal.maxp_ant.maxp_ant_fields.maxPower;
		tpcData.minPower[ant][bw] = tpcFreqPtr->tpcFreq1s2dNoCal.uevm_bw.uevm_bw_fields.ultimateEVM;
		tpcData.minPowerGain[ant][bw] = tpcFreqPtr->tpcFreq1s2dNoCal.ultimateEVMgain & (TPC_GAIN_TABLE_MAX_ROWS - 1);
	}

}

void TPC_buildPout()
{
	int16 gainWord;
	int16 refGainWord;
	uint8 ant,bw;
	int8  refIndex;
	int8  index;

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
		{
			//min power is reference point for buildingpowerout vector
			refIndex = (int8)tpcData.minPower[ant][bw];
			refGainWord = (int16)tpcData.minPowerGain[ant][bw];
			//1. go from refrence down
			index = refIndex;
			gainWord = refGainWord;
			while ((index >= 0) && (gainWord > 0))
			{
                ASSERT (index < TPC_POWER_OUT_MAX_ROWS)
				gPowerOut[ant][bw][index] = (uint8)gainWord;
				index--;
				gainWord -= tpcData.gainStepsInPowerStep;
			}

			//1.1. if reached gain index 0 - complete bottom of power out vector with 0 also
			while (index >= 0)
			{
                ASSERT (index < TPC_POWER_OUT_MAX_ROWS)
				gPowerOut[ant][bw][index] = 0;
				index--;
			}

			//2. go from reference up
			index = refIndex + 1;
			gainWord = refGainWord + tpcData.gainStepsInPowerStep;
			while ((index < TPC_POWER_OUT_MAX_ROWS) && (gainWord < TPC_GAIN_TABLE_MAX_ROWS))
			{
				gPowerOut[ant][bw][index] = (uint8)gainWord;
				index++;
				gainWord += tpcData.gainStepsInPowerStep;
			}

			//2.1. ifreachedmax gain index - complete top of power outvector with max gain index
			while (index < TPC_POWER_OUT_MAX_ROWS)
			{
				gPowerOut[ant][bw][index] = TPC_GAIN_TABLE_MAX_ROWS - 1;
				index++;
			}

			TPC_writePout(&gPowerOut[ant][bw][0], ant, bw);
		}
	}
}

void TPC_setFixedGain()
{
	uint8 powerOut[TPC_POWER_OUT_MAX_ROWS];
	uint8 ant;
	uint8 bw;
	uint8 index;

	for (index = 0; index < tpcData.powerVectorLen; index++)
	{
		powerOut[index] = tpcData.fixedGain;
	}
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
		{
			TPC_writePout(powerOut, ant, bw);
		}
	}
}

void TPC_setFixedPowerAndGain(dutPowerOutVecParams_t * params)
{
	uint32 ramAddr;
	uint32 powerWord;
	uint32 mask;
	uint8  shiftVal;
	
	ramAddr = PHY_TX_BASE_ADDR + TPC_POWER_OUT_START_ADDR +
				(((params->ant * POWER_VEC_ANT_SECTION_SIZE) +
				(params->bandwidth * POWER_VEC_BW_SECTION_SIZE) +
				(params->powerVectorIndex >> 1)) * POWER_VEC_BYTES_WIDTH);

	RegAccess_Read(ramAddr, &powerWord);
	shiftVal = (params->powerVectorIndex & 0x1) * POWER_VECTOR_WORD_WIDTH;
	mask = ((1 << POWER_VECTOR_WORD_WIDTH) - 1) << shiftVal;

	powerWord &= (~mask);
	powerWord |= (params->powerVectorWord << shiftVal);

	RegAccess_Write(ramAddr, powerWord);
	
}

void TPC_loadTSSIparams(uint8 ant, TPC_FREQ* tpcList)
{
	tpcS2dParams_t s2dparams;
	uint32 ramaddr;
	uint32 regVal;
	uint16 ABvals[TPC_MAX_S2D_REGIONS][2] = {0};
	uint8  s2dOffsetGain[TPC_MAX_S2D_REGIONS][2] = {0};
	uint8  bw;
	uint8  index;

	ramaddr = PHY_TX_BASE_ADDR + TPC_TSSI_TABLE_START_ADDR + (4 * TPC_TSSI_NUM_WORDS_PER_ANT * ant);

	s2dparams.ant = ant;
	
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		s2dparams.bw = bw;
		switch (tpcData.tpcFreqLen)
		{
			case TPC_FREQ_LEN_1S2D_NO_CAL:
				ABvals[0][0] = tpcList[bw].tpcFreq1s2dNoCal.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq1s2dNoCal.B1;
				break;
			case TPC_FREQ_LEN_1S2D:
				ABvals[0][0] = tpcList[bw].tpcFreq1s2d.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq1s2d.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq1s2d.uevmg_s2d.uevmg_s2d_fields.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq1s2d.uevmg_s2d.uevmg_s2d_fields.S2Dg1;
				break;
			case TPC_FREQ_LEN_2S2D_NO_CAL:
				ABvals[0][0] = tpcList[bw].tpcFreq2s2dNoCal.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq2s2dNoCal.B1;
				ABvals[1][0] = tpcList[bw].tpcFreq2s2dNoCal.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq2s2dNoCal.B2;
				break;
			case TPC_FREQ_LEN_2S2D:
				ABvals[0][0] = tpcList[bw].tpcFreq2s2d.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq2s2d.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq2s2d.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq2s2d.s2d.s2d_fields.S2Dg1;
				ABvals[1][0] = tpcList[bw].tpcFreq2s2d.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq2s2d.B2;
				s2dOffsetGain[1][0] = tpcList[bw].tpcFreq2s2d.S2Do2;
				s2dOffsetGain[1][1] = tpcList[bw].tpcFreq2s2d.s2d.s2d_fields.S2Dg2;
				break;
			case TPC_FREQ_LEN_3S2D_NO_CAL:
				ABvals[0][0] = tpcList[bw].tpcFreq3s2dNoCal.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq3s2dNoCal.B1;
				ABvals[1][0] = tpcList[bw].tpcFreq3s2dNoCal.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq3s2dNoCal.B2;
				ABvals[2][0] = tpcList[bw].tpcFreq3s2dNoCal.A3;
				ABvals[2][1] = tpcList[bw].tpcFreq3s2dNoCal.B3;
				break;
			case TPC_FREQ_LEN_3S2D:
				ABvals[0][0] = tpcList[bw].tpcFreq3s2d.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq3s2d.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Dg1;
				ABvals[1][0] = tpcList[bw].tpcFreq3s2d.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq3s2d.B2;
				s2dOffsetGain[1][0] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Do2;
				s2dOffsetGain[1][1] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Dg2;
				ABvals[2][0] = tpcList[bw].tpcFreq3s2d.A3;
				ABvals[2][1] = tpcList[bw].tpcFreq3s2d.B3;
				s2dOffsetGain[2][0] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Do3;
				s2dOffsetGain[2][1] = tpcList[bw].tpcFreq3s2d.s2d.s2d_fields.S2Dg3;
				break;
		}

		for (index = 0; index < TPC_MAX_S2D_REGIONS; index++)
		{
			s2dparams.region = index + 1;
			s2dparams.offset = s2dOffsetGain[index][0];
			s2dparams.gain = s2dOffsetGain[index][1];
			s2dparams.pThreshold = TPC_S2D_RAM_PTHRESHOLD_INVALID;
			if (s2dparams.gain != 0)
			{
				TPC_configureS2DGainAndOffset(&s2dparams);
			}

			regVal = (ABvals[index][0] << TPC_S2D_RAM_A_SHIFT) | (ABvals[index][1] << TPC_S2D_RAM_B_SHIFT);
			RegAccess_Write(ramaddr, regVal);
			ramaddr += 8;
		}
		
	}
	

}

bool TPC_calibrateGain(uint8 ant, uint8 bw, uint8 TpcIndex)
{
	//not implemented in Wave500
	return TRUE;
}

uint8 TPC_getLoopType()
{
	return tpcData.tpcLoopType;
}

void TPC_setTimer()
{
	//not implemented in Wave500
}

void TPC_stopTimer()
{
	//not implemented in Wave500
}

void ISR_TPC(void)
{
	//not implemented in Wave500
}

void TPC_writePout(uint8* pOutVec,uint8 ant,uint8 bandwidth)
{
	uint32 ramAddr;
	uint16 pOutWord;
	uint8 index;

	ramAddr = PHY_TX_BASE_ADDR + TPC_POWER_OUT_START_ADDR + ((ant * POWER_VEC_ANT_SECTION_SIZE) + (bandwidth * POWER_VEC_BW_SECTION_SIZE))*POWER_VEC_BYTES_WIDTH;

	for (index = 0; index < tpcData.powerVectorLen; index += 2)
	{
		pOutWord = pOutVec[index] | (pOutVec[index + 1] << POWER_VECTOR_WORD_WIDTH);
		RegAccess_Write(ramAddr, pOutWord);
		ramAddr += POWER_VEC_BYTES_WIDTH;
	}
}

int16 TPC_getRegulationLimit(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimit[bandwidth];
}

int16 TPC_getRegulationLimitMU(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimitMU[bandwidth];
}

int16 TPC_getRegulationLimitBF(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimitBF[bandwidth];
}



int16 TPC_get11bRegulationLimit()
{
	return tpcConfiguration.powerLimit11b;
}

int16 TPC_getMaxEVMpower(uint8 ant,uint8 bandwidth)
{
	return tpcData.maxEvmLimit[ant][bandwidth];
}

int16 TPC_getMinPower(uint8 ant,uint8 bandwidth)
{
	return tpcData.minPower[ant][bandwidth];
}

uint8 TPC_getMinAntBoostOverBws(uint8 ant)
{
	uint8 bw;
	uint8 minBoostPerAnt = MAX_UINT8;
	
	/* Run over all BWs and return the min boost for a specific ant */
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		if (tpcData.antPowerBoost[ant][bw] < minBoostPerAnt)
		{
			minBoostPerAnt = tpcData.antPowerBoost[ant][bw];
		}	
	}
	return minBoostPerAnt;
}

void TPC_initAntParams(tpcAntParams_t* params)
{
	tpcData.tpcFreqLen = params->tpcFreqLen;
	memcpy(&tpcAntParamsArray[params->ant], &params->tpcParams, sizeof(tpcFreqList_t));
}

void TPC_configReq(tpcConfig_t* cfgParams)
{
	RegPhyTxRegsTpcAcc_u	tpcLoopReg;
	
	if (cfgParams->getSetOperation == API_GET_OPERATION)
	{
		memcpy(cfgParams, &tpcConfiguration, sizeof(tpcConfig_t));
	}
	else
	{
		memcpy(&tpcConfiguration, cfgParams, sizeof(tpcConfig_t));
		
		TPC_init(tpcConfiguration.tpcLoopType);
		
		RegAccess_Read(REG_PHY_TX_REGS_TPC_ACC, &tpcLoopReg.val);
		if (cfgParams->tpcLoopType == TPC_OPEN_LOOP)
		{
			tpcLoopReg.bitFields.tssiOpenLoopEn = 1;
		}
		else
		{
			tpcLoopReg.bitFields.tssiOpenLoopEn = 0;
		}
		
		RegAccess_Write(REG_PHY_TX_REGS_TPC_ACC, tpcLoopReg.val);
	}
}

void TPC_GetTxPowerIndex(IN uint8 spectrumMode,OUT uint8 * powerIndex)
{
	*powerIndex = TPC_OPEN_LOOP_INDEX;
}

uint8 TPC_getPowerIndexValue(uint8 powerIndex, Antenna_e ant, Bandwidth_e bw)
{
	uint32 ramAddr;
	uint32 powerWord;
	
	ramAddr = PHY_TX_BASE_ADDR + TPC_POWER_OUT_START_ADDR +
				(((ant * POWER_VEC_ANT_SECTION_SIZE) +
				(bw * POWER_VEC_BW_SECTION_SIZE) +
				(powerIndex >> 1)) * POWER_VEC_BYTES_WIDTH);

	RegAccess_Read(ramAddr, &powerWord);

	return (powerWord);
}

void TPC_shiftPowerOutVector(uint8 ant,uint8 bw,uint8 startIndex, uint8 startGain)
{
	uint8 powerVec[TPC_POWER_OUT_MAX_ROWS];
	int8 index;
	uint8 step;
	uint8 gain;
	int8 index0;

	ASSERT(startIndex < tpcData.powerVectorLen);
	ASSERT(startGain < TPC_GAIN_TABLE_MAX_ROWS);

	step = tpcData.powerVectorStep / tpcData.digGainStep;

	index0 = (startIndex - (startGain / step));

	if (index0 < 0)
	{
		index0 = 0;		
	}
	
	for (index = 0; index < index0; index++)
	{
		powerVec[index] = 0;
	}

	gain = startGain - (step * (startIndex - index0));
	for (index = index0; index < TPC_POWER_OUT_MAX_ROWS; index++)
	{
		if (gain >= TPC_GAIN_TABLE_MAX_ROWS)
		{
			gain = TPC_GAIN_TABLE_MAX_ROWS - 1;
		}
		powerVec[index] = gain;
		gain += step;
	}

	TPC_writePout(powerVec, ant, bw);
}

/******************************************************************************/
/***					      Static Functions											***/
/******************************************************************************/

void TPC_FreqInterpolation_1s2d_no_cal(TPC_FREQ_1S2D_NO_CALIB * outTpcFreq,TPC_FREQ_1S2D_NO_CALIB * tpcFreq1,TPC_FREQ_1S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_1S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
}

void TPC_FreqInterpolation_2s2d_no_cal(TPC_FREQ_2S2D_NO_CALIB * outTpcFreq,TPC_FREQ_2S2D_NO_CALIB * tpcFreq1,TPC_FREQ_2S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_2S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
}

void TPC_FreqInterpolation_3s2d_no_cal(TPC_FREQ_3S2D_NO_CALIB * outTpcFreq,TPC_FREQ_3S2D_NO_CALIB * tpcFreq1,TPC_FREQ_3S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_3S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs(tpcFreq1->chID - outTpcFreq->chID);
	val2 = Abs(tpcFreq2->chID - outTpcFreq->chID);
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//A3, B3
	outTpcFreq->A3 = nearestTpc->A3;
	outTpcFreq->B3 = nearestTpc->B3;
}

void TPC_FreqInterpolation_1s2d(TPC_FREQ_1S2D * outTpcFreq,TPC_FREQ_1S2D * tpcFreq1,TPC_FREQ_1S2D * tpcFreq2)
{
	TPC_FREQ_1S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain * factorA) + (tpcFreq2->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain * factorB);
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.S2Dg1 = nearestTpc->uevmg_s2d.uevmg_s2d_fields.S2Dg1;
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.S2Do1 = nearestTpc->uevmg_s2d.uevmg_s2d_fields.S2Do1;
}

void TPC_FreqInterpolation_2s2d(TPC_FREQ_2S2D * outTpcFreq,TPC_FREQ_2S2D * tpcFreq1,TPC_FREQ_2S2D * tpcFreq2)
{
	TPC_FREQ_2S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->s2d.s2d_fields.S2Dg1 = nearestTpc->s2d.s2d_fields.S2Dg1;
	outTpcFreq->S2Do1 = nearestTpc->S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->s2d.s2d_fields.S2Dg2 = nearestTpc->s2d.s2d_fields.S2Dg2;
	outTpcFreq->S2Do2 = nearestTpc->S2Do2;
}

void TPC_FreqInterpolation_3s2d(TPC_FREQ_3S2D * outTpcFreq,TPC_FREQ_3S2D * tpcFreq1,TPC_FREQ_3S2D * tpcFreq2)
{
	TPC_FREQ_3S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->s2d.s2d_fields.S2Dg1 = nearestTpc->s2d.s2d_fields.S2Dg1;
	outTpcFreq->s2d.s2d_fields.S2Do1 = nearestTpc->s2d.s2d_fields.S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->s2d.s2d_fields.S2Dg2 = nearestTpc->s2d.s2d_fields.S2Dg2;
	outTpcFreq->s2d.s2d_fields.S2Do2 = nearestTpc->s2d.s2d_fields.S2Do2;
	//A3, B3
	outTpcFreq->A3 = nearestTpc->A3;
	outTpcFreq->B3 = nearestTpc->B3;
	//S2D #3
	outTpcFreq->s2d.s2d_fields.S2Dg3 = nearestTpc->s2d.s2d_fields.S2Dg3;
	outTpcFreq->s2d.s2d_fields.S2Do3 = nearestTpc->s2d.s2d_fields.S2Do3;
}

void TPC_bwInterpolation_1s2d_no_cal(TPC_FREQ_1S2D_NO_CALIB * outTpcFreq,TPC_FREQ_1S2D_NO_CALIB * tpcFreq1,TPC_FREQ_1S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_1S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
}

void TPC_bwInterpolation_2s2d_no_cal(TPC_FREQ_2S2D_NO_CALIB * outTpcFreq,TPC_FREQ_2S2D_NO_CALIB * tpcFreq1,TPC_FREQ_2S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_2S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
}

void TPC_bwInterpolation_3s2d_no_cal(TPC_FREQ_3S2D_NO_CALIB * outTpcFreq,TPC_FREQ_3S2D_NO_CALIB * tpcFreq1,TPC_FREQ_3S2D_NO_CALIB * tpcFreq2)
{
	TPC_FREQ_3S2D_NO_CALIB* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs(tpcFreq1->chID - outTpcFreq->chID);
	val2 = Abs(tpcFreq2->chID - outTpcFreq->chID);
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs((int16)(tpcFreq1->chID - tpcFreq2->chID));
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//A3, B3
	outTpcFreq->A3 = nearestTpc->A3;
	outTpcFreq->B3 = nearestTpc->B3;
}

void TPC_bwInterpolation_1s2d(TPC_FREQ_1S2D * outTpcFreq,TPC_FREQ_1S2D * tpcFreq1,TPC_FREQ_1S2D * tpcFreq2)
{
	TPC_FREQ_1S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain * factorA) + (tpcFreq2->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain * factorB);
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.S2Dg1 = nearestTpc->uevmg_s2d.uevmg_s2d_fields.S2Dg1;
	outTpcFreq->uevmg_s2d.uevmg_s2d_fields.S2Do1 = nearestTpc->uevmg_s2d.uevmg_s2d_fields.S2Do1;
}

void TPC_bwInterpolation_2s2d(TPC_FREQ_2S2D * outTpcFreq,TPC_FREQ_2S2D * tpcFreq1,TPC_FREQ_2S2D * tpcFreq2)
{
	TPC_FREQ_2S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->s2d.s2d_fields.S2Dg1 = nearestTpc->s2d.s2d_fields.S2Dg1;
	outTpcFreq->S2Do1 = nearestTpc->S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->s2d.s2d_fields.S2Dg2 = nearestTpc->s2d.s2d_fields.S2Dg2;
	outTpcFreq->S2Do2 = nearestTpc->S2Do2;
}

void TPC_bwInterpolation_3s2d(TPC_FREQ_3S2D * outTpcFreq,TPC_FREQ_3S2D * tpcFreq1,TPC_FREQ_3S2D * tpcFreq2)
{
	TPC_FREQ_3S2D* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM < tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM)
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq1->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	else
	{
		outTpcFreq->uevm_bw.uevm_bw_fields.ultimateEVM = tpcFreq2->uevm_bw.uevm_bw_fields.ultimateEVM;
	}
	//max power
	if (tpcFreq1->maxp_ant.maxp_ant_fields.maxPower < tpcFreq2->maxp_ant.maxp_ant_fields.maxPower)
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq1->maxp_ant.maxp_ant_fields.maxPower;
	}
	else
	{
		outTpcFreq->maxp_ant.maxp_ant_fields.maxPower = tpcFreq2->maxp_ant.maxp_ant_fields.maxPower;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->ultimateEVMgain * factorA) + (tpcFreq2->ultimateEVMgain * factorB);
	outTpcFreq->ultimateEVMgain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->s2d.s2d_fields.S2Dg1 = nearestTpc->s2d.s2d_fields.S2Dg1;
	outTpcFreq->s2d.s2d_fields.S2Do1 = nearestTpc->s2d.s2d_fields.S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->s2d.s2d_fields.S2Dg2 = nearestTpc->s2d.s2d_fields.S2Dg2;
	outTpcFreq->s2d.s2d_fields.S2Do2 = nearestTpc->s2d.s2d_fields.S2Do2;
	//A3, B3
	outTpcFreq->A3 = nearestTpc->A3;
	outTpcFreq->B3 = nearestTpc->B3;
	//S2D #3
	outTpcFreq->s2d.s2d_fields.S2Dg3 = nearestTpc->s2d.s2d_fields.S2Dg3;
	outTpcFreq->s2d.s2d_fields.S2Do3 = nearestTpc->s2d.s2d_fields.S2Do3;
}

void TPC_configureS2DGainAndOffset(tpcS2dParams_t *s2dParams)
{
	uint32 ramaddr;
	uint32 offset;
	uint32 gain;
	uint32 regValue;
	uint32 pthreshold;
	ramaddr = PHY_TX_BASE_ADDR + TPC_TSSI_TABLE_START_ADDR + (PRE_DRV_WORD_WIDTH * ((TPC_TSSI_NUM_WORDS_IN_REGION * s2dParams->region-1)
		+ (TPC_TSSI_TABLE_BW_SECTION_SIZE * s2dParams->bw)+ (TPC_TSSI_NUM_WORDS_PER_ANT * s2dParams->ant)));
	offset = s2dParams->offset << TPC_S2D_RAM_OFFSET_SHIFT;
	gain = s2dParams->gain << (TPC_S2D_RAM_OFFSET_SHIFT + TPC_S2D_RAM_GAIN_SHIFT);

	RegAccess_Read(ramaddr,&regValue);

	if (s2dParams->pThreshold == TPC_S2D_RAM_PTHRESHOLD_INVALID)
	{
		pthreshold = regValue & TPC_S2D_RAM_PTHRESHOLD_MASK;
	}
	else
	{
		pthreshold = s2dParams->pThreshold << (TPC_S2D_RAM_PTHRESHOLD_SHIFT);
	}
	
	regValue = regValue & ~TPC_S2D_RAM_MASK;
	
	RegAccess_Write(ramaddr,regValue | offset | gain | pthreshold);
}

void TPC_setTxPowerTableOffset(uint8 ant, TPC_FREQ* tpcFreqParams){}

void TPC_setGetTxTableOffset(uint8 antIdx, uint8 bwIdx, short* rf_power_offset_val, uint8 setGet ){}
