/******************************************************************************/
/***					      Include Files									***/
/******************************************************************************/
#include "System_MainApi.h"
#include "TpcClbrHndlr.h"
#include "CalibrationsDefines.h"
#include "HwMemoryMap.h"
#include "RegAccess_Api.h"
#include "PhyTxRegs.h"
#include "PhyRxTdRegs.h"
#include "MT_Math.h"
#include "RficDriver_API.h"
#include "loggerAPI.h"
#include "PSD.h"
#include "stringLibApi.h"
#include "ShramTpc.h"
#include "lib_wav600_api.h"
#include "lib_wrx654_api.h"
#include "lib_wrx654_sequences.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 19

/******************************************************************************/
/***					      Definitions													***/
/******************************************************************************/
#define TPC_FREQ_LEN_1_REGION		9
#define TPC_FREQ_LEN_2_REGION		14
#define TPC_FREQ_LEN_3_REGION		20


#define MAX_NUM_OF_TPC_FREQ (4)
/******************************************************************************/
/***					      Static Variables												***/
/******************************************************************************/

uint8 gPowerOut[MAX_NUM_OF_ANTENNAS][MAX_NUM_OF_BW][TPC_POWER_OUT_MAX_ROWS];

tpcConfig_t	tpcConfiguration;
tpc_params_t tpcData;

tpcFreqList_t tpcAntParamsArray[MAX_NUM_OF_ANTENNAS];
uint8 tpcAntBandwidth;


extern uint16 digGainWords[TPC_NUM_DIGITAL_WORDS];
extern uint8 preDrvGainWords[TPC_NUM_PRE_DRV_WORDS];
extern uint8 pgcGainWords[TPC_NUM_PGC_WORDS];

#define PGC_WORD_0_EXT_PA	2
#define PGC_WORD_1_EXT_PA	3
#define EXT_PA_NUM_DIFF_REGS	15



/******************************************************************************/
/***					      Public Functions											***/
/******************************************************************************/

void TPC_fillDefines(void)
{
	uint8 index1, index2;
    memset(&tpcData, 0, sizeof(tpc_params_t));
    memset(&tpcConfiguration, 0, sizeof(tpcConfig_t));
    memset(tpcAntParamsArray, 0, sizeof(tpcFreqList_t)*MAX_NUM_OF_ANTENNAS);
	
	tpcData.digGainMax = DIG_MAX_GAIN_DEFAULT;
	tpcData.digGainMin = DIG_LOWER_LIMIT_DEFAULT;
	tpcData.digGainStep = DIG_STEP_DEFAULT;
	tpcData.digGainUpperLimit = DIG_UPPER_LIMIT_DEFAULT;
	tpcData.digitalMinGain = DIG_MIN_GAIN_DEFAULT;
	tpcData.digWordOffset = DIG_WORD_OFFSET_DEFAULT;
	tpcData.digWordShift = DIG_WORD_SHIFT_DEFAULT;
	tpcData.numberOfTssiCutPts = NUM_TSSI_CUT_POINTS_DEFAULT;
	tpcData.pgc2lowerLimit = PGC_LOWER_LIMIT_DEFAULT;
	tpcData.pgc2step = PGC_STEP_DEFAULT;
	tpcData.pgc2upperLimit = PGC_UPPER_LIMIT_DEFAULT;
	tpcData.pgcWordOffset = PGC_WORD_OFFSET_DEFAULT;
	tpcData.pgcWordShift = PGC_WORD_SHIFT_DEFAULT;
	tpcData.powerVectorLen = POWER_VECTOR_LEN_DEFAULT;
	tpcData.powerVectorStep = POWER_VEC_STEP_DEFAULT;
	tpcData.preDrvMinGain = PRE_DRV_MIN_GAIN_DEFAULT;
	tpcData.preDrvStep = PRE_DRV_STEP_DEFAULT;
	tpcData.preDrvUpperLimit = PRE_DRV_UPPER_LIMIT_DEFAULT;
	tpcData.preDrvWordOffset = PRE_DRV_WORD_OFFSET_DEFAULT;
	tpcData.preDrvWordShift = PRE_DRV_WORD_SHIFT_DEFAULT;
	tpcData.tssiCharacteriztionType = TPC_CHARACTERIZATION_DEFAULT;
	tpcData.gainStepsInPowerStep = tpcData.powerVectorStep / tpcData.digGainStep;
	tpcData.fixedGain = NO_FIXED_GAIN;
	tpcData.powerLimitOffset = 0;

	for (index1 = 0; index1 < MAX_NUM_OF_ANTENNAS; index1++)
	{
		for (index2 = 0; index2 < MAX_NUM_OF_BW; index2++)
		{
			tpcData.minPowerGain[index1][index2] = MIN_POWER_INDEX_DEFAULT;
			tpcData.minPower[index1][index2] = MIN_POWER_DEFAULT;
			tpcData.maxEvmLimit[index1][index2] = MAX_EVM_DEFAULT;
		}
	}
}

void TPC_init(uint8 tpcType)
{
	UNUSED_PARAM(tpcType);
}

void TPC_loadChannelData(uint16 chID,uint8 bandwidth)
{
	uint8 ant;
	uint8 isvalid = 0;
	uint32 antMask = Hdk_GetTxAntMask();
	
#if LM_CALIBRATION_LOGS_ON
ILOG0_DDD("TPC_loadChannelData chID = 0x%x, bw = %d, antMask=%d", chID, bandwidth,antMask);
#endif

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if((1<<ant) & antMask)
		{
			if(tpcData.tpcFreqLen == TPC_FREQ_LEN_1_REGION || tpcData.tpcFreqLen == TPC_FREQ_LEN_2_REGION || tpcData.tpcFreqLen == TPC_FREQ_LEN_3_REGION)
			{
			  TPC_extractTpcFreqParams(ant, chID, bandwidth);
			  isvalid++;
			}	
		}
	}

	if(isvalid)
	  TPC_calcAntennaBoosts();
	
#ifndef ENET_USE_DUMMY_PHY
	if (tpcData.fixedGain == NO_FIXED_GAIN)
	{
		TPC_buildPout();
	}
#endif
}


void TPC_getPowerToIndexLUT(LA_SET_TPC_DATA_t* tpcInfo)
{
	tpcInfo->indexOffset= 0;
	tpcInfo->tpcPowerVectorLen = tpcData.powerVectorLen;
	tpcInfo->tpcpPwerVectorStep = tpcData.powerVectorStep;

}



void TPC_extractTpcFreqParams(uint8 ant, uint16 channel, uint8 bandwidth)
{
	TPC_FREQ				tpcFreqParams[MAX_NUM_OF_TPC_FREQ];
	uint8					tpcList[MAX_NUM_OF_BW][2] = {0};
	TPC_FREQ_1_REGION*		tpcFreqPtr;
	uint8					numElements[MAX_NUM_OF_BW] = {0};
	uint8					index;
	uint16					bw;
	uint8					tpcBwBitmap;
	SetChannelParams_t* 	pSetChannelParams;
	uint8					rfband;
	
	pSetChannelParams = HDK_getSetChannelParams();
	rfband = pSetChannelParams->band;
	
#if LM_CALIBRATION_LOGS_ON
	ILOG0_DDD("TPC_extractTpcFreqParams ant = 0x%x, channel = 0x%x, bw = %d", ant, channel, bandwidth);
	ILOG0_DD("TPC_extractTpcFreqParams channel = 0x%x, rfBand = %d", channel, rfband);
#endif

	//1. sort elements by bandwidth
	for (index = 0; index < TPC_FREQ_STR_NUM; index++)
	{
		tpcFreqPtr = (TPC_FREQ_1_REGION*)(&tpcAntParamsArray[ant].tpcFreq[index]);
		if (tpcFreqPtr->chID != TPC_INVALID_CHID)
		{
			bw = tpcFreqPtr->bw;
            ASSERT(bw < MAX_NUM_OF_BW);
            ASSERT(numElements[bw] < 2);
			tpcList[bw][numElements[bw]] = index;
			numElements[bw]++;
		}
	}
	//2. for each bandwidth:
	//	2.1. if only one element - take this element
	//	2.2. if 2 elements - interpolate
	tpcBwBitmap = 0;
	for (index = 0; index < MAX_NUM_OF_BW; index++)
	{
		if (numElements[index] == 1)
		{
			tpcFreqParams[index].tpcFreq1Region.chID = channel;
			MEMCPY(&tpcFreqParams[index], &tpcAntParamsArray[ant].tpcFreq[tpcList[index][0]], sizeof(TPC_FREQ));
			tpcBwBitmap |= (1 << index);
		}
		else if (numElements[index] == 2)
		{
			tpcFreqParams[index].tpcFreq1Region.chID = channel;
			TPC_interpolateFrequencies(&tpcFreqParams[index],&tpcAntParamsArray[ant].tpcFreq[tpcList[index][0]], &tpcAntParamsArray[ant].tpcFreq[tpcList[index][1]]);
			tpcBwBitmap |= (1 << index);
		}
	}

	//3. check if some bandwidths are missing and interpolate from other bandwidth
	switch (tpcBwBitmap)
	{
		case 1: //only 20MHz exist
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[0], sizeof(TPC_FREQ)); //40 = 20
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[0], sizeof(TPC_FREQ)); //80 = 20
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[0], sizeof(TPC_FREQ)); //160 = 20
			break;
		case 2:	//only 40MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ));	//20 = 40
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[1], sizeof(TPC_FREQ)); //80 = 40
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[1], sizeof(TPC_FREQ)); //160 = 40
			break;
		case 3:	//only 20 & 40 MHz exist
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[1], sizeof(TPC_FREQ)); //80 = 40
			if(rfband == BAND_2_4_GHZ)
				MEMCPY(&tpcFreqParams[3], &tpcFreqParams[0], sizeof(TPC_FREQ)); //160 = 20
			else
				MEMCPY(&tpcFreqParams[3], &tpcFreqParams[1], sizeof(TPC_FREQ)); //160 = 40
			break;
		case 4: //only 80MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[2], sizeof(TPC_FREQ)); //20 = 80
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[2], sizeof(TPC_FREQ)); //40 = 80
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[2], sizeof(TPC_FREQ)); //160 = 80
			break;
		case 5: //only 20 & 80 MHz exist
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[0], sizeof(TPC_FREQ)); //40 = 20
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[2], sizeof(TPC_FREQ)); //160 = 80
			break;
		case 6: //only 40 & 80 MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ)); //20 = 40
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[2], sizeof(TPC_FREQ)); //160 = 80
			break;
		case 7: //20 & 40 & 80 MHz exist
			MEMCPY(&tpcFreqParams[3], &tpcFreqParams[2], sizeof(TPC_FREQ)); //160 = 80
			break;
		case 8: //only 160 MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[3], sizeof(TPC_FREQ)); //20 = 160
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[3], sizeof(TPC_FREQ)); //40 = 160
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[3], sizeof(TPC_FREQ)); //80 = 160
			break;
		case 9: //only 20 & 160 MHz exist
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[0], sizeof(TPC_FREQ)); //40 = 20
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[3], sizeof(TPC_FREQ)); //80 = 160
			break;
		case 10: //only 40 & 160 MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ)); //20 = 40
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[3], sizeof(TPC_FREQ)); //80 = 160
			break;
		case 11: //20 & 40 & 160 MHz exist
			MEMCPY(&tpcFreqParams[2], &tpcFreqParams[1], sizeof(TPC_FREQ)); //80 = 40
			break;
		case 12: //only 80 & 160 MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[2], sizeof(TPC_FREQ)); //20 = 80
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[2], sizeof(TPC_FREQ)); //40 = 80
			break;
		case 13: //20 & 80 & 160 MHz exist
			MEMCPY(&tpcFreqParams[1], &tpcFreqParams[0], sizeof(TPC_FREQ)); //40 = 20
			break;
		case 14: //40  & 80 & 160 MHz exist
			MEMCPY(&tpcFreqParams[0], &tpcFreqParams[1], sizeof(TPC_FREQ)); //20 = 40
			break;
		default: //must be that all bandwidths exist, otherwise - error (non exist)
			//ASSERT(tpcBwBitmap);
			break;			
	}

	//3.1 restore bandwidth after interpolation
	for (index = 0; index < MAX_NUM_OF_BW; index++)
	{
		tpcFreqParams[index].tpcFreq1Region.bw = index;
	}
	//4. build TPC list for current channel and all bandwidths
	TPC_buildTpcLimitsFromTpcFreq(ant, tpcFreqParams);
	tpcAntBandwidth = bandwidth;
#ifndef ENET_USE_DUMMY_PHY
	TPC_loadTSSIparams(ant, tpcFreqParams);
	TPC_setTxPowerTableOffset(ant, tpcFreqParams);
#endif
}

void TPC_calcAntennaBoosts()
{
	uint8	bw;
	uint8	ant;
	uint8 	minPower;
	uint8 	boost;
	uint8 	minPowerAnt;
	uint32	antMask = Hdk_GetTxAntMask();

	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		//first - find antenna with minimal maximum EVM power
		minPower = tpcData.powerVectorLen;
		minPowerAnt = 0;
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != 0)
			{
				if (tpcData.maxEvmLimit[ant][bw] < minPower)
				{
					minPower = tpcData.maxEvmLimit[ant][bw];
					minPowerAnt = ant;
				}
			}
		}

		//now boost of each antenna is maximum EVM of antenna minus minimal maximum EVM power
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (((1 << ant) & antMask) != 0)
			{
				tpcData.minBoostAntIndex[bw] = minPowerAnt;
				boost = tpcData.maxEvmLimit[ant][bw] - minPower;
				 //round down to 1dB resolution and limit by 3 dB
				tpcData.antPowerBoost[ant][bw] = (boost * tpcData.powerVectorStep) >> GAIN_RESOLUTION;
				if (tpcData.antPowerBoost[ant][bw] > TPC_MAX_BOOST)
				{
					tpcData.antPowerBoost[ant][bw] = TPC_MAX_BOOST;
				}
			}
		}
	}
}

void TPC_interpolateFrequencies(TPC_FREQ * outTpcFreq,TPC_FREQ * tpcFreq1,TPC_FREQ * tpcFreq2)
{
	switch (tpcData.tpcFreqLen)
	{
		case TPC_FREQ_LEN_1_REGION:
			TPC_FreqInterpolation_1s2d((TPC_FREQ_1_REGION*)outTpcFreq, (TPC_FREQ_1_REGION*)tpcFreq1, (TPC_FREQ_1_REGION*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_2_REGION:
			TPC_FreqInterpolation_2s2d((TPC_FREQ_2_REGION*)outTpcFreq, (TPC_FREQ_2_REGION*)tpcFreq1, (TPC_FREQ_2_REGION*)tpcFreq2);
			break;
		case TPC_FREQ_LEN_3_REGION:
			TPC_FreqInterpolation_3s2d((TPC_FREQ_3_REGION*)outTpcFreq, (TPC_FREQ_3_REGION*)tpcFreq1, (TPC_FREQ_3_REGION*)tpcFreq2);
			break;
	}
}

void TPC_buildTpcLimitsFromTpcFreq(uint8 ant, TPC_FREQ* tpcList)
{
	TPC_FREQ* tpcFreqPtr;
	uint8 bw;
	
	// copy parameters
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		tpcFreqPtr = (TPC_FREQ*)(&tpcList[bw]);
		tpcData.maxEvmLimit[ant][bw] = tpcFreqPtr->tpcFreq1Region.maxp;  // send to LA
		tpcData.minPower[ant][bw] = tpcFreqPtr->tpcFreq1Region.uevm_index; // send to LA
		tpcData.minPowerGain[ant][bw] = tpcFreqPtr->tpcFreq1Region.uevm_gain & (TPC_GAIN_TABLE_MAX_ROWS - 1);
	}

}


void TPC_buildPout()
{
	int16 gainWord;
	int16 refGainWord;
	uint8 ant,bw;
	int8  refIndex;
	int8  index;
	uint32	antMask = Hdk_GetTxAntMask();

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
	  if (((1 << ant) & antMask) != 0)
	  {
		for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
		{
			//min power is reference point for buildingpowerout vector
			refIndex = (int8)tpcData.minPower[ant][bw];
			refGainWord = (int16)tpcData.minPowerGain[ant][bw];
			//1. go from refrence down
			index = refIndex;
			gainWord = refGainWord;
			while ((index >= 0) && (gainWord > 0)) // KW IGNORE INFINITE_LOOP.LOCAL This is not an infinite loop as index-- is done below.
			{
                ASSERT (index < TPC_POWER_OUT_MAX_ROWS)
				gPowerOut[ant][bw][index] = (uint8)gainWord;
#if LM_CALIBRATION_LOGS_ON
ILOG0_DDDD("TPC_buildPout down gPowerOut: , gPowerOut 0x%x, ant 0x%x, bw 0x%x, index 0x%x",gPowerOut[ant][bw][index], ant, bw, index);
#endif
				index--;
				gainWord -= tpcData.gainStepsInPowerStep;
			}

			//1.1. if reached gain index 0 - complete bottom of power out vector with 0 also
			while (index >= 0)
			{
                ASSERT (index < TPC_POWER_OUT_MAX_ROWS)
				gPowerOut[ant][bw][index] = 0;
				index--;
			}

			//2. go from reference up
			index = refIndex + 1;
			gainWord = refGainWord + tpcData.gainStepsInPowerStep;
			while ((index < TPC_POWER_OUT_MAX_ROWS) && (gainWord < TPC_GAIN_TABLE_MAX_ROWS))
			{
				gPowerOut[ant][bw][index] = (uint8)gainWord;
#if LM_CALIBRATION_LOGS_ON
ILOG0_DDDD("TPC_buildPout up gPowerOut: , gPowerOut 0x%x, ant 0x%x, bw 0x%x, index 0x%x",gPowerOut[ant][bw][index], ant, bw, index);
#endif
				index++;
				gainWord += tpcData.gainStepsInPowerStep;
			}

			//2.1. ifreachedmax gain index - complete top of power outvector with max gain index
			while (index < TPC_POWER_OUT_MAX_ROWS)
			{
				gPowerOut[ant][bw][index] = TPC_GAIN_TABLE_MAX_ROWS - 1;
#if LM_CALIBRATION_LOGS_ON
ILOG0_DDDD("TPC_buildPout max gPowerOut: , gPowerOut 0x%x, ant 0x%x, bw 0x%x, index 0x%x",gPowerOut[ant][bw][index], ant, bw, index);
#endif
				index++;
			}
			TPC_writePout(gPowerOut[ant][bw], ant, bw);
		}
	  }
	}
}



void TPC_setFixedGain()
{
	uint8 powerOut[TPC_POWER_OUT_MAX_ROWS];
	uint8 ant;
	uint8 bw;
	uint8 index;

	for (index = 0; index < tpcData.powerVectorLen; index++)
	{
		powerOut[index] = tpcData.fixedGain;
	}
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
		{
			TPC_writePout(powerOut, ant, bw);
		}
	}
}


// not called
void TPC_setFixedPowerAndGain(dutPowerOutVecParams_t * params)
{
	uint32 ramAddr;
	uint32 powerWord;
	uint32 mask;
	uint8  shiftVal;

	ramAddr = HYP_TD_ANTENNA_0_TPC_POUT +
			(((params->ant * (HYP_TD_ANTENNA_1_TPC_POUT-HYP_TD_ANTENNA_0_TPC_POUT)) +
			(params->bandwidth * POWER_VEC_BW_SECTION_SIZE) +
			(params->powerVectorIndex >> 1)) * POWER_VEC_BYTES_WIDTH);

	RegAccess_Read(ramAddr, &powerWord);
	shiftVal = (params->powerVectorIndex & 0x1) * POWER_VECTOR_WORD_WIDTH;
	mask = ((1 << POWER_VECTOR_WORD_WIDTH) - 1) << shiftVal;

	powerWord &= (~mask);
	powerWord |= (params->powerVectorWord << shiftVal);

	RegAccess_Write(ramAddr, powerWord);
}

void TPC_loadTSSIparams(uint8 ant, TPC_FREQ* tpcList)
{
	tpcS2dParams_t s2dparams;
	int16 ABvals[TPC_MAX_S2D_REGIONS][2] = {0};
	uint8  s2dOffsetGain[TPC_MAX_S2D_REGIONS][2] = {0};
	uint8  bw;
	uint8  index;
	
	s2dparams.ant = ant;
	
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		s2dparams.bw = bw;
#if LM_CALIBRATION_LOGS_ON
		ILOG0_DD("TPC_loadTSSIParams bw = %d, tpcFreqLen = %d", s2dparams.bw, tpcData.tpcFreqLen);
#endif
		switch (tpcData.tpcFreqLen)
		{
			case TPC_FREQ_LEN_1_REGION:
				ABvals[0][0] = tpcList[bw].tpcFreq1Region.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq1Region.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq1Region.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq1Region.S2Dg1;
				break;
			case TPC_FREQ_LEN_2_REGION:
				ABvals[0][0] = tpcList[bw].tpcFreq2Region.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq2Region.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq2Region.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq2Region.S2Dg1;
				ABvals[1][0] = tpcList[bw].tpcFreq2Region.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq2Region.B2;
				s2dOffsetGain[1][0] = tpcList[bw].tpcFreq2Region.S2Do2;
				s2dOffsetGain[1][1] = tpcList[bw].tpcFreq2Region.S2Dg2;
				break;
			case TPC_FREQ_LEN_3_REGION:
				ABvals[0][0] = tpcList[bw].tpcFreq3Region.A1;
				ABvals[0][1] = tpcList[bw].tpcFreq3Region.B1;
				s2dOffsetGain[0][0] = tpcList[bw].tpcFreq3Region.S2Do1;
				s2dOffsetGain[0][1] = tpcList[bw].tpcFreq3Region.S2Dg1;
				ABvals[1][0] = tpcList[bw].tpcFreq3Region.A2;
				ABvals[1][1] = tpcList[bw].tpcFreq3Region.B2;
				s2dOffsetGain[1][0] = tpcList[bw].tpcFreq3Region.S2Do2;
				s2dOffsetGain[1][1] = tpcList[bw].tpcFreq3Region.S2Dg2;
				ABvals[2][0] = tpcList[bw].tpcFreq3Region.A3;
				ABvals[2][1] = tpcList[bw].tpcFreq3Region.B3;
				s2dOffsetGain[2][0] = tpcList[bw].tpcFreq3Region.S2Do3;
				s2dOffsetGain[2][1] = tpcList[bw].tpcFreq3Region.S2Dg3;
				break;
		}

		for (index = 0; index < TPC_MAX_S2D_REGIONS; index++)
		{
			s2dparams.region = index + 1;
			s2dparams.offset = s2dOffsetGain[index][0];
			s2dparams.gain = s2dOffsetGain[index][1];
			s2dparams.pThreshold = TPC_S2D_RAM_PTHRESHOLD_INVALID;
			if (s2dparams.gain != 0 && bw == tpcAntBandwidth)
			{
#if LM_CALIBRATION_LOGS_ON
						ILOG0_DDD("TPC_loadTSSIParams tpcAntBandwidth = %d, bw = %d, tpcFreqLen = %d", tpcAntBandwidth, s2dparams.bw, tpcData.tpcFreqLen);
#endif

				TPC_configureS2DGainAndOffset(&s2dparams);
			}
			bbic_set_tx_power_coef(s2dparams.ant, s2dparams.bw, index, ABvals[index][0], ABvals[index][1]);
		}
		
	}

}


void TPC_setGetTxTableOffset(uint8 antIdx, uint8 bwIdx, short* rf_power_offset_val, uint8 setGet )
{
	if(setGet == 1) //set=1
	{
		bbic_set_tx_power_table_offset(antIdx, bwIdx, *rf_power_offset_val);
	}
	else
	{
		bbic_get_tx_power_table_offset(antIdx, bwIdx, rf_power_offset_val);
	}
}
void TPC_setTxPowerTableOffset(uint8 ant, TPC_FREQ* tpcFreqParams)
{
	uint8 index;

	//set tx power table offset
	//uevm_gain(uint8 defined in TPC_FREQ_x_REGION) is the rf power table (gain table) offset

	for (index = 0; index < MAX_NUM_OF_BW; index++)
	{
			bbic_set_tx_power_table_offset(ant, tpcFreqParams[index].tpcFreq1Region.bw , tpcFreqParams[index].tpcFreq1Region.uevm_gain);
#if LM_CALIBRATION_LOGS_ON
			ILOG0_DDD("TPC_setTxPowerTableOffset ant = %d, bw = %d, uevm_gain = %d", ant,tpcFreqParams[index].tpcFreq1Region.bw , tpcFreqParams[index].tpcFreq1Region.uevm_gain);
#endif
	}
}

bool TPC_calibrateGain(uint8 ant, uint8 bw, uint8 TpcIndex)
{
	UNUSED_PARAM(ant);	
	UNUSED_PARAM(bw);	
	UNUSED_PARAM(TpcIndex);	
	//not implemented in Wave500
	return TRUE;
}

uint8 TPC_getLoopType()
{
	return tpcData.tpcLoopType;
}

void TPC_setTimer()
{
	//not implemented in Wave500
}

void TPC_stopTimer()
{
	//not implemented in Wave500
}

void ISR_TPC(void)
{
	//not implemented in Wave500
}


void TPC_writePout(uint8* pOutVec,uint8 ant,uint8 bandwidth)
{
	uint32 ramAddr;
	uint32 mask;
	uint32 pOutWord;
	uint8  index;
 	uint16 antenna_offset = 0x4000;
	
 	mask = 0xFF << (8 * bandwidth);
	ramAddr = HYP_TD_ANTENNA_0_TPC_POUT + ((ant * (antenna_offset))) ;
	

	for (index = 0; index < tpcData.powerVectorLen; index++)
	{
		pOutWord= pOutVec[index] << (8 * bandwidth);
		RegAccess_WriteMasked(ramAddr, mask ,pOutWord );
		
		ramAddr += POWER_VEC_BYTES_WIDTH;
	}

}

int16 TPC_getRegulationLimit(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimit[bandwidth];
}

int16 TPC_getRegulationLimitMU(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimitMU[bandwidth];
}

int16 TPC_getRegulationLimitBF(uint8 bandwidth)
{
	return tpcConfiguration.regulationLimitBF[bandwidth];
}



int16 TPC_get11bRegulationLimit()
{
	return tpcConfiguration.powerLimit11b;
}

int16 TPC_getMaxEVMpower(uint8 ant,uint8 bandwidth)
{
	// Use 18 dBm for max EVM and max power until it will be parsed from cal file
	return tpcData.maxEvmLimit[ant][bandwidth];
}

int16 TPC_getMinPower(uint8 ant,uint8 bandwidth)
{
	return tpcData.minPower[ant][bandwidth];	//UtimateEvm
}

uint8 TPC_getMinAntBoostOverBws(uint8 ant)
{
	uint8 bw;
	uint8 minBoostPerAnt = MAX_UINT8;
	
	/* Run over all BWs and return the min boost for a specific ant */
	for (bw = 0; bw < MAX_NUM_OF_BW; bw++)
	{
		if (tpcData.antPowerBoost[ant][bw] < minBoostPerAnt)
		{
			minBoostPerAnt = tpcData.antPowerBoost[ant][bw];
		}	
	}
	return minBoostPerAnt;
}
void TPC_initAntParams(tpcAntParams_t* params)
{
	tpcData.tpcFreqLen = params->tpcFreqLen;
	MEMCPY(&tpcAntParamsArray[params->ant], &params->tpcParams, sizeof(tpcFreqList_t));
}

void TPC_configReq(tpcConfig_t* cfgParams)
{
#ifndef ENET_USE_DUMMY_PHY
	uint32 antMask = Hdk_GetTxAntMask();

	if (cfgParams->tpcLoopType == TPC_OPEN_LOOP)
	{
		bbic_set_tx_open_loop(antMask, TRUE);
	}
	else
	{
		bbic_set_tx_open_loop(antMask, FALSE);
	}
#endif
	
	MEMCPY(&tpcConfiguration, cfgParams, sizeof(tpcConfig_t));
}

void TPC_GetTxPowerIndex(IN uint8 spectrumMode,OUT uint8 * powerIndex)
{
	UNUSED_PARAM(spectrumMode);	
	UNUSED_PARAM(powerIndex);	
	*powerIndex = TPC_OPEN_LOOP_INDEX;
}

// not needed; uevm
uint8 TPC_getPowerIndexValue(uint8 powerIndex, Antenna_e ant, Bandwidth_e bw)
{
	uint32 ramAddr;
	uint32 powerWord;
	
	ramAddr = HYP_TD_ANTENNA_0_TPC_POUT +
				((ant * (HYP_TD_ANTENNA_1_TPC_POUT-HYP_TD_ANTENNA_0_TPC_POUT)) +
				((bw * POWER_VEC_BW_SECTION_SIZE) +
				(powerIndex >> 1)) * POWER_VEC_BYTES_WIDTH);
	
	RegAccess_Read(ramAddr, &powerWord);
	return (powerWord);
}

void TPC_shiftPowerOutVector(uint8 ant,uint8 bw,uint8 startIndex, uint8 startGain)
{
	uint8 powerVec[TPC_POWER_OUT_MAX_ROWS];//TPC_POWER_OUT_MAX_ROWS = 96;
	int8 index;
	uint8 step;
	uint8 gain;
	int8 index0;
	//powerVectorLen 0-95
	//digGainStep = 2 Const

	ASSERT(startIndex < tpcData.powerVectorLen);
	ASSERT(startGain < TPC_GAIN_TABLE_MAX_ROWS);

	step = tpcData.powerVectorStep / tpcData.digGainStep;

	index0 = (startIndex - (startGain / step));

	if (index0 < 0)
	{
		index0 = 0;		
	}
	
	for (index = 0; index < index0; index++)
	{
		powerVec[index] = 0;
	}

	gain = startGain - (step * (startIndex - index0));
	for (index = index0; index < TPC_POWER_OUT_MAX_ROWS; index++)
	{
		if (gain >= TPC_GAIN_TABLE_MAX_ROWS)
		{
			gain = TPC_GAIN_TABLE_MAX_ROWS - 1;
		}
		powerVec[index] = gain;
		gain += step;
	}

	TPC_writePout(powerVec, ant, bw);
}

/******************************************************************************/
/***					      Static Functions											***/
/******************************************************************************/
void TPC_FreqInterpolation_1s2d(TPC_FREQ_1_REGION * outTpcFreq,TPC_FREQ_1_REGION * tpcFreq1,TPC_FREQ_1_REGION * tpcFreq2)
{
	TPC_FREQ_1_REGION * nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
#if LM_CALIBRATION_LOGS_ON
	ILOG0_DDD("TPC_FreqInterpolation_1s2d outTpcFreq->chID = %d, tpcFreq1->chID = %d, tpcFreq2->chID = %d", outTpcFreq->chID,tpcFreq1->chID,tpcFreq2->chID);
#endif
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_index < tpcFreq2->uevm_index)
	{
		outTpcFreq->uevm_index = tpcFreq1->uevm_index;
	}
	else
	{
		outTpcFreq->uevm_index = tpcFreq2->uevm_index;
	}
	//max power
	if (tpcFreq1->maxp < tpcFreq2->maxp)
	{
		outTpcFreq->maxp = tpcFreq1->maxp;
	}
	else
	{
		outTpcFreq->maxp = tpcFreq2->maxp;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->uevm_gain * factorA) + (tpcFreq2->uevm_gain * factorB);
	outTpcFreq->uevm_gain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->S2Dg1 = nearestTpc->S2Dg1;
	outTpcFreq->S2Do1 = nearestTpc->S2Do1;
}

void TPC_FreqInterpolation_2s2d(TPC_FREQ_2_REGION * outTpcFreq,TPC_FREQ_2_REGION * tpcFreq1,TPC_FREQ_2_REGION * tpcFreq2)
{
	TPC_FREQ_2_REGION* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_index < tpcFreq2->uevm_index)
	{
		outTpcFreq->uevm_index = tpcFreq1->uevm_index;
	}
	else
	{
		outTpcFreq->uevm_index = tpcFreq2->uevm_index;
	}
	//max power
	if (tpcFreq1->maxp < tpcFreq2->maxp)
	{
		outTpcFreq->maxp = tpcFreq1->maxp;
	}
	else
	{
		outTpcFreq->maxp = tpcFreq2->maxp;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->uevm_gain * factorA) + (tpcFreq2->uevm_gain * factorB);
	outTpcFreq->uevm_gain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->S2Dg1 = nearestTpc->S2Dg1;
	outTpcFreq->S2Do1 = nearestTpc->S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->S2Dg2 = nearestTpc->S2Dg2;
	outTpcFreq->S2Do2 = nearestTpc->S2Do2;
}

void TPC_FreqInterpolation_3s2d(TPC_FREQ_3_REGION * outTpcFreq,TPC_FREQ_3_REGION * tpcFreq1,TPC_FREQ_3_REGION * tpcFreq2)
{
	TPC_FREQ_3_REGION* nearestTpc;
	uint16 val1;
	uint16 val2;
	uint32 interpolatedVal;
	uint16 factorA;
	uint16 factorB;
	uint16 freqDiff;
	uint16 halfFreqDiff;

	//find nearest frequency
	val1 = Abs((int16)(tpcFreq1->chID - outTpcFreq->chID));
	val2 = Abs((int16)(tpcFreq2->chID - outTpcFreq->chID));
	if (val1 < val2)
	{
		nearestTpc = tpcFreq1;
	}
	else
	{
		nearestTpc = tpcFreq2;
	}

	//find interpolation factors
	freqDiff = Abs(tpcFreq1->chID - tpcFreq2->chID);
	halfFreqDiff = freqDiff >> 1;
	factorA = ((val2 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	factorB = ((val1 << TPC_INTERPOLATION_RES) + halfFreqDiff) / freqDiff;
	
	//ultimate EVM
	if (tpcFreq1->uevm_index < tpcFreq2->uevm_index)
	{
		outTpcFreq->uevm_index = tpcFreq1->uevm_index;
	}
	else
	{
		outTpcFreq->uevm_index = tpcFreq2->uevm_index;
	}
	//max power
	if (tpcFreq1->maxp < tpcFreq2->maxp)
	{
		outTpcFreq->maxp = tpcFreq1->maxp;
	}
	else
	{
		outTpcFreq->maxp = tpcFreq2->maxp;
	}
	//ultimate EVM gain
	interpolatedVal = (tpcFreq1->uevm_gain * factorA) + (tpcFreq2->uevm_gain * factorB);
	outTpcFreq->uevm_gain = (interpolatedVal + TPC_INTERPOLATION_ROUND_FACTOR) >> TPC_INTERPOLATION_RES;
	//A1, B1
	outTpcFreq->A1 = nearestTpc->A1;
	outTpcFreq->B1 = nearestTpc->B1;
	//S2D #1
	outTpcFreq->S2Dg1 = nearestTpc->S2Dg1;
	outTpcFreq->S2Do1 = nearestTpc->S2Do1;
	//A2, B2
	outTpcFreq->A2 = nearestTpc->A2;
	outTpcFreq->B2 = nearestTpc->B2;
	//S2D #2
	outTpcFreq->S2Dg2 = nearestTpc->S2Dg2;
	outTpcFreq->S2Do2 = nearestTpc->S2Do2;
	//A3, B3
	outTpcFreq->A3 = nearestTpc->A3;
	outTpcFreq->B3 = nearestTpc->B3;
	//S2D #3
	outTpcFreq->S2Dg3 = nearestTpc->S2Dg3;
	outTpcFreq->S2Do3 = nearestTpc->S2Do3;
}

void TPC_configureS2DGainAndOffset(tpcS2dParams_t *s2dParams)
{

#if LM_CALIBRATION_LOGS_ON
	ILOG0_DD("TPC_configureS2DGainAndOffset ant = %d, bw = %d", s2dParams->ant, s2dParams->bw);
	ILOG0_DDD("TPC_configureS2DGainAndOffset region = %d, gain = %d, offset = %d",s2dParams->region, s2dParams->gain, s2dParams->offset);
#endif
#if !defined(ENET_INC_HW_FPGA) || defined(HDK_RF_EMULATOR)
	rfic_config_tssi((1<<s2dParams->ant), s2dParams->region-1,(int8) (s2dParams->gain<<1)-10, s2dParams->offset);
#else
	UNUSED_PARAM(s2dParams);
#endif
}


