/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "TxIQMismatchClbrHndlr.h"
#include "CalibrationsDefines.h"
#include "System_GlobalDefinitions.h"
#include "mt_cnfg.h"
#include "RegAccess_Api.h"
#include "mt_sysrst.h"
#include "MT_Math.h"
#include "Hdk_Api.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "RficDriver_API.h"
#include "Afe_API.h"
#include "CalibrationHandler.h"
#include "CalibrationHandlerUtils.h"
#include "ErrorHandler_Api.h"
#include "Shram_ClbrDataBuffer.h"
#include "LmHdk_API.h"
#include "CoC_Api.h"



#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 20

/******************************************************************************/
/***						Type Definitions								***/
/******************************************************************************/ 

extern Calibration_params_otf_t Calibration_params_otf;	






/******************************************************************************/
/***						static Variables								***/
/******************************************************************************/

/* During on line calibration we divide the calibration to avoid long calibration 
   the following struct is used to manage on line calibration
*/
static TxIqOnlineDetails_t txIqOnlineDetails;
static bool useSsbLoop;
/******************************************************************************/
/***					Static Function Declarations						***/
/******************************************************************************/ 
static RetVal_e run(IN ClbrType_e inCalibType, OUT TxIq_elements_t* outElements_p );
static void fillCalibrationDetails(IN ClbrType_e inCalibType, OUT TxIq_elements_t* outElements_p, TxIqCalibRangeDetails_t *outCalibratedRangeDetails_p);
static void  initAntennaOnlineIterationDetails(bool inFirstTime, OUT TxIq_elements_t* outElements_p);
static void fillNextOnlineIterationDetails(IN TxIq_elements_t* inElements_p, bool *outIsLastIteration_p);
static RetVal_e calibrateSingleAntenna(IN ClbrType_e inCalibType, uint8 inRfCalibAnt,TxIqCalibRangeDetails_t *inCalibratedRangeDetails, OUT TxIq_elements_t* outElements_p);
static __inline void initHwRegisters(RficLoopbackAntennas_t *inAntInfo_p, IN TxIq_elements_t* inElements_p);
static __inline void restoreHwRegisters(RficLoopbackAntennas_t *inAntInfo_p);
static void setTxIqRfLoop(RficLoopbackAntennas_t *inAntInfo_p, IN TxIq_elements_t* inElements_p);
static void setTxIqAdaptiveGain(Antenna_e rxAnt, TxIq_elements_t* inElements_p);
static int32 calcIMR(CorrRes_t* Result);
static RetVal_e calibrateAllBinsInRange(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,TxIqCalibRangeDetails_t *inCalibratedRangeDetails,int8 inBinsSign,bool inMeasureTime,OUT TxIq_elements_t* outElements_p);
static RetVal_e calibrateAndVerifySingleBin(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,int16 inBin, uint8 inBinIndex, bool inMeasureTime, OUT TxIq_elements_t* outElements_p);
static void calcMinCoefficientValues(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,OUT TxIq_elements_t* outElements_p);
static void measureEnergy(uint8 rxAnt, int16 inBin,int32 *outEnergy, OUT TxIq_elements_t* outElements_p);
static void measureEnergy1(uint8 rxAnt, int16 inBin, TxIq_elements_t* outElements_p, int32 *outEnergy_p);
static void findMinimalIndex(int32 *inEnergyValues, IN TxIq_elements_t* inElements_p, int16 *outMinIndex);
static void findPolynomParameters(int32 inMatA[][TX_IQ_PFIT_MAX_PNTS] ,int32 *inEnergyValues,int32 *outPolynom);
static RetVal_e verifyCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,bool inMeasureTime,OUT TxIq_elements_t* outElements_p);
static RetVal_e verifyRfLoopCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,bool inMeasureTime,OUT TxIq_elements_t* outElements_p);
static RetVal_e fixFailedBins(uint8 inRfCalibAnt,uint16 *inCalibratedBins,int8 inRangeSign,OUT TxIq_elements_t* outElements_p);
static void calcInterpolatedCoefficients(uint8 inRfCalibAnt,uint16 *inCalibratedBins,uint8 inBinIndex,uint8 inNumOfCalibratedBins,int8 inBinSign,bool inAfterFix,bool inMeasureTime,OUT TxIq_elements_t* outElements_p);
static void TxIQClbr_StoreResults(void);
static uint16* getCalibrateBinsArray(TxIq_elements_t* outElements_p);
static uint8 getNumCalibBinsSinglePolarity(TxIq_elements_t* outElements_p);
static uint8 getHighestTransmissionBin(void);
static void resetTxIQCoefficientsTable(Antenna_e ant);
static void IQTD_calibrateSingleAntenna(uint8 ant, TxIq_elements_t* outElements_p);
static void IQTD_calcFMatrix(uint8 ant, int8 bin, int32 F[][2]);
static uint32 IQTD_complexAbs(int16 real, int16 image);
static void interpolateOutOfBandCoefficients(Antenna_e RfCalibAnt, int8 firstBin, int8 lastBin, int8 inBinsSign, uint8 highestTransmissionBin );


/******************************************************************************/
/***						Macros								            ***/
/******************************************************************************/

//Find index in [antenna][bin] arrays (a,b,binStatus,energyOffset)
#define TX_IQ_RESULTS_INDEX(binSign,binIndex,elements_p) \
	binSign>0 ? (binIndex):((getNumCalibBinsSinglePolarity(elements_p))+(binIndex))

//linear interpolation: find coeff_k when m<k<l
#define TX_IQ_INTERPOLATE(m,k,l,coeff_m,coeff_l) (((k-m)*coeff_l+(l-k)*coeff_m)/(l-m))

/******************************************************************************/
/***						Static Function Definitions						***/
/******************************************************************************/
//************************************
// Method:    run
// FullName:  run
// Access:    public static 
// Returns:   RetVal_e
//				Offline - RET_VAL_SUCCESS/RET_VAL_FAIL
//				Online - RET_VAL_SUCCESS(completed all fragmented steps)/RET_VAL_FRAGMENTED(completed step successfully)/RET_VAL_FAIL
// Qualifier:
// Parameter: IN ClbrType_e inCalibType
// Parameter: OUT TxIq_elements_t * outElements_p
//************************************
static RetVal_e run(IN ClbrType_e inCalibType, OUT TxIq_elements_t* outElements_p )
{
    TxIqCalibRangeDetails_t calibratedRangeDetails;
    uint32 StartCalTime;
	RetVal_e rc = RET_VAL_SUCCESS; 
    uint8 RfCalibratedAntenna;
	bool isLastIteration;
    CalDataState_e calStoreState;    
    StartCalTime = TIME_STAMP(START_TIME,0);

	useSsbLoop = RficDriver_isSsbLoop();
	ILOG2_D("use ssb loop %d",useSsbLoop);

    fillCalibrationDetails(inCalibType, outElements_p, &calibratedRangeDetails);
    
    for (RfCalibratedAntenna = 0; RfCalibratedAntenna < MAX_NUM_OF_ANTENNAS; RfCalibratedAntenna++)
    { 
        /* Perform the calibration only for the antennas that are on in antMask - 
		   single antenna in ONLINE, all Tx enabled antennas in OFFLINE
		 */
        if (ANTENNA_MASK_IS_BIT_ON(calibratedRangeDetails.antMask, RfCalibratedAntenna))
        {
			//set calibrated antenna to CAL_STATUS_IN_PROCESS
			//offline
			if(inCalibType == CLBR_TYPE_OFFLINE)
			{
				outElements_p->results.status[RfCalibratedAntenna] = CAL_STATUS_IN_PROCESS;
			}
			//online first antenna iteration - positive range bin 0
			else if((calibratedRangeDetails.rangeType == POSITIVE_RANGE) && (calibratedRangeDetails.firstCalibBinIndex == 0))
			{
				outElements_p->results.status[RfCalibratedAntenna] = CAL_STATUS_IN_PROCESS;
			}
            if(calibrateSingleAntenna(inCalibType, RfCalibratedAntenna,&calibratedRangeDetails,outElements_p) != RET_VAL_SUCCESS)
				rc = RET_VAL_FAIL;
			if (HDK_getChannelWidth() != BANDWIDTH_TWENTY)
			{
				IQTD_calibrateSingleAntenna(RfCalibratedAntenna, outElements_p);
			}

		}
    }
    
    if(inCalibType == CLBR_TYPE_ONLINE)
    {
		fillNextOnlineIterationDetails(outElements_p, &isLastIteration);
 
		//no matter if antenna SUCCESS/FAIL we want to continue to next antenna in online
		if(!isLastIteration) rc=RET_VAL_FRAGMENTED;
    }
    else
    {
        /* At the end of the offline calibration init online calibration to its initial state */
        initAntennaOnlineIterationDetails(TRUE, outElements_p);
    }

    outElements_p->results.lastRunTime = TIME_STAMP(END_TIME,StartCalTime);

 	ILOG2_DD("Tx IQ: Finished Calibration. Ret Code = %d, Run Time = %d", rc, outElements_p->results.lastRunTime);

        /* Store the calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if(calStoreState == CAL_DATA_STATE_STORE)
	{
		TxIQClbr_StoreResults();
	}    
	return (rc);
}	

/********************************************************************************

fillCalibrationDetails

Description:
------------
    Fill the ongoing calibration details    

Input:
-----
  inCalibType   

Output:
-------
	outElements_p
    outCalibRangeDetails_p		- Details of the current calibration iteration.
    
********************************************************************************/
static void fillCalibrationDetails(IN ClbrType_e inCalibType, OUT TxIq_elements_t* outElements_p, TxIqCalibRangeDetails_t *outCalibRangeDetails_p)
{
	uint8 antMask, numOfAnts;
	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &antMask);
	
	outCalibRangeDetails_p->calibratedBins = getCalibrateBinsArray(outElements_p);
	outCalibRangeDetails_p->totalNumCalibratedBins = getNumCalibBinsSinglePolarity(outElements_p);
    
    if(inCalibType == CLBR_TYPE_ONLINE)
    {
         /* Calibrating all 3 antennas is too long to execute during on line calibration, therefore we divide the calibration, 
            each iteration we calibrate numBinsOnlineIteration of a single antenna 
			(or less if this is the last iteration of positive/negative bin - calibrate remaining bins)
		  */
        outCalibRangeDetails_p->antMask  = ANTENNA_BIT(txIqOnlineDetails.calibAnt);
       
        outCalibRangeDetails_p->rangeType = txIqOnlineDetails.rangeType;
        outCalibRangeDetails_p->firstCalibBinIndex = txIqOnlineDetails.firstCalibBinIndex;
        // Update the total number of calibrated bins for the online calibration
        outCalibRangeDetails_p->totalNumCalibratedBins  = outElements_p->params.numBinsOnlineIteration;  
    }
    else //offline
    {
        outCalibRangeDetails_p->antMask = antMask;
        outCalibRangeDetails_p->rangeType = ALL_BIN_RANGES;
        outCalibRangeDetails_p->firstCalibBinIndex = 0;        
    }
}
/********************************************************************************

initAntennaOnlineIterationDetails

Description:
------------
   Initialize antenna online calibration to its starting state (first bin index and positive range)
    
Input:
-----
    inFirstTime - TRUE if this is the first iteration after offline

********************************************************************************/
static void initAntennaOnlineIterationDetails(bool inFirstTime, OUT TxIq_elements_t* outElements_p)
{
	txIqOnlineDetails.positiveRangeFix =FALSE;
	txIqOnlineDetails.negativeRangeFix =FALSE;
    txIqOnlineDetails.firstCalibBinIndex = 0;
    txIqOnlineDetails.rangeType = POSITIVE_RANGE;

	//Find last enabled antenna 
	//don't assume that enabled antennas can't change during online without moving back to offline
	FindTxEnabledAntenna(0,PREVIOUS_ENABLED_ANTENNA_SEARCH,&(txIqOnlineDetails.lastEnabledAntenna));

#if LM_CALIBRATION_LOGS_ON
	ILOG2_D("Tx IQ initAntennaOnlineIterationDetails: last antenna = %d",txIqOnlineDetails.lastEnabledAntenna);
#endif

	//First iteration after offline
	if(inFirstTime){

		//Start with first enabled antenna
		FindTxEnabledAntenna(MAX_NUM_OF_ANTENNAS-1,NEXT_ENABLED_ANTENNA_SEARCH,&(txIqOnlineDetails.calibAnt));
	}
	//next online antenna
	else{
		//Continue with next enabled antenna
		FindTxEnabledAntenna(txIqOnlineDetails.calibAnt,NEXT_ENABLED_ANTENNA_SEARCH,&(txIqOnlineDetails.calibAnt));
	}
}

/********************************************************************************

fillNextOnlineIterationDetails

Description:
------------
    Fill all the required data (range type,antenna,first bin) for the next online iteration:
    The calibration sequence is for each Tx enabled antenna, numBinsOnlineIteration(2) at a time.
	First all positive bins are calibrated and then all negative bins
    
Input:
-----
    inElements_p

Return:
-------
    outIsLastIteration_p - Flag that indicates end of on line calibration

********************************************************************************/
static void fillNextOnlineIterationDetails(IN TxIq_elements_t* inElements_p, bool *outIsLastIteration_p)
{  
	//initialize - in case we continue with current antenna
	*outIsLastIteration_p = FALSE;

	//calibrate all positive bins and then all negative bins - numBinsOnlineIteration at a time
	if(txIqOnlineDetails.rangeType == POSITIVE_RANGE){
		//continue with POSITIVE_RANGE if non-calibrated positive bins remain
		if(txIqOnlineDetails.firstCalibBinIndex + inElements_p->params.numBinsOnlineIteration < getNumCalibBinsSinglePolarity(inElements_p))
			txIqOnlineDetails.firstCalibBinIndex += inElements_p->params.numBinsOnlineIteration;
		//continue to NEGATIVE_RANGE
		else{
			txIqOnlineDetails.rangeType = NEGATIVE_RANGE;
			txIqOnlineDetails.firstCalibBinIndex = 0;
		}
	}
	//NEGATIVE_RANGE
	else{
		//continue with NEGATIVE_RANGE if non-calibrated negative bins remain
		if(txIqOnlineDetails.firstCalibBinIndex + inElements_p->params.numBinsOnlineIteration < getNumCalibBinsSinglePolarity(inElements_p))
			txIqOnlineDetails.firstCalibBinIndex += inElements_p->params.numBinsOnlineIteration;
		//antenna done
		else{		

			/*If online calibration done for last enabled antenna,  
				Set flag that indicates on line last iteration so we can move on to the next calibration*/
			*outIsLastIteration_p = (txIqOnlineDetails.calibAnt==txIqOnlineDetails.lastEnabledAntenna);
  
			initAntennaOnlineIterationDetails(FALSE, inElements_p);
		}
	}

    return;
}

/********************************************************************************

calibrateSingleAntenna

Description:
------------
    Find and set TxIQ mismatch compensation equalizer for chosen antenna (& chosen bins)

Input:
-----
	inCalibType
    inRfCalibAnt		- The calibrated RF antenna. 
    inCalibratedRangeDetails_p	- All needed calibration details(first bin to calibrate,number of bins to calibrate,negative or positive bins)

Output:
-------
	outElements_p

Return:
-------
    (RET_VAL_SUCCESS)	- Successful 
    (RET_VAL_FAIL)		- Failed - there are bins where calibration/verification/fix failed
********************************************************************************/
static RetVal_e calibrateSingleAntenna(IN ClbrType_e inCalibType, uint8 inRfCalibAnt, TxIqCalibRangeDetails_t *inCalibratedRangeDetails_p, OUT TxIq_elements_t* outElements_p)
{
    RetVal_e RetFailOk = RET_VAL_SUCCESS;
    uint32 StartTime;
	bool measureTime = FALSE, fixOffline = FALSE;
	RficLoopbackAntennas_t	antInfo;	//Loopback antennas
	
	antInfo.RfCalibAnt = (Antenna_e)inRfCalibAnt;
	
	
	if (inCalibType == CLBR_TYPE_OFFLINE)
	{
		//Reset TxIQ table
		resetTxIQCoefficientsTable(antInfo.RfCalibAnt);
	}	
	
	RficDriver_GetDigitalLoopBackAnt(&antInfo); 
	ILOG2_DDD("TXIQ calibrateSingleAntenna:RF Ant %d,LoopBackRxAnt %d, LoopBackTxAnt %d",antInfo.RfCalibAnt,antInfo.RxLoopbackAnt,antInfo.TxLoopbackAnt);

 	if (useSsbLoop)
	{
		initHwRegisters(&antInfo, outElements_p);
	}
	else
	{
		setTxIqRfLoop(&antInfo, outElements_p);
	}
	
    if(inCalibratedRangeDetails_p->rangeType & POSITIVE_RANGE)
    {
		//do time measurements only for positive range and index 0
		measureTime = (inCalibratedRangeDetails_p->firstCalibBinIndex == 0);

		RetFailOk = calibrateAllBinsInRange(inCalibType,&antInfo,inCalibratedRangeDetails_p,1,measureTime,outElements_p);

		/* completed all positive bins of antenna at current iteration */
		if(((inCalibType == CLBR_TYPE_OFFLINE) && (RetFailOk != RET_VAL_SUCCESS))||			//offline + there are failed bins
			((inCalibType == CLBR_TYPE_ONLINE) && txIqOnlineDetails.positiveRangeFix))		//online + last positive bins iteration (unknown if there are failed bins in a certain iteration)
		{
			//galia - ONLINE: check if fix+interpolation can be done (as is implemented now) in the last step 
			//(together with last bins calibration iteration) or we need additional steps
			//galia - measure time for worst case fix: simulation that only one bin succeeded and all others need to be fixed
				
			//Fix all failed bins in the positive range and interpolate in between fixed bins
			if(fixFailedBins(inRfCalibAnt,inCalibratedRangeDetails_p->calibratedBins,1,outElements_p) != RET_VAL_SUCCESS)
				RetFailOk = RET_VAL_FAIL;
		}
    }
    
    if(inCalibratedRangeDetails_p->rangeType & NEGATIVE_RANGE)
    {
        if(calibrateAllBinsInRange(inCalibType,&antInfo,inCalibratedRangeDetails_p,-1,FALSE,outElements_p) != RET_VAL_SUCCESS){
			RetFailOk = RET_VAL_FAIL;
			if(inCalibType == CLBR_TYPE_OFFLINE)
				fixOffline = TRUE;			
		}
		/* All negative bins of antenna were calibrated */
		if(fixOffline ||																	//offline + there are failed bins
			((inCalibType == CLBR_TYPE_ONLINE) && txIqOnlineDetails.negativeRangeFix))		//online + last negative bins iteration (unknown if there are failed bins in a certain iteration)
		{
			//Fix all failed bins in the negative range and interpolate in between fixed bins
			if(fixFailedBins(inRfCalibAnt,inCalibratedRangeDetails_p->calibratedBins,-1,outElements_p) != RET_VAL_SUCCESS)
				RetFailOk = RET_VAL_FAIL;
		}
    }

	//galia - measure time to decide if restore will not be done per antenna but after all antennas complete
	//measure time only if positive range and first index = 0
	if(measureTime)
		StartTime = TIME_STAMP(START_TIME,0);
	
    restoreHwRegisters(&antInfo);

	if(measureTime){
		outElements_p->results.restoreHwRegistersRunTime = TIME_STAMP(END_TIME,StartTime);

 		ILOG2_D("Tx IQ calibrateSingleAntenna(): Restore Hw Registers Run Time = %d", outElements_p->results.restoreHwRegistersRunTime);
	}

    return (RetFailOk);
}

/********************************************************************************

initHwRegisters

Description:
------------
    Set RF,AFE & Phy(Digital) relevant registers for TXIQ calibration(Using loop back in the RFIC)  

Input:
-----
    inAntInfo_p 
	inElements_p
    
********************************************************************************/
static __inline void initHwRegisters(RficLoopbackAntennas_t *inAntInfo_p, IN TxIq_elements_t* inElements_p)
{
	int32 initStartTime= TIME_STAMP(START_TIME,0);
    RficPathTypeIndex_e rficGoertzelBranch;
    bool iqSwap = FALSE;
	uint8 bw = HDK_getChannelWidth();
	Goertzel_e goertzelNum;
	UNUSED_PARAM(initStartTime);
	
	RficDriver_SetTxGain(inAntInfo_p->RfCalibAnt, 3);

	// Set RF loop back 
	
    RficDriver_SetLoopback(inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt, RFIC_LOOPBACK_TYPE_TX_AMDET);	
	
#if defined (RFIC_WRX300)
	/*set Rx gains */	
	SetRxGains((AntennaBitmaps_e)(ANTENNA_BIT(inAntInfo_p->RfCalibAnt)),HDK_LNA_INDEX_HIGH_GAIN,0,0,inElements_p->params.pgc3); 
#endif

	// Initialize PHY
    PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_TX_IQ_MISMATCH,inAntInfo_p->RfCalibAnt); 
	AFE_SetLoopBack(inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt, AFE_MODE_LOOPBACK, 0);
	PhyCalDrv_SetToneGenParams(0, 0, (1 << inAntInfo_p->RfCalibAnt), 0);
	
	inElements_p->params.toneAmplitude = TX_IQ_TONE_AMPLITUDE - bw * TX_IQ_TONE_AMPLITUDE_BW_OFFSET;
	
    PhyCalDrv_ToneGenInit(inElements_p->params.toneAmplitude);

	/* Init the Goertzel */
	goertzelNum = (Goertzel_e)(inAntInfo_p->RxLoopbackAnt & 2); // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	
    RficDriver_GetAmDetBranch(HDK_getBand(), &rficGoertzelBranch);
	PhyCalDrv_GoertzelInit(goertzelNum, inAntInfo_p->RxLoopbackAnt,rficGoertzelBranch, iqSwap,inElements_p->params.grtzlNumCycles); 
	ILOG2_DD("amplitude: %d digital gain: %d",inElements_p->params.toneAmplitude, PhyCalDrv_GetDigitalGain(0,0));
	ILOG2_D("Tx IQ: Init Done. Duration = %d", TIME_STAMP(END_TIME, initStartTime));
}

/********************************************************************************

restoreHwRegisters

Description:
------------
    Restore all RF,AFE & Phy(Digital) registers to their values 
    prior to the calibration execution

Input:
-----
    None
    
********************************************************************************/
static __inline void restoreHwRegisters(RficLoopbackAntennas_t *inAntInfo_p)
{
	int32 restoreStartTime= TIME_STAMP(START_TIME,0);

	UNUSED_PARAM(restoreStartTime);
		
    // Restore PHY 
    PhyCalDrv_ToneGenRestore();
    PhyCalDrv_RestoreOperationalTx();
	AFE_RestoreFromLoopBack();
    // Restore RF
	RFIC_RestoreFromLoopBack(inAntInfo_p->RfCalibAnt);

	ILOG2_D("Tx IQ: Restore Done. Duration = %d", TIME_STAMP(END_TIME, restoreStartTime));
}

static void setTxIqRfLoop(RficLoopbackAntennas_t *inAntInfo_p, IN TxIq_elements_t* inElements_p)
{
	int8 iir_mu;
	uint8 bw = HDK_getChannelWidth();
	AntennaBitmaps_e txAntMask = (AntennaBitmaps_e)(1 << inAntInfo_p->RfCalibAnt);
	AntennaBitmaps_e rxAntMask = (AntennaBitmaps_e)(1 << inAntInfo_p->RxLoopbackAnt);

	RficDriver_ActivateRxAntennas(rxAntMask);
	RficDriver_ActivateTxAntennas(txAntMask);
	
	/*set Tx gains */	
	RficDriver_SetTxGain(inAntInfo_p->RfCalibAnt, 3);//ForHarel
//	RficDriver_SetTxGain(inAntInfo_p->RfCalibAnt, 0);
	RficDriver_setRffeMode(T_STATE_MODE, txAntMask);

	/*set Rx gains */	
	RficDriver_AbstractSetRxGains(rxAntMask, TX_IQ_LNA, TX_IQ_PGC1, TX_IQ_PGC2);
	RficDriver_setRffeMode(T_STATE_MODE, rxAntMask);
	
	// Initialize PHY
    PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_TX_IQ_MISMATCH,inAntInfo_p->RfCalibAnt); 
	AFE_SetLoopBack(inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt, AFE_MODE_LOOPBACK, 0);
	
	// Initialize Tone generator
	PhyCalDrv_SetToneGenParams(0, 0, txAntMask, 0);
	
	inElements_p->params.toneAmplitude = 10;
	if (HDK_getChannelWidth() == BANDWIDTH_EIGHTY)
	{
		inElements_p->params.toneAmplitude = 9;
	}
    PhyCalDrv_ToneGenInit(inElements_p->params.toneAmplitude);
			
	// Initialize Accumulator
	iir_mu = TX_IQ_IIR_MU - bw;

	PhyCalDrv_CorrInit(CORR_MODE, rxAntMask, TX_IQ_NUM_SAMPLES, TX_IQ_ROUND, iir_mu);

}

static void setTxIqAdaptiveGain(Antenna_e rxAnt, TxIq_elements_t* inElements_p)
{
	CorrRes_t Result;
	int8 Delay_us;
	uint8 bw = HDK_getChannelWidth();
	uint32 accumNorm, accumDiff;
	uint8 coarseHigh, coarseLow, fineHigh, fineLow;
	int8 amplitude, newAmplitude;
	uint16 digitalGain, newDigitalGain;
	
	Delay_us = DELAY_AFTER_PHY_RFIC_CHANGE-(DELAY_CHANGE_PER_BW*bw);

	// Set default amplitude and digital gain
	PhyCalDev_SetToneGenAmplitude(inElements_p->params.toneAmplitude);
	PhyCalDrv_SetDigitalGain(0,0,DIGITAL_GAIN_0DB);

	// Accumulator measurement
	PhyCalDrv_CorrMeasure(rxAnt, &Result, Delay_us, PHY_CAL_DRV_IGNORE_IQ_SWAP);
	accumNorm = (Result.II / TX_IQ_NUM_SAMPLES ) << TX_IQ_ROUND;
		
	while((accumNorm < TX_IQ_ADAPTIVE_GAIN_ACCUM_LOW) || (accumNorm > TX_IQ_ADAPTIVE_GAIN_ACCUM_HIGH))
	{
		ILOG2_DD("setTxIqAdaptiveGain. acc result(II) = %d. accumNorm = %d", Result.II, accumNorm);
		
		/* Init */
		coarseHigh = 0;
		coarseLow = 0;
		fineHigh = 0;
		fineLow = 0;
		
		if (accumNorm < TX_IQ_ADAPTIVE_GAIN_ACCUM_LOW)
		{
			// Accumulator measurement is low - increase gain 
			
			accumDiff = (accumNorm << 8)/TX_IQ_ADAPTIVE_GAIN_ACCUM_LOW;
			
			if (accumDiff < 180) // 0.7 * 2^8
				coarseHigh = 1;
			else
				fineHigh = 1;
			
			ILOG2_D("Accumulator measurement is low - increase gain. course = %d", coarseHigh);
		}
		if (accumNorm > TX_IQ_ADAPTIVE_GAIN_ACCUM_HIGH)
		{
			// Accumulator measurement is high - decrease gain 
			
			accumDiff = (accumNorm << 8)/TX_IQ_ADAPTIVE_GAIN_ACCUM_HIGH;
			
			if (accumDiff > 384)// 1.5 * 2^8
				coarseLow = 1;
			else
				fineLow = 1;
			ILOG2_D("Accumulator measurement is high - decrease gain. course = %d", coarseLow);
		}
		
		/* Read current amplitude and digital gain */
		
		amplitude = PhyCalDrv_GetToneGenAmplitude();
		digitalGain = PhyCalDrv_GetDigitalGain(0,0);

		/* Stop conditions */
		if ((amplitude == TX_IQ_ADAPTIVE_GAIN_AMP_LOW) || (amplitude == TX_IQ_ADAPTIVE_GAIN_AMP_HIGH)
			|| (digitalGain < TX_IQ_ADAPTIVE_GAIN_DIG_GAIN_LOW) || (digitalGain > TX_IQ_ADAPTIVE_GAIN_DIG_GAIN_HIGH))
            break;

		/* Set new amplitude and digital gain */
		
		newAmplitude = amplitude + coarseLow - coarseHigh;
		newDigitalGain = digitalGain + 30 * (fineHigh - fineLow); // 30 is approximatley 1 dB change

		PhyCalDev_SetToneGenAmplitude(newAmplitude);
		PhyCalDrv_SetDigitalGain(0, 0, newDigitalGain);

		ILOG2_DDDD("old Amp: %d new Amp: %d old digital: %d new digital: %d", amplitude, newAmplitude, digitalGain, newDigitalGain);

		// Accumulator measurement
		PhyCalDrv_CorrMeasure(rxAnt, &Result, Delay_us, PHY_CAL_DRV_IGNORE_IQ_SWAP);
		accumNorm = (Result.II / TX_IQ_NUM_SAMPLES ) << TX_IQ_ROUND;	
	}
}

static int32 calcIMR(CorrRes_t* Result)
{	
	int32 phiSquared, IMR, a;
	uint8 res = IMR_RESOLUTION_BITS;
	int64 IMR_num,IMR_denom, iRes;
	
	if ((Result->QQ > (1 << I_Q_TRUNCATION_BITS)) && ((Result->II > (1 << I_Q_TRUNCATION_BITS))))
	{
		Result->QQ >>= I_Q_TRUNCATION_BITS;
		Result->II >>= I_Q_TRUNCATION_BITS;
	}
	iRes = (int64)Result->II << res;
	
	DEBUG_ASSERT(Result->QQ > 0);

	// a = sqrt(II/QQ)
	a = MATH_Sqrt(iRes / Result->QQ);

	// Phi^2 = IQ^2/(II*QQ)
	phiSquared = ((int64)(Result->IQ << (res - (2 * I_Q_TRUNCATION_BITS))) * Result->IQ) / (Result->II * Result->QQ);

	// IMR_num = (a-1)^2 +a*Phi^2
	IMR_num = (((int64)((a << (res >> 1))- (1 << res))) * ((int64)((a << (res >> 1))- (1 << res)))) + ((int64) (a << (res >> 1))) * phiSquared;
	
	// IMR_denom = (a+1)^2 -a*Phi^2
	IMR_denom = (((int64)(a + (1 << (res >> 1)))) * (a + (1 << (res >> 1)))) - ((int64)(a * (phiSquared >> res)));

	// IMR = IMR_num/IMR_denom
	IMR = IMR_num/IMR_denom;
		
	ILOG2_DDD("a: %d phiSquared: %d IMR: %d",a, phiSquared, IMR);

	return (IMR);
}

/********************************************************************************

calibrateAllBinsInRange

Description:
------------
   Calibrate all bins in the required range  

Input:
-----
	inCalibType
	inAntInfo_p
    inCalibratedRangeDetails    - Pointer to a struct that holds all needed calibration details(first bin to calibrate,number of bins to calibrate)
	inBinsSign					- negative or positive bins
	inMeasureTime				- do time measurements

Output:
-------
outElements_p

Return:
-------
	(RET_VAL_SUCCESS)	- Successful 
	(RET_VAL_FAIL)		- Failed - at least one bin failed (verification failed)
********************************************************************************/
static RetVal_e calibrateAllBinsInRange(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,TxIqCalibRangeDetails_t *inCalibratedRangeDetails,int8 inBinsSign,bool inMeasureTime,OUT TxIq_elements_t* outElements_p)
{
    RetVal_e RetFailOk = RET_VAL_SUCCESS;
    uint8 iterationNum;
	uint8 resultIndex;
	int16 binNum;
	uint8 binIndex = inCalibratedRangeDetails->firstCalibBinIndex;
    uint16 *calibratedBins = inCalibratedRangeDetails->calibratedBins;
	uint8 numArrayBins = getNumCalibBinsSinglePolarity(outElements_p);
	uint8 highestTransmissionBin = getHighestTransmissionBin();
	
	//positive/negative range completed at this iteration
	if (inCalibType == CLBR_TYPE_ONLINE)
	{
		if((binIndex + inCalibratedRangeDetails->totalNumCalibratedBins) >= numArrayBins)
		{
			if(inBinsSign > 0)
				txIqOnlineDetails.positiveRangeFix = TRUE;
			else
				txIqOnlineDetails.negativeRangeFix = TRUE;
		}
	}

    for (iterationNum = 0; 
		((iterationNum < inCalibratedRangeDetails->totalNumCalibratedBins) && (binIndex < numArrayBins)); 
		binIndex++,iterationNum++)
    {
		binNum = inBinsSign*calibratedBins[binIndex];

		//calibrate and verify
		if(calibrateAndVerifySingleBin(inCalibType,inAntInfo_p,binNum,binIndex,((iterationNum == 0) && inMeasureTime),outElements_p) != RET_VAL_SUCCESS)
			RetFailOk = RET_VAL_FAIL;
		//bin calibration succeeded - interpolate with previous bin if exists (and only if previous bin succeeded)
		else 
        {
            if(binIndex > 0)
            {
                calcInterpolatedCoefficients(inAntInfo_p->RfCalibAnt,calibratedBins,binIndex-1,2,inBinsSign,FALSE,((iterationNum == 1) && inMeasureTime),outElements_p); //interpolation of bin pair
            }
			else
			{
				resultIndex = TX_IQ_RESULTS_INDEX(inBinsSign, 0, outElements_p);
				for (binNum = 1; binNum < calibratedBins[0]; binNum ++)
				{
					PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->RfCalibAnt, (binNum * inBinsSign), outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultIndex], outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultIndex]);
				}
			}

            /* The last calibrated bins is not always the highest bin that is used for transmission:
                         (AR10) - We Calibrate Only 2 Bins and interpolated by them, results are not accurate on these bins (AR10)
                         (Wave300) - When calibrating  4 multiplier bins the calibration fails because of unknown reason (may be spur)  */
            if(binIndex == TX_IQ_MAX_CALIB_BINS_SINGLE_POLARITY-1)
            {   
				interpolateOutOfBandCoefficients(inAntInfo_p->RfCalibAnt,calibratedBins[0],calibratedBins[TX_IQ_MAX_CALIB_BINS_SINGLE_POLARITY-1], inBinsSign, highestTransmissionBin);
            }
		}
	}
    return (RetFailOk);
}
/********************************************************************************

calibrateAndVerifySingleBin

Description:
------------
    Find TxIQ mismatch compensation equalizer for single bin.    
	Verify calibration.

Input:
-----
	inCalibType
	inAntInfo_p
    inBin					- The calibrated bin value (positive or negative)
	inBinIndex				- Calibrated bin index (positive)
    inMeasureTime			- Measure time duration for single bin

	Output:
	-------
	outElements_p

Return:
-------
	(RET_VAL_SUCCESS)	- Successful 
	(RET_VAL_FAIL)		- Failed - verification failed
********************************************************************************/
int32 fin_ref_energy;
static RetVal_e calibrateAndVerifySingleBin(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,int16 inBin, uint8 inBinIndex, bool inMeasureTime, OUT TxIq_elements_t* outElements_p)
{
    RetVal_e RetFailOk;
    int16 a_min_orig,b_min_orig; 
	uint32 startTime;
	uint8 resultsIndex;
	int32 tmp_energy = 0;
	int64 fin_energy = 0;
	uint8 k;
	uint32 txAntenna;

	//measure time duration of single bin calibration only if first index is 0 positive
	if(inMeasureTime)
		startTime = TIME_STAMP(START_TIME,0);

 	ILOG2_D("Tx IQ calibrateAndVerifySingleBin:Bin %d",(int32)inBin);

	//Generate tone in the calibrated bin 
    PhyCalDrv_ToggleTxTone(inBin,TONE_GEN_TOGGLE_ON); 
	if (useSsbLoop)
	{
		// Decrease 12 dB in ref measurments to prevent Gohertzl saturation
		PhyCalDev_SetToneGenAmplitude(outElements_p->params.toneAmplitude + (TONE_AMPLITUDE_REFERENCE_SIGN * 2));
			
		PhyCalDrv_ToggleTxTone(TX_IQ_VERIFY_REFERENCE_BIN, TONE_GEN_TOGGLE_ON);
		for (k = 0; k < outElements_p->params.verifyNumMeasurements; k++)
		{
			measureEnergy1(inAntInfo_p->RxLoopbackAnt, TX_IQ_VERIFY_REFERENCE_BIN - inBin, outElements_p, &tmp_energy);
			fin_energy += tmp_energy;
		}
		fin_ref_energy=MATH_10Log10Res((uint32)(fin_energy/outElements_p->params.verifyNumMeasurements), TX_IQ_ENERGY_LOG_RESOLUTION);
		fin_ref_energy >>= TX_IQ_ENERGY_LOG_RESOLUTION;
		
		//Compensate for the two tone ref Goertzel measurments that decreases 11 dB for 1 amplitude level change
		fin_ref_energy += 20;
		ILOG2_DDD("Reference energy (bin %d - %d): %d", TX_IQ_VERIFY_REFERENCE_BIN, inBin, fin_ref_energy);

		PhyCalDrv_ToggleTxTone(TX_IQ_VERIFY_REFERENCE_BIN, TONE_GEN_TOGGLE_OFF);  // clear reference tone
		
		PhyCalDev_SetToneGenAmplitude(outElements_p->params.toneAmplitude);
		
		PhyCalDrv_ToggleTxTone(inBin,TONE_GEN_TOGGLE_ON); // generate calibrated tone
	}
	else
	{
		setTxIqAdaptiveGain(inAntInfo_p->RxLoopbackAnt, outElements_p);
		if (PhyCalDrv_GetToneGenAmplitude() < 8)
		{
			// Open PA and set adaptive gain from scratch
			txAntenna = inAntInfo_p->TxLoopbackAnt;
			RficDriver_setRffeMode(TX_MODE,(AntennaBitmaps_e)(1 << txAntenna));
			setTxIqAdaptiveGain(inAntInfo_p->RxLoopbackAnt, outElements_p);
		}
	}
	
	/* Save the values of the TxIQ coefficients of the digital Tx antenna used during the calibration sequence -
       done only if we run over values of a different antenna during the calibration algorithm (as a result of the loop back design in the RFIC - platform specific)
	   Save only if inLoopBackTxAnt != inRfCalibAnt */
	if(inAntInfo_p->RfCalibAnt != inAntInfo_p->TxLoopbackAnt)
		PHYCalDrv_GetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt, inBin, &a_min_orig, &b_min_orig);

    calcMinCoefficientValues(inCalibType,inAntInfo_p,inBin,inBinIndex,outElements_p);
	
	// Set the IQ coefficients we calculated for the calibrated RF antenna - 
	//Coefficients are set before verify (no matter if verify fails)
	resultsIndex = TX_IQ_RESULTS_INDEX(inBin,inBinIndex,outElements_p);

	PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->RfCalibAnt,inBin,outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex],outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]);
	if (useSsbLoop)
		RetFailOk = verifyCalibrationResults(inAntInfo_p,inBin,inBinIndex,inMeasureTime,outElements_p);
	else
		RetFailOk = verifyRfLoopCalibrationResults(inAntInfo_p,inBin,inBinIndex,inMeasureTime,outElements_p);
	
     // clear tone
    PhyCalDrv_ToggleTxTone(inBin,TONE_GEN_TOGGLE_OFF); 
    // Restore the saved IQ coefficients of the relevant digital antenna - if necessary
	if(inAntInfo_p->RfCalibAnt != inAntInfo_p->TxLoopbackAnt)
	    PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt, inBin, a_min_orig, b_min_orig); 

	//measure time duration of single bin calibration only for index 0 positive
	if(inMeasureTime)
	{
		outElements_p->results.singleBinCalibRunTime = TIME_STAMP(END_TIME,startTime);

 		ILOG2_D("Tx IQ calibrateAndVerifySingleBin(): Run Time = %d",outElements_p->results.singleBinCalibRunTime);
	}

    return (RetFailOk);
}

/********************************************************************************

calcMinCoefficientValues

Description:
------------
    Finds the minimum equalizer coefficients(a,b) for the relevant bin      

Input:
-----
    inCalibType
	inAntInfo_p
	inBin						- The calibrated bin value (positive or negative)
	inBinIndex					- Calibrated bin index (positive)

Output:
-------
    outElements_p -  Minimum coefficient values(a & b) calculated for the required bin and antenna.

********************************************************************************/
static void calcMinCoefficientValues(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,OUT TxIq_elements_t* outElements_p)
{
	CorrRes_t Result;
	uint8 bw = HDK_getChannelWidth();
	int8 Delay_us;
    int32 TxEnergy[TX_IQ_PFIT_MAX_PNTS];
    int16 a_value=0, b_value=0;
    uint8 i;
	uint8 resultsIndex;
	
	Delay_us = DELAY_AFTER_PHY_RFIC_CHANGE-(DELAY_CHANGE_PER_BW*bw);
	
	resultsIndex = TX_IQ_RESULTS_INDEX(inBin,inBinIndex,outElements_p);

    /* If online calibration is performed,use the last calibrated minimal value of A coefficient */
    if (inCalibType == CLBR_TYPE_ONLINE)
    { 
		//PHYCalDrv_GetTxIQCoefficients(inRfCalibAnt, inBin ,&a_value, &b_value);
		//Get internal parameter
		a_value = outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex]; 
    }
	/* Calculate minimum value for A coefficient */
		for(i=0;i<outElements_p->params.numPointsX;i++)
		{
			a_value=outElements_p->params.x[i];
			PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt,inBin,a_value,b_value);
			if (useSsbLoop)	
			{
				measureEnergy(inAntInfo_p->RxLoopbackAnt,inBin,&TxEnergy[i],outElements_p);
			}
			else
			{
				PhyCalDrv_CorrMeasure(inAntInfo_p->RxLoopbackAnt, &Result, Delay_us, PHY_CAL_DRV_IGNORE_IQ_SWAP);
				ILOG2_DD("tx ant: %d. rx ant: %d",inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt);
				ILOG2_DDD("II: %d QQ: %d IQ: %d",Result.II,Result.QQ, Result.IQ);
				TxEnergy[i] = calcIMR(&Result);
			}
		}
		ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): a energy x0=%d x1=%d x2=%d", TxEnergy[0],TxEnergy[1],TxEnergy[2]);
		ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): a energy x3=%d x4=%d x5=%d", TxEnergy[3],TxEnergy[4],TxEnergy[5]);
		ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): a energy x6=%d x7=%d x8=%d", TxEnergy[6],TxEnergy[7],TxEnergy[8]);

		findMinimalIndex(TxEnergy,outElements_p,&a_value);
		// save calculated minimum a coefficient
		outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex] = a_value;

    /* Calculate minimum value for B coefficient */
    for (i=0; i< outElements_p->params.numPointsX; i++) 
    {
        b_value=outElements_p->params.x[i];
        PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt,inBin,a_value,b_value);
		
        if (useSsbLoop)	
		{
			measureEnergy(inAntInfo_p->RxLoopbackAnt,inBin,&TxEnergy[i],outElements_p);
		}
		else
		{
			PhyCalDrv_CorrMeasure(inAntInfo_p->RxLoopbackAnt, &Result, Delay_us, PHY_CAL_DRV_IGNORE_IQ_SWAP);
			ILOG2_DD("tx ant: %d. rx ant: %d",inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt);
			ILOG2_DDD("II: %d QQ: %d IQ: %d",Result.II,Result.QQ, Result.IQ);
			TxEnergy[i] = calcIMR(&Result);
		}
    }
	ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): b energy x0=%d x1=%d x2=%d", TxEnergy[0],TxEnergy[1],TxEnergy[2]);
	ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): b energy x3=%d x4=%d x5=%d", TxEnergy[3],TxEnergy[4],TxEnergy[5]);
	ILOG2_DDD("Tx IQ calcMinimumCoeffcientsValue(): b energy x6=%d x7=%d x8=%d", TxEnergy[6],TxEnergy[7],TxEnergy[8]);

    findMinimalIndex(TxEnergy,outElements_p,&b_value);
    // save calculated minimum b coefficient
    outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]=b_value;

	//mark bin calibration success
	outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] = BIN_CALIB_SUCCESS;

	ILOG2_DD("Tx IQ calcMinCoeffcientValues(): a=%d, b=%d",outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex],outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]);

    return ;
}

/********************************************************************************

measureEnergy

Description:
------------
    Measure energy level of (a+jb) equalizer (using goertzel)    

Input:
-----
	inBin                 - The calibrated bin 

Output:
-------
    outTxEnergy_p - Energy (linear not log)
	outElements_p

********************************************************************************/
static void measureEnergy(uint8 rxAnt, int16 inBin,int32 *outTxEnergy_p, OUT TxIq_elements_t* outElements_p)
{
    GoertzelRes_t	Results[TOTAL_GOERTZELS];
	uint8 goertzelNum, goertzelMask;
	
	goertzelNum = rxAnt & 2; // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	goertzelMask = 1 << goertzelNum;
    
    // Restart tone generator
    PhyCalDrv_ToneGenRestart();
    
	//Goertzel does assert in case of failure
    PhyCalDrv_GoertzelMeasure(goertzelMask, Results,(inBin*2),DELAY_AFTER_PHY_TX_CHANGE);

	Results[goertzelNum].Res_I_Real = Results[goertzelNum].Res_I_Real >> TX_IQ_ENERGY_RESOLUTION;	
    Results[goertzelNum].Res_Q_Real = Results[goertzelNum].Res_Q_Real >> TX_IQ_ENERGY_RESOLUTION;	

    *outTxEnergy_p=Results[goertzelNum].Res_I_Real*Results[goertzelNum].Res_I_Real+Results[goertzelNum].Res_Q_Real*Results[goertzelNum].Res_Q_Real;
       
    return;
}

static void measureEnergy1(uint8 rxAnt, int16 inBin, TxIq_elements_t* outElements_p, int32 *outEnergy_p)
{
	GoertzelRes_t	Results[TOTAL_GOERTZELS];
	uint8 goertzelNum, goertzelMask;
	
	goertzelNum = rxAnt & 2; // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	goertzelMask = 1 << goertzelNum;

	PhyCalDrv_GoertzelMeasure(goertzelMask, Results,inBin,DELAY_AFTER_PHY_RFIC_CHANGE);

	Results[goertzelNum].Res_I_Real = Results[goertzelNum].Res_I_Real >> TX_IQ_ENERGY_RESOLUTION;	
    Results[goertzelNum].Res_Q_Real = Results[goertzelNum].Res_Q_Real >> TX_IQ_ENERGY_RESOLUTION;	

    *outEnergy_p = Results[goertzelNum].Res_I_Real*Results[goertzelNum].Res_I_Real+Results[goertzelNum].Res_Q_Real*Results[goertzelNum].Res_Q_Real;

	return ;
}


/********************************************************************************

findMinimalIndex

Description:
------------
    Find index of minimal energy assuming the energy function is parabolic using Least-Square Calculation.

Input:
-----
    inTxEnergies - Vector of energy values.
	inElements_p
      
Output:
-------
     outMinIndex_p - Index of minimal energy i.e. for (Ax^2+Bx+C) min is -B/(2*A).

********************************************************************************/
static void findMinimalIndex(int32 *inTxEnergies, IN TxIq_elements_t* inElements_p, int16 *outMinIndex_p)
{
    /*variables for rounding*/
    int32 a,b,c,d,e,f,h,pos;
    int32 polynom[2];
    
    findPolynomParameters(inElements_p->params.mat_A,inTxEnergies,polynom);
    
    /*round*/
    b=-polynom[1];
    a=2*polynom[0];
    c=b/a;
    d=b%a;
	if (b < 0)
	{
		d += a;
	}
    e=Abs(Abs(a)-Abs(d));
    f=(Abs(d)<e);
    if(c>0)
        pos=1;
    else
        pos=-1;
    
    h=c+pos*f;
	*outMinIndex_p = (int16)h;
    
	return;
}

/********************************************************************************

findPolynomParameters

Description:
------------
  Find parameters of 2nd degree polynom (reflect energy as a function of equalizer)
  using Least-Square Calculation.  

Input:
-----
    inMatA - Least-Square Matrix
    inTxEnergies

Output:
-------
	outPolynom - polynomial values (Ax^2+Bx+C) C is omitted (not needed):

********************************************************************************/
static void findPolynomParameters(int32 inMatA[][TX_IQ_PFIT_MAX_PNTS] ,int32 *inTxEnergies,int32 *outPolynom)
{
    int16	i,j;
    int16	max_i=2;
    int16	max_j=TX_IQ_PFIT_MAX_PNTS;
    uint64	temp=0;
    
    for(i=0;i<max_i;i++)
    {
        for(j=0;j<max_j;j++)
        {
            temp=temp+(uint64)inMatA[i][j]*(uint64)inTxEnergies[j];
        }
        
        outPolynom[i]=(int32)(temp>>16);
        temp=0;
	}
}

/********************************************************************************
 
verifyCalibrationResults:

Description:
------------
    

Input:
-----
	inAntInfo_p
	inBin					- The calibrated bin value (positive or negative)
	inBinIndex				- Calibrated bin index (positive)
	inMeasureTime			- Measure time of single bin verify

Output:
-------
    outElements_p

Return:
-------
	RET_VAL_SUCCESS/RET_VAL_FAIL
********************************************************************************/
static RetVal_e verifyCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,bool inMeasureTime,OUT TxIq_elements_t* outElements_p)
{
    int16 fin_log_energy;
	int64 fin_energy;
    int32 fin_energy_tmp;
	RetVal_e RetFailOk;
    uint8 i;
	uint32 startTime;
	uint8 resultsIndex;

	//measure time duration of verify for a single bin only for index 0 positive
	if(inMeasureTime)
		startTime = TIME_STAMP(START_TIME,0);

	resultsIndex = TX_IQ_RESULTS_INDEX(inBin,inBinIndex,outElements_p);

	//verify only bins that were calibrated successfully 
	DEBUG_ASSERT(outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] == BIN_CALIB_SUCCESS);

	//Set digital antenna
    PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt,inBin,outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex],outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]);

    /* Measure the IQ mismatch energy after the calibration*/
    for(i=0,fin_energy=0; i<outElements_p->params.verifyNumMeasurements; i++)
    {
        measureEnergy(inAntInfo_p->RxLoopbackAnt,inBin,&fin_energy_tmp,outElements_p);
             
        fin_energy+=fin_energy_tmp;	
    }

	// NOTE: the 64 bit number is casted to 32 bit under the assumption of low value of verifyNumMeasurements (<=4)
	//MATH_10Log10 returns (10log10)<<4
	fin_log_energy = MATH_10Log10Res((uint32)(fin_energy/outElements_p->params.verifyNumMeasurements), TX_IQ_ENERGY_LOG_RESOLUTION);
	fin_log_energy >>= TX_IQ_ENERGY_LOG_RESOLUTION;

	outElements_p->results.energyOffset[inAntInfo_p->RfCalibAnt][resultsIndex] = fin_log_energy;
    
 	ILOG2_D("Tx IQ VerifyCalibrationResults(): Energy(log) = %d",fin_log_energy);

    // Check if IQ mismatch is above the maximum allowed TH. 
    if(Abs(fin_ref_energy - fin_log_energy) < TX_IQ_MAX_VERIFY_ENERGY)
    {
		//mark bin verify failure
		outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] = BIN_VERIFY_FAIL;       
		TxIQClbr_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_FAIL);
		RetFailOk = RET_VAL_FAIL;

 		ILOG2_DD("Tx IQ verifyCalibrationResults():Energy exceeds max TH = %d status = %d",outElements_p->params.maxVerifyEnergy,outElements_p->results.status[inAntInfo_p->RfCalibAnt]);
    }
    else
    {
		//mark bin calibration success
		outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] = BIN_CALIB_SUCCESS;
        TxIQClbr_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_PASS);
		RetFailOk = RET_VAL_SUCCESS;
    }

	//measure time duration of verify for a single bin only for TX_IQ_FIRST_CALIBRATED_INDEX(0) and positive
	if(inMeasureTime){
		outElements_p->results.singleBinVerifyRunTime = TIME_STAMP(END_TIME,startTime);

 		ILOG2_D("Tx IQ verifyCalibrationResults(): Single bin verify Run Time = %d",outElements_p->results.singleBinVerifyRunTime);
	}

    return (RetFailOk);
}

static RetVal_e verifyRfLoopCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p,int16 inBin,uint8 inBinIndex,bool inMeasureTime,OUT TxIq_elements_t* outElements_p)
{
	CorrRes_t Result;
	uint8 bw = HDK_getChannelWidth();
	int8 Delay_us;
	RetVal_e RetFailOk;
	uint32 startTime;
	uint8 resultsIndex;
	int32 IMR,IMRdB;
	
	Delay_us = DELAY_AFTER_PHY_RFIC_CHANGE-(DELAY_CHANGE_PER_BW*bw);
	
	//measure time duration of verify for a single bin only for index 0 positive
	if(inMeasureTime)
		startTime = TIME_STAMP(START_TIME,0);

	resultsIndex = TX_IQ_RESULTS_INDEX(inBin,inBinIndex,outElements_p);

	//verify only bins that were calibrated successfully 
	DEBUG_ASSERT(outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] == BIN_CALIB_SUCCESS);

	//Set digital antenna
    PhyCalDrv_SetTxIQCoefficients(inAntInfo_p->TxLoopbackAnt,inBin,outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex],outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]);
	ILOG2_DD("Verify. a = %d b = %d", outElements_p->results.a[inAntInfo_p->RfCalibAnt][resultsIndex],outElements_p->results.b[inAntInfo_p->RfCalibAnt][resultsIndex]); 
    /* Measure the IQ mismatch energy after the calibration*/
	
    PhyCalDrv_CorrMeasure(inAntInfo_p->RxLoopbackAnt, &Result, Delay_us, PHY_CAL_DRV_IGNORE_IQ_SWAP);
	ILOG2_DD("Verify. tx ant: %d, rx ant: %d. ",inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt);
	ILOG2_DDD("II: %d QQ: %d IQ: %d",Result.II,Result.QQ, Result.IQ);
	IMR = calcIMR(&Result);
	
	IMRdB = MATH_10Log10Res((uint32)IMR, TX_IQ_ENERGY_LOG_RESOLUTION);
	IMRdB -= (IMR_RESOLUTION_dB << TX_IQ_ENERGY_LOG_RESOLUTION);
	IMRdB >>= TX_IQ_ENERGY_LOG_RESOLUTION;
	
	ILOG2_D("Verify. IMR(dB) = %d", IMRdB); 

    // Check if IQ mismatch is above the maximum allowed TH. 
    if (IMRdB > -50)
    {
		//mark bin verify failure
		outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] = BIN_VERIFY_FAIL;       
		TxIQClbr_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_FAIL);
		RetFailOk = RET_VAL_FAIL;

 		ILOG2_DD("Tx IQ verifyCalibrationResults():Energy exceeds max TH = %d status = %d",outElements_p->params.maxVerifyEnergy,outElements_p->results.status[inAntInfo_p->RfCalibAnt]);
    }
    else
    {
		//mark bin calibration success
		outElements_p->results.binStatus[inAntInfo_p->RfCalibAnt][resultsIndex] = BIN_CALIB_SUCCESS;
        TxIQClbr_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_PASS);
		RetFailOk = RET_VAL_SUCCESS;
    }

	//measure time duration of verify for a single bin only for TX_IQ_FIRST_CALIBRATED_INDEX(0) and positive
	if(inMeasureTime){
		outElements_p->results.singleBinVerifyRunTime = TIME_STAMP(END_TIME,startTime);

 		ILOG2_D("Tx IQ verifyCalibrationResults(): Single bin verify Run Time = %d",outElements_p->results.singleBinVerifyRunTime);
	}

    return (RetFailOk);
}

/********************************************************************************

fixFailedBins

Description:
------------
	For positive/negative failed bins (because verification failed) - 
	try to fix calibration coefficients by interpolating nearest successful bins.

	a. If the failed coefficient is in between 2 successful bins - 
		use linear interpolation:
			For m < k < l (m,l - successful bins k - failed bin)
			a_k = ((k-m)/(l-m))*a_l + ((l-k)/(l-m))*a_m	
			b_k = ((k-m)/(l-m))*b_l + ((l-k)/(l-m))*b_m	
	b. Otherwise, if the failed coefficient is an edge coefficient 
		(has successful coefficients only in one of its sides) -
		use the same coefficients as the nearest successful bin.
	c. If there are no successful coefficients at either side - then fix failed (occurs when there are no successful bins in the whole range)
		(function return RET_VAL_FAIL)

!!!!!!!!!!!!!!!!!!!!!!galia!!!!!!!!!!!!!!
Note: It should be tested that the fixed interpolated coefficients indeed improve traffic performance 
	vs.	the original calibrated coefficients for which verification failed
!!!!!!!!!!!!!!!!!!!!!!galia!!!!!!!!!!!!!!

Input:
-----
    inRfCalibAnt   - The calibrated RF antenna.
	inCalibratedBins        - Pointer to bin array (index 0) 
    inRangeSign             - Range sign 1 - Positive , (-1) - Negative

Output
------
	outElements_p

Return
------
	RET_VAL_SUCCESS - fixed all failed bins
	RET_VAL_FAIL - failed to fix all bins (fix fails only if there are no successful bins in the whole range)

********************************************************************************/
static RetVal_e fixFailedBins(uint8 inRfCalibAnt,uint16 *inCalibratedBins,int8 inRangeSign,OUT TxIq_elements_t* outElements_p)
{
    int8 idx,resultsIdx,firstResultsIdx,results_i,m_idx,l_idx;
	int8 m,k,l;
	int16 a_l,a_m,b_l,b_m,a_k,b_k;
	bool m_found,l_found,fixed=FALSE;
	uint32 startTime;
	uint8 failed_counter = 0, fixed_counter = 0;	
	RetVal_e ret = RET_VAL_SUCCESS;

	//measure time for entire function 
	startTime = TIME_STAMP(START_TIME,0);


	/* Search failed bins */
    for (idx=0, firstResultsIdx = TX_IQ_RESULTS_INDEX(inRangeSign,idx,outElements_p), resultsIdx = firstResultsIdx; 
		idx < getNumCalibBinsSinglePolarity(outElements_p); 
		idx++, resultsIdx++)
    {      
		//found failed bin
		if(outElements_p->results.binStatus[inRfCalibAnt][resultsIdx] != BIN_CALIB_SUCCESS)
		{
			failed_counter++;	//counter of failed bins

			//Get failed bin k
			k = inRangeSign*inCalibratedBins[idx]; 

			//search to left for successful bin
			for(results_i=resultsIdx-1, m_found=FALSE; 
				results_i>=firstResultsIdx ; 
				results_i--)
			{
				//found successful left index m
				if(outElements_p->results.binStatus[inRfCalibAnt][results_i] == BIN_CALIB_SUCCESS){
					m_found=TRUE;
					m_idx = results_i-firstResultsIdx;
					//Get bin
					m = inRangeSign*inCalibratedBins[m_idx]; 
					//Get bin coefficients
					a_m = outElements_p->results.a[inRfCalibAnt][results_i]; 
					b_m = outElements_p->results.b[inRfCalibAnt][results_i]; 
					break;
				}
			}

			//search to right for successful bin
			for(results_i=resultsIdx+1, l_found=FALSE; 
				results_i<firstResultsIdx+getNumCalibBinsSinglePolarity(outElements_p) ; 
				results_i++)
			{
				//found successful right index l
				if(outElements_p->results.binStatus[inRfCalibAnt][results_i] == BIN_CALIB_SUCCESS){
					l_found=TRUE;
					l_idx = results_i-firstResultsIdx;
					//Get bin
					l = inRangeSign*inCalibratedBins[l_idx]; 
					//Get bin coefficients
					a_l = outElements_p->results.a[inRfCalibAnt][results_i]; 
					b_l = outElements_p->results.b[inRfCalibAnt][results_i]; 
					break;
				}

			}

			/* interpolate between successful bins */
			//initialize
			fixed = FALSE;
			//found left and right successful bins - use interpolated coefficients
			if(m_found && l_found){
				a_k = TX_IQ_INTERPOLATE(m,k,l,a_m,a_l);
				b_k = TX_IQ_INTERPOLATE(m,k,l,b_m,b_l);
				fixed = TRUE;
			}				
			//found left successful bin - use coefficients of left bin
			else if(m_found){
				a_k = a_m;
				b_k = b_m;
				fixed = TRUE;
			}
			//found right successful bin - - use coefficients of right bin
			else if(l_found){
				a_k = a_l;
				b_k = b_l;
				fixed = TRUE;
			}
			//can't fix - no successful bins at both sides - the whole range failed !!!
			//remain with failed verified result (BIN_VERIFY_FAIL)
			else{
				ret = RET_VAL_FAIL;
 				ILOG2_V("Tx IQ fixFailedBins(): Can't fix bins - calibration of all bins in range failed !!!");

				break;			//all bins in range are failed and therefore no point attempting to fix other bins
			}

			//fix succeeded
			if(fixed){
				fixed_counter++;	//counter of fixed bins

				PhyCalDrv_SetTxIQCoefficients(inRfCalibAnt,k,a_k,b_k);
				// save fixed a,b coefficients
				outElements_p->results.a[inRfCalibAnt][resultsIdx] = a_k;
				outElements_p->results.b[inRfCalibAnt][resultsIdx] = b_k;
 				ILOG2_DDD("Tx IQ fixFailedBins():Minimal Coefficient Values: bin=%d a=%d, b=%d",k,a_k,b_k);

				//mark fix success
				outElements_p->results.binStatus[inRfCalibAnt][resultsIdx] = BIN_FIX_SUCCESS;
			}
		}
	}

 	ILOG2_DD("Tx IQ fixFailedBins(): Failed bins %d, Fixed bins %d",failed_counter,fixed_counter);

	//there are failed bins - 
	//even if all fix attempts failed (fixed_counter = 0) try to interpolate those bins with status BIN_VERIFY_FAIL using the calibrated (failed verified) values
	if(failed_counter>0)
		//re-interpolate the whole range with fixed/non-verified bin coefficient values (only fixed/non-verified bins will be interpolated)
		calcInterpolatedCoefficients(inRfCalibAnt,inCalibratedBins,0,getNumCalibBinsSinglePolarity(outElements_p),inRangeSign,TRUE,FALSE,outElements_p); 

	//reset
	if(inRangeSign == 1)
		txIqOnlineDetails.positiveRangeFix = FALSE;
	else
		txIqOnlineDetails.negativeRangeFix = FALSE;

	//measure time duration for entire function
	outElements_p->results.fixBinsRunTime = TIME_STAMP(END_TIME,startTime);

 	ILOG2_DD("Tx IQ fixFailedBins(): Range sign %d, Run Time = %d",inRangeSign,outElements_p->results.fixBinsRunTime);

	return(ret);
}

/********************************************************************************

calcInterpolatedCoefficients

Description:
------------
	Do linear interpolation in between all relevant bin pairs.

    Calculate the coefficients of all relevant bins using linear interpolation
    For m < k < l (m,l - measured bins k - required bin)
	a_k = ((k-m)/(l-m))*a_l + ((l-k)/(l-m))*a_m	
	b_k = ((k-m)/(l-m))*b_l + ((l-k)/(l-m))*b_m	

	This function is called in two situations:
	1.	Before bins calibration fix attempt - 
		in which case only bin pairs with both binStatus equal BIN_CALIB_SUCCESS are interpolated.
	2.	After bins calibration fix attempt -
		in which case only bin pairs where at least one of the bins has some attempt for a reasonable value are interpolated:
		a. either fixed successfully (BIN_FIX_SUCCESS) or
		b. calibration succeeded, verify failed and fix failed (BIN_VERIFY_FAIL).
		
Input:
-----
    inRfCalibAnt   - The calibrated RF antenna.
    inCalibratedBins        - Pointer to bin array (index 0) 
	inBinIndex				- first bin index to interpolate
    inNumOfCalibratedBins   - number of calibrated bins to interpolate
    inBinSign               - Calibrated bins sign 1 - Positive , (-1) - Negative
	inAfterFix				- TRUE if after calibration bin fix attempt
	inMeasureTime			- Measure time on first bin pair (indexes 0,1 positive)

Output
------
	outElements_p

********************************************************************************/
static void calcInterpolatedCoefficients(uint8 inRfCalibAnt,uint16 *inCalibratedBins,uint8 inBinIndex,uint8 inNumOfCalibratedBins,int8 inBinSign,bool inAfterFix,bool inMeasureTime,OUT TxIq_elements_t* outElements_p)
{
    int8 index;
	int8 m,k,l;
	int16 a_l,a_m,b_l,b_m,a_k,b_k;
	uint32 startTime;
	uint8 resultsIndex;
	bool interpolate;

    /* Update all other TxIQ coefficients using linear interpolation */
    for (index=inBinIndex; index < (inBinIndex + inNumOfCalibratedBins - 1); index++)
    {      
		//measure time duration of single interpolation only for first bin pair with index 0
		if((index == 0) && inMeasureTime)
			startTime = TIME_STAMP(START_TIME,0);

		m = inBinSign*inCalibratedBins[index]; 
        l = inBinSign*inCalibratedBins[index+1];

		//True only if inCalibratedBins points to beginning of range (index 0) !!!
		resultsIndex = TX_IQ_RESULTS_INDEX(inBinSign,index,outElements_p);

		/* check if interpolation required */
		interpolate = FALSE;
		//before fix
		if(!inAfterFix){
			//interpolate if both bins calibration and verification succeeded
			if((outElements_p->results.binStatus[inRfCalibAnt][resultsIndex] == BIN_CALIB_SUCCESS) && 
				(outElements_p->results.binStatus[inRfCalibAnt][resultsIndex+1] == BIN_CALIB_SUCCESS))
				interpolate = TRUE;
		}
		//after fix
		else{
			//interpolate if one of the pair bins has status fix calibration succeeded or verify failed - which means it was not interpolated before fix attempt

			if(((outElements_p->results.binStatus[inRfCalibAnt][resultsIndex] == BIN_FIX_SUCCESS) || (outElements_p->results.binStatus[inRfCalibAnt][resultsIndex] == BIN_VERIFY_FAIL)) ||

				((outElements_p->results.binStatus[inRfCalibAnt][resultsIndex+1] == BIN_FIX_SUCCESS) || (outElements_p->results.binStatus[inRfCalibAnt][resultsIndex+1] == BIN_VERIFY_FAIL)))
			
				interpolate = TRUE;
		}

		if(interpolate)
		{	
			/* Get the measured coefficients*/

			//PHYCalDrv_GetTxIQCoefficients(inRfCalibAnt, m, &a_m, &b_m);	
			//Get internal parameters
			a_m = outElements_p->results.a[inRfCalibAnt][resultsIndex]; 
			b_m = outElements_p->results.b[inRfCalibAnt][resultsIndex]; 

			//PHYCalDrv_GetTxIQCoefficients(inRfCalibAnt, l, &a_l, &b_l);   
			//Get internal parameters
			a_l = outElements_p->results.a[inRfCalibAnt][resultsIndex+1]; 
			b_l = outElements_p->results.b[inRfCalibAnt][resultsIndex+1]; 

			/* Calculate coefficients using linear interpolation for all bins in the range m:l 
			   and update relevant phy registers accordingly*/
			for(k=m;k!=l;k=k+inBinSign)
			{    
				a_k = TX_IQ_INTERPOLATE(m,k,l,a_m,a_l);
				b_k = TX_IQ_INTERPOLATE(m,k,l,b_m,b_l);
           
				PhyCalDrv_SetTxIQCoefficients(inRfCalibAnt,k,a_k,b_k);
			}   

			//measure time duration of single interpolation only for first bin pair with index 0
			if((index == 0) && inMeasureTime){
				outElements_p->results.singleBinPairInterpolationRunTime = TIME_STAMP(END_TIME,startTime);

 				ILOG2_D("Tx IQ calcInterpolatedCoefficients(): Interpolation between 2 bins [0,1] Run Time = %d",outElements_p->results.singleBinPairInterpolationRunTime);
			}
		}
    }
}

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
RetVal_e TxIQClbr_Calibrate(IN ClbrType_e inCalibType)
{
	if (CoC_getNumActiveTxAnts() > 1)
	{
	return (run(inCalibType, &Calibration_params_otf.TxIqParams));
	}
	else
	{
		return RET_VAL_SUCCESS;
	}
}

void TxIQClbr_SetStatus( IN uint32 inAntenna, IN CalStatus_e inStatus )
{
	Calibration_params_otf.TxIqParams.results.status[inAntenna] = inStatus;
}

CalStatus_e TxIQClbr_GetStatus( IN uint32 inAntenna )
{
	return (Calibration_params_otf.TxIqParams.results.status[inAntenna]);
}

uint32 TxIQClbr_GetLastRunTime(void)
{
	return (Calibration_params_otf.TxIqParams.results.lastRunTime);
}
void TxIQClbr_LoadResults(void)
{
    Antenna_e uAntenna;
	uint8 uCBMode,numOfCalibratedBins,binIndex,resultsIndex;
	uint16 uAMin,uBMin;
	uint16 *CalibratedBins;
	uint8 i,j;
    uint8 highestTransmissionBin = getHighestTransmissionBin();
	int16 binNum;
    
	uCBMode = ShramClbrDataBufferStoreChanData.txIqData.cbMode;

	//verify that loaded results have the same CB/nCB mode as the running station


	ASSERT(uCBMode == HDK_getChannelWidth());
	
	//Note: store+load results - now that we support change of number and values of calibrated bins -
	//we assume that the loaded data was calibrated with the same bins array as used in the software:
	//CalibratedBinsCb/CalibratedBinsNcb

	CalibratedBins = getCalibrateBinsArray(&Calibration_params_otf.TxIqParams);
	numOfCalibratedBins = getNumCalibBinsSinglePolarity(&Calibration_params_otf.TxIqParams);

	for(uAntenna = ANTENNA_0; uAntenna < MAX_NUM_OF_ANTENNAS; uAntenna++)
	{
		//Reset TxIQ table
		resetTxIQCoefficientsTable(uAntenna);
		
		for(binIndex=0; binIndex < numOfCalibratedBins ;binIndex++)
		{
			// Load Positive bin value
			uAMin = ShramClbrDataBufferStoreChanData.txIqData.a[uAntenna][binIndex];
			uBMin = ShramClbrDataBufferStoreChanData.txIqData.b[uAntenna][binIndex];
			PhyCalDrv_SetTxIQCoefficients(uAntenna, CalibratedBins[binIndex], uAMin, uBMin);

			//Set internal parameters
			Calibration_params_otf.TxIqParams.results.a[uAntenna][binIndex] = uAMin;
			Calibration_params_otf.TxIqParams.results.b[uAntenna][binIndex] = uBMin;


			Calibration_params_otf.TxIqParams.results.binStatus[uAntenna][binIndex] = BIN_CALIB_SUCCESS;

			// Load Negative bin value 
			resultsIndex = TX_IQ_RESULTS_INDEX(-1,binIndex,&Calibration_params_otf.TxIqParams);
			uAMin = ShramClbrDataBufferStoreChanData.txIqData.a[uAntenna][resultsIndex];
			uBMin = ShramClbrDataBufferStoreChanData.txIqData.b[uAntenna][resultsIndex];
			PhyCalDrv_SetTxIQCoefficients( uAntenna, -CalibratedBins[binIndex], uAMin, uBMin );


			//Set internal parameters
			resultsIndex = TX_IQ_RESULTS_INDEX(-1,binIndex,&Calibration_params_otf.TxIqParams);
			Calibration_params_otf.TxIqParams.results.a[uAntenna][resultsIndex] = uAMin;
			Calibration_params_otf.TxIqParams.results.b[uAntenna][resultsIndex] = uBMin;

			Calibration_params_otf.TxIqParams.results.binStatus[uAntenna][resultsIndex] = BIN_CALIB_SUCCESS;

		}

		// Calculate interpolation values of positive bins 
		for (binNum = 1; binNum < CalibratedBins[0]; binNum ++)
		{
			PhyCalDrv_SetTxIQCoefficients(uAntenna, binNum, Calibration_params_otf.TxIqParams.results.a[uAntenna][0], Calibration_params_otf.TxIqParams.results.b[uAntenna][0]);
		}
		calcInterpolatedCoefficients(uAntenna,CalibratedBins,0,numOfCalibratedBins,1,FALSE,FALSE,&Calibration_params_otf.TxIqParams); 
		
		// Calculate interpolation values of negative bins 
		resultsIndex = TX_IQ_RESULTS_INDEX(-1, 0, &Calibration_params_otf.TxIqParams);
		for (binNum = 1; binNum < CalibratedBins[0]; binNum ++)
		{
			PhyCalDrv_SetTxIQCoefficients(uAntenna, (-1) * binNum, Calibration_params_otf.TxIqParams.results.a[uAntenna][resultsIndex], Calibration_params_otf.TxIqParams.results.b[uAntenna][resultsIndex]);
		}
		calcInterpolatedCoefficients(uAntenna,CalibratedBins,0,numOfCalibratedBins,-1,FALSE,FALSE,&Calibration_params_otf.TxIqParams); 


       /* The last calibrated bins is not always the highest bin that is used for transmission,
            We update the remaining bins coefficients with the value of the last bin calibrated*/
        
       /*interpolation out of band values of positive bins*/
       interpolateOutOfBandCoefficients(uAntenna,CalibratedBins[0],CalibratedBins[TX_IQ_MAX_CALIB_BINS_SINGLE_POLARITY-1],1, highestTransmissionBin);
       /*interpolation out of band values of negative bins*/
       interpolateOutOfBandCoefficients(uAntenna,CalibratedBins[0],CalibratedBins[TX_IQ_MAX_CALIB_BINS_SINGLE_POLARITY-1],-1, highestTransmissionBin);    

	   if (uCBMode != BANDWIDTH_TWENTY)
	   {
			// Load IQ TD parameters
	   
	   	   for (i = 0; i < 2; i++)
		   {
		   		for (j = 0; j < 2; j++)
				{
					Calibration_params_otf.TxIqParams.results.SSB_F_20M[uAntenna][i][j] = ShramClbrDataBufferStoreChanData.txIqData.SSB_F_20M[uAntenna][i][j]; 
					Calibration_params_otf.TxIqParams.results.SSB_F_40M[uAntenna][i][j] = ShramClbrDataBufferStoreChanData.txIqData.SSB_F_40M[uAntenna][i][j]; 
				}
	   	   }
	   
		   // Set IQTD matrix
		   PhyCalDrv_Set_IQTD_Matrix(&Calibration_params_otf.TxIqParams, uCBMode, uAntenna);
	   }
	   
       TxIQClbr_SetStatus(uAntenna, ShramClbrDataBufferStoreChanData.txIqData.calStatus[uAntenna]);
    }
}
//************************************
// Method:      TxIQ_StoreResults
// Purpose:    Stores TXIQ calibration data to a buffer
// Parameter: RetVal_e calRetVal - calibration return value
// Returns:     none
//************************************
static void TxIQClbr_StoreResults(void)
{
	Antenna_e           ant;
	bool 				CalState = TRUE;
	CalStatus_e 		antStatus;           
	uint8               numOfCalibratedBins, binIndex, resultsIndex;
	uint8 i, j;

	numOfCalibratedBins = getNumCalibBinsSinglePolarity(&Calibration_params_otf.TxIqParams);

	ShramClbrDataBufferStoreChanData.txIqData.cbMode = HDK_getChannelWidth();

#if LM_CALIBRATION_LOGS_ON
	ILOG2_D("TxIQClbr_StoreResults(): CB mode %d",ShramClbrDataBufferStoreChanData.txIqData.cbMode);
#endif

	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++ )
	{
        antStatus = TxIQClbr_GetStatus(ant);
		ShramClbrDataBufferStoreChanData.txIqData.calStatus[ant] = TxIQClbr_GetStatus(ant);
		if ((antStatus == CAL_STATUS_FAIL) || (antStatus == CAL_STATUS_IN_PROCESS))
		{
			CalState = FALSE;
		}
		for(binIndex = 0; binIndex < numOfCalibratedBins; binIndex++)
		{
			// Save positive bin value in calibration buffer
			//PHYCalDrv_GetTxIQCoefficients(uAntenna, CalibratedBins[binIndex], &iAMin, &iBMin );

			//Get internal parameters
			ShramClbrDataBufferStoreChanData.txIqData.a[ant][binIndex] = Calibration_params_otf.TxIqParams.results.a[ant][binIndex];
			ShramClbrDataBufferStoreChanData.txIqData.b[ant][binIndex] = Calibration_params_otf.TxIqParams.results.b[ant][binIndex];

#if LM_CALIBRATION_LOGS_ON
			ILOG2_DDD("TxIQClbr_StoreResults(): binIndex %d a=%d b=%d", binIndex, ShramClbrDataBufferStoreChanData.txIqData.a[ant][binIndex], ShramClbrDataBufferStoreChanData.txIqData.b[ant][binIndex]);
#endif

			// Save negative bin value in calibration buffer
			//PHYCalDrv_GetTxIQCoefficients(uAntenna,  -CalibratedBins[binIndex] , &iAMin, &iBMin );

			//Get internal parameters
			resultsIndex = TX_IQ_RESULTS_INDEX(-1,binIndex,&Calibration_params_otf.TxIqParams);
			ShramClbrDataBufferStoreChanData.txIqData.a[ant][resultsIndex] = Calibration_params_otf.TxIqParams.results.a[ant][resultsIndex];
			ShramClbrDataBufferStoreChanData.txIqData.b[ant][resultsIndex] = Calibration_params_otf.TxIqParams.results.b[ant][resultsIndex];

#if LM_CALIBRATION_LOGS_ON
			ILOG2_DDD("TxIQClbr_StoreResults(): binIndex %d a=%d b=%d", resultsIndex, ShramClbrDataBufferStoreChanData.txIqData.a[ant][resultsIndex], ShramClbrDataBufferStoreChanData.txIqData.b[ant][resultsIndex]);
#endif
		}
		
		// Store IQTD parameters
		
		for (i = 0; i < 2; i++)
		{
			for (j = 0; j < 2; j++)
			{
				ShramClbrDataBufferStoreChanData.txIqData.SSB_F_20M[ant][i][j] = Calibration_params_otf.TxIqParams.results.SSB_F_20M[ant][i][j];
				ShramClbrDataBufferStoreChanData.txIqData.SSB_F_40M[ant][i][j] = Calibration_params_otf.TxIqParams.results.SSB_F_40M[ant][i][j];
			}
		}
	}
	ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_TX_IQ_MISMATCH, CalState);    
}

uint32* TxIQClbr_GetElement(void)
{
	return	((uint32*)&Calibration_params_otf.TxIqParams);
}

static void interpolateOutOfBandCoefficients(Antenna_e RfCalibAnt, int8 firstCalBin, int8 lastCalBin, int8 inBinsSign, uint8 highestTransmissionBin )
{
  
	int16 a_m,b_m,a_l,b_l,IqCoeff_a,IqCoeff_b;
	int8 extendedBinNum;
      
	PHYCalDrv_GetTxIQCoefficients(RfCalibAnt ,inBinsSign*firstCalBin,&a_m,&b_m);
    PHYCalDrv_GetTxIQCoefficients(RfCalibAnt ,inBinsSign*lastCalBin,&a_l,&b_l);

    for(extendedBinNum = lastCalBin+1; extendedBinNum <= highestTransmissionBin; extendedBinNum++)
    {
		IqCoeff_a = TX_IQ_INTERPOLATE(firstCalBin,extendedBinNum,lastCalBin,a_m,a_l);
		IqCoeff_b = TX_IQ_INTERPOLATE(firstCalBin,extendedBinNum,lastCalBin,b_m,b_l);
        PhyCalDrv_SetTxIQCoefficients(RfCalibAnt ,inBinsSign*extendedBinNum,IqCoeff_a,IqCoeff_b);
    }
}

static uint16* getCalibrateBinsArray(TxIq_elements_t* outElements_p)
{
	uint8 chanWidth = HDK_getChannelWidth();
	uint16* calibratedBins;
	
    // Choose the required bins array depending on the spectrum mode (CB\nCB) 
    
    if (chanWidth == BANDWIDTH_TWENTY)
	{
		calibratedBins = outElements_p->params.CalibBinsNCb;
	}
	else if (chanWidth == BANDWIDTH_FOURTY)
	{
		calibratedBins = outElements_p->params.CalibBinsCb;
	}
	else
	{
		calibratedBins = outElements_p->params.CalibBinsCb80;
	}
	return (calibratedBins);
}

static uint8 getNumCalibBinsSinglePolarity(TxIq_elements_t* outElements_p)
{
	uint8 chanWidth = HDK_getChannelWidth();
	uint8 NumOfcalibratedBins;
	
    // Choose num of bins depending on the spectrum mode (CB\nCB) 
    
    if (chanWidth == BANDWIDTH_TWENTY)
	{
		NumOfcalibratedBins = outElements_p->params.numBinsNCbSinglePolarity;
	}
	else if (chanWidth == BANDWIDTH_FOURTY)
	{
		NumOfcalibratedBins = outElements_p->params.numBinsCbSinglePolarity;
	}
	else
	{
		NumOfcalibratedBins = outElements_p->params.numBinsCb80SinglePolarity;
	}
	return (NumOfcalibratedBins);
}

uint8 getHighestTransmissionBin()
{
	uint8 chanWidth = HDK_getChannelWidth();
	uint8 highestTransmissionBin;
	
	if (chanWidth == BANDWIDTH_TWENTY)
	{
		highestTransmissionBin = HIGHEST_TRANSMISSION_BIN_BW20;
	}
	else if (chanWidth == BANDWIDTH_FOURTY)
	{
		highestTransmissionBin = HIGHEST_TRANSMISSION_BIN_BW40;
	}
	else
	{
		highestTransmissionBin = HIGHEST_TRANSMISSION_BIN_BW80;
	}
	return (highestTransmissionBin);
}
static void resetTxIQCoefficientsTable(Antenna_e ant)
{
	uint8 binIndex;
	uint8 lastBin = 128;
	
	for (binIndex = 0; binIndex <= lastBin; binIndex++)
	{
		PhyCalDrv_SetTxIQCoefficients(ant, binIndex, 0,0);
		PhyCalDrv_SetTxIQCoefficients(ant, -binIndex, 0,0);
	}
}
static void IQTD_calibrateSingleAntenna(uint8 ant, TxIq_elements_t* outElements_p)
{
	uint32 chanWidth = HDK_getChannelWidth();
	uint8 ssbMode;
	int8 bin20 = 0, bin40 = 0;

	// Get ssb mode
	ssbMode = PhyDrv_GetTxPrimary();
	
	
	if (chanWidth == BANDWIDTH_EIGHTY)
	{
		switch (ssbMode)
		{
			case TX_PRIMARY_TYPE_USB_40_USB_20:
				bin20 = outElements_p->params.USB40_USB20_20M;
				bin40 = outElements_p->params.USB40_USB20_40M;
				break;
			case TX_PRIMARY_TYPE_USB_40_LSB_20:
				bin20 = outElements_p->params.USB40_LSB20_20M;
				bin40 = outElements_p->params.USB40_LSB20_40M;
				break;
			case TX_PRIMARY_TYPE_LSB_40_USB_20:
				bin20 = outElements_p->params.LSB40_USB20_20M;
				bin40 = outElements_p->params.LSB40_USB20_40M;
				break;
			case TX_PRIMARY_TYPE_LSB_40_LSB_20:
				bin20 = outElements_p->params.LSB40_LSB20_20M;
				bin40 = outElements_p->params.LSB40_LSB20_40M;
				break;
			default:
				DEBUG_ASSERT(0);
		}
		
		// Calculate F matrix for 20M
		IQTD_calcFMatrix(ant, bin20, outElements_p->results.SSB_F_20M[ant]);
			
		// Calculate F matrix for 40M
		IQTD_calcFMatrix(ant, bin40, outElements_p->results.SSB_F_40M[ant]);
	}
	else if (chanWidth == BANDWIDTH_FOURTY)
	{
		switch (ssbMode)
		{
			case TX_PRIMARY_TYPE_LSB_40_USB_20:
				bin20 = outElements_p->params.USB20;
				break;
			case TX_PRIMARY_TYPE_LSB_40_LSB_20:
				bin20 = outElements_p->params.LSB20;
				break;
			case TX_PRIMARY_TYPE_USB_40_USB_20:
			case TX_PRIMARY_TYPE_USB_40_LSB_20:
				return;
			default:
				DEBUG_ASSERT(0);
		}
		
		// Calculate F matrix for 20M
		IQTD_calcFMatrix(ant, bin20, outElements_p->results.SSB_F_20M[ant]);
	}

	// Write matrices to PHY registers
	PhyCalDrv_Set_IQTD_Matrix(outElements_p, chanWidth, ant);
}
static void IQTD_calcFMatrix(uint8 ant, int8 bin, int32 F[][2])
{
	uint8 i, j;
	int16 a_k, b_k, a, b_conj_real, b_conj_image, x, p_div_2, e;

	// Get IQ mismatch coefficients
	PHYCalDrv_GetTxIQCoefficients(ant, bin, &a_k, &b_k);

	ILOG2_DDDD("ant: %d bin: %d a_k: %d b_k: %d", ant, bin, a_k, b_k);
	b_conj_real = -1*(a_k);
	b_conj_image = b_k;

	a = (1 << TXIQ_COEF_FIXED_POINT_SHIFT);
	
	// x = image(a+b*)/real(a+b*)
	x = ((b_conj_image << TXIQ_ARCTAN_FIXED_POINT_SHIFT)/(a + b_conj_real));
	
	// p/2 = arctan(x)
	p_div_2 = MATH_ArctanRes(x, TXIQ_ARCTAN_FIXED_POINT_SHIFT);

	// e = |a + b*| - 1<<10
	e = IQTD_complexAbs(a + b_conj_real, b_conj_image) - (1 << TXIQ_COEF_FIXED_POINT_SHIFT);
	
	ILOG2_DDD("x: %d, p/2: %d, e: %d", x, p_div_2, e);
	
	// F calculation
	F[0][0] = ((1 << TXIQ_COEF_FIXED_POINT_SHIFT) - e) * MATH_Cos(p_div_2, TXIQ_ARCTAN_FIXED_POINT_SHIFT);
	F[0][1] = ((1 << TXIQ_COEF_FIXED_POINT_SHIFT) + e) * MATH_Sin(p_div_2);
	F[1][0] = ((1 << TXIQ_COEF_FIXED_POINT_SHIFT) - e) * MATH_Sin(p_div_2);
	F[1][1] = ((1 << TXIQ_COEF_FIXED_POINT_SHIFT) + e) * MATH_Cos(p_div_2, TXIQ_ARCTAN_FIXED_POINT_SHIFT);

	for (i = 0; i < 2; i++)
	{
		for (j = 0; j < 2; j++)
		{
			F[i][j] = DIVIDE_AND_ROUND((F[i][j]),(1 << (TXIQ_ARCTAN_FIXED_POINT_SHIFT + 2)));
		}
	}
	ILOG2_DDDD("F00: %d, F01: %d, F10: %d, F11: %d", F[0][0], F[0][1], F[1][0], F[1][1]);
}
static uint32 IQTD_complexAbs(int16 real, int16 image)
{
	uint32 a, b, e, sqrte, retVal;
	uint8 res = TXIQ_COEF_FIXED_POINT_SHIFT;
	
	e = ((image << res) * (image << res))/(real * real);

	a = (1 << (2*res)) - (((e >> 2)*((1<<(2*res)) - (e>>1))) >> (2*res));
	
	b = ((e >> 1) * a) >> (2 * res);

	sqrte = (1 << (2 * res)) + b;
	
	retVal = ((real * sqrte) >> (res << 1));
	
	ILOG2_DDD("e %d sqrte %d ret %d",e,sqrte,retVal);
	
	return retVal;
}



