/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "TxIQMismatchClbrHndlr.h"
#include "CalibrationsDefines.h"
#include "System_GlobalDefinitions.h"
#include "mt_cnfg.h"
#include "RegAccess_Api.h"
#include "mt_sysrst.h"
#include "MT_Math.h"
#include "Hdk_Api.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "RficDriver_API.h"
#include "Afe_API.h"
#include "CalibrationHandler.h"
#include "CalibrationHandlerUtils.h"
#include "ErrorHandler_Api.h"
#include "Shram_ClbrDataBuffer.h"
#include "LmHdk_API.h"
#include "loggerAPI.h"

#define LOG_LOCAL_GID GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 24

/******************************************************************************/
/***						Type Definitions								***/
/******************************************************************************/ 
typedef struct TxIQ_Coeffs
{
	int16 a[MAX_NUM_OF_ANTENNAS];
	int16 b[MAX_NUM_OF_ANTENNAS];
}TxIQ_Coeffs_t;

typedef struct TxIQ_Cal_Param
{
	int32 startindex;
	uint32 incremental;
	uint32 numberOfSteps;
}TxIQ_Cal_Param_t;

typedef struct TxIQ_Store_Param
{
	uint32 	txIQTxScale;
	int16 	txIQDigGain[MAX_NUM_OF_ANTENNAS];
	uint8 	txIQDifi2Gain[MAX_NUM_OF_ANTENNAS];
}TxIQ_Store_Param_t;


typedef enum
{
    BIN_INDEX,
    BIN_INDEX_START,
    BIN_INDEX_END,
	MAX_BIN_INDEX
}START_END_BIN_INDEX;

typedef enum
{
    CALIBRATE_START, 
	CALIBRATE_POSITIVE = CALIBRATE_START,
    CALIBRATE_NEGATIVE,
    CALIBRATE_END,
    CALIBRATE_ALL = CALIBRATE_END
}CALIBRATE_TYPE;

typedef struct TxIQParams
{
	uint32 lastRunTime;
	int16 binArray[CALIBRATE_END][MAX_BIN_INDEX];
	
	TxIQ_Coeffs_t savedCoeffs[CALIBRATE_END];
	
	CalStatus_e status[MAX_NUM_OF_ANTENNAS];	
	CALIBRATE_TYPE nextOnlineCalib;
}TxIQParams_t;


/******************************************************************************/
/***						static Variables								***/
/******************************************************************************/
TxIQParams_t txIqParams;
bool firstCalib = TRUE;

TxIQ_Cal_Param_t TxiqCalibStepArray[TX_CALIB_STEP_NUM] = {
		{TX_CALIB_STEP_1_START_OFFSET, TX_CALIB_STEP_1_JUMP_SIZE, TX_CALIB_STEP_1_JUMPS_NUM},
		{TX_CALIB_STEP_2_START_OFFSET, TX_CALIB_STEP_2_JUMP_SIZE, TX_CALIB_STEP_2_JUMPS_NUM},
		{TX_CALIB_STEP_3_START_OFFSET, TX_CALIB_STEP_3_JUMP_SIZE, TX_CALIB_STEP_3_JUMPS_NUM},
		{TX_CALIB_STEP_4_START_OFFSET, TX_CALIB_STEP_4_JUMP_SIZE, TX_CALIB_STEP_4_JUMPS_NUM} };


/******************************************************************************/
/***					Static Function Declarations						***/
/******************************************************************************/ 
static RetVal_e run(IN ClbrType_e inCalibType);

// 			--	save before setup & setup calibration --					  //
static void configureTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam);
static void configureRficForTxIQCalibration(uint8 antMask);
static void configurePhyForTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam);
static void setTxIQParams(uint8 antMask);
static void saveCurrentTxIqCoeffs(uint8 antMask);
static void fillBinArray(void);

// 					--	restore after calibration  --						  //
static void restoreFromTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam);
static void restoreRficFromTxIQCalibration(uint8 antMask);
static void restorePhyFromTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam);
static void handleFailure(uint8 antMask);

// 								--  calibration  --							  //
static RetVal_e calibrateTxIQ(ClbrType_e inCalibType, uint8 antMask);
static void calibrateAllBinsInRange(uint8 antMask, CALIBRATE_TYPE calibrate);
static void calibrateBin(uint8 antMask, int16 bin, int16 binStartIndex, int16 binEndIndex, bool isFragment, TxIQ_Coeffs_t *srcCoeffs);
static void calcMinCoefficientValues(uint8 antMask, int16 bin, TxIQ_Coeffs_t* txiqCoeffs, bool isFragment);
static void calcBestTxIQCoeff(uint8 antMask, int16 bin, TxIQ_Coeffs_t* txiqCoeffs, int16* calculatedCoeff, TxIQ_Cal_Param_t stepDescriptor);
static void verifyTxIqCalibrationResults(uint8* failedAntMask);

/******************************************************************************/
/***						Static Function Definitions						***/
/******************************************************************************/
static RetVal_e run(IN ClbrType_e inCalibType)
{
	uint8 antMask, maxAnts;
	RetVal_e rc = RET_VAL_SUCCESS; 
	CalDataState_e calStoreState;
	TxIQ_Store_Param_t storeParam;
	uint32 startTime = TIME_STAMP(START_TIME,0);
	
#ifdef WRX600_BU_LOGS
	ILOG0_D("Run TxIQ calibration. CalibType %d", inCalibType);
#endif	
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	/* Configuration */
	configureTxIQCalibration(antMask, &storeParam);
	/* Calibrate */ 
	rc = calibrateTxIQ(inCalibType, antMask);
	/* Restore	*/
	restoreFromTxIQCalibration(antMask, &storeParam);
	/* Store calibration results in OFFLINE state*/
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if((CLBR_TYPE_OFFLINE == inCalibType) && (CAL_DATA_STATE_STORE == calStoreState))
	{
		ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_TX_IQ_MISMATCH, TRUE); 
	}
	
	txIqParams.lastRunTime = TIME_STAMP(END_TIME,startTime);
#ifdef WRX600_BU_LOGS
	ILOG0_D("TxIQ Calibration done. elapsed time = %d usec",txIqParams.lastRunTime);
#endif
	return (rc);
}	

static void configureTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam)
{
	/* Set TxIQ Database */
	setTxIQParams(antMask);
	/* PHY configuration */
	configurePhyForTxIQCalibration(antMask, storeParam);
	/* RF configuration */
	configureRficForTxIQCalibration(antMask);
}

static void configureRficForTxIQCalibration(uint8 antMask)
{	
		uint8 band = HDK_getBand();
				
#ifdef WRX600_BU_LOGS
		ILOG0_D("configureRficForTxIQCalibration antMask 0x%x", antMask);
#endif
		RficDriver_ActivateTxAntennas(antMask); //turn the bbic_tx off & then on again
	
		//configure rx gains to initial gain and lna, since it is configured to lowest pgc index
		RficDriver_SetRxGains(antMask, TX_INITIAL_LNA_INDEX, TX_INITIAL_PGC_GAIN_DB);
		RficDriver_SetPgcbwidth (antMask, TX_PGC_BW);
	
		RficDriver_DeactivateTxAntennas(antMask);
	
		// Configure SSB loop
		RficDriver_SetLoopback(antMask, RFIC_LOOPBACK_TYPE_TX_AMDET);
	
		if (band == BAND_5_2_GHZ)
		{
			RficDriver_SetTxGain(antMask, TX_INITIAL_TPC_INDEX, TX_PAD_GAIN_DB_5G);
		}
		else
		{
			RficDriver_SetTxGain(antMask, TX_INITIAL_TPC_INDEX, TX_PAD_GAIN_DB_2_4G);
		}
}


static void configurePhyForTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam)
{
		uint32 goertzelLength[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_LENGTH, TX_GRTZL_LENGTH, TX_GRTZL_LENGTH, TX_GRTZL_LENGTH};	
		uint8 goertzelCycles[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES};
		int16 goertzelBin[MAX_NUM_OF_ANTENNAS] = {TX_IQ_CALIB_BIN, TX_IQ_CALIB_BIN, TX_IQ_CALIB_BIN, TX_IQ_CALIB_BIN};
		uint8 difi2Gain[MAX_NUM_OF_ANTENNAS] = {TX_DIFI_2_GAIN, TX_DIFI_2_GAIN, TX_DIFI_2_GAIN, TX_DIFI_2_GAIN};
		int16 digGain[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN};
	
#ifdef WRX600_BU_LOGS
		ILOG0_D("configurePhyForTxLOCalibration antMask 0x%x", antMask);
#endif
		storeParam->txIQTxScale = PhyCalDrv_GetToneGenScale();
		PhyCalDrv_GetTxDigitalGain(antMask,storeParam->txIQDigGain);
	
		//reset rx td
		PhyCalDrv_resetRxtd(TRUE);
		PhyCalDrv_resetRxtd(FALSE);
		
		AFE_RxOn(antMask);
		AFE_TxOff(antMask);
		AFE_TxOn(antMask);
		AFE_AuxadcOff(antMask);
	
		/* Store current difi 2 gain, use RXDC_DIFI_2_GAIN for calibration */
		PhyCalDrv_getDifi2Gain(antMask, storeParam->txIQDifi2Gain);
		PhyCalDrv_setDifi2Gain(antMask, difi2Gain);
	
		PhyCalDrv_enableDifiGain(antMask, FALSE);
		PhyCalDrv_setDcCancellationMode(antMask, FALSE);
	
		PhyCalDrv_SetPgc2Gain(antMask, TX_INITIAL_TPC_INDEX);
		
		// Configure and start Tone Generator
		PhyCalDrv_EnableGclkBypass(TRUE);
		PhyCalDrv_RunToneGen(antMask, TX_IQ_CALIB_BIN, TX_GRTZL_TONE_SCALE, digGain);
	
		// Configure Goertzel
		PhyCalDrv_GoertzelConfig(antMask, goertzelLength, goertzelCycles, goertzelBin, goertzelBin);
}

static void restoreFromTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam)
{
	/* Restore failed antenas from last calibration */
	handleFailure(antMask);
	
	/* Restore RF configuration */
	restoreRficFromTxIQCalibration(antMask);

	/* Restore PHY configuration */
	restorePhyFromTxIQCalibration(antMask, storeParam);
}


static void handleFailure(uint8 antMask)
{
	uint8 ant, FailedAntMask = 0, calibIndex;

	/* Load calibrated bins */
    for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {		
		if ((antMask & (1 << ant)) && ( CAL_STATUS_FAIL == TxIQClbr_GetStatus(ant)))
		{
			FailedAntMask |= (1 << ant);
		}
    }

	if (0 != FailedAntMask)
	{
#ifdef WRX600_TXIQ_ONLINE_CALB_LOGS
		ILOG0_D("handleFailure restoring for FailedAntMask 0x%x", FailedAntMask);
#endif
	
		for (calibIndex = CALIBRATE_START ; calibIndex < CALIBRATE_END ; calibIndex++) 
		{
			PhyCalDrv_SetTxIQCoefficientsAllBins(FailedAntMask, 
									txIqParams.savedCoeffs[calibIndex].a, txIqParams.savedCoeffs[calibIndex].b, 
									txIqParams.binArray[calibIndex][BIN_INDEX_START], txIqParams.binArray[calibIndex][BIN_INDEX_END]);
		}
	}
}

static void restoreRficFromTxIQCalibration(uint8 antMask)
{
	// Restore SSB loop
	RFIC_RestoreFromLoopBack(antMask);
	
	//RficDriver_readTxbandFromRf(storedTxBand);

}

static void restorePhyFromTxIQCalibration(uint8 antMask, TxIQ_Store_Param_t* storeParam)
{
	PhyCalDrv_enableDifiGain(antMask, TRUE);
	PhyCalDrv_setDcCancellationMode(antMask, TRUE);
	// AFE loopback restore 
	AFE_RxOff(antMask);
	AFE_TxOff(antMask);

	// Clear and disable Tone Gen
	PhyCalDrv_EnableGclkBypass(FALSE);
	PhyCalDrv_ToneGenClearAndDisable(antMask);

	// Restore difi2 gain
	PhyCalDrv_setDifi2Gain(antMask, storeParam->txIQDifi2Gain);
	PhyCalDrv_SetToneGenScale(antMask, storeParam->txIQTxScale);
	PhyCalDrv_SetDigitalGain(antMask, storeParam->txIQDigGain);
}


static RetVal_e calibrateTxIQ(ClbrType_e inCalibType, uint8 antMask)
{
	uint8 failedAntMask = antMask;
	uint8 tryCounter = 0;
	
	uint8 maxTries = TXIQ_MAX_TRIES_OFFLINE;	//assume offline
	CALIBRATE_TYPE calibrate = CALIBRATE_ALL;
	RetVal_e rc = RET_VAL_SUCCESS;

	if (inCalibType == CLBR_TYPE_OFFLINE)
	{	
		/* Reset coefficients table in PHY RAM */
		PhyCalDrv_ClearTxIQCoefficients(antMask);
		txIqParams.nextOnlineCalib = CALIBRATE_START;
	}

	else if (inCalibType == CLBR_TYPE_ONLINE)
	{	
		/* setup for current fragment */
		rc = RET_VAL_FRAGMENTED;
		calibrate = txIqParams.nextOnlineCalib;
		maxTries = TXIQ_MAX_TRIES_ONLINE;

		/* setup for next fragment */
		txIqParams.nextOnlineCalib++; 
		
		/* we coverd all the fragments */
		if (txIqParams.nextOnlineCalib == CALIBRATE_END) 
		{
			txIqParams.nextOnlineCalib = CALIBRATE_START;
			rc = RET_VAL_SUCCESS;
		}
	}

	for(tryCounter = 0 ; (failedAntMask != 0) && (tryCounter < maxTries) ; tryCounter++)
	{
#ifdef WRX600_BU_LOGS
		ILOG0_DD("calibratedAntMask 0x%x, tryCounter %d",failedAntMask, tryCounter);
#endif
		calibrateAllBinsInRange(failedAntMask, calibrate);
		verifyTxIqCalibrationResults(&failedAntMask);
	}

	/* indicate antennas status */
	TxIQClbr_SetStatus(antMask,CAL_STATUS_PASS);
	TxIQClbr_SetStatus(failedAntMask,CAL_STATUS_FAIL);

	return (rc);
}

static void calibrateAllBinsInRange(uint8 antMask, CALIBRATE_TYPE calibrate)
{	
	bool isOnline = TRUE;
	CALIBRATE_TYPE calibIndex;

	/* offline - do all fragments */
	if (calibrate == CALIBRATE_ALL) 
	{
		isOnline = FALSE;

		for (calibIndex = CALIBRATE_START ; calibIndex < CALIBRATE_END ; calibIndex++) 
		{
			calibrateBin(antMask, txIqParams.binArray[calibIndex][BIN_INDEX],
								txIqParams.binArray[calibIndex][BIN_INDEX_START], 
								txIqParams.binArray[calibIndex][BIN_INDEX_END], isOnline, NULL);
		}
	}

	/* online - do one fragment */
	else
	{
		calibrateBin(antMask, txIqParams.binArray[calibrate][BIN_INDEX],
					txIqParams.binArray[calibrate][BIN_INDEX_START], 
					txIqParams.binArray[calibrate][BIN_INDEX_END], isOnline, 
					&txIqParams.savedCoeffs[calibrate]);
	}
}

static void calibrateBin(uint8 antMask, int16 bin, int16 binStartIndex, int16 binEndIndex, bool isFragment, TxIQ_Coeffs_t *srcCoeffs)
{	
	TxIQ_Coeffs_t txIqCoeffs = {0};
	int16 digGain[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN};
	uint8 ant;
	
	/* use A & B from PHY only for partial / online calibration */
	if (isFragment)
	{
		DEBUG_ASSERT( NULL != srcCoeffs );

		/* copy it for fine tuning */
		for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (antMask & (1 << ant))
			{
				txIqCoeffs.a[ant] = srcCoeffs->a[ant];
				txIqCoeffs.b[ant] = srcCoeffs->b[ant];
#ifdef WRX600_TXIQ_ONLINE_CALB_LOGS
				ILOG0_DD("calibrateBin read a:%d, b:%d",txIqCoeffs.a[ant], txIqCoeffs.b[ant]);
#endif	
			}
		}
	}
	
	/* Transmit tone and config Goertzel to the calibrated bin */
	PhyCalDrv_RunToneGen(antMask, bin, TX_GRTZL_TONE_SCALE, digGain);
	PhyCalDrv_GoertzelSetTone(antMask, bin << 1);	

	/* Calc best A and B */
	calcMinCoefficientValues(antMask, bin, &txIqCoeffs, isFragment);
	
#ifdef WRX600_BU_LOGS
	for(ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{
			ILOG0_DD("calibrateBin min a:%d, b:%d",txIqCoeffs.a[ant], txIqCoeffs.b[ant]);
		}
	}
#endif
	
	//work around use bin 4 for all
	PhyCalDrv_SetTxIQCoefficientsAllBinsFullCal(antMask,txIqCoeffs.a, txIqCoeffs.b,binStartIndex,binEndIndex);
}
static void calcMinCoefficientValues(uint8 antMask, int16 bin, TxIQ_Coeffs_t* txiqCoeffs, bool isFragment)
{
	uint8 step = 0;
	
	/* online calibration skips the first step */
	if (TRUE == isFragment)	
		step++;

	for( ; step < TX_CALIB_STEP_NUM ; step++)
	{
		/* Calculate minimum value for A coefficient */
		calcBestTxIQCoeff(antMask, bin, txiqCoeffs, txiqCoeffs->a, TxiqCalibStepArray[step]);
	    /* Calculate minimum value for B coefficient */
		calcBestTxIQCoeff(antMask, bin, txiqCoeffs, txiqCoeffs->b, TxiqCalibStepArray[step]);	 
	}
}

static void calcBestTxIQCoeff(uint8 antMask, int16 bin, TxIQ_Coeffs_t* txiqCoeffs, int16* calculatedCoeff, TxIQ_Cal_Param_t stepDescriptor)
{
    uint32 txEnergy[MAX_NUM_OF_ANTENNAS];
	int32 startIndexArray[MAX_NUM_OF_ANTENNAS];
	int32 MinimumIndex[MAX_NUM_OF_ANTENNAS];
	uint32 MinimumEnergy[MAX_NUM_OF_ANTENNAS];
    uint8 index, ant;	
	
#ifdef WRX600_TXIQ_ONLINE_CALB_LOGS
	int32 StartCalTime, EndCalTime;
	StartCalTime = TIME_STAMP(START_TIME,0);
#endif
	// init arrays
	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		startIndexArray[ant] = calculatedCoeff[ant] + stepDescriptor.startindex;
		MinimumIndex[ant] = 0;
		MinimumEnergy[ant] = (uint32)-1;
	}

	for(index = 0; index < stepDescriptor.numberOfSteps; index++)
	{
		for(ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (antMask & (1 << ant))
			{					
				calculatedCoeff[ant] = startIndexArray[ant] + (stepDescriptor.incremental)*index;
#ifdef WRX600_BU_LOGS
				ILOG0_DDD("txiqCoeffs->a[%d]:%d, b:%d",ant,txiqCoeffs->a[ant], txiqCoeffs->b[ant]);
#endif
			}
		}
		PhyCalDrv_SetTxIQCoefficients(antMask, bin, txiqCoeffs->a, txiqCoeffs->b);
		PhyCalDrv_GoertzelMeasureEnergy(antMask, txEnergy, TX_GRTZL_ENERGY_RESOLUTION, TX_GRTZL_TIMEOUT);

		for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (antMask & (1 << ant))
			{					
				if(txEnergy[ant] < MinimumEnergy[ant])
				{
					 MinimumEnergy[ant] = txEnergy[ant];
					 MinimumIndex[ant] = calculatedCoeff[ant];
#ifdef WRX600_BU_LOGS
				ILOG0_DDD("minimum choosen ant :%d, MinimumEnergy[ant]:%d,MinimumIndex[ant]:%d",ant,MinimumEnergy[ant], MinimumIndex[ant]);
#endif
				}
			}
		}
	}

	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{					
				calculatedCoeff[ant] = MinimumIndex[ant];
#ifdef WRX600_BU_LOGS				
				ILOG0_DDD("txiqCoeffs->a[%d]:%d, b:%d",ant,txiqCoeffs->a[ant], txiqCoeffs->b[ant]);
#endif
		}
	}
	
#ifdef WRX600_TXIQ_ONLINE_CALB_LOGS	
	EndCalTime = TIME_STAMP(END_TIME,StartCalTime);
	ILOG0_DDD("calcBestTxIQCoeff-> mask:%x  steps:%d, time: %d",antMask, stepDescriptor.numberOfSteps, EndCalTime);
#endif
}

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
RetVal_e TxIQClbr_Calibrate(IN ClbrType_e inCalibType)
{
	return run(inCalibType);
}

void TxIQClbr_SetStatus( IN uint32 inAntennaMask, IN CalStatus_e inStatus )
{
	uint8 ant;

	ILOG0_DD("TxIQClbr_SetStatus inStatus:%d antMask 0x%x", inStatus, inAntennaMask);
	
	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{		
		if (inAntennaMask & (1 << ant))
		{
			txIqParams.status[ant] = inStatus;
		}
	}
}

CalStatus_e TxIQClbr_GetStatus( IN uint32 inAntenna )
{
	return txIqParams.status[inAntenna];
}

uint32 TxIQClbr_GetLastRunTime(void)
{
	return txIqParams.lastRunTime;
}

static void setTxIQParams(uint8 antMask)
{
	fillBinArray();
#ifdef WRX600_BU_logs
	ILOG0_V("setTxIQParams");
	SLOG0(0, 0, TxIQParams_t, &txIqParams);
#endif

	saveCurrentTxIqCoeffs(antMask);
}
static void saveCurrentTxIqCoeffs(uint8 antMask)
{
	uint8 ant;
	CALIBRATE_TYPE calibIndex;

	/*if this is the first time in TXIQ callibration there is no point at saving the registers content */
	if (firstCalib) 
	{
		for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			for (calibIndex = CALIBRATE_START ; calibIndex < CALIBRATE_END ; calibIndex++) 
			{
				txIqParams.savedCoeffs[calibIndex].a[ant] = 0;
				txIqParams.savedCoeffs[calibIndex].b[ant] = 0;
			}
		}
		
		firstCalib = FALSE;
		txIqParams.nextOnlineCalib = CALIBRATE_START;
		return;
	}
	
	/* read A & B from PHY */
	for (calibIndex = CALIBRATE_START ; calibIndex < CALIBRATE_END ; calibIndex++) 
	{
		PhyCalDrv_GetTxIQCoefficients(antMask, txIqParams.binArray[calibIndex][BIN_INDEX], txIqParams.savedCoeffs[calibIndex].a, txIqParams.savedCoeffs[calibIndex].b);

#ifdef WRX600_TXIQ_ONLINE_CALB_LOGS
		ILOG0_D("saveCurrentTxIqCoeffs  calibration type :%d", calibIndex);
		ILOG0_DDDD("saveCurrentTxIqCoeffs read a[0]:%d, a[1]:%d, a[2]:%d, a[3]:%d",txIqParams.savedCoeffs[calibIndex].a[0], txIqParams.savedCoeffs[calibIndex].a[1], txIqParams.savedCoeffs[calibIndex].a[2], txIqParams.savedCoeffs[calibIndex].a[3]);
		ILOG0_DDDD("saveCurrentTxIqCoeffs read b[0]:%d, b[1]:%d, b[2]:%d, b[3]:%d",txIqParams.savedCoeffs[calibIndex].b[0], txIqParams.savedCoeffs[calibIndex].b[1], txIqParams.savedCoeffs[calibIndex].b[2], txIqParams.savedCoeffs[calibIndex].b[3]);
#endif
	}
}

static void fillBinArray()
{
	uint8 bw = HDK_getChannelWidth();

	/* for bins 0 till 255 */
	txIqParams.binArray[CALIBRATE_POSITIVE][BIN_INDEX_START] = TXIQ_CALIBRATED_BIN_START;
	txIqParams.binArray[CALIBRATE_POSITIVE][BIN_INDEX_END] = TXIQ_CALIBRATED_BIN_END;

	/* for bins -256 till -1 */
	txIqParams.binArray[CALIBRATE_NEGATIVE][BIN_INDEX_START] = -TXIQ_CALIBRATED_BIN_END;
	txIqParams.binArray[CALIBRATE_NEGATIVE][BIN_INDEX_END] = TXIQ_CALIBRATED_BIN_START;
	
	switch (bw)
	{
		case BANDWIDTH_TWENTY:		
			txIqParams.binArray[CALIBRATE_POSITIVE][BIN_INDEX] = TXIQ_20_CALIBRATED_BIN;
			txIqParams.binArray[CALIBRATE_NEGATIVE][BIN_INDEX] = -TXIQ_20_CALIBRATED_BIN;
			break;
		case BANDWIDTH_FOURTY:			
			txIqParams.binArray[CALIBRATE_POSITIVE][BIN_INDEX] = TXIQ_40_CALIBRATED_BIN;
			txIqParams.binArray[CALIBRATE_NEGATIVE][BIN_INDEX] = -TXIQ_40_CALIBRATED_BIN;
			break;
		case BANDWIDTH_EIGHTY :
		case BANDWIDTH_ONE_HUNDRED_SIXTY:
			txIqParams.binArray[CALIBRATE_POSITIVE][BIN_INDEX] = TXIQ_80_CALIBRATED_BIN;
			txIqParams.binArray[CALIBRATE_NEGATIVE][BIN_INDEX] = -TXIQ_80_CALIBRATED_BIN;
			break;
		default:
			FATAL("Un-supported BW");
			break;
	}
			
}

static void verifyTxIqCalibrationResults(uint8* failedAntMask)
{
	uint8 ant;
	uint32 minEnergy[MAX_NUM_OF_ANTENNAS];

	PhyCalDrv_SetTxToneGen();
	PhyCalDrv_GoertzelMeasureEnergy(*failedAntMask, minEnergy, TX_GRTZL_ENERGY_RESOLUTION, TX_GRTZL_TIMEOUT);
	
	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (*failedAntMask & (1 << ant))
		{
#ifdef WRX600_BU_LOGS
			ILOG0_DD("verifyTxIqCalibrationResults. ant %d energy %d", ant, minEnergy[ant]);
#endif
			if (minEnergy[ant] < TX_IQ_ENERGY_MAX_VALUE)
			{
#ifdef WRX600_BU_LOGS
				ILOG0_D("ant %d passed", ant);
#endif
				//pass calibration
				*failedAntMask &= (~(1 << ant));
			}
#ifdef WRX600_BU_LOGS			
			else
			{
				ILOG0_D("ant %d failed", ant);
			}
#endif
		}
	}
}


