#include "System_GlobalDefinitions.h"
#include "OSAL_Api.h"
#include <protocol.h>
#include <enet_pas.h>
#include <System_Configuration.h>
#include <PhyDriver_API.h>

#include "System_GlobalDefinitions.h"
#include "Afe_API.h"

#include <lmi.h>
#include "RegAccess_Api.h"
#include "CalibrationHandler.h"
#include "CalibrationHandlerDB.h"
#include "CalibrationHandlerUtils.h"
#include <PhyCalDriver_API.h>
#include "TxLOLeakageClbrHndlr.h"
#include "CalibrationsDefines.h"
#include <mt_sysrst.h>
#include "loggerAPI.h"
#include "RficDriver_API.h"
#include "RFICDefines.h"
#include "MT_Math.h"
#include "TpcClbrHndlr.h"
#include "ErrorHandler_Api.h"
#include "Hdk_Api.h"
#include "Shram_ClbrDataBuffer.h"
#include "LmHdk_API.h"
#include "CoC_Api.h"


#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 12




extern Calibration_params_otf_t Calibration_params_otf;	



/* During on line calibration we divide the calibration to avoid long calibration - one antenna and one TPC per step
   The following struct is used to manage on line calibration
*/
static TxLoOnlineDetails_t txLoOnlineDetails;

//galia static volatile int8 txLoRefAmplitude = 10;

/******************************************************************************/
/***					Private Function Declarations						***/
/******************************************************************************/
static RetVal_e run(IN ClbrType_e inCalibType, TxLO_elements_t*  outElements_p);
static void fillCalibrationDetails(IN ClbrType_e inCalibType, IN TxLO_elements_t* inElements_p, TxLoCalibRangeDetails_t *outCalibRangeDetails_p);
static void initAntennaOnlineIterationDetails(bool inFirstTime, TxLO_elements_t*  outElements_p);
static void fillNextOnlineIterationDetails(IN int8 inNextCalibTpc, TxLO_elements_t* inElements_p, bool *outIsLastIteration_p);
static RetVal_e calibrateSingleAntenna(IN ClbrType_e inCalibType, uint8 inRfCalibAnt, uint8 inCbMode, TxLoCalibRangeDetails_t *inCalibRangeDetails_p, TxLO_elements_t* outElements_p);
static __inline void initHwRegisters(RficLoopbackAntennas_t *inAntInfo_p, IN TxLO_elements_t* inElements_p, int8 *outDigitalDacI_p, int8 *outDigitalDacQ_p);
static __inline void restoreHwRegisters(RficLoopbackAntennas_t *inAntInfo_p,  int8 inDigitalDacI, int8 inDigitalDacQ, IN TxLO_elements_t* inElements_p);
static RetVal_e calibrateAllTpcsInRange(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,uint8 inCbMode,TxLoCalibRangeDetails_t *inCalibRangeDetails_p,OUT TxLO_elements_t* outElements_p);
static void initAntennaOfflineIterationDetails(uint8 inRfCalibAnt, TxLO_elements_t* outElements_p, TxLoCalibRangeDetails_t *outCalibRangeDetails_p);
static RetVal_e calibrateAndVerifySingleTpc(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,uint8 inCbMode,uint8 inTpcIndex, RficDriverTxLoTpcInfo_t *inTpcInfo_p, bool inMeasureTime, OUT TxLO_elements_t* outElements_p);
static void digitalDacUpdateRequired(int8 inDacIndex, TxLoDigitalDacInfo_t	*outDigitalDacInfo_p);
static void addDigitalDac(RficLoopbackAntennas_t *inAntInfo_p, bool inIsIqSwapped, TxLoDigitalDacInfo_t *outDigitalDacInfo_p, TxLO_elements_t* outElements_p);
static RetVal_e verifyCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p, uint8 inTpcIndex, uint8 inCbMode, int8 *inIndexes_p, RficDriverTxLoTpcInfo_t *inTpcInfo_p, OUT TxLO_elements_t* outElements_p);
static void updateTpcsToNextCalibTpc(uint8 inRfCalibAnt,uint8 inCurrCalibTpc,RficDriverTxLoTpcInfo_t *inTpcInfo_p,bool inUpdate,TxLO_elements_t* outElements_p,int8 *outNextCalibTpc_p);
static void updateTpcsToStart(uint8 inRfCalibAnt,uint8 inFirstCalibTpc,TxLO_elements_t* outElements_p);
static void TxLoLeakage_StoreResults(void);
static void TxLoLeakage_ResetOfflineStatus(void);
static void measureEnergy(uint8 inTpcIndex, int8 inValueI ,int8 inValueQ, int8 inBin, RficLoopbackAntennas_t *inAntInfo_p, TxLO_elements_t* outElements_p, int32 *outEnergy_p);

////////////////////////
// Methods
////////////////////////

//************************************
// Method:    run
// FullName:  run
// Access:    public static 
// Returns:   RetVal_e
// Qualifier:
// Parameter: IN ClbrType_e inCalibType
// Parameter: IN uint8 * inTpcMaxPowerIndexArray
// Parameter: TxLO_elements_t * outElements_p
//************************************
static RetVal_e run(IN ClbrType_e inCalibType, TxLO_elements_t*  outElements_p)
{
	RetVal_e RetFailOk=RET_VAL_SUCCESS;
	uint8 RfCalibAnt;
	int32 startTime = TIME_STAMP(START_TIME,0);
	TxLoCalibRangeDetails_t CalibRangeDetails;
	bool isLastIteration;
	CalDataState_e calStoreState;
	uint16 digGain;
	int8 toneAmp;
	
#if defined (RFIC_WRX300)	
	uint16 readVal[3];
	RficDriver_ReadRegister(REG_RFIC_L41_BC_EN_2G_INTPA, readVal);

	if ((readVal[0] & REG_RFIC_L41_BC_EN_2G_INTPA_MASK) == 0) //if external PA board
	{
		digGain = LO_EXT_PA_DIGITAL_GAIN;
		toneAmp = LO_EXT_PA_TONE_AMPLITUDE;
	}
	else
	{
		digGain = LO_DIGITAL_GAIN;
		toneAmp = LO_TONE_AMPLITUDE;
	}
	
#else
	digGain = LO_DIGITAL_GAIN;
	toneAmp = LO_TONE_AMPLITUDE;
#endif	

	outElements_p->params.toneAmplitude = toneAmp;

	PhyCalDrv_SetDigitalGain(0, 0, digGain);
	PhyCalDrv_SetDigitalGain(1, 0, digGain);

	TxLoLeakage_ResetOfflineStatus();
 
#if LM_CALIBRATION_LOGS_ON
	ILOG9_D("Tx Lo: Executing Calibration: CalibType(offline(1)/online(2)) = %d,",inCalibType); 
#endif    

	/*set Rx Gains*/
#if defined (RFIC_WRX300)
	SetRxGains((AntennaBitmaps_e)7, HDK_LNA_INDEX_HIGH_GAIN, 0, outElements_p->params.pgc2, outElements_p->params.pgc3); 
#endif
	//offline: calibrate all Tx enabled antennas and all TPCs. online: calibrate single antenna and numTpcsOnlineIteration TPCs

	fillCalibrationDetails(inCalibType, outElements_p, &CalibRangeDetails);

	for (RfCalibAnt = 0; RfCalibAnt < MAX_NUM_OF_ANTENNAS; RfCalibAnt++) 
	{
		if ( ANTENNA_MASK_IS_BIT_ON(CalibRangeDetails.antMask,RfCalibAnt) )
		{
		 	if (calibrateSingleAntenna(inCalibType, RfCalibAnt, HDK_getChannelWidth(), &CalibRangeDetails, outElements_p) == RET_VAL_FAIL) {
				RetFailOk = RET_VAL_FAIL;
			}
		}
	}
	
	if(inCalibType == CLBR_TYPE_ONLINE)
	{
		fillNextOnlineIterationDetails(CalibRangeDetails.nextCalibTpcIndex, outElements_p, &isLastIteration);
 
		//no matter if antenna SUCCESS/FAIL we want to continue to next antenna in online
		if(!isLastIteration) RetFailOk=RET_VAL_FRAGMENTED;
	}
	else
	{
		/* At the end of the offline calibration init online calibration to its initial state */
		initAntennaOnlineIterationDetails(TRUE,outElements_p);
	}
	
	outElements_p->results.lastRunTime = TIME_STAMP(END_TIME,startTime);

	PhyCalDrv_RestoreOperationalTx();
#if LM_CALIBRATION_LOGS_ON 
	ILOG9_DD("Tx Lo: Finished Calibration. Ret Code = %d, Run Time = %d", RetFailOk, outElements_p->results.lastRunTime);
#endif
    /* Store calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if(calStoreState == CAL_DATA_STATE_STORE)
	{
		TxLoLeakage_StoreResults();
	}
	return RetFailOk;
}

/********************************************************************************

fillCalibrationDetails

Description:
------------
    Fill the ongoing calibration details    

Input:
-----
	inCalibType
	inElements_p

Output:
-------
	outCalibRangeDetails_p   - Details of the current calibration iteration.
    
********************************************************************************/
static void fillCalibrationDetails(IN ClbrType_e inCalibType, IN TxLO_elements_t* inElements_p, TxLoCalibRangeDetails_t *outCalibRangeDetails_p)
{
	
	uint8 antMask, numOfAnts;
	HDK_GetMaxActiveAntennasNumAndMask(&numOfAnts, &antMask);
	
    if(inCalibType == CLBR_TYPE_ONLINE)
    {
         /* Calibrating all 3 antennas is too long to execute during on line calibration, therefore we divide the calibration, 
            each iteration we calibrate numTpcsOnlineIteration of a single antenna 
			(or less if this is the last iteration - calibrate remaining TPCs)
		  */
        outCalibRangeDetails_p->antMask  = ANTENNA_BIT(txLoOnlineDetails.calibAnt);
       
		outCalibRangeDetails_p->tpcInfo_p = &(txLoOnlineDetails.tpcInfo);
		outCalibRangeDetails_p->nextCalibTpcIndex = txLoOnlineDetails.nextCalibTpcIndex;
		//outCalibRangeDetails_p->numCalibDoneTpcs = txLoOnlineDetails.numCalibDoneTpcs;

        // Update the total number of calibrated Tpcs for single online iteration
        outCalibRangeDetails_p->totalNumCalibTpcs  = inElements_p->params.numTpcsOnlineIteration;  
    }
    else
    {
        outCalibRangeDetails_p->antMask = antMask;
		//tpcInfo (including nextCalibTpcIndex & totalNumCalibratedTpcs) will be read from RFIC per antenna when looping on antennas
    }
}
/********************************************************************************

initAntennaOnlineIterationDetails

Description:
------------
   Initialize antenna online calibration to its starting state (digital DAC calibration TPC)
    
Input:
-----
    inFirstTime - TRUE if this is the first iteration after offline

Output:
-----
	outElements_p
	
********************************************************************************/
static void initAntennaOnlineIterationDetails(bool inFirstTime, TxLO_elements_t*  outElements_p)
{
	//txLoOnlineDetails.numCalibDoneTpcs = 0;
	
	//Find last enabled antenna
	//don't asssume that enabled antennas can't change during online without moving back to offline	
	FindTxEnabledAntenna(0,PREVIOUS_ENABLED_ANTENNA_SEARCH,&txLoOnlineDetails.lastEnabledAntenna);

	//First iteration after offline
	if(inFirstTime){
		//Start with first enabled antenna
		FindTxEnabledAntenna(MAX_NUM_OF_ANTENNAS-1,NEXT_ENABLED_ANTENNA_SEARCH,&txLoOnlineDetails.calibAnt);
	}
	//next online antenna
	else{
		//Continue with next enabled antenna
		FindTxEnabledAntenna(txLoOnlineDetails.calibAnt,NEXT_ENABLED_ANTENNA_SEARCH,&txLoOnlineDetails.calibAnt);
	}
	
	//get TPC info
	RficDriver_TxLoGetTpcInfo((Antenna_e)(txLoOnlineDetails.calibAnt),&(txLoOnlineDetails.tpcInfo));

	//start with digital DAC calibration TPC
	txLoOnlineDetails.nextCalibTpcIndex = outElements_p->params.digitalDacTpcIdx;
}

/********************************************************************************

fillNextOnlineIterationDetails

Description:
------------
    Fill all the required data (antenna,TPC) for the next online iteration:
    The calibration sequence is for each Tx enabled antenna, numTpcsOnlineIteration TPCs at a time.
	First the digital DAC TPC is calibrated and the all other calibration TPC's from array start to end
    
Input:
-----
    inNextCalibTpc
	inElements_p

Return:
-------
    outIsLastIteration_p - Flag that indicates end of on line calibration

********************************************************************************/
static void fillNextOnlineIterationDetails(IN int8 inNextCalibTpc, TxLO_elements_t*  inElements_p, bool *outIsLastIteration_p)
{  
	//initialize - in case we continue with current antenna
	*outIsLastIteration_p = FALSE;

	//continue with next antenna if all calibrated TPCs done
	if(inNextCalibTpc == -1){ 

		/*If online calibration done for last enabled antenna,  
			Set flag that indicates on line last iteration so we can move on to the next calibration*/
		*outIsLastIteration_p = (txLoOnlineDetails.calibAnt==txLoOnlineDetails.lastEnabledAntenna);
  
		initAntennaOnlineIterationDetails(FALSE,inElements_p);
	}
	else
	{
		txLoOnlineDetails.nextCalibTpcIndex = inNextCalibTpc;
	}

    return;
}

/********************************************************************************
 calibrateSingleAntenna:

Description:
------------
Find and set TxLo compensation equalizer (I&Q DAC values) per single tx antenna.

Input:
-----
1. inCalibType
2. inRfCalibAnt
3. inCbMode
4. inCalibRangeDetails_p

Output:
-------
outElements_p

Return:
-------
RetVal_e
********************************************************************************/
static RetVal_e calibrateSingleAntenna(IN ClbrType_e inCalibType, uint8 inRfCalibAnt, uint8 inCbMode, TxLoCalibRangeDetails_t *inCalibRangeDetails_p, TxLO_elements_t* outElements_p)
{
	RetVal_e RetFailOk=RET_VAL_SUCCESS;
	int8 i_digital_dac,q_digital_dac;
	int32 StartTime;
	RficLoopbackAntennas_t	antInfo;	//Loopback antennas

	/*get rx/tx loopback antennas (AM detector loop back) */
	antInfo.RfCalibAnt = (Antenna_e)inRfCalibAnt;
	RficDriver_GetDigitalLoopBackAnt(&antInfo);

#if LM_CALIBRATION_LOGS_ON 
	ILOG9_DDD("TXLo calibrateSingleAntenna:RF Ant %d,LoopBackRxAnt %d, LoopBackTxAnt %d",antInfo.RfCalibAnt,antInfo.RxLoopbackAnt,antInfo.TxLoopbackAnt);
#endif    

	initHwRegisters(&antInfo,outElements_p,&i_digital_dac,&q_digital_dac);

	RetFailOk = calibrateAllTpcsInRange(inCalibType,&antInfo,inCbMode,inCalibRangeDetails_p,outElements_p);

	StartTime = TIME_STAMP(START_TIME,0);

	//done even if calibration/verification failed
	restoreHwRegisters(&antInfo, i_digital_dac, q_digital_dac, outElements_p);

	outElements_p->results.restoreHwRegistersRunTime = TIME_STAMP(END_TIME,StartTime);

#if LM_CALIBRATION_LOGS_ON 
	ILOG9_D("TxLo calibrateSingleAntenna(): Restore Hw Registers Run Time = %d", outElements_p->results.restoreHwRegistersRunTime);
#endif
	return RetFailOk;
}

/********************************************************************************

initHwRegisters

Description:
------------
    Set RF,AFE & Phy(Digital) relevant registers for TxLo calibration(Using loop back in the RFIC)  
	of single antenna

Input:
-----
    inAntInfo_p
	inBranch (I/Q)
	inElements_p

Output:
-------
	outDigitalDacI_p
	outDigitalDacQ_p
    
********************************************************************************/
static __inline void initHwRegisters(RficLoopbackAntennas_t *inAntInfo_p, IN TxLO_elements_t* inElements_p, int8 *outDigitalDacI_p, int8 *outDigitalDacQ_p)
{
	//int32 initStartTime= TIME_STAMP(START_TIME,0);
    RficPathTypeIndex_e rficGoertzelBranch;
	uint8 antMask;
    bool iqSwap = FALSE;
	Goertzel_e goertzelNum;

	// Set RF loopback 
	
	RficDriver_SetLoopback(inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt,RFIC_LOOPBACK_TYPE_TX_AMDET);
	
	/* Init*/ 

	// Initialize PHY
	PhyCalDrv_SetLoopBackMode(CLBR_PROC_TYPE_TX_LO_LEAKAGE,inAntInfo_p->RfCalibAnt);
	AFE_SetLoopBack(inAntInfo_p->RfCalibAnt, inAntInfo_p->RxLoopbackAnt, AFE_MODE_LOOPBACK, 0);

   	/* Init the Goertzel */
	goertzelNum = (Goertzel_e)(inAntInfo_p->RxLoopbackAnt & 2); // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	
    RficDriver_GetAmDetBranch(HDK_getBand(), &rficGoertzelBranch);

	PhyCalDrv_GoertzelInit(goertzelNum, inAntInfo_p->RxLoopbackAnt,rficGoertzelBranch, iqSwap,inElements_p->params.grtzlNumCycles);

	antMask = 1 << inAntInfo_p->RfCalibAnt;
	PhyCalDrv_SetToneGenParams(0, 0, antMask, 0);
	if (HDK_getChannelWidth() == BANDWIDTH_EIGHTY)
	{
		inElements_p->params.toneAmplitude = LO_TONE_AMPLITUDE - 1;
	}
	PhyCalDrv_ToneGenInit(inElements_p->params.toneAmplitude);
	
	PhyCalDrv_ToggleTxTone(inElements_p->params.calibBin,TONE_GEN_TOGGLE_ON); // generate calibrated tone

	/* save original TxLoopbackAnt values to restore at the end
		done only if we run over values of a different antenna during the calibration algorithm 
		(as a result of the loop back design in the RFIC - platform specific)
		Save only if TxLoopbackAnt != RfCalibAnt 
	*/
	if(inAntInfo_p->TxLoopbackAnt != inAntInfo_p->RfCalibAnt){
		PhyCalDrv_GetDigitalLoCancelPair(inAntInfo_p->TxLoopbackAnt,outDigitalDacI_p,outDigitalDacQ_p);
	}
	//Start with zero digital DACs
	PhyCalDrv_SetDigitalLoCancelPair(inAntInfo_p->TxLoopbackAnt,0,0);	

	return;
}

/********************************************************************************

restoreHwRegisters

Description:
------------
    Restore RF,AFE & Phy(Digital) registers to their values prior to TxLo calibration
	of a single antenna

Input:
-----
    inAntInfo_p
	inDigitalDacI
	inDigitalDacQ
	inElements_p

Output:
-------
    
********************************************************************************/
static __inline void restoreHwRegisters(RficLoopbackAntennas_t *inAntInfo_p,  int8 inDigitalDacI, int8 inDigitalDacQ, IN TxLO_elements_t* inElements_p)
{
	int8 q_temp_digital_dac,i_temp_digital_dac;
	//int32 restoreStartTime= TIME_STAMP(START_TIME,0);

	//only if Rf Tx and Loopback Tx antennas are different (platform dependent)
	if(inAntInfo_p->TxLoopbackAnt != inAntInfo_p->RfCalibAnt){

		/* Restore PHY */
		PhyCalDrv_GetDigitalLoCancelPair(inAntInfo_p->TxLoopbackAnt,&i_temp_digital_dac,&q_temp_digital_dac);
		//set RF antenna with calibrated values
		PhyCalDrv_SetDigitalLoCancelPair(inAntInfo_p->RfCalibAnt,i_temp_digital_dac,q_temp_digital_dac);	
		
		//restore loopback Tx antenna
		PhyCalDrv_SetDigitalLoCancelPair(inAntInfo_p->TxLoopbackAnt,inDigitalDacI,inDigitalDacQ);	
	}	

	PhyCalDrv_ToggleTxTone(inElements_p->params.calibBin,TONE_GEN_TOGGLE_OFF);  // clear tone
	PhyCalDrv_ToneGenRestore();
	PhyCalDrv_RestoreOperationalTx();
	AFE_RestoreFromLoopBack();

	/*RF*/
	RFIC_RestoreFromLoopBack(inAntInfo_p->RfCalibAnt);

	return;
}

/********************************************************************************

calibrateAllTpcsInRange

Description:
------------
   Calibrate all TPCs in the required range  

Input:
-----
	inCalibType
	inAntInfo_p
	inCbMode
    inCalibRangeDetails_p   - Pointer to a struct that holds all needed calibration details(next TPC to calibrate,number of TPCs to calibrate)
	inMeasureTime			- do time measurements

Output:
-------
outElements_p

Return:
-------
	(RET_VAL_SUCCESS)	- Successful 
	(RET_VAL_FAIL)		- Failed - at least one TPC failed (verification failed)
********************************************************************************/
static RetVal_e calibrateAllTpcsInRange(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,uint8 inCbMode,TxLoCalibRangeDetails_t *inCalibRangeDetails_p,OUT TxLO_elements_t* outElements_p)
{
    RetVal_e RetFailOk = RET_VAL_SUCCESS;
    int8 tpcIndex;
	uint8 iterationNum;
	RficDriverTxLoTpcInfo_t offlineTpcInfo;
	bool update;

	//online mode RFIC TPC info get was done once in first antenna iteration

	//offline mode - get antenna RFIC TPC info
	if(inCalibType == CLBR_TYPE_OFFLINE)
	{
		inCalibRangeDetails_p->tpcInfo_p = &offlineTpcInfo;
		initAntennaOfflineIterationDetails(inAntInfo_p->RfCalibAnt, outElements_p, inCalibRangeDetails_p);
	}
	//online and first TPC
	else if(inCalibRangeDetails_p->nextCalibTpcIndex == outElements_p->params.digitalDacTpcIdx)
	{
		outElements_p->results.status[inAntInfo_p->RfCalibAnt] = CAL_STATUS_IN_PROCESS;
	}

	//Loop on all antenna TPC values that should be calibrated in current state 
	for(iterationNum = 0; 
		(iterationNum < inCalibRangeDetails_p->totalNumCalibTpcs) && ((tpcIndex = inCalibRangeDetails_p->nextCalibTpcIndex) != -1); 
		iterationNum++)
    {

		//calibrate and verify
		if(calibrateAndVerifySingleTpc(inCalibType,inAntInfo_p,inCbMode,tpcIndex,inCalibRangeDetails_p->tpcInfo_p,(iterationNum == 0),outElements_p) != RET_VAL_SUCCESS){
//			update = FALSE; //don't update backward/forward TPC's if TPC calibration(verification) failed - if we implement fix failed calibrated TPC's
			update = TRUE; //update forward/backward TPC's even if TPC calibration failed due to verification failure
			RetFailOk = RET_VAL_FAIL;
		}
		else
			update = TRUE;

		//Update forward/backward TPCs until next calibrated TPC
		//update forward/backward DACs even if verification failed
		updateTpcsToNextCalibTpc(inAntInfo_p->RfCalibAnt,(uint8)tpcIndex,inCalibRangeDetails_p->tpcInfo_p,update,outElements_p,&(inCalibRangeDetails_p->nextCalibTpcIndex));
    }

    return RetFailOk;
}

/********************************************************************************

initAntennaOfflineIterationDetails

Description:
------------
   Initialize antenna offline calibration to its starting state (digital DAC calibration TPC)
    
Input:
-----
inRfCalibAnt

Output:
------
outElements_p
outCalibRangeDetails_p

********************************************************************************/
static void initAntennaOfflineIterationDetails(uint8 inRfCalibAnt, TxLO_elements_t* outElements_p, TxLoCalibRangeDetails_t *outCalibRangeDetails_p)
{
	outElements_p->results.status[inRfCalibAnt]=CAL_STATUS_IN_PROCESS;

	//get TPC info
	RficDriver_TxLoGetTpcInfo((Antenna_e)(inRfCalibAnt),outCalibRangeDetails_p->tpcInfo_p);

	//start with digital DAC calibration TPC
	outCalibRangeDetails_p->nextCalibTpcIndex = outElements_p->params.digitalDacTpcIdx;
	//calibrate all calibrated TPCs in one step
	outCalibRangeDetails_p->totalNumCalibTpcs  = outCalibRangeDetails_p->tpcInfo_p->numCalibTpcs; 
}

/********************************************************************************

calibrateAndVerifySingleTpc

Description:
------------
    Find TxLo mismatch compensation equalizer for single TPC.    
	Verify calibration.

Input:
-----
	inCalibType
	inAntInfo_p
	inCbMode
	inTpcIndex				- Calibrated TPC index (positive)
	inTpcInfo_p
    inMeasureTime			- Measure time duration for single TPC

	Output:
	-------
	outElements_p

Return:
-------
	(RET_VAL_SUCCESS)	- Successful 
	(RET_VAL_FAIL)		- Failed - verification failed
********************************************************************************/
static RetVal_e calibrateAndVerifySingleTpc(IN ClbrType_e inCalibType,RficLoopbackAntennas_t *inAntInfo_p,uint8 inCbMode,uint8 inTpcIndex, RficDriverTxLoTpcInfo_t *inTpcInfo_p, bool inMeasureTime, OUT TxLO_elements_t* outElements_p)
{
    RetVal_e RetFailOk;
	int32 startTime;
	int8 i_q_indexes[RFIC_PATH_TYPE_MAX];
	int8 i, repeat;
	TxLoDigitalDacInfo_t	digitalDacInfo;
	bool digitalDacIter;	
	int32 fin_energy_tmp = 0;
	int64 fin_energy = 0;
	int8 maxIndex = 0;
	int8 maxIndexQ = 0;
	int8 minIndex = 0;	
	int8 minIndexQ = 0;
	int8 toneAmplitude = 0;
	uint16 readVal[3];
	uint32 isInternalPA = 0;
	
	//measure time duration of single tpc calibration only if first index is 0 positive
	if(inMeasureTime)
		startTime = TIME_STAMP(START_TIME,0);

#if LM_CALIBRATION_LOGS_ON 
	ILOG9_D("TxLo calibrateAndVerifySingleTpc: TPC %d",inTpcIndex);
#endif

	//Set TPC
	RficDriver_SetTxGain(inAntInfo_p->RfCalibAnt,inTpcIndex);

	/*
		galia - NOTE: the reference verify energy must be measured on the first calibrated TPC (usually TPC index 0)
		since all other TPC's are compared to this energy during TPC verify after TPC calibration.
		However, if the digital DAC TPC is not TPC index 0 then the digital DAC TPC will be calibrated first 
		and not the reference energy TPC and verify will not work!!!
	*/
	//measure reference bin energy for calibration verify using reference TPC index (0) and I,Q DACs 0
	if(inTpcIndex == outElements_p->params.verifyRefTpcIdx){

		// Decrease 6 dB in ref measurments to prevent Gohertzl saturation
		PhyCalDev_SetToneGenAmplitude(outElements_p->params.toneAmplitude + (TONE_AMPLITUDE_REFERENCE_SIGN * 1));

		//transmit tone at verify reference bin
		PhyCalDrv_ToggleTxTone(outElements_p->params.verifyRefBin,TONE_GEN_TOGGLE_ON); // generate reference tone

		//read energy using average on verificationNumMeasurements
		fin_energy=0;
		for (i=0; i<outElements_p->params.verifyNumMeasurements; i++) {
			//measure energy for I=Q=0 and BIN=verifyRefBin(16)-calibBin(3)
			measureEnergy(inTpcIndex, 0, 0, outElements_p->params.verifyRefBin-outElements_p->params.calibBin, inAntInfo_p, outElements_p, &fin_energy_tmp);
			fin_energy+= fin_energy_tmp;
		}
		// NOTE: the 64 bit number is casted to 32 bit under the assumption of low value of verifyNumMeasurements (<=4)
		//MATH_10Log10Res returns (10log10)<<5
		outElements_p->results.refEnergy[inAntInfo_p->RfCalibAnt] = MATH_10Log10Res((uint32)(fin_energy/outElements_p->params.verifyNumMeasurements), LO_ENERGY_LOG_RESOLUTION);
		//shift back >>5 to get 10log10
		outElements_p->results.refEnergy[inAntInfo_p->RfCalibAnt] >>= LO_ENERGY_LOG_RESOLUTION;

		//Compensate for the two tone ref Goertzel measurments that decreases 11 dB for 1 amplitude level change
		outElements_p->results.refEnergy[inAntInfo_p->RfCalibAnt] += 10;

#if LM_CALIBRATION_LOGS_ON 
		ILOG9_D("Tx Lo calibrateAndVerifySingleTpc(): Verify reference Energy(log) = %d",outElements_p->results.refEnergy[inAntInfo_p->RfCalibAnt]);
#endif
			
		PhyCalDrv_ToggleTxTone(outElements_p->params.verifyRefBin,TONE_GEN_TOGGLE_OFF);  // clear reference tone

		PhyCalDev_SetToneGenAmplitude(outElements_p->params.toneAmplitude);
		
		PhyCalDrv_ToggleTxTone(outElements_p->params.calibBin,TONE_GEN_TOGGLE_ON); // generate calibrated tone

	}

	if (inCalibType == CLBR_TYPE_ONLINE)
	{
		//online: initial q DAC is the previous calibrated value
		RficDriver_GetLoDACs(inAntInfo_p->RfCalibAnt,(RficTxGainTpcIndex_e)inTpcIndex,i_q_indexes);
		//online: repeat 1 iteration
		repeat=1;											
	}
	else
	{
		//offline: initial i,q DACs are 0
		i_q_indexes[RFIC_PATH_TYPE_I] = 0;
		i_q_indexes[RFIC_PATH_TYPE_Q] = 0;			
		//offline: repeat DacNumIterOffline iterations
		repeat = outElements_p->params.dacNumIterOffline;	
	}

	//initialize digital DAC clipping
	digitalDacInfo.doClip = FALSE;
	
	//Perform digital DAC clipping only if in offline mode and TPC is the digital DAC TPC
	digitalDacIter = ((inCalibType == CLBR_TYPE_OFFLINE) && (outElements_p->params.digitalDacTpcIdx == inTpcIndex));

	if(digitalDacIter)
	{
		digitalDacInfo.maxIter = outElements_p->params.digitalDacMaxIter;
		digitalDacInfo.clipMin = outElements_p->params.clipMin;
		digitalDacInfo.clipMax = outElements_p->params.clipMax;
		digitalDacInfo.iterCounter=0;
	}

		RficDriver_ReadRegister(REG_RFIC_L41_BC_EN_2G_INTPA, readVal);
		if ((readVal[0] & REG_RFIC_L41_BC_EN_2G_INTPA_MASK) == 0) //if external PA board
		{
			isInternalPA = 0;
		}
		else
		{
			isInternalPA = 1;
		}

	do
	{
		if (digitalDacInfo.doClip)
		{
			digitalDacInfo.doClip = FALSE;
			digitalDacInfo.i = i_q_indexes[RFIC_PATH_TYPE_I];
			digitalDacInfo.q = i_q_indexes[RFIC_PATH_TYPE_Q];
			addDigitalDac(inAntInfo_p,FALSE,&digitalDacInfo,outElements_p);
			i_q_indexes[RFIC_PATH_TYPE_I]=0;
			i_q_indexes[RFIC_PATH_TYPE_Q]=0;
		}
	
		//Calibrate DACs
		minIndex = outElements_p->params.dacMin;
		minIndexQ = outElements_p->params.dacMin;
		maxIndex = outElements_p->params.dacMax;
		maxIndexQ = outElements_p->params.dacMax;
		 
		
		for (i=0; i<repeat; i++) 
		{
			
			toneAmplitude =outElements_p->params.toneAmplitude;



			getMinimumIndex(&minIndex,&minIndexQ,i,i_q_indexes[RFIC_PATH_TYPE_I],i_q_indexes[RFIC_PATH_TYPE_Q],toneAmplitude,inCalibType,inAntInfo_p->RfCalibAnt);
	

			if(minIndex < LO_DAC_MIN)
			{

				minIndex = LO_DAC_MIN;
			}
			if(minIndexQ < LO_DAC_MIN)
			{
				minIndexQ = LO_DAC_MIN;
			}

			/* calibrate I */
#if LM_CALIBRATION_LOGS_ON 			
			ILOG9_DD(" calibrateAndVerifySingleTpc() starting I - Iteration = %d Q = %d", i, i_q_indexes[RFIC_PATH_TYPE_Q]);
#endif

			doAxisIteration(minIndex, maxIndex, i_q_indexes[RFIC_PATH_TYPE_Q], /*is_q*/ FALSE, inAntInfo_p, inTpcIndex, outElements_p, &i_q_indexes[RFIC_PATH_TYPE_I],isInternalPA,i,inCalibType);
#if LM_CALIBRATION_LOGS_ON 
			ILOG9_DD(" TxLo doAxisIteration(). Iteration = %d, I = %d", i, i_q_indexes[RFIC_PATH_TYPE_I]);
#endif

			//Update if digital DAC clipping required
			if(digitalDacIter)
				digitalDacUpdateRequired(i_q_indexes[RFIC_PATH_TYPE_I],&digitalDacInfo);

			
			/* calibrate Q */
#if LM_CALIBRATION_LOGS_ON
			ILOG9_DD(" calibrateAndVerifySingleTpc(). starting Q - Iteration = %d I = %d", i, i_q_indexes[RFIC_PATH_TYPE_I]);
#endif

			doAxisIteration(minIndexQ, maxIndexQ, i_q_indexes[RFIC_PATH_TYPE_I], /*is_q*/ TRUE, inAntInfo_p, inTpcIndex, outElements_p, &i_q_indexes[RFIC_PATH_TYPE_Q],isInternalPA,i, inCalibType);
#if LM_CALIBRATION_LOGS_ON 
			ILOG9_DD(" calibrateAndVerifySingleTpc(). Iteration = %d, Q = %d", i, i_q_indexes[RFIC_PATH_TYPE_Q]);
#endif

			//Update if digital DAC clipping required
			if(digitalDacIter)
				digitalDacUpdateRequired(i_q_indexes[RFIC_PATH_TYPE_Q],&digitalDacInfo);
		}
	}
	// do it again with more DC at the digital DAC if we are in offline,
	while (digitalDacInfo.doClip); //doClip is updated only in digitalDacIter

	if(digitalDacIter){ 
#if LM_CALIBRATION_LOGS_ON 
		ILOG9_DD("TxLo calibrateAndVerifySingleTpc digitalDacI = %d, digitalDacQ = %d",outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt],outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt]);
#endif
	}

	outElements_p->results.dacs[inAntInfo_p->RfCalibAnt][inTpcIndex][RFIC_PATH_TYPE_I] = i_q_indexes[RFIC_PATH_TYPE_I];
	outElements_p->results.dacs[inAntInfo_p->RfCalibAnt][inTpcIndex][RFIC_PATH_TYPE_Q] = i_q_indexes[RFIC_PATH_TYPE_Q];

#if LM_CALIBRATION_LOGS_ON 
	ILOG9_DD("TxLo calibrateAndVerifySingleTpc dacI = %d, dacQ = %d",i_q_indexes[RFIC_PATH_TYPE_I],i_q_indexes[RFIC_PATH_TYPE_Q]);
#endif

	RetFailOk = verifyCalibrationResults(inAntInfo_p,inTpcIndex,inCbMode,i_q_indexes,inTpcInfo_p,outElements_p);

	//measure time duration of single TPC calibration only for index 0 positive
	if(inMeasureTime)
	{
		outElements_p->results.singleTpcCalibRunTime = TIME_STAMP(END_TIME,startTime);

#if LM_CALIBRATION_LOGS_ON 
		ILOG9_D("TxLo calibrateAndVerifySingleTpc(): Run Time = %d",outElements_p->results.singleTpcCalibRunTime);
#endif
	}

    return RetFailOk;
}


/********************************************************************************
 digitalDacUpdateRequired:	

Description:
------------
Check if calibrated DAC value is outside of allowed range 
in which case another iteration of digital DAC clipping is required.

Input:
-----
inDacIndex

Output:
-------
outDigitalDacInfo_p - doClip = TRUE/FALSE
********************************************************************************/
static void digitalDacUpdateRequired(int8 inDacIndex, TxLoDigitalDacInfo_t	*outDigitalDacInfo_p)
{
	outDigitalDacInfo_p->doClip = FALSE;

	if (inDacIndex > outDigitalDacInfo_p->clipMax || inDacIndex < outDigitalDacInfo_p->clipMin){
		//did not exceed max digital DAC iterations
		if (outDigitalDacInfo_p->iterCounter < outDigitalDacInfo_p->maxIter) 
			outDigitalDacInfo_p->doClip = TRUE;
		//exceeded max digital DAC iterations
		else{
#if LM_CALIBRATION_LOGS_ON
			ILOG9_DD("TxLo digitalDacUpdateRequired() DAC  %d exceeds allowed range but digital DAC max iterations %d exceeded",inDacIndex,outDigitalDacInfo_p->maxIter);
#endif
		}
	}
}

/********************************************************************************
 addDigitalDac:	

Description:
------------
In case of failure we use the Base band DC .
here we read the BB DC and add DC if necessary.
for better understanding to which direction we need to improve , 
if we have reached LO_CLIP_MIN(-58) at the Q DAC , we will add at the Q BB dc path

Input:
-----
1. inAntInfo_p
2. inIsIqSwapped

Output:
-------
outDigitalDacInfo_p
outElements_p
********************************************************************************/
static void addDigitalDac(RficLoopbackAntennas_t *inAntInfo_p, bool inIsIqSwapped, TxLoDigitalDacInfo_t *outDigitalDacInfo_p, TxLO_elements_t* outElements_p)
{

//	int8 q_index_clip,i_index_clip;
	uint8 i_Branch,q_Branch;
	int32 newDacValue;

	if(inIsIqSwapped)
	{
		i_Branch = RFIC_PATH_TYPE_Q;
		q_Branch = RFIC_PATH_TYPE_I;
	}
	else
	{
		i_Branch = RFIC_PATH_TYPE_I;
		q_Branch = RFIC_PATH_TYPE_Q;
	}
	
	outDigitalDacInfo_p->divider = 1 << outDigitalDacInfo_p->iterCounter;

	if (outDigitalDacInfo_p->q > outDigitalDacInfo_p->clipMax)
	{
		newDacValue = (int32)outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] - LO_DIGITAL_DAC_ADDITION/outDigitalDacInfo_p->divider;
		if (newDacValue < LO_DIGITAL_DAC_MIN)
		{
			outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] = LO_DIGITAL_DAC_MIN;
		}
		else
		{
			outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] = (int8)newDacValue;
		}
		PhyCalDrv_SetDigitalLoCancel(q_Branch,inAntInfo_p->TxLoopbackAnt,outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt]);	
	}
	else if (outDigitalDacInfo_p->q < outDigitalDacInfo_p->clipMin)
	{
		newDacValue = (int32)outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] + LO_DIGITAL_DAC_ADDITION/outDigitalDacInfo_p->divider;
		if (newDacValue > LO_DIGITAL_DAC_MAX)
		{
			outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] = LO_DIGITAL_DAC_MAX;
		}
		else
		{
			outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt] = (int8)newDacValue;
		}
		PhyCalDrv_SetDigitalLoCancel(q_Branch,inAntInfo_p->TxLoopbackAnt,outElements_p->results.digitalDacQ[inAntInfo_p->RfCalibAnt]);	
	}
	
	if (outDigitalDacInfo_p->i > outDigitalDacInfo_p->clipMax)
	{
		newDacValue = (int32)outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] - LO_DIGITAL_DAC_ADDITION/outDigitalDacInfo_p->divider;
		if (newDacValue < LO_DIGITAL_DAC_MIN)
		{
			outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] = LO_DIGITAL_DAC_MIN;
		}
		else
		{
			outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] = (int8)newDacValue;
		}
		PhyCalDrv_SetDigitalLoCancel(i_Branch,inAntInfo_p->TxLoopbackAnt,outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt]);
	}
	else if (outDigitalDacInfo_p->i < outDigitalDacInfo_p->clipMin)
	{
		newDacValue = (int32)outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] + LO_DIGITAL_DAC_ADDITION/outDigitalDacInfo_p->divider;
		if (newDacValue > LO_DIGITAL_DAC_MAX)
		{
			outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] = LO_DIGITAL_DAC_MAX;
		}
		else
		{
			outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt] = (int8)newDacValue;
		}
		PhyCalDrv_SetDigitalLoCancel(i_Branch,inAntInfo_p->TxLoopbackAnt,outElements_p->results.digitalDacI[inAntInfo_p->RfCalibAnt]);	
	}

	outDigitalDacInfo_p->iterCounter++;
}

/********************************************************************************
 
verifyCalibrationResults:

Description:
------------ 

Input:
-----
	1. inAntInfo_p
	2. inTpcIndex
	3. inCbMode
	4. inIndexes_p
	5. inTpcInfo_p

Output:
-------
    outElements_p

Return:
-------
	RET_VAL_SUCCESS/RET_VAL_FAIL
********************************************************************************/
static RetVal_e verifyCalibrationResults(RficLoopbackAntennas_t *inAntInfo_p, uint8 inTpcIndex, uint8 inCbMode, int8 *inIndexes_p, RficDriverTxLoTpcInfo_t *inTpcInfo_p, OUT TxLO_elements_t* outElements_p)
{
	RetVal_e RetFailOk;
	int32 fin_energy_tmp = 0;
	int64 fin_energy=0;
	uint8 i;
	//galia w300 verify int16 log_energy,fin_ll;
	uint32 startTime;

	startTime = TIME_STAMP(START_TIME,0);

	//read energy using average on verificationNumMeasurements
	fin_energy=0;
	for (i=0; i<outElements_p->params.verifyNumMeasurements; i++) {
		measureEnergy(inTpcIndex, inIndexes_p[RFIC_PATH_TYPE_I], inIndexes_p[RFIC_PATH_TYPE_Q], outElements_p->params.calibBin, inAntInfo_p, outElements_p, &fin_energy_tmp);
		fin_energy+= fin_energy_tmp;
	}

	// NOTE: the 64 bit number is casted to 32 bit under the assumption of low value of verifyNumMeasurements (<=4)
	//MATH_10Log10 returns (10log10)<<4
	outElements_p->results.tpcEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex] = MATH_10Log10Res((uint32)(fin_energy/outElements_p->params.verifyNumMeasurements), LO_ENERGY_LOG_RESOLUTION);
	//shift back >>4 to get 10log10
	outElements_p->results.tpcEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex] >>= LO_ENERGY_LOG_RESOLUTION;

	outElements_p->results.diffEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex] = 
		outElements_p->results.refEnergy[inAntInfo_p->RfCalibAnt] + (inTpcInfo_p->tpcDiffFromRef[inTpcIndex] << 1) - 
		outElements_p->results.tpcEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex];

#if LM_CALIBRATION_LOGS_ON 
	ILOG9_DD("Tx Lo VerifyCalibrationResults(): TPC Energy(log) = %d Diff from Ref %d",outElements_p->results.tpcEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex], inTpcInfo_p->tpcDiffFromRef[inTpcIndex]);
#endif

	//verify failed 
	if(outElements_p->results.diffEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex] < outElements_p->params.maxVerifyEnergyDiff)
	{
		TxLoLeakage_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_FAIL);
		RetFailOk = RET_VAL_FAIL;

#if LM_CALIBRATION_LOGS_ON
		ILOG9_DDD("Tx Lo verifyCalibrationResults():Energy diff from ref %d below TH = %d status = %d",outElements_p->results.diffEnergy[inAntInfo_p->RfCalibAnt][inTpcIndex],outElements_p->params.maxVerifyEnergyDiff,outElements_p->results.status[inAntInfo_p->RfCalibAnt]);
#endif
	}
	else{
		TxLoLeakage_SetStatus(inAntInfo_p->RfCalibAnt,CAL_STATUS_PASS);
		RetFailOk = RET_VAL_SUCCESS;

#if LM_CALIBRATION_LOGS_ON
		ILOG9_D("TxLo verifyCalibrationResults status = %d",outElements_p->results.status[inAntInfo_p->RfCalibAnt]);
#endif
	}

	outElements_p->results.singleTpcVerifyRunTime = TIME_STAMP(END_TIME,startTime);

#if LM_CALIBRATION_LOGS_ON
	ILOG9_D("TxLo verifyCalibrationResults(): Single TPC verify Run Time = %d",outElements_p->results.singleTpcVerifyRunTime);
#endif

	return(RetFailOk);
}


/********************************************************************************

updateTpcsToNextCalibTpc

Description:
------------
    Update all DAC values of the TPCs following current calibrated TPC until next TPC to calibrate is found.
	Non-calibrated TPCs are updated with DAC values equal to those of the currently calibrated DAC.

	Note: The TPC used for digital DAC calibration may be located anywhere in the TPC array and
		is always calibrated first.

Input:
-----
	inRfCalibAnt
    inCurrCalibTpc  - The currently calibrated TPC. 
    inTpcInfo_p		
	inUpdate		- update forward/backward TPC's 
		(currently TRUE even if TPC calibrate & verify failed, 
		if we implement fix then update will not be done in the calibration stage if verify failed but only after the fix stage)

Output:
-------
	outElements_p
    outNextCalibTpc_p - (-1) if no more TPCs to calibrate
********************************************************************************/
static void updateTpcsToNextCalibTpc(uint8 inRfCalibAnt,uint8 inCurrCalibTpc,RficDriverTxLoTpcInfo_t *inTpcInfo_p,bool inUpdate,TxLO_elements_t* outElements_p,int8 *outNextCalibTpc_p)
{
	uint8 idx;
	bool updateTpcs;
	uint8 tempNextCalibTpc;
	static uint8 updateToStart = FALSE;	

	//calibrated TPC is the first index in array and is non-zero - update backwards all TPCs from array start (index zero) to current TPC
	if(updateToStart){
		if(inUpdate)
			updateTpcsToStart(inRfCalibAnt,inCurrCalibTpc,outElements_p);
		updateToStart = FALSE;
	}

	updateTpcs = TRUE;
	//update all successive TPCs until next calibrated TPC
	for (idx = inCurrCalibTpc+1, *outNextCalibTpc_p = -1; idx < inTpcInfo_p->tpcArraySize; idx++)
	{
		//found next calibrated TPC
		if(LO_TPC_MASK_IS_BIT_ON(inTpcInfo_p->calibTpcMask,idx)){

			//skip digital DAC TPC and find the next calibrated TPC
			if(idx == outElements_p->params.digitalDacTpcIdx){
				//stop TPC updates (already done after digital DAC TPC calibration)
				updateTpcs = FALSE;
			}
			//found new calibrated DAC (not digital DAC TPC)
			else{
				*outNextCalibTpc_p = idx;
				break;
			}
		}

		//non-calibrated TPCs - update TPC DACs
		if(inUpdate && updateTpcs){
			//RFIC
			RficDriver_SetLoDACs((Antenna_e)inRfCalibAnt, (RficTxGainTpcIndex_e)idx, outElements_p->results.dacs[inRfCalibAnt][inCurrCalibTpc]);

			//internal variables
			outElements_p->results.dacs[inRfCalibAnt][idx][RFIC_PATH_TYPE_I] = outElements_p->results.dacs[inRfCalibAnt][inCurrCalibTpc][RFIC_PATH_TYPE_I];
			outElements_p->results.dacs[inRfCalibAnt][idx][RFIC_PATH_TYPE_Q] = outElements_p->results.dacs[inRfCalibAnt][inCurrCalibTpc][RFIC_PATH_TYPE_Q];
		}
	}

	//current TPC is the digital DAC TPC which is always calibrated first regardless of its location in the array
	if(inCurrCalibTpc == outElements_p->params.digitalDacTpcIdx){
		//If the digital DAC TPC is not the first array entry, then continue calibration from first calibration TPC from array start
		if(inCurrCalibTpc != 0){
			//save first calibrated TPC after digital DAC TPC
			tempNextCalibTpc = *outNextCalibTpc_p;

			//search first calibration TPC from array start until digital DAC TPC
			for (idx = 0, *outNextCalibTpc_p = -1; idx < inCurrCalibTpc; idx++)
			{
				//found first calibrated TPC < digital DAC TPC
				if(LO_TPC_MASK_IS_BIT_ON(inTpcInfo_p->calibTpcMask,idx)){
					*outNextCalibTpc_p = idx;
					break;
				}
			}
			//no TPCs calibrated before digital DAC TPC - digital DAC is the first calibrated TPC index
			if(*outNextCalibTpc_p == -1){
				//continue from the first calibrated TPC after digital DAC TPC
				*outNextCalibTpc_p = tempNextCalibTpc;

				//currently calibrated digital DAC TPC is the first calibrated TPC
				//calibrate from start to digital DAC TPC (first calibrated index)
				if(inUpdate)
					updateTpcsToStart(inRfCalibAnt,inCurrCalibTpc,outElements_p);
			}
			//first array index smaller than digital DAC TPC and non-zero - 
			//raise flag so that in the next iteration when first index is calibrated
			//we will update from array start until first calibrated TPC
			else if(*outNextCalibTpc_p != 0)
				updateToStart = TRUE;			
		}
	}

//debug
#if LM_CALIBRATION_LOGS_ON
	ILOG9_D("TxLo updateTpcsToNextCalibTpc: next TPC = %d",*outNextCalibTpc_p);
#endif

    return;
}

/********************************************************************************
 measureEnergy:

Description:
------------
Measure Lo energy level of (a+jb) equalizer (using goertzel).

Input:
-----
1. inTpcIndex
2. I value
3. Q value
4. inBin
5. inRfCalibAnt

Output:
-------
outElements_p->status
outEnergy_p - measured energy (linear not log)
********************************************************************************/
static void measureEnergy(uint8 inTpcIndex, int8 inValueI ,int8 inValueQ, int8 inBin, RficLoopbackAntennas_t *inAntInfo_p, TxLO_elements_t* outElements_p, int32 *outEnergy_p)
{
	GoertzelRes_t	Results[TOTAL_GOERTZELS];
	int8 inValues[RFIC_PATH_TYPE_MAX];
	uint8 goertzelNum, goertzelMask;
	
	goertzelNum = inAntInfo_p->RxLoopbackAnt & 2; // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	goertzelMask = 1 << goertzelNum;

	/*set I and Q parameters*/
	inValues[RFIC_PATH_TYPE_I] = inValueI;
	inValues[RFIC_PATH_TYPE_Q] = inValueQ;
	RficDriver_SetLoDACs(inAntInfo_p->RfCalibAnt,(RficTxGainTpcIndex_e)inTpcIndex,inValues);
    
	PhyCalDrv_GoertzelMeasure(goertzelMask, Results,(int16)inBin,DELAY_AFTER_PHY_RFIC_CHANGE);

	Results[goertzelNum].Res_I_Real >>= LO_ENERGY_RESOLUTION; 
	Results[goertzelNum].Res_Q_Real >>= LO_ENERGY_RESOLUTION;
	*outEnergy_p = Results[goertzelNum].Res_I_Real*Results[goertzelNum].Res_I_Real+Results[goertzelNum].Res_Q_Real*Results[goertzelNum].Res_Q_Real;

    return;
}

/********************************************************************************

updateTpcsToStart

Description:
------------
    For the smallest calibrated TPC index, if non-zero, we must update all TPCs
	from array start (index 0) to first non-zero calibrated TPC to equal first TPC DAC values

Input:
-----
	inRfCalibAnt
    inFirstCalibTpc  - The first calibrated TPC (non-zero index)

Output:
-------
	outElements_p
********************************************************************************/
static void updateTpcsToStart(uint8 inRfCalibAnt,uint8 inFirstCalibTpc,TxLO_elements_t* outElements_p)
{
	uint8 idx;
	
	//update all TPCs from array start until first calibrated TPC
	for (idx = 0; idx < inFirstCalibTpc; idx++)
	{
		RficDriver_SetLoDACs((Antenna_e)inRfCalibAnt, (RficTxGainTpcIndex_e)idx, outElements_p->results.dacs[inRfCalibAnt][inFirstCalibTpc]);

		//internal variables
		outElements_p->results.dacs[inRfCalibAnt][idx][RFIC_PATH_TYPE_I] = outElements_p->results.dacs[inRfCalibAnt][inFirstCalibTpc][RFIC_PATH_TYPE_I];
		outElements_p->results.dacs[inRfCalibAnt][idx][RFIC_PATH_TYPE_Q] = outElements_p->results.dacs[inRfCalibAnt][inFirstCalibTpc][RFIC_PATH_TYPE_Q];
	}
}

//************************************
// Method:    TxLoLeakage_Calibrate
// FullName:  TxLoLeakage_Calibrate
// Access:    public 
// Returns:   RetVal_e
// Qualifier:
// Parameter: IN ClbrType_e inCalibType
//************************************
RetVal_e TxLoLeakage_Calibrate(IN ClbrType_e inCalibType)
{
	if (CoC_getNumActiveTxAnts() > 1)
	{
		return run(inCalibType, &Calibration_params_otf.TxLoParams);
	}
	else
	{
		return RET_VAL_SUCCESS;
	}
}

void TxLoLeakage_LoadResults(void)
{
	int8 uDigitalLOCancelI, uDigitalLOCancelQ;
	uint8 tpcIndex;
	Antenna_e uAntenna;
	RficDriverTxLoTpcInfo_t tpcInfo;
	uint8 numCalibTpcs;
    
    for(uAntenna = ANTENNA_0; uAntenna < MAX_NUM_OF_ANTENNAS; uAntenna++)
    {
        uDigitalLOCancelI = ShramClbrDataBufferStoreChanData.txLoData.digitalLoCancelPair[uAntenna][HDK_PATH_TYPE_I];
        uDigitalLOCancelQ = ShramClbrDataBufferStoreChanData.txLoData.digitalLoCancelPair[uAntenna][HDK_PATH_TYPE_Q];

        //Update internal variables
        Calibration_params_otf.TxLoParams.results.digitalDacI[uAntenna] = uDigitalLOCancelI;
        Calibration_params_otf.TxLoParams.results.digitalDacQ[uAntenna] = uDigitalLOCancelQ;

        //set hardware
        PhyCalDrv_SetDigitalLoCancelPair(uAntenna, uDigitalLOCancelI, uDigitalLOCancelQ);
    
        //get TPC info with number of calibrated TPC's currently being used
        RficDriver_TxLoGetTpcInfo(uAntenna, &tpcInfo);            
        //Read number of calibrated TPC's from stored buffer
        numCalibTpcs = ShramClbrDataBufferStoreChanData.txLoData.numOfCalibTpcs[uAntenna];
    
        /* We assume that the loaded data was calibrated with the same calibrated TPC's
               as used in the running software:
               verify that the number of calibrated TPC's while storing the data is equal to the number 
               of calibrated TPC's used in the current running software*/
        DEBUG_ASSERT(tpcInfo.numCalibTpcs == numCalibTpcs);
    
        for(tpcIndex = 0; tpcIndex < numCalibTpcs; tpcIndex++)
        {
            //Update internal variables
            Calibration_params_otf.TxLoParams.results.dacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_I] = ShramClbrDataBufferStoreChanData.txLoData.loDacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_I];
            Calibration_params_otf.TxLoParams.results.dacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_Q] = ShramClbrDataBufferStoreChanData.txLoData.loDacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_Q];    
            //set hardware
            RficDriver_SetLoDACs(uAntenna, (RficTxGainTpcIndex_e)tpcIndex, Calibration_params_otf.TxLoParams.results.dacs[uAntenna][tpcIndex]);
        }
    
        TxLoLeakage_SetStatus(uAntenna, ShramClbrDataBufferStoreChanData.txLoData.calStatus[uAntenna]);
    }
}

//************************************
// Method:      TxLoLeakage_StoreResults
// Purpose:     Saves TXIQ calibration data to a buffer
// Parameter:   None
// Returns:     None
//************************************
static void TxLoLeakage_StoreResults(void)
{
	int8 uDigitalLOCancelI,uDigitalLOCancelQ;
	int8 uIQ[RFIC_PATH_TYPE_MAX];
	uint8 tpcIndex;
    Antenna_e uAntenna;
	CalStatus_e antStatus;
	bool CalState = TRUE;
	RficDriverTxLoTpcInfo_t tpcInfo;

	for( uAntenna = ANTENNA_0; uAntenna < MAX_NUM_OF_ANTENNAS; uAntenna++) 
	{
		antStatus = TxLoLeakage_GetStatus(uAntenna);
		
		ShramClbrDataBufferStoreChanData.txLoData.calStatus[uAntenna] = TxLoLeakage_GetStatus(uAntenna);
		
		if ((antStatus == CAL_STATUS_FAIL) || (antStatus == CAL_STATUS_IN_PROCESS))
		{
			CalState = FALSE;
		}	
		

		PhyCalDrv_GetDigitalLoCancelPair( uAntenna, &uDigitalLOCancelI, &uDigitalLOCancelQ );

		ShramClbrDataBufferStoreChanData.txLoData.digitalLoCancelPair[uAntenna][HDK_PATH_TYPE_I] = uDigitalLOCancelI;
		ShramClbrDataBufferStoreChanData.txLoData.digitalLoCancelPair[uAntenna][HDK_PATH_TYPE_Q] = uDigitalLOCancelQ;

#if LM_CALIBRATION_LOGS_ON
		ILOG9_DDD("TxLoLeakage_StoreResults(): Ant %d Digital DAC: I=%d Q=%d", uAntenna, uDigitalLOCancelI, uDigitalLOCancelQ);
#endif

		//get TPC info with number of calibrated TPC's
		RficDriver_TxLoGetTpcInfo(uAntenna, &tpcInfo);
		ShramClbrDataBufferStoreChanData.txLoData.numOfCalibTpcs[uAntenna] = tpcInfo.numCalibTpcs;

#if LM_CALIBRATION_LOGS_ON
		ILOG9_DD("TxLoLeakage_StoreResults(): Ant %d Num Calibrated TPCs=%d", uAntenna, tpcInfo.numCalibTpcs);
#endif

		for(tpcIndex = 0; tpcIndex < tpcInfo.numCalibTpcs ; tpcIndex++)
		{
			RficDriver_GetLoDACs(uAntenna, (RficTxGainTpcIndex_e)tpcIndex, uIQ);

			ShramClbrDataBufferStoreChanData.txLoData.loDacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_I] = uIQ[RFIC_PATH_TYPE_I];
			ShramClbrDataBufferStoreChanData.txLoData.loDacs[uAntenna][tpcIndex][RFIC_PATH_TYPE_Q] = uIQ[RFIC_PATH_TYPE_Q];

#if LM_CALIBRATION_LOGS_ON
			ILOG9_DDDD("TxLoLeakage_StoreResults(): Ant %d TPC %d DAC: I=%d Q=%d", uAntenna, tpcIndex, uIQ[RFIC_PATH_TYPE_I], uIQ[RFIC_PATH_TYPE_Q]);
#endif
		}
	}
	ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_TX_LO_LEAKAGE, CalState);    
}


void TxLoLeakage_SetStatus( IN uint32 inAntenna, IN CalStatus_e inStatus )
{
	Calibration_params_otf.TxLoParams.results.status[inAntenna] = inStatus;
}

CalStatus_e TxLoLeakage_GetStatus( IN uint32 inAntenna )
{
	return Calibration_params_otf.TxLoParams.results.status[inAntenna];
}

uint32 TxLoLeakage_GetLastRunTime(void)
{
	return Calibration_params_otf.TxLoParams.results.lastRunTime;
}
static void TxLoLeakage_ResetOfflineStatus(void)
{
	uint8 i=0;

	for(i=0;i< MAX_NUM_OF_ANTENNAS ;i++)
	{
		TxLoLeakage_SetStatus(i, CAL_STATUS_INIT);
	}
}




