#include "System_GlobalDefinitions.h"
#include "OSAL_Api.h"
#include <protocol.h>
#include <enet_pas.h>
#include <System_Configuration.h>
#include <PhyDriver_API.h>

#include "System_GlobalDefinitions.h"
#include "Afe_API.h"

#include <lmi.h>
#include "RegAccess_Api.h"
#include "CalibrationHandler.h"
#include "CalibrationHandlerDB.h"
#include "CalibrationHandlerUtils.h"
#include <PhyCalDriver_API.h>
#include "TxLOLeakageClbrHndlr.h"
#include "CalibrationsDefines.h"
#include <mt_sysrst.h>
#include "loggerAPI.h"
#include "RficDriver_API.h"
#include "RFICDefines.h"
#include "MT_Math.h"
#include "TpcClbrHndlr.h"
#include "ErrorHandler_Api.h"
#include "Hdk_Api.h"
#include "Shram_ClbrDataBuffer.h"
#include "LmHdk_API.h"
#include "Utils_Api.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 21

/******************************************************************************/
/***						Type Definitions							    ***/
/******************************************************************************/	
typedef struct TxLo_DacIndexes
{
	int16 iDig[MAX_NUM_OF_ANTENNAS];
	int16 qDig[MAX_NUM_OF_ANTENNAS];
	int8 i[MAX_NUM_OF_ANTENNAS];
	int8 q[MAX_NUM_OF_ANTENNAS];
}TxLo_DacIndexes_t;

typedef struct TxLo_StoreParam
{
	TxLo_DacIndexes_t dacIndexes;
	uint32	txScale;
	int16	digGain[MAX_NUM_OF_ANTENNAS];
	uint8	difi2Gain[MAX_NUM_OF_ANTENNAS];
}
TxLo_StoreParam_t;

typedef struct TxLo_Params
{
	uint32 lastRunTime;
	CalStatus_e status[MAX_NUM_OF_ANTENNAS];	
}TxLo_Params_t;

typedef struct TxLo_CalParam
{
	int32 startindex;
	uint32 incremental;
	uint32 numberOfSteps;
}TxLo_CalParam_t;


/******************************************************************************/
/***						static Variables								***/
/******************************************************************************/
TxLo_CalParam_t TxLoCalibStepArray[TX_CALIB_STEP_NUM] = {
		{TX_CALIB_STEP_1_START_OFFSET, TX_CALIB_STEP_1_JUMP_SIZE, TX_CALIB_STEP_1_JUMPS_NUM},
		{TX_CALIB_STEP_2_START_OFFSET, TX_CALIB_STEP_2_JUMP_SIZE, TX_CALIB_STEP_2_JUMPS_NUM},
		{TX_CALIB_STEP_3_START_OFFSET, TX_CALIB_STEP_3_JUMP_SIZE, TX_CALIB_STEP_3_JUMPS_NUM},
		{TX_CALIB_STEP_4_START_OFFSET, TX_CALIB_STEP_4_JUMP_SIZE, TX_CALIB_STEP_4_JUMPS_NUM} };



/******************************************************************************/
/***					Private Function Declarations						***/
/******************************************************************************/
static RetVal_e run(IN ClbrType_e inCalibType);
// -----------------------------------  configure ---------------------------------------------- //
static void configureTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam);
static void storeDacIndexesBaforeTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam);
static void configurePhyForTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam);
static void configureRficForTxLOCalibration(uint8 antMask);
// -----------------------------------   restore  ---------------------------------------------- //
static void restoreFromTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam);
static void handleFailure(uint8 antMask, TxLo_StoreParam_t* storedParam);
static void restoreRficFromTxLOCalibration(uint8 antMask);
static void restorePhyFromTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam);
// -----------------------------------  calibrate ---------------------------------------------- //
static void verifyTxLoCalibrationResults(uint8* failedAntMask, TxLo_DacIndexes_t* dacIndexes);
static RetVal_e calibrateTxLO(ClbrType_e inCalibType, uint8 antMask);
static void calcOptDigIndexes(uint8 antMask, TxLo_DacIndexes_t* dacIndexes, bool isPartialCal);
static void calcBestTxLOCoeff(uint8 antMask, TxLo_DacIndexes_t* txloCoeffs, int16* calculatedCoeff, TxLo_CalParam_t stepDescriptor);
static void setTxLoAnalogDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes);
static void setTxLoDigitalDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes);
static void getTxLoAnalogAndDigitalDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes);


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

TxLo_Params_t txLoParams = {0};
bool TxLoFirstTime = TRUE;

////////////////////////
// Methods
////////////////////////

//************************************
// Method:    run
// FullName:  run
// Access:    public static 
// Returns:   RetVal_e
// Qualifier:
// Parameter: IN ClbrType_e inCalibType
// Parameter: IN uint8 * inTpcMaxPowerIndexArray
// Parameter: TxLO_elements_t * outElements_p
//************************************
static RetVal_e run(IN ClbrType_e inCalibType)
{
	RetVal_e RetFailOk = RET_VAL_SUCCESS;
	CalDataState_e calStoreState;
	uint8 antMask, maxAnts;
	TxLo_StoreParam_t storeParam = {0};
	uint32 startTime = TIME_STAMP(START_TIME,0);
	
#ifdef WRX600_BU_LOGS
	ILOG0_D("Run TxLO calibration. CalibType %d", inCalibType);
#endif	
	HDK_GetMaxActiveAntennasNumAndMask(&maxAnts, &antMask);

	/* Configuration */
	configureTxLOCalibration(antMask, &storeParam);

	/* Calibrate */	
	RetFailOk = calibrateTxLO(inCalibType, antMask);

	/* Restore  */
	restoreFromTxLOCalibration(antMask, &storeParam);
	
    /* Store calibration results */
	ClbrHndlr_GetCalibrationDataState(&calStoreState);
	if(calStoreState == CAL_DATA_STATE_STORE)
	{
		ClbrHndlr_UpdateCalState(CLBR_PROC_TYPE_TX_LO_LEAKAGE, TRUE); 
	}
	
	txLoParams.lastRunTime = TIME_STAMP(END_TIME,startTime);
#ifdef WRX600_BU_LOGS
	ILOG0_D("TxLO Calibration done. elapsed time = %d usec",txLoParams.lastRunTime);
#endif
	return (RetFailOk);
}

static void configureTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam)
{
	/* store dacIndexes */
	storeDacIndexesBaforeTxLOCalibration(antMask, storedParam);
	
	/* PHY configuration */
	configurePhyForTxLOCalibration(antMask, storedParam);
	
	/* RF configuration */
	configureRficForTxLOCalibration(antMask);
}

static void storeDacIndexesBaforeTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam)
{	
	if (TRUE == TxLoFirstTime)
	{
	    TxLoFirstTime = FALSE;
		return;
	}

	else
	{
		getTxLoAnalogAndDigitalDacs(antMask, &(storedParam->dacIndexes));
	}
}

static void configureRficForTxLOCalibration(uint8 antMask)
{	
	uint8 band = HDK_getBand();
			
#ifdef WRX600_BU_LOGS
	ILOG0_D("configureRficForTxLOCalibration antMask 0x%x", antMask);
#endif
	RficDriver_ActivateTxAntennas(antMask); //turn the bbic_tx off & then on again

	//configure rx gains to initial gain and lna, since it is configured to lowest pgc index
	RficDriver_SetRxGains(antMask, TX_INITIAL_LNA_INDEX, TX_INITIAL_PGC_GAIN_DB);
	RficDriver_SetPgcbwidth (antMask, TX_PGC_BW);

	RficDriver_DeactivateTxAntennas(antMask);

	// Configure SSB loop
	RficDriver_SetLoopback(antMask, RFIC_LOOPBACK_TYPE_TX_AMDET);

	if (band == BAND_5_2_GHZ)
	{
		RficDriver_SetTxGain(antMask, TX_INITIAL_TPC_INDEX, TX_PAD_GAIN_DB_5G);
	}
	else
	{
		RficDriver_SetTxGain(antMask, TX_INITIAL_TPC_INDEX, TX_PAD_GAIN_DB_2_4G);
	}
}


static void configurePhyForTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam)
{
	uint32 goertzelLength[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_LENGTH, TX_GRTZL_LENGTH, TX_GRTZL_LENGTH, TX_GRTZL_LENGTH};	
	uint8 goertzelCycles[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES, TX_GRTZL_NUM_OF_CYCLES};
	int16 goertzelBin[MAX_NUM_OF_ANTENNAS] = {LO_CALIB_BIN, LO_CALIB_BIN, LO_CALIB_BIN, LO_CALIB_BIN};
	uint8 difi2Gain[MAX_NUM_OF_ANTENNAS] = {TX_DIFI_2_GAIN, TX_DIFI_2_GAIN, TX_DIFI_2_GAIN, TX_DIFI_2_GAIN};
	int16 digGain[MAX_NUM_OF_ANTENNAS] = {TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN, TX_GRTZL_TONE_DIGGAIN};

#ifdef WRX600_BU_LOGS
	ILOG0_D("configurePhyForTxLOCalibration antMask 0x%x", antMask);
#endif
	storedParam->txScale = PhyCalDrv_GetToneGenScale();
	PhyCalDrv_GetTxDigitalGain(antMask,storedParam->digGain);

	//reset rx td
	PhyCalDrv_resetRxtd(TRUE);
	PhyCalDrv_resetRxtd(FALSE);

	AFE_RxOn(antMask);
	AFE_TxOff(antMask);
	AFE_TxOn(antMask);
	AFE_AuxadcOff(antMask);

	/* Store current difi 2 gain, use RXDC_DIFI_2_GAIN for calibration */
	PhyCalDrv_getDifi2Gain(antMask, storedParam->difi2Gain);
	PhyCalDrv_setDifi2Gain(antMask, difi2Gain);

	PhyCalDrv_enableDifiGain(antMask, FALSE);
	PhyCalDrv_setDcCancellationMode(antMask, FALSE);

	PhyCalDrv_SetPgc2Gain(antMask, TX_INITIAL_TPC_INDEX);
	
	// Configure and start Tone Generator
	PhyCalDrv_EnableGclkBypass(TRUE);
	PhyCalDrv_RunToneGen(antMask, LO_CALIB_BIN, TX_GRTZL_TONE_SCALE, digGain);

	// Configure Goertzel
	PhyCalDrv_GoertzelConfig(antMask, goertzelLength, goertzelCycles, goertzelBin, goertzelBin);
}

static void restoreFromTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam)
{
	/* Restore dac Indexes for failed antennas */
	handleFailure(antMask, storedParam);

	/* Restore RF configuration */
	restoreRficFromTxLOCalibration(antMask);

	/* Restore PHY configuration */
	restorePhyFromTxLOCalibration(antMask, storedParam);
}

static void handleFailure(uint8 antMask, TxLo_StoreParam_t* storedParam)
{
	uint8 ant, FailedAntMask = 0;

	/* get status of antennas */
    for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {		
		if ((antMask & (1 << ant)) && ( CAL_STATUS_FAIL == TxLoLeakage_GetStatus(ant)))
		{
			FailedAntMask |= (1 << ant);
		}
    }

	/* fix failed antennas */
	if (0 != FailedAntMask)
	{
		setTxLoDigitalDacs(FailedAntMask, &(storedParam->dacIndexes));
	}
}


static void restoreRficFromTxLOCalibration(uint8 antMask)
{
	// Restore SSB loop
	RFIC_RestoreFromLoopBack(antMask);
}

static void restorePhyFromTxLOCalibration(uint8 antMask, TxLo_StoreParam_t* storedParam)
{
	PhyCalDrv_enableDifiGain(antMask, TRUE);
	PhyCalDrv_setDcCancellationMode(antMask, TRUE);

	// AFE loopback restore	
	AFE_RxOff(antMask);
	AFE_TxOff(antMask);

	// Clear and disable Tone Gen
	PhyCalDrv_EnableGclkBypass(FALSE);
	PhyCalDrv_ToneGenClearAndDisable(antMask);

	// Restore difi2 gain
	PhyCalDrv_setDifi2Gain(antMask, storedParam->difi2Gain);
	PhyCalDrv_SetToneGenScale(antMask, storedParam->txScale);
	PhyCalDrv_SetDigitalGain(antMask, storedParam->digGain);
}

static void verifyTxLoCalibrationResults(uint8* failedAntMask, TxLo_DacIndexes_t *dacIndexes)
{
	uint32 txEnergy[MAX_NUM_OF_ANTENNAS];
	uint8 ant;

	setTxLoDigitalDacs(*failedAntMask, dacIndexes);
	PhyCalDrv_GoertzelMeasureEnergy(*failedAntMask, txEnergy, TX_GRTZL_ENERGY_RESOLUTION, TX_GRTZL_TIMEOUT);

	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)//verification
	{
		if (*failedAntMask & (1 << ant))
		{
#ifdef WRX600_BU_LOGS
			ILOG0_DDDD("verifyTxLoCalibrationResults ant %d test q:%d, i:%d, energy:%d",ant, dacIndexes->iDig[ant],dacIndexes->qDig[ant], txEnergy[ant]);
#endif	
			if (txEnergy[ant] < TXLO_ENERGY_MAX_VALUE)
			{//pass calibration
				*failedAntMask &= (~(1 << ant));
			}
#ifdef WRX600_BU_LOGS
			else
			{
				ILOG0_DDDD("ant %d failed in TxLoverification test q:%d, i:%d, energy:%d",ant, dacIndexes->iDig[ant],dacIndexes->qDig[ant], txEnergy[ant]);
			}
#endif	
		}
	}
}

static RetVal_e calibrateTxLO(ClbrType_e inCalibType, uint8 antMask)
{
	uint8 failedAntMask = antMask;
	uint8 tryCounter = 0, maxTryCounter = 0;
	TxLo_DacIndexes_t dacIndexes = {0};
	bool isPartialCal = FALSE;

	// init dac
	setTxLoAnalogDacs(antMask, &dacIndexes);

	if (inCalibType == CLBR_TYPE_OFFLINE)
	{		
		maxTryCounter = TXLO_MAX_TRIES_OFFLINE;
		setTxLoDigitalDacs(antMask, &dacIndexes);
	}
	
	else
	{
		ASSERT(inCalibType == CLBR_TYPE_ONLINE);
		
		maxTryCounter = TXLO_MAX_TRIES_ONLINE;
		isPartialCal = TRUE;
		getTxLoAnalogAndDigitalDacs(antMask, &dacIndexes);		
	}

	for (tryCounter = 0; (failedAntMask != 0) && (tryCounter < maxTryCounter) ; tryCounter++)
	{
#ifdef WRX600_BU_LOGS
		ILOG0_DD("failedAntMask 0x%x, tryCounter %d",failedAntMask, tryCounter);
#endif
		calcOptDigIndexes(failedAntMask, &dacIndexes, isPartialCal);		
		verifyTxLoCalibrationResults(&failedAntMask, &dacIndexes);
	}

	// handl end of calibration
	TxLoLeakage_SetStatus(antMask & failedAntMask, CAL_STATUS_FAIL);
	TxLoLeakage_SetStatus(antMask & (~failedAntMask), CAL_STATUS_PASS);

	return RET_VAL_SUCCESS;
}

static void calcOptDigIndexes(uint8 antMask, TxLo_DacIndexes_t* dacIndexes, bool isPartialCal)
{
	uint8 step = 0;

	/* online calibration skips the first step */
	if (TRUE == isPartialCal)
	{
		step++;
	}

	for( ; step < TX_CALIB_STEP_NUM ; step++)
	{
		/* Calculate minimum value for i coefficient */
		calcBestTxLOCoeff(antMask, dacIndexes, dacIndexes->iDig, TxLoCalibStepArray[step]);
	    /* Calculate minimum value for q coefficient */
		calcBestTxLOCoeff(antMask, dacIndexes, dacIndexes->qDig, TxLoCalibStepArray[step]);	 
	}
}

static void calcBestTxLOCoeff(uint8 antMask, TxLo_DacIndexes_t* txloCoeffs, int16* calculatedCoeff, TxLo_CalParam_t stepDescriptor)
{
    uint32 txEnergy[MAX_NUM_OF_ANTENNAS];
	int32 startIndexArray[MAX_NUM_OF_ANTENNAS];
	int32 MinimumIndex[MAX_NUM_OF_ANTENNAS];
	uint32 MinimumEnergy[MAX_NUM_OF_ANTENNAS];
    uint8 index, ant;	
	
#ifdef WRX600_TXLO_ONLINE_CALB_LOGS
	int32 StartCalTime, EndCalTime;
	StartCalTime = TIME_STAMP(START_TIME,0);
#endif
	// init arrays
	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		startIndexArray[ant] = calculatedCoeff[ant] + stepDescriptor.startindex;
		MinimumIndex[ant] = 0;
		MinimumEnergy[ant] = (uint32)-1;
	}

	for(index = 0; index < stepDescriptor.numberOfSteps; index++)
	{
		for(ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (antMask & (1 << ant))
			{					
				calculatedCoeff[ant] = startIndexArray[ant] + (stepDescriptor.incremental)*index;
#ifdef WRX600_BU_LOGS
				ILOG0_DDD("txloCoeffs->iDig[%d]:%d, qDig:%d",ant,txloCoeffs->iDig[ant], txloCoeffs->qDig[ant]);
#endif
			}
		}
		
		setTxLoDigitalDacs(antMask, txloCoeffs);
		PhyCalDrv_GoertzelMeasureEnergy(antMask, txEnergy, TX_GRTZL_ENERGY_RESOLUTION, TX_GRTZL_TIMEOUT);

		for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (antMask & (1 << ant))
			{					
				if(txEnergy[ant] < MinimumEnergy[ant])
				{
					 MinimumEnergy[ant] = txEnergy[ant];
					 MinimumIndex[ant] = calculatedCoeff[ant];
#ifdef WRX600_BU_LOGS
				ILOG0_DDD("minimum choosen ant :%d, MinimumEnergy[ant]:%d,MinimumIndex[ant]:%d",ant,MinimumEnergy[ant], MinimumIndex[ant]);
#endif
				}
			}
		}
	}

	for(ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{					
				calculatedCoeff[ant] = MinimumIndex[ant];
#ifdef WRX600_BU_LOGS		
				ILOG0_DDD("txloCoeffs->iDig[%d]:%d, qDig:%d",ant,txloCoeffs->iDig[ant], txloCoeffs->qDig[ant]);
#endif
		}
	}
	
#ifdef WRX600_TXLO_ONLINE_CALB_LOGS	
	EndCalTime = TIME_STAMP(END_TIME,StartCalTime);
	ILOG0_DDD("calcBestTxLoCoeff-> mask:%x  steps:%d, time: %d",antMask, stepDescriptor.numberOfSteps, EndCalTime);
#endif
}

static void setTxLoAnalogDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes)
{
	uint8 ant;
#ifdef WRX600_BU_LOGS
	ILOG0_D("setTxLoAnalogDacs. antMask = 0x%x", antMask);
	SLOG0(0, 0, TxLo_DacIndexes_t, dacIndexes);
#endif	
	
	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{
			RficDriver_SetLoDACs(1 << ant, dacIndexes->i[ant], dacIndexes->q[ant]);
		}
	}
}

static void setTxLoDigitalDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes)
{
#ifdef WRX600_BU_LOGS
	ILOG0_D("setTxLoDigitalDacs. antMask = 0x%x", antMask);
	SLOG0(0, 0, TxLo_DacIndexes_t, dacIndexes);
#endif	
	
	PhyCalDrv_SetDigitalLoCancelPair(antMask, dacIndexes->iDig, dacIndexes->qDig, TX_INITIAL_TPC_INDEX);
}


 
static void getTxLoAnalogAndDigitalDacs(uint8 antMask, TxLo_DacIndexes_t* dacIndexes)
{
#ifdef WRX600_BU_LOGS
	ILOG0_DD("getTxLoDacsPerGain. antMask = 0x%x tpcIndex = %d",antMask, TX_INITIAL_TPC_INDEX);
#endif	
	RficDriver_GetLoDACs(antMask, TX_INITIAL_TPC_INDEX, dacIndexes->i, dacIndexes->q);
	PhyCalDrv_GetDigitalLoCancelPair(antMask, dacIndexes->iDig, dacIndexes->qDig, TX_INITIAL_TPC_INDEX);
#ifdef WRX600_BU_LOGS
	SLOG0(0, 0, TxLo_DacIndexes_t, dacIndexes);
#endif	
}

/******************************************************************************/
/***							Public Functions							***/
/******************************************************************************/

RetVal_e TxLoLeakage_Calibrate(IN ClbrType_e inCalibType)
{
	return run(inCalibType);
}

void TxLoLeakage_SetStatus(uint8 antMask, CalStatus_e inStatus)
{
	uint8 ant;
	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		if (antMask & (1 << ant))
		{
			txLoParams.status[ant] = inStatus;
		}
	}
}

CalStatus_e TxLoLeakage_GetStatus( IN uint32 inAntenna )
{
	return (txLoParams.status[inAntenna]);
}

uint32 TxLoLeakage_GetLastRunTime(void)
{
	return (txLoParams.lastRunTime);
}
