#ifndef __CalibrationUtils_h__
#define __CalibrationUtils_h__

#include "System_GlobalDefinitions.h"
#include "HdkGlobalDefs.h"
#include "CalibrationDefs.h"
#include "RficDriver_API.h"

/* Calibrations store and load results definitions */
#ifndef ENET_INC_ARCH_WAVE600
#define RX_DC_NUM_OF_PGC1_VALUES            (1 + (PGC1_MAX_VALUE - PGC1_MIN_VALUE) / PGC1_STEP)
#define RX_DC_NUM_OF_PGC2_VALUES            (1 + (PGC2_MAX_VALUE - PGC2_MIN_VALUE) / PGC2_STEP)
#endif
#define RSSI_PATH_MAX_MEASUREMENTS          (20)    //Moved here from RssiPathClbrHndlr.c


/* Antenna bit mask manipulations */
#define ANTENNA_BIT(antNum)							(1<<(antNum))						// Antenna bit
#define ANTENNA_BIT_ALL								(0x7)								// Antenna bit on for all 3 antennas
#define ANTENNA_MASK_ADD_BIT(antMask,antNum)		((antMask) |= ANTENNA_BIT(antNum))	// Add antenna to antenna bit mask 
#define ANTENNA_MASK_IS_BIT_ON(antMask,antNum)		((antMask) & ANTENNA_BIT(antNum))	// Is antenna bit on in antenna mask

/* Is given AntNum enabled for receive */
#define RX_ANTENNA_ENABLED(rxAntMask,antNum)		ANTENNA_MASK_IS_BIT_ON(rxAntMask,antNum)
/* Is given AntNum enabled for transmit */
#define TX_ANTENNA_ENABLED(txAntMask,antNum)		ANTENNA_MASK_IS_BIT_ON(txAntMask,antNum)

#define NEXT_ENABLED_ANTENNA_SEARCH       (1)
#define PREVIOUS_ENABLED_ANTENNA_SEARCH   (-1)

// dB adjustments
#define	REDUCE_BY_0_3_dB(x)		(((x)>>1)+((x)>>2)+((x)>>3)+((x)>>4))           // -0.3dB = 0.93 =~ 1/2+1/4+1/8+1/16 = 0.9375
#define	REDUCE_BY_0_6_dB(x)		(((x)>>1)+((x)>>2)+((x)>>3))                    // -0.6dB = 0.87 =~ 1/2+1/4+1/8 = 0.875
#define REDUCE_BY_0_81_dB(x)	(((x)>>1)+((x)>>2)+((x)>>4)+((x)>>6)+((x)>>9))  // -0.81dB =  multiply in 0.83 and 1/2+1/4+1/16+1/32 =~ 0.83
#define REDUCE_BY_0_9_dB(x)		(((x)>>1)+((x)>>2)+((x)>>4))                    // -0.9dB = 0.81 =~ 1/2+1/4+1/16 = 0.8125
#define REDUCE_BY_1_dB(x)		(((x)>>1)+((x)>>2)+((x)>>5)+((x)>>6))           // -1dB =  multiply in 0.8 and 1/2+1/4+1/32+1/64 =~ 0.8
#define REDUCE_BY_1_25_dB(x)	(((x)>>1)+((x)>>2))                             // -1.25dB = multiply in 0.75 and 1/2+1/4 = 0.75
#define REDUCE_BY_1_8_dB(x)		(((x)>>1)+((x)>>3)+((x)>>5))                    // -1.8dB = multiply in 0.66 and 1/2+1/8+1/32 =~ 0.656
#define REDUCE_BY_1_9_dB(x)		(((x)>>1)+((x)>>3)+((x)>>6)+((x)>>8))           // -1.9dB = multiply in 0.6456 and 1/2+1/8+1/64+1/256 =~ 0.6445
#define REDUCE_BY_2_05_dB(x)	(((x)>>1)+((x)>>3))                             // -2.05dB =  multiply in 0.6237 and 1/2+1/8 =~ 0.623
#define REDUCE_BY_3_dB(x)		((x)>>1)                                        // -3dB =  multiply in 0.5
#define	REDUCE_BY_3_6_dB(x)		(((x)>>2)+((x)>>3)+((x)>>4))                    // -3.6dB = 0.436 =~ 1/4+1/8+1/16 = 0.4375
#define REDUCE_BY_3_92_dB(x)	(((x)>>2)+((x)>>3)+((x)>>5))                    // -3.92dB =  multiply in 0.4 and 1/4+1/8+1/32 =~ 0.4
#define REDUCE_BY_8_5_dB(x)		(((x)>>3)+((x)>>6))                             // -8.5dB = 0.141 =~ 1/8+ 1/64 = 0.1406
#define REDUCE_BY_10_dB(x)		(((x)>>4)+((x)>>5)+((x)>>8)+((x)>>9)+((x)>>10)) // -10dB = multiply by 0.1 =~ 1/16+1/32+1/256+1/512+1/1024 =~ 0.1
#define REDUCE_BY_14_dB(x)		(((x)>>5)+((x)>>7)+((x)>>10))                   // -14dB = multiply by 0.0398 =~ 1/32+1/128+1/1024 =~ 0.04
#define REDUCE_BY_15_dB(x)		(((x)>>5)+((x)>>12))                            // -15dB = multiply in 0.1 and 1/32+1/4096 =~ 0.031
#define REDUCE_BY_21_dB(x)		(((x)>>7)+((x)>>13)+((x)>>14))                  // -21dB = 0.00794 =~ 1/(2^7)+1/(2^13)+1/(2^14) = 0.008
#define REDUCE_BY_24_dB(x)		(REDUCE_BY_25_dB(x))                            // Ariel - place holder
#define REDUCE_BY_25_dB(x)		(((x)>>9)+((x)>>10)+((x)>>12))                  // -25dB = multiply in 0.0032 and 1/512+1/1024+1/4096 =~ 0.0032
#define REDUCE_BY_26_dB(x)		(REDUCE_BY_25_dB(x))                            // Ariel - place holder
#define REDUCE_BY_35_dB(x)		(((x)>>12)+((x)>>14)+((x)>>17)+((x)>>19)+((x)>>20)+((x)>>21)) // -35dB = multiply by 0.000316 =~ 1/(2^12)+1/(2^14)+1/(2^17)+1/(2^19)+1/(2^20)+1/(2^21) =~ 0.000316
#define REDUCE_BY_40_dB(x)		(((x)>>14)+((x)>>15)+((x)>>16))                 // -40dB = 0.0001 =~ 1/(2^14)+1/(2^15)+1/(2^16) = 0.000107

#define INCREASE_BY_0_6_dB(x)	((x) + ((x)>>3)+((x)>>6)+((x)>>8)+((x)>>9)+((x)>>10)) // +0.6dB = multiply by 1.148 =~ 1+1/8+1/64+1/256+1/512+1/1024=1.14746
#define INCREASE_BY_1_dB(x)		((x) + ((x)>>2))                        // +1dB = multiply in 1.25
#define INCREASE_BY_2_0_5_dB(x) ((x) + ((x)>>1)+((x)>>4)+((x)>>5))      // +2.05dB =~ multiply by 1.6 (1/0.625)
#define INCREASE_BY_5_dB(x)		(((x)<<1)+x+((x)>>3)+((x)>>5)+((x)>>8)) // +5dB = multiply in 3.162 and(2+1+1/8+1/32+1/256+1/512) = 3.162
#define INCREASE_BY_6_dB(x)		((x)<<2)                                // +6dB = multiply by 3.98

#define NUM_OF_GAINS_FOR_DCTBlCHIPMODE 5

#ifndef ENET_INC_ARCH_WAVE600
/* TBD - gen5 should be defined same as gen6.  
Should be changed after all gen6 calibrations will be ready */

typedef struct DcTblChipMode
{
	uint16 gain[NUM_OF_GAINS_FOR_DCTBlCHIPMODE][TOTAL_ANTENNAS];
	uint16 pgc2rxdc0_radar[TOTAL_ANTENNAS];
	uint16 pgc2rxdc1[TOTAL_ANTENNAS];
	uint16 pgc2rxdc2[TOTAL_ANTENNAS];
	uint16 mixDc[TOTAL_ANTENNAS];
} DcTblChipMode_t;

typedef struct RxDcStoreChanData
{
	uint32 DcDigCorrectionTable[TOTAL_ANTENNAS][RXDC_TABLE1_CORRECTION_LEN];
    CalStatus_e calStatus[TOTAL_ANTENNAS][HDK_PATH_TYPE_MAX];
	//for chip mode
	DcTblChipMode_t dcTblChipMode;
    int8        dcTbl1Entry[TOTAL_ANTENNAS][HDK_LNA_INDEX_NUM_OF_INDEXES][RX_DC_NUM_OF_PGC1_VALUES][HDK_PATH_TYPE_MAX];
    int8        dcTbl1CorrectionEntry[TOTAL_ANTENNAS][HDK_LNA_INDEX_NUM_OF_INDEXES][RX_DC_NUM_OF_PGC1_VALUES][HDK_PATH_TYPE_MAX];
    int8        dcTbl2Entry[TOTAL_ANTENNAS][RX_DC_NUM_OF_PGC2_VALUES][HDK_PATH_TYPE_MAX];
} RxDcStoreChanData_t;

typedef struct LpfStoreChanData
{
    uint8       lpfWord[TOTAL_ANTENNAS][HDK_PATH_TYPE_MAX];
    CalStatus_e calStatus[TOTAL_ANTENNAS];
} LpfStoreChanData_t;

typedef struct RssiPathStoreChanData
{
    int16       a[TOTAL_ANTENNAS][RSSI_PATH_MAX_MEASUREMENTS];
    uint8       rssi[TOTAL_ANTENNAS][RSSI_PATH_MAX_MEASUREMENTS];
    uint8       section[TOTAL_ANTENNAS][RSSI_PATH_MAX_MEASUREMENTS];
    CalStatus_e calStatus[TOTAL_ANTENNAS];    
} RssiPathStoreChanData_t;

typedef struct TxLoStoreChanData
{
    int8        loDacs[MAX_NUM_OF_ANTENNAS][RFIC_TX_GAIN_TPC_INDEX_MAX][HDK_PATH_TYPE_MAX];
    int8        digitalLoCancelPair[MAX_NUM_OF_ANTENNAS][HDK_PATH_TYPE_MAX];
    uint8       numOfCalibTpcs[MAX_NUM_OF_ANTENNAS];
    CalStatus_e calStatus[MAX_NUM_OF_ANTENNAS];
} TxLoStoreChanData_t;

typedef struct TxIqStoreChanData
{
    int16       a[MAX_NUM_OF_ANTENNAS][TX_IQ_MAX_CALIBRATED_BINS];
    int16       b[MAX_NUM_OF_ANTENNAS][TX_IQ_MAX_CALIBRATED_BINS];
	int16		SSB_F_20M[MAX_NUM_OF_ANTENNAS][2][2];
	int16		SSB_F_40M[MAX_NUM_OF_ANTENNAS][2][2];
    CalStatus_e calStatus[MAX_NUM_OF_ANTENNAS];
    uint8       cbMode;
} TxIqStoreChanData_t;

typedef struct RxIqStoreChanData
{
    int16       w1[MAX_NUM_OF_ANTENNAS];
    int16       w2[MAX_NUM_OF_ANTENNAS];
    CalStatus_e calStatus[MAX_NUM_OF_ANTENNAS];
} RxIqStoreChanData_t;

typedef struct MsSsiAdcStoreChanData
{
    uint8       gain[TOTAL_ANTENNAS];
    uint8       offset[TOTAL_ANTENNAS];
    CalStatus_e calStatus[TOTAL_ANTENNAS];
} MsSsiAdcStoreChanData_t;

typedef struct RficBiasStoreChanData
{
    uint8       Rint;
    CalStatus_e calStatus;
} RficBiasStoreChanData_t;

typedef struct RssiDcStoreChanData
{
    uint8       rssiDcOffsets[TOTAL_ANTENNAS][RFIC_RSSI_STG_TOTAL];
    CalStatus_e calStatus[TOTAL_ANTENNAS];
} RssiDcStoreChanData_t;

typedef struct ClbrStoreChanData
{
	
    RxDcStoreChanData_t     rxDcData;
    LpfStoreChanData_t      lpfData;
    RssiPathStoreChanData_t rssiPathData;
    TxLoStoreChanData_t     txLoData;
    TxIqStoreChanData_t     txIqData;
    RxIqStoreChanData_t     rxIqData;

    MsSsiAdcStoreChanData_t msSsiAdcData;
    RficBiasStoreChanData_t rficBiasData;
    RssiDcStoreChanData_t   rssiDcData;
	uint8                   stateArray[CLBR_PROC_TYPE_NUM];    
} ClbrStoreChanData_t;

#else // ENET_INC_ARCH_WAVE600

typedef struct DcTblChipMode{
	uint16 gain[NUM_OF_GAINS_FOR_DCTBlCHIPMODE];
	uint16 pgc2rxdc0_radar;
	uint16 pgc2rxdc1;
} DcTblChipMode_t;

typedef struct RxDcStoreChanData
{
	uint16 dcAnalog[HDK_LNA_INDEX_NUM_OF_INDEXES];
	int16 dcI[HDK_LNA_INDEX_NUM_OF_INDEXES][RXDC_NUM_OF_PGC_GAINS];
	int16 dcQ[HDK_LNA_INDEX_NUM_OF_INDEXES][RXDC_NUM_OF_PGC_GAINS];
    CalStatus_e calStatus;
} RxDcStoreChanData_t;

typedef struct LpfStoreChanData
{
    uint8       lpfWord[HDK_PATH_TYPE_MAX];
    CalStatus_e calStatus;
} LpfStoreChanData_t;

typedef struct RssiPathStoreChanData
{
    int16       a[RSSI_PATH_MAX_MEASUREMENTS];
    uint8       rssi[RSSI_PATH_MAX_MEASUREMENTS];
    uint8       section[RSSI_PATH_MAX_MEASUREMENTS];
    CalStatus_e calStatus;    
} RssiPathStoreChanData_t;

typedef struct TxLoStoreChanData
{
    int8        loDacs[HDK_PATH_TYPE_MAX];
    int16       digitalLoCancelPair[HDK_PATH_TYPE_MAX];
    CalStatus_e calStatus;
} TxLoStoreChanData_t;

typedef struct TxIqStoreChanData
{
    int16       aNeg;
    int16       aPos;	
    int16       bNeg;
    int16       bPos;
    CalStatus_e calStatus;
} TxIqStoreChanData_t;

typedef struct RxIqStoreChanData
{
    uint16       w1;
    int16        w2;
    CalStatus_e calStatus;
} RxIqStoreChanData_t;

typedef struct MsSsiAdcStoreChanData
{
    uint8       gain;
    uint8       offset;
    CalStatus_e calStatus;
} MsSsiAdcStoreChanData_t;

typedef struct RficBiasStoreChanData
{
    uint8       Rint;
    CalStatus_e calStatus;
} RficBiasStoreChanData_t;

typedef struct RssiDcStoreChanData
{
    uint8       rssiDcOffsets[RFIC_RSSI_STG_TOTAL];
    CalStatus_e calStatus;
} RssiDcStoreChanData_t;

typedef struct ClbrStoreChanDataAnt
{
    RxDcStoreChanData_t     rxDcData;
    LpfStoreChanData_t      lpfData;
    RssiPathStoreChanData_t rssiPathData;
    TxLoStoreChanData_t     txLoData;
    TxIqStoreChanData_t     txIqData;
    RxIqStoreChanData_t     rxIqData;

    MsSsiAdcStoreChanData_t msSsiAdcData;
    RficBiasStoreChanData_t rficBiasData;
    RssiDcStoreChanData_t   rssiDcData;
} ClbrStoreChanDataAnt_t;

typedef struct ClbrStoreChanData
{
	uint8                   stateArray[CLBR_PROC_TYPE_NUM]; 
	ClbrStoreChanDataAnt_t clbrStoreChanDataPerAnt[MAX_NUM_OF_ANTENNAS];
} ClbrStoreChanData_t;

#endif


////////////////////////
// Public Members
////////////////////////

/********************************************************************************

FindTxEnabledAntenna

Description:
------------
    Find the next or previous enabled antenna.

Input:
-----
    inCurrentAntenna    - The current calibrated antenna. 
    inPreviousOrNext    - Required search type - NEXT_ENABLED_ANTENNA_SEARCH / PREVIOUS_ENABLED_ANTENNA_SEARCH

Output:
-------
    outEnabledAntenna_p - 
********************************************************************************/
void FindTxEnabledAntenna(uint8 inCurrentAntenna,int8 inPreviousOrNext,uint8 *outEnabledAntenna_p);
//************************************
// Method:    GetLSB
// Purpose:   Get the next calibration procedure indicated in the bitmap (which is assumed to be non-zero!!!)
// Parameter: uint32 x - bitmask
// Returns:   ClbrProcType_e
//************************************
ClbrProcType_e GetNextProc(uint32 x);

#endif // __CalibrationUtils_h__

