
#include "System_GlobalDefinitions.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "CalibrationManager.h"
#include "CalibrationHandler.h"
#include "HdkGlobalDefs.h"
#include "shram_man_msgs.h"
#include "init_ifmsg.h"
#include "lmi.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "mib_ms.h"
#include "CoC_Api.h"
#include "bss_manage_task.h"
#include "BSSmanager_API.h"
#include "ProcessManager_Api.h"
#include "Hdk_Api.h"
#include "DmaManager_Api.h"
#include "EventsManager_api.h"
#include "HdkTask.h"
#include "CalibrationManager.h"
#include "ErrorHandler_Api.h"
#include "Shram_ClbrDataBuffer.h"
#include "mt_sysrst.h"
#include "Dut_Api.h"
#include "ServicesHandler_Api.h"
#include "Pauser_Api.h"
#include "HdkCdbManagerTask_api.h"
#include "loggerAPI.h"
#include "Utils_Api.h"

#define LOG_LOCAL_GID   GLOBAL_GID_CALIBRATIONS
#define LOG_LOCAL_FID 2


#ifdef DEBUG_MODE_PROB
#define STATIC
#else
#define STATIC static
#endif



////////////////////////
// 		Definitions		//
////////////////////////

// Temperature sensor defines
#define MINIMAL_TIMER_VALUE         		  10  /* 10ms */
#define TEMPERATURE_DELTA_PARAMETER           10

#define DEFAULT_OFFLINE_INIT_CALBR_PROCS (CLBR_PROC_BIT_LPF)

typedef enum CalibrationManagerStatistics
{
	CM_STATISTICS_ONLINE_CAL_NET_DURATION,
	CM_STATISTICS_ONLINE_CAL_OVERALL_DURATION,
	CM_STATISTICS_NUM,
	CM_STATISTICS_ILLEGAL
}CalibrationManagerStatistics_e;

 
#define ONLINE_CLBR_FRAGMENT_MAX_TIME_VAL	(6000) // 6 msec 


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
STATIC void loadStoreSmInit(void);
STATIC uint32 getBufferAddressForCalibration(void);
STATIC uint32 getLSB(uint32 x);
STATIC void issueHostDataTransaction(uint8 direction);
STATIC void fillSetChannelMsgFromCalibrateDb(K_MSG *setChannelMsg);
#if !defined ENET_INC_UMAC
STATIC void sendErrorsToHostByHdkCdbManager(CALIBR_ALGO_EVENT* errors);
STATIC void activateDmaReqToHdkCdbMan(uint32 sourceAddress, uint32 destinationAddress, uint32 dmaLength, uint8 direction);
#endif
#if defined ENET_INC_UMAC
STATIC void sendErrorsToHost(CALIBR_ALGO_EVENT* errors);
void ClbrMngr_ActivateDma(uint32 sourceAddress, uint32 destinationAddress, uint32 dmaLength, uint8 direction);
#endif

STATIC void temperatureDiffCheck(void);
STATIC void temperatureUpdate(void);
STATIC uint32 getHostRamPointer(void);

////////////////////////
// Private Members
////////////////////////

STATIC const LoadStoreStateMachineFunctionEntry_t loadStoreStateMachineFunc[LOAD_STORE_NUMBER_OF_STATES][LOAD_STORE_NUMBER_OF_EVENTS] = 
{
	/*STATE - LOAD_STORE_STATE_IDLE  */
	{
		loadStoreTriggerLoadStateHandler,				/*EVENT - LOAD_STORE_EVENT_TRIGGER_LOAD			*/
		loadStoreInvalidEvent,							/*EVENT - LOAD_STORE_EVENT_WAIT_DMA_DONE			*/
		loadStoreTriggerStoreStateHandler, 				/*EVENT - LOAD_STORE_EVENT_TRIGGER_STORE			*/
	},

	/*STATE - LOAD_STORE_STATE_WAIT_DMA_LOAD  */
	{
		loadStoreInvalidEvent,							/*EVENT - LOAD_STORE_EVENT_TRIGGER_LOAD			  */
		loadStoreDmaDoneStateHandler,					/*EVENT - LOAD_STORE_EVENT_DMA_DONE			        */
		loadStoreInvalidEvent, 							/*EVENT - LOAD_STORE_EVENT_TRIGGER_STORE			  */
	},
	/*STATE - LOAD_STORE_STATE_WAIT_DMA_STORE  */
	{
		loadStoreInvalidEvent,							/*EVENT - LOAD_STORE_EVENT_TRIGGER_LOAD			  */
		loadStoreDmaDoneStateHandler,					/*EVENT - LOAD_STORE_EVENT_WAIT_DMA_DONE			  */
		loadStoreInvalidEvent, 							/*EVENT - LOAD_STORE_EVENT_TRIGGER_STORE			  */
	}
};

LoadStoreState_e loadStoreSmState;

STATIC uint32 m_calibProcBitmap[CLBR_TYPE_NUM];
STATIC ClbrProcBit_e m_curOnlineProc = CLBR_PROC_BIT_MIN; // Current running caibration process
STATIC uint32 m_curOfflineProcBitmap = 0x0; // The bitmap of the calibration processes that are still to be done
ClbrMngr_CalibrateDB_t ClbrMngr_Calibrate;
ClbrMngr_TemperatureDB_t ClbrMngr_TemperatureDb;
uint32 OnlineMaskUpdate;
#if defined ENET_INC_UMAC
uint8 dmaManagerClientId;
#endif


//Seamless calibration data
STATIC bool m_isFirstOfflineCalib = TRUE;

STATIC CalibrSchedData_t m_clbrSchedData;
STATIC int32	m_onlineCalNetStartTime;

/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/
//#define CALIBRATION_MANAGER_LOGS

uint32 calibrationManagerStatistics[CM_STATISTICS_NUM];



////////////////////////
// 	   Private Methods		//
////////////////////////

STATIC void loadStoreSmInit(void)
{
	setLoadStoreSmState(LOAD_STORE_STATE_IDLE);
}

STATIC uint32 getHostRamPointer(void)
{
	SetChannelParams_t* pSetChannelParams = HDK_getSetChannelParams();
	
	uint32 uReturnAddress = pSetChannelParams->calibrationBufferBaseAddress;
	uint32 uDataIndex ;
	uDataIndex = HDK_convertChannelNumToIndex(pSetChannelParams->setChannelReqParams.low_chan_num, pSetChannelParams->setChannelReqParams.chan_width, pSetChannelParams->band);
	if(uDataIndex == INVALID_INDEX)
		return INVALID_INDEX; 
	uReturnAddress += CHANNEL_CALIBRATION_DATA_SIZE * uDataIndex;
	return uReturnAddress;
}
STATIC void issueHostDataTransaction(uint8 direction)
{
	uint32 hostAddress, shramAddress, sourceAddress, destAddress, length;

	// Get position of host buffer to use for storage
	hostAddress = getHostRamPointer();
	
	// Address in SHRAM for storage
	shramAddress = getBufferAddressForCalibration();
	
	//Set the data size
	length = CHANNEL_CALIBRATION_DATA_SIZE;
	if (direction == DMA_DIRECTION_HOST_TO_SHRAM)
	{
		sourceAddress = hostAddress;
		destAddress = shramAddress;
	}
	else
	{
		sourceAddress = shramAddress;
		destAddress = hostAddress;
	}
#if defined ENET_INC_UMAC
/*DMA run only from UM, if call from LM send message to UM to run DMA,
	otherwize call DMA from calibration manager directly*/
	ClbrMngr_ActivateDma(sourceAddress,destAddress, length, direction);
#else
	activateDmaReqToHdkCdbMan(sourceAddress,destAddress, length, direction);
#endif
}

/**
 * Uses a free FSDU to get a buffer to use with seamless calibration data
 *
 * @return Address in shared RAM to use
 */
STATIC uint32 getBufferAddressForCalibration(void)
{ 
    return ((uint32)(&ShramClbrDataBufferStoreChanData));
}

STATIC uint32 getLSB(uint32 x)
{
	uint32 index = 0;

	if(!x)
	{
	    return x;
	}

	while((!(x & 1<<index)) && (index < 32))
	{
		index++;
	}

	return (1<<index);
}

STATIC inline void resetOnlineProc(void)
{
	m_curOnlineProc = getLSB(m_calibProcBitmap[CLBR_TYPE_ONLINE]);
}

void ClbrMngr_handleOfflineCalibDone(void)
{
	bool isStore = (m_clbrSchedData.dataStoreState == CAL_DATA_STATE_STORE ? TRUE : FALSE);
	
#ifdef HDK_CDB_WO_DUT
	Dut_updateCalibResults(m_clbrSchedData.errors.u32ErrCode);
#endif
	// Notify the host if any errors were ecountered
	if (m_clbrSchedData.errors.u32ErrCode != 0)
	{
#if defined ENET_INC_UMAC
		sendErrorsToHost(&m_clbrSchedData.errors);
#else //LMAC
		/*When calibration manager compiled in LM message will be sent through Hdk Cdb manager in UM*/		
		sendErrorsToHostByHdkCdbManager(&m_clbrSchedData.errors);
#endif
	}

	m_clbrSchedData.dataStoreState = CAL_DATA_STATE_NONE;
	
#ifndef ENET_INC_HW_FPGA
	RficDriver_ReadTemperature(&ClbrMngr_TemperatureDb.antCurrentTemperature);	
#endif

	if (isStore)
	{
		//Move calibration data to host memory
		if (HDK_GetBaseHostDataAddrPointer() != 0) 
		{
			loadStoreTriggerStoreEventHandler();
			return;
		}
	}	
#ifdef HDK_CDB_WO_DUT
	DUT_HandleCalibDone();
#endif	
	/* Send Calibration Results Stored event to Set Channel SM */
	setChannelCalResultsStoredEventHandler(NULL);	
}
STATIC void fillSetChannelMsgFromCalibrateDb(K_MSG *setChannelMsg)
{
	HdkSetChannelReqParams_t *pHdkSetChannelRequestParams;
	
	pHdkSetChannelRequestParams = ((HdkSetChannelReqParams_t *)setChannelMsg->abData);
	pHdkSetChannelRequestParams->low_chan_num = ClbrMngr_Calibrate.channels[ClbrMngr_Calibrate.currentCalibratedChannelIndex];
	pHdkSetChannelRequestParams->low_chan_num2 = 0;
	pHdkSetChannelRequestParams->chan_width = ClbrMngr_Calibrate.chandWidth;
	pHdkSetChannelRequestParams->primary_chan_idx = 0;
	pHdkSetChannelRequestParams->isContinuousInterfererDetectionNeeded = FALSE;
	pHdkSetChannelRequestParams->isRadarDetectionNeeded = FALSE;
	pHdkSetChannelRequestParams->switchType = ST_CALIBRATE;
	pHdkSetChannelRequestParams->channelNotification = CHANNEL_NOTIFICATION_NORMAL;
	pHdkSetChannelRequestParams->block_tx_post = FALSE;
	pHdkSetChannelRequestParams->RegulationType = HDK_getSetChannelReqParams()->RegulationType;

}
#if !defined ENET_INC_UMAC
STATIC void sendErrorsToHostByHdkCdbManager(CALIBR_ALGO_EVENT* errors)
{
	K_MSG *pErrorMsgToHost;
	HdkCdbManCalibrationErrorMsgToHost_t*	pErrorMsgToHostParams;
	uint8 	vapId;

	pErrorMsgToHost = OSAL_GET_MESSAGE(sizeof(HdkCdbManCalibrationErrorMsgToHost_t));
	pErrorMsgToHostParams = (HdkCdbManCalibrationErrorMsgToHost_t*)pK_MSG_DATA(pErrorMsgToHost);

	pErrorMsgToHostParams->calibrAlgoType = errors->u32calibrAlgoType;
	pErrorMsgToHostParams->errCode = errors->u32ErrCode;
	pErrorMsgToHostParams->eventId = EVENT_CALIBR_ALGO_FAILURE;
	pErrorMsgToHostParams->timeStamp = GET_TSF_TIMER_LOW();

	vapId = ConfigurationManager_GetFirstVapForBand(ConfigurationManager_GetMyBand());

	OSAL_SEND_MESSAGE(HDK_CDB_MAN_SEND_CAL_ERROR_MSG_TO_HOST_REQ, TASK_HDK_CDB_MANAGER,pErrorMsgToHost, vapId);

}
#endif
#if defined ENET_INC_UMAC
STATIC void sendErrorsToHost(CALIBR_ALGO_EVENT* errors)
{
	MAC_EVENT sMACEvent;
	
	sMACEvent.u.sCalibrationEvent.u32calibrAlgoType = errors->u32calibrAlgoType; 
	sMACEvent.u.sCalibrationEvent.u32ErrCode = errors->u32ErrCode;
	sMACEvent.u32EventID = EVENT_CALIBR_ALGO_FAILURE;
	sMACEvent.u32Timestamp = GET_TSF_TIMER_LOW();

	vIF_SendMsg(&sUmiIndFreeQueue, UMI_MC_MAN_MAC_EVENT_IND, TASK_UM_IF_TASK, &sMACEvent, sizeof(MAC_EVENT), 0);
}
#endif

void ClbrMngr_EnableDisableOnlineCal(bool enableCalibration, bool runImmediatly)
{
	uint8 switchType = hdkGetChannelSwitchType();
	K_MSG* pMsg;
	HdkCdbManagerOnlineCalSmMsg_t* pEnDisOnlineCal;
	BandId_e myBand = ConfigurationManager_GetMyBand();
	uint8 vapId = ConfigurationManager_GetFirstVapForBand(myBand);
	pMsg = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerOnlineCalSmMsg_t));
	pEnDisOnlineCal = (HdkCdbManagerOnlineCalSmMsg_t *)pK_MSG_DATA(pMsg);
	pEnDisOnlineCal->band = myBand;
	pEnDisOnlineCal->runImmediateOnlineCal = runImmediatly;

	
	if(switchType == ST_RSSI || switchType == ST_CALIBRATE || switchType == ST_SCAN || OnlineMaskUpdate == 0)
	{
		pEnDisOnlineCal->enableDisableCal = DISABLED;
		pEnDisOnlineCal->runImmediateOnlineCal = FALSE;
	}
	else
	{
		// Set online calibration timer
		pEnDisOnlineCal->enableDisableCal = enableCalibration;
		pEnDisOnlineCal->runImmediateOnlineCal = runImmediatly;

	}
	
	/*Send message to HdkCdbManager*/
	OSAL_SEND_MESSAGE(HDK_CDB_MAN_BAND_EN_DIS_ONLINE_CAL_REQ, TASK_HDK_CDB_MANAGER, pMsg, vapId);	//HdkCdbMan should know the sender vapId
}



////////////////////////
// Public Methods
////////////////////////

/**********************    Event Handlers - Load & Store SM    *****************************/

void loadStoreTriggerLoadEventHandler(void)
{
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG0_D("Load Store event: LOAD_STORE_EVENT_TRIGGER_LOAD. current state: %d", loadStoreSmState);
#endif

	loadStoreStateMachineFunc[loadStoreSmState][LOAD_STORE_EVENT_TRIGGER_LOAD]();
}

void loadStoreDmaDoneEventHandler(void)
{
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG0_D("Load Store event: LOAD_STORE_EVENT_DMA_DONE. current state: %d", loadStoreSmState);
#endif

	loadStoreStateMachineFunc[loadStoreSmState][LOAD_STORE_EVENT_DMA_DONE]();
}

void loadStoreTriggerStoreEventHandler(void)
{
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG0_D("Load Store event: LOAD_STORE_EVENT_TRIGGER_STORE. current state: %d", loadStoreSmState);
#endif

	loadStoreStateMachineFunc[loadStoreSmState][LOAD_STORE_EVENT_TRIGGER_STORE]();
}

/**********************    State Handlers - Load & Store SM    *****************************/

void loadStoreTriggerLoadStateHandler(void)
{
	/* Trigger DMA */
	issueHostDataTransaction(DMA_DIRECTION_HOST_TO_SHRAM);
	
	/* Change data store state */
	m_clbrSchedData.dataStoreState = CAL_DATA_STATE_LOAD;

	/* Switch Load and Store state */
	setLoadStoreSmState(LOAD_STORE_STATE_WAIT_DMA_LOAD);
}

void loadStoreDmaDoneStateHandler(void)
{	
	LoadStoreState_e loadStoreState = loadStoreSmState;
	
#if defined ENET_INC_UMAC 
	/* Allow DMA Manager to send MSG again (it was disabled since the MSG was sent)
		If CalibrationManager compiled in UMAC, DMA handler will done through calibration manager and not HdkCdbManager,
		In that case EventManager_TurnOnEvent should be done here. 
		If calibration manager compiled in LM, DMA handler will be done through HdkCdbManager so EventManager_TurnOnEvent will be done there
	*/
	EventManager_TurnOnEvent(EVENT_ID_DMA_HDK_BAND0);	
#endif
	/* Switch Load and Store state */
	setLoadStoreSmState(LOAD_STORE_STATE_IDLE);

	if (loadStoreState == LOAD_STORE_STATE_WAIT_DMA_LOAD)
	{
		if (CoC_getState() == COC_STATE_WAIT_FOR_HDK_SET_AND_CFM)
		{
			ClbrMngr_RunCocCalibrations(HDK_GetCocSaveMsg());
		}
		else
		{
			/* Send Calibration Results Loaded event to Set Channel SM */
			setChannelCalResultsLoadedEventHandler(NULL);
		}
	}
	else if (loadStoreState == LOAD_STORE_STATE_WAIT_DMA_STORE)
	{
		/* Send Calibration Results Stored event to Set Channel SM */
		setChannelCalResultsStoredEventHandler(NULL);
	}

}

void loadStoreTriggerStoreStateHandler(void)
{
	issueHostDataTransaction(DMA_DIRECTION_SHRAM_TO_HOST);

	/* Switch Load and Store state */
	setLoadStoreSmState(LOAD_STORE_STATE_WAIT_DMA_STORE);
}

void loadStoreInvalidEvent(void)
{
	ASSERT(0);
}

void loadStoreIgnoreEvent(void)
{

}

/**********************    State Setter  *****************************/

void setLoadStoreSmState(LoadStoreState_e state)
{
#ifdef CALIBRATION_MANAGER_LOGS
	if (state != LOAD_STORE_STATE_IDLE)
	{
		ILOG0_D("setLoadStoreSmState. state: %d", state);
	}
#endif
 
	loadStoreSmState = state;
}

/**********************    State Getter  *****************************/

LoadStoreState_e getLoadStoreSmState(void)
{
	return loadStoreSmState;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void ClbrMngr_Init(bool isNeedLoad)
{
	UNUSED_PARAM(isNeedLoad);	
	ClbrHndlr_Init();
	loadStoreSmInit();
	ClbrMngr_CalibrateParamsInit();
	ClbrMngr_TemperatureDbInit();
}
#if defined ENET_INC_UMAC
void ClbrMngr_PostInitDmaInit(void)
{
	dmaRegistration_t dmaRegister;

	// Register as a client with DMA Manager
	// Fill parameters
	dmaRegister.priority = DMA_PRIORITY_5;
	dmaRegister.eventId = EVENT_ID_DMA_HDK_BAND0;
	// Register and get clientID
	dmaManagerClientId = DmaManager_Register(&dmaRegister);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

void ClbrMngr_ActivateDma(uint32 sourceAddress, uint32 destinationAddress, uint32 dmaLength, uint8 direction)
{
	dmaDesc_t*	pDmaDesc = NULL;
	
	// Allocate DMA descriptor
	pDmaDesc = DmaManager_AllocateDesc();

	// Fill DMA descriptor
	pDmaDesc->source 		= sourceAddress;
	pDmaDesc->destination 	= destinationAddress;
	pDmaDesc->direction 	= direction;
	pDmaDesc->length		= dmaLength;
	pDmaDesc->clientId 		= dmaManagerClientId;

	// Kick DMA manager
	DmaManager_NewDmaRequest(pDmaDesc);	
}
#endif
void ClbrMngr_TemperatureDbInit(void)
{
	memset(&ClbrMngr_TemperatureDb,0, sizeof(ClbrMngr_TemperatureDB_t));
	ClbrMngr_TemperatureDb.temperatureDeltaParam = TEMPERATURE_DELTA_PARAMETER;
}

void ClbrMngr_CalibrateParamsInit(void)
{
	memset(&ClbrMngr_Calibrate,0, sizeof(ClbrMngr_CalibrateDB_t));
}


void ClbrMngr_Reset(clbrHndlrResetConf_t* pResetConf)
{
	// the offline mask arrives with the init procs enabled as well
	uint32 offlineMask = (pResetConf->offlineCalMask & SUPPORTED_OFFLINE_CALIBRATION_MASK); 
	uint32 initMask = offlineMask;
	
	initMask &= DEFAULT_OFFLINE_INIT_CALBR_PROCS;
	offlineMask &= (~DEFAULT_OFFLINE_INIT_CALBR_PROCS);
	
	pResetConf->onlineCalMask &= SUPPORTED_ONLINE_CALIBRATION_MASK;

	// Internal members
	ClbrMngr_ResetOfflineProcs();
	ClbrMngr_SetCalibProcMask(CLBR_TYPE_INIT, initMask); 
	ClbrMngr_SetCalibProcMask(CLBR_TYPE_OFFLINE, offlineMask); // the mask arrives with the init procs enabled as well
	OnlineMaskUpdate = pResetConf->onlineCalMask;

	// Calibration Handler reset
	ClbrHndlr_Reset(pResetConf);
}

void ClbrMngr_SetCalibProcMask( IN ClbrType_e calibrationType, IN uint32 mask )
{
	m_calibProcBitmap[calibrationType] = mask;	
}

uint32 ClbrMngr_GetCalibProcMask( IN ClbrType_e calibrationType)
{
	return	(m_calibProcBitmap[calibrationType]);	
}

void ClbrMngr_RunOnlineCalibration(void)
{
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG0_V("ClbrMngr_RunOnlineCalibration");
#endif
#if defined(ENET_INC_HW_FPGA)
	ASSERT(0);/* In case we're running on FPGA,  we should't get here as calibration mask should be zero */
#endif
	// Mark that we begun online process
	m_clbrSchedData.dataStoreState = CAL_DATA_STATE_NONE;
	m_clbrSchedData.calibrSMState = CLBR_TYPE_ONLINE;
	m_clbrSchedData.calBitmap = m_curOnlineProc;

	m_onlineCalNetStartTime = TIME_STAMP(START_TIME, 0);
	
	/* Calibrate */
	ClbrHndlr_Calibrate();
}

void ClbrMngr_ResetStatus(void)
{
	ClbrHndlr_ResetStatus();
}

void ClbrMngr_RunOfflineCalibration(void)
{
	uint8 switchType;
	HdkCdbManagerOfflineCalRequester_e offlineCalRequester = OFFLINE_CAL_SET_CHANNEL_REQUESTER;
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG0_V("ClbrMngr_RunOfflineCalibration");
#endif

#ifndef ENET_INC_HW_FPGA
	RficDriver_ReadTemperature(&ClbrMngr_TemperatureDb.antCurrentTemperature);	
#endif

	// Reset the online ref - to begin from first after offline
	resetOnlineProc();
	m_curOfflineProcBitmap = m_calibProcBitmap[CLBR_TYPE_OFFLINE];
	switchType = hdkGetChannelSwitchType();
#ifdef HDK_LOGS	
	ILOG0_D("SWITCH TYPE %d ++++++++ ", switchType);							
#endif

	switch(switchType)
	{
		case ST_RSSI:
			m_curOfflineProcBitmap = m_curOfflineProcBitmap & SUPPORTED_RSSI_CHANNEL_CALIBRATION_MASK;
			break;
		case ST_CALIBRATE:
			if (m_isFirstOfflineCalib)
			{
				m_curOfflineProcBitmap = m_curOfflineProcBitmap & SUPPORTED_MASK_FOR_CALIBRATE_REQ_FIRST_CHANNEL;
			}
			else
			{
				m_curOfflineProcBitmap = m_curOfflineProcBitmap & SUPPORTED_MASK_FOR_CALIBRATE_REQ_REST_CHANNELS;
			}
			offlineCalRequester = OFFLINE_CAL_CALIBRATE_REQUESTER;
			break;
	
		case ST_SCAN:
	
			m_curOfflineProcBitmap = m_curOfflineProcBitmap & SUPPORTED_SCAN_CHANNEL_CALIBRATION_MASK;
			break;
		default:
			break;
	}

#ifdef HDK_LOGS
		ILOG0_D("BITMAP BEFORE m_curOfflineProcBitmap = %d", m_curOfflineProcBitmap);
#endif

	HDK_FreqJumpCalMapHandler(&m_curOfflineProcBitmap);  
	
	if(m_isFirstOfflineCalib)
	{
		m_curOfflineProcBitmap |= m_calibProcBitmap[CLBR_TYPE_INIT];
		m_isFirstOfflineCalib = FALSE;
	}

	m_clbrSchedData.calibrSMState = CLBR_TYPE_OFFLINE;
	m_clbrSchedData.calBitmap = m_curOfflineProcBitmap;

	
/* Calibrate */
	setNextOfflineCalibrationsBitmaps(&m_clbrSchedData, FALSE); //force load will be set according to switchType in setNextOfflineCalibrationsBitmaps
	sendDualBandOfflineCalReq(offlineCalRequester, m_clbrSchedData.calBitmap,NULL);
	offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_RUN_CALIBRATION);

}

void ClbrMngr_RunCalibrationOnDemand(uint32 calibrateMask, K_MSG* reqesterMsgPtr)
{
	m_clbrSchedData.dataStoreState = CAL_DATA_STATE_NONE;
	m_clbrSchedData.calibrSMState = CLBR_TYPE_OFFLINE;
	m_clbrSchedData.calBitmap = calibrateMask & SUPPORTED_OFFLINE_CALIBRATION_MASK;
	
	/* Calibrate */
	sendDualBandOfflineCalReq(OFFLINE_CAL_ON_DEMAND_REQUESTER, m_clbrSchedData.calBitmap,reqesterMsgPtr);
	offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_RUN_CALIBRATION);
}

void ClbrMngr_RunCocCalibrations(K_MSG* pCocMsg)
{
		m_clbrSchedData.calibrSMState = CLBR_TYPE_OFFLINE;
		m_clbrSchedData.dataStoreState = CAL_DATA_STATE_LOAD;
		m_clbrSchedData.calBitmap = m_calibProcBitmap[CLBR_TYPE_OFFLINE];
		/* Calibrate */
		setNextOfflineCalibrationsBitmaps(&m_clbrSchedData, TRUE);
		sendDualBandOfflineCalReq(OFFLINE_CAL_COC_REQUESTER, m_clbrSchedData.calBitmap,pCocMsg);
		offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_RUN_CALIBRATION);
}
void ClbrMngr_RunRadioStateChangeCalibration(bool isUp)
{
	if (isUp == TRUE)
	{
		m_clbrSchedData.calibrSMState = CLBR_TYPE_OFFLINE;
		m_clbrSchedData.dataStoreState = CAL_DATA_STATE_LOAD;
		m_clbrSchedData.calBitmap = m_calibProcBitmap[CLBR_TYPE_OFFLINE];
		
		/* Calibrate */
		setNextOfflineCalibrationsBitmaps(&m_clbrSchedData, TRUE);
		sendDualBandOfflineCalReq(OFFLINE_CAL_RADIO_STATE_REQUESTER, m_clbrSchedData.calBitmap, NULL);
		offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_RUN_CALIBRATION);

	}
	else
	{
		/* Send Process Finished event to HDK Process SM */
		hdkProcessProcessFinishedEventHandler(NULL, HDK_PROCESS_RADIO_ON_OFF);
	}
}

void ClbrMngr_RunErpCalibration(HdkCdbManagerOfflineCalRequester_e offlineCalReqester)
{
		m_clbrSchedData.calibrSMState = CLBR_TYPE_OFFLINE;
		m_clbrSchedData.dataStoreState = CAL_DATA_STATE_LOAD;
		m_clbrSchedData.calBitmap = m_calibProcBitmap[CLBR_TYPE_OFFLINE];
		/* Calibrate */
		setNextOfflineCalibrationsBitmaps(&m_clbrSchedData, TRUE);
		sendDualBandOfflineCalReq(offlineCalReqester, m_clbrSchedData.calBitmap, NULL);
		offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_RUN_CALIBRATION);
}

void ClbrMngr_handleOnlineCalibDone(void)
{
	bool isOnlineProcFinished = m_clbrSchedData.isOnlineProcComplete;

	if(isOnlineProcFinished)
	{
		// Find next valid online proc
		DEBUG_ASSERT(m_calibProcBitmap[CLBR_TYPE_ONLINE]);
		do 
		{
			// Cycle the procs
			m_curOnlineProc <<= 1;
			if(m_curOnlineProc > CLBR_PROC_BIT_MAX)
			{
                ClbrMngr_Calibrate.moreFrags = FALSE;
#ifdef HDK_CDB_WO_DUT
				Dut_UpdateCalibrationCounter();
#endif
				temperatureUpdate();
				
				m_curOnlineProc = CLBR_PROC_BIT_MIN; // Start from the beginning
			}
		} while ((m_calibProcBitmap[CLBR_TYPE_ONLINE] & m_curOnlineProc) == 0);
	}
	calibrationManagerStatistics[CM_STATISTICS_ONLINE_CAL_NET_DURATION] = TIME_STAMP(END_TIME, m_onlineCalNetStartTime);
#ifdef CALIBRATION_MANAGER_LOGS
	ILOG2_DD("CM_STATISTICS_ONLINE_CAL_NET_DURATION %d proc %d", calibrationManagerStatistics[CM_STATISTICS_ONLINE_CAL_NET_DURATION], m_curOnlineProc);
#endif	
	DEBUG_ASSERT(calibrationManagerStatistics[CM_STATISTICS_ONLINE_CAL_NET_DURATION] <= ONLINE_CLBR_FRAGMENT_MAX_TIME_VAL);
}

void ClbrMngr_FinalizeOnlineCalibrationProcess(K_MSG *psMsg)
{
	/* Send an event to HDK Process SM */
	hdkProcessExecutionCompletedEventHandler(psMsg, HDK_PROCESS_ONLINE_CAL);
}

void ClbrMngr_HandleCalibrateProcessStart(K_MSG *psMsg)
{
	ProcessManagerReturnParams_t*	processMangerReturnPrams = (ProcessManagerReturnParams_t*)pK_MSG_DATA(psMsg);
	K_MSG *requesterParams = (K_MSG *)processMangerReturnPrams->requesterParams;
	UMI_CALIBRATE_PARAMS *calibrationRequestParams =  (UMI_CALIBRATE_PARAMS *)pK_MSG_DATA(requesterParams);

	/* Send Process Scheduled event to HDK Process SM */
	hdkProcessProcessScheduledEventHandler(NULL, HDK_PROCESS_CALIBRATE);

	/* Fill Calibrate database*/
	ClbrMngr_Calibrate.chandWidth = calibrationRequestParams->chan_width;
	ClbrMngr_Calibrate.numberOfChannels = calibrationRequestParams->num_chans;
	MEMCPY(ClbrMngr_Calibrate.channels, calibrationRequestParams->chan_nums, ClbrMngr_Calibrate.numberOfChannels);
	ClbrMngr_Calibrate.chandWidth = calibrationRequestParams->chan_width;
	
	ILOG0_V("ClbrMngr_HandleCalibrateProcessStart"); 

#ifdef CALIBRATION_MANAGER_LOGS	
	SLOG0(0, 0, ClbrMngr_CalibrateDB_t, &ClbrMngr_Calibrate);
#endif

	/*Send message to CDB manager offline cal SM*/
	sendDualBandOfflineCalReq(OFFLINE_CAL_CALIBRATE_REQUESTER, m_clbrSchedData.calBitmap, requesterParams);
	offlineCalSetEvent(HDK_OFFLINE_CAL_EVENT_START_CALIBRATION_PROC);

}
void ClbrMngr_RunCalibrationOnFirstChannel(K_MSG *psMsg)
{
	K_MSG *setChannelMsg;
	UNUSED_PARAM(psMsg);
	/* Fill Set Channel message*/
	setChannelMsg = OSAL_GET_MESSAGE(sizeof(HdkSetChannelReqParams_t));
	fillSetChannelMsgFromCalibrateDb(setChannelMsg);
	/* Send Set Frequency event to Set Channel SM */
	setChannelSetFrequencyEventHandler(setChannelMsg);
	ILOG0_V("ClbrMngr_RunCalibrationOnFirstChannel");
}
void ClbrMngr_RunCalibrationOnNextChannel(K_MSG *psMsg)
{	
	/* Send Offline Calibration Done event to Set Channel SM */
	UNUSED_PARAM(psMsg);	
	setChannelOfflineCalDoneEventHandler(NULL);
}

void ClbrMngr_HandleCalibrateProcessEnd(K_MSG *psMsg)
{
	/* Send an event to HDK Process SM */
	hdkProcessExecutionCompletedEventHandler(psMsg, HDK_PROCESS_CALIBRATE);
}

/**
* Handles DMA complete message for a chunk of data for seamless calibration
*/
void ClbrMngr_HostDataChunkDmaCompleteCb(K_MSG *in_psMsg)
{	
	UNUSED_PARAM(in_psMsg);	
	loadStoreDmaDoneEventHandler();
}

void ClbrMngr_ResetOfflineProcs(void)
{
	m_curOfflineProcBitmap = 0x0;
	m_isFirstOfflineCalib = TRUE;
}

void ClbrMngr_CalibrateChannelDone(void)
{
	K_MSG *setChannelMsg; 	
	
#ifdef CALIBRATION_MANAGER_LOGS	
	ILOG0_V("ClbrMngr_CalibrateChannelDone");	
	SLOG0(0, 0, ClbrMngr_CalibrateDB_t, &ClbrMngr_Calibrate);
#endif

	if ((ClbrMngr_Calibrate.currentCalibratedChannelIndex + 1) != ClbrMngr_Calibrate.numberOfChannels)
	{
		/* Calibrate next channel */
		ClbrMngr_Calibrate.currentCalibratedChannelIndex++;
		
		/* Fill new Set Channel message */
		setChannelMsg = OSAL_GET_MESSAGE(sizeof(HdkSetChannelReqParams_t));
		fillSetChannelMsgFromCalibrateDb(setChannelMsg);

		/* Send Set Frequency event to Set Channel SM */
		setChannelSetFrequencyEventHandler(setChannelMsg);
	}
	else
	{
		sendMsgToHdkCdbOfflineCalSm(HDK_CDB_MAN_OFFLINE_CALIBRATION_FINISHED_ON_LAST_CH);
	}
}

void ClbrMngr_WatchDogTimerExpired(K_MSG *pMsg)
{
	UNUSED_PARAM(pMsg);	
	FATAL("Calibration WD Timer Expired");
}

STATIC void temperatureDiffCheck(void)
{
	uint8 temperatureDiff;
	ClbrMngr_TemperatureDb.toCalibrate = FALSE;

	//Compare between the current temperature to the previous stored temperature
	temperatureDiff = ABS_DIFFERENCE(ClbrMngr_TemperatureDb.antCurrentTemperature, ClbrMngr_TemperatureDb.antPreviousTemperature);
	if (temperatureDiff >= ClbrMngr_TemperatureDb.temperatureDeltaParam)
	{
		ClbrMngr_TemperatureDb.toCalibrate = TRUE;
	}
}

STATIC void temperatureUpdate(void)
{
	ClbrMngr_TemperatureDb.antPreviousTemperature = ClbrMngr_TemperatureDb.antCurrentTemperature;
}

void sendDualBandOfflineCalReq(HdkCdbManagerOfflineCalRequester_e requester, uint32 calBitmap, K_MSG* pRequesterMsg)
{
	K_MSG* pMsgToHdkCdbManager;
	HdkCdbManagerOfflineCalSmMsg_t* pOfflineCalMsgtoBand;
	BandId_e myBand;

	myBand = ConfigurationManager_GetMyBand();
	pMsgToHdkCdbManager = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerOfflineCalSmMsg_t));
	pMsgToHdkCdbManager->header.vapId = ConfigurationManager_GetFirstVapForBand(myBand);
	pOfflineCalMsgtoBand = (HdkCdbManagerOfflineCalSmMsg_t *)pK_MSG_DATA(pMsgToHdkCdbManager);
	pOfflineCalMsgtoBand->band = myBand;
	pOfflineCalMsgtoBand->retMsg = HDK_CDB_MAN_INVALID_MSG; 
	pOfflineCalMsgtoBand->retTask = HDK_CDB_MAN_INVALID_TASK;
	pOfflineCalMsgtoBand->requester = requester;
	pOfflineCalMsgtoBand->requesterKmsgPtr = pRequesterMsg;

	/*Send run calibration request to HdkCdbManager for offline cal SM (set other core in pause during calibration)*/
	if (calBitmap != 0) //only if we should actually run calibration
	{
		/*Send message to HdkCdbManager to run offline*/
		pOfflineCalMsgtoBand->dualBandOfflineCalType = DUAL_BAND_OFFLINE_CAL;
	}
	else
	{
		pOfflineCalMsgtoBand->dualBandOfflineCalType = SINGLE_BAND_OFFLINE_CAL;
	}
	ILOG0_V("sendDualBandOfflineCalReq");
	SLOG0(0, 0, HdkCdbManagerOfflineCalSmMsg_t, pOfflineCalMsgtoBand);
	
	OSAL_SEND_MESSAGE(HDK_CDB_MAN_RUN_OFFLINE_CALIBRATION_REQ, TASK_HDK_CDB_MANAGER, pMsgToHdkCdbManager, pMsgToHdkCdbManager->header.vapId); 
}

void sendMsgToHdkCdbOfflineCalSm(K_MSG_TYPE msgType)
{
	K_MSG* pMsgToHdkCdbManager;
	HdkCdbManagerOfflineCalSmMsg_t* pOfflineCalMsgtoBand;
	BandId_e myBand;

	myBand = ConfigurationManager_GetMyBand();
	pMsgToHdkCdbManager = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerOfflineCalSmMsg_t));
	pMsgToHdkCdbManager->header.vapId = ConfigurationManager_GetFirstVapForBand(myBand);
	pOfflineCalMsgtoBand = (HdkCdbManagerOfflineCalSmMsg_t *)pK_MSG_DATA(pMsgToHdkCdbManager);
	pOfflineCalMsgtoBand->band = myBand;
	pOfflineCalMsgtoBand->retMsg = HDK_CDB_MAN_INVALID_MSG; 
	pOfflineCalMsgtoBand->retTask = HDK_CDB_MAN_INVALID_TASK;
	pOfflineCalMsgtoBand->requester = OFFLINE_CAL_INVALID_REQUESTER;
	pOfflineCalMsgtoBand->dualBandOfflineCalType = DUAL_BAND_OFFLINE_CAL; //default value - depand also in CDB offline cal SM
	pOfflineCalMsgtoBand->requesterKmsgPtr = NULL;

	/*Send run calibration request to HdkCdbManager for offline cal SM (set other core in pause during calibration)*/
	ILOG0_V("sendMsgToHdkCdbOfflineCalSm");
	SLOG0(0, 0, HdkCdbManagerOfflineCalSmMsg_t, pOfflineCalMsgtoBand);
	OSAL_SEND_MESSAGE(msgType, TASK_HDK_CDB_MANAGER, pMsgToHdkCdbManager, ConfigurationManager_GetFirstVapForBand(myBand)); 

}

CalibrSchedData_t* ClbrMngr_GetCalibDataPtr(void)
{
	return &m_clbrSchedData;
}

void hdkRunOnlineCalibrationReqEventHandler(K_MSG* pMsg)
{
	bool calibrate = FALSE;
	ClbrProcType_e	nextCalibration = CLBR_PROC_TYPE_INVALID;
    ProcessManagerReturnParams_t* processManagerReturnParams;
	processManagerReturnParams = (ProcessManagerReturnParams_t*)pK_MSG_DATA(pMsg);		
	
	if (processManagerReturnParams->processStatus == PROCESS_STATUS_REQUEST_REJECTED)
	{
		setHdkProcessSmState(HDK_PROCESS_STATE_WAIT_FOR_PROCESS_END);
		hdkProcessExecutionCompletedEventHandler(pMsg, HDK_PROCESS_ONLINE_CAL);
	}
    else
    {
		hdkProcessProcessScheduledEventHandler(NULL, HDK_PROCESS_ONLINE_CAL);

		if (ClbrMngr_Calibrate.moreFrags == FALSE)
		{
			ClbrMngr_SetCalibProcMask(CLBR_TYPE_ONLINE, OnlineMaskUpdate);
			resetOnlineProc();
			if (m_calibProcBitmap[CLBR_TYPE_ONLINE] != 0)
			{
#ifndef ENET_INC_HW_FPGA
				/* Read temperature sensor */						
				RficDriver_ReadTemperature(&ClbrMngr_TemperatureDb.antCurrentTemperature);
#endif
				/* Check temperature diffs */
				temperatureDiffCheck();

				if((ClbrMngr_Calibrate.forceOnlineCal == TRUE) || (ClbrMngr_TemperatureDb.toCalibrate == TRUE)) // if calibration time arrived or temp changed
				{
					ClbrMngr_Calibrate.moreFrags = TRUE;
					calibrate = TRUE;
					ClbrMngr_TemperatureDb.toCalibrate = FALSE;
				}
			}
		}
		else
		{
			calibrate = TRUE;
		}	
		if (calibrate == TRUE)
		{
			/* Run online calibration */
			ClbrMngr_RunOnlineCalibration();
			ClbrMngr_handleOnlineCalibDone();
		}
		/*Send calibration end to HDK CDB manager calibration SM*/
		if (m_curOnlineProc != 0)
		{
			nextCalibration = GetNextProc(m_curOnlineProc);
		}
		
		sendOnlineCalMsgToHdkCdbMan(HDK_CDB_MAN_BAND_ONLINE_CALIBRATION_ENDED, ClbrMngr_Calibrate.moreFrags, calibrate,nextCalibration, ENABLED);
    }
}
void ClbrMngr_SetForceOnlineCal(bool runForceCal)
{
	/*Run calibration without temperature consideration*/
	ClbrMngr_Calibrate.forceOnlineCal = runForceCal;
}

#if !defined ENET_INC_UMAC
/*DMA run only from UM, if called from LM send message to UM to run DMA otherwize call DMA from calibration manager directly*/
static void activateDmaReqToHdkCdbMan(uint32 sourceAddress, uint32 destinationAddress, uint32 dmaLength, uint8 direction)
{
	
	K_MSG *pActivateDmaMsg;
	HdkCdbManagerActivateDmaMsg_t*	pDmaParams;
	uint8 	vapId;

	/* Send indication to HdkCdbMan to activate DMA - DMA will be activated from UM*/	
	pActivateDmaMsg = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerActivateDmaMsg_t));
	pDmaParams = (HdkCdbManagerActivateDmaMsg_t*)pK_MSG_DATA(pActivateDmaMsg);
	pDmaParams->band = ConfigurationManager_GetMyBand();
	pDmaParams->dmaParams.source = sourceAddress;
	pDmaParams->dmaParams.destination = destinationAddress;
	pDmaParams->dmaParams.length = dmaLength;
	pDmaParams->dmaParams.direction = direction;

	vapId = ConfigurationManager_GetFirstVapForBand(pDmaParams->band);

	OSAL_SEND_MESSAGE(HDK_CDB_MAN_ACTIVATE_DMA_REQ, TASK_HDK_CDB_MANAGER,pActivateDmaMsg, vapId);
}
#endif

