#ifndef AFE_API_H_
#define AFE_API_H_

#include "System_GlobalDefinitions.h"
#include "HdkGlobalDefs.h"
#include "Hdk_Api.h"
#include "CalibrationsDefines.h"
#include "OSAL_Kmsg.h"
/**************************************************************************************/

#define AFE_LOOPBACK_REGISTER_MASK	(0xFF)

// describing the desired loop-back mode
typedef enum
{
  AFE_MODE_LOOPBACK, // Enable DAC & ADC according to the loop back desired and antenna number
  AFE_MODE_RX,// All DACs turned off, all ADC's turned on,Aux ADC are turned off
}AFE_LoopBack_Mode_e;

// AFE Antenna state
typedef enum
{
  AFE_ANTENNA_RX,
  AFE_ANTENNA_TX,
  AFE_ANTENNA_STANDBY,
  AFE_ANTENNA_POWER_DOWN,
  AFE_ANTENNA_UNINITIALZED,
}AFE_ANTENNA_mode_e;

// this enum represent the state of the DAC/ADC/AUX ADC.
typedef enum
{
  AFE_ADC,
  AFE_DAC,
  AFE_AUX_ADC,
  AFE_MAX_SUB_BLOACK = AFE_AUX_ADC,
  AFE_TOTAL_SUB_BLOACKS,
  AFE_ALL_SUB_BLOACKS = AFE_TOTAL_SUB_BLOACKS,
}AFE_SUB_BLOCK_e;

// this enum helps to represent the state of the DAC/ADC/AUX ADC.
typedef enum
{
  BLOCK_ACTIVE,
  BLOCK_STANDBY,
  BLOCK_POWER_DOWN,
  BLOCK_MAX_MODE = BLOCK_POWER_DOWN,
  BLOCK_TOTAL_MODES,
  BLOCK_ALL_MODES = BLOCK_TOTAL_MODES,
}AFE_SUB_BLOCK_mode_e;

// AFE T/R state
typedef enum
{
  AFE_TR_SWITCH_MODE_NONE,
  AFE_TR_SWITCH_MODE_TX,
  AFE_TR_SWITCH_MODE_RX,
  AFE_TR_SWITCH_MODE_RXTX,
  AFE_TR_SWITCH_MAX_MODE = AFE_TR_SWITCH_MODE_RXTX,
  AFE_TR_SWITCH_TOTAL_MODES,
}AFE_TR_SWITCH_mode_e;

/* AFE MS-SSI ADC Modes */
typedef enum AfeMsSsiAdcMode
{
    AFE_MS_SSI_ADC_MODE_FUNCTIONAL,
    AFE_MS_SSI_ADC_MODE_GAIN_CALIBRATION,
    AFE_MS_SSI_ADC_MODE_OFFSET_CALIBRATION,
    AFE_MS_SSI_ADC_MODE_MAX,
} AfeMsSsiAdcMode_e;

// Dither tone frequency mode
typedef enum
{
  NOMINAL_FREQUENCY = 0,
  DOUBLED_FREQUENCY = 1,
}AFE_DITHER_TONE_FREQ_e;

typedef struct AfePSDRegistersDB
{
	uint32 regAddr;
	uint32 mask;
	uint16 regData[FCSI_TOTAL_CHANNELS];
} AfePSDRegistersDB_t;



/*********************************************************************************
Method:       AFE_Init
Description:  Initialization of the AFE block. This function was created mainly
              for the Wave 400 that requires an initialization sequence unlike
              the Wave 300 that its default values are sufficient.
Parameter:    IN void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_Init(IN void* conf_seg, bool resetCentral);

/*********************************************************************************
Method:       AFE_Init
Description:  Initialization of the AFE block. This function was created mainly
              for the Wave 400 that requires an initialization sequence unlike
              the Wave 300 that its default values are sufficient.
Parameter:    IN void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_InitFuse(IN void* conf_seg);
void AFE_InitFuseAnt(void);
void AFE_ADC_Calibration(void);

/*********************************************************************************
Method:       AFE_SetLoopBack
Description:  Set the AFE to loop-back mode based on the relevant calibration
              type, antenna number. This function also save the current registers
              state into a structure that will be used later on during restore operation.
Parameter:    IN Antenna_e antenna - the relevant antenna for loop-back
              IN AFE_LoopBack_Mode_e lp_mode - All Rx ADC open or open only relevant DAC/ADC
			  IN uint8 isDebug - Whether to enable the PA for RFIC debug purposes 
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_SetLoopBack( IN Antenna_e antenna, Antenna_e rxAntenna, IN AFE_LoopBack_Mode_e lp_mode, IN uint8 isDebug );

/*********************************************************************************
Method:       AFE_RestoreFromLoopBack
Description:  Restore the previous registers settings prior to loop-back mode.
Parameter:    void
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_RestoreFromLoopBack( void );

void AFE_TxOn(uint8 antMask);
void AFE_TxOff(uint8 antMask);
void AFE_RxOn(uint8 antMask);
void AFE_RxOff(uint8 antMask);


void AFE_AuxadcOff(uint8 antMask);
void AFE_AuxadcOn(uint8 antMask);

void AFE_SetSpectrumMode(IN SpectrumMode_e spectrumMode);


/*********************************************************************************
Method:       AFE_SetAntennaMode
Description:  Set the AFE per antenna state (Rx, Tx, standby, power down). The
              AFE setting should be as follow:
              Mode       | IQADC     | IQDAC      | AuxADC     | PLL        | LDO/BG
              -----------|-----------|------------------------------------------------
              RX MODE    | Active    | Standby    | Active     | Active     | Active
              TX MODE    | Standby   | Active     | Active     | Active     | Active
              Standby    | Standby   | Standby    | Standby    | Active     | Active
              Power Down |Power down | Power down | Power down | Power down | Power down

Parameter:    IN Antenna_e antenna - the requested antenna
              IN AFE_ANTENNA_mode_e antenna_mode - the AFE antenna state
Returns:      void
*********************************************************************************/
RetVal_e AFE_SetAntennaMode( IN Antenna_e antenna, IN AFE_ANTENNA_mode_e antenna_mode );

/*********************************************************************************
Method:       AFE_GetAntennaMode
Description:  Get the current state of the AFE per antenna (Rx, Tx, standby, power down)
Parameter:    IN Antenna_e antenna - the requested antenna
              OUT AFE_ANTENNA_mode_e *antenna_mode - the AFE antenna state
Returns:
*********************************************************************************/
RetVal_e AFE_GetAntennaMode( IN Antenna_e antenna, OUT AFE_ANTENNA_mode_e *antenna_mode );
/*********************************************************************************
Method:      AFE_GetCtune
Description:  Get Ctune value
Parameter:   
Returns:
*********************************************************************************/
int16 AFE_GetCtune(void);
/*********************************************************************************
Method:      AFE_GetQTrim
Description:  Get qTeim value
Parameter:   
Returns:
*********************************************************************************/
int8 AFE_GetQTrim(void);
/***************************************************************************
**
** NAME          AFE_EnableDisableAfeAntennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void AFE_LdoConfig(uint8 antMask);

void AFE_EnableDisableAntennas(uint8 numAnts, PowerSaveStates_e powerSaveState);

/*********************************************************************************
Method:       AFE_SetSubBlockMode
Description:  Set, per antenna, each ADC/DAC/AUX to active, standby or power
              down mode
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              IN AFE_BLOCK_mode_e mode: active/standby/power-down
Returns:      Status_e
*********************************************************************************/
#if defined (AFE_SUB_BLOCK_SUPPORT)
RetVal_e AFE_SetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, IN AFE_SUB_BLOCK_mode_e mode );

/*********************************************************************************
Method:       AFE_GetSubBlockMode
Description:  Get, per antenna, each ADC/DAC/AUX state
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              OUT AFE_SUB_BLOCK_mode_e *mode: active/standby/power-down
Returns:
*********************************************************************************/
RetVal_e AFE_GetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, OUT AFE_SUB_BLOCK_mode_e *mode );

/*********************************************************************************
Method:       AFE_SetTRSwitchMode
Description:  Set, per antenna, the t/r switch mode
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_TR_SWITCH_mode_e mode: (tx/rx/txrx)
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_SetTRSwitchMode( IN Antenna_e antenna, IN AFE_TR_SWITCH_mode_e mode );
#endif

/*********************************************************************************
Method:       AFE_GetTRSwitchMode
Description:  Get, per antenna, the t/r switch mode
Parameter:    IN Antenna_e antenna: 0-2
              OUT AFE_TR_SWITCH_mode_e *mode: (tx/rx/txrx)
Returns:
*********************************************************************************/
RetVal_e AFE_GetTRSwitchMode( IN Antenna_e antenna, OUT AFE_TR_SWITCH_mode_e *mode );


/*********************************************************************************
Method:       AFE_SetRegister
Description:  Set AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between the mapped
              registers and the ones accessed by FSCI
Parameter:    IN uint32 reg_addr
              IN uint32 reg_value
Returns:      None
*********************************************************************************/
void AFE_SetRegister( IN uint32 reg_addr, IN uint32 reg_value );

void AFE_SetRegisterMasked(uint32 ant_mask, uint32 reg_addr, uint32 reg_mask, uint32 reg_value);

/*********************************************************************************
Method:       AFE_GetRegister
Description:  Get AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between
              the mapped registers and the ones accessed by FSCI
Parameter:    IN uint32 reg_addr
              OUT uint32 *reg_value
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e AFE_GetRegister( IN uint32 reg_addr, OUT uint16 *reg_value );


/*********************************************************************************
Method:       AFE_SetRssiAdcGainAndOffset
Description:  Stores RSSI ADC gain and offset trim values for the selected antenna list
Parameter:    IN AntennaBitmaps_e antList - The antenna list to update
              IN uint8 *gainTrims         - An array of size TOTAL_ANTENNAS with
                                            the gain trim values to store
              IN uint8 *offsetTrim        - An array of size TOTAL_ANTENNAS with
                                            the offset trim values to store
              IN AfeMsSsiAdcMode_e mode   - The mode of operation:
                                            Functional, gain calibration or offset calibration
Returns:      None
*********************************************************************************/
void AFE_SetRssiAdcGainAndOffset(IN AntennaBitmaps_e antList, IN uint8 *gainTrims, IN uint8 *offsetTrim, IN AfeMsSsiAdcMode_e mode);

/*********************************************************************************
Method:		AFE_Set_Dither_Tone_Mode
Description:  
Parameter:    	IN uint8 mode (nominal/double)
Returns:      	void
Remarks:		Set frequency of Dither tone, and save original value for restore option
*********************************************************************************/
void AFE_Set_Dither_Tone_Frequency(IN uint8 mode);

/*********************************************************************************
Method:		AFE_Restore_Dither_Tone_Frequency
Description:  
Parameter:     None
Returns:      	void
Remarks:		Set original Dither tone
*********************************************************************************/
void AFE_Restore_Dither_Tone_Frequency(void);

/*********************************************************************************
Method:			AFE_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_ReadRegister(IN uint32 regNum,OUT uint16 *regVal);

/*********************************************************************************
Method:			AFE_WriteRegisters
Description:  Writes 3 diferent values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues - the value to write (W300) or an array of 3 values to write per antenna (AR10)
Parameter:     IN AntennaBitmaps_e antList
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 *regValues);

/*********************************************************************************
Method:			AFE_WriteMaskRegisters
Description:  Writes 3 diferent masked values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues - the value to write (W300) or an array of 3 values to write per antenna (AR10)
Parameter:     IN AntennaBitmaps_e antList
Parameter:     IN uint32 regMask - register mask to write to
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteMaskRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 *regValues, IN uint32 regMask);
void AFE_CopyOneRowFromPsd(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint32 *regValues, IN uint32 regMask,uint32 rowIndex);
void AFE_resetPsdArray(void);



void AFE_RxDCInitAnt(uint8 ant);

void AFE_RxDCrestoreAnt(uint8 ant);
void AFE_RxDCrestoreAntAll(void);
void AFE_RxDCinit(void);
void AFE_RxDCrestore(void);
void AFE_SetEfuseData(K_MSG* psMsg);

#endif /* AFE_API_H_ */
