/*******************************************************************************
*
*   Source File:
*       Afe.c
*
*  AUTHOR: Oshri Yahav
*
*   Description: Low Level Driver for the AFE module.  
*
*   Copyright:
*       (C) Metalink Ltd.
*       All rights are strictly reserved. Reproduction or divulgence in any
*       form whatsoever is not permitted without written authority from the
*       copyright owner. Issued by Metalink Transmission Devices Ltd in
*       Israel - 11/94.
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***                        Include Files                                   ***/
/******************************************************************************/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "System_Information.h"
#include "PhyRxTdIfRiscPage0TdRegs.h"
#include "Antenna_fcsi_regpkg.h"
#include "PhyRxTdIfRiscPage0TdRegs.h"
#include "CalibrationsDefines.h"
#include "ErrorHandler_Api.h"
#include "Afe_API.h"
#include "Afe.h"
#include "PhyRegsIncluder.h"
#include "RegAccess_Api.h"
#include "mem_access.h"
#include "Indirect_API.h"
#include "AFEDefines.h"
#include "mt_sysrst.h"
#include "loggerAPI.h"
#include "lib_abb_api.h"

#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE_1
#define LOG_LOCAL_FID 0

/******************************************************************************/
/***                        Internal Constants                              ***/
/******************************************************************************/
#define AFE_REG_DEFAULT_MASK               (0xFFFF)
#define AFE_REG_DEFAULT_MASK_SHIFT         (0)
#define AFE_REG_NO_SHIFT                   (0)
#define AFE_STBY_OFF            	       (0)
#define AFE_STBY_ON             	       (1)
#define AFE_LOOPBACK_REGISTER_MASK			(0xFF)
#define AFE_LOOPBACK_TX_ANTS_SHIFT			(4)




/* AFE Stand BY Mode register value */
#define AFE_DYN_CTRL_ALL_DAC_STANDBY_MODE  (REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
											REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
											REG_AFE_DYN_CTRL_TX2_STBY_MASK)| \
											REG_AFE_DYN_CTRL_TX3_STBY_MASK)
// All AUX ADC on standby mode
#define AFE_DYN_CTRL_ALL_AUX_ADC_STANDBY  (REG_AFE_DYN_CTRL_AUX0_ADC_STBY_MASK | \
										   REG_AFE_DYN_CTRL_AUX1_ADC_STBY_MASK | \
										   REG_AFE_DYN_CTRL_AUX2_ADC_STBY_MASK)| \
										   REG_AFE_DYN_CTRL_AUX3_ADC_STBY_MASK)

// Loop-back on ANT0 --> ADC0-On, DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT0 (REG_AFE_DYN_CTRL_RX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT1 --> ADC1-On, DAC0-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT1 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT2 --> ADC2-On DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT2 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT2 --> ADC2-On DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT3 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK)

/* MS-SSI ADC calibration constants */
#define AFE_SSI_ADC_MODE_VALUE_FUNCTIONAL   (0x03)
#define AFE_SSI_ADC_MODE_VALUE_GAIN_CALIB   (0x13)
#define AFE_SSI_ADC_MODE_VALUE_OFFSET_CALIB (0x23)

/* MS-SSI ADC calibration init/restore values */
#define AFE_SSI_ADC_INIT_L1_VALUE           (0x0770)
#define AFE_SSI_ADC_RESTORE_L1_VALUE        (0x0270)
/* RC Replica tuning constants */
#define AFE_RC_TUNING_POWER_UP              (0x01)
#define AFE_RC_TUNING_POWER_DOWN            (0x00)

#define AFE_RC_TUNING_MEAS_WAIT_TIME        (3000) /* Time to wait for rctuning_meas_done (in microseconds) */
#define AFE_RC_TUNING_MEAS_WAIT_MAX_RETRIES (3)
#define AFE_RC_TUNING_MEAS_DONE_HIGH        (1)

#define AFE_RC_TUNING_1ST_MEAS_SWAP_BIT     (0)
#define AFE_RC_TUNING_2ND_MEAS_SWAP_BIT     (1)

#define AFE_RC_TUNING_FORMULA_A0            (1530539) /* Tuning formula: A0 constant */
#define AFE_RC_TUNING_FORMULA_A1            (2458)    /* Tuning formula: A1 constant */
#define AFE_RC_TUNING_FORMULA_A2            (-17)     /* Tuning formula: A2 constant */

#define AFE_RC_TUNING_GRX350_FACTOR			(58245)
#define AFE_RC_TUNING_FACTOR_SHIFT			(1)
#define AFE_RC_TUNING_GRX350_OFFSET			(9)

#define AFE_RC_TUNING_NLPF_MAX_VALUE        (31)
#define AFE_RC_TUNING_NLPF_MIN_VALUE        (0)

/* AFE init register values constants */
/* set clock-buffer R/I-value */
#define AFE_INIT_L19_REG_VALUE              (0x22A0)
/* Enable LDO */
#define AFE_INIT_L21_REG_VALUE              (0xA988)
/* Write default n/m values */
#define AFE_INIT_L17_REG_VALUE              (0x0261)
/* Enable Clocking */
#define AFE_INIT_L24_REG_VALUE              (0x0771)
/* release all resets, enable clk in channel */
#define AFE_INIT_L7_REG_VALUE               (0x089F)
/* config SSI-ADC */
#define AFE_INIT_L0_REG_VALUE               (0x8403)
#define AFE_INIT_L1_REG_VALUE               (0x0270)
#define AFE_INIT_L2_REG_VALUE               (0x52EC)
/* config IQ-ADC */
#define AFE_INIT_L3_CB_MODE_REG_VALUE       (0x1A0C)
#define AFE_INIT_L3_CB_MODE_REG_MASK        (~REG_AFE_L3_RCTUNE_MASK)
#define AFE_INIT_L3_CB_BW					(0)
#define AFE_INIT_L3_NCB_BW					(1)

#define AFE_INIT_L3_NCB_MODE_REG_VALUE      (0x1B0C)
#define AFE_INIT_L3_NCB_MODE_REG_MASK       (~REG_AFE_L3_RCTUNE_MASK)
#define AFE_INIT_L4_REG_VALUE               (0x4127)
#define AFE_INIT_L5_CB_MODE_REG_VALUE       (0x001E)
#define AFE_INIT_L5_NCB_MODE_REG_VALUE      (0x009E)
#define AFE_INIT_L5_CB_EN_FB_LO				(0)
#define AFE_INIT_L5_NCB_EN_FB_LO			(1)

#define AFE_INIT_L6_REG_VALUE               (0x0000)
/* Set Bias */
#define AFE_INIT_L20_REG_VALUE              (0x009F)

#define AFE_INIT_GRX500_L20_VALUE			(0x8800)
#define AFE_INIT_GRX500_L21_VALUE			(0x8000)
#define AFE_INIT_GRX500_L7_VALUE			(0x189F)
#define AFE_INIT_GRX500_L3_VALUE			(0x100F)
#define AFE_INIT_GRX500_L4_VALUE			(0x1527)
#define AFE_INIT_GRX500_L5_VALUE			(0x0002)

#define AFE_INIT_GLOBAL_MEM_ADDR			(0)
#define AFE_INIT_GLOBAL_MEM_RESET_MASK		(1 << 13)

#define AFE_INIT_GRX500_L7_RX_MUX_ON		(0x989F)
#define AFE_INIT_GRX500_L7_RX_ON			(0x999F)
#define AFE_INIT_GRX500_L7_RX_MUX_OFF_RESET_RX_SYNCOFF (0x189D)
#define AFE_INIT_GRX500_L7_RX_MUX_OFF		(0x189F)

#define AFE_INIT_RX_DELAY_US				(20)


/******************************************************************************/
/***                     Internal Data Structures                           ***/
/******************************************************************************/
// struct definition to save the current register settings
typedef struct
{
  uint32  afe_power_down_reg;
  uint32  afe_dyn_ctrl_reg;
}AFE_BLOCK_STATE_t;

// bit operation, used in various set mode functions
typedef enum
{
  AFE_CLR_BIT = 0,
  AFE_SET_BIT = 1,
}AFE_SUB_BLOCK_OP_e;

// struct definition help configure each AFE sub-block
typedef struct
{
  uint32               afe_dyn_ctrl_bit;
  AFE_SUB_BLOCK_OP_e   afe_dyn_ctrl_op;
  uint32               afe_power_down_bit;
  AFE_SUB_BLOCK_OP_e   afe_power_down_op;
}AFE_SUB_BLOCK_STATE_t;

/******************************************************************************/
/***                                Macros                                  ***/
/******************************************************************************/


/******************************************************************************/
/***                      Static Function Declaration                       ***/
/******************************************************************************/
static void writeToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint16 *regValues, IN uint32 mask, IN uint32 shift);
static void writeSameValueToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint32 regValue, IN uint32 mask, IN uint32 shift);
static void writeAfePsdTableFromDB(void);

/******************************************************************************/
/***                           Static Variables                             ***/
/******************************************************************************/
// This struct used to save the current state of the registers before loop-back
// and restore after loop-back
static uint32 afe_blk_state;

// Boolean variable indicating if we are in loop-back or not.
static bool in_loop_back = FALSE;

// Global struct that represent the current state of each antenna
static AFE_ANTENNA_mode_e afe_antenna_mode_array[MAX_NUM_OF_ANTENNAS] =
    {    AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED	};


// This is 3D array hold the registers configuration for active/standby/power_down modes
// Array dimensions [3][3][3].
// For active state we clear the standby bit and the power down bit
// For standby state we set the standby bit and clear the power down bit
// For power down state we set the standby bit and the power down bit
#if defined (AFE_SUB_BLOCK_SUPPORT)
static AFE_SUB_BLOCK_STATE_t afe_sub_blocks_configuration
  [TOTAL_ANTENNAS * AFE_TOTAL_SUB_BLOACKS * BLOCK_TOTAL_MODES] =
{
  // ANTENNA_0
  // ADC
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

  // ANTENNA_1
  // ADC
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

  // ANTENNA_2
  // ADC
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

   // ANTENNA_3
  // ADC
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_SET_BIT}, //POWER_DOWN
};
#endif

static AfePSDRegistersDB_t AfeDriverPSD[AFE_FCSI_MAX_ADDRESS - AFE_FCSI_MIN_ADDRESS];
static uint32 afeInitNumberOfrows = 0;

uint8 afeEfuseData[AFE_CALIBRATION_DATA_SIZE_GEN5] = {0};

uint8 ctuneValue;

/******************************************************************************/
/***                             Debug Section                              ***/
/******************************************************************************/

/******************************************************************************/
/***                        Public Functions Definitions                    ***/
/******************************************************************************/
/*********************************************************************************
Method:       AFE_Init
Description:  Initialization of the AFE block. This function was created mainly
              for the Wave 400 that requires an initialization sequence unlike
              the Wave 300 that its default values are sufficient.
Parameter:    IN void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_Init(IN void* conf_seg, bool resetCentral)
{
	uint16 readVals[FCSI_TOTAL_CHANNELS];
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf137_u resetReg;
	
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_FIRST_WRITE);
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_SECOND_WRITE);
	RegAccess_Write(REG_MIPS_PLL_CLOCK_3, REG_MIPS_PLL_CLOCK_3_ENABLE);
	
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, &resetReg.val);
	if (resetCentral)
	{
		resetReg.bitFields.fcsiMsResetCentralN = 0;
	}
	resetReg.bitFields.fcsiMsResetAntennasN = 0;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
	MT_DELAY(10);
	if (resetCentral)
	{
		resetReg.bitFields.fcsiMsResetCentralN = 1;
		RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
		abb11ac_central_init();
		MT_DELAY(2);
	}
	resetReg.bitFields.fcsiMsResetAntennasN = 1;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
	MT_DELAY(5);
	abb11ac_antenna_init(15);
	MT_DELAY(2);
		
	

	//second time abb11ac_fusing
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_FIRST_WRITE);
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_SECOND_WRITE);
	RegAccess_Write(REG_MIPS_PLL_CLOCK_3, REG_MIPS_PLL_CLOCK_3_DISABLE);

	AFE_ReadRegister(ANTENNA_FCSI_IQADCREG0, readVals);
	ctuneValue = ANTENNA_FCSI_IQADCREG0_CTUNE__GET(readVals[0]);
    return RET_VAL_SUCCESS;
}

RetVal_e AFE_InitFuse(IN void* conf_seg)
{
	uint16 readVals[FCSI_TOTAL_CHANNELS];
	int ctune = -1;
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_FIRST_WRITE);
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_SECOND_WRITE);
	RegAccess_Write(REG_MIPS_PLL_CLOCK_3, REG_MIPS_PLL_CLOCK_3_ENABLE);


	ctune = abb11ac_fusing(afeEfuseData);//second time
	
	if (ctune == -1)//second time
	{
	}
	else
	{	
	    // config antenna for 11ac-mode
	    abb11ac_ant_config(15, 40, ctune);
	}	
	//second time abb11ac_fusing
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_FIRST_WRITE);
	RegAccess_Write(REG_MIPS_SECURE_WRITE, REG_MIPS_SECURE_WRITE_SECOND_WRITE);
	RegAccess_Write(REG_MIPS_PLL_CLOCK_3, REG_MIPS_PLL_CLOCK_3_DISABLE);

	AFE_ReadRegister(ANTENNA_FCSI_IQADCREG0, readVals);
	ctuneValue = ANTENNA_FCSI_IQADCREG0_CTUNE__GET(readVals[0]);
	writeAfePsdTableFromDB();
    return RET_VAL_SUCCESS;
	
}
void AFE_InitFuseAnt(void)
{
}


/*********************************************************************************
Method:       AFE_SetLoopBack
Description:  Set the AFE to loop-back mode based on the relevant calibration
              type, antenna number. This function also save the current registers
              state into a structure that will be used later on during restore operation.
Parameter:    IN Antenna_e antenna - the relevant antenna for loop-back
              IN AFE_LoopBack_Mode_e lp_mode - All Rx ADC open or open only relevant DAC/ADC
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_SetLoopBack( IN Antenna_e antenna, Antenna_e rxAntenna, IN AFE_LoopBack_Mode_e lp_mode, IN uint8 isDebug )
{
	RegPhyRxTdIfRiscPage0TdAfeDynCtrl_u regVal;
	uint8 afeLoopbackMask;

	// if already in loopback return.
	if(in_loop_back)
	{
		return RET_VAL_ALREADY_EXISTS;
	}

	afeLoopbackMask = AFE_LOOPBACK_REGISTER_MASK;
	afeLoopbackMask = ~((1 << (antenna + AFE_LOOPBACK_TX_ANTS_SHIFT)) | (1 << rxAntenna));
	
	// Save current registers settings

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_DYN_CTRL,&afe_blk_state);
	regVal.val = afe_blk_state;

	if(lp_mode == AFE_MODE_RX)
	{
    	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_DYN_CTRL,afe_blk_state|0xff);
		regVal.bitFields.rx0Stby = AFE_STBY_OFF;
		regVal.bitFields.rx1Stby = AFE_STBY_OFF;
		regVal.bitFields.rx2Stby = AFE_STBY_OFF;
		regVal.bitFields.rx3Stby = AFE_STBY_OFF;

		regVal.bitFields.tx0Stby = AFE_STBY_ON;
		regVal.bitFields.tx1Stby = AFE_STBY_ON;
		regVal.bitFields.tx2Stby = AFE_STBY_ON;
		regVal.bitFields.tx3Stby = AFE_STBY_ON;

		regVal.bitFields.aux0AdcStby = AFE_STBY_OFF;
		regVal.bitFields.aux1AdcStby = AFE_STBY_OFF;
		regVal.bitFields.aux2AdcStby = AFE_STBY_OFF;
		regVal.bitFields.aux3AdcStby = AFE_STBY_OFF;

		RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_DYN_CTRL,regVal.val);
	}
	else
	{
			RegAccess_WriteMasked(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_DYN_CTRL, AFE_LOOPBACK_REGISTER_MASK, afeLoopbackMask);
	}

	// indicate that we are in loop-back and restore is permitted
	in_loop_back = TRUE;

	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_RestoreFromLoopBack
Description:  Restore the previous registers settings prior to loop-back mode.
Parameter:    void
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_RestoreFromLoopBack( void )
{
	// if loop-back was not set before we don't allow restore
	if(!in_loop_back)
	{
		return RET_VAL_UNINITIALIZED;
	}

	// Restore previous registers configuration
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_DYN_CTRL,afe_blk_state);

	// indicate that we are out of loop-back mode
	in_loop_back = FALSE;

	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_SetAntennaMode
Description:  Set the AFE per antenna state (Rx, Tx, standby, power down). The
              AFE setting should be as follow:
              Mode       | IQADC     | IQDAC      | AuxADC     | PLL        | LDO/BG
              -----------|-----------|------------------------------------------------
              RX MODE    | Active    | Standby    | Active     | Active     | Active
              TX MODE    | Standby   | Active     | Active     | Active     | Active
              Standby    | Standby   | Standby    | Standby    | Active     | Active
              Power Down |Power down | Power down | Power down | Power down | Power down

Parameter:    IN Antenna_e antenna - the requested antenna
              IN AFE_ANTENNA_mode_e antenna_mode - the AFE antenna state
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_SetAntennaMode( IN Antenna_e antenna, IN AFE_ANTENNA_mode_e antenna_mode )
{
  uint32 afe_dyn_ctrl_reg = 0;
  uint32 afe_power_down_reg = 0;

  // Read current register settings
  afe_power_down_reg = MT_RdReg(PHY_TD_BASE_ADDR,REG_AFE_PD_CTRL_RX0_PD);	
  afe_dyn_ctrl_reg = MT_RdReg(PHY_TD_BASE_ADDR,REG_AFE_DYN_CTRL_RX0_STBY);

  switch(antenna_mode)
  {
	case AFE_ANTENNA_RX:
	{
	  // ADC active, set RXx_ADC_STANDBY_MODE bit to 0
	  // AUX ADC active, set AUXx_STANDBY_MODE bit to 0
	  // DAC standby, set TXx_DAC_STANDBY_MODE bit to 1
	  // power up the antenna
	  switch(antenna)
	  {
	  case ANTENNA_0:
		afe_dyn_ctrl_reg = AFE_ADC0_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC0_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC0_ACTIVE_MODE(afe_dyn_ctrl_reg);
						   
		afe_power_down_reg = AFE_ADC0_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC0_POWER_UP(afe_power_down_reg);
		break;

	  case ANTENNA_1:
		afe_dyn_ctrl_reg = AFE_ADC1_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC1_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC1_ACTIVE_MODE(afe_dyn_ctrl_reg);
							 
		afe_power_down_reg = AFE_ADC1_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC1_POWER_UP(afe_power_down_reg);
		break;

	  case ANTENNA_2:
		afe_dyn_ctrl_reg = AFE_ADC2_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC2_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC2_ACTIVE_MODE(afe_dyn_ctrl_reg);
							   
		afe_power_down_reg = AFE_ADC2_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC2_POWER_UP(afe_power_down_reg);
		break;
		
	  case ANTENNA_3:
		afe_dyn_ctrl_reg = AFE_ADC3_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC3_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC3_ACTIVE_MODE(afe_dyn_ctrl_reg);
							   
		afe_power_down_reg = AFE_ADC3_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC3_POWER_UP(afe_power_down_reg);
		break;
		
	  default:
		break;
	  }
	}
	break;

	case AFE_ANTENNA_TX:
	{
	  // ADC standby, set RXx_ADC_STANDBY_MODE bit to 1
	  // DAC active, set TXx_DAC_STANDBY_MODE bit to 0
	  // AUX ADC active, set AUXx_STANDBY_MODE bit to 0
	  // power up the antenna
	  switch(antenna)
	  {
	  case ANTENNA_0:
		afe_dyn_ctrl_reg = AFE_DAC0_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_ADC0_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC0_ACTIVE_MODE(afe_dyn_ctrl_reg);
		
		afe_power_down_reg = AFE_DAC0_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC0_POWER_UP(afe_power_down_reg);
		break;

	  case ANTENNA_1:
		afe_dyn_ctrl_reg = AFE_DAC1_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_ADC1_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC1_ACTIVE_MODE(afe_dyn_ctrl_reg);
		  
		afe_power_down_reg = AFE_DAC1_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC1_POWER_UP(afe_power_down_reg);
		break;


	  case ANTENNA_2:
		afe_dyn_ctrl_reg = AFE_DAC2_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_ADC2_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC2_ACTIVE_MODE(afe_dyn_ctrl_reg);
		
		afe_power_down_reg = AFE_DAC2_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC2_POWER_UP(afe_power_down_reg);
		break;
		
	  case ANTENNA_3:
		afe_dyn_ctrl_reg = AFE_DAC3_ACTIVE_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_ADC3_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC3_ACTIVE_MODE(afe_dyn_ctrl_reg);
	  
		afe_power_down_reg = AFE_DAC3_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC3_POWER_UP(afe_power_down_reg);
	  break;

	  default:
		break;
	  }
	}
	break;

	case AFE_ANTENNA_STANDBY:
	{
	  // ADC standby, set RXx_ADC_STANDBY_MODE bit to 1
	  // DAC standby, set TXx_DAC_STANDBY_MODE bit to 1
	  // AUX ADC standby, set AUXx_STANDBY_MODE bit to 1
	  // power up the antenna
	  switch(antenna)
	  {
	  case ANTENNA_0:
		afe_dyn_ctrl_reg = AFE_ADC0_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC0_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC0_STANDBY_MODE(afe_dyn_ctrl_reg);
		
		afe_power_down_reg = AFE_ADC0_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC0_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC0_POWER_UP(afe_power_down_reg); 				   
		break;

	  case ANTENNA_1:
		afe_dyn_ctrl_reg = AFE_ADC1_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC1_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC1_STANDBY_MODE(afe_dyn_ctrl_reg);
	  
		afe_power_down_reg = AFE_ADC1_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC1_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC1_POWER_UP(afe_power_down_reg); 				 
		break;


	  case ANTENNA_2:
		afe_dyn_ctrl_reg = AFE_ADC2_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC2_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC2_STANDBY_MODE(afe_dyn_ctrl_reg);
		  
		afe_power_down_reg = AFE_ADC2_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC2_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC2_POWER_UP(afe_power_down_reg); 			   
		break;
		
	  case ANTENNA_3:
		afe_dyn_ctrl_reg = AFE_ADC3_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_DAC3_STANDBY_MODE(afe_dyn_ctrl_reg);
		afe_dyn_ctrl_reg = AFE_AUX_ADC3_STANDBY_MODE(afe_dyn_ctrl_reg);
		  
		afe_power_down_reg = AFE_ADC3_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC3_POWER_UP(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC3_POWER_UP(afe_power_down_reg); 			   
		break;

	  default:
		break;
	  }
	}
	break;

	case AFE_ANTENNA_POWER_DOWN:
	{
	  // Power Down: set PDN_CTRL_RX_x_PD and AFE_PDN_CTRL_TX_x_PD as 1.
	  switch(antenna)
	  {
	  case ANTENNA_0:
		afe_power_down_reg = AFE_ADC0_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC0_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC0_POWER_DOWN(afe_power_down_reg);						  
		break;
		
	  case ANTENNA_1:
		afe_power_down_reg = AFE_ADC1_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC1_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC1_POWER_DOWN(afe_power_down_reg);						  
		break;
		
	  case ANTENNA_2:
		afe_power_down_reg = AFE_ADC2_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC2_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC2_POWER_DOWN(afe_power_down_reg);						  
		break;

	  case ANTENNA_3:
		afe_power_down_reg = AFE_ADC3_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_DAC3_POWER_DOWN(afe_power_down_reg);
		afe_power_down_reg = AFE_AUX_ADC3_POWER_DOWN(afe_power_down_reg);						  
		break;
		
	  default:
		break;
	  }
	}
	break;

	default:
	  break;
  }

  MT_WrReg(PHY_TD_BASE_ADDR,REG_AFE_DYN_CTRL_RX0_STBY, afe_dyn_ctrl_reg);
  MT_WrReg(PHY_TD_BASE_ADDR,REG_AFE_PD_CTRL_RX0_PD, afe_power_down_reg);

  afe_antenna_mode_array[antenna] = antenna_mode;

  return RET_VAL_SUCCESS;
}


/*********************************************************************************
Method:       AFE_GetAntennaMode
Description:  Get the current state of the AFE per antenna (Rx, Tx, standby, power down)
Parameter:    IN Antenna_e antenna - the requested antenna
              OUT AFE_ANTENNA_mode_e *antenna_mode - the AFE antenna state
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetAntennaMode( IN Antenna_e antenna, OUT AFE_ANTENNA_mode_e *antenna_mode )
{
  if(antenna > MAX_ANTENNA)
  {
    return RET_VAL_BAD_PARAM;
  }

  *antenna_mode = afe_antenna_mode_array[antenna];

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:      AFE_GetCtune
Description:  Get Ctune value
Parameter:   
Returns:
*********************************************************************************/
int16 AFE_GetCtune(void)
{
	return (int16)ctuneValue;
}

void AFE_LdoConfig(uint8 antMask)
{
	writeSameValueToAfeRegister(0xf, ANTENNA_FCSI_REGLDO, ANTENNA_FCSI_REGLDO_PUPLDOANT_OFF__SVAL, ANTENNA_FCSI_REGLDO_PUPLDOANT__SMSK, AFE_REG_DEFAULT_MASK_SHIFT);
	writeSameValueToAfeRegister(antMask, ANTENNA_FCSI_REGLDO, ANTENNA_FCSI_REGLDO_PUPLDOANT_ON__SVAL, ANTENNA_FCSI_REGLDO_PUPLDOANT__SMSK, AFE_REG_DEFAULT_MASK_SHIFT);
}

/***************************************************************************
**
** NAME          AFE_EnableDisableAfeAntennas
**
** PARAMETERS   New number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void AFE_EnableDisableAntennas(uint8 antMask, PowerSaveStates_e powerSaveState)
{
	uint8 chan_width;
	Antenna_e ant;
	AFE_ANTENNA_mode_e antMode;
	int32 bandwidth;
	int32 ctune40M = 0; 
	uint32 AUXmask = AUX0_CH_EN | AUX1_CH_EN | AUX2_CH_EN | AUX3_CH_EN;
	uint32 AUXval = antMask;
	
	// Power down all antennas
	AFE_LdoConfig(antMask);
	abb11ac_ant_config(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, 0, 0);

	// Power up antennas in mask
	
	MT_WrRegMask(PHY_TD_BASE_ADDR, REG_AFE_AUX_CTRL_AUX0_CH_EN, AUXmask, AUXval);
	
    for (ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ant++)
	{
		if ((1 << ant) & antMask)
		{
			antMode = AFE_ANTENNA_STANDBY;
		}	
		else
		{
			antMode = AFE_ANTENNA_POWER_DOWN;
		}			
		AFE_SetAntennaMode(ant, antMode);
	}
	
	chan_width = HDK_getChannelWidth();
	switch (chan_width)
	{
		case BANDWIDTH_TWENTY:
			bandwidth = 10;
			break;
		case BANDWIDTH_FOURTY:
			bandwidth = 20;
			break;	
		case BANDWIDTH_EIGHTY:
			bandwidth = 40;
			break;	
		default:
			DEBUG_ASSERT(0);
	}
	
	ctune40M = AFE_GetCtune(); // TBD Yossi - implement this.
	bandwidth = 40;//Fix afe filter to BW 40 for all BW
	abb11ac_ant_config(antMask, bandwidth, ctune40M);
}
/*********************************************************************************
Method:       AFE_SetSubBlockMode
Description:  Set, per antenna, each ADC/DAC/AUX to active, standby or power
              down mode
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              IN AFE_BLOCK_mode_e mode: active/standby/power-down
Returns:      RetVal_e
*********************************************************************************/
#if defined (AFE_SUB_BLOCK_SUPPORT)
RetVal_e AFE_SetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, IN AFE_SUB_BLOCK_mode_e mode)
{
  uint32 afe_dyn_ctrl_reg = 0;
  uint32 afe_power_down_reg = 0;
  uint32 value = 0;
  // For each antenna we have 9 options, for each block we have 3 options
  uint32 index = (antenna*(AFE_TOTAL_SUB_BLOACKS*BLOCK_TOTAL_MODES) + block*BLOCK_TOTAL_MODES + mode);

  afe_dyn_ctrl_reg = READ( AFE_DYN_CTRL, HW_REGISTER );
  afe_power_down_reg = READ( AFE_PDN_CTRL, HW_REGISTER );

  // AFE_DYN_CTRL settings:
  // Clear the relevant bit
  afe_dyn_ctrl_reg &= ~(1 << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);

  // Set the relevant bit value
  value = (afe_sub_blocks_configuration[index].afe_dyn_ctrl_op << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);
  afe_dyn_ctrl_reg |= value;


  // AFE_PDN_CTRL settings
  // Clear the relevant bit
  afe_power_down_reg &= ~(1 << afe_sub_blocks_configuration[index].afe_power_down_bit);

  // Set the relevant bit value
  value = (afe_sub_blocks_configuration[index].afe_power_down_op << afe_sub_blocks_configuration[index].afe_power_down_bit);
  afe_power_down_reg |= value;

  WRITE( AFE_DYN_CTRL, HW_REGISTER, afe_dyn_ctrl_reg );
  WRITE( AFE_PDN_CTRL, HW_REGISTER, afe_power_down_reg );

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_GetSubBlockMode
Description:  Get, per antenna, each ADC/DAC/AUX state
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              OUT AFE_SUB_BLOCK_mode_e *mode: active/standby/power-down
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, OUT AFE_SUB_BLOCK_mode_e *mode)
{
  uint32 afe_dyn_ctrl_reg = 0;
  uint32 afe_power_down_reg = 0;
  // For each antenna we have 9 options, for each block we have 3 options
  uint32 index = (antenna*(AFE_TOTAL_SUB_BLOACKS*BLOCK_TOTAL_MODES) + block*BLOCK_TOTAL_MODES);

  // Get the AFE_PDN_CTRL settings

  afe_power_down_reg = READ( AFE_PDN_CTRL, HW_REGISTER );

  // if power down bit is set, return block in power down
  afe_power_down_reg &= (1 << afe_sub_blocks_configuration[index].afe_power_down_bit);
  if(afe_power_down_reg)
  {
    *mode = BLOCK_POWER_DOWN;
  }
  else
  {
    // Get the AFE_DYN_CTRL settings
    afe_dyn_ctrl_reg = READ( AFE_DYN_CTRL, HW_REGISTER );

    afe_dyn_ctrl_reg &= (1 << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);
    if(afe_dyn_ctrl_reg)
    {
      // if standby bit is set, return block in standby
      *mode = BLOCK_STANDBY;
    }
    else
    {
      // else return active
      *mode = BLOCK_ACTIVE;
    }
  }

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_SetTRSwitchMode
Description:   
Parameter:
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_SetTRSwitchMode( IN Antenna_e antenna,  IN AFE_TR_SWITCH_mode_e mode)
{
  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_GetTRSwitchMode
Description:   
Parameter:
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetTRSwitchMode( IN Antenna_e antenna, OUT AFE_TR_SWITCH_mode_e *mode)
{
  return RET_VAL_SUCCESS;
}
#endif


/*********************************************************************************
Method:       AFE_SetRegister
Description:  Set AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between the mapped
              registers and the ones accessed by FSCI
              Writes the same value to all 3 antennas registers. To write different values, use AFE_WriteRegisters
Parameter:    IN uint32 reg_addr
                    IN uint32 reg_value
Returns:      None
*********************************************************************************/
void AFE_SetRegister(IN uint32 reg_addr, IN uint32 reg_value)
{
    /* if the desired address is memory mapped */
    if( ( reg_addr >= AFE_MIN_ADDRESS ) && ( reg_addr <= AFE_MAX_ADDRESS) )
    {
        WRITE( reg_addr, HW_REGISTER, reg_value );
    }
    else /* if the desired address is register mapped */
    {
        writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5, reg_addr, reg_value, AFE_REG_DEFAULT_MASK, AFE_REG_DEFAULT_MASK_SHIFT);
    }
}

void AFE_SetRegisterMasked(uint32 ant_mask, uint32 reg_addr,uint32 reg_mask,uint32 reg_value)
{
	/* if the desired address is memory mapped */
	if( ( reg_addr >= AFE_MIN_ADDRESS ) && ( reg_addr <= AFE_MAX_ADDRESS) )
	{
		WRITE( reg_addr, HW_REGISTER, reg_value );
	}
	else /* if the desired address is register mapped */
	{
		writeSameValueToAfeRegister(ant_mask, reg_addr, reg_value, reg_mask, AFE_REG_DEFAULT_MASK_SHIFT);
	}
}

/*********************************************************************************
Method:       AFE_GetRegister
Description:  Get AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between
              the mapped registers and the ones accessed by FSCI
Parameter:    IN uint32 reg_addr
              OUT uint32 *reg_value
Returns:      RetVal_e - success or failure
*********************************************************************************/
RetVal_e AFE_GetRegister(IN uint32 reg_addr, OUT uint16 *reg_value)
{
    INDIRECT_MULTI_ARG_t arg;
    RetVal_e             retVal = RET_VAL_UNINITIALIZED;
    FcsiChannels_e		 channel;

    /* if the desired address is memory mapped */
    if( ( reg_addr >= AFE_MIN_ADDRESS ) && ( reg_addr <= AFE_MAX_ADDRESS) )
    {
        *reg_value = (uint16)READ( reg_addr, HW_REGISTER );
        retVal = RET_VAL_SUCCESS;
    }
    else /* if the desired address is register mapped */
    {
        /* Prepare the address we want to read */
        for (channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
        {
            arg.addr[channel] = reg_addr;
        }

        retVal = IndirectMultipleRead( INDIRECT_BLOCK_AFE, &arg);

        /* Return the array we read */
        for (channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
        {
            reg_value[channel] = (uint16)arg.value[channel];
        }
    }

    return retVal;
}

/*********************************************************************************
Method:       AFE_SetRssiAdcGainAndOffset
Description:  Stores RSSI ADC gain and offset trim values for the selected antenna list
Parameter:    IN AntennaBitmaps_e antList - The antenna list to update
              IN uint8 *gainTrims         - An array of size TOTAL_ANTENNAS with
                                            the gain trim values to store
              IN uint8 *offsetTrim        - An array of size TOTAL_ANTENNAS with
                                            the offset trim values to store
              IN AfeMsSsiAdcMode_e mode   - The mode of operation:
                                            Functional, gain calibration or offset calibration
Returns:      None
*********************************************************************************/
void AFE_SetRssiAdcGainAndOffset(IN AntennaBitmaps_e antList, IN uint8 *gainTrims,
                                        IN uint8 *offsetTrim, IN AfeMsSsiAdcMode_e mode)
{
    uint8     adcMode  = 0;
    Antenna_e ant = ANTENNA_0;
    uint16    regVals[FCSI_TOTAL_CHANNELS];

    switch (mode)
    {
        case AFE_MS_SSI_ADC_MODE_FUNCTIONAL:
            adcMode = AFE_SSI_ADC_MODE_VALUE_FUNCTIONAL;
            for (ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ++ant)
            {
                regVals[ant] = ((offsetTrim[ant]) << REG_AFE_L0_CTRL_OFF_SHIFT) |
                               ((gainTrims[ant])  << REG_AFE_L0_CTRL_GAIN_SHIFT)|
                                (adcMode << REG_AFE_L0_PUP_SSIADC_SHIFT);
            }
            break;

        case AFE_MS_SSI_ADC_MODE_GAIN_CALIBRATION:
            adcMode = AFE_SSI_ADC_MODE_VALUE_GAIN_CALIB;
            for (ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ++ant)
            {
                regVals[ant] = ((gainTrims[ant]) << REG_AFE_L0_CTRL_GAIN_SHIFT) |
                                (adcMode << REG_AFE_L0_PUP_SSIADC_SHIFT);
            }
            break;

        case AFE_MS_SSI_ADC_MODE_OFFSET_CALIBRATION:
            adcMode = AFE_SSI_ADC_MODE_VALUE_OFFSET_CALIB;
            for (ant = ANTENNA_0; ant < TOTAL_ANTENNAS; ++ant)
            {
                regVals[ant] = ((offsetTrim[ant]) << REG_AFE_L0_CTRL_OFF_SHIFT) |
                               ((gainTrims[ant])  << REG_AFE_L0_CTRL_GAIN_SHIFT)|
                                (adcMode << REG_AFE_L0_PUP_SSIADC_SHIFT);
            }
            break;
    }

    writeToAfeRegister(antList, REG_AFE_L0_SSI_ADCREG0, regVals, AFE_REG_DEFAULT_MASK, AFE_REG_DEFAULT_MASK_SHIFT);
}

/*********************************************************************************
Method:		AFE_Set_Dither_Tone_Mode
Description:  not need in Wave500, Dummy Function

*********************************************************************************/
void AFE_Set_Dither_Tone_Frequency(IN uint8 mode)
{			
}

/*********************************************************************************
Method:		AFE_Restore_Dither_Tone_Frequency
Description:  Wave500 Dummy 
Parameter:     None
Returns:      	void
Remarks:		Set original Dither tone
*********************************************************************************/
void AFE_Restore_Dither_Tone_Frequency(void)
{
}


/*********************************************************************************
Method:			AFE_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_ReadRegister(IN uint32 regNum,OUT uint16 *regVal)
{
	AFE_GetRegister(regNum,regVal);
}

/*********************************************************************************
Method:			AFE_WriteRegisters
Description:   Writes 3 diferent values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues
Parameter:     IN AntennaBitmaps_e antList
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 *regValues)
{
    writeToAfeRegister(channelList, regNum, regValues, AFE_REG_DEFAULT_MASK, AFE_REG_DEFAULT_MASK_SHIFT);
}


/*********************************************************************************
Method:			AFE_WriteMaskRegisters
Description:  Writes 3 diferent masked values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues - the value to write (W300) or an array of 3 values to write per antenna (AR10)
Parameter:     IN AntennaBitmaps_e antList
Parameter:     IN uint32 regMask - register mask to write to
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteMaskRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 * regValues,IN uint32 regMask)
{
	writeToAfeRegister(channelList, regNum, regValues, regMask, AFE_REG_DEFAULT_MASK_SHIFT);
}
void AFE_CopyOneRowFromPsd(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint32 *regValues, IN uint32 regMask,uint32 rowIndex)
{
	FcsiChannels_e		 channel;
	afeInitNumberOfrows++;
	AfeDriverPSD[rowIndex].regAddr = regNum;
	AfeDriverPSD[rowIndex].mask= regMask;
	for (channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
    {
		AfeDriverPSD[rowIndex].regData[channel] = regValues[channel];
	}
}
static void writeAfePsdTableFromDB(void)
{
	uint32 rowIndex;
	uint32 index = 0;
	uint16 regData[FCSI_TOTAL_CHANNELS];
	for (rowIndex = 0; rowIndex < afeInitNumberOfrows; rowIndex++)
    {
		for (index = 0; index < FCSI_TOTAL_CHANNELS; index++)
    	{
			regData[index] = AfeDriverPSD[rowIndex].regData[index];
		}
		
		writeToAfeRegister(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5,AfeDriverPSD[rowIndex].regAddr,regData,AfeDriverPSD[rowIndex].mask,AFE_REG_DEFAULT_MASK_SHIFT);
	}
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void AFE_resetPsdArray(void)
{
	memset(&AfeDriverPSD, 0, sizeof(AfeDriverPSD));
	afeInitNumberOfrows = 0;
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


/******************************************************************************/
/***                        Static Functions Definitions                    ***/
/******************************************************************************/

/*********************************************************************************
Method:       writeToAfeRegister
Description:  Writes a value to the AFE registers considering the right mask and shift
Parameter:    IN uint32 regAddr    - register address
*             IN uint32 *regValue - the value we want to write
*             IN uint32 mask       - mask for written value
*             IN uint32 shift      - shift of the written value inside the register
Returns:      None
*********************************************************************************/
static void writeSameValueToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint32 regValue, IN uint32 mask, IN uint32 shift)
{
    uint16    regValues[FCSI_TOTAL_CHANNELS];
    FcsiChannels_e channel;

    for(channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
    {
        regValues[channel] = (uint16)regValue;
    }

    writeToAfeRegister(channelList, regAddr, regValues, mask, shift);
}

/*********************************************************************************
Method:       writeToAfeRegister
Description:  Writes a value to the AFE registers considering the right mask and shift
Parameter:    IN uint32 regAddr    - register address
*             IN uint32 *regValues - the values we want to write
*             IN uint32 mask       - mask for written value
*             IN uint32 shift      - shift of the written value inside the register
Returns:      None
*********************************************************************************/
static void writeToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint16 *regValues, IN uint32 mask, IN uint32 shift)
{
    uint16               readVals[FCSI_TOTAL_CHANNELS];
    FcsiChannels_e		 channel;
    INDIRECT_MULTI_ARG_t multiArgVals;
    RetVal_e             retVal = RET_VAL_UNINITIALIZED;

	UNUSED_PARAM(retVal); //Fix compilation warning

    /* Read, */
    AFE_GetRegister(regAddr, readVals);

    /* Modify,  */
    for (channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
    {
        if (0 != ((0x1 << channel) & channelList))
        {
            /* channel is in the list - Modify the value we read */
            multiArgVals.value[channel] = (((regValues[channel] << shift) & mask) | (readVals[channel] & (~mask)));
        }
        else
        {
            /* channel is not in the list - Keep the value we read */
            multiArgVals.value[channel] = readVals[channel];
        }

        multiArgVals.addr[channel]  = regAddr;
    }

    /* Write */
    retVal = IndirectMultipleWrite(INDIRECT_BLOCK_AFE, &multiArgVals);
	if (retVal != RET_VAL_SUCCESS)
	{
		DEBUG_ASSERT(0);
	}
}

uint16 afeLowPowerBackup[FCSI_TOTAL_CHANNELS];
void AFE_RxDCInitAnt(uint8 ant)
{
	uint8 antMask;
	uint16 afeLowPowerTemp[FCSI_TOTAL_CHANNELS];
	
	antMask = 1 << ant;
	AFE_ReadRegister(ANTENNA_FCSI_IQADCREG0, afeLowPowerTemp);
	afeLowPowerBackup[ant] = afeLowPowerTemp[ant];
		
	AFE_SetRegisterMasked(antMask, ANTENNA_FCSI_IQADCREG0,
							(ANTENNA_FCSI_IQADCREG0_ENLOWPOW__SMSK | ANTENNA_FCSI_IQADCREG0_ENCD__SMSK),
							(ANTENNA_FCSI_IQADCREG0_ENLOWPOW_ENLOWPOW__SVAL | ANTENNA_FCSI_IQADCREG0_ENCD_DISCD__SVAL));
}

void AFE_RxDCrestoreAntAll()
{
	uint8 ant;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		AFE_RxDCrestoreAnt(ant);
	}
}

void AFE_RxDCrestoreAnt(uint8 ant)
{
	uint8 antMask;
	
	antMask = 1 << ant;
	afeLowPowerBackup[ant] &= ANTENNA_FCSI_IQADCREG0_ENLOWPOW__SMSK;
	afeLowPowerBackup[ant] |= ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL;
	AFE_SetRegisterMasked(antMask, ANTENNA_FCSI_IQADCREG0, (ANTENNA_FCSI_IQADCREG0_ENLOWPOW__SMSK | ANTENNA_FCSI_IQADCREG0_ENCD__SMSK), afeLowPowerBackup[ant]);
}

void AFE_RxDCinit()
{
	uint8 ant;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		AFE_RxDCInitAnt(ant);
	}
}

void AFE_RxDCrestore()
{
}

void AFE_SetEfuseData(K_MSG * psMsg)
{
	afe_calibration_data_t* params;
	uint8 index;

	params = (afe_calibration_data_t*)pK_MSG_DATA(psMsg);
	for (index = 0; index < AFE_CALIBRATION_DATA_SIZE_GEN5; index++)
	{
		afeEfuseData[index] = params->calibrationData[index];
	}
}

