/*******************************************************************************
*
*   Source File:
*       Afe.c
*
*  AUTHOR: Oshri Yahav
*
*   Description: Low Level Driver for the AFE module.  
*
*   Copyright:
*       (C) Metalink Ltd.
*       All rights are strictly reserved. Reproduction or divulgence in any
*       form whatsoever is not permitted without written authority from the
*       copyright owner. Issued by Metalink Transmission Devices Ltd in
*       Israel - 11/94.
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***                        Include Files                                   ***/
/******************************************************************************/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "System_Information.h"
#include "PhyRxTdIfRegs.h"
#include "interface_abb.h"
#include "PhyRxTdIfRegs.h"
#include "CalibrationsDefines.h"
#include "ErrorHandler_Api.h"
#include "Afe_API.h"
#include "Afe.h"
#include "PhyRegsIncluder.h"
#include "RegAccess_Api.h"
#include "mem_access.h"
#include "Indirect_API.h"
#include "AFEDefines.h"
#include "mt_sysrst.h"
#include "loggerAPI.h"
#include "lib_abb_api.h"
#include "HwMemoryMap.h"
#include "lib_wav600_api.h"
#include "abb_mode_defines.h"
#include "phy_lib.h"

#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 2

/******************************************************************************/
/***                        Internal Constants                              ***/
/******************************************************************************/
#define AFE_REG_DEFAULT_MASK               (0xFFFF)
#define AFE_REG_DEFAULT_MASK_SHIFT         (0)
#define AFE_REG_NO_SHIFT                   (0)
#define AFE_STBY_OFF            	       (0)
#define AFE_STBY_ON             	       (1)
#define AFE_LOOPBACK_REGISTER_MASK			(0xFF)
#define AFE_LOOPBACK_TX_ANTS_SHIFT			(4)




/* AFE Stand BY Mode register value */
#define AFE_DYN_CTRL_ALL_DAC_STANDBY_MODE  (REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
											REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
											REG_AFE_DYN_CTRL_TX2_STBY_MASK)| \
											REG_AFE_DYN_CTRL_TX3_STBY_MASK)
// All AUX ADC on standby mode
#define AFE_DYN_CTRL_ALL_AUX_ADC_STANDBY  (REG_AFE_DYN_CTRL_AUX0_ADC_STBY_MASK | \
										   REG_AFE_DYN_CTRL_AUX1_ADC_STBY_MASK | \
										   REG_AFE_DYN_CTRL_AUX2_ADC_STBY_MASK)| \
										   REG_AFE_DYN_CTRL_AUX3_ADC_STBY_MASK)

// Loop-back on ANT0 --> ADC0-On, DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT0 (REG_AFE_DYN_CTRL_RX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT1 --> ADC1-On, DAC0-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT1 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT2 --> ADC2-On DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT2 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX3_STBY_MASK)

// Loop-back on ANT2 --> ADC2-On DAC1-On, all others on standby
#define AFE_DYN_CTRL_LOOPBACK_ANT3 (REG_AFE_DYN_CTRL_RX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX2_STBY_MASK | \
									REG_AFE_DYN_CTRL_RX3_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX0_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX1_STBY_MASK | \
									REG_AFE_DYN_CTRL_TX2_STBY_MASK)

/* MS-SSI ADC calibration constants */
#define AFE_SSI_ADC_MODE_VALUE_FUNCTIONAL   (0x03)
#define AFE_SSI_ADC_MODE_VALUE_GAIN_CALIB   (0x13)
#define AFE_SSI_ADC_MODE_VALUE_OFFSET_CALIB (0x23)

/* MS-SSI ADC calibration init/restore values */
#define AFE_SSI_ADC_INIT_L1_VALUE           (0x0770)
#define AFE_SSI_ADC_RESTORE_L1_VALUE        (0x0270)
/* RC Replica tuning constants */
#define AFE_RC_TUNING_POWER_UP              (0x01)
#define AFE_RC_TUNING_POWER_DOWN            (0x00)

#define AFE_RC_TUNING_MEAS_WAIT_TIME        (3000) /* Time to wait for rctuning_meas_done (in microseconds) */
#define AFE_RC_TUNING_MEAS_WAIT_MAX_RETRIES (3)
#define AFE_RC_TUNING_MEAS_DONE_HIGH        (1)

#define AFE_RC_TUNING_1ST_MEAS_SWAP_BIT     (0)
#define AFE_RC_TUNING_2ND_MEAS_SWAP_BIT     (1)

#define AFE_RC_TUNING_FORMULA_A0            (1530539) /* Tuning formula: A0 constant */
#define AFE_RC_TUNING_FORMULA_A1            (2458)    /* Tuning formula: A1 constant */
#define AFE_RC_TUNING_FORMULA_A2            (-17)     /* Tuning formula: A2 constant */

#define AFE_RC_TUNING_GRX350_FACTOR			(58245)
#define AFE_RC_TUNING_FACTOR_SHIFT			(1)
#define AFE_RC_TUNING_GRX350_OFFSET			(9)

#define AFE_RC_TUNING_NLPF_MAX_VALUE        (31)
#define AFE_RC_TUNING_NLPF_MIN_VALUE        (0)

/* AFE init register values constants */
/* set clock-buffer R/I-value */
#define AFE_INIT_L19_REG_VALUE              (0x22A0)
/* Enable LDO */
#define AFE_INIT_L21_REG_VALUE              (0xA988)
/* Write default n/m values */
#define AFE_INIT_L17_REG_VALUE              (0x0261)
/* Enable Clocking */
#define AFE_INIT_L24_REG_VALUE              (0x0771)
/* release all resets, enable clk in channel */
#define AFE_INIT_L7_REG_VALUE               (0x089F)
/* config SSI-ADC */
#define AFE_INIT_L0_REG_VALUE               (0x8403)
#define AFE_INIT_L1_REG_VALUE               (0x0270)
#define AFE_INIT_L2_REG_VALUE               (0x52EC)
/* config IQ-ADC */
#define AFE_INIT_L3_CB_MODE_REG_VALUE       (0x1A0C)
#define AFE_INIT_L3_CB_MODE_REG_MASK        (~REG_AFE_L3_RCTUNE_MASK)
#define AFE_INIT_L3_CB_BW					(0)
#define AFE_INIT_L3_NCB_BW					(1)

#define AFE_INIT_L3_NCB_MODE_REG_VALUE      (0x1B0C)
#define AFE_INIT_L3_NCB_MODE_REG_MASK       (~REG_AFE_L3_RCTUNE_MASK)
#define AFE_INIT_L4_REG_VALUE               (0x4127)
#define AFE_INIT_L5_CB_MODE_REG_VALUE       (0x001E)
#define AFE_INIT_L5_NCB_MODE_REG_VALUE      (0x009E)
#define AFE_INIT_L5_CB_EN_FB_LO				(0)
#define AFE_INIT_L5_NCB_EN_FB_LO			(1)

#define AFE_INIT_L6_REG_VALUE               (0x0000)
/* Set Bias */
#define AFE_INIT_L20_REG_VALUE              (0x009F)

#define AFE_INIT_GRX500_L20_VALUE			(0x8800)
#define AFE_INIT_GRX500_L21_VALUE			(0x8000)
#define AFE_INIT_GRX500_L7_VALUE			(0x189F)
#define AFE_INIT_GRX500_L3_VALUE			(0x100F)
#define AFE_INIT_GRX500_L4_VALUE			(0x1527)
#define AFE_INIT_GRX500_L5_VALUE			(0x0002)

#define AFE_INIT_GLOBAL_MEM_ADDR			(0)
#define AFE_INIT_GLOBAL_MEM_RESET_MASK		(1 << 13)

#define AFE_INIT_GRX500_L7_RX_MUX_ON		(0x989F)
#define AFE_INIT_GRX500_L7_RX_ON			(0x999F)
#define AFE_INIT_GRX500_L7_RX_MUX_OFF_RESET_RX_SYNCOFF (0x189D)
#define AFE_INIT_GRX500_L7_RX_MUX_OFF		(0x189F)

#define AFE_INIT_RX_DELAY_US				(20)


/******************************************************************************/
/***                     Internal Data Structures                           ***/
/******************************************************************************/
// struct definition to save the current register settings
typedef struct
{
  uint32  afe_power_down_reg;
  uint32  afe_dyn_ctrl_reg;
}AFE_BLOCK_STATE_t;

// bit operation, used in various set mode functions
typedef enum
{
  AFE_CLR_BIT = 0,
  AFE_SET_BIT = 1,
}AFE_SUB_BLOCK_OP_e;

// struct definition help configure each AFE sub-block
typedef struct
{
  uint32               afe_dyn_ctrl_bit;
  AFE_SUB_BLOCK_OP_e   afe_dyn_ctrl_op;
  uint32               afe_power_down_bit;
  AFE_SUB_BLOCK_OP_e   afe_power_down_op;
}AFE_SUB_BLOCK_STATE_t;

// RC-Replica tuning parameters and results
typedef struct RcReplicaTuningParameters
{
    uint32 measWait; /* Time to wait for rctuning_meas_done (in microseconds) */
    int32 A0;        /* Tuning formula: A0 constant */
    int32 A1;        /* Tuning formula: A1 constant */
    int32 A2;        /* Tuning formula: A2 constant */
} RcReplicaTuningParameters_t;

typedef struct RcReplicaTuningResults
{
    uint32 Nlpf;     /* Tuning formula: The tuning word to store */
    uint32 rcCnt1;   /* Tuning formula: 1st RC measurement value */
    uint32 rcCnt2;   /* Tuning formula: 2nd RC measurement value */
    uint32 rcCnt;    /* Tuning formula: Average between the two measurements */
} RcReplicaTuningResults_t;

typedef struct RcReplicaTuningElements
{
    RcReplicaTuningParameters_t params;
    RcReplicaTuningResults_t    results;
} RcReplicaTuningElements_t;

#define ADC_CAL_N_COMP	5
#define ADC_CAL_N_CAL	10
#define ADC_CAL_MAT_ROW_SIZE	3
#define ADC_CAL_MAT_COL_SIZE	32
#define ADC_CAL_NOS	100000
#define ADC_CAL_ACCUM_READ_TO	1000

typedef struct AdcAntCalMat
{
	uint8 mAntCal[ADC_CAL_MAT_ROW_SIZE][ADC_CAL_MAT_COL_SIZE];
} AdcAntCalMat_t;

typedef struct AdcCalMat
{
	AdcAntCalMat_t mCal[MAX_NUM_OF_ANTENNAS];
} AdcCalMat_t;



/******************************************************************************/
/***                                Macros                                  ***/
/******************************************************************************/

/******************************************************************************/
/***                      Static Function Declaration                       ***/
/******************************************************************************/
static void writeToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint16 *regValues, IN uint32 mask, IN uint32 shift);
static void writeSameValueToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint32 regValue, IN uint32 mask, IN uint32 shift);
static void writeAfePsdTableFromDB(void);

static void AFE_ADC_CalAccum(uint8 nComp, AdcCalMat_t* calMat);

/******************************************************************************/
/***                           Static Variables                             ***/
/******************************************************************************/
// Global struct that represent the current state of each antenna
static AFE_ANTENNA_mode_e afe_antenna_mode_array[MAX_NUM_OF_ANTENNAS] =
    {    AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED,
         AFE_ANTENNA_UNINITIALZED	};


// This is 3D array hold the registers configuration for active/standby/power_down modes
// Array dimensions [3][3][3].
// For active state we clear the standby bit and the power down bit
// For standby state we set the standby bit and clear the power down bit
// For power down state we set the standby bit and the power down bit
#if defined (AFE_SUB_BLOCK_SUPPORT)
static AFE_SUB_BLOCK_STATE_t afe_sub_blocks_configuration
  [TOTAL_ANTENNAS * AFE_TOTAL_SUB_BLOACKS * BLOCK_TOTAL_MODES] =
{
  // ANTENNA_0
  // ADC
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX0_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_0_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX0_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_0_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX0_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_0_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

  // ANTENNA_1
  // ADC
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX1_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_1_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX1_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_1_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX1_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_1_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

  // ANTENNA_2
  // ADC
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX2_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_2_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX2_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_2_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX2_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_2_PD_BIT, AFE_SET_BIT}, //POWER_DOWN

   // ANTENNA_3
  // ADC
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_RX3_ADC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_RX_3_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  // DAC
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_CLR_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_TX3_DAC_STANDBY_MODE_BIT, AFE_SET_BIT, AFE_PDN_CTRL_TX_3_PD_BIT,      AFE_SET_BIT}, //POWER_DOWN
  //AUX
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_CLR_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_CLR_BIT}, //ACTIVE
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_CLR_BIT}, //STANDBY
  {AFE_DYN_AUX3_STANDBY_MODE_BIT,    AFE_SET_BIT, AFE_PDN_CTRL_AUX_ADC_3_PD_BIT, AFE_SET_BIT}, //POWER_DOWN
};
#endif

static AfePSDRegistersDB_t AfeDriverPSD[AFE_FCSI_MAX_ADDRESS - AFE_FCSI_MIN_ADDRESS];
static uint32 afeInitNumberOfrows = 0;

uint8 afeEfuseData[AFE_CALIBRATION_DATA_SIZE_GEN6] = {0};

int16 ctuneValue;
int8  qTrimValue;

/******************************************************************************/
/***                             Debug Section                              ***/
/******************************************************************************/

/******************************************************************************/
/***                        Public Functions Definitions                    ***/
/******************************************************************************/
/*********************************************************************************
Method:       AFE_Init
Description:  Initialization of the AFE block. This function was created mainly
              for the Wave 400 that requires an initialization sequence unlike
              the Wave 300 that its default values are sufficient.
Parameter:    IN void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e
*********************************************************************************/
RetVal_e AFE_Init(IN void* conf_seg, bool resetCentral)
{
	UNUSED_PARAM(conf_seg);
	UNUSED_PARAM(resetCentral);	
	ctuneValue = -1;
	qTrimValue = -1;
	
    return RET_VAL_SUCCESS;
}

RetVal_e AFE_InitFuse(IN void* conf_seg)
{	
	UNUSED_PARAM(conf_seg);	
    return RET_VAL_SUCCESS;
}

void AFE_InitFuseAnt(void)
{
	uint8 antMask = Hdk_GetTxAntMask();
	
	abb_fcsirf(1, 1, 1);
	ctuneValue = abb_rx_rctune(afeEfuseData);
	abb_antenna_init(antMask);
	abb_antenna_fusing(antMask, afeEfuseData);
	qTrimValue = abb_antenna_rx_calibration(antMask);
	ASSERT(qTrimValue >= 0);
	writeAfePsdTableFromDB();
}

void AFE_AuxadcOff(uint8 antMask)
{
	bbic_auxadc_off(antMask);
}

void AFE_AuxadcOn(uint8 antMask)
{
	bbic_auxadc_on(antMask);
}

void AFE_TxOn(uint8 antMask)
{
	bbic_txon(antMask);
}
void AFE_TxOff(uint8 antMask)
{
	bbic_txoff(antMask);
}
void AFE_RxOn(uint8 antMask)
{
	bbic_rxon(antMask);
}
void AFE_RxOff(uint8 antMask)
{
	bbic_rxoff(antMask);
}

/*********************************************************************************
Method:       AFE_SetAntennaMode
Description:  Set the AFE per antenna state (Rx, Tx, standby, power down). The
              AFE setting should be as follow:
              Mode       | IQADC     | IQDAC      | AuxADC     | PLL        | LDO/BG
              -----------|-----------|------------------------------------------------
              RX MODE    | Active    | Standby    | Active     | Active     | Active
              TX MODE    | Standby   | Active     | Active     | Active     | Active
              Standby    | Standby   | Standby    | Standby    | Active     | Active
              Power Down |Power down | Power down | Power down | Power down | Power down

Parameter:    IN Antenna_e antenna - the requested antenna
              IN AFE_ANTENNA_mode_e antenna_mode - the AFE antenna state
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_SetAntennaMode( IN Antenna_e antenna, IN AFE_ANTENNA_mode_e antenna_mode )
{
	UNUSED_PARAM(antenna);	
	UNUSED_PARAM(antenna_mode);	
	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_GetAntennaMode
Description:  Get the current state of the AFE per antenna (Rx, Tx, standby, power down)
Parameter:    IN Antenna_e antenna - the requested antenna
              OUT AFE_ANTENNA_mode_e *antenna_mode - the AFE antenna state
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetAntennaMode( IN Antenna_e antenna, OUT AFE_ANTENNA_mode_e *antenna_mode )
{
  if(antenna > MAX_ANTENNA)
  {
    return RET_VAL_BAD_PARAM;
  }

  *antenna_mode = afe_antenna_mode_array[antenna];

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:      AFE_GetCtune
Description:  Get Ctune value
Parameter:   
Returns:
*********************************************************************************/
int16 AFE_GetCtune(void)
{
	return ctuneValue;
}
/*********************************************************************************
Method:      AFE_GetQTrim
Description:  Get qTrim value
Parameter:   
Returns:
*********************************************************************************/
int8 AFE_GetQTrim(void)
{
	return qTrimValue;
}
/***************************************************************************
**
** NAME          AFE_EnableDisableAfeAntennas
**
** PARAMETERS   New number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
uint32 afeAntennaFuseCounter = 0;
void AFE_EnableDisableAntennas(uint8 antMask, PowerSaveStates_e powerSaveState)
{
	int16 ctune;
	int8 qtrim;
	uint8 abbmode;
	UNUSED_PARAM(antMask);	
	if(afeAntennaFuseCounter == 0)
	{
		AFE_InitFuseAnt();
		afeAntennaFuseCounter++;
	}
	ctune = AFE_GetCtune();
	qtrim = AFE_GetQTrim();
	abbmode = (powerSaveState == RADIO_OFF_ERP_DUTY_CYCLE) ? ABB_STANDBY : ABB_11AX;
	
	ILOG0_DDD("abb_antenna_config() abbmode: %d ctune: %d qtrim: %d", abbmode, ctune, qtrim); 
	abb_antenna_config(Hdk_GetTxAntMask(), abbmode, ctune, qtrim);
}
/*********************************************************************************
Method:       AFE_SetSubBlockMode
Description:  Set, per antenna, each ADC/DAC/AUX to active, standby or power
              down mode
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              IN AFE_BLOCK_mode_e mode: active/standby/power-down
Returns:      RetVal_e
*********************************************************************************/
#if defined (AFE_SUB_BLOCK_SUPPORT)
RetVal_e AFE_SetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, IN AFE_SUB_BLOCK_mode_e mode)
{
  uint32 afe_dyn_ctrl_reg = 0;
  uint32 afe_power_down_reg = 0;
  uint32 value = 0;
  // For each antenna we have 9 options, for each block we have 3 options
  uint32 index = (antenna*(AFE_TOTAL_SUB_BLOACKS*BLOCK_TOTAL_MODES) + block*BLOCK_TOTAL_MODES + mode);

  afe_dyn_ctrl_reg = READ( AFE_DYN_CTRL, HW_REGISTER );
  afe_power_down_reg = READ( AFE_PDN_CTRL, HW_REGISTER );

  // AFE_DYN_CTRL settings:
  // Clear the relevant bit
  afe_dyn_ctrl_reg &= ~(1 << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);

  // Set the relevant bit value
  value = (afe_sub_blocks_configuration[index].afe_dyn_ctrl_op << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);
  afe_dyn_ctrl_reg |= value;


  // AFE_PDN_CTRL settings
  // Clear the relevant bit
  afe_power_down_reg &= ~(1 << afe_sub_blocks_configuration[index].afe_power_down_bit);

  // Set the relevant bit value
  value = (afe_sub_blocks_configuration[index].afe_power_down_op << afe_sub_blocks_configuration[index].afe_power_down_bit);
  afe_power_down_reg |= value;

  WRITE( AFE_DYN_CTRL, HW_REGISTER, afe_dyn_ctrl_reg );
  WRITE( AFE_PDN_CTRL, HW_REGISTER, afe_power_down_reg );

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_GetSubBlockMode
Description:  Get, per antenna, each ADC/DAC/AUX state
Parameter:    IN Antenna_e antenna: 0-2
              IN AFE_SUB_BLOCK_e block: ADC/DAC/AUX
              OUT AFE_SUB_BLOCK_mode_e *mode: active/standby/power-down
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetSubBlockMode( IN Antenna_e antenna, IN AFE_SUB_BLOCK_e block, OUT AFE_SUB_BLOCK_mode_e *mode)
{
  uint32 afe_dyn_ctrl_reg = 0;
  uint32 afe_power_down_reg = 0;
  // For each antenna we have 9 options, for each block we have 3 options
  uint32 index = (antenna*(AFE_TOTAL_SUB_BLOACKS*BLOCK_TOTAL_MODES) + block*BLOCK_TOTAL_MODES);

  // Get the AFE_PDN_CTRL settings

  afe_power_down_reg = READ( AFE_PDN_CTRL, HW_REGISTER );

  // if power down bit is set, return block in power down
  afe_power_down_reg &= (1 << afe_sub_blocks_configuration[index].afe_power_down_bit);
  if(afe_power_down_reg)
  {
    *mode = BLOCK_POWER_DOWN;
  }
  else
  {
    // Get the AFE_DYN_CTRL settings
    afe_dyn_ctrl_reg = READ( AFE_DYN_CTRL, HW_REGISTER );

    afe_dyn_ctrl_reg &= (1 << afe_sub_blocks_configuration[index].afe_dyn_ctrl_bit);
    if(afe_dyn_ctrl_reg)
    {
      // if standby bit is set, return block in standby
      *mode = BLOCK_STANDBY;
    }
    else
    {
      // else return active
      *mode = BLOCK_ACTIVE;
    }
  }

  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_SetTRSwitchMode
Description:   
Parameter:
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_SetTRSwitchMode( IN Antenna_e antenna,  IN AFE_TR_SWITCH_mode_e mode)
{
  return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_GetTRSwitchMode
Description:  
Parameter:
Returns:      RetVal_e
*********************************************************************************/
RetVal_e AFE_GetTRSwitchMode( IN Antenna_e antenna, OUT AFE_TR_SWITCH_mode_e *mode)
{
  return RET_VAL_SUCCESS;
}
#endif


/*********************************************************************************
Method:       AFE_SetRegister
Description:  Set AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between the mapped
              registers and the ones accessed by FSCI
              Writes the same value to all 3 antennas registers. To write different values, use AFE_WriteRegisters
Parameter:    IN uint32 reg_addr
                    IN uint32 reg_value
Returns:      None
*********************************************************************************/
void AFE_SetRegister(IN uint32 reg_addr, IN uint32 reg_value)
{
    /* if the desired address is memory mapped */
    if( ( reg_addr >= AFE_MIN_ADDRESS ) && ( reg_addr <= AFE_MAX_ADDRESS) )
    {
        WRITE( reg_addr, HW_REGISTER, reg_value );
    }
    else /* if the desired address is register mapped */
    {
        writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5, reg_addr, reg_value, AFE_REG_DEFAULT_MASK, AFE_REG_DEFAULT_MASK_SHIFT);
    }
}

void AFE_SetRegisterMasked(uint32 ant_mask, uint32 reg_addr,uint32 reg_mask,uint32 reg_value)
{
	/* if the desired address is memory mapped */
	if( ( reg_addr >= AFE_MIN_ADDRESS ) && ( reg_addr <= AFE_MAX_ADDRESS) )
	{
		WRITE( reg_addr, HW_REGISTER, reg_value );
	}
	else /* if the desired address is register mapped */
	{
		writeSameValueToAfeRegister(ant_mask, reg_addr, reg_value, reg_mask, AFE_REG_DEFAULT_MASK_SHIFT);
	}
}

/*********************************************************************************
Method:       AFE_GetRegister
Description:  Get AFE register value. since in Wave 400 some of the register
              can be accessed only by FCSI, we need to differentiate between
              the mapped registers and the ones accessed by FSCI
Parameter:    IN uint32 reg_addr
              OUT uint32 *reg_value
Returns:      RetVal_e - success or failure
*********************************************************************************/
RetVal_e AFE_GetRegister(IN uint32 reg_addr, OUT uint16 *reg_value)
{
	uint32		antMask;
	bool		success;

	antMask = Hdk_GetTxAntMask();

	FCSI_READ_TRANSACTION_CENTRAL(1, 0, reg_addr, &reg_value[FCSI_CHANNEL_E], &success);
	ASSERT(success);

	FCSI_READ_TRANSACTION_ANT(1, 0, antMask, reg_addr, reg_value, &success);
	ASSERT(success);
	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       AFE_SetRssiAdcGainAndOffset
Description:  Stores RSSI ADC gain and offset trim values for the selected antenna list
Parameter:    IN AntennaBitmaps_e antList - The antenna list to update
              IN uint8 *gainTrims         - An array of size TOTAL_ANTENNAS with
                                            the gain trim values to store
              IN uint8 *offsetTrim        - An array of size TOTAL_ANTENNAS with
                                            the offset trim values to store
              IN AfeMsSsiAdcMode_e mode   - The mode of operation:
                                            Functional, gain calibration or offset calibration
Returns:      None
*********************************************************************************/
void AFE_SetRssiAdcGainAndOffset(IN AntennaBitmaps_e antList, IN uint8 *gainTrims,
                                        IN uint8 *offsetTrim, IN AfeMsSsiAdcMode_e mode)
{
	UNUSED_PARAM(antList);	
	UNUSED_PARAM(gainTrims);	
	UNUSED_PARAM(offsetTrim);	
	UNUSED_PARAM(mode);	
}



/*********************************************************************************
Method:       AFE_SetSpectrumMode
Description:  Sets the AFE registers related to CB/nCB mode
Parameter:    IN SpectrumMode_e spectrumMode - The required spectrum mode
                                               SPECTRUM_MODE_CB or SPECTRUM_MODE_nCB
Returns:      void
*********************************************************************************/
void AFE_SetSpectrumMode(IN SpectrumMode_e spectrumMode)
{
    switch(spectrumMode)
    {
        case SPECTRUM_MODE_CB:
            writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_ANTENNAS, REG_AFE_L3_IQ_ADCREG0, AFE_INIT_L3_CB_BW, REG_AFE_L3_BW_MASK, REG_AFE_L3_BW_SHIFT);
            writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_ANTENNAS, REG_AFE_L5_IQ_DACREG0, AFE_INIT_L5_CB_EN_FB_LO, REG_AFE_L5_EN_FB_LO_MASK, REG_AFE_L5_EN_FB_LO_SHIFT);
            break;

        case SPECTRUM_MODE_NCB:
            writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_ANTENNAS, REG_AFE_L3_IQ_ADCREG0, AFE_INIT_L3_NCB_BW, REG_AFE_L3_BW_MASK, REG_AFE_L3_BW_SHIFT);
            writeSameValueToAfeRegister(ANTENNA_BITMAP_ALL_ANTENNAS, REG_AFE_L5_IQ_DACREG0, AFE_INIT_L5_NCB_EN_FB_LO, REG_AFE_L5_EN_FB_LO_MASK, REG_AFE_L5_EN_FB_LO_SHIFT);
            break;

        default:
			return; //temporal for 80MHz configuration
    }
}

/*********************************************************************************
Method:		AFE_Set_Dither_Tone_Mode
Description:  not need in Wave500, Dummy Function

*********************************************************************************/
void AFE_Set_Dither_Tone_Frequency(IN uint8 mode)
{	
	UNUSED_PARAM(mode);	
}

/*********************************************************************************
Method:		AFE_Restore_Dither_Tone_Frequency
Description:  Wave500 Dummy 
Parameter:     None
Returns:      	void
Remarks:		Set original Dither tone
*********************************************************************************/
void AFE_Restore_Dither_Tone_Frequency(void)
{
}


/*********************************************************************************
Method:			AFE_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_ReadRegister(IN uint32 regNum,OUT uint16 *regVal)
{
	AFE_GetRegister(regNum,regVal);
}

/*********************************************************************************
Method:			AFE_WriteRegisters
Description:   Writes 3 diferent values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues
Parameter:     IN AntennaBitmaps_e antList
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 *regValues)
{
    writeToAfeRegister(channelList, regNum, regValues, AFE_REG_DEFAULT_MASK, AFE_REG_DEFAULT_MASK_SHIFT);
}


/*********************************************************************************
Method:			AFE_WriteMaskRegisters
Description:  Writes 3 diferent masked values to each antenna's register. To write same value use AFE_SetRegister
Parameter:     IN uint32 regNum
Parameter:     IN uint32 *regValues - the value to write (W300) or an array of 3 values to write per antenna (AR10)
Parameter:     IN AntennaBitmaps_e antList
Parameter:     IN uint32 regMask - register mask to write to
Returns:      	void
Remarks:		None
*********************************************************************************/
void AFE_WriteMaskRegisters(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint16 * regValues,IN uint32 regMask)
{
	writeToAfeRegister(channelList, regNum, regValues, regMask, AFE_REG_DEFAULT_MASK_SHIFT);
}
void AFE_CopyOneRowFromPsd(IN AntennaBitmaps_e channelList, IN uint32 regNum,IN uint32 *regValues, IN uint32 regMask,uint32 rowIndex)
{
	FcsiChannels_e		 channel;
	UNUSED_PARAM(channelList);	
	afeInitNumberOfrows++;
	AfeDriverPSD[rowIndex].regAddr = regNum;
	AfeDriverPSD[rowIndex].mask= regMask;
	for (channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
    {
		AfeDriverPSD[rowIndex].regData[channel] = regValues[channel];
	}
}
static void writeAfePsdTableFromDB(void)
{
	uint32 rowIndex;
	uint32 index = 0;
	uint16 regData[FCSI_TOTAL_CHANNELS];
	for (rowIndex = 0; rowIndex < afeInitNumberOfrows; rowIndex++)
    {
		for (index = 0; index < FCSI_TOTAL_CHANNELS; index++)
    	{
			regData[index] = AfeDriverPSD[rowIndex].regData[index];
		}
		
		writeToAfeRegister(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5,AfeDriverPSD[rowIndex].regAddr,regData,AfeDriverPSD[rowIndex].mask,AFE_REG_DEFAULT_MASK_SHIFT);
	}
}


void AFE_resetPsdArray(void)
{
	memset(&AfeDriverPSD, 0, sizeof(AfeDriverPSD));
	afeInitNumberOfrows = 0;
}

/******************************************************************************/
/***                        Static Functions Definitions                    ***/
/******************************************************************************/
/*********************************************************************************
Method:       writeToAfeRegister
Description:  Writes a value to the AFE registers considering the right mask and shift
Parameter:    IN uint32 regAddr    - register address
*             IN uint32 *regValue - the value we want to write
*             IN uint32 mask       - mask for written value
*             IN uint32 shift      - shift of the written value inside the register
Returns:      None
*********************************************************************************/
static void writeSameValueToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint32 regValue, IN uint32 mask, IN uint32 shift)
{
    uint16    regValues[FCSI_TOTAL_CHANNELS];
    FcsiChannels_e channel;

    for(channel = FCSI_CHANNEL_A; channel < FCSI_TOTAL_CHANNELS; channel++)
    {
        regValues[channel] = (uint16)regValue;
    }

    writeToAfeRegister(channelList, regAddr, regValues, mask, shift);
}

/*********************************************************************************
Method:       writeToAfeRegister
Description:  Writes a value to the AFE registers considering the right mask and shift
Parameter:    IN uint32 regAddr    - register address
*             IN uint32 *regValues - the values we want to write
*             IN uint32 mask       - mask for written value
*             IN uint32 shift      - shift of the written value inside the register
Returns:      None
*********************************************************************************/
static void writeToAfeRegister(IN uint32 channelList, IN uint32 regAddr, IN uint16 *regValues, IN uint32 mask, IN uint32 shift)
{
	uint16		centralVal = 0;
	uint16 		antVal[MAX_NUM_OF_ANTENNAS] = {0};
	uint8		ant;
	bool		success;

	channelList &= (Hdk_GetTxAntMask() | (1 << FCSI_CHANNEL_E)); //mask out antennas not assigned to band (keep central)

	if ((channelList & (1 << FCSI_CHANNEL_E)) != 0) //if central
	{
		FCSI_READ_TRANSACTION_CENTRAL(1, 0, regAddr, &centralVal, &success);
		centralVal = (((regValues[FCSI_CHANNEL_E] << shift) & mask) | (centralVal & (~mask)));
		FCSI_WRITE_TRANSACTION_CENTRAL(1, 0, regAddr, centralVal, &success);
		ASSERT(success);		
		channelList &= ~(1 << FCSI_CHANNEL_E);
	}
	
	if (channelList  != 0)
	{
		FCSI_READ_TRANSACTION_ANT(1, 0, channelList, regAddr, antVal, &success);
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if ((channelList & (1 << ant)) != 0)
			{
				antVal[ant] = (((regValues[ant] << shift) & mask) | (antVal[ant] & (~mask)));
			}
		}
		FCSI_WRITE_TRANSACTION_ANT(1, 0, channelList, regAddr, antVal, &success);
		ASSERT(success);
	}
}

void AFE_RxDCInitAnt(uint8 ant)
{
	UNUSED_PARAM(ant);	
}

void AFE_RxDCrestoreAntAll()
{
}
void AFE_RxDCrestoreAnt(uint8 ant)
{
	UNUSED_PARAM(ant);
}

void AFE_RxDCinit()
{
}

void AFE_RxDCrestore()
{
}

void AFE_SetEfuseData(K_MSG * psMsg)
{	
	afe_calibration_data_t* params;
	uint8 index;

	params = (afe_calibration_data_t*)pK_MSG_DATA(psMsg);
	for (index = 0; index < AFE_CALIBRATION_DATA_SIZE_GEN6; index++)
	{
		afeEfuseData[index] = params->calibrationData[index];
	}
}

void AFE_ADC_Calibration(void)
{
	
	uint8 antMask = Hdk_GetTxAntMask();
	uint8 antNumber;
	AdcCalMat_t calmat;
	AFE_ADC_CalAccum(ADC_CAL_N_COMP, &calmat);

	ILOG0_V("glick ADC_Calibration");
	SLOG0(0, 0, AdcCalMat_t, &calmat);

	abb_antenna_rx_qtmanen(antMask, 1);
	for (antNumber = 0; antNumber < MAX_NUM_OF_ANTENNAS; antNumber++)
	{
		if(antMask&(1<<antNumber))
		{
			abb_antenna_rx_adccalman(1<<antNumber, calmat.mCal[antNumber].mAntCal, ADC_CAL_N_COMP);
		}
	}
	abb_antenna_rx_qtmanen(antMask, 0);
}
CorrResults_t	corrRes[MAX_NUM_OF_ANTENNAS];
	CorrOF_t		corrOf[MAX_NUM_OF_ANTENNAS];
	uint8 iCal;
	uint8 vUse[ADC_CAL_MAT_COL_SIZE];
static void AFE_ADC_CalAccum(uint8 nComp, AdcCalMat_t * calMat)
{
	
	uint8 vAddr[ADC_CAL_N_COMP] = {14, 15, 16, 17, 18};
	uint8 mVal[ADC_CAL_N_CAL][ADC_CAL_N_COMP] = {{0, 0, 0, 0, 0},//0
												 {0, 0, 0, 15, 0},// 1
												 {0, 0, 15, 15, 0},// 2
												 {0, 0, 7, 15, 0},// 3
												 {0, 7, 7, 15, 0},// 4
												 {0, 7, 15, 15, 0},// 5
												 {0, 7, 15, 0, 0},//6
												 {0, 7, 7, 0, 0},//7
												 {0, 7, 0, 0, 0},//8
												 {0, 7, 0, 15, 0}};//9
	uint32 nos[MAX_NUM_OF_ANTENNAS] = {ADC_CAL_NOS, ADC_CAL_NOS, ADC_CAL_NOS, ADC_CAL_NOS};
	uint8 iirShift[MAX_NUM_OF_ANTENNAS] = {6, 6, 6, 6};
	uint8 zeroAntValues[MAX_NUM_OF_ANTENNAS] = {0, 0, 0, 0};
	uint8 accumshift[MAX_NUM_OF_ANTENNAS] = {4, 4, 4, 4};
	
	uint32 IAccumMin[MAX_NUM_OF_ANTENNAS] = {MAX_UINT32, MAX_UINT32, MAX_UINT32, MAX_UINT32};
	uint32 QAccumMin[MAX_NUM_OF_ANTENNAS] = {MAX_UINT32, MAX_UINT32, MAX_UINT32, MAX_UINT32};
	uint8 IAccumMinIdx[MAX_NUM_OF_ANTENNAS] = {0, 0, 0, 0};
	uint8 QAccumMinIdx[MAX_NUM_OF_ANTENNAS] = {0, 0, 0, 0};
	
	uint8 iUse;
	uint8 iComp;
	uint8 nUse;
	uint8 iAnt;
	uint8 antMask = Hdk_GetTxAntMask();
	uint8 validMask;
	bbic_set_iir_shift(antMask , iirShift);
	bbic_enable_iir(antMask, 1);
	MT_Delay(500);
	int i;
	bbic_set_accum_params(antMask, nos, zeroAntValues, zeroAntValues, zeroAntValues, accumshift);

	for (iCal = 0; iCal < ADC_CAL_N_CAL; iCal++)
	{
		if (iCal == 0)
		{
			nUse = ADC_CAL_N_COMP;
			for (iUse = 0; iUse < nUse; iUse++)
			{
				vUse[iUse] = iUse;
			}
		}
		else
		{
			nUse = 0;
			for (iComp = 0; iComp < ADC_CAL_N_COMP; iComp++)
			{
				if (mVal[iCal - 1][iComp] != mVal[iCal][iComp])
				{
					vUse[nUse] = iComp;
					nUse++;
				}
			}
		}
		for(i=0;i<5;i++)
		{
			ILOG0_DD("vUse[%d]:%d",i,vUse[i]);
		}
		ILOG0_D("iCal:%d",iCal);
		for (iUse = 0; iUse < nUse; iUse++)
		{
			ILOG0_DDD("iUse:%d, vUse[iUse]:%d, mVal[iCal][vUse[iUse]]:%d",iUse,vUse[iUse],mVal[iCal][vUse[iUse]]);
			calMat->mCal[0].mAntCal[0][iUse] = vAddr[vUse[iUse]];
			calMat->mCal[0].mAntCal[1][iUse] = mVal[iCal][vUse[iUse]];
			calMat->mCal[0].mAntCal[2][iUse] = mVal[iCal][vUse[iUse]];
		}
		
		SLOG0(0, 0, AdcAntCalMat_t, &(calMat->mCal[0]));
		abb_antenna_rx_adccalman(antMask, calMat->mCal[0].mAntCal, nUse);
		validMask = bbic_read_accum(antMask, ADC_CAL_ACCUM_READ_TO*10, corrRes, corrOf);
		if(validMask != antMask)
		{
			ASSERT(0);
		}
		if(corrOf[0].II == 1)
		{
			ASSERT(0);
		}
		if(corrOf[1].II == 1)
		{
			ASSERT(0);
		}if(corrOf[2].II == 1)
		{
			ASSERT(0);
		}
		if(corrOf[3].II == 1)
		{
			ASSERT(0);
		}
		
		for (iAnt = 0; iAnt < MAX_NUM_OF_ANTENNAS; iAnt++)
		{
			if (corrRes[iAnt].II.pwrVal < IAccumMin[iAnt])
			{
				IAccumMin[iAnt] = corrRes[iAnt].II.pwrVal;
				IAccumMinIdx[iAnt] = iCal;
			}
			if (corrRes[iAnt].QQ.pwrVal < QAccumMin[iAnt])
			{
				QAccumMin[iAnt] = corrRes[iAnt].QQ.pwrVal;
				QAccumMinIdx[iAnt] = iCal;
			}
			ILOG0_DDDD("ant:%d, iCal:%d icomp:%d qcomp:%d",iAnt,iCal,corrRes[iAnt].II.pwrVal, corrRes[iAnt].QQ.pwrVal);
		}
	}

	for (iAnt = 0; iAnt < MAX_NUM_OF_ANTENNAS; iAnt++)
	{
		for (iComp = 0; iComp < nComp; iComp++)
		{
			calMat->mCal[iAnt].mAntCal[0][iComp] = vAddr[iComp];
			calMat->mCal[iAnt].mAntCal[1][iComp] = mVal[IAccumMinIdx[iAnt]][iComp];
			calMat->mCal[iAnt].mAntCal[2][iComp] = mVal[QAccumMinIdx[iAnt]][iComp];
			ILOG0_DDD("min!!!!! ant:%d icomp:%d qcomp:%d",iAnt,IAccumMinIdx[iAnt],QAccumMinIdx[iAnt]);
		}
	}

	abb_antenna_rx_qtmanen(antMask, 0);
}


