#ifndef INDIRECT_API_H_
#define INDIRECT_API_H_

#include "System_GlobalDefinitions.h"
#include "HdkGlobalDefs.h"
/**************************************************************************************/

// This enum will help us distinct between the 2 bus type we support
typedef enum
{
  INDIRECT_ACCESS_BUS_PIF,
  INDIRECT_ACCESS_BUS_FCSI,
}INDIRECT_BUS_TYPE_e;


// interfaces enumeration
typedef enum
{
  INDIRECT_BLOCK_RF_EXT, // external
  INDIRECT_BLOCK_RF_INT, // internal
  INDIRECT_BLOCK_AFE,
  INDIRECT_BLOCK_MAX = INDIRECT_BLOCK_RF_INT,
  INDIRECT_BLOCK_ALL,
}INDIRECT_BLOCK_e;

// This structure will be used for the multiple read/write API
typedef struct
{
  uint32 addr[FCSI_TOTAL_CHANNELS];
  uint32 value[FCSI_TOTAL_CHANNELS];
}INDIRECT_MULTI_ARG_t;

/*********************************************************************************
Method:       IndirectGetType
Description:  Return the indirect bus type (currently FCSI or PIF)
Parameter:    None
Returns:      INDIRECT_BUS_TYPE_e
*********************************************************************************/
INDIRECT_BUS_TYPE_e IndirectGetType( void );

/*********************************************************************************
Method:       IndirectReset
Description:  Reset the indirect access bus controller to default values.
Parameter:    void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectReset( void* conf_seg );
void IndirectResetAnt(void);

/*********************************************************************************
Method:       IndirectSingleWrite
Description:  Set a single indirect register.  
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal. WAVE300
                support only external RF.
              IN uint32 address  register address in the block.
              IN uint32 value  the value to be set.
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectSingleWrite( IN INDIRECT_BLOCK_e block,
                              IN uint32 address,
                              IN uint32 value);

/*********************************************************************************
Method:       IndirectMultipleWrite
Description:  Set 3 indirect registers simultaneously. WAVE300 does not support this API.
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal.
              IN INDIRECT_MULTI_ARG_t arg - pointer to a struct that will
                contain the 3 addresses and the values
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectMultipleWrite( IN INDIRECT_BLOCK_e block,
                                IN INDIRECT_MULTI_ARG_t *arg);

/*********************************************************************************
Method:       IndirectSingleRead
Description:  Get a single indirect register. 
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal.
              IN uint32 address  register address in this module.
              OUT uint32 *value  the return value.
Returns:      Status_e - success or failure.
*********************************************************************************/
RetVal_e IndirectSingleRead( IN  INDIRECT_BLOCK_e block,
                             IN  uint32 address,
                             OUT uint32 *value);

/*********************************************************************************
Method:       IndirectMultipleRead
Description:  Get a single indirect register. WAVE300 does not support this API.
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external, 
                and RF external on Wave300.
              INOUT INDIRECT_MULTI_ARG_t *arg - pointer to a struct that will
                contain the 3 addresses and the values
Returns:      Status_e - success or failure.
*********************************************************************************/
RetVal_e IndirectMultipleRead( IN    INDIRECT_BLOCK_e block,
                               INOUT INDIRECT_MULTI_ARG_t *arg);

#endif /* INDIRECT_API_H_ */
