/*******************************************************************************
*
*   Source File:
*       Fcsi.c
*
*  AUTHOR: Oshri Yahav
*
*   Description: Low Level Driver for the FCSI module.  
*
*   Copyright:
*       (C) Metalink Ltd.
*       All rights are strictly reserved. Reproduction or divulgence in any
*       form whatsoever is not permitted without written authority from the
*       copyright owner. Issued by Metalink Transmission Devices Ltd in
*       Israel - 11/94.
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***                        Include Files                                   ***/
/******************************************************************************/
#include "OSAL_Api.h"
#include "Indirect_API.h"
#include "PhyRegsIncluder.h"
#include "HdkGlobalDefs.h"
#include "RegAccess_Api.h"
#include "PhyRxTdIfRiscPage0TdRegs.h"
#include "PhyRxTdIfRiscPage0FdRegs.h"
#include "PhyRxTdRegs.h"
#include "PhyTxRegsRegs.h"
#include "mt_sysrst.h"
#include "PhyDriver_API.h"


/******************************************************************************/
/***                        Internal Constants                              ***/
/******************************************************************************/
#define RISC_ARBITER_GCLK_BYPASS_ENABLE 0x00800000
#define FCSI_CLOCK_DIVIDERS_VALUES		0x0F02

#define FCSI_DONE_DELAY 300 // the delay time to checking the indirect command completion

#define FCSI_DATA_A		0
#define FCSI_DATA_B		1
#define FCSI_DATA_C		2
#define FCSI_DATA_D		3
#define FCSI_DATA_E		4

#define FCSI_ADRESS_A	0
#define FCSI_ADRESS_B	1
#define FCSI_ADRESS_C	2
#define FCSI_ADRESS_D	3
#define FCSI_ADRESS_E	4

#define FCSI_READ_BIT   (1) /* when using multiple read, we need to turn this bit on in address A's register */
#define FCSI_WRITE_BIT  (0) /* when using multiple write, we set the read/write bit to 0 in address A's register */
/******************************************************************************/
/***                     Internal Data Structures                           ***/
/******************************************************************************/

typedef enum FcsiReadWrite
{
    FCSI_WRITE = 0,
    FCSI_READ = 1,
} FcsiReadWrite_e;

/******************************************************************************/
/***                                Macros                                  ***/
/******************************************************************************/




/******************************************************************************/
/***                      Static Function Declaration                       ***/
/******************************************************************************/
static void configureFcsiAdressRegisters(IN INDIRECT_BLOCK_e block,IN uint32 *addr, IN FcsiReadWrite_e readOrWrite);
static RetVal_e waitFcsiIdle(void);
/******************************************************************************/
/***                           Static Variables                             ***/
/******************************************************************************/

/******************************************************************************/
/***                             Debug Section                              ***/
/******************************************************************************/

/******************************************************************************/
/***                        Public Functions Definitions                    ***/
/******************************************************************************/

/*********************************************************************************
Method:       IndirectGetType
Description:  Return the indirect bus type (currently FCSI or PIF)
Parameter:    None
Returns:      INDIRECT_BUS_TYPE_e
*********************************************************************************/
INDIRECT_BUS_TYPE_e IndirectGetType( void )
{
  return INDIRECT_ACCESS_BUS_FCSI;
}

/*********************************************************************************
Method:       IndirectReset
Description:  Reset the indirect access bus controller to default values.
Parameter:    void* conf_seg - part of the prog-model the F/W receives at init
              time.
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectReset( void* conf_seg )
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf137_u resetReg;
	
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, 0xFFFF);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG80, 0x840000);
	
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, &resetReg.val);
	resetReg.bitFields.fcsiRfResetN = 0;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
	MT_DELAY(10);
	resetReg.bitFields.fcsiRfResetN = 1;
	resetReg.bitFields.fcsiMsResetAntennasN = 1;
	resetReg.bitFields.fcsiMsResetCentralN =1;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
	
	RegAccess_WriteMasked(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF128, 0x3, 0x2);
	return RET_VAL_SUCCESS;
}
void IndirectResetAnt(void)
{
}

/*********************************************************************************
Method:       IndirectSingleWrite
Description:  Set a single indirect register.  
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal.
              IN uint32 address  register address in the block.
              IN uint32 value  the value to be set.
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectSingleWrite( IN INDIRECT_BLOCK_e block, IN uint32 address, IN uint32 value)
{
  return RET_VAL_NOT_SUPPORTED;
}

/*********************************************************************************
Method:       IndirectMultipleWrite
Description:  Set 3 indirect registers simultaneously.
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal.
              IN INDIRECT_MULTI_ARG_t arg - pointer to a struct that will
                contain the 3 addresses and the values
Returns:      Status_e - success or failure
*********************************************************************************/
RetVal_e IndirectMultipleWrite( IN INDIRECT_BLOCK_e block, IN INDIRECT_MULTI_ARG_t *arg)
{
	//Verify FCSI is not Busy 
	waitFcsiIdle(); 


	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12C, arg->value[FCSI_DATA_A]);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12D, arg->value[FCSI_DATA_B]);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12E, arg->value[FCSI_DATA_C]);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12F, arg->value[FCSI_DATA_D]);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF130, arg->value[FCSI_DATA_E]);

	// Configure Fcsi Addresses Registers & Initiate Write transaction
	configureFcsiAdressRegisters(block,arg->addr,FCSI_WRITE);
	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       IndirectSingleRead
Description:  Get a single indirect register.  
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external or RF internal.
              IN uint32 address  register address in this module.
              OUT uint32 *value  the return value.
Returns:      Status_e - success or failure.
*********************************************************************************/
RetVal_e IndirectSingleRead( IN INDIRECT_BLOCK_e block, IN uint32 address, OUT uint32 *value)
{
  return RET_VAL_NOT_SUPPORTED;
}



/*********************************************************************************
Method:       IndirectMultipleRead
Description:  Get a single indirect register.
Parameter:    IN INDIRECT_BLOCK_e block - AFE, RF external, 
              INOUT INDIRECT_MULTI_ARG_t *arg - pointer to a struct that will
                contain the 3 addresses and the values
Returns:      Status_e - success or failure.
*********************************************************************************/
RetVal_e IndirectMultipleRead( IN INDIRECT_BLOCK_e block, INOUT INDIRECT_MULTI_ARG_t *arg)
{
	//Verify FCSI is not Busy
	waitFcsiIdle(); 

	// Configure Fcsi Addresses Registers & Initiate Read transaction
	configureFcsiAdressRegisters(block,arg->addr,FCSI_READ);

	// Wait for Fcsi transaction to end
	waitFcsiIdle();

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF131, &(arg->value[FCSI_DATA_A]));
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF132, &(arg->value[FCSI_DATA_B]));
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF133, &(arg->value[FCSI_DATA_C]));
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF134, &(arg->value[FCSI_DATA_D]));
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF135, &(arg->value[FCSI_DATA_E]));
	return RET_VAL_SUCCESS;
}

/*********************************************************************************
Method:       configureAdressRegisters
Description:  
Parameter:    IN INDIRECT_BLOCK_e block
Parameter:    IN uint32 * addr
Parameter:    IN FcsiReadWrite_e readOrWrite
Returns:      void
*********************************************************************************/
static void configureFcsiAdressRegisters(IN INDIRECT_BLOCK_e block,IN uint32 *addr, IN FcsiReadWrite_e readOrWrite)
{

	RegPhyRxTdIfRiscPage0TdPhyRxtdIf12A_u reg_BCdata;
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf12B_u reg_DEdata;
	RegPhyRxTdIfRiscPage0TdPhyRxtdIfFcsiAccess_u reg_FcsiAccess;

    // Configure Fcsi Address B & Address C Register
    // Write Address B & Address C
 	reg_BCdata.bitFields.fcsiGrAddrB = addr[FCSI_ADRESS_B];
	reg_BCdata.bitFields.fcsiGrAddrC = addr[FCSI_ADRESS_C];

	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12A, reg_BCdata.val);

 	reg_DEdata.bitFields.fcsiGrAddrD = addr[FCSI_ADRESS_D];
	reg_DEdata.bitFields.fcsiGrAddrE = addr[FCSI_ADRESS_E];

	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF12B, reg_DEdata.val);

	reg_FcsiAccess.val = 0;
	reg_FcsiAccess.bitFields.fcsiGrAddrA = addr[FCSI_ADRESS_A];
	reg_FcsiAccess.bitFields.fcsiGrRdWr = readOrWrite;

	if (block == INDIRECT_BLOCK_AFE)
	{
		reg_FcsiAccess.bitFields.fcsiGrMsAccess = 1;
	}
	else
	{
		reg_FcsiAccess.bitFields.fcsiGrRfAccess = 1;
	}
	

	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF_FCSI_ACCESS, reg_FcsiAccess.val);
}

/*********************************************************************************
Method:       waitFcsiIdle
Description:  
Parameter:    void
Returns:      RetVal_e
*********************************************************************************/
static RetVal_e waitFcsiIdle(void)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf136_u  fcsiBusy;
	uint16  delayCounter = 0;

	do
	{
		// read the status register
		RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF136, &fcsiBusy.val);
	} while ((fcsiBusy.bitFields.fcsiBusy) && ((++delayCounter) <= FCSI_DONE_DELAY));


	// check for timeout
	if (delayCounter > FCSI_DONE_DELAY)
	{
		return RET_VAL_TIMEOUT;
	}

	return RET_VAL_SUCCESS;
}



