/****************************************************************************
 ** COMPONENT:      ENET SW
 **
 ** MODULE:         PhyDriver.h
 **
 ** DESCRIPTION:    PHY header internal for module, for all PHY types (real & dummy)
 **
 ****************************************************************************/
#ifndef PHY_DRIVER_H
#define PHY_DRIVER_H

#include "PhyDriver_API.h"
#include "ieee80211.h"
#include "System_Configuration.h"

typedef struct PhyDriverInterfererDetectionParameters_s
{
	uint32  continuousInterfererDetectionReadCounter;

	uint8   isContinuousInterfererDetectionEnabled;
	uint8   isNewChannelSetRequestPending;
	uint8   newChannel;
	uint8   isContinuousInterfererDetectionNeededForNewChannel;

	uint16 	continuousInterfererFifoOverflowCounter;
#if defined RADAR_DETECTION_ENABLED
	uint32  radarDetectionReadCounter;
	uint32  radarFifoOverflowCounter;

	uint8	isRadarDetectionEnabled;
	uint8	isRadarDetectionNeededForNewChannel;
#endif
	uint16	reserved;
}PhyDriverInterfererDetectionParameters_t;

typedef struct PhyDriverInterfererRadarDetectionSample
{    
	uint32  timeStamp;
	uint16  pulseWidth;
	int8    amp;
	uint8   reserved;
} PhyDriverInterfererRadarDetectionSample_t;

typedef struct PhyDriverInterfererContinuousInterfererDetectionSample
{    
	uint8  reserved1;
	int8   metric;
	uint16 reserved2;
} PhyDriverInterfererContinuousInterfererDetectionSample_t;

/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
/* Phy Interrupt Cause */
#define PHY_DRIVER_GEN_RISC_INTERRUPT_MASK			0x1 
#define PHY_DRIVER_TX_PACKET_INTERRUPT_MASK			0x2 
#define PHY_DRIVER_WATCH_DOG_INTERRUPT_MASK		    0x4         
#define PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK	    (PHY_DRIVER_GEN_RISC_INTERRUPT_MASK | PHY_DRIVER_WATCH_DOG_INTERRUPT_MASK | PHY_DRIVER_TX_PACKET_INTERRUPT_MASK )
#define PHY_DRIVER_INTERRUPT_EANBLE_MASK            (PHY_DRIVER_WATCH_DOG_INTERRUPT_MASK | PHY_DRIVER_GEN_RISC_INTERRUPT_MASK)

/* Phy genrisc Interrupt Cause */
#define PHY_GEN_RISC_INTEERUPT_CONTINUOUS_INTERFERER_FIFO_OVERFLOW		0x2
#define PHY_GEN_RISC_INTEERUPT_MIPS_HALT_DONE                        	0x4
#define PHY_GEN_RISC_INTEERUPT_RADAR_FIFO_OVERFLOW		                0x8
#define PHY_GEN_RISC_INTEERUPT_ALL_CAUSES_MASK                          (PHY_GEN_RISC_INTEERUPT_CONTINUOUS_INTERFERER_FIFO_OVERFLOW |PHY_GEN_RISC_INTEERUPT_MIPS_HALT_DONE | PHY_GEN_RISC_INTEERUPT_RADAR_FIFO_OVERFLOW)

#define NS_TO_CLOCKS(delay)         (uint32)((((delay)*(System_GetMainCoreFrequency()))+500)/1000)
#define NS_TO_CLOCKS_REDUCED(delay) (uint32)((((delay)*(System_GetMainCoreFrequency()>>1))+500)/1000)

/* the PHY_WATCH_DOG_LIMIT is defined in cycles */
#define PHY_WATCH_DOG_LIMIT  0xFFFFFF 

#define PHY_GENERAL_DISABLE_VALUE 0
#define PHY_RX_DISABLE_VALUE 0
#define PHY_RX_MODEM_DISABLE_VALUE 0
#define PHY_TX_DISABLE_VALUE 0

#define PHY_GENERAL_ENABLE_VALUE 	1
#define PHY_RX_ENABLE_VALUE 		(0xFFFF)
#define PHY_RX_MODEM_ENABLE_VALUE 	(0x1)
#define PHY_RX_FCSI_EN_VALUE		(0x7)
#define PHY_TX_ENABLE_VALUE 		(0xFFFFFFFF)

#define PHY_GENERAL_ONE				1

#define PHY_TX_START_READ_DELAY 0x01b0
#define END_DELAY_ENABLE 0x0010
#define PHY_TX_MODEM_11B_START_AND_END_DELAY_ENABLE 0x1e00000

#define PHY_TX_START_READ_DELAY_AND_END_DELAY_ENABLE_AC_MODE  0x0200500
#define PHY_TX_MODEM_11B_START_AND_END_DELAY_ENABLE_AC_MODE   0x500000c

#define PHY_TX_MODEM_11B_START_DELAY_ENABLE 0x0000 
#define PHY_TX_MODEM_11B_END_DELAY_ENABLE 0x01e0
#define PHY_RFIC_RELEASE_RESET	7

#define PHY_DRIVER_INTERFERER_DETECTION_REDUCE_AMP_VALUE 256

#define PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS (MEM_RADAR_BUF_START) 
#define PHY_DRIVER_RADAR_DETECTION_FIFO_END_ADDRESS (MEM_RADAR_BUF_END) 
#define PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_POINTER_ADDRESS (MEM_RADAR_WR_POINTER) 
#define PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_COUNTER_ADDRESS (MEM_RADAR_WR_COUNT) 
#define PHY_DRIVER_RADAR_DETECTION_FIFO_READ_COUNTER_ADDRESS (MEM_RADAR_RE_COUNT) 
#define PHY_DRIVER_RADAR_DETECTION_ENABLE_ADDRESS (MEM_RADAR_ENABLE_MAC) 
#define PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS (MEM_RADAR_THRESHOLD)
#define PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE (32)
#define PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE_MASK (PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE - 1)


#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_BASE_ADDRESS (MEM_BT_BUF_START) 
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_END_ADDRESS (MEM_BT_BUF_END) 
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_POINTER_ADDRESS (MEM_BT_WR_POINTER) 
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_COUNTER_ADDRESS (MEM_BT_WR_COUNT) 
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_READ_COUNTER_ADDRESS (MEM_BT_RE_COUNT) 
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_ENABLE_ADDRESS (BT_MEASURE_ENABLE)
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE (32)
#define PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE_MASK (PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE - 1)
#define PHY_ALL_ANT_MASK   0xF

/* set TX window size */
#define FCC_5GHZ_EDGE_CHANNEL_ARR_LEN (6u)
#define FCC_2GHZ_EDGE_CHANNEL_ARR_LEN (2u)
#define CHANNEL_SIZE (4u)
#define EDGES_TYPES_NUM (2u)
#define TX_WINDOW_SIZE (4u)
#define TX_WINDOW_MASK (EXPONENT_TO_VALUE(TX_WINDOW_SIZE))

#if !defined(ENET_INC_ARCH_WAVE600D2) && defined(ENET_INC_ARCH_WAVE600B)
/* set ERP Phy Connected STA indication */
#define PHY_DRIVER_ERP_IS_STA_CONNECTED_ADDRESS   	   (IS_STA_CONNECTED)
#define PHY_DRIVER_ERP_IS_STA_CONNECTED_OFFSET     (0)

#define PHY_DRIVER_ERP_IS_STA_CONNECTED_IND_SHIFT  	   (0)
#define PHY_DRIVER_ERP_IS_STA_CONNECTED_IND_MASK       (0x1)

#define PHY_DRIVER_ERP_IS_STA_CONNECTED_ENABLE_SHIFT   (1)
#define PHY_DRIVER_ERP_IS_STA_CONNECTED_ENABLE_MASK       (0x1)
#endif /*not valid for 600D2/600*/


/***************************************************************************/
/***                              Registers                              ***/
/***************************************************************************/

#define SCP_IN_FW_EXTENDED_RATE_MASK     0x1
#define STBC_IN_FW_EXTENDED_RATE_MASK    0x1
#define LDPC_IN_FW_EXTENDED_RATE_MASK    0x1

/* TCR 0 fields*/
//Phy mode
#define TCR0_PHY_MODE_NUMOFBITS		2
#define TCR0_PHY_MODE_SHIFT			0
#define TCR0_PHY_MODE_MASK 		  	MASK(TCR0_PHY_MODE_NUMOFBITS,TCR0_PHY_MODE_SHIFT,uint32)
//LDPC
#define TCR0_CBW_NUMOFBITS		    2
#define TCR0_CBW_SHIFT			    2
#define TCR0_CBW_MASK 		  	    MASK(TCR0_CBW_NUMOFBITS,TCR0_CBW_SHIFT,uint32)


/* TCR 1 fields*/
//MCS
#define TCR1_MCS_NUMOFBITS		    6
#define TCR1_MCS_SHIFT			    0
#define TCR1_MCS_MASK 		  	    MASK(TCR1_MCS_NUMOFBITS,TCR1_MCS_SHIFT,uint32)
//SCP = Short GI = Shot Cicle Prefix
#define TCR1_SCP_NUMOFBITS		    1
#define TCR1_SCP_SHIFT			    8
#define TCR1_SCP_MASK 		  	    MASK(TCR1_SCP_NUMOFBITS,TCR1_SCP_SHIFT,uint32)
//LDPC
#define TCR1_LDPC_NUMOFBITS		    1
#define TCR1_LDPC_SHIFT			    9
#define TCR1_LDPC_MASK 		  	    MASK(TCR1_LDPC_NUMOFBITS,TCR1_LDPC_SHIFT,uint32)
//STBC
#define TCR1_STBC_NUMOFBITS		    1
#define TCR1_STBC_SHIFT			    10
#define TCR1_STBC_MASK 		  	    MASK(TCR1_STBC_NUMOFBITS,TCR1_STBC_SHIFT,uint32)

/* TCR 3 fields*/
//Aggregate
#define TCR3_AGG_NUMOFBITS		    1
#define TCR3_AGG_SHIFT			    0
#define TCR3_AGG_MASK 		  	    MASK(TCR3_AGG_NUMOFBITS,TCR3_AGG_SHIFT,uint32)

/* TCR 3 fields*/
//length
#define TCR4_LENGTH_NUMOFBITS		16
#define TCR4_LENGTH_SHIFT			0
#define TCR4_LENGTH_MASK 		  	MASK(TCR4_LENGTH_NUMOFBITS,TCR4_LENGTH_SHIFT,uint32)



#endif /* PHY_DRIVER_H */
