/****************************************************************************
 ** COMPONENT:      ENET SW
 **
 ** MODULE:         PhyDriver.c
 **
 ** DESCRIPTION:    PHY driver for Metalink phy that support 802.11A & 802.1N
 **
 ****************************************************************************/


/***************************************************************************/
/***                         Include Files                               ***/
/***************************************************************************/
#include "System_GlobalDefinitions.h"
#include "PhyDriver.h"
#include "PhyDriverCommon.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "PhyRegsIncluder.h"
#include "PhyTxRegsRegs.h"
#include "Protocol_PhyAttributes.h"
#include "ieee80211.h"
#include "database.h"
#include "mib_id.h"
#include "mib_ms.h"
#include "frame.h"
#include "lmi.h"
#include "lm.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "mt_cnfg.h"
#include "mt_sysdefs.h"
#include "mt_sysrst.h"
#include "MT_Emerald_Env_regs.h"
#include "MacHtExtensionsRegs.h"
#include "lminfra.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "RegAccess_Api.h"
#include "PSD.h"
#include "loggerAPI.h"
#include "CpuLoad_Api.h"
#include "PhyRxTdIfRiscPage0TdRegs.h"
#include "PhyRxTdIfRiscPage0TxbRegs.h"
#include "HwSemaphore_API.h"
#include "fast_mem_psd2mips.h"


#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 10

/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
#define NUMBER_OF_ACTIVE_TX_ANTENNAS 3

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/

/***************************************************************************/
/***                     Private Function Prototypes                     ***/
/***************************************************************************/

/***************************************************************************/
/***                         Private Data                                ***/
/***************************************************************************/
#if defined(ENET_INC_UMAC)
static PhyConfigParams_t m_configParams;
void PhyDrv_SetTxAntOperationSet(uint8 opSet)
{
    /* In dummy Phy the numer of active antennas is always 3 */
	m_configParams.txEnabledAntennas = NUMBER_OF_ACTIVE_TX_ANTENNAS;
}
void PhyDrv_SetRxAntOperationSet(uint8 opSet)
{
    uint32 antennaDataToWrite;
	
	m_configParams.rxEnabledAntennas = opSet;
	antennaDataToWrite = MASK(opSet, REG_PHY_RXTD_REG3C_HOST_ANTENNA_EN_SHIFT, uint32);
	MT_WrRegMask (PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG3C_HOST_ANTENNA_EN,REG_PHY_RXTD_REG3C_HOST_ANTENNA_EN_MASK,antennaDataToWrite);	
}

uint8 PhyDrv_GetRxEnabledAntennas(void)
{
	return m_configParams.rxEnabledAntennas;
}

uint8 PhyDrv_GetTxEnabledAntennas(void)
{
	return m_configParams.txEnabledAntennas;
}

void PhyDrv_GetLnaActiveValue(HdkLnaIndex_e inLnaIndex, uint8 *outLnaValue)
{
	*outLnaValue = 0;
}

RetVal_e PhyDrv_DisableBB()
{
	ILOG2_V("PhyDrv_DisableBB");
	
    PhyDrv_DisablePhy();
   
    return RET_VAL_SUCCESS;
}

RetVal_e PhyDrv_EnableBB(void)
{
	ILOG2_V("PhyDrv_EnableBB");
    //PHY activating sequence
    //Enable TD, TX (b, n), BE, FD, AFE modules
    PhyDrv_ResetAndEnable();

	return RET_VAL_SUCCESS;
}

void PhyDrv_ResetBB()
{
	PhyDrv_DisableBB();
}

void PhyDrv_EnableRx(void)
{
}

void PhyDrv_ReActivateBB()
{
	PhyDrv_EnableBB();
}
void PhyDrv_SetMacAntConfig(uint8 antMask)
{
}
void PhyDrv_preCal()
{
}

void PhyDrv_postCal()
{
}
void PhyDrv_SetRxHalfBand(PhyRxHalfBandType_e halfBandType)
{

}

void PhyDrv_SetTxHalfBand(PhyTxHalfBandType_e halfBandType)
{

}

void PhyDrv_Init( void )
{   
    uint8 rxAntennas;
	
	/* In the dummy Phy in Tx there are always 3 antennas set */
	m_configParams.txEnabledAntennas = 3;
    
	rxAntennas = Hdk_GetRxAntMask();
	PhyDrv_SetRxAntOperationSet(rxAntennas);	
}

uint8 PhyDrv_GetTxPrimary(void)
{
	uint32 txPrimaryType;
	
	RegAccess_Read(REG_PHY_TX_REGS_TX_BE_REG_48, &txPrimaryType);
	return (txPrimaryType & TX_PRIMARY_TYPE_MASK);
}

void PhyDrv_UpdatePrimaryChannel(uint8 band, uint8 u8SpectrumMode, uint8 primary_chan_idx)
{
}

void PhyDrv_SetSpectrum(uint8 band, uint8 chanWidth, uint8 primary_chan_idx, bool isBWChanged)
{
}

void PhyDrv_StartRisc(void)
{
}
void PhyDrv_StopRisc(void)
{
}

void PhyDrv_ReadVersion(PhyVersion_t *outPhyVersion)
{
}

void PhyDrv_SetSpacelessTransmission(uint8 value)
{
}

void PhyDrv_SetScramblerMode(uint8 mode)
{
}


void PhyDrv_ReadRssi(OUT uint8 *rssiValues, OUT uint8 *energyEstNumOfSamples)
{
}

void PhyDrv_ReadPacketCounter(OUT uint32 *rxPacketCounter,OUT uint32 *rxCrcErrorPacketCounter)
{
}

void PhyDrv_ResetPacketCounter(void)
{
}

/**********************************************************************************

isr_PhyDrv_PhyInterrupt 


Description:
------------
Handles interrupt from the PHY


Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void isr_PhyDrv_PhyInterrupt(void)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf86_u interruptStatusRegister;
	RegPhyRxTdIfRiscPage0TxbTxErrorIndication_u txPacketErrorCauseRegister;

	ACCUMULATE_CPU_IDLE_TIME();

	interruptStatusRegister.val = 0;
	txPacketErrorCauseRegister.val = 0;
	
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF86, &interruptStatusRegister.val);

	ASSERT(interruptStatusRegister.val & PHY_DRIVER_INTERRUPT_EANBLE_MASK);
	/* Phy genrisc interrupt or Tx packet error should not occur in dummy PHY */
	DEBUG_ASSERT(!(interruptStatusRegister.bitFields.phyMacIntCauseReg || interruptStatusRegister.bitFields.txPacketIntCauseReg));
	DEBUG_ASSERT(!(interruptStatusRegister.val & ~(PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK)));

    /* The cause and the clear bits are correlating so the value can be used to clear the interrupts */
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF85, interruptStatusRegister.val);

	/* WD interrupt, unknown problem cause the system to stuck - halt system.
	   "watchdog" raise every ~70mSec and check that the GenRisc is not stuck, if so,
	   reset operation is activated, and MAC receive an indication. */
	if(interruptStatusRegister.bitFields.wdIntCauseReg & (PHY_DRIVER_INTERRUPT_EANBLE_MASK & PHY_DRIVER_WATCH_DOG_INTERRUPT_MASK))
	{	
		FATAL("WD(phy) interrupt occured");
	}

	if(interruptStatusRegister.bitFields.txPacketIntCauseReg & (PHY_DRIVER_INTERRUPT_EANBLE_MASK & PHY_DRIVER_TX_PACKET_INTERRUPT_MASK))
	{	
		/* This register is clear each transmission so till the interrupt occurs this value might be irrelevant already. */
		RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_TX_ERROR_INDICATION, &txPacketErrorCauseRegister.val);
		FATAL("Tx packet interrupt"); //should be decided what to do in this case
	}
}

/**********************************************************************************

PhyDrvRadarDetection_SetRssiTh 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/
void PhyDrvRadarDetection_SetRssiTh(int32 value)
{

}

/**********************************************************************************

PhyDrvRadarDetection_GetRssiTh 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/

int32 PhyDrvRadarDetection_GetRssiTh(void)
{
	return 0;
}
/**********************************************************************************

PhyDrvInterfererDetection_SetChannelRequest 


Description:
------------
Empty function

Input:
-----





Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_SetChannelRequest(uint8 newChannel, uint8 isRadarDetectionNeeded, uint8 isContinuousInterfererDetectionNeeded)
{
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushRadarDetectionFifo 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushRadarDetectionFifo()
{
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(uint8 isNotificationNeeded)
{
}

/**********************************************************************************

PhyDrv_ClearPhyInterrupts 


Description:
------------
Clear interrupts from the PHY (used in intialization)

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#pragma ghs section text=".initialization" 

void PhyDrv_ClearPhyInterrupts()
{
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF85, PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK);
}


/**********************************************************************************

PhyDrvInterfererDetection_Initialize 


Description:
------------
Empty function 

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_Initialize()
{
}
#pragma ghs section text=default

/**********************************************************************************

PhyDrv_SetCcaTh 


Description:
------------
set CCA thresholds 

Input:
-----


Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void PhyDrv_SetCcaTh(K_MSG * psMsg)
{
}

void PhyDrv_GetCcaTh(K_MSG* psMsg)
{
}

/********************************************************************************
PhyDrv_Set_11B_RxAntSelection:

Description:
------------

Set Rx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_RxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{

}


/********************************************************************************
PhyDrv_Set_11B_TxAntSelection:

Description:
------------

Set Tx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_TxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{

}

/********************************************************************************
PhyDrv_Set_11B_roundRobinSelection:

Description:
------------

Set roundRobin mode for tx and rx. (802.11.b)
Input:
-----
rate for switching ant

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_roundRobinSelection(uint8 rate)
{

}

void PhyDrv_SetVhtNonAggregate(uint8 enable)
{

}

/********************************************************************************
PhyDrv_GetRxEvm: Dummy

********************************************************************************/
void PhyDrv_GetRxEvm(OUT uint8 *rxEvm_p)
{

}

void PhyDrv_SetCompressedFourierPhases(uint8 numOfAnts)
{

}
void PhyDrv_SetFrequency()
{

}

void PhyDrv_WriteRxThToPhy(int THval)
{

}

void PhyDrv_PowerSave_RadioOff(void)
{

}

void PhyDrv_WriteRxDutyCycleReq(uint32 onTime, uint32 offTime)
{	

}

void PhyDrv_WriteMinRssiToPhy(uint8 regVal)
{

}

void PhyDrv_ActivateSemaphoreForAocs(uint32 activate)
{
	if(activate == TRUE)
	{
		HwSemaphore_Lock(HW_SEMAPHORE_18_2_AOCS_INFO);
	}
	else
	{
		HwSemaphore_Free(HW_SEMAPHORE_18_2_AOCS_INFO);
	}
}

uint32 PhyDrv_GetRxAbortCounter(void)
{
	return REGISTER(MEM_PACKET_DROPPED_MAC_ABORT);
}

#endif

void PhyDrv_BandEdgeFix(SetChannelParams_t* channelReqParams)
{
}

void phyDrv_SetTxWindow(uint8 windowLenMode)
{
}


bool PhyDrv_IsAcMode(void)
{	
	uint32 acSupportReg;
	bool retValue;
	
	RegAccess_Read((DUMMY_PHY_BASE_ADDR + 0x78),&acSupportReg);

	
	retValue =  acSupportReg & 0x1;

	return retValue;
}

/***************************************************************************
** NAME         PhyDrv_DisablePhy
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino resets  PHY operation.
****************************************************************************/
void PhyDrv_DisablePhy()
{


	MASK_HW_ERROR_RX();
		
	/* deactivate Rx */
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG3A_ADC_ACTIVE_REG,PHY_RX_MODEM_DISABLE_VALUE);
	/* set Tx to reset */
	MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_00_SW_RESET_N_REG,PHY_TX_DISABLE_VALUE);
	/* set Rx to reset */
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG00_SW_RESET_N_REG,PHY_RX_DISABLE_VALUE);
}

/***************************************************************************
** NAME         PhyDrv_ResetAndEnable
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino Enables PHY operation.
****************************************************************************/

void PhyDrv_ResetAndEnable(void)
{
	bool isAcMode;

	isAcMode = PhyDrv_IsAcMode();	

	if(isAcMode)
	{
		/* Remove TX from reset */
		MT_WrReg(PHY_TX_BASE_ADDR, REG_TX_BE_REG_00_SW_RESET_N_REG ,PHY_TX_ENABLE_VALUE);
		/* Configure start_read_del and end_delay_len */
		MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_0E_START_READ_DEL,PHY_TX_START_READ_DELAY_AND_END_DELAY_ENABLE_AC_MODE);
		/* Configure modem_11b_phy_dis_dly and modem_11b_phy_ena_dly */
		MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_57_MODEM_11B_PHY_ENA_DLY,PHY_TX_MODEM_11B_START_AND_END_DELAY_ENABLE_AC_MODE);
		/*	Remove RX from reset */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG00_SW_RESET_N_REG, PHY_RX_ENABLE_VALUE);
		/* Configure hb Mode  0  - 80 2 -20 3 -40*/
		MT_WrReg(PHY_TD_BASE_ADDR,0x2C, 0);			/*TBD - Change to defines*/
		/* Set Num of Antennas */
		MT_WrRegMask(PHY_TD_BASE_ADDR,0xF0,0xF0,0xF0);	/*TBD  - Change to defines*/
		/* Enable/disable watch dog */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG189_WATCH_DOG_LIMIT, PHY_WATCH_DOG_LIMIT);
		/*	Activate RX */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG3A_ADC_ACTIVE_REG, PHY_RX_MODEM_ENABLE_VALUE);
		/* Configure  "legacy ready" and "legacy bf ready" timing */ 
		MT_WrReg(PHY_TX_BASE_ADDR, 0xCB4, 0x040011B0);  			/*TBD - Change to defines*/
		/* Configure non legacy bf ready timing */
		MT_WrReg(PHY_TX_BASE_ADDR, 0xCB8, 0x1530);  				/*TBD - Change to defines*/
	}
	else
	{
		/* Remove TX from reset */
		MT_WrReg(PHY_TX_BASE_ADDR, REG_TX_BE_REG_00_SW_RESET_N_REG ,PHY_TX_ENABLE_VALUE);
		/* Configure start_read_del and end_delay_len */
		MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_0E_START_READ_DEL,PHY_TX_START_READ_DELAY_AND_END_DELAY_ENABLE);
		/* Configure modem_11b_phy_dis_dly and modem_11b_phy_ena_dly */
		MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_57_MODEM_11B_PHY_ENA_DLY,PHY_TX_MODEM_11B_START_AND_END_DELAY_ENABLE);
		/*	Remove RX from reset */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG00_SW_RESET_N_REG, PHY_RX_ENABLE_VALUE);
		/* Enable/disable watch dog */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG189_WATCH_DOG_LIMIT, PHY_WATCH_DOG_LIMIT);
		/*	Activate RX */
		MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG3A_ADC_ACTIVE_REG, PHY_RX_MODEM_ENABLE_VALUE);
	}
}

void PhyDrv_macEmuStop(void)
{

}

void PhyDrv_macEmuRun(uint8 antMask, uint8 RfPow, uint16 bw, uint8 packet_format, uint8 mcs, uint8 nss, uint32 packet_length, uint8 ldpc, uint8 gap, uint16 repeats)
{

}
void PhyDrv_setPgcFiltLut(int8 AntMsk , uint8 PgcFiltLUT_local[5][11])
{
}


