/****************************************************************************
 ** COMPONENT:      ENET SW
 **
 ** MODULE:         PhyDriver.c
 **
 ** DESCRIPTION:    PHY driver for Metalink phy that support 802.11A & 802.1N
 **
 ****************************************************************************/
/***************************************************************************/
/***                         Include Files                               ***/
/***************************************************************************/
#include "System_GlobalDefinitions.h"
#include "PhyDriver.h"
#include "PhyDriverCommon.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "PhyRegsIncluder.h"
#include "PhyRxbeRegs.h"
#include "PhyTxRegsRegs.h"
#include "PhyRxTdRegs.h"
#include "PhyRxTdIfRiscPage0FdRegs.h"
#include "PhyRxTdIfRiscPage0BeRegs.h"
#include "PhyRxTdIfRiscPage0TdRegs.h"
#include "PhyRxTdIfRiscPage0TxbRegs.h"
#include "Protocol_PhyAttributes.h"
#include "ieee80211.h"
#include "database.h"
#include "mib_id.h"
#include "mib_ms.h"
#include "frame.h"
#include "lmi.h"
#include "lm.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "mt_cnfg.h"
#include "mt_sysdefs.h"
#include "mt_sysrst.h"
#include "MT_Emerald_Env_regs.h"
#include "MacHtExtensionsRegs.h"
#include "lminfra.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "RegAccess_Api.h"
#include "Indirect_API.h"
#include "PSD.h"
#include "Hdk_Api.h"
#include "loggerAPI.h"
#include "HwSemaphore_API.h"
#include "PhyRxbeRegs.h"
#include "fast_mem_psd2mips.h"
#include "PSD.h"
#ifdef ENET_INC_UMAC
#include "InterfererDetection_Api.h"
#include "OSAL_UpperMacMessages.h"
#include "OSAL_Tasks.h"
#include "CpuLoad_Api.h"
#endif /* ifdef ENET_INC_UMAC */

#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 11



#define PHY_DRIVER_LOGS
/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
#define STOP_RISC_ENABLE_ADDRESS 0x20008
#define CCA_REG_OFFSET (65)

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/
/***************************************************************************/
/***                     Private Function Prototypes                     ***/
/***************************************************************************/
#ifdef ENET_INC_UMAC
static void phyDrvInterfererDetectionDisablePhy(void);
static void phyDrvPhyGenriscInterrupt(void);
#endif /* #ifdef ENET_INC_UMAC */

/***************************************************************************/
/***                         Private Data                                ***/
/***************************************************************************/
#ifdef ENET_INC_UMAC
static PhyConfigParams_t m_configParams;
#endif
#ifdef ENET_INC_UMAC
static PhyDriverInterfererDetectionParameters_t PhyDriverInterfererDetectionParameters;
uint16 phyErrorIndication[PHY_ERROR_NUM_OF_CAUSE_TX_ERROR] = {0};
uint32 AAA_PHY_GENRISC_REGS[24] = {0};
#endif /* #ifdef ENET_INC_UMAC */
#define PHY_IQ_ENABLE_MASK	(0x2)
#define TX_IQ_CAL_BYPASS_OFF (0)
#define SHORT_RADAR_TIME_FOR_SUBBAND_BITMAP (0x8)
#define LONG_RADAR_ON_DC_MASK (0x6)//bit 1 and 2 enabled zero and 3 disable
#define SUBBAND_BITMAP_FOR_ALL_SUBBAND (0xf)
#define SUBBAND_THRESHOLD_FROM_MAX (10)
#define MINIMUM_VALUE_FOR_AMPLITUDE (-128)

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
#ifdef ENET_INC_UMAC
/**********************************************************************************

phyDrvInterfererDetectionDisablePhy 


Description:
------------
Handles the disabling Phy procedure from the interferer detection module point of
view

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void phyDrvInterfererDetectionDisablePhy()
{
	InterfererDetectionSetChannelMessageParameters_t *interfererDetectionSetChannelMessageParameters = NULL;
	K_MSG *interfererDetectionMessage = NULL; 
	
	if(PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending)
	{
		/* Disable Phy after set channel request */
		PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending = FALSE;
		if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
		{
			PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter = 0;
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_POINTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_READ_COUNTER_ADDRESS, 0);
		}
#if defined RADAR_DETECTION_ENABLED
		if(PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled)
		{
			PhyDriverInterfererDetectionParameters.radarDetectionReadCounter = 0;
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_POINTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_READ_COUNTER_ADDRESS, 0);
		}
#endif
		PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled = PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel;
#if defined RADAR_DETECTION_ENABLED
		PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled = PhyDriverInterfererDetectionParameters.isRadarDetectionNeededForNewChannel;
#endif
		RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_ENABLE_ADDRESS, PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel); 
#if defined RADAR_DETECTION_ENABLED
	    RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_ENABLE_ADDRESS, PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled);
#endif
    
		/*The following code should happen only in the first disable Phy of the set channel request proccess */
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionSetChannelMessageParameters_t));
		interfererDetectionSetChannelMessageParameters = ((InterfererDetectionSetChannelMessageParameters_t *)interfererDetectionMessage->abData);
		interfererDetectionSetChannelMessageParameters->channel =  PhyDriverInterfererDetectionParameters.newChannel;
#if defined RADAR_DETECTION_ENABLED
		interfererDetectionSetChannelMessageParameters->isRadarDetectionEnabled =  PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled;
#endif
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_SET_CHANNEL, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	else
	{
		/* Disable Phy not after set channel request - flush the FIFOs only for Continuous Interferer in 500b the timestamp are TSF sync */
		if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
		{
			PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(FALSE); 
		}
	}
}

/**********************************************************************************

phyDrvPhyGenriscInterrupt 


Description:
------------
Handles the interrupt triggered by the PHY genrsic

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void phyDrvPhyGenriscInterrupt()
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf08_u phyGenriscInterruptCauseRegister;

	phyGenriscInterruptCauseRegister.val = 0;
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF08, &phyGenriscInterruptCauseRegister.val);
   	DEBUG_ASSERT(!(phyGenriscInterruptCauseRegister.val & ~(PHY_GEN_RISC_INTEERUPT_ALL_CAUSES_MASK)));
	/* Clear the causes the PHY genrsic wrote */
	//there might be a race with the Phy genrisc and this must be solved in changing the HW */
	RegAccess_WriteMasked(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF08, phyGenriscInterruptCauseRegister.val, 0);
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_CONTINUOUS_INTERFERER_FIFO_OVERFLOW)
	{
		PhyDriverInterfererDetectionParameters.continuousInterfererFifoOverflowCounter++;
	}
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_MIPS_HALT_DONE)
	{
		FATAL("MIPS halt done");//TBD should be decided what to do in this case
	}
#if defined RADAR_DETECTION_ENABLED
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_RADAR_FIFO_OVERFLOW)
	{
		PhyDriverInterfererDetectionParameters.radarFifoOverflowCounter++;
		//ignore indication
		//FATAL("Radar FIFO overflow");//TBD should be decided what to do in this case
	}
#endif
}
#endif  /* #ifdef ENET_INC_UMAC */

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/***************************************************************************/
/***                        Private Functions                            ***/
/***************************************************************************/
bool PhyDrv_IsAcMode(void)
{	
	return 1; //dummy function - we only supprt wave500b and wave600. Is this question necessary?
}

/***************************************************************************
** NAME         PhyDrv_DisablePhy
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino resets  PHY operation.
****************************************************************************/
void PhyDrv_DisablePhy()
{
	ILOG0_V("PhyDrv_DisablePhy");
	
	PhyDrv_StopRisc(); //turn off RISC
//	MASK_HW_ERROR_RX(); //disable Rx PHY MAC interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF1AB, PHY_GENERAL_ONE);
	MT_DELAY(1);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG3A, PHY_RX_DISABLE_VALUE); //deactivate Rx
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_00, PHY_TX_DISABLE_VALUE); //reset Tx
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_06, PHY_TX_DISABLE_VALUE); //reset Tx
	//RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, PHY_RX_DISABLE_VALUE); //reset Rx
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF100, PHY_RX_DISABLE_VALUE); //reset RxFD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF180, PHY_RX_DISABLE_VALUE); //reset RxBE interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF8F, PHY_RX_DISABLE_VALUE); //reset RxTD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C5, PHY_TX_DISABLE_VALUE); //reset Tx interface
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_01, PHY_TX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_14, PHY_TX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF101, PHY_RX_DISABLE_VALUE); //reset RxFD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF181, PHY_RX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF90, PHY_RX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C6, PHY_RX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG_SW_RESET_N_GEN, PHY_TX_DISABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, PHY_RX_DISABLE_VALUE);
#ifdef ENET_INC_UMAC	
	phyDrvInterfererDetectionDisablePhy();
#endif /* #ifdef ENET_INC_UMAC */
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG01, PHY_RX_DISABLE_VALUE);
}

#ifdef ENET_INC_UMAC
/***************************************************************************
** NAME         PhyDrv_ResetAndEnable
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino Enables PHY operation.
****************************************************************************/
void PhyDrv_ResetAndEnable(void)
{
	PhyDrv_ReleaseSWreset();
	PhyDrv_EnableRx();
	PhyDrv_StartRisc();
}

void PhyDrv_ResetBB()
{
	PhyDrv_DisableBB();
	PhyDrv_ReleaseSWreset();
	PhyDrv_EnableRx();
}

void PhyDrv_ReActivateBB()
{
	PhyDrv_DisableBB();
	MT_DELAY(100);
	PhyDrv_EnableBB();
}
void PhyDrv_SetMacAntConfig(uint8 antMask)
{
}
uint8 PhyDrv_ReadMacAntConfig(void)
{
	ASSERT(0); //This register does not exist here.
	return 0;
}

void PhyDrv_preCal()
{
	RegPhyRxTdPhyRxtdReg23F_u regVal;

	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG23F, &regVal.val);
	regVal.bitFields.blockDetectionOnBands = 0xF;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG23F, regVal.val);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C5, PHY_TX_DISABLE_VALUE); //reset Tx interface
	REGISTER(REG_PHY_RX_TD_PHY_RXTD_REG80) |= 0x10;//enable Gclock to RxIqEstimator
} 

void PhyDrv_postCal()
{
	RegPhyRxTdPhyRxtdReg23F_u regVal;

	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG23F, &regVal.val);
	regVal.bitFields.blockDetectionOnBands = 0;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG23F, regVal.val);
	REGISTER(REG_PHY_RX_TD_PHY_RXTD_REG80) &= 0xffffffef;//disable Gclock to RxIqEstimator
}

void PhyDrv_ReleaseSWreset(void)
{
#ifdef PHY_DRIVER_LOGS
	ILOG0_V("PhyDrv_ReleaseSWreset");
#endif
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_00, PHY_TX_ENABLE_VALUE); //enable Tx
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_06, PHY_TX_ENABLE_VALUE); //enable Tx
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_01, PHY_TX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_TX_REGS_TX_BE_REG_14, PHY_TX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00, PHY_RX_ENABLE_VALUE); //enable Rx
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG01, PHY_RX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF100, PHY_RX_ENABLE_VALUE); //enable Rx FD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF101, PHY_RX_ENABLE_VALUE); //reset RxFD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF180, PHY_RX_ENABLE_VALUE); //enable Rx BE interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF181, PHY_RX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF8F, PHY_RX_ENABLE_VALUE); //enable Rx TD interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF90, PHY_RX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C5, PHY_RX_ENABLE_VALUE); //enable Tx interface
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C6, PHY_TX_ENABLE_VALUE);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG_SW_RESET_N_GEN, PHY_TX_ENABLE_VALUE);
	RegAccess_WriteMasked(REG_PHY_TX_REGS_TX_BE_REG_48, PHY_IQ_ENABLE_MASK, TX_IQ_CAL_BYPASS_OFF);
}


void PhyDrv_EnableRx(void)
{
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG3A, PHY_RX_MODEM_ENABLE_VALUE); //activate Rx
}

void PhyDrv_RFICreset(void)
{
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF137, PHY_GENERAL_DISABLE_VALUE);
}

void PhyDrv_RFICreleaseReset(void)
{
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF137, PHY_RFIC_RELEASE_RESET);
}

void PhyDrv_SetRxHalfBand(PhyRxHalfBandType_e halfBandType)
{
	RegPhyRxTdPhyRxtdReg0B_u halfbandVal;

	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG0B, &(halfbandVal.val));
	halfbandVal.bitFields.hbMode = halfBandType;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG0B, halfbandVal.val);
}

void PhyDrv_SetTxAntOperationSet(uint8 opSet)
{
	m_configParams.txEnabledAntennas = opSet;
	RegAccess_WriteMasked(REG_PHY_TX_REGS_TX_BE_REG_07, TX_PORT_ENABLE_MASK, (opSet << TX_PORT_ENABLE_SHIFT));
}

void PhyDrv_SetRxAntOperationSet(uint8 opSet)
{
    RegPhyRxTdPhyRxtdReg3C_u antennaDataToWrite;
	
	m_configParams.rxEnabledAntennas = opSet;
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG3C, &(antennaDataToWrite.val));
	antennaDataToWrite.bitFields.hostAntennaEn = opSet;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG3C, antennaDataToWrite.val);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_AFE_AUX_CTRL,Hdk_GetRxAntMask());
}

void PhyDrv_SetRxPrimary(PhyRxPrimaryChType_e rxPrimaryType)
{
	RegAccess_WriteMasked(REG_PHY_RX_TD_PHY_RXTD_REG35, RX_PRIMARY_TYPE_MASK, (uint32)rxPrimaryType);
}

void PhyDrv_SetTxHalfBand(PhyTxHalfBandType_e halfBandType)
{
	RegAccess_WriteMasked(REG_PHY_TX_REGS_TX_BE_REG_07, TX_HALF_BAND_TYPE_MASK, (uint32)halfBandType);
}

void PhyDrv_SetTxPrimary(PhyTxPrimaryChType_e txPrimaryType)
{
	RegAccess_WriteMasked(REG_PHY_TX_REGS_TX_BE_REG_48, TX_PRIMARY_TYPE_MASK, (uint32)txPrimaryType);
}
uint8 PhyDrv_GetTxPrimary(void)
{
	uint32 txPrimaryType;
	
	RegAccess_Read(REG_PHY_TX_REGS_TX_BE_REG_48, &txPrimaryType);
	return (txPrimaryType & TX_PRIMARY_TYPE_MASK);
}

uint8 PhyDrv_GetRxEnabledAntennas(void)
{
	return m_configParams.rxEnabledAntennas;
}

uint8 PhyDrv_GetTxEnabledAntennas(void)
{
	return (Hdk_GetTxAntMask());
}

void PhyDrv_GetLnaActiveValue(HdkLnaIndex_e inLnaIndex, uint8 *outLnaValue)
{
	*outLnaValue = inLnaIndex;
}

RetVal_e PhyDrv_DisableBB()
{
#ifdef PHY_DRIVER_LOGS
	ILOG0_V("PhyDrv_DisableBB");
#endif
    PhyDrv_DisablePhy();
    return RET_VAL_SUCCESS;
}

RetVal_e PhyDrv_EnableBB(void)
{
#ifdef PHY_DRIVER_LOGS
	ILOG0_V("PhyDrv_EnableBB");
#endif
    //PHY activating sequence
    //Enable TD, TX (b, n), BE, FD, AFE modules
    PhyDrv_ResetAndEnable();
	return RET_VAL_SUCCESS;
}

void PhyDrv_Init( void )
{   
}

void PhyDrv_UpdatePrimaryChannel(uint8 band, uint8 u8SpectrumMode, uint8 primary_chan_idx)
{
	RegPhyRxTdIfRiscPage0TxbPhyTxbRiscReg1C1_u txbRiscReg1c1;

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C1,&txbRiscReg1c1.val);
	if (u8SpectrumMode == BANDWIDTH_TWENTY)
	{
		PhyDrv_SetRxHalfBand(RX_HALF_BAND_TYPE_20);
		PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_LSB_40_LSB_20);
		PhyDrv_SetTxHalfBand(TX_HALF_BAND_TYPE_20);
		PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_LSB_40_LSB_20);
		txbRiscReg1c1.bitFields.rxFreqShiftEn = 0;
		RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C1,txbRiscReg1c1.val); 
	}
	else if (u8SpectrumMode == BANDWIDTH_FOURTY)
	{
		PhyDrv_SetRxHalfBand(RX_HALF_BAND_TYPE_40);
		PhyDrv_SetTxHalfBand(TX_HALF_BAND_TYPE_40);
		switch 	(primary_chan_idx)
		{
			case 0:
				/* Primary Channel Is Lower,Secondary Channel Is Upper*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_LSB_40_LSB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_LSB_40_LSB_20);
				txbRiscReg1c1.bitFields.rxFreqShiftEn = 1;
				txbRiscReg1c1.bitFields.rxFreqShiftMode= 0;
				RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C1,txbRiscReg1c1.val); 
				break;
			case 1:
				/* Primary Channel Is Upper*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_LSB_40_USB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_LSB_40_USB_20);
				txbRiscReg1c1.bitFields.rxFreqShiftEn = 1;
				txbRiscReg1c1.bitFields.rxFreqShiftMode= 1;
				RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C1,txbRiscReg1c1.val);  
				break;
			default:
				DEBUG_ASSERT(0);
				break;
		}
	}
	else //80MHz
	{
		PhyDrv_SetRxHalfBand(RX_HALF_BAND_TYPE_FULL_CB_80);
		PhyDrv_SetTxHalfBand(TX_HALF_BAND_TYPE_CB80);
		switch 	(primary_chan_idx)
		{
			case 0:
				/* Primary Channel Is Lower, Lower*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_LSB_40_LSB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_LSB_40_LSB_20);
				break;
			case 1:
				/* Primary Channel Is Lower, Upper*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_LSB_40_USB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_LSB_40_USB_20);
				break;
			case 2:
				/* Primary Channel Is Upper, Lower*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_USB_40_LSB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_USB_40_LSB_20);
				break;
			case 3:
				/* Primary Channel Is Upper, Upper*/
				PhyDrv_SetRxPrimary(RX_PRIMARY_TYPE_USB_40_USB_20);
				PhyDrv_SetTxPrimary(TX_PRIMARY_TYPE_USB_40_USB_20);
				break;
			default:
				DEBUG_ASSERT(0);
				break;
		}
		txbRiscReg1c1.bitFields.rxFreqShiftEn = 0; 
		txbRiscReg1c1.bitFields.rxFreqShiftMode= 0;
		RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_PHY_TXB_RISC_REG1C1,txbRiscReg1c1.val); 
	}
}

void PhyDrv_SetSpectrum(uint8 band, uint8 chanWidth, uint8 primary_chan_idx, bool isBWChanged)
{
	PhyDrv_UpdatePrimaryChannel(band, chanWidth, primary_chan_idx);
}

void PhyDrv_StartRisc(void)
{
	uint32 starttime;
	uint32 isUp = 0;
	uint32 timeDiff = 0;
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf84_u phyRxTdIf84Reg;

	RegAccess_Write(REG_PHY_RISC_INIT_STATUS, isUp);	// write 0 to Init Status
	phyRxTdIf84Reg.val = 0;
	MT_WrRegMask(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG00_SW_RESET_N_REG, 0x40, 0x40);
	MT_WrRegMask(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG01_BLOCK_EN, 0x40, 0x40);
	REGISTER(PHY_TD_BASE_ADDR + STOP_RISC_ENABLE_ADDRESS) = 0x0;
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG051_RISC_START_OP, 1);
	MT_Delay(1);
	/* Unmask  Phy WD interrupt and Phy genrisc to mips*/
	//TBD should be moved to different function
	phyRxTdIf84Reg.bitFields.wdIntMask = TRUE;
	phyRxTdIf84Reg.bitFields.phyMacIntMask = TRUE;
	phyRxTdIf84Reg.bitFields.txPacketIntMask = TRUE;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF84,phyRxTdIf84Reg.val);
	starttime = GET_TSF_TIMER_LOW();
	while ((isUp == 0) && (timeDiff < RISC_INIT_TIMEOUT))
	{
		timeDiff = GET_TSF_TIMER_LOW() - starttime;
		RegAccess_Read(REG_PHY_RISC_INIT_STATUS, &isUp);
	}
	ASSERT(isUp != 0);
	PhyDrv_RestoreRegsAfterStartRisc();
	set11bAntennaSelection(NULL, FALSE);
}
#endif //#ifdef ENET_INC_UMAC

void PhyDrv_StopRisc(void)
{
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_IF67_CLEAR_WD_COUNTER,REG_PHY_RXTD_IF67_CLEAR_WD_COUNTER_MASK);		// clear watchdog counter
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_IF67_WD_COUNTER_EN,0x0);											// disable watchdog
	MT_WrRegMask(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG00_SW_RESET_N_REG,0x40,0);									// Reset GenRisc
	MT_WrRegMask(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG01_BLOCK_EN, 0x40,0);										// Reset GenRisc
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_REG051_RISC_START_OP,0);
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_IF67_CLEAR_WD_COUNTER,0x10);										// clear watchdog counter
	MT_WrReg(PHY_TD_BASE_ADDR,REG_PHY_RXTD_IF67_WD_COUNTER_EN,0x0);											// disable watchdog
}

#ifdef ENET_INC_UMAC
void PhyDrv_ReadVersion(PhyVersion_t *outPhyVersion)
{
}

void PhyDrv_SetSpacelessTransmission(uint8 value)
{
	if(value == 0x0)
	{
		REGISTER(MEM_STAY_AT_TX) = 0x0;	
	}
	else
	{
		REGISTER(MEM_STAY_AT_TX) = 0x1;
	}

	MT_WrRegMask(PHY_TX_BASE_ADDR, REG_TX_BE_REG_04_TUNING_MODE, REG_TX_BE_REG_04_TUNING_MODE_MASK, value);
}

void PhyDrv_SetScramblerMode(uint8 mode)
{
	uint32 regVal = ((mode==1) ? PHY_SCRAMBLER_MODE_ON:PHY_SCRAMBLER_MODE_OFF);

	MT_WrReg(PHY_TX_BASE_ADDR,REG_TX_BE_REG_12_SW_SCR_INIT,regVal);
}


void PhyDrv_ReadRssi(OUT uint8 *rssiValues, OUT uint8 *energyEstNumOfSamples)
{
	uint8 ant;
	RegPhyRxTdIfRiscPage0BePhyRxbeIf184_u regPhyRxTdIfRiscPage0BePhyRxbeIf184;
	RegPhyRxTdPhyRxtdReg35_u regPhyRxTdPhyRxtdReg35;

	for (ant=ANTENNA_0; ant<TOTAL_ANTENNAS; ant++)
	{
		RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_BE_PHY_RXBE_IF184 + 4*ant, &regPhyRxTdIfRiscPage0BePhyRxbeIf184.val);
		rssiValues[ant] = regPhyRxTdIfRiscPage0BePhyRxbeIf184.bitFields.ant1AdRssi;
	}
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG35, &regPhyRxTdPhyRxtdReg35.val);
	*energyEstNumOfSamples = (regPhyRxTdPhyRxtdReg35.bitFields.ccaAccSamples) + 1;
}

void PhyDrv_ReadPacketCounter(OUT uint32 *rxPacketCounter,OUT uint32 *rxCrcErrorPacketCounter)
{
	// Latch Phy Counters
	MT_WrReg(PHY_BE_BASE_ADDR,REG_PHY_RXBE_REG25_LATCH_COUNTERS,1);
	//Read Counters from Phy
	*rxPacketCounter = MT_RdReg(PHY_BE_BASE_ADDR,REG_PHY_RXBE_REG37_CRC_PACKET_CNT_LATCHED);
	*rxCrcErrorPacketCounter = MT_RdReg(PHY_BE_BASE_ADDR,REG_PHY_RXBE_REG38_CRC_ERR_PACKET_CNT_LATCHED);
}

void PhyDrv_ResetPacketCounter(void)
{
	MT_WrReg(PHY_BE_BASE_ADDR,REG_PHY_RXBE_REG24_CLEAR_COUNTERS,1);
}

/**********************************************************************************
Name: PhyDrv_GetRxEvm

Description: on debug mode, get rx EVM from register, for all Antena's

Input/Output: get and fill struct pointer of arr[4]
	
**********************************************************************************/
void PhyDrv_GetRxEvm(OUT uint8 *rxEvm_p)
{
	RegAccess_Read(REG_PHY_RXBE_PHY_RXBE_REG70, (uint32 *)(rxEvm_p));	
}

/**********************************************************************************

isr_PhyDrv_PhyInterrupt 


Description:
------------
Handles interrupt from the PHY


Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void isr_PhyDrv_PhyInterrupt(void)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf86_u interruptStatusRegister;
	RegPhyRxTdIfRiscPage0TxbTxErrorIndication_u txPacketErrorCauseRegister;
	uint32 txPacketErrorCauseBitmap = 0;
	uint16 txPacketErrorCauseIndex = 0;
	uint32 regIndex = 0;

	ACCUMULATE_CPU_IDLE_TIME();
	interruptStatusRegister.val = 0;
	txPacketErrorCauseRegister.val = 0;
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF86, &interruptStatusRegister.val);
	DEBUG_ASSERT(interruptStatusRegister.val & PHY_DRIVER_INTERRUPT_EANBLE_MASK);
	DEBUG_ASSERT(!(interruptStatusRegister.val & ~(PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK)));
    /* The cause and the clear bits are correlating so the value can be used to clear the interrupts */
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF85, interruptStatusRegister.val);
	if(interruptStatusRegister.bitFields.txPacketIntCauseReg && (PHY_DRIVER_INTERRUPT_EANBLE_MASK & PHY_DRIVER_TX_PACKET_INTERRUPT_MASK))
	{	
		/* This register is clear each transmission so till the interrupt occurs this value might be irrelevant already. */
		RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TXB_TX_ERROR_INDICATION, &txPacketErrorCauseRegister.val);
		/* Making sure that at least one cause is set and no other bits than cause error are set */
		DEBUG_ASSERT(txPacketErrorCauseRegister.val);
		DEBUG_ASSERT(txPacketErrorCauseRegister.val < 0x10000); 
		txPacketErrorCauseBitmap = txPacketErrorCauseRegister.val;
		while(txPacketErrorCauseBitmap)
		{
			txPacketErrorCauseIndex = Utils_FindFirstSetAndClear(&txPacketErrorCauseBitmap);
			phyErrorIndication[txPacketErrorCauseIndex]++;
		}
	}
	if(interruptStatusRegister.bitFields.phyMacIntCauseReg && (PHY_DRIVER_INTERRUPT_EANBLE_MASK & PHY_DRIVER_GEN_RISC_INTERRUPT_MASK)) 
	{
		phyDrvPhyGenriscInterrupt();		
	}
	/* WD interrupt, unknown problem cause the system to stuck - halt system.
	   "watchdog" raise every ~70mSec and check that the GenRisc is not stuck, if so,
	   reset operation is activated, and MAC receive an indication. */
	if(interruptStatusRegister.bitFields.wdIntCauseReg && (PHY_DRIVER_INTERRUPT_EANBLE_MASK & PHY_DRIVER_WATCH_DOG_INTERRUPT_MASK))
	{
		// Read genrisc registers (registers + PC + some more). can be seen in SoC online.
		MT_WrRegMask(PHY_TD_BASE_ADDR, REG_PHY_RXTD_REG050_RISC_PAGE, REG_PHY_RXTD_REG050_RISC_PAGE_MASK, REG_PHY_RXTD_REG050_RISC_PAGE_MASK);	
		for (regIndex = 0 ; regIndex < 20 ; regIndex++)
		{
			AAA_PHY_GENRISC_REGS[regIndex] = MT_RdReg(PHY_TD_BASE_ADDR, (0x8000+(4*regIndex)));
		}
		AAA_PHY_GENRISC_REGS[20] = MT_RdReg(PHY_TD_BASE_ADDR, 0x804c); // Read the PC 4 more times, since PHY can be in a loop
		AAA_PHY_GENRISC_REGS[21] = MT_RdReg(PHY_TD_BASE_ADDR, 0x804c);
		AAA_PHY_GENRISC_REGS[22] = MT_RdReg(PHY_TD_BASE_ADDR, 0x804c);
		AAA_PHY_GENRISC_REGS[23] = MT_RdReg(PHY_TD_BASE_ADDR, 0x804c);
		MT_WrRegMask(PHY_TD_BASE_ADDR, REG_PHY_RXTD_REG050_RISC_PAGE, REG_PHY_RXTD_REG050_RISC_PAGE_MASK, 0);	
		FATAL("WD(phy) interrupt occured");
	}
}

/**********************************************************************************

PhyDrvRadarDetection_SetRssiTh 


Description:
------------
This Function called with initialized value by FW while its init.
The QA has the ability to change it on-the-fly for the radar tests,
by setting iwpriv and Driver will send a msg with new values.


Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/
void PhyDrvRadarDetection_SetRssiTh(int32 value)
{
     //The PHY configuration(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS) for minimum RSSI value in received signal.
     //Below this threshold, the PHY won't check if the input signal is radar.
	 RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS, value);
}

/**********************************************************************************

PhyDrvRadarDetection_GetRssiTh 


Description:
------------
return RSSI TH

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/

int32 PhyDrvRadarDetection_GetRssiTh(void)
{
	uint32 val;

	RegAccess_Read(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS, &val);

	return (int32)val;
}
/**********************************************************************************

PhyDrvInterfererDetection_SetChannelRequest 


Description:
------------
Handles set channel request

Input:
-----
new channel - the new channel that will be set
isRadarDetectionNeeded - a flag that indicates if radar detection needed
isContinuousInterfererDetectionNeeded - a flag that indicates if continuous 
interferer detection needed




Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_SetChannelRequest(uint8 newChannel, uint8 isRadarDetectionNeeded, uint8 isContinuousInterfererDetectionNeeded)
{
	PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending = TRUE;
	PhyDriverInterfererDetectionParameters.newChannel = newChannel;
#if defined RADAR_DETECTION_ENABLED
	PhyDriverInterfererDetectionParameters.isRadarDetectionNeededForNewChannel = isRadarDetectionNeeded;
#endif
	PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel = isContinuousInterfererDetectionNeeded;
    /* Stop sampling in PHY genrisc in order to get overflow */
	if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
	{
		RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_ENABLE_ADDRESS, 0); 
	}
#if defined RADAR_DETECTION_ENABLED
	if(PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled)
	{
		RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_ENABLE_ADDRESS, 0); 
	}
#endif
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushRadarDetectionFifo 


Description:
------------
Process the radar detection FIFO and sends message to the Interferer task if needed

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if defined RADAR_DETECTION_ENABLED
void PhyDrvInterfererDetection_FlushRadarDetectionFifo()
{
	InterfererDetectionRadarDetectionSamplesMessageParameters_t *interfererDetectionRadarSamplesMessageParameters = NULL;
	K_MSG* interfererDetectionMessage = NULL;
	uint8 numberOfValidSamples = 0;
	uint32 writeCounter = 0;
	PhyDriverInterfererRadarDetectionSample_t *radarDetectionSample = NULL;
	uint8 numberOfSamplesInMessage = 0;
	uint8 index = 0;
	uint8 subBandIndex = 0;
	uint8 numberOfsubBandIndex = 0;
	int8 subBandIndexMaxAmplitude;

	if(HDK_getChannelWidth()  == BANDWIDTH_TWENTY)
	{
		numberOfsubBandIndex = 1;
	}
	else
	{
		if(HDK_getChannelWidth()  == BANDWIDTH_FOURTY)
		{
			numberOfsubBandIndex = 2;
		}
		else
		{
			numberOfsubBandIndex = 4;
		}
	}

	RegAccess_Read(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, &writeCounter);
	numberOfValidSamples = writeCounter - PhyDriverInterfererDetectionParameters.radarDetectionReadCounter;

	ASSERT(numberOfValidSamples <= PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE);

    radarDetectionSample = (PhyDriverInterfererRadarDetectionSample_t *)PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS;
	radarDetectionSample += PhyDriverInterfererDetectionParameters.radarDetectionReadCounter%PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE;
	while(numberOfValidSamples)
	{
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionRadarDetectionSamplesMessageParameters_t));
		interfererDetectionRadarSamplesMessageParameters = ((InterfererDetectionRadarDetectionSamplesMessageParameters_t *)interfererDetectionMessage->abData);
		if(numberOfValidSamples > MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE)
		{
			numberOfSamplesInMessage = MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE;
		}
		else
		{
			numberOfSamplesInMessage = numberOfValidSamples;
		}
		numberOfValidSamples -= numberOfSamplesInMessage;
		interfererDetectionRadarSamplesMessageParameters->numberOfSamples = numberOfSamplesInMessage;
		for(index = 0; index < numberOfSamplesInMessage; index++)
		{
#ifdef PHY_DRIVER_LOGS
			ILOG0_D("-------------------- RAW SAMPLE %d --------------------", radarDetectionSample->pulseWidth);
#endif
			interfererDetectionRadarSamplesMessageParameters->pulses[index].timeStamp = radarDetectionSample->timeStamp;
            interfererDetectionRadarSamplesMessageParameters->pulses[index].pulseWidth = DIVIDE_BY_20(radarDetectionSample->pulseWidth);
			interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap = 0;

			subBandIndexMaxAmplitude = MINIMUM_VALUE_FOR_AMPLITUDE;
			for(subBandIndex = 0; subBandIndex < numberOfsubBandIndex; subBandIndex++)
			{
				if(radarDetectionSample->amp[subBandIndex]  > subBandIndexMaxAmplitude)
				{
					subBandIndexMaxAmplitude = radarDetectionSample->amp[subBandIndex];
				}
			}
			
			for(subBandIndex = 0; subBandIndex < numberOfsubBandIndex; subBandIndex++)
			{
#ifdef PHY_DRIVER_LOGS
				ILOG0_D("PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo	radarDetectionSample->amp[subBandIndex] %d", radarDetectionSample->amp[subBandIndex] -40);
#endif
				if((int16)(radarDetectionSample->amp[subBandIndex] + SUBBAND_THRESHOLD_FROM_MAX) > subBandIndexMaxAmplitude)
				{
					interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap |= 1 << ((numberOfsubBandIndex-1) - subBandIndex);
					
#ifdef PHY_DRIVER_LOGS
					ILOG0_D("interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap = %d", interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap );
#endif
				}
			}
			
			if(interfererDetectionRadarSamplesMessageParameters->pulses[index].pulseWidth < SHORT_RADAR_TIME_FOR_SUBBAND_BITMAP)
			{
				interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap = SUBBAND_BITMAP_FOR_ALL_SUBBAND;
			}
			else
			{
			 	if(interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap == SUBBAND_BITMAP_FOR_ALL_SUBBAND)//case that radar is on DC
			 	{
			 		interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap = LONG_RADAR_ON_DC_MASK;
			 	}
			}
			radarDetectionSample++;
			if((uint32)radarDetectionSample == PHY_DRIVER_RADAR_DETECTION_FIFO_END_ADDRESS)
			{
				/* Wrap around - go to begining of FIFO */
				radarDetectionSample = (PhyDriverInterfererRadarDetectionSample_t *)PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS;
			}
		}
		PhyDriverInterfererDetectionParameters.radarDetectionReadCounter += numberOfSamplesInMessage;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_RADAR_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_READ_COUNTER_ADDRESS, PhyDriverInterfererDetectionParameters.radarDetectionReadCounter);
}
#endif

/**********************************************************************************

PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo 


Description:
------------
Process the continuous interferer detection FIFO and sends message to the 
Interferer task if needed

Input:
-----
isNotificationNeeded - a flag for the interferer task if notification is needed


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(uint8 isNotificationNeeded)
{
	InterfererDetectionBtInterfererDetectionSamplesMessageParameters *interfererDetectionContinuousInterfererSamplesMessageParameters = NULL;
	K_MSG* interfererDetectionMessage = NULL;
	uint8 numberOfValidSamples = 0;
	uint32 writeCounter = 0;
	uint8 numberOfSamplesInMessage = 0;
	int32 continuousInterfererSample;
	uint8 index = 0;
	uint32 continuousInterfererSampleAddr = 0;
	
    RegAccess_Read(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, &writeCounter);
	numberOfValidSamples = writeCounter - PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter;
	ASSERT(numberOfValidSamples <= PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE);
	continuousInterfererSampleAddr = PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_BASE_ADDRESS + 4*(PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter & PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE_MASK);
	if(isNotificationNeeded && (0 == numberOfValidSamples))
	{
		/* In case a notification is needed but there are no valid samples a message still needs to be sent to the interferer detection module */
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionBtInterfererDetectionSamplesMessageParameters));
		interfererDetectionContinuousInterfererSamplesMessageParameters = ((InterfererDetectionBtInterfererDetectionSamplesMessageParameters *)interfererDetectionMessage->abData);
		interfererDetectionContinuousInterfererSamplesMessageParameters->numberOfSamples = 0;
		interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = TRUE;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	while(numberOfValidSamples)
	{
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionBtInterfererDetectionSamplesMessageParameters));
		interfererDetectionContinuousInterfererSamplesMessageParameters = ((InterfererDetectionBtInterfererDetectionSamplesMessageParameters *)interfererDetectionMessage->abData);
		if(numberOfValidSamples > MAX_NUMBER_OF_BT_INTERFERER_SAMPLES_IN_MESSAGE)
		{
			interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = FALSE;
			numberOfSamplesInMessage = MAX_NUMBER_OF_BT_INTERFERER_SAMPLES_IN_MESSAGE;
		}
		else
		{
			/* Insert the flag only in the last message */
			interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = isNotificationNeeded;
			numberOfSamplesInMessage = numberOfValidSamples;
		}
		numberOfValidSamples -= numberOfSamplesInMessage;
		interfererDetectionContinuousInterfererSamplesMessageParameters->numberOfSamples = numberOfSamplesInMessage;
		//ILOG0_D("PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo	numberOfSamplesInMessage %d", numberOfSamplesInMessage );
		for(index = 0; index < numberOfSamplesInMessage; index++)
		{
			continuousInterfererSample = REGISTER(continuousInterfererSampleAddr);
			continuousInterfererSample = continuousInterfererSample  >> 23;
			interfererDetectionContinuousInterfererSamplesMessageParameters->measurements[index] = continuousInterfererSample;
#ifdef PHY_DRIVER_LOGS
			ILOG2_DD("PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo index = %d continuousInterfererSample = %d ", index, continuousInterfererSample);
#endif			
			continuousInterfererSampleAddr = continuousInterfererSampleAddr+4;
			if(continuousInterfererSampleAddr == PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_END_ADDRESS)
			{
				/* Wrap around - go to begining of FIFO */
				continuousInterfererSampleAddr = PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_BASE_ADDRESS;
			}
		}
		PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter += numberOfSamplesInMessage;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_READ_COUNTER_ADDRESS, PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter);
}

/**********************************************************************************

PhyDrv_ClearPhyInterrupts 


Description:
------------
Clear interrupts from the PHY (used in intialization)

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void PhyDrv_ClearPhyInterrupts()
{
	/* Clear also interrupt cause register from the PHY genrisc */
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF08,0);
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF85, PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK);
}


/**********************************************************************************

PhyDrvInterfererDetection_Initialize 


Description:
------------
Initialize the parameters of the interferer detection in the PHY driver 

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_Initialize()
{
	memset(&PhyDriverInterfererDetectionParameters, 0, sizeof(PhyDriverInterfererDetectionParameters_t));
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
/**********************************************************************************

PhyDrv_SetCcaTh 


Description:
------------
set CCA thresholds 

Input:
-----


Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void PhyDrv_SetCcaTh(K_MSG * psMsg)
{
	UMI_CCA_TH_t* params;
	RegPhyRxTdPhyRxtdReg1Fd_u midPktReg;
	RegPhyRxTdPhyRxtdReg36_u ccaReg;

	params = (UMI_CCA_TH_t*)pK_MSG_DATA(psMsg);
//	ILOG0_D("PhyDrv_SetCcaTh  params->midPktSecondary20CCA  =%d", params->midPktSecondary20CCA);
//	ILOG0_D("PhyDrv_SetCcaTh  params->midPktSecondary40CCA  =%d", params->midPktSecondary40CCA);
//	ILOG0_D("PhyDrv_SetCcaTh  params->midPktPrimaryCCA  =%d", params->midPktPrimaryCCA);
//	ILOG0_D("PhyDrv_SetCcaTh  params->primaryChCCA  =%d",params->primaryChCCA);
//	ILOG0_D("PhyDrv_SetCcaTh  params->secondaryChCCA  =%d", params->secondaryChCCA);
	midPktReg.bitFields.ccaSecMidTh20 = params->midPktSecondary20CCA + CCA_REG_OFFSET;
	midPktReg.bitFields.ccaSecMidTh40 = params->midPktSecondary40CCA + CCA_REG_OFFSET;
	midPktReg.bitFields.ccaPrimMidTh = params->midPktPrimaryCCA + CCA_REG_OFFSET;
	
	ccaReg.bitFields.ccaEdThreshold = params->primaryChCCA + CCA_REG_OFFSET;
	ccaReg.bitFields.ccaEdThreshold40 = params->secondaryChCCA + CCA_REG_OFFSET;

	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG36, ccaReg.val);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG1FD, midPktReg.val);
}

void PhyDrv_GetCcaTh(K_MSG* psMsg)
{
	UMI_CCA_TH_t* params;
	RegPhyRxTdPhyRxtdReg1Fd_u midPktReg;
	RegPhyRxTdPhyRxtdReg36_u ccaReg;

	params = (UMI_CCA_TH_t*)pK_MSG_DATA(psMsg);
	ccaReg.val = REGISTER(REG_PHY_RX_TD_PHY_RXTD_REG36);
	midPktReg.val = REGISTER(REG_PHY_RX_TD_PHY_RXTD_REG36);

	params->midPktSecondary20CCA = midPktReg.bitFields.ccaSecMidTh20 - CCA_REG_OFFSET;
	params->midPktSecondary40CCA = midPktReg.bitFields.ccaSecMidTh40 - CCA_REG_OFFSET;
	params->midPktPrimaryCCA = midPktReg.bitFields.ccaPrimMidTh - CCA_REG_OFFSET;
	
	params->primaryChCCA = ccaReg.bitFields.ccaEdThreshold - CCA_REG_OFFSET;
	params->secondaryChCCA = ccaReg.bitFields.ccaEdThreshold40 - CCA_REG_OFFSET;

//	ILOG0_D("PhyDrv_getCcaTh  params->midPktSecondary20CCA  =%d", params->midPktSecondary20CCA);
//	ILOG0_D("PhyDrv_getCcaTh  params->midPktSecondary40CCA  =%d", params->midPktSecondary40CCA);
//	ILOG0_D("PhyDrv_getCcaTh  params->midPktPrimaryCCA  =%d", params->midPktPrimaryCCA);
//	ILOG0_D("PhyDrv_getCcaTh  params->primaryChCCA  =%d",params->primaryChCCA);
//	ILOG0_D("PhyDrv_getCcaTh  params->secondaryChCCA  =%d", params->secondaryChCCA);
	
}


/**********************************************************************************

Description:
------------
restores fields which were reset in start RISC 

Input:
-----
Output:
-------
Returns:
--------
**********************************************************************************/
void PhyDrv_RestoreRegsAfterStartRisc()
{
	RegPhyRxTdIfRiscPage0FdPhyRxfdIf12D_u regVal;

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF12D, &regVal.val);
	regVal.bitFields.genriscBfEn = 1;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_FD_PHY_RXFD_IF12D, regVal.val);
}

/********************************************************************************
PhyDrv_Set_11B_RxAntSelection:

Description:
------------

Set Rx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_RxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIfd9_u AntSel11bReg;
	
	DEBUG_ASSERT(Ant_selection!=ROUNDROBIN);

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, &AntSel11bReg.val);

	/*select Rx antenna */
	AntSel11bReg.bitFields.bOverrideRxAntSel = Ant_selection;
	
	/*set override bit */
	AntSel11bReg.bitFields.bOverrideRxAntEn = TRUE;
	
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, AntSel11bReg.val);
}

/********************************************************************************
PhyDrv_Set_11B_TxAntSelection:

Description:
------------

Set Tx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_TxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIfd9_u AntSel11bReg;
	
	DEBUG_ASSERT(Ant_selection!=ROUNDROBIN);
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, &AntSel11bReg.val);
	/*select Tx antenna */
	AntSel11bReg.bitFields.bOverrideTxAntSel = Ant_selection;
	/*set override bit */
	AntSel11bReg.bitFields.bOverrideTxAntEn = TRUE;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, AntSel11bReg.val);
}

/********************************************************************************
PhyDrv_Set_11B_roundRobinSelection:

Description:
------------

Set roundRobin mode for tx and rx. (802.11.b)
Input:
-----
rate for switching ant

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_roundRobinSelection(uint8 rate)
{
	RegPhyRxTdPhyRxtdReg17B_u antSel11bCounterReg;
	RegPhyRxTdIfRiscPage0TdPhyRxtdIfd9_u AntSel11bReg;
	
	DEBUG_ASSERT(rate > 0);

	/* set the switch rate in [mesc] units*/
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG17B, &antSel11bCounterReg.val);
	antSel11bCounterReg.bitFields.switch11BCounterLimit = rate;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG17B, antSel11bCounterReg.val);

	/* reset override bit*/
	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, &AntSel11bReg.val);
	AntSel11bReg.bitFields.bOverrideRxAntEn = FALSE;
	AntSel11bReg.bitFields.bOverrideTxAntEn = FALSE;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IFD9, AntSel11bReg.val);
}

void PhyDrv_SetVhtNonAggregate(uint8 enable)
{
	RegPhyRxbePhyRxbeReg67_u vHTNonAggregate;
	vHTNonAggregate.val = REGISTER(REG_PHY_RXBE_PHY_RXBE_REG67);
	vHTNonAggregate.bitFields.vhtNonAggregate = enable;
	REGISTER(REG_PHY_RXBE_PHY_RXBE_REG67) = vHTNonAggregate.val;
}
void PhyDrv_PowerSave_RadioOff(void)
{
	RegPhyRxTdIfRiscPage0TdPhyRxtdIf137_u resetReg;

	RegAccess_Read(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, &resetReg.val);
	resetReg.bitFields.fcsiMsResetAntennasN = 0;
	resetReg.bitFields.fcsiRfResetN = 0;
	RegAccess_Write(REG_PHY_RX_TD_IF_RISC_PAGE_0_TD_PHY_RXTD_IF137, resetReg.val);
}
void PhyDrv_SetCompressedFourierPhases(uint8 numOfAnts)
{
	PhyCompFourierPhasesTableDefinition_t* pCompressedFourierTableDef;
	uint8 tableSize;
	uint8 rowIndex = 0;
	uint32 address, mask, value;  
#ifdef PHY_DRIVER_LOGS
	ILOG0_V("PhyDrv_SetCompressedFourierPhases, Enter function"); 
#endif
	pCompressedFourierTableDef = getCompressedFourierPhasesTable(numOfAnts);

	if (pCompressedFourierTableDef != NULL) //if table was  update
	{
		tableSize = pCompressedFourierTableDef->phyAntConfigTableSize;
#ifdef PHY_DRIVER_LOGS
		ILOG0_D("PhyDrv_SetCompressedFourierPhases, tableSize = %d",tableSize); 
#endif
		for (rowIndex = 0; rowIndex < tableSize; rowIndex++)
		{
			address = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].address;
			mask    = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].mask;
			value   = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].value;
			RegAccess_WriteMasked(address, mask , value);
#ifdef PHY_DRIVER_LOGS
			ILOG0_D("PhyDrv_SetCompressedFourierPhases, address = %X",address); 
#endif
		}
	}

}
void PhyDrv_SetFrequency()
{
}

void PhyDrv_WriteRxThToPhy(int THval)
{
	MT_WrReg(PHY_TD_BASE_ADDR, REG_DETECTOR_POWER_THRESHOLD, THval << DET_POWER_TH_SHIFT);	
}

void PhyDrv_WriteRxDutyCycleReq(uint32 onTime, uint32 offTime)
{	
	MT_WrReg(PHY_TD_BASE_ADDR, REG_DET_TH_ON_TIMER, onTime);
	MT_WrReg(PHY_TD_BASE_ADDR, REG_DET_TH_OFF_TIMER, offTime);
}
void PhyDrv_WriteMinRssiToPhy(uint8 regVal)
{

	MT_WrReg(PHY_TD_BASE_ADDR, REG_PHY_MIN_RSSI, regVal << RX_TH_PHY_REPORT_SHIFT);
}
void PhyDrv_ActivateSemaphoreForAocs(uint32 activate)
{
	if(activate == TRUE)
	{
		HwSemaphore_Lock(HW_SEMAPHORE_18_2_AOCS_INFO);
	}
	else
	{
		HwSemaphore_Free(HW_SEMAPHORE_18_2_AOCS_INFO);
	}
}
uint32 PhyDrv_GetRxAbortCounter(void)
{
	return REGISTER(MEM_PACKET_DROPPED_MAC_ABORT);
}

void PhyDrv_macEmuStop(void)
{

}

void PhyDrv_macEmuRun(uint8 antMask, uint8 RfPow, uint16 bw, uint8 packet_format, uint8 mcs, uint8 nss, uint32 packet_length, uint8 ldpc, uint8 gap, uint16 repeats)
{
}
void PhyDrv_setPgcFiltLut(int8 AntMsk , uint8 PgcFiltLUT_local[5][11])
{
}

void PhyDrv_BandEdgeFix(SetChannelParams_t* channelReqParams)
{
}

void phyDrv_SetTxWindow(uint8 windowLenMode)
{

}

#endif //#ifdef ENET_INC_UMAC

