/****************************************************************************
 ** COMPONENT:      ENET SW
 **
 ** MODULE:         PhyDriver.c
 **
 ** DESCRIPTION:    PHY driver for Metalink phy that support 802.11A & 802.1N
 **
 ****************************************************************************/


/***************************************************************************/
/***                         Include Files                               ***/
/***************************************************************************/
#include "System_GlobalDefinitions.h"
#include "PhyDriver.h"
#include "PhyDriverCommon.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "PhyRegsIncluder.h"
#include "PhyTxRegs.h"
#include "Protocol_PhyAttributes.h"
#include "ieee80211.h"
#include "database.h"
#include "mib_id.h"
#include "mib_ms.h"
#include "frame.h"
#include "lmi.h"
#include "lm.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "mt_cnfg.h"
#include "mt_sysdefs.h"
#include "mt_sysrst.h"
#include "MacHtExtensionsRegs.h"
#include "lminfra.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "RegAccess_Api.h"
#include "PSD.h"
#include "loggerAPI.h"
#include "CpuLoad_Api.h"
#include "PhyRxTdIfRegs.h"
#include "PhyTxbIfRegs.h"
#include "HwSemaphore_API.h"
#include "PhyRxTdRegs.h"
#include "HDK_utils.h"
#include "DummyPhyRegs.h"
#include "DescriptorsDefinitions.h"
#include "ConfigurationManager_api.h"
#include "MacGeneralRegs.h"
#include "phy_lib.h"
#include "PhyTxMacEmuRegs.h"
#include "fast_mem_psd2mips.h"

#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 12



/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
#define NUMBER_OF_ACTIVE_TX_ANTENNAS 3

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/


/***************************************************************************/
/***                     Private Function Prototypes                     ***/
/***************************************************************************/

/***************************************************************************/
/***                         Private Data                                ***/
/***************************************************************************/
static PhyConfigParams_t m_configParams;
bool   PhyLocker = FALSE;
/***************************************************************************/
/***                        Private Functions                            ***/
/***************************************************************************/
void PhyDrv_LockReActivateBB(bool lock)
{
	PhyLocker = lock;
	ILOG0_D("PhyDrv_LockReActivateBB() PhyLocker: %d ",PhyLocker);
}

bool PhyDrv_IsAcMode(void)
{	
	bool retValue;

#ifdef HDK_REL_2_0

	retValue = 1; // no change in release 2.0 between ac and non ac mode

#else

	RegDummyPhyDphyRegAcCapable_u acSupportReg;
	
	RegAccess_Read((REG_DUMMY_PHY_DPHY_REG_AC_CAPABLE),&acSupportReg.val);

	
	retValue =  acSupportReg.bitFields.acCapable;

#endif	

	return retValue;
}

/***************************************************************************
** NAME         PhyDrv_DisablePhy
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino resets  PHY operation.
****************************************************************************/
void PhyDrv_DisablePhy()
{

	RegPhyRxTdPhyRxtdReg18_u adcActiveReg;
	RegPhyTxTxSwReset_u txSwResetReg;
	RegPhyRxTdPhyRxtdReg00_u rxSwResetReg;
	
	/* deactivate Rx */
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG18,&adcActiveReg.val);
	adcActiveReg.bitFields.adcActiveReg = PHY_RX_MODEM_DISABLE_VALUE;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG18,adcActiveReg.val);

	/* set Tx to reset */
	RegAccess_Read(REG_PHY_TX_TX_SW_RESET,&txSwResetReg.val);
	txSwResetReg.bitFields.swResetNReg = PHY_TX_DISABLE_VALUE;
	RegAccess_Write(REG_PHY_TX_TX_SW_RESET,txSwResetReg.val);
	/* set Rx to reset */
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG00,&rxSwResetReg.val);
	rxSwResetReg.bitFields.swResetNReg = PHY_RX_DISABLE_VALUE;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00,rxSwResetReg.val);
}

/***************************************************************************
** NAME         PhyDrv_ResetAndEnable
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino Enables PHY operation @640MHz.
****************************************************************************/

void PhyDrv_ResetAndEnable(void)
{
	bool isAcMode;
	RegPhyTxTxSwReset_u txSwResetReg;
	RegPhyTxTxBeReg0E_u txRegsTxBeReg0E;
	RegPhyTxTxBeReg57_u txRegsTxBeReg57;
	RegPhyRxTdPhyRxtdReg00_u rxSwResetReg;
	RegPhyRxTdPhyRxtdReg05_u rxTdReg05;
	RegPhyRxTdPhyRxtdReg039_u rxTdAntReg39;
	RegPhyRxTdPhyRxtdReg18_u adcActiveReg;
	isAcMode = PhyDrv_IsAcMode();	

	if(isAcMode)
	{
		/* Remove TX from reset */
		RegAccess_Read(REG_PHY_TX_TX_SW_RESET,&txSwResetReg.val);
		txSwResetReg.bitFields.swResetNReg = PHY_TX_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_TX_TX_SW_RESET,txSwResetReg.val);
		/* Configure start_read_del and end_delay_len */
		RegAccess_Read(REG_PHY_TX_TX_BE_REG_0E,&txRegsTxBeReg0E.val);
		txRegsTxBeReg0E.val = PHY_TX_START_READ_DELAY_AND_END_DELAY_ENABLE_AC_MODE;
		RegAccess_Write(REG_PHY_TX_TX_BE_REG_0E,txRegsTxBeReg0E.val);
		/* Configure modem_11b_phy_dis_dly and modem_11b_phy_ena_dly */
		RegAccess_Read(REG_PHY_TX_TX_BE_REG_57,&txRegsTxBeReg57.val);
		txRegsTxBeReg57.val = PHY_TX_MODEM_11B_START_AND_END_DELAY_ENABLE_AC_MODE;
		RegAccess_Write(REG_PHY_TX_TX_BE_REG_57,txRegsTxBeReg57.val);
		/*	Remove RX from reset */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG00,&rxSwResetReg.val);
		rxSwResetReg.bitFields.swResetNReg = PHY_RX_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00,rxSwResetReg.val);
		/* Configure hb Mode  0  - 80 2 -20 3 -40*/
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG05,&rxTdReg05.val);
		rxTdReg05.bitFields.bwMode = 0;
		rxTdReg05.bitFields.hbMode = 0;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG05,rxTdReg05.val);
		/* Enable/disable watch dog */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG039,&rxTdAntReg39.val);
		rxTdAntReg39.bitFields.watchDogLimit = PHY_WATCH_DOG_LIMIT;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG039, rxTdAntReg39.val);
		/*	Activate RX */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG18,&adcActiveReg.val);
		adcActiveReg.bitFields.adcActiveReg = PHY_RX_MODEM_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG18, adcActiveReg.val);
	}
	else
	{
		/* Remove TX from reset */
		RegAccess_Read(REG_PHY_TX_TX_SW_RESET,&txSwResetReg.val);
		txSwResetReg.bitFields.swResetNReg = PHY_TX_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_TX_TX_SW_RESET,txSwResetReg.val);
		/* Configure start_read_del and end_delay_len */
		RegAccess_Read(REG_PHY_TX_TX_BE_REG_0E,&txRegsTxBeReg0E.val);
		txRegsTxBeReg0E.bitFields.startReadDel = PHY_TX_START_READ_DELAY;
		txRegsTxBeReg0E.bitFields.endDelayLen = END_DELAY_ENABLE;
		RegAccess_Write(REG_PHY_TX_TX_BE_REG_0E,txRegsTxBeReg0E.val);
		/* Configure modem_11b_phy_dis_dly and modem_11b_phy_ena_dly */
		RegAccess_Read(REG_PHY_TX_TX_BE_REG_57,&txRegsTxBeReg57.val);
		txRegsTxBeReg57.bitFields.modem11BPhyDisDly = PHY_TX_MODEM_11B_START_DELAY_ENABLE;
		txRegsTxBeReg57.bitFields.modem11BPhyEnaDly = PHY_TX_MODEM_11B_END_DELAY_ENABLE;
		RegAccess_Write(REG_PHY_TX_TX_BE_REG_57,txRegsTxBeReg57.val);
		/*	Remove RX from reset */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG00,&rxSwResetReg.val);
		rxSwResetReg.bitFields.swResetNReg = PHY_RX_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG00,rxSwResetReg.val);
		/* Enable/disable watch dog */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG039,&rxTdAntReg39.val);
		rxTdAntReg39.bitFields.watchDogLimit = PHY_WATCH_DOG_LIMIT;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG039, rxTdAntReg39.val);
		/*	Activate RX */
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG18,&adcActiveReg.val);
		adcActiveReg.bitFields.adcActiveReg = PHY_RX_MODEM_ENABLE_VALUE;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG18, adcActiveReg.val);
	}
}

void PhyDrv_SetTxAntOperationSet(uint8 opSet)
{
    /* In dummy Phy the numer of active antennas is always 3 */
	UNUSED_PARAM(opSet);
	m_configParams.txEnabledAntennas = NUMBER_OF_ACTIVE_TX_ANTENNAS;
}

void PhyDrv_SetRxAntOperationSet(uint8 opSet)
{
	RegPhyRxTdPhyRxtdReg07B_u rxTdAntReg7B;
	
	m_configParams.rxEnabledAntennas = opSet;
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG07B,&rxTdAntReg7B.val);
	/* WLANRTSYS-13381: W/A for bug in Dummy PHY 600-2 REL2.0 */
	rxTdAntReg7B.val >>= 4;
	rxTdAntReg7B.bitFields.hostAntennaEn = opSet;	
	rxTdAntReg7B.val <<= 4;
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG07B,rxTdAntReg7B.val);
}

uint8 PhyDrv_GetRxEnabledAntennas(void)
{
	return m_configParams.rxEnabledAntennas;
}

uint8 PhyDrv_GetTxEnabledAntennas(void)
{
	return m_configParams.txEnabledAntennas;
}

RetVal_e PhyDrv_DisableBB()
{
	ILOG2_V("PhyDrv_DisableBB");
	
    PhyDrv_DisablePhy();

    return RET_VAL_SUCCESS;
}

RetVal_e PhyDrv_EnableBB(void)
{
	ILOG2_V("PhyDrv_EnableBB");
    //PHY activating sequence
    //Enable TD, TX (b, n), BE, FD, AFE modules
    PhyDrv_ResetAndEnable();

	return RET_VAL_SUCCESS;
}

void PhyDrv_ResetBB()
{
	
	PhyDrv_DisableBB();
}

void PhyDrv_ReleaseSWreset(void)
{
}

void PhyDrv_EnableRx(void)
{
}

void PhyDrv_ReActivateBB()
{
	PhyDrv_EnableBB();
}
void PhyDrv_SetMacAntConfig(uint8 antMask)
{
	uint8 band = ConfigurationManager_GetMyBand();
	
	PHY_SET_MAC_ANT_CONFIG(band, antMask);
}
uint8 PhyDrv_ReadMacAntConfig(void)
{
	uint8 antMask;
	RegMacGeneralPhyBandConfig_u phyBandConfigReg;

	RegAccess_Read(REG_MAC_GENERAL_PHY_BAND_CONFIG, &phyBandConfigReg.val);

	if (ConfigurationManager_GetMyBand() == CONFIGURATION_MANAGER_BAND_0)
	{
		antMask = ((!phyBandConfigReg.bitFields.phyAntenna0Sel) |
					((!phyBandConfigReg.bitFields.phyAntenna1Sel)<<1)|
					((!phyBandConfigReg.bitFields.phyAntenna2Sel)<<2)|
					((!phyBandConfigReg.bitFields.phyAntenna3Sel)<<3));
	}
	else
	{
		antMask = ((phyBandConfigReg.bitFields.phyAntenna0Sel) |
					((phyBandConfigReg.bitFields.phyAntenna1Sel)<<1)|
					((phyBandConfigReg.bitFields.phyAntenna2Sel)<<2)|
					((phyBandConfigReg.bitFields.phyAntenna3Sel)<<3));
	}

	return antMask;
}

void PhyDrv_preCal()
{
}

void PhyDrv_postCal()
{
}
void PhyDrv_SetRxHalfBand(PhyRxHalfBandType_e halfBandType)
{
	UNUSED_PARAM(halfBandType);
}

void PhyDrv_SetTxHalfBand(PhyTxHalfBandType_e halfBandType)
{
	UNUSED_PARAM(halfBandType);
}

void PhyDrv_Init( void )
{   
    uint8 rxAntennas;
	
	/* In the dummy Phy in Tx there are always 3 antennas set */
	m_configParams.txEnabledAntennas = 3;
    
	rxAntennas = PHY_ALL_ANT_MASK;
	PhyDrv_SetRxAntOperationSet(rxAntennas);
}

void PhyDrv_SetBand(uint8 phyType,uint8 *band)
{
	UNUSED_PARAM(phyType);
	UNUSED_PARAM(band);
}
uint8 PhyDrv_GetTxPrimary(void)
{
	uint32 txPrimaryType;
	
	RegAccess_Read(REG_PHY_TX_TX_BE_REG_48, &txPrimaryType);
	return (txPrimaryType & TX_PRIMARY_TYPE_MASK);
}

void PhyDrv_UpdatePrimaryChannel(uint8 band, uint8 u8SpectrumMode, uint8 primary_chan_idx)
{
	UNUSED_PARAM(band);
	UNUSED_PARAM(u8SpectrumMode);
	UNUSED_PARAM(primary_chan_idx);	
}
void PhyDrv_SetSpectrum(uint8 band, uint8 chanWidth, uint8 primary_chan_idx, bool isBWChanged)
{
	UNUSED_PARAM(band);
	UNUSED_PARAM(chanWidth);
	UNUSED_PARAM(primary_chan_idx);	
	UNUSED_PARAM(isBWChanged);	
}

void PhyDrv_StartRisc(void)
{
}
void PhyDrv_StopRisc(void)
{
}

void PhyDrv_ReadVersion(PhyVersion_t *outPhyVersion)
{
	UNUSED_PARAM(outPhyVersion);	
}

void PhyDrv_SetSpacelessTransmission(uint8 value)
{
	UNUSED_PARAM(value);	
}

void PhyDrv_SetScramblerMode(uint8 mode)
{
	UNUSED_PARAM(mode);	
}


void PhyDrv_ReadRssi(OUT uint8 *rssiValues, OUT uint8 *energyEstNumOfSamples)
{
	UNUSED_PARAM(rssiValues);	
	UNUSED_PARAM(energyEstNumOfSamples);	
}

void PhyDrv_ReadPacketCounter(OUT uint32 *rxPacketCounter,OUT uint32 *rxCrcErrorPacketCounter)
{
	UNUSED_PARAM(rxPacketCounter);	
	UNUSED_PARAM(rxCrcErrorPacketCounter);
}

void PhyDrv_ResetPacketCounter(void)
{
}

/**********************************************************************************

isr_PhyDrv_PhyInterrupt 


Description:
------------
Handles interrupt from the PHY


Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void isr_PhyDrv_PhyInterrupt(void)
{

	RegPhyTxMacEmuPhyMacError_u interruptStatusRegister;

	ACCUMULATE_CPU_IDLE_TIME();

	interruptStatusRegister.val = 0;

	RegAccess_Read(REG_PHY_TX_MAC_EMU_PHY_MAC_ERROR, &interruptStatusRegister.val);

	ASSERT(interruptStatusRegister.val != 0);
	
	FATAL("isr_PhyDrv_PhyInterrupt");

}

/**********************************************************************************

PhyDrvRadarDetection_SetRssiTh 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/
void PhyDrvRadarDetection_SetRssiTh(int32 value)
{
	UNUSED_PARAM(value);	
}


/**********************************************************************************

PhyDrvRadarDetection_GetRssiTh 


Description:
------------
return RSSI TH

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/

int32 PhyDrvRadarDetection_GetRssiTh(void)
{
	return 0;
}


/**********************************************************************************

PhyDrvInterfererDetection_SetChannelRequest 


Description:
------------
Empty function

Input:
-----





Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_SetChannelRequest(uint8 newChannel, uint8 isRadarDetectionNeeded, uint8 isContinuousInterfererDetectionNeeded)
{
	UNUSED_PARAM(newChannel);
	UNUSED_PARAM(isRadarDetectionNeeded);
	UNUSED_PARAM(isContinuousInterfererDetectionNeeded);
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushRadarDetectionFifo 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushRadarDetectionFifo()
{
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo 


Description:
------------
Empty function

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(uint8 isNotificationNeeded)
{
	UNUSED_PARAM(isNotificationNeeded);
}

/**********************************************************************************

PhyDrv_ClearPhyInterrupts 


Description:
------------
Clear interrupts from the PHY (used in intialization)

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrv_ClearPhyInterrupts()
{
	RegAccess_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF85, PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK);
}


/**********************************************************************************

PhyDrvInterfererDetection_Initialize 


Description:
------------
Empty function 

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_Initialize()
{
}

/**********************************************************************************

PhyDrv_SetCcaTh 


Description:
------------
set CCA thresholds 

Input:
-----


Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void PhyDrv_SetCcaTh(K_MSG * psMsg)
{
	UNUSED_PARAM(psMsg);
}

void PhyDrv_GetCcaTh(K_MSG* psMsg)
{
	UNUSED_PARAM(psMsg);
}


/**********************************************************************************

Description:
------------
restores fields which were reset in start RISC 

Input:
-----
Output:
-------
Returns:
--------
**********************************************************************************/
void PhyDrv_RestoreRegsAfterStartRisc()
{
}

/********************************************************************************
PhyDrv_Set_11B_RxAntSelection:

Description:
------------

Set Rx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_RxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	UNUSED_PARAM(Ant_selection);
}


/********************************************************************************
PhyDrv_Set_11B_TxAntSelection:

Description:
------------

Set Tx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_TxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	UNUSED_PARAM(Ant_selection);
}

/********************************************************************************
PhyDrv_Set_11B_roundRobinSelection:

Description:
------------

Set roundRobin mode for tx and rx. (802.11.b)
Input:
-----
rate for switching ant

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_roundRobinSelection(uint8 rate)
{
	UNUSED_PARAM(rate);
}
void PhyDrv_SetVhtNonAggregate(uint8 enable)
{
	UNUSED_PARAM(enable);
}

/********************************************************************************
PhyDrv_GetRxEvm: Dummy

********************************************************************************/
void PhyDrv_GetRxEvm(OUT uint8 *rxEvm_p) 
{
	UNUSED_PARAM(rxEvm_p);
}
void PhyDrv_SetCompressedFourierPhases(uint8 numOfAnts)
{
	UNUSED_PARAM(numOfAnts);
}
void PhyDrv_SetFrequency()
{
}

void PhyDrv_SetBssColor(uint8 vapId, uint8 bssColor)
{
	UNUSED_PARAM(vapId);
	UNUSED_PARAM(bssColor);
}
void PhyDrv_WriteRxThToPhy(int THval)
{
	UNUSED_PARAM(THval);
}
void PhyDrv_WriteRxDutyCycleReq(uint32 onTime, uint32 offTime)
{	
	UNUSED_PARAM(onTime);
	UNUSED_PARAM(offTime);
}

void PhyDrv_ActivateSemaphoreForAocs(uint32 activate)
{
	if(activate == TRUE)
	{
		HW_SEMAPHORE_LOCK(HW_SEMAPHORE_AOCS_INFO);
	}
	else
	{
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_AOCS_INFO);
	}
}
uint32 PhyDrv_GetRxAbortCounter(void)
{
	return REGISTER(MEM_PACKET_DROPPED_MAC_ABORT);
}

void PhyDrv_macEmuStop(void)
{

}

void PhyDrv_macEmuRun(uint8 antMask, uint8 RfPow, uint16 bw, uint8 packet_format, uint8 mcs, uint8 nss, uint32 packet_length, uint8 ldpc, uint8 gap, uint16 repeats)
{
	UNUSED_PARAM(antMask);
	UNUSED_PARAM(RfPow);
	UNUSED_PARAM(bw);
	UNUSED_PARAM(packet_format);
	UNUSED_PARAM(mcs);
	UNUSED_PARAM(nss);
	UNUSED_PARAM(packet_length);
	UNUSED_PARAM(ldpc);
	UNUSED_PARAM(gap);
	UNUSED_PARAM(repeats);
}
void PhyDrv_setPgcFiltLut(int8 AntMsk , uint8 PgcFiltLUT_local[5][11])
{
	UNUSED_PARAM(AntMsk);
	UNUSED_PARAM(PgcFiltLUT_local);
}

void phyDrv_SetTxWindow(uint8 windowLenMode)
{
	UNUSED_PARAM(windowLenMode);
}

void phyDrv_SetBandEdgeFixForTxWindow(uint32* windowFixsTable)
{
	RegAccess_Write(TX_WIN_BANDEDGE_FIX_ENABLER, 0);
	UNUSED_PARAM(windowFixsTable); 
}

void PhyDrv_BandEdgeFix(SetChannelParams_t* channelReqParams)
{
	UNUSED_PARAM(channelReqParams); 
}



void PhyDrv_SetConnectedSTAind(uint8 isSTAConnected)
{
	UNUSED_PARAM(isSTAConnected); 
}

void PhyDrv_SetPowerSave(uint8 enableDisable)
{
	UNUSED_PARAM(enableDisable);
}

void PhyDrv_PowerSaveInit(void)
{

}

