/****************************************************************************
 ** COMPONENT:      ENET SW
 **
 ** MODULE:         PhyDriver.c
 **
 ** DESCRIPTION:    PHY driver for Metalink phy that support 802.11A & 802.1N
 **
 ****************************************************************************/
/***************************************************************************/
/***                         Include Files                               ***/
/***************************************************************************/
#include "System_GlobalDefinitions.h"
#include "PhyDriver.h"
#include "PhyDriverCommon.h"
#include "PhyDriver_API.h"
#include "PhyCalDriver_API.h"
#include "PhyRegsIncluder.h"
#ifdef ENET_INC_ARCH_WAVE600D2	
#include "PhyRxBeRegs.h"
#else
#include "PhyRxbeRegs.h"
#endif
#include "PhyTxRegs.h"
#include "PhyRxTdRegs.h"
#include "PhyRxTdIfRegs.h"
#include "PhyRxFdIfRegs.h"
#include "PhyRxBeIfRegs.h"
#include "Protocol_PhyAttributes.h"
#include "ieee80211.h"
#include "database.h"
#include "mib_id.h"
#include "mib_ms.h"
#include "frame.h"
#include "lmi.h"
#include "lm.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "mt_cnfg.h"
#include "mt_sysdefs.h"
#include "mt_sysrst.h"
#include "EmeraldEnvRegs.h"
#include "MacHtExtensionsRegs.h"
#include "lminfra.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "RegAccess_Api.h"
#include "Indirect_API.h"
#include "PSD.h"
#include "Hdk_Api.h"
#include "loggerAPI.h"
#include "CpuLoad_Api.h"
#include "PhyTxbIfRegs.h"
#include "HwSemaphore_API.h"
#include "fast_mem_psd2mips.h"
#include "HwEventsAndErrors_Api.h"
#include "phy_lib.h"
#include "HDK_utils.h"
#include "DescriptorsDefinitions.h"
#include "ConfigurationManager_api.h"
#include "MacGeneralRegs.h"
#include "RficDriver_API.h"
#include "InterfererDetection_Api.h"
#include "OSAL_UpperMacMessages.h"
#include "OSAL_Tasks.h"
#include "CpuLoad_Api.h"
#include "PSD.h"
#include "Utils_Api.h"
#include "PhyTestBus_API.h"
#include "lib_wav600_api.h"
#ifdef ENET_INC_ARCH_WAVE600D2	
#include "PhyTxtdAnt0Regs.h"
#include "PhyTxtdAnt1Regs.h"
#include "PhyTxtdAnt2Regs.h"
#include "PhyTxtdAnt3Regs.h"
#else
#include "PhyTxTdAnt0Regs.h"
#include "PhyTxTdAnt1Regs.h"
#include "PhyTxTdAnt2Regs.h"
#include "PhyTxTdAnt3Regs.h"
#endif




#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 21

/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
#define ENABLE_RADAR_DEBUG 

/***************************************************************************/
/***                              Defines                                ***/
/***************************************************************************/
#ifdef ENABLE_RADAR_DEBUG
typedef struct PhyRadarDetectionDebug_s
{
	PhyDriverInterfererRadarDetectionSample_t	mirrorFifo[PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE];
	PhyDriverInterfererRadarDetectionSample_t *	startingSample_p;
	uint32 										writeCounter;
	uint8 										numberOfValidSamples;	
}PhyRadarDetectionDebug_t;

PhyRadarDetectionDebug_t phyRadarDetectionDebug;
#endif

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/
typedef struct _PhyMuGroupUspEntry_t
{
	uint32 enable		: 1;
	uint32 userPosition	: 2;
	uint32				: 0;
} PhyMuGroupUspEntry_t;

#define PHY_NUM_OF_MU_GROUPS			(64)

/***************************************************************************/
/***                     Private Function Prototypes                     ***/
/***************************************************************************/
static void phyDrvInterfererDetectionDisablePhy(void);
static void phyDrvPhyGenriscInterrupt(void);
static void phyDrvGetBandEdgeCahnnels(uint8 regulation, uint8 band, uint8 **outEdgeChannelList, uint8 *ListLen);
static uint8 phyDrvGetNewLowChannel(uint8 lowChannel, uint8 PrimaryChannelIndex, uint8 newBandWidth);
static bool phyDrvIsBandEdgeChannel(uint8 lowChannel, uint8 regulation, uint8 bandWidth, uint8 band);


/***************************************************************************/
/***                         Private Data                                ***/
/***************************************************************************/
static PhyConfigParams_t m_configParams;
static PhyDriverInterfererDetectionParameters_t PhyDriverInterfererDetectionParameters;

uint32 AAA_PHY_GENRISC_REGS[24] = {0};
RegPhyTxMacEmuPhyMacError_u txPacketErrorCauseRegister;
bool   PhyLocker = FALSE;

#define TX_IQ_CAL_BYPASS_OFF (0)
#define SHORT_RADAR_TIME_FOR_SUBBAND_BITMAP (0x8)
#define LONG_RADAR_ON_DC_MASK (0x6)//bit 1 and 2 enabled zero and 3 disable
#define SUBBAND_BITMAP_FOR_ALL_SUBBAND (0xf)
#define SUBBAND_THRESHOLD_FROM_MAX (10)
#define MINIMUM_VALUE_FOR_AMPLITUDE (-127) // In Wave500B real phy - this value was changed to -128. We don't take the change since this is not used yet in gen6.
#define CCA_MEASUREMENT_OFFSET_DB (0) 

/* set TX window size */
bool GotTxWindowFixFromPsd = FALSE;
uint8 FCC_5GHZ_EDGE_CHANNEL_ARR[FCC_5GHZ_EDGE_CHANNEL_ARR_LEN] = {36u, 64u, 100u, 144u, 149u, 165u};
uint8 FCC_2GHZ_EDGE_CHANNEL_ARR[FCC_2GHZ_EDGE_CHANNEL_ARR_LEN] = {1u, 11u};
uint8 TX_WINDOW[EDGES_TYPES_NUM][BANDWIDTH_ONE_HUNDRED_SIXTY+1] = {{0, 0, 0, 0},  //false
																   {0, 0, 0, 0}}; //true

 
/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

phyDrvInterfererDetectionDisablePhy 


Description:
------------
Handles the disabling Phy procedure from the interferer detection module point of
view

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void phyDrvInterfererDetectionDisablePhy()
{
	InterfererDetectionSetChannelMessageParameters_t *interfererDetectionSetChannelMessageParameters = NULL;
	K_MSG *interfererDetectionMessage = NULL; 
	
	if(PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending)
	{
		/* Disable Phy after set channel request */
		PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending = FALSE;
		if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
		{
			PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter = 0;
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_POINTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_READ_COUNTER_ADDRESS, 0);
		}
#if defined RADAR_DETECTION_ENABLED
		if(PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled)
		{
			PhyDriverInterfererDetectionParameters.radarDetectionReadCounter = 0;
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_POINTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, 0);
			RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_READ_COUNTER_ADDRESS, 0);
		}
#endif
		PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled = PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel;
#if defined RADAR_DETECTION_ENABLED
		PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled = PhyDriverInterfererDetectionParameters.isRadarDetectionNeededForNewChannel;
#endif
		RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_ENABLE_ADDRESS, PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel); 
#if defined RADAR_DETECTION_ENABLED
	    RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_ENABLE_ADDRESS, PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled);
#endif
    
		/*The following code should happen only in the first disable Phy of the set channel request proccess */
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionSetChannelMessageParameters_t));
		interfererDetectionSetChannelMessageParameters = ((InterfererDetectionSetChannelMessageParameters_t *)interfererDetectionMessage->abData);
		interfererDetectionSetChannelMessageParameters->channel =  PhyDriverInterfererDetectionParameters.newChannel;
#if defined RADAR_DETECTION_ENABLED
		interfererDetectionSetChannelMessageParameters->isRadarDetectionEnabled =  PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled;
#endif
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_SET_CHANNEL, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	else
	{
		/* Disable Phy not after set channel request - flush the FIFOs only for Continuous Interferer in 500b the timestamp are TSF sync */
		if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
		{
			PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(FALSE); 
		}
	}
}

/**********************************************************************************

phyDrvPhyGenriscInterrupt 


Description:
------------
Handles the interrupt triggered by the PHY genrsic

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
static void phyDrvPhyGenriscInterrupt()
{
	RegPhyRxTdIfPhyRxtdIf08_u phyGenriscInterruptCauseRegister;

	phyGenriscInterruptCauseRegister.val = 0;
	RegAccess_Read(REG_PHY_RX_TD_IF_PHY_RXTD_IF08, &phyGenriscInterruptCauseRegister.val);
   	ASSERT(!(phyGenriscInterruptCauseRegister.val & ~(PHY_GEN_RISC_INTEERUPT_ALL_CAUSES_MASK)));
	// Clear the causes the PHY genrsic wrote
	//there might be a race with the Phy genrisc and this must be solved in changing the HW
	RegAccess_WriteMasked(REG_PHY_RX_TD_IF_PHY_RXTD_IF08, phyGenriscInterruptCauseRegister.val, 0);
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_CONTINUOUS_INTERFERER_FIFO_OVERFLOW)
	{
		PhyDriverInterfererDetectionParameters.continuousInterfererFifoOverflowCounter++;
	}
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_MIPS_HALT_DONE)
	{
		FATAL("MIPS halt done");//TBD should be decided what to do in this case
	}
#if defined RADAR_DETECTION_ENABLED
	if(phyGenriscInterruptCauseRegister.val & PHY_GEN_RISC_INTEERUPT_RADAR_FIFO_OVERFLOW)
	{
		PhyDriverInterfererDetectionParameters.radarFifoOverflowCounter++;
		//ignore indication
		//FATAL("Radar FIFO overflow");//TBD should be decided what to do in this case
	}
#endif
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
void PhyDrv_LockReActivateBB(bool lock)
{
	PhyLocker = lock;
	ILOG0_D("PhyDrv_LockReActivateBB() PhyLocker: %d ",PhyLocker);
}
/***************************************************************************/
/***                        Private Functions                            ***/
/***************************************************************************/
/***************************************************************************
** NAME         PhyDrv_DisablePhy
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino resets  PHY operation.
****************************************************************************/
void PhyDrv_DisablePhy()
{
	PhyDrv_StopRisc();
	phy_disable();
	phyDrvInterfererDetectionDisablePhy();
}

/***************************************************************************
** NAME         PhyDrv_ResetAndEnable
** PARAMETERS   None
** RETURNS      None
** DESCRIPTION  This Functino Enables PHY operation.
****************************************************************************/
void PhyDrv_ResetAndEnable(void)
{
	PhyDrv_ReleaseSWreset();
	PhyDrv_EnableRx();
	PhyDrv_StartRisc();
}

void PhyDrv_ResetBB()
{
	PhyDrv_DisableBB();
	PhyDrv_ReleaseSWreset();
	PhyDrv_EnableRx();
}

void PhyDrv_ReActivateBB()
{
	PhyDrv_DisableBB();
	MT_DELAY(100);
	PhyDrv_EnableBB();
}
void PhyDrv_SetMacAntConfig(uint8 antMask)
{
	uint8 band = ConfigurationManager_GetMyBand();
	
	PHY_SET_MAC_ANT_CONFIG(band, antMask);
}
uint8 PhyDrv_ReadMacAntConfig(void)
{
	uint8 antMask;
	RegMacGeneralPhyBandConfig_u phyBandConfigReg;

	RegAccess_Read(REG_MAC_GENERAL_PHY_BAND_CONFIG, &phyBandConfigReg.val);

	if (ConfigurationManager_GetMyBand() == CONFIGURATION_MANAGER_BAND_0)
	{
		antMask = ((!phyBandConfigReg.bitFields.phyAntenna0Sel) |
					((!phyBandConfigReg.bitFields.phyAntenna1Sel)<<1)|
					((!phyBandConfigReg.bitFields.phyAntenna2Sel)<<2)|
					((!phyBandConfigReg.bitFields.phyAntenna3Sel)<<3));
	}
	else
	{
		antMask = ((phyBandConfigReg.bitFields.phyAntenna0Sel) |
					((phyBandConfigReg.bitFields.phyAntenna1Sel)<<1)|
					((phyBandConfigReg.bitFields.phyAntenna2Sel)<<2)|
					((phyBandConfigReg.bitFields.phyAntenna3Sel)<<3));
	}

	return antMask;
}

void PhyDrv_preCal()
{
	PHY_PRE_CAL();
} 

void PhyDrv_postCal()
{
	PHY_POST_CAL();
}

void PhyDrv_SWresetViaRisc(void)
{
}

void PhyDrv_ReleaseSWreset(void)
{
	phy_enable();
}


void PhyDrv_EnableRx(void)
{
	PHY_ENABLE_RX();
}

void PhyDrv_RFICreset(void)
{
	PHY_RFIC_RESET();
}

void PhyDrv_RFICreleaseReset(void)
{
	PHY_RFIC_RESET_RELEASE();
}

void PhyDrv_SetRxHalfBand(PhyRxHalfBandType_e halfBandType)
{
	PHY_SET_RX_HALFBAND(halfBandType);
}

void PhyDrv_SetTxAntOperationSet(uint8 opSet)
{
	m_configParams.txEnabledAntennas = opSet;
	PHY_SET_TX_ANT_OPERATION_SET(opSet);
}

void PhyDrv_SetRxAntOperationSet(uint8 opSet)
{
	m_configParams.rxEnabledAntennas = opSet;
	PHY_SET_RX_ANT_OPERATION_SET(opSet);
}

uint8 PhyDrv_GetTxPrimary(void)
{
	//PHYTBD
	return 0;
}

uint8 PhyDrv_GetRxEnabledAntennas(void)
{
	return m_configParams.rxEnabledAntennas;
}

uint8 PhyDrv_GetTxEnabledAntennas(void)
{
	return m_configParams.txEnabledAntennas;
}

RetVal_e PhyDrv_DisableBB()
{
	ILOG2_V("PhyDrv_DisableBB");
    PhyDrv_DisablePhy();
    return RET_VAL_SUCCESS;
}

RetVal_e PhyDrv_EnableBB(void)
{
	ILOG2_V("PhyDrv_EnableBB");
    //PHY activating sequence
    //Enable TD, TX (b, n), BE, FD, AFE modules
    PhyDrv_ResetAndEnable();
	return RET_VAL_SUCCESS;
}

void PhyDrv_Init( void )
{   	
}

void PhyDrv_UpdatePrimaryChannel(uint8 band, uint8 u8SpectrumMode, uint8 primary_chan_idx)
{
	uint8 halfbandmode;
	uint8 antMask = Hdk_GetRxAntMask();
	UNUSED_PARAM(band);	
	switch (u8SpectrumMode)
	{
		case BANDWIDTH_TWENTY:
			halfbandmode = 3;
			break;
		case BANDWIDTH_FOURTY:
			halfbandmode = 2;
			break;
		case BANDWIDTH_EIGHTY:
			halfbandmode = 1;
			break;
		default: //160MHz
			halfbandmode = 0;
			break;
	}

	PHY_SET_RX_HALFBAND(halfbandmode);
	PHY_SET_RX_PRIMARY(primary_chan_idx ,antMask);
	PHY_SET_TX_HALFBAND(halfbandmode);
	PHY_SET_TX_PRIMARY(primary_chan_idx,halfbandmode, antMask);
	PhyDrv_SetFrequency();
	
}

void PhyDrv_SetSpectrum(uint8 band, uint8 chanWidth, uint8 primary_chan_idx, bool isBWChanged)
{
	UNUSED_PARAM(isBWChanged);	
	PhyDrv_UpdatePrimaryChannel(band, chanWidth, primary_chan_idx);
}

void PhyDrv_StartRisc(void)
{
	uint32 isUp = 0;
	RegPhyRxTdIfPhyRxtdIf84_u regPhyRxTdIfPhyRxtdIf84;
	
	if (PhyLocker == FALSE)
	{
		if (Hdk_isTestBusEnable() == TRUE)
		{
			PhyTestBus_Open();
		}

		PHY_WRITE_REG(GENRISC_FINISHED_LOADING, 0, 0);
#if defined(ENET_INC_LMAC1) && !defined(ENET_INC_ARCH_WAVE600B)//HW trig patch for A0.
		RegAccess_Write(CDB_BAND, 1);
#endif
		PhyStartRiscOperation();
		PHY_READ_REG(GENRISC_FINISHED_LOADING, 0, &isUp);
		ASSERT(isUp != 0);
		//enable the WD int in the Phy
		RegAccess_Read(REG_PHY_RX_TD_IF_PHY_RXTD_IF84, &regPhyRxTdIfPhyRxtdIf84.val);
		regPhyRxTdIfPhyRxtdIf84.bitFields.wdIntMask = 1;	
		RegAccess_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF84, regPhyRxTdIfPhyRxtdIf84.val);

#ifndef GEN6_REL3_5_HW_BUG_PATCH
		PhyDrv_RestoreRegsAfterStartRisc();
#endif
	}
	
}

void PhyDrv_StopRisc(void)
{
	PhyStopRiscOperation();
}

void PhyDrv_ReadVersion(PhyVersion_t *outPhyVersion)
{
	UNUSED_PARAM(outPhyVersion);	
}

void PhyDrv_SetSpacelessTransmission(uint8 value)
{
	UNUSED_PARAM(value);	
	PHY_SET_SPACELESS();
}

void PhyDrv_SetScramblerMode(uint8 mode)
{
	RegPhyTxTxBeReg12_u scramblerReg;
	uint32 regVal = ((mode==1) ? PHY_SCRAMBLER_MODE_ON:PHY_SCRAMBLER_MODE_OFF);
	
	RegAccess_Read(REG_PHY_TX_TX_BE_REG_12, &scramblerReg.val);
	scramblerReg.bitFields.swScrInit0 = regVal;
	RegAccess_Write(REG_PHY_TX_TX_BE_REG_12,scramblerReg.val);
}


void PhyDrv_ReadRssi(OUT uint8 *rssiValues, OUT uint8 *energyEstNumOfSamples)
{
	PHY_READ_RSSI(rssiValues, energyEstNumOfSamples);
}

void PhyDrv_ReadPacketCounter(OUT uint32 *rxPacketCounter,OUT uint32 *rxCrcErrorPacketCounter)
{
	RegPhyRxBeIfPhyRxbeIf1A4_u gclkBypassReg;
	RegPhyRxBePhyRxbeReg25_u latchCountersReg;

	// Gclck bypass
	gclkBypassReg.val = 0;
	gclkBypassReg.bitFields.swGclkBypass = 0x3FFF;
	RegAccess_Write(REG_PHY_RX_BE_IF_PHY_RXBE_IF1A4, gclkBypassReg.val);

	// Latch Phy Counters
	latchCountersReg.val = 0;
	latchCountersReg.bitFields.latchCounters = 1;
	RegAccess_Write(REG_PHY_RX_BE_PHY_RXBE_REG25, latchCountersReg.val);
	
	//Read Counters from Phy
	RegAccess_Read(REG_PHY_RX_BE_PHY_RXBE_REG37,rxPacketCounter);
	RegAccess_Read(REG_PHY_RX_BE_PHY_RXBE_REG38,rxCrcErrorPacketCounter);
}

void PhyDrv_ResetPacketCounter(void)
{
	RegPhyRxBePhyRxbeReg24_u clearCountersReg;

	clearCountersReg.val = 0;
	clearCountersReg.bitFields.clearCounters = 1;
	RegAccess_Write(REG_PHY_RX_BE_PHY_RXBE_REG24, clearCountersReg.val);
}

void PhyDrv_WriteRxDutyCycleReq(uint32 onTime, uint32 offTime)
{	
	RegAccess_Write(DET_TH_TIMER_ON, onTime);
	RegAccess_Write(DET_TH_TIMER_OFF, offTime);
}

void PhyDrv_WriteRxThToPhy(int THval)
{
	RegAccess_Write(FAR_STATION_THRESHOLD, THval << DET_POWER_TH_SHIFT);
}

uint32 PhyDrv_GetRxAbortCounter(void)
{
	return REGISTER(MEM_PACKET_DROPPED_MAC_ABORT);
}

void PhyDrv_ActivateSemaphoreForAocs(uint32 activate)
{
	if(activate == TRUE)
	{
		HW_SEMAPHORE_LOCK(HW_SEMAPHORE_AOCS_INFO);
	}
	else
	{
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_AOCS_INFO);
	}
}

/**********************************************************************************
Name: PhyDrv_GetRxEvm

Description: on debug mode, get rx EVM from register, for all Antena's

Input/Output: get and fill struct pointer of arr[4]
	
**********************************************************************************/
void PhyDrv_GetRxEvm(OUT uint8 *rxEvm_p)
{
	uint32 ant;
	uint32 rxEvmForAllAntennas;
	
	RegAccess_Read(EVM_BUF_START, &rxEvmForAllAntennas);
	for(ant =0 ; ant < MAX_NUM_OF_DIGITAL_ANTENNAS; ant++)
	{
    	rxEvm_p[ant] = (rxEvmForAllAntennas >> (ant*8))&0xff;	
	}
}

/**********************************************************************************

isr_PhyDrv_PhyInterrupt 


Description:
------------
Handles interrupt from the PHY


Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void isr_PhyDrv_PhyInterrupt(void)
{
	RegPhyRxTdIfPhyRxtdIf86_u interruptStatusRegister;
	RegPhyRxTdIfPhyRxtdIf86_u interruptStatusRegisterMasked;
	RegPhyRxTdPhyRxtdReg021_u riscPageReg;
	uint32 regIndex = 0;

	ACCUMULATE_CPU_IDLE_TIME();
	interruptStatusRegister.val = 0;
	
	RegAccess_Read(REG_PHY_RX_TD_IF_PHY_RXTD_IF86, &interruptStatusRegister.val);
	ASSERT(interruptStatusRegister.val & PHY_DRIVER_INTERRUPT_EANBLE_MASK);
	DEBUG_ASSERT(!(interruptStatusRegister.val & ~(PHY_DRIVER_ALL_POSSIBLE_INTERRUPT_MASK)));
    // The cause and the clear bits are correlating so the value can be used to clear the interrupts
	RegAccess_Write(REG_PHY_RX_TD_IF_PHY_RXTD_IF85, interruptStatusRegister.val);

	interruptStatusRegisterMasked.val = (interruptStatusRegister.val & PHY_DRIVER_INTERRUPT_EANBLE_MASK);

	if(interruptStatusRegisterMasked.bitFields.txPacketIntCauseReg)
	{	
		RegAccess_Read(REG_PHY_TX_MAC_EMU_PHY_MAC_ERROR, &txPacketErrorCauseRegister.val);
		DEBUG_FATAL("Tx packet error interrupt occured");	
	}
	if(interruptStatusRegisterMasked.bitFields.phyMacIntCauseReg)
	{
		phyDrvPhyGenriscInterrupt();		
	}
	// WD interrupt, unknown problem cause the system to stuck - halt system.
	//   "watchdog" raise every ~70mSec and check that the GenRisc is not stuck, if so,
	//   reset operation is activated, and MAC receive an indication.
	if(interruptStatusRegisterMasked.bitFields.wdIntCauseReg)
	{
		// Read genrisc registers (registers + PC + some more). can be seen in SoC online.
		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG021, &riscPageReg.val);
		riscPageReg.bitFields.riscPage = 1;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG021, riscPageReg.val);
		
		for (regIndex = 0 ; regIndex < 20 ; regIndex++)
		{
			AAA_PHY_GENRISC_REGS[regIndex] = MT_RdReg(PHY_RX_TD_BASE_ADDRESS, (0x8000+(4*regIndex)));
		}
		AAA_PHY_GENRISC_REGS[20] = MT_RdReg(PHY_RX_TD_BASE_ADDRESS, 0x804c); // Read the PC 4 more times, since PHY can be in a loop
		AAA_PHY_GENRISC_REGS[21] = MT_RdReg(PHY_RX_TD_BASE_ADDRESS, 0x804c);
		AAA_PHY_GENRISC_REGS[22] = MT_RdReg(PHY_RX_TD_BASE_ADDRESS, 0x804c);
		AAA_PHY_GENRISC_REGS[23] = MT_RdReg(PHY_RX_TD_BASE_ADDRESS, 0x804c);

		RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG021, &riscPageReg.val);
		riscPageReg.bitFields.riscPage = 0;
		RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG021, riscPageReg.val);
		FATAL("WD(phy) interrupt occured");	
	}
}

/**********************************************************************************

PhyDrvRadarDetection_SetRssiTh 


Description:
------------
This Function called with initialized value by FW while its init.
The QA has the ability to change it on-the-fly for the radar tests,
by setting iwpriv and Driver will send a msg with new values.


Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/
void PhyDrvRadarDetection_SetRssiTh(int32 value)
{
     //The PHY configuration(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS) for minimum RSSI value in received signal.
     //Below this threshold, the PHY won't check if the input signal is radar.
	 RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS, value);
}


/**********************************************************************************

PhyDrvRadarDetection_GetRssiTh 


Description:
------------
return RSSI TH

Input:
-----


Output:
-------
	

Returns:
--------

**********************************************************************************/

int32 PhyDrvRadarDetection_GetRssiTh(void)
{
	uint32 val;

	RegAccess_Read(PHY_DRIVER_RADAR_DETECTION_RSSI_TH_ADDRESS, &val);

	return (int32)val;
}


/**********************************************************************************

PhyDrvInterfererDetection_SetChannelRequest 


Description:
------------
Handles set channel request

Input:
-----
new channel - the new channel that will be set
isRadarDetectionNeeded - a flag that indicates if radar detection needed
isContinuousInterfererDetectionNeeded - a flag that indicates if continuous 
interferer detection needed




Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_SetChannelRequest(uint8 newChannel, uint8 isRadarDetectionNeeded, uint8 isContinuousInterfererDetectionNeeded)
{
	PhyDriverInterfererDetectionParameters.isNewChannelSetRequestPending = TRUE;
	PhyDriverInterfererDetectionParameters.newChannel = newChannel;
#if defined RADAR_DETECTION_ENABLED
	PhyDriverInterfererDetectionParameters.isRadarDetectionNeededForNewChannel = isRadarDetectionNeeded;
#endif
	PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionNeededForNewChannel = isContinuousInterfererDetectionNeeded;
    /* Stop sampling in PHY genrisc in order to get overflow */
	if(PhyDriverInterfererDetectionParameters.isContinuousInterfererDetectionEnabled)
	{
		RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_ENABLE_ADDRESS, 0); 
	}
#if defined RADAR_DETECTION_ENABLED
	if(PhyDriverInterfererDetectionParameters.isRadarDetectionEnabled)
	{
		RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_ENABLE_ADDRESS, 0); 
	}
#endif
}

/**********************************************************************************

PhyDrvInterfererDetection_FlushRadarDetectionFifo 


Description:
------------
Process the radar detection FIFO and sends message to the Interferer task if needed

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if defined RADAR_DETECTION_ENABLED
void PhyDrvInterfererDetection_FlushRadarDetectionFifo()
{
	InterfererDetectionRadarDetectionSamplesMessageParameters_t *interfererDetectionRadarSamplesMessageParameters = NULL;
	K_MSG* interfererDetectionMessage = NULL;
	uint8 numberOfValidSamples = 0;
	uint32 writeCounter = 0;
	PhyDriverInterfererRadarDetectionSample_t *radarDetectionSample = NULL;
	uint8 numberOfSamplesInMessage = 0;
	uint8 index = 0;

	
	RegAccess_Read(PHY_DRIVER_RADAR_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, &writeCounter);
	numberOfValidSamples = writeCounter - PhyDriverInterfererDetectionParameters.radarDetectionReadCounter;

	ASSERT(numberOfValidSamples <= PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE);

    radarDetectionSample = (PhyDriverInterfererRadarDetectionSample_t *)PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS;
	radarDetectionSample += PhyDriverInterfererDetectionParameters.radarDetectionReadCounter &  PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE_MASK;

#ifdef ENABLE_RADAR_DEBUG
	/* Store: first sample location, write counter and the number of valid samples */
	phyRadarDetectionDebug.startingSample_p = radarDetectionSample;
	phyRadarDetectionDebug.writeCounter = writeCounter;
	phyRadarDetectionDebug.numberOfValidSamples = numberOfValidSamples;
	
	/* Store the fifo */
	MEMCPY(phyRadarDetectionDebug.mirrorFifo, (void *)PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS, PHY_DRIVER_RADAR_DETECTION_FIFO_SIZE * sizeof(PhyDriverInterfererRadarDetectionSample_t));

#endif

	while(numberOfValidSamples)
	{
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionRadarDetectionSamplesMessageParameters_t));
		interfererDetectionRadarSamplesMessageParameters = ((InterfererDetectionRadarDetectionSamplesMessageParameters_t *)interfererDetectionMessage->abData);
		if(numberOfValidSamples > MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE)
		{
			numberOfSamplesInMessage = MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE;
		}
		else
		{
			numberOfSamplesInMessage = numberOfValidSamples;
		}
		numberOfValidSamples -= numberOfSamplesInMessage;
		interfererDetectionRadarSamplesMessageParameters->numberOfSamples = numberOfSamplesInMessage;
		
		ASSERT(numberOfSamplesInMessage <= MAX_NUMBER_OF_RADAR_SAMPLES_IN_MESSAGE); //KW_FIX_FW_G
		//RADAR_DETECTION_SAMPLE_SIZE (sizeof(RadarDetectionSample_t))
		//KW IGNORE ABV.GENERAL sizeof (RadarDetectionSample_t) is not being calculated by kw. When RADAR_DETECTION_SAMPLE_SIZE is hardcoded to 8 this issue goes off.
		for(index = 0; index < numberOfSamplesInMessage; index++)
		{
			interfererDetectionRadarSamplesMessageParameters->pulses[index].timeStamp = radarDetectionSample->timeStamp;
			interfererDetectionRadarSamplesMessageParameters->pulses[index].pulseWidth = DIVIDE_BY_20(radarDetectionSample->pulseWidth & 0xFFFF);
			// TBD HDK - subband DFS was  not ported to gen6 yet
			interfererDetectionRadarSamplesMessageParameters->pulses[index].ampSubBandBitMap = 0xF; //radarDetectionSample->amp - PHY_DRIVER_INTERFERER_DETECTION_REDUCE_AMP_VALUE;
			radarDetectionSample++;
			if((uint32)radarDetectionSample == PHY_DRIVER_RADAR_DETECTION_FIFO_END_ADDRESS)
			{
				// Wrap around - go to begining of FIFO 
				radarDetectionSample = (PhyDriverInterfererRadarDetectionSample_t *)PHY_DRIVER_RADAR_DETECTION_FIFO_BASE_ADDRESS;
			}
		}
		PhyDriverInterfererDetectionParameters.radarDetectionReadCounter += numberOfSamplesInMessage;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_RADAR_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	RegAccess_Write(PHY_DRIVER_RADAR_DETECTION_FIFO_READ_COUNTER_ADDRESS, PhyDriverInterfererDetectionParameters.radarDetectionReadCounter);
}
#endif

/**********************************************************************************

PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo 


Description:
------------
Process the continuous interferer detection FIFO and sends message to the 
Interferer task if needed

Input:
-----
isNotificationNeeded - a flag for the interferer task if notification is needed


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo(uint8 isNotificationNeeded)
{

	InterfererDetectionBtInterfererDetectionSamplesMessageParameters *interfererDetectionContinuousInterfererSamplesMessageParameters = NULL;
	K_MSG* interfererDetectionMessage = NULL;
	uint8 numberOfValidSamples = 0;
	uint32 writeCounter = 0;
	uint8 numberOfSamplesInMessage = 0;
	int32 continuousInterfererSample;
	uint8 index = 0;
	uint32 continuousInterfererSampleAddr = 0;
	
    RegAccess_Read(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_WRITE_COUNTER_ADDRESS, &writeCounter);
	numberOfValidSamples = writeCounter - PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter;
	ASSERT(numberOfValidSamples <= PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE);
	continuousInterfererSampleAddr = PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_BASE_ADDRESS + 4*(PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter & PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE_MASK);
	if(isNotificationNeeded && (0 == numberOfValidSamples))
	{
		// In case a notification is needed but there are no valid samples a message still needs to be sent to the interferer detection module
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionBtInterfererDetectionSamplesMessageParameters));
		interfererDetectionContinuousInterfererSamplesMessageParameters = ((InterfererDetectionBtInterfererDetectionSamplesMessageParameters *)interfererDetectionMessage->abData);
		interfererDetectionContinuousInterfererSamplesMessageParameters->numberOfSamples = 0;
		interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = TRUE;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	while(numberOfValidSamples)
	{
		interfererDetectionMessage = OSAL_GET_MESSAGE(sizeof(InterfererDetectionBtInterfererDetectionSamplesMessageParameters));
		interfererDetectionContinuousInterfererSamplesMessageParameters = ((InterfererDetectionBtInterfererDetectionSamplesMessageParameters *)interfererDetectionMessage->abData);
#if (PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_SIZE > MAX_NUMBER_OF_BT_INTERFERER_SAMPLES_IN_MESSAGE) //KW_FIX_FW_G
		if(numberOfValidSamples > MAX_NUMBER_OF_BT_INTERFERER_SAMPLES_IN_MESSAGE)
		{
			interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = FALSE;
			numberOfSamplesInMessage = MAX_NUMBER_OF_BT_INTERFERER_SAMPLES_IN_MESSAGE;
		}
		else
#endif
		{
			// Insert the flag only in the last message
			interfererDetectionContinuousInterfererSamplesMessageParameters->isNotificationNeeded = isNotificationNeeded;
			numberOfSamplesInMessage = numberOfValidSamples;
		}
		numberOfValidSamples -= numberOfSamplesInMessage;
		interfererDetectionContinuousInterfererSamplesMessageParameters->numberOfSamples = numberOfSamplesInMessage;
		//ILOG0_D("PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo	numberOfSamplesInMessage %d", numberOfSamplesInMessage );
		for(index = 0; index < numberOfSamplesInMessage; index++)
		{
			continuousInterfererSample = REGISTER(continuousInterfererSampleAddr);
			continuousInterfererSample >>= 24;
			interfererDetectionContinuousInterfererSamplesMessageParameters->measurements[index] = continuousInterfererSample;
			//ILOG0_DDD("PhyDrvInterfererDetection_FlushContinuousInterfererDetectionFifo. index = %d addr in FIFO 0x%x continuousInterfererSample(>>24) = %d ", index, continuousInterfererSampleAddr, continuousInterfererSample);
			
			continuousInterfererSampleAddr = continuousInterfererSampleAddr+4;
			if(continuousInterfererSampleAddr == PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_END_ADDRESS)
			{
				// Wrap around - go to begining of FIFO
				continuousInterfererSampleAddr = PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_BASE_ADDRESS;
			}
		}
		PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter += numberOfSamplesInMessage;
		OSAL_SEND_MESSAGE(INTERFERER_DETECTION_CONTINUOUS_INTERFERER_DETECTION_SAMPLES, TASK_INTERFERER_DETECTION, interfererDetectionMessage, GET_DEFAULT_VAP_FOR_MY_BAND());
	}
	RegAccess_Write(PHY_DRIVER_CONTINUOUS_INTERFERER_DETECTION_FIFO_READ_COUNTER_ADDRESS, PhyDriverInterfererDetectionParameters.continuousInterfererDetectionReadCounter);
}

/**********************************************************************************

PhyDrv_ClearPhyInterrupts 


Description:
------------
Clear interrupts from the PHY (used in intialization)

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrv_ClearPhyInterrupts()
{
	PHY_CLEAR_PHY_INTERRUPTS();
}


/**********************************************************************************

PhyDrvInterfererDetection_Initialize 


Description:
------------
Initialize the parameters of the interferer detection in the PHY driver 

Input:
-----


Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void PhyDrvInterfererDetection_Initialize()
{
	memset(&PhyDriverInterfererDetectionParameters, 0, sizeof(PhyDriverInterfererDetectionParameters_t));
}

/**********************************************************************************

PhyDrv_SetCcaTh 


Description:
------------
set CCA thresholds 

Input:
-----


Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void PhyDrv_SetCcaTh(K_MSG * psMsg)
{
	UMI_CCA_TH_t* params;
	RegPhyRxTdPhyRxtdReg07A_u ccaThReg;

	params = (UMI_CCA_TH_t*)pK_MSG_DATA(psMsg);
	ILOG0_D("PhyDrv_SetCcaTh  params->primaryChCCA	= %d",params->primaryChCCA);
	ILOG0_D("PhyDrv_SetCcaTh  params->secondaryChCCA  = %d", params->secondaryChCCA);
	ILOG0_D("PhyDrv_SetCcaTh  params->sec40CcaTh  = %d",params->sec40CcaTh);
	ILOG0_D("PhyDrv_SetCcaTh  params->sec80CcaTh  =%d", params->sec80CcaTh);
	
	ccaThReg.bitFields.pmCcaPdTh20Prim = params->primaryChCCA	  + CCA_MEASUREMENT_OFFSET_DB;
	ccaThReg.bitFields.pmCcaPdTh20Sec = params->secondaryChCCA	  + CCA_MEASUREMENT_OFFSET_DB;
	ccaThReg.bitFields.pmCcaPdTh40 = params->sec40CcaTh 		  + CCA_MEASUREMENT_OFFSET_DB;		
	ccaThReg.bitFields.pmCcaPdTh80 = params->sec80CcaTh 		  + CCA_MEASUREMENT_OFFSET_DB;		

	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG07A, ccaThReg.val);

}

void PhyDrv_GetCcaTh(K_MSG* psMsg)
{
	UMI_CCA_TH_t* params;
	RegPhyRxTdPhyRxtdReg07A_u ccaThReg;

	params = (UMI_CCA_TH_t*)pK_MSG_DATA(psMsg);
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG07A,&ccaThReg.val);

	params->primaryChCCA = ccaThReg.bitFields.pmCcaPdTh20Prim;
	params->secondaryChCCA = ccaThReg.bitFields.pmCcaPdTh20Sec;
	params->sec40CcaTh = ccaThReg.bitFields.pmCcaPdTh40; 
	params->sec80CcaTh = ccaThReg.bitFields.pmCcaPdTh80;

	ILOG0_D("PhyDrv_GetCcaTh  params->primaryChCCA	= %d",params->primaryChCCA);
	ILOG0_D("PhyDrv_GetCcaTh  params->secondaryChCCA  = %d", params->secondaryChCCA);
	ILOG0_D("PhyDrv_GetCcaTh  params->sec40CcaTh  = %d",params->sec40CcaTh);
	ILOG0_D("PhyDrv_GetCcaTh  params->sec80CcaTh  =%d", params->sec80CcaTh);
}


/**********************************************************************************

Description:
------------
restores fields which were reset in start RISC 

Input:
-----
Output:
-------
Returns:
--------
**********************************************************************************/
void PhyDrv_RestoreRegsAfterStartRisc()
{
	RegPhyRxFdIfPhyRxfdIf12D_u regIF12D;

	RegAccess_Read(REG_PHY_RX_FD_IF_PHY_RXFD_IF12D, &regIF12D.val);
	regIF12D.bitFields.genriscBfEn = 1;
	RegAccess_Write(REG_PHY_RX_FD_IF_PHY_RXFD_IF12D, regIF12D.val);
}

/********************************************************************************
PhyDrv_Set_11B_RxAntSelection:

Description:
------------

Set Rx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_RxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	PHY_SET_11B_RX_ANT_SELECTION(Ant_selection);
}

/********************************************************************************
PhyDrv_Set_11B_TxAntSelection:

Description:
------------

Set Tx selected antenna. (802.11.b)
Input:
-----

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_TxAntSelection(PhyDrv_11BAntSelection_e Ant_selection)
{
	PHY_SET_11B_TX_ANT_SELECTION(Ant_selection);
}

/********************************************************************************
PhyDrv_Set_11B_roundRobinSelection:

Description:
------------

Set roundRobin mode for tx and rx. (802.11.b)
Input:
-----
rate for switching ant

Output:
-------

********************************************************************************/
void PhyDrv_Set_11B_roundRobinSelection(uint8 rate)
{
	PHY_SET_11B_ROUNDROBIN(rate);
}

void PhyDrv_SetVhtNonAggregate(uint8 enable)
{
	PHY_SET_VHT_NONAGGREGATE(enable);
}
void PhyDrv_SetCompressedFourierPhases(uint8 numOfAnts)
{
	PhyCompFourierPhasesTableDefinition_t* pCompressedFourierTableDef;
	uint8 tableSize;
	uint8 rowIndex = 0;
	uint32 address, mask, value;  

	ILOG0_V("PhyDrv_SetCompressedFourierPhases, Enter function"); 

	if (numOfAnts > 0)
	{
		pCompressedFourierTableDef = getCompressedFourierPhasesTable(numOfAnts);

		if (pCompressedFourierTableDef != NULL) //if table was  update
		{
			tableSize = pCompressedFourierTableDef->phyAntConfigTableSize;
			ILOG0_D("PhyDrv_SetCompressedFourierPhases, tableSize = %d",tableSize); 

			for (rowIndex = 0; rowIndex < tableSize; rowIndex++)
			{
				address = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].address;
				mask    = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].mask;
				value   = pCompressedFourierTableDef->PhyAntConfigTablePtr[rowIndex].value;
				RegAccess_WriteMasked(address, mask , value);
				ILOG0_D("PhyDrv_SetCompressedFourierPhases, address = %X",address); 
			}
		}
	}
}
void PhyDrv_SetFrequency()
{
	uint32 freq = RficDriver_CalcLoFrequency(HDK_getSetChannelParams());
	uint8 mask = Hdk_GetRxAntMask();
	PHY_SET_FREQUENCY(freq, mask);
	PHY_SET_BAND(HDK_getBand());
}

void PhyDrv_SetBssColor(uint8 vapId, uint8 bssColor)
{
	PHY_SET_BSS_COLOR(vapId, bssColor);
}

void PhyDrv_PowerSave_RadioOff(void)
{

}

void PhyDrv_macEmuStop(void)
{
	bbic_mac_emu_deactivate();
}

void PhyDrv_macEmuRun(uint8 antMask, uint8 RfPow, uint16 bw, uint8 packet_format, uint8 mcs, uint8 nss, uint32 packet_length, uint8 ldpc, uint8 gap, uint16 repeats)
{
	bbic_mac_emu_set_tcr(antMask, RfPow, bw, packet_format, mcs, nss, packet_length, ldpc);
	bbic_mac_emu_set_inter_packet_gap(gap);
	bbic_mac_emu_set_num_of_packets(repeats);//num of packets 1-0xffff
	bbic_mac_emu_activate();
}

void PhyDrv_setPgcFiltLut(int8 AntMsk , uint8 PgcFiltLUT_local[5][11])
{
	bbic_set_PgcFiltLut(AntMsk , PgcFiltLUT_local);
}


static void phyDrvGetBandEdgeCahnnels(uint8 regulation, uint8 band, uint8 **outEdgeChannelList, uint8 *ListLen)
{
	*outEdgeChannelList = NULL;
	*ListLen = 0;

	if(regulation == REGD_CODE_FCC)
	{
		if (band == BAND_5_2_GHZ)
		{
			*outEdgeChannelList = FCC_5GHZ_EDGE_CHANNEL_ARR;
			*ListLen = FCC_5GHZ_EDGE_CHANNEL_ARR_LEN;
		}

		else if (band == BAND_2_4_GHZ)
		{
			*outEdgeChannelList = FCC_2GHZ_EDGE_CHANNEL_ARR;
			*ListLen = FCC_2GHZ_EDGE_CHANNEL_ARR_LEN;
		}
	}
}

/********************************************************************************
phyDrvGetNewLowChannel:

Description:
------------
use a low channel, primary index and the new BW and calculate the new low channel. 
assumption - the new BW is smaller then the old one

Input:
-----
low channel, primary index and the new BW


Output:
-------
calculated new low channel

********************************************************************************/
static uint8 phyDrvGetNewLowChannel(uint8 lowChannel, uint8 PrimaryChannelIndex, uint8 newBandWidth)
{
	uint8 NewLowChannel;
	
	NewLowChannel = lowChannel + CHANNEL_SIZE*((PrimaryChannelIndex>>newBandWidth)<<newBandWidth);

	return(NewLowChannel);
}


/********************************************************************************
phyDrvIsBandEdgeChannel:

Description:
------------
use phyDrvGetBandEdgeCahnnels to get a list of band edge channel.
calculate the high channel 
return true if the low channel or the high channel are in the band edge channel list,
if not, returns false.

Input:
-----
low channel, regulation, band and BW


Output:
-------
true - if the given channel is a band edge channel in the given regulation
false - if not

********************************************************************************/
static bool phyDrvIsBandEdgeChannel(uint8 lowChannel, uint8 regulation, uint8 bandWidth, uint8 band)
{
	bool isEdge = FALSE;
	uint8 index, listLen = 0;
	uint8 *edgeList = NULL;
	uint8 highChannel = lowChannel + CHANNEL_SIZE*(EXPONENT_TO_VALUE(bandWidth));

	phyDrvGetBandEdgeCahnnels(regulation, band, &edgeList, &listLen);
	
	ILOG0_DD("in phyDrvIsBandEdgeChannel - lowChannel: %d, highChannel: %d", lowChannel, highChannel);
	
	if (edgeList == NULL)
	{
		return(FALSE);
	}

	//go through the list and see if the lowChannel \ highChannel are in it 
	for (index = 0 ; index < listLen ; index++)
	{
		isEdge = ((edgeList[index] == lowChannel) || (edgeList[index] == highChannel));
		
		ILOG0_DDD("in phyDrvIsBandEdgeChannel - index: %d, value: %d, isEdge: %d", index, edgeList[index], isEdge);
		
		if (isEdge)
		{
			break;
		}
	}
	
	return(isEdge);
}


/********************************************************************************
PhyDrv_BandEdgeFix:

Description:
------------
use the set channel parameters to determine if we will be a band edge channel 
at diffrent boundrates.
Use the resoult to update a table in the fast mem. 
The PHY uses these values to determin the TX window size per packet 

Input:
-----
set channel parameters. use the band, regulation, and the channel parameters


Output:
-------

********************************************************************************/
void PhyDrv_BandEdgeFix(SetChannelParams_t* setChannelReqParams)
{
	HdkSetChannelReqParams_t *channelReqParams = &(setChannelReqParams->setChannelReqParams);
	
	uint8 tempLowChannel = 0, BwIndex = 0;
	uint32 newTxWindowValues = 0;
	bool isEdge = FALSE;
	
	//we didn't get values from PSD - our TX window values are invalid
	if (!GotTxWindowFixFromPsd) 
	{
		ILOG0_D("in PhyDrv_BandEdgeFix - flag: %d", GotTxWindowFixFromPsd);
		return;
	}
	
	for (BwIndex = BANDWIDTH_TWENTY ; BwIndex <= BANDWIDTH_ONE_HUNDRED_SIXTY ; BwIndex++)
	{
		/* 1) when we are band edge channel, we will stay band edge channel
		   2) dont bother with unreachable BW */
		if ((!isEdge) && (BwIndex <= channelReqParams->chan_width))
		{
			tempLowChannel = phyDrvGetNewLowChannel(channelReqParams->low_chan_num, 
												channelReqParams->primary_chan_idx, BwIndex);
		
			isEdge = phyDrvIsBandEdgeChannel(tempLowChannel, channelReqParams->RegulationType, 
												BwIndex, setChannelReqParams->band);
		}
		
		newTxWindowValues |= (((uint32)TX_WINDOW[isEdge][BwIndex]) << (TX_WINDOW_SIZE*BwIndex));
	}

	// save for the PHY use
	ILOG0_D("in PhyDrv_BandEdgeFix - Tx Window Values: %d", newTxWindowValues);
	RegAccess_Write(TX_WIN_BANDEDGE_FIX_TABLE, newTxWindowValues);	
	
	/* to avoid a race condition at startup so the PHY will not read a non valid value before we set it */
	RegAccess_Write(TX_WIN_BANDEDGE_FIX_ENABLER, 1);	
}

/********************************************************************************
phyDrv_SetBandEdgeFixForTxWindow:

Description:
------------
read the TX window sizes from PSD for band edge fix

Input:
-----
table of BW x isEdge

Output:
-------

********************************************************************************/
void phyDrv_SetBandEdgeFixForTxWindow(uint32* windowFixsTable)
{
	uint8 BwIndex, type;
	uint32 TxWindowValue;
	for (BwIndex = BANDWIDTH_TWENTY ; BwIndex <= BANDWIDTH_ONE_HUNDRED_SIXTY ; BwIndex++)
	{
		for (type = 0 ; type < EDGES_TYPES_NUM  ; type++)
		{
			TxWindowValue = windowFixsTable[type*(BANDWIDTH_ONE_HUNDRED_SIXTY+1) + BwIndex];
			ASSERT((TX_WINDOW_MASK & TxWindowValue) == TxWindowValue); //make sure the TX window parameter is in the expected size
			TX_WINDOW[type][BwIndex] = TxWindowValue;
		}
	}

	GotTxWindowFixFromPsd = TRUE;
	RegAccess_Write(TX_WIN_BANDEDGE_FIX_ENABLER, 0);
}

void phyDrv_SetTxWindow(uint8 windowLenMode)
{
	RegPhyTxtdAnt0TxtdWindowing_u antWindowingReg;
	int8 i;
	uint8 bandAntMask = Hdk_GetRxAntMask();
	uint8 antIterator;
	for(i=0, antIterator = bandAntMask; ((i < MAX_NUM_OF_ANTENNAS) &&  (antIterator != 0)); i++, antIterator >>= 0x1)
	{
		if ((antIterator & 0x1) == 0x1)
		{
			RegAccess_Read(REG_PHY_TXTD_ANT0_TXTD_WINDOWING + i*TX_ANT_REGS_OFFSET,&(antWindowingReg.val));
			antWindowingReg.bitFields.pmTxwinCp4 = windowLenMode;
			antWindowingReg.bitFields.pmTxwinCp8Ss = windowLenMode;
			RegAccess_Write(REG_PHY_TXTD_ANT0_TXTD_WINDOWING + i*TX_ANT_REGS_OFFSET,antWindowingReg.val);
			/*DEBUG*/
			SLOG0(0, 0, RegPhyTxtdAnt0TxtdWindowing_u, &(antWindowingReg.val));
		}
	}
}

void PhyDrv_SetConnectedSTAind(uint8 isSTAConnected)
{
#if !defined(ENET_INC_ARCH_WAVE600D2) && defined(ENET_INC_ARCH_WAVE600B)
	PHY_WRITE_REG_FIELD(PHY_DRIVER_ERP_IS_STA_CONNECTED_ADDRESS,PHY_DRIVER_ERP_IS_STA_CONNECTED_OFFSET,PHY_DRIVER_ERP_IS_STA_CONNECTED_IND_SHIFT,PHY_DRIVER_ERP_IS_STA_CONNECTED_IND_MASK,isSTAConnected << PHY_DRIVER_ERP_IS_STA_CONNECTED_IND_SHIFT);
#else/*not valid for 600D2/600*/
	UNUSED_PARAM(isSTAConnected);
#endif
}

void PhyDrv_SetPowerSave(uint8 enableDisable)
{
#if !defined(ENET_INC_ARCH_WAVE600D2) && defined(ENET_INC_ARCH_WAVE600B)
	PHY_WRITE_REG_FIELD(PHY_DRIVER_ERP_IS_STA_CONNECTED_ADDRESS,PHY_DRIVER_ERP_IS_STA_CONNECTED_OFFSET,PHY_DRIVER_ERP_IS_STA_CONNECTED_ENABLE_SHIFT,PHY_DRIVER_ERP_IS_STA_CONNECTED_ENABLE_MASK,enableDisable << PHY_DRIVER_ERP_IS_STA_CONNECTED_ENABLE_SHIFT);
#else/*not valid for 600D2/600*/
	UNUSED_PARAM(enableDisable);
#endif
}

void PhyDrv_PowerSaveInit(void)
{
#if !defined(ENET_INC_ARCH_WAVE600D2) && defined(ENET_INC_ARCH_WAVE600B)
	PhyDrv_SetConnectedSTAind(ENABLED);
	PhyDrv_SetPowerSave(ENABLED);
#endif
}
