/*****************************************************************************
*    
*   Source File:
*       RficDriver_API.h
*
*   AUTHOR: Amir Hitron
*
*   Description:Driver of the Rfic  
*       
*   Copyright: 
*       (C) Metalink Ltd.
*       All rights are strictly reserved. Reproduction or divulgence in any 
*       form whatsoever is not permitted without written authority from the 
*       copyright owner. Issued by Metalink Transmission Devices Ltd in 
*       Israel - 11/94.
*   Revision History:
*
*****************************************************************************/
#ifndef __RFIC_DRIVER_API_H
#define __RFIC_DRIVER_API_H

/******************************************************************************/
/***                        Include Files                                   ***/
/******************************************************************************/
#include "HdkGlobalDefs.h"
#include "RFICDefines.h"
#include "System_GlobalDefinitions.h"
#include "calibrationsParamsOtf.h"
#include "CalibrationsDefines.h"
#include "lmi.h"
#include "PhyCalDriver_API.h"
#include "mhi_dut.h"

/******************************************************************************/
/***                        Constant Defines                                ***/
/******************************************************************************/
/* Central bias calibration Rint min/Max values */
#define RFIC_CENT_BIAS_RINT_MAX          (0x1F)
#define RFIC_CENT_BIAS_RINT_MIN          (0x00)
#define NUMOFBANDS 2

/******************************************************************************/
/***                        Enum Definition                                 ***/
/******************************************************************************/

typedef struct _IRE_t
{
	uint32 bandRange[NUMOFBANDS];
	uint32 value[NUMOFBANDS];
	uint32 mask[NUMOFBANDS];
} IRE_t;

typedef enum
{
    RFIC_DRIVER_PSD_DB = 0,
	RFIC_DRIVER_ZWDFS_PSD_DB,
	
} RfregPsdDb_e;


typedef enum
{
    DRIVER = 0,
	FW,
} RficDriverOrFWSelectFilter;


typedef enum
{
    RFIC_DAC0 = 0,
    RFIC_DAC1,
    RFIC_DAC2,
    RFIC_DAC_INDEX_MAX,
} RficDacIndex_e;

/* Rfic loop back types*/
typedef enum RficLoopbackTypeIndex
{
    RFIC_LOOPBACK_TYPE_TX_AMDET  = 0,   //Used in Tx LO & Tx IQ MM calibrations
    RFIC_LOOPBACK_TYPE_RX_LPF,          //Used in Rx LPF calibration
    RFIC_LOOPBACK_TYPE_TX_LPF,          //Used in Tx LPF calibration
    RFIC_LOOPBACK_TYPE_RSSI,            //Used in RSSI Path calibration
    //In VCO,Dc & TPC calibrations there are no loop backs
    RFIC_LOOPBACK_TYPE_MAX,
} RficLoopbackTypeIndex_e;

typedef enum RficRxPgcInputPowerState
{
    RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_ON   = 0,
    RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_OFF = 1,
	RFIC_RX_PGC_INPUT_POWER_STATE_ALL_ON = 2,
    RFIC_RX_PGC1_SHORT_INPUT_TO_GROUND_MAX,
} RficRxPgcInputPowerState_e;

typedef enum
{
    RFIC_PLL_LOCK_BIT_INVALID,
    RFIC_PLL_LOCK_BIT_VALID,
    RFIC_PLL_LOCK_BIT_MAX,
} RficDriverPllLockBit_e;

/* RSSI DAC indexes in array*/
typedef enum RficRssiDacIndexes
{
	RFIC_RSSI_STG_I_1_DAC = 0,
	RFIC_RSSI_STG_I_2_DAC,
	RFIC_RSSI_STG_I_3_DAC,
	RFIC_RSSI_STG_I_4_DAC,
	RFIC_RSSI_STG_I_5_DAC,
	RFIC_RSSI_STG_Q_1_DAC,
	RFIC_RSSI_STG_Q_2_DAC,
	RFIC_RSSI_STG_Q_3_DAC,
	RFIC_RSSI_STG_Q_4_DAC,
	RFIC_RSSI_STG_Q_5_DAC,
	RFIC_RSSI_STG_TOTAL
} RficRssiDacIndexes_e;

/* AM detector branch, used by Goertzel init                                    */
/* The AM detector block of the 2.4GHz RF is connected to the I branch,         */
/* and the AM detector block of the 5.2GHz RF is connected to the Q branch      */
/* The enum is defined this way since in th HW (Goertzel) 0 stands for Q branch */
typedef enum RficGoertzelAmDetBranch
{
    RFIC_AM_DET_Q_BRANCH = 0,
    RFIC_AM_DET_I_BRANCH = 1,
} RficAmDetBranch_e;

//Rffe enum Definition
typedef enum CalRFFEState
{
	TX_MODE,
	RX_MODE,
	T_STATE_MODE,
	PA_MODE,
	EXT_LNA_BYPASS_MODE,
	DUMMY_VALUE_1,
	DUMMY_VALUE_2,
	DUMMY_VALUE_3,
	DUMMY_VALUE_4,
	DUMMY_VALUE_5,
	RXIQ_MODE,
	NUM_OF_RFFE_MODE
}CalRFFEState_e;

typedef enum RffeID
{
	TX_ID=65,
	RX_ID,
	T_STATE_ID
}RffeID_e;

/* Pages enum to support Wave300's Rx page (0) and Tx page(1) in the RF registers */
typedef enum RficRegistersPage
{
    RF_REGISTERS_PAGE_NONE = 0,
    RF_REGISTERS_PAGE_RX = 0,
    RF_REGISTERS_PAGE_TX,
    RF_REGISTERS_PAGE_MAX,
} RficRegistersPage_e;

typedef enum RxDcOffset_ChipMode
{
	GAIN0_CH_MODE0 = 	  0x071,
	GAIN0_CH_MODE1 = 	  0x072,
	GAIN0_CH_MODE2 = 	  0x073,
	GAIN1_CH_MODE0 = 	  0x074,
	GAIN1_CH_MODE1 = 	  0x075,
	GAIN1_CH_MODE2 = 	  0x076,
	GAIN2_CH_MODE0 = 	  0x077,
	GAIN2_CH_MODE1 = 	  0x078,
	GAIN2_CH_MODE2 = 	  0x079,
	GAIN3_CH_MODE0 = 	  0x07A,
	GAIN3_CH_MODE1 = 	  0x07B,
	GAIN3_CH_MODE2 = 	  0x07C,
	GAIN4_CH_MODE0 = 	  0x07D,
	GAIN4_CH_MODE1 =   	  0x07E,
	GAIN4_CH_MODE2 = 	  0x07F,

	pgc2rxdc0_radar = 	0x055,//gain 18
	pgc2rxdc1 = 	  	0x056,//gain 12
	pgc2rxdc2 =			0x057,
	mixDc     = 		0x06B,
} RxDcOffset_ChipMode;

#define RXDC_DIG_CORRECTION1_MASK	(0x1)
#define RXDC_DIG_CORRECTION1_SHIFT	(0)
#define RXDC_DIG_CORRECTION2_MASK	(0x2)
#define RXDC_DIG_CORRECTION2_SHIFT	(1)
#define RXDC_DIG_CORRECTION3_MASK	(0x4)
#define RXDC_DIG_CORRECTION3_SHIFT	(2)

#define RXDC_NUM_SELECTION	 3 //RxDcOffset_ChipMode adresses to jump from GAIN0_CH_MODE0 to GAIN1_CH_MODE0

/******************************************************************************/
/***                        Type Definition                                 ***/
/******************************************************************************/
/* This struct is used to describe a register in the RF per antenna */
typedef struct RficRfRegistersDB
{
    uint16 regAddr;
#ifdef ENET_INC_ARCH_WAVE600	
    uint16 regData[FCSI_TOTAL_CHANNELS];
#else
	uint16 regData[MAX_NUM_OF_ANTENNAS];
#endif
} RficRfRegistersDB_t;

typedef struct RficPSDRegistersDB
{
    uint16 regAddr;
	uint32 mask;
    uint16 regData[MAX_NUM_OF_ANTENNAS];
} RficPSDRegistersDB_t;

typedef struct RficTxGains
{
	uint8 bbGain;
	uint8 paDrivergain;
} RficTxGains_t;

typedef struct RficRxGains
{
    int8  dac0Values[RFIC_PATH_TYPE_MAX];
    int8  dac1Values[RFIC_PATH_TYPE_MAX];
    int8  dac2Values[RFIC_PATH_TYPE_MAX];
    HdkLnaIndex_e lnaIndex;
    uint8 lnaActiveValue;
    int32  pgc1Gain;
    int32  pgc2Gain;
    int32  pgc3Gain;
} RficRxGains_t;

#ifndef ENET_INC_ARCH_WAVE600
typedef struct RficDriverTxLoTpcInfo
{
    uint8  tpcDiffFromRef[RFIC_TX_GAIN_TPC_INDEX_MAX]; /* Difference, in dB, from reference verify TPC index gain */
    uint8  tpcArraySize; /* array size of TPC's for which I,Q Lo DACs should be set in the RFIC */
    uint8  numCalibTpcs; /* Number of calibrated TPC's <= tpcArraySize */
    byte   calibTpcMask; /* bitmask of calibrated TPC's subset inside the tpcArray */
} RficDriverTxLoTpcInfo_t;//Digital DAC calibration details

typedef struct{
	uint8	maxIter;
	int8	clipMin;
	int8	clipMax;
	bool	doClip;	
	uint8	iterCounter;
	uint8	divider;
	int8	i;
	int8	q;
}TxLoDigitalDacInfo_t;

typedef struct 
{
	RficDriverTxLoTpcInfo_t *tpcInfo_p;
	uint8 antMask;
	int8 nextCalibTpcIndex;		//The TPC that should be calibrated next, -1 if done
	uint8 totalNumCalibTpcs;	//Total number of TPCs that should be calibrated in one iteration offline/online
}TxLoCalibRangeDetails_t;

//Online calibration details
typedef struct 
{
	RficDriverTxLoTpcInfo_t tpcInfo;
	uint8 calibAnt;
	int8 nextCalibTpcIndex;			//The TPC that should be calibrated next, -1 if done
	uint8 lastEnabledAntenna;
}TxLoOnlineDetails_t;
#endif

typedef struct TemperatureSensorParams
{
	int32 offset;
	uint16 refTemp;
	uint16 refResult;
	uint16 slope;
} TemperatureSensorParams_t;

//WRX 300
#define NUM_OF_TEMPERATURE_SAMPLES			 0x8
#define TEMPERATURE_SENSOR_MODE				 0x3
#define RSSI_DATA_MODE						 0x2

// WRX 500
#define TEMPERATURE_DEFAULT_REF_TEMP_VAL	 25
#define TEMPERATURE_DEFAULT_REF_RESULT_VAL	 1430
#define TEMPERATURE_SLOPE_VAL				 455  // (1/4.5) << 11
 
#define RFIC_DEFAULT_MASK					 0xFFFF

#define RFIC_DEFAULT_MASK_SHIFT              0x0
#define RFIC_NO_SHIFT                        0x0
#define RFIC_INVALID_REG_VALUE               0xFFFF

#define RXDC_CAL_MODE_NONE	2
#define RXDC_CAL_MODE_CHIP	0
#define RXDC_CAL_MODE_SW	1

#define RFIC_DELAY_BETWEEN_ANTENNA_SET 1000

//for selectBandFilter
#define LOW_BAND  0
#define HIGH_BAND 1
#define BAND_RANGE_CONFIG_SHIFT 15
#define BAND_RANGE_MASK 0x8000
#define ZWDFS_FECTL_MASK 		0x3
#define MAX_NUM_OF_BITS			5


/******************************************************************************/
/***                        Macros                                          ***/
/******************************************************************************/


/******************************************************************************/
/***                        Public Function Declaration                     ***/
/******************************************************************************/

/*****************************************************************************
*    Function Name:  RficDriver_Init
*    Description:    initializes RF
*    Parameters:     None
*    Returns:        None
*    Remarks:        
*    Restrictions:   None
******************************************************************************/
void RficDriver_Init(void);
void rfic_set_mode_lp(uint8 AntMsk, uint8 mode);
uint8 rfic_get_mode_lp(uint8 AntMsk);
void RficDriver_enable_pll_lowpower(void);
void RficDriver_disable_pll_lowpower(void);
void RficDriver_set_rfbandwidthAndMode(uint8 AntMsk, uint8 mode, bool inCOC);
void RficDriver_BandInit(void);

/*****************************************************************************
*    Function Name:  RficDriver_GetRfChipID
*    Description:    Reads the revision code of the RFIC chip.
*    Parameters:     outRevId - Here we will put the returned data
*    Returns:        The revision code according to the specific RFIC chip
*    Remarks:        Implements RficDriver_PublicApi_t->GetRevisionID
*                    
*                    registers.
*                    Bits[10:0]    01  01  01     01  001  (default)
*                                  Vx  y   A/B/C  z1  z2   (0x2A9)
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetRfChipID(uint32 *outRevId);

/*****************************************************************************
*    Function Name:  RficDriver_UpdateShort0ShadowRegister
*    Description:    
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_UpdateShort0ShadowRegister(AntennaBitmaps_e inListOfAntennas);
/*****************************************************************************
*    Function Name:  RficDriver_ActivateRxAntennas
*    Description:    Enables the receive antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateRxAntennas(AntennaBitmaps_e inListOfAntennas);
/*****************************************************************************
*    Function Name:  RficDriver_ActivateTxAntennas
*    Description:    Enables the transmit antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ActivateRxAntennas
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateTxAntennas(AntennaBitmaps_e inListOfAntennas);
/*****************************************************************************
*    Function Name:  RficDriver_DeactivateRxAntennas
*    Description:    Disables the receive antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to deactivate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DeactivateRxAntennas(AntennaBitmaps_e inListOfAntennas);
/*****************************************************************************
*    Function Name:  RficDriver_DeactivateTxAntennas
*    Description:    Disables the transmit antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to deactivate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DeactivateTxAntennas(AntennaBitmaps_e inListOfAntennas);
void RficDriver_SetAntennas(AntennaBitmaps_e rfAntMask, uint8 afeAntMask, uint8 numOfAnts, PowerSaveStates_e powerSaveState);
/***************************************************************************
**
** NAME          RficDriver_EnableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_EnableRFantennas(AntennaBitmaps_e AntMask);
/***************************************************************************
**
** NAME          LmHdk_DisableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : disables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_DisableRFantennas(uint32 rfAntMask, uint8 afeAntMask);

/*****************************************************************************
*    Function Name:  RficDriver_SetStandbyMode
*    Description:    Sets the RFIC registers related to Rx/Tx to standby mode.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*    Returns:        None
*    Remarks:        None
*    Restrictions:   For calibration purposes
******************************************************************************/
void RficDriver_SetStandbyMode(AntennaBitmaps_e inListOfAntennas);
/*****************************************************************************
*    Function Name:  RficDriver_SetLoopback
*    Description:    Configure the loop back for calibrations.
*    Parameters:     inAntenna  - The antenna we want to loop
*                    inLoopType - The type of loop we want to configure in the 
*                                 RFIC (RF, BB, AMDET)
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetLoopback
*    Restrictions:   None
******************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600	
void RficDriver_SetLoopback(uint8 antMask, RficLoopbackTypeIndex_e inLoopType);
#else
void RficDriver_SetLoopback(Antenna_e txAnt, Antenna_e rxAnt, RficLoopbackTypeIndex_e inLoopType);
#endif
/*****************************************************************************
*    Function Name:  RficDriver_GetDigitalLoopBackAnt
*    Description:    Get the Tx and Rx antennas to use in the digital path of the loopback
*    Parameters:     inOutLoopbackAnts - an input/output struct pointer:
*                    inOutLoopbackAnts->RfCalibAnt    - input: The calibrated antenna
*                    inOutLoopbackAnts->RxLoopbackAnt - output: Here we put the Rx loopback antenna to return
*                    inOutLoopbackAnts->TxLoopbackAnt - output: Here we put the Tx loopback antenna to return
*    Returns:        The Tx and Rx loopback antennas to use according to the
*                    calibrated antenna input
*    Remarks:        Implements RficDriver_PublicApi_t->GetDigitalLoopBackAnt
*                    
*                     
*                    (Tx1 -> Rx1,Tx2->Rx2 Tx3 ->Rx3) because we have pin sharing
*                    (Tx\Rx), the digital path used for TX is different from the
*                    digital Rx path
*                    
*                    Calibrated Antenna = 0: Digital Tx0 -> Digital Rx1
*                    Calibrated Antenna = 1: Digital Tx1 -> Digital Rx0
*                    Calibrated Antenna = 2: Digital Tx2 -> Digital Rx1
*                    
*    Restrictions:   None
******************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600	
void RFIC_RestoreFromLoopBack(Antenna_e RfCalibAnt);
#else
void RFIC_RestoreFromLoopBack(uint8 antMask);
void RficDriver_GetDigitalLoopBackAnt(RficLoopbackAntennas_t *inOutLoopbackAnts);
#endif
/*****************************************************************************
*    Function Name:  RficDriver_SetTuningWords
*    Description:    Set RFIC values according to selected channel (frequency).
*    Parameters:     inFreq - The desired LO frequency
*                    inFvco - The calculated VCO frequency
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetTuningWords
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetTuningWords(uint32 inFreq, uint32 inFvco);
/*****************************************************************************
*    Function Name:  RficDriver_CopyRegValuesFromProgModel
*    Description:    Copy default registers values, taken from the ProgModel.
*    Parameters:     inSrcAddress - Starting address of the data.
*                    inLength     - Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyRegValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyRegValuesFromProgModel(uint32 *inSrcAddress, uint32 inLength);
/*****************************************************************************
*    Function Name:  RficDriver_CopyDataBaseValuesFromProgModel
*    Description:    Copy database values from the ProgModel.
*    Parameters:     inMinorID    - Minor ID of the tuning word address.
*                    inSrcAddress - Starting address of the database data to copy.
*                    inLength     -Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyDataBaseValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress, uint32 inLength);
/*****************************************************************************
*    Function Name:  RficDriver_LoadRegDB
*    Description:    Loads all registers from SW DB to RFIC
*    Parameters:     Nane
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->LoadRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_LoadRegDB(void);
/*****************************************************************************
*    Function Name:  RficDriver_StoreRegDB
*    Description:    Stores all RFIC registers to SW DB
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->StoreRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_StoreRegDB(void);
void RficDriver_writeSWCalToRfic(uint8 lnaIndex, int8 inRxDacValue[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX]);

/*****************************************************************************
*    Function Name:  RficDriver_SetRxDACs
*    Description:    Set the PGCs' from Rx path, DAC values 
*    Parameters:     inAnt        - The updated antenna
*                    inPathType   - I, Q or both
*                    inDacIndex   - The required DAC to update
*                    inRxDacValue - Values to set
*                    inLnaIndex    - LNA index
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetRxDACs (AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType, RficDacIndex_e inDacIndex, int8 inRxDacValue[][2], HdkLnaIndex_e inLnaIndex);
/*****************************************************************************
*    Function Name:  RficDriver_GetRxDac0
*    Description:    Retrives the DAC0 look-up table from the RF registers
*    Parameters:     inAntBitmap   - The updated antenna
*                    inLnaIndex    - The LNA index
*                    outRxDacValue - Here we put the DAC0 values we read (I and Q) per requested antenna
*    Returns:        The I and Q DAC0 gain values for the required antenna and LNA gain
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetRxDac0(AntennaBitmaps_e inAntBitmap, HdkLnaIndex_e inLnaIndex, int8 inRxDacValue[][RFIC_PATH_TYPE_I_AND_Q]);

/*****************************************************************************
*	 Function Name:  SetRxGains
*	 Description:	 Set Rx Gains and corresponding DACs
*	 Parameters:	 inAnt		   - The updated antenna
*					 inRficRxGains - The required gains and DACs values
*	 Returns:		 None
*	 Remarks:		 Implements RficDriver_PublicApi_t->SetRxGains
*	 Restrictions:	 None
******************************************************************************/

void SetRxGains(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain);

/*****************************************************************************
*	 Function Name:  SetRxGainsForRssi
*	 Description:	 Set Rx Gains and corresponding DACs
*	 Parameters:	 inAnt		   - The updated antenna
*					 inRficRxGains - The required gains and DACs values
*	 Returns:		 None
*	 Remarks:		 Implements RficDriver_PublicApi_t->SetRxGains
*	 Restrictions:	 None
******************************************************************************/

void SetRxGainsForRssi(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain);


/*****************************************************************************
*    Function Name:  RficDriver_SetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     inAnt         - The updated antenna
*                    inRficRxGains - The required gains and DACs values
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxGains
*    Restrictions:   None
******************************************************************************/
void RficDriver_AbstractSetRxGains(uint8 antMask, uint8 lnaGain, int32 pgc1Gain, int32 pgc2Gain);
/*****************************************************************************
*    Function Name:  RficDriver_AbstractSetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     inAnt         - The updated antenna
*                    inRficRxGains - The required gains and DACs values
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxGains
*    Restrictions:   None
******************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
void RficDriver_DisableLna(uint8 AntMsk);
void RficDriver_EnableLna(uint8 AntMsk);
void RficDriver_SetBandWidth(uint8 antMask);
void RficDriver_SetPgcbwidth (uint8 AntMsk, uint8 rfBandwidth);

void RficDriver_SetRxGains(uint8 antMask, uint8 lnaIndex, int8 pgcGainDb);
#else
void RficDriver_SetRxGains(Antenna_e inAnt,RficRxGains_t *inRficRxgains);
#endif

/*****************************************************************************
*    Function Name:  RficDriver_GetRxGains
*    Description:    Get Rx Gains 
*    Parameters:     inAnt         - The required antenna gains
*    Returns:		 inRficRxGains - The required gains        
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxGains
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetRxGains(Antenna_e inAnt, RficRxGains_t *outRficRxGains);
/*****************************************************************************
*    Function Name:  RficDriver_SetLoDACs
*    Description:    Set the Tx DACs for LO leakage cancellation
*    Parameters:     inAnt        - The updated antenna
*                    inTpcIndex   - The Tx gain index to use
*                    inTxDacValue - The required Tx DAC value
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetLLDACs
*    Restrictions:   inTxDacValues is an array of size RFIC_PATH_TYPE_MAX
******************************************************************************/
#ifdef RFIC_WRX600
void RficDriver_SetLoDACs(uint8 antMask, int8 dacI, int8 dacQ);
void RficDriver_SetLoDACsPerGain(uint8 antMask, int8* dacI, int8* dacQ, uint8 tpcIndex);
uint16 RficDriver_GetRxdcOffset (uint8 Ant, uint8 rfrxgainidx, uint8 rxdcidx);
void RficDriver_SetRxdcOffset (uint8 Ant, uint8 rfrxgainidx, uint8 rxdcidx, uint16 valToWrite);

#else
void RficDriver_SetLoDACs(Antenna_e inAnt, RficTxGainTpcIndex_e inTpcValue, int8 *inTxDacValue);
#endif
/*****************************************************************************
*    Function Name:  RficDriver_GetLoDACs
*    Description:    Get the current Tx DACs values for the required antenna
*    Parameters:     inAnt       - The updated antenna
*                    inTpcIndex  - The Tx gain index to use
*                    outDacValue - Here we put the returned Tx DAC value
*    Returns:        The Tx DAC value
*    Remarks:        Implements RficDriver_PublicApi_t->GetLLDACs
*    Restrictions:   outDacValue is an array of size RFIC_PATH_TYPE_MAX
******************************************************************************/
#ifdef RFIC_WRX600
void RficDriver_GetLoDACs(uint8 antMask, uint8 tpcIndex,int8* dacI, int8* dacQ);
#else
void RficDriver_GetLoDACs(Antenna_e inAnt, RficTxGainTpcIndex_e inTpcValue, int8 *outDacValue);
#endif
/*****************************************************************************
*    Function Name:  RficDriver_ShortRxPgcInputToGround
*    Description:    Short PGC1s input to ground for all antennas
*    Parameters:     onOff - 0=Short is On; 1=Short is Off
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ShortRxPgcInputToGround
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxPgcInputPowerState(RficRxPgcInputPowerState_e state);
/*****************************************************************************
*    Function Name:  RficDriver_SetRxCalGain
*    Description:    Select calibration buffer gain (used only for calibrations)
*    Parameters:     inAnt       - The updated antenna
*                    inRxCalGain - FALSE (0) <-> (-18dB)
*                                  TRUE  (1) <-> (+6dB)
*					 int8 * outCalGainDb_p - the difference in dB
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxCalGain
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxCalGain(Antenna_e inAnt, bool inRxCalGain, int8 *outCalGainDb_p);
/*****************************************************************************
*    Function Name:  RficDriver_SetRxAllPGCsTuneWord
*    Description:    Set the PGC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxAllPGCsTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8* inLpfWord);
/*****************************************************************************
*    Function Name:  RficDriver_SetRxPGC3TuneWord
*    Description:    Set the PGC3 tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxPGC3TuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord);
/*****************************************************************************
*    Function Name:  RficDriver_GetRxPGCTuneWord
*    Description:    Get the PGC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetRxPGCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetRxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType, uint8 *outLpfWord);
/*****************************************************************************
*    Function Name:  RficDriver_SetTxPGCTuneWord
*    Description:    Set the TPC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*				     inIsIqSwap - Are the IQ branches swapped in teh digital
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->settxtpcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetTxPGCTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8* inLpfWord);
/*****************************************************************************
*    Function Name:  RficDriver_GetTxPGCTuneWord
*    Description:    Get the TPC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetTxTPCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetTxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType, uint8 *outLpfWord);
/*****************************************************************************
*    Function Name:  RficDriver_GtTxGain
*    Description:    Gt the TPC index
*    Parameters:     OUT dutRficTxGainsParams_t *dutRficTxGainsParams_p
*    Returns:        None
*    Remarks:         
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetTxGain(dutRficTxGainsParams_t *dutRficTxGainsParams_p);
/*****************************************************************************
*    Function Name:  RficDriver_SetTxGain
*    Description:    Set the TPC index
*    Parameters:     inAnt      - The updated antenna
*                    inTpcIndex - The required index for the TPC value
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetTPCGain
*    Restrictions:   None
******************************************************************************/
#ifdef RFIC_WRX600
void RficDriver_SetTxGain(uint8 antMask, uint8 tpcgainidx, uint8 padGainDb);
void RficDriver_SetTpcGainIndex(uint8 antMask, uint8 tpcgainidx);
#else
void RficDriver_SetTxGain(Antenna_e inAnt, uint8 inTpcIndex);
#endif
void RficDriver_createPgcfiltLut(uint16 sA, int16 sB);
void RficDriver_getPgcFiltLut(uint8 mLut[5][11]);
void RficDriver_AbstractSetTxGain(AntennaBitmaps_e antList, int pgcGain, int pd);

void RficDriver_SetRxPgcGain(uint8 antMask, uint8 pgcGainDb);

/*****************************************************************************
*    Function Name:  RficDriver_TxLoGetTpcInfo
*    Description:    Set the TPC value
*    Parameters:     inAnt          - The required antenna
*                    outTxLoTpcInfo - Here we put the info requested for the required antenna
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->TxLoGetTpcInfo
*    Restrictions:   None
******************************************************************************/
#ifndef ENET_INC_ARCH_WAVE600
void RficDriver_TxLoGetTpcInfo(Antenna_e inAnt, RficDriverTxLoTpcInfo_t *outTxLoTpcInfo);
#endif
/*****************************************************************************
*    Function Name:  RficDriver_PowerUpBiasCalBlock
*    Description:    Power-up the bias central constant current calibration
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->PowerUpBiasCalBlock
*    Restrictions:   None
******************************************************************************/
void RficDriver_PowerUpBiasCalBlock(void);
/*****************************************************************************
*    Function Name:  RficDriver_SetRintValue
*    Description:    Programs the Rint resistor's value
*    Parameters:     inValue - Value to set Rint to
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRintValue
*    Restrictions:   max=31, min=0, default=16
******************************************************************************/
void RficDriver_SetRintValue(uint8 inValue);
/*****************************************************************************
*    Function Name:  RficDriver_ReadBiasCalComparator
*    Description:    Reads and returns the comparator's output
*    Parameters:     outCompResult - Here we put the value read from the register
*    Returns:        The output of the bias calibration comparator's output
*    Remarks:        Implements RficDriver_PublicApi_t->ReadBiasCalComperator
*    Restrictions:   None
******************************************************************************/
void RficDriver_ReadBiasCalComparator(bool *outCompResult);
/*****************************************************************************
*    Function Name:  RficDriver_InitRssiDcOffsetCalib
*    Description:    Initializes the RSSI DC calibration
*    Parameters:     inAntList - The required antennas to calibrate
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->InitRssiDcOffsetCalib
******************************************************************************/
void RficDriver_InitRssiDcOffsetCalib(AntennaBitmaps_e inAntList);
/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetCalibStage
*    Description:    Sets the calibration stage number in the calibration stage
*                    register as well as the path type in the same register for
*                    the selected antenna list (bitmap), path type (I or Q but NOT both)
*                    and stage number (0 through 4)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 1 and 5
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetSetCalibStage
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetCalibStage(AntennaBitmaps_e inAntList, RficPathTypeIndex_e inInputPathType, uint8 inStageNumber);
/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetDacValues
*    Description:    Sets the values of the DACs in selected path, stage and antenna
*                    list (bitmap), according to the array of DAC values (inDacValues)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 0 and 4
*                    inDacValues     - The values to set in the DACs
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetCalibStg
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetDacValues(AntennaBitmaps_e inAntList, RficPathTypeIndex_e inInputPathType, uint8 inStageNumber, uint8 *inDacValues);
/*****************************************************************************
*    Function Name:  RficDriver_StoreRssiDcOffsetCalibValues
*    Description:    Stores all DC-offset values for the specific antenna
*    Parameters:     inAntList      - The antenna list to update
*                    inAllDacValues - The DAC values ordered by RficRssiDacIndexes_e
*    Returns:        None
*    Remarks:        None
*    Restrictions:   inAllDacValues is 10 slots, not-more, no-less
******************************************************************************/
void RficDriver_StoreRssiDcOffsetCalibValues(AntennaBitmaps_e inAntList, uint8 (*inAllDacValues)[RFIC_RSSI_STG_TOTAL]);
/*****************************************************************************
*    Function Name:  RficDriver_SetFrequency
*    Description:    Sets the frequency of the system
*    Parameters:     inFreqSettingsParams - A structure with all the parameters
*                                           needed to set a frequency in the system:
*                         -> rfOscilFreq  - Frequency of the RF oscillator
*                         -> channel    - Channel we are using
*                         -> spectrumMode - CB or nCB
*                         -> upperLower   - When full Spectrum is used, which other
*                                           channel is taken, the upper->1 or lower->0
*                         -> band         - BAND_2_4_GHZ or BAND_5_2_GHZ
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetFrequency
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetFrequency_COC(uint8 AntMsk, bool inCOC);
uint8 RficDriver_GetRfBw(Bandwidth_e fwBw);
RetVal_e RficDriver_SetFrequency(SetChannelParams_t* inSetChannelParams);
/*****************************************************************************
*    Function Name:  RficDriver_SetRxBandLUT
*    Description:    
*    Parameters:     rxBandLUTtemp[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT]
*    Returns:        none
*    Remarks:        
*    Restrictions:   None
******************************************************************************/

void RficDriver_SetRxBandLUT(uint16 *rxBandLUTtemp);
void  RficDriver_SetTxBandLUT(uint16 *txBandLUTtemp);
void RficDriver_readTxbandFromRf(uint8 AntMsk, uint16 *txBand);
void RficDriver_setTxband(uint8 AntMsk, uint8 bandSel);

/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingSet
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function changes the frequency
*                    if the band number and channel number require that.
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingSet
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingSet(void);
/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingReset
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function resets the frequency
*                    back to operational mode after the calibration was done
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingReset
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingReset(void);
/*****************************************************************************
*    Function Name:  RficDriver_CalculateFlo
*    Description:    Returns the frequency of the local oscillator according to
*                    inputs
*    Parameters:     inChannelNum   - Channel we are using
*                    inSpectrumMode - CB or nCB
*                    inUpperLower   - When full Spectrum is used, which other
*                                     channel is taken, the upper->1 or lower->0
*                    inBand         - BAND_2_4_GHZ or BAND_5_2_GHZ
*                    outFlo         - Here we will set the output of the function
*    Returns:        The local oscillator's frequency
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_CalculateFlo(uint8 inChannelNum, SpectrumMode_e inSpectrumMode, uint8 inUpperLower, uint8 inBand, uint32 *outFlo);
uint32 RficDriver_CalcLoFrequency(SetChannelParams_t* inSetChannelParams);

/*****************************************************************************
*    Function Name:  RficDriver_SetXtalValue
*    Description:    Set the Xtal value 
*    Parameters:     uint32 - xtalVal
*    Returns:        void
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetXtalValue(uint32 xtalVal);

/*****************************************************************************
*    Function Name:  RficDriver_XtalInitPreventAmpChange
*    Description:    To avoid clock amplitude change during start, (which causes DSL problems),
*                    we need to add the following writes to the init part.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_XtalInitPreventAmpChange(void);
//************************************
// Method:    RficDriver_GetXtalValue
// Description:
// Parameter: uint32 * xtalVal
// Returns:   void
// Remarks:   None
//************************************
void RficDriver_GetXtalValue(uint32 *xtalVal);
/*****************************************************************************
*    Function Name:  RficDriver_GetAmDetBranch
*    Description:    This function returns the AM detector branch according to the frequency band given
*    Parameters:     inFreqBand     - The required frequency band (BAND_2_4_GHZ or BAND_5_2_GHZ only)
*                    outAmDetBranch - Here we will put the returned value
*    Returns:        None
*    Remarks:        This function ignores IQ swap if exists. User must check IQ swap with PHY driver
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetAmDetBranch(uint8 inFreqBand, RficPathTypeIndex_e *outAmDetBranch);
/*********************************************************************************
Method:			RficDriver_WriteRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteRegister(IN uint32 regNum,IN uint16 *regVal);
/*********************************************************************************
Method:			RficDriver_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	OUT uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_ReadRegister(IN uint32 regNum,OUT uint16 *regVal);
/*********************************************************************************
Method: 		RficDriver_WriteMaskRegister
Description:  
Parameter:		IN uint32 regNum
Parameter:		IN uint32 regVal
Parameter:		IN uint32 regMask
Returns:		void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteMaskRegister(IN uint32 regNum,IN uint16 *regVal, IN uint32 regMask);

/*****************************************************************************
*    Function Name:  RficDriver_rffe_CopyDataBaseValuesFromProgModel.
*    Description:    Save the values for rffe mode.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_rffe_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress,uint32 inLength);
/*****************************************************************************
*    Function Name:  RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
*    Description:    set rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/

void RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant);
void RficDriver_disableRffeManMode(uint8 antMask);
void RficDriver_setAntennaFectlMan(uint8 AntMsk, uint8 fectl);
void RficDriver_enableBbloop(uint8 AntMsk, uint8 HbLb, uint8 LoopType, uint8 IqSel);
void RficDriver_disableBbloop(uint8 AntMsk, uint8 HbLb);

#ifndef ENET_INC_ARCH_WAVE600

/*****************************************************************************
*    Function Name:  getMinimumIndex()
*    Description:    Txlo ar10/WRx300 Linear search calculates the minimal index backoff according to the "iterationNumber"
*    Parameters:     
*					 *minIndex 		 - 	pointer to update I cntrl
 new min cntrl 
*					 *minIndexQ      - 	pointer to update Q cntrl
 new min cntrl
*					 iterationNumber - 	Iteration number
*					 iqIndexI 		 - 	Current I cntrl minimal index
*					 iqIndexQ 		 - 	Current Q cntrl minimal index
*					 toneAmplitude 	 - 	ToneGen amlitude
*					 inCalibType 	 - 	Offline/Online
*					 RfCalibAnt 	 - 	Calibrated Antenna.

*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void getMinimumIndex(int8 *minIndex,int8* minIndexQ,int8 iterationNumber,int8 iqIndexI,int8 iqIndexQ,int8 toneAmplitude,ClbrType_e inCalibType,Antenna_e RfCalibAnt);

/*****************************************************************************
*    Function Name:  getNumberOfRuns()
*    Description:    Txlo ar10/WRx300 Linear search calculates Number of iterations for on the range of cntrls that was calculated 
					 [getMinimumIndex() : Max cntrl]
*    Parameters:     
*					*numberOfRuns
*					runsIncrement
*					iterationNumber

*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/

void getNumberOfRuns(uint32 *numberOfRuns,uint32* runsIncrement,int8 iterationNumber);
#endif


/*****************************************************************************
*    Function Name:  RficDriver_getRffeMode
*    Description:    get rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
uint8 RficDriver_getRffeMode(CalRFFEState_e mode,Antenna_e ant);

void SetRffeMode(uint8 inRxAntMask);

/*****************************************************************************
*    Function Name:  RficDriver_EnableLnaNoise(uint8 AntMsk);
*    Description:    set lna noise
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/

void RficDriver_EnableLnaNoise(uint8 AntMsk);


/*****************************************************************************
*    Function Name:  RficDriver_DisableLnaNoise(uint8 AntMsk);
*    Description:    Disable lna noise
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/

void RficDriver_DisableLnaNoise(uint8 AntMsk);

/********************************************************************************
Function Name:  RficCalDrv_RestoreOperational:

Description:  Restores the RFIC to operational state:
			1) Rfic rx off.
			2) disable LnaNoise.
********************************************************************************/

void RficCalDrv_RestoreOperational(uint8 rxAntMask);

/*****************************************************************************
*    Function Name:  writeOneValueToAntList
*    Description:    Writes inValue into inRfRegAddr on all antennas, using inMask
*                    as the write mask.
*    Parameters:     inRfRegAddr - The RF register address we write to.
*                    inValue     - The value we want to write into the registers.
*                    inMask      - A mask of where in the register we write
*                                  inValue into.
*                    inUpdateDB  - TRUE <->Write data to local DB as well
*                                  FALSE<->Don't write to local DB
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void writeToRficRegister(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint16 *inValues, uint16 inMask, uint16 inShift, bool inUpdateDB);
void writeOneValueToAntList(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint32 inValue, uint16 inMask, uint16 inShift, bool inUpdateDB);

#ifndef ENET_INC_ARCH_WAVE600

/********************************************************************************
*    Function Name:  doAxisIteration:
*	 Description:    Find equalizer parameter (DAC value) of lowest Lo energy 
*
*	 Parameters:	1. inLo - 			minimal index
					2. inHi -			maximal index
					3. inOtherAxisVal - index of other axis (I for Q calibration and vice versa)
					4. inIsQ - 			defines whether Q or I is calibrated
					5. *inAntInfo_p -	pointer to antenna RX loopback / TX calibrated antenna
					6. inTpcIndex - 	calibrated TPC index
					7. *outElements_p - pointer to Txlo Param/result struct.
					8. *outIndex_p - 	returnd cntrl output struct.
					9. isInternalPA - 	1-ext PA, 0 -int PA
					10.IterationNumber - wrx300 - LinearSearch pass "repeat" from calibrateAndVerifySingleTpc() 
										 wrx500/514 - Binary search pass outElements_p->params.binaryMaxSteps
					11.inCalibType-		Offline/Online
	
*	 Returns:        None
*    Remarks:        None
*    Restrictions:   None
	
********************************************************************************/

void doAxisIteration(int8 inLo, int8 inHi, int8 inOtherAxisVal, bool inIsQ, RficLoopbackAntennas_t *inAntInfo_p, uint8 inTpcIndex, TxLO_elements_t* outElements_p, int8 *outIndex_p,uint32 isInternalPA,int8 IterationNumber,ClbrType_e inCalibType);

/********************************************************************************
 *    Function Name: updateMinimumEnergy:	
 *	  Description: Checks if measured energy is minimum in the binary search and update accordingly 
 *	  Parameters:
			inValue	- I/Q value for which inEnergy was measured
			inEnergy - currently measured energy
*			*outMinEnergy_p - pointer occupies minimum energy measured .
*			*outMinIndex_p - pointer occupies the cntrl index with minimum energy.

*	 Returns:        None
*    Remarks:        None
*    Restrictions:   None

********************************************************************************/
void updateMinimumEnergy(int8 inValue, int32 inEnergy, int32 *outMinEnergy_p, int8 *outMinIndex_p);
#endif

/********************************************************************************
 *    Function Name: measureEnergy()
 * 	  Description: Measure Lo energy level of (a+jb) equalizer (using goertzel).
 *	  Parameters:
				1. inTpcIndex - 	TPC calibrated index
				2. I value	  -		Txlo cntrl index for I path
				3. Q value	  - 	Txlo cntrl index for Q path
				4. inBin	  - 	Bin to be measured by Gortzel block 
				5. *inAntInfo_p - 	TX/RX calibrated/loopback antenna struct pointer
				6. * outElements_p -TxLO_elements_t parameters
				7. *outEnergy_p  -  pointer gets the total measured energy (I^2+Q^2) 

*	 Returns:        None
*    Remarks:        None
*    Restrictions:   None

********************************************************************************/

void prepareDac0Values(AntennaBitmaps_e inAntBitmap,RficPathTypeIndex_e inPathType, int8 inRxDacValue[][2], uint32 *outDacValues);

void RficDriver_RxDcInitAntAll(void);

void RficDriver_RxDcInitAnt(uint8 ant);
void RficDriver_RxDcInit(uint8 antMask, uint8 intBw);
bool RficDriver_RunRxDcCal(uint8 antMask, uint8 lnaGainIndex, uint16 analogTimeout);
void RficDriver_RxDcInitBW(uint8 bw);
bool RficDriver_readDigitalCorrectionfromPhy(void);

void RficDriver_RxDcActivateDac2ChipCalAll(int32 Pgc2Gain, uint8 intBw, uint8 numTries, int32 failTH, uint32 corrSamples);
void RficDriver_RxDcActivateDac0ChipCalAll(RficRxGains_t* rxGains, uint8 intBw,int32 failTH, uint8 numTries, uint32 corrSamples);
void RficDriver_RxDcActivateDac2ChipCal(uint8 ant, int32 Pgc2Gain, uint8 intBw, uint8 numTries, int32 failTH, uint32 corrSamples);
void RficDriver_RxDcActivateDac0ChipCal(uint8 ant, RficRxGains_t* rxGains, uint8 intBw,int32 failTH, uint8 numTries, uint32 corrSamples);
void RficDriver_RxDcCalibrateDAC2(uint8 gainIndex);
#ifdef RFIC_WRX600
void RficDriver_RxDcStopDcCal(uint8 antMask);
#else
void RficDriver_RxDcStopDcCal(void);
#endif
void RficDriver_SetDCcalType(uint8 calType);
bool RficDriver_SetDCspecialGainTo1stTable(uint8 index, uint8 lnaIndex, uint8 pgc1index, uint8 pgc2index);
bool RficDriver_SetDCspecialGainTo2ndTable(uint8 lnaIndex, uint8 pgc2index, uint8 pgc1start, uint8 pgc1end);
bool RficDriver_GetDCspecialGainFrom1stTable(uint8 index, uint8 *lnaIndex, uint8 *pgc1index, uint8 *pgc2index);
bool RficDriver_GetDCspecialGainFrom2ndTable(uint8 *lnaIndex, uint8 *pgc2index, uint8* pgc1start, uint8* pgc1end);
void RficDriver_DCwriteSpecialGain(CorrRes_t *results, uint8 gainIndex, uint8 bw);

void RficDriver_SetLnaGainLUT(void);
void RficDriver_SetLnaConversionsTable(Antenna_e ant, uint8 index, uint8 val);
void RficDriver_OverrideLnaConversionsTable(Antenna_e ant, uint8 val);
void RficDriver_RestoreLnaConversionsTable(Antenna_e ant);
uint8 RficDriver_GetLnaGain(Antenna_e ant, uint8 index);
void RficDriver_GetSetRxBandAll(void);
uint16 RficDriver_GetTxBand(void);
uint16 RficDriver_GetRxBand(uint8 ant);
void RficDriver_SetRxBand(uint16 rxBand);
void RficDriver_SetTxBand(uint16 txBand);


void RficDriver_SetCalibrationData(K_MSG* psMsg);
void RficDriver_resetPsdArray(void);
void RficDriver_resetZwdfsPsdArray(void);
void RficDriver_IREInit(void);
void RficDriver_ReadTemperature(int32* temperature);
void RficDriver_SelectBandFilterZwdfs(int8 setAPHighLowFilterBank, uint8 driver_fw, uint8 low_chan_num);
void RficDriver_SelectBandFilterB(int8 setAPHighLowFilterBank, uint8 driver_fw, uint8 low_chan_num);
void RficDriver_handleIRETable_B(void);
void RficDriver_handleZwdfsTable(void);
void RficDriver_handleIRETable_A(void);
bool RficDriver_isSsbLoop(void);
uint32 checkPsdTableType(void);
bool RficDriver_writeDigitalCorrectionToPhy(uint32 dcCancellationword);
bool RficDriver_writeDcModeToPhy(uint32 dcMode);
int RficDriver_readDcMode(void);
uint8 readDcModeFromPhy(void);
void RficDriver_InitBwChanDC(void);
void RficDriver_AssociateAntennasToMyBand(uint8 antMask);
void RficDriver_dumpRegs(void);
void RficDriver_copyOneRowFromPsdTable(uint32 regNum, uint32 *regVal, uint32 regMask,uint16 rowIndex, RfregPsdDb_e dbTag);
void RficDriver_writePsdTableFromDB(void);

#ifdef HDK_RF_EMULATOR
void RficDriver_SetRfEmuBand(uint8 band, uint8 antMask);
#endif

/******************************************************************************/
/***                        external Public Variables                       ***/
/******************************************************************************/

/******************************************************************************/
/***                        Statistics & Debug                              ***/
/******************************************************************************/

#endif//__RFIC_DRIVER_API_H
