/*****************************************************************************
*   COPYRIGHT:
*       (C) Copyright 2010 of Lantiq
*       All rights are strictly reserved. Reproduction or divulging in any
*       form whatsoever is not permitted without written authority from the
*       copyright owner. Issued by Lantiq
*   MODULE NAME:
*       RficDriver.c
*   DESCRIPTION:
*       Driver for the RFIC of AR10 platform
*   AUTHOR:
*       Ariel Groenteman
*   DATE:
*       July 2011
*****************************************************************************/

/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "mt_sysrst.h"
#include "stringLibApi.h"
#include "HdkGlobalDefs.h"
#include "RficDriver_API.h"
#include "Indirect_API.h"
#include "lib_abb_api.h"
#include "lib_rf_4x4_api.h"
#include "lib_rf_4x4_sequences.h"
#include "interface_rf_4x4.h"
#include "rf_antenna0_fcsi_regpkg.h"
#include "RficLoopBackTypes.h"
#include "RFICDefines.h"
#include "RficCommon.h"
#include "ErrorHandler_Api.h"
#include "rf_antenna0_fcsi_regpkg.h"
#include "PhyDriver_API.h"
#include "Afe_API.h"
#include "PhyRxAgcPage1Regs.h"
#include "PhyRxTdRegs.h"
#include "PSD.h"
#include "CalibrationHandler.h"
#include "CoC_Api.h"
#include "MT_Math.h"
#include "RxDcOffsetClbrHndlr.h"
#include "loggerAPI.h"
#include "fast_mem_psd2mips.h"
#include "Dut_Api.h"
#include "lib_rf_4x4_api.h"

#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE_1
#define LOG_LOCAL_FID 2

/******************************************************************************/
/***						  Constants										***/
/******************************************************************************/
/* Tuning words LO Channel Band Select */
#define RFIC_TUNE_WORD_LO_CHAN_HIGH_BAND_VCO_LOW  (0)
#define RFIC_TUNE_WORD_LO_CHAN_HIGH_BAND_VCO_HIGH (1)

/* Rx PGC x Power-up/down */
#define RFIC_RX_PGC_POWER_UP_VALUE       1
#define RFIC_RX_PGC_POWER_DOWN_VALUE     0
/* Rx PGC 1 */
#define RFIC_RX_PGC1_GAIN_MAX_VALUE      (24) /* dB */
#define RFIC_RX_PGC1_GAIN_MIN_VALUE      (-6) /* dB */
/* Rx PGC 2 */
#define RFIC_RX_PGC2_GAIN_MAX_VALUE      (24) /* dB */
#define RFIC_RX_PGC2_GAIN_MIN_VALUE      (-6) /* dB */
/* Rx PGC3*/
#define RFIC_RX_PGC3_GAIN_MAX_VALUE      (12) /* dB */
#define RFIC_RX_PGC3_GAIN_MIN_VALUE      (-3) /* dB */
/* Reference frequency of the VCO */
#define RFIC_VCO_REF_FREQUENCY           36 /* Frequency (in GHz) generated with the crystal in the RF */

#define RFIC_NUM_OF_SHORT_REGS           4
#define RFIC_NUM_OF_LONG_REGS            64
#define RFIC_NUM_OF_REGS                 (RFIC_NUM_OF_SHORT_REGS + RFIC_NUM_OF_LONG_REGS)
#define RFIC_FIRST_SHORT_REG_ADDR        (REG_RFIC_S0_SHORT_0)
#define RFIC_LAST_SHORT_REG_ADDR         (REG_RFIC_S3_SHORT_3)
#define RFIC_FIRST_LONG_REG_ADDR         (REG_RFIC_L0_RSSI_REG1)
#define RFIC_LAST_LONG_REG_ADDR          (REG_RFIC_L63_RF_PLL_8)

#define RFIC_NUM_OF_DAC0_REGS            (HDK_LNA_INDEX_MAX_INDEX)

#define RFIC_RESET_VALUE                 1

#define RFIC_STANDBY_MODE_VALUE          0
#define RFIC_STANDBY_MODE_MASK           (0xFFFF)

#define REG_RFIC_L49_DEFAULT_VALUE		 (0xC000)

/* Power-up Rx Path values, masks and shifts */
/* Power Up PGC3 */
#define RFIC_POWER_UP_PGC3_VALUE         (0x01)
#define RFIC_POWER_UP_PGC3_MASK          (REG_RFIC_L32_EN_PGC3_MASK)
#define RFIC_POWER_UP_PGC3_SHIFT         (REG_RFIC_L32_EN_PGC3_SHIFT)
/* Power Up PGC1/2 */
#define RFIC_POWER_UP_PGC1_AND_PGC2_VALUE (0x7F)
#define RFIC_POWER_UP_PGC1_AND_PGC2_MASK  (REG_RFIC_L10_EN_OC1_PGC2_MASK|REG_RFIC_L10_EN_OP2_PGC2_MASK|REG_RFIC_L10_EN_OP1_PGC2_MASK|REG_RFIC_L10_EN_OC_PGC1_MASK|REG_RFIC_L10_EN_OFFC_BIAS_CUR_MASK|REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_MASK|REG_RFIC_L10_EN_PGC1_MASK)
#define RFIC_POWER_UP_PGC1_AND_PGC2_SHIFT (REG_RFIC_L10_EN_OC1_PGC2_SHIFT) /* Minimal shift in register L10 for powering up PGC1 and 2 */
/* Power Up Down-Mixer */
#define RFIC_POWER_UP_MIXER_VALUE        (0x0F)
#define RFIC_POWER_UP_MIXER_MASK         (REG_RFIC_L8_EN_MIXQ_MASK|REG_RFIC_L8_EN_MIXI_MASK|REG_RFIC_L8_WU_MIXER_MASK|REG_RFIC_L8_EN_MIXER_MASK)
#define RFIC_POWER_UP_MIXER_SHIFT        (REG_RFIC_L8_EN_MIXQ_SHIFT) /* Minimal shift in register L8 for powering up Rx mixer */
/* Power Up LNA */
#define RFIC_POWER_UP_LNA_VALUE          (0x05)
#define RFIC_POWER_UP_LNA_MASK           (REG_RFIC_L6_WU_LNA5G_MASK|REG_RFIC_L6_EN_LNA_MASK)
#define RFIC_POWER_UP_LNA_SHIFT          (REG_RFIC_L6_WU_LNA5G_SHIFT) /* Minimal shift in register L6 for powering up the LNA */
/* Power Up RSSI */
#define RFIC_POWER_UP_RSSI_VALUE         (0x01)
#define RFIC_POWER_UP_RSSI_MASK          (REG_RFIC_L3_EN_RSSI_MASK)
#define RFIC_POWER_UP_RSSI_SHIFT         (REG_RFIC_L3_EN_RSSI_SHIFT)
/* Power Up channel LO */
#define RFIC_POWER_UP_CHANNEL_LO_VALUE   (0x01)
#define RFIC_POWER_UP_CHANNEL_LO_MASK    (REG_RFIC_L14_EN_MASK)
#define RFIC_POWER_UP_CHANNEL_LO_SHIFT   (REG_RFIC_L14_EN_SHIFT)

/* RX and TX enable/disable values */
#define RFIC_RX_ENABLE_VALUE             1
#define RFIC_RX_DISABLE_VALUE            0
#define RFIC_TX_ENABLE_VALUE             1
#define RFIC_TX_DISABLE_VALUE            0

/* Set Rx-on bit and Tx-on bit to 0 */
#define RFIC_RX_OFF_AND_TX_OFF_VALUE     ((RFIC_RX_DISABLE_VALUE<<REG_RFIC_S0_RXON_SHIFT)|(RFIC_TX_DISABLE_VALUE<<REG_RFIC_S0_TXON_SHIFT))
#define RFIC_RX_OFF_AND_TX_OFF_MASK      (REG_RFIC_S0_RXON_MASK|REG_RFIC_S0_TXON_MASK)

/* Set Rx on bit to 1 and Tx-on bit to 0 */
#define RFIC_RX_ON_AND_TX_OFF_VALUE     ((RFIC_RX_ENABLE_VALUE<<REG_RFIC_S0_RXON_SHIFT)|(RFIC_TX_DISABLE_VALUE<<REG_RFIC_S0_TXON_SHIFT))
#define RFIC_RX_ON_AND_TX_OFF_MASK      (REG_RFIC_S0_RXON_MASK|REG_RFIC_S0_TXON_MASK)

/* Power-down PGC1 */
#define RFIC_POWER_DOWN_PGC1_VALUE       ((RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_PGC1_SHIFT)|(RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OC_PGC1_SHIFT))
#define RFIC_POWER_DOWN_PGC1_MASK        (REG_RFIC_L10_EN_PGC1_MASK|REG_RFIC_L10_EN_OC_PGC1_MASK)

/* Power-up PGC1 */
#define RFIC_POWER_UP_PGC1_VALUE         ((RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_PGC1_SHIFT)| \
										  (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OFFC_BIAS_CUR_SHIFT)| \
										  (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OC_PGC1_SHIFT))
#define RFIC_POWER_UP_PGC1_MASK          (REG_RFIC_L10_EN_PGC1_MASK|REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_MASK| \
                                          REG_RFIC_L10_EN_OFFC_BIAS_CUR_MASK|REG_RFIC_L10_EN_OC_PGC1_MASK)

/* Power-up PGC2 */
#define RFIC_POWER_UP_PGC2_VALUE         ((RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OFFC_BIAS_CUR_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OP1_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OP2_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OC1_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_UP_VALUE<<REG_RFIC_L10_EN_OC2_PGC2_SHIFT))
#define RFIC_POWER_UP_PGC2_MASK          (REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_MASK|REG_RFIC_L10_EN_OFFC_BIAS_CUR_MASK|\
                                          REG_RFIC_L10_EN_OP1_PGC2_MASK|REG_RFIC_L10_EN_OP2_PGC2_MASK|\
                                          REG_RFIC_L10_EN_OC1_PGC2_MASK|REG_RFIC_L10_EN_OC2_PGC2_MASK)

/* Power-down PGC2 */
#define RFIC_POWER_DOWN_PGC2_VALUE       ((RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OPAMPS_BIAS_CUR_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OFFC_BIAS_CUR_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OP1_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OP2_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OC1_PGC2_SHIFT)| \
                                          (RFIC_RX_PGC_POWER_DOWN_VALUE<<REG_RFIC_L10_EN_OC2_PGC2_SHIFT))
#define RFIC_POWER_DOWN_PGC2_MASK        (RFIC_POWER_UP_PGC2_MASK)


#define RFIC_CENT_BIAS_COMP_OUT_VAL_PASS 1

/* RSSI DC offset calibration step number min/Max values */
#define RFIC_RSSI_DC_CAL_MIN_STEP_NUM    0 /* regular operation */
#define RFIC_RSSI_DC_CAL_MAX_STEP_NUM    5 /* cal step 5 */

#define RFIC_RSSI_DC_CAL_INIT_S0_VALUE   (0x0002) /* Set antennas to standby mode */
#define RFIC_RSSI_DC_CAL_INIT_S0_MASK    (RFIC_DEFAULT_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L2_VALUE   (0x0040) /* Short RSSI input to GND */
#define RFIC_RSSI_DC_CAL_INIT_L2_MASK    (REG_RFIC_L2_SHORT_INPUT_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L4_VALUE   (0x0010) /* Set ANA-MUX to RSSI and to step 0 (0..4) */
#define RFIC_RSSI_DC_CAL_INIT_L4_MASK    (REG_RFIC_L4_SEL_IN_MASK|\
                                          REG_RFIC_L4_SEL_RSSI_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L5_VALUE   (0xD100) /* ANA-MUX setting for RSSI DC calibration */
#define RFIC_RSSI_DC_CAL_INIT_L5_MASK    (REG_RFIC_L5_EN_ANAMUX_MASK|\
                                          REG_RFIC_L5_SEL_OFFS_MASK|\
                                          REG_RFIC_L5_SEL_RB_MASK|\
                                          REG_RFIC_L5_SEL_FB_MASK)

#define RFIC_RSSI_DC_CAL_STEP_L2_MASK    (REG_RFIC_L2_CAL_STEP_MASK|\
                                          REG_RFIC_L2_CAL_Q_MASK|\
                                          REG_RFIC_L2_CAL_I_MASK)

/* RSSI DC offset calibration Store all DAC values - register L2 mask */
#define RFIC_RSSI_DC_CAL_STORE_ALL_REG_L2_MASK (REG_RFIC_L2_CAL_STEP_MASK|REG_RFIC_L2_SHORT_INPUT_MASK| \
                                                REG_RFIC_L2_CAL_Q_MASK|REG_RFIC_L2_CAL_I_SHIFT| \
                                                REG_RFIC_L2_Q_RSSI_CAL5_MASK|REG_RFIC_L2_I_RSSI_CAL5_MASK)

/* VCO and PLL constants */
#define RFIC_VCO_PLL_LOCKBIT_LOCKED      1
#define RFIC_VCO_PLL_LOCKBIT_UNLOCKED    0
#define RFIC_VCO_PLL_LOCK_MAX_RETRIES    3

#define RFIC_FREQ_12GHZ_VCO_MIN          10400 /* Min VCO frequency to use 12GHz divider */

/* VCO main divider integer bins bounds */
/* Numbers are according to MatLab file: setRFPLLFrequency.m */
/* All values are shifted by 21 bits left to make frequency setting work with
 * 32 bit integers and not with fixed-point fractions */
#define RFIC_VCO_MAIN_DIV_BIN_BOUND_65   (65 << 21)
#define RFIC_VCO_MAIN_DIV_BIN_BOUND_71   (71 << 21)
#define RFIC_VCO_MAIN_DIV_BIN_BOUND_77   (77 << 21)
#define RFIC_VCO_MAIN_DIV_BIN_BOUND_83   (83 << 21)

/* VCO main divider integer bins values */
/* Numbers are according to MatLab file: setRFPLLFrequency.m */
/* All values are shifted by 21 bits left to make frequency setting work with
 * 32 bit integers and not with fixed-point fractions */
#define RFIC_VCO_MAIN_DIV_BIN_VALUE_64   (64 << 21)
#define RFIC_VCO_MAIN_DIV_BIN_VALUE_70   (70 << 21)
#define RFIC_VCO_MAIN_DIV_BIN_VALUE_76   (76 << 21)
#define RFIC_MMD_VAL_FOR_BIN_VALUE_64    (30)
#define RFIC_MMD_VAL_FOR_BIN_VALUE_70    (33)
#define RFIC_MMD_VAL_FOR_BIN_VALUE_OTHER (36)

#define RFIC_VCO_HIGH_BAND_RANGE_SELECT  (10000)
#define RFIC_VCO_HIGH_BAND_VCO_SEL_VALUE (1)
#define RFIC_VCO_LOW_BAND_VCO_SEL_VALUE  (0)

#define RFIC_RF_PLL_RESET_DO             (0)
#define RFIC_RF_PLL_RESET_UNDO           (1)

/* Tx LO TPC info constants */
#define RFIC_TPC_INFO_ARRAY_SIZE         (uint8)(3)
#define RFIC_TPC_INFO_NUM_CALIB_TPCS     (uint8)(3)
#define RFIC_TPC_INFO_CALIB_MASK         (byte)(0xFF)

/* Tx LO registers constants */
#define RFIC_TX_LO_TPC_DC_OFF_GAIN_MASK  (REG_RFIC_L16_TPC_DCOFF_I_GAIN0_MASK | REG_RFIC_L16_TPC_DCOFF_Q_GAIN0_MASK)
#define RFIC_TX_LO_TPC_DC_OFF_GAIN_SHIFT (REG_RFIC_L16_TPC_DCOFF_Q_GAIN0_SHIFT) /* Q shift is 0 < I shift (== 7)*/

/* DAC0 gains constants */
#define RFIC_DAC0_SIGN_MASK              (0x0020)
#define RFIC_DAC0_SIGN_EXTAND            (0x00C0)

/* Rffe constants */
#define TR_SWITCH_MASK		             (0x00C0)
#define PA_ON_MASK  		             (0x000C)
#define S0_PA_AND_TRS_MASK               (0x1B00)
#define TWO_LSB_BITS                     (0X3)

#define TR_SWITCH_SHIFT		             (6)
#define PA_ON_SHIFT  		             (2)
#define TR_SWITCH_INTO_S0_SHIFT          (8)
#define PA_ON_INTO_S0_SHIFT				 (11)

/* XTAL (DCXO) amplitude change prevention constants */
#define RFIC_SET_XTAL_L50_VALUE          (0x0008) /* Set DCXO bias mode and rest to default, enable FSYS output clock */
#define RFIC_SET_XTAL_L51_ONLY_BG_VALUE  (0x0001) /* Enable Bandgap */
#define RFIC_SET_XTAL_L51_FULL_VALUE     (0x1841) /* Enable Bandgap, Enable Central bias V2I const and default Bgp and temp trimming, default iconst */
#define RFIC_SET_XTAL_L52_VALUE          (0x8617) /* Enable Central bias V2I ref, PLL reset, soft reset released, Set DCXO LDO to default */
#define RFIC_SET_XTAL_L53_VALUE          (0x1948) /* Set FCSI LDO to default, DCXO current control default, FSYS strength default */

#define RF_INIT_DELAY_1		(50000)
#define RF_INIT_DELAY_2		(10)
#define RF_INIT_DELAY_3		(10)
#define RF_INIT_DELAY_4		(50)


#define RFE_CTRL_BASE_ADDRESS (0x20dc0)
#define RFIC_RFFE_SHIFT_NO_LNA (2)
#define RFIC_RFFE_MASK (0x1f << RFIC_RFFE_SHIFT_NO_LNA)


#define RXDC_DAC1_WIDTH	6
#define RXDC_DAC1_MASK	((1 << RXDC_DAC1_WIDTH) - 1)
#define RXDC_DAC2_WIDTH	5
#define RXDC_DAC2_MASK	((1 << RXDC_DAC2_WIDTH) - 1)

#define RF_CHANNEL_WIDTH_MHZ	5
#define RF_SPECTRUM_20_SHIFT	0
#define RF_SPECTRUM_40_SHIFT	10
#define RF_SPECTRUM_80_SHIFT	30

#define RXDC_SHORT0_RX_BAND_SHIFT	0
#define RXDC_SHORT0_TX_BAND_SHIFT	3
#define RXDC_SHORT0_BW_SHIFT		8
#define RXDC_SHORT0_SEL_CH_SHIFT	10
#define RXDC_SHORT0_SEL_RXDC_SHIFT	13
#define RXDC_SHORT0_RXON_SHIFT		15
#define RXDC_SHORT0_DEFAULT_VAL		(0x8000)

#define RXDC_SPECIAL_GAIN_TABLE1_LNA_MASK	(0x7)
#define RXDC_SPECIAL_GAIN_TABLE1_LNA_SHIFT	(0)
#define RXDC_SPECIAL_GAIN_TABLE1_PGC1_MASK	(0xf8)
#define RXDC_SPECIAL_GAIN_TABLE1_PGC1_SHIFT	(3)
#define RXDC_SPECIAL_GAIN_TABLE1_PGC2_MASK	(0xf00)
#define RXDC_SPECIAL_GAIN_TABLE1_PGC2_SHIFT	(8)
#define RXDC_SPECIAL_GAIN_RESULT_MASK		(0x7FF)
#define RXDC_SPECIAL_GAIN_RESULT_SHIFT		(11)

#define LNA_GAIN01_CONVERSION_MASK		(RF_ANTENNA0_FCSI_FECTL01_LNA_GAIN0__SMSK | RF_ANTENNA0_FCSI_FECTL01_FECTL_GAIN0__SMSK | RF_ANTENNA0_FCSI_FECTL01_LNA_GAIN1__SMSK | RF_ANTENNA0_FCSI_FECTL01_FECTL_GAIN1__SMSK)
#define LNA_GAIN0_CONVERSION_SHIFT		(0)
#define LNA_GAIN1_CONVERSION_SHIFT		(8)

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/ 
/* The structure of the word we send the RFIC (using the Indirect API) */
typedef struct RficProgModel
{
    uint16 Data;
    uint16 Register;
} RficProgModel_t;

typedef struct
{
    uint32 freq:16;
    uint32 word:16;
} RficFreqTuneWord_T;

/******************************************************************************/
/***					Internal Data Structures							***/
/******************************************************************************/

/******************************************************************************/
/***						Public Variables								***/
/******************************************************************************/



/******************************************************************************/
/***						Static Variables								***/
/******************************************************************************/

uint8 bandwidthOffset[MAX_NUM_OF_BW] = {RF_SPECTRUM_20_SHIFT, RF_SPECTRUM_40_SHIFT, RF_SPECTRUM_80_SHIFT};
uint8 LNA_LUT[MAX_NUM_OF_ANTENNAS][8];

/*The internal registers DB - mirror of the RF registers. */ 
static RficRfRegistersDB_t RficDriverRegistersLocalDB[RFIC_NUM_OF_REGS];
static RficPSDRegistersDB_t RficDriverPSD[RFIC_NUM_OF_REGS];

/* Tuning words per frequency - Some of the components in the RF are set to
** different values depended on the channel(frequency) that was chosen
** these values are taken from the progmodel and  stored in following arrays.
** When we set the channel we use the relevant values for the chosen frequency */
static RficFreqTuneWord_t lnaBandSelect5GTuneWord       = {{{5100,0},{5300,1},{5500,2},{5600,3},{5800,4},{6100,5},{6400,6},{RFIC_INVALID_TUNE_WORD_ENTRIE,7},{RFIC_INVALID_TUNE_WORD_ENTRIE,7},{RFIC_INVALID_TUNE_WORD_ENTRIE,7}}, REG_RFIC_L6_BAND_5G, REG_RFIC_L6_BAND_5G_MASK, REG_RFIC_L6_BAND_5G_SHIFT};
static RficFreqTuneWord_t txMixerBandSelect5GTuneWord   = {{{4900,0},{5200,3},{5400,2},{5600,1},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0}}, REG_RFIC_L28_SWC1_MOD5G, (REG_RFIC_L28_SWC1_MOD5G_MASK|REG_RFIC_L28_SWC2_MOD5G_MASK), REG_RFIC_L28_SWC1_MOD5G_SHIFT};
static RficFreqTuneWord_t predriverBandSelect5GTuneWord = {{{4900,0},{5100,3},{5300,2},{5500,1},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0},{RFIC_INVALID_TUNE_WORD_ENTRIE,0}}, REG_RFIC_L27_PAD5G_BW, REG_RFIC_L27_PAD5G_BW_MASK, REG_RFIC_L27_PAD5G_BW_SHIFT};

static const uint32 rficTxGainValues[RFIC_TX_GAIN_TPC_INDEX_MAX + 1] =
{
    0x8000, /*  TPC gain (-6) */
    0x8001, /*  TPC gain (-2) */
    0x8002, /*  TPC gain (2)  */
	0x800c, /*  TpcGain (-6)PD gain -6  */
};

//Difference in dB between TPC and reference TPC (at index 0)
static const uint8 rficTpcDiff[RFIC_TX_GAIN_TPC_INDEX_MAX] = {0,4,8};

static const uint32 rficTxGainRegs[RFIC_TX_GAIN_TPC_INDEX_MAX] =
{		
    RF_ANTENNA0_FCSI_PGC2TXDC0, 
    RF_ANTENNA0_FCSI_PGC2TXDC1,
    RF_ANTENNA0_FCSI_PGC2TXDC2, 				 
} ;

#define RFIC_CIS_NUM_REGS	16
uint16 rficCalibrationData[RFIC_CIS_NUM_REGS] = {0};
uint8 rficCalLoaded = 0;

uint8 rficCisIndexToReg[RFIC_CIS_NUM_REGS] =
{
	2,
	3,
	4,
	5,
	6,
	7,
	8,
	9,
	11,
	14,
	15,
	0,
	1,
	10,
	12,
	13
};

TemperatureSensorParams_t tempSensorParams = {0};

/******************************************************************************/
/***						Macros											***/
/******************************************************************************/
#define CALC_PGC1_INDEX(x) (((x) + 6) / 6)
#define CALC_PGC2_INDEX(x) (((x) + PGC2_INDEX_TO_GAIN_BIAS) / PGC2_INDEX_TO_GAIN_FACTOR)
#define CALC_PGC3_INDEX(x)  ((x) + 3)

#define CALC_PGC1_GAIN(x)	(6*((x) - 1)) // PGC Gain[dbm] = (pgc1GainIndex - 1) *6
#define CALC_PGC2_GAIN(x)	(6*((x) - 1)) // PGC Gain[dbm] = (pgc1GainIndex - 1) *6
#define CALC_PGC3_GAIN(x)	((x) - 3))    // PGC Gain[dbm] = (pgc1GainIndex - 3)

#define CALC_TX_LO_DAC_INDEX_FROM_VAL(x) (x)
#define CALC_TX_LO_DAC_VAL_FROM_INDEX(x) (x)

/* Nfrac will be the fractional part left after the integer bin value divider is
 * subtracted from the main divider value, minus 4 */
#define GET_FRACTIONAL_PART(N_MAIN_DIV, N_INT) ((N_MAIN_DIV) - (N_INT) - (4 << 21))

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static void readFromRficRegister(uint32 inRfRegAddr, uint16 *outValues);
static inline void setRxDac1(AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType, uint16 *inRxDacValues);
static inline void setRxDac2(AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType, uint16 *inRxDacValues);
static inline void updateDB(INDIRECT_MULTI_ARG_t *inMultiArgVals, uint16 inMask);
static void setAMDetectorLoop(Antenna_e inTxAnt, Antenna_e inRxAnt);
static void setRxLpfLoop(Antenna_e inTxAnt, Antenna_e inRxAnt);
static void setTxLpfLoop(Antenna_e inTxAnt, Antenna_e inRxAnt);
static void setRssiLoop(void);
static void writePsdTableFromDB(void);
static void RficDriver_setS2dGain(void);

static void getTempSensorParams(void);
static void tempSensorStartClock(void);
static void tempSensorRead(int32* temperature);
static void tempSensorStopClock(void);
static void updateMinimumEnergy(int8 inValue, int32 inEnergy, int32 *outMinEnergy_p, int8 *outMinIndex_p);
static void measureEnergy(uint8 inTpcIndex, int8 inValueI ,int8 inValueQ, int8 inBin, RficLoopbackAntennas_t *inAntInfo_p, TxLO_elements_t* outElements_p, int32 *outEnergy_p);
static void GetTxloPivot(TxLO_elements_t*outElements_p, int8 fixedcntrl, uint8 IterationNumber);


void RficDriver_set_lo_frequencies(void);
void RficDriver_set_rfchannel(uint8_t AntMsk,  uint8 chanl_bit);
void RficDriver_set_rxdc(uint8_t AntMsk,  uint8 setChnlBw);

void pollRegField(uint32 regAddr, uint32 regMask, uint32 pollVal);

/******************************************************************************/
/***							Debug Section								***/
/******************************************************************************/

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
void RficDriver_Init(void)
{
	// Reset RFIC registers
#if !defined(ENET_INC_HW_FPGA)
    IndirectReset(NULL);
#endif
	rfic_seq_init();
	writePsdTableFromDB();
	RficDriver_SetLnaGainLUT();
	RficDriver_setS2dGain();

	rfic_read_efuse_values(1, rficCalibrationData);

	getTempSensorParams();
}
void RficDriver_BandInit(void)
{
}


void RficDriver_disable_pll_lowpower(void)
{
	rfic_disable_pll_lowpower();
}
void RficDriver_enable_pll_lowpower(void)
{
	rfic_enable_pll_lowpower();
}

uint8 rfic_get_mode_lp(uint8 AntMsk)//should get the exact index and then dont need a loop
{
	uint16 antArr[MAX_NUM_OF_ANTENNAS] = {0};
	uint8 antIndex = 0;
	uint8 mode;
	int debugCount=0;
	readFromRficRegister(RF_ANTENNA0_FCSI_SR0, antArr);
	for (antIndex = 0; antIndex<MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		if (0 != ((0x1 << antIndex) & AntMsk))
		{
			mode = (antArr[antIndex] & RF_ANTENNA0_FCSI_SR0_MODE__SMSK) >> RF_ANTENNA0_FCSI_SR0_MODE__POS;
		}
		else
		{
			debugCount++;
		}
	}
	return mode;
}


void rfic_set_mode_lp(uint8 AntMsk, uint8 mode)
{
	RFIC_FCSI_RMW(AntMsk, RF_ANTENNA0_FCSI_SR0,
			RF_ANTENNA0_FCSI_SR0_MODE__ISMSK,
			mode << RF_ANTENNA0_FCSI_SR0_MODE__POS);
}

void RficDriver_set_rfbandwidthAndMode(uint8 AntMsk, uint8 mode, bool inCOC)
{
	RficDriver_SetFrequency_COC(AntMsk, inCOC);
	rfic_set_mode_lp(AntMsk, mode);
}



static void RficDriver_setS2dGain(void)
{
	uint32 regValue = REGISTER(REG_PHY_RX_AGC_PAGE_1_PHY_RXAGC_IF11A);
	rfic_config_s2dgain(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,regValue&0x3,(regValue&0xc)>>2,(regValue&0x30)>>4,(regValue&0xc0)>>6,(regValue&0x300)>>8,(regValue&0xc00)>>10); 	
}

static void getTempSensorParams(void)
{
	tempSensorParams.refTemp = rfic_read_efuse_handlertemp(rficCalibrationData);
	tempSensorParams.refResult = rfic_read_efuse_tempsensorresult(rficCalibrationData);
	if ((tempSensorParams.refTemp == 0) && (tempSensorParams.refResult == 0))
	{		
		tempSensorParams.refTemp = TEMPERATURE_DEFAULT_REF_TEMP_VAL;
		tempSensorParams.refResult = TEMPERATURE_DEFAULT_REF_RESULT_VAL;
	}
	
	tempSensorParams.slope = TEMPERATURE_SLOPE_VAL; // shifted left 11 bits
	tempSensorParams.offset = (tempSensorParams.refTemp << 11) - (tempSensorParams.slope * tempSensorParams.refResult); // shifted left 11 bits
}

/*****************************************************************************
*    Function Name:  RficDriver_GetRfChipID
*    Description:    Reads the revision code of the RFIC chip.
*    Parameters:     outRevId - Here we will put the returned data
*    Returns:        The revision code according to the specific RFIC chip
*    Remarks:        Implements RficDriver_PublicApi_t->GetRevisionID
*                    
*                    registers.
*                    Bits[10:0]    01  01  01     01  001  (default)
*                                  Vx  y   A/B/C  z1  z2   (0x2A9)
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetRfChipID(uint32 *outRevId)
{
    uint16 readVals[MAX_NUM_OF_ANTENNAS];

    readFromRficRegister(REG_RFIC_L41_VERSION, readVals);
    *outRevId = ((readVals[ANTENNA_0] & REG_RFIC_L41_VERSION_MASK) >> REG_RFIC_L41_VERSION_SHIFT);
}

/*****************************************************************************
*    Function Name:  RficDriver_UpdateShort0ShadowRegister
*    Description:    
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_UpdateShort0ShadowRegister(AntennaBitmaps_e inListOfAntennas)
{
	uint8 index;
	uint32 shadow;

	for (index = 0; index < MAX_NUM_OF_ANTENNAS; index++)
	{
		shadow = RficDriverRegistersLocalDB[RF_ANTENNA0_FCSI_SR0].regData[index] & (~(1 << RF_ANTENNA0_FCSI_SR0_RXON__POS));
		shadow |= (inListOfAntennas & 1) << RF_ANTENNA0_FCSI_SR0_RXON__POS;
		RegAccess_Write(RF_FCSI_S0_BACKUP0 + (index << 2), shadow);
		inListOfAntennas >>= 1;
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_ActivateRxAntennas
*    Description:    Enables the receive antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateRxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	rfic_rxoff(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
	rfic_rxon ((uint8)inListOfAntennas);

//	RficDriver_UpdateShort0ShadowRegister(inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_ActivateTxAntennas
*    Description:    Enables the transmit antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ActivateRxAntennas
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateTxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	rfic_txoff(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
	rfic_txon ((uint8)inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetAntennas
*    Description:    Set new antennas configuration according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                   		 addAntennas - 0x1 = add ants / 0x0 = remove ants
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetAntennas(AntennaBitmaps_e rfAntMask, uint8 afeAntMask, uint8 numOfAnts, PowerSaveStates_e powerSaveState)
{
	uint8 antMaskOff = (~((uint8)rfAntMask)) & 0xF;
	Antenna_e antIndex;
	
	AFE_EnableDisableAntennas(afeAntMask);

	// Turn on /off antennas
	rfic_antenna_off(antMaskOff, RFIC_DELAY_BETWEEN_ANTENNA_SET);

	for (antIndex = ANTENNA_0; antIndex < MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		if (rfAntMask & (1 << antIndex))
		{
			RficDriver_RestoreLnaConversionsTable(antIndex);
		}
		else
		{
			RficDriver_OverrideLnaConversionsTable(antIndex, RficDriver_getRffeMode(T_STATE_MODE, antIndex));
		}
	}
}

/***************************************************************************
**
** NAME          RficDriver_EnableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_EnableRFantennas(AntennaBitmaps_e AntMask)
{

}

/***************************************************************************
**
** NAME          LmHdk_DisableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : disables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_DisableRFantennas(uint32 rfAntMask, uint8 afeAntMask)
{

}


/*****************************************************************************
*    Function Name:  RficDriver_SetStandbyMode
*    Description:    Sets the RFIC registers related to Rx/Tx to standby mode.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*    Returns:        None
*    Remarks:        None
*    Restrictions:   For calibration purposes
******************************************************************************/
void RficDriver_SetStandbyMode(AntennaBitmaps_e inListOfAntennas)
{
    writeOneValueToAntList(inListOfAntennas, REG_RFIC_S0_SHORT_0, RFIC_STANDBY_MODE_VALUE, RFIC_STANDBY_MODE_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetLoopback
*    Description:    Configure the loop back for calibrations.
*    Parameters:     inAntenna  - The antenna we want to loop
*                    inLoopType - The type of loop we want to configure in the
*                                 RFIC (RF, BB, AMDET)
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetLoopback
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetLoopback(Antenna_e txAnt, Antenna_e rxAnt, RficLoopbackTypeIndex_e inLoopType)
{
    switch(inLoopType)
    {
        /* AM Detector Loop Back */
        case RFIC_LOOPBACK_TYPE_TX_AMDET :
            setAMDetectorLoop(txAnt, rxAnt);
            break;

        /* Rx LPF Loop Back */
        case RFIC_LOOPBACK_TYPE_RX_LPF:
            setRxLpfLoop(txAnt, rxAnt);
            break;

        case RFIC_LOOPBACK_TYPE_TX_LPF:
            setTxLpfLoop(txAnt, rxAnt);
            break;

        case RFIC_LOOPBACK_TYPE_RSSI:
            setRssiLoop();
            break;

        default:
            DEBUG_ASSERT(0);
            break;
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_GetDigitalLoopBackAnt
*    Description:    Get the Tx and Rx antennas to use in the digital path of the loopback
*    Parameters:     inOutLoopbackAnts - an input/output struct pointer:
*                    inOutLoopbackAnts->RfCalibAnt    - input: The calibrated antenna
*                    inOutLoopbackAnts->RxLoopbackAnt - output: Here we put the Rx loopback antenna to return
*                    inOutLoopbackAnts->TxLoopbackAnt - output: Here we put the Tx loopback antenna to return
*    Returns:        The Tx and Rx loopback antennas to use according to the
*                    calibrated antenna input
*    Remarks:        Implements RficDriver_PublicApi_t->GetDigitalLoopBackAnt
*                    
*                    
*                    (Tx1 -> Rx1,Tx2->Rx2 Tx3 ->Rx3) because we have pin sharing
*                    (Tx\Rx), the digital path used for TX is different from the
*                    digital Rx path
*                    
*                    Calibrated Antenna = 0: Digital Tx0 -> Digital Rx1
*                    Calibrated Antenna = 1: Digital Tx1 -> Digital Rx0
*                    Calibrated Antenna = 2: Digital Tx2 -> Digital Rx1
*                    
*    Restrictions:   None
******************************************************************************/
void RFIC_RestoreFromLoopBack(Antenna_e RfCalibAnt)
{
	uint8 txAntMask = 1 << RfCalibAnt;
	
	rfic_disable_ssbcal(txAntMask);
	
	rfic_txoff(txAntMask);

	rfic_disable_bbio_x_y();
}

void RficDriver_GetDigitalLoopBackAnt(RficLoopbackAntennas_t *inOutLoopbackAnts)
{
	uint8 rxAntEnabledMask;
	uint8 antIndex;
	uint8 rxAnt;
	uint8 txIsOdd = (inOutLoopbackAnts->RfCalibAnt & 1);

	rxAntEnabledMask = PhyDrv_GetRxEnabledAntennas();
	rxAntEnabledMask &= (~(1 << inOutLoopbackAnts->RfCalibAnt));
	ASSERT(rxAntEnabledMask != 0); // only 1 antenna enabled

	// Try optimal pairs: 0<-->1 and 2<-->3
	rxAnt = inOutLoopbackAnts->RfCalibAnt + (1 - (txIsOdd << 1));
	if ((rxAntEnabledMask & (1 << rxAnt)) == FALSE)
	{
		// If not possible, try 1<-->2
		rxAnt = inOutLoopbackAnts->RfCalibAnt + ((-1) + (txIsOdd << 1));
		
		if ((rxAnt > MAX_ANTENNA) || (rxAntEnabledMask & (1 << rxAnt)) == FALSE)
		{
			// If not possible, search for an active antenna
			for (antIndex = 0; antIndex < MAX_NUM_OF_ANTENNAS; antIndex++)
			{
				if ((rxAntEnabledMask & (1 << antIndex)) != FALSE)
				{
					rxAnt = antIndex;
					rxAntEnabledMask = 0; //kill off loop
				}
			}
		}
	}	
	inOutLoopbackAnts->RxLoopbackAnt = (Antenna_e)rxAnt;	
	inOutLoopbackAnts->TxLoopbackAnt = inOutLoopbackAnts->RfCalibAnt;
}


/*****************************************************************************
*    Function Name:  RficDriver_SetTuningWords
*    Description:    Set RFIC values according to selected channel (frequency).
*    Parameters:     inFreq - The desired LO frequency
*                    inFvco - The calculated VCO frequency
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetTuningWords
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetTuningWords(uint32 inFreq, uint32 inFvco)
{
    uint8  i         = 0;
    uint32 twLoChan  = RFIC_TUNE_WORD_LO_CHAN_HIGH_BAND_VCO_LOW;
    uint32 twLnaBand = lnaBandSelect5GTuneWord.tuneWords[RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES-1].word;
    uint32 twTxMixer = txMixerBandSelect5GTuneWord.tuneWords[RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES-1].word;
    uint32 twPreDrv  = predriverBandSelect5GTuneWord.tuneWords[RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES-1].word;

    //Look-Up for LO Channel frequency range select for high band
    if (RFIC_VCO_HIGH_BAND_RANGE_SELECT >= inFvco)
    {
        /* Usually selected for 2.4GHz */
        twLoChan  = RFIC_TUNE_WORD_LO_CHAN_HIGH_BAND_VCO_LOW;
    }
    else
    {
        /* Usually selected for 5.2GHz */
        twLoChan = RFIC_TUNE_WORD_LO_CHAN_HIGH_BAND_VCO_HIGH;
    }

    //Look-up for LNA band select (5GHz) tune word
    for (i=0; i<RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES; i++)
    {
        if (inFreq < lnaBandSelect5GTuneWord.tuneWords[i].freq)
        {
            twLnaBand = lnaBandSelect5GTuneWord.tuneWords[i].word;
            break;
        }
    }

    //Look-up for Tx mixer band select (5GHz) tune word
    for (i=0; i<RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES; i++)
    {
        if (inFreq < txMixerBandSelect5GTuneWord.tuneWords[i].freq)
        {
            twTxMixer = txMixerBandSelect5GTuneWord.tuneWords[i].word;
            break;
        }
    }

    //Look-up for Pre-driver band select (5GHz) tune word
    for (i=0; i<RFIC_MAX_NUMBER_OF_TUNE_WORD_ENTRIES; i++)
    {
        if (inFreq < predriverBandSelect5GTuneWord.tuneWords[i].freq)
        {
            twPreDrv = predriverBandSelect5GTuneWord.tuneWords[i].word;
            break;
        }
    }

    //Set LO channel high-band frequency range select tune word
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                           REG_RFIC_L14_SEL_FREQ_HB,
                           twLoChan,
                           REG_RFIC_L14_SEL_FREQ_HB_MASK,
                           REG_RFIC_L14_SEL_FREQ_HB_SHIFT,
                           TRUE);

    //Set LNA band select (5GHz) tune word
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                           lnaBandSelect5GTuneWord.regAddr,
                           twLnaBand,
                           lnaBandSelect5GTuneWord.regMask,
                           lnaBandSelect5GTuneWord.regShift,
                           TRUE);

    //Set Tx mixer band select (5GHz) tune word
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                           txMixerBandSelect5GTuneWord.regAddr,
                           twTxMixer,
                           txMixerBandSelect5GTuneWord.regMask,
                           txMixerBandSelect5GTuneWord.regShift,
                           TRUE);

    //Set Pre-driver band select (5GHz) tune word
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                           predriverBandSelect5GTuneWord.regAddr,
                           twPreDrv,
                           predriverBandSelect5GTuneWord.regMask,
                           predriverBandSelect5GTuneWord.regShift,
                           TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_CopyRegValuesFromProgModel
*    Description:    Copy default registers values, taken from the ProgModel.
*    Parameters:     inSrcAddress - Starting address of the data.
*                    inLength     - Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyRegValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyRegValuesFromProgModel(uint32 *inSrcAddress, uint32 inLength)
{
    RficProgModel_t *curData = (RficProgModel_t*)inSrcAddress;
    uint32          i        = 0;

    // Init the indirect bus to the RF/AFE modules
    //IndirectReset(NULL);

    for (i = 0; i < inLength; i++)
    {
        writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, curData->Register, curData->Data, RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, FALSE);
        curData=(RficProgModel_t*)(++inSrcAddress);
        //TODO (Ariel Groenteman) - check if we really need this delay
        //Delay next write, because we use an adjacent branch in the address tree
        MT_DELAY(500);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_CopyDataBaseValuesFromProgModel
*    Description:    Copy database values from the ProgModel.
*    Parameters:     inMinorID    - Minor ID of the tuning word address.
*                    inSrcAddress - Starting address of the database data to copy.
*                    inLength     -Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyDataBaseValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress,
                                                uint32 inLength)
{
    /* Not needed in AR10, Tuning words are hard coded as arrays */
}

/*****************************************************************************
*    Function Name:  RficDriver_LoadRegDB
*    Description:    Loads all registers from SW DB to RFIC
*    Parameters:     Nane
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->LoadRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_LoadRegDB(void)
{
    uint8  entryInd = 0;

    for (entryInd = 0;entryInd < RFIC_NUM_OF_REGS; entryInd++)
    {
        writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                            RficDriverRegistersLocalDB[entryInd].regAddr,
                            RficDriverRegistersLocalDB[entryInd].regData,
                            RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, FALSE);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_StoreRegDB
*    Description:    Stores all RFIC registers to SW DB
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->StoreRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_StoreRegDB(void)
{
    uint32 entryInd  = 0;
    uint32 addr      = 0;
    uint32 dbIndex   = 0;

    //Store short registers
    for (entryInd = 0; entryInd < RFIC_NUM_OF_SHORT_REGS; entryInd++)
    {
        /* Calculate DB index */
        dbIndex = entryInd;

        /* Configure the address to read (and also save it to the DB) */
        addr = entryInd + RFIC_FIRST_SHORT_REG_ADDR;
        RficDriverRegistersLocalDB[dbIndex].regAddr = addr;

        /* Read the values from the RF registers into the DB */
        readFromRficRegister(addr, RficDriverRegistersLocalDB[dbIndex].regData);
    }

    //Store long registers
    for (entryInd = 0; entryInd < RFIC_NUM_OF_LONG_REGS; entryInd++)
    {
        /* Calculate DB index */
        dbIndex = entryInd+ RFIC_NUM_OF_SHORT_REGS;

        /* Configure the address to read (and also save it to the DB) */
        addr = entryInd + RFIC_FIRST_LONG_REG_ADDR;
        RficDriverRegistersLocalDB[dbIndex].regAddr = addr;

        /* Read the values from the RF registers into the DB */
        readFromRficRegister(addr, RficDriverRegistersLocalDB[dbIndex].regData);
    }
}

void RficDriver_writeSWCalToRfic(uint8 lnaIndex, int8 inRxDacValue[MAX_NUM_OF_ANTENNAS][RFIC_PATH_TYPE_MAX])
{

	uint16 dacValues[MAX_NUM_OF_ANTENNAS];
	uint8 ant;
	int reg_address;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		dacValues[ant] = ((1<<RF_ANTENNA0_FCSI_RXMIXDC00_SUCCESS_I__POS)|
							(1<<RF_ANTENNA0_FCSI_RXMIXDC00_SUCCESS_Q__POS)|
							(inRxDacValue[ant][RFIC_PATH_TYPE_I] & RF_ANTENNA0_FCSI_RXMIXDC00_DACI__MSK) << RF_ANTENNA0_FCSI_RXMIXDC00_DACI__POS) |
							((inRxDacValue[ant][RFIC_PATH_TYPE_Q] & RF_ANTENNA0_FCSI_RXMIXDC00_DACQ__MSK) << RF_ANTENNA0_FCSI_RXMIXDC00_DACQ__POS);
	}
	reg_address=GAIN0_CH_MODE0;
	reg_address+= RXDC_NUM_SELECTION*lnaIndex;
	RficDriver_WriteRegister(reg_address, dacValues);

}


/*****************************************************************************
*    Function Name:  RficDriver_SetRxDACs
*    Description:    Set the PGCs from Rx path, DAC values 
*    Parameters:     inAnt        - The updated antenna
*                    inPathType   - I, Q or both
*                    inDacIndex   - The required DAC to update
*                    inRxDacValue - Values to set
*                    inLnaIndex   - LNA index
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxDACs
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetRxDACs(AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType,
                          RficDacIndex_e inDacIndex, int8 inRxDacValue[][2], HdkLnaIndex_e inLnaIndex)
{
    uint16 dacValues[MAX_NUM_OF_ANTENNAS];
	uint8 ant;

    switch(inDacIndex)
    {
        case RFIC_DAC0:
            break;
        case RFIC_DAC1:
			for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
			{
				dacValues[ant] = ((inRxDacValue[ant][RFIC_PATH_TYPE_I] & RXDC_DAC1_MASK) << RF_ANTENNA0_FCSI_SR3_I_OFFSET_DAC0__POS) |
									((inRxDacValue[ant][RFIC_PATH_TYPE_Q] & RXDC_DAC1_MASK) << RF_ANTENNA0_FCSI_SR3_Q_OFFSET_DAC0__POS);
			}
            setRxDac1(inAntBitmap, inPathType, dacValues);
            break;
        case RFIC_DAC2:
			for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
			{
				dacValues[ant] = ((inRxDacValue[ant][RFIC_PATH_TYPE_I] & RXDC_DAC2_MASK) << RF_ANTENNA0_FCSI_PGC2RXDC2_DACI__POS) |
									((inRxDacValue[ant][RFIC_PATH_TYPE_Q] & RXDC_DAC2_MASK) << RF_ANTENNA0_FCSI_PGC2RXDC2_DACQ__POS);
			}
            setRxDac2(inAntBitmap, inPathType, dacValues);
            break;
        default:
            DEBUG_ASSERT(0);
    }
}

/*****************************************************************************
*    Function Name:  SetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     antmask         - The updated antenna mask
*                   	 inLnaIndex - The required lna gain values
*				inPgc1Gain,inPgc2Gain,inPgc3Gain - PGC gains value
*    Returns:        None
*    Remarks:        Implements RficDriver
*    Restrictions:   None
******************************************************************************/
void SetRxGains(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain)
{
	Antenna_e i;
	RficRxGains_t RficRxGains;
	
	RficRxGains.lnaIndex = inLnaIndex;
	RficRxGains.pgc1Gain = inPgc1Gain;
	RficRxGains.pgc2Gain = inPgc2Gain;
	
	for(i = ANTENNA_0; i < MAX_NUM_OF_ANTENNAS; ++i)
	{
		if( (1<<i) & antMask)
		{
			RficDriver_SetRxGains(i,&RficRxGains); 		
		}
	}
}

void SetRxGainsForRssi(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain)
{
	SetRxGains(antMask,inLnaIndex,inPgc1Gain,inPgc2Gain,inPgc3Gain);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     inAnt         - The updated antenna
*                    inRficRxGains - The required gains and DACs values
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxGains
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetRxGains(Antenna_e inAnt, RficRxGains_t *inRficRxGains)
{
	uint8 antMask;
	uint8 lnaGain;

	lnaGain = RficDriver_GetLnaGain(inAnt, inRficRxGains->lnaIndex);

	antMask = 1 << inAnt;
	rfic_set_rxgain(antMask, lnaGain, inRficRxGains->pgc1Gain, inRficRxGains->pgc2Gain);
}

void RficDriver_AbstractSetRxGains(uint8 antMask, uint8 lnaGain, int32 pgc1Gain, int32 pgc2Gain)
{
	rfic_set_rxgain(antMask, lnaGain, pgc1Gain, pgc2Gain);
}

void RficDriver_GetRxGains(Antenna_e inAnt, RficRxGains_t *outRficRxGains)
{
    uint16 regVal[MAX_NUM_OF_ANTENNAS];
   
    readFromRficRegister(REG_RFIC_S2_PGC1GAIN,regVal);
     // Extract PGC1 gain index and calculate actual Gain
    outRficRxGains->pgc1Gain=(int8)(CALC_PGC1_GAIN((regVal[inAnt] & REG_RFIC_S2_PGC1GAIN_MASK) >> REG_RFIC_S2_PGC1GAIN_SHIFT));
    // Extract PGC2 gain index and calculate actual Gain
    outRficRxGains->pgc2Gain=(int8)(CALC_PGC2_GAIN((regVal[inAnt] & REG_RFIC_S2_PGC2GAIN_MASK) >> REG_RFIC_S2_PGC2GAIN_SHIFT));
    // LNA Gain & PGC 3
    readFromRficRegister(REG_RFIC_S1_LNAGAIN,regVal);
    // Extract PGC3(VGA) gain index and calculate actual Gain
    outRficRxGains->pgc3Gain=(int8)((regVal[inAnt] & REG_RFIC_S1_PGC3GAIN_MASK) >> REG_RFIC_S1_PGC3GAIN_SHIFT);
     // Extract lna gain index and calculate actual Gain
    outRficRxGains->lnaActiveValue=(int8)((regVal[inAnt] & REG_RFIC_S1_LNAGAIN_MASK) >> REG_RFIC_S1_LNAGAIN_SHIFT);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetLoDACs
*    Description:    Set the Tx DACs for LO leakage cancellation
*    Parameters:     inAnt        - The updated antenna
*                    inTpcIndex   - The Tx gain index to use
*                    inTxDacValue - The required Tx DAC value
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetLLDACs
*    Restrictions:   inTxDacValues is an array of size RFIC_PATH_TYPE_MAX
******************************************************************************/
void RficDriver_SetLoDACs(Antenna_e inAnt,  RficTxGainTpcIndex_e inTpcIndex, int8 *inTxDacValue)
{
    AntennaBitmaps_e antList   = (AntennaBitmaps_e)(1 << inAnt);
    uint32           regAddr   = rficTxGainRegs[inTpcIndex];
    uint32           regValue  = 0;

    /* Check that input DAC values are valid */
    ASSERT((inTxDacValue[RFIC_PATH_TYPE_I] <= RFIC_DAC_MAX_VALUE) &&
           (inTxDacValue[RFIC_PATH_TYPE_I] >= RFIC_DAC_MIN_VALUE) &&
           (inTxDacValue[RFIC_PATH_TYPE_Q] <= RFIC_DAC_MAX_VALUE) &&
           (inTxDacValue[RFIC_PATH_TYPE_Q] >= RFIC_DAC_MIN_VALUE));

    /* Prepare values for writing */
    regValue = (((CALC_TX_LO_DAC_INDEX_FROM_VAL(inTxDacValue[RFIC_PATH_TYPE_I])) << RF_ANTENNA0_FCSI_PGC2TXDC0_I_GAIN0__POS) & RF_ANTENNA0_FCSI_PGC2TXDC0_I_GAIN0__SMSK)|
               (((CALC_TX_LO_DAC_INDEX_FROM_VAL(inTxDacValue[RFIC_PATH_TYPE_Q])) << RF_ANTENNA0_FCSI_PGC2TXDC0_Q_GAIN0__POS) & RF_ANTENNA0_FCSI_PGC2TXDC0_Q_GAIN0__SMSK);

	RFIC_FCSI_Write(antList,regAddr,regValue);
}

/*****************************************************************************
*    Function Name:  RficDriver_GetLoDACs
*    Description:    Get the current Tx DACs values for the required antenna
*    Parameters:     inAnt       - The updated antenna
*                    inTpcIndex  - The Tx gain index to use
*                    outDacValue - Here we put the returned Tx DAC value
*    Returns:        The Tx DAC value
*    Remarks:        Implements RficDriver_PublicApi_t->GetLLDACs
*    Restrictions:   outDacValue is an array of size RFIC_PATH_TYPE_MAX
******************************************************************************/
void RficDriver_GetLoDACs(Antenna_e inAnt, RficTxGainTpcIndex_e inTpcIndex, int8 *outDacValue)
{
    uint16 OutTxLOCal[MAX_NUM_OF_ANTENNAS];
    uint16 regAddr  = rficTxGainRegs[inTpcIndex];

    /* Read the register */
    readFromRficRegister(regAddr, OutTxLOCal);

    /* Parse the values we read */
    outDacValue[RFIC_PATH_TYPE_I] = CALC_TX_LO_DAC_VAL_FROM_INDEX(((OutTxLOCal[inAnt]) & RF_ANTENNA0_FCSI_PGC2TXDC0_I_GAIN0__SMSK) >> RF_ANTENNA0_FCSI_PGC2TXDC0_I_GAIN0__POS);
    outDacValue[RFIC_PATH_TYPE_Q] = CALC_TX_LO_DAC_VAL_FROM_INDEX(((OutTxLOCal[inAnt]) & RF_ANTENNA0_FCSI_PGC2TXDC0_Q_GAIN0__SMSK) >> RF_ANTENNA0_FCSI_PGC2TXDC0_Q_GAIN0__POS);
	if(outDacValue[RFIC_PATH_TYPE_I]  > RFIC_DAC_MAX_VALUE)
	{
		outDacValue[RFIC_PATH_TYPE_I] = outDacValue[RFIC_PATH_TYPE_I] | 0xe0;
	}
	if(outDacValue[RFIC_PATH_TYPE_Q]  > RFIC_DAC_MAX_VALUE)
	{
		outDacValue[RFIC_PATH_TYPE_Q] = outDacValue[RFIC_PATH_TYPE_Q] | 0xe0;
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_ShortRxPgcInputToGround
*    Description:    Short PGC1s input to ground for all antennas
*    Parameters:     onOff - 0=Short is On; 1=Short is Off
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ShortRxPgcInputToGround
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxPgcInputPowerState(RficRxPgcInputPowerState_e state)
{
    if(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_ON == state)
    {
		RFIC_FCSI_RMW(15, RF_ANTENNA0_FCSI_FILTER,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL__ISMSK,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL_0__SVAL);
    }
    else if(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_OFF == state)
    {
		RFIC_FCSI_RMW(15, RF_ANTENNA0_FCSI_FILTER,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL__ISMSK,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL_1__SVAL);
    }
    else
    {
        DEBUG_ASSERT(0);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxCalGain
*    Description:    Select calibration buffer gain (used only for calibrations)
*					  
*    Parameters:     inAnt       - The updated antenna
*                    inRxCalGain - 
*					  
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxCalGain
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxCalGain(Antenna_e inAnt, bool inRxCalGain, int8 *outCalGainDb_p)
{
    /* Not Supported in AR10 - Empty Function, to support generic RFIC API */
	*outCalGainDb_p = 0;
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxPGC3TuneWord
*    Description:    Set the PGC3 LPF tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxPGC3TuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
	uint16	regVal   = 0;
	uint16	regMask  = 0;
	uint8	qLpfWord = inLpfWord[RFIC_PATH_TYPE_Q];
	uint8	iLpfWord = inLpfWord[RFIC_PATH_TYPE_I];

	switch(inPathType)
	{
	case RFIC_PATH_TYPE_I:
		/* Write PGC3's LPF word */
		writeOneValueToAntList(inAnts, REG_RFIC_L33_FTRIM_I, iLpfWord, REG_RFIC_L33_FTRIM_I_MASK, REG_RFIC_L33_FTRIM_I_SHIFT, TRUE);
		break;

	case RFIC_PATH_TYPE_Q:
		/* Write PGC3's LPF word */
		writeOneValueToAntList(inAnts, REG_RFIC_L33_FTRIM_Q, qLpfWord, REG_RFIC_L33_FTRIM_Q_MASK, REG_RFIC_L33_FTRIM_Q_SHIFT, TRUE);
		break;

	case RFIC_PATH_TYPE_I_AND_Q:
		/* Write PGC3's LPF word */
		regVal  = (qLpfWord << REG_RFIC_L33_FTRIM_Q_SHIFT) | (iLpfWord << REG_RFIC_L33_FTRIM_I_SHIFT);
		regMask = (REG_RFIC_L33_FTRIM_Q_MASK | REG_RFIC_L33_FTRIM_I_MASK);
		writeOneValueToAntList(inAnts, REG_RFIC_L33_FTRIM_I, regVal, regMask, RFIC_NO_SHIFT, TRUE);
		break;

	default:
		DEBUG_ASSERT(0);
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxAllPGCsTuneWord
*    Description:    Set the PGC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxAllPGCsTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
    uint16	regVal   = 0;
    uint16	regMask  = 0;
	uint8	qLpfWord = inLpfWord[RFIC_PATH_TYPE_Q];
	uint8	iLpfWord = inLpfWord[RFIC_PATH_TYPE_I];

	// Write to PGC3
	RficDriver_SetRxPGC3TuneWord(inAnts, inPathType, inLpfWord);
	
	// Write to PGC12 as well
	switch(inPathType)
    {
        case RFIC_PATH_TYPE_I:
            /* Write PGC1's and PGC2's LPF word */
            writeOneValueToAntList(inAnts, REG_RFIC_L9_PGC12_I_FTRIM, iLpfWord, REG_RFIC_L9_PGC12_I_FTRIM_MASK, REG_RFIC_L9_PGC12_I_FTRIM_SHIFT, TRUE);
            break;

        case RFIC_PATH_TYPE_Q:
            /* Write PGC1's and PGC2's LPF word */
            writeOneValueToAntList(inAnts, REG_RFIC_L9_PGC12_Q_FTRIM, qLpfWord, REG_RFIC_L9_PGC12_Q_FTRIM_MASK, REG_RFIC_L9_PGC12_Q_FTRIM_SHIFT, TRUE);
            break;

        case RFIC_PATH_TYPE_I_AND_Q:
            /* Write PGC1's and PGC2's LPF word */
            regVal  = (qLpfWord << REG_RFIC_L9_PGC12_Q_FTRIM_SHIFT) | (iLpfWord << REG_RFIC_L9_PGC12_I_FTRIM_SHIFT);
            regMask = (REG_RFIC_L9_PGC12_Q_FTRIM_MASK | REG_RFIC_L9_PGC12_I_FTRIM_MASK);
            writeOneValueToAntList(inAnts, REG_RFIC_L9_PGC12_I_FTRIM, regVal, regMask, RFIC_NO_SHIFT, TRUE);
            break;

        default:
            DEBUG_ASSERT(0);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_GetRxPGCTuneWord
*    Description:    Get the PGC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetRxPGCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetRxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType,
                             uint8 *outLpfWord)
{
    uint16 tmpVal[MAX_NUM_OF_ANTENNAS];

    switch(inPathType)
    {
    case RFIC_PATH_TYPE_I:
        /* Read PGC1's and PGC2's LPF word */
        readFromRficRegister(REG_RFIC_L9_PGC12_I_FTRIM, tmpVal);
        outLpfWord[RFIC_PATH_TYPE_I] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L9_PGC12_I_FTRIM_MASK) >> REG_RFIC_L9_PGC12_I_FTRIM_SHIFT);
        break;

    case RFIC_PATH_TYPE_Q:
        /* Read PGC1's and PGC2's LPF word */
        readFromRficRegister(REG_RFIC_L9_PGC12_Q_FTRIM, tmpVal);
        outLpfWord[RFIC_PATH_TYPE_Q] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L9_PGC12_Q_FTRIM_MASK) >> REG_RFIC_L9_PGC12_Q_FTRIM_SHIFT);
        break;

    case RFIC_PATH_TYPE_I_AND_Q:
        /* Read PGC1's and PGC2's LPF word */
        readFromRficRegister(REG_RFIC_L9_PGC12_I_FTRIM, tmpVal);
        outLpfWord[RFIC_PATH_TYPE_I] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L9_PGC12_I_FTRIM_MASK) >> REG_RFIC_L9_PGC12_I_FTRIM_SHIFT);
        outLpfWord[RFIC_PATH_TYPE_Q] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L9_PGC12_Q_FTRIM_MASK) >> REG_RFIC_L9_PGC12_Q_FTRIM_SHIFT);
        break;

    default:
        DEBUG_ASSERT(0);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_SetTxPGCTuneWord
*    Description:    Set the TPC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->settxtpcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetTxPGCTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
	uint16	regVal   = 0;
	uint16	regMask  = 0;
	uint8	qLpfWord = inLpfWord[RFIC_PATH_TYPE_Q];
	uint8	iLpfWord = inLpfWord[RFIC_PATH_TYPE_I];

	switch(inPathType)
	{
	case RFIC_PATH_TYPE_I:
		/* Write TPC's LPF word I trim */
		writeOneValueToAntList(inAnts, REG_RFIC_L24_FTRIM_I, iLpfWord, REG_RFIC_L24_FTRIM_I_MASK, REG_RFIC_L24_FTRIM_I_SHIFT, TRUE);
		break;

	case RFIC_PATH_TYPE_Q:
		/* Write TPC's LPF word Q trim */
		writeOneValueToAntList(inAnts, REG_RFIC_L24_FTRIM_Q, qLpfWord, REG_RFIC_L24_FTRIM_Q_MASK, REG_RFIC_L24_FTRIM_Q_SHIFT, TRUE);
		break;

	case RFIC_PATH_TYPE_I_AND_Q:
		/* Write TPC's LPF word I & Q trims */
		regVal  = (qLpfWord << REG_RFIC_L24_FTRIM_Q_SHIFT) | (iLpfWord << REG_RFIC_L24_FTRIM_I_SHIFT);
		regMask = REG_RFIC_L24_FTRIM_Q_MASK | REG_RFIC_L24_FTRIM_I_MASK;
		writeOneValueToAntList(inAnts, REG_RFIC_L24_FTRIM_I, regVal, regMask, RFIC_NO_SHIFT, TRUE);
		break;

	default:
		DEBUG_ASSERT(0);
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_GetTxPGCTuneWord
*    Description:    Get the PGC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetTxTPCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetTxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType, uint8 *outLpfWord)
{
	uint16 tmpVal[MAX_NUM_OF_ANTENNAS];

	switch(inPathType)
	{
	case RFIC_PATH_TYPE_I:
		/* Read TPC's LPF word I trim */
		readFromRficRegister(REG_RFIC_L24_FTRIM_I, tmpVal);
		outLpfWord[RFIC_PATH_TYPE_I] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L24_FTRIM_I_MASK) >> REG_RFIC_L24_FTRIM_I_SHIFT);
		break;

	case RFIC_PATH_TYPE_Q:
		/* Read TPC's LPF word Q trim */
		readFromRficRegister(REG_RFIC_L24_FTRIM_Q, tmpVal);
		outLpfWord[RFIC_PATH_TYPE_Q] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L24_FTRIM_Q_MASK) >> REG_RFIC_L24_FTRIM_Q_SHIFT);
		break;

	case RFIC_PATH_TYPE_I_AND_Q:
		/* Read TPC's LPF word I & Q trims */
		readFromRficRegister(REG_RFIC_L24_FTRIM_I, tmpVal);
		outLpfWord[RFIC_PATH_TYPE_I] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L24_FTRIM_I_MASK) >> REG_RFIC_L24_FTRIM_I_SHIFT);
		outLpfWord[RFIC_PATH_TYPE_Q] = (uint8)(((tmpVal[inAnt]) & REG_RFIC_L24_FTRIM_Q_MASK) >> REG_RFIC_L24_FTRIM_Q_SHIFT);
		break;

	default:
		DEBUG_ASSERT(0);
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_GetTxGain
*    Description:    Get the Gain value of all antennas
*    Parameters:     OUT dutRficTxGainsParams_t *dutRficTxGainsParams_p
*    Returns:        None
*    Remarks:        
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetTxGain(OUT dutRficTxGainsParams_t *dutRficTxGainsParams_p)
{
	uint16 antArr[MAX_NUM_OF_ANTENNAS] = {0};
	uint8 antIndex = 0;
	
	readFromRficRegister(RF_ANTENNA0_FCSI_SR2, antArr);

	for (antIndex = 0; antIndex<MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		dutRficTxGainsParams_p->antTxGains[antIndex].bbGain = 
			(antArr[antIndex] & RF_ANTENNA0_FCSI_SR2_PGC2_TXGAIN_CONF__SMSK) >> RF_ANTENNA0_FCSI_SR2_PGC2_TXGAIN_CONF__POS;
		dutRficTxGainsParams_p->antTxGains[antIndex].paDriverGain = 
			(antArr[antIndex] & RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__SMSK) >> RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__POS;
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_SetTxGain
*    Description:    Set the TPC value
*    Parameters:     inAnt      - The updated antenna
*                    inTpcIndex - The required index for the TPC value
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetTPCGain
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetTxGain(Antenna_e inAnt, uint8 inTpcIndex)
{
    AntennaBitmaps_e antList   = (AntennaBitmaps_e)(1 << inAnt);

	uint8 pgcGain = rficTxGainValues[inTpcIndex] & RF_ANTENNA0_FCSI_SR2_PGC2_TXGAIN_CONF__SMSK;
	uint8 pd = (rficTxGainValues[inTpcIndex] & RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__SMSK) >> RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__POS;

	RficDriver_AbstractSetTxGain(antList, pgcGain, (pd << 1)); //convert PD gain to dB
}

void RficDriver_AbstractSetTxGain(AntennaBitmaps_e antList, int pgcGain, int pd)
{
	rfic_set_txgain(antList, pgcGain, pd);
}
void RficDriver_createPgcfiltLut(uint16 sA, int16 sB)
{
}
void RficDriver_getPgcFiltLut(uint8 mLut[5][11])
{
}


/*****************************************************************************
*    Function Name:  RficDriver_TxLoGetTpcInfo
*    Description:    Set the TPC value
*    Parameters:     inAnt          - The required antenna
*                    outTxLoTpcInfo - Here we put the info requested for the required antenna
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->TxLoGetTpcInfo
*    Restrictions:   None
******************************************************************************/
void RficDriver_TxLoGetTpcInfo(Antenna_e inAnt, RficDriverTxLoTpcInfo_t *outTxLoTpcInfo)
{
    memcpy(outTxLoTpcInfo->tpcDiffFromRef, rficTpcDiff, RFIC_TX_GAIN_TPC_INDEX_MAX);
    outTxLoTpcInfo->tpcArraySize = RFIC_TPC_INFO_ARRAY_SIZE;
    outTxLoTpcInfo->numCalibTpcs = RFIC_TPC_INFO_NUM_CALIB_TPCS;
    outTxLoTpcInfo->calibTpcMask = RFIC_TPC_INFO_CALIB_MASK;
}

/*****************************************************************************
*    Function Name:  RficDriver_PowerUpBiasCalBlock
*    Description:    Power-up the bias central constant current calibration
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->PowerUpBiasCalBlock
*    Restrictions:   None
******************************************************************************/
void RficDriver_PowerUpBiasCalBlock(void)
{
    uint32 regVal  = 0;
    uint16 regMask = 0;

    /* Power-up the bias calibration blocks */
    regVal = 1 << REG_RFIC_L51_EN_V2I_CAL_SHIFT;
    /* Set Rint to 0 is already done since regVal is initialized to 0 */
    /* Set masks */
    regMask = REG_RFIC_L51_EN_V2I_CAL_MASK | REG_RFIC_L51_CAL_ICONST_MASK;
    /* write value to register L51 */
    writeOneValueToAntList(ANTENNA_BITMAP_ANTENNA_0, REG_RFIC_L51_EN_BGP, regVal, regMask, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRintValue
*    Description:    Programs the Rint resistor's value
*    Parameters:     inValue - Value to set Rint to
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRintValue
*    Restrictions:   max=31, min=0, default=16
******************************************************************************/
void RficDriver_SetRintValue(uint8 inValue)
{
    DEBUG_ASSERT(RFIC_CENT_BIAS_RINT_MAX >= inValue);

    /* Long registers after L32 exist only for antenna 0 */
    writeOneValueToAntList(ANTENNA_BITMAP_ANTENNA_0, REG_RFIC_L51_CAL_ICONST, inValue, REG_RFIC_L51_CAL_ICONST_MASK, REG_RFIC_L51_CAL_ICONST_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_ReadBiasCalComparator
*    Description:    Reads and returns the comparator's output
*    Parameters:     outCompResult - Here we put the value read from the register
*    Returns:        The output of the bias calibration comparator's output
*    Remarks:        Implements RficDriver_PublicApi_t->ReadBiasCalComperator
*    Restrictions:   None
******************************************************************************/
void RficDriver_ReadBiasCalComparator(bool *outCompResult)
{
    uint16 compOut[MAX_NUM_OF_ANTENNAS];

    readFromRficRegister(REG_RFIC_L51_CAL_ICONST_COMP_O, compOut);

    compOut[ANTENNA_0]  &= REG_RFIC_L51_CAL_ICONST_COMP_O_MASK;
    compOut[ANTENNA_0] >>= REG_RFIC_L51_CAL_ICONST_COMP_O_SHIFT;

    if (RFIC_CENT_BIAS_COMP_OUT_VAL_PASS == compOut[ANTENNA_0])
    {
        *outCompResult = TRUE;
    }
    else
    {
        *outCompResult = FALSE;
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_InitRssiDcOffsetCalib
*    Description:    Initializes the RSSI DC calibration
*    Parameters:     inAntList       - The required antennas to calibrate
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->InitRssiDcOffsetCalib
******************************************************************************/
void RficDriver_InitRssiDcOffsetCalib(AntennaBitmaps_e inAntList)
{
    /* Set antennas to standby mode */
    writeOneValueToAntList(inAntList, REG_RFIC_S0_SHORT_0, RFIC_RSSI_DC_CAL_INIT_S0_VALUE, RFIC_RSSI_DC_CAL_INIT_S0_MASK, RFIC_NO_SHIFT, FALSE);
    /* Short RSSI input to GND */
    writeOneValueToAntList(inAntList, REG_RFIC_L2_RSSI_REG3, RFIC_RSSI_DC_CAL_INIT_L2_VALUE, RFIC_RSSI_DC_CAL_INIT_L2_MASK, RFIC_NO_SHIFT, FALSE);

    /* Set ANA-MUX to RSSI and to step 0 (out of 0 to 4) */
    writeOneValueToAntList(inAntList, REG_RFIC_L4_ANA_MUX, RFIC_RSSI_DC_CAL_INIT_L4_VALUE, RFIC_RSSI_DC_CAL_INIT_L4_MASK, RFIC_NO_SHIFT, FALSE);

    /* ANA-MUX setting for RSSI DC calibration */
    writeOneValueToAntList(inAntList, REG_RFIC_L5_ANA_MUX_PROCMON_TEMPS, RFIC_RSSI_DC_CAL_INIT_L5_VALUE, RFIC_RSSI_DC_CAL_INIT_L5_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetCalibStage
*    Description:    Sets the calibration stage number in the calibration stage
*                    register as well as the path type in the same register for
*                    the selected antenna list (bitmap), path type (I or Q but NOT both)
*                    and stage number (0 through 4)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 1 and 5
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetSetCalibStage
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetCalibStage(AntennaBitmaps_e inAntList,
                                          RficPathTypeIndex_e inInputPathType,
                                          uint8 inStageNumber)
{
    uint32 regVal = 0;

    /* Update step in ANA-MUX (steps here are 0 to 4) */
    writeOneValueToAntList(inAntList, REG_RFIC_L4_SEL_RSSI, inStageNumber, REG_RFIC_L4_SEL_RSSI_MASK, REG_RFIC_L4_SEL_RSSI_SHIFT, FALSE);

    ++inStageNumber; /* Calibration stage numbers are 1 to 5 */
    ASSERT(RFIC_RSSI_DC_CAL_MAX_STEP_NUM >= inStageNumber);

    regVal = inStageNumber << REG_RFIC_L2_CAL_STEP_SHIFT;

    switch(inInputPathType)
    {
        case RFIC_PATH_TYPE_I:
            regVal |= 1 << REG_RFIC_L2_CAL_I_SHIFT;
            regVal |= 0 << REG_RFIC_L2_CAL_Q_SHIFT;
            break;
        case RFIC_PATH_TYPE_Q:
            regVal |= 0 << REG_RFIC_L2_CAL_I_SHIFT;
            regVal |= 1 << REG_RFIC_L2_CAL_Q_SHIFT;
            break;
        default:
            DEBUG_ASSERT(0);
    }

    /* Update step and path */
    writeOneValueToAntList(inAntList, REG_RFIC_L2_CAL_STEP, regVal, RFIC_RSSI_DC_CAL_STEP_L2_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetDacValues
*    Description:    Sets the values of the DACs in selected path, stage and antenna
*                    list (bitmap), according to the array of DAC values (inDacValues)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 0 and 4
*                    inDacValues     - The values to set in the DACs
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetCalibStg
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetDacValues(AntennaBitmaps_e inAntList,
                                         RficPathTypeIndex_e inInputPathType,
                                         uint8 inStageNumber, uint8 *inDacValues)
{
    uint16    regAddr  = 0;
    uint16    regMask  = 0;
    uint16    regShift = 0;
	uint16    regValues[MAX_NUM_OF_ANTENNAS];
	Antenna_e ant      = ANTENNA_0;

    ++inStageNumber; /* Calibration stage numbers are 1 through 5 */

    switch(inInputPathType)
    {
        case RFIC_PATH_TYPE_I:
            if (inStageNumber < RFIC_RSSI_DC_CAL_MAX_STEP_NUM)
            {
                regAddr  = REG_RFIC_L0_RSSI_REG1;
                /* stage 1 mask is 0xF000, stage 2 mask is 0xF00, stage 3 mask is 0xF0 and stage 4 mask is 0xF */
                regMask  = REG_RFIC_L0_I_RSSI_CAL1_MASK >> (REG_RFIC_L0_I_RSSI_CAL1_WIDTH * (inStageNumber - 1));
                regShift = REG_RFIC_L0_I_RSSI_CAL1_SHIFT - (REG_RFIC_L0_I_RSSI_CAL1_WIDTH * (inStageNumber - 1));
            }
            else /* stage number 5 */
            {
                regAddr  = REG_RFIC_L2_I_RSSI_CAL5;
                regMask  = REG_RFIC_L2_I_RSSI_CAL5_MASK;
                regShift = REG_RFIC_L2_I_RSSI_CAL5_SHIFT;
            }
            break;

        case RFIC_PATH_TYPE_Q:
            if (inStageNumber < RFIC_RSSI_DC_CAL_MAX_STEP_NUM)
            {
                regAddr  = REG_RFIC_L1_RSSI_REG2;
                /* stage 1 mask is 0xF000, stage 2 mask is 0xF00, stage 3 mask is 0xF0 and stage 4 mask is 0xF */
                regMask  = REG_RFIC_L1_Q_RSSI_CAL1_MASK >> (REG_RFIC_L1_Q_RSSI_CAL1_WIDTH * (inStageNumber - 1));
                regShift = REG_RFIC_L1_Q_RSSI_CAL1_SHIFT - (REG_RFIC_L1_Q_RSSI_CAL1_WIDTH * (inStageNumber - 1));
            }
            else /* stage number 5 */
            {
                regAddr  = REG_RFIC_L2_Q_RSSI_CAL5;
                regMask  = REG_RFIC_L2_Q_RSSI_CAL5_MASK;
                regShift = REG_RFIC_L2_Q_RSSI_CAL5_SHIFT;
            }
            break;

        default:
            DEBUG_ASSERT(0);
    }

	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		regValues[ant] = (uint32)inDacValues[ant];
	}

    writeToRficRegister(inAntList, regAddr, regValues, regMask, regShift, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_StoreRssiDcOffsetCalibValues
*    Description:    Stores all DC-offset values for the specific antenna
*    Parameters:     inAntList      - The antenna list to update
*                    inAllDacValues - The DAC values ordered by RficRssiDacIndexes_e
*    Returns:        None
*    Remarks:        None
*    Restrictions:   inAllDacValues is 10 slots, not-more, no-less
******************************************************************************/
void RficDriver_StoreRssiDcOffsetCalibValues(AntennaBitmaps_e inAntList, uint8 (*inAllDacValues)[RFIC_RSSI_STG_TOTAL])
{
    Antenna_e ant = ANTENNA_0;
    uint16    regL0Vals[MAX_NUM_OF_ANTENNAS];
    uint16    regL1Vals[MAX_NUM_OF_ANTENNAS];
    uint16    regL2Vals[MAX_NUM_OF_ANTENNAS];

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ++ant)
    {
        /* Prepare values for register L0 */
        regL0Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_I_4_DAC]) << REG_RFIC_L0_I_RSSI_CAL4_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_3_DAC]) << REG_RFIC_L0_I_RSSI_CAL3_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_2_DAC]) << REG_RFIC_L0_I_RSSI_CAL2_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_1_DAC]) << REG_RFIC_L0_I_RSSI_CAL1_SHIFT);
        /* Prepare values for register L1 */
        regL1Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_Q_4_DAC]) << REG_RFIC_L1_Q_RSSI_CAL4_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_3_DAC]) << REG_RFIC_L1_Q_RSSI_CAL3_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_2_DAC]) << REG_RFIC_L1_Q_RSSI_CAL2_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_1_DAC]) << REG_RFIC_L1_Q_RSSI_CAL1_SHIFT);
        /* Prepare values for register L2 */
        regL2Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_Q_5_DAC]) << REG_RFIC_L2_Q_RSSI_CAL5_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_5_DAC]) << REG_RFIC_L2_I_RSSI_CAL5_SHIFT);
    }

    /* Write the values prepared */
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L0_RSSI_REG1, regL0Vals, RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L1_RSSI_REG2, regL1Vals, RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L2_RSSI_REG3, regL2Vals, RFIC_RSSI_DC_CAL_STORE_ALL_REG_L2_MASK, RFIC_NO_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetFrequency
*    Description:    Sets the frequency of the system
*    Parameters:     inSetChannelParams - A structure with all the parameters
*                                           needed to set a frequency in the system:
*                         -> rfOscilFreq  - Frequency of the RF oscillator
*                         -> channel    - Channel we are using
*                         -> spectrumMode - CB or nCB
*                         -> upperLower   - When full Spectrum is used, which other
*                                           channel is taken, the upper->1 or lower->0
*                         -> band         - BAND_2_4_GHZ or BAND_5_2_GHZ
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetFrequency
*    Restrictions:   None
******************************************************************************/
uint16 rxBandLUT[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT] = 

	{{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
										} ;

uint16 txBandLUT[6][2] =
{
	2100, 0,
	2100, 1,
	2100, 2,
	2500, 3,
	2900, 4,
	2900, 5
};


uint16 chosenTxBand;
uint16 chosenRxBand[MAX_NUM_OF_ANTENNAS];

void  RficDriver_SetRxBandLUT(uint16 *rxBandLUTtemp)
{
	memcpy(&rxBandLUT,rxBandLUTtemp,MAX_NUMBER_OF_LNA_CROSSING_POINT*MAX_NUM_OF_ANTENNAS*2);
}

uint8 RficDriver_GetRfBw(Bandwidth_e bw)
{
	uint8 rf_bw = 80;
	if (BANDWIDTH_TWENTY == bw)
	{
		rf_bw = 20;
	}
	else if (BANDWIDTH_FOURTY == bw)
	{
		rf_bw = 40;
	}
	else if (BANDWIDTH_EIGHTY == bw)
	{
		rf_bw = 80;
	}
	else
	{
		ASSERT(0);
	}
	return rf_bw;

}

void RficDriver_SetFrequency_COC(uint8 AntMsk, bool inCOC)
{
	uint32	freq;
	uint8	rf_bw;
	Bandwidth_e bw;
	uint16 lock[MAX_NUM_OF_ANTENNAS];

	SetChannelParams_t inSetChannelParams;
	memcpy(&inSetChannelParams,HDK_getSetChannelParams(),sizeof(SetChannelParams_t));
	if(TRUE == inCOC)
	{
		bw = BANDWIDTH_TWENTY;
		inSetChannelParams.setChannelReqParams.chan_width = bw;//BANDWIDTH_TWENTY
		inSetChannelParams.setChannelReqParams.low_chan_num = inSetChannelParams.setChannelReqParams.low_chan_num + 4*inSetChannelParams.setChannelReqParams.primary_chan_idx;
	}
	else 
	{
		bw = (Bandwidth_e)inSetChannelParams.setChannelReqParams.chan_width;
	}
	freq = RficDriver_CalcLoFrequency(&inSetChannelParams);
	rfic_set_pllreset();
	rfic_enable_lotree();
	if(TRUE == inCOC)
	{
		rfic_set_lo_frequencies (freq,freq,freq,freq,freq,freq);
	}
	else
	{
		HDK_SetFreqArr(freq,(uint8)bw);
		RficDriver_set_lo_frequencies();
	}
	MT_DELAY(100);
	rfic_release_pllreset();
	rf_bw = RficDriver_GetRfBw(bw);
	rfic_set_rfbandwidth(AntMsk, rf_bw);
	MT_DELAY(1000);
	RficDriver_ReadRegister(RF_ANTENNA0_FCSI_PLLSDM0, lock);
	if (RF_ANTENNA0_FCSI_PLLSDM0_LOCK_O__GET(lock[0]) == RF_ANTENNA0_FCSI_PLLSDM0_LOCK_O_0__SVAL)
	{
		ASSERT(0);//if enter here first try to MT_DELAY(1000) before the check
	}
}


RetVal_e RficDriver_SetFrequency(SetChannelParams_t* inSetChannelParams)
{				   
	uint32 antMask;
	uint32 freq;
	uint8  bw;
	uint8  index;
	uint32 ant;
	uint16 lock[MAX_NUM_OF_ANTENNAS];
	uint8  position = 0;
	bw = inSetChannelParams->setChannelReqParams.chan_width;
	freq = RficDriver_CalcLoFrequency(inSetChannelParams);
#ifdef BEEROCK_DEBUG
	ILOG0_DD("RficDriver_SetFrequency freq = %d, bw = %d", freq, bw);
#endif
	
	rfic_set_pllreset();
	rfic_enable_lotree();

	ILOG0_DD("DRIVER set frequency %d ....FREQ FOR CALIBRATE, is diffrent from LoFrequencies_G[2]=%d", freq,HDK_GetLoFrequency(2)); 
	SLOG0(0, 0, SetChannelParams_t, inSetChannelParams);
	ILOG0_DD("LoFrequencies_G[1]=%d, .LoFrequencies_G[2]=%d", HDK_GetLoFrequency(1),HDK_GetLoFrequency(2));

	if(HDK_IsFrequencyJumpEnabled() == TRUE)
	{
		RficDriver_set_rfchannel(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,20 << bw);
		RficDriver_set_rxdc(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, 20 << bw);	
		position = HDK_GetRfBw(bw);
		HDK_SetFreqArr(freq,position);/*if FJ Enable && bw=requested && if ST_Type = Calibrate, and the Arr is not ready in advance*/
	}
	else
	{
		/* if FJ disabled, all the freq set in position 0 */
		HDK_SetFreqArr(freq,position);
	}
	
	RficDriver_set_lo_frequencies();
	
	MT_DELAY(100);
	rfic_release_pllreset();

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		for (index = 0; index < MAX_NUMBER_OF_LNA_CROSSING_POINT; index++)
		{
			if(freq < (uint32)(rxBandLUT[ant][index]))
			{
				break;
			}
			
		}
		chosenRxBand[ant] = index;
		rfic_set_rxband(0x1<<ant, (uint8_t)chosenRxBand[ant]);
	}
	
	index = 0;
	while (freq > (uint32)(txBandLUT[index][0]))
	{
		index++;
	}
	chosenTxBand = txBandLUT[index][1];
	rfic_set_txband(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, (uint8_t)chosenTxBand);

	rfic_set_rfbandwidth(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, 20 << bw);

	antMask = Hdk_GetRxAntMask();
	
	for (index = 0; index < MAX_NUM_OF_ANTENNAS; index++)
	{
		RficDriverRegistersLocalDB[RF_ANTENNA0_FCSI_SR0].regData[index] |= (antMask & 1) << RF_ANTENNA0_FCSI_SR0_RXON__POS;
		RegAccess_Write(RF_FCSI_S0_BACKUP0 + (index << 2), RficDriverRegistersLocalDB[RF_ANTENNA0_FCSI_SR0].regData[index]);
		antMask >>= 1;
	}

//	ASSERT(0);
	MT_DELAY(1000);
	RficDriver_ReadRegister(RF_ANTENNA0_FCSI_PLLSDM0, lock);
	if (RF_ANTENNA0_FCSI_PLLSDM0_LOCK_O__GET(lock[0]) == RF_ANTENNA0_FCSI_PLLSDM0_LOCK_O_0__SVAL)
	{
		return RET_VAL_FAIL;
	}
    return RET_VAL_SUCCESS;
}

uint16 RficDriver_GetTxBand()
{
	return chosenTxBand;
}
void RficDriver_GetSetRxBandAll(void)
{
	uint8 ant;	
	uint16 RxBand;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		RxBand=chosenRxBand[ant];
		rfic_set_rxband(1 << ant,RxBand);
	}
}


uint16 RficDriver_GetRxBand(uint8 ant)
{
	return chosenRxBand[ant];
}
void RficDriver_SetRxBand(uint16 rxBand)
{
	rfic_set_rxband(0xf,rxBand);
}

/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingSet
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function changes the frequency
*                    if the band number and channel number require that.
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingSet
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingSet(void)
{
    /* Not Supported in AR10 - Empty Function, to support generic RFIC API */
}

/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingReset
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function resets the frequency
*                    back to operational mode after the calibration was done
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingReset
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingReset(void)
{
    /* Not Supported in wave500 - Empty Function, to support generic RFIC API */
}

/*****************************************************************************
*    Function Name:  RficDriver_CalculateFlo
*    Description:    Returns the frequency of the local oscillator according to
*                    inputs
*    Parameters:     inChannelNum   - Channel we are using
*                    inSpectrumMode - CB or nCB
*                    inUpperLower   - When full Spectrum is used, which other
*                                     channel is taken, the upper->1 or lower->0
*                    inBand         - BAND_2_4_GHZ or BAND_5_2_GHZ
*                    outFlo         - Here we will set the output of the function
*    Returns:        The local oscillator's frequency
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_CalculateFlo(uint8 inChannelNum, SpectrumMode_e inSpectrumMode,
                             uint8 inUpperLower, uint8 inBand, uint32 *outFlo)
{
	DEBUG_ASSERT(0);
}

uint32 RficDriver_CalcLoFrequency(SetChannelParams_t * inSetChannelParams)
{
	uint32 freq;
	uint32 freqStart;

	if(BAND_5_2_GHZ == inSetChannelParams->band)
	{
		if((inSetChannelParams->setChannelReqParams.low_chan_num >= RFIC_FREQ_FIRST_CHANNEL_5G_4) &&
			(inSetChannelParams->setChannelReqParams.low_chan_num <= RFIC_FREQ_LAST_CHANNEL_5G_4))
		{
			freqStart = RFIC_FREQ_STARTING_POINT_5G_4;
		}
		else
		{
			freqStart = RFIC_FREQ_STARTING_POINT_5G_5;
		}
	}
	else
	{

		if((inSetChannelParams->setChannelReqParams.low_chan_num == 14) && 
		   (inSetChannelParams->setChannelReqParams.chan_width == BANDWIDTH_TWENTY))
		{
			freqStart = RFIC_FREQ_STARTING_POINT_2_4G_CHANNEL_14;
		}
		else
		{
			freqStart = RFIC_FREQ_STARTING_POINT_2_4G;
		}
	}

	freq = freqStart + (inSetChannelParams->setChannelReqParams.low_chan_num * RF_CHANNEL_WIDTH_MHZ) +
			bandwidthOffset[(uint8)inSetChannelParams->setChannelReqParams.chan_width];

	return freq;
}

/*****************************************************************************
*    Function Name:  RficDriver_SetXtalValue
*    Description:    Set the Xtal value 
*    Parameters:     uint32 - xtalVal
*    Returns:        void
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetXtalValue(uint32 xtalVal)
{
	rfic_trim_dcxo (xtalVal, xtalVal);
}

/*****************************************************************************
*    Function Name:  RficDriver_XtalInitPreventAmpChange
*    Description:    To avoid clock amplitude change during start, (which causes DSL problems),
*                    we need to add the following writes to the init part.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_XtalInitPreventAmpChange(void)
{
    /* Set FCSI LDO to default, DCXO current control default, FSYS strength default */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L53_FCSI_LDO_OTHERS, RFIC_SET_XTAL_L53_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    /* Enable Bandgap */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L51_BIAS_CENTRAL1,RFIC_SET_XTAL_L51_ONLY_BG_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    /* Delay between two writes to same register */
    MT_DELAY(100);

    /* Enable Bandgap, Enable Centralbias V2I const and default Bgp and temp trimming, default iconst */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L51_BIAS_CENTRAL1,RFIC_SET_XTAL_L51_FULL_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    /* Enable Centralbias V2I ref, PLL reset, soft reset released, set DCXO LDO to default */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L52_BIAS_CENTRAL2_RESET_DCXO_LDO,RFIC_SET_XTAL_L52_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    /* Set DCXO bias mode and rest to default, enable FSYS output clock */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L50_DCXO_REG2_FSYS,RFIC_SET_XTAL_L50_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_GetXtalValue
*    Description:    Get the Xtal value  from the RFIC
*    Parameters:     uint32 - xtalVal
*    Returns:        void
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetXtalValue(uint32 *xtalVal)
{
	uint16 regVal[MAX_NUM_OF_ANTENNAS];	

	/* Read Xtal Value xtal Value*/
	readFromRficRegister(RF_ANTENNA1_FCSI_DCXO0, regVal);
	
	*xtalVal = (regVal[ANTENNA_1] & RF_ANTENNA1_FCSI_DCXO0_CAP_TRIM_X1__MSK) >> RF_ANTENNA1_FCSI_DCXO0_CAP_TRIM_X1__POS;
}

/*****************************************************************************
*    Function Name:  RficDriver_GetAmDetBranch
*    Description:    This function returns the AM detector branch according to the frequency band given
*    Parameters:     inFreqBand     - The required frequency band (BAND_2_4_GHZ or BAND_5_2_GHZ only)
*                    outAmDetBranch - Here we will put the returned value
*    Returns:        None
*    Remarks:        This function ignores IQ swap if exists. User must check IQ swap with PHY driver
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetAmDetBranch(uint8 inFreqBand, RficPathTypeIndex_e *outAmDetBranch)
{
	*outAmDetBranch = RFIC_PATH_TYPE_Q;
}

/*********************************************************************************
Method:			RficDriver_WriteRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteRegister(IN uint32 regNum,IN uint16 *regVal)
{
	writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,regNum,regVal,RFIC_DEFAULT_MASK,RFIC_DEFAULT_MASK_SHIFT,FALSE);
}
/*********************************************************************************
Method:			RficDriver_WriteMaskRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 regVal
Parameter:    	IN uint32 regMask
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteMaskRegister(IN uint32 regNum,IN uint16 *regVal, IN uint32 regMask)
{
	writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, regNum, regVal, regMask, RFIC_DEFAULT_MASK_SHIFT, FALSE);
}
/*********************************************************************************
Method:			RficDriver_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	OUT uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_ReadRegister(IN uint32 regNum,OUT uint16 *regVal)
{
	readFromRficRegister(regNum,regVal);
}

/******************************************************************************
 ** General functions unrelated to the the type of RFIC we are working with ***
 ******************************************************************************/

/******************************************************************************/
/***						Staticic Functions Definitions					***/
/******************************************************************************/
/*****************************************************************************
*    Function Name:  writeToRficRegister
*    Description:    Writes inValue into inRfRegAddr, using inMask as write mask.
*    Parameters:     inAntList   - The antenna bitmap to update
*                    inRfRegAddr - The RF register address we write to.
*                    inValues    - The values we want to write into the register.
*                    inMask      - A mask of where in the register we write
*                                  inValue into.
*                    inUpdateDB  - TRUE<->Write data to local DB as well
*                                  FALSE<->Don't write to local DB
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void writeToRficRegister(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint16 *inValues, uint16 inMask, uint16 inShift, bool inUpdateDB)
{
    INDIRECT_MULTI_ARG_t multiArgVals;
    RetVal_e             retVal = RET_VAL_UNINITIALIZED;
    Antenna_e            ant        = ANTENNA_0;
    uint16               maskedVals[MAX_NUM_OF_ANTENNAS];
    uint16               readValues[MAX_NUM_OF_ANTENNAS];

    /* Read original values from registers */
    readFromRficRegister(inRfRegAddr, readValues);

    /* Change values in the antennas we need to update */
    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ++ant)
    {
        if (0 != ((0x1 << ant) & inAntList))
        {
            /* Antenna 'ant' is in the list - Modify the value we read */
            maskedVals[ant] = (((inValues[ant] << inShift) & inMask) | (readValues[ant] & (~inMask)));
        }
        else
        {
            /* Antenna 'ant' is not in the list - Keep the value we read */
            maskedVals[ant] = readValues[ant];
        }
    }

    /* Prepare the indirect struct for writing */
    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
        multiArgVals.addr[ant]  = inRfRegAddr;
        multiArgVals.value[ant] = maskedVals[ant];
    }

    /* write value to RF registers using Indirect_API */
    retVal = IndirectMultipleWrite(INDIRECT_BLOCK_RF_EXT, &multiArgVals);
    if (RET_VAL_SUCCESS != retVal)
    {
		DEBUG_ASSERT(0);
    }

    /* Update DB if needed */
    if (TRUE == inUpdateDB)
    {
        updateDB(&multiArgVals, inMask);
    }
}

/*****************************************************************************
*    Function Name:  writeOneValueToAntList
*    Description:    Writes inValue into inRfRegAddr on all antennas, using inMask
*                    as the write mask.
*    Parameters:     inRfRegAddr - The RF register address we write to.
*                    inValue     - The value we want to write into the registers.
*                    inMask      - A mask of where in the register we write
*                                  inValue into.
*                    inUpdateDB  - TRUE <->Write data to local DB as well
*                                  FALSE<->Don't write to local DB
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void writeOneValueToAntList(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint32 inValue, uint16 inMask, uint16 inShift, bool inUpdateDB)
{
    uint16 regVals[MAX_NUM_OF_ANTENNAS];
    Antenna_e ant = ANTENNA_0;

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ++ant)
    {
        regVals[ant] = inValue;
    }

    writeToRficRegister(inAntList, inRfRegAddr, regVals, inMask, inShift, inUpdateDB);
}

/*****************************************************************************
*    Function Name:  readFromRficRegister
*    Description:    Reads a value from inRfRegAddr into outValues.
*    Parameters:     inRfRegAddr - The RF register address we read from.
*                    outValues   - Here we will put the values read from the registers.
*    Returns:        The contents of the register
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void readFromRficRegister(uint32 inRfRegAddr, uint16 *outValues)
{
    INDIRECT_MULTI_ARG_t multiArgVals;
    RetVal_e             retVal = RET_VAL_UNINITIALIZED;
    Antenna_e            ant    = ANTENNA_0;

    /* Prepare the address we want to read */
    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
        multiArgVals.addr[ant] = inRfRegAddr;
    }

    retVal = IndirectMultipleRead(INDIRECT_BLOCK_RF_EXT, &multiArgVals);
    if (RET_VAL_SUCCESS != retVal)
    {
		DEBUG_ASSERT(0);
    }

    /* Return the array we read */
    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
        outValues[ant] = (uint16)multiArgVals.value[ant];
    }
}

/*****************************************************************************
*    Function Name:  setRxDac1
*    Description:    Set Rx DAC1(For DC cancellation)
*    Parameters:     inAntBitmap   - The updated antenna bitmap
*                    inPathType    - I, Q or both
*                    inRxDacValues - The required DAC values
*    Returns:        None
*    Remarks:        Due to parallel register access, 3 antennas take same time
*                    as one antenna, but 2 antennas take longer (about twice the time)
*    Restrictions:   None
******************************************************************************/
static inline void setRxDac1(AntennaBitmaps_e inAntBitmap,RficPathTypeIndex_e inPathType, uint16 *inRxDacValues)
{
    writeToRficRegister(inAntBitmap, RF_ANTENNA0_FCSI_SR3, inRxDacValues, (RF_ANTENNA0_FCSI_SR3_I_OFFSET_DAC0__SMSK | RF_ANTENNA0_FCSI_SR3_Q_OFFSET_DAC0__SMSK), RFIC_NO_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  setRxDac2
*    Description:    Set Rx DAC2(For DC cancellation)
*    Parameters:     inAntBitmap   - The updated antenna
*                    inPathType    - I, Q or both
*                    inRxDacValues - The required DAC values
*    Returns:        None
*    Remarks:        Due to parallel register access, 3 antennas take same time
*                    as one antenna, but 2 antennas take longer (about twice the time)
*    Restrictions:   None
******************************************************************************/
static inline void setRxDac2(AntennaBitmaps_e inAntBitmap,RficPathTypeIndex_e inPathType, uint16 *inRxDacValues)
{
    writeToRficRegister(inAntBitmap, RF_ANTENNA0_FCSI_PGC2RXDC2, inRxDacValues, (RF_ANTENNA0_FCSI_PGC2RXDC2_DACI__SMSK | RF_ANTENNA0_FCSI_PGC2RXDC2_DACQ__SMSK), RFIC_NO_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  updateDB
*    Description:    Updates the local DB of the RF registers with the 
*    Parameters:     inMultiArgVals - A pointer to the values and register addresses
*                                     we wrote to the RF and want to update the DB with
*                    inMask         - The mask of the bits to update
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static inline void updateDB(INDIRECT_MULTI_ARG_t *inMultiArgVals, uint16 inMask)
{
    uint16    arrayIndex = 0;
    Antenna_e ant        = ANTENNA_0;
    uint32    rfRegAddr  = inMultiArgVals->addr[ANTENNA_0];

    /* Calculate index of register in the DB array */
    if (RFIC_LAST_SHORT_REG_ADDR >= rfRegAddr)
    {
        /* Register is one of the short registers */
        arrayIndex = rfRegAddr;
    }
    else if ((RFIC_FIRST_LONG_REG_ADDR <= rfRegAddr) &&
             (RFIC_LAST_LONG_REG_ADDR  >= rfRegAddr))
    {
        /* Register is one of the long registers */
        arrayIndex = rfRegAddr - RFIC_FIRST_LONG_REG_ADDR + RFIC_NUM_OF_SHORT_REGS;
    }
    else
    {
        /* Invalid Register address */
        DEBUG_ASSERT(0);
    }

    /* write value to RF driver's DB */
    RficDriverRegistersLocalDB[arrayIndex].regAddr = rfRegAddr;

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
        RficDriverRegistersLocalDB[arrayIndex].regData[ant] = ((inMultiArgVals->value[ant]) & inMask) |
                                                              ((RficDriverRegistersLocalDB[arrayIndex].regData[ant]) & (~inMask));
    }
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void RficDriver_resetPsdArray()
{
	memset(&RficDriverPSD, 0, sizeof(RficDriverPSD));
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

void RficDriver_IREInit()
{
}

void RficDriver_copyOneRowFromPsdTable(uint32 regNum, uint32 *regVal, uint32 regMask)
{
    uint16    arrayIndex;
    uint8 ant;
	
    /* Calculate index of register in the DB array */
    if (RFIC_LAST_SHORT_REG_ADDR >= regNum)
    {
        /* Register is one of the short registers */
        arrayIndex = regNum;
    }
    else if ((RFIC_FIRST_LONG_REG_ADDR <= regNum) &&
             (RFIC_LAST_LONG_REG_ADDR  >= regNum))
    {
        /* Register is one of the long registers */
        arrayIndex = regNum - RFIC_FIRST_LONG_REG_ADDR + RFIC_NUM_OF_SHORT_REGS;
    }
    else
    {
        /* Invalid Register address */
        DEBUG_ASSERT(0);
    }
	
    /* write value to RF PSD DB */
    RficDriverPSD[arrayIndex].regAddr = regNum;
	RficDriverPSD[arrayIndex].mask |= regMask;

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
        RficDriverPSD[arrayIndex].regData[ant] |= (uint16)(regVal[ant] & regMask);
	}
}

static void writePsdTableFromDB(void)
{
    uint16    arrayIndex;
    uint8 ant;
	uint32 address, mask;
	uint16 values[MAX_NUM_OF_ANTENNAS];
	
   for (arrayIndex = 0; arrayIndex < RFIC_NUM_OF_REGS; arrayIndex++)
	{
		address = RficDriverPSD[arrayIndex].regAddr;
		mask    = RficDriverPSD[arrayIndex].mask;

		// Get values per antenna
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			values[ant] = RficDriverPSD[arrayIndex].regData[ant];
		}
		RficDriver_WriteMaskRegister(address, values, mask);
	}
}

uint32  checkPsdTableType(void)
{
	return PSD_BAND_2_4GHZ_TABLE;
}


/*****************************************************************************
*    Function Name:  setAMDetectorLoop
*    Description:    Sets the AM Detector loop-back
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setAMDetectorLoop(Antenna_e inTxAnt, Antenna_e inRxAnt)
{
    AntennaBitmaps_e txAntMask = (AntennaBitmaps_e)(1 << inTxAnt);

	rfic_rxoff(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
	
	rfic_txon(txAntMask);
	
	rfic_enable_ssbcal(txAntMask);

	rfic_enable_bbio_x_y(inTxAnt, inRxAnt);
}


void getMinimumIndex(int8 *minIndex,int8* minIndexQ,int8 iterationNumber,int8 iqIndexI,int8 iqIndexQ,int8 toneAmplitude,ClbrType_e inCalibType,Antenna_e RfCalibAnt)
{

}
void getNumberOfRuns(uint32 *numberOfRuns,uint32* runsIncrement,int8 iterationNumber)
{
	
}
/*****************************************************************************
*    Function Name:  setRxLpfLoop
*    Description:    Sets the Rx LPF loop-back
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setRxLpfLoop(Antenna_e inTxAnt, Antenna_e inRxAnt)
{
    AntennaBitmaps_e TxAnt = (AntennaBitmaps_e)(1 << inTxAnt);
    AntennaBitmaps_e RxAnt = (AntennaBitmaps_e)(1 << inRxAnt);

    /* Configuration of the calibrated antenna */
	/* Set Tx ON */
	writeOneValueToAntList(TxAnt, REG_RFIC_S0_SHORT_0, RFIC_LOOP_BACK_RX_LPF_TX_ANT_S0_VALUE, RFIC_LOOP_BACK_RX_LPF_TX_ANT_S0_MASK, RFIC_NO_SHIFT, FALSE);
    /* Disable TPC and set output to mix - in case Tx was set before loop */
    writeOneValueToAntList(TxAnt, REG_RFIC_L24_TPC_REG, RFIC_LOOP_BACK_RX_LPF_L24_VALUE, RFIC_LOOP_BACK_RX_LPF_L24_MASK, RFIC_NO_SHIFT, FALSE);
    /* Set PGC1 input selector to DAC */
    writeOneValueToAntList(TxAnt, REG_RFIC_L9_PGC12_REG1, RFIC_LOOP_BACK_RX_LPF_L9_VALUE, RFIC_LOOP_BACK_RX_LPF_L9_MASK, RFIC_NO_SHIFT, FALSE);
    /* Power Up PGC1/2 */
    writeOneValueToAntList(TxAnt, REG_RFIC_L10_PGC12_REG2, RFIC_LOOP_BACK_RX_LPF_L10_VALUE, RFIC_LOOP_BACK_RX_LPF_L10_MASK, RFIC_NO_SHIFT, FALSE);

    /* Configuration of the loop-back antenna */
    /* Set loop-back antenna to stand-by mode */
    writeOneValueToAntList(RxAnt, REG_RFIC_S0_SHORT_0, RFIC_LOOP_BACK_RX_LPF_RX_ANT_S0_VALUE, RFIC_LOOP_BACK_RX_LPF_RX_ANT_S0_MASK, RFIC_NO_SHIFT, FALSE);
    if (ANTENNA_2 == inTxAnt)
    {
        /* Power Up PGC3 Rx ant, input select C */
        writeOneValueToAntList(RxAnt, REG_RFIC_L32_PGC3_REG1, RFIC_LOOP_BACK_RX_LPF_L32_SELECT_C_VALUE, RFIC_LOOP_BACK_RX_LPF_L32_MASK, RFIC_NO_SHIFT, FALSE);
    }
    else
    {
        /* Power Up PGC3 Rx ant, input select B */
        writeOneValueToAntList(RxAnt, REG_RFIC_L32_PGC3_REG1, RFIC_LOOP_BACK_RX_LPF_L32_SELECT_B_VALUE, RFIC_LOOP_BACK_RX_LPF_L32_MASK, RFIC_NO_SHIFT, FALSE);
    }
	//TxAnt:Tri-St
	RficDriver_setRffeMode(TX_MODE,ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
	//RxAnt:Tri-St
	RficDriver_setRffeMode(T_STATE_MODE,TxAnt);

}

/*****************************************************************************
*    Function Name:  setTxLpfLoop
*    Description:    Sets the Tx LPF loop-back. NOTE: assumes existing Rx LPF loopback
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setTxLpfLoop(Antenna_e inTxAnt, Antenna_e inRxAnt)
{
    AntennaBitmaps_e TxAnt = (AntennaBitmaps_e)(1 << inTxAnt);

    /* Power Up TPC and set test mode */
    writeOneValueToAntList(TxAnt, REG_RFIC_S0_SHORT_0, RFIC_LOOP_BACK_TX_LPF_S0_VALUE, RFIC_LOOP_BACK_TX_LPF_S0_MASK, RFIC_NO_SHIFT, FALSE);
    writeOneValueToAntList(TxAnt, REG_RFIC_L9_PGC12_REG1, RFIC_LOOP_BACK_TX_LPF_L9_VALUE, RFIC_LOOP_BACK_TX_LPF_L9_MASK, RFIC_NO_SHIFT, FALSE);
    writeOneValueToAntList(TxAnt, REG_RFIC_L24_TPC_REG, RFIC_LOOP_BACK_TX_LPF_L24_VALUE, RFIC_LOOP_BACK_TX_LPF_L24_MASK, RFIC_NO_SHIFT, FALSE);

	//TxAnt:Tri-St
	RficDriver_setRffeMode(TX_MODE,ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
	//RxAnt:Tri-St
	RficDriver_setRffeMode(T_STATE_MODE,TxAnt);

}

/*****************************************************************************
*    Function Name:  setRssiLoop
*    Description:    Sets the RSSI loop-back
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setRssiLoop(void)
{
    /* Set stand-by mode, and select SSI PATH - RSSI */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_S0_SHORT_0, RFIC_LOOP_BACK_RSSI_S0_VALUE, RFIC_LOOP_BACK_RSSI_S0_MASK, RFIC_NO_SHIFT, FALSE);
    /* Select RSSI input - DAC output */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L2_RSSI_REG3, RFIC_LOOP_BACK_RSSI_L2_VALUE, RFIC_LOOP_BACK_RSSI_L2_MASK, RFIC_NO_SHIFT, FALSE);

	//Tri-St All ants
	RficDriver_setRffeMode(T_STATE_MODE,ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
}

static void tempSensorStartClock(void)
{
	rfic_stop_tempsensor_clock();
	
	rfic_start_tempsensor_clock();
}

static void tempSensorRead(int32* temperature)
{
	uint16 tempResult;
	
	rfic_start_tempsensor();

	MT_DELAY(150);
	
	tempResult = rfic_read_tempsensor();

	temperature[0] = (tempSensorParams.slope * tempResult)  + tempSensorParams.offset;
	
	temperature[0] >>= 11;
	
	temperature[1] = temperature[0];
	temperature[2] = temperature[0];
}

static void tempSensorStopClock(void)
{
	rfic_stop_tempsensor_clock();
}

/*****************************************************************************
*    Function Name:  SetRffeMode(uint8 inRxAntMask)
*    Description:    call set rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void SetRffeMode(uint8 inRxAntMask)
{
	RficDriver_setRffeMode(T_STATE_MODE,(AntennaBitmaps_e)inRxAntMask);
}

/*****************************************************************************
*    Function Name:  RficDriver_getRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
*    Description:    get rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
	
uint8 RficDriver_getRffeMode(CalRFFEState_e mode,Antenna_e ant)
{
	uint32 regAddress;
	uint8 val;
	
	DEBUG_ASSERT(mode < NUM_OF_RFFE_MODE);

	regAddress = PHY_TD_BASE_ADDR + RFE_CTRL_BASE_ADDRESS + ant*4 + mode*0x10;
	val = REGISTER(regAddress);

	return val;
}

/*****************************************************************************
*    Function Name:  RficDriver_rffe_CopyDataBaseValuesFromProgModel.
*    Description:    Save the values for rffe mode.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_rffe_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress,uint32 inLength)
{
}

/*****************************************************************************
*    Function Name:  RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
*    Description:    set rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
	
void RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
{
	uint32 antIndex;
	uint32 regAddress;
	uint16 values[MAX_NUM_OF_ANTENNAS];
	
	DEBUG_ASSERT(mode < NUM_OF_RFFE_MODE);
	for (antIndex = 0; antIndex < MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		if (ant & (1 << antIndex))
		{
			
			regAddress = PHY_TD_BASE_ADDR + RFE_CTRL_BASE_ADDRESS + antIndex*4 + mode*0x10;
			values[antIndex] = REGISTER(regAddress);
		}
	}
	writeToRficRegister(ant, REG_RFIC_S1_SHORT_1, values, RFIC_RFFE_MASK, RFIC_DEFAULT_MASK_SHIFT, FALSE);
}

void RficDriver_RxDcInitAntAll(void)
{
	uint8 ant;
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		RficDriver_RxDcInitAnt(ant);
	}
}

void RficDriver_RxDcInitAnt(uint8 ant)
{
	uint8 antMask;

	antMask = 1 << ant;
	rfic_rxon(antMask);
	rfic_start_dc_clock();
}
void RficDriver_RxDcActivateDac2ChipCalAll(int32 Pgc2Gain, uint8 intBw, uint8 numTries, int32 failTH, uint32 corrSamples)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	int32 dcLevelI[MAX_NUM_OF_ANTENNAS];
	int32 dcLevelQ[MAX_NUM_OF_ANTENNAS];
	
	uint16 pollingMask;
	uint8 tries;
	bool success;
	int success_counter;
	uint8 ant;
	pollingMask = RF_ANTENNA0_FCSI_PGC2DCOCTL_RDY2START_DCO_I__SMSK | RF_ANTENNA0_FCSI_PGC2DCOCTL_RDY2START_DCO_Q__SMSK;
	
	success = FALSE;
	tries = 0;

	while ((success == FALSE) && (tries < numTries))
	{
		success_counter=0;
		rfic_set_dcintegrator(intBw, intBw);
		pollRegField(RF_ANTENNA0_FCSI_PGC2DCOCTL, pollingMask, pollingMask);
		rfic_start_dccancel2();
		pollRegField(RF_ANTENNA0_FCSI_PGC2DCOCTL, pollingMask, pollingMask);
		rfic_stop_dccancel2();
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			dcLevelI[ant] = Abs(results[ant].II) / corrSamples;
			dcLevelI[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			dcLevelQ[ant] = Abs(results[ant].QQ) / corrSamples;
			dcLevelQ[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			if ((dcLevelI[ant] < failTH) && (dcLevelQ[ant] < failTH))
			{
				success_counter++;
			}
		}
		if (MAX_NUM_OF_ANTENNAS==success_counter)
		{
			success=TRUE;
		}
		tries++;
	}
	if (success == FALSE)
	{
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if (dcLevelI[ant] >= failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
			}
			if (dcLevelQ[ant] >= failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
			}
		}
	}
}

void RficDriver_RxDcActivateDac0ChipCalAll(RficRxGains_t* rxGains, uint8 intBw,int32 failTH, uint8 numTries, uint32 corrSamples)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelI[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelQ[MAX_NUM_OF_ANTENNAS];
	uint16 pollingMask;
	uint16 txBand;
	uint16 bw;
	uint8  tries;
	bool   success;
	int success_counter;
	uint8 ant;

	bw = HDK_getChannelWidth();
	txBand = RficDriver_GetTxBand();
	RficDriver_GetSetRxBandAll();
	rfic_set_txband(ANTENNA_BITMAP_ALL_ANTENNAS, txBand);
	rfic_set_rfbandwidth(ANTENNA_BITMAP_ALL_ANTENNAS, 20 << bw);
	rfic_rxon(ANTENNA_BITMAP_ALL_ANTENNAS);

	pollingMask = RF_ANTENNA0_FCSI_RXMIXERDCOCTL_RDY2START_DCO_I__SMSK | RF_ANTENNA0_FCSI_RXMIXERDCOCTL_RDY2START_DCO_Q__SMSK;

	success = FALSE;
	tries = 0;

	while ((success == FALSE) && (tries < numTries))
	{
		rfic_set_dcintegrator(intBw, intBw);
		pollRegField(RF_ANTENNA0_FCSI_RXMIXERDCOCTL, pollingMask, pollingMask);
		rfic_start_dccancel0();
		pollRegField(RF_ANTENNA0_FCSI_RXMIXERDCOCTL, pollingMask, pollingMask);
		rfic_stop_dccancel0();
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		for (ant=0,success_counter=0;ant<MAX_NUM_OF_ANTENNAS;ant++)
		{
			dcLevelI[ant] = Abs(results[ant].II) / corrSamples;
			dcLevelI[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			dcLevelQ[ant] = Abs(results[ant].QQ) / corrSamples;
			dcLevelQ[ant] >>= RXDC_CORRELATOR_SHIFT_FACTOR;
			if ((dcLevelI[ant] < failTH) && (dcLevelQ[ant] < failTH))
			{
				success_counter++;
			}
		}
		if (MAX_NUM_OF_ANTENNAS==success_counter)
		{
			success=TRUE;
		}
		tries++;
	}

	if (success == FALSE)
	{
		for (ant=0,success_counter=0;ant<MAX_NUM_OF_ANTENNAS;ant++)
		{
			if (dcLevelI[ant] >= failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
			}
			if (dcLevelQ[ant] >= failTH)
			{
				RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
			}
		}
	}
	
}

void RficDriver_RxDcActivateDac2ChipCal(uint8 ant, int32 Pgc2Gain, uint8 intBw, uint8 numTries, int32 failTH, uint32 corrSamples)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	int32 dcLevelI;
	int32 dcLevelQ;
	
	uint16 pollingMask;
	uint8 tries;
	bool success;

	pollingMask = RF_ANTENNA0_FCSI_PGC2DCOCTL_RDY2START_DCO_I__SMSK | RF_ANTENNA0_FCSI_PGC2DCOCTL_RDY2START_DCO_Q__SMSK;
	
	success = FALSE;
	tries = 0;

	while ((success == FALSE) && (tries < numTries))
	{
		rfic_set_dcintegrator(intBw, intBw);
		pollRegField(RF_ANTENNA0_FCSI_PGC2DCOCTL, pollingMask, pollingMask);
		rfic_start_dccancel2();
		pollRegField(RF_ANTENNA0_FCSI_PGC2DCOCTL, pollingMask, pollingMask);
		rfic_stop_dccancel2();
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		dcLevelI = Abs(results[ant].II) / corrSamples;
		dcLevelI >>= RXDC_CORRELATOR_SHIFT_FACTOR;
		dcLevelQ = Abs(results[ant].QQ) / corrSamples;
		dcLevelQ >>= RXDC_CORRELATOR_SHIFT_FACTOR;
		if ((dcLevelI < failTH) && (dcLevelQ < failTH))
		{
			success = TRUE;
		}
		tries++;
	}
	if (success == FALSE)
	{
		if (dcLevelI >= failTH)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
		}
		if (dcLevelQ >= failTH)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
		}
	}
}

void RficDriver_RxDcActivateDac0ChipCal(uint8 ant, RficRxGains_t* rxGains, uint8 intBw,int32 failTH, uint8 numTries, uint32 corrSamples)
{
	CorrRes_t results[MAX_NUM_OF_ANTENNAS];
	uint32 dcLevelI;
	uint32 dcLevelQ;
	uint16 pollingMask;
	uint16 txBand;
	uint16 rxBand;
	uint16 bw;
	uint8  tries;
	bool   success;

	bw = HDK_getChannelWidth();
	rxBand = RficDriver_GetRxBand(ant);
	txBand = RficDriver_GetTxBand();
	
	rfic_set_rxband(1 << ant, rxBand);
	rfic_set_txband(1 << ant, txBand);
	rfic_set_rfbandwidth(1 << ant, 20 << bw);
	rfic_rxon(1 << ant);

	pollingMask = RF_ANTENNA0_FCSI_RXMIXERDCOCTL_RDY2START_DCO_I__SMSK | RF_ANTENNA0_FCSI_RXMIXERDCOCTL_RDY2START_DCO_Q__SMSK;

	success = FALSE;
	tries = 0;

	while ((success == FALSE) && (tries < numTries))
	{
		rfic_set_dcintegrator(intBw, intBw);
		pollRegField(RF_ANTENNA0_FCSI_RXMIXERDCOCTL, pollingMask, pollingMask);
		rfic_start_dccancel0();
		pollRegField(RF_ANTENNA0_FCSI_RXMIXERDCOCTL, pollingMask, pollingMask);
		rfic_stop_dccancel0();
		PhyCalDrv_CorrMeasureAll(results, DELAY_AFTER_PHY_RFIC_CHANGE, FALSE);
		dcLevelI = Abs(results[ant].II) / corrSamples;
		dcLevelI >>= RXDC_CORRELATOR_SHIFT_FACTOR;
		dcLevelQ = Abs(results[ant].QQ) / corrSamples;
		dcLevelQ >>= RXDC_CORRELATOR_SHIFT_FACTOR;
		if ((dcLevelI < failTH) && (dcLevelQ < failTH))
		{
			success = TRUE;
		}
		tries++;
	}

	if (success == FALSE)
	{
		if (dcLevelI >= failTH)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_I, CAL_STATUS_FAIL);
		}
		if (dcLevelQ >= failTH)
		{
			RxDcOffset_SetStatus(ant, RFIC_PATH_TYPE_Q, CAL_STATUS_FAIL);
		}
	}
	
}

void RficDriver_RxDcStopDcCal()
{
	rfic_stop_dc_clock();
}

void RficDriver_RxDcInitBW(uint8 bw)
{
}

void RficDriver_SetDCcalType(uint8 calType)
{
	uint16 regVal[MAX_NUM_OF_ANTENNAS];
	uint16 val;
	uint8 ant;

	val = 0;
	if (calType == RXDC_CAL_MODE_SW) //SW calibrated special gains
	{
		val = RF_ANTENNA0_FCSI_SR3_FORCE_DAC0_1__SVAL | RF_ANTENNA0_FCSI_SR3_FORCE_DAC2_1__SVAL;
	}
	else
	{
		val = RF_ANTENNA0_FCSI_SR3_FORCE_DAC0_0__SVAL | RF_ANTENNA0_FCSI_SR3_FORCE_DAC2_0__SVAL;
	}
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		regVal[ant] = val;
	}
	RficDriver_WriteMaskRegister(RF_ANTENNA0_FCSI_SR3, regVal, (RF_ANTENNA0_FCSI_SR3_FORCE_DAC0__SMSK | RF_ANTENNA0_FCSI_SR3_FORCE_DAC2__SMSK));
}

bool RficDriver_readDigitalCorrectionfromPhy(){
	RegPhyRxAgcPage1PhyRxtdAgcAcc1F7_u digcancl1;
	RegPhyRxTdPhyRxtdReg2A6_u digcancl2and3;
	uint32 dig1on;
	uint32 dig2on;
	uint32 dig3on;
	RegAccess_Read(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1F7, &digcancl1.val);
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG2A6, &digcancl2and3.val);
	dig1on = digcancl1.bitFields.agcDcCancellationMode;
	dig2on = digcancl2and3.bitFields.rxDcRadarEnable;
	dig3on = digcancl2and3.bitFields.rxDcRadarHwSelect;
	if (dig1on && dig2on && dig3on)
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}


bool RficDriver_writeDigitalCorrectionToPhy(uint32 dcCancellationword){
	uint32 digcancellation1;
	uint32 digcancellation2;
	uint32 digcancellation3;
	RegPhyRxAgcPage1PhyRxtdAgcAcc1F7_u digcancl1;
	RegPhyRxTdPhyRxtdReg2A6_u digcancl2and3;

	RegAccess_Read(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1F7, &digcancl1.val);
	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG2A6, &digcancl2and3.val);
	
	digcancellation1 = (dcCancellationword & RXDC_DIG_CORRECTION1_MASK) >> RXDC_DIG_CORRECTION1_SHIFT;
	digcancellation2 = (dcCancellationword & RXDC_DIG_CORRECTION2_MASK) >> RXDC_DIG_CORRECTION2_SHIFT;
	digcancellation3 = (dcCancellationword & RXDC_DIG_CORRECTION3_MASK) >> RXDC_DIG_CORRECTION3_SHIFT;
	
	digcancl1.bitFields.agcDcCancellationMode = digcancellation1;
	digcancl2and3.bitFields.rxDcRadarEnable = digcancellation2;
	digcancl2and3.bitFields.rxDcRadarHwSelect = digcancellation3;
	
	RegAccess_Write(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1F7, digcancl1.val);
	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG2A6, digcancl2and3.val);
	
	return TRUE;
}

bool RficDriver_writeDcModeToPhy(uint32 dcMode){
	RegPhyRxAgcPage1PhyRxtdAgcAcc14C_u ModeToPhy;
	uint32 addr;
	addr = REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC14C;

	RegAccess_Read(addr, &ModeToPhy.val);
	
	ModeToPhy.bitFields.dcMode = dcMode;

	RegAccess_Write(addr, ModeToPhy.val);

	return TRUE;
}



bool RficDriver_SetDCspecialGainTo1stTable(uint8 index, uint8 lnaIndex, uint8 pgc1index, uint8 pgc2index)
{
	uint32 addr;
	uint32 mixedModeConfig0;

	addr = REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A8 + (index * 4);
	mixedModeConfig0=(lnaIndex<<RXDC_SPECIAL_GAIN_TABLE1_LNA_SHIFT)|(pgc1index<<RXDC_SPECIAL_GAIN_TABLE1_PGC1_SHIFT)|(pgc2index<<RXDC_SPECIAL_GAIN_TABLE1_PGC2_SHIFT);

	RegAccess_Write(addr, mixedModeConfig0);
	return TRUE;
}

bool RficDriver_SetDCspecialGainTo2ndTable(uint8 lnaIndex, uint8 pgc2index, uint8 pgc1start, uint8 pgc1end)
{
	RegPhyRxAgcPage1PhyRxtdAgcAcc1A6_u regLnaPgc2;
	RegPhyRxAgcPage1PhyRxtdAgcAcc1A7_u regPgc1;
	regLnaPgc2.bitFields.mixedModeRadarLnaIndex=lnaIndex;
	regLnaPgc2.bitFields.mixedModeRadarPgc2Index=pgc2index;
	regPgc1.bitFields.mixedModeRadarPgc1IndexStart=pgc1start;
	regPgc1.bitFields.mixedModeRadarPgc1IndexEnd=pgc1end;

	RegAccess_Write(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A6, regLnaPgc2.val);
	RegAccess_Write(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A7, regPgc1.val);

	return TRUE;
}

bool RficDriver_GetDCspecialGainFrom1stTable(uint8 index, uint8 *lnaIndex, uint8 *pgc1index, uint8 *pgc2index)
{
	RegPhyRxAgcPage1PhyRxtdAgcAcc1A8_u regVal;
	uint32 addr;

	addr = REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A8 + (index * 4);
	RegAccess_Read(addr, &regVal.val);

	*lnaIndex = (regVal.bitFields.mixedModeConfig0 & RXDC_SPECIAL_GAIN_TABLE1_LNA_MASK) >> RXDC_SPECIAL_GAIN_TABLE1_LNA_SHIFT;
	*pgc1index = (regVal.bitFields.mixedModeConfig0 & RXDC_SPECIAL_GAIN_TABLE1_PGC1_MASK) >> RXDC_SPECIAL_GAIN_TABLE1_PGC1_SHIFT;
	*pgc2index = (regVal.bitFields.mixedModeConfig0 & RXDC_SPECIAL_GAIN_TABLE1_PGC2_MASK) >> RXDC_SPECIAL_GAIN_TABLE1_PGC2_SHIFT;

	return TRUE;
}

bool RficDriver_GetDCspecialGainFrom2ndTable(uint8 *lnaIndex, uint8 *pgc2index, uint8* pgc1start, uint8* pgc1end)
{
	RegPhyRxAgcPage1PhyRxtdAgcAcc1A6_u regLnaPgc2;
	RegPhyRxAgcPage1PhyRxtdAgcAcc1A7_u regPgc1;

	RegAccess_Read(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A6, &regLnaPgc2.val);
	RegAccess_Read(REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC1A7, &regPgc1.val);

	*lnaIndex = regLnaPgc2.bitFields.mixedModeRadarLnaIndex;
	*pgc2index = regLnaPgc2.bitFields.mixedModeRadarPgc2Index;
	*pgc1start = regPgc1.bitFields.mixedModeRadarPgc1IndexStart;
	*pgc1end = regPgc1.bitFields.mixedModeRadarPgc1IndexEnd;

	return TRUE;
}

void RficDriver_DCwriteSpecialGain(CorrRes_t *results, uint8 gainIndex, uint8 bw)
{
	uint32 addr;
	uint32 regVal;
	uint8 ant;

	addr = RXDC_TABLE1_START_ADDRESS + (RXDC_TABLE1_BYTE_WIDTH * (((MAXIMUM_BANDWIDTHS_GEN5 * RXDC_TABLE1_SECTION_LEN) + (bw * RXDC_TABLE1_CORRECTION_LEN)) + gainIndex));

	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		regVal = ((-results[ant].II) & RXDC_SPECIAL_GAIN_RESULT_MASK) | (((-results[ant].QQ) & RXDC_SPECIAL_GAIN_RESULT_MASK) << RXDC_SPECIAL_GAIN_RESULT_SHIFT);
		RegAccess_Write(addr, regVal);
		addr += RXDC_TABLE1_BYTE_WIDTH * (MAXIMUM_BANDWIDTHS_GEN5 * (RXDC_TABLE1_SECTION_LEN + RXDC_TABLE1_CORRECTION_LEN));
	}
}

void pollRegField(uint32 regAddr, uint32 regMask, uint32 pollVal)
{
	uint8 ant;
	uint8 antMask;
	uint8 curAntMask;
	uint16 readVal[MAX_NUM_OF_ANTENNAS];

	antMask = Hdk_GetTxAntMask();
	
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		curAntMask = 1 << ant;
		if ((curAntMask & antMask) != FALSE)
		{
			do
			{
				RFIC_FCSI_Read(curAntMask, regAddr, readVal);
			} while ((readVal[ant] & regMask) != pollVal);
		}
	}
}

uint32 setLnaGainLUTCounter =0;
void RficDriver_SetLnaGainLUT()
{
	uint8 ant, valsIndex;
	uint16 rfVal[MAX_NUM_OF_ANTENNAS];
	uint32 startAddress, val;

	for (valsIndex = 0; valsIndex < 3; valsIndex++)
	{
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			startAddress = LNA_CONVERSIONS_START_ADDRESS + (ant * LNA_CONVERSIONS_ANT_OFFSET) + (valsIndex << 3);
			if(setLnaGainLUTCounter==0)
			{
				RegAccess_Read(startAddress, &val);
				LNA_LUT[ant][valsIndex << 1] = (uint8)val;
			
				RegAccess_Read(startAddress + 4, &val);
				LNA_LUT[ant][(valsIndex << 1)+1] = (uint8)val;
			}
			rfVal[ant] = (LNA_LUT[ant][valsIndex << 1] << LNA_GAIN0_CONVERSION_SHIFT) | (LNA_LUT[ant][(valsIndex << 1)+1] << LNA_GAIN1_CONVERSION_SHIFT);
		}
		RficDriver_WriteMaskRegister(RF_ANTENNA0_FCSI_FECTL01 + valsIndex, rfVal, LNA_GAIN01_CONVERSION_MASK);
	}
	setLnaGainLUTCounter++;
}

void RficDriver_SetLnaConversionsTable(Antenna_e ant, uint8 index, uint8 val)
{
	uint32 address = LNA_CONVERSIONS_START_ADDRESS + (ant * LNA_CONVERSIONS_ANT_OFFSET) + (index << 2);
	RegAccess_Write(address, val);
}

uint8 RficDriver_GetLnaGain(Antenna_e ant, uint8 index)
{
	return LNA_LUT[ant][index];
}

void RficDriver_OverrideLnaConversionsTable(Antenna_e ant, uint8 val)
{
	uint8 lnaLutIndex;
	
	for (lnaLutIndex = 0; lnaLutIndex < HDK_LNA_INDEX_NUM_OF_INDEXES; lnaLutIndex++)
	{
		RficDriver_SetLnaConversionsTable(ant, lnaLutIndex,  val);
	}
}

void RficDriver_RestoreLnaConversionsTable(Antenna_e ant)
{
	uint8 lnaLutIndex;
	
	for (lnaLutIndex = 0; lnaLutIndex < HDK_LNA_INDEX_NUM_OF_INDEXES; lnaLutIndex++)
	{
		RficDriver_SetLnaConversionsTable(ant, lnaLutIndex, RficDriver_GetLnaGain(ant, lnaLutIndex));
	}
}

/********************************************************************************
Function Name:  RficCalDrv_RestoreOperational:

Description:  Restores the RFIC to operational state:
			1) Rfic rx off.
			2) disable LnaNoise.
********************************************************************************/
void RficCalDrv_RestoreOperational(uint8 rxAntMask)
{
	rfic_rxoff(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5);
	
	rfic_disable_lnanoise(rxAntMask);

}

/*****************************************************************************
*    Function Name:  RficDriver_EnableLnaNise(uint8_t AntMsk)
*    Description:    set RFIC + FEM config
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_EnableLnaNoise(uint8_t AntMsk)
{	
	rfic_enable_lnanoise(AntMsk);
}

/*****************************************************************************
*    Function Name:  RficDriver_EnableLnaNise(uint8_t AntMsk)
*    Description:    set RFIC + FEM config
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DisableLnaNoise(uint8_t AntMsk)
{	
	rfic_disable_lnanoise(AntMsk);
}

/********************************************************************************
*    Function Name: doAxisIteration:
*    Description: Find equalizer parameter (DAC value) of lowest Lo energy 
*    Parameters:     see RficDriver_API.c
*    Returns:        see RficDriver_API.c
*    Remarks:        see RficDriver_API.c
*    Restrictions:   see RficDriver_API.c
********************************************************************************/
void doAxisIteration(int8 inLo, int8 inHi, int8 inOtherAxisVal, bool inIsQ, RficLoopbackAntennas_t *inAntInfo_p, uint8 inTpcIndex, TxLO_elements_t* outElements_p, int8 *outIndex_p,uint32 isInternalPA,int8 IterationNumber,ClbrType_e inCalibType)

{
	int32 energy_lo, energy_hi;
	int8 i;
	//minimum measured energy & index during I/Q min search calibration
	int32	minEnergyMeasured = LO_INFINITE_ENERGY;	// "infinite" energy
	int8	minIndexMeasured= 0;					// arbitrary value   

    if (inIsQ) 
	{
		measureEnergy(inTpcIndex, inOtherAxisVal, inLo, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
		//check and update if the currently measured energy gives the minimum energy until now 
		updateMinimumEnergy(inLo, energy_lo, &minEnergyMeasured, &minIndexMeasured);
		
		measureEnergy(inTpcIndex, inOtherAxisVal, inHi, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
		//check and update if the currently measured energy gives the minimum energy until now 
		updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
	} 
	else
	{
		measureEnergy(inTpcIndex, inLo, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
		//check and update if the currently measured energy gives the minimum energy until now 
		updateMinimumEnergy(inLo, energy_lo, &minEnergyMeasured, &minIndexMeasured);
		
		measureEnergy(inTpcIndex, inHi, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
		//check and update if the currently measured energy gives the minimum energy until now 
		updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
	}
	//ILOG9_DDDDD("ant: %d. inHi: %d inLow %d energy high %d energy low: %d",inAntInfo_p->RfCalibAnt, inHi,inLo,energy_hi,energy_lo);
	// Binary search
	for (i = 0; i < outElements_p->params.binaryMaxSteps; i++)
    {
		if (energy_lo < energy_hi) 
        {
        	if((inHi - inLo) > 1) //range can still be split - continue binary search
			{
				outElements_p->params.binaryPivot = inHi;
				//ILOG9_D("doAxisIteration() stepping in GetTxloPivot() outElements_p->params.binaryPivot : %d ",outElements_p->params.binaryPivot);
				GetTxloPivot(outElements_p, inLo, i);
				
				inHi = outElements_p->params.binaryPivot;
				//ILOG9_D(" calculating pivot from Hi direction : %d ",inHi);
				if (inIsQ) 
				{
					measureEnergy(inTpcIndex, inOtherAxisVal, inHi, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
					//check and update if the currently measured energy gives the minimum energy until now 
					updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
				}
				else
				{
					measureEnergy(inTpcIndex, inHi, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
					//check and update if the currently measured energy gives the minimum energy until now 
					updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
				}
        	}
		} 
        else 
        {
        	if((inHi - inLo) > 1) //range can still be split - continue binary search
			{
				outElements_p->params.binaryPivot = inLo;
				//ILOG9_D(" outElements_p->params.binaryPivot = %d ",outElements_p->params.binaryPivot);
				GetTxloPivot(outElements_p, inHi,i);
				
				inLo = outElements_p->params.binaryPivot;

				//ILOG9_D(" calculating pivot from Lo direction : %d ",inLo);
				
				if (inIsQ)
				{
					measureEnergy(inTpcIndex, inOtherAxisVal, inLo, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
					//check and update if the currently measured energy gives the minimum energy until now 
					updateMinimumEnergy(inLo, energy_lo, &minEnergyMeasured, &minIndexMeasured);
				}
				else 
				{
					measureEnergy(inTpcIndex, inLo, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
					//check and update if the currently measured energy gives the minimum energy until now 
					updateMinimumEnergy(inLo, energy_lo, &minEnergyMeasured, &minIndexMeasured);
				}
        	}
		}		
		//ILOG9_DDDDD(" ant: %d. inHi: %d inLow %d energy high %d energy low: %d",inAntInfo_p->RfCalibAnt, inHi,inLo,energy_hi,energy_lo);
	}
	// Linear search - fine tuning
	for (i = 0; i < outElements_p->params.linearMaxSteps; i++)
	{
		if (energy_lo < energy_hi) 
        {
			inHi--;
			if (inIsQ) 
			{
				measureEnergy(inTpcIndex, inOtherAxisVal, inHi, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
				//check and update if the currently measured energy gives the minimum energy until now 
				updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
			}
			else
			{
				measureEnergy(inTpcIndex, inHi, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_hi);
				//check and update if the currently measured energy gives the minimum energy until now 
				updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
			}
		}
		else
		{
			inLo++;
			if (inIsQ) 
			{
				measureEnergy(inTpcIndex, inOtherAxisVal, inHi, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
				//check and update if the currently measured energy gives the minimum energy until now 
				updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
			}
			else
			{
				measureEnergy(inTpcIndex, inHi, inOtherAxisVal, outElements_p->params.calibBin, inAntInfo_p, outElements_p, &energy_lo);
				//check and update if the currently measured energy gives the minimum energy until now 
				updateMinimumEnergy(inHi, energy_hi, &minEnergyMeasured, &minIndexMeasured);
			}
		}
	}
	*outIndex_p = minIndexMeasured;
	
	return ;
}

/********************************************************************************
 measureEnergy:

Description:
------------
Measure Lo energy level of (a+jb) equalizer (using goertzel).

Input:
-----
1. inTpcIndex
2. I value
3. Q value
4. inBin
5. inRfCalibAnt

Output:
-------
outElements_p->status
outEnergy_p - measured energy (linear not log)
********************************************************************************/
static void measureEnergy(uint8 inTpcIndex, int8 inValueI ,int8 inValueQ, int8 inBin, RficLoopbackAntennas_t *inAntInfo_p, TxLO_elements_t* outElements_p, int32 *outEnergy_p)
{
	GoertzelRes_t	Results[TOTAL_GOERTZELS];
	int8 inValues[RFIC_PATH_TYPE_MAX];
	uint8 goertzelNum, goertzelMask;
	
	goertzelNum = inAntInfo_p->RxLoopbackAnt & 2; // Use Goertzel0 for Rx ants 0 and 1, and Goertzel2 (in B11) for Rx ants 2 and 3
	goertzelMask = 1 << goertzelNum;

	/*set I and Q parameters*/
	inValues[RFIC_PATH_TYPE_I] = inValueI;
	inValues[RFIC_PATH_TYPE_Q] = inValueQ;
	RficDriver_SetLoDACs(inAntInfo_p->RfCalibAnt,(RficTxGainTpcIndex_e)inTpcIndex,inValues);
    
	PhyCalDrv_GoertzelMeasure(goertzelMask, Results,(int16)inBin,DELAY_AFTER_PHY_RFIC_CHANGE);

	Results[goertzelNum].Res_I_Real >>= LO_ENERGY_RESOLUTION; 
	Results[goertzelNum].Res_Q_Real >>= LO_ENERGY_RESOLUTION;
	*outEnergy_p = Results[goertzelNum].Res_I_Real*Results[goertzelNum].Res_I_Real+Results[goertzelNum].Res_Q_Real*Results[goertzelNum].Res_Q_Real;

	return ;
}

/********************************************************************************
 updateMinimumEnergy:	

Description:
------------
Check if measured energy is minimum in the binary search and update accordingly 

Input:
-----
inValue	- I/Q value for which inEnergy was measured
inEnergy - currently measured energy

Output:
-------
outMinEnergy_p - minimum energy measured until now
outMinIndex_p - index with minimum energy until now

********************************************************************************/
static void updateMinimumEnergy(int8 inValue, int32 inEnergy, int32 *outMinEnergy_p, int8 *outMinIndex_p)
{
	//check if the currently measured energy gives the minimum energy until now 
	if (inEnergy < *outMinEnergy_p)
	{
		*outMinEnergy_p = inEnergy;
		*outMinIndex_p = inValue;
	}
}

void RficDriver_SetCalibrationData(K_MSG * psMsg)
{
	rfic_calibration_data_t* params;
	uint8 index;
	uint8 numRegsInCis;
	uint8 regNum;

	params = (rfic_calibration_data_t*)pK_MSG_DATA(psMsg);
	numRegsInCis = params->rficCisSize >> 1; //each register is of 16bit so for each one there are 2 bytes in CIS

	for (index = 0; index < numRegsInCis; index++)
	{
		regNum = rficCisIndexToReg[index];
		rficCalibrationData[regNum] = params->rficCis[index << 1] | (params->rficCis[(index << 1) + 1] << 8);
	}
	rficCalibrationData[12] = 0x4210;
	rficCalibrationData[13] = 0x4210;
	rficCalLoaded = 1;
}

bool RficDriver_isSsbLoop(void)
{
	uint16 regVals[MAX_NUM_OF_ANTENNAS];
	bool isSsbLoop = TRUE;
	
	RficDriver_ReadRegister(RF_ANTENNA3_FCSI_EFUSE15, regVals);
	if (RF_ANTENNA3_FCSI_EFUSE15_VERSIONNUMBER__GET(regVals[3]) == 0)
	{
		isSsbLoop = FALSE;
	}	

	return isSsbLoop;
}

void RficDriver_ReadTemperature(int32* temperature)
{
	PhyDrv_StopRisc();
	
	tempSensorStartClock();

	tempSensorRead(temperature);

	tempSensorStopClock();

	PhyDrv_StartRisc();
}

void RficDriver_SelectBandFilterB(int8 setAPHighLowFilterBank, uint8 driver_fw, uint8 low_chan_num)
{
}

void RficDriver_handleIRETable_B(void)
{
}

uint8 readDcModeFromPhy(){
	RegPhyRxAgcPage1PhyRxtdAgcAcc14C_u ModeFromPhy;
	uint32 addr;
	uint8 dc_mode;
	addr=REG_PHY_RX_AGC_PAGE_1_PHY_RXTD_AGC_ACC14C;
	RegAccess_Read(addr, &ModeFromPhy.val);
	dc_mode=ModeFromPhy.bitFields.dcMode;
	return dc_mode;
}


/*****************************************************************************
*    Function Name:  set_lo_frequencies(uint8 freq, uint8 bw)
*    Description:    set lo frequencies {position 0 = frequency of bw 80MHz, pos 1 =freq Bw 40MHz, pos 2 =freq Bw 20MHz
*    Parameters:     BW{80MHz= 0, 40MHz =1, 20MHz =2}
******************************************************************************/
void RficDriver_set_lo_frequencies(void)
{ 			
	rfic_set_lo_frequencies (HDK_GetLoFrequency(RF_BW_80),HDK_GetLoFrequency(RF_BW_40),HDK_GetLoFrequency(RF_BW_20),HDK_GetLoFrequency(3),HDK_GetLoFrequency(4),HDK_GetLoFrequency(5));			
}

void RficDriver_set_rfchannel(uint8_t AntMsk,  uint8 setChnlBw)
{
uint8_t ch_bit;
	
			if (setChnlBw == 80)
				ch_bit = 0;
			else if (setChnlBw == 40)
				ch_bit = 1;
			else if (setChnlBw == 20)
				ch_bit = 2;
			else
				ch_bit = 0;
	RFIC_FCSI_RMW(AntMsk, RF_ANTENNA0_FCSI_SR0,
				 RF_ANTENNA0_FCSI_SR0_SEL_CH__ISMSK,
				 ch_bit << RF_ANTENNA0_FCSI_SR0_SEL_CH__POS);
}
void RficDriver_set_rxdc(uint8_t AntMsk,  uint8 setChnlBw)
{
	
	uint8_t rxdc_bit;
		
				if (setChnlBw == 80)
					rxdc_bit = 0;
				else if (setChnlBw == 40)
					rxdc_bit = 1;
				else if (setChnlBw == 20)
					rxdc_bit = 2;
				else
					rxdc_bit = 0;
		RFIC_FCSI_RMW(AntMsk, RF_ANTENNA0_FCSI_SR0,
					 RF_ANTENNA0_FCSI_SR0_SEL_RXDC__ISMSK,
					 rxdc_bit << RF_ANTENNA0_FCSI_SR0_SEL_RXDC__POS);
}
void RficDriver_InitBwChanDC(void)
{
	uint32 val;
		val = 0x4B00;
		RegAccess_Write(SET_BW_CH_DC_20M,val );
		val = 0x2600;
		RegAccess_Write(SET_BW_CH_DC_40M,val );
		val = 0;
		RegAccess_Write(SET_BW_CH_DC_80M,val );
}



static void GetTxloPivot(TxLO_elements_t* outElements_p ,int8 fixedcntrl, uint8 IterationNumber)
{

	int8 i=0 ;
	uint8 MaxIteration = outElements_p->params.binaryMaxSteps;
	int8 pivot = outElements_p->params.binaryPivot;
	uint8 PivotDiv = 2;
	//ILOG9_D(" GetTxloPivot() pivot : %d ",pivot);
	//ILOG9_D(" GetTxloPivot() Maxiteration : %d ",MaxIteration);

	if (IterationNumber > MaxIteration >> 1) 
		{
			PivotDiv = 1; 
		}
	else
		{
			PivotDiv = 2;
		}

	//ILOG9_D(" GetTxloPivot() PivotDiv : %d ", PivotDiv);

	for (i=0 ; i < PivotDiv ; i++)

		{
			fixedcntrl = (pivot + fixedcntrl) >> 1;
			//ILOG9_DD(" GetTxloPivot() itiration: %d fixedcntrl: %d",i, fixedcntrl);
		}

	outElements_p->params.binaryPivot = fixedcntrl;
	//ILOG9_D(" GetTxloPivot() outElements_p->params.binaryPivot: %d ",outElements_p->params.binaryPivot);

}

void RficDriver_AssociateAntennasToMyBand(uint8 antMask)
{

}





