/*****************************************************************************
*   COPYRIGHT:
*       (C) Copyright 2010 of Lantiq
*       All rights are strictly reserved. Reproduction or divulging in any
*       form whatsoever is not permitted without written authority from the
*       copyright owner. Issued by Lantiq
*   MODULE NAME:
*       RficDriver.c
*   DESCRIPTION:
*       Driver for the RFIC of AR10 platform
*   AUTHOR:
*       Ariel Groenteman
*   DATE:
*       July 2011
*****************************************************************************/

/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "mt_sysrst.h"
#include "stringLibApi.h"
#include "HdkGlobalDefs.h"
#include "RficDriver_API.h"
#include "Indirect_API.h"
#include "lib_abb_api.h"
#include "lib_wrx654_api.h"
#include "lib_wrx654_sequences.h"
#include "interface_wrx654.h"
#include "wrx654_rf_antenna_fcsi_regpkg.h"
#include "RficLoopBackTypes.h"
#include "RFICDefines.h"
#include "RficCommon.h"
#include "ErrorHandler_Api.h"
#include "PhyDriver_API.h"
#include "Afe_API.h"
#ifndef ENET_INC_ARCH_WAVE600D2
#include "PhyRxAgcIfRegs.h"
#endif
#include "PhyRxTdRegs.h"
#ifdef ENET_INC_ARCH_WAVE600D2	
#include "PhyRxtdAnt0Regs.h"
#else
#include "PhyRxTdAnt0Regs.h"
#endif
#include "PSD.h"
#include "CalibrationHandler.h"
#include "CoC_Api.h"
#include "MT_Math.h"
#include "RxDcOffsetClbrHndlr.h"
#include "loggerAPI.h"
#include "fast_mem_psd2mips.h"
#include "Dut_Api.h"
#include "Hdk_Api.h"
#include "phy_lib.h"
#ifdef HDK_RF_EMULATOR
#include "RfEmulatorRegs.h"
#endif
#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 0

/******************************************************************************/
/***						  Constants										***/
/******************************************************************************/

#define RFIC_NUM_OF_SHORT_REGS           4
#define RFIC_LONG_FECTL_REG_NUM			 4
#define RFIC_NUM_OF_LONG_REGS            64
#define RFIC_NUM_OF_REGS                 (RFIC_NUM_OF_SHORT_REGS + RFIC_NUM_OF_LONG_REGS)
#define RFIC_FIRST_SHORT_REG_ADDR        (REG_RFIC_S0_SHORT_0)
#define RFIC_LAST_SHORT_REG_ADDR         (REG_RFIC_S3_SHORT_3)
#define RFIC_FIRST_LONG_REG_ADDR         (REG_RFIC_L0_RSSI_REG1)
#define RFIC_LAST_LONG_REG_ADDR          (REG_RFIC_L63_RF_PLL_8)

#define RFIC_STANDBY_MODE_VALUE          0
#define RFIC_STANDBY_MODE_MASK           (0xFFFF)

#define RFIC_CENT_BIAS_COMP_OUT_VAL_PASS 1

/* RSSI DC offset calibration step number min/Max values */
#define RFIC_RSSI_DC_CAL_MIN_STEP_NUM    0 /* regular operation */
#define RFIC_RSSI_DC_CAL_MAX_STEP_NUM    5 /* cal step 5 */

#define RFIC_RSSI_DC_CAL_INIT_S0_VALUE   (0x0002) /* Set antennas to standby mode */
#define RFIC_RSSI_DC_CAL_INIT_S0_MASK    (RFIC_DEFAULT_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L2_VALUE   (0x0040) /* Short RSSI input to GND */
#define RFIC_RSSI_DC_CAL_INIT_L2_MASK    (REG_RFIC_L2_SHORT_INPUT_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L4_VALUE   (0x0010) /* Set ANA-MUX to RSSI and to step 0 (0..4) */
#define RFIC_RSSI_DC_CAL_INIT_L4_MASK    (REG_RFIC_L4_SEL_IN_MASK|\
                                          REG_RFIC_L4_SEL_RSSI_MASK)

#define RFIC_RSSI_DC_CAL_INIT_L5_VALUE   (0xD100) /* ANA-MUX setting for RSSI DC calibration */
#define RFIC_RSSI_DC_CAL_INIT_L5_MASK    (REG_RFIC_L5_EN_ANAMUX_MASK|\
                                          REG_RFIC_L5_SEL_OFFS_MASK|\
                                          REG_RFIC_L5_SEL_RB_MASK|\
                                          REG_RFIC_L5_SEL_FB_MASK)

#define RFIC_RSSI_DC_CAL_STEP_L2_MASK    (REG_RFIC_L2_CAL_STEP_MASK|\
                                          REG_RFIC_L2_CAL_Q_MASK|\
                                          REG_RFIC_L2_CAL_I_MASK)

/* RSSI DC offset calibration Store all DAC values - register L2 mask */
#define RFIC_RSSI_DC_CAL_STORE_ALL_REG_L2_MASK (REG_RFIC_L2_CAL_STEP_MASK|REG_RFIC_L2_SHORT_INPUT_MASK| \
                                                REG_RFIC_L2_CAL_Q_MASK|REG_RFIC_L2_CAL_I_SHIFT| \
                                                REG_RFIC_L2_Q_RSSI_CAL5_MASK|REG_RFIC_L2_I_RSSI_CAL5_MASK)

#define RFIC_VCO_HIGH_BAND_RANGE_SELECT  (10000)

/* XTAL (DCXO) amplitude change prevention constants */
#define RFIC_SET_XTAL_L50_VALUE          (0x0008) /* Set DCXO bias mode and rest to default, enable FSYS output clock */
#define RFIC_SET_XTAL_L51_ONLY_BG_VALUE  (0x0001) /* Enable Bandgap */
#define RFIC_SET_XTAL_L51_FULL_VALUE     (0x1841) /* Enable Bandgap, Enable Central bias V2I const and default Bgp and temp trimming, default iconst */
#define RFIC_SET_XTAL_L52_VALUE          (0x8617) /* Enable Central bias V2I ref, PLL reset, soft reset released, Set DCXO LDO to default */
#define RFIC_SET_XTAL_L53_VALUE          (0x1948) /* Set FCSI LDO to default, DCXO current control default, FSYS strength default */

#define RFE_CTRL_BASE_ADDRESS (0x20dc0)
#define RFIC_RFFE_SHIFT_NO_LNA (2)
#define RFIC_RFFE_MASK (0x1f << RFIC_RFFE_SHIFT_NO_LNA)

#define RF_CHANNEL_WIDTH_MHZ	5
#define RF_SPECTRUM_20_SHIFT	0
#define RF_SPECTRUM_40_SHIFT	10
#define RF_SPECTRUM_80_SHIFT	30
#define RF_SPECTRUM_160_SHIFT	70

#define SR2_FECTL_MASK  (WRX654_RF_ANTENNA_FCSI_SR2_FECTL0__SMSK | \
				    	WRX654_RF_ANTENNA_FCSI_SR2_FECTL1__SMSK  | \
				    	WRX654_RF_ANTENNA_FCSI_SR2_FECTL2__SMSK  | \
						WRX654_RF_ANTENNA_FCSI_SR2_FECTL3__SMSK  | \
						WRX654_RF_ANTENNA_FCSI_SR2_FECTL4__SMSK  | \
						WRX654_RF_ANTENNA_FCSI_SR2_FECTL5__SMSK)
					
#define SR2_FECTL_SHIFT (WRX654_RF_ANTENNA_FCSI_SR2_FECTL0__POS)

/******************************************************************************/
/***						Type Definition									***/
/******************************************************************************/ 
/* The structure of the word we send the RFIC (using the Indirect API) */
typedef struct RficProgModel
{
    uint16 Data;
    uint16 Register;
} RficProgModel_t;

typedef struct
{
    uint32 freq:16;
    uint32 word:16;
} RficFreqTuneWord_T;

/******************************************************************************/
/***					Internal Data Structures							***/
/******************************************************************************/

/******************************************************************************/
/***						Public Variables								***/
/******************************************************************************/



/******************************************************************************/
/***						Static Variables								***/
/******************************************************************************/

uint8 bandwidthOffset[MAX_NUM_OF_BW] = {RF_SPECTRUM_20_SHIFT, RF_SPECTRUM_40_SHIFT, RF_SPECTRUM_80_SHIFT, RF_SPECTRUM_160_SHIFT};
IRE_t ireDbSwitchB;
IRE_t ireDbSwitchA;
IRE_t ZwdfsDbSwitch;

/*The internal registers DB - mirror of the RF registers. */ 
static RficRfRegistersDB_t RficDriverRegistersLocalDB[RFIC_NUM_OF_REGS];
static RficPSDRegistersDB_t RficDriverPSD[RFIC_NUM_OF_REGS];
static RficPSDRegistersDB_t RficDriverZwdfsPSD[RFIC_LONG_FECTL_REG_NUM];


#define RFIC_CIS_NUM_REGS	16
uint16 rficCalibrationData[RFIC_CIS_NUM_REGS] = {0};
uint8 rficCalLoaded = 0;
uint8 AAA_InitDone = 0;
uint16 AAA_RficRegs[70][5] = {0};
uint8 rficCisIndexToReg[RFIC_CIS_NUM_REGS] =
{
	2,
	3,
	4,
	5,
	6,
	7,
	8,
	9,
	11,
	14,
	15,
	0,
	1,
	10,
	12,
	13
};

TemperatureSensorParams_t tempSensorParams = {0};

/******************************************************************************/
/***						Macros											***/
/******************************************************************************/
#define SIGN_EXTEND(value, Bits)   if(value  >= (1<< (Bits-1) ) ) value-=(1<<Bits)

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static void readFromRficRegister(uint32 inRfRegAddr, uint16 *outValues);
#ifdef SDL_IGNORE_UNUSED_FUNCTION
static inline void setRxDac1(AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType, uint16 *inRxDacValues);
static inline void setRxDac2(AntennaBitmaps_e inAntBitmap, RficPathTypeIndex_e inPathType, uint16 *inRxDacValues);
static uint8 getBandIndex(uint16 freq, uint8 ant, uint16 bandLUT[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT]);
#endif
static inline void updateDB(uint32 rfRegAddr, uint8 ant, uint16 value);
static void setAMDetectorLoop(uint8 antMask);
static void setRxLpfLoop(uint8 antMask);
static void setTxLpfLoop(uint8 antMask);
static void setRssiLoop(void);
static void writePsdTableFromDB(void);
static void OverwriteZwdfsPsdArray(void);
static void rficSetConfig(uint8 antMask);
static void getRficEfuse(void);
static RficPSDRegistersDB_t * rficGetPsdDB (RfregPsdDb_e dBtag, uint16 arrayIndex, uint16 rowIndex);
static void getTempSensorParams(void);
static void tempSensorStartClock(void);
static void tempSensorRead(int32* temperature);
static void tempSensorStopClock(void);
static uint8 getMyPllId(void);
void RficDriver_set_lo_frequencies(void);
void RficDriver_set_rfchannel(uint8_t AntMsk,  uint8 chanl_bit);
void RficDriver_set_rxdc(uint8_t AntMsk,  uint8 setChnlBw);


void pollRegField(uint32 regAddr, uint32 regMask, uint32 pollVal);

/******************************************************************************/
/***							Debug Section								***/
/******************************************************************************/

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
void RficDriver_Init(void)
{	
#ifdef WRX600_BU_LOGS
	ILOG0_V("RficDriver_Init");
#endif
	// RFIC init
	rfic_central_init();	
	getRficEfuse();
	// write efuse data to central regs only
	rfic_read_efuse_values(1, rficCalibrationData);
}

void RficDriver_BandInit(void)
{
	uint8 antMask = Hdk_GetTxAntMask();	
#ifdef WRX600_BU_LOGS
	ILOG0_DD("RficDriver_BandInit. band %d antMask 0x%x", ConfigurationManager_GetMyBand(), antMask);
#endif
	// RFIC Band init
	rfic_band_init(antMask);	
	rficSetConfig(antMask);
	writePsdTableFromDB();
	// Write efuse data to antennas regs only
	rfic_read_efuse_values(2, rficCalibrationData);
	getTempSensorParams();
	
	AAA_InitDone = 1;
}

static RficPSDRegistersDB_t * rficGetPsdDB (RfregPsdDb_e dBtag, uint16 arrayIndex, uint16 rowIndex)
{
	RficPSDRegistersDB_t * ret = NULL; //KW_FIX_FW_G
	ASSERT(arrayIndex <= RFIC_NUM_OF_REGS);
	
	switch (dBtag)
	{
		case RFIC_DRIVER_PSD_DB:   
		{
			ret = (&RficDriverPSD[arrayIndex]);
			break;
		}
		case RFIC_DRIVER_ZWDFS_PSD_DB:	
		{
			ASSERT(rowIndex <= RFIC_LONG_FECTL_REG_NUM);
			ret = (&RficDriverZwdfsPSD[rowIndex]);
			break;
		}
	
	}
	return ret;
}

static void rficSetConfig(uint8 antMask)
{
	// Config default values. operational values should be overriden by PSD

	// TPC gains
	rfic_config_tpcgain(antMask, -6, -4, -2, 0);
	// Rx Gain
	rfic_config_lnagain(antMask, 3, 2, 1, 0, 0, 0, 0);
	// RSSI 0
	rfic_config_rssi(antMask, 0, 8, 1, 6);	
	// RSSI 1
	rfic_config_rssi(antMask, 1, -4, 2, 3);
	// TSSI
	rfic_config_tssi(antMask, 0, -8, 1);
}

static void getRficEfuse(void)
{
	rfic_get_efuse_values(rficCalibrationData);
	
}

static void getTempSensorParams(void)
{
	tempSensorParams.refTemp = rfic_read_efuse_handlertemp(rficCalibrationData);
	tempSensorParams.refResult = rfic_read_efuse_tempsensorresult(rficCalibrationData);
	if ((tempSensorParams.refTemp == 0) && (tempSensorParams.refResult == 0))
	{		
		tempSensorParams.refTemp = TEMPERATURE_DEFAULT_REF_TEMP_VAL;
		tempSensorParams.refResult = TEMPERATURE_DEFAULT_REF_RESULT_VAL;
	}
	
	tempSensorParams.slope = TEMPERATURE_SLOPE_VAL; // shifted left 11 bits
	tempSensorParams.offset = (tempSensorParams.refTemp << 11) - (tempSensorParams.slope * tempSensorParams.refResult); // shifted left 11 bits
}

/*****************************************************************************
*    Function Name:  RficDriver_GetRfChipID
*    Description:    Reads the revision code of the RFIC chip.
*    Parameters:     outRevId - Here we will put the returned data
*    Returns:        The revision code according to the specific RFIC chip
*    Remarks:        Implements RficDriver_PublicApi_t->GetRevisionID
*                     
*                    registers.
*                    Bits[10:0]    01  01  01     01  001  (default)
*                                  Vx  y   A/B/C  z1  z2   (0x2A9)
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetRfChipID(uint32 *outRevId)
{
    uint16 readVals[FCSI_TOTAL_CHANNELS];

    readFromRficRegister(REG_RFIC_L41_VERSION, readVals);
    *outRevId = ((readVals[ANTENNA_0] & REG_RFIC_L41_VERSION_MASK) >> REG_RFIC_L41_VERSION_SHIFT);
}

/*****************************************************************************
*    Function Name:  RficDriver_UpdateShort0ShadowRegister
*    Description:    
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_UpdateShort0ShadowRegister(AntennaBitmaps_e inListOfAntennas)
{
	UNUSED_PARAM(inListOfAntennas);	
/*RFICTBD: rewrite with wrx654 field defines
	uint8 index;
	uint32 shadow;

	for (index = 0; index < MAX_NUM_OF_ANTENNAS; index++)
	{
		shadow = RficDriverRegistersLocalDB[RF_ANTENNA0_FCSI_SR0].regData[index] & (~(1 << RF_ANTENNA0_FCSI_SR0_RXON__POS));
		shadow |= (inListOfAntennas & 1) << RF_ANTENNA0_FCSI_SR0_RXON__POS;
		RegAccess_Write(RF_FCSI_S0_BACKUP0 + (index << 2), shadow);
		inListOfAntennas >>= 1;
	}
*/	
}

/*****************************************************************************
*    Function Name:  RficDriver_ActivateRxAntennas
*    Description:    Enables the receive antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateRxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	//on bbic
	uint8 rxAntMask = Hdk_GetRxAntMask();
	rfic_rxoff(rxAntMask);
	rfic_rxon((uint8)inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_ActivateTxAntennas
*    Description:    Enables the transmit antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ActivateRxAntennas
*    Restrictions:   None
******************************************************************************/
void RficDriver_ActivateTxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	uint8 txAntMask = Hdk_GetTxAntMask();
	rfic_txoff(txAntMask);
	rfic_txon((uint8)inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_DeactivateRxAntennas
*    Description:    Disables the receive antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to deactivate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DeactivateRxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	rfic_rxoff((uint8)inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_DeactivateTxAntennas
*    Description:    Disables the transmit antennas according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to deactivate.
*                    ant_i <-> bit regInd
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DeactivateTxAntennas(AntennaBitmaps_e inListOfAntennas)
{
	rfic_txoff((uint8)inListOfAntennas);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetAntennas
*    Description:    Set new antennas configuration according to the antenna bitmap.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*                   		 addAntennas - 0x1 = add ants / 0x0 = remove ants
*    Returns:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetAntennas(AntennaBitmaps_e rfAntMask, uint8 afeAntMask, uint8 numOfAnts, PowerSaveStates_e powerSaveState)
{
	uint8 antMaskOff = (~((uint8)rfAntMask)) & 0xF;
	UNUSED_PARAM(numOfAnts);		
#if !defined(HDK_RF_EMULATOR)	
	AFE_EnableDisableAntennas(afeAntMask, powerSaveState);
#endif
	// Turn on /off antennas
	ILOG0_D("rfic_antenna_off(antMaskOff, RFIC_DELAY_BETWEEN_ANTENNA_SET) antMaskOff: %d", antMaskOff);
	ILOG0_D("_set_antenna_fectl_man(antMaskOff,0)  antMaskOff: %d", antMaskOff);
	rfic_antenna_off(antMaskOff, RFIC_DELAY_BETWEEN_ANTENNA_SET);
	rfic_set_antenna_fectl_man(antMaskOff,0);
	rfic_disable_antenna_fectl(rfAntMask);

}

/***************************************************************************
**
** NAME          RficDriver_EnableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : enables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_EnableRFantennas(AntennaBitmaps_e AntMask)
{
	UNUSED_PARAM(AntMask);
}

/***************************************************************************
**
** NAME          LmHdk_DisableRFantennas
**
** PARAMETERS   current and new number of enabled antennas
** DESCRIPTION : disables RF antennas from current number of antennas to new
**                 
****************************************************************************/
void RficDriver_DisableRFantennas(uint32 rfAntMask, uint8 afeAntMask)
{
	UNUSED_PARAM(rfAntMask);
	UNUSED_PARAM(afeAntMask);
}

//void RficDriver_CoCcalib(AntennaBitmaps_e EnabledAntMask, uint8 AntNum, bool isUp)
//{
//	if (isUp == TRUE)
//	{
		// Re-run calibrations when adding antennas
//		ClbrMngr_RunCocCalibrations();
//	}
//}

/*****************************************************************************
*    Function Name:  RficDriver_SetStandbyMode
*    Description:    Sets the RFIC registers related to Rx/Tx to standby mode.
*    Parameters:     inListOfAntennas - Bitmap of the list of antennas to activate.
*    Returns:        None
*    Remarks:        None
*    Restrictions:   For calibration purposes
******************************************************************************/
void RficDriver_SetStandbyMode(AntennaBitmaps_e inListOfAntennas)
{
    writeOneValueToAntList(inListOfAntennas, REG_RFIC_S0_SHORT_0, RFIC_STANDBY_MODE_VALUE, RFIC_STANDBY_MODE_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetLoopback
*    Description:    Configure the loop back for calibrations.
*    Parameters:     inAntenna  - The antenna we want to loop
*                    inLoopType - The type of loop we want to configure in the
*                                 RFIC (RF, BB, AMDET)
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetLoopback
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetLoopback(uint8 antMask, RficLoopbackTypeIndex_e inLoopType)
{
    switch(inLoopType)
    {
        /* AM Detector Loop Back */
        case RFIC_LOOPBACK_TYPE_TX_AMDET :
            setAMDetectorLoop(antMask);
            break;

        /* Rx LPF Loop Back */
        case RFIC_LOOPBACK_TYPE_RX_LPF:
            setRxLpfLoop(antMask);
            break;

        case RFIC_LOOPBACK_TYPE_TX_LPF:
            setTxLpfLoop(antMask);
            break;

        case RFIC_LOOPBACK_TYPE_RSSI:
            setRssiLoop();
            break;

        default:
            DEBUG_ASSERT(0);
            break;
    }
}

void RFIC_RestoreFromLoopBack(uint8 antMask)
{	
	uint8 band = HDK_getBand();

	if (band == BAND_5_2_GHZ)
	{
		rfic_disable_ssbcal_hb(antMask);
	}
	else
	{
		rfic_disable_ssbcal_lb(antMask);
	}
	rfic_txoff(antMask);
	rfic_disable_antenna_fectl(antMask);
}

/*****************************************************************************
*    Function Name:  RficDriver_CopyRegValuesFromProgModel
*    Description:    Copy default registers values, taken from the ProgModel.
*    Parameters:     inSrcAddress - Starting address of the data.
*                    inLength     - Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyRegValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyRegValuesFromProgModel(uint32 *inSrcAddress, uint32 inLength)
{
	UNUSED_PARAM(inSrcAddress);	
	UNUSED_PARAM(inLength);	
}

/*****************************************************************************
*    Function Name:  RficDriver_CopyDataBaseValuesFromProgModel
*    Description:    Copy database values from the ProgModel.
*    Parameters:     inMinorID    - Minor ID of the tuning word address.
*                    inSrcAddress - Starting address of the database data to copy.
*                    inLength     -Length of the data to copy.
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->CopyDataBaseValuesFromProgModel
*    Restrictions:   None
******************************************************************************/
void RficDriver_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress,
                                                uint32 inLength)
{
    /* Not needed in AR10, Tuning words are hard coded as arrays */
	UNUSED_PARAM(inMinorID);	
	UNUSED_PARAM(inSrcAddress);	
	UNUSED_PARAM(inLength);	
}

/*****************************************************************************
*    Function Name:  RficDriver_LoadRegDB
*    Description:    Loads all registers from SW DB to RFIC
*    Parameters:     Nane
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->LoadRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_LoadRegDB(void)
{
    uint8  entryInd = 0;

    for (entryInd = 0;entryInd < RFIC_NUM_OF_REGS; entryInd++)
    {
        writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,
                            RficDriverRegistersLocalDB[entryInd].regAddr,
                            RficDriverRegistersLocalDB[entryInd].regData,
                            RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, FALSE);
    }
}

/*****************************************************************************
*    Function Name:  RficDriver_StoreRegDB
*    Description:    Stores all RFIC registers to SW DB
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->StoreRegDB
*    Restrictions:   None
******************************************************************************/
void RficDriver_StoreRegDB(void)
{
    uint32 entryInd  = 0;
    uint32 addr      = 0;
    uint32 dbIndex   = 0;

    //Store short registers
    for (entryInd = 0; entryInd < RFIC_NUM_OF_SHORT_REGS; entryInd++)
    {
        /* Calculate DB index */
        dbIndex = entryInd;

        /* Configure the address to read (and also save it to the DB) */
        addr = entryInd + RFIC_FIRST_SHORT_REG_ADDR;
        RficDriverRegistersLocalDB[dbIndex].regAddr = addr;

        /* Read the values from the RF registers into the DB */
        readFromRficRegister(addr, RficDriverRegistersLocalDB[dbIndex].regData);
    }

    //Store long registers
    for (entryInd = 0; entryInd < RFIC_NUM_OF_LONG_REGS; entryInd++)
    {
        /* Calculate DB index */
        dbIndex = entryInd+ RFIC_NUM_OF_SHORT_REGS;

        /* Configure the address to read (and also save it to the DB) */
        addr = entryInd + RFIC_FIRST_LONG_REG_ADDR;
        RficDriverRegistersLocalDB[dbIndex].regAddr = addr;

        /* Read the values from the RF registers into the DB */
        readFromRficRegister(addr, RficDriverRegistersLocalDB[dbIndex].regData);
    }
}
/*****************************************************************************
*    Function Name:  SetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     antmask         - The updated antenna mask
*                   	 inLnaIndex - The required lna gain values
*				inPgc1Gain,inPgc2Gain,inPgc3Gain - PGC gains value
*    Returns:        None
*    Remarks:        Implements RficDriver
*    Restrictions:   None
******************************************************************************/
void SetRxGains(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain)
{
	//if pgcIn not in dB we should do...=pgcGainDb (inPgc1Gain * 2)
	UNUSED_PARAM(inPgc2Gain);	
	UNUSED_PARAM(inPgc3Gain);	
	RficDriver_SetRxGains(antMask, inLnaIndex, inPgc1Gain);						 
}

void SetRxGainsForRssi(AntennaBitmaps_e antMask, HdkLnaIndex_e inLnaIndex, int8 inPgc1Gain,int8 inPgc2Gain,int8 inPgc3Gain)
{

	SetRxGains(antMask,inLnaIndex,inPgc1Gain,inPgc2Gain,inPgc3Gain);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxGains
*    Description:    Set Rx Gains and corresponding DACs
*    Parameters:     inAnt         - The updated antenna
*                    inRficRxGains - The required gains and DACs values
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxGains
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetRxGains(uint8 antMask, uint8 lnaIndex, int8 pgcGainDb)
{
	uint8 bw = HDK_getChannelWidth();
	
	rfic_set_rxgain_bwidth(antMask, lnaIndex, pgcGainDb, 20 << bw);
}

void RficDriver_SetPgcbwidth (uint8 AntMsk, uint8 rfBandwidth)
{
	rfic_set_pgcbwidth (AntMsk, rfBandwidth);

}

void RficDriver_SetRxdcOffset (uint8 antMask, uint8 rfrxgainidx, uint8 rxdcidx, uint16 valToWrite)
{
	rfic_set_rxdcoffset_reg (antMask, rfrxgainidx, rxdcidx, valToWrite);
}

uint16 RficDriver_GetRxdcOffset (uint8 Ant, uint8 rfrxgainidx, uint8 rxdcidx)
{
	return rfic_get_rxdcoffset(Ant, rfrxgainidx, rxdcidx);
}

void RficDriver_SetLoDACs(uint8 antMask, int8 dacI, int8 dacQ)
{	
#ifdef WRX600_BU_LOGS
	ILOG0_V("RficDriver_SetLoDACs");
#endif
	rfic_set_txdcoffset(antMask, dacI, dacQ);
}

void RficDriver_SetLoDACsPerGain(uint8 antMask, int8* dacI, int8* dacQ, uint8 tpcIndex)
{	
#ifdef WRX600_BU_LOGS
	ILOG0_V("RficDriver_SetLoDACs");
#endif
	rfic_set_txdc_tpccfg_par(antMask, dacI, dacQ, tpcIndex);
}

/*****************************************************************************
*    Function Name:  RficDriver_GetLoDACs
*    Description:    Get the current Tx DACs values for the required antenna
*    Parameters:     inAnt       - The updated antenna
*                    inTpcIndex  - The Tx gain index to use
*                    outDacValue - Here we put the returned Tx DAC value
*    Returns:        The Tx DAC value
*    Remarks:        Implements RficDriver_PublicApi_t->GetLLDACs
*    Restrictions:   outDacValue is an array of size RFIC_PATH_TYPE_MAX
******************************************************************************/
void RficDriver_GetLoDACs(uint8 antMask, uint8 tpcIndex, int8* dacI, int8* dacQ)
{
	uint8 ant;
	uint16 dacsRegVal[FCSI_TOTAL_CHANNELS];
	uint32 regAddr=WRX654_RF_ANTENNA_FCSI_TPCOFF01;
	uint32 shiftI=WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__POS;
	uint32 shiftQ=WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__POS;
	
	switch (tpcIndex)
	{
		case RFIC_TX_GAIN_TPC_INDEX_0:		
			regAddr = WRX654_RF_ANTENNA_FCSI_TPCOFF01;
			shiftI = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__POS;			
			shiftQ = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__POS;
			break;
		case RFIC_TX_GAIN_TPC_INDEX_1:
			regAddr = WRX654_RF_ANTENNA_FCSI_TPCOFF01;
			shiftI = WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN1__POS;			
			shiftQ = WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN1__POS;
			break;
		case RFIC_TX_GAIN_TPC_INDEX_2:
			regAddr = WRX654_RF_ANTENNA_FCSI_TPCOFF23;
			shiftI = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN2__POS;			
			shiftQ = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN2__POS;		
			break;
		case RFIC_TX_GAIN_TPC_INDEX_MAX:
			regAddr = WRX654_RF_ANTENNA_FCSI_TPCOFF23;
			shiftI = WRX654_RF_ANTENNA_FCSI_TPCOFF23_I_GAIN3__POS;			
			shiftQ = WRX654_RF_ANTENNA_FCSI_TPCOFF23_Q_GAIN3__POS;				
			break;		
	}	
	RficDriver_ReadRegister(regAddr,dacsRegVal);
	for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{	
		if ((1 << ant) & antMask)
		{
			dacI[ant] = ((dacsRegVal[ant] >> shiftI) & WRX654_RF_ANTENNA_FCSI_TPCOFF01_I_GAIN0__MSK);
			dacQ[ant] = ((dacsRegVal[ant] >> shiftQ) & WRX654_RF_ANTENNA_FCSI_TPCOFF01_Q_GAIN0__MSK);
			SIGN_EXTEND(dacI[ant], 4); 	
			SIGN_EXTEND(dacQ[ant], 4);	
		}
	}
}

/*****************************************************************************
*    Function Name:  RficDriver_ShortRxPgcInputToGround
*    Description:    Short PGC1s input to ground for all antennas
*    Parameters:     onOff - 0=Short is On; 1=Short is Off
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->ShortRxPgcInputToGround
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxPgcInputPowerState(RficRxPgcInputPowerState_e state)
{
	UNUSED_PARAM(state);	
/* RFICTBD: update to wrx654 fields

    if(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_ON == state)
    {
		RFIC_FCSI_RMW(15, RF_ANTENNA0_FCSI_FILTER,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL__ISMSK,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL_0__SVAL);
    }
    else if(RFIC_RX_PGC_INPUT_POWER_STATE_SHORT_GROUND_OFF == state)
    {
		RFIC_FCSI_RMW(15, RF_ANTENNA0_FCSI_FILTER,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL__ISMSK,
				RF_ANTENNA1_FCSI_FILTER_EN_OFFSET_CAL_1__SVAL);
    }
    else
    {
        DEBUG_ASSERT(0);
    }
*/    
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxCalGain
*    Description:    Select calibration buffer gain (used only for calibrations)
*					  
*    Parameters:     inAnt       - The updated antenna
*                    inRxCalGain - 
*					 int8 * outCalGainDb_p - the difference in dB )
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRxCalGain
*    Restrictions:   This function is for calibration purpose only, thus it
*                    does not save any data in the mirror DB
******************************************************************************/
void RficDriver_SetRxCalGain(Antenna_e inAnt, bool inRxCalGain, int8 *outCalGainDb_p)
{
	UNUSED_PARAM(inAnt);	
	UNUSED_PARAM(inRxCalGain);	
    /* Not Supported in AR10 - Empty Function, to support generic RFIC API */
	*outCalGainDb_p = 0;
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxPGC3TuneWord
*    Description:    Set the PGC3 LPF tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxPGC3TuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
	UNUSED_PARAM(inAnts);	
	UNUSED_PARAM(inPathType);	
	UNUSED_PARAM(inLpfWord);	
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRxAllPGCsTuneWord
*    Description:    Set the PGC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->setrxpgcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetRxAllPGCsTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
	UNUSED_PARAM(inAnts);	
	UNUSED_PARAM(inPathType);	
	UNUSED_PARAM(inLpfWord);	
}

/*****************************************************************************
*    Function Name:  RficDriver_GetRxPGCTuneWord
*    Description:    Get the PGC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetRxPGCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetRxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType,
                             uint8 *outLpfWord)
{
	UNUSED_PARAM(inAnt);	
	UNUSED_PARAM(inPathType);	
	UNUSED_PARAM(outLpfWord);	
}

/*****************************************************************************
*    Function Name:  RficDriver_SetTxPGCTuneWord
*    Description:    Set the TPC tune word for the relevant antenna
*    Parameters:     inAnt	      - The bitmap of antennas to update
*                    inPathType - I, Q or both
*                    inLpfWord  - An array of two elements: One value for I path
*                    and one for Q path. The value is a 5 bits word
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->settxtpcword
*                    if inPathType is I_AND_Q, the same inLpfWord is written to
*                    both I and Q
*    Restrictions:   inPathType can be I Q or both
******************************************************************************/
void RficDriver_SetTxPGCTuneWord(AntennaBitmaps_e inAnts, RficPathTypeIndex_e inPathType, uint8 *inLpfWord)
{
	UNUSED_PARAM(inAnts);	
	UNUSED_PARAM(inPathType);	
	UNUSED_PARAM(inLpfWord);	
}

/*****************************************************************************
*    Function Name:  RficDriver_GetTxPGCTuneWord
*    Description:    Get the PGC tune word for the relevant antenna
*    Parameters:     inAnt      - The required antenna
*                    inPathType - I, Q or both
*                    outLpfWord - Here we put the returned LPF tune word(s)
*    Returns:        The LPF tune word that determines the LPF corners
*    Remarks:        Implements RficDriver_PublicApi_t->GetTxTPCWord
*    Restrictions:   inPathType can be I, Q or both
******************************************************************************/
void RficDriver_GetTxPGCTuneWord(Antenna_e inAnt, RficPathTypeIndex_e inPathType, uint8 *outLpfWord)
{
	UNUSED_PARAM(inAnt);	
	UNUSED_PARAM(inPathType);	
	UNUSED_PARAM(outLpfWord);	
}

/*****************************************************************************
*    Function Name:  RficDriver_GetTxGain
*    Description:    Get the Gain value of all antennas
*    Parameters:     OUT dutRficTxGainsParams_t *dutRficTxGainsParams_p
*    Returns:        None
*    Remarks:        
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetTxGain(OUT dutRficTxGainsParams_t *dutRficTxGainsParams_p)
{
	UNUSED_PARAM(dutRficTxGainsParams_p);	
/* RFICTBD: update to wrx654 fields
	uint16 antArr[MAX_NUM_OF_ANTENNAS] = {0};
	uint8 antIndex = 0;
	
	readFromRficRegister(RF_ANTENNA0_FCSI_SR2, antArr);

	for (antIndex = 0; antIndex<MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		dutRficTxGainsParams_p->antTxGains[antIndex].bbGain = 
			(antArr[antIndex] & RF_ANTENNA0_FCSI_SR2_PGC2_TXGAIN_CONF__SMSK) >> RF_ANTENNA0_FCSI_SR2_PGC2_TXGAIN_CONF__POS;
		dutRficTxGainsParams_p->antTxGains[antIndex].paDriverGain = 
			(antArr[antIndex] & RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__SMSK) >> RF_ANTENNA0_FCSI_SR2_PA_DRV_GAIN__POS;
	}
*/	
}

/*****************************************************************************
*    Function Name:  RficDriver_SetTxGain
*    Description:    Set the TPC value
*    Parameters:     inAnt      - The updated antenna
*                    inTpcIndex - The required index for the TPC value
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetTPCGain
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetTxGain(uint8 antMask, uint8 tpcgainidx, uint8 padGainDb)
{
	// Set TPC index and pad gain 
	rfic_set_txgain(antMask, tpcgainidx, padGainDb);
}

void RficDriver_AbstractSetTxGain(AntennaBitmaps_e antList, int pgcGain, int pd)
{
	rfic_set_txgain(antList, pgcGain, pd);
}
void RficDriver_createPgcfiltLut(uint16 sA, int16 sB)
{
	rfic_create_pgcfiltlut(sA, sB);
}
void RficDriver_getPgcFiltLut(uint8 mLut[5][11])
{
	rfic_get_pgcfiltlut(mLut);
}

void RficDriver_SetTpcGainIndex(uint8 antMask, uint8 tpcgainidx)
{
	// Set only TPC index
	rfic_set_tpcgainidx(antMask, tpcgainidx);
}

void RficDriver_SetRxPgcGain(uint8 antMask, uint8 pgcGainDb)
{
	// Set Rx PGC gain
	rfic_set_pgcgain(antMask, pgcGainDb);
}

/*****************************************************************************
*    Function Name:  RficDriver_PowerUpBiasCalBlock
*    Description:    Power-up the bias central constant current calibration
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->PowerUpBiasCalBlock
*    Restrictions:   None
******************************************************************************/
void RficDriver_PowerUpBiasCalBlock(void)
{
}

/*****************************************************************************
*    Function Name:  RficDriver_SetRintValue
*    Description:    Programs the Rint resistor's value
*    Parameters:     inValue - Value to set Rint to
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetRintValue
*    Restrictions:   max=31, min=0, default=16
******************************************************************************/
void RficDriver_SetRintValue(uint8 inValue)
{
	UNUSED_PARAM(inValue);	
}

/*****************************************************************************
*    Function Name:  RficDriver_ReadBiasCalComparator
*    Description:    Reads and returns the comparator's output
*    Parameters:     outCompResult - Here we put the value read from the register
*    Returns:        The output of the bias calibration comparator's output
*    Remarks:        Implements RficDriver_PublicApi_t->ReadBiasCalComperator
*    Restrictions:   None
******************************************************************************/
void RficDriver_ReadBiasCalComparator(bool *outCompResult)
{
	UNUSED_PARAM(outCompResult);	
}

/*****************************************************************************
*    Function Name:  RficDriver_InitRssiDcOffsetCalib
*    Description:    Initializes the RSSI DC calibration
*    Parameters:     inAntList       - The required antennas to calibrate
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->InitRssiDcOffsetCalib
******************************************************************************/
void RficDriver_InitRssiDcOffsetCalib(AntennaBitmaps_e inAntList)
{
    /* Set antennas to standby mode */
    writeOneValueToAntList(inAntList, REG_RFIC_S0_SHORT_0, RFIC_RSSI_DC_CAL_INIT_S0_VALUE, RFIC_RSSI_DC_CAL_INIT_S0_MASK, RFIC_NO_SHIFT, FALSE);
    /* Short RSSI input to GND */
    writeOneValueToAntList(inAntList, REG_RFIC_L2_RSSI_REG3, RFIC_RSSI_DC_CAL_INIT_L2_VALUE, RFIC_RSSI_DC_CAL_INIT_L2_MASK, RFIC_NO_SHIFT, FALSE);

    /* Set ANA-MUX to RSSI and to step 0 (out of 0 to 4) */
    writeOneValueToAntList(inAntList, REG_RFIC_L4_ANA_MUX, RFIC_RSSI_DC_CAL_INIT_L4_VALUE, RFIC_RSSI_DC_CAL_INIT_L4_MASK, RFIC_NO_SHIFT, FALSE);

    /* ANA-MUX setting for RSSI DC calibration */
    writeOneValueToAntList(inAntList, REG_RFIC_L5_ANA_MUX_PROCMON_TEMPS, RFIC_RSSI_DC_CAL_INIT_L5_VALUE, RFIC_RSSI_DC_CAL_INIT_L5_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetCalibStage
*    Description:    Sets the calibration stage number in the calibration stage
*                    register as well as the path type in the same register for
*                    the selected antenna list (bitmap), path type (I or Q but NOT both)
*                    and stage number (0 through 4)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 1 and 5
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetSetCalibStage
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetCalibStage(AntennaBitmaps_e inAntList,
                                          RficPathTypeIndex_e inInputPathType,
                                          uint8 inStageNumber)
{
    uint32 regVal = 0;

    /* Update step in ANA-MUX (steps here are 0 to 4) */
    writeOneValueToAntList(inAntList, REG_RFIC_L4_SEL_RSSI, inStageNumber, REG_RFIC_L4_SEL_RSSI_MASK, REG_RFIC_L4_SEL_RSSI_SHIFT, FALSE);

    ++inStageNumber; /* Calibration stage numbers are 1 to 5 */
    ASSERT(RFIC_RSSI_DC_CAL_MAX_STEP_NUM >= inStageNumber);

    regVal = inStageNumber << REG_RFIC_L2_CAL_STEP_SHIFT;

    switch(inInputPathType)
    {
        case RFIC_PATH_TYPE_I:
            regVal |= 1 << REG_RFIC_L2_CAL_I_SHIFT;
            regVal |= 0 << REG_RFIC_L2_CAL_Q_SHIFT;
            break;
        case RFIC_PATH_TYPE_Q:
            regVal |= 0 << REG_RFIC_L2_CAL_I_SHIFT;
            regVal |= 1 << REG_RFIC_L2_CAL_Q_SHIFT;
            break;
        default:
            DEBUG_ASSERT(0);
    }

    /* Update step and path */
    writeOneValueToAntList(inAntList, REG_RFIC_L2_CAL_STEP, regVal, RFIC_RSSI_DC_CAL_STEP_L2_MASK, RFIC_NO_SHIFT, FALSE);
}

/*****************************************************************************
*    Function Name:  RficDriver_RssiDcOffsetSetDacValues
*    Description:    Sets the values of the DACs in selected path, stage and antenna
*                    list (bitmap), according to the array of DAC values (inDacValues)
*    Parameters:     inAntList       - The antenna list to calibrate
*                    inInputPathType - I or Q but NOT both
*                    inStageNumber   - Stage number between 0 and 4
*                    inDacValues     - The values to set in the DACs
*    Returns:        The value of the DAC that minimized the DC offset of the required
*                    antenna, path and stage
*    Remarks:        Implements RficDriver_PublicApi_t->RssiDcOffsetCalibStg
*    Restrictions:   inInputPathType can be I or Q but NOT both
******************************************************************************/
void RficDriver_RssiDcOffsetSetDacValues(AntennaBitmaps_e inAntList,
                                         RficPathTypeIndex_e inInputPathType,
                                         uint8 inStageNumber, uint8 *inDacValues)
{
    uint16    regAddr  = 0;
    uint16    regMask  = 0;
    uint16    regShift = 0;
	uint16    regValues[MAX_NUM_OF_ANTENNAS];
	Antenna_e ant      = ANTENNA_0;

    ++inStageNumber; /* Calibration stage numbers are 1 through 5 */

    switch(inInputPathType)
    {
        case RFIC_PATH_TYPE_I:
            if (inStageNumber < RFIC_RSSI_DC_CAL_MAX_STEP_NUM)
            {
                regAddr  = REG_RFIC_L0_RSSI_REG1;
                /* stage 1 mask is 0xF000, stage 2 mask is 0xF00, stage 3 mask is 0xF0 and stage 4 mask is 0xF */
                regMask  = REG_RFIC_L0_I_RSSI_CAL1_MASK >> (REG_RFIC_L0_I_RSSI_CAL1_WIDTH * (inStageNumber - 1));
                regShift = REG_RFIC_L0_I_RSSI_CAL1_SHIFT - (REG_RFIC_L0_I_RSSI_CAL1_WIDTH * (inStageNumber - 1));
            }
            else /* stage number 5 */
            {
                regAddr  = REG_RFIC_L2_I_RSSI_CAL5;
                regMask  = REG_RFIC_L2_I_RSSI_CAL5_MASK;
                regShift = REG_RFIC_L2_I_RSSI_CAL5_SHIFT;
            }
            break;

        case RFIC_PATH_TYPE_Q:
            if (inStageNumber < RFIC_RSSI_DC_CAL_MAX_STEP_NUM)
            {
                regAddr  = REG_RFIC_L1_RSSI_REG2;
                /* stage 1 mask is 0xF000, stage 2 mask is 0xF00, stage 3 mask is 0xF0 and stage 4 mask is 0xF */
                regMask  = REG_RFIC_L1_Q_RSSI_CAL1_MASK >> (REG_RFIC_L1_Q_RSSI_CAL1_WIDTH * (inStageNumber - 1));
                regShift = REG_RFIC_L1_Q_RSSI_CAL1_SHIFT - (REG_RFIC_L1_Q_RSSI_CAL1_WIDTH * (inStageNumber - 1));
            }
            else /* stage number 5 */
            {
                regAddr  = REG_RFIC_L2_Q_RSSI_CAL5;
                regMask  = REG_RFIC_L2_Q_RSSI_CAL5_MASK;
                regShift = REG_RFIC_L2_Q_RSSI_CAL5_SHIFT;
            }
            break;

        default:
            DEBUG_ASSERT(0);
    }

	for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
	{
		regValues[ant] = (uint32)inDacValues[ant];
	}

    writeToRficRegister(inAntList, regAddr, regValues, regMask, regShift, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_StoreRssiDcOffsetCalibValues
*    Description:    Stores all DC-offset values for the specific antenna
*    Parameters:     inAntList      - The antenna list to update
*                    inAllDacValues - The DAC values ordered by RficRssiDacIndexes_e
*    Returns:        None
*    Remarks:        None
*    Restrictions:   inAllDacValues is 10 slots, not-more, no-less
******************************************************************************/
void RficDriver_StoreRssiDcOffsetCalibValues(AntennaBitmaps_e inAntList, uint8 (*inAllDacValues)[RFIC_RSSI_STG_TOTAL])
{
    Antenna_e ant = ANTENNA_0;
    uint16    regL0Vals[MAX_NUM_OF_ANTENNAS];
    uint16    regL1Vals[MAX_NUM_OF_ANTENNAS];
    uint16    regL2Vals[MAX_NUM_OF_ANTENNAS];
	UNUSED_PARAM(inAntList);
    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ++ant)
    {
        /* Prepare values for register L0 */
        regL0Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_I_4_DAC]) << REG_RFIC_L0_I_RSSI_CAL4_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_3_DAC]) << REG_RFIC_L0_I_RSSI_CAL3_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_2_DAC]) << REG_RFIC_L0_I_RSSI_CAL2_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_1_DAC]) << REG_RFIC_L0_I_RSSI_CAL1_SHIFT);
        /* Prepare values for register L1 */
        regL1Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_Q_4_DAC]) << REG_RFIC_L1_Q_RSSI_CAL4_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_3_DAC]) << REG_RFIC_L1_Q_RSSI_CAL3_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_2_DAC]) << REG_RFIC_L1_Q_RSSI_CAL2_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_Q_1_DAC]) << REG_RFIC_L1_Q_RSSI_CAL1_SHIFT);
        /* Prepare values for register L2 */
        regL2Vals[ant] = ((inAllDacValues[ant][RFIC_RSSI_STG_Q_5_DAC]) << REG_RFIC_L2_Q_RSSI_CAL5_SHIFT) |
                         ((inAllDacValues[ant][RFIC_RSSI_STG_I_5_DAC]) << REG_RFIC_L2_I_RSSI_CAL5_SHIFT);
    }

    /* Write the values prepared */
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L0_RSSI_REG1, regL0Vals, RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L1_RSSI_REG2, regL1Vals, RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
    writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L2_RSSI_REG3, regL2Vals, RFIC_RSSI_DC_CAL_STORE_ALL_REG_L2_MASK, RFIC_NO_SHIFT, TRUE);
}

/*****************************************************************************
*    Function Name:  RficDriver_SetFrequency
*    Description:    Sets the frequency of the system
*    Parameters:     inSetChannelParams - A structure with all the parameters
*                                           needed to set a frequency in the system:
*                         -> rfOscilFreq  - Frequency of the RF oscillator
*                         -> channel    - Channel we are using
*                         -> spectrumMode - CB or nCB
*                         -> upperLower   - When full Spectrum is used, which other
*                                           channel is taken, the upper->1 or lower->0
*                         -> band         - BAND_2_4_GHZ or BAND_5_2_GHZ
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->SetFrequency
*    Restrictions:   None
******************************************************************************/
uint16 rxBandLUT[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT] = 
{
	/* Default values. should be overriden from cal file */
	{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
	{4900,4900,4950,5150,5350,5600,5850},
};
uint16 txBandLUT[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT] = 
{
	/* Default values. should be overriden from cal file */
	{4970,5180,5380,5570,5700,5840,6000},
	{4970,5180,5380,5570,5700,5840,6000},
	{4970,5180,5380,5570,5700,5840,6000},
	{4970,5180,5380,5570,5700,5840,6000},
};


uint8 lnaRxsubband_update = 0;
uint8 lnaTxsubband_update = 0;

void  RficDriver_SetRxBandLUT(uint16 *rxBandLUTtemp)
{
	MEMCPY(&rxBandLUT,rxBandLUTtemp,MAX_NUMBER_OF_LNA_CROSSING_POINT*MAX_NUM_OF_ANTENNAS*2);
	lnaRxsubband_update = TRUE;
}
void  RficDriver_SetTxBandLUT(uint16 *txBandLUTtemp)
{
	MEMCPY(&txBandLUT,txBandLUTtemp,MAX_NUMBER_OF_LNA_CROSSING_POINT*MAX_NUM_OF_ANTENNAS*2);
	lnaTxsubband_update =TRUE;
}

void RficDriver_readTxbandFromRf(uint8 AntMsk, uint16 *txBand)
{
	rfic_read_txband(AntMsk, txBand);
}

void RficDriver_setTxband(uint8 AntMsk, uint8 bandSel)
{
	rfic_set_txband(AntMsk, bandSel);
}

RetVal_e RficDriver_SetFrequency(SetChannelParams_t* inSetChannelParams)
{				   
	uint32 antMask = inSetChannelParams->txAntMask;
	uint32 freq = RficDriver_CalcLoFrequency(inSetChannelParams);
	uint8  bw = inSetChannelParams->setChannelReqParams.chan_width;
	uint8 pll = getMyPllId();
	uint8 rxBandIndex=0, txBandIndex=0;
    int8 index;
	uint8 freqJumpChan = 0;
	RetVal_e ret = RET_VAL_FAIL;
	uint32 ant;

#ifdef WRX600_BU_LOGS
	SLOG0(0, 0, SetChannelParams_t, inSetChannelParams);
	ILOG0_DDDD("RficDriver_SetFrequency. pll %d antMask 0x%x freq %d bw %d", pll, antMask, freq, bw);
#endif
		
	if(HDK_IsFrequencyJumpEnabled() == TRUE)
	{
		// Freq jump - TBD
	}
	else
	{
#ifdef WRX600_BU_LOGS
		ILOG0_V("Frequency jump is disabled. lock PLL");
#endif
		rfic_lock_pll(pll, freq);	
	}
	// Call RficDriver_AssociateAntennasToMyBand after PLL is configured
	RficDriver_AssociateAntennasToMyBand(antMask);

	if(lnaRxsubband_update == TRUE)
	{
		for( ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if(antMask&(0x1<<ant))
			{
				rxBandIndex = 0;
                
				for (index = (MAX_NUMBER_OF_LNA_CROSSING_POINT-1); index >= 0; index--)
				{
					if (freq >= rxBandLUT[ant][index])
					{
					  rxBandIndex = index+1;
					  break;
					}
				}
				rfic_set_rxband((0x1<<ant), rxBandIndex);
			}
		}
	}
	else
	{
		rxBandIndex = rfic_get_rxband(freq);
		rfic_set_rxband(antMask, rxBandIndex);
	}

	if(lnaTxsubband_update == TRUE)
	{
		for( ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if(antMask&(0x1<<ant))
			{
				txBandIndex = 0;
                
				for (index = (MAX_NUMBER_OF_LNA_CROSSING_POINT-1); index >= 0; index--)
				{
					if (freq >= txBandLUT[ant][index])
					{
					  txBandIndex = index+1;
					  break;
					}
				}
				rfic_set_txband((0x1<<ant), txBandIndex);
			}
		}
	}
	else
	{
		txBandIndex = rfic_get_txband(freq);
		rfic_set_txband(antMask, txBandIndex);

	}
	
	rfic_set_rfch_dc_bw(antMask, freqJumpChan, bw, 20 << bw);

#ifdef WRX600_BU_LOGS
	ILOG0_DD("rxBandIndex %d txBandIndex %d", rxBandIndex, txBandIndex);
	RficDriver_dumpRegs();
#endif	
#ifdef HDK_RF_EMULATOR
	RficDriver_SetRfEmuBand(inSetChannelParams->band, antMask);
#endif
	if (rfic_poll_pll_lock(pll) == TRUE)
	{
		ret = RET_VAL_SUCCESS;		
	}
    return ret;
}

void RficDriver_SetRxBand(uint16 rxBand)
{
	rfic_set_rxband(0xf,rxBand);
}

void RficDriver_SetTxBand(uint16 txBand)
{
	rfic_set_txband(0xf,txBand);
}

/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingSet
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function changes the frequency
*                    if the band number and channel number require that.
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingSet
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingSet(void)
{
    /* Not Supported in AR10 - Empty Function, to support generic RFIC API */
}

/*****************************************************************************
*    Function Name:  RficDriver_RxIqFreqSpursCancellingReset
*    Description:    In 2.4GHz we have Rx RF problems (spurs) in odd frequencies
*                    (even channels), therefore we calibrate the RxIQ mismatch
*                    with offset of 1MHz from our chosen central frequency if the
*                    channel number is even. This function resets the frequency
*                    back to operational mode after the calibration was done
*    Parameters:     None
*    Returns:        None
*    Remarks:        Implements RficDriver_PublicApi_t->RxIqFreqSpursCancellingReset
*    Restrictions:   None
******************************************************************************/
void RficDriver_RxIqFreqSpursCancellingReset(void)
{
    /* Not Supported in wave500 - Empty Function, to support generic RFIC API */
}

/*****************************************************************************
*    Function Name:  RficDriver_CalculateFlo
*    Description:    Returns the frequency of the local oscillator according to
*                    inputs
*    Parameters:     inChannelNum   - Channel we are using
*                    inSpectrumMode - CB or nCB
*                    inUpperLower   - When full Spectrum is used, which other
*                                     channel is taken, the upper->1 or lower->0
*                    inBand         - BAND_2_4_GHZ or BAND_5_2_GHZ
*                    outFlo         - Here we will set the output of the function
*    Returns:        The local oscillator's frequency
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_CalculateFlo(uint8 inChannelNum, SpectrumMode_e inSpectrumMode,
                             uint8 inUpperLower, uint8 inBand, uint32 *outFlo)
{
	UNUSED_PARAM(inChannelNum);	
	UNUSED_PARAM(inSpectrumMode);	
	UNUSED_PARAM(inUpperLower);	
	UNUSED_PARAM(inBand);	
	UNUSED_PARAM(outFlo);	
}

uint32 RficDriver_CalcLoFrequency(SetChannelParams_t * inSetChannelParams)
{
	uint32 freq;
	uint32 freqStart;

	if(BAND_5_2_GHZ == inSetChannelParams->band)
	{
		if((inSetChannelParams->setChannelReqParams.low_chan_num >= RFIC_FREQ_FIRST_CHANNEL_5G_4) &&
			(inSetChannelParams->setChannelReqParams.low_chan_num <= RFIC_FREQ_LAST_CHANNEL_5G_4))
		{
			freqStart = RFIC_FREQ_STARTING_POINT_5G_4;
		}
		else
		{
			freqStart = RFIC_FREQ_STARTING_POINT_5G_5;
		}
	}
	else
	{

		if((inSetChannelParams->setChannelReqParams.low_chan_num == 14) && 
		   (inSetChannelParams->setChannelReqParams.chan_width == BANDWIDTH_TWENTY))
		{
			freqStart = RFIC_FREQ_STARTING_POINT_2_4G_CHANNEL_14;
		}
		else
		{
			freqStart = RFIC_FREQ_STARTING_POINT_2_4G;
		}
	}

	freq = freqStart + (inSetChannelParams->setChannelReqParams.low_chan_num * RF_CHANNEL_WIDTH_MHZ) +
			bandwidthOffset[(uint8)inSetChannelParams->setChannelReqParams.chan_width];

	return freq;
}

/*****************************************************************************
*    Function Name:  RficDriver_SetXtalValue
*    Description:    Set the Xtal value 
*    Parameters:     uint32 - xtalVal
*    Returns:        void
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_SetXtalValue(uint32 xtalVal)
{
	rfic_trim_dcxo (xtalVal, xtalVal);
}

/*****************************************************************************
*    Function Name:  RficDriver_XtalInitPreventAmpChange
*    Description:    To avoid clock amplitude change during start, (which causes DSL problems),
*                    we need to add the following writes to the init part.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_XtalInitPreventAmpChange(void)
{
/*RFICTBD: update to wrx654 
    // Set FCSI LDO to default, DCXO current control default, FSYS strength default 
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L53_FCSI_LDO_OTHERS, RFIC_SET_XTAL_L53_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    // Enable Bandgap 
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L51_BIAS_CENTRAL1,RFIC_SET_XTAL_L51_ONLY_BG_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    // Delay between two writes to same register 
    MT_DELAY(100);

    // Enable Bandgap, Enable Centralbias V2I const and default Bgp and temp trimming, default iconst 
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L51_BIAS_CENTRAL1,RFIC_SET_XTAL_L51_FULL_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    // Enable Centralbias V2I ref, PLL reset, soft reset released, set DCXO LDO to default 
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L52_BIAS_CENTRAL2_RESET_DCXO_LDO,RFIC_SET_XTAL_L52_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);

    // Set DCXO bias mode and rest to default, enable FSYS output clock 
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L50_DCXO_REG2_FSYS,RFIC_SET_XTAL_L50_VALUE , RFIC_DEFAULT_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
*/	
}

/*****************************************************************************
*    Function Name:  RficDriver_GetXtalValue
*    Description:    Get the Xtal value  from the RFIC
*    Parameters:     uint32 - xtalVal
*    Returns:        void
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetXtalValue(uint32 *xtalVal)
{
	uint16 regVal[FCSI_TOTAL_CHANNELS];	

	// Read Xtal Value xtal Value
	readFromRficRegister(WRX654_RF_CENTRAL_FCSI_DCXO0, regVal);//WRX654_RF_CENTRAL_FCSI_DCXO0   RF_ANTENNA1_FCSI_DCXO0
	
	*xtalVal = (regVal[FCSI_CHANNEL_E] & WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X1__MSK) >> WRX654_RF_CENTRAL_FCSI_DCXO0_CAP_TRIM_X1__POS;
}

/*****************************************************************************
*    Function Name:  RficDriver_GetAmDetBranch
*    Description:    This function returns the AM detector branch according to the frequency band given
*    Parameters:     inFreqBand     - The required frequency band (BAND_2_4_GHZ or BAND_5_2_GHZ only)
*                    outAmDetBranch - Here we will put the returned value
*    Returns:        None
*    Remarks:        This function ignores IQ swap if exists. User must check IQ swap with PHY driver
*    Restrictions:   None
******************************************************************************/
void RficDriver_GetAmDetBranch(uint8 inFreqBand, RficPathTypeIndex_e *outAmDetBranch)
{
	UNUSED_PARAM(inFreqBand);	
	*outAmDetBranch = RFIC_PATH_TYPE_Q;
}

/*********************************************************************************
Method:			RficDriver_WriteRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteRegister(IN uint32 regNum,IN uint16 *regVal)
{
	writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5,regNum,regVal,RFIC_DEFAULT_MASK,RFIC_DEFAULT_MASK_SHIFT,FALSE);
}
/*********************************************************************************
Method:			RficDriver_WriteMaskRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	IN uint32 regVal
Parameter:    	IN uint32 regMask
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_WriteMaskRegister(IN uint32 regNum,IN uint16 *regVal, IN uint32 regMask)
{
	writeToRficRegister(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, regNum, regVal, regMask, RFIC_DEFAULT_MASK_SHIFT, FALSE);
}
/*********************************************************************************
Method:			RficDriver_ReadRegister
Description:  
Parameter:    	IN uint32 regNum
Parameter:    	OUT uint32 * regVal
Returns:      	void
Remarks:		None
*********************************************************************************/
void RficDriver_ReadRegister(IN uint32 regNum,OUT uint16 *regVal)
{
	readFromRficRegister(regNum,regVal);
}

/******************************************************************************
 ** General functions unrelated to the the type of RFIC we are working with ***
 ******************************************************************************/

/******************************************************************************/
/***						Staticic Functions Definitions					***/
/******************************************************************************/
/*****************************************************************************
*    Function Name:  writeToRficRegister
*    Description:    Writes inValue into inRfRegAddr, using inMask as write mask.
*    Parameters:     inAntList   - The antenna bitmap to update
*                    inRfRegAddr - The RF register address we write to.
*                    inValues    - The values we want to write into the register.
*                    inMask      - A mask of where in the register we write
*                                  inValue into.
*                    inUpdateDB  - TRUE<->Write data to local DB as well
*                                  FALSE<->Don't write to local DB
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void writeToRficRegister(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint16 *inValues, uint16 inMask, uint16 inShift, bool inUpdateDB)
{
	uint16		centralVal = 0;
	uint16 		antVal[MAX_NUM_OF_ANTENNAS] = {0};
	uint8		ant;
	bool		success;

	inAntList &= (Hdk_GetTxAntMask() | (1 << FCSI_CHANNEL_E)); //mask out antennas not assigned to band (keep central)

	if ((inAntList & (1 << FCSI_CHANNEL_E)) != 0) //if central
	{
		FCSI_READ_TRANSACTION_CENTRAL(0, 1, inRfRegAddr, &centralVal, &success);
		ASSERT(success);
		centralVal = (((inValues[FCSI_CHANNEL_E] << inShift) & inMask) | (centralVal & (~inMask)));
		FCSI_WRITE_TRANSACTION_CENTRAL(0, 1, inRfRegAddr, centralVal, &success);
		ASSERT(success);
		if (inUpdateDB == TRUE)
		{
			updateDB(inRfRegAddr, FCSI_CHANNEL_E, centralVal);
		}		
		inAntList &= ~(1 << FCSI_CHANNEL_E);
	}

	if (inAntList  != 0)
	{
		FCSI_READ_TRANSACTION_ANT(0, 1, inAntList, inRfRegAddr, antVal, &success);
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			if ((inAntList & (1 << ant)) != 0)
			{
				antVal[ant] = (((inValues[ant] << inShift) & inMask) | (antVal[ant] & (~inMask)));
				if (inUpdateDB == TRUE)
				{
					updateDB(inRfRegAddr, ant, antVal[ant]);
				}
			}
		}
		FCSI_WRITE_TRANSACTION_ANT(0, 1, inAntList, inRfRegAddr, antVal, &success);
		ASSERT(success);
	}	
}

/*****************************************************************************
*    Function Name:  writeOneValueToAntList
*    Description:    Writes inValue into inRfRegAddr on all antennas, using inMask
*                    as the write mask.
*    Parameters:     inRfRegAddr - The RF register address we write to.
*                    inValue     - The value we want to write into the registers.
*                    inMask      - A mask of where in the register we write
*                                  inValue into.
*                    inUpdateDB  - TRUE <->Write data to local DB as well
*                                  FALSE<->Don't write to local DB
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void writeOneValueToAntList(AntennaBitmaps_e inAntList, uint32 inRfRegAddr, uint32 inValue, uint16 inMask, uint16 inShift, bool inUpdateDB)
{
    uint16 regVals[FCSI_TOTAL_CHANNELS];
    Antenna_e ant = ANTENNA_0;

    for (ant = ANTENNA_0; ant < FCSI_TOTAL_CHANNELS; ++ant)
    {
        regVals[ant] = inValue;
    }

    writeToRficRegister(inAntList, inRfRegAddr, regVals, inMask, inShift, inUpdateDB);
}

/*****************************************************************************
*    Function Name:  readFromRficRegister
*    Description:    Reads a value from inRfRegAddr into outValues.
*    Parameters:     inRfRegAddr - The RF register address we read from.
*                    outValues   - Here we will put the values read from the registers.
*    Returns:        The contents of the register
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void readFromRficRegister(uint32 inRfRegAddr, uint16 *outValues)
{
	uint32		antMask;
	bool		success;

	antMask = Hdk_GetTxAntMask();

	FCSI_READ_TRANSACTION_CENTRAL(0, 1, inRfRegAddr, &outValues[FCSI_CHANNEL_E], &success);
	ASSERT(success);

	FCSI_READ_TRANSACTION_ANT(0, 1, antMask, inRfRegAddr, outValues, &success);
	ASSERT(success);
}
#ifdef SDL_IGNORE_UNUSED_FUNCTION
/*****************************************************************************
*    Function Name:  setRxDac1
*    Description:    Set Rx DAC1(For DC cancellation)
*    Parameters:     inAntBitmap   - The updated antenna bitmap
*                    inPathType    - I, Q or both
*                    inRxDacValues - The required DAC values
*    Returns:        None
*    Remarks:        Due to parallel register access, 3 antennas take same time
*                    as one antenna, but 2 antennas take longer (about twice the time)
*    Restrictions:   None
******************************************************************************/
static inline void setRxDac1(AntennaBitmaps_e inAntBitmap,RficPathTypeIndex_e inPathType, uint16 *inRxDacValues)
{
/*RFICTBD: update to wrx654 
    writeToRficRegister(inAntBitmap, RF_ANTENNA0_FCSI_SR3, inRxDacValues, (RF_ANTENNA0_FCSI_SR3_I_OFFSET_DAC0__SMSK | RF_ANTENNA0_FCSI_SR3_Q_OFFSET_DAC0__SMSK), RFIC_NO_SHIFT, TRUE);
*/
}

/*****************************************************************************
*    Function Name:  setRxDac2
*    Description:    Set Rx DAC2(For DC cancellation)
*    Parameters:     inAntBitmap   - The updated antenna
*                    inPathType    - I, Q or both
*                    inRxDacValues - The required DAC values
*    Returns:        None
*    Remarks:        Due to parallel register access, 3 antennas take same time
*                    as one antenna, but 2 antennas take longer (about twice the time)
*    Restrictions:   None
******************************************************************************/
static inline void setRxDac2(AntennaBitmaps_e inAntBitmap,RficPathTypeIndex_e inPathType, uint16 *inRxDacValues)
{
/*RFICTBD: update to wrx654 
    writeToRficRegister(inAntBitmap, RF_ANTENNA0_FCSI_PGC2RXDC2, inRxDacValues, (RF_ANTENNA0_FCSI_PGC2RXDC2_DACI__SMSK | RF_ANTENNA0_FCSI_PGC2RXDC2_DACQ__SMSK), RFIC_NO_SHIFT, TRUE);
*/
}
#endif
/*****************************************************************************
*    Function Name:  updateDB
*    Description:    Updates the local DB of the RF registers with the 
*    Parameters:     inMultiArgVals - A pointer to the values and register addresses
*                                     we wrote to the RF and want to update the DB with
*                    inMask         - The mask of the bits to update
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static inline void updateDB(uint32 rfRegAddr, uint8 ant, uint16 value)
{
    uint16    arrayIndex = 0;

    /* Calculate index of register in the DB array */
    if (RFIC_LAST_SHORT_REG_ADDR >= rfRegAddr)
    {
        /* Register is one of the short registers */
        arrayIndex = rfRegAddr;
    }
    else if ((RFIC_FIRST_LONG_REG_ADDR <= rfRegAddr) &&
             (RFIC_LAST_LONG_REG_ADDR  >= rfRegAddr))
    {
        /* Register is one of the long registers */
        arrayIndex = rfRegAddr - RFIC_FIRST_LONG_REG_ADDR + RFIC_NUM_OF_SHORT_REGS;
    }

    /* write value to RF driver's DB */
    RficDriverRegistersLocalDB[arrayIndex].regAddr = rfRegAddr;

    RficDriverRegistersLocalDB[arrayIndex].regData[ant] = value;
}

void RficDriver_resetPsdArray(void)
{
	memset(&RficDriverPSD, 0, sizeof(RficDriverPSD));
}

void RficDriver_resetZwdfsPsdArray(void)
{
	memset(&RficDriverZwdfsPSD, 0, sizeof(RficDriverZwdfsPSD));
}


void RficDriver_writePsdTableFromDB(void)
{
	
	if (ConfigurationManager_IsZwdfsVap(ConfigurationManager_GetFirstVapForMyBand()))
	{
		OverwriteZwdfsPsdArray();
	}

	writePsdTableFromDB();
}


static void OverwriteZwdfsPsdArray(void)
{

	uint32 	  regNum;
	uint16    zwdfsDbIndex;
	uint16    PsdDbIndex=0;
    uint8 	  ant;
	
	for  (zwdfsDbIndex = 0; zwdfsDbIndex < RFIC_LONG_FECTL_REG_NUM; zwdfsDbIndex++)
	{
		regNum = RficDriverZwdfsPSD[zwdfsDbIndex].regAddr;
		
		/* Calculate index of register in the DB array */
	    if (RFIC_LAST_SHORT_REG_ADDR >= regNum)
	    {
	        /* Register is one of the short registers */
	        PsdDbIndex = regNum;
	    }
	    else if ((RFIC_FIRST_LONG_REG_ADDR <= regNum) &&
	             (RFIC_LAST_LONG_REG_ADDR  >= regNum))
	    {
	        /* Register is one of the long registers */
	        PsdDbIndex = regNum - RFIC_FIRST_LONG_REG_ADDR + RFIC_NUM_OF_SHORT_REGS;
	    }
	    else
	    {
	        /* Invalid Register address */
	        DEBUG_ASSERT(0);
	    }
#ifdef HDK_LOGS
		ILOG0_DDD("OverwriteZwdfsPsdArray() zwdfsDbIndex:%d regNum:%d PsdDbIndex:%d",zwdfsDbIndex,regNum, PsdDbIndex);
#endif
		RficDriverPSD[PsdDbIndex].regAddr = RficDriverZwdfsPSD[zwdfsDbIndex].regAddr;
		RficDriverPSD[PsdDbIndex].mask = RficDriverZwdfsPSD[zwdfsDbIndex].mask;
		
		for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
#ifdef HDK_LOGS
			ILOG0_DD("OverwriteZwdfsPsdArray() ant:%d RficDriverZwdfsPSD[zwdfsDbIndex].regData[ant]:%d ",ant,RficDriverZwdfsPSD[zwdfsDbIndex].regData[ant]);
#endif

	        RficDriverPSD[PsdDbIndex].regData[ant] = RficDriverZwdfsPSD[zwdfsDbIndex].regData[ant];
		}


	}
}

void RficDriver_copyOneRowFromPsdTable(uint32 regNum, uint32 *regVal, uint32 regMask, uint16 rowIndex, RfregPsdDb_e dbTag)
{
    uint16    arrayIndex = 0;
    uint8 ant;
	RficPSDRegistersDB_t* RficDb;
	
    /* Calculate index of register in the DB array */
    if (RFIC_LAST_SHORT_REG_ADDR >= regNum)
    {
        /* Register is one of the short registers */
        arrayIndex = regNum;
    }
    else if ((RFIC_FIRST_LONG_REG_ADDR <= regNum) &&
             (RFIC_LAST_LONG_REG_ADDR  >= regNum))
    {
        /* Register is one of the long registers */
        arrayIndex = regNum - RFIC_FIRST_LONG_REG_ADDR + RFIC_NUM_OF_SHORT_REGS;
    }
    else
    {
        /* Invalid Register address */
        DEBUG_ASSERT(0);
    }
	
    /* write value to RF PSD DB */
	RficDb = rficGetPsdDB(dbTag, arrayIndex, rowIndex);
	ASSERT(RficDb != NULL); //KW_FIX_F_G
	RficDb->regAddr = regNum;
	RficDb->mask |= regMask;

    for (ant = ANTENNA_0; ant < MAX_NUM_OF_ANTENNAS; ant++)
    {
    
        RficDb->regData[ant] |= (uint16)(regVal[ant] & regMask);
	
	}
}

static void writePsdTableFromDB(void)
{
    uint16    arrayIndex;
    uint8 ant;
	uint32 address, mask;
	uint16 values[MAX_NUM_OF_ANTENNAS];
	// TBD - change from MAX_NUM_OF_ANTENNAS to FCSI_TOTAL_CHANNELS (for writing to central)
	
   for (arrayIndex = 0; arrayIndex < RFIC_NUM_OF_REGS; arrayIndex++)
	{
		address = RficDriverPSD[arrayIndex].regAddr;
		mask    = RficDriverPSD[arrayIndex].mask;

		// Get values per antenna
		for (ant = 0; ant < MAX_NUM_OF_ANTENNAS; ant++)
		{
			values[ant] = RficDriverPSD[arrayIndex].regData[ant];
		}
		RficDriver_WriteMaskRegister(address, values, mask);
	}
}

uint32  checkPsdTableType(void)
{
	return PSD_BAND_5GHZ_TABLE;
}


/*****************************************************************************
*    Function Name:  setAMDetectorLoop
*    Description:    Sets the AM Detector loop-back
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setAMDetectorLoop(uint8 antMask)
{	
	uint8 band = HDK_getBand();
#ifdef WRX600_BU_LOGS
	ILOG0_DD("setAMDetectorLoop antMask 0x%x band %d", antMask, band);
#endif

	if (band == BAND_5_2_GHZ)
	{
		rfic_enable_ssbcal_hb(antMask);
	}
	else
	{
		rfic_enable_ssbcal_lb(antMask);
	}	
	rfic_rxoff(antMask);
	/* W/A for issue in RFIC, high gain should be set before Tx on. Desired gain may be set later */
	rfic_set_tpcgainidx(antMask, 3);
	/* */
	//rfic_txon(antMask);
	RficDriver_setRffeMode(TX_MODE, antMask);
}

/*****************************************************************************
*    Function Name:  setRxLpfLoop
*    Description:    Sets the Rx LPF loop-back
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setRxLpfLoop(uint8 antMask)
{
	UNUSED_PARAM(antMask);	
	DEBUG_ASSERT(0)
}

/*****************************************************************************
*    Function Name:  setTxLpfLoop
*    Description:    Sets the Tx LPF loop-back. NOTE: assumes existing Rx LPF loopback
*    Parameters:     inTxAnt - The antenna used for Tx in the loop-back
*                    inRxAnt - The antenna used for Rx in the loop-back
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setTxLpfLoop(uint8 antMask)
{
	UNUSED_PARAM(antMask);
	DEBUG_ASSERT(0)
}

/*****************************************************************************
*    Function Name:  setRssiLoop
*    Description:    Sets the RSSI loop-back
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
static void setRssiLoop(void)
{
    /* Set stand-by mode, and select SSI PATH - RSSI */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_S0_SHORT_0, RFIC_LOOP_BACK_RSSI_S0_VALUE, RFIC_LOOP_BACK_RSSI_S0_MASK, RFIC_NO_SHIFT, FALSE);
    /* Select RSSI input - DAC output */
    writeOneValueToAntList(ANTENNA_BITMAP_ALL_ANTENNAS_GEN5, REG_RFIC_L2_RSSI_REG3, RFIC_LOOP_BACK_RSSI_L2_VALUE, RFIC_LOOP_BACK_RSSI_L2_MASK, RFIC_NO_SHIFT, FALSE);

	//Tri-St All ants
	RficDriver_setRffeMode(T_STATE_MODE,ANTENNA_BITMAP_ALL_ANTENNAS_GEN5);
}

static void tempSensorStartClock(void)
{
	rfic_stop_tempsensor_clock();
	
	rfic_start_tempsensor_clock();
}

static void tempSensorRead(int32* temperature)
{
	uint16 tempResult;
	
	rfic_start_tempsensor();

	MT_DELAY(150);
	
	tempResult = rfic_read_tempsensor();

	*temperature = (tempSensorParams.slope * tempResult)  + tempSensorParams.offset;
	
	*temperature >>= 11;
}

static void tempSensorStopClock(void)
{
	rfic_stop_tempsensor_clock();
}

/*****************************************************************************
*    Function Name:  RficDriver_getRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
*    Description:    get rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
	
uint8 RficDriver_getRffeMode(CalRFFEState_e mode,Antenna_e ant)
{
	uint32 regAddress;
	uint8 val;
	
	DEBUG_ASSERT(mode < NUM_OF_RFFE_MODE);

	regAddress = B0_PHY_RX_TD_BASE_ADDR + RFE_CTRL_BASE_ADDRESS + ant*4 + mode*0x10;
	val = REGISTER(regAddress);

	return val;
}

/*****************************************************************************
*    Function Name:  RficDriver_rffe_CopyDataBaseValuesFromProgModel.
*    Description:    Save the values for rffe mode.
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_rffe_CopyDataBaseValuesFromProgModel(uint32 inMinorID, uint32 *inSrcAddress,uint32 inLength)
{
	UNUSED_PARAM(inMinorID);	
	UNUSED_PARAM(inSrcAddress);	
	UNUSED_PARAM(inLength);	
}

/*****************************************************************************
*    Function Name:  RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e ant)
*    Description:    set rffe mode
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/

void RficDriver_setRffeMode(CalRFFEState_e mode,AntennaBitmaps_e antMask)
{
	uint32 antIndex;
	uint32 regAddress;
	uint32 regVal;
	uint8 values[MAX_NUM_OF_ANTENNAS];
	
	DEBUG_ASSERT(mode < NUM_OF_RFFE_MODE);
	for (antIndex = 0; antIndex < MAX_NUM_OF_ANTENNAS; antIndex++)
	{
		if (antMask & (1 << antIndex))
		{
			regAddress = RF_TX0_COMMAND + antIndex*4 + mode*0x10;// RF_TX0_COMMAND(INSTEAD OF RFE_CTRL_BASE_ADDRESS)  change RFE_CTRL_BASE_ADDRESS
			regVal = REGISTER(regAddress);
			values[antIndex] = (regVal & SR2_FECTL_MASK) >> SR2_FECTL_SHIFT;
		}
	}
	
	/* rfic_set_antenna_fectl_man uses manual FEM setting.
		 rfic_disable_antenna_fectl  must be called to disable the manual setting*/	
	rfic_set_antenna_fectl_man_par(antMask, values);
}
void RficDriver_enableBbloop(uint8 AntMsk, uint8 HbLb, uint8 LoopType, uint8 IqSel)
{
	rfic_enable_bbloop( AntMsk, HbLb, LoopType, IqSel);
}

void RficDriver_disableBbloop(uint8 AntMsk, uint8 HbLb)
{
	rfic_disable_bbloop( AntMsk, HbLb);
}

void RficDriver_setAntennaFectlMan(uint8 AntMsk, uint8 fectl)
{
	rfic_set_antenna_fectl_man(AntMsk,fectl);
}
void RficDriver_disableRffeManMode(uint8 antMask)
{
	rfic_disable_antenna_fectl(antMask);
}

void RficDriver_RxDcInit(uint8 antMask, uint8 intBw)
{
	rfic_rxon(antMask);
	rfic_txoff(antMask);	
	rfic_set_dcintegrator(antMask, intBw);
}

bool RficDriver_RunRxDcCal(uint8 antMask, uint8 lnaGainIndex, uint16 analogTimeout)
{
	uint32 startTime, timeDiff = 0;
	bool analogSuccess = FALSE; 
	
	rfic_stop_dccancel (antMask);
	RficDriver_SetRxGains(antMask, lnaGainIndex, RXDC_PGC_GAIN_FOR_ANALOG_CALIBRATION_DB);
	rfic_start_dc_clock (antMask);
	rfic_start_dccancel (antMask);
	
	startTime = GET_TSF_TIMER_LOW();
	while ((analogSuccess == FALSE) && (timeDiff < analogTimeout))
	{
		timeDiff = GET_TSF_TIMER_LOW() - startTime;
		analogSuccess = rfic_check_dccancel(antMask);
	}
#ifdef WRX600_BU_LOGS
	ILOG0_DD("RficDriver_RunRxDcCal. analogSuccess %d time %d", analogSuccess, timeDiff);
#endif

	return analogSuccess;
}

void RficDriver_RxDcStopDcCal(uint8 antMask)
{
	rfic_stop_dccancel(antMask);
	rfic_stop_dc_clock(antMask);
}

/********************************************************************************
Function Name:  RficCalDrv_RestoreOperational:

Description:  Restores the RFIC to operational state:
			1) Rfic rx off.
			2) disable LnaNoise.
********************************************************************************/
void RficCalDrv_RestoreOperational(uint8 rxAntMask)
{
	rfic_rxoff(ANTENNA_BITMAP_ALL_AFE_CHANNELS_GEN5);
	
	rfic_disable_lnanoise(rxAntMask);

}

/*****************************************************************************
*    Function Name:  RficDriver_EnableLnaNise(uint8_t AntMsk)
*    Description:    set RFIC + FEM config
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_EnableLnaNoise(uint8_t AntMsk)
{	
	rfic_enable_lnanoise(AntMsk);
	rfic_disable_lna(AntMsk);
}

void RficDriver_DisableLna(uint8 AntMsk)
{	
	rfic_disable_lna(AntMsk);
}
void RficDriver_EnableLna(uint8 AntMsk)
{	
	rfic_enable_lna(AntMsk);
}

/*****************************************************************************
*    Function Name:  RficDriver_EnableLnaNise(uint8_t AntMsk)
*    Description:    set RFIC + FEM config
*    Parameters:     None
*    Returns:        None
*    Remarks:        None
*    Restrictions:   None
******************************************************************************/
void RficDriver_DisableLnaNoise(uint8_t AntMsk)
{	
	rfic_disable_lnanoise(AntMsk);
	rfic_enable_lna(AntMsk);
}
void RficDriver_SetCalibrationData(K_MSG * psMsg)
{
	rfic_calibration_data_t* params;
	uint8 index;
	uint8 numRegsInCis;
	uint8 regNum;

	params = (rfic_calibration_data_t*)pK_MSG_DATA(psMsg);
	numRegsInCis = params->rficCisSize >> 1; //each register is of 16bit so for each one there are 2 bytes in CIS

	for (index = 0; index < numRegsInCis; index++)
	{
		regNum = rficCisIndexToReg[index];
		rficCalibrationData[regNum] = params->rficCis[index << 1] | (params->rficCis[(index << 1) + 1] << 8);
	}
	rficCalibrationData[12] = 0x4210;
	rficCalibrationData[13] = 0x4210;
	rficCalLoaded = 1;
}

void RficDriver_ReadTemperature(int32* temperature)
{
	PhyDrv_StopRisc();
	
	tempSensorStartClock();

	tempSensorRead(temperature);

	tempSensorStopClock();

	PhyDrv_StartRisc();
}


void RficDriver_SelectBandFilterZwdfs(int8 setAPHighLowFilterBank, uint8 driver_fw, uint8 low_chan_num)
{
	uint8 index;
	uint16 value;
	uint16 value_array[MAX_NUM_OF_BITS];
	uint32 antmask;
	uint8 lowHigh;
	
	if (ZwdfsDbSwitch.bandRange[LOW_BAND] != 0)//verify struct is not default
	{
		if(driver_fw == DRIVER)
		{
			if (setAPHighLowFilterBank == SET_AP_LOW_CHANNEL)
			{
				lowHigh = LOW_BAND;
			}
			else //	SET_AP_HIGH_CHANNEL
			{
				lowHigh = HIGH_BAND;
			}

		} 
		else 
		{
			if (low_chan_num <= ZwdfsDbSwitch.bandRange[LOW_BAND])
			{
				lowHigh = LOW_BAND;
			}
			else
			{
				lowHigh = HIGH_BAND;
			}
		}
		antmask = ZwdfsDbSwitch.mask[lowHigh];
		value = ZwdfsDbSwitch.value[lowHigh];
		for (index = 0; index < MAX_NUM_OF_BITS; index++)
		{
			if ((antmask & (1<<index)) != 0)
			{
				value_array[index] = value ;
			}
		}
	
		if(driver_fw == DRIVER)
		{
			PhyDrv_StopRisc();
		}
		
		writeToRficRegister(antmask, REG_RFIC_S2_SHORT_2, value_array,ZWDFS_FECTL_MASK, RFIC_DEFAULT_MASK_SHIFT, TRUE);
		
		if(driver_fw == DRIVER)
		{
			PhyDrv_StartRisc();
		}
	}
}


void RficDriver_SelectBandFilterB(int8 setAPHighLowFilterBank, uint8 driver_fw, uint8 low_chan_num)
{
	uint8 index;
	//uint32 ch;
	uint16 value_array[MAX_NUM_OF_BITS];
	uint16 value;
	uint32 mask;
	uint8 lowHigh;
	
	//ch = HDK_getLowChannel();
	if (ireDbSwitchB.bandRange[LOW_BAND] != 0)//verify struct is not default
	{
		if(driver_fw == DRIVER)
		{
			if (setAPHighLowFilterBank == SET_AP_LOW_CHANNEL)
			{
				lowHigh = LOW_BAND;
			}
			else //	SET_AP_HIGH_CHANNEL
			{
				lowHigh = HIGH_BAND;
			}

		} 
		else 
		{
			if (low_chan_num <= ireDbSwitchB.bandRange[LOW_BAND])
			{
				lowHigh = LOW_BAND;
			}
			else
			{
				lowHigh = HIGH_BAND;
			}
		}
		mask = ireDbSwitchB.mask[lowHigh];
		value = ireDbSwitchB.value[lowHigh];
		for (index = 0; index < MAX_NUM_OF_BITS; index++)
		{
			if ((mask & (1<<index)) != 0)
			{
				value_array[index] = ((value & (1<<index)) << (BAND_RANGE_CONFIG_SHIFT - index));
			}
		}
		//ASSERT(0);
		if(driver_fw == DRIVER)
		{
			PhyDrv_StopRisc();
		}
		writeToRficRegister((AntennaBitmaps_e)mask, REG_RFIC_S3_SHORT_3, value_array, BAND_RANGE_MASK, 0, TRUE);
		if(driver_fw == DRIVER)
		{
			PhyDrv_StartRisc();
		}
	}
}

void RficDriver_handleIRETable_B(void)
{
	uint16 cols = pShramAddress[PSD_NUM_OF_COLS_INDEX];
	uint16 bandIndex;
	for (bandIndex = 0; bandIndex < NUMOFBANDS; bandIndex++)
	{
		ireDbSwitchB.bandRange[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 0];
		ireDbSwitchB.value[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 1];
		ireDbSwitchB.mask[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 2];
	}
}

void RficDriver_handleIRETable_A(void)
{
	uint16 cols = pShramAddress[PSD_NUM_OF_COLS_INDEX];
	uint16 bandIndex;
	for (bandIndex = 0; bandIndex < NUMOFBANDS; bandIndex++)
	{
		ireDbSwitchA.bandRange[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 0];
		ireDbSwitchA.value[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 1];
		ireDbSwitchA.mask[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 2];
	}
}

void RficDriver_handleZwdfsTable(void)
{
	uint16 cols = pShramAddress[PSD_NUM_OF_COLS_INDEX];
	uint16 bandIndex;
	for (bandIndex = 0; bandIndex < NUMOFBANDS; bandIndex++)
	{
		ZwdfsDbSwitch.bandRange[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 0];
		ZwdfsDbSwitch.value[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 1];
		ZwdfsDbSwitch.mask[bandIndex] = pShramAddress[PSD_TABLE_START_INDEX + cols * bandIndex + 2];
	}
}

void RficDriver_IREInit()
{
	memset32(&ireDbSwitchB, 0, CONVERT_BYTES_TO_WORDS(sizeof(IRE_t)));
	memset32(&ireDbSwitchA, 0, CONVERT_BYTES_TO_WORDS(sizeof(IRE_t)));
	memset32(&ZwdfsDbSwitch, 0, CONVERT_BYTES_TO_WORDS(sizeof(IRE_t)));

}

/*****************************************************************************
*    Function Name:  set_lo_frequencies(uint8 freq, uint8 bw)
*    Description:    set lo frequencies {position 0 = frequency of bw 80MHz, pos 1 =freq Bw 40MHz, pos 2 =freq Bw 20MHz
*    Parameters:     BW{80MHz= 0, 40MHz =1, 20MHz =2}
******************************************************************************/
void RficDriver_set_lo_frequencies(void)
{ 			
/* RFICTBD: update to wrx654 fields
	rfic_set_lo_frequencies (HDK_GetLoFrequency(RF_BW_80),HDK_GetLoFrequency(RF_BW_40),HDK_GetLoFrequency(RF_BW_20),HDK_GetLoFrequency(3),HDK_GetLoFrequency(4),HDK_GetLoFrequency(5));			
*/
}

void RficDriver_set_rfchannel(uint8_t AntMsk,  uint8 setChnlBw)
{
	UNUSED_PARAM(AntMsk);	
	UNUSED_PARAM(setChnlBw);	
	/* RFICTBD: update to wrx654 fields
	uint8_t ch_bit;
	
			if (setChnlBw == 80)
				ch_bit = 0;
			else if (setChnlBw == 40)
				ch_bit = 1;
			else if (setChnlBw == 20)
				ch_bit = 2;
			else
				ch_bit = 0;
	RFIC_FCSI_RMW(AntMsk, RF_ANTENNA0_FCSI_SR0,
				 RF_ANTENNA0_FCSI_SR0_SEL_CH__ISMSK,
				 ch_bit << RF_ANTENNA0_FCSI_SR0_SEL_CH__POS);
*/					 				 
}
void RficDriver_set_rxdc(uint8_t AntMsk,  uint8 setChnlBw)
{
	UNUSED_PARAM(AntMsk);	
	UNUSED_PARAM(setChnlBw);
/* RFICTBD: update to wrx654 fields
	
	uint8_t rxdc_bit;
		
	if (setChnlBw == 80)
		rxdc_bit = 0;
	else if (setChnlBw == 40)
		rxdc_bit = 1;
	else if (setChnlBw == 20)
		rxdc_bit = 2;
	else
		rxdc_bit = 0;
	RFIC_FCSI_RMW(AntMsk, RF_ANTENNA0_FCSI_SR0,
				 RF_ANTENNA0_FCSI_SR0_SEL_RXDC__ISMSK,
				 rxdc_bit << RF_ANTENNA0_FCSI_SR0_SEL_RXDC__POS);
*/					 
}
void RficDriver_InitBwChanDC(void)
{
	uint32 val;
	val = 0x4B00;
	RegAccess_Write(SET_BW_CH_DC_20M,val );
	val = 0x2600;
	RegAccess_Write(SET_BW_CH_DC_40M,val );
	val = 0;
	RegAccess_Write(SET_BW_CH_DC_80M,val );
}
void RficDriver_AssociateAntennasToMyBand(uint8 antMask)
{
#ifdef WRX600_BU_LOGS
	ILOG0_DDD("RficDriver_AssociateAntennasToMyBand. pll %d antMask 0x%x hbLb %d", getMyPllId(), antMask, 1-HDK_getBand());
#endif
	rfic_connect_to_pll(antMask, getMyPllId(), TRUE);
}

static uint8 getMyPllId(void)
{
	// Use Band ID as PLL ID
	return ConfigurationManager_GetMyBand();
}

#ifdef SDL_IGNORE_UNUSED_FUNCTION
static uint8 getBandIndex(uint16 freq, uint8 ant, uint16 bandLUT[MAX_NUM_OF_ANTENNAS][MAX_NUMBER_OF_LNA_CROSSING_POINT])
{
	uint8 index;

	for (index = 0; index < MAX_NUMBER_OF_LNA_CROSSING_POINT; index++)
	{
		if(freq < (bandLUT[ant][index]))
		{
			break;
		}
		
	}
	return index;
}
#endif
/***********************************************************************************************************/
/* CoC functions stubs - TODO: implement for gen6                                                          */
/***********************************************************************************************************/

uint8 rfic_get_mode_lp(uint8 AntMsk)//should get the exact index and then dont need a loop
{
	UNUSED_PARAM(AntMsk);	
	return 0;
}

void RficDriver_enable_pll_lowpower(void)
{	
}

void RficDriver_set_rfbandwidthAndMode(uint8 AntMsk, Bandwidth_e bw, uint8 mode)
{
	UNUSED_PARAM(AntMsk);
	UNUSED_PARAM(bw);
	UNUSED_PARAM(mode);
}

void RficDriver_disable_pll_lowpower(void)
{
}
/***********************************************************************************************************/

void RficDriver_dumpRegs(void)
{
	uint8 index = 0;
	uint8 reg;
	
	for (reg = 0; reg < 4; reg++)
	{
		readFromRficRegister(reg,AAA_RficRegs[index]);
		index++;
	}
	for (reg = 0x40; reg < 0x80; reg++)
	{
		readFromRficRegister(reg,AAA_RficRegs[index]);
		index++;
	}
}
#ifdef HDK_RF_EMULATOR
void RficDriver_SetRfEmuBand(uint8 band, uint8 antMask)
{
	uint8 rfEmuHbLbMask;

	if (band == BAND_2_4_GHZ)
	{
		rfEmuHbLbMask = 0;
	}
	else
	{
		rfEmuHbLbMask = 0xF;
	}
	RegAccess_WriteMasked(REG_RF_EMULATOR_RF_CONTROL, antMask, rfEmuHbLbMask);
}
#endif

void RficDriver_SetBandWidth(uint8 antMask)
{
	uint8 bw = HDK_getChannelWidth();
	rfic_set_bandwidth(antMask, 20 << bw);
}

