/**************************************************************************
***************************************************************************
**
** COMPONENT:        Lower MAC/UpperMAC
**
** MODULE:           $Workfile$
**
** VERSION:          $Revision: #1 $
**
** DATED:            $Date: 31/07/2005 $
**
** AUTHOR:           Moche Cohen
**
** DESCRIPTION:      Programming model generic loader.
**
** CHANGE HISTORY:
**
**
** LAST MODIFIED BY:   $Author: prh $
**                     $Modtime$
**
** 	ToDo:	1) update the address of the handhshae section.
			2) send notification over PCI
			3) update calls to Prome/Athe...
			4) Disable interrupts at entry...
****************************************************************************/
/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
//#include "enet_sw.h"
#include <ProgModelLoader.h>
#include "shram_rx_buffer.h"
#include <mt_cnfg.h>
#include "System_Configuration.h"
#include "PhyDriver_API.h"
#include "RficDriver_API.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#define LOG_LOCAL_GID   GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 17


/******************************************************************************/
/***						  Constants										***/
/******************************************************************************/
#define DB_RFIC_MOD_TUNE			             1
#define DB_RFIC_TPC_TUNE			             2
#define DB_RFIC_DRIVER_TUNE			             3
#define DB_RFIC_IQ_CHAIN_TUNE		  	         4
#define DB_RFIC_MIXER_TUNE				         5
#define DB_RFIC_TXBBGAIN_TUNE			         6
#define DB_RFIC_HW_BOARD                         7


/*********************/
/*	  Data Bases	 */
/*********************/
#define DB_TPC0				0x0
#define DB_TPC1				0x1
#define DB_RFIC				0x3
#define DB_RFFE				0x4

#define DB_MAJOR_ID_MASK	0xF
#define DB_MAJOR_ID_OFFSET	4
#define DB_MINOR_ID_MASK	0xF
#define DB_MINOR_ID_OFFSET	0

/*********************/
/*		Devices		 */
/*********************/ 
#define DEV_PROMETHEUS		0x100
#define DEV_ATHENA			0x101
#define DEV_RFIC			0x102
#define DEV_FERRONIA		0x103


#define ARC_UNCACHED_ADDRESS_PREFIX	(0xa0000000)

/******************************************************************************/
/***					Internal Data Structures							***/
/******************************************************************************/
typedef enum SECTION_TYPE
{
	RAW=0,
	DB=1,
	DEVICE=2,
	VERSION_INFO
} eSECTIONTYPE;

typedef struct SECTION_HEADER
{
	uint32 u32Type;
	uint32 u32StartAddress;
	uint32 u32Length;
} sSECTIONHEADER;

typedef struct DB_HANDLER
{
	uint32 ID;
	void (*Handler)(uint32 minorID,uint32 *u32SrcAddress,uint32 u32Length);
} DB_HANDLER_T;

typedef struct DEVICE_HANDLER
{
	uint32 ID;
	void (*Handler)(uint32 *u32SrcAddress,uint32 u32Length);
} DEVICE_HANDLER_T;

/******************************************************************************/
/***						Macros											***/
/******************************************************************************/
#define NUM_OF_DBs		sizeof(DB_handlers)/sizeof(DB_HANDLER_T)
#define NUM_OF_DEVICEs	sizeof(arrDevicesHandlers)/sizeof(DEVICE_HANDLER_T)

#define GET_DB_MAJOR_ID(x)  ((x >> DB_MAJOR_ID_OFFSET) & DB_MAJOR_ID_MASK)
#define GET_DB_MINOR_ID(x)  ((x >> DB_MINOR_ID_OFFSET) & DB_MINOR_ID_MASK)


/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static void vRAWHandler(uint32 *DestAddress,uint32 u32Length, uint32 *u32SrcAddress);
static void vDBHandler(uint32 u32DBID,uint32 u32Length, uint32 *u32SrcAddress);
static void vDeviceHandler(uint32 u32DevID,uint32 u32Length, uint32 *u32SrcAddress);


#if defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)
static __inline void (*GetDBHandler(uint32 inDB_ID))(uint32 minorID,uint32 *u32SrcAddress,uint32 u32Length);
static __inline void (*GetDeviceHandler(uint32 u32DevID))(uint32 *Src,uint32 Length);
#endif

/******************************************************************************/
/***						Static Variables								***/
/******************************************************************************/
//The following arrays hold pointers to handlers for each section type.
#if defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)

static DEVICE_HANDLER_T arrDevicesHandlers[]=
{
	{DEV_RFIC	,RficDriver_CopyRegValuesFromProgModel	}
};

static DB_HANDLER_T DB_handlers[] =
{
	{DB_TPC0		,NULL},
	{DB_TPC1		,NULL},
	{DB_RFIC		,RficDriver_CopyDataBaseValuesFromProgModel},
	{DB_RFFE		,RficDriver_rffe_CopyDataBaseValuesFromProgModel}

};
#else
/*	Currently there no DBs and devices in the UM:
	If there will be a need for the above define the 
	arrays also for UM.
*/
#endif//#if defined(ENET_INC_LMAC) && (ENET_REAL_PHY ==  PHY_REAL_MT_11A_11N)

/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/
/***************************************************************************
**
** NAME         void vLoadProgModel()
**
** PARAMETERS   None
**
** RETURNS      None
**
** DESCRIPTION  
**		Main loading loop. Based on the handshake data, loops through all the
**		sections in the cyclic buffer. For each section, calls the appropriate handler
**		based on the Type.
**		After a chunk of data has beend handled, notifies the upper layer:
**			1) for LM , just updates the handshake area.
**			2) for UM, wait for LM and then notify NDIS over the PCI.
**
****************************************************************************/
void vLoadProgModel(uint32 antRegOffset)
{
	sSECTIONHEADER sCurSection;
	uint32 *p32CurAddress;
	p32CurAddress = hdkGetPsdProgmodelShramAddress();
	//3.Read next section data:
	MEMCPY(&sCurSection,p32CurAddress, sizeof(sSECTIONHEADER));

	while ( (sCurSection.u32StartAddress !=0 ) && (sCurSection.u32Length !=0 ))
	{
		p32CurAddress+=sizeof(sSECTIONHEADER)/sizeof(uint32);

		sCurSection.u32StartAddress |= ARC_UNCACHED_ADDRESS_PREFIX;
		
		switch ( sCurSection.u32Type)
		{
			case RAW:
				vRAWHandler((uint32*)(sCurSection.u32StartAddress + antRegOffset),sCurSection.u32Length,p32CurAddress);
				break;
			case DB:
				vDBHandler(sCurSection.u32StartAddress,sCurSection.u32Length,p32CurAddress);
				break;
			case DEVICE:
				vDeviceHandler(sCurSection.u32StartAddress,sCurSection.u32Length,p32CurAddress);
				break;
			default:
				return;
		};

		//Update address pointer and read next section:
		p32CurAddress+=sCurSection.u32Length;
		//Read next section data:
		MEMCPY(&sCurSection,p32CurAddress, sizeof(sSECTIONHEADER));
	}

}

/******************************************************************************/
/***						Static Functions Definitions					***/
/******************************************************************************/
/***************************************************************************
**
** NAME         void vRAWHandler(uint32 *u32DestAddress,uint32 u32Length, uint32 *u32SrcAddress)
**
** PARAMETERS   DestAddress : destination address of the data.
**				SrcAddress	: Source address (in the cyclic buffer )
**				Length 		: Length ( in 32 bit)
** RETURNS      None
**
** DESCRIPTION  
**			generic loader, just call memcpy. Performed in LMAc only.
**
****************************************************************************/
static void vRAWHandler(uint32 *u32DestAddress,uint32 u32Length, uint32 *u32SrcAddress)
{
	//just memcpy:
	memcpy32(u32DestAddress, u32SrcAddress, u32Length);
}

/***************************************************************************
**
** NAME         vDBHandler
**
** PARAMETERS   u32DBID	- The DB we want to update
**				u32Length - Length of the data
**				u32SrcAddress - Source address(in the cyclic buffer )
**					
** RETURNS      None
**
** DESCRIPTION  
**				Look for the suitable DB handler and execute this handler
****************************************************************************/
static void vDBHandler(uint32 u32DBID,uint32 u32Length, uint32 *u32SrcAddress)
{	
#if defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)
	
	uint8 majorId,minorId;
	void  (*DB_Handler)(uint32 minorID,uint32 *u32SrcAddress,uint32 u32Length);

	majorId = GET_DB_MAJOR_ID(u32DBID); 
	minorId = u32DBID; /* Until we synchronize with progmodel values (generated using BB ) we will use u32DBID instead of GET_DB_MINOR_ID(u32DBID);*/ 

	/* Get DB Handler*/
	DB_Handler=GetDBHandler(majorId);

	/* Run DB Handler*/
	if(DB_Handler!=NULL)
	{
		DB_Handler(minorId,u32SrcAddress, u32Length);
	}
#else
	UNUSED_PARAM(u32DBID);
	UNUSED_PARAM(u32Length);
	UNUSED_PARAM(u32SrcAddress);
#endif /*defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)*/
}


/***************************************************************************
**
** NAME			vDeviceHandler         
**
** PARAMETERS   u32DevID - The device we want to update 
**				u32Length - Length of the data
**				u32SrcAddress -  Source address(in the cyclic buffer )
**							
** RETURNS      None
**
** DESCRIPTION  :
**				Look for the suitable device handler and execute this handler
**
****************************************************************************/
static void vDeviceHandler(uint32 u32DevID,uint32 u32Length, uint32 *u32SrcAddress)
{
#if defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)
	void  (*pu32DevHandler)(uint32 *Src,uint32 Length);

	pu32DevHandler=GetDeviceHandler(u32DevID);
	
	if(pu32DevHandler!=NULL)
	{
		pu32DevHandler( u32SrcAddress, u32Length );
	}
#else
	UNUSED_PARAM(u32DevID);
	UNUSED_PARAM(u32Length);
	UNUSED_PARAM(u32SrcAddress);
#endif /*defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)*/
}

#if defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0)
/***************************************************************************
**
** NAME         GetDBHandler
**
** PARAMETERS   DBID:Databade ID
**						
** RETURNS      DB Handler - handler for the requested DB. 
**
** DESCRIPTION  
**			Looks for a specific DB handler 
**
****************************************************************************/
static __inline  void (*GetDBHandler(uint32 inDB_ID))(uint32 minorID,uint32 *u32SrcAddress,uint32 u32Length)
{
	uint32 i=0;

	for (i=0;i<NUM_OF_DBs;++i)
	{
		if(inDB_ID==DB_handlers[i].ID)
			return DB_handlers[i].Handler;
	}
	return NULL;
}

/***************************************************************************
**
** NAME         GetDeviceHandler
**
** PARAMETERS   DevID: Device ID.
**				
**
** RETURNS      Device Handler - handler for the requested DB.
**
** DESCRIPTION  
**			Looks for a specific device handler and then calls it.
**
****************************************************************************/
static __inline void (*GetDeviceHandler(uint32 u32DevID))(uint32 *Src,uint32 Length)
{
	uint32 i=0;

	for (i=0;i<NUM_OF_DEVICEs;++i)
	{
		if(u32DevID==arrDevicesHandlers[i].ID)
			return arrDevicesHandlers[i].Handler;
	}
	return NULL;
}


#endif /* defined (ENET_INC_LMAC) && !defined (HDK_REL_2_0) */







