// ----------------------------------------------------------------------------
// Project    : abb11ac
// ----------------------------------------------------------------------------
// File       : lib_abb_api.c
// Updated    : 28.10.2014
// Standard   : C
// ----------------------------------------------------------------------------
#include <stdint.h> // standard lib
//#include <math.h>
#include "fcsi_access_functions.h"
#include "target.h" // Lantiq specific lib
#include "lib_abb_api.h"


#include "tb_abb11ac_fcsi2_n_defines.h" // defines such as ports etc.
#include "interface_abb.h"          // macros

/* register packages */
/*#include "ant_fcsi_addrmap.h"
#include "antenna_clockbiasldoanamux_fcsi_regpkg.h"
#include "antenna_dac40m_macro_fcsi_regpkg.h"
#include "antenna_sdadc40m_macro_fcsi_regpkg.h"
#include "antenna_ssiadc_fcsi_regpkg.h"*/
#include "antenna_fcsi_regpkg.h"

/*#include "central_centralstuff_fcsi_regpkg.h"
#include "central_fcsi_addrmap.h"
#include "central_lcpll_macro_fcsi_regpkg.h"*/
#include "central_fcsi_regpkg.h"
/*
 * Short description:
 * direct control lines are forced within this testbench
 * 0.  direct control: 	all direct controls are set to 0
 * 1.  direct control: 	fcsi reset is assigned
 * 2.  direct control: 	power-up bbpll
 * 3.  fcsi control:	pup antenna ldo (wait 10us)
 * 4.  fcsi control:	pup antenna bias
 * 5.  fcsi control:	bring adc and dac into stby (placed at start in this testcase) and running all parallel
 * 6.  fcsi control: 	pup antenna clock (pll lock required)
 * 8   fcsi control: 	reset is asigned to adc, dac, and syncronizers
 * 9.  direct control:	via direct control enable clocks are asigned for sd-adc, dac, ssi-adc (pll lock required)
 * 10. fcsi control:	reset is assigned for rx, tx and ssi
 * 11.
*
*/

void abb11ac_switchoff_lcpll_op_buf_pdn(void)
{
    // at startup all pins should be zero except lcpll_op_buf_pdn-bus
    // with 1 it is in power-down
    //ABB_DIR_CTRL_WRITES(lcpll_op_buf_pdn_d2a, 5, 0x1F);
    ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a, 0x1);
    ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+1, 0x1);
    ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+2, 0x1);
    ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+3, 0x1);
    ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+4, 0x1);
}

void abb11ac_init(void)
{
    ABB_DIR_CTRL_WRITE(usb_spnd_wup_d2a, 0);
    ABB_DIR_CTRL_WRITE(fcsi_rf_oen, 0);
    // wait 5 us;
}

void abb11ac_release_resetn_fcsi(void)
{
   // release fcsi reset
    ABB_DIR_CTRL_WRITE(FCSI2_RST_N,1);
}

void abb11ac_pllsettings(void)
{
    // biasing is in correct mode by reset (reg 0x5CH<15:0>=00F6H(reset))
    
    // PLL settings
	
    // set lcpll output freqs
    ABB_DIR_CTRL_WRITES(lcpll_op_freq_sel_clk1_d2a, 2, 0x1); // 480 MHz
    ABB_DIR_CTRL_WRITES(lcpll_op_freq_sel_clk2_d2a, 2, 0x1); // 320 MHz
    ABB_DIR_CTRL_WRITES(lcpll_op_freq_sel_clk3_d2a, 2, 0x1); // 480 MHz
    ABB_DIR_CTRL_WRITES(lcpll_op_freq_sel_clk4_d2a, 2, 0x0); // 320 MHz
    ABB_DIR_CTRL_WRITES(lcpll_op_freq_sel_clk5_d2a, 2, 0x0); // 1280 MHz
    // wait 2 us
    
}

void abb11ac_pll_pup(void)
{
    ABB_DIR_CTRL_WRITE(lcpll_pup_d2a, 1);
    // wait 2 us 
}

void abb11ac_pll_release_reset(void)
{
    ABB_DIR_CTRL_WRITE(lcpll_resetn_d2a, 1);
    // wait for lock signal
}

void abb11ac_pup_clkldo(void)
{
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG19FSYSLDOCLKLDO, \
     CENTRAL_FCSI_REG19FSYSLDOCLKLDO_PUPLDOCLK__ISMSK, \
     CENTRAL_FCSI_REG19FSYSLDOCLKLDO_PUPLDOCLK_ON__SVAL);
}

void abb11ac_pup_clkdistribution(void)
{
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG21CLOCKING,\
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF1280M__ISMSK &
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF320M__ISMSK  & \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF1280M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF320M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF1280M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF320M__ISMSK, \
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF1280M_67OHM__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF320M_183OHM__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF1280M_6MA__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF320M_2000UA__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF1280M_PUP__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF320M_PUP__SVAL);
     
    // wait 5 us
}

void abb11ac_init1_antenna(int antenna)
{

    // switch on testload
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_TLLDOANT__ISMSK, ANTENNA_FCSI_REGLDO_TLLDOANT_ON__SVAL);
	
    // switch on clk inside antennas
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGCLOCKING, \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC__ISMSK & \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC__ISMSK, \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC_PUP__SVAL | \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC_PUP__SVAL);
    
    /*ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGCLOCKING, \
    ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC__ISMSK, ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC_PUP__SVAL);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGCLOCKING, \
    ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC__ISMSK, ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC_PUP__SVAL);
	*/
	
    // pup antenna ldo
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_PUPLDOANT__ISMSK, ANTENNA_FCSI_REGLDO_PUPLDOANT_ON__SVAL);
    // wait 2 us
}

void abb11ac_init2_antenna(int antenna)
{    
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_TLLDOANT__ISMSK, \
    ANTENNA_FCSI_REGLDO_TLLDOANT_OFF__SVAL);
   	
    /*ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_TLLDOANT__ISMSK, ANTENNA_FCSI_REGLDO_TLLDOANT_OFF__SVAL);
*/	
	
    // pup antenna biasing
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGBIASING, \
    ANTENNA_FCSI_REGBIASING_PUPLB__ISMSK, ANTENNA_FCSI_REGBIASING_PUPLB_ON__SVAL);
    // config is reset
    // wait 1 us
}

// TX-DAC functions
void abb11ac_pup_dacmatrix(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_PUPDAC__ISMSK, ANTENNA_FCSI_IQDACREG0_PUPDAC_ON__SVAL);
    // wait 5 us
}

void abb11ac_pup_dacbuf(int antenna)
{  
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_PUPBUF__ISMSK, ANTENNA_FCSI_IQDACREG0_PUPBUF_ON__SVAL);
    // wait 5 us
}

void abb11ac_release_resetn_dac(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_RESQ__ISMSK, ANTENNA_FCSI_IQDACREG0_RESQ_OFF__SVAL);
}

void abb11ac_dac_11ac_mode(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_FMODE__ISMSK, ANTENNA_FCSI_IQDACREG0_FMODE_11AC__SVAL);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
    ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK & \
    ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK, \
	ANTENNA_FCSI_IQDACREG1_DEMMODE_CLA6F__SVAL | ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL);
    //ANTENNA_FCSI_IQDACREG1_DEMMODE_CLA6F__SVAL | ANTENNA_FCSI_IQDACREG1_ENDEM_ON__SVAL);

}

void abb11ac_release_resetn_txiq(int antenna)
{
    ABB_DIR_CTRL_WRITES(TX_IQ_RSTN, 4, antenna);
}	

void abb11ac_set_tx_on(int antenna)
{
    ABB_DIR_CTRL_WRITES(TX_ON, 4, antenna);
}

void abb11ac_set_tx_data_valid(int antenna)
{
    ABB_DIR_CTRL_WRITES(TX_IQ_DATA_VALID, 4, antenna);
}

void abb11ac_TXDAC_default(int antenna)
{   
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0__H_MSK, ANTENNA_FCSI_IQDACREG0__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
    ANTENNA_FCSI_IQDACREG1__H_MSK, ANTENNA_FCSI_IQDACREG1__INIT);  
}


// SSI functions
void abb11ac_SSI_init(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
     ANTENNA_FCSI_REG0_RESN__ISMSK, ANTENNA_FCSI_REG0_RESN_NO_RESET__SVAL);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
     ANTENNA_FCSI_REG0_PD__ISMSK, ANTENNA_FCSI_REG0_PD_PUP__SVAL);
    // config fits for 11ac mode by reset/default
}

void abb11ac_set_ssi_on(int antenna)
{
    ABB_DIR_CTRL_WRITES(SSI_ON, 4, antenna);
}

void abb11ac_SSIADC_default(int antenna)
{      
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
    ANTENNA_FCSI_REG0__H_MSK, ANTENNA_FCSI_REG0__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
    ANTENNA_FCSI_REG1__H_MSK, ANTENNA_FCSI_REG1__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG2, \
    ANTENNA_FCSI_REG2__H_MSK, ANTENNA_FCSI_REG2__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG3, \
    ANTENNA_FCSI_REG3__H_MSK, ANTENNA_FCSI_REG3__INIT);
}


// RX-ADC functions
void abb11ac_ensleep_on_rxadc(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
    ANTENNA_FCSI_IQADCREG1_ENSLEEP__ISMSK, ANTENNA_FCSI_IQADCREG1_ENSLEEP_ON__SVAL);
    // wait 3 us
}
      
      
void abb11ac_config_rx_adc(int antenna, int ctune)
{
    // ADC pup, but still not active because of rx_on is 0
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
    ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK & \
    ANTENNA_FCSI_IQADCREG0_PDADC__ISMSK & \
    ANTENNA_FCSI_IQADCREG0_ENLOWPOW__ISMSK & \
    ANTENNA_FCSI_IQADCREG0_EN20M__ISMSK & \
    ANTENNA_FCSI_IQADCREG0_ENCD__ISMSK & \
    ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE__ISMSK, \
    ctune | ANTENNA_FCSI_IQADCREG0_PDADC_PUP__SVAL | \
    ANTENNA_FCSI_IQADCREG0_ENLOWPOW_DEFAULT__SVAL | \
    ANTENNA_FCSI_IQADCREG0_EN20M_DEFAULT__SVAL | \
    ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL | \
    ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE_DISLFRESETPULSE__SVAL);
	
    
    /* ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
    ANTENNA_FCSI_IQADCREG1_ENSLEEP__ISMSK, ANTENNA_FCSI_IQADCREG1_ENSLEEP_ON__SVAL);*/
    // reg 0x42 and 0x43 are by default in reset
    // ctune & ADC clk inv from efuse
}

void abb11ac_release_resetn_rxiq(int antenna)
{
    ABB_DIR_CTRL_WRITES(RX_IQ_RSTN, 4, antenna);
}

void abb11ac_set_rx_on(int antenna)
{
    ABB_DIR_CTRL_WRITES(RX_ON, 4, antenna);
}

void abb11ac_RXADC_default(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
    ANTENNA_FCSI_IQADCREG0__H_MSK, ANTENNA_FCSI_IQADCREG0__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
    ANTENNA_FCSI_IQADCREG1__H_MSK, ANTENNA_FCSI_IQADCREG1__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG2, \
    ANTENNA_FCSI_IQADCREG2__H_MSK, ANTENNA_FCSI_IQADCREG2__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG3, \
    ANTENNA_FCSI_IQADCREG3__H_MSK, ANTENNA_FCSI_IQADCREG3__INIT);
}


int abb11ac_rcmeas(uint16_t *rccount)
{
    uint16_t data[NR_OF_ANTENNAS];
    uint16_t measdone = 0;
    uint16_t countvalue1 = 0;
    uint16_t countvalue2 = 0;
    uint16_t count_calc = 0;
    uint16_t i = 0;
    
    // settings and start meas1
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING__ISMSK, CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING_PUP__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK, CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_NORESET__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, CENTRAL_FCSI_REG22RCMEAS_START_START__SVAL);

    
    // read measdone and countvalue
    while ((measdone == 0) && (i < 15))
    {
        ABB_FCSI_Read(ABB_CENTRAL, CENTRAL_FCSI_REG23RCMEASI, data); // expected low
	measdone = CENTRAL_FCSI_REG23RCMEASI_RCTUNINGMEASDONE__GET(data[ABB_CENTRAL_INDEX]);
	i = i+1;
    }
    
    if (measdone == 1)
    {
        // countvalue1 = CENTRAL_FCSI_REG23RCMEASI_RCCOUNT__GET(data[ABB_CENTRAL_INDEX]);
	abb11ac_rcmeas_read_measdone(&countvalue1);
    }
    else
    {
       return -1;
    }
    

    // reset rcmeas, change swap bit, and start 2nd measurement
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_RESET__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_START_OFF__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_SWAP__ISMSK, CENTRAL_FCSI_REG22RCMEAS_SWAP_SWAP__SVAL);
	
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK, CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_NORESET__SVAL);
	
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, CENTRAL_FCSI_REG22RCMEAS_START_START__SVAL);
	
	
    // read measdone and 2nd countvalue
    measdone = 0;
    i = 0;
    
    while ((measdone == 0) && (i < 15))
    {
        ABB_FCSI_Read(ABB_CENTRAL, CENTRAL_FCSI_REG23RCMEASI, data); // expected low
	measdone = CENTRAL_FCSI_REG23RCMEASI_RCTUNINGMEASDONE__GET(data[ABB_CENTRAL_INDEX]);
	i = i+1;
    }
    
    if (measdone == 1)
    {
        //countvalue2 = CENTRAL_FCSI_REG23RCMEASI_RCCOUNT__GET(data[ABB_CENTRAL_INDEX]);
	abb11ac_rcmeas_read_measdone(&countvalue2);
    }
    else
    {
       return -1;
    }	
       
      count_calc = (countvalue1 + countvalue2)/2;
       
    // set rcmeas back to default values
    /*ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_SWAP__ISMSK, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING_PD__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_RESET__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_START_OFF__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_SWAP_OFF__SVAL);*/
	
    abb11ac_rcmeas_default();
	
    rccount[0] = count_calc;
    rccount[1] = countvalue1;
    rccount[2] = countvalue2;
	
    return (int) count_calc;  
}





uint16_t abb11ac_rcmeas_read_measdone(uint16_t *countval)
{
    uint16_t data[NR_OF_ANTENNAS];
    ABB_FCSI_Read(ABB_CENTRAL, CENTRAL_FCSI_REG23RCMEASI, data); // expected low
    *countval = CENTRAL_FCSI_REG23RCMEASI_RCCOUNT__GET(data[ABB_CENTRAL_INDEX]);
    return CENTRAL_FCSI_REG23RCMEASI_RCTUNINGMEASDONE__GET(data[ABB_CENTRAL_INDEX]);
}

void abb11ac_rcmeas_default()
{
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_SWAP__ISMSK, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING_PD__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_RESET__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_START_OFF__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_SWAP_OFF__SVAL);
}

/*// rcmeas
void abb11ac_rcmeas1(void)
{
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING__ISMSK, CENTRAL_FCSI_REG22RCMEAS_PUPRCTUNING_PUP__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK, CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_NORESET__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, CENTRAL_FCSI_REG22RCMEAS_START_START__SVAL);
}

uint16_t abb11ac_rcmeas2()
{
    uint16_t countval1;
    
    // read countvalue of first measurement
    countval1 = abb11ac_rcmeas_read_measdone();
  
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK & \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_RESET__SVAL | \
	CENTRAL_FCSI_REG22RCMEAS_START_OFF__SVAL);
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_SWAP__ISMSK, CENTRAL_FCSI_REG22RCMEAS_SWAP_SWAP__SVAL);
	
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_RESETNCNT__ISMSK, CENTRAL_FCSI_REG22RCMEAS_RESETNCNT_NORESET__SVAL);
	
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, \
	CENTRAL_FCSI_REG22RCMEAS_START__ISMSK, CENTRAL_FCSI_REG22RCMEAS_START_START__SVAL);
    return countval1;
}

uint16_t abb11ac_rcmeas3(uint16_t countval1)
{
    uint16_t countval2;
    uint16_t count_calc;
    
    // read countvalue of second measurement
    countval2 = abb11ac_rcmeas_read_measdone();
    
    count_calc = (countval1 + countval2)/2;
    
    
    // set rcmeas back to default values
    abb11ac_rcmeas_default();
    return count_calc;  
}*/





int abb11ac_adctune(uint16_t count_calc, uint16_t count_nom, uint16_t* ctune)
{
  uint16_t ctune_calc = 0;
  if (count_calc != 0)
  {
     // formula updated on 18. 9. 2017 (ctune value increased by 1)
     /*ctune_calc = ((uint16_t) (16.5 * ((double) count_nom/(double) count_calc) - 7.5) + 1);
     if (ctune_calc > 15)
     {
         ctune_calc = 15;
     }
     if (ctune_calc < 2)
     {
         ctune_calc = 2;
     }
     *ctune = ctune_calc;
      
     //ctune[1] = ctune[0] - 2;
     return 0;*/
     
     // VT-splitlot shows countcalc values from
     // 300 to 450 and no temperature dependancy
     // ctune 5 up to 12 are realistic
     if(count_calc < 271)
     {
         ctune_calc = 15;
     }
     else if(count_calc < 285)
     {
         ctune_calc = 14;
     }
     else if(count_calc < 299)
     {
         ctune_calc = 13;
     }
     else if(count_calc < 315)
     {
         ctune_calc = 12;
     }
     else if(count_calc < 333)
     {
         ctune_calc = 11;
     }
     else if(count_calc < 354)
     {
         ctune_calc = 10;
     }
     else if(count_calc < 376)
     {
         ctune_calc = 9;
     }
     else if(count_calc < 402)
     {
         ctune_calc = 8;
     }
     else if(count_calc < 432)
     {
         ctune_calc = 7;
     }
     else if(count_calc < 466)
     {
         ctune_calc = 6;
     }
     else if(count_calc < 507)
     {
         ctune_calc = 5;
     }
     else if(count_calc < 555)
     {
         ctune_calc = 4;
     }
     else if(count_calc < 614)
     {
         ctune_calc = 3;
     }
     else // < 686 -> 1
     {
         ctune_calc = 2;
     }
     *ctune = ctune_calc;
      
     return 0;
  }
  else
  {
     return -1;
  }
}

void abb11ac_observe_vref_iref(void)
{
   // workaround: anamuxsel set to 2 -> biasldo @ BB_VREF; gnd @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_2__SVAL); 
	
   // anamuxsel set to 1 -> vbg @ BB_VREF, itrim @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_1__SVAL); 
}


void abb11ac_observe_biasldo(void)
{
   // anamuxsel set to 2 -> biasldo @ BB_VREF; gnd @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_2__SVAL); 
}


void abb11ac_observe_pll(int diagmodsel)
{
   // anamuxsel set to 3 -> pll @ BB_VREF; gnd @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_3__SVAL); 
	
   // diagmodsel -> refer to LCPLLDIAG Sheet
   // value already converted to position; therefore multiply with 8, as it starts with bit3
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_FCSI10, \
	CENTRAL_FCSI_FCSI10_DIAGENABLE__ISMSK & \
	CENTRAL_FCSI_FCSI10_DIAGMODULESEL__ISMSK, CENTRAL_FCSI_FCSI10_DIAGENABLE_ON__VAL | (diagmodsel*8)); 
   		
}

void abb11ac_observe_mosmeas_vref_iref(void)
{
   // mose meas-settins
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG25APM, \
   CENTRAL_FCSI_REG25APM_PDAPM__ISMSK & \
   CENTRAL_FCSI_REG25APM_RESETN__ISMSK & \
   CENTRAL_FCSI_REG25APM_TESTVREF800MV__ISMSK,
   CENTRAL_FCSI_REG25APM_PDAPM_PUP__SVAL | \
   CENTRAL_FCSI_REG25APM_RESETN_NORES__SVAL | \
   CENTRAL_FCSI_REG25APM_TESTVREF800MV_ON__SVAL);
   
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG24APM, \
   CENTRAL_FCSI_REG24APM_MEASUREREF__ISMSK, \
   CENTRAL_FCSI_REG24APM_MEASUREREF_ON__SVAL);
   
   // anamuxsel set to 4 -> mosmeas800mV @ BB_VREF; mosmeas100u @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_4__SVAL);
   
}

void abb11ac_observe_antldo(int antindex)
{ 
   // anamuxsel set to 5 -> test_ant_n @ BB_VREF; test_ant_p @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK & \
	CENTRAL_FCSI_REG29ANAMUX_ENMUXANT__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_5__SVAL | (antindex << 4)); 
 	
   ABB_FCSI_RMW(1 << antindex, ANTENNA_FCSI_REGANAMUX, \
        ANTENNA_FCSI_REGANAMUX_SELMUXANT__ISMSK, ANTENNA_FCSI_REGANAMUX_SELMUXANT_LDO__SVAL);	
}

void abb11ac_observe_antssi_vref(int antindex)
{
   int ant = 1 << antindex;
   abb11ac_SSI_init(ant);
   //debug_printf("antinex = %d; 2^antindex %d\n",antindex, ant);
   // anamuxsel set to 5 -> test_ant_n @ BB_VREF; test_ant_p @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK & \
	CENTRAL_FCSI_REG29ANAMUX_ENMUXANT__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_5__SVAL | (antindex << 4)); 
   ABB_FCSI_RMW(ant, ANTENNA_FCSI_REGANAMUX, \
        ANTENNA_FCSI_REGANAMUX_SELMUXANT__ISMSK, ANTENNA_FCSI_REGANAMUX_SELMUXANT_SSI__SVAL);
   // SSI settings		
   ABB_FCSI_RMW(ant, ANTENNA_FCSI_REG0, \
        ANTENNA_FCSI_REG0_OBSPP__ISMSK & \
	ANTENNA_FCSI_REG0_OBSPPEN__ISMSK, \
	ANTENNA_FCSI_REG0_OBSPP_OBS4__SVAL | ANTENNA_FCSI_REG0_OBSPPEN_EN__SVAL);		
}

void abb11ac_observe_antssi_vcm(int antindex)
{
   int ant = 1 << antindex;
   abb11ac_SSI_init(ant);
   // anamuxsel set to 5 -> test_ant_n @ BB_VREF; test_ant_p @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK & \
	CENTRAL_FCSI_REG29ANAMUX_ENMUXANT__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_5__SVAL | (antindex << 4)); 
   ABB_FCSI_RMW(ant, ANTENNA_FCSI_REGANAMUX, \
        ANTENNA_FCSI_REGANAMUX_SELMUXANT__ISMSK, ANTENNA_FCSI_REGANAMUX_SELMUXANT_SSI__SVAL);
   // SSI settings		
   ABB_FCSI_RMW(ant, ANTENNA_FCSI_REG0, \
        ANTENNA_FCSI_REG0_OBSPP__ISMSK & \
	ANTENNA_FCSI_REG0_OBSPPEN__ISMSK, \
	ANTENNA_FCSI_REG0_OBSPP_OBS7__SVAL | ANTENNA_FCSI_REG0_OBSPPEN_EN__SVAL);		
}

void abb11ac_observe_fsysldo(void)
{
   // anamuxsel set to 6  -> gnd @ BB_VREF; fsysldo @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_6__SVAL); 	
}

void abb11ac_observe_clkldo(void)
{
   // anamuxsel set to 7 -> gnd @ BB_VREF; clkldo @ BB_ITRIM
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_7__SVAL); 	
}

void abb11ac_observe_default(void)
{
    // set default to central anamux
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX__H_MSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__INIT);
	
    // set default to ant anamux	
    ABB_FCSI_RMW(ABB_ANT3210, ANTENNA_FCSI_REGANAMUX, \
        ANTENNA_FCSI_REGANAMUX_SELMUXANT__ISMSK, ANTENNA_FCSI_REGANAMUX_SELMUXANT_LDO__SVAL);
	
    // set SSI values back
    ABB_FCSI_RMW(ABB_ANT3210, ANTENNA_FCSI_REG0, \
        ANTENNA_FCSI_REG0_OBSPP__ISMSK & \
	ANTENNA_FCSI_REG0_OBSPPEN__ISMSK, \
	ANTENNA_FCSI_REG0_OBSPP_OBS0__SVAL | ANTENNA_FCSI_REG0_OBSPPEN_DIS__SVAL);
	
   // mose meas-settins
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG25APM, \
       CENTRAL_FCSI_REG25APM_PDAPM__ISMSK & \
       CENTRAL_FCSI_REG25APM_RESETN__ISMSK & \
       CENTRAL_FCSI_REG25APM_TESTVREF800MV__ISMSK,
       CENTRAL_FCSI_REG25APM_PDAPM_PD__SVAL | \
       CENTRAL_FCSI_REG25APM_RESETN_RES__SVAL | \
       CENTRAL_FCSI_REG25APM_TESTVREF800MV_OFF__SVAL);
   
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG24APM, \
       CENTRAL_FCSI_REG24APM_MEASUREREF__ISMSK, \
       CENTRAL_FCSI_REG24APM_MEASUREREF_OFF__SVAL);
   
   // PLL settings   
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_FCSI10, \
	CENTRAL_FCSI_FCSI10_DIAGENABLE__ISMSK & \
	CENTRAL_FCSI_FCSI10_DIAGMODULESEL__ISMSK, CENTRAL_FCSI_FCSI10_DIAGENABLE_OFF__VAL | 0); 	
}

void abb11ac_bias_softfuse(int vref, int iref)
{
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG28BIAS, \
	CENTRAL_FCSI_REG28BIAS_BGPVREFCONF__ISMSK & \
	CENTRAL_FCSI_REG28BIAS_IBIASTRIM__ISMSK, (vref << 8) | (iref << 4));
}


/*void abb11ac_adc_efuse(int antenna, int ctune, int clkinv)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
        ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK, ctune);
	ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
        ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, clkinv);
}*/

void abb11ac_antenna_default(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGCLOCKING, \
        ANTENNA_FCSI_REGCLOCKING__H_MSK, ANTENNA_FCSI_REGCLOCKING__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGBIASING, \
        ANTENNA_FCSI_REGBIASING__H_MSK, ANTENNA_FCSI_REGBIASING__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
        ANTENNA_FCSI_REGLDO__H_MSK, ANTENNA_FCSI_REGLDO__INIT);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGANAMUX, \
        ANTENNA_FCSI_REGANAMUX__H_MSK, ANTENNA_FCSI_REGANAMUX__INIT);
}

void abb11ac_rffcsi(int value_smt, int value_sr, int value_e)
{
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG30RFFCSI, \
            CENTRAL_FCSI_REG30RFFCSI_SMT__ISMSK & \
	    CENTRAL_FCSI_REG30RFFCSI_SR__ISMSK & \
	    CENTRAL_FCSI_REG30RFFCSI_E__ISMSK, value_smt | value_sr | value_e);
}

void abb11ac_rf_oen(int value)
{
    ABB_DIR_CTRL_WRITE(fcsi_rf_oen, value);
}

void abb11ac_rf_usb(int value)
{
    ABB_DIR_CTRL_WRITE(usb_spnd_wup_d2a, value);
}


void abb11ac_anamux_central_programming(int value)
{
   // workaround: anamuxsel set to 7 -> gnd @ testn (vref-node); VDDA1V1CLK @ testp (itrim-node)
   // also set to 6 would be possible -> gnd @ testn (vref-node); VDDA1V1FSYS @ testp (itrim-node)
   // no setting were gnd at both test-nodes (as in GRX500)
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL_7__SVAL); 
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ANAMUXSEL__ISMSK, value); 
}

void abb11ac_anamux_central_antsel(int value)
{
   ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, \
	CENTRAL_FCSI_REG29ANAMUX_ENMUXANT__ISMSK, value*16); 
}

void abb11ac_anamux_ant_sel(int antenna, int value)
{
   ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGANAMUX, \
	ANTENNA_FCSI_REGANAMUX_SELMUXANT__ISMSK, value); 
}


void abb11ac_ssi_test(int antenna)
{
    // set input for ssi (wrx, q-dac, i-dac)
    //ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
//	ANTENNA_FCSI_REG1_SSISEL__ISMSK, ANTENNA_FCSI_REG1_SSISEL_IDAC__SVAL);
	
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
	ANTENNA_FCSI_REG0_OBSPPEN__ISMSK, ANTENNA_FCSI_REG0_OBSPPEN_EN__SVAL);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
	ANTENNA_FCSI_REG0_OBSPP__ISMSK, ANTENNA_FCSI_REG0_OBSPP_OBS1__SVAL);
    //ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
//	ANTENNA_FCSI_REG0_OBSPPSEL__ISMSK, ANTENNA_FCSI_REG0_OBSPPSEL_PP1__SVAL);	
}


void abb11ac_central_init(void)
{
    /*// pup clock ldo
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG19FSYSLDOCLKLDO, \
     CENTRAL_FCSI_REG19FSYSLDOCLKLDO_PUPLDOCLK__ISMSK, \
     CENTRAL_FCSI_REG19FSYSLDOCLKLDO_PUPLDOCLK_ON__SVAL);*/
     
    // pup clock ldo: reset = 0x0201 -> 0x0301
    ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG19FSYSLDOCLKLDO, 0x0301); 
    
    /*// pup clk-distribution
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG21CLOCKING,\
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF1280M__ISMSK &
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF320M__ISMSK  & \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF1280M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF320M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF1280M__ISMSK & \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF320M__ISMSK, \
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF1280M_67OHM__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_RSELCMLBUF320M_183OHM__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF1280M_6MA__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_IPRGCMLBUF320M_2000UA__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF1280M_PUP__SVAL | \
     CENTRAL_FCSI_REG21CLOCKING_PDCMLBUF320M_PUP__SVAL);*/
    
    // pup and program clock distribution: reset = 0x0399 -> 0x0055 
    ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG21CLOCKING,0x0055);
    
     
    // Write these two registers to initial value, because reset does not come through
    // at startup phase, as its supply is not there -> bug
    ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, CENTRAL_FCSI_REG22RCMEAS__INIT);
   
    // read register
    ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG23RCMEASI, CENTRAL_FCSI_REG23RCMEASI__INIT);   
}

void abb11ac_antenna_init(int antenna)
{
    // switch on testload
    // not used FCSI_WRITE as fusing was done before and this register contains fuse bits
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_TLLDOANT__ISMSK, ANTENNA_FCSI_REGLDO_TLLDOANT_ON__SVAL);
   
    // pup antenna biasing
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGBIASING, \
    ANTENNA_FCSI_REGBIASING_PUPLB__ISMSK, ANTENNA_FCSI_REGBIASING_PUPLB_ON__SVAL);	
    
    
    // switch on clk inside antennas
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGCLOCKING, \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC__ISMSK & \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC__ISMSK, \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSDAC_PUP__SVAL | \
     ANTENNA_FCSI_REGCLOCKING_PDCML2CMOSADC_PUP__SVAL);
    
	
    // pup antenna ldo
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_PUPLDOANT__ISMSK, ANTENNA_FCSI_REGLDO_PUPLDOANT_ON__SVAL);
    
    // switch off testload, prog ldovoltage
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_TLLDOANT__ISMSK, \
    ANTENNA_FCSI_REGLDO_TLLDOANT_OFF__SVAL);
    
    
    
    // ensleep of SDADC
    // not used FCSI WRITE as there is adcclkinv fuse bit in this register
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
    ANTENNA_FCSI_IQADCREG1_ENSLEEP__ISMSK, ANTENNA_FCSI_IQADCREG1_ENSLEEP_ON__SVAL);
    
    
    // set CLA6F algorithm and activate it
    // not used FCSI_WRTIE as there is absgain fuse bits in this register
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
    ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK & \
    ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK, \
    ANTENNA_FCSI_IQDACREG1_DEMMODE_CLA6F__SVAL | ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL);
 //  ANTENNA_FCSI_IQDACREG1_DEMMODE_CLA6F__SVAL | ANTENNA_FCSI_IQDACREG1_ENDEM_ON__SVAL);
   
    // release resetn dac
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_RESQ__ISMSK, ANTENNA_FCSI_IQDACREG0_RESQ_OFF__SVAL);
    
}


void abb11ac_ant_config(int antenna, int bandwidth, int ctune)
{
    if(bandwidth == 40)
    {
      // SDADC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
      ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_EN20M__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_PDADC__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENCD__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE__ISMSK, \
      ctune | ANTENNA_FCSI_IQADCREG0_EN20M_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_PDADC_PUP__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE_DISLFRESETPULSE__SVAL);
	  
	  // Change ADC IBIAS to +31.25u for 11AC
	  ABB_FCSI_RMW(antenna,  ANTENNA_FCSI_REGBIASING, \
	  ANTENNA_FCSI_REGBIASING_PROGSDADC25U__ISMSK,  \
      ANTENNA_FCSI_REGBIASING_PROGSDADC25U_31U25__SVAL);
	  
      // DAC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
      ANTENNA_FCSI_IQDACREG0_FMODE__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPBUF__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPDAC__ISMSK, \
      ANTENNA_FCSI_IQDACREG0_FMODE_11AC__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPBUF_ON__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPDAC_ON__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
      ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK & \
      ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK, \
      ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL | \
      ANTENNA_FCSI_IQDACREG1_DEMMODE_CLA6F__SVAL);
    //  ANTENNA_FCSI_IQDACREG1_ENDEM_ON__SVAL | \
      
      // SSI settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
      ANTENNA_FCSI_REG0_PD__ISMSK & \
      ANTENNA_FCSI_REG0_RESN__ISMSK, \
      ANTENNA_FCSI_REG0_PD_PUP__SVAL | \
      ANTENNA_FCSI_REG0_RESN_NO_RESET__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF__ISMSK, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF_11AC__SVAL);
    }
    else if(bandwidth == 20)
    {
      // SDADC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
      ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_EN20M__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_PDADC__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENCD__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE__ISMSK, \
      (ctune -2) | ANTENNA_FCSI_IQADCREG0_EN20M_EN20MHZ__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_PDADC_PUP__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE_DISLFRESETPULSE__SVAL);
      
	  ABB_FCSI_RMW(antenna,  ANTENNA_FCSI_REGBIASING, \
	  ANTENNA_FCSI_REGBIASING_PROGSDADC25U__ISMSK,  \
      ANTENNA_FCSI_REGBIASING_PROGSDADC25U_25U__SVAL);

      // DAC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
      ANTENNA_FCSI_IQDACREG0_FMODE__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPBUF__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPDAC__ISMSK, \
      ANTENNA_FCSI_IQDACREG0_FMODE_CB__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPBUF_ON__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPDAC_ON__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
      ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK & \
      ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK, \
      ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL | \
      ANTENNA_FCSI_IQDACREG1_DEMMODE_FIB4__SVAL);
      
      // SSI settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
      ANTENNA_FCSI_REG0_PD__ISMSK & \
      ANTENNA_FCSI_REG0_RESN__ISMSK, \
      ANTENNA_FCSI_REG0_PD_PUP__SVAL | \
      ANTENNA_FCSI_REG0_RESN_NO_RESET__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF__ISMSK, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF_11N__SVAL);
    }
    else if(bandwidth == 10) // low power mode
    {
      // SDADC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
      ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_EN20M__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_PDADC__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENCD__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE__ISMSK, \
      ctune | ANTENNA_FCSI_IQADCREG0_EN20M_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW_ENLOWPOW__SVAL | \
      ANTENNA_FCSI_IQADCREG0_PDADC_PUP__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE_DISLFRESETPULSE__SVAL);

	  ABB_FCSI_RMW(antenna,  ANTENNA_FCSI_REGBIASING, \
	  ANTENNA_FCSI_REGBIASING_PROGSDADC25U__ISMSK,  \
      ANTENNA_FCSI_REGBIASING_PROGSDADC25U_25U__SVAL);
	  
      // DAC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
      ANTENNA_FCSI_IQDACREG0_FMODE__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPBUF__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPDAC__ISMSK, \
      ANTENNA_FCSI_IQDACREG0_FMODE_NCB__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPBUF_ON__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPDAC_ON__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
      ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK & \
      ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK, \
      ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL | \
      ANTENNA_FCSI_IQDACREG1_DEMMODE_FIB4__SVAL);
      
      // SSI settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
      ANTENNA_FCSI_REG0_PD__ISMSK & \
      ANTENNA_FCSI_REG0_RESN__ISMSK, \
      ANTENNA_FCSI_REG0_PD_PUP__SVAL | \
      ANTENNA_FCSI_REG0_RESN_NO_RESET__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF__ISMSK, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF_LP__SVAL);
    }
    else if(bandwidth == 0) // power-down mode
    {
      // SDADC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG0, \
      ANTENNA_FCSI_IQADCREG0_CTUNE__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_EN20M__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_PDADC__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENCD__ISMSK & \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE__ISMSK, \
      ANTENNA_FCSI_IQADCREG0_CTUNE_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_EN20M_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLOWPOW_DEFAULT__SVAL | \
      ANTENNA_FCSI_IQADCREG0_PDADC_PD__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENCD_ENCD__SVAL | \
      ANTENNA_FCSI_IQADCREG0_ENLFRESETPULSE_DISLFRESETPULSE__SVAL);      
 
      // DAC settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
      ANTENNA_FCSI_IQDACREG0_FMODE__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPBUF__ISMSK & \
      ANTENNA_FCSI_IQDACREG0_PUPDAC__ISMSK, \
      ANTENNA_FCSI_IQDACREG0_FMODE_CB__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPBUF_OFF__SVAL | \
      ANTENNA_FCSI_IQDACREG0_PUPDAC_OFF__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG1, \
      ANTENNA_FCSI_IQDACREG1_ENDEM__ISMSK & \
      ANTENNA_FCSI_IQDACREG1_DEMMODE__ISMSK, \
      ANTENNA_FCSI_IQDACREG1_ENDEM_OFF__SVAL | \
      ANTENNA_FCSI_IQDACREG1_DEMMODE_FIB4__SVAL);
      
      // SSI settings
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG0, \
      ANTENNA_FCSI_REG0_PD__ISMSK & \
      ANTENNA_FCSI_REG0_RESN__ISMSK, \
      ANTENNA_FCSI_REG0_PD_PD__SVAL | \
      ANTENNA_FCSI_REG0_RESN_RESET__SVAL);
      
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF__ISMSK, \
      ANTENNA_FCSI_REG1_SSIBUFBIASCONF_11AC__SVAL);
    }
}

void abb11ac_ssi_se_mode(int antenna)
{
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSISEL__ISMSK, \
      ANTENNA_FCSI_REG1_SSISEL_SINGLE__SVAL);	
}

void abb11ac_ssi_diff_mode(int antenna)
{
      ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REG1, \
      ANTENNA_FCSI_REG1_SSISEL__ISMSK, \
      ANTENNA_FCSI_REG1_SSISEL_WRX__SVAL);	
}


void abb11ac_reset_FCSI_central()
{
   // CENTRAL
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI0, CENTRAL_FCSI_FCSI0__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI1, CENTRAL_FCSI_FCSI1__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI2, CENTRAL_FCSI_FCSI2__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI3, CENTRAL_FCSI_FCSI3__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI4, CENTRAL_FCSI_FCSI4__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI5, CENTRAL_FCSI_FCSI5__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI6, CENTRAL_FCSI_FCSI6__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI7, CENTRAL_FCSI_FCSI7__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI8, CENTRAL_FCSI_FCSI8__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI9, CENTRAL_FCSI_FCSI9__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI10, CENTRAL_FCSI_FCSI10__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_FCSI11, CENTRAL_FCSI_FCSI11__INIT);
   
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG16LCPLL, CENTRAL_FCSI_REG16LCPLL__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG17LCPLL, CENTRAL_FCSI_REG17LCPLL__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG18LCPLL, CENTRAL_FCSI_REG18LCPLL__INIT);
   
   // init value + pup of clkldo to keep supply for antenna fcsi
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG19FSYSLDOCLKLDO, CENTRAL_FCSI_REG19FSYSLDOCLKLDO__INIT | CENTRAL_FCSI_REG19FSYSLDOCLKLDO_PUPLDOCLK_ON__SVAL);
   
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG20FSYS, CENTRAL_FCSI_REG20FSYS__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG21CLOCKING, CENTRAL_FCSI_REG21CLOCKING__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG22RCMEAS, CENTRAL_FCSI_REG22RCMEAS__INIT);
   
   // read register
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG23RCMEASI, CENTRAL_FCSI_REG23RCMEASI__INIT);
   
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG24APM, CENTRAL_FCSI_REG24APM__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG25APM, CENTRAL_FCSI_REG25APM__INIT);
   
   // read register
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG26APMI, CENTRAL_FCSI_REG26APMI__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG27APMI, CENTRAL_FCSI_REG27APMI__INIT);
   
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG28BIAS, CENTRAL_FCSI_REG28BIAS__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG29ANAMUX, CENTRAL_FCSI_REG29ANAMUX__INIT);
   ABB_FCSI_Write(ABB_CENTRAL, CENTRAL_FCSI_REG30RFFCSI, CENTRAL_FCSI_REG30RFFCSI__INIT);
}   


void abb11ac_reset_FCSI_antenna()
{
   // Antenna
   // SDADC (RX)
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQADCREG0, ANTENNA_FCSI_IQADCREG0__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQADCREG1, ANTENNA_FCSI_IQADCREG1__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQADCREG2, ANTENNA_FCSI_IQADCREG2__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQADCREG3, ANTENNA_FCSI_IQADCREG3__INIT);
   
   // infra
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REGCLOCKING, ANTENNA_FCSI_REGCLOCKING__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REGBIASING, ANTENNA_FCSI_REGBIASING__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REGLDO, ANTENNA_FCSI_REGLDO__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REGANAMUX, ANTENNA_FCSI_REGANAMUX__INIT);
   
   // DAC (TX)
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQDACREG0, ANTENNA_FCSI_IQDACREG0__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_IQDACREG1, ANTENNA_FCSI_IQDACREG1__INIT);
   
   // SSI
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REG0, ANTENNA_FCSI_REG0__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REG1, ANTENNA_FCSI_REG1__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REG2, ANTENNA_FCSI_REG2__INIT);
   ABB_FCSI_Write(ABB_ANT3210, ANTENNA_FCSI_REG3, ANTENNA_FCSI_REG3__INIT);
}



void abb11ac_reset_FCSI()
{
   abb11ac_reset_FCSI_central();
   abb11ac_reset_FCSI_antenna();
}


void abb11ac_reset_pulse_dac(int antenna)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_RESQ__ISMSK, ANTENNA_FCSI_IQDACREG0_RESQ_ON__SVAL);
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQDACREG0, \
    ANTENNA_FCSI_IQDACREG0_RESQ__ISMSK, ANTENNA_FCSI_IQDACREG0_RESQ_OFF__SVAL);
}


void abb11ac_antenna_fuse_values(int antenna, int adc_clkinv)
{
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_IQADCREG1, \
      ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, adc_clkinv);
    
    ABB_FCSI_RMW(antenna, ANTENNA_FCSI_REGLDO, \
    ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
    ANTENNA_FCSI_REGLDO_PRGLDOANT_1__SVAL);	  
}


int abb11ac_fusing(uint8_t *fuse) //, uint16_t *readbackval)
{
    uint16_t rccountp[3];
    int rccount_calc = 0;
    // mapping
    
    // reg 20
    // reg20, bit 7
    int trim_indicator = fuse[0] >> 7;
    // reg20, bit 6:3
    int iref = ((fuse[0] >> 3) & 0x0F);
    // reg20, bit 2:0
    int vref = (fuse[0] & 0x07);
    
    
    // reg21
    // reg21, bit 5:3
    int dacgain_ant1 = ((fuse[1] >> 3) & 0x07);
    // reg 21, bit 2:0
    int dacgain_ant0 = (fuse[1] & 0x07);
    
    // reg22
    // reg22, bit 6:4
    int ldo_ant0 = ((fuse[2] >> 4) & 0x07);
    // reg22, bit 3:1
    int dacgain_ant3 = ((fuse[2] >> 1) & 0x07);
    // reg22, bit 0 + reg21, bit 7:6
    int dacgain_ant2 = ((fuse[2] & 0x01) << 2) + ((fuse[1] >> 6) & 0x03);
    
    
    // reg23
    // reg23, bit 7:5
    int ldo_ant3 = ((fuse[3] >> 5) & 0x03);
    // reg23, bit 4:2
    int ldo_ant2 = ((fuse[3] >> 2) & 0x03);
    // reg23, bit 1:0 + reg22, bit 7
    int ldo_ant1 = ((fuse[3] & 0x03) << 1) + (fuse[2] >> 7);
    
    // reg24
    // reg24, bit 7:6
    int ssi_ant3 = ((fuse[4] >> 6) & 0x03);
    // reg24, bit 5:4
    int ssi_ant2 = ((fuse[4] >> 4) & 0x03);
    // reg24, bit 3:2
    int ssi_ant1 = ((fuse[4] >> 2) & 0x03);
    // reg24, bit 1:0
    int ssi_ant0 = (fuse[4] & 0x03);
    
    // rccount
    int rccount = ((fuse[6] & 0x1F) << 4) + (fuse[5] >> 4);
    
    
    // reg25
    // reg25, bit 3
    int clkinv3 = ((fuse[5] >> 3) & 0x01);
    // reg25, bit 2
    int clkinv2 = ((fuse[5] >> 2) & 0x01);
    // reg25, bit 1
    int clkinv1 = ((fuse[5] >> 1) & 0x01);
    // reg25, bit 0
    int clkinv0 = (fuse[5] & 0x01);
    
    int ret_val_adctune = 0;
    
    // pointer for adctune function
    uint16_t ctune40m[1] = {0};
    
    debug_printf("fuse[0]        = %d;\n",fuse[0]);
    debug_printf("fuse[1]        = %d;\n",fuse[1]);
    debug_printf("fuse[2]        = %d;\n",fuse[2]);
    debug_printf("fuse[3]        = %d;\n",fuse[3]);
    debug_printf("fuse[4]        = %d;\n",fuse[4]);
    debug_printf("fuse[5]        = %d;\n",fuse[5]);
    debug_printf("fuse[6]        = %d;\n",fuse[6]);
    debug_printf("fuse[7]        = %d;\n",fuse[7]);
    debug_printf("fuse[8]        = %d;\n",fuse[8]);
    debug_printf("fuse[9]        = %d;\n",fuse[9]);
    debug_printf("fuse[10]       = %d;\n",fuse[10]);
    
    
    debug_printf("trim_indicator = %d;\n",trim_indicator);
    debug_printf("iref           = %d;\n",iref);
    debug_printf("vref           = %d;\n",vref);
    
    debug_printf("dacgain_ant0   = %d;\n",dacgain_ant0);
    debug_printf("dacgain_ant1   = %d;\n",dacgain_ant1);
    debug_printf("dacgain_ant2   = %d;\n",dacgain_ant2);
    debug_printf("dacgain_ant3   = %d;\n",dacgain_ant3);
    
    debug_printf("ssi_ant0       = %d;\n",ssi_ant0);
    debug_printf("ssi_ant1       = %d;\n",ssi_ant1);
    debug_printf("ssi_ant2       = %d;\n",ssi_ant2);
    debug_printf("ssi_ant3       = %d;\n",ssi_ant3);
    
    debug_printf("ldo_ant0       = %d;\n",ldo_ant0);
    debug_printf("ldo_ant1       = %d;\n",ldo_ant1);
    debug_printf("ldo_ant2       = %d;\n",ldo_ant2);
    debug_printf("ldo_ant3       = %d;\n",ldo_ant3);
    
    debug_printf("clkinv0        = %d;\n",clkinv0);
    debug_printf("clkinv1        = %d;\n",clkinv1);
    debug_printf("clkinv2        = %d;\n",clkinv2);
    debug_printf("clkinv3        = %d;\n",clkinv3);
    
    debug_printf("rccount        = %d;\n",rccount);
    
    /*// mosmeas
    int rvt_core_M =
    int rvt_core_P =
    int lvt_core_M =
    int lvt_core_P =
    int io_M =
    int io_P = 
    
    readbackval[0] = rcccount;
    readbackval[1] = rvt_core_M;
    readbackval[2] = rvt_core_P;
    readbackval[3] = lvt_core_M;
    readbackval[4] = lvt_core_P;
    readbackval[5] = io_M;
    readbackval[6] = io_P;*/
    
    
    if (trim_indicator == 1)
    {
        // vref, iref
    ABB_FCSI_RMW(ABB_CENTRAL, CENTRAL_FCSI_REG28BIAS, \
	CENTRAL_FCSI_REG28BIAS_BGPVREFCONF__ISMSK & \
	CENTRAL_FCSI_REG28BIAS_IBIASTRIM__ISMSK, \
	CENTRAL_FCSI_REG28BIAS_BGPVREFCONF__PUT(vref,0) | \
	CENTRAL_FCSI_REG28BIAS_IBIASTRIM__PUT(iref,0)); 
	
	// dacgain
	ABB_FCSI_RMW(ABB_ANT3, ANTENNA_FCSI_IQDACREG1, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__ISMSK, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__PUT(dacgain_ant3, 0));
	
	ABB_FCSI_RMW(ABB_ANT2, ANTENNA_FCSI_IQDACREG1, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__ISMSK, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__PUT(dacgain_ant2, 0));
	
	ABB_FCSI_RMW(ABB_ANT1, ANTENNA_FCSI_IQDACREG1, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__ISMSK, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__PUT(dacgain_ant1, 0));
	
	ABB_FCSI_RMW(ABB_ANT0, ANTENNA_FCSI_IQDACREG1, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__ISMSK, \
	ANTENNA_FCSI_IQDACREG1_ABSGAIN__PUT(dacgain_ant0, 0));
	
	// ssi offset
	ABB_FCSI_RMW(ABB_ANT3, ANTENNA_FCSI_REG1, \
	ANTENNA_FCSI_REG1_TRIMOFF__ISMSK, \
	ANTENNA_FCSI_REG1_TRIMOFF__PUT(ssi_ant3, 0));
	
	ABB_FCSI_RMW(ABB_ANT2, ANTENNA_FCSI_REG1, \
	ANTENNA_FCSI_REG1_TRIMOFF__ISMSK, \
	ANTENNA_FCSI_REG1_TRIMOFF__PUT(ssi_ant2, 0));
	
	ABB_FCSI_RMW(ABB_ANT1, ANTENNA_FCSI_REG1, \
	ANTENNA_FCSI_REG1_TRIMOFF__ISMSK, \
	ANTENNA_FCSI_REG1_TRIMOFF__PUT(ssi_ant1, 0));
	
	ABB_FCSI_RMW(ABB_ANT0, ANTENNA_FCSI_REG1, \
	ANTENNA_FCSI_REG1_TRIMOFF__ISMSK, \
	ANTENNA_FCSI_REG1_TRIMOFF__PUT(ssi_ant0, 0));
	
	// trim ldo ant
	ABB_FCSI_RMW(ABB_ANT3, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__PUT(ldo_ant3, 0));
	
	ABB_FCSI_RMW(ABB_ANT2, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__PUT(ldo_ant2, 0));
	
	ABB_FCSI_RMW(ABB_ANT1, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__PUT(ldo_ant1, 0));
	
	ABB_FCSI_RMW(ABB_ANT0, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__PUT(ldo_ant0, 0));
	
	// invert clk adc
	ABB_FCSI_RMW(ABB_ANT3, ANTENNA_FCSI_IQADCREG1, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__PUT(clkinv3, 0));
	
	ABB_FCSI_RMW(ABB_ANT2, ANTENNA_FCSI_IQADCREG1, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__PUT(clkinv2, 0));
	
	ABB_FCSI_RMW(ABB_ANT1, ANTENNA_FCSI_IQADCREG1, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__PUT(clkinv1, 0));
	
	ABB_FCSI_RMW(ABB_ANT0, ANTENNA_FCSI_IQADCREG1, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__ISMSK, \
	ANTENNA_FCSI_IQADCREG1_ADCCLKINV__PUT(clkinv0, 0));
	
	ret_val_adctune = abb11ac_adctune(rccount, RCCOUNTNOM, ctune40m);
	
	if (ret_val_adctune == -1)
	{
	    return -1;
	}
	else
	{
	    return (int) ctune40m[0];	 
	}
    }
    else
    {
    
        // trim ldo ant to 1.17V
	ABB_FCSI_RMW(ABB_ANT3, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT_1__SVAL);
	
	ABB_FCSI_RMW(ABB_ANT2, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT_1__SVAL);
	
	ABB_FCSI_RMW(ABB_ANT1, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT_1__SVAL);
	
	ABB_FCSI_RMW(ABB_ANT0, ANTENNA_FCSI_REGLDO, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT__ISMSK, \
	ANTENNA_FCSI_REGLDO_PRGLDOANT_1__SVAL);
	
	
        //return 8; // -> ctune will be set to 8 if chip not fused; request from ISR
	
	// switch on rcmeas clock; (all clks pup)
	//ABB_DIR_CTRL_WRITES(lcpll_op_buf_pdn_d2a, 5, 0x00);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+2, 0x0);
	rccount_calc = abb11ac_rcmeas(rccountp);
	// switch off rcmeas clock; (all clks pup except rcmeas)
	//ABB_DIR_CTRL_WRITES(lcpll_op_buf_pdn_d2a, 5, 0x04);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+2, 0x1);
	
	if (rccount_calc == -1)
	{
	   return -1;
	}
	else
	{
	    ret_val_adctune = abb11ac_adctune(rccount_calc, RCCOUNTNOM, ctune40m);
	    if (ret_val_adctune == -1)
	    {
	        return -1;
	    }
	    else
	    {
	        return (int) ctune40m[0];	 
	    }
	}
	
    }

}

int abb11ac_pllpup_sequence()
{
        int i = 0;
	
	// just to guarantee that output buffer of PLL is in PD
	//ABB_DIR_CTRL_WRITES(lcpll_op_buf_pdn_d2a, 5, 0x1F);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a, 0x1);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+1, 0x1);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+2, 0x1);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+3, 0x1);
	ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+4, 0x1);
	
	abb11ac_pll_pup();
	ABB_wait(DEL_PLLUP,T_USEC);
	
        // pll_pup before ABB-release_reset_fcsi as there are some registers inisde PLL that are supplied by PLLLDO.
	// this supply must be there to guarantee reset values

	// program output divider
	abb11ac_pllsettings();
	
	abb11ac_pll_release_reset();
	
	// wait for lock signal; -> direct pin output in ABB11AC; need SOC Adress
	ABB_wait(DEL_LCLOCKMIN,T_USEC);
        for (i = 0; i<=10; i++)
	{
	    if (ABB_DIR_CTRL_READ(lcpll_lock_a2d) == 1)
	    {
	        debug_printf("PLL locked\n");
		// switchon pll clk output buffer, except clk3 and clk1
		//ABB_DIR_CTRL_WRITES(lcpll_op_buf_pdn_d2a, 5, 0x04);
		ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+1, 0x0); // switch on clk2 (wlan_clk)
		ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+3, 0x0); // switch on clk4 (dac_clk)
		ABB_DIR_CTRL_WRITE(lcpll_op_buf_pdn_d2a+4, 0x0); // switch on clk5 (adc_clk)
		// at this time clk2, clk4 and clk5 should run
		
		// release reset of central_fcsi and antenna fcsi
	        // on SOC this is just one fcsi_ms_resetn
                ABB_DIR_CTRL_WRITE(FCSI2_RST_N_C, 0x1);
	        ABB_DIR_CTRL_WRITE(FCSI2_RST_N_0, 0x1);
	        ABB_DIR_CTRL_WRITE(FCSI2_RST_N_1, 0x1);
	        ABB_DIR_CTRL_WRITE(FCSI2_RST_N_2, 0x1);
                ABB_DIR_CTRL_WRITE(FCSI2_RST_N_3, 0x1);
		
		// ABB_wait xxx; 
	        // here the switch from driver to FW-mode should be done
	        // switch also from refclock to 320MHz clock
		
		return 0;
	    }
	    else
	    {
	        if (i == 10)
		{
		   debug_printf("Timeout PLL locking.\n"); 
		   return -1;
		}
	    }
	    ABB_wait(50, T_USEC);
	}
	return 0;
}


int abb11ac_ABB_pup_sequence(uint8_t *fuse)
{
        int ctune = -1;
	
	// at first central init, to pup clkldo; after fcsi_reset of antennas can be released;
	abb11ac_central_init();
	ABB_wait(DEL_LDOUP,T_USEC);
	
	
	// workaround due to antenna FCSI reset was not coming through
	// programs FCSI reset values in antenna registers
	//abb11ac_reset_FCSI_antenna();

	ABB_DIR_CTRL_WRITE(FCSI2_RST_N_0, 0x1);
	abb11ac_ABB_antenna_pup_sequence(fuse);	
    return ctune;
}

int abb11ac_ABB_antenna_pup_sequence(uint8_t *fuse)
{
        int ctune = -1;
	
	// at first central init, to pup clkldo; after fcsi_reset of antennas can be released;
	//abb11ac_central_init();
	//ABB_wait(DEL_LDOUP,T_USEC);
	
	
	// workaround due to antenna FCSI reset was not coming through
	// programs FCSI reset values in antenna registers
	//abb11ac_reset_FCSI_antenna();
	//debug_printf("in antenna power up,  \n");
	
	// init antenna (includes pup of ant ldo)
	abb11ac_antenna_init(ABB_ANT3210);
	ABB_wait(DEL_LDOUP,T_USEC);
	
	// program fusevalues
	ctune = abb11ac_fusing(fuse);
	
	if (ctune == -1)
	{
	    return -1;
	}
	else
	{	
	    // config antenna for 11ac-mode
	    abb11ac_ant_config(ABB_ANT3210, 40, ctune);
	    return 0;
	}
}
