/***********************************************************************************
 File:			HdkRadarDetection.h
 Module:		HDK 
 Purpose: 		Detect none wifi interferers
 Description:   This file is the implementation of the HDK radar detection module 
 				which is responsible to manage the radar detection PHY genrisc FIFO 
 				processing.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HdkRadarDetection.h"
#include "PhyDriver_API.h"
#include "tx_application_Api.h"
#include "System_Timers.h"
#include "RadarDetection.h"
#include "loggerAPI.h"

#if defined RADAR_DETECTION_ENABLED
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HDK_MODULE
#define LOG_LOCAL_FID 4


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static HdkRadarDetectionParameters_t HdkRadarDetectionParameters;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

HdkRadarDetection_TimerExpired 


Description:
------------
Handles the event that radar detection timer has expired


Input:
-----

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void HdkRadarDetection_TimerExpired()
{
	/* If the radar detection is not needed (a new channel was set without radar detection) this event will be ignored */
	if(HdkRadarDetectionParameters.isRadarDetectionEnabled)
	{
		PhyDrvInterfererDetection_FlushRadarDetectionFifo();
		MT_SetModeTimer(MT_TIMER_0, MT_TIMER_ENABLE, MT_TIMER_ONE_SHOT);
		radarAdaptiveSensStateMachineEnterEvent(RADAR_ADAPTIVE_SENS_EVENT_TIMER_EXPIRED);
	}
}

/**********************************************************************************

HdkRadarDetection_SetChannelRequest 


Description:
------------
Handles set channel request message recieved in the HDK module regarding radar
detection processing

Input:
-----
setChannelRequestParameters -  the structure that contains the parameters

		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
void HdkRadarDetection_SetChannelRequest(HdkSetChannelReqParams_t *setChannelRequestParameters)
{
	ILOG0_D("HdkRadarDetection_SetChannelRequest, isRadarDetectionEnabled = %d", HdkRadarDetectionParameters.isRadarDetectionEnabled);
	SLOG0(0, 0, HdkSetChannelReqParams_t, setChannelRequestParameters);
	if(HdkRadarDetectionParameters.isRadarDetectionEnabled && (!setChannelRequestParameters->isRadarDetectionNeeded))
	{		
		radarAdaptiveSensStateMachineEnterEvent(RADAR_ADAPTIVE_SENS_EVENT_SET_NO_RADAR_CHANNEL);
		MT_SetModeTimer(MT_TIMER_0, MT_TIMER_DISABLE, MT_TIMER_ONE_SHOT);		
	}
	else if((!HdkRadarDetectionParameters.isRadarDetectionEnabled) && setChannelRequestParameters->isRadarDetectionNeeded)
	{
		MT_SetModeTimer(MT_TIMER_0, MT_TIMER_ENABLE, MT_TIMER_ONE_SHOT);
	}
	
	HdkRadarDetectionParameters.isRadarDetectionEnabled = setChannelRequestParameters->isRadarDetectionNeeded;

	if (HdkRadarDetectionParameters.isRadarDetectionEnabled)
	{
		if((setChannelRequestParameters->switchType == ST_SCAN) || ((setChannelRequestParameters->switchType == ST_CSA) && (setChannelRequestParameters->block_tx_post == TRUE)))
		{
			radarAdaptiveSensStateMachineEnterEvent(RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_CAC);
		}
		else
		{
			radarAdaptiveSensStateMachineEnterEvent(RADAR_ADAPTIVE_SENS_EVENT_SET_RADAR_CHANNEL_IN_SERVICE);
		}
	}
}


bool HdkRadarDetectionIsRadarDetectionEnabled(void)
{
	return HdkRadarDetectionParameters.isRadarDetectionEnabled;
}
/**********************************************************************************

HdkRadarDetection_Initialize 


Description:
------------
Initialize the HDK radar detection internal parameters and structuress

Input:
-----


		
Output:
-------
	

Returns:
--------

	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void HdkRadarDetection_Initialize(void)
{
	memset(&HdkRadarDetectionParameters, 0, sizeof(HdkRadarDetectionParameters_t));
	
	MT_enableDisableGpTimerIrq(MT_TIMER_0, 1);
	MT_SetTimeTimer(MT_TIMER_0, HDK_RADAR_DETECTION_TIMER_IN_MICROSECONDS);

}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
#endif //RADAR_DETECTION_ENABLED

