/***********************************************************************************
 File:			HeGroupManager_API.h
 Module:		HE Group manager 
 Purpose: 		To create and delete HE multi user groups
 Description:   This file is the API of the HE Group manager which is responsible of 
 				creating and deleting HE multi user groups
************************************************************************************/
#ifndef HE_GROUP_MANAGER_API_H_
#define HE_GROUP_MANAGER_API_H_

/*---------------------------------------------------------------------------------
/						includes					
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "SharedDbTypes.h"
#include "System_Configuration.h"
#include "HwGlobalDefinitions.h"
#include "frame.h"
#include "TxSelector_Api.h"
#include "CommonPlan_Descriptors.h"


/*---------------------------------------------------------------------------------
/						defines				
/----------------------------------------------------------------------------------*/
#define HE_GROUP_MANAGER_MAX_NUMBER_OF_GROUPS 					(32) // if change, should also change ALL_GROUPS_FREE
#define HE_GROUP_MANAGER_MIN_USERS_FOR_MU_MIMO					(2)
#ifdef ENET_INC_HW_FPGA
#define HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP		(16)
#else
#define HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP		(32)
#endif //ENET_INC_HW_FPGA
#define HE_GROUP_MANAGER_INVALID_PLAN_INDEX						(127)

#define HE_GROUP_MANAGER_STATIC_DL_GROUP_ID						(0) 
#define HE_GROUP_MANAGER_STATIC_UL_GROUP_ID						(1) 

/*---------------------------------------------------------------------------------
/						Macros					
/----------------------------------------------------------------------------------*/

			
/*---------------------------------------------------------------------------------
/						Data type definitions					
/----------------------------------------------------------------------------------*/
typedef enum 
{
	DL_MU_TYPE_NONE = 0,
	DL_MU_TYPE_HE,
	DL_MU_TYPE_MIMO,
	DL_MU_TYPE_HE_MIMO
} DlmuType_e;

typedef enum 
{
	UL_MU_TYPE_NONE = 0,
	UL_MU_TYPE_HE,
	UL_MU_TYPE_MIMO
} UlmuType_e;

typedef enum MuType_e
{
    OFDMA = 0,
    MIMO,
    BOTH_OFDMA_AND_MIMO,
} MuType_e;

typedef struct CreatePlanMsg_t
{ 
    MuType_e            muType;
  	uint8 				groupId;
	uint8  				bw;
	uint8				numberOfRepetitions;   
} CreatePlanMsg_t;

typedef struct RemovePlanMsg_t
{ 
  	uint8 groupId;
} RemovePlanMsg_t;

typedef struct HeGm_RecoverFromPdTypeNotAllowedRequest_t
{
	StaId staId;
	uint8 tid;
} HeGm_RecoverFromPdTypeNotAllowedRequest_t;

typedef enum CreatePlanStatus_e
{	
	CREATE_PLAN_SUCCEEDED = 0,
	CREATE_PLAN_FAILED,
} CreatePlanStatus_e;

typedef struct CreatePlanCfmMsg_t
{ 
	CreatePlanStatus_e  status;
	bool				isStatic;
  	uint8 				groupId;
} CreatePlanCfmMsg_t;

typedef struct AddPhaseToPlanMsg_t
{ 
  	uint8 groupId;
} AddPhaseToPlanMsg_t;

typedef struct HeGroupManagerUpdateMetrics_t
{
	uint8 groupIndex;
	uint8 reserved[3];
} HeGroupManagerUpdateMetrics_t;

typedef struct ConfigNumOfStaInGroupMsg_t
{ 
  	uint8 maxStationsInGroup;
    uint8 minStationsInGroup;
} ConfigNumOfStaInGroupMsg_t;

typedef enum PlanManagerState_e
{
	PLAN_MANAGER_IDLE_STATE, 
	PLAN_MANAGER_WAIT_FOR_MEMORY_RESOURCE_ALLOCATION_STATE,
	PLAN_MANAGER_WAIT_FOR_RU_ALLOCATION_STATE, 
	PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_STATE, 
	PLAN_MANAGER_READY_STATE, 
	PLAN_MANAGER_WAIT_FOR_LA_FREE_RESOURCES_STATE, 
	PLAN_MANAGER_WAIT_FOR_LA_REMOVE_FOR_PHASE_ADDITION_STATE, 
	PLAN_MANAGER_WAIT_FOR_LA_SETTINGS_FOR_PHASE_ADDITION_STATE,
} PlanManagerState_e;

typedef enum HeGroupManagerState_e
{	
	HE_GROUP_MANAGER_GROUP_FIRST_STATE = 0,
	HE_GROUP_MANAGER_GROUP_IDLE = HE_GROUP_MANAGER_GROUP_FIRST_STATE,
	HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_CREATION,
	HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_LOCK,
	HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_REMOVAL,
	HE_GROUP_MANAGER_GROUP_ACTIVE,
	HE_GROUP_MANAGER_GROUP_MAX_NUM_OF_STATES = 0xFF,	
} HeGroupManagerState_e;

typedef enum FormationType_e
{	
	DL_OFDMA = 0,
	DL_MIMO,
	UL_OFDMA,
#ifdef ENET_INC_ARCH_WAVE600D2
	UL_MIMO,
#endif
	INVALID_FORMATION_TYPE = 0xFF
} FormationType_e;

typedef enum HeMuSequence_e
{	
	HE_MU_SEQ_MU_BAR = 0,
	HE_MU_SEQ_VHT_LIKE, 
	HE_MU_SEQ_DL_BASIC_TF,
	HE_MU_SEQ_VHT_LIKE_IMM_ACK,
	HE_MU_SEQ_VHT_LIKE_PROTECTION,
	HE_MU_UL,
	HE_MU_BSRP,
	HE_MU_BSRP_UL,
#ifdef ENET_INC_ARCH_WAVE600D2
	HE_MU_NFRP,
#endif
} HeMuSequence_e;

typedef enum TrafficType_e
{
    BOTH_DL_AND_UL = 0,
    DL_ONLY,
    UL_ONLY,
} TrafficType_e;

typedef enum NumOfAntennas_e
{
    ONE_ANT = 1,
    TWO_ANTS,
    THREE_ANTS,
    FOUR_ANTS,
} NumOfAntennas_e;

typedef union HeGroupManagerCharacteristicsBitmap_u
{
	uint32 val;
	struct
	{
		uint32 axSupport  			:1;
		uint32 ulMuMimoSupport	 	:1;
		uint32 dlCandidate    		:1;
		uint32 ulCandidate   		:1;
		uint32 maxBw160        		:1;
		uint32 maxBw80         		:1;
		uint32 maxBw40         		:1;
		uint32 maxBw20         		:1;
		uint32 maxNss1         		:1;
		uint32 maxNss2         		:1;
		uint32 maxNss3         		:1;
		uint32 maxNss4         		:1;
		uint32 highRssi				:1;
		uint32 medRssi				:1;
		uint32 lowRssi				:1;
		uint32 OfdmaInBw20M			:1;
		uint32 Reserved				:16; // note: when adding new charactaristics - max one bit per characteristic!		
	} bitFields;
}HeGroupManagerCharacteristicsBitmap_u;

typedef struct HeGroupManagerPhaseBitmap_t
{
    uint8 dlPhase               :1;
    uint8 ulPhase               :1;
    uint8 soundingPhase			:1;
	uint8 protectionPhase		:1;
	uint8 bsrpPhase				:1;
    uint8 nfrpPhase             :1;
	uint8 reserved				:2;
} HeGroupManagerPhaseBitmap_t;

typedef struct HeGroupManager_GroupDbEntry_t
{
    uint16						            members[HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	uint16						            membersPointingToPlan[HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP];
	CommonPlan_t* 				            planPtr;
	K_MSG*						            pSavedMsgforMixPlanLock;
    HeGroupManagerCharacteristicsBitmap_u   groupCharacteristicsBitmap;	
    HeGroupManagerPhaseBitmap_t	            phasesBitmap;
    FormationType_e				            formationType;
    HeMuSequence_e				            sequence;
	HeGroupManagerState_e 		            heGroupManagerState;
	PlanManagerState_e 			            planManagerState;	
	uint8						            numOfStationsInGroup;
	uint8						            vapId; // vapId of 1 of the station for getting BandId	
#ifdef ENET_INC_ARCH_WAVE600D2
	uint8									mbssRxCtrlSupport;
#endif
} HeGroupManager_GroupDbEntry_t;

typedef enum HeGroupManagerStaticGroupId_e
{	
	HE_GROUP_MANAGER_DL_GID = 0,
	HE_GROUP_MANAGER_UL_GID,		
} HeGroupManagerStaticGroupId_e;

typedef struct HeGroupManagerChannelNotification_t
{
	Bandwidth_e bw;
} HeGroupManagerChannelNotification_t;

typedef struct HeGroupManagerAntConfigNotification_t
{
	uint8 numberOfConnectedAnts;
} HeGroupManagerAntConfigNotification_t;

typedef struct HeGroupManagerStaParamsFromLa_t
{
    K_MSG*      pAddStaMsg;
	Bandwidth_e staBw;
	bool        ofdmaInBW20;
	uint8       staMaxNss;	
} HeGroupManagerStaParamsFromLa_t;

typedef enum HeGroupManagerStationState_e
{
	HE_GROUP_MANAGER_STATION_NOT_ELIGIBLE = 0,
	HE_GROUP_MANAGER_STATION_ELIGIBLE,	// for Phase 0 , means HE capable
	HE_GROUP_MANAGER_STATION_IN_A_GROUP ,
	HE_GROUP_MANAGER_MAX_NUM_OF_STATION_STATES = 0xFF,	
} HeGroupManagerStationState_e;

extern HeGroupManager_GroupDbEntry_t 	  heGroupManagerDb[]; 


/*---------------------------------------------------------------------------------
/						Public functions declarations			
/----------------------------------------------------------------------------------*/

void heGroupManager_Init(void);
uint16 heGroupManagerGetTSManagerTidsBitmap(StaId stationId);
void heGroupManagerSetPlanTidsBitmap(uint32 tidAllocBitmap, uint8 muSequenceType);			
uint16 heGroupManagerGetHeGroupManagerTidsBitmap(StaId stationId);
void heGroupManagerTSManagerEnableTid(StaId stationId, uint8 tidIndex);
void heGroupManagerTSManagerDisableTid(StaId stationId, uint8 tidIndex);
void heGroupManager_TaskEntry(K_MSG *psMsg);
bool heGroupManager_IsStationInStaticPlan(StaId stationId);
uint16 heGroupManager_GetNumOfActiveAntennas(uint8 vapId);
HeGroupManagerStationState_e heGroupManagerGetStationState(StaId stationIndex);

#endif /* HE_GROUP_MANAGER_API_H_ */
