/***********************************************************************************
 File:			GroupsGenerator.c
 Module:		He Group Manager
 Purpose: 		To create MU groups
 Description:   This file is the implementation of the groups generator which is 
 				responsible of creating multi user groups (MIMO & OFDMA)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "HeGroupManager_API.h"
#include "HeGroupManager.h"
#include "Utils_Api.h"
#include "OSAL_Api.h"
#include "loggerAPI.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "queue_utility.h"
#include "mhi_umi.h"
#include "CommonPlan_Descriptors.h"
#include "ShramStationDatabase.h"
#include "PlanManager_API.h"
#include "CoC_Api.h"
#include "HeGroupManagerClassifier.h"
#include "HeGroupManager.h"
#include "GroupsGenerator.h"
#include "LinkAdaptation.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "GroupManager_API.h"
#include "DescriptorsDefinitions.h"
#endif
#include "stringLibApi.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HE_GROUP_MANAGER
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
#ifdef ENET_INC_ARCH_WAVE600D2
#define MBSS_RX_CTRL_MIN_NUM_VAPS 2
#endif

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void  groupsGeneratorSetBwForPlan(HeGroupGeneratorResults_t* pHeGroupGeneratorResults, HeGroupManagerGroupPolicy_t* pGroupPolicy);
Bw_e groupsGeneratorGetDataBwLimit(uint16 stationIndex);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
extern LinkAdaptationStaDatabase_t 		LinkAdaptationStaDatabase[HW_NUM_OF_STATIONS];

/*---------------------------------------------------------------------------------

/						 Functions Definitions									
/----------------------------------------------------------------------------------*/



/**********************************************************************************   
groupsGenerator_SetGroupPoliciesTable
  
Description: called by user. Update heGroupManagerGlobalParameters
------------
    
Input: 

-----                    
Output:
-------
            
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetGroupPoliciesTable(HeGroupManagerGroupPoliciesTable_t groupPoliciesTable)
{
	//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
    MEMCPY(&heGroupManagerGlobalParameters.groupPoliciesTable, &groupPoliciesTable, sizeof(HeGroupManagerGroupPoliciesTable_t));
}


/**********************************************************************************   
groupsGenerator_AddGroupTypeToTable
  
Description: Add group type to group policies table (store in heGroupManagerGlobalParameters)
------------
    
Input: 

-----                    
Output:
-------
            
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_AddGroupTypeToTable(HeGroupManagerGroupPolicy_t groupPolicy)
{
    uint8 numOfPolicies = heGroupManagerGlobalParameters.groupPoliciesTable.numOfPolicies;
	//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
    MEMCPY(&heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[numOfPolicies], &groupPolicy, sizeof(HeGroupManagerGroupPolicy_t));
    heGroupManagerGlobalParameters.groupPoliciesTable.numOfPolicies = numOfPolicies + 1;
}


/**********************************************************************************   
groupsGenerator_GenerateAllGroups
  
Description: Genearte all groups
    
Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
bool groupsGenerator_GenerateAllGroups(void)
{
    uint8					  numOfFreeGroups;
    uint8					  maxGroups;
    uint8                     vapId;
    uint8                     firstVapInBand;
    uint8                     numOfVapsInBand;
    BandId_e                  bandId;
    bool                      groupsInGeneration = FALSE;
    
	ILOG0_V("groupsGenerator_GenerateAllGroups");
    
    for (bandId = CONFIGURATION_MANAGER_BAND_0; bandId < ConfigurationManager_GetNumOfActiveBands(); bandId++)
    {
        maxGroups       = heGroupManagerGlobalParameters.groupManagerConfigurations.maxGroups[bandId];
        numOfFreeGroups = maxGroups - (NUMBER_OF_GROUPS_FOR_PROBING + Utils_CountNumOfOnes(heGroupManagerGlobalParameters.activeGroupsBitmap));
        firstVapInBand  = ConfigurationManager_GetFirstVapForBand(bandId);
        numOfVapsInBand = ConfigurationManager_GetNumOfVapsInBand(bandId);
        
        if (numOfFreeGroups >= 1)
	    {
            for (vapId = firstVapInBand; vapId < (firstVapInBand + numOfVapsInBand); vapId++)
            {
                if (VapDatabaseObj.vapDbSwEntries[vapId].state == VAP_STATE_ADDED)
                {
                    ILOG0_D("groupsGenerator_GenerateGroupsForVap: %d", vapId);
                    groupsGenerator_GenerateGroupsForVap(vapId, bandId, &numOfFreeGroups, &groupsInGeneration);
                }
            }
        }
    }
    return groupsInGeneration;
}


/**********************************************************************************   
groupsGenerator_GenerateGroupsForVap
  
Description:  Create several groups for VAP which meet group policies table 
------------
Run over groups policies table and create groups from stations in VAP (use API with classifier).
For each group send request to build group with group ID, VAP ID, formation type, num of stations in group, stations list and more
    
Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_GenerateGroupsForVap(uint8 vapId, BandId_e bandId, uint8* numOfFreeGroups, bool* groupsInGeneration)
{
    HeGroupManagerCharacteristicsBitmap_u bandLimitationBitmap;
    HeGroupManagerCharacteristicsBitmap_u groupCharacteristicsBitmap;
    HeGroupManagerGroupPolicy_t           groupPolicy;
    HeGroupGeneratorResults_t             groupGeneratorResults;
    StaId 								  numOfStaInCluster;
    uint8                                 groupPolicyId;
    uint8                                 numOfGroupsInCreationForPolicy;
    uint8                                 numOfStationsInGroup = 0;
    StaId                                 staId;
    StaId                                 staLoopIndex;
    uint8                                 maxStationsInGroup;
    uint8                                 minStationsInGroup;
    uint8                                 maxNumOfGroupsForPolicy;
    
    bandLimitationBitmap = heGroupManagerBandParameters[bandId].bandLimitationBitmap;    
    for (groupPolicyId = 0; groupPolicyId < heGroupManagerGlobalParameters.groupPoliciesTable.numOfPolicies; groupPolicyId++)
    { 
        groupPolicy = heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[groupPolicyId];        
        if (groupPolicy.validGroup == 1)
        {
            numOfGroupsInCreationForPolicy = 0;
            maxStationsInGroup = MIN(groupPolicy.maxStationsInGroup, heGroupManagerGlobalParameters.groupManagerConfigurations.maxStationsInGroup);
            minStationsInGroup = MAX(groupPolicy.minStationsInGroup, heGroupManagerGlobalParameters.groupManagerConfigurations.minStationsInGroup);
            /* get next group characteristics bitmap from group policy table */
            groupCharacteristicsBitmap = groupPolicy.groupCharacteristics;
            if ((bandLimitationBitmap.val & groupCharacteristicsBitmap.val) != groupCharacteristicsBitmap.val)
            {
                continue;
            }
            /* create stations cluster according to group characteristic bitmap */
            heGroupManagerClassifier_CreateStationsCluster(vapId, groupCharacteristicsBitmap, &numOfStaInCluster);
            if (numOfStaInCluster < minStationsInGroup)
            {/* for the first attempt for the current group policy */
                continue; 
            }
            maxNumOfGroupsForPolicy = MIN(heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[groupPolicyId].numOfGroupsForPolicy, heGroupManagerGlobalParameters.groupPoliciesTable.maxNumOfGroupsPerPolicy);
            while (numOfGroupsInCreationForPolicy < maxNumOfGroupsForPolicy)
            {
                memset(&groupGeneratorResults, 0x0, sizeof(HeGroupGeneratorResults_t));
                /* check if there are enough free data phases and hasn't reached max num of groups */
                if ((!groupsGenerator_CheckFreeDataPhases(groupPolicy.phasesBitmap, vapId)) || (*numOfFreeGroups < 1))
                {
                    return;
                }
                numOfStaInCluster = heGroupManagerClassifier_GetNumberOfStaInCluster();
                if (numOfStaInCluster < minStationsInGroup)
                {/* for additional attempts to create another group for the same group policy */
                    break; 
                }
                for (staLoopIndex = 0; staLoopIndex < numOfStaInCluster; staLoopIndex++)
                {
                    /* get station from cluster */
                    staId = heGroupManagerClassifier_GetStationFromCluster();
                    ASSERT (staId != INVALID_STA_INDEX);
                    groupGeneratorResults.stationsList[staLoopIndex] = staId;
                    numOfStationsInGroup++;
                    if (numOfStationsInGroup == maxStationsInGroup)
                    {/* create group */
                        break;
                    }                                      
                }
                // next phases - add punishment mechanism, check if group in punishment queue
                /* create group */ 
                numOfGroupsInCreationForPolicy++;
                groupsGenerator_FillGroupParametersForPlan(&groupPolicy, &groupGeneratorResults, vapId, (uint8)numOfStationsInGroup);
                ILOG0_DDD("groupsGenerator: creating a group for vap: %d, policyId: %d, num of sta in group: %d", vapId, groupPolicyId, numOfStationsInGroup);
                groupsGenerator_CreatePlan(&groupGeneratorResults);
                numOfStationsInGroup = 0;
                groupsGenerator_DecreaseNumOfFreeDataPhases(groupPolicy.phasesBitmap, vapId);
                *numOfFreeGroups = (*numOfFreeGroups - 1);
                *groupsInGeneration = TRUE;
            }            
        }   	    
    }
}


/**********************************************************************************   
groupsGenerator_fillGroupParametersForPlan
 
Description:  Fill group parameters 
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_FillGroupParametersForPlan(HeGroupManagerGroupPolicy_t* groupPolicy, HeGroupGeneratorResults_t* groupGeneratorResults, uint8 vapId, uint8 numOfStationsInGroup)
{
    groupGeneratorResults->numOfStationsInGroup = numOfStationsInGroup;
    groupGeneratorResults->vapId                = vapId;
    groupGeneratorResults->phasesBitmap         = groupPolicy->phasesBitmap;
    groupGeneratorResults->sequence             = groupPolicy->sequence;
    groupGeneratorResults->formationType        = groupPolicy->formationType;
    groupGeneratorResults->numberOfRepetitions  = groupPolicy->numberOfRepetitions;
    groupGeneratorResults->groupCharacteristics = groupPolicy->groupCharacteristics;
            
    if ((groupPolicy->formationType == DL_OFDMA) || (groupPolicy->formationType == UL_OFDMA))
    {        
        groupGeneratorResults->muType = OFDMA;
    }
    else // MIMO
    {
        groupGeneratorResults->muType = MIMO;
    }
    groupsGeneratorSetBwForPlan(groupGeneratorResults, groupPolicy);
#ifdef DYNAMIC_GROUPING_DEBUG
    SLOG0(0, 0, HeGroupGeneratorResults_t, groupGeneratorResults);
#endif
}


/**********************************************************************************   
groupsGenerator_GenerateGroup
  
Description:  Build group according to group policy. 
------------
For each group created send message to build Group request with group ID, VAP ID, formation type, num of stations in group, stations list and group classification bitmap

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_GenerateGroup(uint8 vapId, HeGroupManagerGroupPolicy_t groupPolicy)
{
    /* TBD */
    UNUSED_PARAM(vapId);
    UNUSED_PARAM(groupPolicy);
}


/**********************************************************************************   
groupsGenerator_SetMaxGroupsConfiguration
  
Description:  Set group configuration in HE GM DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetMaxGroupsConfiguration(uint8 maxGroupsBand0, uint8 maxGroupsBand1)
{
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxGroups[CONFIGURATION_MANAGER_BAND_0] = maxGroupsBand0;
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxGroups[CONFIGURATION_MANAGER_BAND_1] = maxGroupsBand1;
}


/**********************************************************************************   
groupsGenerator_SetNumOfStaInGroupConfiguration
  
Description:  Set group configuration in HE GM DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetNumOfStaInGroupConfiguration(K_MSG* pMsg)
{
    ConfigNumOfStaInGroupMsg_t *params = (ConfigNumOfStaInGroupMsg_t *)pK_MSG_DATA(pMsg);
    
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxStationsInGroup = params->maxStationsInGroup;
    heGroupManagerGlobalParameters.groupManagerConfigurations.minStationsInGroup = params->minStationsInGroup;
}


/**********************************************************************************   
groupsGenerator_SetGroupConfiguration
  
Description:  Set group configuration in HE GM DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetTrafficConfiguration(TrafficType_e trafficType)
{
    if (trafficType == BOTH_DL_AND_UL)
    {
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.dlCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.dlCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.ulCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.ulCandidate = 1;
    }
    else if (trafficType == DL_ONLY)
    {
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.dlCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.dlCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.ulCandidate = 0;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.ulCandidate = 0;
    }
    else if (trafficType == UL_ONLY)
    {
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.dlCandidate = 0;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.dlCandidate = 0;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.ulCandidate = 1;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.ulCandidate = 1;
    }
}


/**********************************************************************************   
groupsGenerator_SetMuModeConfiguration
  
Description:  Set group configuration in HE GM DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetMuModeConfiguration(FormationType_e dlTypeOpt1, FormationType_e dlTypeOpt2, FormationType_e ulTypeOpt1, FormationType_e ulTypeOpt2)
{
    uint8           groupPolicy;
    FormationType_e formationType;
    

	for (groupPolicy = 0; groupPolicy < heGroupManagerGlobalParameters.groupPoliciesTable.numOfPolicies; groupPolicy++)
	{
		formationType = heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[groupPolicy].formationType;

		/*Update valid bit in policy table according to dl / ul options*/
		if ((formationType == dlTypeOpt1) || (formationType == dlTypeOpt2)||
			(formationType == ulTypeOpt1) || (formationType == ulTypeOpt2))
		{
			heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[groupPolicy].validGroup = 1;
		}
		else
		{
			heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies[groupPolicy].validGroup = 0;
		}
	}
}


/**********************************************************************************   
groupsGenerator_SetAntennasConfiguration
  
Description:  Set antennas configuration in heGroupManagerGlobalParameters DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_SetAntennasConfiguration(NumOfAntennas_e numOfAnt, BandId_e bandId)
{
    if (numOfAnt == ONE_ANT)
    {
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss1 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss2 = 0;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss3 = 0;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss4 = 0;
    }
    else if (numOfAnt == TWO_ANTS)
    {
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss1 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss2 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss3 = 0;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss4 = 0;    
    }
    else if (numOfAnt == THREE_ANTS)
    {
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss1 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss2 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss3 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss4 = 0; 
    }
    else if (numOfAnt == FOUR_ANTS)
    {
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss1 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss2 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss3 = 1;
        heGroupManagerBandParameters[bandId].bandLimitationBitmap.bitFields.maxNss4 = 1;
    }

    heGroupManagerBandParameters[bandId].numOfActiveAntennas = numOfAnt;
}


/**********************************************************************************   
groupsGenerator_CreatePlan
  
Description:  Update Group DB and STA DB with required groups parameters 
------------
Lock Plan
Send message to PlanManager
Change group state


Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void groupsGenerator_CreatePlan(HeGroupGeneratorResults_t *pHeGroupGeneratorResults)
{
    HeGroupManager_GroupDbEntry_t*	pHeGroupManagerDbEntry;
	K_MSG* 							psMsg = NULL;
	CreatePlanMsg_t*				pCreatePlanMsg;
	uint32							notActiveGroupsBitmap;
	uint8							numOfStationsInGroup;
	uint8							uspIndex;	
	StaId							staIndex;
	uint8							groupId = 0;
	
	if(heGroupManagerGlobalParameters.activeGroupsBitmap == 0)
	{
		groupId = 0; 
	}
	else
	{
		notActiveGroupsBitmap = ~(heGroupManagerGlobalParameters.activeGroupsBitmap); 
		groupId = Utils_FindLastSetAndClear(&notActiveGroupsBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(notActiveGroupsBitmap)));
	}
	Utils_SetBitInBitmap(&heGroupManagerGlobalParameters.activeGroupsBitmap, groupId);

	heGroupManagerGlobalParameters.numOfGroupsInCreation++;
	
	pHeGroupManagerDbEntry = &heGroupManagerDb[groupId];

	pHeGroupManagerDbEntry->heGroupManagerState         = HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_CREATION;
	pHeGroupManagerDbEntry->formationType 	            = pHeGroupGeneratorResults->formationType;
    pHeGroupManagerDbEntry->phasesBitmap                = pHeGroupGeneratorResults->phasesBitmap;
    pHeGroupManagerDbEntry->sequence                    = pHeGroupGeneratorResults->sequence;
	pHeGroupManagerDbEntry->vapId			            = pHeGroupGeneratorResults->vapId;
    pHeGroupManagerDbEntry->groupCharacteristicsBitmap  = pHeGroupGeneratorResults->groupCharacteristics;

	numOfStationsInGroup = pHeGroupGeneratorResults->numOfStationsInGroup;
	pHeGroupManagerDbEntry->numOfStationsInGroup = numOfStationsInGroup;

	for(uspIndex = 0; uspIndex < numOfStationsInGroup; uspIndex++)
	{
		staIndex = pHeGroupGeneratorResults->stationsList[uspIndex];
		pHeGroupManagerDbEntry->members[uspIndex] 					    = staIndex;
		pHeGroupManagerDbEntry->membersPointingToPlan[uspIndex]		    = staIndex;
		heGroupManagerStationDb[staIndex].heGroupManagerStationState	= HE_GROUP_MANAGER_STATION_IN_A_GROUP;
		Utils_SetBitInBitmap(&heGroupManagerStationDb[staIndex].groupsBitmap, groupId);
	}

	psMsg = OSAL_GET_MESSAGE(sizeof(CreatePlanMsg_t));
	pCreatePlanMsg = (CreatePlanMsg_t*) pK_MSG_DATA(psMsg);

	psMsg->header.vapId	 					= pHeGroupGeneratorResults->vapId;
	pCreatePlanMsg->groupId 				= groupId;
	pCreatePlanMsg->bw      				= pHeGroupGeneratorResults->bw;
	pCreatePlanMsg->numberOfRepetitions     = pHeGroupGeneratorResults->numberOfRepetitions;    
    pCreatePlanMsg->muType                  = pHeGroupGeneratorResults->muType;

	ILOG0_D("groupsGenerator_CreatePlan for group: %d", groupId);
#ifdef DYNAMIC_GROUPING_DEBUG
    SLOG0(0, 0, HeGroupManager_GroupDbEntry_t, &heGroupManagerDb[groupId]);
#endif
#ifdef OFDMA_DYNAMIC_PLAN_ON		
	OSAL_SEND_MESSAGE(PLAN_MANAGER_CREATE_PLAN, TASK_PLAN_MANAGER, psMsg, psMsg->header.vapId);
#endif
}


/**********************************************************************************

heGroupManagerCreatePlanCfm


Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool groupsGenerator_CreatePlanDone(K_MSG* psMsg)
{
	CreatePlanCfmMsg_t* 		pHeGroupManagerCreatePlanCfmMsg = (CreatePlanCfmMsg_t*)pK_MSG_DATA(psMsg);
	K_MSG* 						pSetStaticCfmMsg = NULL;
	K_MSG* 						heMuDbgStatIndMsg = NULL;
	UMI_DBG_HE_MU_GROUP_STATS*	heMuDbgStat;
	uint8 						groupId;
	uint8						numOfStationsInGroup;
	StaId						staIndex;
	uint8			 			planIndex;
	uint8						userIndex; 
	bool                        allGroupsCreationDone = FALSE;

	groupId = pHeGroupManagerCreatePlanCfmMsg->groupId;

	heGroupManagerGlobalParameters.numOfGroupsInCreation--;
		
	if(pHeGroupManagerCreatePlanCfmMsg->status == CREATE_PLAN_SUCCEEDED)
	{
		ILOG0_V("[groupsGenerator_CreatePlanDone] - Success !");
		SERIAL_TRACE("groupsGenerator_CreatePlanDone - Success !",0,0,0);
		// change state in Group DB entry 
		heGroupManagerDb[groupId].heGroupManagerState = HE_GROUP_MANAGER_GROUP_ACTIVE;

		// Unlock Plan (since Plan not in use is locked)
		planIndex = heGroupManagerDb[groupId].planPtr->planIndex;
		Locker_UnLockPlan(planIndex, heGroupManagerDb[groupId].members[0]);

		// Set selector (index & HE MU EN)
		heGroupManagerEnableHeMuInSelector(groupId);

		/*Send group creation indication for driver*/
		heMuDbgStatIndMsg = OSAL_GET_MESSAGE(sizeof(UMI_DBG_HE_MU_GROUP_STATS));
			
		heMuDbgStat = (UMI_DBG_HE_MU_GROUP_STATS*)pK_MSG_DATA(heMuDbgStatIndMsg); 
		heMuDbgStat->groupId = groupId; 
		heMuDbgStat->planType = heGroupManagerDb[groupId].formationType; 
		for(userIndex = 0; userIndex < heGroupManagerDb[groupId].numOfStationsInGroup; userIndex++)
		{
			heMuDbgStat->stationId[userIndex] = heGroupManagerDb[groupId].members[userIndex]; 
		}
		for(; userIndex < HE_MU_MAX_NUM_OF_USERS_PER_GROUP ; userIndex++)
		{
			heMuDbgStat->stationId[userIndex] = INVALID_SID_FOR_HE_GROUP;
		}
		heMuDbgStat->vapId = heGroupManagerDb[groupId].vapId; 
		heMuDbgStatIndMsg->header.vapId = heGroupManagerDb[groupId].vapId; 
		heMuDbgStat->setReset = HE_MU_GROUP_SET;
		OSAL_SEND_MESSAGE(UMI_MC_MAN_HE_MU_DEBUG_IND, TASK_UM_IF_TASK, heMuDbgStatIndMsg, heMuDbgStatIndMsg->header.vapId);
        if(pHeGroupManagerCreatePlanCfmMsg->isStatic == TRUE)
        {
            heGroupManagerGlobalParameters.numOfActiveStaticPlan++;
        }
	}
	else // creation failed
	{
			SERIAL_TRACE("groupsGenerator_CreatePlanDone - Failed !",0,0,0);
			ILOG0_V("[groupsGenerator_CreatePlanDone] - Failed!");
			// release GroupDbEntry
			Utils_ZeroBitInBitmap(&heGroupManagerGlobalParameters.activeGroupsBitmap, groupId);		

			// remove the stations from the Group
			numOfStationsInGroup = heGroupManagerDb[groupId].numOfStationsInGroup;
			for(userIndex = 0; userIndex < numOfStationsInGroup; userIndex++)
			{
				staIndex = heGroupManagerDb[groupId].members[userIndex];
                heGroupManagerClassifier_RemoveStationFromGroup(staIndex, groupId);
			}
			
			// Reset Group Data Base entry (only for clear debug)
			memset(&heGroupManagerDb[groupId], 0, sizeof(HeGroupManager_GroupDbEntry_t)); // heGroupManagerState = HE_GROUP_MANAGER_GROUP_IDLE;
			
            groupsGenerator_IncreaseNumOfFreeDataPhases(heGroupManagerDb[groupId].phasesBitmap, heGroupManagerDb[groupId].vapId);
	}
	if(heGroupManagerGlobalParameters.numOfGroupsInCreation == 0)
	{
		allGroupsCreationDone = TRUE;
	}
	if(pHeGroupManagerCreatePlanCfmMsg->isStatic == TRUE)
	{
		// send cfm to Host		
		pSetStaticCfmMsg =	heGroupManagerGlobalParameters.returnMsg;
		heGroupManagerGlobalParameters.returnMsg = NULL;
		OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pSetStaticCfmMsg, pSetStaticCfmMsg->header.vapId);
	}	
	return allGroupsCreationDone;
}


/**********************************************************************************
GroupGenerator_CreateStaticPlan 

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
void groupsGenerator_CreateStaticPlan(K_MSG* psMsg)
{
	UMI_STATIC_PLAN_CONFIG*			pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)pK_MSG_DATA(psMsg);
	HeGroupManager_GroupDbEntry_t*	pHeGroupManagerDbEntry;
	uint8							groupId = 0;
	uint8 							uspIndex;
	StaId							stationIndex = 0;
	uint8							numOfStationsInGroup = 0;

	heGroupManagerGlobalParameters.numOfGroupsInCreation++;

	switch(pMuPlanConfig->commonSection.phaseFormat)
	{
		case PHASE_FORMAT_DL_DATA:
		case PHASE_FORMAT_SOUNDING:	
			groupId = HE_GROUP_MANAGER_STATIC_DL_GROUP_ID;
			break; 
		case PHASE_FORMAT_UL_DATA:
			groupId = HE_GROUP_MANAGER_STATIC_UL_GROUP_ID;
			break;
		default: 
			break;
	}

	pHeGroupManagerDbEntry = &heGroupManagerDb[groupId];	
	Utils_SetBitInBitmap(&heGroupManagerGlobalParameters.activeGroupsBitmap, groupId);
	if ((pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_DATA) || (pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_SOUNDING))
	{
		if(pMuPlanConfig->commonSection.muType == OFDMA)
		{
			pHeGroupManagerDbEntry->formationType  = DL_OFDMA;
		}
		else // MU_MIMO
		{
			pHeGroupManagerDbEntry->formationType  = DL_MIMO;
		}
	}
	else // UpLink
	{
		if(pMuPlanConfig->commonSection.muType == OFDMA)
		{
			pHeGroupManagerDbEntry->formationType  = UL_OFDMA;
		}
		else // MU_MIMO
		{
#ifdef ENET_INC_ARCH_WAVE600D2
			pHeGroupManagerDbEntry->formationType  = UL_MIMO;
#else
			FATAL("UL MU-MIMO is unsupported")
#endif
		}
	}
	pHeGroupManagerDbEntry->heGroupManagerState  = HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_CREATION;
	numOfStationsInGroup						 = pMuPlanConfig->commonSection.numOfParticipatingStations;
	pHeGroupManagerDbEntry->numOfStationsInGroup = numOfStationsInGroup;
		
	// Get VapId from the message
	pHeGroupManagerDbEntry->vapId = psMsg->header.vapId;
	
	for(uspIndex = 0; uspIndex < numOfStationsInGroup; uspIndex++)
	{	
		stationIndex = pMuPlanConfig->perUserParameters[uspIndex].uspStationIndexes;
		pHeGroupManagerDbEntry->members[uspIndex] = stationIndex;
		pHeGroupManagerDbEntry->membersPointingToPlan[uspIndex]	= stationIndex;
		heGroupManagerStationDb[stationIndex].heGroupManagerStationState = HE_GROUP_MANAGER_STATION_IN_A_GROUP;		
		Utils_SetBitInBitmap(&heGroupManagerStationDb[stationIndex].groupsBitmap, groupId);
#ifdef ENET_INC_ARCH_WAVE600D2
		if ((pHeGroupManagerDbEntry->formationType == UL_MIMO) && (heGroupManagerStationDb[stationIndex].characteristicsBitmap.bitFields.ulMuMimoSupport == FALSE))
		{
			FATAL("UL MU-MIMO is not supported by station");
		}
#endif
	}
#ifdef ENET_INC_ARCH_WAVE600D2
	groupsGenerator_SetGroupSupportMbssRxCntrl(psMsg, groupId);
#endif
	
	OSAL_SEND_MESSAGE(PLAN_MANAGER_CREATE_STATIC_PLAN, TASK_PLAN_MANAGER, psMsg, psMsg->header.vapId);
}

#ifdef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************
groupsGenerator_GetNumOfVapsInGroup 


**********************************************************************************/
void groupsGenerator_SetGroupSupportMbssRxCntrl(K_MSG* psMsg, uint8 groupId)
{
UMI_STATIC_PLAN_CONFIG* 		pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)pK_MSG_DATA(psMsg);
HeGroupManager_GroupDbEntry_t*	pHeGroupManagerDbEntry =  NULL;
uint8							uspIndex = 0;
StaId							stationIndex = 0;
uint8							numOfStationsInGroup = 0;
uint8							stationVapId = 0xff;
uint32							numOfVapsInGroup = 0;
uint32							staMbssCount = 0;

	pHeGroupManagerDbEntry = &heGroupManagerDb[groupId];	
	numOfStationsInGroup = pMuPlanConfig->commonSection.numOfParticipatingStations;
	for(uspIndex = 0; uspIndex < numOfStationsInGroup; uspIndex++)
	{	
		stationIndex = pMuPlanConfig->perUserParameters[uspIndex].uspStationIndexes;
		
		/* count for Mbss Rx Control, if group have stations from least 2 diffrent vaps */
		if(stationVapId != heGroupManagerStationDb[stationIndex].vapId)
		{
			stationVapId = heGroupManagerStationDb[stationIndex].vapId;
			numOfVapsInGroup++;
		}
		
		/* count all stations support mbss rx control in the HE capability */
		if(GroupManager_GetStaMbssRxCtrlSupport(stationIndex) == TRUE)
		{
			staMbssCount++;
		}
	 }
	
	pHeGroupManagerDbEntry->mbssRxCtrlSupport = MULTI_BSSID_NO_MULTI_BSSID; 
	/* all stations in group should support mbss rx control, and be from least 2 diffrent vaps, in order to turn on mbss indication at phase */
	if(( numOfVapsInGroup >= MBSS_RX_CTRL_MIN_NUM_VAPS) &&(staMbssCount == numOfStationsInGroup))
	{
			pHeGroupManagerDbEntry->mbssRxCtrlSupport = MULTI_BSSID_MULTI_BSSID;
	}	
		
}
#endif
/**********************************************************************************
GroupGenerator_GetNumOfFreeDataPhases 

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
uint8 groupsGenerator_GetNumOfFreeDataPhases(uint8 vapId)
{
    BandId_e    bandId;

    bandId = ConfigurationManager_GetBandForVap(vapId);

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[groupsGenerator_GetNumOfFreeDataPhases], num of free data phases: %d", heGroupManagerBandParameters[bandId].numOfFreeDataPhases);
#endif
    return (heGroupManagerBandParameters[bandId].numOfFreeDataPhases);    
}


/**********************************************************************************
GroupGenerator_CheckFreeDataPhases 

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
bool groupsGenerator_CheckFreeDataPhases(HeGroupManagerPhaseBitmap_t phaseBitmap, uint8 vapId)
{
    uint8 numOfRequiredDataPhases;

    numOfRequiredDataPhases = groupsGenerator_CalcNumOfDataPhases(phaseBitmap);
    return (groupsGenerator_GetNumOfFreeDataPhases(vapId) >= numOfRequiredDataPhases);    
}


/**********************************************************************************
GroupGenerator_IncreaseNumOfFreeDataPhases 

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
void groupsGenerator_IncreaseNumOfFreeDataPhases(HeGroupManagerPhaseBitmap_t phaseBitmap, uint8 vapId)
{
    BandId_e    bandId;
    uint8       numOfFreeDataPhases;

    bandId = ConfigurationManager_GetBandForVap(vapId);
    numOfFreeDataPhases = heGroupManagerBandParameters[bandId].numOfFreeDataPhases;
    numOfFreeDataPhases += groupsGenerator_CalcNumOfDataPhases(phaseBitmap);
    heGroupManagerBandParameters[bandId].numOfFreeDataPhases = numOfFreeDataPhases;

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[groupsGenerator_IncreaseNumOfFreeDataPhases], numOfFreeDataPhases: %d", numOfFreeDataPhases);
#endif    
}


/**********************************************************************************
GroupGenerator_DecreaseNumOfFreeDataPhases

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
void groupsGenerator_DecreaseNumOfFreeDataPhases(HeGroupManagerPhaseBitmap_t phaseBitmap, uint8 vapId)
{
    BandId_e    bandId;
    uint8       numOfFreeDataPhases;

    bandId = ConfigurationManager_GetBandForVap(vapId);
    numOfFreeDataPhases = heGroupManagerBandParameters[bandId].numOfFreeDataPhases;
    numOfFreeDataPhases -= groupsGenerator_CalcNumOfDataPhases(phaseBitmap);  
    ASSERT (numOfFreeDataPhases >= 0);
    heGroupManagerBandParameters[bandId].numOfFreeDataPhases = numOfFreeDataPhases;  

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[groupsGenerator_DecreaseNumOfFreeDataPhases], numOfFreeDataPhases: %d", numOfFreeDataPhases);
#endif
}


/**********************************************************************************
GroupGenerator_CalcNumOfDataPhases

Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
uint8 groupsGenerator_CalcNumOfDataPhases(HeGroupManagerPhaseBitmap_t phaseBitmap)
{
    uint8 numOfDataPhases = 0;

    if (phaseBitmap.dlPhase == 1)
    {
        numOfDataPhases += 2;
    }
    if (phaseBitmap.ulPhase == 1)
    {
        numOfDataPhases += 2;
    }    
    numOfDataPhases += 1; // bsrp phase is always allocated
    
    return numOfDataPhases;
}


/**********************************************************************************
groupsGeneratorSetBwForPlan 

Description:
------------
Calculate the lowest BW between the stations members of the Group Generator results

Input: 
-----
pHeGroupGeneratorResults : pointer to HE Group Generator results 	
		
Output:
-------
	
Returns:
--------

**********************************************************************************/
static void groupsGeneratorSetBwForPlan(HeGroupGeneratorResults_t* pHeGroupGeneratorResults, HeGroupManagerGroupPolicy_t* pGroupPolicy)
{
	Bw_e	currentBw = BW_20MHZ;
	Bw_e    lowestBw;
	Bw_e	stationNum;

    if (TRUE == pGroupPolicy->groupCharacteristics.bitFields.maxBw160)
    {
        pHeGroupGeneratorResults->bw = BW_160MHZ;
    }
    else if (TRUE == pGroupPolicy->groupCharacteristics.bitFields.maxBw80)
    {
        pHeGroupGeneratorResults->bw = BW_80MHZ;
    }
    else if (TRUE == pGroupPolicy->groupCharacteristics.bitFields.maxBw40)
    {
        pHeGroupGeneratorResults->bw = BW_40MHZ;
    }
    else if (TRUE == pGroupPolicy->groupCharacteristics.bitFields.maxBw20)
    {
        pHeGroupGeneratorResults->bw = BW_20MHZ;
    }
    else // no bw was set in group characteristics
    {       
        lowestBw = heGroupManagerBandParameters[ConfigurationManager_GetBandForVap(pHeGroupGeneratorResults->vapId)].maxBw;
	    for(stationNum = 0; stationNum < pHeGroupGeneratorResults->numOfStationsInGroup; stationNum++)
	    {      
            if (TRUE == heGroupManagerStationDb[stationNum].characteristicsBitmap.bitFields.maxBw160)
            {   
                currentBw = BW_160MHZ;
            }
            else if (TRUE == heGroupManagerStationDb[stationNum].characteristicsBitmap.bitFields.maxBw80)
            {
                currentBw = BW_80MHZ;
            }
            else if (TRUE == heGroupManagerStationDb[stationNum].characteristicsBitmap.bitFields.maxBw40)
            {
                currentBw = BW_40MHZ;
            }
            else if (TRUE == heGroupManagerStationDb[stationNum].characteristicsBitmap.bitFields.maxBw20)
            {
                currentBw = BW_20MHZ;
            }
			else
			{
				ASSERT(0);
			}
#ifdef DYNAMIC_GROUPING_DEBUG
            ILOG0_DD("[groupsGeneratorSetBwForPlan], user: %d, bw: %d", stationNum, currentBw);
#endif
		    if(currentBw < lowestBw)
		    {
			    lowestBw = currentBw; 
		    }
	    }
	    pHeGroupGeneratorResults->bw = lowestBw;
    }
#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[groupsGeneratorSetBwForPlan], result bw: %d", pHeGroupGeneratorResults->bw);
#endif
}


/**********************************************************************************
groupsGeneratorGetDataBwLimit

Description:
------------

Input: 
-----

Returns:
--------
	
**********************************************************************************/
Bw_e groupsGeneratorGetDataBwLimit(uint16 stationIndex)
{		
	return (Bw_e)(StaDbHwEntries[stationIndex].common.dataBwLimit);
}


