/***********************************************************************************
 File:			HeGroupManager.c
 Module:		He Group Manager
 Purpose: 		To create and delete multi HE MU groups
 Description:   This file is the implementation of the HE MU group manager which is 
 				responsible of creating and deleting multi user groups (MIMO & OFDMA)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "HeGroupManager_API.h"
#include "HeGroupManager.h"
#include "Utils_Api.h"
#include "OSAL_Api.h"
#include "loggerAPI.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "queue_utility.h"
#include "mhi_umi.h"
#include "CommonPlan_Descriptors.h"
#include "ShramStationDatabase.h"
#include "PlanManager_API.h"
#include "CoC_Api.h"
#include "CommonRamLinkAdaptation.h"
#include "HeGroupManager.h"
#include "GroupsGenerator.h"
#include "HeGroupManager_API.h"
#include "HeGroupManagerClassifier.h"
#include "VapDatabase_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HE_GROUP_MANAGER
#define LOG_LOCAL_FID 2

#define GROUP_EFFICIENCY_TH_PERCENT_UNIT 0 // should be 90
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Function Declaration									
/----------------------------------------------------------------------------------*/
bool groupsHandler_CheckGroupsEfficiency(void);
bool groupsHandler_RemoveStaticPlan(K_MSG* psMsg);
bool groupsHandler_RemoveStation(K_MSG* psMsg);
void groupsHandler_RemoveGroup(uint8 groupId);
bool  groupsHandler_PlanRemovedCfm (K_MSG* psMsg,K_MSG** psMsgOriginal, uint8* returnGroupId);
void groupsHandler_PlanRemovedDoneGroupsInRemoval (K_MSG* psMsgOriginal, uint8 groupId);
void groupsHandler_PlanRemovedDoneGroupsUnderMaintenence (uint8 groupId);
bool groupsHandler_RemoveAllActiveGroups(void);
bool groupsHandler_BandConfigurationChanged(BandId_e bandId);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------

/						 Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

groupsHandler_CheckGroupsEfficiency





Description:
------------
Run over Groups metrics and check group efficiency, if group is not efficient - remove it

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool groupsHandler_CheckGroupsEfficiency(void)
{
	uint32 activeGroupsBitmap 	= 0;
	uint8  groupId   		  	= 0;
	bool groupRemoved			= FALSE;
	
	activeGroupsBitmap = heGroupManagerGlobalParameters.activeGroupsBitmap ;

	while(activeGroupsBitmap != ALL_GROUPS_FREE)
	{
		groupId = Utils_CountTrailingZeros(activeGroupsBitmap);
		Utils_ZeroBitInBitmap(&activeGroupsBitmap, groupId);
		if (heGroupManagerDb[groupId].groupCharacteristicsBitmap.bitFields.dlCandidate == TRUE)
		{
			ILOG0_D("groupsHandler_CheckGroupsEfficiency, dlPsduEfficiency = %d", GroupsMetrics[groupId].dlPsduEfficiency);
			/*DL group*/
			if (GroupsMetrics[groupId].dlPsduEfficiency < GROUP_EFFICIENCY_TH_PERCENT_UNIT)
			{
				groupRemoved = TRUE;
				groupsHandler_RemoveGroup(groupId);
			}
		}

	}	
	return groupRemoved;
}


/**********************************************************************************

groupsHandler_PlanRemovedCfm




Description:
------------


Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool  groupsHandler_PlanRemovedCfm (K_MSG* psMsg,K_MSG** psMsgOriginal, uint8* returnGroupId)
{
	
	RemovePlanMsg_t* 			pHeGroupManagerRemovePlanCfmMsg;
	StaId						stationId;
	uint8			 			stationNum = 0;
	uint8			 			groupId;
	ILOG0_D("[groupsHandler_PlanRemovedCfm] - heGroupManagerGlobalParameters.removalRequestor = %d", heGroupManagerGlobalParameters.removalRequestor);
	SERIAL_TRACE("[heGroupManagerRemovePlanCfm]", heGroupManagerGlobalParameters.removalRequestor, 0, 0);


	pHeGroupManagerRemovePlanCfmMsg = (RemovePlanMsg_t*)pK_MSG_DATA(psMsg);

	// Retrieve groupId from the message
	groupId = pHeGroupManagerRemovePlanCfmMsg->groupId;
	*returnGroupId = groupId;

	heGroupManagerGlobalParameters.numOfGroupsInRemoval --;

	// If the nbr of static plan was different than 0, it means that this remove is necessarily for a static plan
	if(heGroupManagerGlobalParameters.numOfActiveStaticPlan != 0)
	{
		heGroupManagerGlobalParameters.numOfActiveStaticPlan --;
	}

	// Release GroupId
	Utils_ZeroBitInBitmap(&heGroupManagerGlobalParameters.activeGroupsBitmap, groupId);

	//Disconnect all members from the group 
	for(stationNum=0; stationNum<heGroupManagerDb[groupId].numOfStationsInGroup; stationNum++)
	{
		stationId = heGroupManagerDb[groupId].members[stationNum];
		heGroupManagerClassifier_RemoveStationFromGroup(stationId, groupId);
		// Disconnect the Plan in Selector
		heGroupManagerDisableHeMuInSelector(stationId, groupId);		
	}

    groupsGenerator_IncreaseNumOfFreeDataPhases(heGroupManagerDb[groupId].phasesBitmap, heGroupManagerDb[groupId].vapId);
    
	// Reset Group Data Base entry (only for clear debug)
	memset(&heGroupManagerDb[groupId], 0, sizeof(HeGroupManager_GroupDbEntry_t)); // heGroupManagerState = HE_GROUP_MANAGER_GROUP_IDLE;
	*psMsgOriginal = heGroupManagerGlobalParameters.returnMsg;

	return (heGroupManagerGlobalParameters.numOfGroupsInRemoval == 0);
}


/**********************************************************************************

groupsHandler_PlanRemovedDoneGroupsUnderMaintenence




Description:
------------


Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void groupsHandler_PlanRemovedDoneGroupsUnderMaintenence (uint8 groupId)
{
	K_MSG* 			 			heMuDbgStatIndMsg;
	UMI_DBG_HE_MU_GROUP_STATS*	heMuDbgStat;

	heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = INVALID_STA_INDEX;

	/*Send group creation indication for driver*/
	heMuDbgStatIndMsg	= OSAL_GET_MESSAGE(sizeof(UMI_DBG_HE_MU_GROUP_STATS));
	heMuDbgStat = (UMI_DBG_HE_MU_GROUP_STATS*)pK_MSG_DATA(heMuDbgStatIndMsg); 
	heMuDbgStat->groupId = groupId; 
	heMuDbgStat->planType  = heGroupManagerDb[groupId].formationType; 
	heMuDbgStat->vapId = heGroupManagerDb[groupId].vapId; 
	heMuDbgStatIndMsg->header.vapId = heGroupManagerDb[groupId].vapId; 
	heMuDbgStat->setReset = HE_MU_GROUP_RESET;
	OSAL_SEND_MESSAGE(UMI_MC_MAN_HE_MU_DEBUG_IND, TASK_UM_IF_TASK, heMuDbgStatIndMsg, heMuDbgStatIndMsg->header.vapId);	
}


/**********************************************************************************

groupsHandler_PlanRemovedDoneGroupsInRemoval




Description:
------------


Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void groupsHandler_PlanRemovedDoneGroupsInRemoval (K_MSG* psMsgOriginal, uint8 groupId)
{

	K_MSG* 			 			heMuDbgStatIndMsg;
	UMI_DBG_HE_MU_GROUP_STATS*	heMuDbgStat;
	uint8*                      pSetChannelmbufferedMsgVapId;
	uint8                       band;

	switch (heGroupManagerGlobalParameters.removalRequestor)
	{
		case DISABLE_PROCESS:
			if(heGroupManagerGlobalParameters.activeGroupsBitmap == ALL_GROUPS_FREE)
			{
				OSAL_SEND_MESSAGE(UM_MAN_HE_MU_OPERATION_CONFIG_CFM, TASK_UM_IF_TASK, psMsgOriginal, psMsgOriginal->header.vapId);
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);		
			}
			break;
		case DISABLE_DYNAMIC_MU_PROCESS:
			if(heGroupManagerGlobalParameters.numOfGroupsInRemoval == 0)
			{
				OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, psMsgOriginal, psMsgOriginal->header.vapId);
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);		
			}
		  	break;
		case STATION_REMOVAL:
			if(heGroupManagerGlobalParameters.numOfGroupsInRemoval == 0)
			{
				// Send confirmation to Station Manager
				hegroupManager_SendConfirmToStationManager(heGroupManagerGlobalParameters.stationIdCurrentlyProcessed);
				heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = INVALID_STA_INDEX;
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);				
			}
		  	break;
		case REMOVE_GROUPS_FOR_STATION_TID:
			if (heGroupManagerGlobalParameters.numOfGroupsInRemoval == 0)
			{
				//unlock queue to let this station be serviced in su
				Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, heGroupManagerGlobalParameters.stationIdCurrentlyProcessed, 
											(0x1 << heGroupManagerGlobalParameters.tidCurrentlyProcessed));
				heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = INVALID_STA_INDEX;
				
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);
			}
			break;
		case CREATE_STATIC:
			groupsGenerator_CreateStaticPlan(psMsgOriginal);
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_PLAN);
			break;
		case SET_CHANNEL:
			/*Send confirmation to HDK task */
			/*Go over 2 bands - if set channel cfm msg != invalid it means message was buffered-> send cfm to HDK task*/
			for (band = 0; band < ConfigurationManager_GetNumOfActiveBands(); band++) 
			{
				pSetChannelmbufferedMsgVapId = &(heGroupManagerBandParameters[band].setChannelVapIdCfmPendingMsg);
				OSAL_SEND_NO_DATA_MESSAGE(HDK_CHANNEL_NOTIFICATION_CFM, TASK_HDK, *pSetChannelmbufferedMsgVapId);
				*pSetChannelmbufferedMsgVapId = INVALID_VAP; //initiate value after sending message;
			}
			break;
		case SET_ANTENNA:
			/*Send confirmation to HDK task */
			OSAL_SEND_NO_DATA_MESSAGE(HDK_SET_ANT_CONFIG_NOTIFICATION_CFM, TASK_HDK, VAP_ID_DO_NOT_CARE);
			break;
		default: 
			DEBUG_FATAL("heGroupManagerRemovePlanCfm, Not supposed to reach this case");		
			break;
	}

		/*Send group creation indication for driver*/
		heMuDbgStatIndMsg	= OSAL_GET_MESSAGE(sizeof(UMI_DBG_HE_MU_GROUP_STATS));
		heMuDbgStat = (UMI_DBG_HE_MU_GROUP_STATS*)pK_MSG_DATA(heMuDbgStatIndMsg); 
		heMuDbgStat->groupId = groupId; 
		heMuDbgStat->planType  = heGroupManagerDb[groupId].formationType; 
		heMuDbgStat->vapId = heGroupManagerDb[groupId].vapId; 
		heMuDbgStatIndMsg->header.vapId = heGroupManagerDb[groupId].vapId; 
		heMuDbgStat->setReset = HE_MU_GROUP_RESET;
		OSAL_SEND_MESSAGE(UMI_MC_MAN_HE_MU_DEBUG_IND, TASK_UM_IF_TASK, heMuDbgStatIndMsg, heMuDbgStatIndMsg->header.vapId); 

}


/**********************************************************************************

groupsHandler_RemoveStation






Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool groupsHandler_RemoveStation(K_MSG* psMsg)
{
	BssManagerStaManagerReq_t*	pStationManagerRequest = NULL;
	K_MSG*						originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC*			stopTrafficMessageParameters = NULL;
	uint32						groupsBitmap;
	StaId						stationIndex;
	uint8						groupId;
	bool						groupRemoved = FALSE;
	pStationManagerRequest 		 = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	originalRemoveStationMessage = pStationManagerRequest->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
    stationIndex = stopTrafficMessageParameters->u16SID;

	if(heGroupManagerStationDb[stationIndex].heGroupManagerStationState != HE_GROUP_MANAGER_STATION_IN_A_GROUP)
	{
		hegroupManager_SendConfirmToStationManager(stationIndex);
	}
	else
	{
		heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = stationIndex;					

		groupsBitmap = heGroupManagerStationDb[stationIndex].groupsBitmap;

		/*Remove all groups that removed STA is a member of them*/
		while(groupsBitmap != 0)
		{
			groupId = Utils_CountTrailingZeros(groupsBitmap);
			Utils_ZeroBitInBitmap(&groupsBitmap, groupId);
			groupsHandler_RemoveGroup(groupId);
			groupRemoved = TRUE;
		}
	}
    heGroupManagerClassifier_RemoveStation(stationIndex);
    heGroupManagerStationDb[stationIndex].heGroupManagerStationState = HE_GROUP_MANAGER_STATION_NOT_ELIGIBLE;
	return groupRemoved;
		
}	


/**********************************************************************************

groupsHandler_RemoveGroup 


Description:
------------


Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void groupsHandler_RemoveGroup(uint8 groupId)
{
	RequesterLockParams_t	lockRequesterParams;
	uint8					vapId;
	uint8					planIndex;
	CommonPlanCommon_t*		pPlan;
	
	ILOG0_D("[groupsHandler_RemoveGroup], groupId = %d", groupId);
	heGroupManagerDb[groupId].heGroupManagerState = HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_LOCK;

	heGroupManagerGlobalParameters.numOfGroupsInRemoval ++;
	
	vapId = heGroupManagerDb[groupId].vapId;

	// Lock Plan
	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
	lockRequesterParams.callerContext = (void *)((uint32)groupId);
	lockRequesterParams.returnTask	  = TASK_HE_GROUP_MANAGER;
	lockRequesterParams.returnMsg	  = HE_GROUP_MANAGER_LOCK_PLAN_CFM;
	
	pPlan 	  = heGroupManagerDb[groupId].planPtr;
	planIndex = pPlan->planIndex;

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_DD("[groupsHandler_RemoveGroup], vapId: %d, planIndex: %d", vapId, planIndex);
    SLOG0(0, 0, RequesterLockParams_t, &lockRequesterParams);
#endif

	// Call Locker, will be back through message (HE_GROUP_MANAGER_LOCK_PLAN_CFM)	
	Locker_LockPlan(planIndex, vapId, &lockRequesterParams);
}



/**********************************************************************************

groupsHandler_RemoveAllActiveGroups





Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool groupsHandler_RemoveAllActiveGroups(void)
{
	uint32 activeGroupsBitmap = 0;
	uint8  groupId   		  = 0;
	bool   groupRemoved       = FALSE;

    activeGroupsBitmap = heGroupManagerGlobalParameters.activeGroupsBitmap;

	while(activeGroupsBitmap != ALL_GROUPS_FREE)
	{
		groupId = Utils_CountTrailingZeros(activeGroupsBitmap);
		Utils_ZeroBitInBitmap(&activeGroupsBitmap, groupId);
		groupsHandler_RemoveGroup(groupId);
		groupRemoved = TRUE;
	}			
	return groupRemoved;
}


/**********************************************************************************

groupsHandler_BandConfigurationChanged





Description:
------------

Input: 
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
bool groupsHandler_BandConfigurationChanged(BandId_e bandId)
{
	uint32 activeGroupsBitmap = 0;
	uint8  groupId   		  = 0;
	uint32 bandLimitationBitmap ;
	uint32 groupCharacteristicsBitmap ;
	uint8 firstVapInBand;
	uint8 numOfVaps;
	uint8 vapId;
	bool groupRemoved = FALSE;
	
	bandLimitationBitmap = heGroupManagerBandParameters[bandId].bandLimitationBitmap.val;
	activeGroupsBitmap = heGroupManagerGlobalParameters.activeGroupsBitmap ;
		
	firstVapInBand = ConfigurationManager_GetFirstVapForBand(bandId);
	numOfVaps = ConfigurationManager_GetNumOfVapsInBand(bandId);

	for (vapId = firstVapInBand ; vapId < numOfVaps ; vapId++)
	{
		/*Loop over all vaps in band*/
		if (VapDatabaseObj.vapDbSwEntries[vapId].state == VAP_STATE_ADDED)
		{
			while(activeGroupsBitmap != ALL_GROUPS_FREE)
			{
				/*Loop over all groups in VAP*/
				if (heGroupManagerDb[groupId].vapId == vapId)
				{
					/*If group charateristics bitmap does not match to band limitation charateristics bitmap - remove group*/
					groupCharacteristicsBitmap = heGroupManagerDb[groupId].groupCharacteristicsBitmap.val;
					if ((bandLimitationBitmap & groupCharacteristicsBitmap)!= groupCharacteristicsBitmap)
					{
						groupRemoved = TRUE;
						groupsHandler_RemoveGroup(groupId);
					}
				}
				groupId = Utils_CountTrailingZeros(activeGroupsBitmap);
				Utils_ZeroBitInBitmap(&activeGroupsBitmap, groupId);
			}			
		}
	}
	return groupRemoved;
}


