/***********************************************************************************
 File:			HeGroupManager.c
 Module:		He Group Manager
 Purpose: 		To create and delete multi HE MU groups
 Description:   This file is the implementation of the HE MU group manager which is 
 				responsible of creating and deleting multi user groups (MIMO & OFDMA)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "HeGroupManager_API.h"
#include "HeGroupManager.h"
#include "Utils_Api.h"
#include "OSAL_Api.h"
#include "loggerAPI.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "queue_utility.h"
#include "mhi_umi.h"
#include "CommonPlan_Descriptors.h"
#include "ShramStationDatabase.h"
#include "PlanManager_API.h"
#include "CoC_Api.h"
#include "HeGroupManagerClassifier.h"
#include "GroupsGenerator.h"
#include "GroupsHandler.h"
#include "ShramPlannerResources.h"
#include "HdkCdbManagerTask_api.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HE_GROUP_MANAGER
#define LOG_LOCAL_FID 0

#define HE_GROUP_MANAGER_DEBUG
 
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

const bool supportedDlMuType [DYNAMIC_DL_UL_MU_TYPE_NUM] = { 1, 1, 0, 0};
const bool supportedUlMuType [DYNAMIC_DL_UL_MU_TYPE_NUM] = { 1, 0, 0, 0};

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void  heGroupManagerEnableHeMuInSelector(uint8 groupId);
void  heGroupManagerDisableHeMuInSelector(StaId stationId, uint8 groupId);
static void heGroupManagerSetStaticPlanReq(K_MSG* psMsg);
static void heGroupManagerRemoveStaticPlanReq(K_MSG* psMsg);
static void heGroupManagerSetStaticPlan(K_MSG* psMsg);
static void heGroupManagerCreatePlanCfm(K_MSG* psMsg);
static void heGroupManagerRemovePlanCfm(K_MSG* psMsg);
static void heGroupManagerLockPlanCfm(K_MSG* psMsg);
static void heGroupManagerTimerExpired(K_MSG* psMsg);
static void heGroupManagerAddStationReq(K_MSG* psMsg);
static void heGroupManagerRemoveStationReq(K_MSG* psMsg);
static void heGroupManagerLnaChange(K_MSG* psMsg);
static void heGroupManagerRecoverFromPdTypeNotAllowedReq(K_MSG* psMsg);
static void heGroupManagerUpdateCoCInfo(K_MSG* psMsg);
void  heGroupManagerChangeGlobalState(HeGroupManagerGlobalState_e newState);
void  hegroupManager_SendConfirmToStationManager(StaId stationIndex);
static void	 heGroupManagerUpdateSelectorHeBitmap(StaId stationId, bool isHeDl, bool enableUl);
void heGroupManagerInitGroupPoliciesTable(void);
#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A after Jira WLANVLSIIP-2881 is resolved in B0
bool  heGroupManagerStationAcPsTypesAreMixed(uint8 acPsTypeBitmap);
#endif // PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE
void HeGroupManager_TaskEntry(K_MSG *psMsg);
static void heGroupManagerCollectingMetrics(void);
static void heGroupManagerSendEventToGlobalSm(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupMangerAddSta(K_MSG* pMsg);
static void heGroupManagerUpdateMetricsCfm(K_MSG* pMsg);
static void heGroupManagerSetChannelReq(K_MSG* pMsg);
static void heGroupManagerSetAntsReq(K_MSG* pMsg);
static void heGroupManagerSetChannelInDisableState(K_MSG* pMsg);
static void heGroupManagerSetAntsInDisableState(K_MSG* pMsg);
static bool heGroupManagerSetChannel(K_MSG* pMsg);
static void heGroupManagerSetAnts(K_MSG* pMsg);
static void heGroupManagerSetStaAssocParamsFromLa(K_MSG* psMsg);
static void heGroupManagerSetNumOfStaInGroupConfiguration(K_MSG* psMsg);
static void heGroupManagerGetStaAssocParamsFromLa(K_MSG* pAddStaMsg);
/*Global state machine functions*/
static void heGroupManagerGlobalStateMachine(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateDisableHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateIdleHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateCollectingMetricsHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateUnderMaintenenceHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateInRemovalHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateInCreationHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateCreatingStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerGlobalStateRemovingStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg);
static void heGroupManagerRemoveStaticPlanDoneEventHanldler(K_MSG*	psMsgOriginal);
static void heGroupManagerRemoveStationInDisableState(K_MSG* psMsg);
static void heGroupManagerSetDynamicMuParams(K_MSG* psMsg);
static void heGroupManagerSetDynamicMuReq(K_MSG* psMsg);
static void heGroupManagerConvertDlMuType2FormationType(dlUlMuType_e dlMuType, FormationType_e* option1, FormationType_e* option2);
static void heGroupManagerConvertULMuType2FormationType(dlUlMuType_e ulMuType, FormationType_e* option1, FormationType_e* option2);
static void heGroupManagerSetDefaultCdbConfiguration(void);
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

//HEGroupManagerStatistics_t hegroupDbStatistics;

HeGroupManager_GlobalParameters_t  heGroupManagerGlobalParameters;
HeGroupManager_StationDbEntry_t	  heGroupManagerStationDb[HW_NUM_OF_STATIONS];
HeGroupManager_BandParameters_t  heGroupManagerBandParameters[NUM_OF_CONFIGURATION_MANAGER_BANDS];

static const FunctionEntry_t afpTaskTable[TASK_HE_GROUP_MANAGER_END - TASK_HE_GROUP_MANAGER_START] =
{
	/* HE Group manager main state machine messages */
	{heGroupManagerAddStationReq,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_ADD_STATION)},
	{heGroupManagerRemoveStationReq,				DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_REMOVE_STATION)},
	{heGroupManagerSetStaticPlanReq,				DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_SET_STATIC_PLAN_REQ)},
	{heGroupManagerRemoveStaticPlanReq,				DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_REMOVE_STATIC_PLAN_REQ)},
	{heGroupManagerCreatePlanCfm,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_CREATE_PLAN_CFM)},
	{heGroupManagerRemovePlanCfm,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_REMOVE_PLAN_CFM)},
	{heGroupManagerLockPlanCfm, 					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_LOCK_PLAN_CFM)},
	{heGroupManagerTimerExpired,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_TIMER_EXPIRED)},
	{heGroupManagerLnaChange,						DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_LNA_TYPE_HAS_CHANGED_IND)},
	{heGroupManagerRecoverFromPdTypeNotAllowedReq,	DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_RECOVER_FROM_PD_TYPE_NOT_ALLOWED_REQ)},
	{heGroupManagerUpdateCoCInfo,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_UPDATE_COC_INFO)},
	{heGroupManagerUpdateMetricsCfm,				DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_METERICS_CFM)},
	{heGroupManagerSetChannelReq,					DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_SET_CHANNEL)},
	{heGroupManagerSetAntsReq,						DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_SET_ANTENNAS)},
	{heGroupManagerSetStaAssocParamsFromLa,			DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_SET_STA_ASSOC_BW_AND_ANTS_FROM_LA)},
    {heGroupManagerSetNumOfStaInGroupConfiguration,	DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_SET_MAX_MIN_NUM_OF_STA_IN_GROUP)},
	{heGroupManagerSetDynamicMuReq, 				DOUBLE_CHECK_MSG_TYPE(HE_GROUP_MANAGER_ENABLE_DYNAMIC_MU)},
	
};



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
// This Data Base is common to the HE Group Manager and the Plan manager

HeGroupManager_GroupDbEntry_t	   heGroupManagerDb[HE_GROUP_MANAGER_MAX_NUMBER_OF_GROUPS];

HeGroupManagerGroupPolicy_t        heGroupManagerGroupPolicies[MAX_GROUP_POLICIES] = 
{
#ifndef ENET_INC_ARCH_WAVE600B
    {0x15,{1,0,0,0,0,0,0},HE_MU_SEQ_VHT_LIKE,DL_OFDMA,8,2,0,1,1,0} , // DL, 160M, HE_MU_SEQ_VHT_LIKE, DL_OFDMA     // we don't support RU of 160MHZ in A0 (phy issue)
#else
    {0x15,{1,0,0,0,0,0,0},HE_MU_SEQ_VHT_LIKE,DL_OFDMA,8,1,0,1,1,0} , // DL, 160M, HE_MU_SEQ_VHT_LIKE, DL_OFDMA     // num of phase repetitions is 0 for DL according to jira WLANRTSYS-12368    
#endif
    {0x25,{1,0,0,0,0,0,0},HE_MU_SEQ_VHT_LIKE,DL_OFDMA,8,1,0,1,1,0} , // DL, 80M,  HE_MU_SEQ_VHT_LIKE, DL_OFDMA
	{0x45,{1,0,0,0,0,0,0},HE_MU_SEQ_VHT_LIKE,DL_OFDMA,8,1,0,1,1,0} , // DL, 40M,  HE_MU_SEQ_VHT_LIKE, DL_OFDMA
	{0x85,{1,0,0,0,0,0,0},HE_MU_SEQ_VHT_LIKE,DL_OFDMA,8,1,0,1,1,0} , // DL, 20M,  HE_MU_SEQ_VHT_LIKE, DL_OFDMA
    {0x19,{0,1,0,0,0,0,0},HE_MU_UL          ,UL_OFDMA,8,1,1,1,1,0} , // UL, 160M, HE_MU_UL,           UL_OFDMA
	{0x29,{0,1,0,0,0,0,0},HE_MU_UL          ,UL_OFDMA,8,1,1,1,1,0} , // UL, 80M,  HE_MU_UL,           UL_OFDMA
	{0x49,{0,1,0,0,0,0,0},HE_MU_UL		    ,UL_OFDMA,8,1,1,1,1,0} , // UL, 40M,  HE_MU_UL,			  UL_OFDMA
	{0x89,{0,1,0,0,0,0,0},HE_MU_UL		    ,UL_OFDMA,8,1,1,1,1,0} , // UL, 20M,  HE_MU_UL,			  UL_OFDMA
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} ,
    {0x00,{0,0,0,0,0,0,0},0,0,0,0,0,0,0,0} , // groupCharacteristics, phasesBitmap, sequence, formationType, maxStationsInGroup, minStationsInGroup, numberOfRepetitions, validGroup, numOfGroupsForPolicy, reserved
};

/**********************************************************************************
HeGroupManager_TaskEntry 

Description:
------------
the entry point of the group manager task

Input: 
-----
groupManagerMessage - pointer to the message to handle	
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void HeGroupManager_TaskEntry(K_MSG *psMsg)
{
	/* Use common task switching and Table */ 
	vTaskDispatcher(psMsg, afpTaskTable, TASK_HE_GROUP_MANAGER_START, TASK_HE_GROUP_MANAGER_END);
}


static void heGroupManagerSetStaticPlanReq(K_MSG* psMsg)
{
		heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN, psMsg);
}



/**********************************************************************************
heGroupManagerSetStaticPlanReq 

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerSetStaticPlan(K_MSG* psMsg)
{
	UMI_STATIC_PLAN_CONFIG*	pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)psMsg->abData;
	bool groupRemoved = FALSE;
	// Check if not already a plan of same type
	if((heGroupManagerGlobalParameters.numOfActiveStaticPlan != 0) &&
		(((pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_DL_DATA) && (heGroupManagerGlobalParameters.activeGroupsBitmap & (1 << HE_GROUP_MANAGER_DL_GID))) ||
		((pMuPlanConfig->commonSection.phaseFormat == PHASE_FORMAT_UL_DATA) && (heGroupManagerGlobalParameters.activeGroupsBitmap & (1 << HE_GROUP_MANAGER_UL_GID)))))
	{ // there is already a plan for the same type
			
		ILOG0_V("There is already an active Plan for this type (UL/DL) ! "); 
		OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);		
	}
	else
	{
		// store original message for later confirmation
		heGroupManagerGlobalParameters.returnMsg = psMsg;

		if((heGroupManagerGlobalParameters.numOfActiveStaticPlan == 0) &&
		   (heGroupManagerGlobalParameters.activeGroupsBitmap != ALL_GROUPS_FREE))
		{ // There is at least 1 non-static active group
			heGroupManagerGlobalParameters.removalRequestor = CREATE_STATIC;
			groupRemoved = groupsHandler_RemoveAllActiveGroups();
			if (groupRemoved == TRUE)
			{
				heGroupManagerGlobalParameters.removalRequestor = CREATE_STATIC;
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
			}
			else
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_PLAN);
			}
		}
		else
		{
			groupsGenerator_CreateStaticPlan(psMsg);
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_PLAN);
		}
	}
}


static void heGroupManagerRemoveStaticPlanReq(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN,psMsg);
}


/**********************************************************************************
heGroupManagerRemoveStaticPlan 

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
bool groupsHandler_RemoveStaticPlan(K_MSG* psMsg)
{
	UMI_STATIC_PLAN_CONFIG* pMuPlanConfig = (UMI_STATIC_PLAN_CONFIG*)pK_MSG_DATA(psMsg);
	uint8					groupId;
	
	heGroupManagerGlobalParameters.returnMsg = psMsg;

	if(pMuPlanConfig->commonSection.phaseFormat == DOWNLINK_PLAN)
	{
		groupId = 0;
		if((heGroupManagerGlobalParameters.activeGroupsBitmap & 0x1) == 0)
		{
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
			return FALSE;
		}
	}
	else
	{
		groupId = 1;
		if((heGroupManagerGlobalParameters.activeGroupsBitmap & 0x2) == 0)
		{
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
			return FALSE;
		}
	}
	groupsHandler_RemoveGroup(groupId);

	return TRUE;
}


static void heGroupManagerCreatePlanCfm(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE, psMsg);
}


static void heGroupManagerRemovePlanCfm(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE, psMsg);

}
	

/**********************************************************************************
heGroupManagerLockPlanCfm

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerLockPlanCfm(K_MSG* psMsg)
{
	RemovePlanMsg_t* pHeGroupManagerRemovePlanMsg = NULL;
	K_MSG* 			 heGroupManagerMsg = NULL;
	LockPlanReqCb_t* lockPlanReqCb = NULL;
	uint8		 	 groupId;
    
	lockPlanReqCb = (LockPlanReqCb_t *)pK_MSG_DATA(psMsg);

	// Get groupId from Locker message
	groupId = (uint8)lockPlanReqCb->callerContext;

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[heGroupManagerLockPlanCfm], groupId: %d", groupId);
#endif

	heGroupManagerMsg 			 = OSAL_GET_MESSAGE(sizeof(RemovePlanMsg_t));
	pHeGroupManagerRemovePlanMsg = (RemovePlanMsg_t*) pK_MSG_DATA(heGroupManagerMsg);

	// Prepare message to Plan Manager for removing the plan for this group
	pHeGroupManagerRemovePlanMsg->groupId = groupId;
	heGroupManagerMsg->header.vapId 	  = heGroupManagerDb[groupId].vapId;
	heGroupManagerDb[groupId].heGroupManagerState = HE_GROUP_MANAGER_GROUP_WAIT_FOR_PLAN_REMOVAL;

	OSAL_SEND_MESSAGE(PLAN_MANAGER_REMOVE_PLAN, TASK_PLAN_MANAGER, heGroupManagerMsg, heGroupManagerMsg->header.vapId);

}


/**********************************************************************************
heGroupManagerTimerExpired

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerTimerExpired(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED,psMsg);

}


#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A after Jira WLANVLSIIP-2881 is resolved in B0
/**********************************************************************************
heGroupManagerStationTidsPsTypesAreMixed

Description: if TIDs ps types are mixed, some Legacy and some WMM return FALSE, otherwise return TRUE
------------

Input: ti
-----

Returns:
--------
	void - 	
**********************************************************************************/
bool  heGroupManagerStationAcPsTypesAreMixed(uint8 acPsTypeBitmap)
{
 // only 4 LSB bits are relevant from this bitmap
 	uint8 psTypeBitmap = (acPsTypeBitmap & 0xF);
	if ( (psTypeBitmap == 0) || (psTypeBitmap == 0xF))
	{
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}
#endif // PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE


/**********************************************************************************
heGroupManagerFlushPendingQueue

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerFlushPendingQueue(void)
{
	K_MSG*			 pPendingMessage;

	pPendingMessage = FwUtils_Dequeue(&heGroupManagerGlobalParameters.pendingQueue);
	while(pPendingMessage != NULL)
	{// flush the pending messages
		OSAL_SEND_MESSAGE_TO_FRONT(pPendingMessage->header.tKMsgType, TASK_HE_GROUP_MANAGER, pPendingMessage, pPendingMessage->header.vapId);
		pPendingMessage = FwUtils_Dequeue(&heGroupManagerGlobalParameters.pendingQueue);
	}
}


/**********************************************************************************
heGroupManagerLnaChange

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerLnaChange(K_MSG* psMsg)
{
	UNUSED_PARAM(psMsg);	
//	StaId	stationId;
//	uint8	groupId;
//	
//	LaLnaTypeIndMsg_t* plnaTypeMsgParams = (LaLnaTypeIndMsg_t*)pK_MSG_DATA(psMsg);
//
//	stationId = plnaTypeMsgParams->staId;
//
//	
//	switch(heGroupManagerGlobalParameters.globalState)
//	{
//		case HE_GROUP_MANAGER_GLOBAL_STATE_IDLE:
//		case HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED:
//			heGroupManagerStationDb[stationId].lnaType = plnaTypeMsgParams->lnaType;				
//			if(heGroupManagerStationDb[stationId].groupsBitmap != 0)
//			{ // in case of DISABLED it always will be groupsBitmap ==0
//				groupId = Utils_CountLeadingZeros(heGroupManagerStationDb[stationId].groupsBitmap);
//				heGroupManagerGlobalParameters.removalRequestor = REMOVE_GROUP;
//				groupsHandler_RemoveGroup(groupId);
//			}
//			break;
//		default:
//			if(heGroupManagerStationDb[stationId].groupsBitmap == 0)
//			{
//				heGroupManagerStationDb[stationId].lnaType = plnaTypeMsgParams->lnaType;
//			}
//			else
//			{// there is a group this station is member to
//				// enqueue to pending queue
//				FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, psMsg);
//			}
//			break;
//	}			
}


static void heGroupManagerAddStationReq(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA,psMsg);
}


static void heGroupManagerRemoveStationReq(K_MSG* psMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA,psMsg);
}


static void heGroupManagerUpdateCoCInfo(K_MSG* psMsg)
{
	cocUpdateAntMaskMsg_t*	pCoCInfo = (cocUpdateAntMaskMsg_t*)pK_MSG_DATA(psMsg);
	uint8 vapIndex = psMsg->header.vapId;
	BandId_e bandId = ConfigurationManager_GetBandForVap(vapIndex);
	
	groupsGenerator_SetAntennasConfiguration(pCoCInfo->currentTxAntsNum,bandId);
}


/**********************************************************************************
heGroupManagerRecoverFromPdTypeNotAllowedReq

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
static void heGroupManagerRecoverFromPdTypeNotAllowedReq(K_MSG* psMsg)
{
	HeGm_RecoverFromPdTypeNotAllowedRequest_t *pRecoverFromPdTypeNotAllowedRequest = NULL;
	uint32 groupsBitmap;
	StaId stationIndex;
	uint8 groupId;

	pRecoverFromPdTypeNotAllowedRequest = (HeGm_RecoverFromPdTypeNotAllowedRequest_t *)pK_MSG_DATA(psMsg);
	
	stationIndex = pRecoverFromPdTypeNotAllowedRequest->staId;

	if (heGroupManagerStationDb[stationIndex].heGroupManagerStationState != HE_GROUP_MANAGER_STATION_IN_A_GROUP)
	{
		//this can happen when a transmission from another tid of the station was triggered while station was being removed from all groups
		//just need to unlock the queue and continue
		ILOG0_DD("station %d (tid %d) is not part of any he mu group", stationIndex, pRecoverFromPdTypeNotAllowedRequest->tid);
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, stationIndex, (0x1 << pRecoverFromPdTypeNotAllowedRequest->tid));
	}
	else
	{
		switch(heGroupManagerGlobalParameters.globalState)
		{
			case HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED:
				FATAL("OTFA should not have ask for remove all groups for this sta/tid because he gm is disabled");
				break;
			case HE_GROUP_MANAGER_GLOBAL_STATE_IDLE:
				heGroupManagerGlobalParameters.removalRequestor = REMOVE_GROUPS_FOR_STATION_TID;
				heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = stationIndex;
				heGroupManagerGlobalParameters.tidCurrentlyProcessed = pRecoverFromPdTypeNotAllowedRequest->tid;
				
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);

				groupsBitmap = heGroupManagerStationDb[stationIndex].groupsBitmap;
				while(groupsBitmap != 0)
				{
					groupId = Utils_CountTrailingZeros(groupsBitmap);
					Utils_ZeroBitInBitmap(&groupsBitmap, groupId);
					groupsHandler_RemoveGroup(groupId);
				}
				break;
			default:
				// enqueue to pending list	
				FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, psMsg);
				break;
		}
	}
		
}	


/**********************************************************************************
hegroupManager_SendConfirmToStationManager

Description:
------------
Sends Confirmation message on removal of the station to BSS manager.
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void hegroupManager_SendConfirmToStationManager(StaId stationIndex)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *pConfirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	pConfirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	pConfirmMessage->clientId = BSS_MANAGER_STA_MANAGER_HE_GM_CLIENT;
	/*Set STA ID*/
	pConfirmMessage->sid = stationIndex;
	/*Send confirmation message*/
#ifdef HE_GROUP_MANAGER_DEBUG								
	ILOG0_D("HE Group Manager, Send Confirmation, stationIndex %d", stationIndex);
#endif
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}


/**********************************************************************************
heGroupManagerChangeGlobalState

Description:
------------
change the global state

newState - the new state to change to
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerChangeGlobalState(HeGroupManagerGlobalState_e newState)
{
#ifdef DYNAMIC_GROUPING_DEBUG								
    ILOG0_DD("heGroupManagerChangeGlobalState: current state: %d, new state %d", heGroupManagerGlobalParameters.globalState , newState);
#endif

	heGroupManagerGlobalParameters.globalState = newState;

	if(newState == HE_GROUP_MANAGER_GLOBAL_STATE_IDLE)
	{
		heGroupManagerFlushPendingQueue();
		OSAL_SET_TIMER_EXPLICIT(HE_GROUP_MANAGER_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(HE_GROUP_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_HE_GROUP_MANAGER);
	}
	else if ((newState == HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED)|| (newState == HE_GROUP_MANAGER_GLOBAL_STATE_STATIC_PLAN))
	{
		heGroupManagerFlushPendingQueue();		
		OSAL_RESET_TIMER_EXPLICIT(HE_GROUP_MANAGER_TIMER_EXPIRED, TASK_HE_GROUP_MANAGER);
	}
}


/**********************************************************************************
heGroupManagerEnableHeMuInSelector

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerEnableHeMuInSelector(uint8 groupId)
{
	StaId	stationId;
	uint8	stationNum = 0;
	uint8	planIndex;
	bool 	isDl = TRUE;
	TX_INTERRUPT_SAVE_AREA;
	
	planIndex = heGroupManagerDb[groupId].planPtr->planIndex;
	for(stationNum=0; stationNum<heGroupManagerDb[groupId].numOfStationsInGroup; stationNum++)
	{	
		stationId = heGroupManagerDb[groupId].members[stationNum];

#ifdef ENET_INC_ARCH_WAVE600D2
		if((heGroupManagerDb[groupId].formationType == UL_OFDMA) || (heGroupManagerDb[groupId].formationType == UL_MIMO))
#else
		if(heGroupManagerDb[groupId].formationType == UL_OFDMA)
#endif
		{
			ILOG0_V("heGroupManagerEnableHeMuInSelector - UL MU"); 
			isDl = FALSE;
			TxSelector_SetULPlanIndex(stationId, planIndex);
		}
		else
		{
			ILOG0_V("heGroupManagerEnableHeMuInSelector - DL MU"); 
			TxSelector_SetDLPlanIndex(stationId, planIndex);
		}

		// The following code should be atomic so disable interrupt
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM = heGroupManagerGlobalParameters.tidsHeMuEnableByPlan; //TID_BITMAP_ALL_TIDS;
		heGroupManagerUpdateSelectorHeBitmap(stationId, isDl, TRUE);
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
}


/**********************************************************************************
heGroupManagerDisableHeMuInSelector

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerDisableHeMuInSelector(StaId stationId, uint8 groupId)
{
	bool isHeDl = TRUE;
	TX_INTERRUPT_SAVE_AREA;
	// Set Locked Plan as planIndex
#ifdef ENET_INC_ARCH_WAVE600D2
	if((heGroupManagerDb[groupId].formationType == UL_OFDMA) || (heGroupManagerDb[groupId].formationType == UL_MIMO))
#else
	if(heGroupManagerDb[groupId].formationType == UL_OFDMA)
#endif
	{
		isHeDl = FALSE;
	 	TxSelector_SetULPlanIndex(stationId, HE_GROUP_MANAGER_INVALID_PLAN_INDEX);
	}
	else
	{
		 TxSelector_SetDLPlanIndex(stationId, HE_GROUP_MANAGER_INVALID_PLAN_INDEX);
	}

	if(heGroupManagerStationDb[stationId].groupsBitmap == 0)
	{// if this station is not member in any other group, disable HE MU in selector
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM = NONE_TIDS;
		heGroupManagerUpdateSelectorHeBitmap(stationId, isHeDl, FALSE);	
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
		
}


/**********************************************************************************
heGroupManager_GetTSManagerBitmap

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
uint16 heGroupManagerGetTSManagerTidsBitmap(StaId stationId)
{
	return heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager;
}

    
/**********************************************************************************
heGroupManager_GetNumOfActiveAntennasPerBand

Description:
------------
		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
uint16 heGroupManager_GetNumOfActiveAntennas(uint8 vapId)
{
	BandId_e band = ConfigurationManager_GetBandForVap(vapId);
	
	return heGroupManagerBandParameters[band].numOfActiveAntennas;
}


/***********************************************************************
* heGroupManagerSetPlanTidsBitmap
* 
* Description:
* ------------
* Store the TIDs that are enable for MU HE. They are enabled by Plan manager.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void heGroupManagerSetPlanTidsBitmap(uint32 tidAllocBitmap, uint8 muSequenceType)
{	
	ILOG0_D("[NO AGG] heGroupManagerSetPlanTidsBitmap 1 tidAllocBitmap = 0x%x", tidAllocBitmap);

	/* Management frames are not part of the plan (since for example the sta will send ACK that shall colide with our: MU-BAR TF) */
	switch(muSequenceType)
	{
		case HE_MU_SEQ_MU_BAR:
		case HE_MU_SEQ_VHT_LIKE:
		case HE_MU_SEQ_DL_BASIC_TF:
		case HE_MU_SEQ_VHT_LIKE_IMM_ACK:
		case HE_MU_SEQ_VHT_LIKE_PROTECTION:
			tidAllocBitmap &= ~(TID_BITMAP_TID_MGMT);
			break;			
		case HE_MU_UL:
		case HE_MU_BSRP:
		case HE_MU_BSRP_UL:
			break;
		default:
			FATAL("[heGroupManagerSetPlanTidsBitmap], UNKNOWN muSequenceType");
			break;
	}
	ILOG0_D("[NO AGG] heGroupManagerSetPlanTidsBitmap 2 tidAllocBitmap = 0x%x", tidAllocBitmap);
	heGroupManagerGlobalParameters.tidsHeMuEnableByPlan = tidAllocBitmap;
}


/**********************************************************************************
heGroupManager_GetHeGroupManagerBitmap

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
uint16 heGroupManagerGetHeGroupManagerTidsBitmap(StaId stationId)
{
	return heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM;
}


/**********************************************************************************
heGroupManager_TSManagerEnableTid

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerTSManagerEnableTid(StaId stationId, uint8 tidIndex)
{
	TX_INTERRUPT_SAVE_AREA;
		
	ILOG0_DDDD("heGroupManagerTSManagerEnableTid BEFORE, StationId: %d, tidIndex %d, current status of bitmaps: tidsHeMuEnablebyTSManager: 0x%X, tidsHeMuEnablebyHeGM 0x%X", stationId, tidIndex, heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager, heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager |= (1 << tidIndex); 
	/* The assumption is that BAR Tx is relevant only for DL plan. if UL plan could include DL phase, HE UL should be enabled too*/
	heGroupManagerUpdateSelectorHeBitmap(stationId, TRUE, TRUE); 
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

	ILOG0_DDDD("heGroupManagerTSManagerEnableTid AFTER, StationId: %d, tidIndex %d, current status of bitmaps: tidsHeMuEnablebyTSManager: 0x%X, tidsHeMuEnablebyHeGM 0x%X", stationId, tidIndex, heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager, heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM);

}


/**********************************************************************************
heGroupManagerTSManagerDisableTid

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerTSManagerDisableTid(StaId stationId, uint8 tidIndex)
{
	uint32 mask;
	TX_INTERRUPT_SAVE_AREA;

	ILOG0_DDDD("heGroupManagerTSManagerDisableTid BEFORE, StationId: %d, tidIndex %d, current status of bitmaps: tidsHeMuEnablebyTSManager: 0x%X, tidsHeMuEnablebyHeGM 0x%X", stationId, tidIndex, heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager, heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM);
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	mask = ~(1 << tidIndex);
	heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager &= mask;
	/* The assumption is that BAR Tx is relevant only for DL plan. if UL plan could include DL phase, HE UL should be disabled too*/
	heGroupManagerUpdateSelectorHeBitmap(stationId, TRUE, FALSE);

	ILOG0_DDDD("heGroupManagerTSManagerDisableTid AFTER, StationId: %d, tidIndex %d, current status of bitmaps: tidsHeMuEnablebyTSManager: 0x%X, tidsHeMuEnablebyHeGM 0x%X", stationId, tidIndex, heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager, heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


/**********************************************************************************
heGroupManagerUpdateSelectorHeBitmap

Description:
------------
 		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerUpdateSelectorHeBitmap(StaId stationId, bool isHeDl, bool enableUl)
{
	uint32 tidBitmap;

#ifdef ENET_INC_ARCH_WAVE600B
	if (isHeDl == TRUE)
	{
		tidBitmap = heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM & heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager;
		TxSelector_HeMuDlTidUpdate(stationId, tidBitmap);
	}
	else
	{
		TxSelector_HeMuUlStationUpdate(stationId, enableUl);
	}
#else
	UNUSED_PARAM(isHeDl);
	UNUSED_PARAM(enableUl);	
	tidBitmap = heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM & heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager;
	TxSelector_HeMuTidUpdate(stationId, tidBitmap);
#endif
}


/**********************************************************************************
heGroupManager_IsStationInStaticPlan

Description:
------------

Input: 
-----
		
Output:
-------

Returns:
--------

**********************************************************************************/
bool heGroupManager_IsStationInStaticPlan(StaId stationId)
{
	bool retVal;

	retVal = ((heGroupManagerGlobalParameters.numOfActiveStaticPlan != 0) && (heGroupManagerStationDb[stationId].heGroupManagerStationState == HE_GROUP_MANAGER_STATION_IN_A_GROUP));

	return retVal;
}


/**********************************************************************************
heGroupManagerInitGroupPoliciesTable

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerInitGroupPoliciesTable(void)
{
    heGroupManagerGlobalParameters.groupPoliciesTable.maxNumOfGroupsPerPolicy      = 1;
    heGroupManagerGlobalParameters.groupPoliciesTable.numOfPolicies                = 8;
    heGroupManagerGlobalParameters.groupPoliciesTable.pHeGroupManagerGroupPolicies = heGroupManagerGroupPolicies;
}


static void heGroupManagerSendEventToGlobalSm(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	heGroupManagerGlobalStateMachine(event, pMsg);
}


static void heGroupManagerGlobalStateMachine(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	HeGroupManagerGlobalState_e globalState = heGroupManagerGlobalParameters.globalState;

	ILOG0_DD("heGroupManagerGlobalStateMachine, current state: %d event: %d", globalState, event);
	ASSERT(globalState <= HE_GROUP_MANAGER_GLOBAL_LAST_STATE);
	ASSERT(event <= HE_GROUP_MANAGER_GLOBAL_LAST_EVENT);

	switch (globalState)
	{
		case HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED:
		{
			heGroupManagerGlobalStateDisableHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_IDLE:
		{
			heGroupManagerGlobalStateIdleHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_COLLECTING_METRICS:
		{
			heGroupManagerGlobalStateCollectingMetricsHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_UNDER_MAINTENENCE:
		{
			heGroupManagerGlobalStateUnderMaintenenceHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_GROUPS_IN_CREATION:
		{
			heGroupManagerGlobalStateInCreationHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL:
		{
			heGroupManagerGlobalStateInRemovalHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_CREATING_STATIC_PLAN:
		{
			heGroupManagerGlobalStateCreatingStaticPlanHandler(event, pMsg);
			break;
		}
		case HE_GROUP_MANAGER_GLOBAL_STATE_STATIC_PLAN:
		{
			heGroupManagerGlobalStateStaticPlanHandler(event, pMsg);
			break;
		}
		
		case HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN:
		{
			heGroupManagerGlobalStateRemovingStaticPlanHandler(event, pMsg);
			break;
		}
	}
}

static void heGroupManagerCollectingMetrics(void)
{
	K_MSG* pConfigSmMsg;
	HdkCdbManagerConfigMsg_t* pConfigSmParams;
	BandConfigurationMode_e bandConfiguration = ConfigurationManager_GetBandConfigurationMode();
	/*Send message to configuration SM to set add vap common configuration in bands*/
	pConfigSmMsg = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerConfigMsg_t));
	pConfigSmParams = (HdkCdbManagerConfigMsg_t *)pK_MSG_DATA(pConfigSmMsg);
	pConfigSmParams->cbMessage = HE_GROUP_MANAGER_METERICS_CFM;
	pConfigSmParams->cbTask = TASK_HE_GROUP_MANAGER;
	pConfigSmParams->cbVapId = ConfigurationManager_GetFirstVapForBand(ConfigurationManager_GetMyBand());
	pConfigSmParams->numberOfStages = 1;
	if (bandConfiguration == CONFIGURATION_MODE_DUAL_BAND)
	{
		pConfigSmParams->numberOfStages = 2; //Send message to each band one after the other, after one cfm send to the other band
	}
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_0].message = LINK_ADAPTATION_UPDATE_GROUPS_METRICS_REQ;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_0].task = TASK_LINK_ADAPTATION;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_0].isDualBandMessgae = FALSE;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_0].band = CONFIGURATION_MANAGER_BAND_0;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_1].message = LINK_ADAPTATION_UPDATE_GROUPS_METRICS_REQ;;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_1].task = TASK_LINK_ADAPTATION;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_1].isDualBandMessgae = FALSE;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_1].band = CONFIGURATION_MANAGER_BAND_1;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_2].message = HDK_CDB_MAN_INVALID_MSG;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_2].task = TASK_LINK_ADAPTATION;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_2].isDualBandMessgae = FALSE;
	pConfigSmParams->stageDb[CONFIG_SM_STAGE_2].band = CONFIGURATION_MANAGER_BAND_0;
	pConfigSmParams->hostMsgAddress = NULL;
	pConfigSmParams->configureVapId = ConfigurationManager_GetFirstVapForBand(CONFIGURATION_MANAGER_BAND_0); //not mandatory
	OSAL_SEND_MESSAGE(HDK_CDB_MAN_CONFIG_SM_CONFIG_REQ, TASK_HDK_CDB_MANAGER,pConfigSmMsg,pConfigSmParams->configureVapId); 
	
}


static void heGroupManagerUpdateMetricsCfm(K_MSG* pMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY, pMsg);
}


static void heGroupMangerAddSta(K_MSG* pMsg)
{
	BssManagerStaManagerReq_t*	pStationManagerRequest = NULL;
	K_MSG*						originalAddStationMessage = NULL;
	UMI_STA_ADD*				addStationMessageParameters = NULL;
	StaId						stationIndex;

	pStationManagerRequest		= (BssManagerStaManagerReq_t *)pK_MSG_DATA(pMsg);
	originalAddStationMessage	= pStationManagerRequest->psMsg;
	addStationMessageParameters = (UMI_STA_ADD *)pK_MSG_DATA(originalAddStationMessage);
	stationIndex = addStationMessageParameters->u16SID;
	heGroupManagerStationDb[stationIndex].vapId = addStationMessageParameters->u8VapIndex;
	ILOG0_V("[heGroupManagerAddStation]");
	SLOG0(0, 0, UMI_STA_ADD, addStationMessageParameters);
	heGroupManagerClassifier_AddSta(stationIndex, addStationMessageParameters);
	heGroupManagerGetStaAssocParamsFromLa(originalAddStationMessage);
}


static void heGroupManagerSetChannelReq(K_MSG* pMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL, pMsg);
}


static void heGroupManagerSetAntsReq(K_MSG* pMsg)
{
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS, pMsg);
}


static void heGroupManagerSetChannelInDisableState(K_MSG* pMsg)
{
	HeGroupManagerChannelNotification_t *params_p = (HeGroupManagerChannelNotification_t *)pK_MSG_DATA(pMsg);
	BandId_e band = ConfigurationManager_GetBandForVap(pMsg->header.vapId);

	heGroupManager_SetBwConfiguration(params_p->bw,&heGroupManagerBandParameters[band].bandLimitationBitmap);
    heGroupManagerBandParameters[band].maxBw = params_p->bw;

	/*Send confirmation to HDK task */
	OSAL_SEND_NO_DATA_MESSAGE(HDK_CHANNEL_NOTIFICATION_CFM, TASK_HDK, pMsg->header.vapId);	
}


static void heGroupManagerSetAntsInDisableState(K_MSG* pMsg)
{
	HeGroupManagerAntConfigNotification_t *params_p = (HeGroupManagerAntConfigNotification_t *)pK_MSG_DATA(pMsg);
	BandId_e band = ConfigurationManager_GetBandForVap(pMsg->header.vapId);
	
	groupsGenerator_SetAntennasConfiguration(params_p->numberOfConnectedAnts,band);

	/*Send confirmation to HDK task */
	OSAL_SEND_NO_DATA_MESSAGE(HDK_SET_ANT_CONFIG_NOTIFICATION_CFM, TASK_HDK, pMsg->header.vapId);	
}


static bool heGroupManagerSetChannel(K_MSG* pMsg)
{
	HeGroupManagerChannelNotification_t *params_p = (HeGroupManagerChannelNotification_t *)pK_MSG_DATA(pMsg);
	BandId_e band = ConfigurationManager_GetBandForVap(pMsg->header.vapId);
	bool groupRemoved = FALSE;
	
	heGroupManager_SetBwConfiguration(params_p->bw, &heGroupManagerBandParameters[band].bandLimitationBitmap);
    heGroupManagerBandParameters[band].maxBw = params_p->bw;
	groupRemoved = groupsHandler_BandConfigurationChanged(band);
	if (groupRemoved == TRUE)
	{
		heGroupManagerGlobalParameters.removalRequestor = SET_CHANNEL;
		heGroupManagerBandParameters[band].setChannelVapIdCfmPendingMsg = pMsg->header.vapId;
	}
	else
	{
		/*Send confirmation to HDK task */
		OSAL_SEND_NO_DATA_MESSAGE(HDK_CHANNEL_NOTIFICATION_CFM, TASK_HDK, pMsg->header.vapId);
	}
    return groupRemoved;
}


static void heGroupManagerSetAnts(K_MSG* pMsg)
{
	HeGroupManagerAntConfigNotification_t *params_p = (HeGroupManagerAntConfigNotification_t *)pK_MSG_DATA(pMsg);
	BandId_e band = ConfigurationManager_GetBandForVap(pMsg->header.vapId);
	
	groupsGenerator_SetAntennasConfiguration(params_p->numberOfConnectedAnts,band);
	/*Send confirmation to HDK task */
	OSAL_SEND_NO_DATA_MESSAGE(HDK_SET_ANT_CONFIG_NOTIFICATION_CFM, TASK_HDK, pMsg->header.vapId);

	
}


static void heGroupManagerSetStaAssocParamsFromLa(K_MSG* psMsg)
{
	
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA, psMsg);
}


static void heGroupManagerSetNumOfStaInGroupConfiguration(K_MSG* psMsg)
{
	
	heGroupManagerSendEventToGlobalSm(HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP, psMsg);
}


static void heGroupManagerGetStaAssocParamsFromLa(K_MSG* pAddStaMsg)
{

	K_MSG* pAddStaLaParamsMsg = OSAL_GET_MESSAGE(sizeof(HdkCdbManagerConfigBandReq_t));
	HeGroupManagerStaParamsFromLa_t* pAddStaLaParams = (HeGroupManagerStaParamsFromLa_t *)pK_MSG_DATA(pAddStaLaParamsMsg);
	pAddStaLaParams->pAddStaMsg = pAddStaMsg;
	uint8 vapId = pAddStaMsg->header.vapId;
	
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_GET_ASSOC_PARAMS_FROM_LA, TASK_LINK_ADAPTATION,pAddStaLaParamsMsg,vapId); 
}


/**********************************************************************************   
heGroupManager_SetBwConfiguration
  
Description:  Set Bw configuration in heGroupManagerGlobalParameters DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void heGroupManager_SetBwConfiguration(Bw_e maxBw, HeGroupManagerCharacteristicsBitmap_u* chararcteristicsBitmap)
{
    if (maxBw == BW_20MHZ)
    {
        chararcteristicsBitmap->bitFields.maxBw20  = 1;
        chararcteristicsBitmap->bitFields.maxBw40  = 0;
        chararcteristicsBitmap->bitFields.maxBw80  = 0;
        chararcteristicsBitmap->bitFields.maxBw160 = 0;
    }
    else if (maxBw == BW_40MHZ)
    {
        chararcteristicsBitmap->bitFields.maxBw20  = 1;
        chararcteristicsBitmap->bitFields.maxBw40  = 1;
        chararcteristicsBitmap->bitFields.maxBw80  = 0;
        chararcteristicsBitmap->bitFields.maxBw160 = 0;
    }
    else if (maxBw == BW_80MHZ)
    {
        chararcteristicsBitmap->bitFields.maxBw20  = 1;
        chararcteristicsBitmap->bitFields.maxBw40  = 1;
        chararcteristicsBitmap->bitFields.maxBw80  = 1;
        chararcteristicsBitmap->bitFields.maxBw160 = 0;
    }
    else if (maxBw == BW_160MHZ)
    {
        chararcteristicsBitmap->bitFields.maxBw20  = 1;
        chararcteristicsBitmap->bitFields.maxBw40  = 1;
        chararcteristicsBitmap->bitFields.maxBw80  = 1;
        chararcteristicsBitmap->bitFields.maxBw160 = 1;
    }
  
}


static void heGroupManagerGlobalStateDisableHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			heGroupManagerSetDynamicMuParams(pMsg);
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);				
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
			// send confirmation to host
			OSAL_SEND_MESSAGE(UM_MAN_HE_MU_OPERATION_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
			// send confirmation to host
			heGroupManagerSetDynamicMuParams(pMsg);
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
			heGroupManagerRemoveStationInDisableState(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
			heGroupManagerSetStaticPlan(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plan active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
			heGroupManagerSetChannelInDisableState(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
			heGroupManagerSetAntsInDisableState(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            groupsGenerator_SetNumOfStaInGroupConfiguration(pMsg);
            break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateIdleHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool groupRemovedMsgSent;
	
	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
			if(heGroupManagerGlobalParameters.activeGroupsBitmap != ALL_GROUPS_FREE)
			{
				heGroupManagerGlobalParameters.returnMsg = pMsg;
				heGroupManagerGlobalParameters.removalRequestor = DISABLE_PROCESS;
				groupsHandler_RemoveAllActiveGroups();
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
			}
			else
			{// no need to remove any group
				OSAL_SEND_MESSAGE(UM_MAN_HE_MU_OPERATION_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);
			}			
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
			if(heGroupManagerGlobalParameters.activeGroupsBitmap != ALL_GROUPS_FREE)
			{
				heGroupManagerGlobalParameters.returnMsg = pMsg;
				heGroupManagerGlobalParameters.removalRequestor = DISABLE_DYNAMIC_MU_PROCESS;
				groupsHandler_RemoveAllActiveGroups();
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
			}
			else
			{// No need to remove any group
				OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);
			}			
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
			heGroupManagerCollectingMetrics();	//Send message to bands for collecting metrics
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_COLLECTING_METRICS);
			/*Timer will be disable until state will be changed to idle again*/
			OSAL_RESET_TIMER_EXPLICIT(HE_GROUP_MANAGER_TIMER_EXPIRED, TASK_HE_GROUP_MANAGER);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
			heGroupManagerSetStaticPlan(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
			groupRemovedMsgSent = groupsHandler_RemoveStation(pMsg);
			if (groupRemovedMsgSent == TRUE)
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
				heGroupManagerGlobalParameters.removalRequestor = STATION_REMOVAL;
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
			groupRemovedMsgSent = heGroupManagerSetChannel(pMsg);
            if (groupRemovedMsgSent == TRUE)
            {
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
            }
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
			heGroupManagerSetAnts(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			SERIAL_TRACE("Enable Dynamic MU command ignored, fetaure is already enabled - In order to change parameter, set disable first",0,0,0);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plan active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
		/*Events to buffer*/
		/*No buffered event*/
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateCollectingMetricsHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool groupGenerationReqSent = FALSE;
	bool groupRemovedMsgSent = FALSE;
	
	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
			
			heGroupManagerClassifier_UpdateClassificationBitmapFromMetrics();
			groupRemovedMsgSent = groupsHandler_CheckGroupsEfficiency();
			if (groupRemovedMsgSent == TRUE)
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_UNDER_MAINTENENCE);
			}
			else
			{
				/*No group has been removed jump to group generation*/
				groupGenerationReqSent = groupsGenerator_GenerateAllGroups();
				if (groupGenerationReqSent == TRUE)
				{
					heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_GROUPS_IN_CREATION);
				}
				else
				{
					/*No group created*/
					heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);
				}
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
			groupRemovedMsgSent = groupsHandler_RemoveStation(pMsg);
			if (groupRemovedMsgSent == TRUE)
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IN_REMOVAL);
				heGroupManagerGlobalParameters.removalRequestor = STATION_REMOVAL;
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
			heGroupManagerSetAnts(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plan active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateUnderMaintenenceHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool allGroupsRemovedDone = FALSE;
	bool groupGenerationReqSent = FALSE;
	K_MSG*	psMsgOriginal;
	uint8 	groupId;
	
	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
			allGroupsRemovedDone = groupsHandler_PlanRemovedCfm(pMsg,&psMsgOriginal, &groupId);
			if (allGroupsRemovedDone == TRUE)
			{
				groupsHandler_PlanRemovedDoneGroupsUnderMaintenence(groupId);
				groupGenerationReqSent = groupsGenerator_GenerateAllGroups();
				if (groupGenerationReqSent == TRUE)
				{
					heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_GROUPS_IN_CREATION);
				}
				else
				{
					/*No group created*/
					heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);
				}
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plan active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateInRemovalHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool 	allGroupsRemovedDone = FALSE;
	K_MSG*	psMsgOriginal;
	uint8 	groupId;

	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
			allGroupsRemovedDone = groupsHandler_PlanRemovedCfm(pMsg,&psMsgOriginal,&groupId);
			if (allGroupsRemovedDone == TRUE)
			{
				groupsHandler_PlanRemovedDoneGroupsInRemoval(psMsgOriginal,groupId);
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is already a plan in removal!"); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
        case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
        case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
			break;
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateInCreationHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool allGroupsCreationDone = FALSE;
	
	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			allGroupsCreationDone = groupsGenerator_CreatePlanDone(pMsg);
			if (allGroupsCreationDone == TRUE)
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
	    case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			SERIAL_TRACE("Enable Dynamic MU command ignored, fetaure is already enabled - In order to change parameter, set disable first",0,0,0);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plan active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateCreatingStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{

	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			groupsGenerator_CreatePlanDone(pMsg);
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_STATIC_PLAN);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			SERIAL_TRACE("Enable Dynamic MU command ignored, fetaure is already enabled - In order to change parameter, set disable first",0,0,0);
			break;
		/*No events to ignore*/
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	bool 	groupRemovedMsgSent = FALSE;

	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ASSERT(heGroupManagerGlobalParameters.activeGroupsBitmap != ALL_GROUPS_FREE);
			groupRemovedMsgSent = groupsHandler_RemoveStaticPlan(pMsg);
            if (TRUE == groupRemovedMsgSent)
            {
			    heGroupManagerGlobalParameters.removalRequestor = REMOVE_STATIC;
			    heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN);
            }
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
			heGroupManagerGlobalParameters.returnMsg = pMsg;
			heGroupManagerGlobalParameters.removalRequestor = DISABLE_PROCESS;
			groupsHandler_RemoveAllActiveGroups();
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN);
            break;
	case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
            SERIAL_TRACE("Disable dynamic group is not valid when static plan is enable",0,0,0);
			// send confirmation to host
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
			groupRemovedMsgSent = heGroupManagerSetChannel(pMsg);
            if (groupRemovedMsgSent == TRUE)
            {
                heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN);
            }
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
			heGroupManagerSetAnts(pMsg);
			break;
		
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
			groupRemovedMsgSent = groupsHandler_RemoveStation(pMsg);
			if (groupRemovedMsgSent == TRUE)
			{
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN);
				heGroupManagerGlobalParameters.removalRequestor = STATION_REMOVAL;
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			ASSERT(heGroupManagerGlobalParameters.activeGroupsBitmap != ALL_GROUPS_FREE);
			/*Switch to dynamic*/
			heGroupManagerGlobalParameters.returnMsg = pMsg;
			groupsHandler_RemoveAllActiveGroups();
			heGroupManagerGlobalParameters.removalRequestor = CHANGE_FROM_STATIC_TO_DYNAMIC;
			heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_REMOVING_STATIC_PLAN);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		/*No events to ignore*/
		/*No events to buffer*/
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
			ASSERT(0);
	}
}


static void heGroupManagerGlobalStateRemovingStaticPlanHandler(HeGroupManagerGlobalEvent_e event, K_MSG* pMsg)
{
	K_MSG*	psMsgOriginal;
	uint8 	groupId;
    bool    allGroupsRemovedDone = FALSE;

	switch (event)
	{
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_REMOVED_DONE:
            allGroupsRemovedDone = groupsHandler_PlanRemovedCfm(pMsg,&psMsgOriginal,&groupId);
			if (allGroupsRemovedDone == TRUE)
			{
                heGroupManagerRemoveStaticPlanDoneEventHanldler(psMsgOriginal);
			}
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ADD_STA:
			heGroupMangerAddSta(pMsg);
			break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STA_ASSOC_PARAMS_FROM_LA:
			heGroupManagerClassifier_SetStaAssocParamsFromLa(pMsg);
			break;
        case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_NUM_OF_STA_IN_GROUP:
            SERIAL_TRACE("[heGroupManagerSetNumOfStaInGroup], num of stations in group configurations can be changed only when feature is disabled!",0,0,0);
            break;
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STATIC_PLAN:
			ILOG0_V("There is no static plam active "); 
			OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);		
			break;
		/*Events to ignore*/
		/*No events to ignore*/
		/*Events to buffer*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_CHANNEL:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_ANTENNAS:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_REMOVE_STA:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_SET_STATIC_PLAN:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU:
			FwUtils_EnqueueToHead(&heGroupManagerGlobalParameters.pendingQueue, pMsg);
			break;
		/*Non valid events*/
		case HE_GROUP_MANAGER_GLOBAL_EVENT_TIMER_EXPIRED:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_METRICS_READY:
		case HE_GROUP_MANAGER_GLOBAL_EVENT_PLAN_CREATION_DONE:
			ASSERT(0);
	}
}


static void heGroupManagerRemoveStaticPlanDoneEventHanldler(K_MSG*	psMsgOriginal)
{
    uint8* pSetChannelmbufferedMsgVapId;
    uint8  band;
	if (heGroupManagerGlobalParameters.removalRequestor == REMOVE_STATIC)
	{
		heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);
        OSAL_SEND_MESSAGE(UM_MAN_STATIC_PLAN_CONFIG_CFM , TASK_UM_IF_TASK, psMsgOriginal, psMsgOriginal->header.vapId);
	}
	else if(heGroupManagerGlobalParameters.removalRequestor == CHANGE_FROM_STATIC_TO_DYNAMIC)
	{
		heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_IDLE);
		OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM, TASK_UM_IF_TASK, psMsgOriginal, psMsgOriginal->header.vapId);
	}
    else if (heGroupManagerGlobalParameters.removalRequestor == SET_CHANNEL)
    {
        heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);
        /*Send confirmation to HDK task */
		/*Go over 2 bands - if set channel cfm msg != invalid it means message was buffered-> send cfm to HDK task*/
		for (band = 0; band < ConfigurationManager_GetNumOfActiveBands(); band++) 
		{
				pSetChannelmbufferedMsgVapId = &(heGroupManagerBandParameters[band].setChannelVapIdCfmPendingMsg);
				OSAL_SEND_NO_DATA_MESSAGE(HDK_CHANNEL_NOTIFICATION_CFM, TASK_HDK, *pSetChannelmbufferedMsgVapId);
				*pSetChannelmbufferedMsgVapId = INVALID_VAP; //initiate value after sending message;
		}
    }
    else if (heGroupManagerGlobalParameters.removalRequestor == DISABLE_PROCESS)
    {
		OSAL_SEND_MESSAGE(UM_MAN_HE_MU_OPERATION_CONFIG_CFM, TASK_UM_IF_TASK, psMsgOriginal, psMsgOriginal->header.vapId);
		heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);		
    }
	else if (heGroupManagerGlobalParameters.removalRequestor == STATION_REMOVAL)
    {
			if(heGroupManagerGlobalParameters.numOfGroupsInRemoval == 0)
			{
				// Send confirmation to Station Manager
				hegroupManager_SendConfirmToStationManager(heGroupManagerGlobalParameters.stationIdCurrentlyProcessed);
				heGroupManagerGlobalParameters.stationIdCurrentlyProcessed = INVALID_STA_INDEX;
				heGroupManagerChangeGlobalState(HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED);				
			}
	}
	else
	{
		ASSERT(0);
	}	
}


/**********************************************************************************
heGroupManagerRemoveStationInDisableState

Description:
------------
Function call when group manager is in disable state -> in this state we just need to send confirmation to station manager
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerRemoveStationInDisableState(K_MSG* psMsg)
{
	BssManagerStaManagerReq_t*	pStationManagerRequest = NULL;
	K_MSG*						originalRemoveStationMessage = NULL;
	UMI_STOP_TRAFFIC*			stopTrafficMessageParameters = NULL;
	StaId						stationIndex;


	pStationManagerRequest 		 = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	originalRemoveStationMessage = pStationManagerRequest->psMsg;
	stopTrafficMessageParameters = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(originalRemoveStationMessage);
    stationIndex = stopTrafficMessageParameters->u16SID;

	heGroupManagerStationDb[stationIndex].heGroupManagerStationState = HE_GROUP_MANAGER_STATION_NOT_ELIGIBLE;
	heGroupManagerClassifier_RemoveStation(stationIndex);
	hegroupManager_SendConfirmToStationManager(stationIndex);		

}
/**********************************************************************************
heGroupManagerSetDynamicMuReq


Description:
------------
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/

static void heGroupManagerSetDynamicMuReq(K_MSG* psMsg)
{
	UMI_DYNAMIC_MU_TYPE* pDynamicMuType = (UMI_DYNAMIC_MU_TYPE *)pK_MSG_DATA(psMsg);

	pDynamicMuType->Status = UMI_OK; 

	if (API_SET_OPERATION == pDynamicMuType->getSetOperation)
	{
		if (supportedDlMuType[(dlUlMuType_e)pDynamicMuType->dlMuType] && supportedUlMuType[(dlUlMuType_e)pDynamicMuType->ulMuType])
		{
			if ((DYNAMIC_MU_TYPE_NONE == (dlUlMuType_e)pDynamicMuType->dlMuType) && (DYNAMIC_MU_TYPE_NONE == (dlUlMuType_e)pDynamicMuType->ulMuType))
			{
				heGroupManagerGlobalStateMachine(HE_GROUP_MANAGER_GLOBAL_EVENT_DISABLE_DYNAMIC_MU, psMsg);
			}
			else
			{
				heGroupManagerGlobalStateMachine(HE_GROUP_MANAGER_GLOBAL_EVENT_ENABLE_DYNAMIC_MU, psMsg);
			}
		}
		else
		{
			pDynamicMuType->Status = UMI_BAD_PARAMETER;
			SERIAL_TRACE("[heGroupManagerSetDynamicMu], Dynamic group isn't enabled - MU is disable, OR dl + ul Mu Type combination is not supported ",0,0,0);
			OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM , TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
		}
	}
	else //get API
	{
		if (heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].numOfFreeDataPhases == 0)
		{
			pDynamicMuType->cdbConfig = DYNAMIC_MU_BAND1;
		}
		else if(heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].numOfFreeDataPhases == 0)
		{
			pDynamicMuType->cdbConfig = DYNAMIC_MU_BAND0;
		}
		else
		{
			pDynamicMuType->cdbConfig = DYNAMIC_MU_CDB;
		}
		pDynamicMuType->dlMuType = heGroupManagerGlobalParameters.groupManagerConfigurations.dlMuType;
		pDynamicMuType->ulMuType = heGroupManagerGlobalParameters.groupManagerConfigurations.ulMuType;
		pDynamicMuType->minStationsInGroup = heGroupManagerGlobalParameters.groupManagerConfigurations.minStationsInGroup;
		pDynamicMuType->maxStationsInGroup = heGroupManagerGlobalParameters.groupManagerConfigurations.maxStationsInGroup;
		OSAL_SEND_MESSAGE(UMI_MC_SET_DYNAMIC_MU_TYPE_CFM , TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
	}

}

/**********************************************************************************
heGroupManagerSetDynamicMuParams



Description:
------------
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void heGroupManagerSetDynamicMuParams(K_MSG* psMsg)
{
	UMI_DYNAMIC_MU_TYPE* pDynamicMuType = (UMI_DYNAMIC_MU_TYPE *)pK_MSG_DATA(psMsg);
	FormationType_e dlFormationType1;
	FormationType_e dlFormationType2;
	FormationType_e ulFormationType1;
	FormationType_e ulFormationType2;
	dlUlMuType_e dlMuType = (dlUlMuType_e)pDynamicMuType->dlMuType;
	dlUlMuType_e ulMuType = (dlUlMuType_e)pDynamicMuType->ulMuType;
	
	heMuGroupCdbMode_e cdbConfig = (heMuGroupCdbMode_e)pDynamicMuType->cdbConfig;
		
	heGroupManagerConvertDlMuType2FormationType(dlMuType, &dlFormationType1, &dlFormationType2);
	heGroupManagerConvertULMuType2FormationType(ulMuType, &ulFormationType1, &ulFormationType2);

	groupsGenerator_SetMuModeConfiguration(dlFormationType1, dlFormationType2,ulFormationType1, ulFormationType2);

	heGroupManagerGlobalParameters.groupManagerConfigurations.dlMuType = dlMuType;
	heGroupManagerGlobalParameters.groupManagerConfigurations.ulMuType = ulMuType;
	heGroupManagerGlobalParameters.groupManagerConfigurations.minStationsInGroup = pDynamicMuType->minStationsInGroup;
	heGroupManagerGlobalParameters.groupManagerConfigurations.maxStationsInGroup = pDynamicMuType->maxStationsInGroup;

	if (DYNAMIC_MU_BAND0 == cdbConfig)
	{
		/*Set only band0 -> zero band1 free data phases so no group will be generated for this band*/
		heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].numOfFreeDataPhases = 0;
	}
	else if (DYNAMIC_MU_BAND1 == cdbConfig)
	{
		/*Set only band1 -> zero band1 free data phases so no group will be generated for this band*/
		heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].numOfFreeDataPhases = 0;
	}
	else
	{
		/*CDB mode-> Set defaults values*/
		heGroupManagerSetDefaultCdbConfiguration();
	}
}
	
/**********************************************************************************
heGroupManagerConvertDlMuType2FormationType





Description:
------------
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/

static void heGroupManagerConvertDlMuType2FormationType(dlUlMuType_e dlMuType, FormationType_e* option1, FormationType_e* option2)
{
	if (dlMuType == DYNAMIC_MU_TYPE_OFDMA)
	{
		*option1 = DL_OFDMA;
		*option2 = DL_OFDMA;
	}
	else if(dlMuType == DYNAMIC_MU_TYPE_MIMO)
	{
		*option1 = DL_MIMO;
		*option2 = DL_MIMO;
	}
	else if (dlMuType == DYNAMIC_MU_TYPE_BOTH)
	{
		*option1 = DL_OFDMA;
		*option2 = DL_MIMO;
	}
	else
	{
		*option1 = INVALID_FORMATION_TYPE;
		*option2 = INVALID_FORMATION_TYPE;
	}

}
/**********************************************************************************
heGroupManagerConvertULMuType2FormationType





Description:
------------
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/

static void heGroupManagerConvertULMuType2FormationType(dlUlMuType_e ulMuType, FormationType_e* option1, FormationType_e* option2)
{
	if (ulMuType == DYNAMIC_MU_TYPE_OFDMA)
	{
		*option1 = UL_OFDMA;
		*option2 = UL_OFDMA;
	}
#ifdef ENET_INC_ARCH_WAVE600D2
	else if(ulMuType == DYNAMIC_MU_TYPE_MIMO)
	{

		*option1 = UL_MIMO;
		*option2 = UL_MIMO;
	}
#endif
	else if (ulMuType == DYNAMIC_MU_TYPE_BOTH)
	{
		*option1 = UL_OFDMA;
#ifdef ENET_INC_ARCH_WAVE600D2
		*option2 = UL_MIMO;
#else
		*option2 = UL_OFDMA;
#endif
	}
	else
	{
		*option1 = INVALID_FORMATION_TYPE;
		*option2 = INVALID_FORMATION_TYPE;
	}

}

/**********************************************************************************
heGroupManagerSetDefaultCdbConfiguration





Description:
------------
Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/

static void heGroupManagerSetDefaultCdbConfiguration()
{
		if (ConfigurationManager_GetNumOfActiveBands() == 1)
		{
			heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].numOfFreeDataPhases	= MAX_NUMBER_OF_DATA_PHASES;
		}
		else // CDB
		{
			heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].numOfFreeDataPhases		= MAX_NUMBER_OF_DATA_PHASES_PER_BAND;
			heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].numOfFreeDataPhases		= MAX_NUMBER_OF_DATA_PHASES_PER_BAND;
		}  
}
/**********************************************************************************
heGroupManager_Init

Description:
------------

Input: 
-----
			
Output:
-------	

Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManager_Init(void)
{
	StaId stationId;	

	// Init heGroupManagerGlobalParameters	
	heGroupManagerGlobalParameters.activeGroupsBitmap   		= ALL_GROUPS_FREE;
	heGroupManagerGlobalParameters.globalState          		= HE_GROUP_MANAGER_GLOBAL_STATE_DISABLED;
	heGroupManagerGlobalParameters.returnMsg					= NULL;
	heGroupManagerGlobalParameters.numOfGroupsInCreation		= 0;
	heGroupManagerGlobalParameters.numOfGroupsInRemoval			= 0;
	heGroupManagerGlobalParameters.numOfActiveStaticPlan		= 0;
	heGroupManagerGlobalParameters.stationIdCurrentlyProcessed	= INVALID_STA_INDEX;
	heGroupManagerGlobalParameters.tidCurrentlyProcessed		= 0;
    heGroupManagerGlobalParameters.tidsHeMuEnableByPlan         = 0;
	heGroupManagerGlobalParameters.removalRequestor				= NO_REMOVAL_REQUESTOR;
	heGroupManagerGlobalParameters.creationRequestor			= NO_CREATION_REQUESTOR;

    heGroupManagerGlobalParameters.groupManagerConfigurations.dlMuType                                  = DYNAMIC_MU_TYPE_NONE;
    heGroupManagerGlobalParameters.groupManagerConfigurations.ulMuType                                  = DYNAMIC_MU_TYPE_NONE;
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxGroups[CONFIGURATION_MANAGER_BAND_0]   = MAX_NUM_OF_HE_GROUPS;
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxGroups[CONFIGURATION_MANAGER_BAND_1]   = MAX_NUM_OF_HE_GROUPS;
    heGroupManagerGlobalParameters.groupManagerConfigurations.maxStationsInGroup                        = MIN_STA_IN_HE_GROUP;
    heGroupManagerGlobalParameters.groupManagerConfigurations.minStationsInGroup                        = MAX_STA_IN_HE_GROUP;

    
    heGroupManagerInitGroupPoliciesTable();
    
	FwUtils_InitQueue(&heGroupManagerGlobalParameters.pendingQueue);

	// Init heGroupManagerDb
	memset(&heGroupManagerDb, 0, sizeof(heGroupManagerDb));

	// Init heGroupManagerStationDb - Set all stations to be not eligible (HE_GROUP_MANAGER_STATION_NOT_ELIGIBLE)
	memset(&heGroupManagerStationDb, 0, sizeof(heGroupManagerStationDb));
	
	for(stationId=0; stationId<HW_NUM_OF_STATIONS; stationId++)
	{
		ASSERT(stationId < HW_NUM_OF_STATIONS)
		heGroupManagerStationDb[stationId].tidsHeMuEnablebyTSManager  = TID_BITMAP_ALL_TIDS;
		// Set Locked Plan index for DL & UL 
		TxSelector_SetULPlanIndex(stationId, HE_GROUP_MANAGER_INVALID_PLAN_INDEX);
		TxSelector_SetDLPlanIndex(stationId, HE_GROUP_MANAGER_INVALID_PLAN_INDEX);	
	}

    // Init heGroupManagerBandParameters
    memset(&(heGroupManagerBandParameters[0]), 0, sizeof(heGroupManagerBandParameters)); // KW_FIX_FW_G
    heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.val = 0xFFFFFFFF;
    heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].maxBw                    = BW_160MHZ;
    heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].numOfActiveAntennas      = FOUR_ANTS;
#ifndef ENET_INC_ARCH_WAVE600D2
    heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].bandLimitationBitmap.bitFields.ulMuMimoSupport = 0;
#endif

	heGroupManagerSetDefaultCdbConfiguration();

    if (ConfigurationManager_GetNumOfActiveBands() > 1)
    {
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.val = 0xFFFFFFFF;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].maxBw                    = BW_160MHZ;
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].numOfActiveAntennas      = FOUR_ANTS;
#ifndef ENET_INC_ARCH_WAVE600D2
        heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].bandLimitationBitmap.bitFields.ulMuMimoSupport = 0;
#endif
    }  
	
	heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_0].setChannelVapIdCfmPendingMsg = INVALID_VAP;
	heGroupManagerBandParameters[CONFIGURATION_MANAGER_BAND_1].setChannelVapIdCfmPendingMsg = INVALID_VAP;
	heGroupManager_Classifier_Init();
}
