/***********************************************************************************
 File:			HeGroupManagerClassifier.c
 Module:		He Group Manager
 Purpose: 		
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "BSSmanager_API.h"
#include "HeGroupManager_API.h"
#include "HeGroupManager.h"
#include "Utils_Api.h"
#include "OSAL_Api.h"
#include "loggerAPI.h"
#include "TxSelector_Api.h"
#include "Locker_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "queue_utility.h"
#include "mhi_umi.h"
#include "CommonPlan_Descriptors.h"
#include "ShramStationDatabase.h"
#include "PlanManager_API.h"
#include "CoC_Api.h"
#include "VapDatabase_Api.h"
#include "StaDatabase_Api.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_HE_GROUP_MANAGER
#define LOG_LOCAL_FID 3

#define HE_GROUP_MANAGER_DEBUG
#define STA_CLUSTER_SIZE HW_NUM_OF_STATIONS
/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define CLASSIFIER_CONVERT_RSSI_TO_DBM(x) (x-65)
#define MAX_RSSI_VAL 0
#define HIGH_RSSI_HIGH_BOUNDRY (HIGHER_RSSI_VAL)
#define MED_RSSI_HIGH_BOUNDRY (-60)
#define LOW_RSSI_HIGH_BOUNDRY (-70)
#define MIN_RSSI_VAL (-128)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						 Function Declaration									
/----------------------------------------------------------------------------------*/
void heGroupManagerClassifier_CreateStationsCluster(uint8 vapId, HeGroupManagerCharacteristicsBitmap_u groupPolicyChar, StaId* itemsInCluster);
StaId heGroupManagerClassifier_GetStationFromCluster(void);
StaId heGroupManagerClassifier_GetNumberOfStaInCluster(void);
void heGroupManagerClassifier_RemoveStationFromGroup(StaId stationId, uint8 groupId);
void heGroupManagerClassifier_UpdateClassificationBitmapFromMetrics(void);
void heGroupManagerClassifier_AddSta(StaId stationIndex, UMI_STA_ADD *pAddStationMessageParameters);
static void heGroupManagerClassifierPutStationInCluster(StaId staId, StaId writeIndex);
static void heGroupManagerClassifierUpdateRssiRange(HeGroupManagerCharacteristicsBitmap_u* staCharBitmap, int8 rssi);
static bool heGroupManagerClassifierCharBitmapCmpr(HeGroupManagerCharacteristicsBitmap_u staCharacteristics, HeGroupManagerCharacteristicsBitmap_u groupCharacteristics);
void heGroupManagerClassifierGroupCreationCanceled(StaId staionslistFromCanceledGroup[HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP]);
void heGroupManagerClassifier_SetStaAssocParamsFromLa(K_MSG* pMsg);
void heGroupManagerClassifier_SetAntennasConfiguration(uint8 maxNss, HeGroupManagerCharacteristicsBitmap_u* pStaChararcteristicsBitmap);
void heGroupManagerClassifier_RemoveStation(StaId stationId);
void heGroupManager_Classifier_Init(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
typedef struct HeGroupManagerClassifierStaElement_t
{
	StaId	   staIndex;
} HeGroupManagerClassifierStaElement_t;

typedef struct HeGroupManagerClassifierStaCluster_t
{
    HeGroupManagerCharacteristicsBitmap_u characteristicsBitmap;
	HeGroupManagerClassifierStaElement_t  classifierClusterElement[HW_NUM_OF_STATIONS];
	StaId readIndex;
	StaId itemsInCluster;
} HeGroupManagerClassifierStaCluster_t;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

static HeGroupManagerClassifierStaCluster_t ClassifierStationsCluster;
/*---------------------------------------------------------------------------------
/						Function Definition									
/----------------------------------------------------------------------------------*/
void heGroupManagerClassifier_CreateStationsCluster(uint8 vapId, HeGroupManagerCharacteristicsBitmap_u groupPolicyCharacteristics, StaId* itemsInCluster)
{
	StaId staId = VapDb_GetFirstConnectedStaIndex(vapId);
	StaId writeIndex = 0;

	ClassifierStationsCluster.readIndex = 0;
	ClassifierStationsCluster.itemsInCluster = 0;
	ClassifierStationsCluster.characteristicsBitmap.val = groupPolicyCharacteristics.val;

	while (staId != DB_ASYNC_SID)
	{
		if (heGroupManagerClassifierCharBitmapCmpr(heGroupManagerStationDb[staId].characteristicsBitmap, groupPolicyCharacteristics))
		{
			heGroupManagerClassifierPutStationInCluster(staId,writeIndex++);
			ClassifierStationsCluster.itemsInCluster++;
		}
		staId = StaDb_GetNextSid(staId);
	}
	*itemsInCluster =  ClassifierStationsCluster.itemsInCluster;
	ILOG0_D("heGroupManagerClassifier_CreateStationsCluster, itemsInCluster = %d", *itemsInCluster);
}


StaId heGroupManagerClassifier_GetStationFromCluster(void)
{
	StaId staIndex = INVALID_STA_INDEX;
	StaId readIndex = ClassifierStationsCluster.readIndex;

	if ((ClassifierStationsCluster.itemsInCluster != 0) && 
		(ClassifierStationsCluster.readIndex != ClassifierStationsCluster.itemsInCluster))
	{
		staIndex = ClassifierStationsCluster.classifierClusterElement[readIndex].staIndex;
		/*STA can be part of one dl group and one ul group -> Clear  candidate bit according to current cluster charateristics*/
		if(ClassifierStationsCluster.characteristicsBitmap.bitFields.dlCandidate == TRUE)
		{
			heGroupManagerStationDb[staIndex].characteristicsBitmap.bitFields.dlCandidate = FALSE;
		}
		if(ClassifierStationsCluster.characteristicsBitmap.bitFields.ulCandidate == TRUE)
		{
			heGroupManagerStationDb[staIndex].characteristicsBitmap.bitFields.ulCandidate = FALSE;
		}
		
		ClassifierStationsCluster.readIndex++;
	}
	ILOG0_DDDD("heGroupManagerClassifier_GetStationFromCluster, staIndex = %d, dlCandidate = %d, ulCandidate = %d, readIndex = %d,", 
			staIndex, heGroupManagerStationDb[staIndex].characteristicsBitmap.bitFields.dlCandidate, heGroupManagerStationDb[staIndex].characteristicsBitmap.bitFields.ulCandidate,ClassifierStationsCluster.readIndex);

	return staIndex;
}


static void heGroupManagerClassifierPutStationInCluster(StaId staId, StaId writeIndex)
{
	ASSERT(writeIndex < STA_CLUSTER_SIZE);
	ClassifierStationsCluster.classifierClusterElement[writeIndex].staIndex = staId;
}


StaId heGroupManagerClassifier_GetNumberOfStaInCluster(void)
{
	return (ClassifierStationsCluster.itemsInCluster - ClassifierStationsCluster.readIndex);
}


/**********************************************************************************
heGroupManagerClassifier_AddSta

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerClassifier_AddSta(StaId stationIndex, UMI_STA_ADD *pAddStationMessageParameters)
{
	HeGroupManagerCharacteristicsBitmap_u* pStaCharBitmap = &(heGroupManagerStationDb[stationIndex].characteristicsBitmap);
	int16 rssi = 0;	
#ifdef ENET_INC_ARCH_WAVE600D2
	HE_MAC_PHY_CAPABILITIES_INFO* heMacPhyCapabilities = (HE_MAC_PHY_CAPABILITIES_INFO*)&pAddStationMessageParameters->u8HE_Mac_Phy_Cap_Info[0];
#endif

	ILOG0_D("[heGroupManagerClassifier_AddSta], 	StaIndex = %d", stationIndex);
	heGroupManagerStationDb[stationIndex].characteristicsBitmap.val = 0; //Reset when STA connected

	// Set the station to "eligible" in StationDB if the station is HE capable
#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A when HW issue WLANVLSIIP-2881 will be fixed in B0
	if((MTLK_BFIELD_GET(pAddStationMessageParameters->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE) == TRUE) && (heGroupManagerStationAcPsTypesAreMixed(pAddStationMessageParameters->u8UAPSD_Queues) == FALSE))
#else
	if(MTLK_BFIELD_GET(pAddStationMessageParameters->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE) == TRUE)
#endif
	{
		/*Every HE STA support */
		heGroupManagerStationDb[stationIndex].heGroupManagerStationState = HE_GROUP_MANAGER_STATION_ELIGIBLE;
		heGroupManagerStationDb[stationIndex].characteristicsBitmap.bitFields.axSupport = TRUE;
		heGroupManagerStationDb[stationIndex].characteristicsBitmap.bitFields.dlCandidate = TRUE;
		heGroupManagerStationDb[stationIndex].characteristicsBitmap.bitFields.ulCandidate = TRUE;
#ifdef ENET_INC_ARCH_WAVE600D2
		/* UL MU-MIMO capability */
		heGroupManagerStationDb[stationIndex].characteristicsBitmap.bitFields.ulMuMimoSupport = heMacPhyCapabilities->hePhyCapInfo.HE_PHY_FULL_BW_UL_MU;
#endif
	}
	rssi = MAX(MIN_INT8, CLASSIFIER_CONVERT_RSSI_TO_DBM(pAddStationMessageParameters->rssi));

	heGroupManagerClassifierUpdateRssiRange(pStaCharBitmap, (int8)rssi);

	SLOG0(0, 0, UMI_STA_ADD, pAddStationMessageParameters);
}


static void heGroupManagerClassifierUpdateRssiRange(HeGroupManagerCharacteristicsBitmap_u* staCharBitmap,int8 rssi)
{
	if (rssi >= MED_RSSI_HIGH_BOUNDRY)
	{
		staCharBitmap->bitFields.highRssi = TRUE;
		staCharBitmap->bitFields.medRssi = FALSE;
		staCharBitmap->bitFields.lowRssi = FALSE;
	}
	else if(rssi >= LOW_RSSI_HIGH_BOUNDRY)
	{
		staCharBitmap->bitFields.highRssi = FALSE;
		staCharBitmap->bitFields.medRssi = TRUE;
		staCharBitmap->bitFields.lowRssi = FALSE;
	}
	else if(rssi >= MIN_RSSI_VAL)
	{
		staCharBitmap->bitFields.highRssi = FALSE;
		staCharBitmap->bitFields.medRssi = FALSE;
		staCharBitmap->bitFields.lowRssi = TRUE;
	}
}

/***********************************************************************
* heGroupManagerGetStationState
* 
* Description:
* ------------
* Get station state
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
HeGroupManagerStationState_e heGroupManagerGetStationState(StaId stationIndex)
{
	return(heGroupManagerStationDb[stationIndex].heGroupManagerStationState);
}

/**********************************************************************************
heGroupManagerClassifier_SetStaAssocParamsFromLa

Description:
------------

Input: 
-----
			
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
void heGroupManagerClassifier_SetStaAssocParamsFromLa(K_MSG* pMsg)
{
	HeGroupManagerStaParamsFromLa_t* pStaParams = (HeGroupManagerStaParamsFromLa_t *)pK_MSG_DATA(pMsg);
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pStaParams->pAddStaMsg);
	StaId staId = pAddSta->u16SID;
	HeGroupManagerCharacteristicsBitmap_u* pStaChararcteristicsBitmap = &heGroupManagerStationDb[staId].characteristicsBitmap;

	/*set STA BW*/
#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_DD("[heGroupManagerClassifier_SetStaAssocParamsFromLa], station: %d, bw: %d", staId, pStaParams->staBw);
#endif
	heGroupManager_SetBwConfiguration(pStaParams->staBw, pStaChararcteristicsBitmap);
	pStaChararcteristicsBitmap->bitFields.OfdmaInBw20M = pStaParams->ofdmaInBW20;

	/*set STA MaxNss*/
	heGroupManagerClassifier_SetAntennasConfiguration(pStaParams->staMaxNss,pStaChararcteristicsBitmap);
	/*Send confirmation to station manager*/
	hegroupManager_SendConfirmToStationManager(staId);
	SLOG0(0, 0, HeGroupManager_StationDbEntry_t, &heGroupManagerStationDb[staId]);

}


static bool heGroupManagerClassifierCharBitmapCmpr(HeGroupManagerCharacteristicsBitmap_u staCharacteristics, HeGroupManagerCharacteristicsBitmap_u groupCharacteristics)
{
	return ((staCharacteristics.val & groupCharacteristics.val) == groupCharacteristics.val);
}


void heGroupManagerClassifier_RemoveStationFromGroup(StaId stationId, uint8 groupId)
{
	HeGroupManager_GroupDbEntry_t* pGroupDb = &heGroupManagerDb[groupId];

	/*Set STA candidation according to dissasambled group*/
	heGroupManagerStationDb[stationId].characteristicsBitmap.bitFields.dlCandidate = pGroupDb->groupCharacteristicsBitmap.bitFields.dlCandidate;
	heGroupManagerStationDb[stationId].characteristicsBitmap.bitFields.ulCandidate = pGroupDb->groupCharacteristicsBitmap.bitFields.ulCandidate;
	
	Utils_ZeroBitInBitmap(&heGroupManagerStationDb[stationId].groupsBitmap, groupId);
	heGroupManagerStationDb[stationId].tidsHeMuEnablebyHeGM = NONE_TIDS;
	heGroupManagerStationDb[stationId].heGroupManagerStationState = HE_GROUP_MANAGER_STATION_ELIGIBLE;
}


void heGroupManagerClassifierGroupCreationCanceled(StaId staionslistFromCanceledGroup[HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP])
{
	/*Return stations to be candidates for group after group was removed*/
	StaId calceledStaIndex;
	StaId staId;
	
	for (calceledStaIndex = 0; calceledStaIndex < HE_GROUP_MANAGER_MAX_NUMBER_OF_STATIONS_IN_GROUP; calceledStaIndex++) 
	{
		staId = staionslistFromCanceledGroup[calceledStaIndex];
		/*Set STA candidation current cluster */
		heGroupManagerStationDb[staId].characteristicsBitmap.bitFields.dlCandidate = ClassifierStationsCluster.characteristicsBitmap.bitFields.dlCandidate;
		heGroupManagerStationDb[staId].characteristicsBitmap.bitFields.ulCandidate = ClassifierStationsCluster.characteristicsBitmap.bitFields.ulCandidate;
	}
}


void heGroupManagerClassifier_UpdateClassificationBitmapFromMetrics(void)
{
	
}


/**********************************************************************************   
heGroupManagerClassifier_SetAntennasConfiguration
  
Description:  Set antennas configuration in heGroupManagerGlobalParameters DB
------------

Input: 
-----        
            
Output:
-------       
    
Returns:
--------
    void - 
**********************************************************************************/
void heGroupManagerClassifier_SetAntennasConfiguration(uint8 maxNss, HeGroupManagerCharacteristicsBitmap_u* pStaChararcteristicsBitmap)
{
    if (maxNss == 1)
    {
        pStaChararcteristicsBitmap->bitFields.maxNss1 = 1;
        pStaChararcteristicsBitmap->bitFields.maxNss2 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss3 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss4 = 0;
    }
    else if (maxNss == 2)
    {
        pStaChararcteristicsBitmap->bitFields.maxNss1 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss2 = 1;
        pStaChararcteristicsBitmap->bitFields.maxNss3 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss4 = 0;    
    }
    else if (maxNss == 3)
    {
        pStaChararcteristicsBitmap->bitFields.maxNss1 = 0;
       	pStaChararcteristicsBitmap->bitFields.maxNss2 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss3 = 1;
        pStaChararcteristicsBitmap->bitFields.maxNss4 = 0; 
    }
    else if (maxNss == 4)
    {
       	pStaChararcteristicsBitmap->bitFields.maxNss1 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss2 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss3 = 0;
        pStaChararcteristicsBitmap->bitFields.maxNss4 = 1;
    }

}

void heGroupManagerClassifier_RemoveStation(StaId stationId)
{

	/*Clear ax support when sta disconnect so it will not chosen to next group*/
	heGroupManagerStationDb[stationId].characteristicsBitmap.bitFields.axSupport = FALSE;
	
}

void heGroupManager_Classifier_Init(void)
{
	memset(&ClassifierStationsCluster, 0, sizeof(HeGroupManagerClassifierStaCluster_t));
}


