
/***************************************************************************
****************************************************************************
**
** COMPONENT:        HIM
**
** MODULE:           $File: //bwp/enet/demo153_sw/develop/src/him_cardbus/him.c $
**
** VERSION:          $Revision: #8 $
**
** DATED:            $Date: 2004/03/10 $
**
** AUTHOR:           Laurence Evans
**
** DESCRIPTION:      Host Interface Manager
**
** CHANGE HISTORY:
**
** LAST MODIFIED BY:   $Author: prh $
**
****************************************************************************
*
* Copyright (c) 2003 TTPCom Limited
*
****************************************************************************/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "System_Information.h"
#include "ErrorHandler_Api.h"
#include "OSAL_Api.h"
#include "HostInterface_API.h"
#include "HostInterface.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "HostIfRegs.h"
#endif
#include "stringLibApi.h"
#include "TxPacketsClassifier_API.h"
#include "TxMulticastHandler_API.h"
#include "HostInterfaceAcc_Api.h"
#include "Debug_Trace.h"
#include "CalibrationManager.h"
#include "BSSmanager_API.h"
#include "lm_interface.h"
#include "umi.h"
#include "Pac_Api.h"
#ifdef ENET_INC_ARCH_WAVE600
#include "MacHtExtensionsRegs.h"
#include "MacGeneralRegs.h"
#include "ConfigurationManager_api.h"
#else
#include "MT_MAC_HT_extensions_regs.h"
#include "MacRabRegs.h"
#endif
#include "RegAccess_Api.h"
#include "int_gen.h"
#include "shram_logger_buffers.h"
#include "enet_pas.h"
#include "enet_hrc.h"
#include "shram_dataHostIf.h"
#include "CpuLoad_Api.h"
#include "EventsManager_api.h"
#include "ShramPacketDescriptors.h"
#include "HwQManager_API.h"
#include "ShramRxDescriptors.h"
#include "TxMulticastHandler_API.h"
#include "HwMemoryMap.h"
#include "loggerDef.h"
#include "um_interface.h"
#include "StatisticsManager_api.h"
#ifdef CBM_FRAGMENTATION_WORKAROUND
#include "ShramHdAddrRing.h"
#endif //CBM_FRAGMENTATION_WORKAROUND
#include "HostIfAccRegs.h"
#include "Atf_Api.h"
#include "VapDatabase_Api.h"
#include "HdkCdbManagerTask_api.h"
#include "um_interface.h"

#if defined (ENET_INC_UMAC)

/******************************************/
/*        DO NOT REMOVE THIS LINE!        */
/******************************************/
#include "loggerAPI.h"
#define LOG_LOCAL_GID   GLOBAL_GID_HOST_IF
#define LOG_LOCAL_FID 0


/***************************************************************************/
/***                       Types and Defines                             ***/
/***************************************************************************/
#ifndef TX_DATA_UM_PATH
#define HOST_INTERFACE_FIRST_READY_LIST	HW_Q_MANAGER_READY_LIST_UNICAST_PD
#define HOST_INTERFACE_LAST_READY_LIST	HW_Q_MANAGER_READY_LIST_MULTICAST_PD
#define HOST_INTERFACE_NUM_READY_LIST   (HOST_INTERFACE_LAST_READY_LIST - HOST_INTERFACE_FIRST_READY_LIST + 1)
#endif

typedef struct HostInterfaceGlobalDb_s
{
	uint32 reliableMulticastBitmap; // bit per VAP  (TRUE = reliable multicast)
	uint32 stopVapTrafficBitmap;	// bit per VAP (TRUE = VAP is in STOP TRAFFIC)	
	uint16 hostInterfaceStaNumNonQoS;
	uint16 hostInterfaceNumAtfStas;
#if defined (ENET_INC_ARCH_WAVE600)	
    uint8  atf_AgeingTtlCriteria[NUM_OF_CONFIGURATION_MANAGER_BANDS];
  	uint8  atf_AgeingEnable[NUM_OF_CONFIGURATION_MANAGER_BANDS];
  	uint8  atf_End_of_interval_ttl[NUM_OF_CONFIGURATION_MANAGER_BANDS];
#else
	uint8  atf_AgeingTtlCriteria;
	uint8  atf_AgeingEnable;
	uint8  atf_End_of_interval_ttl;
#endif
	uint8  restrictedAcModeEnable; // options: {FALSE: disabled, TRUE: enabled}
	uint8  restrictedAcBitmap;	   // bitmap per AC to be given priority when restricted mode is enabled: bit 0-BE; bit 1-BK; bit 2-VI; bit 3-VO
	uint16 restrictedAcThreshEnter;
	uint16 restrictedAcThreshExit; 
} HostInterfaceGlobalDb_t;


typedef struct HostInterfaceRestrictedAcDb_s
{
	uint8 enable;					// options: {FALSE: disabled, TRUE: enabled}
} HostInterfaceRestrictedAcDb_t;

typedef struct HostInterfaceVapDb_s
{
	uint8 enabled;
} HostInterfaceVapDb_t;

typedef struct HostInterfaceCounters_s
{
	uint32 host2fwManReqCnt;
	uint32 host2fwDbgReqCnt;
	uint32 host2fwManCfmCnt;
	uint32 host2fwDbgCfmCnt;
} HostInterfaceCounters_t;

#ifndef TX_DATA_UM_PATH
typedef void (*HostInterfaceListHandler)(TxPd_t *headPd, TxPd_t *tailPd);
#endif

//------------ Debug HIM code ---vvv--------


/*
 * This message is sent during initialisation from the HIM to itself to
 * start the second stage of the initialisation process. There is no reply.
 */



/***************************************************************************/
/***                         Private Data                                ***/
/***************************************************************************/
static const uint8 protectedManagementFrames[MGMT_FRAME_SUBTYPE_LAST] = {
	0,	// MGMT_FRAME_SUBTYPE_ASSOCIATION_REQ
	0,	// MGMT_FRAME_SUBTYPE_ASSOCIATION_RES
	0,	// MGMT_FRAME_SUBTYPE_REASSOCIATION_REQ
	0,	// MGMT_FRAME_SUBTYPE_REASSOCIATION_RES
	0,	// MGMT_FRAME_SUBTYPE_PROBE_REQ
	0,	// MGMT_FRAME_SUBTYPE_PROBE_RES
	0,	// MGMT_FRAME_SUBTYPE_TIMING_ADVERTISEMENT
	0,	// MGMT_FRAME_SUBTYPE_RESERVED1
	0,	// MGMT_FRAME_SUBTYPE_BEACON
	0,	// MGMT_FRAME_SUBTYPE_ATIM
	1,	// MGMT_FRAME_SUBTYPE_DISASSOCIATION
	0,	// MGMT_FRAME_SUBTYPE_AUTHENTICATION
	1,	// MGMT_FRAME_SUBTYPE_DEAUTHENTICATION
	0,	// MGMT_FRAME_SUBTYPE_ACTION - Action Need to be handled separetely
	0,  // MGMT_FRAME_SUBTYPE_ACTION_NO_ACK
	0,	// MGMT_FRAME_SUBTYPE_RESERVED2
};

static uint8 protectedActionFrames[NUM_ACTION_CATEGORY];


static HostInterfaceGlobalDb_t HostInterface_GlobalDb;
static HostInterfaceRestrictedAcDb_t HostInterfaceRestrictedAcDb;
static HostInterfaceVapDb_t	HostInterfaceVapDb[HW_NUM_OF_VAPS];

static HostInterfaceCounters_t HostInterfaceCounters;
	
/***************************************************************************/
/***                  Private Function Prototypes                        ***/
/***************************************************************************/
extern SHRAM_IND_MSG asMAN_IndData[NUM_MAN_INDS]; 
extern BUFFER_DAT_IND_MSG_DESC ShramLoggerVectorAreaDescriptor[NUM_OF_LOGGER_BUFFER_DESCRIPTORS];
extern  uint8 shram_LoggerRegularBuffers[LOGGER_NUM_OF_BUFFERS][LOGGER_BUFFER_SIZE];
/* Helper Functions. */

static void HIM_PutMessageOnHostFifo(uint32 *fifoDescriptor);
static void HostInterface_StaManagerSendConfirm(StaId sid);
static void hostInterface_StaAdd(K_MSG *psMsg);
static void hostInterface_StaSetFilter(K_MSG *psMsg);
static void hostInterface_StaStopTraffic(K_MSG *psMsg);
static void hostInterface_StaRemove(K_MSG *psMsg);
static void hostInterface_VapSetBss(K_MSG* psMsg);
static void hostInterface_SetWmm(K_MSG* psMsg);
static void hostInterface_AddVap(K_MSG* psMsg); 
static void hostInterface_StopVapTraffic(K_MSG* psMsg);
static void hostInterface_SetVapMulticastMode(K_MSG* psMsg);
static void hostInterfaceRingsSetAtfAgeing(K_MSG *psMsg);
static void hostInterfaceRingsSetAtfBufferEnable(K_MSG *psMsg);
static void HostInterface_RestrictedAcMsg(K_MSG *psMsg);


/* current indexes in reqres and cfmind q */
static uint32 u32ReqResIdx;
static uint32 u32CfmIndIdx;

#ifndef TX_DATA_UM_PATH
static void hostInterfaceUnicastListHandler(TxPd_t *headPd, TxPd_t *tailPd);
static void hostInterfaceMulticastListHandler(TxPd_t *headPd, TxPd_t *tailPd);
#endif

#if defined (ENET_INC_ARCH_WAVE600)	
bool hostInterface_atfCheckAgeingCondition(uint8 ttlCount, uint8 bandId);
#else
bool hostInterface_atfCheckAgeingCondition(uint8 ttlCount);
#endif




static const FunctionEntry_t afpTaskTable[TASK_HIM_END - TASK_HIM_START]=
{
	{vHIM_ForwardMcMsgs,							DOUBLE_CHECK_MSG_TYPE(HIM_MC_DATA_INT)},
    {hostInterfaceRingsNewHdsIndication,            DOUBLE_CHECK_MSG_TYPE(HIM_NEW_HDS_IN_HOST_RING)},
    {hostInterfaceRings_DoneListsHandler,           DOUBLE_CHECK_MSG_TYPE(HIM_DESCRIPTORS_ON_DONE_LISTS)},
    {hostInterfaceRings_HandleDmaComplete,          DOUBLE_CHECK_MSG_TYPE(HIM_RINGS_DMA_DONE)},
#ifndef TX_DATA_UM_PATH
	{hostInterface_ReadyListsHandler,				DOUBLE_CHECK_MSG_TYPE(HIM_DESCRIPTORS_ON_READY_LISTS)},
#endif
	{hostInterface_StaAdd,          				DOUBLE_CHECK_MSG_TYPE(HIM_STA_ADD)},
	{hostInterface_StaSetFilter,					DOUBLE_CHECK_MSG_TYPE(HIM_STA_SET_FILTER)},
	{hostInterface_StaStopTraffic,					DOUBLE_CHECK_MSG_TYPE(HIM_STA_STOP_TRAFFIC)},
	{hostInterface_StaRemove,						DOUBLE_CHECK_MSG_TYPE(HIM_STA_REMOVE)},
	{hostInterface_VapSetBss,						DOUBLE_CHECK_MSG_TYPE(HIM_VAP_SET_BSS)},	
	{hostInterface_SetWmm,							DOUBLE_CHECK_MSG_TYPE(HIM_SET_WMM_PARAMS)},
	{hostInterface_AddVap,							DOUBLE_CHECK_MSG_TYPE(HIM_POST_VAP_ACTIVATION)},
	{hostInterface_StopVapTraffic,					DOUBLE_CHECK_MSG_TYPE(HIM_VAP_STOP_TRAFFIC)},
	{hostInterface_SetVapMulticastMode,				DOUBLE_CHECK_MSG_TYPE(HIM_SET_MULTICAST_MODE)},
    {hostInterfaceRingsSetAtfAgeing,		    	DOUBLE_CHECK_MSG_TYPE(HIM_RINGS_ATF_AGEING)},
    {hostInterfaceRingsSetAtfBufferEnable,			DOUBLE_CHECK_MSG_TYPE(HIM_RINGS_ATF_BUFFER_STATION)},
    {HostInterface_RestrictedAcMsg,					DOUBLE_CHECK_MSG_TYPE(HIM_SET_RESTRICTED_AC_MODE)},
};


#ifndef TX_DATA_UM_PATH
static const HostInterfaceListHandler HostInterfaceListsHandlers[HOST_INTERFACE_NUM_READY_LIST] =
{
	hostInterfaceUnicastListHandler,  	  /* HW_Q_MANAGER_READY_LIST_UNICAST_PD */
	hostInterfaceMulticastListHandler,   /* HW_Q_MANAGER_READY_LIST_MULTICAST_PD */	
};	
#endif

/***************************************************************************/
/***                         Public Functions                            ***/
/***************************************************************************/


/**********************************************************************************
hostInterfaceRingsSetAtfEnable  

Description:
------------
        Msg, set the atf enable for specific station in static mode,
        indicating the pds need to pushed to buffer list and not to queues list
Input:
-----
    stationID
Output:
------- 
    Cfm msg
Returns:
--------

**********************************************************************************/
static void hostInterfaceRingsSetAtfAgeing(K_MSG *psMsg)
{
    AtfAgeingMsg_t *AtfAgeingHim = (AtfAgeingMsg_t *)pK_MSG_DATA(psMsg);


#if defined (ENET_INC_ARCH_WAVE600)
	uint8 bandId = ConfigurationManager_GetBandForVap(psMsg->header.vapId);

	HostInterface_GlobalDb.atf_AgeingEnable[bandId] 		= AtfAgeingHim->AgeingEnable;
	HostInterface_GlobalDb.atf_AgeingTtlCriteria[bandId]	= AtfAgeingHim->AgeingTtlCriteria;
	HostInterface_GlobalDb.atf_End_of_interval_ttl[bandId] 	= AtfAgeingHim->end_of_interval_ttl;
#else
    HostInterface_GlobalDb.atf_AgeingEnable 				= AtfAgeingHim->AgeingEnable;
    HostInterface_GlobalDb.atf_AgeingTtlCriteria			= AtfAgeingHim->AgeingTtlCriteria;
    HostInterface_GlobalDb.atf_End_of_interval_ttl 			= AtfAgeingHim->end_of_interval_ttl;
#endif
}

/**********************************************************************************
hostInterfaceRingsSetAtfEnable  

Description:
------------
        Msg, set the atf enable for specific station in static mode,
        indicating the pds need to pushed to buffer list and not to queues list
Input:
-----
    stationID
Output:
------- 
    Cfm msg
Returns:
--------

**********************************************************************************/
static void hostInterfaceRingsSetAtfBufferEnable(K_MSG *psMsg)
{
    AtfMsg_t *AtfHim = (AtfMsg_t *)pK_MSG_DATA(psMsg);
    
#ifdef ATF_DEBUG_LOCAL    
   	ILOG0_D("[hostInterfaceRingsSetAtfBufferEnable] stationId:%d", AtfHim->stationId);
#endif

	if (hostInterface_IsStaQoS(AtfHim->stationId)== TRUE)
	{
		/* open route packets to UMAC */
		HostInterfaceStaDb[AtfHim->stationId].pdDist = PD_DIST_UMAC;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_UMAC ,AtfHim->stationId);
	}

	// Count number of ATF STAs which are handled in host IF.
	HostInterface_GlobalDb.hostInterfaceNumAtfStas++;


    HostInterfaceStaDb[AtfHim->stationId].atf_buffered = TRUE;
  	OSAL_SEND_MESSAGE(ATF_BUFFERING_HIM_BUFFER_CFM, TASK_ATF, psMsg, VAP_ID_DO_NOT_CARE);
    
}


/**********************************************************************************
hostInterfaceRingsSetAtfEnable  

Description:
------------
        Function call, set the atf disable for specific station in static mode,
        indicating the pds need to pushed to queues list and not to buffer list
Input:
-----

Output:
------- 

Returns:
--------

**********************************************************************************/
void hostInterfaceRingsSetAtfBufferDisable(StaId stationId)
{
#ifdef ATF_DEBUG_LOCAL    
	ILOG0_V("[hostInterfaceRingsSetAtfBufferDisable]");
#endif
	if (hostInterface_IsStaQoS(stationId)== TRUE)
	{
		/* open route packets to TXQ */
		HostInterfaceStaDb[stationId].pdDist = PD_DIST_TXQ;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_TXQ ,stationId);
	}


	// Count number of ATF STAs which are handled in host IF.
	HostInterface_GlobalDb.hostInterfaceNumAtfStas--;

    HostInterfaceStaDb[stationId].atf_buffered = FALSE;
}


/********************************************************************************
HostInterface_RestrictedAcMsg


Description:
------------
	Msg from Driver
	
Input:
-----

Output:
-------

Returns:
--------

********************************************************************************/
void HostInterface_RestrictedAcMsg(K_MSG *psMsg)
{
	K_MSG *agerMsg;
	UMI_SET_RESTRICTED_AC* umiSetRestrictedAc = NULL;
	UMI_SET_RESTRICTED_AC* pSetREstrictedAc = (UMI_SET_RESTRICTED_AC*)pK_MSG_DATA(psMsg);	

	if (pSetREstrictedAc->getSetOperation == API_GET_OPERATION)
	{
		pSetREstrictedAc->restrictedAcModeEnable = HostInterface_GlobalDb.restrictedAcModeEnable;
		pSetREstrictedAc->restrictedAcThreshEnter = HostInterface_GlobalDb.restrictedAcThreshEnter;
		pSetREstrictedAc->restrictedAcThreshExit = HostInterface_GlobalDb.restrictedAcThreshExit;
		pSetREstrictedAc->acRestrictedBitmap = HostInterface_GlobalDb.restrictedAcBitmap;
	}
	else
	{
		HostInterface_GlobalDb.restrictedAcModeEnable = pSetREstrictedAc->restrictedAcModeEnable;
		HostInterface_GlobalDb.restrictedAcBitmap = pSetREstrictedAc->acRestrictedBitmap;
		HostInterface_GlobalDb.restrictedAcThreshEnter = pSetREstrictedAc->restrictedAcThreshEnter;
		HostInterface_GlobalDb.restrictedAcThreshExit = pSetREstrictedAc->restrictedAcThreshExit;
		
		// if RestrictedAC mode is enabled than bitMap must be set
		ASSERT((HostInterface_GlobalDb.restrictedAcModeEnable == FALSE)||(HostInterface_GlobalDb.restrictedAcBitmap != 0))
		// restrictedAcThreshExit > restrictedAcThreshEnter
		ASSERT(HostInterface_GlobalDb.restrictedAcThreshExit > HostInterface_GlobalDb.restrictedAcThreshEnter)
		
		/* Use MAX_UINT16 to disable the usage of restricted AC by HOST I/F TASK, but still send it QoS TASK */
		if (HostInterface_GlobalDb.restrictedAcThreshExit == NUM_OF_TX_DESC)
		{
			HostInterface_GlobalDb.restrictedAcModeEnable = FALSE;
			ILOG0_V("HostInterface_RestrictedAcMsg: User indicates that Restricted AC should not be used by HOST I/F TASK. QoS TASK shall use it.");
		}
		
		/* Inform AGER that Restricted AC parameters have changed */
		agerMsg = OSAL_GET_MESSAGE(sizeof(UMI_SET_RESTRICTED_AC));
		umiSetRestrictedAc = (UMI_SET_RESTRICTED_AC*) pK_MSG_DATA(agerMsg);
		umiSetRestrictedAc->restrictedAcModeEnable = pSetREstrictedAc->restrictedAcModeEnable;
		umiSetRestrictedAc->acRestrictedBitmap = pSetREstrictedAc->acRestrictedBitmap; 
		umiSetRestrictedAc->restrictedAcThreshEnter = pSetREstrictedAc->restrictedAcThreshEnter;
		umiSetRestrictedAc->restrictedAcThreshExit = pSetREstrictedAc->restrictedAcThreshExit;
		OSAL_SEND_MESSAGE(QOS_AGER_SET_RESTRICTED_PARAMS, TASK_QOS, agerMsg, psMsg->header.vapId);
	}
	
	// Send confirmation to host
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_RESTRICTED_AC_MODE_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
}

/********************************************************************************
HostInterface_IsRestrictedAc


Description:
------------
This is API for the data  (unicast & multicast) packets to check whether given AC 
will be given a higher priority over other ACs given PD thresholds

Input:
-----
	tid

Output:
-------

Returns:
--------
	bool param allowAllocation indicating if to allow PD allocation to queue or not
********************************************************************************/
bool HostInterface_IsRestrictedAc(uint8 tid)
{
	uint16 numberOfFreePdsNeeded;
	uint16 currentNumberOfFreePds;
	uint16 numberOfFreePdsNeededToExitRestriction;
	uint8  acIndex;
	bool   allowAllocation = TRUE; 

	//if system restricted mode is disabled always allow PD allocation to queue otherwise verify:
	if (HostInterface_GlobalDb.restrictedAcModeEnable == TRUE)
	{	
		//convert tid to AC
		acIndex = au8Convert8021dToWmeUserPriority[tid];
		acIndex = (1<<acIndex);

		//check number of free PDs 
		numberOfFreePdsNeeded = HostInterface_GlobalDb.restrictedAcThreshEnter;
		currentNumberOfFreePds = HostIfAcc_GetCurrentFreePdsNumber();
		numberOfFreePdsNeededToExitRestriction = HostInterface_GlobalDb.restrictedAcThreshExit;
	//	ILOG0_DDDD("[HostInterface_IsRestrictedAc], HostInterface_GlobalDb.restrictedAcModeEnable = %d, HostInterface_GlobalDb.restrictedAcBitmap = %d, HostInterface_GlobalDb.restrictedAcThreshEnter = %d, HostInterface_GlobalDb.restrictedAcThreshExit = %d", HostInterface_GlobalDb.restrictedAcModeEnable, HostInterface_GlobalDb.restrictedAcBitmap, HostInterface_GlobalDb.restrictedAcThreshEnter, HostInterface_GlobalDb.restrictedAcThreshExit);
		//ILOG0_D("[HostInterface_IsRestrictedAc], HostInterfaceRestrictedAcDb.enable = %d", HostInterfaceRestrictedAcDb.enable);
		//ILOG0_DDD("[HostInterface_IsRestrictedAc], numberOfFreePdsNeeded = %d, currentNumberOfFreePds = %d, numberOfFreePdsNeededToExitRestriction = %d", numberOfFreePdsNeeded, currentNumberOfFreePds, numberOfFreePdsNeededToExitRestriction);
		//ILOG0_DD("[HostInterface_IsRestrictedAc], tid = %d, acIndex = %d", tid, acIndex);

		// system defined to allow restricted AC mode need to check current mode
		if(HostInterfaceRestrictedAcDb.enable == FALSE)
		{	
			//system is not in restricted state check amount of PDs
			if(currentNumberOfFreePds < numberOfFreePdsNeeded)
			{	
				//system is not in restricted state and the amount of available PDs does not suffice - change state to restricted and check AC
				HostInterfaceRestrictedAcDb.enable = TRUE;
				//check if AC is given priority
				if (HostInterface_GlobalDb.restrictedAcBitmap & acIndex)
				{
					//AC is prioritzied therefore allow PD allocation to queue
					allowAllocation = TRUE;
				}
				else
				{
					//AC is not prioritzied therefore do not allow PD allocation to queue
					allowAllocation = FALSE;
				}
			}
			else
			{
				//system is not in restricted state and the amount of available PDs suffices - allow allocation of PD to queue
				allowAllocation = TRUE;
			}
		}
		else
		{
			//system was in restricted AC state check if should stay restricted
			if(currentNumberOfFreePds > numberOfFreePdsNeededToExitRestriction)
			{
				//amount of available PDs is larger than hytstersis threhold therefore can exit restricted state
				HostInterfaceRestrictedAcDb.enable = FALSE;
				allowAllocation = TRUE;
			}
			else
			{
				//stay restricted, check if AC is given priority
				if (HostInterface_GlobalDb.restrictedAcBitmap & acIndex)
				{
					//AC is prioritzied therefore allow PD allocation to queue
					allowAllocation = TRUE;
				}
				else
				{
					//AC is not prioritzied therefore do not allow PD allocation to queue
					allowAllocation = FALSE;
				}			
			}	
		 }
	}
	
//	ILOG0_D("[HostInterface_IsRestrictedAc], HostInterfaceRestrictedAcDb.enable = %d", HostInterfaceRestrictedAcDb.enable);
//	ILOG0_D("[HostInterface_IsRestrictedAc], allowAllocation = %d", allowAllocation);
	return(allowAllocation);
}


/***************************************************************************
**
** NAME         vHIM_Init
**
** PARAMETERS   na
**
** RETURNS      na
**
** DESCRIPTION  This function should be called once during initialisation to
**              initialise the Host Interface Manager.
**
****************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void vHIM_Init(void)
{
#if defined (ENET_INC_ARCH_WAVE600)
#ifdef ENET_INC_ARCH_WAVE600D2
	RegHostIfNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#else
	RegMacGeneralNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#endif
#else // wave500
	RegMacRabNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#endif

#ifdef ATOMIC_COUNTER_ENABLER
	EventManager_TurnOffEvent(EVENT_ID_MAILBOX_NOT_EMPTY);
#else
	EventManager_TurnOffEvent(EVENT_ID_MC_DATA_INT);
#endif

     /* clear interrupts */
	npu2UpiInterruptClearReg.bitFields.npu2UpiInterruptClear = 0x1 << RAB_HOST_INT_INDEX;
#if defined (ENET_INC_ARCH_WAVE600)
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_HOST_IF_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);  
#else
	RegAccess_Write(REG_MAC_GENERAL_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);  
#endif
#else // wave500
	RegAccess_Write(REG_MAC_RAB_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);	 
#endif

	/* enable interrupts */
	InterruptManager_EnableNpuRabInt(RAB_HOST_INT_INDEX);

	memset(&HostInterfaceCounters,0, sizeof(HostInterfaceCounters_t));

}

/***************************************************************************
**
** NAME         vHIM_Enable
**
** PARAMETERS   na
**
** RETURNS      na
**
** DESCRIPTION  Enables the HIM interface
**
****************************************************************************/
void vHIM_Enable(void)
{
	int32 iLoop;
	
    /* init mips debug interface */
    sCHIvectorArea.sMipsControl.sHeader.u32ExtensionMagic    = HOST_EXTENSION_MAGIC;
    sCHIvectorArea.sMipsControl.sHeader.u32ExtensionID       = VECTOR_AREA_MIPS_CONTROL_EXTENSION_ID;
    sCHIvectorArea.sMipsControl.sHeader.u32ExtensionDataSize = sizeof(sCHIvectorArea.sMipsControl.sData);
    sCHIvectorArea.sMipsControl.sData.u32DescriptorLocation  = ErrorHandler_GetCtrlStructure() - PAS_BASE_ADDRESS; 

    /* Ready to receive data */
#ifdef ATOMIC_COUNTER_ENABLER
	EventManager_TurnOnEvent(EVENT_ID_MAILBOX_NOT_EMPTY);
#else	
	EventManager_TurnOnEvent(EVENT_ID_MC_DATA_INT);
#endif

    /* clear FIFOs indexes */
    u32CfmIndIdx = u32ReqResIdx = 0;
  
    /* clear FIFOs queues */
	for (iLoop = 0; iLoop < HIM_CIRCULAR_BUFFER_SIZE; iLoop++)
	{
		au32FIFOReqResQ[iLoop] = 0;
		au32FIFOCfmIndQ[iLoop] = 0;
	}
	ASSERT(CALIBR_HOST_BUFFER_SIZE < 128000)//if larger cant malloc in driver

    /* init CHI area vector */
    sCHIvectorArea.sBasic.sFifoQ.u32IndNumOfElements = HIM_CIRCULAR_BUFFER_SIZE;
    sCHIvectorArea.sBasic.sFifoQ.u32IndStartOffset = (uint32)au32FIFOCfmIndQ - PAS_BASE_ADDRESS;
    sCHIvectorArea.sBasic.sFifoQ.u32ReqNumOfElements = HIM_CIRCULAR_BUFFER_SIZE;
    sCHIvectorArea.sBasic.sFifoQ.u32ReqStartOffset = (uint32)au32FIFOReqResQ - PAS_BASE_ADDRESS;

    sCHIvectorArea.sBasic.sDAT.u32IndNumOfElements = NUM_DAT_INDS;
    sCHIvectorArea.sBasic.sDAT.u32IndStartOffset = (uint32)asHIMdatIndDesc - PAS_BASE_ADDRESS;
	sCHIvectorArea.sBasic.sDAT.u32ReqNumOfElements = 0;
	sCHIvectorArea.sBasic.sDAT.u32ReqStartOffset = 0;


	HostInterfaceRings_FillChiHostRingsInfo(&sCHIvectorArea.sBasic);

    sCHIvectorArea.sBasic.sMAN.u32IndNumOfElements = NUM_MAN_INDS;
    sCHIvectorArea.sBasic.sMAN.u32IndStartOffset = (uint32)asMAN_IndData - PAS_BASE_ADDRESS;
    sCHIvectorArea.sBasic.sMAN.u32ReqNumOfElements = NUM_MAN_REQS;
    sCHIvectorArea.sBasic.sMAN.u32ReqStartOffset = (uint32)asHIMmanReq - PAS_BASE_ADDRESS;
    
    sCHIvectorArea.sBasic.sDBG.u32IndNumOfElements = NUM_DBG_INDS;
    sCHIvectorArea.sBasic.sDBG.u32IndStartOffset = (uint32)asHIMdbgInd - PAS_BASE_ADDRESS;
    sCHIvectorArea.sBasic.sDBG.u32ReqNumOfElements = NUM_DBG_REQS;
    sCHIvectorArea.sBasic.sDBG.u32ReqStartOffset = (uint32)asHIMdbgReq - PAS_BASE_ADDRESS;

 	/*this section is D/C for driver - there was plan in past to use this mehanism for the management ring, but now the management ring works in different way not connected to this ind/req logic at all
 	It will be totally removed in future with sync with the driver team	*/
    sCHIvectorArea.sBasic.sBSS.u32IndNumOfElements = NUM_BSS_INDS;
    sCHIvectorArea.sBasic.sBSS.u32IndStartOffset = (uint32)asHIMBssIndDesc - PAS_BASE_ADDRESS;
    sCHIvectorArea.sBasic.sBSS.u32ReqNumOfElements = NUM_BSS_REQS;
    sCHIvectorArea.sBasic.sBSS.u32ReqStartOffset = (uint32)asHIMBssReq - PAS_BASE_ADDRESS;

    sCHIvectorArea.sCalibr.sHeader.u32ExtensionMagic    = HOST_EXTENSION_MAGIC;
    sCHIvectorArea.sCalibr.sHeader.u32ExtensionID       = VECTOR_AREA_CALIBR_EXTENSION_ID;
    sCHIvectorArea.sCalibr.sHeader.u32ExtensionDataSize = sizeof(sCHIvectorArea.sCalibr.sData);
    sCHIvectorArea.sCalibr.sData.u32BufferRequestedSize = CALIBR_HOST_BUFFER_SIZE; 
#ifdef ENET_INC_ARCH_WAVE600D2
	sCHIvectorArea.sCalibr.sData.pPsdProgmodelAddress = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)p32PsdProgmodelShramAddress) - SHARED_RAM_GRAY_SIZE; 
#else
	sCHIvectorArea.sCalibr.sData.pPsdProgmodelAddress = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)p32PsdProgmodelShramAddress); 
#endif


	

    sCHIvectorArea.sLoggerExt.sHeader.u32ExtensionMagic            = HOST_EXTENSION_MAGIC;
    sCHIvectorArea.sLoggerExt.sHeader.u32ExtensionID               = VECTOR_AREA_LOGGER_EXTENSION_ID;
    sCHIvectorArea.sLoggerExt.sHeader.u32ExtensionDataSize         = sizeof(sCHIvectorArea.sLoggerExt.sData);
    sCHIvectorArea.sLoggerExt.sData.u32BufferDescriptorsLocation   = (uint32)&ShramLoggerVectorAreaDescriptor - PAS_BASE_ADDRESS;
    sCHIvectorArea.sLoggerExt.sData.u32NumOfBufferDescriptors      = NUM_OF_LOGGER_BUFFER_DESCRIPTORS;
    
	sCHIvectorArea.sFwCapabilitiesNumOfStationsExt.sHeader.u32ExtensionMagic     = HOST_EXTENSION_MAGIC;
	sCHIvectorArea.sFwCapabilitiesNumOfStationsExt.sHeader.u32ExtensionID        = VECTOR_AREA_FW_CAPABILITIES_NUM_OF_STATIONS_EXTENSION_ID;
	sCHIvectorArea.sFwCapabilitiesNumOfStationsExt.sHeader.u32ExtensionDataSize	= sizeof(sCHIvectorArea.sFwCapabilitiesNumOfStationsExt.sData);
	sCHIvectorArea.sFwCapabilitiesNumOfStationsExt.sData.u32NumOfStations		= HW_NUM_OF_STATIONS;

	sCHIvectorArea.sFwCapabilitiesNumOfVapsExt.sHeader.u32ExtensionMagic     = HOST_EXTENSION_MAGIC;
	sCHIvectorArea.sFwCapabilitiesNumOfVapsExt.sHeader.u32ExtensionID        = VECTOR_AREA_FW_CAPABILITIES_NUM_OF_VAPS_EXTENSION_ID;
	sCHIvectorArea.sFwCapabilitiesNumOfVapsExt.sHeader.u32ExtensionDataSize	= sizeof(sCHIvectorArea.sFwCapabilitiesNumOfVapsExt.sData);
	sCHIvectorArea.sFwCapabilitiesNumOfVapsExt.sData.u32NumOfVaps			= HW_NUM_OF_VAPS;

    sCHIvectorArea.sLoggerBufInfoExt.sHeader.u32ExtensionMagic      = HOST_EXTENSION_MAGIC;
    sCHIvectorArea.sLoggerBufInfoExt.sHeader.u32ExtensionID         = VECTOR_AREA_LOGGER_BUF_INFO_EXTENSION_ID;
    sCHIvectorArea.sLoggerBufInfoExt.sHeader.u32ExtensionDataSize   = sizeof(sCHIvectorArea.sLoggerBufInfoExt.sData);
    
    sCHIvectorArea.sLoggerBufInfoExt.sData.lmBufferInfo.u32Size                  = LOGGER_LM_BUFFER_SIZE;
	sCHIvectorArea.sLoggerBufInfoExt.sData.lmBufferInfo.u32NumberOfBuffers		 = LOGGER_NUM_OF_BUFFERS;
    sCHIvectorArea.sLoggerBufInfoExt.sData.lmBufferInfo.u32BufferPoolAddress     = LoggerAPIGetPoolStart((void *)shram_LmLoggerRegularBuffers, LOGGER_NUM_OF_BUFFERS) - PAS_BASE_ADDRESS;

	sCHIvectorArea.sLoggerBufInfoExt.sData.umBufferInfo.u32Size                  = LOGGER_UM_BUFFER_SIZE;
    sCHIvectorArea.sLoggerBufInfoExt.sData.umBufferInfo.u32NumberOfBuffers       = LOGGER_NUM_OF_BUFFERS;
    sCHIvectorArea.sLoggerBufInfoExt.sData.umBufferInfo.u32BufferPoolAddress     = (uint32)shram_LoggerRegularBuffers - PAS_BASE_ADDRESS;

    // RD logger
	//this extension was removed entirely on Master
	sCHIvectorArea.sLoggerRdExt.sHeader.u32ExtensionMagic       			 = HOST_EXTENSION_MAGIC;
    sCHIvectorArea.sLoggerRdExt.sHeader.u32ExtensionID          			 = VECTOR_AREA_LOGGER_RD_EXTENSION_ID;
    sCHIvectorArea.sLoggerRdExt.sHeader.u32ExtensionDataSize    			 = sizeof(sCHIvectorArea.sLoggerRdExt.sData);    
    sCHIvectorArea.sLoggerRdExt.sData.u16LoggerRdExtensionFirstRdIndex       = 0xDEAD; //the field is don't care in ER7, and was removed entirely on Master
    sCHIvectorArea.sLoggerRdExt.sData.u16LoggerRdExtensionNumOfRDs           = 0xDEAD; //the field is don't care in ER7, and was removed entirely on Master

#ifdef CBM_FRAGMENTATION_WORKAROUND
	//CBM W/A fragmentation buffer address
	sCHIvectorArea.sHdFragmentationRingExt.sHeader.u32ExtensionMagic 			 = HOST_EXTENSION_MAGIC;
	sCHIvectorArea.sHdFragmentationRingExt.sHeader.u32ExtensionID				 = VECTOR_AREA_FRAGMENTATION_RING_EXTENSION_ID;
	sCHIvectorArea.sHdFragmentationRingExt.sHeader.u32ExtensionDataSize			 = sizeof(sCHIvectorArea.sHdFragmentationRingExt.sData);
	sCHIvectorArea.sHdFragmentationRingExt.sData.u32FragmentationRingAddress	 = CONVERT_WLAN_SHRAM_ADDR_TO_OFFSET_FROM_BAR0((uint32)HdAddrRing);
	sCHIvectorArea.sHdFragmentationRingExt.sData.u32FragmentationRingNumOfEntries = HD_ADDR_RING_NUM_OF_ENTRIES;
#endif //CBM_FRAGMENTATION_WORKAROUND


	sCHIvectorArea.sDebugExt.sHeader.u32ExtensionMagic   	 = HOST_EXTENSION_MAGIC;
	sCHIvectorArea.sDebugExt.sHeader.u32ExtensionID      	 = VECTOR_AREA_VECTOR_AREA_DEBUG_EXTENSION_ID;
	sCHIvectorArea.sDebugExt.sHeader.u32ExtensionDataSize	 = sizeof(sCHIvectorArea.sDebugExt.sData);

	/* clear all req pools */
	memset(asHIMmanReq, 0, sizeof(asHIMmanReq));
	memset(asHIMdbgReq, 0, sizeof(asHIMdbgReq));
	memset(asHIMBssReq, 0, sizeof(asHIMBssReq));

}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
/***************************************************************************
**
** NAME         vHIM_Task
**
** PARAMETERS   psKnlMsg      A pointer to the message to be processed by
**                            the HIM task.
**
** RETURNS      na
**
** DESCRIPTION  This function will be called by the Kernel each time that a
**              message is sent to the HIM task.
**
****************************************************************************/
void vHIM_Task(K_MSG *psKnlMsg)
{
	vTaskDispatcher(psKnlMsg, afpTaskTable, TASK_HIM_START, TASK_HIM_END);
}


/***************************************************************************
**
** NAME         vHIM_ForwardMcMsgs
**
** PARAMETERS   psKnlMsg      A pointer to the message to be processed by
**                            the HIM task.
**
** RETURNS      na
**
** DESCRIPTION  This function will be called by the Kernel each time that a
**              message is sent to the HIM task.
**
****************************************************************************/
void vHIM_ForwardMcMsgs( K_MSG *psMsg)
{
	uint32 fifoDescriptor;
	uint8 descType;
	uint16 descIndex;
    uint8 messages_count;	
	uint8 vapIndex;
#ifdef ATOMIC_COUNTER_ENABLER	
	uint32 numOfPendingMessages;
#endif
	TX_INTERRUPT_SAVE_AREA;
	UNUSED_PARAM(psMsg);	

   /*
    * Check the Request/Response message queue. Process all messages in the
    * queue before completing. Note that we take a copy of the head index at
    * the start of the loop because we need to manipulate it. We write it back
    * at the end of each loop to free space in the circular buffer.
    */

#ifdef ATOMIC_COUNTER_ENABLER
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Read(REG_HOST_IF_HOST_MAC_MAILBOX_ACCUM_COUNT, &numOfPendingMessages);
#else
	RegAccess_Read(REG_HOST_IF_ACC_HOST_MAC_MAILBOX_ACCUM_COUNT, &numOfPendingMessages);
#endif
	DEBUG_ASSERT(numOfPendingMessages > 0); // We must have at least 1 driver's message to handle
	for(messages_count = 0; messages_count < numOfPendingMessages; messages_count++)
	{
#else
    for ( messages_count=0; ;messages_count++ ) // loop forever !!!
    {
#endif

        K_MSG *psKnlMsg = PNULL;
		K_MSG *psKnlMsgTemp = PNULL;
        /* We need to lock against getting a new RX interrupt during this stage */
        
        OSAL_DISABLE_INTERRUPTS(&interrupt_save);
        fifoDescriptor = au32FIFOReqResQ[u32ReqResIdx];
		descType = MTLK_BFIELD_GET(fifoDescriptor, HIM_DESC_TYPE);
		descIndex = MTLK_BFIELD_GET(fifoDescriptor, HIM_DESC_INDEX);
		vapIndex = MTLK_BFIELD_GET(fifoDescriptor, HIM_DESC_VAP);

		if (descType == MSG_TYPE_NULL)
        {
            /* Allow ISR task to send HIM interrupt indication */

#ifdef ATOMIC_COUNTER_ENABLER
			EventManager_TurnOnEvent(EVENT_ID_MAILBOX_NOT_EMPTY);
#else				
			EventManager_TurnOnEvent(EVENT_ID_MC_DATA_INT);
#endif
            OSAL_ENABLE_INTERRUPTS(interrupt_save);
            break;
        }

        au32FIFOReqResQ[u32ReqResIdx] = 0;
        OSAL_ENABLE_INTERRUPTS(interrupt_save);
        
        /* get message from appropriate pool according to sDesc.u8Type */
        switch(descType)
        {
        case MSG_TYPE_DAT_IND:
		case MSG_TYPE_BSS_IND:
		case MSG_TYPE_BSS_REQ:
		case MSG_TYPE_DAT_LOGGER_IND:
			DEBUG_ASSERT(0);
			break;
        case MSG_TYPE_MAN_IND:
            psKnlMsg = (K_MSG *)((uint32)asMAN_IndData + descIndex * sizeof(SHRAM_IND_MSG));
            /* init message as persistent */
            OSAL_INIT_PERSISTENT_MSG(psKnlMsg, sizeof(UMI_IND));
            /* change type of msg from ind  to res*/
			MSG_MAKE_REPLY(psKnlMsg->header.tKMsgType);
            break;
        case MSG_TYPE_MAN_REQ:
			//WLANRTSYS-11712 - As part of secured host message implementation, when driver post a message it is copied
			//to B1 Tx Circular buffer before validation and processing. This is part of secured
			//Host message implementation. B1 Tx Circular buffer is a region which host cannot access.
			HostInterfaceCounters.host2fwManReqCnt++;
			psKnlMsg = (K_MSG *)((uint32)asHIMmanReq + descIndex * sizeof(SHRAM_MAN_MSG));
			psKnlMsgTemp = (K_MSG *)((uint32)asHIMmanReqSecured + descIndex * sizeof(SHRAM_MAN_MSG));
			MEMCPY((void *)psKnlMsgTemp,(void const *)psKnlMsg, sizeof(SHRAM_MAN_MSG));
			psKnlMsg = psKnlMsgTemp;
			/* init message as persistent */
			OSAL_INIT_PERSISTENT_MSG(psKnlMsg, sizeof(UMI_MAN));
            break;
        case MSG_TYPE_DBG_IND:
            psKnlMsg = (K_MSG *)((uint32)asHIMdbgInd + descIndex * sizeof(SHRAM_DBG_MSG));
			
            /* init message as persistent */
            OSAL_INIT_PERSISTENT_MSG(psKnlMsg, sizeof(UMI_DBG));
            /* change type of msg from ind  to res*/
			MSG_MAKE_REPLY(psKnlMsg->header.tKMsgType);
            break;
        case MSG_TYPE_DBG_REQ:
			HostInterfaceCounters.host2fwDbgReqCnt++;
			psKnlMsg = (K_MSG *)((uint32)asHIMdbgReq + descIndex * sizeof(SHRAM_DBG_MSG));
			psKnlMsgTemp = (K_MSG *)((uint32)asHIMdbgReqSecured + descIndex * sizeof(SHRAM_DBG_MSG));
			MEMCPY((void *)psKnlMsgTemp,(void const *)psKnlMsg, sizeof(SHRAM_DBG_MSG));
			psKnlMsg = psKnlMsgTemp;
			/* init message as persistent */
			OSAL_INIT_PERSISTENT_MSG(psKnlMsg, sizeof(UMI_DBG));
            break;
        
        default:
            FATAL("Error buffer descriptor in FIFO!");
        }


        /* Increment Request/Response index and check for wrap. */
        u32ReqResIdx++;
        if (u32ReqResIdx == HIM_CIRCULAR_BUFFER_SIZE)
		{	
			u32ReqResIdx = 0;
		}

		if (psKnlMsg != NULL)
		{
			uint16 u16Status;
			UNUSED_PARAM(u16Status);
			/*
			* Forward the message to the MAC to handle. Note that we do not check
			* the validity of the message ID- this will be tested by the MAC which
			* can then take whatever action it deems appropriate.
			*/
#if defined (ENET_INC_ARCH_WAVE600)
			psKnlMsg->header.vapId = ConfigurationManager_ConvertVapAndRadioToActualVap(MTLK_BFIELD_GET(fifoDescriptor, HIM_DESC_RADIO), vapIndex);  
#else
			psKnlMsg->header.vapId = vapIndex; 
#endif
			psKnlMsg->header.sFrom.taskID = MSG_FROM_DRIVER;
			psKnlMsg->header.sTo.taskID = MSG_TO_FW;
			u16Status = u16UMIsendMsg((UMI_MSG *)psKnlMsg);
			DEBUG_ASSERT (u16Status == UMI_OK);
		}
    }
#ifdef ATOMIC_COUNTER_ENABLER
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_HOST_IF_HOST_MAC_MAILBOX_ACCUM_SUB, numOfPendingMessages);
#else
	RegAccess_Write(REG_HOST_IF_ACC_HOST_MAC_MAILBOX_ACCUM_SUB, numOfPendingMessages);
#endif
	EventManager_TurnOnEvent(EVENT_ID_MAILBOX_NOT_EMPTY);
#endif
}

/***************************************************************************
**
** NAME         vHIM_HostISR
**
** PARAMETERS   na
**
** RETURNS      na
**
** DESCRIPTION  This function will be called each time that the Host
**              generates an interrupt to the Upper MAC CPU. The ISR will
**              empty the Host to HIM message queue and send each message to
**              the HIM task for processing.
**
****************************************************************************/
void vHIM_HostISR(void)
{

#if defined (ENET_INC_ARCH_WAVE600)
#ifdef ENET_INC_ARCH_WAVE600D2
		RegHostIfNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#else
		RegMacGeneralNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#endif
#else // wave500
		RegMacRabNpu2UpiInterruptClear_u npu2UpiInterruptClearReg;
#endif
	
	ACCUMULATE_CPU_IDLE_TIME();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vHIM_HostISR,0,0);
    
	npu2UpiInterruptClearReg.bitFields.npu2UpiInterruptClear = 0x1 << RAB_HOST_INT_INDEX;

#if defined (ENET_INC_ARCH_WAVE600)
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_HOST_IF_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);
#else
	RegAccess_Write(REG_MAC_GENERAL_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);
#endif
#else 
	RegAccess_Write(REG_MAC_RAB_NPU2UPI_INTERRUPT_CLEAR, npu2UpiInterruptClearReg.val);
#endif


#ifdef ATOMIC_COUNTER_ENABLER
	EventManager_TurnOffEvent(EVENT_ID_MAILBOX_NOT_EMPTY);
#else
	EventManager_TurnOffEvent(EVENT_ID_MC_DATA_INT);
#endif

	OSAL_SEND_NO_DATA_MESSAGE(HIM_MC_DATA_INT, TASK_HIM, VAP_ID_DO_NOT_CARE);

	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vHIM_HostISR,1,1);
		
}


/***************************************************************************/
/***                        Private Functions                            ***/
/***************************************************************************/
static void HIM_PutMessageOnHostFifo(uint32 *fifoDescriptor)
{

    // calculate new index 
    uint32 uNewIndex = u32CfmIndIdx + 1;

    if (uNewIndex == HIM_CIRCULAR_BUFFER_SIZE) 
    {
        uNewIndex = 0;
    }

    /* Insert the message desc into the circular queue at the current position */
    
    /* clear next pos in buffer */
    au32FIFOCfmIndQ[uNewIndex] = 0;

    /* write message desc */
  
    au32FIFOCfmIndQ[u32CfmIndIdx] = *fifoDescriptor;
    
    /* update current index */
    u32CfmIndIdx = uNewIndex;
}

static void HostInterface_StaManagerSendConfirm(StaId sid)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_HIM_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = sid;
	/*Send confirmation message*/
	ILOG2_DD("Ts Manager, Send Confirmation, Client %d, SID %d", confirmMessage->clientId, sid);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

#ifndef TX_DATA_UM_PATH
static void hostInterfaceUnicastListHandler(TxPd_t *headPd, TxPd_t *tailPd)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
#if defined (ENET_INC_ARCH_WAVE600)	
	uint8 bandId;
#endif
	// KW IGNORE CWARN.MEM.NONPOD there is no NONPOD in C
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	/*If all STAs are QoS and don't participate in ATF - just send to liberator*/
	if ((HostInterface_GlobalDb.hostInterfaceStaNumNonQoS == 0) && (HostInterface_GlobalDb.hostInterfaceNumAtfStas == 0))
	{
		/*Send to Liberator*/
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		hwQueueManagerRequestParams.pHeadDesc = headPd;
		hwQueueManagerRequestParams.pTailDesc = tailPd;
#ifdef ENET_INC_ARCH_WAVE600
		bandId = ConfigurationManager_GetBandForVap(headPd->txQVapId);
		GeneralStatistics.dropReasonClassifier[bandId] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
		GeneralStatistics.dropReasonClassifier += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
	}
	else
	{
		TxPd_t *nextPacketDescriptor = NULL;
		TxPd_t *currentPacketDescriptor = NULL;
		TxPd_t *headDropList = NULL;
		TxPd_t *tailDropList = NULL;
		StaId sid;
		uint8 isQosStation;
		
		currentPacketDescriptor = headPd;
		while (currentPacketDescriptor != (TxPd_t *)NULL_PD)
		{	
			nextPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
			sid = currentPacketDescriptor->txQStaId;
			isQosStation = MTLK_BFIELD_GET(HostInterfaceStaDb[sid].flags, STA_ADD_FLAGS_WME);
 			/* If STA QoS or Disabled or not part of ATF */

			if ((HostInterfaceStaDb[sid].open == FALSE) || ((isQosStation == TRUE) && (HostInterfaceStaDb[sid].atf_buffered == FALSE)))
			{
				/*Add to Drop List*/
				if (headDropList == NULL)
				{
					headDropList = currentPacketDescriptor;
				}
				else
				{
					tailDropList->nextPdPointer = SET_NEXT_PD(currentPacketDescriptor);
				}
				tailDropList = currentPacketDescriptor;
			}
			else
			{

				if (isQosStation == FALSE)
				{
#ifdef ENET_INC_ARCH_WAVE600 
					// There length of this packet was increased in host-if counter per STA/TID. Since we are about to change the TID, we need to change the counters accordingly.
					hostInterfaceDecrementPdByteCount(currentPacketDescriptor->txQTid, currentPacketDescriptor->txQStaId, currentPacketDescriptor->dataLength);
					hostInterfaceIncrementPdByteCount(IEEE802_1D_BE_1, currentPacketDescriptor->txQStaId, currentPacketDescriptor->dataLength);					
#endif		
					/*Override TID*/
					currentPacketDescriptor->txQTid = IEEE802_1D_BE_1;
				}

				if (HostInterfaceStaDb[sid].atf_buffered == FALSE)
				{
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
					hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
					hwQueueManagerRequestParams.primaryAddr = sid;
					hwQueueManagerRequestParams.secondaryAddr = currentPacketDescriptor->txQTid;
					hwQueueManagerRequestParams.pHeadDesc = currentPacketDescriptor;

					HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);				  
				}
				else
				{
					// check if PD will be aged. since gap of time is higher than criteria
#if defined (ENET_INC_ARCH_WAVE600)	
					bandId = ConfigurationManager_GetBandForStation(sid);

					if ( hostInterface_atfCheckAgeingCondition(currentPacketDescriptor->ttlCount, bandId))
#else
					if ( hostInterface_atfCheckAgeingCondition(currentPacketDescriptor->ttlCount))
#endif
					{
						/*Send to Liberator*/
						hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
						hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
						hwQueueManagerRequestParams.pHeadDesc = currentPacketDescriptor;
						currentPacketDescriptor->status = PD_STATUS_NACK;
#if defined (ENET_INC_ARCH_WAVE600)
						GeneralStatistics.dropReasonClassifier[bandId]++;
#else
						GeneralStatistics.dropReasonClassifier++;
#endif //ENET_INC_ARCH_WAVE600
						/*Send to Liberator*/
						HwQManager_PushPacketToTail(&hwQueueManagerRequestParams); 
					} 
					else
					{
						/*Send to Atf Buffer list*/
						Atf_Buffering_PushPacketToTail(currentPacketDescriptor); 
					}
				}
			}
		
			currentPacketDescriptor = nextPacketDescriptor;
		}
		/*Handle Drop List*/
		if (headDropList != NULL)
		{
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			hwQueueManagerRequestParams.pHeadDesc = headDropList;
			hwQueueManagerRequestParams.pTailDesc = tailDropList;
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
#if defined (ENET_INC_ARCH_WAVE600)
			bandId = ConfigurationManager_GetBandForVap(headDropList->txQVapId);
			GeneralStatistics.dropReasonClassifier[bandId] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
			GeneralStatistics.dropReasonClassifier += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600
			/*Send to Liberator*/
			HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
		}
	}
}

static void hostInterfaceMulticastListHandler(TxPd_t *headPd, TxPd_t *tailPd)
{
	TxPd_t* pDesc = (TxPd_t *)headPd;
	TxPd_t* pNextPd;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;	
	UNUSED_PARAM(tailPd);

	// KW IGNORE CWARN.MEM.NONPOD there is no NONPOD in C
	memset(&hwQueueManagerRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));		
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE; 		
	while(pDesc != (TxPd_t *)NULL_PD)
	{
		pNextPd = (TxPd_t *)GET_NEXT_PD(pDesc);
		pDesc->nextPdPointer = NEXT_PD_NULL;
		if (HostInterfaceVapDb[pDesc->txQVapId].enabled == TRUE)
		{
			MulticastHandler_NewMulticastPacket(pDesc);
		}
		else
		{
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
			hwQueueManagerRequestParams.pHeadDesc = pDesc;
			/*Send to Liberator*/
			HwQManager_PushPacketToTail(&hwQueueManagerRequestParams); 
		}
		pDesc = pNextPd;
	}
}

void hostInterface_ReadyListsHandler(K_MSG *psMsg)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint16 emptyDoneList = 0;
	uint16 NotEmptyDoneList = 0;
	uint8 listIndex = HOST_INTERFACE_FIRST_READY_LIST;
	uint8 handlerIndex = 0;
	UNUSED_PARAM(psMsg);
	// KW IGNORE CWARN.MEM.NONPOD there is no NONPOD in C
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	emptyDoneList = HwQManager_GetEmptyTxReadyHiPriLists();
	ASSERT(!emptyDoneList);

	NotEmptyDoneList = HwQManager_GetNotEmptyTxReadyHiPriLists();
	DEBUG_ASSERT(NotEmptyDoneList);

	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

	while(NotEmptyDoneList)
	{
	    if(HW_Q_MANAGER_LIST_BIT_IS_SET_MASK & NotEmptyDoneList)
	    {
			HwQManager_ClearTxListsInt(listIndex);
	    	hwQueueManagerRequestParams.dplIndex = listIndex;
	    	HwQManager_FlushQ(&hwQueueManagerRequestParams);
			ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
			HostInterfaceListsHandlers[handlerIndex]((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc,(TxPd_t *)hwQueueManagerRequestParams.pTailDesc);
	    }
		NotEmptyDoneList >>= HW_Q_MANAGER_SHIFT_TO_NEXT_LIST;
		listIndex++;
		handlerIndex++;
	}
	EventManager_TurnOnEvent(EVENT_ID_TX_LIST_READY_PDS_HIGH_PRI_NOT_EMPTY);
}
#endif

static void hostInterface_StaAdd(K_MSG *psMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pStaAdd = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	StaId sid;
	uint8 isXFilterOpen;
	uint8 isQosStation;

	sid = pStaAdd->u16SID;
	isXFilterOpen = MTLK_BFIELD_GET(pStaAdd->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);
	isQosStation = MTLK_BFIELD_GET(pStaAdd->u8Flags, STA_ADD_FLAGS_WME);
    /* clear buffering */
    HostInterfaceStaDb[sid].atf_buffered = FALSE;
	/*Store flags for later use*/
	HostInterfaceStaDb[sid].flags = pStaAdd->u8Flags;
	/*Mark STA as enabled*/
	HostInterfaceStaDb[sid].enabled = TRUE;
	/*Clear filter */
	HostInterfaceStaDb[sid].open = FALSE;
	/*Clear use PMF flag*/
	HostInterfaceStaDb[sid].use_pmf = FALSE;
	/*Check if filter is open*/
	if (isXFilterOpen == TRUE)
	{
		/*Mark filter as open*/
		HostInterfaceStaDb[sid].open = TRUE;
	}
	/*if STA is non-QoS increment number of non-QoS Stations*/
	if (isQosStation == FALSE)
	{
		SERIAL_TRACE("hostInterface_StaAdd - non qos sta",0,0,0);
		HostInterface_GlobalDb.hostInterfaceStaNumNonQoS++;
		/*open route packets to TXQ*/
		HostInterfaceStaDb[sid].pdDist = PD_DIST_UMAC;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_UMAC,sid);
	}
	/*if STA is QoS and filter is open route packets to TXQ now, else wait for set filter*/
	else if (isXFilterOpen == TRUE)
	{
		/*open route packets to TXQ*/
		HostInterfaceStaDb[sid].pdDist = PD_DIST_TXQ;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_TXQ ,sid);
	}
	/*Send confirmation to STA Manager*/
	HostInterface_StaManagerSendConfirm(sid);
}

static void hostInterface_StaSetFilter(K_MSG *psMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_802_1X_FILTER* p8021xFilter = (UMI_802_1X_FILTER*)pK_MSG_DATA(pMsg);
	StaId sid = p8021xFilter->u16SID;
	uint8 isQosStation = MTLK_BFIELD_GET(HostInterfaceStaDb[sid].flags, STA_ADD_FLAGS_WME);
	uint8 isPmf = MTLK_BFIELD_GET(HostInterfaceStaDb[sid].flags, STA_ADD_FLAGS_MFP);

	/*Mark filter as opened*/
	HostInterfaceStaDb[sid].open = TRUE;
	if (isPmf == TRUE)
	{
		/*If STA is PMF, set use PMF flag*/
		HostInterfaceStaDb[sid].use_pmf = TRUE;
	}
	/*if STA is QoS route packets to TXQ now*/
	if (isQosStation == TRUE)
	{
		/*open route packets to TXQ*/
		HostInterfaceStaDb[sid].pdDist = PD_DIST_TXQ;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_TXQ ,sid);
	}
	/*Send confirmation to STA Manager*/
	HostInterface_StaManagerSendConfirm(sid);
}

static void hostInterface_StaStopTraffic(K_MSG *psMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(pMsg);
	StaId sid = pStopTraffic->u16SID;
	uint8 isQosStation = MTLK_BFIELD_GET(HostInterfaceStaDb[sid].flags, STA_ADD_FLAGS_WME);

	/*Mark filter as closed*/
	HostInterfaceStaDb[sid].open = FALSE;
	if (isQosStation == FALSE)
	{
		/*If non-QoS decrement number of non-QoS Stations*/
		HostInterface_GlobalDb.hostInterfaceStaNumNonQoS--;
	}
	else if (HostInterfaceStaDb[sid].pdDist == PD_DIST_TXQ)
	{
		/*route back to UMAC*/
		HostInterfaceStaDb[sid].pdDist = PD_DIST_UMAC;
		HostIfAcc_ChangeStaAllTidPdDist(PD_DIST_UMAC,sid);
	}
	/*Send confirmation to STA Manager*/
	HostInterface_StaManagerSendConfirm(sid);
}

static void hostInterface_StaRemove(K_MSG *psMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_REMOVE *pStaRemove = (UMI_STA_REMOVE *)pK_MSG_DATA(pMsg);
	StaId sid = pStaRemove->u16SID;

	/*Mark STA as disabled*/
	HostInterfaceStaDb[sid].enabled = FALSE;
	HostInterface_StaManagerSendConfirm(sid);
}

static void hostInterface_VapSetBss(K_MSG* psMsg) 
{
	UMI_SET_BSS* 	pSetBss;
	uint8 			vapId; 
	bool 			isReliableMulticast;
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	pSetBss = (UMI_SET_BSS*) EXTRACT_VAP_MANAGER_MSG(psMsg);
	vapId = pSetBss->vapId;
	isReliableMulticast = MTLK_BFIELD_GET(pSetBss->flags, VAP_ADD_FLAGS_RELIABLE_MCAST);

	// Clear and Save reliable multicast flag for later (will be used in post vap activation)
	HostInterface_GlobalDb.reliableMulticastBitmap &= ~(0x1 << vapId);
	HostInterface_GlobalDb.reliableMulticastBitmap |= (isReliableMulticast << vapId);
	
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_SET_BSS, BSS_MANAGER_VAP_MANAGER_HIM_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
}

static void hostInterface_SetWmm(K_MSG* psMsg) 
{
	UMI_SET_WMM_PARAMETERS* pSetWmmMsg; 
	uint8 					vapId;
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	pSetWmmMsg = (UMI_SET_WMM_PARAMETERS*) EXTRACT_VAP_MANAGER_MSG(psMsg);
	vapId = pSetWmmMsg->vapId;

	if (HostInterfaceVapDb[vapId].enabled == TRUE)
	{
		/*If this is a STA VAP*/
		if (VapDb_GetVapMode(vapId) == VAP_MODE_STA)
		{
			StaId sid;
			/*Get First STA ID*/
		
			sid = VapDb_GetFirstConnectedStaIndex(vapId);
			/*In STA VAP there can be only one STA ...*/
			if (sid != DB_ASYNC_SID)
			{
				/*Get ACM from SET WMM parameters*/
				HostInterfaceStaDb[sid].acm_flags = (pSetWmmMsg->wmmParamsPerAc[0].acm_flag) 	|
													(pSetWmmMsg->wmmParamsPerAc[1].acm_flag << 1)| 
													(pSetWmmMsg->wmmParamsPerAc[2].acm_flag << 2)| 
													(pSetWmmMsg->wmmParamsPerAc[3].acm_flag << 3);
			}
		}
	}

	/* Send message to BSS manager - confirm on Add vap event  */
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_SET_WMM_PARAMS, BSS_MANAGER_VAP_MANAGER_HIM_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
}
	

static void hostInterface_AddVap(K_MSG* psMsg) 
{
	UMI_SET_WMM_PARAMETERS* pPostVapActivationMsg; 
	uint8 					vapId;
	bool 					isReliableMulticast;
	
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast.
	pPostVapActivationMsg = (UMI_SET_WMM_PARAMETERS*) EXTRACT_VAP_MANAGER_MSG(psMsg);
	vapId = pPostVapActivationMsg->vapId;

	HostInterfaceVapDb[vapId].enabled = TRUE;
	// Get stored flag for reliable multicast (was sent during SET_BSS)
	isReliableMulticast = ((HostInterface_GlobalDb.reliableMulticastBitmap >> vapId) & 0x1);
	if (isReliableMulticast == TRUE)
	{
		// Reliable multicast - send PDs to UMAC (for duplication)
		HostIfAcc_ChangeVapMcastPdDist(PD_DIST_UMAC, vapId);   
		// Set initial multicast mode of VAP
		MulticastHandler_SetVapMulticastMode(vapId, RELIABLE_MULTICAST);
	}
	else
	{
		// Regular multicast - send PDs directly to TX Queues
		HostIfAcc_ChangeVapMcastPdDist(PD_DIST_TXQ, vapId);
		// Set initial multicast mode of VAP
		MulticastHandler_SetVapMulticastMode(vapId, REGULAR_MULTICAST);
	}

	// Clear stop VAP traffic bitmap (might be set if this vap was once active)
	HostInterface_GlobalDb.stopVapTrafficBitmap &= ~(0x1 << vapId);	

	/* Send message to BSS manager - confirm on Add vap event  */
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_POST_VAP_ACTIVATION, BSS_MANAGER_VAP_MANAGER_HIM_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
}


static void hostInterface_StopVapTraffic(K_MSG* psMsg) 
{
	UMI_STOP_VAP_TRAFFIC* stopTraffic;
	uint8 vapId;

	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	stopTraffic = (UMI_STOP_VAP_TRAFFIC *)EXTRACT_VAP_MANAGER_MSG(psMsg); //need a stop traffic for VAP
	vapId = stopTraffic->vapId;  // Should be taken from stop traffic

	// Mark that the VAP is during stop MCAST traffic. Used in order not to change PD distribution to TXQ when getting "UM_MAN_SET_MULTICAST_MODE_REQ"
	HostInterface_GlobalDb.stopVapTrafficBitmap |= (0x1 << vapId);	

	HostIfAcc_ChangeVapMcastPdDist(PD_DIST_UMAC, vapId);  // configure hostIf acc to forward VAP's MC packet to FW  

	HostInterfaceVapDb[vapId].enabled = FALSE;

	/* Send message to BSS manager - confirm on Remove vap event  */
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_STOP_TRAFFIC, BSS_MANAGER_VAP_MANAGER_HIM_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
}

static void hostInterface_SetVapMulticastMode(K_MSG* psMsg) 
{
	UMI_MULTICAST_MODE*		pMulticastMode = (UMI_MULTICAST_MODE*)pK_MSG_DATA(psMsg);
	uint8					vapIndex;
	bool					isVapInStopTraffic;


#if defined (ENET_INC_ARCH_WAVE600)
	// Driver sends the vap index not as actual vap but as offset from 0 in the same band.
	// Need to define if FW continue to do this override or driver do it instead.
	pMulticastMode->u8VapID = psMsg->header.vapId;
#endif

	SLOG0(0, 0, UMI_MULTICAST_MODE, pMulticastMode);

	vapIndex = pMulticastMode->u8VapID;
	
	if (pMulticastMode->getSetOperation == API_GET_OPERATION)
	{
		pMulticastMode->u8Action = MulticastHandler_GetVapMulticastMode(pMulticastMode->u8VapID);
	}
	else
	{
		// Check if VAP is during stop traffic
		isVapInStopTraffic = ((HostInterface_GlobalDb.stopVapTrafficBitmap >> vapIndex) & 0x1);
		
		// We don't change mode during stop traffic. At stop traffic we want multicast to go to UMAC anyway.
		if (isVapInStopTraffic == FALSE)
		{
			if (pMulticastMode->u8Action == ENABLE_RELIABLE_MULTICAST)
			{
				// Reliable multicast - send PDs to UMAC (for duplication)
				HostIfAcc_ChangeVapMcastPdDist(PD_DIST_UMAC, vapIndex);   
				// Set initial multicast mode of VAP
				MulticastHandler_SetVapMulticastMode(vapIndex, RELIABLE_MULTICAST);
			}
			else
			{
				// Regular multicast - send PDs directly to TX Queues
				HostIfAcc_ChangeVapMcastPdDist(PD_DIST_TXQ, vapIndex);
				// Set initial multicast mode of VAP
				MulticastHandler_SetVapMulticastMode(vapIndex, REGULAR_MULTICAST);
			}
		}
	}

	// Send confirmation to host
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_MULTICAST_MODE_CFM, TASK_UM_IF_TASK, psMsg, vapIndex);
}



/***************************************************************************
**
** NAME         vHIM_SendMsgToHost
**
** PARAMETERS   psKnlMsg      A pointer to the message that is to be sent to
**                            the Host. Note that the pointer should refer to
**                            the Kernel Header of the message to be sent
**                            because this contains the Message ID.
**
** RETURNS      na
**
** DESCRIPTION  This function should be called to place a message on the HIM
**              to Host message queue and generate a Host interrupt.
**
****************************************************************************/

void vHIM_SendMsgToHost(K_MSG *psKnlMsg)
{    
	uint32 fifoDescriptor = 0;
	uint8 descType;
	uint16 descIndex = 0xFFFF;	// default value is only for logs.
	uint8 BandId = 0;

	TX_INTERRUPT_SAVE_AREA;

	K_MSG * psKnlMsgTemp = PNULL;
#if defined (ENET_INC_ARCH_WAVE600)
	// Convert the actual vap index to vap + band (radio)
	psKnlMsg->header.vapId = ConfigurationManager_ConvertActualVapToVapAndRadio(psKnlMsg->header.vapId, &BandId);
#endif

    // set type of pool according to message type 
    descType = MSG_GET_TYPE(psKnlMsg->header.tKMsgType);

    // check if buffer desc is valid in message type 
    DEBUG_ASSERT(descType);

    // calculate and set index and info in desc 
    switch(descType)
    {
    case MSG_TYPE_DAT_IND:
    case MSG_TYPE_BSS_IND:  		
    case MSG_TYPE_BSS_REQ:
	case MSG_TYPE_DAT_LOGGER_IND:
		DEBUG_ASSERT(0);
		break;
		
    case MSG_TYPE_MAN_IND:
         /* calculate index */
        descIndex = ((uint32)psKnlMsg - (uint32)asMAN_IndData) / sizeof(SHRAM_IND_MSG);
        break;
    case MSG_TYPE_MAN_REQ:
		HostInterfaceCounters.host2fwManCfmCnt++;
         /* calculate index */
		//WLANRTSYS-11712 - As part of secured host message implementation, When the message processing 
		//is completed the message is copied back to the shared RAM buffer from B1 Tx Circular buffer
		//Before the host is interrupted
		descIndex = ((uint32)psKnlMsg - (uint32)asHIMmanReqSecured) / sizeof(SHRAM_MAN_MSG);
		psKnlMsgTemp = (K_MSG *)((uint32)asHIMmanReq + descIndex * sizeof(SHRAM_MAN_MSG));
		MEMCPY((void *)psKnlMsgTemp,(void const *)psKnlMsg, sizeof(SHRAM_MAN_MSG));
		psKnlMsg = psKnlMsgTemp;
        break;
    case MSG_TYPE_DBG_IND:
         /* calculate index */
        descIndex = ((uint32)psKnlMsg - (uint32)asHIMdbgInd) / sizeof(SHRAM_DBG_MSG);
        break;
    case MSG_TYPE_DBG_REQ:
		HostInterfaceCounters.host2fwDbgCfmCnt++;
         /* calculate index */
		//WLANRTSYS-11712 - As part of secured host message implementation, When the message processing 
		//is completed the message is copied back to the shared RAM buffer from B1 Tx Circular buffer
		//Before the host is interrupted
		descIndex = ((uint32)psKnlMsg - (uint32)asHIMdbgReqSecured) / sizeof(SHRAM_DBG_MSG);
		psKnlMsgTemp = (K_MSG *)((uint32)asHIMdbgReq + descIndex * sizeof(SHRAM_DBG_MSG));
		MEMCPY((void *)psKnlMsgTemp,(void const *)psKnlMsg, sizeof(SHRAM_DBG_MSG));
		psKnlMsg = psKnlMsgTemp;
        break;
    default:
        FATAL("Error buffer descriptor in FIFO!");
    }

	// Set the fields in the descriptor
	MTLK_BFIELD_SET(fifoDescriptor, HIM_DESC_TYPE, 	descType);
	MTLK_BFIELD_SET(fifoDescriptor, HIM_DESC_INDEX, descIndex);
	MTLK_BFIELD_SET(fifoDescriptor, HIM_DESC_VAP, 	psKnlMsg->header.vapId);
	MTLK_BFIELD_SET(fifoDescriptor, HIM_DESC_RADIO, BandId);

	 
	LOGGER_API_LOG_KERNEL_EVENT(KERNEL_EVENT_SEND_TO_HOST_MSG, psKnlMsg);
	
    OSAL_DISABLE_INTERRUPTS(&interrupt_save); 
    HIM_PutMessageOnHostFifo(&fifoDescriptor);

#ifdef ATOMIC_COUNTER_ENABLER
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_HOST_IF_MAC_HOST_MAILBOX_ACCUM_ADD, 0x1);
#else
	RegAccess_Write(REG_HOST_IF_ACC_MAC_HOST_MAILBOX_ACCUM_ADD, 0x1);
#endif
#else
    vHRC_GenerateHostInterrupt(ENET_HRT_LIC_INTERRUPT_HOST);
#endif
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/***************************************************************************
**
** NAME         vHRC_GenerateHostInterrupt
**
** PARAMETERS   na
**
** RETURNS      na
**
** DESCRIPTION  This function should be called to generate an interrupt to
**              the host.
**
****************************************************************************/
#if defined (ENET_INC_UMAC)
void vHRC_GenerateHostInterrupt(uint8 intType)
{
#ifdef ENET_INC_ARCH_WAVE600
#ifdef ENET_INC_ARCH_WAVE600D2
	RegHostIfPhiInterruptSet_u phiInterruptSetReg;
#else
	RegMacGeneralPhiInterruptSet_u phiInterruptSetReg;
#endif

	phiInterruptSetReg.bitFields.phiInterruptSet = intType;
#ifdef ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_HOST_IF_PHI_INTERRUPT_SET, phiInterruptSetReg.val);	
#else
	RegAccess_Write(REG_MAC_GENERAL_PHI_INTERRUPT_SET, phiInterruptSetReg.val);	
#endif
 

#else
	RegMacRabPhiInterruptSet_u phiInterruptSetReg;

	phiInterruptSetReg.bitFields.phiInterruptSet = intType;
	RegAccess_Write(REG_MAC_RAB_PHI_INTERRUPT_SET, phiInterruptSetReg.val);
#endif //ENET_INC_ARCH_WAVE600
}
#endif
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization_start" 
#endif
void sendInitializationCompleteToHost()
{

    /* Signal to host that the interface is ready */
    sCHIvectorArea.sBasic.u32Magic     				= HOST_MAGIC;// remove once driver print on console the 3 next line values
	sCHIvectorArea.sDebugExt.sData.u32Core0_info 	= HOST_MAGIC;
	sCHIvectorArea.sDebugExt.sData.u32Core1_info 	= HOST_MAGIC;
	sCHIvectorArea.sDebugExt.sData.u32Core2_info 	= HOST_MAGIC;

    /* generate interrupt */
    vHRC_GenerateHostInterrupt(ENET_HRT_LIC_INTERRUPT_HOST);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
bool hostInterface_IsRobustManagementFrame(uint8 subType, uint8 action)
{
	/*Action Frames are special ...*/
	if (subType == MGMT_FRAME_SUBTYPE_ACTION)
	{
		return protectedActionFrames[action];
	}
	else
	{
		return protectedManagementFrames[subType];
	}
}



/**********************************************************************************

hostInterface_atfCheckAgeingCondition


Description:
------------
    
    
Input:
-----
    

**********************************************************************************/
#if defined (ENET_INC_ARCH_WAVE600)	
bool hostInterface_atfCheckAgeingCondition(uint8 ttlCount, uint8 bandId)
{
    uint8 ttlGap;
    bool willAged = FALSE;
    
    if (HostInterface_GlobalDb.atf_AgeingEnable[bandId] == TRUE)
    {
        // calculate the gap
        ttlGap = (HostInterface_GlobalDb.atf_End_of_interval_ttl[bandId] - ttlCount);
        /* In case of wrap around the last bit is ignored */
        ttlGap &= HOST_INTERFACE_TTL_MASK;
        // if condition is TRUE that PD will be aged. since gap of time is higher than criteria
        willAged = (ttlGap > HostInterface_GlobalDb.atf_AgeingTtlCriteria[bandId]);
    }
#ifdef ATF_DEBUG_LOCAL    
    ILOG0_DDDD("HostInterface_GlobalDb.atf_AgeingEnable %d ttlCount %d, HostInterface_GlobalDb.atf_End_of_interval_ttl %d, HostInterface_GlobalDb.atf_AgeingTtlCriteria %d",HostInterface_GlobalDb.atf_AgeingEnable[bandId], ttlCount, HostInterface_GlobalDb.atf_End_of_interval_ttl[bandId], HostInterface_GlobalDb.atf_AgeingTtlCriteria[bandId]);
    ILOG0_D("willAged%d",willAged);
#endif    
    return (willAged);
}

#else
bool hostInterface_atfCheckAgeingCondition(uint8 ttlCount)
{
    uint8 ttlGap;
    bool willAged = FALSE;
    
    if (HostInterface_GlobalDb.atf_AgeingEnable == TRUE)
    {
        // calculate the gap
        ttlGap = (HostInterface_GlobalDb.atf_End_of_interval_ttl - ttlCount);
        /* In case of wrap around the last bit is ignored */
        ttlGap &= HOST_INTERFACE_TTL_MASK;
        // if condition is TRUE that PD will be aged. since gap of time is higher than criteria
        willAged = (ttlGap > HostInterface_GlobalDb.atf_AgeingTtlCriteria);
    }
#ifdef ATF_DEBUG_LOCAL    
    ILOG0_DDDD("HostInterface_GlobalDb.atf_AgeingEnable %d ttlCount %d, HostInterface_GlobalDb.atf_End_of_interval_ttl %d, HostInterface_GlobalDb.atf_AgeingTtlCriteria %d",HostInterface_GlobalDb.atf_AgeingEnable, ttlCount, HostInterface_GlobalDb.atf_End_of_interval_ttl, HostInterface_GlobalDb.atf_AgeingTtlCriteria);
    ILOG0_D("willAged%d",willAged);
#endif    
    return (willAged);
}
#endif


bool hostInterface_IsStaEnabled(StaId sid)
{
	if (HostInterfaceStaDb[sid].enabled == TRUE)
	{
		return (TRUE);
	}
	return (FALSE);
}

bool hostInterface_IsStaMfp(StaId sid)
{
	if (HostInterfaceStaDb[sid].use_pmf == TRUE)
	{
		return (TRUE);
	}
	return (FALSE);
}

bool hostInterface_IsStaAtfBuffered(StaId sid)
{
	if (HostInterfaceStaDb[sid].atf_buffered == TRUE)
	{
		return (TRUE);
	}
	return (FALSE);
}

bool hostInterface_IsVapEnabled(uint8 vapId)
{
	if (HostInterfaceVapDb[vapId].enabled == TRUE)
	{
		return (TRUE);
	}
	return (FALSE);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void HostInterface_PostInit()
{
	StaId i;

	/*Set protected Action Frames*/
	memset(protectedActionFrames, 0, NUM_ACTION_CATEGORY);
	protectedActionFrames[SPECTRUM_MAN_CATEGORY] = 1; // 0 - Spectrum Management
	protectedActionFrames[QOS_CATEGORY] = 1; // 1 - QoS
	protectedActionFrames[DLS_CATEGORY] = 1; // 2 - DLS
	protectedActionFrames[BLOCK_ACK_CATEGORY] = 1; // 3 - Block ACK
	protectedActionFrames[RADIO_MEAS_CATEGORY] = 1; // 5 - Radio Measurement
	protectedActionFrames[FAST_BSS_CATEGORY] = 1; // 6 - Fast BSS transition
	protectedActionFrames[SA_QUERY_CATEGORY] = 1; // 8 - SA Query
	protectedActionFrames[PROTECTED_DUAL_CATEGORY] = 1; // 9 - Protected Dual
	protectedActionFrames[WNM_CATEGORY] = 1;  // 10 - WNM
	protectedActionFrames[MESH] = 1;  // 13 - Mesh
	protectedActionFrames[MULTIHOP] = 1;  // 14 - Multihop
	protectedActionFrames[VENDOR_SPECIFIC_PROTECTED] = 1;  // 126 - Vendor Specific Protected
	
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		HostInterfaceStaDb[i].enabled = FALSE;
		HostInterfaceStaDb[i].open = FALSE;
		HostInterfaceStaDb[i].use_pmf = FALSE;        
		HostInterfaceStaDb[i].atf_buffered = FALSE;
		HostInterfaceStaDb[i].pdDist = PD_DIST_UMAC;
		HostInterfaceStaDb[i].flags = 0;
		HostInterfaceStaDb[i].acm_flags = ACM_ALL_ACS_ALLOWED;
	}
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		HostInterfaceVapDb[i].enabled = FALSE;
	}
	HostInterface_GlobalDb.reliableMulticastBitmap 	= 0;
	HostInterface_GlobalDb.stopVapTrafficBitmap 	= 0;
	HostInterface_GlobalDb.hostInterfaceNumAtfStas = 0;
	HostInterface_GlobalDb.hostInterfaceStaNumNonQoS = 0;
#if defined (ENET_INC_ARCH_WAVE600)	
	HostInterface_GlobalDb.atf_AgeingEnable[CONFIGURATION_MANAGER_BAND_0] = FALSE;
	HostInterface_GlobalDb.atf_AgeingEnable[CONFIGURATION_MANAGER_BAND_1] = FALSE;
#else	
	HostInterface_GlobalDb.atf_AgeingEnable = FALSE;
#endif
	HostInterface_GlobalDb.restrictedAcModeEnable = FALSE;
	HostInterface_GlobalDb.restrictedAcBitmap = 0;
	HostInterface_GlobalDb.restrictedAcThreshEnter = HOST_INTERFACE_FREE_PDS_THRESH_RESTRICTED_AC_ENABLE;
	HostInterface_GlobalDb.restrictedAcThreshExit = HOST_INTERFACE_FREE_PDS_THRESH_RESTRICTED_AC_DISABLE;

	HostInterfaceRestrictedAcDb.enable = FALSE;
    }
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
#endif
/* End of file him.c. */











