/***********************************************************************************
 File:		HostInterfaceAcc.c
 Module:		Host Interface
 Purpose: 		
 Description:	
************************************************************************************/
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HostAccCode.h"
#include "HostIfAccRegs.h"
#include "stringLibApi.h"
#include "PacketDescriptor.h"
#include "RegAccess_Api.h"
#include "HostGenRiscInterface_Api.h"
#include "HwQManager_API.h"
#include "shram_dataHostIf.h"
#include "HostInterfaceAcc_Api.h"
#include "RxDescriptor.h"
#include "loggerAPI.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_HOST_IF
#define LOG_LOCAL_FID 1

#define TX_IN_HOST_IF_FW_CODE		(HOST_IF_ACC_BASE_ADDRESS + 0x800)
#define TX_OUT_HOST_IF_FW_CODE  	(HOST_IF_ACC_BASE_ADDRESS + 0x900)
#define RX_IN_HOST_IF_FW_CODE   	(HOST_IF_ACC_BASE_ADDRESS + 0xa00)
#define RX_OUT_HOST_IF_FW_CODE  	(HOST_IF_ACC_BASE_ADDRESS + 0xb00)
#define TX_IN_FIXED_VALUES 			(HOST_IF_ACC_BASE_ADDRESS + 0x8C4)
#define TX_OUT_FIXED_VALUES 		(HOST_IF_ACC_BASE_ADDRESS + 0x9C4)
#define RX_IN_FIXED_VALUES 			(HOST_IF_ACC_BASE_ADDRESS + 0xaC4)
#define RX_OUT_FIXED_VALUES 		(HOST_IF_ACC_BASE_ADDRESS + 0xbC4)
#define DIST_BITMAP_ADDRESS     	(HOST_IF_ACC_BASE_ADDRESS + 0xC00)
#define UNICAST_BITMAP_SIZE     	(1024)
#define VAPS_BITMAP_SIZE		 	(16)
#define ALIGN_VAP_BITMAP_2_WORD  	(16)
#define GROUPS_BITMAP_SIZE       	(128)

typedef enum 
{
	HOST_IF_ACC_START_CORE = 0,
	HOST_IF_ACC_HALT_CORE,
}HostIFAccCoreConfig_e;

typedef enum 
{
	HOST_IF_ACC_NO_SWAP_MODE = 0,
	HOST_IF_ACC_SWAP_MODE ,
}HostIfAccSwapMode_e;

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static 	void hostIfAccLoadTxInCode(DataPathMode_e dataPthMode);
static void hostIfAccLoadTxOutCode(DataPathMode_e dataPthMode);
static void hostIfAccLoadRxInCode(DataPathMode_e dataPthMode);
static void hostIfAccLoadRxOutCode(DataPathMode_e dataPthMode);

//---------------------------------------------------------------------------------
//						Static Function Definitions				
//---------------------------------------------------------------------------------
static void hostIfAccLoadTxInCode(DataPathMode_e dataPthMode)
{
	if (dataPthMode == DATA_PATH_MODE_DC_MODE_1)
	{
		memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void *)PumaPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxInAccCode)));
	}
	else if (dataPthMode == DATA_PATH_MODE_DC_NONE)
	{
		memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)SwPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathTxInAccCode)));
	}
	else
	{
		memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)FastPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxInAccCode)));
	}
	memcpy32((void *)TX_IN_FIXED_VALUES, (void*)TxInAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(TxInAccFixedValues)));
}

static void hostIfAccLoadTxOutCode(DataPathMode_e dataPthMode)
{
	if (dataPthMode == DATA_PATH_MODE_DC_MODE_1)
	{
		memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void *)PumaPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxOutAccCode)));
	}	
	else if (dataPthMode == DATA_PATH_MODE_DC_NONE)
	{
		memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)SwPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathTxOutAccCode)));
	}
	else
	{
		memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)FastPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxOutAccCode)));
	}    
	memcpy32((void *)TX_OUT_FIXED_VALUES,   (void*)TxOutAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(TxOutAccFixedValues)));
}

static void hostIfAccLoadRxInCode(DataPathMode_e dataPthMode)
{
	if (dataPthMode == DATA_PATH_MODE_DC_MODE_1)
	{
		memcpy32((void *)RX_IN_HOST_IF_FW_CODE, (void *)PumaPathRxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathRxInAccCode)));
	}
	else if (dataPthMode == DATA_PATH_MODE_DC_NONE)
	{
		memcpy32((void *)RX_IN_HOST_IF_FW_CODE, (void*)SwPathRxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathRxInAccCode)));
	}
	memcpy32((void *)RX_IN_FIXED_VALUES,   (void*)RxInAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(RxInAccFixedValues)));
}

static void hostIfAccLoadRxOutCode(DataPathMode_e dataPthMode)
{	
	if (dataPthMode == DATA_PATH_MODE_DC_MODE_1)
	{
		memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void *)PumaPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathRxOutAccCode)));
	}
	else if (dataPthMode == DATA_PATH_MODE_DC_NONE)
	{
		memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)SwPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathRxOutAccCode)));
	}
	else
	{
		memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)FastPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathRxOutAccCode)));
	}
	memcpy32((void *)RX_OUT_FIXED_VALUES, (void*)RxOutAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(RxOutAccFixedValues)));
}

//---------------------------------------------------------------------------------
//						Public Function Definitions				
//---------------------------------------------------------------------------------
void HostIfAcc_SetMcastDistMode(McastDistMode_e newMcastMode)
{
	RegHostIfAccMulticastPdDistribCfg_u mcastDistCfg;

	mcastDistCfg.val = 0;
	RegAccess_Read(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,&mcastDistCfg.val);
	mcastDistCfg.bitFields.groupVapPrio = newMcastMode;
	RegAccess_Write(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,mcastDistCfg.val);
}

static void hostIfAcc_changePdDist(PdDist_e distValue , DistType_e distType , uint16 index)
{
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	pdDistCfg.bitFields.pdDistAddr = index;
	pdDistCfg.bitFields.pdDistType = distType;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
}

void hostIfAccChangeStaTidPdDist(PdDist_e distValue ,StaId staId,uint8 TID)
{
	uint16 index = 0; // 3 lsb - TID, 7 msb - staId 

	index =  (TID&INDEX_TID_MASK)|(staId << INDEX_STA_ID_OFFSET);  
	hostIfAcc_changePdDist(distValue,DIST_TYPE_UNICAST,index); 
}

void HostIfAcc_ChangeStaAllTidPdDist(PdDist_e distValue ,StaId staId)
{
#ifndef TX_DATA_UM_PATH
	uint16 tidIndex; // 3 lsb - TID, 7 msb - staId

	for(tidIndex  = 0; tidIndex < NUM_OF_TID; tidIndex++)
	{
		hostIfAccChangeStaTidPdDist(distValue, staId, tidIndex); 
	}
#endif
}

void HostIfAcc_ChangeVapMcastPdDist(PdDist_e distValue ,uint16 vapId)
{
#ifndef TX_DATA_UM_PATH
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	pdDistCfg.bitFields.pdDistAddr = vapId;
	pdDistCfg.bitFields.pdDistType = DIST_TYPE_MULTICAST_VAP;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
#endif
}

void HostIfAcc_ChangeMcastGroupPdDist(PdDist_e distValue ,uint16 groupId)
{
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	pdDistCfg.bitFields.pdDistAddr = groupId;
	pdDistCfg.bitFields.pdDistType = DIST_TYPE_MULTICAST_GROUP;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
}

/**********************************************************************************

HostIfAcc_ConfigureRingsSize 

Description:
------------
	Configure the Host IF Acc with Ring size -> The Acc uses this vali=ue to calucatre how many HD
	to forward towards GenRisc so it will not exceeds Ring end.
Input:
-----
	dataPathInitParams - Int params Structuer recieved from Driver

Output:
-------
	None

**********************************************************************************/
void HostIfAcc_ConfigureRingsSize(UMI_DATA_PATH_INIT_PARAMS* dataPathInitParams)
{
	RegHostIfAccTxRingCfg_u	txRingCfgReg;
	RegHostIfAccRxRingCfg_u rxRingCfgReg;
    RegHostIfAccTxInConvCfg_u txInCfg;
    RegHostIfAccTxOutConvCfg_u txOutCfg;
	RegHostIfAccRxInConvCfg_u  rxInCfg;
	RegHostIfAccRxOutConvCfg_u rxOutCfg;
	RegHostIfAccHostType_u hostTypeReg;
	RegHostIfAccOwnBitValue_u ownBitValueReg;
	RegHostIfAccTxInSwInstruction_u txInSwInstrReg;
	RegHostIfAccTxOutSwInstruction_u txOutSwInstrReg;
	RegHostIfAccRxInSwInstruction_u rxInSwInstrReg;
	RegHostIfAccRxOutSwInstruction_u rxOutSwInstrReg;
	HostIfDataPathHdSize_t *hostIfDataHdSize;	
	RegHostIfAccFreeRdInit_u freeRdInitReg;
	DataPathMode_e dataPathMode;
	uint8 swapMode;
	uint8 rxInSwHalt;

	
	freeRdInitReg.val = 0;

	dataPathMode = (DataPathMode_e)dataPathInitParams->dataPathMode;
	hostIfDataHdSize = HostInterfaceRings_GetHdSizes(dataPathMode);
	if (dataPathMode == DATA_PATH_MODE_DC_MODE_1)
	{
		rxInSwHalt = HOST_IF_ACC_START_CORE;	// PUMA Direct Connect: Rx is active
		swapMode = HOST_IF_ACC_SWAP_MODE;		// PUMA Direct Connect: HD is BIG Endianness (PUMA cable card is writing in BIG Endianness to the RING)
		
		freeRdInitReg.bitFields.freeRdInit = (NUM_OF_HOST_DATA_RX_DESCRIPTORS - NUM_OF_FORWARD_RX_DESCRIPTORS);
	}
	else 
	{
		
		freeRdInitReg.bitFields.freeRdInit = (dataPathInitParams->numOfAllocatedRxHds - NUM_OF_FORWARD_RX_DESCRIPTORS);
		if(dataPathMode == DATA_PATH_MODE_DC_NONE)
		{
			rxInSwHalt = HOST_IF_ACC_START_CORE;
			swapMode = HOST_IF_ACC_NO_SWAP_MODE;		
		}
		else
		{
			rxInSwHalt = HOST_IF_ACC_HALT_CORE; /* When working on Fast Data Path Mode (GSWIP) Rx Is Halted */
			if(dataPathInitParams->hostEndianessMode == HOST_ENDIANESS_MODE_BE)
			{
				swapMode = HOST_IF_ACC_SWAP_MODE;
			}
			else
			{
				swapMode = HOST_IF_ACC_NO_SWAP_MODE;
			}
		}
	}
	
	/* Configure Number of Free RDs for TH Counter - it is different between DC_DP mode one and other modes because of the actual amount of RDs that pushed to free list in those cases*/
	RegAccess_Write(REG_HOST_IF_ACC_FREE_RD_INIT,freeRdInitReg.val);
	/* Configure Tx Out Ring Size*/
	RegAccess_Read(REG_HOST_IF_ACC_TX_RING_CFG,&txRingCfgReg.val);
	txRingCfgReg.bitFields.txOutRingSize = (dataPathInitParams->txOutRingSizeBytes / hostIfDataHdSize->txOutHdSize); // HDS;
	RegAccess_Write(REG_HOST_IF_ACC_TX_RING_CFG,txRingCfgReg.val); 
	/* Configure Rx Out Ring Size*/
	RegAccess_Read(REG_HOST_IF_ACC_RX_RING_CFG,&rxRingCfgReg.val);
	rxRingCfgReg.bitFields.rxOutRingSize = (dataPathInitParams->rxOutRingSizeBytes/ hostIfDataHdSize->rxOutHdSize); // HDS 	
	RegAccess_Write(REG_HOST_IF_ACC_RX_RING_CFG,rxRingCfgReg.val); 
	/* Configure Tx In */
	txInCfg.val = 0;
	txInCfg.bitFields.txInHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->txInHdSize)-1;
	txInCfg.bitFields.txInPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(TxPd_t))-1;
	txInCfg.bitFields.txInTxqIdLine = PCK_DSCPTR_TID_LINE_NUM;
	txInCfg.bitFields.txInTxqIdOffset = PCK_DSCPTR_TID_OFFSET; 
	txInCfg.bitFields.txInDescAllZeros = TRUE;
	txInCfg.bitFields.txInSwap = swapMode; 
	txInCfg.bitFields.txInSetNullPop = TRUE; 
	txInCfg.bitFields.txInSetNullPush = TRUE; 
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_CONV_CFG,txInCfg.val); 
	/* Configure Tx Out */
	txOutCfg.val = 0;
	txOutCfg.bitFields.txOutHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->txOutHdSize)-1;;
	txOutCfg.bitFields.txOutPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(TxPd_t))-1;
	txOutCfg.bitFields.txOutTxqIdLine = PCK_DSCPTR_TID_LINE_NUM;
	txOutCfg.bitFields.txOutTxqIdOffset = PCK_DSCPTR_TID_OFFSET;
	txOutCfg.bitFields.txOutSwap = swapMode; 
	txOutCfg.bitFields.txOutSetNullPop = TRUE; 
	txOutCfg.bitFields.txOutSetNullPush = TRUE; 
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_CONV_CFG,txOutCfg.val); 
	/* Configure Rx In */
	rxInCfg.val = 0;
	rxInCfg.bitFields.rxInHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->rxInHdSize)-1;
	rxInCfg.bitFields.rxInPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(Rd_t ))-1;
	rxInCfg.bitFields.rxInSetNullPop = TRUE;
	rxInCfg.bitFields.rxInSetNullPush = TRUE;
	rxInCfg.bitFields.rxInSwap = swapMode;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_CONV_CFG,rxInCfg.val); 
	/* Configure Rx Out */
	rxOutCfg.val = 0;
	rxOutCfg.bitFields.rxOutHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->rxOutHdSize)-1;
	rxOutCfg.bitFields.rxOutPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(Rd_t ))-1;
	rxOutCfg.bitFields.rxOutTxqIdLine = RD_TX_QID_LINE_NUM;
	rxOutCfg.bitFields.rxOutTxqIdOffset = RD_TX_QID_OFFSET;
	rxOutCfg.bitFields.rxOutSetNullPop = TRUE;
	rxOutCfg.bitFields.rxOutSetNullPush = TRUE;
	rxOutCfg.bitFields.rxOutSwap = swapMode; 
	rxOutCfg.bitFields.rxOutSetNullPop = TRUE;
	rxOutCfg.bitFields.rxOutSetNullPush = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_CONV_CFG,rxOutCfg.val); 
	/* Configure all Rings to  Fast Path (Gswip)\Sw Path(No GSWIP) */
	hostTypeReg.val = 0;
	hostTypeReg.bitFields.txInGswip = dataPathMode;	
	hostTypeReg.bitFields.txOutGswip = dataPathMode;	
	hostTypeReg.bitFields.rxInGswip = dataPathMode;	
	hostTypeReg.bitFields.rxOutGswip = dataPathMode;
#ifdef CBM_FRAGMENTATION_WORKAROUND
	if (dataPathInitParams->cbmFragmentationWaEnable)
	{
		hostTypeReg.bitFields.rxOutGswip = DATA_PATH_MODE_DC_NONE;
	}
#endif //CBM_FRAGMENTATION_WORKAROUND
	
	RegAccess_Write(REG_HOST_IF_ACC_HOST_TYPE,hostTypeReg.val); 
	/* Own Bit Value Configuration*/
	ownBitValueReg.bitFields.ownBitValue = dataPathInitParams->hdOwnBitValue;
	ownBitValueReg.bitFields.ownBitLine	 = PCK_DSCPTR_OWN_LINE_NUM;
	ownBitValueReg.bitFields.ownBitOffset = PCK_DSCPTR_OWN_BIT_OFFSET;
	RegAccess_Write(REG_HOST_IF_ACC_OWN_BIT_VALUE,ownBitValueReg.val); 	
    hostIfAccLoadTxInCode(dataPathMode);
    hostIfAccLoadTxOutCode(dataPathMode);
    hostIfAccLoadRxInCode(dataPathMode);
    hostIfAccLoadRxOutCode(dataPathMode);
	/* Start  Hw Acc Cores */
	txInSwInstrReg.val	= 0;
	txOutSwInstrReg.val = 0;
	rxInSwInstrReg.val	= 0;
	rxOutSwInstrReg.val = 0;
	txInSwInstrReg.bitFields.txInSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_SW_INSTRUCTION,txInSwInstrReg.val); 
	txOutSwInstrReg.bitFields.txOutSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_SW_INSTRUCTION,txOutSwInstrReg.val); 
	rxOutSwInstrReg.bitFields.rxOutSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_SW_INSTRUCTION,rxOutSwInstrReg.val); 
	rxInSwInstrReg.bitFields.rxInSwHalt = rxInSwHalt;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_SW_INSTRUCTION,rxInSwInstrReg.val); 
}

/**********************************************************************************

HostIfAcc_GetCurrentFreePdsNumber 


Description:
------------
Returns the current number of free PDs

Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
uint16 HostIfAcc_GetCurrentFreePdsNumber(void)
{
	RegHostIfAccFreePdCount_u numberOfFreePdsRegister;

	numberOfFreePdsRegister.val = 0;
	RegAccess_Read(REG_HOST_IF_ACC_FREE_PD_COUNT, &numberOfFreePdsRegister.val);
	return numberOfFreePdsRegister.bitFields.freePdCount;
}

/**********************************************************************************

HostIfAcc_ClearFreePdsThresholdEvent 


Description:
------------
Clears the event that the number of free PDs has reached to the threshold (this event goes 
to HW events)

Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
void HostIfAcc_ClearFreePdsThresholdEvent(void)
{
	RegHostIfAccThresholdClr_u clearThresholdInterruptRegister;

	clearThresholdInterruptRegister.val = 0;
	clearThresholdInterruptRegister.bitFields.pdThresReachedIntClr = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_THRESHOLD_CLR, clearThresholdInterruptRegister.val); 
}

/**********************************************************************************

HostIfAcc_SetPdThreshold 


Description:
------------
Sets the PD threshold
Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
void HostIfAcc_SetPdThreshold(uint16 pdThreshold)
{
	RegHostIfAccPdThreshold_u freePdsThresholdRegister;

	ILOG0_D("HostIfAcc_SetPdThreshold - %d", pdThreshold);
	freePdsThresholdRegister.val = 0;
	freePdsThresholdRegister.bitFields.pdThres = pdThreshold;
	RegAccess_Write(REG_HOST_IF_ACC_PD_THRESHOLD,freePdsThresholdRegister.val);
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void HostIfAcc_Init(void)
{
    RegHostIfAccTxInListIdx_u txInListIdx;
    RegHostIfAccTxOutListIdx_u txOutListIdx;
    RegHostIfAccRxInListIdx_u  rxInListIdx;
    RegHostIfAccRxOutListIdx_u rxOutListIdx;   
	RegHostIfAccRxInSwInstruction_u rxInSwInstrReg;
	RegHostIfAccMulticastPdDistribCfg_u pdMcCfg; 
	RegHostIfAccTtlScaleSet_u ttlScaleRegister;
	RegHostIfAccFreePdInit_u  numberOfFreePdsRegister;
	RegHostIfAccPdThreshold_u freePdsThresholdRegister;
	RegHostIfAccHostCountersCfg_u counterControlRegister;
	RegHostIfAccTxInMulticastCfg_u multicastTxInRegister;
	RegHostIfAccRxOutMulticastCfg_u multicastRxOutRegister;

	ttlScaleRegister.val = 0;
	numberOfFreePdsRegister.val = 0;
	freePdsThresholdRegister.val = 0;
	counterControlRegister.val = 0;	
	multicastTxInRegister.val = 0;
	multicastRxOutRegister.val = 0;
    // config shared RAM buffers addresses
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxInShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxInShramBuffers[1][0]- SHARED_RAM_BASE_ADDR));  
    RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxOutShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxOutShramBuffers[1][0]- SHARED_RAM_BASE_ADDR)); 
    RegAccess_Write(REG_HOST_IF_ACC_RX_IN_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxInShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_RX_IN_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxInShramBuffers[1][0]- SHARED_RAM_BASE_ADDR));  
    RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxOutShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxOutShramBuffers[1][0]- SHARED_RAM_BASE_ADDR)); 
	/* Halt All Hw Acc Cores*/
	rxInSwInstrReg.val = 0;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_SW_INSTRUCTION,rxInSwInstrReg.val); 	
    // config DLM indexes
    txInListIdx.val = 0;
    txInListIdx.bitFields.txInPopListIdx = HW_Q_MANAGER_TX_POOL_LIST_DATA_PD;
    txInListIdx.bitFields.umacUcPushListIdx = HW_Q_MANAGER_READY_LIST_UNICAST_PD;
#ifndef TX_DATA_UM_PATH
	txInListIdx.bitFields.umacMcPushListIdx = HW_Q_MANAGER_READY_LIST_MULTICAST_PD;
#else	
    txInListIdx.bitFields.umacMcPushListIdx = HW_Q_MANAGER_READY_LIST_UNICAST_PD;
#endif
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_LIST_IDX,txInListIdx.val); 
	txOutListIdx.val = 0;
	txOutListIdx.bitFields.txOutPopListIdx = HW_Q_MANAGER_LOW_PR_READY_LIST_TEMP_HOST_Q;
	txOutListIdx.bitFields.txOutPushListIdx = HW_Q_MANAGER_TX_POOL_LIST_DATA_PD;
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_LIST_IDX,txOutListIdx.val); 
	rxInListIdx.val = 0;
	rxInListIdx.bitFields.rxInPopListIdx = HW_Q_MANAGER_RX_HOST_INTERFACE_TEMP_LIST;
	rxInListIdx.bitFields.rxInPushListIdx = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_LIST_IDX,rxInListIdx.val); 
	rxOutListIdx.val = 0;
	rxOutListIdx.bitFields.rxOutPopListIdx = HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT;
	rxOutListIdx.bitFields.rxOutPushListIdx = HW_Q_MANAGER_RX_HOST_INTERFACE_TEMP_LIST;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_LIST_IDX,rxOutListIdx.val);
	pdMcCfg.val = 0; 
#ifndef TX_DATA_UM_PATH
	pdMcCfg.bitFields.groupVapPrio = MCAST_DIST_MODE_VAP_BITMAPS;
#else
	pdMcCfg.bitFields.groupVapPrio = MCAST_DIST_MODE_IGNORE_BITMAPS;
#endif
	pdMcCfg.bitFields.mcBitLine = PCK_DSCPTR_MC_LINE_NUM; 
	pdMcCfg.bitFields.mcBitOffset = PCK_DSCPTR_MC_BIT_OFFSET; 
	pdMcCfg.bitFields.groupIdxLine = 0x0;//PCK_DSCPTR_GROUP_IDX_LINE_NUM; 
	pdMcCfg.bitFields.groupIdxOffset = 0x0;//PCK_DSCPTR_GROUP_IDX_BIT_OFFSET; 
	RegAccess_Write(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,pdMcCfg.val);
    ttlScaleRegister.bitFields.ttlScale = HOST_INTERFACE_ACCELERATOR_TTL_SCALE;
	RegAccess_Write(REG_HOST_IF_ACC_TTL_SCALE_SET,ttlScaleRegister.val);
	numberOfFreePdsRegister.bitFields.freePdInit = NUM_OF_TX_DESC;
	RegAccess_Write(REG_HOST_IF_ACC_FREE_PD_INIT,numberOfFreePdsRegister.val);
	freePdsThresholdRegister.bitFields.pdThres = HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD_DISABLED;
	RegAccess_Write(REG_HOST_IF_ACC_PD_THRESHOLD,freePdsThresholdRegister.val);
	/* Enable Host IF Counters*/
	counterControlRegister.val = 0;
	counterControlRegister.bitFields.qosTxEn = TRUE;
	counterControlRegister.bitFields.txInUniHdCountEn = TRUE;
	counterControlRegister.bitFields.txInMultiHdCountEn = TRUE;
	counterControlRegister.bitFields.txInBroadcastHdCountEn = TRUE;
	counterControlRegister.bitFields.txInUniBytesCountEn = TRUE;
	counterControlRegister.bitFields.txInMultiBytesCountEn = TRUE;
	counterControlRegister.bitFields.txInBroadcastBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutUniHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutMultiHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutBroadcastHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutUniBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutMultiBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutBroadcastBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutStaBytesCountEn = TRUE;
	counterControlRegister.bitFields.agerCountEn = TRUE;
	counterControlRegister.bitFields.qosTxDecEn = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_COUNTERS_CFG,counterControlRegister.val);
	/* Enable HOSTIF_COUNTERS_TX_IN_MULTICAST_HD Counter*/
	multicastTxInRegister.val = 0;
	multicastTxInRegister.bitFields.txInByteCountLine = 0x0;	//dataLength field resides in word1 of PD in bits 16..29
	multicastTxInRegister.bitFields.txInByteCountOffset = 0x30;	//dataLength field resides in word1 of PD in bits 16..29
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_MULTICAST_CFG,multicastTxInRegister.val);
	/* Enable HOSTIF_COUNTERS_RX_OUT_MULTICAST_HD Counter*/
	multicastRxOutRegister.val = 0;
	multicastRxOutRegister.bitFields.rxOutByteCountLine = 0x0;
	multicastRxOutRegister.bitFields.rxOutByteCountOffset = 0x0;
	multicastRxOutRegister.bitFields.rxOutMcBitLine = 0x0;
	multicastRxOutRegister.bitFields.rxOutMcBitOffset = 0x38;
	multicastRxOutRegister.bitFields.rxOutBcBitLine = 0x0;
	multicastRxOutRegister.bitFields.rxOutBcBitOffset = 0x37;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_MULTICAST_CFG,multicastRxOutRegister.val);
	memset((void*)DIST_BITMAP_ADDRESS,0xFF,CONVERT_BIT_INDEX_TO_BYTES(UNICAST_BITMAP_SIZE + VAPS_BITMAP_SIZE + ALIGN_VAP_BITMAP_2_WORD + GROUPS_BITMAP_SIZE)); // init all bitmaps to '1'
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

