/***********************************************************************************
 File:			HostInterfaceRings.c			
 Module:			HostInterface		
 Purpose:			
 Description:		
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "HostInterface_API.h"
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "HostInterface.h"
#include "HwEventsAndErrors_Api.h"
#include "PacketDescriptor.h"
#include "HostIfAccRegs.h"
#include "shram_dataHostIf.h"
#include "shramTxDesc.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "TxPacketsClassifier_API.h"
#include "loggerAPI.h"
#include "HostInterfaceHw.h"
#include "DmaManager_Api.h"
#include "Dut_Api.h"
#include "StaDatabase_Api.h"
#include "StatisticsManager_api.h"
#ifdef HOST_IF_COUNTERS_WORKAROUND
#include "HostInterface_ScratchPadApi.h"
#endif // HOST_IF_COUNTERS_WORKAROUND


#define LOG_LOCAL_GID   GLOBAL_GID_HOST_IF
#define LOG_LOCAL_FID 6

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_ARCH_WAVE600)
#define AAA_RINGS_DEBUG 1
#else
// #define AAA_RINGS_DEBUG 1
#endif
/*---------------------------------------------------------------------------------
/						Externs					
/----------------------------------------------------------------------------------*/
#ifdef DEBUG_STA_MODE_RSSI
extern bool PowerManagementIndication;
#endif

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
#if defined (AAA_RINGS_DEBUG)
uint8 AAA_RINGS_SUBTYPE[4][16] = {0};
uint8 mangMcf[2] = {0};
uint8 mangRetryCount[16] = {0};
uint8 dataMcf[2] = {0};
#endif

HostInterfaceStaDb_t HostInterfaceStaDb[HW_NUM_OF_STATIONS];

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
uint16 hostInterfaceRingsHandleOutBuffer(HdRingDb_t* pRingDb,HostIfShRamBuf_t *pBufferDb);


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
hostInterfaceRingsGetFreeBufferAddr  

Description:
------------
get free shared RAM buffer
Input:
-----
buffDb_t -  buffers DB		

Output:
-------
	

Returns:
--------
HostIfShRamBuf_t - pointer to specific buffer
	
**********************************************************************************/
HostIfShRamBuf_t* hostInterfaceRingsGetFreeBufferAddr(buffDb_t* pbuffDb)
{
	HostIfShRamBuf_t* pBuffer = NULL;
	uint32 bufferIndex = pbuffDb->nextEmptyBufferIndex;

	if (pbuffDb->buffer[bufferIndex].status == BUFFER_STATUS_EMPTY)
	{
		pBuffer = &pbuffDb->buffer[bufferIndex];
	}
	return (pBuffer);
}

/**********************************************************************************
hostInterfaceRingsMoveToNextFreeBuffer  

Description:
------------
mark buffer as full and move pointer to the next buffer ("ping-pong")
Input:
-----
buffDb_t - buffers DB		

Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void hostInterfaceRingsMoveToNextFreeBuffer(buffDb_t* pbuffDb)
{
	pbuffDb->buffer[pbuffDb->nextEmptyBufferIndex].status = BUFFER_STATUS_FULL;
	pbuffDb->nextEmptyBufferIndex = (1 + pbuffDb->nextEmptyBufferIndex) % NUM_OF_SHARED_RAM_BUFFERS;
    pbuffDb->numOfFreeBuffers--;
}

/**********************************************************************************
hostInterfaceRingsGetFullBufferAddr  

Description:
------------
get full shared RAM  buffer
Input:
-----
buffDb_t -  buffers DB		

Output:
-------
	

Returns:
--------
HostIfShRamBuf_t - pointer to specific buffer
	
**********************************************************************************/
HostIfShRamBuf_t* hostInterfaceRingsGetFullBufferAddr(buffDb_t* pbuffDb)
{
	HostIfShRamBuf_t* pBuffer = NULL;
	uint32 bufferIndex = pbuffDb->nextFullBufferIndex;
	
	if (pbuffDb->buffer[bufferIndex].status == BUFFER_STATUS_FULL)
	{
		pBuffer = &pbuffDb->buffer[bufferIndex];
	}
	return (pBuffer);
}

/**********************************************************************************
hostInterfaceRingsMoveToNextFullBuffer  

Description:
------------
mark buffer as empty and move pointer to the next buffer ("ping-pong")
Input:
-----
buffDb_t - buffers DB		

Output:
-------
	

Returns:
--------
	
**********************************************************************************/
void hostInterfaceRingsMoveToNextFullBuffer(buffDb_t* pbuffDb)
{	
	pbuffDb->buffer[pbuffDb->nextFullBufferIndex].status = BUFFER_STATUS_EMPTY;
	pbuffDb->nextFullBufferIndex = (1 + pbuffDb->nextFullBufferIndex)%NUM_OF_SHARED_RAM_BUFFERS;
    pbuffDb->numOfFreeBuffers++;
}

void hostInterfaceRingsNewHdsIndication(K_MSG* pHdsOnRingMsg)
{
	RingDoneData_t* pRingDoneData = NULL;

	pRingDoneData = (RingDoneData_t*)pHdsOnRingMsg->abData;
	pRingsDb[pRingDoneData->ringType]->msgPending = FALSE;
	hostInterfaceRingsNewHds(pRingsDb[pRingDoneData->ringType]);
}

void hostInterfaceRingsNewHds(HdRingDb_t* pRingDb)
{
    TX_INTERRUPT_SAVE_AREA;
	uint16 numHdToCopy = 0;
	uint32 dmaHostAddr;
	uint32 dmaLength;
	uint8* dmaShRamAddr;
	uint32 ddrReadyHds =0;
	uint32 ddrReadAddr;
	HostIfShRamBuf_t* pInBuf = NULL;
	dmaDesc_t* pDmaDesc = NULL;
	uint32 totalHdsCopied = 0;
	
	ddrReadyHds = hostInterfaceGetNumOfReadyHdsInHostRing(pRingDb);
	// if we have free buffer && free PDs && pending HDs in the DDR
	while ((pRingDb->inBuffers.numOfFreeBuffers > 0) && (pRingDb->numOfFreeDescriptors > 0) && (ddrReadyHds > 0))
	{
        ddrReadAddr = pRingDb->readAddress; // get Fw read pointer
		numHdToCopy = ddrReadyHds;
		if (numHdToCopy > pRingDb->numOfFreeDescriptors)
		{
			numHdToCopy = pRingDb->numOfFreeDescriptors;		
		}
		if (numHdToCopy > pRingDb->inBuffers.maxHdsInBuffer)
		{			
			numHdToCopy = pRingDb->inBuffers.maxHdsInBuffer;
		}
		dmaHostAddr = ddrReadAddr;
		dmaLength = numHdToCopy * sizeof(Hd_t);
		if (pRingDb->ringEndAddr <= (dmaHostAddr + dmaLength))
		{
			dmaLength = pRingDb->ringEndAddr - ddrReadAddr;
//			numHdToCopy = dmaLength / sizeof(Hd_t);
			numHdToCopy = DIVIDE_BY_20(dmaLength);
			pRingDb->readAddress= pRingDb->ringStartAddr;
		}
		else
		{
			pRingDb->readAddress = pRingDb->readAddress + dmaLength;
		}
       	pRingDb->numOfFreeDescriptors -= numHdToCopy;
		pInBuf = hostInterfaceRingsGetFreeBufferAddr(&pRingDb->inBuffers);
		dmaShRamAddr = pInBuf->pBuffer;
	    pInBuf->numHdInBuf = numHdToCopy;
		hostInterfaceRingsMoveToNextFreeBuffer(&pRingDb->inBuffers);
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		GeneralQ_PushItemToTail(&GlobalHostIfRingsDb.pendingDmaDoneQueue,(GeneralQueueItem_t *)pInBuf);
	    pDmaDesc = DmaManager_AllocateDesc();
	    pDmaDesc->destination = (uint32)dmaShRamAddr;
	    pDmaDesc->direction = DMA_DIRECTION_HOST_TO_SHRAM; 	   
	    pDmaDesc->source = dmaHostAddr;
	    pDmaDesc->length = dmaLength;
	    pDmaDesc->clientId = hostIfRingsDmaClientId;
		DmaManager_NewDmaRequest(pDmaDesc);	   
		ddrReadyHds -= numHdToCopy;
		totalHdsCopied += numHdToCopy;
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	if(totalHdsCopied != 0)
	{
		hostInterfaceIncNumOfDoneInHdsCounter(pRingDb , totalHdsCopied);
	}
}

uint16 hostInterfaceRingsHandleOutBuffer(HdRingDb_t* pRingDb,HostIfShRamBuf_t *pBufferDb)
{
	TX_INTERRUPT_SAVE_AREA;
	uint16 maxDescsToCopy = 0;
	uint16 HdsToEndRing = 0;
	uint16 dmaLength = 0;
	uint32 dmaWriteAddr;
    uint16 descsToCopy =0;
	dmaDesc_t* pDmaDesc = NULL;

	/*Get current address*/
	dmaWriteAddr = pRingDb->writeAddress;
	/*Get the max number of Descriptors we caqn copy*/
	maxDescsToCopy = pRingDb->outBuffers.maxHdsInBuffer;
	/*Calculate number of HDs to end of ring*/
	HdsToEndRing = DIVIDE_BY_20(pRingDb->ringEndAddr - pRingDb->writeAddress);
	/*If we are at end of ring*/
    if (HdsToEndRing == 0)
    {
    	/*Move to start of rings*/
    	pRingDb->writeAddress =  pRingDb->ringStartAddr;
	  	dmaWriteAddr = pRingDb->ringStartAddr;
		/*Number of HDs to end of ring is the number of HDs in the ring*/
      	HdsToEndRing = pRingDb->ringSizeHds; 
    }
	/*Check If the max number of HDs we can copy is greater than the number of available HDs till end of ring*/
	if (maxDescsToCopy >= HdsToEndRing)
	{
		/*Set maximum number of HDs we can copy to the number of HDs available till end of ring*/
		maxDescsToCopy = HdsToEndRing;
	}
	/*Get descriptors to copy - this will be written to the buffer*/
    descsToCopy = pRingDb->descsDoneListHandler((Hd_t*)pBufferDb->pBuffer, maxDescsToCopy);
	/*Any descriptors to copy*/
	if (descsToCopy > 0)
	{
        pRingDb->numOfFreeDescriptors += descsToCopy;
        pBufferDb->numHdInBuf = descsToCopy;
		dmaLength = (descsToCopy * sizeof(Hd_t));
   		pRingDb->writeAddress = pRingDb->writeAddress + dmaLength;
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		hostInterfaceRingsMoveToNextFreeBuffer(&pRingDb->outBuffers);
		/*Add buffer to pending DMA queue*/
		GeneralQ_PushItemToTail(&GlobalHostIfRingsDb.pendingDmaDoneQueue,(GeneralQueueItem_t *)pBufferDb);
		/*Send DMA request*/
	    pDmaDesc = DmaManager_AllocateDesc();
	    pDmaDesc->destination = dmaWriteAddr;
	    pDmaDesc->direction = DMA_DIRECTION_SHRAM_TO_HOST; 	   
	    pDmaDesc->source = (uint32)pBufferDb->pBuffer;
	    pDmaDesc->length = dmaLength;
	    pDmaDesc->clientId = hostIfRingsDmaClientId;
		DmaManager_NewDmaRequest(pDmaDesc);	   
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
    return (descsToCopy);
}

#ifdef DEBUG_STA_MODE_RSSI
void hostInterfaceRingsSetPowerManagementBit(Hd_t* tempHostDesc, bool PowerManagementVal)
{
	tempHostDesc->powerManagement = PowerManagementVal;
}
#endif


void hostInterfaceRingsConvertDriverHdsToPds(Hd_t* pHostDesc ,uint8 numOfHds)
{
    TxPd_t* pDesc;
    Hd_t tempHostDesc;
	uint8 hdIndex;

	for (hdIndex = 0; hdIndex < numOfHds ; hdIndex++)
    {	
    	/*Get PD from Management from Host Pool*/
        pDesc = ResourceManager_GetDescriptor(DESC_POOL_MANAGEMENT_FROM_HOST);
		ASSERT((TxPd_t *)NULL_PD != pDesc);
		/*Copy HD to local HD*/
        memcpy32(&tempHostDesc,pHostDesc,CONVERT_BYTES_TO_WORDS(sizeof(Hd_t)));
		/*Copy fields from local HD to PD*/
    	pDesc->txQTid = tempHostDesc.class;
#ifdef DEBUG_STA_MODE_RSSI
		hostInterfaceRingsSetPowerManagementBit(&tempHostDesc, PowerManagementIndication);
#endif
#ifndef ENET_INC_ARCH_WAVE600
		pDesc->powerManagement = tempHostDesc.powerManagement;
#endif
#ifdef ENET_INC_ARCH_WAVE600 
		// In gen6, the VAP ID is composed from 4 bits from vapID fields and 1 bit from EP field
    	pDesc->txQVapId = (tempHostDesc.vapId | ((tempHostDesc.ep & 0x1) << 4));
#else
		pDesc->txQVapId = tempHostDesc.vapId;
#endif
    	pDesc->dataLength = tempHostDesc.length;
    	pDesc->packetPointer = (tempHostDesc.pointerToDramWordOffset | tempHostDesc.pointerToDramByteOffset); 
    	pDesc->mcUnicast = tempHostDesc.multicastIndication;
    	pDesc->txQStaId = tempHostDesc.staId;
    	pDesc->bdIndex = tempHostDesc.bdIndex;
    	pDesc->nextPdPointer = NEXT_PD_NULL;
		/* Currently there are only management and data on these rings */
		if(tempHostDesc.type == FM_CONTROL_FTYPE_MANAGEMENT)
		{	
			/*Copy sub type to management PD*/
#ifdef ENET_INC_ARCH_WAVE600 
			pDesc->mgmtFrameSubtype = tempHostDesc.subtype;
#else
			pDesc->mgmtSubtype = tempHostDesc.subtype;
#endif
			if (tempHostDesc.subtype == MGMT_FRAME_SUBTYPE_PROBE_RES) /* Probe response */
			{
				pDesc->pdType = PD_TYPE_PROBE_RES;
				/*Set "beacon" sub type to normal - this overwrites the sub type field */
#ifdef ENET_INC_ARCH_WAVE600
				pDesc->beaconFrameSubtype= BEACON_FRAME_SUBTYPE_NORMAL;

				/*set total length since this is the last (and only) PD (used by Sender in order to execute DMA only once following errata https://jira-chd.intel.com/browse/WLANSW-7097)
				It is done here also since in Sender beacon and probe_res is a common code*/
				pDesc->beaconProbeRespTotalLength = tempHostDesc.length;
				pDesc->beaconProbeRespLastDmaActivation = TRUE;
				pDesc->beaconProbeRespActivateDma = TRUE;
				pDesc->beaconProbeRespOffsetFromPacketStart = 0x0;

#else
				pDesc->beaconProberesSubtype = PD_BEACON_PROBE_RESP_SUBTYPE_NORMAL;
#endif
				pDesc->pdCounter = 1;
			}
			else
			{
				/*If STA is MFP, Disassociate & Deauthenticast has to be transmitted with protection*/
				if ((tempHostDesc.multicastIndication == MULTICAST) ||
					 (tempHostDesc.unknownSidInd == TRUE) ||
					 (hostInterface_IsStaEnabled(tempHostDesc.staId) == FALSE) ||
					 (hostInterface_IsStaMfp(tempHostDesc.staId) == FALSE) ||
					 (hostInterface_IsRobustManagementFrame(tempHostDesc.subtype, tempHostDesc.action) == FALSE))
				{
					pDesc->pdType = PD_TYPE_MANAGEMENT_UNENC;
				}
				else
				{
					pDesc->pdType = PD_TYPE_MANAGEMENT_ENC;
				}
			}
#ifdef ENET_INC_ARCH_WAVE600
			DEBUG_ASSERT(pDesc->txQTid == MANAGEMENT_TID);
#endif
#ifdef ENET_INC_ARCH_WAVE600B		//update the unnkown Sid ind
			pDesc->unknownSidInd = tempHostDesc.unknownSidInd;

#else
			pDesc->unnkownSid = tempHostDesc.unknownSidInd;
#endif

			TxPacketsClassifier_SendManagementPacketFromHost(pDesc);
		}
		else
		{
			/* Currently there are only NDP and EAPOLs on these rings */
			if(tempHostDesc.subtype == FRAME_SUBTYPE_NULL_DATA_PACKET || tempHostDesc.subtype == FRAME_SUBTYPE_QOS_NULL_DATA_PACKET)
			{
				pDesc->pdType = PD_TYPE_NDP;
				DEBUG_ASSERT(pDesc->dataLength == 0);
			}
			else
			{
				pDesc->ethType = tempHostDesc.etype;
				pDesc->pdType = PD_TYPE_DATA;
#ifdef ENET_INC_ARCH_WAVE600 
				hostInterfaceIncrementPdByteCount(pDesc->txQTid,pDesc->txQStaId,pDesc->dataLength);					
#endif
			}
			/*Must be unicast to known STA*/
			DEBUG_ASSERT(tempHostDesc.multicastIndication == UNICAST);
			DEBUG_ASSERT(tempHostDesc.unknownSidInd == FALSE);
			/*If STA non-QoS put packet on TID 0*/
			if (hostInterface_IsStaQoS(tempHostDesc.staId) == FALSE)
			{
#ifdef ENET_INC_ARCH_WAVE600 
				// There length of this packet was increased in host-if counter per STA/TID. Since we are about to change the TID, we need to change the counters accordingly.
				hostInterfaceDecrementPdByteCount(pDesc->txQTid, pDesc->txQStaId, pDesc->dataLength);
				hostInterfaceIncrementPdByteCount(IEEE802_1D_BE_1,pDesc->txQStaId,pDesc->dataLength);					
#endif		
				pDesc->txQTid = IEEE802_1D_BE_1;
			}
			TxPacketsClassifier_SendDataPacketFromHost(pDesc);	
		}
		/*Move to next HD*/
	 	pHostDesc++;
    }
}

void hostInterfaceRingsConvertDriverHdsToRds(Hd_t* pHostDesc ,uint8 numOfHds)
{
    Rd_t * pDesc;
    rxDescriptorQueue_t rdList;
	uint8 hdIndex;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    Hd_t tempHostDesc;
	
    rdList.pHead = (Rd_t *)NULL_RD;
    rdList.pTail = (Rd_t *)NULL_RD;
	/*Loop thru returned HDs*/
	for (hdIndex = 0; hdIndex < numOfHds ; hdIndex++)
    {
        /*Get RD from pending list*/
        pDesc = hostInterfaceRingsGetRdFromList(&driverRdsTempList);
		ASSERT(pDesc != (Rd_t *)NULL_RD);
		memcpy32(&tempHostDesc, pHostDesc, CONVERT_BYTES_TO_WORDS(sizeof(Hd_t)));
		/*Copy new DRAM pointer and BD index to RD*/
        pDesc->dramPointer = CONVERT_BYTE_ADDRESS_TO_DRAM_POINTER(tempHostDesc.pointerToDramWordOffset);
        pDesc->bdIndex = tempHostDesc.bdIndex;

		hostInterfaceRings_PutOnRdsList(pDesc, pDesc, &rdList);
		/*Move to next Host Descriptor*/
        pHostDesc++;


    }
	/*Move RDs to liberator*/
	ASSERT(rdList.pHead != (Rd_t *)NULL_RD);
    hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
    hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
    hwQueueManagerRequestParams.pHeadDesc = rdList.pHead; 
    hwQueueManagerRequestParams.pTailDesc= rdList.pTail;
    HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
}

uint16 hostInterfaceRingsConvertDriverRdsToHds(Hd_t* pHostDesc ,uint8 numOfHds)
{
	uint16 rd2HdCounter = 0;
    Rd_t * rxDescriptor = (Rd_t *)NULL_RD;
    Hd_t tempHostDesc;	
	BaaCounters_t *pBaaCounters = (BaaCounters_t *)statisticsGetBaaCountersAddress();

	/*Loop while there is room in ring and done RDs*/
    while ((rd2HdCounter < numOfHds)&&(driverRdsDoneList.pHead != (Rd_t *)NULL_RD))
    {   
    	/*get RD from Done List*/
		rxDescriptor = hostInterfaceRingsGetRdFromList(&driverRdsDoneList);
		/*In DUT mode need to intercept RDs here*/
		if(hostInterfaceCheckDutMulticastManagement(rxDescriptor) == TRUE)
		{
			continue;
		}
		/*Copy fields from RD to local HD*/
	    tempHostDesc.class = rxDescriptor->rxQueueTid;
#ifdef ENET_INC_ARCH_WAVE600 
		// In gen6, the VAP ID is splitted to 4 bits in vapID fields and 1 bit in EP field
        tempHostDesc.vapId = (rxDescriptor->rxQueueVapId & 0xF);
		tempHostDesc.ep = ((rxDescriptor->rxQueueVapId >> 4) & 0x1);
#else
        tempHostDesc.vapId = rxDescriptor->rxQueueVapId;
#endif
        tempHostDesc.length = rxDescriptor->dataLength;
        tempHostDesc.pointerToDramWordOffset = CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer)|GET_BIT4_IN_RD_OFFSET(rxDescriptor->dramByteOffset);
        tempHostDesc.staId = rxDescriptor->rxQueueStaId;
        tempHostDesc.subtype = rxDescriptor->frameSubtype;
        tempHostDesc.type = rxDescriptor->frameType;
		tempHostDesc.etype = rxDescriptor->ethType;
        tempHostDesc.bdIndex = rxDescriptor->bdIndex;
		tempHostDesc.pointerToDramByteOffset = CONVERT_RD_OFFSET_TO_HD_OFFSET(rxDescriptor->dramByteOffset);
		tempHostDesc.eop = rxDescriptor->eop;
		tempHostDesc.sop = rxDescriptor->sop;
#ifdef ENET_INC_ARCH_WAVE600 
		tempHostDesc.own = 1;
#else
		tempHostDesc.Own = 1;
#endif

		tempHostDesc.multicastIndication = rxDescriptor->multicast;
		if ((rxDescriptor->frameType == FM_CONTROL_FTYPE_MANAGEMENT) && ((rxDescriptor->frameSubtype == MGMT_FRAME_SUBTYPE_BEACON) || (rxDescriptor->frameSubtype == MGMT_FRAME_SUBTYPE_PROBE_RES)))
		{
			pBaaCounters->beaconProbeResponsePhyType[ConfigurationManager_GetBandForVap(rxDescriptor->rxQueueVapId)]++;

		}
#if defined (AAA_RINGS_DEBUG)
		mangMcf[rxDescriptor->multicast]++;
		AAA_RINGS_SUBTYPE[tempHostDesc.type][tempHostDesc.subtype]++;
#endif		
		/*Copy local HD to HD*/
	    memcpy32(pHostDesc,&tempHostDesc,CONVERT_BYTES_TO_WORDS(sizeof(Hd_t)));
		/*Move to next HD*/
        pHostDesc++;
		/*Increment number of HDs handles*/
        rd2HdCounter++;        

		/*Put RD to pending list*/
	    hostInterfaceRings_PutOnRdsList(rxDescriptor,rxDescriptor,&driverRdsTempList);

    }         
	/*Return number of HDs copied to*/
	return (rd2HdCounter);
}

uint16 hostInterfaceRingsConvertDriverPdsToHds(Hd_t* pHostDesc ,uint8 numOfHds)
{
    uint16 pd2HdCounter = 0;
    TxPd_t *packetDescriptor = (TxPd_t*)NULL_PD;
    Hd_t tempHostDesc;

	/*Loop while there is room in ring and done PDs*/
    while ((pd2HdCounter < numOfHds)&&(driverPdsDoneList.pHead != (TxPd_t*)NULL_PD))
    {  
    	/*Get PD from done list*/
		packetDescriptor = hostInterfaceRingsGetPdFromList(&driverPdsDoneList);
		/*Increment number of HDs copied to*/
        pd2HdCounter++; 
		/*Copy fields from PD to local HD*/
        tempHostDesc.class = packetDescriptor->txQTid;
#ifdef ENET_INC_ARCH_WAVE600 
		// In gen6, the VAP ID is splitted to 4 bits in vapID fields and 1 bit in EP field
        tempHostDesc.vapId = (packetDescriptor->txQVapId & 0xF);
		tempHostDesc.ep = ((packetDescriptor->txQVapId >> 4) & 0x1);
#else
		tempHostDesc.vapId = packetDescriptor->txQVapId;
#endif
        tempHostDesc.length = packetDescriptor->dataLength;
        tempHostDesc.pointerToDramWordOffset = packetDescriptor->packetPointer & 0x0FFFFFF;
		tempHostDesc.pointerToDramByteOffset = (tempHostDesc.pointerToDramWordOffset & (0x7));
		tempHostDesc.pointerToDramWordOffset = (tempHostDesc.pointerToDramWordOffset & (~(0x7)));
        tempHostDesc.multicastIndication = packetDescriptor->mcUnicast;
        tempHostDesc.staId = packetDescriptor->txQStaId;
		/*Don't check retry bit - if packet was never transmitted and status is ACK it means that we flushed it either due to STA or VAP removal so
		we don't care ... */
		tempHostDesc.status = (packetDescriptor->status == PD_STATUS_ACK_RECEIVED);
		if (packetDescriptor->pdType == PD_TYPE_PROBE_RES)
		{
			/*For Probe RSP we overwrote the sub type*/
			tempHostDesc.subtype = MGMT_FRAME_SUBTYPE_PROBE_RES;
			tempHostDesc.type = FM_CONTROL_FTYPE_MANAGEMENT;
		}
		else if (packetDescriptor->pdType == PD_TYPE_DATA)
		{
			if (hostInterface_IsStaQoS(tempHostDesc.staId))
			{
        		tempHostDesc.subtype = FRAME_SUBTYPE_QOS_DATA_PACKET;
			}
			else
			{
				tempHostDesc.subtype = FRAME_SUBTYPE_DATA_PACKET;
			}
		 	tempHostDesc.type = FM_CONTROL_FTYPE_DATA;

#ifdef ENET_INC_ARCH_WAVE600 
			hostInterfaceDecrementPdByteCount(packetDescriptor->txQTid, packetDescriptor->txQStaId, packetDescriptor->dataLength);
#endif
		}
		else if (packetDescriptor->pdType == PD_TYPE_NDP)
		{
			if (hostInterface_IsStaQoS(tempHostDesc.staId))
		  	{
          		tempHostDesc.subtype = FRAME_SUBTYPE_QOS_NULL_DATA_PACKET;
		  	}
			else
			{
				tempHostDesc.subtype = FRAME_SUBTYPE_NULL_DATA_PACKET;
			}
			tempHostDesc.type = FM_CONTROL_FTYPE_DATA;
		}
		else
		{
			/*All Other PD types - use sub type from PD*/
#ifdef ENET_INC_ARCH_WAVE600 
			tempHostDesc.subtype = packetDescriptor->mgmtFrameSubtype;
#else
			tempHostDesc.subtype = packetDescriptor->mgmtSubtype;
#endif
			tempHostDesc.type = FM_CONTROL_FTYPE_MANAGEMENT;
		}
        tempHostDesc.bdIndex = packetDescriptor->bdIndex;
#ifdef ENET_INC_ARCH_WAVE600B		//update the unnkown Sid ind
		tempHostDesc.unknownSidInd = packetDescriptor->unknownSidInd;	
#else
		tempHostDesc.unknownSidInd = packetDescriptor->unnkownSid;
#endif

#if defined (AAA_RINGS_DEBUG)
		mangRetryCount[packetDescriptor->retryCount]++;
#endif
		/*Copy local HD to HD*/
        memcpy32(pHostDesc,&tempHostDesc,CONVERT_BYTES_TO_WORDS(sizeof(Hd_t)));  
		/*Move to next HD*/
        pHostDesc++;
		/*We are done with PD - release it*/
        ResourceManager_ReleaseDescriptor(packetDescriptor,DESC_POOL_MANAGEMENT_FROM_HOST);
    }         
	/*Return number of HDs copied to*/
    return (pd2HdCounter);
}

#ifdef ENET_INC_ARCH_WAVE600 

void hostInterfaceIncrementPdByteCount(uint8 tid, StaId staId, uint16 byteCount)
{
	volatile RegHostIfAccTxPdByteCountUpdate_u regHostIfAccTxPdByteCountUpdate;
#ifdef HOST_IF_COUNTERS_WORKAROUND
	volatile HostInterface_ScratchPadApiParams_t * pHostInterface_ScratchPadApiParams = (HostInterface_ScratchPadApiParams_t *) (MAC_GENRISC_HOST_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_HOST_INTERFACE_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	uint8	dummyParam = 0; 
#endif
	regHostIfAccTxPdByteCountUpdate.val = 0;

	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaTidByteCountNum = byteCount;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdTidNum = tid;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaNum = staId;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaTidByteCountInc = TRUE;

#ifdef HOST_IF_COUNTERS_WORKAROUND
	while(1)
	{
		pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 1;
		// Klocwork: "countersSemaphoreGenrisc" is a Semaphore set in Genrisc, klocwork will give error as "TAINTED.LOOP_BOUND:" ignore it.
		if(pHostInterface_ScratchPadApiParams->countersSemaphoreGenrisc == 1)
		{
			pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 0;
			// next line is only in order to "waste" time & not accessing the SCPAD continuously
			dummyParam = 1;
		}
		else
		{ // semaphore is taken by FW
			break; 
		}
	}
#endif // HOST_IF_COUNTERS_WORKAROUND

	RegAccess_Write(REG_HOST_IF_ACC_TX_PD_BYTE_COUNT_UPDATE,regHostIfAccTxPdByteCountUpdate.val);
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_TX_PD_BYTE_COUNT_UPDATE,&regHostIfAccTxPdByteCountUpdate.val);
	}while (regHostIfAccTxPdByteCountUpdate.bitFields.txPdUpdateDone == FALSE);

#ifdef HOST_IF_COUNTERS_WORKAROUND	
	pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 0;
#endif // HOST_IF_COUNTERS_WORKAROUND	
}

void hostInterfaceDecrementPdByteCount(uint8 tid, StaId staId, uint16 byteCount)
{
	volatile RegHostIfAccTxPdByteCountUpdate_u regHostIfAccTxPdByteCountUpdate;

#ifdef HOST_IF_COUNTERS_WORKAROUND
	volatile HostInterface_ScratchPadApiParams_t * pHostInterface_ScratchPadApiParams = (HostInterface_ScratchPadApiParams_t *) (MAC_GENRISC_HOST_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_HOST_INTERFACE_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	uint8	dummyParam = 0; 
#endif

	regHostIfAccTxPdByteCountUpdate.val = 0;

	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaTidByteCountNum = byteCount;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdTidNum = tid;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaNum = staId;
	regHostIfAccTxPdByteCountUpdate.bitFields.txPdStaTidByteCountInc = FALSE;
#ifdef HOST_IF_COUNTERS_WORKAROUND
	while(1)
	{
		pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 1;
		// Klocwork: "countersSemaphoreGenrisc" is a Semaphore set in Genrisc, klocwork will give error as "TAINTED.LOOP_BOUND:" ignore it.
		if(pHostInterface_ScratchPadApiParams->countersSemaphoreGenrisc == 1)
		{
			pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 0;
			// next line is only in order to "waste" time & not accessing the SCPAD continuously
			dummyParam = 1;
		}
		else
		{ // semaphore is taken by FW
			break; 
		}
	}
#endif // HOST_IF_COUNTERS_WORKAROUND

	RegAccess_Write(REG_HOST_IF_ACC_TX_PD_BYTE_COUNT_UPDATE,regHostIfAccTxPdByteCountUpdate.val);

	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_TX_PD_BYTE_COUNT_UPDATE,&regHostIfAccTxPdByteCountUpdate.val);
	}while (regHostIfAccTxPdByteCountUpdate.bitFields.txPdUpdateDone == FALSE);

#ifdef HOST_IF_COUNTERS_WORKAROUND	
	pHostInterface_ScratchPadApiParams->countersSemaphoreFw = 0;
#endif // HOST_IF_COUNTERS_WORKAROUND
}
#endif // ENET_INC_ARCH_WAVE600

Rd_t * hostInterfaceRingsGetRdFromList(rxDescriptorQueue_t* pRdList)
{
	TX_INTERRUPT_SAVE_AREA;
	Rd_t * pRxDesc = (Rd_t *)NULL_RD;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
    if (pRdList->pHead != (Rd_t *)NULL_RD) 	
	{
	    pRxDesc = pRdList->pHead;
	    pRdList->pHead =(Rd_t *)GET_NEXT_RD(pRxDesc);
	    pRxDesc->nextRd = NEXT_RD_NULL;
		pRxDesc->tailRd = SET_NEXT_RD(pRxDesc);
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	return (pRxDesc);
}

TxPd_t* hostInterfaceRingsGetPdFromList(PacketDescriptorQueue_t* pPdList)
{
	TX_INTERRUPT_SAVE_AREA;
	TxPd_t* pPd = (TxPd_t*)NULL_PD;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
    if (pPdList->pHead !=(TxPd_t*)NULL_PD) 	
	{
	    pPd = pPdList->pHead;
	    pPdList->pHead =(TxPd_t*)GET_NEXT_PD(pPd);
	    pPd->nextPdPointer = NEXT_PD_NULL;
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	return (pPd);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
void hostInterfaceRings_DoneListsHandler(K_MSG *psMsg)
{
	UNUSED_PARAM(psMsg);
	EventManager_TurnOnEvent(EVENT_ID_DESCRIPTORS_PENDING_FOR_HOST_IF);
	hostInterfaceCheckForNewDesc();
}

void hostInterfaceRingsNewDesc(HdRingDb_t *pRingDb)
{
    bool doneListStatus;
	HostIfShRamBuf_t* pBufferDb = NULL;
    uint8 newFreeDescs = 0;

    doneListStatus = TRUE;
    while ((pRingDb->outBuffers.numOfFreeBuffers > 0)&&(doneListStatus == TRUE))
    {
    	pBufferDb = hostInterfaceRingsGetFreeBufferAddr(&pRingDb->outBuffers);// get free buffer address
		newFreeDescs = hostInterfaceRingsHandleOutBuffer(pRingDb,pBufferDb);  //check for more PDs on the PD-Done-list
        if (newFreeDescs == 0)
        {
        	doneListStatus = FALSE; // no more PDs wating in the done list
        }
    }
}

void hostInterfaceRings_HandleDmaComplete(K_MSG *psMsg)
{
	TX_INTERRUPT_SAVE_AREA;
	HostIfShRamBuf_t* pBufferDb = NULL;
	uint32 	dmaTransferCount = 0;
	uint8 ringIndex;
	uint8 ringDirection;
	UNUSED_PARAM(psMsg);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	dmaTransferCount = DmaManager_GetTransferCount(hostIfRingsDmaClientId);
	while(GlobalHostIfRingsDb.dmaTransferCount != dmaTransferCount)
	{
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		pBufferDb = (HostIfShRamBuf_t *)GeneralQ_PopItemFromHead(&GlobalHostIfRingsDb.pendingDmaDoneQueue);
		DEBUG_ASSERT(pBufferDb);
		ringIndex = pBufferDb->ringIndex;
		ringDirection = pBufferDb->ringDirection;
		if (ringDirection == HOST_INTERFACE_RING_DIRECTION_OUT)
		{
			hostInterfaceIncNumOfDoneOutHdsCounter(pRingsDb[ringIndex] , pBufferDb->numHdInBuf);
			hostInterfaceRingsMoveToNextFullBuffer(&(pRingsDb[ringIndex]->outBuffers));// free the buffer
		}
		else
		{
			pRingsDb[ringIndex]->inDmaDoneHandler((Hd_t*)pBufferDb->pBuffer,pBufferDb->numHdInBuf);
			hostInterfaceRingsMoveToNextFullBuffer(&(pRingsDb[ringIndex]->inBuffers));
		}
		GlobalHostIfRingsDb.dmaTransferCount++;
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		dmaTransferCount = DmaManager_GetTransferCount(hostIfRingsDmaClientId);
	}
	EventManager_TurnOnEvent(EVENT_ID_DMA_HOST_IF_DONE);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	// check for pending Descriptors
	hostInterfaceCheckForNewDesc();
	// check for pending Hds in the ring
	hostInterfaceCheckForNewHdsOnDmaDone();
}

void hostInterfaceRings_PutOnRdsList(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor,rxDescriptorQueue_t* pRdList)
{
	TX_INTERRUPT_SAVE_AREA;

	DEBUG_ASSERT((Rd_t  *)NULL_RD != headRxDescriptor);
	DEBUG_ASSERT((Rd_t  *)NULL_RD != tailRxDescriptor);
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	if (pRdList->pHead != (Rd_t *)NULL_RD)
	{
		pRdList->pTail->nextRd = SET_NEXT_RD(headRxDescriptor);
	}
	else
	{
		pRdList->pHead = headRxDescriptor;
	}
	pRdList->pTail = tailRxDescriptor;
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

bool hostInterface_IsStaOpen(StaId sid)
{
	if (HostInterfaceStaDb[sid].open == TRUE)
	{
		return (TRUE);
	}
	return (FALSE);
}

bool hostInterface_IsStaQoS(StaId sid)
{
	uint8 isStaQos = MTLK_BFIELD_GET(HostInterfaceStaDb[sid].flags, STA_ADD_FLAGS_WME);
	return (isStaQos);
}

bool hostInterface_IsStaTidAcmAllowed(StaId sid, uint8 tid)
{
	uint8 ac;
	ac = au8Convert8021dToWmeUserPriority[tid];
	if (HostInterfaceStaDb[sid].acm_flags == ACM_ALL_ACS_ALLOWED)
	{
		return (TRUE);
	}
	if (!(HostInterfaceStaDb[sid].acm_flags & (1 << ac)))
	{
		return (TRUE);
	}
	return (FALSE);
}

void hostInterfaceRings_PutOnPdsList(TxPd_t* pPacketDescHead , TxPd_t* pPacketDescTail,PacketDescriptorQueue_t* pPdList)
{
	TX_INTERRUPT_SAVE_AREA;

	ASSERT((TxPd_t *)NULL_PD != pPacketDescHead);
	ASSERT((TxPd_t *)NULL_PD != pPacketDescTail);
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	if (pPdList->pHead != (TxPd_t*)NULL_PD)
	{
		pPdList->pTail->nextPdPointer = SET_NEXT_PD(pPacketDescHead);
	}
	else
	{
		pPdList->pHead = pPacketDescHead;
	}
	pPdList->pTail = pPacketDescTail;
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

// convert TID to AC
const uint8 au8Convert8021dToWmeUserPriority[NUM_802_1D_FRAME_PRIORITIES] = {
    ACCESS_CATEGORY_BE, // 0 IEEE802_1D_BK_1 1 BK (Lowest)
    ACCESS_CATEGORY_BK, // 1 IEEE802_1D_BK_2 2 -
    ACCESS_CATEGORY_BK, // 1 IEEE802_1D_BE_1 0 BE
    ACCESS_CATEGORY_BE, // 0 IEEE802_1D_BE_2 3 EE
    ACCESS_CATEGORY_VI, // 2 IEEE802_1D_VI_1 4 CL
    ACCESS_CATEGORY_VI, // 2 IEEE802_1D_VI_2 5 VI
    ACCESS_CATEGORY_VO, // 3 IEEE802_1D_VO_1 6 VO
    ACCESS_CATEGORY_VO  // 3 IEEE802_1D_VO_2 7 NC (Highest)
};

