/***********************************************************************************
 File:		HostInterfaceAcc.c
 Module:		Host Interface
 Purpose: 		
 Description:	
************************************************************************************/
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HostAccCode.h"
#include "HostIfAccRegs.h"
#include "stringLibApi.h"
#include "PacketDescriptor.h"
#include "RegAccess_Api.h"
#include "HostGenRiscInterface_Api.h"
#include "HwQManager_API.h"
#include "shram_dataHostIf.h"
#include "HostInterfaceAcc_Api.h"
#include "RxDescriptor.h"
#include "loggerAPI.h"
#include "ConfigurationManager_api.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_HOST_IF
#define LOG_LOCAL_FID 7

#define TX_IN_HOST_IF_FW_CODE		(HOST_IF_ACC_FW_CODE_RAM + 0x000)
#define TX_OUT_HOST_IF_FW_CODE  	(HOST_IF_ACC_FW_CODE_RAM + 0x100)
#define RX_IN_HOST_IF_FW_CODE   	(HOST_IF_ACC_FW_CODE_RAM + 0x200)
#define RX_OUT_HOST_IF_FW_CODE  	(HOST_IF_ACC_FW_CODE_RAM + 0x300)
#define TX_IN_FIXED_VALUES 			(TX_IN_HOST_IF_FW_CODE	 + 0xC4) //0xC0 is reserved for TSF, configurable fixes values start 0xC4
#define TX_OUT_DATA_PATH_PORT_VALUE (TX_OUT_HOST_IF_FW_CODE	 + 0xC4) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define TX_OUT_DW1_FIXED_VALUES 	(TX_OUT_HOST_IF_FW_CODE	 + 0xC8) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define TX_OUT_FIXED_VALUES 		(TX_OUT_HOST_IF_FW_CODE	 + 0xCC) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define RX_IN_FIXED_VALUES 			(RX_IN_HOST_IF_FW_CODE	 + 0xC4) //0xC0 is reserved for TSF, configurable fixes values start 0xC4
#define RX_OUT_DATA_PATH_PORT_VALUE (RX_OUT_HOST_IF_FW_CODE	 + 0xC4) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define RX_OUT_DW3_FIXED_VALUES 	(RX_OUT_HOST_IF_FW_CODE	 + 0xC8) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define RX_OUT_FIXED_VALUES 		(RX_OUT_HOST_IF_FW_CODE	 + 0xCC) //0xC0 is reserved for TSF, configurable fixes values start 0xCC
#define DIST_BITMAP_ADDRESS     	(HOST_IF_ACC_DIST_RAM)
#define UNICAST_BITMAP_SIZE     	(HW_NUM_OF_STATIONS * NUM_OF_TOTAL_TIDS)
#define VAPS_BITMAP_SIZE		 	(HW_NUM_OF_VAPS)
#define GROUPS_BITMAP_SIZE       	(128)

/* TX QID filed definitions*/
#define INDEX_TID_MASK_1			(0x7) //Mask for 3 first bits of TID - TID in total has 4 bits
#define INDEX_TID_MASK_2			(0x8) // Mask for the forth bit of TID 
#define INDEX_STA_ID_OFFSET			(0x3)
#define INDEX_LAST_TID_BIT_SHIFT	(0xB - 0x3) // Offset for the last bit of TID which is written after Station Id
#define VAP_ID_MASK					(0x1F)
#define GROUP_ID_MASK				(0x7F)

#define HD_DSCPTR_LENGTH_LINE_NUM	(0x3)
#define HD_DSCPTR_LENGTH_BIT_OFFSET	(0x0)

#define RX_FREE_LIST_THRESHOLD		(4096) //minimum RD free list threshold





typedef enum 
{
	HOST_IF_ACC_START_CORE = 0,
	HOST_IF_ACC_HALT_CORE,
}HostIFAccCoreConfig_e;

typedef enum 
{
	HOST_IF_ACC_NO_SWAP_MODE = 0,
	HOST_IF_ACC_SWAP_MODE ,
}HostIfAccSwapMode_e;

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void hostIfAccLoadTxInCode(DataPathMode_e dataPthMode);
static void hostIfAccLoadTxOutCode(DataPathMode_e dataPthMode, uint8 dataPathPort, uint32 txOutDw1FixedValues);
static void hostIfAccLoadRxInCode(DataPathMode_e dataPthMode);
static void hostIfAccLoadRxOutCode(DataPathMode_e dataPthMode, uint8 dataPathPort, uint32 rxOutDw3FixedValues);

RegHostIfAccHostCountersCfg_u debug_counterControlRegister;
RegHostIfAccTxInMulticastCfg_u debug_multicastTxInRegister;
RegHostIfAccRxOutMulticastCfg_u debug_multicastRxOutRegister;

//---------------------------------------------------------------------------------
//						Static Function Definitions				
//---------------------------------------------------------------------------------
static void hostIfAccLoadTxInCode(DataPathMode_e dataPthMode)
{
	switch (dataPthMode)
	{
		case DATA_PATH_MODE_DC_MODE_1:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void *)PumaPathTxInAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxInAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void *)PumaPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxInAccCode)));
			}
			break;
		}
#ifdef ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_MODE_2:
		{
			memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void *)LgmPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(LgmPathTxInAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_3:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)FlmPathTxInAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(FlmPathTxInAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)FlmPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FlmPathTxInAccCode)));
			}
			break;
		}
#endif //ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_NONE:
		{
			memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)SwPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathTxInAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_0:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)FastPathTxInAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxInAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)TX_IN_HOST_IF_FW_CODE, (void*)FastPathTxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxInAccCode)));
			}
			break;
		}
		default:
			FATAL("hostIfAccLoadTxInCode, DC_MODE not supported");
	}
			
	memcpy32((void *)TX_IN_FIXED_VALUES, (void*)TxInAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(TxInAccFixedValues)));
}

static void hostIfAccLoadTxOutCode(DataPathMode_e dataPthMode, uint8 dataPathPort, uint32 txOutDw1FixedValues)
{
	uint32 dataPathPortWord = dataPathPort;

	switch (dataPthMode)
	{
	
		case DATA_PATH_MODE_DC_MODE_1:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void *)PumaPathTxOutAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxOutAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void *)PumaPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathTxOutAccCode)));
			}
			break;
		}	
#ifdef ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_MODE_2:
		{
			memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void *)LgmPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(LgmPathTxOutAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_3:
		{
			memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void *)FlmPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FlmPathTxOutAccCode)));
			break;
		}
#endif //ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_NONE:
		{
			memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)SwPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathTxOutAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_0:
		{
#ifdef HOST_IF_COUNTERS_WORKAROUND
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)FastPathTxOutAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxOutAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)FastPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxOutAccCode)));
			}
#else		
			memcpy32((void *)TX_OUT_HOST_IF_FW_CODE, (void*)FastPathTxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathTxOutAccCode)));
#endif //HOST_IF_COUNTERS_WORKAROUND
			break;
		}

		default:
			FATAL("hostIfAccLoadTxOutCode, DC_MODE not supported");			
	}
	
	//copy dataPathPort value to host i/f acc ram
	memcpy32((void *)TX_OUT_DATA_PATH_PORT_VALUE, (void*)&dataPathPortWord, CONVERT_BYTES_TO_WORDS(sizeof(dataPathPortWord)));
	//copy txOutDw1FixedValues value to host i/f acc ram
	memcpy32((void *)TX_OUT_DW1_FIXED_VALUES, (void*)&txOutDw1FixedValues, CONVERT_BYTES_TO_WORDS(sizeof(txOutDw1FixedValues)));
	//copy fixed values to host i/f acc ram
	memcpy32((void *)TX_OUT_FIXED_VALUES,   (void*)TxOutAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(TxOutAccFixedValues)));
}

static void hostIfAccLoadRxInCode(DataPathMode_e dataPthMode)
{
	switch (dataPthMode)
	{	
		case DATA_PATH_MODE_DC_MODE_1:
		{
			memcpy32((void *)RX_IN_HOST_IF_FW_CODE, (void *)PumaPathRxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathRxInAccCode)));
			break;
		}
#ifdef ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_MODE_2:
		{
			memcpy32((void *)RX_IN_HOST_IF_FW_CODE, (void *)LgmPathRxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(LgmPathRxInAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_3:
			//no code for FLM, no Rx IN ACC in this case
			break;
#endif // ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_NONE:
		{
			memcpy32((void *)RX_IN_HOST_IF_FW_CODE, (void*)SwPathRxInAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathRxInAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_0:
			//no code for Fast path, no Rx IN ACC in this case
			break;

		default:
			FATAL("hostIfAccLoadRxInCode, DC_MODE not supported");
	}
		memcpy32((void *)RX_IN_FIXED_VALUES, (void*)RxInAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(RxInAccFixedValues)));	
}

static void hostIfAccLoadRxOutCode(DataPathMode_e dataPthMode, uint8 dataPathPort, uint32 rxOutDw3FixedValues)
{	
	uint32 dataPathPortWord = dataPathPort;

	switch (dataPthMode)
	{		
		case DATA_PATH_MODE_DC_MODE_1:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void *)PumaPathRxOutAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathRxOutAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void *)PumaPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(PumaPathRxOutAccCode)));
			}
			break;
		}
#ifdef ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_MODE_2:
		{
			memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void *)LgmPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(LgmPathRxOutAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_3:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)FlmPathRxOutAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(FlmPathRxOutAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)FlmPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FlmPathRxOutAccCode)));
			}
			break;
		}
#endif // ENET_INC_ARCH_WAVE600B
		case DATA_PATH_MODE_DC_NONE:
		{
			memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)SwPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(SwPathRxOutAccCode)));
			break;
		}
		case DATA_PATH_MODE_DC_MODE_0:
		{
			if (CONFIGURATION_MODE_DUAL_BAND == ConfigurationManager_GetBandConfigurationMode())
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)FastPathRxOutAccCodeCDB, CONVERT_BYTES_TO_WORDS(sizeof(FastPathRxOutAccCodeCDB)));
			}
			else
			{
				memcpy32((void *)RX_OUT_HOST_IF_FW_CODE, (void*)FastPathRxOutAccCode, CONVERT_BYTES_TO_WORDS(sizeof(FastPathRxOutAccCode)));
			}
			break;
		}
		default:
			FATAL("hostIfAccLoadRxOutCode, DC_MODE not supported");
	}
	
	//copy dataPathPort value to host i/f acc ram
	memcpy32((void *)RX_OUT_DATA_PATH_PORT_VALUE, (void*)&dataPathPortWord, CONVERT_BYTES_TO_WORDS(sizeof(dataPathPortWord)));	
	//copy rxOutDw3FixedValues value to host i/f acc ram
	memcpy32((void *)RX_OUT_DW3_FIXED_VALUES, (void*)&rxOutDw3FixedValues, CONVERT_BYTES_TO_WORDS(sizeof(rxOutDw3FixedValues)));
	//copy fixed values to host i/f acc ram
	memcpy32((void *)RX_OUT_FIXED_VALUES, (void*)RxOutAccFixedValues, CONVERT_BYTES_TO_WORDS(sizeof(RxOutAccFixedValues)));
}

//---------------------------------------------------------------------------------
//						Public Function Definitions				
//---------------------------------------------------------------------------------
void HostIfAcc_SetMcastDistMode(McastDistMode_e newMcastMode)
{
#if !defined(TX_DATA_UM_PATH) && !defined(HOST_IF_DIST_TYPE_MULTICAST_VAP_WORKAROUND)
	RegHostIfAccMulticastPdDistribCfg_u mcastDistCfg;

	mcastDistCfg.val = 0;
	RegAccess_Read(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,&mcastDistCfg.val);
	mcastDistCfg.bitFields.groupVapPrio = newMcastMode;
	RegAccess_Write(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,mcastDistCfg.val);
#else
	UNUSED_PARAM(newMcastMode);	
#endif
}

static void hostIfAcc_changePdDist(PdDist_e distValue , DistType_e distType , uint16 index)
{
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	pdDistCfg.bitFields.pdDistAddr = index;
	pdDistCfg.bitFields.pdDistType = distType;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
}

void hostIfAccChangeStaTidPdDist(PdDist_e distValue ,StaId staId,uint8 TID)
{
	// index parameter is copied to pdDistAddr field in REG_HOST_IF_ACC_PD_DISTRIB_UPDATE register and is constructed 
	// as follows:
	// TID is 4 bits long. TID bits 2:0 copied to bits 2:0 of pdDistAddr
	// staId is 8 bits long and is copied to bits 10:3 in pdDistAddr. 
	// the last bit of TID is copied to bit 11 in pdDistAddr
	// TID bits are splitted due to HW design of memory
	uint16 index = 0;

	index =  (TID&INDEX_TID_MASK_1)|(staId << INDEX_STA_ID_OFFSET)|((TID&INDEX_TID_MASK_2) << INDEX_LAST_TID_BIT_SHIFT);  
	hostIfAcc_changePdDist(distValue,DIST_TYPE_UNICAST,index); 
}

void HostIfAcc_ChangeStaAllTidPdDist(PdDist_e distValue ,StaId staId)
{
#ifndef TX_DATA_UM_PATH
	uint16 tidIndex; // 3 lsb - TID, 7 msb - staId

	for(tidIndex  = 0; tidIndex < NUM_OF_TID; tidIndex++)
	{
		hostIfAccChangeStaTidPdDist(distValue,staId, tidIndex); 
	}
#endif
}

void HostIfAcc_ChangeVapMcastPdDist(PdDist_e distValue ,uint16 vapId)
{
#if !defined(TX_DATA_UM_PATH) && !defined(HOST_IF_DIST_TYPE_MULTICAST_VAP_WORKAROUND)
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	// VAP id is set to bit 4:0 of pdDistAddr the rest fo the bits must be zero
	pdDistCfg.bitFields.pdDistAddr = (vapId & VAP_ID_MASK);
	pdDistCfg.bitFields.pdDistType = DIST_TYPE_MULTICAST_VAP;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
#else
	UNUSED_PARAM(distValue);	
	UNUSED_PARAM(vapId);
#endif
}

void HostIfAcc_ChangeMcastGroupPdDist(PdDist_e distValue ,uint16 groupId)
{
	RegHostIfAccPdDistribUpdate_u pdDistCfg;

	pdDistCfg.val = 0;
	// Group id is set to bit 6:0 of pdDistAddr the rest fo the bits must be zero
	pdDistCfg.bitFields.pdDistAddr = (groupId & GROUP_ID_MASK);
	pdDistCfg.bitFields.pdDistType = DIST_TYPE_MULTICAST_GROUP;
	pdDistCfg.bitFields.pdDistBitValue = distValue;
	RegAccess_Write(REG_HOST_IF_ACC_PD_DISTRIB_UPDATE,pdDistCfg.val);
}

/**********************************************************************************

HostIfAcc_ConfigureRingsSize 

Description:
------------
	Configure the Host IF Acc with Ring size -> The Acc uses this vali=ue to calucatre how many HD
	to forward towards GenRisc so it will not exceeds Ring end.
Input:
-----
	dataPathInitParams - Int params Structuer recieved from Driver

Output:
-------
	None

**********************************************************************************/
void HostIfAcc_ConfigureRingsSize(UMI_DATA_PATH_INIT_PARAMS* dataPathInitParams)
{
	RegHostIfAccTxRingCfg_u	txRingCfgReg;
	RegHostIfAccRxRingCfg_u rxRingCfgReg;
    RegHostIfAccTxInConvCfg_u txInCfg;
    RegHostIfAccTxOutConvCfg_u txOutCfg;
	RegHostIfAccRxInConvCfg_u  rxInCfg;
	RegHostIfAccRxOutConvCfg_u rxOutCfg;
	RegHostIfAccHostType_u hostTypeReg;
	RegHostIfAccOwnBitValue_u ownBitValueReg;
	RegHostIfAccTxInSwInstruction_u txInSwInstrReg;
	RegHostIfAccTxOutSwInstruction_u txOutSwInstrReg;
	RegHostIfAccRxInSwInstruction_u rxInSwInstrReg;
	RegHostIfAccRxOutSwInstruction_u rxOutSwInstrReg;	
	RegHostIfAccFreeRdInit_u freeRdInitReg;	
	RegHostIfAccRdThreshold_u rdThreshodldReg;
	HostIfDataPathHdSize_t *hostIfDataHdSize;
	DataPathMode_e dataPathMode;
	uint32	rxOutDw3FixedValues; //used for LGM & FLM
	uint32	txOutDw1FixedValues; //used for LGM & FLM	
	uint8 dataPathPort;
	uint8 swapMode;
	uint8 rxInSwHalt;
	bool isGswip = FALSE;


	dataPathMode = (DataPathMode_e)dataPathInitParams->dataPathMode;
	dataPathPort = dataPathInitParams->dataPathPort;
	rxOutDw3FixedValues = dataPathInitParams->rxOutDw3FixedValues;
	txOutDw1FixedValues = dataPathInitParams->txOutDw1FixedValues;
	hostIfDataHdSize = HostInterfaceRings_GetHdSizes(dataPathMode);

	freeRdInitReg.val = 0;
	rdThreshodldReg.val = 0;
	
	if ((dataPathMode != DATA_PATH_MODE_DC_MODE_2) && (dataPathMode != DATA_PATH_MODE_DC_MODE_3))
	{
		dataPathPort >>= 1; //bit0 is the wlan radio id and had different handling in all platforms excapt FLM & LGM (if FLM/LGM we need to leave the value as is completely)
	}
	
	if ((dataPathMode == DATA_PATH_MODE_DC_MODE_1) || (dataPathMode == DATA_PATH_MODE_DC_MODE_2) || (dataPathMode == DATA_PATH_MODE_DC_NONE))
	{
		rxInSwHalt = HOST_IF_ACC_START_CORE;	// PUMA Direct Connect: Rx is active
		// PUMA Direct Connect: HD is BIG Endianness (PUMA cable card is writing in BIG Endianness to the RING)
		swapMode = (dataPathMode == DATA_PATH_MODE_DC_MODE_1) ? HOST_IF_ACC_SWAP_MODE : HOST_IF_ACC_NO_SWAP_MODE;
		
		freeRdInitReg.bitFields.freeRdInit = 0; //all RDs in pending list at init, so no RDs in free list, this will increased by the liberator once RDs will be pushed to free list via the rx in flow
	}
	else //GRX Fast path, FLM
	{		
		freeRdInitReg.bitFields.freeRdInit = dataPathInitParams->numOfAllocatedRxHds;
		
		rxInSwHalt = HOST_IF_ACC_HALT_CORE; /* When working on Fast Data Path Mode GSWIP/FLM, Rx Is Halted */
		if(dataPathInitParams->hostEndianessMode == HOST_ENDIANESS_MODE_BE)
		{
			swapMode = HOST_IF_ACC_SWAP_MODE;
		}
		else
		{
			swapMode = HOST_IF_ACC_NO_SWAP_MODE;
		}
	}

	//set threshold to RD free list - this threshold will be checked in host interface genrisc
	rdThreshodldReg.bitFields.rdThres = RX_FREE_LIST_THRESHOLD;

	/* Configure Number of Free RDs for TH Counter*/
	RegAccess_Write(REG_HOST_IF_ACC_FREE_RD_INIT,freeRdInitReg.val);
	/* Configure RD TH*/
	RegAccess_Write(REG_HOST_IF_ACC_RD_THRESHOLD,rdThreshodldReg.val);	
	/* Configure Tx Out Ring Size*/
	RegAccess_Read(REG_HOST_IF_ACC_TX_RING_CFG,&txRingCfgReg.val);
	txRingCfgReg.bitFields.txOutRingSize = (dataPathInitParams->txOutRingSizeBytes / hostIfDataHdSize->txOutHdSize); // HDS;
	RegAccess_Write(REG_HOST_IF_ACC_TX_RING_CFG,txRingCfgReg.val); 
	/* Configure Rx Out Ring Size*/
	RegAccess_Read(REG_HOST_IF_ACC_RX_RING_CFG,&rxRingCfgReg.val);
	rxRingCfgReg.bitFields.rxOutRingSize = (dataPathInitParams->rxOutRingSizeBytes/ hostIfDataHdSize->rxOutHdSize); // HDS 	
	RegAccess_Write(REG_HOST_IF_ACC_RX_RING_CFG,rxRingCfgReg.val); 
	/* Configure Tx In */
	txInCfg.val = 0;
	txInCfg.bitFields.txInHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->txInHdSize)-1;
	txInCfg.bitFields.txInPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(TxPd_t))-1;
	txInCfg.bitFields.txInTxqIdLine = PCK_DSCPTR_TID_LINE_NUM;
	txInCfg.bitFields.txInTxqIdOffset = PCK_DSCPTR_TID_OFFSET; 
	//txInCfg.bitFields.txInDescAllZeros = TRUE; //field does not exist in gen6
	txInCfg.bitFields.txInSwap = swapMode; 
	txInCfg.bitFields.txInSetNullPop = TRUE; 
	txInCfg.bitFields.txInSetNullPush = TRUE; 
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_CONV_CFG,txInCfg.val); 
	/* Configure Tx Out */
	txOutCfg.val = 0;
#ifdef HOST_IF_COUNTERS_WORKAROUND
	if (dataPathMode == DATA_PATH_MODE_DC_MODE_0)
	{
		txOutCfg.bitFields.txOutHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->txInHdSize)-1; //in fast path host i/f counter w/a we use same hd for tx out as for tx in
	}
	else
#endif //HOST_IF_COUNTERS_WORKAROUND
	{
		txOutCfg.bitFields.txOutHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->txOutHdSize)-1;
	}
	txOutCfg.bitFields.txOutPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(TxPd_t))-1;
	txOutCfg.bitFields.txOutTxqIdLine = PCK_DSCPTR_TID_LINE_NUM;
	txOutCfg.bitFields.txOutTxqIdOffset = PCK_DSCPTR_TID_OFFSET;
	txOutCfg.bitFields.txOutSwap = swapMode; 
	txOutCfg.bitFields.txOutSetNullPop = TRUE; 
	txOutCfg.bitFields.txOutSetNullPush = TRUE; 
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_CONV_CFG,txOutCfg.val); 
	/* Configure Rx In */
	rxInCfg.val = 0;
	rxInCfg.bitFields.rxInHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->rxInHdSize)-1;
	rxInCfg.bitFields.rxInPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(Rd_t ))-1;
	rxInCfg.bitFields.rxInSetNullPop = TRUE;
	rxInCfg.bitFields.rxInSetNullPush = TRUE;
	rxInCfg.bitFields.rxInSwap = swapMode;
	rxInCfg.bitFields.rxInMaskConversionEn = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_CONV_CFG,rxInCfg.val); 
	/* Configure Rx Out */
	rxOutCfg.val = 0;
	rxOutCfg.bitFields.rxOutHdLineNum = CONVERT_BYTES_TO_WORDS(hostIfDataHdSize->rxOutHdSize)-1;
	rxOutCfg.bitFields.rxOutPdLineNum = CONVERT_BYTES_TO_WORDS(sizeof(Rd_t ))-1;
	rxOutCfg.bitFields.rxOutTxqIdLine = RD_TX_QID_LINE_NUM;
	rxOutCfg.bitFields.rxOutTxqIdOffset = RD_TX_QID_OFFSET;
	rxOutCfg.bitFields.rxOutSetNullPop = TRUE;
	rxOutCfg.bitFields.rxOutSetNullPush = TRUE;
	rxOutCfg.bitFields.rxOutSwap = swapMode; 
	rxOutCfg.bitFields.rxOutSetNullPop = TRUE;
	rxOutCfg.bitFields.rxOutSetNullPush = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_CONV_CFG,rxOutCfg.val); 
	/* Configure all Rings to  Fast Path (Gswip)\Sw Path(No GSWIP) */
	hostTypeReg.val = 0;

	isGswip =  ((dataPathMode == DATA_PATH_MODE_DC_MODE_0) || (dataPathMode == DATA_PATH_MODE_DC_MODE_2) || (dataPathMode == DATA_PATH_MODE_DC_MODE_3))? TRUE:FALSE;
	
	hostTypeReg.bitFields.txInGswip = isGswip;	
	hostTypeReg.bitFields.txOutGswip = isGswip;	
	hostTypeReg.bitFields.rxInGswip = isGswip;	
	hostTypeReg.bitFields.rxOutGswip = isGswip;

#ifdef CBM_FRAGMENTATION_WORKAROUND
	if (dataPathInitParams->cbmFragmentationWaEnable)
	{
		hostTypeReg.bitFields.rxOutGswip = FALSE;
	}
#endif //CBM_FRAGMENTATION_WORKAROUND

#ifdef HOST_IF_COUNTERS_WORKAROUND
	// Host I/F acc to ignore the DMA done for txIn - it is handled by Host IF GenRisc 
	hostTypeReg.bitFields.txInGswip = FALSE;	
#endif //HOST_IF_COUNTERS_WORKAROUND

	RegAccess_Write(REG_HOST_IF_ACC_HOST_TYPE,hostTypeReg.val); 
	/* Own Bit Value Configuration*/
	ownBitValueReg.bitFields.ownBitValue = dataPathInitParams->hdOwnBitValue;
	ownBitValueReg.bitFields.ownBitLine	 = PCK_DSCPTR_OWN_LINE_NUM;
	ownBitValueReg.bitFields.ownBitOffset = PCK_DSCPTR_OWN_BIT_OFFSET;
	RegAccess_Write(REG_HOST_IF_ACC_OWN_BIT_VALUE,ownBitValueReg.val); 	
    hostIfAccLoadTxInCode(dataPathMode);
    hostIfAccLoadTxOutCode(dataPathMode, dataPathPort, txOutDw1FixedValues);
    hostIfAccLoadRxInCode(dataPathMode);
    hostIfAccLoadRxOutCode(dataPathMode, dataPathPort, rxOutDw3FixedValues);
	/* Start  Hw Acc Cores */
	txInSwInstrReg.val	= 0;
	txOutSwInstrReg.val = 0;
	rxInSwInstrReg.val	= 0;
	rxOutSwInstrReg.val = 0;
	txInSwInstrReg.bitFields.txInSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_SW_INSTRUCTION,txInSwInstrReg.val); 
	txOutSwInstrReg.bitFields.txOutSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_SW_INSTRUCTION,txOutSwInstrReg.val); 
	rxOutSwInstrReg.bitFields.rxOutSwHalt = 0;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_SW_INSTRUCTION,rxOutSwInstrReg.val); 
	rxInSwInstrReg.bitFields.rxInSwHalt = rxInSwHalt;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_SW_INSTRUCTION,rxInSwInstrReg.val); 
}

/**********************************************************************************

HostIfAcc_GetCurrentFreePdsNumber 


Description:
------------
Returns the current number of free PDs

Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
uint32 HostIfAcc_GetCurrentFreePdsNumber(void)
{
	RegHostIfAccFreePdCount_u numberOfFreePdsRegister;

	numberOfFreePdsRegister.val = 0;
	RegAccess_Read(REG_HOST_IF_ACC_FREE_PD_COUNT, &numberOfFreePdsRegister.val);
	return numberOfFreePdsRegister.bitFields.freePdCount;
}

/**********************************************************************************

HostIfAcc_ClearFreePdsThresholdEvent 


Description:
------------
Clears the event that the number of free PDs has reached to the threshold (this event goes 
to HW events)

Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
void HostIfAcc_ClearFreePdsThresholdEvent(void)
{
	RegHostIfAccThresholdClr_u clearThresholdInterruptRegister;

	clearThresholdInterruptRegister.val = 0;
	clearThresholdInterruptRegister.bitFields.pdThresReachedIntClr = TRUE;
	RegAccess_Write(REG_HOST_IF_ACC_THRESHOLD_CLR, clearThresholdInterruptRegister.val); 
}

/**********************************************************************************

HostIfAcc_SetPdThreshold 


Description:
------------
Sets the PD threshold
Input: 
-----	

			
Output:
-------
	

Returns:
--------
number of free PDs 
	
**********************************************************************************/
void HostIfAcc_SetPdThreshold(uint16 pdThreshold)
{
	RegHostIfAccPdThreshold_u freePdsThresholdRegister;

	ILOG0_D("HostIfAcc_SetPdThreshold - %d", pdThreshold);
	freePdsThresholdRegister.val = 0;
	freePdsThresholdRegister.bitFields.pdThres = pdThreshold;
	RegAccess_Write(REG_HOST_IF_ACC_PD_THRESHOLD,freePdsThresholdRegister.val);
}

void HostIfAcc_Init(void)
{
    RegHostIfAccTxInListIdx_u txInListIdx;
    RegHostIfAccTxOutListIdx_u txOutListIdx;
    RegHostIfAccRxInListIdx_u  rxInListIdx;
    RegHostIfAccRxOutListIdx_u rxOutListIdx;   
	RegHostIfAccRxInSwInstruction_u rxInSwInstrReg;
	RegHostIfAccMulticastPdDistribCfg_u pdMcCfg; 
	RegHostIfAccTtlScaleSet_u ttlScaleRegister;
	RegHostIfAccFreePdInit_u  numberOfFreePdsRegister;
	RegHostIfAccPdThreshold_u freePdsThresholdRegister;
	RegHostIfAccHostCountersCfg_u counterControlRegister;
	RegHostIfAccTxInMulticastCfg_u multicastTxInRegister;
	RegHostIfAccRxOutMulticastCfg_u multicastRxOutRegister;
	RegHostIfAccPdLengthLocationCfg_u pdLengthLocationCfg;
	ttlScaleRegister.val = 0;
	numberOfFreePdsRegister.val = 0;
	freePdsThresholdRegister.val = 0;
	counterControlRegister.val = 0;	
	multicastTxInRegister.val = 0;
	multicastRxOutRegister.val = 0;
	pdLengthLocationCfg.val = 0;
	debug_counterControlRegister.val = 0;
	debug_multicastTxInRegister.val = 0;
	debug_multicastRxOutRegister.val = 0;				
    // config shared RAM buffers addresses
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxInShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxInShramBuffers[1][0]- SHARED_RAM_BASE_ADDR));  
    RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxOutShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.TxOutShramBuffers[1][0]- SHARED_RAM_BASE_ADDR)); 
    RegAccess_Write(REG_HOST_IF_ACC_RX_IN_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxInShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_RX_IN_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxInShramBuffers[1][0]- SHARED_RAM_BASE_ADDR));  
    RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_HD_BUF0_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxOutShramBuffers[0][0]- SHARED_RAM_BASE_ADDR));
    RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_HD_BUF1_BASE_ADDR,CONVERT_BYTES_TO_WORDS((uint32)&ShramHostIfDb.RxOutShramBuffers[1][0]- SHARED_RAM_BASE_ADDR)); 
	/* Halt All Hw Acc Cores*/
	rxInSwInstrReg.val = 0;
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_SW_INSTRUCTION,rxInSwInstrReg.val); 	
    // config DLM indexes
    txInListIdx.val = 0;
    txInListIdx.bitFields.txInPopListIdx = HW_Q_MANAGER_TX_POOL_LIST_DATA_PD;
    txInListIdx.bitFields.umacUcPushListIdx = HW_Q_MANAGER_READY_LIST_UNICAST_PD;
#ifndef TX_DATA_UM_PATH
	txInListIdx.bitFields.umacMcPushListIdx = HW_Q_MANAGER_READY_LIST_MULTICAST_PD;
#else	
    txInListIdx.bitFields.umacMcPushListIdx = HW_Q_MANAGER_READY_LIST_UNICAST_PD;
#endif
    RegAccess_Write(REG_HOST_IF_ACC_TX_IN_LIST_IDX,txInListIdx.val); 
	txOutListIdx.val = 0;
	txOutListIdx.bitFields.txOutPopListIdx = HW_Q_MANAGER_LOW_PR_READY_LIST_TEMP_HOST_Q;
	txOutListIdx.bitFields.txOutPushListIdx = HW_Q_MANAGER_TX_POOL_LIST_DATA_PD;
	RegAccess_Write(REG_HOST_IF_ACC_TX_OUT_LIST_IDX,txOutListIdx.val); 
	rxInListIdx.val = 0;
	rxInListIdx.bitFields.rxInPopListIdx = HW_Q_MANAGER_RX_HOST_INTERFACE_TEMP_LIST;
	rxInListIdx.bitFields.rxInPushListIdx = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST; //push directly to free list. This will enable counter free and busy counters to be valid
	RegAccess_Write(REG_HOST_IF_ACC_RX_IN_LIST_IDX,rxInListIdx.val); 
	rxOutListIdx.val = 0;
	rxOutListIdx.bitFields.rxOutPopListIdx = HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT;
	rxOutListIdx.bitFields.rxOutPushListIdx = HW_Q_MANAGER_RX_HOST_INTERFACE_TEMP_LIST;
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_LIST_IDX,rxOutListIdx.val);
	pdMcCfg.val = 0; 
#if !defined(TX_DATA_UM_PATH) && !defined(HOST_IF_DIST_TYPE_MULTICAST_VAP_WORKAROUND)
	pdMcCfg.bitFields.groupVapPrio = MCAST_DIST_MODE_VAP_BITMAPS;
#else
	pdMcCfg.bitFields.groupVapPrio = MCAST_DIST_MODE_IGNORE_BITMAPS;
#endif

	pdMcCfg.bitFields.mcBitLine = PCK_DSCPTR_MC_LINE_NUM; 
	pdMcCfg.bitFields.mcBitOffset = PCK_DSCPTR_MC_BIT_OFFSET; 
	pdMcCfg.bitFields.groupIdxLine = PCK_DSCPTR_GROUP_IDX_LINE_NUM; 
	pdMcCfg.bitFields.groupIdxOffset = PCK_DSCPTR_GROUP_IDX_BIT_OFFSET; 

	RegAccess_Write(REG_HOST_IF_ACC_MULTICAST_PD_DISTRIB_CFG,pdMcCfg.val);
	
    ttlScaleRegister.bitFields.ttlScale = HOST_INTERFACE_ACCELERATOR_TTL_SCALE;
	RegAccess_Write(REG_HOST_IF_ACC_TTL_SCALE_SET,ttlScaleRegister.val);
	numberOfFreePdsRegister.bitFields.freePdInit = NUM_OF_TX_DESC;
	RegAccess_Write(REG_HOST_IF_ACC_FREE_PD_INIT,numberOfFreePdsRegister.val);
	freePdsThresholdRegister.bitFields.pdThres = HOST_INTERFACE_ACCELERATOR_FREE_PDS_THRESHOLD_DISABLED;
	RegAccess_Write(REG_HOST_IF_ACC_PD_THRESHOLD,freePdsThresholdRegister.val);
	/* Enable Host IF Counters*/
	counterControlRegister.val = 0;
	counterControlRegister.bitFields.txInUniHdCountEn = TRUE;
	counterControlRegister.bitFields.txInMultiHdCountEn = TRUE;
	counterControlRegister.bitFields.txInBroadcastHdCountEn = TRUE;
	counterControlRegister.bitFields.txInUniBytesCountEn = TRUE;
	counterControlRegister.bitFields.txInMultiBytesCountEn = TRUE;
	counterControlRegister.bitFields.txInBroadcastBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutUniHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutMultiHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutBroadcastHdCountEn = TRUE;
	counterControlRegister.bitFields.rxOutUniBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutMultiBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutBroadcastBytesCountEn = TRUE;
	counterControlRegister.bitFields.rxOutStaBytesCountEn = TRUE;
	counterControlRegister.bitFields.agerCountEn = TRUE;
#ifdef HOST_IF_COUNTERS_WORKAROUND
	// Host I/F accelerator shouldn't update the PD & Bytes counters
	counterControlRegister.bitFields.qosTxEn = FALSE;
	counterControlRegister.bitFields.qosTxDecEn = FALSE;
	counterControlRegister.bitFields.qosStaTidByteCountEn = FALSE;
	counterControlRegister.bitFields.qosStaTidByteCountDecEn = FALSE;
#else
	counterControlRegister.bitFields.qosTxStaTidPdCountIncEn = TRUE; //Enable in Host IF acc logic to increment count per STA-TID
	counterControlRegister.bitFields.qosTxStaTidPdCountDecEn = TRUE; //Enable in Host IF acc logic to decrement count per STA-TID
	counterControlRegister.bitFields.qosTxVapAcPdCountIncEn  = TRUE; //Enable in Host IF acc logic to increment count per VAP-AC
	counterControlRegister.bitFields.qosTxVapAcPdCountDecEn  = TRUE; //Enable in Host IF acc logic to decrement count per VAP-AC
	counterControlRegister.bitFields.qosStaTidByteCountIncEn = TRUE;
	counterControlRegister.bitFields.qosStaTidByteCountDecEn = TRUE;
#endif
#ifdef ENET_INC_ARCH_WAVE600B
	counterControlRegister.bitFields.agerPdNoTransmitCountEn = TRUE;
#endif
	RegAccess_Write(REG_HOST_IF_ACC_HOST_COUNTERS_CFG,counterControlRegister.val);
	RegAccess_Read(REG_HOST_IF_ACC_HOST_COUNTERS_CFG,&debug_counterControlRegister.val);
	/* Enable HOSTIF_COUNTERS_TX_IN_MULTICAST_HD Counter*/
	multicastTxInRegister.val = 0;
	multicastTxInRegister.bitFields.txInByteCountLine = 0x0;		//dataLength field resides in word0 of PD in bits 0..13
	multicastTxInRegister.bitFields.txInByteCountOffset = 0x0;		//dataLength field resides in word0 of PD in bits 0..13
	RegAccess_Write(REG_HOST_IF_ACC_TX_IN_MULTICAST_CFG,multicastTxInRegister.val);
	RegAccess_Read(REG_HOST_IF_ACC_TX_IN_MULTICAST_CFG,&debug_multicastTxInRegister.val);
	/* Enable HOSTIF_COUNTERS_RX_OUT_MULTICAST_HD Counter*/
	multicastRxOutRegister.val = 0;
	multicastRxOutRegister.bitFields.rxOutByteCountLine = 0x0;		//dataLength field resides in word1 of RD in bits 0..13
	multicastRxOutRegister.bitFields.rxOutByteCountOffset = 0x20;	//dataLength field resides in word1 of RD in bits 0..13
	multicastRxOutRegister.bitFields.rxOutMcBitLine = 0x1;			//multicast field resides in word2 of RD in bit 27
	multicastRxOutRegister.bitFields.rxOutMcBitOffset = 0x1B;		//multicast field resides in word2 of RD in bit 27
	multicastRxOutRegister.bitFields.rxOutBcBitLine = 0x1;			//broadcast field resides in word2 of RD in bit 26
	multicastRxOutRegister.bitFields.rxOutBcBitOffset = 0x1A;		//broadcast field resides in word2 of RD in bit 26
	RegAccess_Write(REG_HOST_IF_ACC_RX_OUT_MULTICAST_CFG, multicastRxOutRegister.val);
	RegAccess_Read(REG_HOST_IF_ACC_RX_OUT_MULTICAST_CFG, &debug_multicastRxOutRegister.val);

	pdLengthLocationCfg.bitFields.txLengthInPdLineNum = PCK_DSCPTR_LENGTH_LINE_NUM;	//Line number in PD for "length" field. Used at TX Out core.
	pdLengthLocationCfg.bitFields.txLengthInPdOffset = PCK_DSCPTR_LENGTH_BIT_OFFSET;//Offset in PD for "length" field. Used at TX Out core.
	pdLengthLocationCfg.bitFields.txLengthInHdLineNum = HD_DSCPTR_LENGTH_LINE_NUM;	//Line number in HD for "length" field. Used at TX In core
	pdLengthLocationCfg.bitFields.txLengthInHdOffset = HD_DSCPTR_LENGTH_BIT_OFFSET;	//Offset in HD for "length" field. Used at TX In core.
	RegAccess_Write(REG_HOST_IF_ACC_PD_LENGTH_LOCATION_CFG, pdLengthLocationCfg.val);

	memset32((void*)DIST_BITMAP_ADDRESS, 0xFFFFFFFF,CONVERT_BYTES_TO_WORDS( CONVERT_BIT_INDEX_TO_BYTES(UNICAST_BITMAP_SIZE + VAPS_BITMAP_SIZE + GROUPS_BITMAP_SIZE))); // init all bitmaps to '1'
}

