/***************************************************************
 File:		Pac_Api.h
 Module:	  	Pac Drivers
 Purpose: 	Create common interface for accessing the PAC registers 
 Description:  This file is the API for acessing the PAC registers
***************************************************************/
#ifndef PAC_API_H_
#define PAC_API_H_
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "mhi_ieee_address.h"
#include "lmi.h"
#include "RxcRam.h"



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define TIM_FAST_DIVIDE_RATIO               (1000000) /* 1MHz */
#define TIM_SLOT_DIVIDE_RATIO               8000000 /* 8MHz */
#define TIM_IFS_DIVIDE_RATIO                (8000000) /* 8MHz */
#define TIM_NAV_SLOW_INT_RATIO              (1000000) /* 1MHz */
#define TIM_NAV_SLOW_INCR_RATIO             (512) /* 512MHz */

/* during Scan we will use Long Filter 0 as the address 1 filter*/
#define SCAN_DEFAULT_MAC_ADDRESS_FILTER 0


/* Receive Conditions used in Wave 500*/
#define RCD_MPDU_FIELDS_G3  		    0x0  /* 00 */
#define RCD_MPDU_FIELDS_G2 		        0x1  /* 01 */
#define RCD_MPDU_FIELDS_G1  		    0x2  /* 02 */
#define RCD_MPDU_FIELDS_G0  		    0x3  /* 03 */
#define RCD_NO_ACK_FILTERS  		    0x4  /* 04 */
#define RCD_ERR_CAT_NOACK   		    0x5  /* 05 */
#define RCD_ERR_CAT_ACK				    0x6  /* 06 */
#define RCD_ERR						    0x7  /* 07 */
#define RCD_RTS						    0x8  /* 08 */
#define RCD_PSPOLL					    0x9  /* 09 */
#define RCD_NAV_RESET				    0xa  /* 10 */
#define RCD_BAR_BACK				    0xb  /* 11 */
#define RCD_BAR_ACK					    0xc  /* 12 */
#define RCD_NDP						    0xd  /* 13 */
#define RCD_BFPOLL					    0xe  /* 14 */
#define RCD_NDP_ERR					    0xf  /* 15 */
#define RCD_UQDN_NOACK_AGG			    0x10 /* 16 */
#define RCD_UQDN_NOACK				    0x11 /* 17 */
#define RCD_UQDN_ACK_AGG			    0x12 /* 18 */
#define RCD_UQDN_ACK				    0x13 /* 19 */
#define RCD_UDATA					    0x14 /* 20 */
#define RCD_BEACON					    0x15 /* 21 */
#define RCD_PROBE_RESPONSE			    0x16 /* 22 */
#define RCD_ACTION_LAST_IN_AGG_NO_BACK 	0x17 /* 23 */
#define RCD_ACTION_LAST_IN_AGG 			0x18 /* 24 */
#define RCD_UMGMT_NOACK				    0x19 /* 25 */
#define RCD_UMGMT					    0x1a /* 26 */
#define RCD_MULTI					    0x1b /* 27 */
#define RCD_NAV_LOAD				    0x1c /* 28 */

#define RXD_CONDITION_MATRIX_ROWS   (34)



/* Time Definitions*/
#define ONE_USEC 			(1)
#define ONE_EIGTH_US_IN_NS 	(125)


/* Hw Delay Id*/
typedef enum
{
	PAC_DELAY_TXC_SIFS_ELEMENT = 0,
	PAC_DELAY_DELIA_IFS_OFDM,
	PAC_DELAY_DELIA_IFS_11B,
	PAC_DELAY_RXC_CCA,
	PAC_DELAY_RXC_RX_READY,
	PAC_DELAY_TIM_TX_NAV_IFS, 
	PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TX_OFDM,
	PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TX_11B,
	PAC_DELAY_TX_SEQ_SIFS_COMP_POST_RX_OFDM,
	PAC_DELAY_TX_SEQ_SIFS_COMP_POST_RX_11B,
	PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TIMEOUT,
	PAC_DELAY_TOTAL_NUM,
	PAC_DELAY_MAX = 0xFF,
}PacHwDelays_e;


typedef enum
{
	RXF_FIFO_READ_INVALID = 0,
	RXF_FIFO_READ_VALID  = 1
} RxfFifoReadValid_e;

typedef enum
{
	RXF_FILTER_STATUS_DISABLE = 0,
	RXF_FILTER_STATUS_ENABLE  = 1
} RxfFilterStatus_e;

typedef enum
{
	RXF_BSS_MODE_IBSS  = 0, // 00
	RXF_BSS_MODE_STA   = 1, // 01 
	RXF_BSS_MODE_AP    = 2, // 10  
	RXF_BSS_MODE_WDS   = 3, // 11
	RXF_BSS_MODE_MAX   = 4,
} RxfBssMode_e;


typedef enum
{
	RXF_AND_LOGIC_TYPE  = 0,
	RXF_OR_LOGIC_TYPE = 1,	
} RxfLogicType_e;


typedef enum
{
	RXF_DONT_USE_LONG_FILTER = 0,
	RXF_USE_LONG_FILTER      = 1,	
} RxfLongFilterUse_e;


typedef enum
{
	RXF_DONT_USE_SHORT_FILTER		= 0, /* 0x0: No use of enhanced short filter results */		
	RXF_USE_SHORT_FILTER_0			= 1, /* 0x1: use only enhanced short filter 0 result */
	RXF_USE_SHORT_FILTER_0_1		= 2, /* 0x2: use only enhanced short filter 0-1 results */
	RXF_USE_SHORT_FILTER_0_2		= 3, /* 0x3: use only enhanced short filter 0-2 results */	
	RXF_USE_SHORT_FILTER_0_3		= 4, /* 0x4: use only enhanced short filter 0-3 results */
} RxfShortFilterUse_e;

typedef enum
{
	RXF_GROUP_FILTER_ID_0 = 0,
	RXF_GROUP_FILTER_ID_1 = 1,
	RXF_GROUP_FILTER_ID_2 = 2,
	RXF_GROUP_FILTER_ID_3 = 3,
	RXF_GROUP_FILTER_NUM  = 4,
} RxfGroupFilterId_e;


typedef enum
{
	RXF_GROUP_SHORT_FILTER_ID_0 = 0,
	RXF_GROUP_SHORT_FILTER_ID_1 = 1,
	RXF_GROUP_SHORT_FILTER_ID_2 = 2,
	RXF_GROUP_SHORT_FILTER_ID_3 = 3,
	RXF_GROUP_SHORT_FILTER_NUM  = 4,
} RxfShortFilterId_e;


typedef enum
{
	RXF_LONG_GROUP_FILTER_SOURCE_PHY_STATUS = 0, 
	RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER = 1
} RxfGroupFilterSource_e;


typedef enum
{
	RXF_GROUP_LONG_FILTER_EQUATION_EQUAL        = 0, 
	RXF_GROUP_LONG_FILTER_EQUATION_NOT_EQUAL    = 1,
	RXF_GROUP_LONG_FILTER_EQUATION_GREATER_THAN = 2,
	RXF_GROUP_LONG_FILTER_EQUATION_BELOW_THAN   = 3
} RxfGroupLongFilterEquation_e;


typedef enum
{
	RXF_GROUP_SHORT_FILTER_EQUATION_EQUAL 			    = 0, 
	RXF_GROUP_SHORT_FILTER_EQUATION_NOT_EQUAL 			= 1,
	RXF_GROUP_SHORT_FILTER_EQUATION_GREATER_THAN 		= 2,
	RXF_GROUP_SHORT_FILTER_EQUATION_BELOW_THAN 			= 3,
	RXF_GROUP_SHORT_FILTER_EQUATION_GREATER_THAN_SIGNED = 6,
	RXF_GROUP_SHORT_FILTER_EQUATION_BELOW_THAN_SIGNED 	= 7
} RxfGroupShortFilterEquation_e;

/* RXC Enum definitions */
typedef enum
{
	RXC_TRAFFIC_INDICATION_DISABLE = 0,
	RXC_TRAFFIC_INDICATION_ENABLE = 1
} RxcTrafficIndicationEnable_e;

typedef enum
{
	RXC_STATE_IDLE		= 0 ,
	RXC_STATE_ENABLED,
	RXC_STATE_FRAMEMARK_TIMESTAMP,
	RXC_STATE_APPEND_RESERVED_SPACE,
	RXC_STATE_PHY_STATUS1,
	RXC_STATE_PHY_STATUS2,
	RXC_STATE_DATA,
	RXC_STATE_WAIT_RD_INFO,
	RXC_STATE_PHY_STATUS0,
	RXC_STATE_PHY_MATRICES,
	RXC_STATE_KEEP_FRAME_DECISION,
	RXC_STATE_DONE,
	RXC_NUM_OF_STATES,
	RXC_MAX_NUM_OF_STATES,
} RxcState_e;


/* Delineator  definitions */
#define DELINEATOR_STATE_IDLE (0)


#define DELINEATOR_RX_RESUMED	(0x0)
#define DELINEATOR_RX_HALTED	(0x1)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
 #define GET_TSF_TIMER_LOW() (Pac_TimGetTsfLow())
 
/*---------------------------------------------------------------------------------
/                           Data Type Definition                        
/----------------------------------------------------------------------------------*/
/* RXF  Data Type Definitions */
typedef struct RxfGroupFilterConfig
{
	RxfGroupFilterId_e     groupId; //              - There are 4 groups (groupId = {0,1,2,3})
	uint16   		 vapEnableBitmap;  // - Bimap of the vaps the filter group is enabled for them
	RxfLogicType_e   logic0; //:1      - The logic between ELF and ESF0.
	RxfLogicType_e   logic1; //:1      - The logic between ESF1 and the result of ELF and ESF0.
	RxfLogicType_e   logic2; //:1      - The logic between ESF2 and the result of ELF, ESF0 and ESF1.
	RxfLogicType_e   logic3; //:1      - The logic between ESF3 and the result of ELF, ESF0, ESF1 and ESF2.
	RxfLongFilterUse_e   useEnhancedLongFilterResult; //:1 - use of enhanced long filter result. 	
	RxfShortFilterUse_e  useEnhancedShortFilterResult; //:3 - use of enhanced short filter result.(the num of results to use)
} RxfGroupFilterConfig_t;

typedef struct RxfEnhancedLongFilterConfig
{
	RxfGroupFilterId_e	 groupId; 			// There are 4 groups (groupId = {0,1,2,3})
	RxfGroupFilterSource_e	groupSource;// 0: Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	RxfGroupLongFilterEquation_e  groupEquation; // Determines the equation between the filter value and the latched filed.{==,!=,>,<}
	uint8   groupOffset;	//    Byte offset of the field (from the start of the PHY Status or MPDU MAC header) 	   
	uint32	groupValue;	    //    MPDU field filter value to compare
	uint32	groupMask;	    //    MPDU field filter mask of results to use
} RxfEnhancedLongFilterConfig_t;


typedef struct RxfEnhancedShortFilterConfig
{
	RxfGroupFilterId_e	 groupId; 			// There are 4 groups (groupId = {0,1,2,3})
	RxfShortFilterId_e	 filterId;          // There are 4 short filter in a group (short filter Id = {0,1,2,3})
	RxfGroupFilterSource_e	groupSource;// 0: Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	RxfGroupShortFilterEquation_e  groupEquation; // Determines the equation between the filter value and the latched filed.{==,!=,>,<,>S,<S}
	uint8   groupOffset;	//    Byte offset of the field (from the start of the PHY Status or MPDU MAC header) 	   
	uint8	groupValue;	    //    MPDU field filter value to compare
	uint8	groupMask;	    //    MPDU field filter mask of results to use
} RxfEnhancedShortFilterConfig_t;

typedef struct
{
	uint16 currentIndexInFifo;
	uint16 numberOfEntries;
} RxfFifoParameters_t;


typedef struct PacDurAutoReplyPowerParameters
{
	uint8 autoReplyPower11B;
	uint8 autoReplyPower11AGBw20;
	uint8 autoReplyPower11AGBw40;
	uint8 autoReplyPower11AGBw80;
	uint8 autoReplyPower11AGBw160;
	uint8 autoReplyPower11axSuExtDcm1Ru242;
	uint8 autoReplyPower11axSuExtDcm0Ru242;
	uint8 autoReplyPower11axSuExtDcm1Ru106;
	uint8 autoReplyPower11axSuExtDcm0Ru106;	
	uint8 reserved[3];
} PacDurAutoReplyPowerParameters_t;


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/                           Public Function Declaration                        
/----------------------------------------------------------------------------------*/

/* Pac General API*/
void Pac_SetHwDelays(uint32 hwMode);
uint32 Pac_GetHwDelays(PacHwDelays_e hwDelayId);

/* Pac Delia API*/
void 	Pac_DeliaInit(void);
#if defined (ENET_INC_ARCH_WAVE600)
void Pac_DeliaTxEndErrorDuringTx(uint32 deliaStatus);
void Pac_DeliaTxEndErrorAtEndOfTx(uint32 deliaStatus);
void Pac_DeliaReadDebugInfoOnAssert(void);
#endif //ENET_INC_ARCH_WAVE600



/* Pac Rxd API*/
#ifndef ENET_INC_ARCH_WAVE600 // there is no RxD in wave600
void Pac_RxdInit(void);
void Pac_RxdLoadConditionMatrix(void);
void Pac_RxdConfigureScanMode(void);
void Pac_RxdConfigureNormalMode(void);
void Pac_RxdDisable(void);
bool Pac_RxdIsRxStored(uint32 event);
bool Pac_RxdIsEventDisabled(uint32 event);
void Pac_RxdDisableRx(uint32 event);
void Pac_RxdEnableRx(uint32 event);
void Pac_RxdConfigureBypassMode(void);
void Pac_RxdConfigureRssiMode(void);
#endif //ENET_INC_ARCH_WAVE600

/* Pac Rxc API*/
#if defined (ENET_INC_ARCH_WAVE600)
void Pac_RxcClearMpduEndErrorCtrl(void);
uint16 Pac_RxcReadMpduCounter(void);
#ifdef ENET_INC_ARCH_WAVE600D2
void Pac_RxcSetAidForStaMode(uint8 vapId, uint16 aid, bool valid);
#endif
void Pac_RxcSetBssModeRejectFilterStatus(uint8 vapId, RxfBssMode_e bssMode, RxfFilterStatus_e status);
void Pac_RxcSetMACAddressfilter (uint8 vapId, IEEE_ADDR *address);
void Pac_RxcClearAddressFilter (uint8 vapId) ;
void Pac_RxcInitializeNoBaAgreementFifo(void);
void Pac_RxcGetFifoParameters(RxfFifoParameters_t *fifoParameters);
void Pac_RxcSetFifoParameters(RxfFifoParameters_t *fifoParameters);
void Pac_RxcSetHeFragViolationFilter(uint8 supportedHeFragLevel, uint8 vapIndex);
void Pac_RxcSetStaFrameClass(uint32 staIndex,uint32 allowedFrameClass);
void Pac_RxcPhyStatusFilterSetGroupIndexForSta(uint8 groupIndex , StaId staIndex);
void Pac_RxcPhyStatusFilterEnableDisable(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, bool isEnable);
void Pac_RxcPhyStatusFilterSetMinMax(RxcPhyStatusFilterType_e filterType, bool isMinimum);
void Pac_RxcPhyStatusFilterSetGreaterOrLessEqual(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, RxcPhyStatusFilterOpration_e filterOperation);
void Pac_RxcPhyStatusFilterSetReferenceAndMask(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, uint8 referenceVal, uint8 maskVal);
void Pac_RxcAutoRespFifoUpdateEnable(bool flag);
void Pac_RxcSnifferModeConfiguration(uint8 bandId); 
void Pac_RxcPauseRx(void);
void Pac_RxcResumeRx(void);
void Pac_RxcClearRxHaltIrq(void);
void Pac_RxcErrorIrq(void);
#endif //ENET_INC_ARCH_WAVE600
void Pac_RxcSetForceCca(void);
void Pac_RxcSetForceCcaToNormalMode(void);
void Pac_RxcUpdateMinimalRssiThreshold(int8 rssiThreshlod);
void Pac_RxcReadRxBufferInPtr(volatile uint32 *rxBufferInPtr);
void Pac_RxcClearRxBufferOverflow(void);
void Pac_RxcEnable(void);  
void Pac_RxcDisable(void);
void Pac_RxcReset(void);
void Pac_RxcInit(void);

void  Pac_DelineatorEnableFifoFullEvent(void);
uint8 Pac_DelineatorGetState(void);
void  Pac_DelineatorAssertToReset(void);
void  Pac_DelineatorReleaseFromReset(void);
void  Pac_DeliniatorInit(void);

void Pac_RxcSetReservedStorageSize(uint8 reservedStorageSize);
void Pac_RxcEnableRssiAbort(void);
void Pac_RxcDisableRssiAbort(void);
void Pac_RxcDisableCcaIndication(void);
RxcState_e Pac_RxcGetState(void);



#if defined (ENET_INC_UMAC)
uint8 Pac_RxcBaAgreementRecipientGetAndClearActivity(StaId stationIndex, uint8 tid);
bool Pac_RxcBaAgreementInitiatorGetAndClearActivity(StaId stationIndex, uint8 tid);
#ifdef ENET_INC_ARCH_WAVE600D2
bool Pac_RxcRxPpGetAndClearActivity(StaId stationIndex, uint8 tid);
#endif 
uint8 Pac_RxcGetAndClearStationActivity(StaId stationIndex);
void Pac_RxcInitializeRamParameters(void);
#endif //ENET_INC_ARCH_WAVE600


/* Pac Rxf API */
#if defined (ENET_INC_LMAC)
void Pac_RxfInit(void);
#endif
void Pac_RxfReset (void);
void Pac_RxfSetMACAddressfilter (uint8 vapId, IEEE_ADDR *address, bool RssiModeEnable);
void Pac_RxfClearAddressFilter (uint8 vapId);
void Pac_RxfUseFixedVapId(uint8 vapId);
void Pac_RxfNormalVapId(uint8 vapId);


#if defined (ENET_INC_UMAC)
void Pac_RxfGetFifoParameters(RxfFifoParameters_t *fifoParameters);
void Pac_RxfSetFifoParameters(RxfFifoParameters_t *fifoParameters);
void Pac_RxfInitializeNoBaAgreementFifo(void);
#endif //ENET_INC_ARCH_WAVE600

 #if defined (ENET_INC_LMAC)
void Pac_RxfSetBssModeRejectFilterStatus(uint8 vapId, RxfBssMode_e bssMode, RxfFilterStatus_e status);
void Pac_RxfConfigureGroupFilter(RxfGroupFilterConfig_t* configuration);
void Pac_RxfConfigureEnhancedLongFilter(RxfEnhancedLongFilterConfig_t* configuration);
void Pac_RxfConfigureEnhancedShortFilter(RxfEnhancedShortFilterConfig_t* configuration);
uint16 Pac_RxcReadRxBufferOverflowCounter(void);
void Pac_RxcClearRxBufferOverflowCounter(void);
#endif /* #if defined (ENET_INC_LMAC) */
 

/* Pac Txc API*/
void Pac_TxcEnablePriority(uint32 priority);
void Pac_TxcDisablePriority(uint32 priority);
void Pac_TxcDisableAllPriorities(void);
void Pac_TxcGetCoincidentPtcs(uint32 *coincidentPtcs);
void Pac_TxcSetAlternativeBackoff(uint32 priority,uint32 alternativeBackoff);
void Pac_TxcInit(void);


/* Pac Tim API*/
void   	Pac_TimProgramHwEventTimer(TSF hwEventTime);
void   	Pac_TimProgramSwEvent2Timer(TSF swEventTime);
void   	Pac_TimProgramSwEvent3Timer(TSF swEventTime);
bool   	Pac_TimIsSwEvent2Occured(void);
bool   	Pac_TimIsSwEvent3Occured(void);
void  	Pac_TimResetSwTimer2(void);
void   	Pac_TimResetSwTimer3(void); 
void   	Pac_TimProgramQueueSwitchTimer( uint8 u8direction, uint32 switchTime);
uint32 	Pac_TimGetSlotTimer(void);
void   	Pac_TimSetSlotTimer(uint32 slotTimerValue);
void   	Pac_TimSetTsf(uint32 tsfLow, uint32 tsfHigh);
TSF    	Pac_TimGetTsf(void);
uint32 	Pac_TimGetTsfHigh(void);
uint32 	Pac_TimGetTsfLow(void);
uint32 	Pac_TimGetDeltaTimeFromTsf(uint32 startTime);
void 	Pac_TimResetQbssIntegrator(void);
void 	Pac_TimResetQbss2Integrator(void);
uint32  Pac_TimReadQbssIntegrator(void);
uint32  Pac_TimReadQbss2Integrator(void);
void   	Pac_TimResetTimers(void);
void   	Pac_TimInit(void);
uint32 	Pac_TimGetTsfLowPerBand(uint8 bandId);
uint32  Pac_TimReadQbssIntegratorPerBand(uint8 bandId);
void 	Pac_TimResetQbssIntegratorPerBand(uint8 bandId);



/* Pac Dur API*/
void pacDurFillRamAutoReplyAirTime(void);
void pacDurFillRamAutoReplyBa11axAirTime(void);
void Pac_DurConfigureAutoReplyRate(uint16 vapIndex);
#ifdef ENET_INC_ARCH_WAVE600
void Pac_DurConfigureGeneralVapAutoReplyRate(void);
#endif //ENET_INC_ARCH_WAVE600

#ifdef ENET_INC_ARCH_WAVE600D2
void Pac_DurConfigureAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters,uint8 autoReplyRate);
void Pac_DurSetExtendedRangeAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters);
#else // ENET_INC_ARCH_WAVE600D2
#if defined (ENET_INC_LMAC)
void Pac_DurConfigureAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters,uint8 autoReplyRate);
#endif /* #if defined (ENET_INC_LMAC)*/
#endif //ENET_INC_ARCH_WAVE600D2

void Pac_DurInit(void);

void Pac_DurConfigureVapMode(uint16 vapIndex, uint16 vapMode);
#ifdef ENET_INC_ARCH_WAVE600D2
void pacDurFillRamAutoReply11axExtAirTime(void);
#endif

uint32 Pac_DurGetTotalAirTimePerBand(uint8 bandId);
uint32 Pac_DurGetTotalAirTime(void);


void Pac_DurResetTotalAirTimePerBand(uint8 bandId);
void Pac_DurResetTotalAirTime(void);


/* Mac Clc API*/
void Pac_ClcInit(void);


#endif /* PAC_API_H_ */

