/***********************************************************************************
 File:		PacDur.c
 Module:		Pac Driver
 Purpose:		
 Description:	FW driver for Dur module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "RegAccess_Api.h"
#include "PacDurRegs.h"
#include "DurRam.h"
#include "HwMemoryMap.h"
#include "Protocol_AirTimeAndDuration.h"
#include "protocol.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "PacDur.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 8


#define RXC_SIFS_PLUS_PHY_RX_DELAY_11AG 				42
#define RXC_SIFS_PLUS_PHY_RX_DELAY_11B_SHORT_PREAMBLE	108
#define RXC_SIFS_PLUS_PHY_RX_DELAY_11B_LONG_PREAMBLE	206



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/

    


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
PacDurRam_t *PacDurRam; 



/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void pacDurFillRamAutoReply(void);

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

pacDurFillRamAutoReplyAirTime

Description:
------------
	1) Fills the Auto Reply Air Time fields  in Pac Dur Ram 
		
**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void pacDurFillRamAutoReplyAirTime(void)
{
	AutoReplyFrameAndSifsAirTime_t autoReplySifsAndAirTime;
	uint32 	rate;

	for(rate = 0; rate < LM_PHY_11A_MAX_RATES; rate++)
	{
		autoReplySifsAndAirTime.bitFields.ackOrCtsAirTime11B = Protocol_GetAckCts11bAirTime(rate)+ SIFS_TIME_11B;
		autoReplySifsAndAirTime.bitFields.ackOrCtsAirTime11AG = Protocol_GetAckCts11agAirTime(rate) + SIFS_TIME_11A;
		autoReplySifsAndAirTime.bitFields.baAirTime11AG = 	Protocol_GetBaAirTime(rate)+ SIFS_TIME_11A;				

		PacDurRam->autoReplyAndSifsAirTime[rate].val = autoReplySifsAndAirTime.val;
	}
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Pac_DurConfigureAutoReplyRate

Description:
------------
	Fills the Autou Reply rate in Pac Dur Ram
Input:
-----
	vapIndex - The Vap index to update (can also be general vap)
		
**********************************************************************************/
void Pac_DurConfigureAutoReplyRate(uint16 vapIndex)
{
	
	/* TBD - Set According to Basic Rate Set*/
	PacDurRam->mcsOut[vapIndex].mscOut11nAc.val= 0x24924890;
	PacDurRam->mcsOut[vapIndex].mscOut11ag.val = 0x924480;
	PacDurRam->mcsOut[vapIndex].mscOut11b.val = 0xFAC688;
}

#ifdef ENET_INC_LMAC
/**********************************************************************************

PacDur_ConfigureAutoReplyPower 


Description:
------------
Configures the auto reply power tables in the PAC DUR RAAM.

Input:
-----
	autoReplyPowerParameters - the struct that conatins all the power parameters
		
Output:
-------
	

Returns:
--------
	 
	
**********************************************************************************/
void Pac_DurConfigureAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters,uint8 autoReplyRate)
{
	AutoReplyPower_t autoReplyPowerValues;

	autoReplyPowerValues.val = 0;

	DEBUG_ASSERT(autoReplyRate < LM_PHY_11A_MAX_RATES);

    /* Currently 160 MHZ is not being used so it is not set, when it is added the value should be set here */
	autoReplyPowerValues.bitFields.txPower11B = autoReplyPowerParameters->autoReplyPower11B;
	autoReplyPowerValues.bitFields.txPower11AGBw20 = autoReplyPowerParameters->autoReplyPower11AGBw20;
	autoReplyPowerValues.bitFields.txPower11AGBw40 = autoReplyPowerParameters->autoReplyPower11AGBw40;
	autoReplyPowerValues.bitFields.txPower11AGBw80 = autoReplyPowerParameters->autoReplyPower11AGBw80;

	PacDurRam->autoReplyPower[autoReplyRate].val = autoReplyPowerValues.val;
}
#endif /* #ifdef ENET_INC_LMAC */


/**********************************************************************************

Pac_DurConfigureVapMode

Description:
------------
	
Input:
-----
	vapIndex - The Vap index to 
		
**********************************************************************************/
void Pac_DurConfigureVapMode(uint16 vapIndex, uint16 vapMode)
{
	RegPacDurTxFrameAddr3SelectVap_u durTxFrameAddr3SelectVap; 
	
	RegAccess_Read(REG_PAC_DUR_TX_FRAME_ADDR3_SELECT_VAP, &durTxFrameAddr3SelectVap.val);
	durTxFrameAddr3SelectVap.bitFields.txFrameAddr3SelectVap &= ~(0x1 << vapIndex);
	durTxFrameAddr3SelectVap.bitFields.txFrameAddr3SelectVap |= (vapMode << vapIndex);
	RegAccess_Write(REG_PAC_DUR_TX_FRAME_ADDR3_SELECT_VAP, durTxFrameAddr3SelectVap.val);
}
/**********************************************************************************

Pac_DurGetTotalAirTime


Description:
------------
	
Input:
-----
	vapIndex - The Vap index to 
		
**********************************************************************************/
uint32 Pac_DurGetTotalAirTime(void)
{
	RegPacDurRxToMeAirTimeCtrRd_u regPacDurRxToMeAirTimeCtrRd;

	regPacDurRxToMeAirTimeCtrRd.val = 0x0;
	RegAccess_Read(REG_PAC_DUR_RX_TO_ME_AIR_TIME_CTR_RD, &regPacDurRxToMeAirTimeCtrRd.val);

	return regPacDurRxToMeAirTimeCtrRd.bitFields.rxToMeAirTimeCtrRd;
}

/**********************************************************************************

Pac_DurResetTotalAirTime



Description:
------------
	
Input:
-----

		
**********************************************************************************/
void Pac_DurResetTotalAirTime(void)
{
	RegPacDurRxToMeAirTimeCtrWr_u regPacDurRxToMeAirTimeCtrWr;

	regPacDurRxToMeAirTimeCtrWr.val = 0x0;
	RegAccess_Write(REG_PAC_DUR_RX_TO_ME_AIR_TIME_CTR_WR, regPacDurRxToMeAirTimeCtrWr.val);
}

/**********************************************************************************

Pac_DurInit

Description:
------------
	Initialize Dur HW

**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void Pac_DurInit(void)
{
	RegPacDurSifsPlusPhyRxReadyAirTimeDur_u durSifsPlusPhyRxReadyAirTimeDurReg;
	RegPacDurRxNavTimerLoadRxdEventsA_u     navSetRxdLines0To15Register;
	RegPacDurRxNavTimerLoadRxdEventsB_u     navSetRxdLines16To31Register;
	RegPacDurRxNavTimerLoadRxdEventsC_u     navSetRxdLines32To33Register;
	RegPacDurAckControl_u					ackControlRegister;

	durSifsPlusPhyRxReadyAirTimeDurReg.val = 0;
	navSetRxdLines0To15Register.val = 0;
	navSetRxdLines16To31Register.val = 0;
	navSetRxdLines32To33Register.val = 0;
	ackControlRegister.val = 0;
	
	PacDurRam = (PacDurRam_t *)DUR_AUTO_RATE_MEM_BASE_ADDR;

	pacDurFillRamAutoReplyAirTime();

	// Fill Auto reply rates for Global VAP
	Pac_DurConfigureAutoReplyRate(PAC_DUR_RAM_MCS_OUT_GENERAL_VAP_INDEX);

	// 11ag: SIFS = 16uSec  Phy RX Delay = (Legacy) 26uSec   Total = 42uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11Ag     = RXC_SIFS_PLUS_PHY_RX_DELAY_11AG; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11a/g/n/ac [microseconds].
	// 11b short preamble: SIFS = 10uSec  Phy RX Delay = (11b short) 98uSec   Total = 108uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11BShort = RXC_SIFS_PLUS_PHY_RX_DELAY_11B_SHORT_PREAMBLE; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11b short preamble [microseconds]
	// 11b long preamble: SIFS = 10uSec  Phy RX Delay = (11b long) 98*2 uSec   Total = 206uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11BLong  = RXC_SIFS_PLUS_PHY_RX_DELAY_11B_LONG_PREAMBLE; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11b long preamble [microseconds]
	RegAccess_Write(REG_PAC_DUR_SIFS_PLUS_PHY_RX_READY_AIR_TIME_DUR, durSifsPlusPhyRxReadyAirTimeDurReg.val);

	navSetRxdLines0To15Register.val = PAC_DUR_NAV_SET_RXD_LINES_0_TO_15_INIT_VALUE;
	RegAccess_Write(REG_PAC_DUR_RX_NAV_TIMER_LOAD_RXD_EVENTS_A, navSetRxdLines0To15Register.val);

	navSetRxdLines16To31Register.val = PAC_DUR_NAV_SET_RXD_LINES_16_TO_31_INIT_VALUE;
	RegAccess_Write(REG_PAC_DUR_RX_NAV_TIMER_LOAD_RXD_EVENTS_B, navSetRxdLines16To31Register.val);
	
	navSetRxdLines32To33Register.val = PAC_DUR_NAV_SET_RXD_LINES_32_TO_33_INIT_VALUE;
	RegAccess_Write(REG_PAC_DUR_RX_NAV_TIMER_LOAD_RXD_EVENTS_C, navSetRxdLines32To33Register.val);

	ackControlRegister.bitFields.ackControlNoAck = (DUR_ACK_CTRL_NO_ACK1_MSK | DUR_ACK_CTRL_NO_ACK2_MSK |	DUR_ACK_CTRL_NO_ACK3_MSK);
	RegAccess_Write(REG_PAC_DUR_ACK_CONTROL, ackControlRegister.val);

}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

