/***********************************************************************************
 File:		PacRxc.c
 Module:		Pac Driver
 Purpose:		
 Description:	FW driver for Rxc module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "PacRxcRegs.h"
#include "Pac_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "RxcRam.h"
#include "RegAccess_Api.h"
#include "RxHandler_InitApi.h"
#include "MT_MAC_PAC_regs.h"

#ifdef ENET_INC_LMAC
#include "stringLibApi.h"
#include "RxHandler_InitApi.h"
#include "PacDurRegs.h" // move to DUR module the config of 
#include "lmi.h"
#include "PhyDriver_API.h"
#include "HwGlobalDefinitions.h"
#endif /* #ifdef ENET_INC_LMAC */





/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 11
	
	
#define RXC_NEAR_FULL_THRESHOLD					(0)
#define RXC_ENABLE_BUFFER_OVERFLOW_IND			(1)
#define RXC_ENABLE_FIFO_OVERFLOW_IND			(1)
#define RXC_ENABLE_DMA_BUFFER_OUT_INCREMENT		(1)

#define RXC_GLOBAL_ENABLE						(1)
#define RXC_GLOBAL_DISABLE						(0)
#define RXC_ENABLE_AUTO_KEEP					(1)  /* Retain only frames selected by decision matrix (RXD) */ 
#define RXC_HW_DELAY							(10) /*  RXC Delay in cycles for CCA and */

#define TRAFFIC_IND_GET_ROW 					(2) /*  */
#define TRAFFIC_IND_GET_COLUMN					(4) /*  */
#define TRAFFIC_IND_MUL_8						(3) /*  */


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define RXC_STA_ACTIVITY_BIT_LOCATION_MASK			  (0x1F)

#define RXC_MINIMAL_RSSI_THRESHOLD_VALUE		  (0xAE) // -128 + 46  --> 0x80 + 0x2e

#define RXC_ENABLE_DISABLE_RXD_LINES_A_TRAFFIC_BA_RECIPIENT_IND   ((1 << RCD_ERR_CAT_ACK) | (1 << RCD_BAR_BACK) | (1 << RCD_UQDN_ACK_AGG) | (1 << RCD_ACTION_LAST_IN_AGG))
#define RXC_ENABLE_DISABLE_RXD_LINES_B_TRAFFIC_BA_RECIPIENT_IND   (0x0)
#define RXC_INDICATION_SET 1

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/


typedef enum
{
	RXC_DISABLE_PHY_RX_ABORT_RSSI = 0,
	RXC_ENABLE_PHY_RX_ABORT_RSSI = 1
} RxcPhyRxAbortRssiThresholdStatus_e;


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/* Pointer to start of RXC traffic indications RAM */
#ifdef ENET_INC_UMAC	
static RxcTrafficIndRam_t* RxcTrafficIndRamPtr; 
#endif
/*---------------------------------------------------------------------------------
/						Static Function Declaration 								
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

void Pac_RxcSetForceCca(void)
{
	uint32 regVal = 0;
	
	/* read */
	RegAccess_Read(REG_PAC_RXC_MT_FORCE_CCA, &regVal);

	ILOG0_D("[PTA][Pac_RxcSetForceCca] read regVal = 0x%x", regVal);

	/* Change MUX to force */
	regVal |= (1 << REG_MT_FORCE_SW_CCA_SHIFT);
	
	/* Modify the value: set CCA to "NOSIY" ('1') */
	regVal &= ~(1 << REG_MT_FORCED_SW_CCA_SHIFT);

	ILOG0_D("[PTA][Pac_RxcSetForceCca] write regVal = 0x%x", regVal);

	/* write */
	RegAccess_Write(REG_PAC_RXC_MT_FORCE_CCA, regVal);
}

void Pac_RxcSetForceCcaToNormalMode(void)
{
	uint32 regVal = 0;
	
	/* read */
	RegAccess_Read(REG_PAC_RXC_MT_FORCE_CCA, &regVal);

	ILOG0_D("[PTA][Pac_RxcSetForceCcaToNormalMode] read regVal = 0x%x", regVal);

	/* Change MUX to normal state */
	regVal &= ~(1 << REG_MT_FORCE_SW_CCA_SHIFT);

	/* Modify the value: set CCA to "NORMAL" ('1') */
	// nothying to do

	ILOG0_D("[PTA][Pac_RxcSetForceCcaToNormalMode] write regVal = 0x%x", regVal);
	
	/* write */
	RegAccess_Write(REG_PAC_RXC_MT_FORCE_CCA, regVal);
}


/**********************************************************************************

Pac_RxcEnable


Description:
------------

Input:
-----
	Enable RXC HW
Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
void Pac_RxcEnable(void)
{
	RegPacRxcRxControl_u rxControlReg; 
	
	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL,&rxControlReg.val);
	rxControlReg.bitFields.rxcEnable = RXC_GLOBAL_ENABLE;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);				
}

/**********************************************************************************

Pac_RxcDisable


Description:
------------
	Disable RXC HW
Input:
-----

Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
void Pac_RxcDisable(void)
{
	RegPacRxcRxControl_u rxControlReg; 

	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL,&rxControlReg.val);
	rxControlReg.bitFields.rxcEnable = RXC_GLOBAL_DISABLE;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);		 
}


/**********************************************************************************

Pac_RxcGetState

Description:
------------
	Return the current state of the EXC SM
Input:
-----
	None
	
Returns:
--------
	RxcState_e - Current state of RXC
	
**********************************************************************************/
RxcState_e Pac_RxcGetState(void)
{
	RegPacRxcRxcState_u rxcStateReg;
	RxcState_e retVal;
	
	RegAccess_Read(REG_PAC_RXC_RXC_STATE,&rxcStateReg.val);

	retVal = (RxcState_e)rxcStateReg.bitFields.rxcState;	

	return retVal;
}


#ifdef ENET_INC_UMAC
/**********************************************************************************

Pac_RxcUpdateMinimalRssiThreshold

Description: 
------------

Input:  
-----
rssiThreshlod

Output:
-------
 None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcUpdateMinimalRssiThreshold(int8 rssiThreshlod)
{
	RegPacRxcMinimalRssiThreshold_u minimalRssiThresholdReg; // Minimal RSSI threshold. , Signed value. , Used for setting PHY abort and NAV update.

	// Set Minimal RSSI threshold value. Used for: 1) Decision of PHY abort as MAX RSSI < threshold 2) Input to RXD as MAX RSSI not below threshold  
	minimalRssiThresholdReg.val = rssiThreshlod+0x2E;
	RegAccess_Write(REG_PAC_RXC_MINIMAL_RSSI_THRESHOLD, minimalRssiThresholdReg.val);
}
/**********************************************************************************
Pac_RxcBaAgreementRecipientGetAndClearActivity


Description:
------------
Reads and clears the activity bit in the BA agreement recipient activity bitmap
	
Input:
-----
stationIndex - the station Index to check
tid - the TID to check
	
Output:
-------
	
	
Returns:
--------
indication if there was activity or not
	
**********************************************************************************/
bool Pac_RxcBaAgreementRecipientGetAndClearActivity(StaId stationIndex, uint8 tid)
{
	bool bitIsSet =  FALSE;
	uint32 mask = 0;
	
	uint8 row = 0;    /* In which row is the station located 0-31. See rxc_rxd_rxf.doc section 2.2.1 */
	uint8 column = 0; /* In which column is the the station located 0-3. */
	
	row = stationIndex >> TRAFFIC_IND_GET_ROW;		/* 2: Every row has 4 stations hence divide by 4 */
	column = stationIndex % TRAFFIC_IND_GET_COLUMN;	/* 4: */

	mask = RXC_INDICATION_SET << (tid + (column << TRAFFIC_IND_MUL_8)); /* 3: multiply column by 8 to get to the correct station (jump over tid0-7 of all the non-interesting stations) and add tid posistion*/ 


    /* Read the bit */
	bitIsSet = ((RxcTrafficIndRamPtr->BaAgreementRecipientActivityBitmap[row].val & mask) >> (tid + (column << TRAFFIC_IND_MUL_8))); /* 3: move bit to location '0' */

	/* Clear the bit - write '1' only to this bit in order to clear it */
	RxcTrafficIndRamPtr->BaAgreementRecipientActivityBitmap[row].val = mask;  
	
	return bitIsSet;
}


/**********************************************************************************
Pac_RxcBaAgreementInitiatorGetAndClearActivity


Description:
------------
Reads and clears the activity bit in the BA agreement initiator activity bitmap
	
Input:
-----
stationIndex - the station Index to check
tid - the TID to check
	
Output:
-------
	
	
Returns:
--------
indication if there was activity or not
	
**********************************************************************************/
bool Pac_RxcBaAgreementInitiatorGetAndClearActivity(StaId stationIndex, uint8 tid)
{
	bool bitIsSet =  FALSE;
	uint32 mask = 0;

	uint8 row = 0;    /* In which row is the station located 0-31. See rxc_rxd_rxf.doc section 2.2.1 */
	uint8 column = 0; /* In which column is the the station located 0-3. */
	
	row = stationIndex >> TRAFFIC_IND_GET_ROW;		/* 2: Every row has 4 stations hence divide by 4 */
	column = stationIndex % TRAFFIC_IND_GET_COLUMN;	/* 4: */

	mask = RXC_INDICATION_SET << (tid + (column << TRAFFIC_IND_MUL_8)); /* 3: multiply column by 8 to get to the correct station (jump over tid0-7 of all the non-interesting stations) and add tid posistion*/ 

    /* Read the bit */
	bitIsSet = ((RxcTrafficIndRamPtr->BaAgreementInitiatorActivityBitmap[row].val & mask) >> (tid + (column << TRAFFIC_IND_MUL_8))); /* 3: move bit to location '0' */

	/* Clear the bit - write '1' only to this bit in order to clear it */
	RxcTrafficIndRamPtr->BaAgreementInitiatorActivityBitmap[row].val = mask;  

	return bitIsSet;
}

/**********************************************************************************
Pac_RxcGetAndClearStationActivity

Description:
------------
	Reads and clears the activity bit of the specific station
Input:
-----
	stationIndex - the station index to check
Output:
-------
	
Returns:
--------
	indication if there was activity or not
**********************************************************************************/
uint8 Pac_RxcGetAndClearStationActivity(StaId stationIndex)
{
	uint32 mask = 0;	
	uint8 bitIsSet =  FALSE;
	uint8 activityBitWordNumber = 0;
	
	activityBitWordNumber = CONVERT_BIT_INDEX_TO_WORDS(stationIndex);	
	mask = RXC_INDICATION_SET << (stationIndex & RXC_STA_ACTIVITY_BIT_LOCATION_MASK);

	/* Check the bit */
	if (RxcTrafficIndRamPtr->stationActivityBitmap[activityBitWordNumber] & mask)
	{
		bitIsSet = TRUE;
	}

	/* Clear the bit - write '1' only to this bit in order to clear it */
	RxcTrafficIndRamPtr->stationActivityBitmap[activityBitWordNumber] = mask;  

	return bitIsSet;
}

/**********************************************************************************

Pac_RxcInitializeRamParameters


Description:
------------
Initialize the RXC RAM parameters
Input:
-----

Output:
-------

Returns:
--------
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void Pac_RxcInitializeRamParameters(void)
{
	RxcTrafficIndRamPtr = (RxcTrafficIndRam_t *)RXC_TRAFFIC_INDICATION_CLEAR_BY_ONE_BITMAP;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

#else /* #ifdef ENET_INC_UMAC */

/**********************************************************************************

Pac_RxcClearRxBufferOverflow

Description:
------------
Clear the RX Cyclic buffer

Input:
-----
	None	
Output:
-------
		None

Returns:
--------
		None
**********************************************************************************/
void  Pac_RxcClearRxBufferOverflow(void)
{
	RegPacRxcRxControl_u rxControlReg;
	
	RegAccess_Read(REG_PAC_RXC_RX_CONTROL,&rxControlReg.val);
	rxControlReg.bitFields.rxBufferOverflowWpulse = RXC_ENABLE_BUFFER_OVERFLOW_IND;
	rxControlReg.bitFields.rxFifoOverflowWpulse = RXC_ENABLE_FIFO_OVERFLOW_IND;

	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);
}

/**********************************************************************************

Pac_RxcReset


Description:
------------

Input:
-----

Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void Pac_RxcReset(void)
{
	RegPacRxcRxBufferInPointerWr_u rxBufferInPtrReg;
	RegPacRxcRxBufferOutPointerWr_u rxBufferOutPtrReg;

	rxBufferInPtrReg.val = 0;
	rxBufferOutPtrReg .val = 0;

	/* Receive circular buffer pointers only initialised at startup */	 
	RegAccess_Write(REG_PAC_RXC_RX_BUFFER_IN_POINTER_WR,rxBufferInPtrReg.val);
	RegAccess_Write(REG_PAC_RXC_RX_BUFFER_OUT_POINTER_WR,rxBufferOutPtrReg.val);
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

/**********************************************************************************

Pac_RxcSetReservedStorageSize

Description: configures reserved size in bytes in the phy header
------------

Input:  reservedStorageSize is 6 bit size
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcSetReservedStorageSize(uint8 reservedStorageSize)
{
	RegPacRxcRxControl_u rxControlReg;
	rxControlReg.val = 0;	
	
	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);
	rxControlReg.bitFields.reservedStorageSize = reservedStorageSize;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);		 

}

/**********************************************************************************

Pac_RxcEnableRssiAbort

Description: enable rssi abort
------------

Input: 
-----
	None

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcEnableRssiAbort(void)
{
	RegPacRxcRxControl_u rxControlReg;
	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);	
	rxControlReg.bitFields.phyRxAbortRssiEn = RXC_ENABLE_PHY_RX_ABORT_RSSI;	
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);
}

/**********************************************************************************

Pac_RxcDisableRssiAbort

Description: disable rssi abort
------------

Input:  6 bit size
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcDisableRssiAbort(void)
{
	RegPacRxcRxControl_u rxControlReg;
	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);	
	rxControlReg.bitFields.phyRxAbortRssiEn = RXC_DISABLE_PHY_RX_ABORT_RSSI;	
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);
}

/**********************************************************************************

Pac_RxcDisableCcaIndication


Description: disable CCA  indication on secondary channels , both for secondary 20Mhz and 40Mhz 
------------

Input:  6 bit size
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcDisableCcaIndication(void)
{
	RegPacRxcMtForceCca_u forceCcaReg;
	forceCcaReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &forceCcaReg.val);	
	forceCcaReg.bitFields.forceCca20SFree = 1; 
	forceCcaReg.bitFields.forceCca40SFree = 1;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, forceCcaReg.val);
}

/**********************************************************************************

Pac_RxcReadRxBufferOverflowCounter

Description: Counts MPDUs that dropped when RXC buffer overflow occured
------------

Input:  6 bit size
-----

Output:
-------
 uint16 counter
Returns:
--------
    None
**********************************************************************************/
uint16 Pac_RxcReadRxBufferOverflowCounter(void)
{
	RegPacRxcBufferOverflowCtr_u rxBufferOverflowCounter;
	rxBufferOverflowCounter.val = 0;

	RegAccess_Read(REG_PAC_RXC_BUFFER_OVERFLOW_CTR, &rxBufferOverflowCounter.val);

	return (rxBufferOverflowCounter.bitFields.bufferOverflowCtr);
}


/**********************************************************************************

Pac_RxcClearRxBufferOverflowCounter

Description: Clear RxC bufer overflow dropped mpdu counter
------------

Input:  
-----
None

Output:
-------
 None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcClearRxBufferOverflowCounter(void)
{
	RegPacRxcBufferOverflowCtrClear_u rxcBufferOverflowCtrClrReg;
	rxcBufferOverflowCtrClrReg.bitFields.bufferOverflowCtrClear = 1;
	
	RegAccess_Write(REG_PAC_RXC_BUFFER_OVERFLOW_CTR_CLEAR, rxcBufferOverflowCtrClrReg.val);
}



/**********************************************************************************

Pac_RxcInit


Description:
------------
	Initialize RXC HW
Input:
-----

Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
#pragma ghs section text=".initialization" 
void Pac_RxcInit(void)
{
	RegPacRxcRxControl_u rxControlReg;
	RegPacRxcNearFullThreshold_u nearFullThresholdReg;
	RegPacRxcCcaIfsCompensation_u ccaIfsCompReg;
	RegPacRxcRxrdyIfsCompensation_u rxReadyIfsCompReg;
	RegPacRxcTrafficIndicatorEn_u trafficIndicatorEnReg; // enable/disable of traffic indications of the three indication bitmaps
	RegPacRxcTrafficBaRecipientRxdLinesA_u trafficBaRecipientRxdLinesAReg; // Each bit per RXD line (lines 0to31), , Indicates to update BA Agreement recipient bitmap due to BA transmission 
	RegPacRxcTrafficBaRecipientRxdLinesB_u trafficBaRecipientRxdLinesBReg; // Each bit per RXD line (lines 32to33), , Indicates to update BA Agreement recipient bitmap due to BA transmission
	RegPacRxcMinimalRssiThreshold_u minimalRssiThresholdReg; // Minimal RSSI threshold. , Signed value. , Used for setting PHY abort and NAV update.
	uint32 ccaIfsCompenstation = 0; 
	uint32 rxReadyIfsCompenstation = 0; 
	
	rxControlReg.val = 0;
	nearFullThresholdReg.val = 0;
	ccaIfsCompReg.val = 0;
	rxReadyIfsCompReg.val = 0;
	trafficIndicatorEnReg.val = 0;
	trafficBaRecipientRxdLinesAReg.val = 0;
	trafficBaRecipientRxdLinesBReg.val = 0;
	minimalRssiThresholdReg.val = 0;

	nearFullThresholdReg.bitFields.nearFullThreshold = RXC_NEAR_FULL_THRESHOLD;
	RegAccess_Write(REG_PAC_RXC_NEAR_FULL_THRESHOLD,nearFullThresholdReg.val);
	
    /* Rxc Compensations*/
	ccaIfsCompenstation = Pac_GetHwDelays(PAC_DELAY_RXC_CCA);
    ccaIfsCompReg.bitFields.ccaIfsCompensation = System_ConvertNsToSystemClk(ccaIfsCompenstation) - RXC_HW_DELAY;
	RegAccess_Write(REG_PAC_RXC_CCA_IFS_COMPENSATION,ccaIfsCompReg.val);
    rxReadyIfsCompenstation = Pac_GetHwDelays(PAC_DELAY_RXC_RX_READY);
	rxReadyIfsCompReg.bitFields.rxrdyIfsCompensation = System_ConvertNsToSystemClk(rxReadyIfsCompenstation) - RXC_HW_DELAY;
	RegAccess_Write(REG_PAC_RXC_RXRDY_IFS_COMPENSATION,rxReadyIfsCompReg.val);
	

    rxControlReg.bitFields.reservedStorageSize = FRAME_PREFIX_RESERVED_LENGTH_IN_WORDS;
	rxControlReg.bitFields.rxcAutoKeep = RXC_ENABLE_AUTO_KEEP;
	rxControlReg.bitFields.rxBufferOverflowWpulse = RXC_ENABLE_BUFFER_OVERFLOW_IND;
	rxControlReg.bitFields.rxFifoOverflowWpulse = RXC_ENABLE_FIFO_OVERFLOW_IND; 
	rxControlReg.bitFields.rxBufOutPtrRxDmaIncEn = RXC_ENABLE_DMA_BUFFER_OUT_INCREMENT;
	
	// Disable abort PHY-MAC session based on RSSI threshold. - wave500 new configuration
	rxControlReg.bitFields.phyRxAbortRssiEn = RXC_ENABLE_PHY_RX_ABORT_RSSI;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);

	// initialize the pointer to the start of RXC RAM  	

	/* Clear (Set to zero) the entire memory of the traffic indication Bitmap */
	memset32((void *)(RXC_TRAFFIC_INDICATION_CLEAR_BY_ONE_BITMAP + RXC_TRAFFIC_INDICATION_REGULAR_BITMAP_OFFSET), 0, CONVERT_BYTES_TO_WORDS(sizeof(RxcTrafficIndRam_t)));

	// enable indications for the traffic indications bitmaps
	trafficIndicatorEnReg.bitFields.trafficStaActivityEn = RXC_TRAFFIC_INDICATION_ENABLE;
	trafficIndicatorEnReg.bitFields.trafficBaAgreementRecipientEn = RXC_TRAFFIC_INDICATION_ENABLE;
	trafficIndicatorEnReg.bitFields.trafficBaAgreementOriginatorEn = RXC_TRAFFIC_INDICATION_ENABLE;
	trafficIndicatorEnReg.bitFields.ignoreMultProbeReqTrafficStaActivityEn = TRUE;
	RegAccess_Write(REG_PAC_RXC_TRAFFIC_INDICATOR_EN, trafficIndicatorEnReg.val);

	// Configure RXD lines if to update BA Aggrement recipient bitmap due to BA transmission
	trafficBaRecipientRxdLinesAReg.val = RXC_ENABLE_DISABLE_RXD_LINES_A_TRAFFIC_BA_RECIPIENT_IND;
	RegAccess_Write(REG_PAC_RXC_TRAFFIC_BA_RECIPIENT_RXD_LINES_A, trafficBaRecipientRxdLinesAReg.val);

	trafficBaRecipientRxdLinesBReg.val = RXC_ENABLE_DISABLE_RXD_LINES_B_TRAFFIC_BA_RECIPIENT_IND;
	RegAccess_Write(REG_PAC_RXC_TRAFFIC_BA_RECIPIENT_RXD_LINES_B, trafficBaRecipientRxdLinesBReg.val);

	// Set Minimal RSSI threshold value. Used for: 1) Decision of PHY abort as MAX RSSI < threshold 2) Input to RXD as MAX RSSI not below threshold  
	minimalRssiThresholdReg.val = RXC_MINIMAL_RSSI_THRESHOLD_VALUE;
	RegAccess_Write(REG_PAC_RXC_MINIMAL_RSSI_THRESHOLD, minimalRssiThresholdReg.val);

}
#pragma ghs section text=default

#endif /* #ifdef ENET_INC_UMAC */


