/***********************************************************************************
 File:		PacRxd.c
 Module:		Pac RXD Driver
 Purpose:		
 Description:	FW driver for Rxd module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "PacRxdRegs.h"
#include "PacRxfRegs.h"
#include "Rxf_Common.h"
#include "RegAccess_Api.h"

#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 13

#define RXD_CONDITION_ROW_LENGTH				(4)
#define RXD_CONDITION_ROW_INIT_VALUE			(0)
#define RXD_CONDITION_ROW_SNIFFER_VALUE			(RXD_COND_MSK_RXC_FRAME_ABORT_MISS)
#define RXD_CONDITION_ROW_RSSI_VALUE	 		(RXD_COND_MSK_RXC_FRAME_ABORT_MISS | RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT)
#define RXD_DISABLE_ALL_RX_EVENTS				(0)
#define RXD_DISABLE_ALL_RX						(0) /* All frames are not stored to cyclic buffer*/
#define RXD_ENABLE_CONCAT						(1)
#define RXD_MAX_LOWER_EVENT_IDX					(31)
#define RXD_MAX_LOWER_EVENTS_NUM				(32)
#define RXD_OVERFLOW_ABORT_ENABLE				(1)
#define RXD_PHY_ERROR_ENABLE					(1)
#define RXD_CRC_ERROR							(1)
#define RXD_OVERFLOW_ABORT_DISABLE				(0)
#define RXD_PHY_ERROR_DISABLE					(0)
#define RXD_CRC_ERROR_DISABLE					(0)

/* Conditions - columns in RXD matrix */

/* Receive condition matrix rows */

/*Notes on Long and Bytes filter usage:
Long 0 = my address
Long 1 = BSS-ID
Long 2 = not used
Byte 0 = more fragment bit
Byte 1 = more data bit
Byte 2 = multicast address bit
Byte 3 = not used
Byte 4 = bit 4 subtype     \
Byte 5 = bit 5 subtype      \__These distinguish the various data and control subtypes when combined in receive matrix
Byte 6 = bit 6 subtype      /
Byte 7 = bit 7 subtype     /
Byte 8 = management type
Byte 9 = data type
byte 10 = control type
byte 11 = not used
*/

/* columns number 0 -31 */
#define RXD_COND_MSK_EMPTY         					(0) 		 // No condition
#define RXD_COND_MSK_BYTE_FILT_0_HIT				(0x1U << 0) // more fragment bit HIT
#define RXD_COND_MSK_BYTE_FILT_0_MISS				(0x1U << 1) // more fragment bit MISS
#define RXD_COND_MSK_BYTE_FILT_1_HIT				(0x1U << 2) // more data bit HIT
#define RXD_COND_MSK_BYTE_FILT_1_MISS				(0x1U << 3) // more data bit MISS
#define RXD_COND_MSK_BYTE_FILT_2_HIT				(0x1U << 4) // multicast address bit HIT  - FW config
#define RXD_COND_MSK_BYTE_FILT_2_MISS				(0x1U << 5) // multicast address bit MISS - FW config
#define RXD_COND_MSK_MULTICAST_3_HIT				(0x1U << 6) // multicast address bit HIT - HW config
#define RXD_COND_MSK_MULTICAST_3_MISS				(0x1U << 7) // multicast address bit MISS - HW config
#define RXD_COND_MSK_SUBTYPE_BIT_0_HIT				(0x1U << 8) // bit 4 subtype HIT
#define RXD_COND_MSK_SUBTYPE_BIT_0_MISS				(0x1U << 9) // bit 4 subtype MISS
#define RXD_COND_MSK_SUBTYPE_BIT_1_HIT				(0x1U << 10) // bit 5 subtype HIT 
#define RXD_COND_MSK_SUBTYPE_BIT_1_MISS				(0x1U << 11) // bit 5 subtype MISS 
#define RXD_COND_MSK_SUBTYPE_BIT_2_HIT				(0x1U << 12) // bit 6 subtype HIT 
#define RXD_COND_MSK_SUBTYPE_BIT_2_MISS				(0x1U << 13) // bit 6 subtype MISS  
#define RXD_COND_MSK_SUBTYPE_BIT_3_HIT				(0x1U << 14) // bit 7 subtype HIT 
#define RXD_COND_MSK_SUBTYPE_BIT_3_MISS				(0x1U << 15) // bit 7 subtype MISS 
#define RXD_COND_MSK_MANAGEMENT						(0x1U << 16) // Management type
#define RXD_COND_MSK_DATA							(0x1U << 17) // Data type
#define RXD_COND_MSK_CONTROL						(0x1U << 18) // control type
#define RXD_COND_MSK_DO_NOT_ACK 					(0x1U << 19)
#define RXD_NO_PHY_RX_ERROR 						(0x1U << 20) // new - set when there is no phy RX error
/*Concatenation Indication - indicates reception of aggregation, when used LF2 Miss is not available*/
#define RXD_COND_MSK_AGGR_IND 		      			(0x1U << 21) // new - Aggregation indication

/* Log filter - 0-2 */
#define RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT  	(0x1U << 22) // Long filter 0 HIT  
#define RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_MISS 	(0x1U << 23) // Long filter 0 MISS
#define RXD_COND_MSK_LONG_FILT_1_BROADCAST_ADD_HIT  (0x1U << 24) // Long filter 1 HIT 
#define RXD_COND_MSK_LONG_FILT_1_BROADCAST_ADD_MISS (0x1U << 25) // Long filter 1 MISS
#define RXD_COND_MSK_LONG_FILT_2_BSS_ID_HIT  		(0x1U << 26) // Long filter 2 HIT
#define RXD_COND_MSK_LONG_FILT_2_BSS_ID_MISS 		(0x1U << 27) // Long filter 2 MISS

/* RXC frame abort */
#define RXD_COND_MSK_RXC_FRAME_ABORT_HIT			(0x1U << 28) // frame abort  (CRC error | PHY error | rx circ buffer overflow HIT
#define RXD_COND_MSK_RXC_FRAME_ABORT_MISS			(0x1U << 29) // frame abort  (CRC error | PHY error | rx circ buffer overflow MISS
/* RX CRC no error */
#define RXD_COND_MSK_RX_CRC_ERR						(0x1U << 30) // CRC error HIT
#define RXD_COND_MSK_RX_CRC_OK						(0x1U << 31) // CRC error MISS

/* columns number 32 -63 */
#define RXD_COND_MSK_STA_DB_IDX_VALID_HIT			(0x1U << 0)  // conditions 32 to 42 are in the HSB register of every event. So the bit location is computed as: condition index - 32 
#define RXD_COND_MSK_STA_DB_IDX_VALID_MISS 			(0x1U << 1)  // condition 33
#define RXD_COND_MSK_MAX_RSSI_NOT_BELOW_THRSHOLD	(0x1U << 2)  // condition 34
#define RXD_COND_MSK_BA_AGGREMENT_STATUS			(0x1U << 3)  // condition 35
#define RXD_COND_MSK_BF_INDICATION					(0x1U << 4)  // condition 36
#define RXD_COND_MSK_PHY_NDP_TIMER_NOT_EXPIRED		(0x1U << 5)  // condition 37
#define RXD_COND_MSK_RXC_REJECT_FILTERS				(0x1U << 6)  // Bss Mode | MPDU type subType | Max MPDU length | A-MPDU with no BA aggr   condition 38
#define RXD_COND_MSK_GROUP_FILT_0					(0x1U << 7)  // MPDU fields group0 filter	condition 39
#define RXD_COND_MSK_GROUP_FILT_1					(0x1U << 8)  // MPDU fields group1 filter   condition 40
#define RXD_COND_MSK_GROUP_FILT_2					(0x1U << 9)  // MPDU fields group2 filter   condition 41
#define RXD_COND_MSK_GROUP_FILT_3					(0x1U << 10) // MPDU fields group3 filter   condition 42
/* Receive ConDitions */

#define RMR_MPDU_FIELDS_G3_H 	(RXD_COND_MSK_GROUP_FILT_3)
#define RMR_MPDU_FIELDS_G3_L 	(RXD_COND_MSK_RXC_FRAME_ABORT_MISS)

#define RMR_MPDU_FIELDS_G2_H	(RXD_COND_MSK_GROUP_FILT_2)
#define RMR_MPDU_FIELDS_G2_L    (RXD_COND_MSK_RXC_FRAME_ABORT_MISS)

#define RMR_MPDU_FIELDS_G1_H  	(RXD_COND_MSK_GROUP_FILT_1)
#define RMR_MPDU_FIELDS_G1_L    (RXD_COND_MSK_RXC_FRAME_ABORT_MISS)

#define RMR_MPDU_FIELDS_G0_H    (RXD_COND_MSK_GROUP_FILT_0)
#define RMR_MPDU_FIELDS_G0_L    (RXD_COND_MSK_RXC_FRAME_ABORT_MISS)

#define RMR_NO_ACK_FILTERS_H    (RXD_COND_MSK_RXC_REJECT_FILTERS)
#define RMR_NO_ACK_FILTERS_L    (RXD_COND_MSK_RXC_FRAME_ABORT_MISS)

#define RMR_ERR_CAT_NOACK_H     RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_ERR_CAT_NOACK_L     (RXD_COND_MSK_RXC_FRAME_ABORT_HIT | RXD_COND_MSK_AGGR_IND | \
							    RXD_COND_MSK_DO_NOT_ACK)

#define RMR_ERR_CAT_ACK_H       RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_ERR_CAT_ACK_L	    (RXD_COND_MSK_RXC_FRAME_ABORT_HIT | RXD_COND_MSK_AGGR_IND | \
							    RXD_NO_PHY_RX_ERROR)

#define RMR_ERR_H				RXD_COND_MSK_EMPTY
#define RMR_ERR_L				RXD_COND_MSK_RXC_FRAME_ABORT_HIT

#define RMR_RTS_H				RXD_COND_MSK_EMPTY
#define RMR_RTS_L				(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_HIT |\
							    RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_HIT | RXD_COND_MSK_SUBTYPE_BIT_0_HIT)

#define RMR_PSPOLL_H			RXD_COND_MSK_EMPTY
#define RMR_PSPOLL_L			(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
								RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_HIT | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_NAV_RESET_H			RXD_COND_MSK_EMPTY
#define RMR_NAV_RESET_L			(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_MISS | RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
								RXD_COND_MSK_SUBTYPE_BIT_2_HIT | RXD_COND_MSK_SUBTYPE_BIT_1_HIT)

#define RMR_BAR_BACK_H          RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_BAR_BACK_L          (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_CONTROL | \
							    RXD_COND_MSK_SUBTYPE_BIT_3_HIT | RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | \
							    RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_BAR_ACK_H           RXD_COND_MSK_EMPTY
#define RMR_BAR_ACK_L           (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
							    RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_NDP_H   		    (RXD_COND_MSK_PHY_NDP_TIMER_NOT_EXPIRED | RXD_COND_MSK_BF_INDICATION | RXD_COND_MSK_STA_DB_IDX_VALID_HIT)	
#define RMR_NDP_L			    (RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_MISS | \
							    RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_BFPOLL_H   		    (RXD_COND_MSK_BF_INDICATION | RXD_COND_MSK_STA_DB_IDX_VALID_HIT)
#define RMR_BFPOLL_L		    (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_MISS | \
							    RXD_COND_MSK_SUBTYPE_BIT_2_HIT | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_NDP_ERR_H   		RXD_COND_MSK_EMPTY
#define RMR_NDP_ERR_L		    (RXD_COND_MSK_CONTROL | RXD_COND_MSK_SUBTYPE_BIT_3_MISS | \
							    RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_UQDN_NOACK_AGG_H    RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_UQDN_NOACK_AGG_L    (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_AGGR_IND | \
								RXD_COND_MSK_DO_NOT_ACK | RXD_COND_MSK_DATA | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
								RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_UQDN_NOACK_H	    RXD_COND_MSK_EMPTY	
#define RMR_UQDN_NOACK_L        (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_DO_NOT_ACK | RXD_COND_MSK_DATA | \
							    RXD_COND_MSK_SUBTYPE_BIT_3_HIT | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_UQDN_ACK_AGG_H	    RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_UQDN_ACK_AGG_L     (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_AGGR_IND | \
							    RXD_COND_MSK_DATA | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
							    RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_UQDN_ACK_H          RXD_COND_MSK_EMPTY
#define RMR_UQDN_ACK_L		    (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_DATA | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
							    RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)
	
#define RMR_UDATA_H			    RXD_COND_MSK_EMPTY	
#define RMR_UDATA_L             (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT| RXD_COND_MSK_DATA)

#define RMR_BEACON_H            RXD_COND_MSK_EMPTY
#define RMR_BEACON_L            (RXD_COND_MSK_LONG_FILT_1_BROADCAST_ADD_HIT | RXD_COND_MSK_MANAGEMENT | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
					            RXD_COND_MSK_SUBTYPE_BIT_2_MISS | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)
 
#define RMR_PROBE_RESPONSE_H	RXD_COND_MSK_EMPTY
#define RMR_PROBE_RESPONSE_L 	(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_MANAGEMENT | RXD_COND_MSK_SUBTYPE_BIT_3_MISS | \
							    RXD_COND_MSK_SUBTYPE_BIT_2_HIT | RXD_COND_MSK_SUBTYPE_BIT_1_MISS | RXD_COND_MSK_SUBTYPE_BIT_0_HIT)


#define RMR_ACTION_LAST_IN_AGG_NO_BACK_H	RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_ACTION_LAST_IN_AGG_NO_BACK_L 	(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_AGGR_IND | RXD_COND_MSK_DO_NOT_ACK | \
							     		 	RXD_COND_MSK_MANAGEMENT |  RXD_COND_MSK_SUBTYPE_BIT_3_HIT | RXD_COND_MSK_SUBTYPE_BIT_2_HIT | \
							     		 	RXD_COND_MSK_SUBTYPE_BIT_1_HIT | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_ACTION_LAST_IN_AGG_H RXD_COND_MSK_BA_AGGREMENT_STATUS
#define RMR_ACTION_LAST_IN_AGG_L (RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_AGGR_IND | \
							     RXD_COND_MSK_MANAGEMENT |  RXD_COND_MSK_SUBTYPE_BIT_3_HIT | RXD_COND_MSK_SUBTYPE_BIT_2_HIT | \
							     RXD_COND_MSK_SUBTYPE_BIT_1_HIT | RXD_COND_MSK_SUBTYPE_BIT_0_MISS)



#define RMR_UMGMT_NOACK_H		RXD_COND_MSK_EMPTY
#define RMR_UMGMT_NOACK_L		(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_MANAGEMENT | RXD_COND_MSK_SUBTYPE_BIT_3_HIT | \
								RXD_COND_MSK_SUBTYPE_BIT_2_HIT | RXD_COND_MSK_SUBTYPE_BIT_1_HIT| RXD_COND_MSK_SUBTYPE_BIT_0_MISS)

#define RMR_UMGMT_H				RXD_COND_MSK_EMPTY
#define RMR_UMGMT_L				(RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_HIT | RXD_COND_MSK_MANAGEMENT)


#define RMR_MULTI_H				RXD_COND_MSK_EMPTY
#define RMR_MULTI_L				RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_MISS | RXD_COND_MSK_MULTICAST_3_HIT		

#define RMR_NAV_LOAD_H			RXD_COND_MSK_MAX_RSSI_NOT_BELOW_THRSHOLD
#define RMR_NAV_LOAD_L 			RXD_COND_MSK_LONG_FILT_0_MY_ADDRESS_MISS		

#define RMR_INVALID         	0x0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

// this matrix holds the lower 32 column bits of every confition
static uint32 RxdLowDefaultRxdMatrix[RXD_CONDITION_MATRIX_ROWS] =
{
	RMR_MPDU_FIELDS_G3_L,              /* 0 */
	RMR_MPDU_FIELDS_G2_L,              /* 1 */
	RMR_MPDU_FIELDS_G1_L,              /* 2 */
	RMR_MPDU_FIELDS_G0_L,              /* 3 */
	RMR_NO_ACK_FILTERS_L,              /* 4 */
	RMR_ERR_CAT_NOACK_L,               /* 5 */
	RMR_ERR_CAT_ACK_L,                 /* 6 */
	RMR_ERR_L,			               /* 7 */
	RMR_RTS_L,			               /* 8 */
	RMR_PSPOLL_L,		               /* 9 */
	RMR_NAV_RESET_L,	               /* 10 */
	RMR_BAR_BACK_L,		               /* 11 */
	RMR_BAR_ACK_L,		               /* 12 */
	RMR_NDP_L,   		               /* 13 */
	RMR_BFPOLL_L,                      /* 14 */
	RMR_NDP_ERR_L,                     /* 15 */
	RMR_UQDN_NOACK_AGG_L,              /* 16 */
	RMR_UQDN_NOACK_L,	               /* 17 */
	RMR_UQDN_ACK_AGG_L,	               /* 18 */
	RMR_UQDN_ACK_L,		               /* 19 */
	RMR_UDATA_L,		               /* 20 */
	RMR_BEACON_L,		               /* 21 */
	RMR_PROBE_RESPONSE_L,              /* 22 */
	RMR_ACTION_LAST_IN_AGG_NO_BACK_L,  /* 23 */
	RMR_ACTION_LAST_IN_AGG_L, 		   /* 24 */
	RMR_UMGMT_NOACK_L,	               /* 25 */
	RMR_UMGMT_L,		               /* 26 */
	RMR_MULTI_L,		               /* 27 */
	RMR_NAV_LOAD_L,		               /* 28 */
	RMR_INVALID,		               /* 29 */
	RMR_INVALID,		               /* 30 */
	RMR_INVALID,		               /* 31 */
	RMR_INVALID,		               /* 32 */
	RMR_INVALID,		               /* 33 */
};


// this matrix holds the higher 32 column bits of every confition
static uint32 RxdHighDefaultRxdMatrix[RXD_CONDITION_MATRIX_ROWS] =
{
	RMR_MPDU_FIELDS_G3_H,               /* 0 */
	RMR_MPDU_FIELDS_G2_H,               /* 1 */
	RMR_MPDU_FIELDS_G1_H,               /* 2 */
	RMR_MPDU_FIELDS_G0_H,               /* 3 */
	RMR_NO_ACK_FILTERS_H,               /* 4 */
	RMR_ERR_CAT_NOACK_H,                /* 5 */
	RMR_ERR_CAT_ACK_H,                  /* 6 */
	RMR_ERR_H,			                /* 7 */
	RMR_RTS_H,			                /* 8 */
	RMR_PSPOLL_H,		                /* 9 */
	RMR_NAV_RESET_H,	                /* 10 */
	RMR_BAR_BACK_H,		                /* 11 */
	RMR_BAR_ACK_H,		                /* 12 */
	RMR_NDP_H,   		                /* 13 */
	RMR_BFPOLL_H,                       /* 14 */
	RMR_NDP_ERR_H,                      /* 15 */
	RMR_UQDN_NOACK_AGG_H,               /* 16 */
	RMR_UQDN_NOACK_H,	                /* 17 */
	RMR_UQDN_ACK_AGG_H,	                /* 18 */
	RMR_UQDN_ACK_H,		                /* 19 */
	RMR_UDATA_H,		                /* 20 */
	RMR_BEACON_H,		                /* 21 */
	RMR_PROBE_RESPONSE_H,               /* 22 */
	RMR_ACTION_LAST_IN_AGG_NO_BACK_H,   /* 23 */
	RMR_ACTION_LAST_IN_AGG_H,		    /* 24 */
	RMR_UMGMT_NOACK_H,	                /* 25 */
	RMR_UMGMT_H,		                /* 26 */
	RMR_MULTI_H,		                /* 27 */
	RMR_NAV_LOAD_H,		                /* 28 */
	RMR_INVALID,		                /* 29 */
	RMR_INVALID,		                /* 30 */
	RMR_INVALID,		                /* 31 */
	RMR_INVALID,		                /* 32 */
	RMR_INVALID,		                /* 33 */
};


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

Pac_RxcDisable

Description:
------------

Input:
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxdDisable(void)
{
	RegPacRxdEventEnableA_u eventEnableAReg; 
	RegPacRxdEventEnableB_u eventEnableBReg; 
	RegPacRxdRxEnableA_u	rxEnableAReg; 
	RegPacRxdRxEnableB_u	rxEnableBReg; 

	eventEnableAReg.val = RXD_DISABLE_ALL_RX_EVENTS;
	eventEnableBReg.val = RXD_DISABLE_ALL_RX_EVENTS;
	
	rxEnableAReg.val = RXD_DISABLE_ALL_RX;
	rxEnableBReg.val = RXD_DISABLE_ALL_RX;

	/*we want to take a special care for clearing EventEnable enable before clearing the RxEnable */
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_A,eventEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_B,eventEnableBReg.val);

	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_A,rxEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_B,rxEnableBReg.val);
}

/**********************************************************************************

Pac_RxdConfigureNormalMode

Description:
------------

Input:
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxdConfigureNormalMode(void)
{

    RegPacRxdEventEnableA_u eventEnableAReg;
	RegPacRxdEventEnableB_u eventEnableBReg;
    RegPacRxdRxEnableA_u rxEnableAReg;
	RegPacRxdRxEnableB_u rxEnableBReg;

    eventEnableAReg.val = 0;
	eventEnableBReg.val = 0;
    rxEnableAReg.val = 0;
	rxEnableBReg.val = 0;

   
   eventEnableAReg.val = ((1 << RCD_MPDU_FIELDS_G3) 				|
						   (1 << RCD_MPDU_FIELDS_G2) 				|
						   (1 << RCD_MPDU_FIELDS_G1) 				|
						   (1 << RCD_MPDU_FIELDS_G0) 				|
						   (1 << RCD_NO_ACK_FILTERS)				|
						   (1 << RCD_ERR_CAT_NOACK)					|
						   (1 << RCD_ERR_CAT_ACK)					|
						   (1 << RCD_ERR)							|
						   (1 << RCD_RTS)							|
						   (1 << RCD_PSPOLL)						|
						   (1 << RCD_NAV_RESET)						|
						   (1 << RCD_BAR_BACK)						|
						   (1 << RCD_BAR_ACK)						|
						   (1 << RCD_NDP)							|
						   (1 << RCD_BFPOLL)                        |
						   (1 << RCD_NDP_ERR)       				|
						   (1 << RCD_UQDN_NOACK_AGG)				|
						   (1 << RCD_UQDN_NOACK)					|
						   (1 << RCD_UQDN_ACK_AGG)					|
						   (1 << RCD_UQDN_ACK)						|
						   (1 << RCD_UDATA)							|
						   (1 << RCD_BEACON)						|
						   (1 << RCD_PROBE_RESPONSE)				|
						   (1 << RCD_ACTION_LAST_IN_AGG_NO_BACK)	|
						   (1 << RCD_ACTION_LAST_IN_AGG)    		|
						   (1 << RCD_UMGMT_NOACK)					| 
						   (1 << RCD_UMGMT)							|
						   (1 << RCD_MULTI)							| 
						   (1 << RCD_NAV_LOAD));
						   
											
	rxEnableAReg.val =    ((1 << RCD_BAR_BACK)						|
						   (1 << RCD_BAR_ACK)						|
						   (1 << RCD_UQDN_NOACK_AGG)				|
						   (1 << RCD_UQDN_NOACK)					|
						   (1 << RCD_UQDN_ACK_AGG)					|
						   (1 << RCD_UQDN_ACK)						|
						   (1 << RCD_UDATA)							|
						   (1 << RCD_BEACON)						|
						   (1 << RCD_PROBE_RESPONSE)				|
						   (1 << RCD_ACTION_LAST_IN_AGG_NO_BACK)	|
						   (1 << RCD_ACTION_LAST_IN_AGG) 			|
						   (1 << RCD_UMGMT_NOACK)					|
						   (1 << RCD_UMGMT)	      					|
						   (1 << RCD_MULTI));

	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_A,eventEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_B,eventEnableBReg.val);

	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_A,rxEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_B,rxEnableBReg.val);
}

/**********************************************************************************

Pac_RxdConfigureScanMode

Description:
------------

Input:
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxdConfigureScanMode(void)
{
    FATAL("Pac_RxdConfigureScanMode");
}


 /**********************************************************************************
 
 Pac_RxdLoadConditionMatrix
 
 Description:
 ------------
    Load the recieved condition matrix from pre defined table defined
 
 Input:
 -----
 
 Output:
 -------
     None
 
 Returns:
 --------
     None
 **********************************************************************************/
void Pac_RxdLoadConditionMatrix(void)
{ 
    /* Initialise receive condition matrix */
	memcpy32((void*)REG_PAC_RXD_CONDITION_0, RxdLowDefaultRxdMatrix, RXD_CONDITION_MATRIX_ROWS);
	memcpy32((void*)REG_PAC_RXD_CONDITION_EXPAND0, RxdHighDefaultRxdMatrix, RXD_CONDITION_MATRIX_ROWS);
}

 /**********************************************************************************
 
 Pac_RxdConfigureBypassMode
 
 Description:
 ------------
    Enable first matrix line only and zero it in order to insure that all packets will pass the RxD filters and that CB pointer will move
 
 Input:
 -----
 	None
 Output:
 -------
     None
 
 Returns:
 --------
     None
 **********************************************************************************/

 void Pac_RxdConfigureBypassMode(void)
 {
	RegPacRxdEventEnableA_u eventEnableAReg;
	RegPacRxdEventEnableB_u eventEnableBReg;
    RegPacRxdRxEnableA_u rxEnableAReg;
	RegPacRxdRxEnableB_u rxEnableBReg;
	RegPacRxdCondition0_u conditionReg;
	RegPacRxdConditionExpand0_u conditionExpandReg;
	RegPacRxdFrameAbort_u frameAbortReg;

	//enable only first line
	eventEnableAReg.val =  0x1;
	eventEnableBReg.val = RXD_DISABLE_ALL_RX_EVENTS;

	//enable rx for first line, means CB pointer will move after the frame will be written to cyclic buffer
	rxEnableAReg.val = 0x1;
	rxEnableBReg.val = RXD_DISABLE_ALL_RX;

	//set condition of first line to 0, to insure it will pass
	conditionReg.val = RXD_CONDITION_ROW_SNIFFER_VALUE; //set bit 29 (not frame abort)
	conditionExpandReg.val = RXD_CONDITION_ROW_INIT_VALUE;


	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_A, eventEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_B, eventEnableBReg.val);
    RegAccess_Write(REG_PAC_RXD_RX_ENABLE_A, rxEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_B, rxEnableBReg.val);
	RegAccess_Write(REG_PAC_RXD_CONDITION_0,conditionReg.val); 
	RegAccess_Write(REG_PAC_RXD_CONDITION_EXPAND0,conditionExpandReg.val);

	frameAbortReg.val = 0;
	frameAbortReg.bitFields.overflowAbort = RXD_OVERFLOW_ABORT_ENABLE;
    frameAbortReg.bitFields.phyError = RXD_PHY_ERROR_ENABLE;
    frameAbortReg.bitFields.crcError = RXD_CRC_ERROR_DISABLE;
    RegAccess_Write(REG_PAC_RXD_FRAME_ABORT,frameAbortReg.val);
	
 }

/**********************************************************************************

Pac_RxdConfigureRssiMode

Description:
------------
   Enable only 2 first lines in RXD: 
   first row will enable writing packet to RXC only if the following conditions are fullfiled:
   	  - CRC OK
(   	  - no RSSI threshold)
(      	  - no BA aggreement)
	  - Long filter address field #2 is equal to input
   second row will disable writing to RXC if all of the conditions in first line are not fullfiled

Input:
-----
   None
Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
void Pac_RxdConfigureRssiMode(void)
{
	// which row # to enable in RXD matrix:
	RegPacRxdEventEnableA_u eventEnableAReg; //rows 0-31
	RegPacRxdEventEnableB_u eventEnableBReg; //rows 32-33

	// which row # will allow writting to RXC:	
    RegPacRxdRxEnableA_u rxEnableAReg;		//rows 0-31
	RegPacRxdRxEnableB_u rxEnableBReg;		//rows 32-33

	// which conditions are set in each row
	RegPacRxdCondition0_u 		conditionReg;		//row 0 - conditions 0-31
	RegPacRxdConditionExpand0_u conditionExpandReg; //row 0 - conditions 32-42
	
	// frame abort register values
	RegPacRxdFrameAbort_u frameAbortReg;

	// long filter control 
	RegPacRxfLongFilter0Control_u longFilter0ControlReg;

	// enable only first 2 lines
	eventEnableAReg.val = 0x3;
	eventEnableBReg.val = RXD_DISABLE_ALL_RX_EVENTS;

	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_A, eventEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_B, eventEnableBReg.val);

	// enable rx for first line, means CB pointer will move after the frame will be written to cyclic buffer
	rxEnableAReg.val = 0x1;
	rxEnableBReg.val = RXD_DISABLE_ALL_RX;
	
    RegAccess_Write(REG_PAC_RXD_RX_ENABLE_A, rxEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_B, rxEnableBReg.val);

	// set condition of first line to: bit 29 (not frame abort) and source filter address field as given
	frameAbortReg.val = 0;
	frameAbortReg.bitFields.overflowAbort = RXD_OVERFLOW_ABORT_ENABLE;
    frameAbortReg.bitFields.phyError = RXD_PHY_ERROR_ENABLE;	      
    frameAbortReg.bitFields.crcError = RXD_CRC_ERROR;		  
    RegAccess_Write(REG_PAC_RXD_FRAME_ABORT,frameAbortReg.val);

	// set control register 0 to point to address 2 offset value (sender)
	longFilter0ControlReg.bitFields.longFilter0Control = RXF_LONG_FILTER_MASK;
	longFilter0ControlReg.bitFields.offsetLongFilter0Control = RXF_LONG_FILTER_ADDRESS_2_OFFSET_VALUE;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_0_CONTROL, longFilter0ControlReg.val);

	// set bit 29 (not frame abort) & set bit 22 (Long filter 0 HIT) 
	conditionReg.val = RXD_CONDITION_ROW_RSSI_VALUE; 
	conditionExpandReg.val = RXD_CONDITION_ROW_INIT_VALUE;
	
	RegAccess_Write(REG_PAC_RXD_CONDITION_0,conditionReg.val); 
	RegAccess_Write(REG_PAC_RXD_CONDITION_EXPAND0,conditionExpandReg.val);

	// set condition of second line to: when conditions don't meet first 
	RegAccess_Write(REG_PAC_RXD_CONDITION_1,~(conditionReg.val)); 
	RegAccess_Write(REG_PAC_RXD_CONDITION_EXPAND1,~(conditionExpandReg.val));	

	// no need to disable other filters such as BA agreement/RSSI threshold since they are not verified in the first two lines
}

/**********************************************************************************

Pac_RxdInit

Description:
------------

Input:
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxdInit(void)
{
    uint8 conditionRowIndex;
    RegPacRxdCondition0_u conditionReg;
    RegPacRxdEventEnableA_u eventEnableAReg;
	RegPacRxdEventEnableB_u eventEnableBReg;
    RegPacRxdRxEnableA_u rxEnableAReg;
	RegPacRxdRxEnableB_u rxEnableBReg;
    RegPacRxdFrameAbort_u frameAbortReg;
	
    uint32 regAddress = 0;
    frameAbortReg.val = 0;

    /* Program RXD block */
    eventEnableAReg.val = RXD_DISABLE_ALL_RX_EVENTS;
	eventEnableBReg.val = RXD_DISABLE_ALL_RX_EVENTS;
    rxEnableAReg.val = RXD_DISABLE_ALL_RX;
	rxEnableBReg.val = RXD_DISABLE_ALL_RX;
    RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_A, eventEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_EVENT_ENABLE_B, eventEnableBReg.val);
    RegAccess_Write(REG_PAC_RXD_RX_ENABLE_A, rxEnableAReg.val);
	RegAccess_Write(REG_PAC_RXD_RX_ENABLE_B, rxEnableBReg.val);
        
    /* frame_abort - Determines which terms are applied to the frame abort matrix columns */
    /* on Rx Buffer or FIFO overflow */
    /* on error signalled by PHY */
    /* Generate frame abort on CRC error */
    frameAbortReg.bitFields.overflowAbort = RXD_OVERFLOW_ABORT_ENABLE;
    frameAbortReg.bitFields.phyError = RXD_PHY_ERROR_ENABLE;
    frameAbortReg.bitFields.crcError = RXD_CRC_ERROR;
    RegAccess_Write(REG_PAC_RXD_FRAME_ABORT,frameAbortReg.val);

    /* Initialise receive condition matrix */
    conditionReg.val = RXD_CONDITION_ROW_INIT_VALUE;

	// initialize the lower 32 columns of the condition table
	regAddress = REG_PAC_RXD_CONDITION_0;
	memset32((void*)regAddress, 0, RXD_CONDITION_MATRIX_ROWS);

	// initialize the higher 32 columns of the condition table
	regAddress = REG_PAC_RXD_CONDITION_EXPAND0;
	memset32((void*)regAddress, 0, RXD_CONDITION_MATRIX_ROWS);

	for (conditionRowIndex = 0; conditionRowIndex < RXD_CONDITION_MATRIX_ROWS; conditionRowIndex++)
    {   
        RegAccess_Write(regAddress,conditionReg.val); 
       
        regAddress = regAddress + RXD_CONDITION_ROW_LENGTH;
    }
}
