/***************************************************************
 File:		Rxf.c
 Module:	  	RXF
 Purpose: 	holds all the implementation of the RXF API functions
 Description:  This module contains the implementation of the functions of the RXF module which is responsible for filtering 
 			the incoming packets in the PAC
***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Pac_Api.h"
#include "PacManager_api.h"
#include "enet_pas.h"
#include "MacRabRegs.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "PacRxfRegs.h"
#include "RegAccess_Api.h"
#include "HwDrivers_Api.h"
#include "stringLibApi.h"
#include "ShramRxfFifo.h"
#include "RxfRam.h"
#include "Rxf_Common.h"
#include "Dut_Api.h"
#ifdef ENET_INC_UMAC
#include "HwEventsAndErrors_Api.h"
#endif /* #ifdef ENET_INC_UMAC */

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 15


#define RXF_MAX_MY_ADDRESS_FILTERS  					(16)
#define RXF_BROADCAST_FILTER 							(16)

/* In order to change the default value (0x3) to the broadcast control register (0x2) the last bit shoould be cleared */
#define RXF_BROADCAST_SELECTION_MASK 					(0xFFFFFFFE)

//mask for disabling all filters
#define RXF_DISABLE_SELECTION_MASK 						(0xFFFFFFFF)

#define RXF_LONG_FILTER_SELECTION_MASK					(0x3)
#define RXF_LONG_FILTER_SELECTION_SIZE 					(2)
#define RXF_LONG_FILTER_SELECTION_MULTIPLY_BY_TWO_SHIFT (1)


#define REG_PAC_RXF_LONG_FILTER_TOTAL_REF_SIZE   		(8)		// refernece val is 6 bytes long and configured in two 32 bits registers (4 LSbytes + 2 HSbyts) 
#define REG_PAC_RXF_LONG_FILTER_TOTAL_REF_SHIFT  		(3)		// amount to shift in order to multiply by 8
#define REG_PAC_RXF_LONG_FILTER_REF1_SIZE        		(4)		// refernece val is 6 bytes long and configured in two 32 bits registers (4 LSbytes + 2 HSbyts) 

#define REG_PAC_RXF_LONG_FILTER_VALUE_ENABLE			(0x0)
#define REG_PAC_RXF_LONG_FILTER_VALUE_DISABLE			(0x3)


#define RXF_SHORT_FILTER_CONFIG_SIZE					(4)

#define RXF_MAX_MPDU_SIZE_PHY_MODE_11_A_G				(MAX_HT_MPDU_IN_A_MPDU_LENGTH_RX)	// Max MPDU size in case of PHY mode 11ag
#define RXF_MAX_MPDU_SIZE_PHY_MODE_11_B					(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX)	// Max MPDU size in case of PHY mode 11b need to be maximum available(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX) - JIRA WLANSW-1907)

#define RXF_MAX_MPDU_SIZE_PHY_MODE_11_N					(MAX_MPDU_LENGTH_OPTION2_HT)  		// Max MPDU size in case of PHY mode 11n
#define RXF_MAX_MPDU_SIZE_PHY_MODE_11_AC				(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX) // Max MPDU size in case of PHY mode 11ac

#define DATA_SUBTYPES_CONFIG							(0xEEEE)
#define MANAGEMENT_SUBTYPES_CONFIG						(0x82C0)
#define CONTROL_SUBTYPES_CONFIG							(0x82C6)

#define RXF_LONG_FILTER_CONFIG_REGS_NUM					(3)		// In enhanced long filter configuration there are three registers per group (ctrl, value, mask)
#define RXF_LONG_FILTER_REGS_SIZE						(HW_REG_SIZE * RXF_LONG_FILTER_CONFIG_REGS_NUM) // size of register block per each of the four groups pf enhanced long filters)

// FIFO RAM 
#define RXF_FIFO_SINGLE_LINE_READ   	  				(1)
#define RXF_FIFO_LINE_SIZE				  				(4) // every line in fifo is 4 bytes	

#define RXF_ENABLE_ALL_VAPS								(0xFFFF)	/* The filter is valid for the VAP index which is set to '1'*/
#define FRAME_CONTROL_FIRST_BYTE_OFFSET					(0x00)		/* BIT2 & BIT3 are located at byte #1 of the MPDU MAC HEADER */
#define FRAME_CONTROL_SECOND_BYTE_OFFSET				(0x01)		/* BIT8 & BIT9 are located at byte #2 of the MPDU MAC HEADER */
#define FRAME_CONTROL_22_BYTE_OFFSET_SEQ_CONTROL		(22)		/* Sequence Control Field is located at byte #24 of the MPDU MAC HEADER */
#define FRAME_CONTROL_24_BYTE_OFFSET_QOS_CONTROL		(24)		/* In Regular case the Qos Control is located at byte #24 of the MPDU MAC HEADER */
#define FRAME_CONTROL_30_BYTE_OFFSET_QOS_CONTROL		(30)		/* In WDS case the Qos Control is located at byte #30 of the MPDU MAC HEADER */
#define FRAME_CONTROL_MASK_TO_DS_FROM_DS				(0x0300)	/* Mask BIT8 & BIT9. After 1 byte offset they are located at 00000011 hence 0x3*/
#define FRAME_CONTROL_TYPE_DATA							(0x08)		/* BIT2 & BIT3 in Frame Control set to 'DATA'*/
#define FRAME_CONTROL_MASK_TYPE							(0x0C)		/* BIT2 & BIT3 in Frame Control Type field mask */
#define FRAME_CONTROL_SUBTYPE_QOSDATA					(0x80)		/* BIT6 & BIT7 in Frame Control set to 'Qos DATA'*/
#define FRAME_CONTROL_MASK_SUBTYPE						(0xC0)		/* BIT6 & BIT7 in Frame Control Subtype field mask */
#define FRAME_CONTROL_MORE_FRAGMENTS_SET				(0x04)		/* B10 in Frame Control set to 'More Fragments' */	
#define FRAME_CONTROL_MASK_MORE_FRAGMENTS				(0x04)		/* B10 in Frame Control 'More Fragments' field mask */
#define FRAME_CONTROL_AMSDU_PRESENT_SET					(0x80)		/* B7 in QoS Control field set to 'A-MSDU Present' */
#define FRAME_CONTROL_MASK_AMSDU_PRESENT				(0x80)		/* B7 in QoS Control 'A-MSDU Present' field */
#define SEQUENCE_CONTROL_FRAGMENT_NUMBER_ZERO			(0x00)		/* B0-B3 bits in Sequence Control field are non-zero */
#define SEQUENCE_CONTROL_FRAGMENT_NUMBER_MASK			(0x0F)		/* B0-B3 bits in Sequence Control field define 'Fragment Number' */


typedef enum
{
	RXF_MPDU_TYPE_DATA		 = 0,
	RXF_MPDU_TYPE_MANAGEMENT = 1,
	RXF_MPDU_TYPE_CONTROL 	 = 2,
	RXF_MPDU_TYPE_NUM 		 = 3
} RxfMpduType_e;

typedef enum
{
	FRAME_CONTROL_TO_DS_FROM_DS_RESERVED0	= 0x0000, 
	FRAME_CONTROL_TO_DS_FROM_DS_REGULAR		= 0x0100,	// Non-WDS case: 01
	FRAME_CONTROL_TO_DS_FROM_DS_RESERVED1	= 0x0200,
	FRAME_CONTROL_TO_DS_FROM_DS_WDS			= 0x0300	// WDS case: 11 		
}RxfElfGroupValue_e;

#define MPDU_TYPE_SUBTYPE_REJECT_FILTER_ARRAY_SIZE (RXF_MPDU_TYPE_NUM * HW_NUM_OF_VAPS)

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
typedef struct RxfAmpduNoBaAggrementFifoManagement
{
	uint16 readIndex;
} RxfAmpduNoBaAggrementFifoManagement_t; // RxfManagement_t;

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
// holds the fifo management parameters
#if defined(ENET_INC_UMAC)
static RxfAmpduNoBaAggrementFifoManagement_t RxfFifoObj;  // RxfManagementObj
#endif

// holds the table that defines per each vap the MPDU type-subtypes that are rejected(Not acknowledged)
// there are 3 types [data,management,control] per each vap. Every type has 16 subtypes.
// so there are 16 bits per type + 16 reserved bits. (3 words per vap) * (16 vaps)
// the filter defines the same default values for all vaps
#if defined(ENET_INC_LMAC)
const static uint32 RxfMpduTypeSubTypeRejectFilterVapsArray[MPDU_TYPE_SUBTYPE_REJECT_FILTER_ARRAY_SIZE] = 
{
	// vap 0
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 1
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 2
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 3
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 4
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 5
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 6
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 7
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 8
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 9
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 10
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 11
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 12
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 13
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 14
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
	// vap 15
	DATA_SUBTYPES_CONFIG,
	MANAGEMENT_SUBTYPES_CONFIG,
	CONTROL_SUBTYPES_CONFIG,
};
#endif /* #ifdef ENET_INC_LMAC */


//---------------------------------------------------------------------------------
//						Static Function Definitions				
//---------------------------------------------------------------------------------



//---------------------------------------------------------------------------------
//						Public Function 			
//---------------------------------------------------------------------------------

#ifdef ENET_INC_UMAC
/**********************************************************************************
Pac_RxfGetFifoParameters


Description:
------------
This function retruns the parameters of the no BA agreement FIFO

Input:
-----
fifoParameters - a pointer to the structure to fill
	
Output:
-------
the fifoParameters strcture is filled with the needed parameters

Returns:
--------
   
**********************************************************************************/
void Pac_RxfGetFifoParameters(RxfFifoParameters_t *fifoParameters)
{
	RegPacRxfRxfFifoDebug_u  rxfFifoStatusRegister;
	RegPacRxfRxfFifoNumEntriesCount_u numberOfValidEntriesRegister;

	rxfFifoStatusRegister.val = 0;
	numberOfValidEntriesRegister.val = 0;

	RegAccess_Read(REG_PAC_RXF_RXF_FIFO_DEBUG, &rxfFifoStatusRegister.val);

	/* Check that errors didnt occur */
	DEBUG_ASSERT(!(rxfFifoStatusRegister.bitFields.rxfFifoFull || rxfFifoStatusRegister.bitFields.rxfFifoDecrementLessThanZero)); 

	RegAccess_Read(REG_PAC_RXF_RXF_FIFO_NUM_ENTRIES_COUNT, &numberOfValidEntriesRegister.val);

	fifoParameters->numberOfEntries = numberOfValidEntriesRegister.bitFields.rxfFifoNumEntriesCount;
	fifoParameters->currentIndexInFifo = RxfFifoObj.readIndex;
}

/**********************************************************************************
Pac_RxfSetFifoParameters


Description:
------------
This function updates the parameters of the no BA agreement FIFO

Input:
-----
fifoParameters - a pointer to the structure with the new parameters
	
Output:
-------


Returns:
--------
   
**********************************************************************************/

void Pac_RxfSetFifoParameters(RxfFifoParameters_t *fifoParameters)
{
	RegPacRxfRxfFifoRdEntriesNum_u numberOfEntriesReadByFwRegister; 

	numberOfEntriesReadByFwRegister.val = 0;

	numberOfEntriesReadByFwRegister.bitFields.rxfFifoRdEntriesNum = fifoParameters->numberOfEntries;
	RxfFifoObj.readIndex = fifoParameters->currentIndexInFifo;

	RegAccess_Write(REG_PAC_RXF_RXF_FIFO_RD_ENTRIES_NUM, numberOfEntriesReadByFwRegister.val);
}


/**********************************************************************************
Pac_RxfInitializeNoBaAgreementFifo


Description:
------------
This function initialize the no BA agreement FIFO

Input:
-----
	
Output:
-------


Returns:
--------
   
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void Pac_RxfInitializeNoBaAgreementFifo(void)
{
	/* The fifo is word alligned */
    memset32(RxfAmpduNoBaAggrementFifo, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxfAmpduNoBaAggrementFifo)));
	RxfFifoObj.readIndex = 0;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

#else /* #ifdef ENET_INC_UMAC */
/**********************************************************************************
Pac_RxfSetBssModeRejectFilterStatus

Description:
------------
	set/reset per vap a reject filter that is based on received packet Bss mode(fromToDs bits)

Input:
-----
	uint8 vapId
	RxfBssMode_e bssMode - [from DS, To DS] = {00,01,10,11}
	RxfFilterStatus_e status - enable/disable

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxfSetBssModeRejectFilterStatus(uint8 vapId, RxfBssMode_e bssMode, RxfFilterStatus_e status)
{
	uint32 mask = RXF_FILTER_STATUS_ENABLE << vapId;
	uint32 data = status << vapId;
	
	switch (bssMode)
	{
		case RXF_BSS_MODE_IBSS:
			RegAccess_WriteMasked(REG_PAC_RXF_BSS_FROM_TO_DS_00_VAP_FILTER_EN, mask, data);	
			break;	
			
		case RXF_BSS_MODE_STA:
			RegAccess_WriteMasked(REG_PAC_RXF_BSS_FROM_TO_DS_01_VAP_FILTER_EN, mask, data);
			break;	
			
		case RXF_BSS_MODE_AP:
			RegAccess_WriteMasked(REG_PAC_RXF_BSS_FROM_TO_DS_10_VAP_FILTER_EN, mask, data);
			break;	
			
		case RXF_BSS_MODE_WDS:
			RegAccess_WriteMasked(REG_PAC_RXF_BSS_FROM_TO_DS_11_VAP_FILTER_EN, mask, data);
			break;	
	
		default:
			DEBUG_ASSERT(0);
			break;
	}		
}


/**********************************************************************************
Pac_RxfConfigureGroupFilter

Description:
------------
Configure the Group filter as specified in the configuration parameter.
	
Input:
-----	
RxfGroupFilterConfig_t contains the following members:
	        Uint8     groupId              - There are 4 groups (groupId = {0,1,2,3})
	        Uint16   vapEnableBitmap - Bimap of the vaps the filter group is enabled for them
	        LogicType_e   logic0:1      - The logic between ELF and ESF0.
	        LogicType_e   logic1:1      - The logic between ESF1 and the result of ELF and ESF0.
	        LogicType_e   logic2:1      - The logic between ESF2 and the result of ELF, ESF0 and ESF1.
		 LogicType_e   logic3:1      - The logic between ESF3 and the result of ELF, ESF0, ESF1 and ESF2.
	        LongFilterUse_e   useEnhancedLongFilterResult:1 - use of enhanced long filter result. 	
		 ShortFilterUse_e  useEnhancedShortFilterResult:3 - use of enhanced short filter result.(the num of results to use)									   of results to use) 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxfConfigureGroupFilter(RxfGroupFilterConfig_t* configuration)
{
	RegPacRxfMpduFieldFilterVapEnGrp0_u mpduFieldFilterVapEnGrp0Reg;
	RegPacRxfMpduFieldFilterLogicGrp0_u mpduFieldFilterLogicGrp0Reg;
	uint32 regAddress = 0;
	mpduFieldFilterVapEnGrp0Reg.val = 0;
	mpduFieldFilterLogicGrp0Reg.val = 0;

	// compute the address of the enable register according to the group ID
	regAddress = REG_PAC_RXF_MPDU_FIELD_FILTER_VAP_EN_GRP0 + (configuration->groupId * HW_REG_SIZE);
	mpduFieldFilterVapEnGrp0Reg.bitFields.mpduFieldFilterVapEnGrp0 = configuration->vapEnableBitmap;
	RegAccess_Write(regAddress, mpduFieldFilterVapEnGrp0Reg.val);

	// compute the address of the logic register according to the group ID
	regAddress = REG_PAC_RXF_MPDU_FIELD_FILTER_LOGIC_GRP0 + (configuration->groupId * HW_REG_SIZE);
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterLogic0Grp0 = configuration->logic0; // The logic between ELF and ESF0. , 0: AND. , 1: OR.
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterLogic1Grp0 = configuration->logic1; // The logic between ESF1 and the result of ELF and ESF0. , 0: AND. , 1: OR.
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterLogic2Grp0 = configuration->logic2; // The logic between ESF2 and the result of ELF, ESF0 and ESF1. , 0: AND. , 1: OR.
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterLogic3Grp0 = configuration->logic3; // The logic between ESF3 and the result of ELF, ESF0, ESF1 and ESF2. , 0: AND. , 1: OR.
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterElfUseGrp0 = configuration->useEnhancedLongFilterResult;  // 0x0: No use of enhanced long filter result. , 0x1: use of enhanced long filter result.
	mpduFieldFilterLogicGrp0Reg.bitFields.mpduFieldFilterEsfUseGrp0 = configuration->useEnhancedShortFilterResult; // 0x0: No use of enhanced short filter results. , 0x1: use only enhanced short filter 0 result. , 0x2: use only enhanced short filter 0-1 results. , 0x3: use only enhanced short filter 0-2 results. , 0x4: use only enhanced short filter 0-3 results.
	RegAccess_Write(regAddress, mpduFieldFilterLogicGrp0Reg.val);
}


/**********************************************************************************
Pac_RxfConfigureEnhancedLongFilter

Description:
------------
Configure the Enhanced long filter as specified in the configuration parameter

Input:
-----	
	RxfEnhancedLongFilterConfig_t contains the following members:
		RxfGroupFilterId_e	 groupId; 			// There are 4 groups (groupId = {0,1,2,3})
		RxfLongGroupFilterSource_e	groupSource;// 0: Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
		RxfLongGroupFilterEquation_e  groupEquation; // Determines the equation between the filter value and the latched filed.{==,!=,>,<}
		uint8   groupOffset;	//    Byte offset of the field (from the start of the PHY Status or MPDU MAC header) 	   
		uint32	groupValue;	    //    MPDU field filter value to compare
		uint32	groupMask;	    //    MPDU field filter mask of results to use

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxfConfigureEnhancedLongFilter(RxfEnhancedLongFilterConfig_t* configuration)
{
	RegPacRxfMpduFieldLongFilterGrp0Ctrl_u mpduFieldLongFilterGrp0CtrlReg;
	RegPacRxfMpduFieldLongFilterGrp0Value_u mpduFieldLongFilterGrp0ValueReg;
	RegPacRxfMpduFieldLongFilterGrp0Mask_u	mpduFieldLongFilterGrp0MaskReg; 
	uint32 regAddress = 0;
	mpduFieldLongFilterGrp0CtrlReg.val  = 0;
	mpduFieldLongFilterGrp0ValueReg.val = 0;
	mpduFieldLongFilterGrp0MaskReg.val  = 0;

	// compute ctrl reg by group Id
	regAddress = REG_PAC_RXF_MPDU_FIELD_LONG_FILTER_GRP0_CTRL + (configuration->groupId * RXF_LONG_FILTER_REGS_SIZE);

	mpduFieldLongFilterGrp0CtrlReg.bitFields.mpduFieldLongFilterGrp0Source = configuration->groupSource;
	mpduFieldLongFilterGrp0CtrlReg.bitFields.mpduFieldLongFilterGrp0Equation = configuration->groupEquation;
	mpduFieldLongFilterGrp0CtrlReg.bitFields.mpduFieldLongFilterGrp0Offset = configuration->groupOffset;
	// write the control register
	RegAccess_Write(regAddress, mpduFieldLongFilterGrp0CtrlReg.val);

	// write the value register
	mpduFieldLongFilterGrp0ValueReg.bitFields.mpduFieldLongFilterGrp0Value = configuration->groupValue;
	regAddress += HW_REG_SIZE; // value register is after control register
	RegAccess_Write(regAddress, mpduFieldLongFilterGrp0ValueReg.val);

	// write the mask register
	mpduFieldLongFilterGrp0MaskReg.bitFields.mpduFieldLongFilterGrp0Mask = configuration->groupMask;
	regAddress += HW_REG_SIZE; // mask register is after value register
	RegAccess_Write(regAddress, mpduFieldLongFilterGrp0MaskReg.val);
}


/**********************************************************************************
Pac_RxfConfigureEnhancedShortFilter

Description:
------------
Configure the Enhanced short filter as specified in the configuration parameter

Input:
-----	
	RxfEnhancedShortFilterConfig_t contains the following members:
		RxfGroupFilterId_e	 groupId;			// There are 4 groups (groupId = {0,1,2,3})
		RxfShortFilterId_e	 filterId;			// There are 4 short filter in a group (short filter Id = {0,1,2,3})
		RxfGroupFilterSource_e	groupSource;// 0: Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
		RxfGroupShortFilterEquation_e  groupEquation; // Determines the equation between the filter value and the latched filed.{==,!=,>,<,>S,<S}
		uint8	groupOffset;	//	  Byte offset of the field (from the start of the PHY Status or MPDU MAC header)	   
		uint8	groupValue; 	//	  MPDU field filter value to compare
		uint8	groupMask;		//	  MPDU field filter mask of results to use

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxfConfigureEnhancedShortFilter(RxfEnhancedShortFilterConfig_t* configuration)
{
	RegPacRxfMpduFieldShortFilter0Grp0Ctrl_u mpduFieldShortFilter0Grp0CtrlReg;

	uint32 shortFilterGroupStartAddr = REG_PAC_RXF_MPDU_FIELD_SHORT_FILTER0_GRP0_CTRL;

	shortFilterGroupStartAddr += (configuration->groupId * RXF_SHORT_FILTER_CONFIG_SIZE * RXF_GROUP_FILTER_NUM) + (configuration->filterId * RXF_SHORT_FILTER_CONFIG_SIZE);

	mpduFieldShortFilter0Grp0CtrlReg.bitFields.mpduFieldShortFilter0Grp0Source = configuration->groupSource;     // 0: Filter on the original PHY status. , 1: Filter on the MPDU MAC header.
	mpduFieldShortFilter0Grp0CtrlReg.bitFields.mpduFieldShortFilter0Grp0Equation = configuration->groupEquation; // Select the equation: , 000: Equal (==). , 001: Not equal (!=). , 010: Greater than (>). , 011: below than (<). , 110: Greater than (>) – signed comparison. , 111: below than (<) – signed comparison.
	mpduFieldShortFilter0Grp0CtrlReg.bitFields.mpduFieldShortFilter0Grp0Offset = configuration->groupOffset;	 // Byte offset of the field (from the start of the PHY Status or MPDU MAC header)
	mpduFieldShortFilter0Grp0CtrlReg.bitFields.mpduFieldShortFilter0Grp0Value= configuration->groupValue;        // MPDU field filter value to compare
	mpduFieldShortFilter0Grp0CtrlReg.bitFields.mpduFieldShortFilter0Grp0Mask = configuration->groupMask;		 // MPDU field filter mask

	RegAccess_Write(shortFilterGroupStartAddr, mpduFieldShortFilter0Grp0CtrlReg.val);
}


/**********************************************************************************
Pac_RxfReset

Description:
------------
	 resets RXF module - all fields and state machine.
Input:
-----
	void

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxfReset(void)
{	
	RegMacRabPasSwResetRegAssert_u rabPasResetRegAssert;
	RegMacRabPasSwResetRegRelease_u rabPasResetRegrelease;
	
	rabPasResetRegAssert.val = 0;
	rabPasResetRegrelease.val = 0;
	
	/* Reset RXF state machine and all fields - return to default value */
	rabPasResetRegAssert.bitFields.swResetAssertRxf = HW_RESET_BLOCK;
	rabPasResetRegAssert.bitFields.swResetAssertRxfReg = HW_RESET_BLOCK;
	RegAccess_Write(REG_MAC_RAB_PAS_SW_RESET_REG_ASSERT,rabPasResetRegAssert.val);
	
	rabPasResetRegrelease.bitFields.swResetReleaseRxf = HW_RESET_BLOCK;
	rabPasResetRegrelease.bitFields.swResetReleaseRxfReg = HW_RESET_BLOCK;
	RegAccess_Write(REG_MAC_RAB_PAS_SW_RESET_REG_RELEASE,rabPasResetRegrelease.val);
}

/*********************************************************************************
Method:		Rxf_SetMACAddressfilter
Description:   Set MAC address filter - my address filter
Parameter:    vapId (filter number), address -  pointer to the address to filter
Returns:      	void
Remarks:		None
*********************************************************************************/
void Pac_RxfSetMACAddressfilter (uint8 vapId, IEEE_ADDR *address, bool RssiModeEnable)
{	
	uint32 selectionShift;
	RegPacRxfLongFilter0Ref1_u		longFilter0Ref1Reg;
	RegPacRxfLongFilter0Ref2_u		longFilter0Ref2Reg;
	RegPacRxfLongFilter0Ref1_u		longFilter0Ref1RegNormal;
	RegPacRxfLongFilter0Ref2_u		longFilter0Ref2RegNormal;
	RegPacRxfLongFiltersSel0To31_u 	longFiltersSel0To31Reg;
	RegPacRxfLongFiltersSel_u 		longFiltersSel32To35Reg;
	PacManagerGlobalParams_t* 		pLocalPacManagerGlobalParameters = PacManager_GetGlobalParams();
	uint32 longFilterRef1Address;
	uint32 longFilterRef2Address;

	longFilter0Ref1Reg.val = 0;
	longFilter0Ref2Reg.val = 0;
	longFilter0Ref1RegNormal.val = 0;
	longFilter0Ref2RegNormal.val = 0;
	longFiltersSel0To31Reg.val = 0;
	longFiltersSel32To35Reg.val = 0;
	
	/* validity check */  
	DEBUG_ASSERT(RXF_MAX_MY_ADDRESS_FILTERS > vapId);

	// compute ref reg address by vap ID 
	longFilterRef1Address = REG_PAC_RXF_LONG_FILTER_0_REF1 + (vapId << REG_PAC_RXF_LONG_FILTER_TOTAL_REF_SHIFT);
	longFilterRef2Address = longFilterRef1Address + REG_PAC_RXF_LONG_FILTER_REF1_SIZE;

	if (RssiModeEnable == TRUE)
	{
		// save normal configuration of the long filter which will be reloaded in NORMAL_AFTER_RSSI mode:
		RegAccess_Read(longFilterRef1Address, &longFilter0Ref1RegNormal.val);
		RegAccess_Read(longFilterRef2Address, &longFilter0Ref2RegNormal.val);
		//save the value in normal mode
		pLocalPacManagerGlobalParameters->longFilter0Ref1RegNormal = longFilter0Ref1RegNormal.val;
		pLocalPacManagerGlobalParameters->longFilter0Ref2RegNormal = longFilter0Ref2RegNormal.val;
#ifdef BEEROCK_DEBUG
		ILOG0_D("[Pac_RxfSetMACAddressfilter], PacManagerGlobalParameters->longFilter0Ref1RegNormal = %x", pLocalPacManagerGlobalParameters->longFilter0Ref1RegNormal);
		ILOG0_D("[Pac_RxfSetMACAddressfilter], PacManagerGlobalParameters->longFilter0Ref2RegNormal = %x", pLocalPacManagerGlobalParameters->longFilter0Ref2RegNormal);
#endif		
		// enable only the filter suitable to the VAP ID and disable all others
		longFiltersSel0To31Reg.val = RXF_DISABLE_SELECTION_MASK;
		RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);

		longFiltersSel32To35Reg.bitFields.selLf16 = REG_PAC_RXF_LONG_FILTER_VALUE_DISABLE;
		longFiltersSel32To35Reg.bitFields.selLf17 = REG_PAC_RXF_LONG_FILTER_VALUE_DISABLE;

		RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL, longFiltersSel32To35Reg.val);
	}
	
	/* Setting the reference data */
	longFilter0Ref1Reg.bitFields.longFilter0Ref1 = 
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_FIRST_BYTE] << RXF_IEEE_ADDRESS_FIRST_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_SECOND_BYTE] << RXF_IEEE_ADDRESS_SECOND_BYTE_SHIFT))  |
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_THIRD_BYTE] << RXF_IEEE_ADDRESS_THIRD_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_FORTH_BYTE] << RXF_IEEE_ADDRESS_FORTH_BYTE_SHIFT)) ; 

	longFilter0Ref2Reg.bitFields.longFilter0Ref2 = 
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_FIFTH_BYTE] << RXF_IEEE_ADDRESS_FIFTH_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXF_IEEE_ADDRESS_SIXT_BYTE] << RXF_IEEE_ADDRESS_SIXT_BYTE_SHIFT));

	RegAccess_Write(longFilterRef1Address, longFilter0Ref1Reg.val);
	
	RegAccess_Write(longFilterRef2Address, longFilter0Ref2Reg.val);

	/* Set selection - compute shift amount (2 bits per vap) */
	selectionShift = vapId << RXF_LONG_FILTER_SELECTION_MULTIPLY_BY_TWO_SHIFT;

	RegAccess_Read(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, &longFiltersSel0To31Reg.val);
	longFiltersSel0To31Reg.val	&= ~ (RXF_LONG_FILTER_SELECTION_MASK << selectionShift);	
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);
#ifdef BEEROCK_DEBUG
	ILOG0_D("[Pac_RxfSetMACAddressfilter], longFilterRef1Address = %x", longFilterRef1Address);
	ILOG0_D("[Pac_RxfSetMACAddressfilter], longFilter0Ref1Reg.val = %x", longFilter0Ref1Reg.val);
	ILOG0_D("[Pac_RxfSetMACAddressfilter], longFilterRef2Address = %x", longFilterRef2Address);
	ILOG0_D("[Pac_RxfSetMACAddressfilter], longFilter0Ref2Reg.val = %x", longFilter0Ref2Reg.val);
#endif
}

/*********************************************************************************
Method:		Rxf_ClearAddressFilter
Description:   Clears MAC address filter - my address filter
Parameter:    vapId (filter number)
Returns:      	void
Remarks:		None
*********************************************************************************/
void Pac_RxfClearAddressFilter (uint8 vapId) 
{
	uint32 selectionShift;
	RegPacRxfLongFiltersSel0To31_u longFiltersSel0To31Reg;
	RegPacRxfLongFilter0Ref1_u	longFilter0Ref1Reg;
	RegPacRxfLongFilter0Ref2_u	longFilter0Ref2Reg;
	uint32 longFilterRefAddress;
	
	/* validity check */  
	DEBUG_ASSERT(RXF_MAX_MY_ADDRESS_FILTERS > vapId);
	
	/* Clearing the reference data */
	// compute ref reg address by vap ID
	longFilter0Ref1Reg.bitFields.longFilter0Ref1 = RXF_REFERENCE_DATA_DEFAULT_VALUE_LOW;
	longFilterRefAddress = REG_PAC_RXF_LONG_FILTER_0_REF1 + (vapId << REG_PAC_RXF_LONG_FILTER_TOTAL_REF_SHIFT);
	RegAccess_Write(longFilterRefAddress, longFilter0Ref1Reg.val);

	longFilter0Ref2Reg.bitFields.longFilter0Ref2 = RXF_REFERENCE_DATA_DEFAULT_VALUE_HIGH;
	longFilterRefAddress += REG_PAC_RXF_LONG_FILTER_REF1_SIZE;
	RegAccess_Write(longFilterRefAddress, longFilter0Ref2Reg.val);

	/* Clear selection */
	selectionShift = vapId * RXF_LONG_FILTER_SELECTION_SIZE;

	RegAccess_Read(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, &longFiltersSel0To31Reg.val);
	longFiltersSel0To31Reg.val	|= (RXF_LONG_FILTER_SELECTION_MASK << selectionShift);	
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);
}

/*********************************************************************************
Method:			PacManagerNormalActivation
Description:   	Configures long filter #0 to normal mode
Parameter:    	void
Returns:      	void
Remarks:		None
*********************************************************************************/
void Pac_RxfNormalVapId(uint8 vapId)
{
	RegPacRxfLongFilter0Control_u 	longFilter0ControlReg;
	RegPacRxfLongFiltersSel0To31_u 	longFiltersSel0To31Reg;
	RegPacRxfLongFiltersSel_u 		longFiltersSel32To35Reg;
	PacManagerGlobalParams_t* 		pLocalPacManagerGlobalParameters = PacManager_GetGlobalParams();
	uint32 selectionShift;
	uint32 longFilterRef1Address;
	uint32 longFilterRef2Address;

	//init
	longFiltersSel0To31Reg.val = 0;
	longFiltersSel32To35Reg.val = 0;

	// rewrite long-filter address to the value in NORMAL mode
	// compute ref reg address by vap ID 0
	longFilterRef1Address = REG_PAC_RXF_LONG_FILTER_0_REF1 + (vapId << REG_PAC_RXF_LONG_FILTER_TOTAL_REF_SHIFT);
	longFilterRef2Address = longFilterRef1Address + REG_PAC_RXF_LONG_FILTER_REF1_SIZE;

	RegAccess_Write(longFilterRef1Address, pLocalPacManagerGlobalParameters->longFilter0Ref1RegNormal);
	
	RegAccess_Write(longFilterRef2Address, pLocalPacManagerGlobalParameters->longFilter0Ref2RegNormal);

	/* Set control register 0  as in Init */	
	longFilter0ControlReg.bitFields.longFilter0Control = RXF_LONG_FILTER_MASK;
	longFilter0ControlReg.bitFields.offsetLongFilter0Control = RXF_LONG_FILTER_ADDRESS_1_OFFSET_VALUE;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_0_CONTROL, longFilter0ControlReg.val);

	//use bitmap in ADD_VAP and remove_VAP to configure long filter #0 as before
	for (vapId = 0; vapId < HW_NUM_OF_VAPS; vapId++)
	{
		if((pLocalPacManagerGlobalParameters->vapIdBitMap>>vapId) & 0x1)
		{
			/* Set selection - compute shift amount (2 bits per vap) */
			selectionShift = vapId << RXF_LONG_FILTER_SELECTION_MULTIPLY_BY_TWO_SHIFT;
			
			RegAccess_Read(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, &longFiltersSel0To31Reg.val);
			longFiltersSel0To31Reg.val	&= ~ (RXF_LONG_FILTER_SELECTION_MASK << selectionShift);	
			RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);			
		}
	}
	
	/* Set selection for broadcast (selection register 1) */
	RegAccess_Read(REG_PAC_RXF_LONG_FILTERS_SEL, &longFiltersSel32To35Reg.val);
	longFiltersSel32To35Reg.val &= RXF_BROADCAST_SELECTION_MASK;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL, longFiltersSel32To35Reg.val);	
#ifdef BEEROCK_DEBUG
	ILOG0_D("[Pac_RxfNormalVapId], pLocalPacManagerGlobalParameters->vapIdBitMap = %d", pLocalPacManagerGlobalParameters->vapIdBitMap);
	ILOG0_D("[Pac_RxfNormalVapId], longFilterRef1Address = %x", longFilterRef1Address);
	ILOG0_D("[Pac_RxfNormalVapId], pLocalPacManagerGlobalParameters->longFilter0Ref1RegNormal = %x", pLocalPacManagerGlobalParameters->longFilter0Ref1RegNormal);
	ILOG0_D("[Pac_RxfNormalVapId], longFilterRef2Address = %x", longFilterRef2Address);
	ILOG0_D("[Pac_RxfNormalVapId], pLocalPacManagerGlobalParameters->longFilter0Ref2RegNormal = %x", pLocalPacManagerGlobalParameters->longFilter0Ref2RegNormal);
#endif
}

/*********************************************************************************
Method:			Pac_RxfUseFixedVapId
Description:   	Configures long filter#0 to recognize all frames as designated to chosen vap id
Parameter:    	void
Returns:      	void
Remarks:		None
*********************************************************************************/
void Pac_RxfUseFixedVapId(uint8 vapId)
{
	RegPacRxfLongFilter0Control_u longFilter0ControlReg;
	RegPacRxfLongFiltersSel0To31_u longFiltersSel0To31Reg;
	RegPacRxfLongFiltersSel_u longFiltersSel32To35Reg;
	uint32 selectionShift;

	longFiltersSel32To35Reg.val = 0;
	longFilter0ControlReg.val = 0;
	
	//configure RxF to recognize all frames as designated to VAP 0. This will be reached by enabling only filter 0 with comparing bitmap of 0, which will give a match always.

	RegAccess_Read(REG_PAC_RXF_LONG_FILTER_0_CONTROL, &longFilter0ControlReg.val);

	longFilter0ControlReg.bitFields.longFilter0Control = 0x0; //means don't compare anything (bitmap of bit per byte, out of 6 possible bytes of Address length)

	// enable only the filter suitable to the VAP ID and disable all others
	longFiltersSel0To31Reg.val = RXF_DISABLE_SELECTION_MASK;

	selectionShift = vapId << RXF_LONG_FILTER_SELECTION_MULTIPLY_BY_TWO_SHIFT;

	longFiltersSel0To31Reg.val	&= ~(RXF_LONG_FILTER_SELECTION_MASK << selectionShift);	
	
	longFiltersSel32To35Reg.bitFields.selLf16 = REG_PAC_RXF_LONG_FILTER_VALUE_DISABLE;
	longFiltersSel32To35Reg.bitFields.selLf17 = REG_PAC_RXF_LONG_FILTER_VALUE_DISABLE;

	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_0_CONTROL, longFilter0ControlReg.val);
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);	
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL, longFiltersSel32To35Reg.val);

}

/*********************************************************************************
Method:		Pac_RxfGroupFilter
Description:   Configures one filter to recognize all frames as designated to chosen vap id
Parameter:    
				groupId - There are 4 filter groups. Indicates which group is to be configures
				qosControlFieldOffset - 24 or 30. The location of QoS Control Field is dependent if WDS is active or not (Address 4 is present ony iin WDS)
				longFilterValue - Represent the "To DS From DS" bits configuration to differentiate WDS from Regular case

Returns:      	void
Remarks:		None
*********************************************************************************/
static void Pac_RxfGroupFilter(RxfGroupFilterId_e groupId, uint8 qosControlFieldOffset, uint32 longFilterValue)
{	
	RxfGroupFilterConfig_t			configuration;
	RxfEnhancedLongFilterConfig_t	longFilter;
	RxfEnhancedShortFilterConfig_t	shortFilter0, shortFilter1, shortFilter2, shortFilter3;

	/* Configure Group control */
	configuration.groupId = groupId;										// 0-3		- There are 4 groups (groupId = {0,1,2,3})
	configuration.vapEnableBitmap = RXF_ENABLE_ALL_VAPS;					// ALL		- Bimap of the vaps the filter group is enabled for them
	configuration.logic0 = RXF_AND_LOGIC_TYPE;								// AND		- The logic between ELF and ESF0.
	configuration.logic1 = RXF_AND_LOGIC_TYPE;								// AND		- The logic between ESF1 and the result of ELF and ESF0.
	configuration.logic2 = RXF_AND_LOGIC_TYPE;								// AND		- The logic between ESF2 and the result of ELF, ESF0 and ESF1.
	configuration.logic3 = RXF_AND_LOGIC_TYPE;								// AND		- The logic between ESF3 and the result of ELF, ESF0, ESF1 and ESF2.
	configuration.useEnhancedLongFilterResult = RXF_USE_LONG_FILTER;		// TRUE		- use of enhanced long filter result.
	configuration.useEnhancedShortFilterResult = RXF_USE_SHORT_FILTER_0_3;	// ALL		- use of enhanced short filter result.(the num of results to use) 

	Pac_RxfConfigureGroupFilter(&configuration);

	/* Configure Long filter to check if this is Regular (non-WDS) */
	longFilter.groupId = groupId;											// 0-3			- There are 4 groups (groupId = {0,1,2,3})
	longFilter.groupSource = RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER;		// 1				- Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	longFilter.groupEquation = RXF_GROUP_LONG_FILTER_EQUATION_EQUAL;		// Equal			- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
	longFilter.groupOffset = FRAME_CONTROL_FIRST_BYTE_OFFSET;				// 0 			- Byte offset of the field (from the start of the PHY Status or MPDU MAC header)
	longFilter.groupValue = longFilterValue;								// WDS/Regular	- MPDU field filter value to compare
	longFilter.groupMask = FRAME_CONTROL_MASK_TO_DS_FROM_DS;				// 11			- Mask B8 and B9 in Frame Control

	Pac_RxfConfigureEnhancedLongFilter(&longFilter);

	/* Configure Short filter 0: is Type == DATA */
	shortFilter0.groupId = groupId;											// 0-3		- There are 4 groups (groupId = {0,1,2,3})
	shortFilter0.filterId = RXF_GROUP_SHORT_FILTER_ID_0;					// 0			- There are 4 short filter in a group (short filter Id = {0,1,2,3})
	shortFilter0.groupSource = RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER; 	// 1			- Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	shortFilter0.groupEquation = RXF_GROUP_SHORT_FILTER_EQUATION_EQUAL;		// Equal		- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
	shortFilter0.groupOffset = FRAME_CONTROL_FIRST_BYTE_OFFSET;				// 0			- Set the offset to the location of 'Type' (B2 and B3 at Frame Control first byte) */
	shortFilter0.groupValue = FRAME_CONTROL_TYPE_DATA;						// 10		- Compare the value of 'Type' to 'DATA' 
	shortFilter0.groupMask = FRAME_CONTROL_MASK_TYPE;						// 1100		- MASK Type field (BIT2 & BIT3) in Frame Control 

	Pac_RxfConfigureEnhancedShortFilter(&shortFilter0);

	/* Configure Short filter 1: is Subtype == QoS DATA */
	shortFilter1.groupId = groupId;											// 0-3		- There are 4 groups (groupId = {0,1,2,3})
	shortFilter1.filterId = RXF_GROUP_SHORT_FILTER_ID_1; 					// 0			- There are 4 short filter in a group (short filter Id = {0,1,2,3})
	shortFilter1.groupSource = RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER;		// 1			- Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	shortFilter1.groupEquation = RXF_GROUP_SHORT_FILTER_EQUATION_EQUAL;		// Equal		- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
	shortFilter1.groupOffset = FRAME_CONTROL_FIRST_BYTE_OFFSET;				// 0 		- Set the offset to the location of 'Subtype' (B4 to B7 at Frame Control first byte) */
	shortFilter1.groupValue = FRAME_CONTROL_SUBTYPE_QOSDATA;				// 10		- Compare the value of 'Subtype' to 'QoS DATA' using only 2 MSBs (B6 & B7) enough to include all Qos Data
	shortFilter1.groupMask = FRAME_CONTROL_MASK_SUBTYPE;					// 1100		- Mask 2 MSBs of Subtype field (BIT6 & BIT7) in Frame Control 

	Pac_RxfConfigureEnhancedShortFilter(&shortFilter1);

	/* Configure Short filter 2: either 'More Fragments' or 'Fragment Number' */
	shortFilter2.groupId = groupId; 													// 0-3		- There are 4 groups (groupId = {0,1,2,3})
	shortFilter2.filterId = RXF_GROUP_SHORT_FILTER_ID_2;								// 2			- There are 4 short filter in a group (short filter Id = {0,1,2,3})
	shortFilter2.groupSource = RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER; 				// 1			- Filter on the MPDU MAC header. 

	if ((groupId == RXF_GROUP_FILTER_ID_0) || (groupId == RXF_GROUP_FILTER_ID_1))
	{
		/* Configure Short filter 2: is More Fragments == TRUE */
		shortFilter2.groupEquation = RXF_GROUP_SHORT_FILTER_EQUATION_EQUAL; 			// Equal		- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
		shortFilter2.groupOffset = FRAME_CONTROL_SECOND_BYTE_OFFSET; 					// 1			- Set the offset to the location of 'More Fragments' (B10 at Frame Control second byte) 
		shortFilter2.groupValue = FRAME_CONTROL_MORE_FRAGMENTS_SET;						// 0100		- B10 in Frame Control set to More Fragments
		shortFilter2.groupMask = FRAME_CONTROL_MASK_MORE_FRAGMENTS;						// 0100		- Mask More Fragments field (B10) in Frame Control 
	}
	else
	{
		/* Configure Short filter 2: is Fragment Number non-zero (handle last fragment) */
		shortFilter2.groupEquation = RXF_GROUP_SHORT_FILTER_EQUATION_NOT_EQUAL; 			// NOT Equal	- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
		shortFilter2.groupOffset = FRAME_CONTROL_22_BYTE_OFFSET_SEQ_CONTROL; 				// 22		- Set the byte offset to the location of 'Sequence Control' field in the MPDU MAC header  
		shortFilter2.groupValue = SEQUENCE_CONTROL_FRAGMENT_NUMBER_ZERO;					// 0			- Fragment Number should be non-zero
		shortFilter2.groupMask = SEQUENCE_CONTROL_FRAGMENT_NUMBER_MASK;						// 0100		- Mask More Fragments field (B10) in Frame Control		
	}

	Pac_RxfConfigureEnhancedShortFilter(&shortFilter2);

	/* Configure Short filter 3: is A-MSDU present == TRUE */
	shortFilter3.groupId = groupId;											// 0-3		- There are 4 groups (groupId = {0,1,2,3})
	shortFilter3.filterId = RXF_GROUP_SHORT_FILTER_ID_3;					// 0			- There are 4 short filter in a group (short filter Id = {0,1,2,3})
	shortFilter3.groupSource = RXF_LONG_GROUP_FILTER_SOURCE_MAC_HEADER; 	// 1			- Filter on the original PHY status. 1: Filter on the MPDU MAC header. 
	shortFilter3.groupEquation = RXF_GROUP_SHORT_FILTER_EQUATION_EQUAL; 	// Equal		- Determines the equation between the filter value and the latched filed.{==,!=,>,<}
	shortFilter3.groupOffset = qosControlFieldOffset; 						// 24/30		- Set the byte offset to the location of 'A-MSDU present' (QoS Control field). 30 - WDS. 24 - Regular 
	shortFilter3.groupValue = FRAME_CONTROL_AMSDU_PRESENT_SET;				// 1000		- Compare the value of 'A-MSDU present' to '1' 
	shortFilter3.groupMask = FRAME_CONTROL_MASK_AMSDU_PRESENT;				// 1000		- Mask More Fragments field (B10) in Frame Control 

	Pac_RxfConfigureEnhancedShortFilter(&shortFilter3);
	
	
}

/*********************************************************************************
Method:			Pac_RxfFilterFragmentedAMSDU
Description:	Filter out packets which are Fragmented & part of AMSDU. 

				Group0 is used for Regular	& A-MSDU set & More Fragments is set
				Group1 is used for WDS 		& A-MSDU set & More Fragments is set			
				Group2 is used for Regular	& A-MSDU set & Fragment Number is non-zero
				Group3 is used for WDS		& A-MSDU set & Fragment Number is non-zero
				
Parameter:    	void
Returns:      	void
Remarks:		None
*********************************************************************************/
static void Pac_RxfFilterFragmentedAMSDU()
{
	/* Configure Group 0 filter: in case of non-WDS, filter out packets which are A-MSDU and More Fragments is set */
	Pac_RxfGroupFilter(RXF_GROUP_FILTER_ID_0, FRAME_CONTROL_24_BYTE_OFFSET_QOS_CONTROL, FRAME_CONTROL_TO_DS_FROM_DS_REGULAR);

	/* Configure Group 1 filter: in case of WDS, filter out packets which are A-MSDU and More Fragments is set */
	Pac_RxfGroupFilter(RXF_GROUP_FILTER_ID_1, FRAME_CONTROL_30_BYTE_OFFSET_QOS_CONTROL, FRAME_CONTROL_TO_DS_FROM_DS_WDS);

	/*Configure Group 2 filter: in case of non-WDS, filter out packets which are A-MSDU and Fragment Number is non-zero */
	Pac_RxfGroupFilter(RXF_GROUP_FILTER_ID_2, FRAME_CONTROL_24_BYTE_OFFSET_QOS_CONTROL, FRAME_CONTROL_TO_DS_FROM_DS_REGULAR);
	
	/*Configure Group 3 filter: in case of WDS, filter out packets which are A-MSDU and Fragment Number is non-zero */
	Pac_RxfGroupFilter(RXF_GROUP_FILTER_ID_3, FRAME_CONTROL_30_BYTE_OFFSET_QOS_CONTROL, FRAME_CONTROL_TO_DS_FROM_DS_WDS);
}

/*********************************************************************************
Method:		Rxf_Initalize
Description:   Initializes RXF module - broadcast filter  ,byte filters and control register 0.
Parameter:    	void
Returns:      	void
Remarks:		None
*********************************************************************************/
#pragma ghs section text=".initialization" 
void Pac_RxfInit()
{	
	RegPacRxfLongFilter0Control_u  longFilter0ControlReg;
	RegPacRxfLongFilter2Control_u  longFilter2ControlReg;
	RegPacRxfLongFilter16Ref1_u	   longFilter16Ref1Reg;
	RegPacRxfLongFilter16Ref2_u	   longFilter16Ref2Reg;
	RegPacRxfLongFiltersSel_u	   longFiltersSelReg;
	RegPacRxfLongFiltersSel0To31_u longFiltersSel0To31Reg;
	RegPacRxfRxfFifoBaseAddr_u 	   fifoBaseAddrReg;
	RegPacRxfRxfFifoDepthMinusOne_u fifoDepthMinusOne;
	RegPacRxfAMpduNoBaAgreementFilterEn_u ampduNoBaAgreementFilterEnReg;
	RegPacRxfMaxMpduLengthFilterEn_u  maxMpduLengthFilterEnReg; 
	RegPacRxfMaxMpduLength11AgB_u  maxMpduLength11AgBReg; 
	RegPacRxfMaxMpduLength11NAc_u  maxMpduLength11NAcReg; 

	fifoBaseAddrReg.val = 0; 
	fifoDepthMinusOne.val = 0;
	maxMpduLengthFilterEnReg.val = 0;


	/* Set control register 0 */	
	longFilter0ControlReg.bitFields.longFilter0Control = RXF_LONG_FILTER_MASK;
	longFilter0ControlReg.bitFields.offsetLongFilter0Control = RXF_LONG_FILTER_ADDRESS_1_OFFSET_VALUE;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_0_CONTROL, longFilter0ControlReg.val);
		
	/* Set broadcast filter */
	/* Set control register 2 - will be used for broadcast */
	longFilter2ControlReg.bitFields.longFilter2Control = RXF_LONG_FILTER_MASK;
	longFilter2ControlReg.bitFields.offsetLongFilter2Control = RXF_LONG_FILTER_ADDRESS_1_OFFSET_VALUE;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_2_CONTROL, longFilter2ControlReg.val);
	
	/* Set reference data registers for broadcast filter */
	longFilter16Ref1Reg.bitFields.longFilter16Ref1 = RXF_BROADCAST_ADDRESS_LOW;
	longFilter16Ref2Reg.bitFields.longFilter16Ref2 = RXF_BROADCAST_ADDRESS_HIGH;

	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_16_REF1, longFilter16Ref1Reg.val);
	RegAccess_Write(REG_PAC_RXF_LONG_FILTER_16_REF2, longFilter16Ref2Reg.val);
	
	/* Set selection to disable (selection register 1) */
	longFiltersSel0To31Reg.val = RXF_DISABLE_SELECTION_MASK;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL_0TO31, longFiltersSel0To31Reg.val);	

	/* Set selection for broadcast (selection register 1) */
	RegAccess_Read(REG_PAC_RXF_LONG_FILTERS_SEL, &longFiltersSelReg.val);
	longFiltersSelReg.val &= RXF_BROADCAST_SELECTION_MASK;
	RegAccess_Write(REG_PAC_RXF_LONG_FILTERS_SEL, longFiltersSelReg.val);	

	// reset RAM Rxf_ConfigureMpduTypeFilter
	memset32((void*)RXF_MPDU_TYPE_FILTER_START, 0, MPDU_TYPE_SUBTYPE_REJECT_FILTER_ARRAY_SIZE);	

	// configure the fifo base address in SHRAM
	fifoBaseAddrReg.bitFields.rxfFifoBaseAddr = (uint32)RxfAmpduNoBaAggrementFifo;
	RegAccess_Write(REG_PAC_RXF_RXF_FIFO_BASE_ADDR, fifoBaseAddrReg.val);	

	// configure the Fifo depth (Max number of entries)
	fifoDepthMinusOne.bitFields.rxfFifoDepthMinusOne = RXF_FIFO_AMPDU_NO_BA_AGGREMENT_DEPTH - 1;
	RegAccess_Write(REG_PAC_RXF_RXF_FIFO_DEPTH_MINUS_ONE, fifoDepthMinusOne.val);	

	// enable the fifo
	ampduNoBaAgreementFilterEnReg.bitFields.aMpduNoBaAgreementFilterEn = RXF_FILTER_STATUS_ENABLE;
	RegAccess_Write(REG_PAC_RXF_A_MPDU_NO_BA_AGREEMENT_FILTER_EN, ampduNoBaAgreementFilterEnReg.val);	

	// Set the value of MaxMpduLengthFilter according to Phy mode. 
	maxMpduLength11AgBReg.bitFields.maxMpduLength11Ag = RXF_MAX_MPDU_SIZE_PHY_MODE_11_A_G;
	maxMpduLength11AgBReg.bitFields.maxMpduLength11B = RXF_MAX_MPDU_SIZE_PHY_MODE_11_B;
	RegAccess_Write(REG_PAC_RXF_MAX_MPDU_LENGTH_11AG_B, maxMpduLength11AgBReg.val);		

	maxMpduLength11NAcReg.bitFields.maxMpduLength11N = RXF_MAX_MPDU_SIZE_PHY_MODE_11_N;
	maxMpduLength11NAcReg.bitFields.maxMpduLength11Ac = RXF_MAX_MPDU_SIZE_PHY_MODE_11_AC;
	RegAccess_Write(REG_PAC_RXF_MAX_MPDU_LENGTH_11N_AC, maxMpduLength11NAcReg.val);	

	// enable maxMpduLength filter for all phy modes
	maxMpduLengthFilterEnReg.bitFields.maxMpduLength11AgFilterEn = RXF_FILTER_STATUS_ENABLE;
	maxMpduLengthFilterEnReg.bitFields.maxMpduLength11BFilterEn = RXF_FILTER_STATUS_ENABLE;
	maxMpduLengthFilterEnReg.bitFields.maxMpduLength11NFilterEn = RXF_FILTER_STATUS_ENABLE;
	maxMpduLengthFilterEnReg.bitFields.maxMpduLength11AcFilterEn = RXF_FILTER_STATUS_ENABLE;
	RegAccess_Write(REG_PAC_RXF_MAX_MPDU_LENGTH_FILTER_EN, maxMpduLengthFilterEnReg.val);	

	//Configure the memory map of the MPDU type reject filter. 
	//Frames with the type and subtype specified as parameters will be filtered if the filter status is enabled.	
	memcpy32((void*)RXF_MPDU_TYPE_FILTER_START, RxfMpduTypeSubTypeRejectFilterVapsArray, MPDU_TYPE_SUBTYPE_REJECT_FILTER_ARRAY_SIZE);

	// Filter Fragmented A-MSDUs (see "Filter Fragmented AMSDU.docx")
	Pac_RxfFilterFragmentedAMSDU();
	
}
#pragma ghs section text=default
#endif /* #ifdef ENET_INC_UMAC */

/* End of file rxf.c. */


