/***********************************************************************************
 File:		PacRxc.c
 Module:		Pac Driver
 Purpose:		
 Description:	FW driver for Rxc module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "Pac_Api.h"
#include "PacTimRegs.h"
#include "PhyDriver_API.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 17

#define TIM_TIMER_ENABLE                (1)
#define TIM_TIMER_DISABLE               (0)
#define TIM_TIMER_OCCURED               (1)
#define TIM_ENABLE_HW_Q_SWITCH_EVENT    (1)
#define TIM_ENABLE_TSF_ADJUST           (1)
#define TIM_TSF_TIMER_ENABLE            (1)
#define TIM_TSF_TIMER_DISABLE           (0)
#define TIM_NAV_ENABLE                  (1)
#define TIM_NAV_DISABLE                 (0)
#define TIM_QBSS_INTEGRATOR_ENABLE_NAV_TIME (1)
#define TIM_QBSS_INTEGRATOR_ENABLE_CCA_TIME (1)
#define TIM_QBSS_INTEGRATOR_RESET_VALUE (0)
#define TIM_QBSS2_INTEGRATOR_ENABLE_NAV_TIME (1)
#define TIM_QBSS2_INTEGRATOR_ENABLE_CCA_TIME (1)
#define TIM_QBSS2_INTEGRATOR_ENABLE_OTX_TIME (1)
#define TIM_QBSS2_INTEGRATOR_RESET_VALUE (0)
#define TIM_TIMERS_RESET_VALUE          (0)



#define TIM_TICK_TIMER_PERIOD_IN_USEC   (1000000/OSAL_TIMERS_K_TICKS_PER_SECOND)
#define TIM_EIGTH_USEC_TICK_PER_USEC    (8)
#define TIM_SLOT_TICK_PER_USEC          (8) /* Slot in counted in 1/8 usec*/
#define TIM_IFS_TICK_PER_USEC           (8) /* Ifs in counted in 1/8 usec */


#define TIM_DIVIDERS_MSB_SHIFT      (8)
#define TIM_SLOT_DIVIDER_MSB_SHIFT  (5)
#define TIM_DIVIDERS_MSB_MASK       (0x1)
#define TIM_DEFAULT_SLOT_TIME		(9)


/* REDUCED_FREQ_PRESCALERS  0x788 Fields Description*/
#define TIM_TSF_FAST_DIVIDE_REDUCED_MASK    (0xFF)
#define TIM_TSF_FAST_DIVIDE_REDUCED_SHIFT   (0)
#define TIM_NAV_FAST_DIVIDE_REDUCED_MASK    (0xFF00)
#define TIM_NAV_FAST_DIVIDE_REDUCED_SHIFT   (0)
#define TIM_IFS_DIVIDE_REDUCED_MASK         (0xFF0000)
#define TIM_IFS_DIVIDE_REDUCED_SHIFT        (16)
#define TIM_SLOT_DIVIDE_REDUCED_MASK        (0x1F000000)
#define TIM_SLOT_DIVIDE_REDUCED_SHIFT       (24)
/* PRESCALERS_EXTENSION_BIT  - Description*/
#define REG_PRESCALERS_EXTENSION_BIT_TSF_FAST_DIVIDE_MASK   (0x00000001)
#define REG_PRESCALERS_EXTENSION_BIT_TSF_FAST_DIVIDE_SHIFT  (0)
#define REG_PRESCALERS_EXTENSION_BIT_NAV_FAST_DIVIDE_MASK   (0x00000002)
#define REG_PRESCALERS_EXTENSION_BIT_NAV_FAST_DIVIDE_SHIFT  (1)
#define REG_PRESCALERS_EXTENSION_BIT_IFS_DIVIDE_MASK        (0x00000004)
#define REG_PRESCALERS_EXTENSION_BIT_IFS_DIVIDE_SHIFT       (2)
#define REG_PRESCALERS_EXTENSION_BIT_SLOT_DIVIDE_MASK       (0x00000008)
#define REG_PRESCALERS_EXTENSION_BIT_SLOT_DIVIDE_SHIFT      (3)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/


#define TIM_RTS_CLR_RX_NAV_TIMER_ENABLE 					1
#define TIM_UPDATE_RX_NAV_TIMER_IF_UNSUPPORTED_MCS_DISABLE  0


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Pac_TimProgramHwEventTimer

Description:
------------
    Program the HW event timer to expire in a specific time according to the TSF value

Input:
-----
    hwEventTime - Expired timer TSF 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimProgramHwEventTimer(TSF hwEventTime)
{
    RegPacTimTsfHwEvent0_u tsfHwEvent0Reg;

    tsfHwEvent0Reg.val = 0;
    tsfHwEvent0Reg.bitFields.tsfHwEvent0Time = hwEventTime;
    tsfHwEvent0Reg.bitFields.tsfHwEvent0HwEnableWr = TIM_TIMER_ENABLE;

    RegAccess_Write(REG_PAC_TIM_TSF_HW_EVENT0,tsfHwEvent0Reg.val);
}

/**********************************************************************************

Pac_TimProgramSwEvent2Timer

Description:
------------
	Program the SW event timer 2 to expire in a specific time according to the TSF value
Input:
-----
    swEventTime - Expired timer TSF

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimProgramSwEvent2Timer(TSF swEventTime)
{
    RegPacTimTsfSwEvent2_u tsfSwEvent2Reg;

    tsfSwEvent2Reg.val = 0;
    tsfSwEvent2Reg.bitFields.tsfSwEvent2Time = swEventTime;
    tsfSwEvent2Reg.bitFields.tsfSwEvent2EnableWr = TIM_TIMER_ENABLE;

    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT2,tsfSwEvent2Reg.val);    
}

/**********************************************************************************

Pac_TimProgramSwEvent3Timer

Description:
------------
    Program the SW event timer 3 to expire in a specific time according to the TSF value
Input:
-----
    swEventTime - Expired timer TSF

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimProgramSwEvent3Timer(TSF swEventTime)
{
    RegPacTimTsfSwEvent3_u tsfSwEvent3Reg;

    tsfSwEvent3Reg.val = 0;
    tsfSwEvent3Reg.bitFields.tsfSwEvent3Time = swEventTime;
    tsfSwEvent3Reg.bitFields.tsfSwEvent3EnableWr = TIM_TIMER_ENABLE;

    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT3,tsfSwEvent3Reg.val);    
        
}

/**********************************************************************************

Pac_TimIsSwEvent2Occured

Description:
------------
    Check If Sw Event 

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
bool Pac_TimIsSwEvent2Occured(void)
{
    bool retVal = FALSE;

    RegPacTimTimerControlRd_u timerControlRd;

    timerControlRd.val = 0;
    RegAccess_Read(REG_PAC_TIM_TIMER_CONTROL_RD,&timerControlRd.val);

    if(timerControlRd.bitFields.tsfSwEvent2Rd == TIM_TIMER_OCCURED)
    {
        retVal = TRUE;
    }
    return retVal;
}
/**********************************************************************************

Pac_TimIsSwEvent3Occured

Description:
------------

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
bool Pac_TimIsSwEvent3Occured(void)
{
    bool retVal = FALSE;

    RegPacTimTimerControlRd_u timerControlRd;

    timerControlRd.val = 0;
    RegAccess_Read(REG_PAC_TIM_TIMER_CONTROL_RD,&timerControlRd.val);

    if(timerControlRd.bitFields.tsfSwEvent3Rd == TIM_TIMER_OCCURED)
    {
        retVal = TRUE;
    }

    return retVal;
}

/**********************************************************************************

Pac_TimResetSwTimer2

Description:
------------

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimResetSwTimer2(void) 
{
    RegPacTimTsfSwEvent2_u tsfSwEvent2Reg;

    tsfSwEvent2Reg.val = 0;
    tsfSwEvent2Reg.bitFields.tsfSwEvent2EnableWr = TIM_TIMER_DISABLE;

    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT2,tsfSwEvent2Reg.val);    
}

/**********************************************************************************

Pac_TimResetSwTimer3

Description:
------------

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimResetSwTimer3(void) 
{
	RegPacTimTsfSwEvent3_u tsfSwEvent3Reg;

    tsfSwEvent3Reg.val = 0;
    tsfSwEvent3Reg.bitFields.tsfSwEvent3EnableWr = TIM_TIMER_DISABLE;

    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT3,tsfSwEvent3Reg.val);    
}


/**********************************************************************************

Pac_TimProgramQueueSwitchTimer

Description:
------------
    These routines programs a queue switch to the respective queue into the hardware.
    The parameter is the time at which the switch is nominally to occur. Note that the actual switch
    time can be deferred by high priority responses.

    
    These routines are designed to be used via the macros PRG_Q_SWI_TMR(q, t), 
    where the queue to switch to is passed as a parameter to the macro.

Input:
-----
    direction
    ulSwitchTime - the time the switch is nominally to occur

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimProgramQueueSwitchTimer( uint8 u8direction, uint32 switchTime)
{
    RegPacTimTsfReqSwitch_u tsfReqSwitchReg;

    tsfReqSwitchReg.val = 0;

    tsfReqSwitchReg.bitFields.tsfReqSwitchTime = switchTime;
    tsfReqSwitchReg.bitFields.tsfReqSwitchHwEnableWr = TIM_ENABLE_HW_Q_SWITCH_EVENT;
    tsfReqSwitchReg.bitFields.tsfReqSwitchTo1 = u8direction;
    
    RegAccess_Write(REG_PAC_TIM_TSF_REQ_SWITCH,tsfReqSwitchReg.val);    
}

/**********************************************************************************

Pac_TimGetSlotTimer

Description:
------------
    Read the configured slot timer  (71-> 9 usec 159-<20 usec)
Input:
-----
    None
Output:
-------
    None

Returns:
--------
    slotTimerValue - Slot Timer Value 
**********************************************************************************/
uint32 Pac_TimGetSlotTimer(void)
{
    uint32 slotTimerValue = 0;
    
    RegPacTimSlotTimer_u slotTimerReg;

    slotTimerReg.val = 0;

    RegAccess_Read(REG_PAC_TIM_SLOT_TIMER,&slotTimerReg.val);
    slotTimerValue = slotTimerReg.bitFields.slotTimerPeriod;

    return slotTimerValue;
}

/**********************************************************************************

Pac_TimSetSlotTimer

Description:
------------

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimSetSlotTimer(uint32 slotTimerValue)
{ 
    RegPacTimSlotTimer_u slotTimerReg;

    slotTimerReg.val = 0;
    
	slotTimerReg.bitFields.slotTimerPeriod = (slotTimerValue - 1);

    RegAccess_Write(REG_PAC_TIM_SLOT_TIMER,slotTimerReg.val);    
}

/**********************************************************************************

Pac_TimSetTsf

Description:
------------
    Set TSF timer to new value

Input:
-----
    tsfLow -
    tsfHigh - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimSetTsf(uint32 tsfLow, uint32 tsfHigh)
{
    RegPacTimTimerControl_u timerControlReg;
    RegPacTimTsfTimerLowWport_u tsfTimerLowWriteReg;
    RegPacTimTsfTimerHighWport_u tsfTimerHighWriteReg;

    timerControlReg.val = 0;
    tsfTimerLowWriteReg.val = 0;
    tsfTimerHighWriteReg.val = 0;
    
    /* Disable the TSF timer so that it's writable */        
    RegAccess_Read(REG_PAC_TIM_TIMER_CONTROL,&timerControlReg.val);
    timerControlReg.bitFields.tsfEnable = TIM_TSF_TIMER_DISABLE;
    RegAccess_Write(REG_PAC_TIM_TIMER_CONTROL,timerControlReg.val); 
    /* Write to the TSF timer */
    tsfTimerLowWriteReg.bitFields.tsfTimerLowWport = tsfLow;
    tsfTimerHighWriteReg.bitFields.tsfTimerHighWport = tsfHigh;
    RegAccess_Write(REG_PAC_TIM_TSF_TIMER_LOW_WPORT,tsfTimerLowWriteReg.val); 
    RegAccess_Write(REG_PAC_TIM_TSF_TIMER_HIGH_WPORT,tsfTimerHighWriteReg.val); 
    /* Re-enable TSF timer */
    timerControlReg.bitFields.tsfEnable = TIM_TSF_TIMER_ENABLE;
    RegAccess_Write(REG_PAC_TIM_TIMER_CONTROL,timerControlReg.val); 
}

/**********************************************************************************

Pac_TimGetTsf


Description:
------------
    This routine performs a read of the TSF counter

Input:
-----
    None

Output:
-------
    None

Returns:
--------
    currentTsfValue -  The current value of the TSF time
**********************************************************************************/
TSF Pac_TimGetTsf(void)
{
    uint32 tsfTimerLowValue;
    uint32 tsfTimerHighValue;

	RegPacTimTsfTimerLowRport_u tsfTimerLowReadReg;
    RegPacTimTsfTimerHighRport_u tsfTimerHighReadReg;
	
    tsfTimerLowReadReg.val = 0;
    tsfTimerHighReadReg.val = 0;
    
    do
    {
        
        RegAccess_Read(REG_PAC_TIM_TSF_TIMER_HIGH_RPORT,&tsfTimerHighReadReg.val);
        tsfTimerHighValue = tsfTimerHighReadReg.bitFields.tsfTimerHighRport;
        
        RegAccess_Read(REG_PAC_TIM_TSF_TIMER_LOW_RPORT,&tsfTimerLowReadReg.val);
        tsfTimerLowValue = tsfTimerLowReadReg.bitFields.tsfTimerLowRport;
        
        RegAccess_Read(REG_PAC_TIM_TSF_TIMER_HIGH_RPORT,&tsfTimerHighReadReg.val);
    }
    while(tsfTimerHighValue != tsfTimerHighReadReg.bitFields.tsfTimerHighRport);
    return ((TSF)tsfTimerHighValue << 32) + tsfTimerLowValue;
}

/**********************************************************************************
Pac_TimGetTsfHigh

Description:
------------

Input:
-----
    None  
Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
uint32 Pac_TimGetTsfHigh(void)
{  
    uint32  currentTsfHigh = 0;
    RegPacTimTsfTimerHighRport_u tsfTimerHighReadReg;

    tsfTimerHighReadReg.val = 0;

    RegAccess_Read(REG_PAC_TIM_TSF_TIMER_HIGH_RPORT,&tsfTimerHighReadReg.val);
    currentTsfHigh = tsfTimerHighReadReg.bitFields.tsfTimerHighRport;
    return currentTsfHigh;
}

/**********************************************************************************

Pac_TimGetTsfLow

Description:
------------

Input:
-----
   None

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
uint32 Pac_TimGetTsfLow(void)
{  
    uint32  currentTsfLow = 0;
    RegPacTimTsfTimerLowRport_u tsfTimerLowReadReg;

    tsfTimerLowReadReg.val = 0;

    RegAccess_Read(REG_PAC_TIM_TSF_TIMER_LOW_RPORT,&tsfTimerLowReadReg.val);
    currentTsfLow = tsfTimerLowReadReg.bitFields.tsfTimerLowRport;
    return currentTsfLow;
}

/**********************************************************************************

Pac_TimGetDeltaTimeFromTsf


Description:
------------
    Returns time delta from input start time 
    
Input:
-----
    startTime - 

Output:
-------
     Delat time in microsec

Returns:
--------
    None
**********************************************************************************/
uint32 Pac_TimGetDeltaTimeFromTsf(uint32 startTime)
{
    uint32 currentTsfLow = 0;
    uint32 deltaTime = 0;

    currentTsfLow = GET_TSF_TIMER_LOW();
    deltaTime = currentTsfLow - startTime;

    return deltaTime;
}


/**********************************************************************************

Pac_TimResetQbssIntegrator

Description:
------------
    

Input:
-----
    

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimResetQbssIntegrator(void)
{
    RegPacTimQbssLoadIntegratorWr_u qbssLoadIntegratorWritereg;

    qbssLoadIntegratorWritereg.val = 0;
    qbssLoadIntegratorWritereg.bitFields.qbssLoadIntegratorWr = TIM_QBSS_INTEGRATOR_RESET_VALUE;
    
    RegAccess_Write(REG_PAC_TIM_QBSS_LOAD_INTEGRATOR_WR,qbssLoadIntegratorWritereg.val);
}

/**********************************************************************************

Pac_TimResetQbss2Integrator

Description:
------------


Input:
-----


Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_TimResetQbss2Integrator(void)
{
    RegPacTimQbss2LoadIntegratorWr_u qbss2LoadIntegratorWritereg;

    qbss2LoadIntegratorWritereg.val = 0;
    qbss2LoadIntegratorWritereg.bitFields.qbss2LoadIntegratorWr = TIM_QBSS2_INTEGRATOR_RESET_VALUE;

    RegAccess_Write(REG_PAC_TIM_QBSS2_LOAD_INTEGRATOR_WR,qbss2LoadIntegratorWritereg.val);
}

/**********************************************************************************

Pac_TimReadQbssIntegrator

Description:
------------
 
Input:
-----
    

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
uint32  Pac_TimReadQbssIntegrator(void)
{    
    RegPacTimQbssLoadIntegratorRd_u qbssLoadIntegratorReadReg;

    qbssLoadIntegratorReadReg.val = 0;

    RegAccess_Read(REG_PAC_TIM_QBSS_LOAD_INTEGRATOR_RD,&qbssLoadIntegratorReadReg.val);

    return qbssLoadIntegratorReadReg.bitFields.qbssLoadIntegratorRd;

}

/**********************************************************************************

Pac_TimReadQbss2Integrator

Description:
------------

Input:
-----


Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
uint32  Pac_TimReadQbss2Integrator(void)
{
    RegPacTimQbss2LoadIntegratorRd_u qbss2LoadIntegratorReadReg;

    qbss2LoadIntegratorReadReg.val = 0;

    RegAccess_Read(REG_PAC_TIM_QBSS2_LOAD_INTEGRATOR_RD,&qbss2LoadIntegratorReadReg.val);

    return qbss2LoadIntegratorReadReg.bitFields.qbss2LoadIntegratorRd;

}



#if defined (ENET_INC_LMAC)
/**********************************************************************************

Pac_TimProgramHwEventTimer

Description:
------------

Input:
-----
    hwEventTime - 

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
#pragma ghs section text=".initialization" 
void Pac_TimResetTimers(void)
{
    RegAccess_Write(REG_PAC_TIM_TSF_HW_EVENT0,TIM_TIMERS_RESET_VALUE);  
    RegAccess_Write(REG_PAC_TIM_TSF_HW_EVENT1,TIM_TIMERS_RESET_VALUE);  
    RegAccess_Write(REG_PAC_TIM_TSF_REQ_SWITCH,TIM_TIMERS_RESET_VALUE);  
    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT2,TIM_TIMERS_RESET_VALUE);  
    RegAccess_Write(REG_PAC_TIM_TSF_SW_EVENT3,TIM_TIMERS_RESET_VALUE);           
}
#pragma ghs section text=default

/**********************************************************************************

Pac_TimInit


Description:
------------

Input:
-----

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
#pragma ghs section text=".initialization" 
void Pac_TimInit(void)
{
    RegPacTimTickTimer_u tickTimerReg;
    RegPacTimTimerControl_u timerControlReg;
    RegPacTimTimerControl2_u timerControl2Reg;
    RegPacTimOneMicroPrescaler_u oneMicroPrescalerReg;
    RegPacTimEighthMicroPrescaler_u eighthMicroPrescalerReg;
	RegPacTimRtsClrRxNavTimerEnable_u rtsClrRxNavTimerEnableReg;
	RegPacTimLspModeEnable_u lspModeEnableReg;
	RegPacTimTxNavTimerIfsCompensation_u txNavTimerCompensationReg;
    uint32 systemClksInOneUsec;
	uint32 systemClksInOneEigthUsec;
    uint32 timTickTimerPeriod;
	uint32 txNavCompensation;

    /* Mips Frequency In Mhz*/
    timTickTimerPeriod = (TIM_TICK_TIMER_PERIOD_IN_USEC - 1);
    
    /* Program Pre scalers*/    
    tickTimerReg.val = 0;
    timerControlReg.val = 0;
    timerControl2Reg.val = 0;
    oneMicroPrescalerReg.val = 0;
    eighthMicroPrescalerReg.val = 0;
	txNavTimerCompensationReg.val = 0;
	rtsClrRxNavTimerEnableReg.val = 0;
	lspModeEnableReg.val = 0;

	systemClksInOneUsec = System_ConvertUsToSystemClk(ONE_USEC);
    oneMicroPrescalerReg.bitFields.oneMicroPrescaler = systemClksInOneUsec - 1;
    oneMicroPrescalerReg.bitFields.oneMicroPrescalerFreqReduced = (systemClksInOneUsec >> 1) - 1;    
    RegAccess_Write(REG_PAC_TIM_ONE_MICRO_PRESCALER,oneMicroPrescalerReg.val);  

	systemClksInOneEigthUsec = System_ConvertNsToSystemClk(ONE_EIGTH_US_IN_NS);
    eighthMicroPrescalerReg.bitFields.eighthMicroPrescaler = systemClksInOneEigthUsec - 1;  
    eighthMicroPrescalerReg.bitFields.eighthMicroPrescalerFreqReduced = (systemClksInOneEigthUsec >> 2) - 1;     
    RegAccess_Write(REG_PAC_TIM_EIGHTH_MICRO_PRESCALER,eighthMicroPrescalerReg.val);    


	txNavCompensation = Pac_GetHwDelays(PAC_DELAY_TIM_TX_NAV_IFS);
	txNavTimerCompensationReg.bitFields.txNavIfsCompensation = System_ConvertNsToSystemClk(txNavCompensation);
	RegAccess_Write(REG_PAC_TIM_TX_NAV_TIMER_IFS_COMPENSATION, txNavTimerCompensationReg.val);
 
	
    /* Configure TIM Tick Timer - Microseconds per kernel  first set period & disable - this makes sure value is written to counter then enable tick timer */
    tickTimerReg.bitFields.tickTimerPeriod = timTickTimerPeriod;
    RegAccess_Write(REG_PAC_TIM_TICK_TIMER,tickTimerReg.val);
    tickTimerReg.bitFields.tickTimerEnable = TIM_TIMER_ENABLE;
    RegAccess_Write(REG_PAC_TIM_TICK_TIMER,tickTimerReg.val);
    /* Enable QBSS1 operation */
    timerControlReg.bitFields.navEnable = TIM_NAV_ENABLE;
    timerControlReg.bitFields.tsfEnable = TIM_TSF_TIMER_ENABLE;
    timerControlReg.bitFields.integrateNavTimeToQbss = TIM_QBSS_INTEGRATOR_ENABLE_NAV_TIME;
    timerControlReg.bitFields.integrateCcaTimeToQbss = TIM_QBSS_INTEGRATOR_ENABLE_CCA_TIME;

    RegAccess_Write(REG_PAC_TIM_TIMER_CONTROL,timerControlReg.val);
    /* Enable QBSS2 operation */
    timerControl2Reg.bitFields.integrateTxTimeToQbss2 = TIM_NAV_ENABLE;
    timerControl2Reg.bitFields.integrateNavTimeToQbss2 = TIM_QBSS2_INTEGRATOR_ENABLE_NAV_TIME;
    timerControl2Reg.bitFields.integrateCcaTimeToQbss2 = TIM_QBSS2_INTEGRATOR_ENABLE_CCA_TIME;

    RegAccess_Write(REG_PAC_TIM_TIMER_CONTROL2,timerControl2Reg.val);
	// rts_clr_rx_nav_timer_enable - RW
	// Enable reset Rx NAV timer in case RTS duration timer expired
	// In case the NAV is updated due to Rx of RTS frame, timer is set. 
	// In case there is no a new Rx session till the expiration of this timer, the Rx NAV timer will be zeroed.
 	rtsClrRxNavTimerEnableReg.bitFields.rtsClrRxNavTimerEnable = TIM_RTS_CLR_RX_NAV_TIMER_ENABLE;
	RegAccess_Write(REG_PAC_TIM_RTS_CLR_RX_NAV_TIMER_ENABLE, rtsClrRxNavTimerEnableReg.val);

	// LSP mode enable - RW (TIM block, offset 0x78)
	// In case PHY recognized Rx frame which is not supported, a short session is opened between PHY and MAC (Delineator).
	// In this session, indication of un-supported MCS and air time based the L-length are passed to MAC. 
	// The Rx NAV is updated with this air time value in case it's greater than the current Rx NAV timer.
	lspModeEnableReg.bitFields.lspEnableModeMcs = TIM_UPDATE_RX_NAV_TIMER_IF_UNSUPPORTED_MCS_DISABLE;
	RegAccess_Write(REG_PAC_TIM_LSP_MODE_ENABLE, lspModeEnableReg.val);
    
    Pac_TimResetTimers();    
}
#pragma ghs section text=default

#endif
