/***********************************************************************************
 File:		PacDur.c
 Module:		Pac Driver
 Purpose:		
 Description:	FW driver for Dur module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "RegAccess_Api.h"
#include "PacDurRegs.h"
#include "DurRam.h"
#include "HwMemoryMap.h"
#include "Protocol_AirTimeAndDuration.h"
#include "protocol.h"
#include "loggerAPI.h"
#include "Pac_Api.h"
#include "ConfigurationManager_api.h"
#include "PacDur.h"
#include "AutoRateMemFw_Descriptors.h"
#include "PacRxcRegs.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 9


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

#define DUR_GET_LOWER_2_BITS_OF_MCS(mcs)	((mcs)&0x3)
#define DUR_GET_UPPER_BIT_OF_MCS(mcs)		(((mcs)>>2)&0x1)

/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/

    


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

AutoRateMemFw_t *PacDurRam;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void pacDurFillRamAutoReply(void);

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Pac_DurResetTotalAirTimePerBand


Description:
------------
	
Input:
-----

		
**********************************************************************************/
void Pac_DurResetTotalAirTimePerBand(uint8 bandId)
{
	RegPacRxcRxToMeAirTimeCtrWr_u regPacDurRxToMeAirTimeCtrWr;

	regPacDurRxToMeAirTimeCtrWr.val = 0x0;
	RegAccess_WritePerBand(REG_PAC_RXC_RX_TO_ME_AIR_TIME_CTR_WR, regPacDurRxToMeAirTimeCtrWr.val,bandId);
}

/**********************************************************************************

Pac_DurResetTotalAirTime



Description:
------------
	
Input:
-----

		
**********************************************************************************/
void Pac_DurResetTotalAirTime()
{
	RegPacRxcRxToMeAirTimeCtrWr_u regPacDurRxToMeAirTimeCtrWr;

	regPacDurRxToMeAirTimeCtrWr.val = 0x0;
	RegAccess_Write(REG_PAC_RXC_RX_TO_ME_AIR_TIME_CTR_WR, regPacDurRxToMeAirTimeCtrWr.val);
}

/**********************************************************************************

Pac_DurGetTotalAirTimePerBand


Description:
------------
	
Input:
-----
	vapIndex - The Vap index to 
		
**********************************************************************************/
uint32 Pac_DurGetTotalAirTimePerBand(uint8 bandId)
{
	RegPacRxcRxToMeAirTimeCtrRd_u regPacDurRxToMeAirTimeCtrRd;

	regPacDurRxToMeAirTimeCtrRd.val = 0x0;
	RegAccess_ReadPerBand(REG_PAC_RXC_RX_TO_ME_AIR_TIME_CTR_RD, &regPacDurRxToMeAirTimeCtrRd.val,bandId);

	return regPacDurRxToMeAirTimeCtrRd.bitFields.rxToMeAirTimeCtrRd;
}
/**********************************************************************************

Pac_DurGetTotalAirTime


Description:
------------
	
Input:
-----
	vapIndex - The Vap index to 
		
**********************************************************************************/
uint32 Pac_DurGetTotalAirTime()
{
	RegPacRxcRxToMeAirTimeCtrRd_u regPacDurRxToMeAirTimeCtrRd;

	regPacDurRxToMeAirTimeCtrRd.val = 0x0;
	RegAccess_Read(REG_PAC_RXC_RX_TO_ME_AIR_TIME_CTR_RD, &regPacDurRxToMeAirTimeCtrRd.val);

	return regPacDurRxToMeAirTimeCtrRd.bitFields.rxToMeAirTimeCtrRd;
}




/**********************************************************************************

pacDurFillRamAutoReplyAirTime

Description:
------------
	1) Fills the Auto Reply Air Time fields  in Pac Dur Ram 
		
**********************************************************************************/
void pacDurFillRamAutoReplyAirTime(void)
{
	AutoReplyFrameAndSifsAirTime_t autoReplySifsAndAirTime;
	uint32 	rate;
	uint32 *ctsAirTimeOffset; 

	for(rate = 0; rate < LM_PHY_11A_MAX_RATES; rate++)
	{
		autoReplySifsAndAirTime.ackOrCtsAirTime11B 	= Protocol_GetAckCts11bAirTime(rate)+ SIFS_TIME_11B;
		autoReplySifsAndAirTime.ackOrCtsAirTime11AG = Protocol_GetAckCts11agAirTime(rate) + SIFS_TIME_11A;
		autoReplySifsAndAirTime.baAirTime11AG 		= Protocol_GetBaAirTime(rate)+ SIFS_TIME_11A;				
		
		// set ctsAirTimeOffset pointer to correct word in PacDurMem to point to AckCtsAirTime of specified rate
		// for each rate the pointer must move 2 words - thats the reason rate value is  multiplied by 2.
		ctsAirTimeOffset = (&PacDurRam->common.word0) + (rate << 1);
		*ctsAirTimeOffset = autoReplySifsAndAirTime.word0;
	}
}

/**********************************************************************************

pacDurFillRamAutoReplyBa11axAirTime


Description:
------------
	1) Fills the C-BA and M-BA Auto Reply Air Time fields for 11ax in Pac Dur Ram 
		
**********************************************************************************/

void pacDurFillRamAutoReplyBa11axAirTime(void)
{
	AutoReplyBaAndSifs11axAirTime_t autoReplyBaAndSifs11axAirTime;
	uint32 	rate;	
	uint32 *airTimeOffset; 

	for(rate = 0; rate < LM_PHY_11A_MAX_RATES; rate++)
	{
		autoReplyBaAndSifs11axAirTime.cba256BitAirTimeSifs11AxSu = protocol11axCba256BitAirTimeByRate[rate] + SIFS_TIME_11A;
		autoReplyBaAndSifs11axAirTime.mbaConstAitTimeSifs11AxSu = protocol11axMba256BitAirTimeByRate[rate] + SIFS_TIME_11A;
		autoReplyBaAndSifs11axAirTime.mbaPreTidAitTime11AxSu = protocol11axMbaPerTidAirTimeByRate[rate];				
		
		// set airTimeOffset pointer to correct word in PacDurMem to point to cba256BitAirTimeSifs11AxSuMcs of specified rate
		// for each rate the pointer must move 2 words - thats the reason rate value is  multiplied by 2.
		airTimeOffset = (&PacDurRam->common.word16) + (rate << 1);
		*airTimeOffset = autoReplyBaAndSifs11axAirTime.word0;
	}
}


#ifdef ENET_INC_ARCH_WAVE600D2
void pacDurFillRamAutoReply11axExtAirTime(void)
{
	PacDurRam->common.word32 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm0Ldpc0Ru106 = protocol11axSuExtAckAirTimeRu106[CONVOLUTIONAL_CODING][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm0Ldpc0Ru106  = protocol11axSuExtBaAirTimeRu106[CONVOLUTIONAL_CODING][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word33 = 0; //reserved

	PacDurRam->common.word34 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm0Ldpc0Ru242  = protocol11axSuExtAckAirTimeRu242[CONVOLUTIONAL_CODING][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm0Ldpc0Ru242  = protocol11axSuExtBaAirTimeRu242[CONVOLUTIONAL_CODING][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word35 = 0; //reserved

	PacDurRam->common.word36 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm0Ldpc1Ru106  = protocol11axSuExtAckAirTimeRu106[LDPC][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm0Ldpc1Ru106  = protocol11axSuExtBaAirTimeRu106[LDPC][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word37 = 0; //reserved

	PacDurRam->common.word38 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm0Ldpc1Ru242  = protocol11axSuExtAckAirTimeRu242[LDPC][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm0Ldpc1Ru242  = protocol11axSuExtBaAirTimeRu242[LDPC][NO_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word39 = 0; //reserved

	PacDurRam->common.word40 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm1Ldpc0Ru106  = protocol11axSuExtAckAirTimeRu106[CONVOLUTIONAL_CODING][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm1Ldpc0Ru106  = protocol11axSuExtBaAirTimeRu106[CONVOLUTIONAL_CODING][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word41 = 0; //reserved

	PacDurRam->common.word42 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm1Ldpc0Ru242  = protocol11axSuExtAckAirTimeRu242[CONVOLUTIONAL_CODING][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm1Ldpc0Ru242  = protocol11axSuExtBaAirTimeRu242[CONVOLUTIONAL_CODING][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word43 = 0; //reserved

	PacDurRam->common.word44 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm1Ldpc1Ru106  = protocol11axSuExtAckAirTimeRu106[LDPC][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm1Ldpc1Ru106  = protocol11axSuExtBaAirTimeRu106[LDPC][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word45 = 0; //reserved

	PacDurRam->common.word46 = 0;
	PacDurRam->common.ackAirTimeSifs11AxSuExtDcm1Ldpc1Ru242  = protocol11axSuExtAckAirTimeRu242[LDPC][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.baAirTimeSifs11AxSuExtDcm1Ldpc1Ru242  = protocol11axSuExtBaAirTimeRu242[LDPC][WITH_DCM] + SIFS_TIME_11A;
	PacDurRam->common.word47 = 0; //reserved
}
#endif //ENET_INC_ARCH_WAVE600D2

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

Pac_DurConfigureAutoReplyRate

Description:
------------
	Fills the Auto Reply rate in Pac Dur Ram
Input:
-----
	vapIndex - The Vap index to update
		
**********************************************************************************/
void Pac_DurConfigureAutoReplyRate(uint16 vapIndex)
{
	AutoRateMemFwPerVap_t mcsOut;
	
	memset32(&mcsOut, 0, (sizeof(AutoRateMemFwPerVap_t) >> 2));

	// PHY mode in = 11n/ac/ax
	mcsOut.mcso11NAcAxMcsi0			= MCS_0; // MCS 0 In	==> MCS 0 Out
	mcsOut.mcso11NAcAxMcsi1 		= MCS_2; // MCS 1 In	==> MCS 2 Out
	mcsOut.mcso11NAcAxMcsi2 		= MCS_2; // MCS 2 In	==> MCS 2 Out
	mcsOut.mcso11NAcAxMcsi3 		= MCS_4; // MCS 3 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi4			= MCS_4; // MCS 4 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi5 		= MCS_4; // MCS 5 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi6 		= MCS_4; // MCS 6 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi7 		= MCS_4; // MCS 7 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi8			= MCS_4; // MCS 8 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi9			= MCS_4; // MCS 9 In	==> MCS 4 Out
	mcsOut.mcso11NAcAxMcsi10Bit10	= DUR_GET_LOWER_2_BITS_OF_MCS(MCS_4); 	// MCS10 In	==> MCS 4 Out (divided over 2 words in DUR RAM)
	mcsOut.mcso11NAcAxMcsi10Bit2 	= DUR_GET_UPPER_BIT_OF_MCS(MCS_4); 		// MCS10 In	==> MCS 4 Out (divided over 2 words in DUR RAM)
	mcsOut.mcso11NAcAxMcsi11 		= MCS_4; // MCS 11 In	==> MCS 4 Out
	
	// PHY mode in = 11a/g
	mcsOut.mcso11AGMcsi0 = MCS_0; // MCS 0 In  ==> MCS 0 Out
	mcsOut.mcso11AGMcsi1 = MCS_0; // MCS 1 In  ==> MCS 0 Out
	mcsOut.mcso11AGMcsi2 = MCS_2; // MCS 2 In  ==> MCS 2 Out
	mcsOut.mcso11AGMcsi3 = MCS_2; // MCS 3 In  ==> MCS 2 Out
	mcsOut.mcso11AGMcsi4 = MCS_4; // MCS 4 In  ==> MCS 4 Out
	mcsOut.mcso11AGMcsi5 = MCS_4; // MCS 5 In  ==> MCS 4 Out
	mcsOut.mcso11AGMcsi6 = MCS_4; // MCS 6 In  ==> MCS 4 Out
	mcsOut.mcso11AGMcsi7 = MCS_4; // MCS 7 In  ==> MCS 4 Out

	// PHY mode in = 11b
	mcsOut.mcso11BMcsi0 = MCS_0; // MCS 0 In  ==> MCS 0 Out
	mcsOut.mcso11BMcsi1 = MCS_1; // MCS 1 In  ==> MCS 1 Out
	mcsOut.mcso11BMcsi2 = MCS_2; // MCS 2 In  ==> MCS 2 Out
	mcsOut.mcso11BMcsi3 = MCS_3; // MCS 3 In  ==> MCS 3 Out 
	mcsOut.mcso11BMcsi4 = MCS_4; // MCS 4 In  ==> MCS 4 Out
	mcsOut.mcso11BMcsi5 = MCS_5; // MCS 5 In  ==> MCS 5 Out
	mcsOut.mcso11BMcsi6 = MCS_6; // MCS 6 In  ==> MCS 6 Out
	mcsOut.mcso11BMcsi7 = MCS_7; // MCS 7 In  ==> MCS 7 Out

	memcpy32(&PacDurRam->perVap[vapIndex], &mcsOut, (sizeof(AutoRateMemFwPerVap_t) >> 2));
	
	// Set auto reply rate for HE SU Extended Range.
	// TBD for HDK - fill values here:
//	AutoResponse_SetExtendedRangeRate( nss,  mcs,  dcm,  isDcmOverride);


}

/**********************************************************************************

Pac_DurConfigureGeneralVapAutoReplyRate


Description:
------------
	Fills the Auto Reply rate of General Vap in Pac Dur Ram
Input:
-----
	
		
**********************************************************************************/

void Pac_DurConfigureGeneralVapAutoReplyRate()
{
	AutoRateMemFwVapGeneral_t mcsOut;
	
	memset32(&mcsOut, 0, (sizeof(AutoRateMemFwVapGeneral_t) >> 2));
	
	// PHY mode in = 11n/ac/ax
	mcsOut.vapGeneralMcso11NAcAxMcsi0		= MCS_0; // MCS 0 In	==> MCS 0 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi1 		= MCS_2; // MCS 1 In	==> MCS 2 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi2 		= MCS_2; // MCS 2 In	==> MCS 2 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi3 		= MCS_4; // MCS 3 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi4		= MCS_4; // MCS 4 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi5 		= MCS_4; // MCS 5 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi6 		= MCS_4; // MCS 6 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi7 		= MCS_4; // MCS 7 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi8		= MCS_4; // MCS 8 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi9		= MCS_4; // MCS 9 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11NAcAxMcsi10Bit10	= DUR_GET_LOWER_2_BITS_OF_MCS(MCS_4); 	// MCS10 In	==> MCS 4 Out (divided over 2 words in DUR RAM)
	mcsOut.vapGeneralMcso11NAcAxMcsi10Bit2 	= DUR_GET_UPPER_BIT_OF_MCS(MCS_4); 		// MCS10 In	==> MCS 4 Out (divided over 2 words in DUR RAM)
	mcsOut.vapGeneralMcso11NAcAxMcsi11 		= MCS_4; // MCS 11 In	==> MCS 4 Out
	
	// PHY mode in = 11a/g
	mcsOut.vapGeneralMcso11AGMcsi0 = MCS_0; // MCS 0 In  ==> MCS 0 Out
	mcsOut.vapGeneralMcso11AGMcsi1 = MCS_0; // MCS 1 In  ==> MCS 0 Out
	mcsOut.vapGeneralMcso11AGMcsi2 = MCS_2; // MCS 2 In  ==> MCS 2 Out
	mcsOut.vapGeneralMcso11AGMcsi3 = MCS_2; // MCS 3 In  ==> MCS 2 Out
	mcsOut.vapGeneralMcso11AGMcsi4 = MCS_4; // MCS 4 In  ==> MCS 4 Out
	mcsOut.vapGeneralMcso11AGMcsi5 = MCS_4; // MCS 5 In  ==> MCS 4 Out
	mcsOut.vapGeneralMcso11AGMcsi6 = MCS_4; // MCS 6 In  ==> MCS 4 Out
	mcsOut.vapGeneralMcso11AGMcsi7 = MCS_4; // MCS 7 In  ==> MCS 4 Out
	

	// PHY mode in = 11b
	mcsOut.vapGeneralMcso11BMcsi0 = MCS_0; // MCS 0 In	==> MCS 0 Out
	mcsOut.vapGeneralMcso11BMcsi1 = MCS_1; // MCS 1 In	==> MCS 1 Out
	mcsOut.vapGeneralMcso11BMcsi2 = MCS_2; // MCS 2 In	==> MCS 2 Out
	mcsOut.vapGeneralMcso11BMcsi3 = MCS_3; // MCS 3 In	==> MCS 3 Out
	mcsOut.vapGeneralMcso11BMcsi4 = MCS_4; // MCS 4 In	==> MCS 4 Out
	mcsOut.vapGeneralMcso11BMcsi5 = MCS_5; // MCS 5 In	==> MCS 5 Out
	mcsOut.vapGeneralMcso11BMcsi6 = MCS_6; // MCS 6 In	==> MCS 6 Out
	mcsOut.vapGeneralMcso11BMcsi7 = MCS_7; // MCS 7 In	==> MCS 7 Out

	memcpy32(&PacDurRam->vapGeneral, &mcsOut, (sizeof(AutoRateMemFwVapGeneral_t) >> 2));
	
	// Set auto reply rate for HE SU Extended Range.
	// TBD for HDK - fill values here:
//	AutoResponse_SetExtendedRangeRate( nss,  mcs,  dcm,  isDcmOverride);


}

/**********************************************************************************

PacDur_ConfigureAutoReplyPower 


Description:
------------
Configures the auto reply power tables in the PAC DUR RAAM.

Input:
-----
	autoReplyPowerParameters - the struct that conatins all the power parameters
		
Output:
-------
	

Returns:
--------
	 
	
**********************************************************************************/

#ifdef ENET_INC_ARCH_WAVE600D2

void Pac_DurConfigureAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters,uint8 autoReplyRate)
{
	AutoReplyPower11Ag_t autoReplyPowerValues;
	AutoReplyPower11b_u autoReplyPower11BValuesU;
	uint32 *txPower11AgAddress = NULL; 
	uint8 *txPower11BAddress = NULL;


	autoReplyPowerValues.word0 = 0;

	ASSERT(autoReplyRate < LM_PHY_11A_MAX_RATES);

	autoReplyPowerValues.txPower11AGBw20 = autoReplyPowerParameters->autoReplyPower11AGBw20;
	autoReplyPowerValues.txPower11AGBw40 = autoReplyPowerParameters->autoReplyPower11AGBw40;
	autoReplyPowerValues.txPower11AGBw80 = autoReplyPowerParameters->autoReplyPower11AGBw80;
	// TBD: 160MHz value isn't set - currently use same as 80MHz
	autoReplyPowerValues.txPower11AGBw160 = autoReplyPowerParameters->autoReplyPower11AGBw80;

	//move the pointer to the offset of provided rate (jump 2 words for each rate)
	txPower11AgAddress = (&PacDurRam->common.word48) + (autoReplyRate << 1);
	*txPower11AgAddress = autoReplyPowerValues.word0;
	// update txPower11B field based on provided rate
	if(autoReplyRate < LM_PHY_11A_RATE_24)
	{
		autoReplyPower11BValuesU.autoReplyPower11BValues.word0 = PacDurRam->common.word64;
		// set the txPower11BAddress to correct offset based on rate
		txPower11BAddress = ((uint8*)&(autoReplyPower11BValuesU.Power11bValues[0])) + autoReplyRate;
		//change relevant byte per rate
		*txPower11BAddress = autoReplyPowerParameters->autoReplyPower11B;
		// write back to the memory
		PacDurRam->common.word64 = autoReplyPower11BValuesU.autoReplyPower11BValues.word0;
	}
	else
	{
		autoReplyPower11BValuesU.autoReplyPower11BValues.word1 = PacDurRam->common.word66;
		// set the txPower11BAddress to correct offset based on rate
		txPower11BAddress = ((uint8*)&(autoReplyPower11BValuesU.Power11bValues[0])) + autoReplyRate;
		//change relevant byte per rate
		*txPower11BAddress = autoReplyPowerParameters->autoReplyPower11B;
		// write back to the memory
		PacDurRam->common.word66 = autoReplyPower11BValuesU.autoReplyPower11BValues.word1;
	}
}




/**********************************************************************************
Pac_DurSetExtendedRangeAutoReplyPower  

Description:
------------
	Set the fixed power which will be used in auto reply for HE SU Extended Range.
	
**********************************************************************************/
void Pac_DurSetExtendedRangeAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters)
{
	AutoRateMemFwCommon_t autoReply11axExtPowerValues;

	
	autoReply11axExtPowerValues.txPower11AxSuExtDcm0Ru106 = autoReplyPowerParameters->autoReplyPower11axSuExtDcm0Ru106;
	autoReply11axExtPowerValues.txPower11AxSuExtDcm1Ru106 = autoReplyPowerParameters->autoReplyPower11axSuExtDcm1Ru106;
	autoReply11axExtPowerValues.txPower11AxSuExtDcm0Ru242 = autoReplyPowerParameters->autoReplyPower11axSuExtDcm0Ru242;
	autoReply11axExtPowerValues.txPower11AxSuExtDcm1Ru242 = autoReplyPowerParameters->autoReplyPower11axSuExtDcm1Ru242;

	PacDurRam->common.word68 = autoReply11axExtPowerValues.word68;	
}


#else

#if defined (ENET_INC_LMAC)
void Pac_DurConfigureAutoReplyPower(PacDurAutoReplyPowerParameters_t *autoReplyPowerParameters,uint8 autoReplyRate)
{
	AutoReplyPower11Ag_t autoReplyPowerValues;
	AutoReplyPower11b_t autoReplyPower11BValues;
	uint32 *txPower11AgAddress = NULL; 
	uint8 *txPower11BAddress = NULL;


	autoReplyPowerValues.word0 = 0;

	ASSERT(autoReplyRate < LM_PHY_11A_MAX_RATES);

	autoReplyPowerValues.txPower11AGBw20 = autoReplyPowerParameters->autoReplyPower11AGBw20;
	autoReplyPowerValues.txPower11AGBw40 = autoReplyPowerParameters->autoReplyPower11AGBw40;
	autoReplyPowerValues.txPower11AGBw80 = autoReplyPowerParameters->autoReplyPower11AGBw80;
	// TBD: 160MHz value isn't set - currently use same as 80MHz
	autoReplyPowerValues.txPower11AGBw160 = autoReplyPowerParameters->autoReplyPower11AGBw80;

	//move the pointer to the offset of provided rate (jump 2 words for each rate)
	txPower11AgAddress = (&PacDurRam->common.word32) + (autoReplyRate << 1);
	*txPower11AgAddress = autoReplyPowerValues.word0;
	// update txPower11B field based on provided rate
	if(autoReplyRate < LM_PHY_11A_RATE_24)
	{
		autoReplyPower11BValues.word0 = PacDurRam->common.word48;
		// set the txPower11BAddress to correct offset based on rate
		txPower11BAddress = ((uint8*)&autoReplyPower11BValues.word0) + autoReplyRate;
		//change relevant byte per rate
		*txPower11BAddress = autoReplyPowerParameters->autoReplyPower11B;
		// write back to the memory
		PacDurRam->common.word48 = autoReplyPower11BValues.word0;
	}
	else
	{
		autoReplyPower11BValues.word1 = PacDurRam->common.word50;
		// set the txPower11BAddress to correct offset based on rate
		txPower11BAddress = ((uint8*)&autoReplyPower11BValues.word0);
		txPower11BAddress += autoReplyRate;
		//change relevant byte per rate
		*txPower11BAddress = autoReplyPowerParameters->autoReplyPower11B;
		// write back to the memory
		PacDurRam->common.word50 = autoReplyPower11BValues.word1;
	}
}
#endif
#endif //ENET_INC_ARCH_WAVE600D2

/**********************************************************************************

Pac_DurConfigureVapMode

Description:
------------
	
Input:
-----
	vapIndex - The Vap index to 
		
**********************************************************************************/
void Pac_DurConfigureVapMode(uint16 vapIndex, uint16 vapMode)
{
	RegPacDurTxFrameAddr3SelectVap_u durTxFrameAddr3SelectVap; 
	
	RegAccess_Read(REG_PAC_DUR_TX_FRAME_ADDR3_SELECT_VAP, &durTxFrameAddr3SelectVap.val);
	durTxFrameAddr3SelectVap.bitFields.txFrameAddr3SelectVap &= ~(0x1 << vapIndex);
	durTxFrameAddr3SelectVap.bitFields.txFrameAddr3SelectVap |= (vapMode << vapIndex);
	RegAccess_Write(REG_PAC_DUR_TX_FRAME_ADDR3_SELECT_VAP, durTxFrameAddr3SelectVap.val);
}



/**********************************************************************************

Pac_DurInit

Description:
------------
	Initialize Dur HW

**********************************************************************************/
void Pac_DurInit(void)
{
#if defined(ENET_INC_LMAC0) 
	PacDurRam = (AutoRateMemFw_t *)B0_DUR_AUTO_RATE_DUR_MEM;
#elif defined(ENET_INC_LMAC1) 
	PacDurRam = (AutoRateMemFw_t *)B0_DUR_AUTO_RATE_DUR_MEM;
#endif

	pacDurFillRamAutoReplyAirTime();
	pacDurFillRamAutoReplyBa11axAirTime();
#ifdef ENET_INC_ARCH_WAVE600D2
	pacDurFillRamAutoReply11axExtAirTime();
#endif //ENET_INC_ARCH_WAVE600D2
	// Fill Auto reply rates for Global VAP
	Pac_DurConfigureGeneralVapAutoReplyRate();


}
