/***********************************************************************************
 File:		PacRxc.c
 Module:		Pac Driver
 Purpose:		
 Description:	FW driver for Rxc module In Pac
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "PacRxcRegs.h"
#include "Pac_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "RxcRam.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "RxfMem_Descriptors.h"
#endif
#include "RegAccess_Api.h"
#include "VapAddrRegRegs.h"
#include "MacHtExtensionsRegs.h"
#include "ShramRxfFifo.h"
#include "SharedDbTypes.h"
#include "stringLibApi.h"
#include "ConfigurationManager_api.h"
#include "ShramStatistics.h"
#ifdef ENET_INC_LMAC
#include "PacDurRegs.h" // move to DUR module the config of 
#include "lmi.h"
#include "PhyDriver_API.h"
#include "HwGlobalDefinitions.h"
#include "RxHandler_InitApi.h"
#include "ShramRxfFifo.h"
#endif /* #ifdef ENET_INC_LMAC */


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_PAC
#define LOG_LOCAL_FID 12
	
	
#define RXC_HW_DELAY							(10) //  RXC Delay in cycles for CCA and RX READY

#define RXC_MAX_MY_ADDRESS_FILTERS				(32)
#define RXC_VAP_ADDR_REFERENCE_REG_SHIFT		(3) // Multiply by 8 to get the correct VAP register
#define RXC_VAP_ADDR_REFERENCE_REG_SIZE    		(4)	// Size of the first reference register (first 4 bytes of the MAC address)

#define RXC_IEEE_ADDRESS_FIRST_BYTE  			(0)
#define RXC_IEEE_ADDRESS_SECOND_BYTE 			(1)
#define RXC_IEEE_ADDRESS_THIRD_BYTE  			(2)
#define RXC_IEEE_ADDRESS_FORTH_BYTE  			(3)
#define RXC_IEEE_ADDRESS_FIFTH_BYTE  			(4)
#define RXC_IEEE_ADDRESS_SIXT_BYTE   			(5)

#define RXC_IEEE_ADDRESS_FIRST_BYTE_SHIFT  		(0)
#define RXC_IEEE_ADDRESS_SECOND_BYTE_SHIFT 		(8)
#define RXC_IEEE_ADDRESS_THIRD_BYTE_SHIFT  		(16)
#define RXC_IEEE_ADDRESS_FORTH_BYTE_SHIFT 		(24)
#define RXC_IEEE_ADDRESS_FIFTH_BYTE_SHIFT  		(0)
#define RXC_IEEE_ADDRESS_SIXT_BYTE_SHIFT   		(8)

#define RXC_MINIMAL_RSSI_THRESHOLD_VALUE		(0xAE) // -128 + 46  --> 0x80 + 0x2e

#define RXC_MAX_MPDU_SIZE_PHY_MODE_11_A_G		(MAX_HT_MPDU_IN_A_MPDU_LENGTH_RX)	// Max MPDU size in case of PHY mode 11ag
#define RXC_MAX_MPDU_SIZE_PHY_MODE_11_B			(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX)	// Max MPDU size in case of PHY mode 11b need to be maximum available(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX) - JIRA WLANSW-1907)

#define RXC_MAX_MPDU_SIZE_PHY_MODE_11_N			(MAX_MPDU_LENGTH_OPTION2_HT)  		// Max MPDU size in case of PHY mode 11n
#define RXC_MAX_MPDU_SIZE_PHY_MODE_11_AC		(MAX_VHT_MPDU_IN_A_MPDU_LENGTH_RX) // Max MPDU size in case of PHY mode 11ac
#define RXC_MAX_MPDU_SIZE_PHY_MODE_11_AX		(MAX_HE_MPDU_IN_A_MPDU_LENGTH_RX) // Max MPDU size in case of PHY mode 11ax

#define RXC_ENABLE_BUFFER_OVERFLOW_IND			(1)
#define RXC_ENABLE_FIFO_OVERFLOW_IND			(1)

/* FORCE_SW_CCA */
#define RXC_FORCE_SW_CCA_SHIFT    				(1)
#define RXC_FORCED_SW_CCA_SHIFT    				(2)

#define RXC_SIFS_PLUS_PHY_RX_DELAY_11AG 				42
#define RXC_SIFS_PLUS_PHY_RX_DELAY_11B_SHORT_PREAMBLE	108
#define RXC_SIFS_PLUS_PHY_RX_DELAY_11B_LONG_PREAMBLE	206


typedef enum
{
	PAC_RXC_FIFO_RX_PM_CLEAR_NOT_EMPTY_IRQ_OFFSET 			= 0x0,
	PAC_RXC_READY_NEGATION_NOMETRICS_OFFSET		  			= 0x1,
	PAC_RXC_GLOBAL_NEGATION_MISSING_METRICS_OFFSET			= 0x2,
	PAC_RXC_MPDU_DROP_DLM_IN_EMPTY_NOT_ZERO_OFFSET			= 0x3,
	PAC_RXC_PHY_MAC_DROP_CTR_NOT_ZERO_OFFSET				= 0x4,
	PAC_RXC_BUFFER_OVERFLOW_CTR_NOT_ZERO_OFFSET				= 0x5,
#ifdef ENET_INC_ARCH_WAVE600B
	PAC_RXC_RXF_FIFO_FULL_DROP_IRQ_OFFSET					= 0x6,
	PAC_RXC_RXF_FIFO_DECREMENT_LESS_THAN_ZERO_OFFSET  		= 0x7,
	PAC_RXC_GLOBAL_NEGATION_MISSING_BF_METRICS_OFFSET		= 0x8,
	PAC_RXC_GLOBAL_NEGATION_MISSING_STATUS_ERR_OFFSET		= 0x9,
	PAC_RXC_GLOBAL_NEGATION_MISSING_STATUS_ERR2_OFFSET		= 0xA,
	PAC_RXC_GLOBAL_NEGATION_MISSING_METRICS_ERR2_OFFSET 	= 0xB,
	PAC_RXC_NOT_MY_STA_FIFO_FULL_DROP_OFFSET				= 0xC,
	PAC_RXC_NOT_MY_STA_FIFO_DECREMENT_LESS_THAN_ZERO_OFFSET = 0xD,
#endif
	PAC_RXC_NUM_OF_ERRORS
}PacRxErrors_e;



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/                       Data Type Definition                        
/----------------------------------------------------------------------------------*/


typedef enum
{
	RXC_DISABLE_PHY_RX_ABORT_RSSI = 0,
	RXC_ENABLE_PHY_RX_ABORT_RSSI = 1
} RxcPhyRxAbortRssiThresholdStatus_e;

typedef struct RxcAmpduNoBaAggrementFifoManagement
{
	uint16 readIndex;
} RxcAmpduNoBaAggrementFifoManagement_t;


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

// Pointer to start of RXC traffic indications RAM
#if defined (ENET_INC_UMAC)	
static RxcTrafficIndRam_t* RxcTrafficIndRamPtr; 
#endif

#if defined(ENET_INC_LMAC)

uint32	RxcErrorIrqCounter[PAC_RXC_NUM_OF_ERRORS] = {0};
RegPacRxcRxcErrIrqStatus_u	rxcErrIrqStatusReg; 

// holds the fifo management parameters
static RxcAmpduNoBaAggrementFifoManagement_t RxcFifoObj;

// Currently this table is not per VAP sine we configure all VAPs to be the same.
// If for some reason we need different configurations per VAP (or per CDB), need to multiply the array by num of VAPs (or bands)
const static uint16 RxcMpduTypeSubTypeRejectFilter[RXC_RXF_MPDU_TYPE_NUM] = 
{
	RXC_RXF_MANAGEMENT_SUBTYPES_CONFIG,
	RXC_RXF_CONTROL_SUBTYPES_CONFIG,
	RXC_RXF_DATA_SUBTYPES_CONFIG,
	RXC_RXF_RESERVED_SUBTYPES_CONFIG,
};
#endif /* #ifdef ENET_INC_LMAC */


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

void Pac_RxcSetForceCca(void)
{
	uint32 regVal = 0;
	
	/* read */
	RegAccess_Read(REG_PAC_RXC_MT_FORCE_CCA, &regVal);

	ILOG0_D("[PTA][Pac_RxcSetForceCca] read regVal = 0x%x", regVal);

	/* Change MUX to force */
	regVal |= (1 << RXC_FORCE_SW_CCA_SHIFT);
	
	/* Modify the value: set CCA to "NOSIY" ('0') */
	regVal &= ~(1 << RXC_FORCED_SW_CCA_SHIFT);

	ILOG0_D("[PTA][Pac_RxcSetForceCca] write regVal = 0x%x", regVal);

	/* write */
	RegAccess_Write(REG_PAC_RXC_MT_FORCE_CCA, regVal);
}

void Pac_RxcSetForceCcaToNormalMode(void)
{
	uint32 regVal = 0;
	
	/* read */
	RegAccess_Read(REG_PAC_RXC_MT_FORCE_CCA, &regVal);

	ILOG0_D("[PTA][Pac_RxcSetForceCcaToNormalMode] read regVal = 0x%x", regVal);

	/* Change MUX to normal state */
	regVal &= ~(1 << RXC_FORCE_SW_CCA_SHIFT);

	/* Modify the value: set CCA to "NORMAL" ('1') */
	// nothying to do

	ILOG0_D("[PTA][Pac_RxcSetForceCcaToNormalMode] write regVal = 0x%x", regVal);
	
	/* write */
	RegAccess_Write(REG_PAC_RXC_MT_FORCE_CCA, regVal);
}


#if defined (ENET_INC_UMAC) // the traffic infication functions are in the UM since it's common for both bands
/**********************************************************************************
Pac_RxcBaAgreementRecipientGetAndClearActivity


Description:
----------
Reads and clears the activity bit in the BA agreement recipient activity bitmap for STA/TID

**********************************************************************************/
bool Pac_RxcBaAgreementRecipientGetAndClearActivity(StaId stationIndex, uint8 tid)
{
	bool 	bitIsSet 	=  FALSE;
	uint32 	mask 		= 0;
	uint8 	row			= 0; // In which row is the station located (0-63). See rxc document, section "Traffic indications"
	uint8 	column 		= 0; // In which column is the the station located 0-3.

	// Every row has 4 stations hence divide by 4
	row = stationIndex >> TRAFFIC_IND_GET_ROW;		
	
	// Modulu 4 ("AND" with 0x3) to get the  position within the row
	column = stationIndex & TRAFFIC_IND_GET_COLUMN;	

	// Multiply column by 8 to get to the correct bits for this station (jump over tid 0-7 of other stations in the row) and add tid position
	mask = (RXC_INDICATION_SET << (tid + (column << TRAFFIC_IND_MUL_8))); 

    // Read the field and move bit to location '0'
	bitIsSet = ((RxcTrafficIndRamPtr->BaAgreementRecipientActivityBitmap[row].val & mask) >> (tid + (column << TRAFFIC_IND_MUL_8))); 

	// Clear the bit - write '1' only to this bit in order to clear it
	RxcTrafficIndRamPtr->BaAgreementRecipientActivityBitmap[row].val = mask;  
	
	return bitIsSet;
}


/**********************************************************************************
Pac_RxcBaAgreementInitiatorGetAndClearActivity


Description:
----------
Reads and clears the activity bit in the BA agreement initiator activity bitmap for STA/TID

**********************************************************************************/
bool Pac_RxcBaAgreementInitiatorGetAndClearActivity(StaId stationIndex, uint8 tid)
{
	bool 	bitIsSet 	=  FALSE;
	uint32 	mask 		= 0;
	uint8 	row			= 0; // In which row is the station located (0-63). See rxc document, section "Traffic indications"
	uint8 	column 		= 0; // In which column is the the station located 0-3.

	// Every row has 4 stations hence divide by 4
	row = stationIndex >> TRAFFIC_IND_GET_ROW;		
	
	// Modulu 4 ("AND" with 0x3) to get the  position within the row
	column = stationIndex & TRAFFIC_IND_GET_COLUMN;	

	// Multiply column by 8 to get to the correct bits for this station (jump over tid 0-7 of other stations in the row) and add tid position
	mask = (RXC_INDICATION_SET << (tid + (column << TRAFFIC_IND_MUL_8))); 

    // Read the field and move bit to location '0'
	bitIsSet = ((RxcTrafficIndRamPtr->BaAgreementInitiatorActivityBitmap[row].val & mask) >> (tid + (column << TRAFFIC_IND_MUL_8))); 

	// Clear the bit - write '1' only to this bit in order to clear it
	RxcTrafficIndRamPtr->BaAgreementInitiatorActivityBitmap[row].val = mask;  
	
	return bitIsSet;
}

#ifdef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************
Pac_RxcRxPpGetAndClearActivity


Description:
----------
Reads and clears the activity bit in the BA agreement RxPP activity bitmap for STA/TID

**********************************************************************************/
bool Pac_RxcRxPpGetAndClearActivity(StaId stationIndex, uint8 tid)
{
	bool 	bitIsSet 	=  FALSE;
	uint32 	mask 		= 0;
	uint8 	row			= 0; // In which row is the station located (0-63). See rxc document, section "Traffic indications"
	uint8 	column 		= 0; // In which column is the the station located 0-3.

	// Every row has 4 stations hence divide by 4
	row = stationIndex >> TRAFFIC_IND_GET_ROW;		
	
	// Modulu 4 ("AND" with 0x3) to get the  position within the row
	column = stationIndex & TRAFFIC_IND_GET_COLUMN;	

	// Multiply column by 8 to get to the correct bits for this station (jump over tid 0-7 of other stations in the row) and add tid position
	mask = (RXC_INDICATION_SET << (tid + (column << TRAFFIC_IND_MUL_8))); 

    // Read the field and move bit to location '0'
	bitIsSet = ((RxcTrafficIndRamPtr->BaAgreementRxPpActivityBitmap[row].val & mask) >> (tid + (column << TRAFFIC_IND_MUL_8))); 

	// Clear the bit - write '1' only to this bit in order to clear it
	RxcTrafficIndRamPtr->BaAgreementRxPpActivityBitmap[row].val = mask;  
	
	return bitIsSet;
}
#endif
/**********************************************************************************
Pac_RxcGetAndClearStationActivity


Description:
----------
Reads and clears the activity bit of the specific station

**********************************************************************************/
uint8 Pac_RxcGetAndClearStationActivity(StaId stationIndex)
{
	uint32 	mask 					= 0;	
	uint8 	bitIsSet 				= FALSE;
	uint8 	activityBitWordNumber 	= 0;
	
	activityBitWordNumber = CONVERT_BIT_INDEX_TO_WORDS(stationIndex);	
	mask = RXC_INDICATION_SET << (stationIndex & RXC_STA_ACTIVITY_BIT_LOCATION_MASK);

	/* Check the bit */
	if (RxcTrafficIndRamPtr->stationActivityBitmap[activityBitWordNumber] & mask)
	{
		bitIsSet = TRUE;
	}

	/* Clear the bit - write '1' only to this bit in order to clear it */
	RxcTrafficIndRamPtr->stationActivityBitmap[activityBitWordNumber] = mask;  

	return bitIsSet;
}


/**********************************************************************************
Pac_RxcInitializeRamParameters


Description:
----------
Initialize the RXC RAM parameters

**********************************************************************************/
void Pac_RxcInitializeRamParameters(void)
{
	// Clear (Set to zero) the entire memory of the traffic indication Bitmap.
	// "TRAFFIC_INDICATOR_MEM" is a regular access memory (unlike "TRAFFIC_INDICATOR_BASE_ADDR" which is cleared by writing "1")
	memset32((void *)TRAFFIC_INDICATOR_MEM, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxcTrafficIndRam_t)));

	// Init the pointer	
	RxcTrafficIndRamPtr = (RxcTrafficIndRam_t *)RXC_TRAFFIC_INDICATION_CLEAR_BY_ONE_BITMAP;
}

#else // the following functions are the driver in the LMs

#ifdef ENET_INC_ARCH_WAVE600D2

/**********************************************************************************
Pac_RxcSetAidForStaMode



Description:
----------
AID configuration in RxC for TF Rx purpose.

**********************************************************************************/
void Pac_RxcSetAidForStaMode(uint8 vapId, uint16 aid, bool valid)
{
	RxfMem_t* pRxfMem = (RxfMem_t*)RXC_RXF_MEM_BASE_ADDR_PER_BAND;

	// Each word holds data for 2 VAPs, so first check if VAP ID is even or odd
	if ((vapId & 0x1) == 0)
	{
		pRxfMem->aidValue[vapId >> 1].aidValueVapIdx0 = aid;		
		pRxfMem->aidValue[vapId >> 1].aidValidVapIdx0 = valid;
	}
	else
	{
		pRxfMem->aidValue[vapId >> 1].aidValueVapIdx1 = aid;		
		pRxfMem->aidValue[vapId >> 1].aidValidVapIdx1 = valid;
	}
}
#endif
/**********************************************************************************
Pac_RxcSetBssModeRejectFilterStatus


Description:
------------
	set/reset per vap a reject filter that is based on received packet Bss mode(fromToDs bits)

Input:
-----
	uint8 vapId
	RxfBssMode_e bssMode - [from DS, To DS] = {00,01,10,11}
	RxfFilterStatus_e status - enable/disable

Output:
-------
    None

Returns:
--------
    None
**********************************************************************************/
void Pac_RxcSetBssModeRejectFilterStatus(uint8 vapId, RxfBssMode_e bssMode, RxfFilterStatus_e status)
{
	uint32 mask = RXF_FILTER_STATUS_ENABLE << vapId;
	uint32 data = status << vapId;
	
	switch (bssMode)
	{
		case RXF_BSS_MODE_IBSS:
			RegAccess_WriteMasked(REG_PAC_RXC_BSS_FROM_TO_DS_00_VAP_FILTER_EN, mask, data);	
			break;	
			
		case RXF_BSS_MODE_STA:
			RegAccess_WriteMasked(REG_PAC_RXC_BSS_FROM_TO_DS_01_VAP_FILTER_EN, mask, data);
			break;	
			
		case RXF_BSS_MODE_AP:
			RegAccess_WriteMasked(REG_PAC_RXC_BSS_FROM_TO_DS_10_VAP_FILTER_EN, mask, data);
			break;	
			
		case RXF_BSS_MODE_WDS:
			RegAccess_WriteMasked(REG_PAC_RXC_BSS_FROM_TO_DS_11_VAP_FILTER_EN, mask, data);
			break;	
	
		default:
			DEBUG_ASSERT(0);
			break;
	}		
}

/**********************************************************************************
Pac_RxcSetMACAddressfilter

Description:
----------
Set the MAC address of a specific VAP in the filters which RXC uses.
Addresses are in a dedicated block called VAP ADDR REG.

**********************************************************************************/
void Pac_RxcSetMACAddressfilter (uint8 vapId, IEEE_ADDR *address)
{	
	RegVapAddrRegVap0Addr1Reference0_u	Vap0Addr1Reference0Reg;
	RegVapAddrRegVap0Addr1Reference1_u	Vap0Addr1Reference1RegS;
	RegVapAddrRegB0VapIdxEn_u			b0VapIdxEnReg;
	RegMacHtExtensionsMacBandId_u		macBand;
	uint32 								longFilterRef0Address;
	uint32 								longFilterRef1Address;	
	uint32								bandRegAddress;
	
	ASSERT(RXC_MAX_MY_ADDRESS_FILTERS > vapId);

	Vap0Addr1Reference0Reg.val 	= 0;
	Vap0Addr1Reference1RegS.val = 0;
	b0VapIdxEnReg.val 			= 0;
	macBand.val 				= 0;

	// Setting the reference data
	Vap0Addr1Reference0Reg.bitFields.vap0Addr1Reference0 = 
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_FIRST_BYTE] << RXC_IEEE_ADDRESS_FIRST_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_SECOND_BYTE] << RXC_IEEE_ADDRESS_SECOND_BYTE_SHIFT))  |
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_THIRD_BYTE] << RXC_IEEE_ADDRESS_THIRD_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_FORTH_BYTE] << RXC_IEEE_ADDRESS_FORTH_BYTE_SHIFT)) ; 

	Vap0Addr1Reference1RegS.bitFields.vap0Addr1Reference1 = 
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_FIFTH_BYTE] << RXC_IEEE_ADDRESS_FIFTH_BYTE_SHIFT)) |
			((uint32)(address->au8Addr[RXC_IEEE_ADDRESS_SIXT_BYTE] << RXC_IEEE_ADDRESS_SIXT_BYTE_SHIFT));

	// compute ref reg address by vap ID
	longFilterRef0Address = REG_VAP_ADDR_REG_VAP0_ADDR1_REFERENCE_0 + (vapId << RXC_VAP_ADDR_REFERENCE_REG_SHIFT);
	longFilterRef1Address = longFilterRef0Address + RXC_VAP_ADDR_REFERENCE_REG_SIZE;

	// refernece val is 6 bytes long and configured in two 32 bits registers (4 LSbytes + 2 HSbyts) 
	RegAccess_Write(longFilterRef0Address, Vap0Addr1Reference0Reg.val);
	RegAccess_Write(longFilterRef1Address, Vap0Addr1Reference1RegS.val);

	//Check the band of this LMAC CPU
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_MAC_BAND_ID, &macBand.val);

	// Get the register for this band
	bandRegAddress = REG_VAP_ADDR_REG_B0_VAP_IDX_EN + (macBand.bitFields.macBandId * (REG_VAP_ADDR_REG_B1_VAP_IDX_EN - REG_VAP_ADDR_REG_B0_VAP_IDX_EN));

	// Set the VAP bit in the register of this band (read-modify-write)
	RegAccess_Read(bandRegAddress, &b0VapIdxEnReg.val);
	b0VapIdxEnReg.val |= (TRUE << vapId);	
	RegAccess_Write(bandRegAddress, b0VapIdxEnReg.val);
}


/**********************************************************************************
Pac_RxcClearAddressFilter


Description:
----------
Clear the MAC address of a specific VAP in the filters which RXC uses.
Addresses are in a dedicated block called VAP ADDR REG.

**********************************************************************************/
void Pac_RxcClearAddressFilter (uint8 vapId) 
{
	RegVapAddrRegVap0Addr1Reference0_u	Vap0Addr1Reference0Reg;
	RegVapAddrRegVap0Addr1Reference1_u	Vap0Addr1Reference1RegS;
	RegVapAddrRegB0VapIdxEn_u			b0VapIdxEnReg;
	RegMacHtExtensionsMacBandId_u		macBand;
	uint32 								longFilterRef0Address;
	uint32 								longFilterRef1Address;	
	uint32								bandRegAddress;


	ASSERT(RXC_MAX_MY_ADDRESS_FILTERS > vapId);

	Vap0Addr1Reference0Reg.val 	= 0;
	Vap0Addr1Reference1RegS.val = 0;
	b0VapIdxEnReg.val 			= 0;
	macBand.val 				= 0;
	
	// compute ref reg address by vap ID
	longFilterRef0Address = REG_VAP_ADDR_REG_VAP0_ADDR1_REFERENCE_0 + (vapId << RXC_VAP_ADDR_REFERENCE_REG_SHIFT);
	longFilterRef1Address = longFilterRef0Address + RXC_VAP_ADDR_REFERENCE_REG_SIZE;

	// Clearing the reference data (set mac address t0 00:00:00:00:00:00)
	Vap0Addr1Reference0Reg.bitFields.vap0Addr1Reference0 = 0;
	RegAccess_Write(longFilterRef0Address, Vap0Addr1Reference0Reg.val);
	Vap0Addr1Reference1RegS.bitFields.vap0Addr1Reference1 = 0;
	RegAccess_Write(longFilterRef1Address, Vap0Addr1Reference1RegS.val);

	//Check the band of this LMAC CPU
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_MAC_BAND_ID, &macBand.val);

	// Get the register for this band
	bandRegAddress = REG_VAP_ADDR_REG_B0_VAP_IDX_EN + (macBand.bitFields.macBandId * (REG_VAP_ADDR_REG_B1_VAP_IDX_EN - REG_VAP_ADDR_REG_B0_VAP_IDX_EN));

	// Clear the VAP bit in the register of this band (read-modify-write)
	RegAccess_Read(bandRegAddress, &b0VapIdxEnReg.val);
	b0VapIdxEnReg.val &= ~(TRUE << vapId);	
	RegAccess_Write(bandRegAddress, b0VapIdxEnReg.val);
}


/**********************************************************************************
Pac_RxcInitializeNoBaAgreementFifo


Description:
----------
Init the fifos of AGG with no BA indications. Each LM init the fifo of its own band

**********************************************************************************/
void Pac_RxcInitializeNoBaAgreementFifo(void)
{
#if defined (ENET_INC_LMAC0)	
	// init fifo of Band 0 (The fifo is word alligned)
	memset32(RxfAmpduNoBaAggrementFifo, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxfAmpduNoBaAggrementFifo)));
	RxcFifoObj.readIndex = 0;
#endif
#if defined (ENET_INC_LMAC1)	
	// init fifo of Band 1 (The fifo is word alligned)
	memset32(RxfAmpduNoBaAggrementFifo_B1, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxfAmpduNoBaAggrementFifo_B1)));
	RxcFifoObj.readIndex = 0;
#endif	
}


/**********************************************************************************
Pac_RxcGetFifoParameters


Description:
----------
Read number of valid entries and current index from AGG WITH NO BA fifo.

**********************************************************************************/
void Pac_RxcGetFifoParameters(RxfFifoParameters_t *fifoParameters)
{
	RegPacRxcRxfFifoDebug_u  			rxfFifoStatusRegister;
	RegPacRxcRxfFifoNumEntriesCount_u 	numberOfValidEntriesRegister;

	rxfFifoStatusRegister.val = 0;
	numberOfValidEntriesRegister.val = 0;

	RegAccess_Read(REG_PAC_RXC_RXF_FIFO_DEBUG, &rxfFifoStatusRegister.val);

	/* Check that errors didnt occur */
	ASSERT(!(rxfFifoStatusRegister.bitFields.rxfFifoFull || rxfFifoStatusRegister.bitFields.rxfFifoDecrementLessThanZero)); 

	RegAccess_Read(REG_PAC_RXC_RXF_FIFO_NUM_ENTRIES_COUNT, &numberOfValidEntriesRegister.val);

	fifoParameters->numberOfEntries = numberOfValidEntriesRegister.bitFields.rxfFifoNumEntriesCount;
	fifoParameters->currentIndexInFifo = RxcFifoObj.readIndex;
}


/**********************************************************************************
Pac_RxcSetFifoParameters


Description:
----------
decease number of valid entries which were already read by FW in AGG WITH NO BA fifo.

**********************************************************************************/
void Pac_RxcSetFifoParameters(RxfFifoParameters_t *fifoParameters)
{
	RegPacRxcRxfFifoRdEntriesNum_u numberOfEntriesReadByFwRegister; 

	numberOfEntriesReadByFwRegister.val = 0;

	numberOfEntriesReadByFwRegister.bitFields.rxfFifoRdEntriesNum = fifoParameters->numberOfEntries;
	RxcFifoObj.readIndex = fifoParameters->currentIndexInFifo;

	RegAccess_Write(REG_PAC_RXC_RXF_FIFO_RD_ENTRIES_NUM, numberOfEntriesReadByFwRegister.val);
}


/**********************************************************************************
Pac_RxcEnable


Description:
----------
Enable the RXC

**********************************************************************************/
void Pac_RxcEnable(void)
{
	RegPacRxcRxControl_u rxControlReg; 
	
	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL,&rxControlReg.val);
	rxControlReg.bitFields.rxcEnable = TRUE;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);				
}


/**********************************************************************************
Pac_RxcDisable


Description:
----------
Disable the RXC

**********************************************************************************/
void Pac_RxcDisable(void)
{
	RegPacRxcRxControl_u rxControlReg; 

	rxControlReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL,&rxControlReg.val);
	rxControlReg.bitFields.rxcEnable = FALSE;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);		 
}


/**********************************************************************************
Pac_RxcUpdateMinimalRssiThreshold


Description:
----------
Update minimal RSSI threshold as sent by driver

**********************************************************************************/
void Pac_RxcUpdateMinimalRssiThreshold(int8 rssiThreshlod)
{
	RegPacRxcMinimalRssiThreshold_u minimalRssiThresholdReg; // Minimal RSSI threshold. , Signed value. , Used for setting PHY abort and NAV update.

	// Set Minimal RSSI threshold value. 
	// Following an alignment with system, changed the hardcoded value of 0x2E (46 dB) to below
	// define as in W600, RSSI is already in dBm	

	minimalRssiThresholdReg.val = rssiThreshlod + GEN6_PHY_RSSI_TO_DBM;

	RegAccess_Write(REG_PAC_RXC_MINIMAL_RSSI_THRESHOLD, minimalRssiThresholdReg.val);
}


/**********************************************************************************
Pac_RxcReadRxBufferOverflowCounter

Description:
----------
Read overflow counter


**********************************************************************************/
uint16 Pac_RxcReadRxBufferOverflowCounter(void)
{
	RegPacRxcDebugCtr1_u debugCtr1Reg;
	debugCtr1Reg.val = 0;

	RegAccess_Read(REG_PAC_RXC_DEBUG_CTR1, &debugCtr1Reg.val);

	return (debugCtr1Reg.bitFields.bufferOverflowCtr);
}

/**********************************************************************************

Pac_RxcClearRxBufferOverflowCounter

Description:
---------
Clear RxC bufer overflow dropped mpdu counter

**********************************************************************************/
void Pac_RxcClearRxBufferOverflowCounter(void)
{
	RegPacRxcDebugCtrClear_u	regPacRxcDebugCtrClear;
	
	regPacRxcDebugCtrClear.val = 0;
	regPacRxcDebugCtrClear.bitFields.bufferOverflowCtrClear = 1;
	RegAccess_Write(REG_PAC_RXC_DEBUG_CTR_CLEAR, regPacRxcDebugCtrClear.val);
}


void Pac_RxcClearMpduEndErrorCtrl(void)
{
	RegPacRxcDebugCtrClear_u	regPacRxcDebugCtrClear;
	
	regPacRxcDebugCtrClear.val = 0;
	regPacRxcDebugCtrClear.bitFields.mpduEndErrorCtrClrStb = 1;
	RegAccess_Write(REG_PAC_RXC_DEBUG_CTR_CLEAR, regPacRxcDebugCtrClear.val);
}

uint16 Pac_RxcReadMpduCounter(void)
{
	RegPacRxcDebugCtr4_u PacRxcDebugCtr;
	RegAccess_Read(REG_PAC_RXC_DEBUG_CTR4, &PacRxcDebugCtr.val);	
	return PacRxcDebugCtr.bitFields.mpduEndErrorCtr;
}


/**********************************************************************************
Pac_RxcErrorIrq

Description:
----------
Handle Error IRQ interrupt

**********************************************************************************/
void Pac_RxcErrorIrq(void)
{
#ifdef PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE
	RegPacRxcRxcErrIrqClr_u 	regPacRxcRxcErrIrqClr;
#endif //PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE


	// Read the status register
	RegAccess_Read(REG_PAC_RXC_RXC_ERR_IRQ_STATUS, &rxcErrIrqStatusReg.val);

	if (rxcErrIrqStatusReg.bitFields.fifoRxPmClrWhenNotEmptyIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_FIFO_RX_PM_CLEAR_NOT_EMPTY_IRQ_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: fifoRxPmClrWhenNotEmptyIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxReadyNegationNoMetricsErrIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_READY_NEGATION_NOMETRICS_OFFSET]++;
#ifdef PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE
		rxcErrIrqStatusReg.bitFields.rxReadyNegationNoMetricsErrIrq = FALSE;
		// Clear IRQ
		regPacRxcRxcErrIrqClr.val = 0;
		regPacRxcRxcErrIrqClr.bitFields.rxReadyNegationNoMetricsErrIrqClr = 1;
		RegAccess_Write(REG_PAC_RXC_RXC_ERR_IRQ_CLR, regPacRxcRxcErrIrqClr.val);
#else
		FATAL("Pac_RxcErrorIrq: rxReadyNegationNoMetricsErrIrq");
#endif //PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE 
	}
	if (rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingMetricsErrIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_GLOBAL_NEGATION_MISSING_METRICS_OFFSET]++;	
#ifdef PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE
		rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingMetricsErrIrq = FALSE;
		// Clear IRQ
		regPacRxcRxcErrIrqClr.val = 0;
		regPacRxcRxcErrIrqClr.bitFields.rxGloblNegationMissingMetricsErrIrqClr = 1;
		RegAccess_Write(REG_PAC_RXC_RXC_ERR_IRQ_CLR, regPacRxcRxcErrIrqClr.val);
#else
		FATAL("Pac_RxcErrorIrq: rxGloblNegationMissingMetricsErrIrq");
#endif //PAC_RXC_NO_ASSERT_FOR_ERRORS_1_2_PHY_ISSUE
	}
	if (rxcErrIrqStatusReg.bitFields.mpduDropDlmInEmptyCtrNotZero == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_MPDU_DROP_DLM_IN_EMPTY_NOT_ZERO_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: mpduDropDlmInEmptyCtrNotZero");
	}
	if (rxcErrIrqStatusReg.bitFields.phyMacFifoFullDropCtrNotZero == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_PHY_MAC_DROP_CTR_NOT_ZERO_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: phyMacFifoFullDropCtrNotZero");
	}
	if (rxcErrIrqStatusReg.bitFields.bufferOverflowCtrNotZero == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_BUFFER_OVERFLOW_CTR_NOT_ZERO_OFFSET]++;
#ifdef	PAC_RXC_NO_ASSERT_FOR_ERROR_5_RXC_ISSUE
		rxcErrIrqStatusReg.bitFields.bufferOverflowCtrNotZero = FALSE;
		
		ILOG0_D("Pac_RxcErrorIrq: CyclicBufferOverflowCounter = %d", CyclicBufferOverflowCounter);

		// We clear the counter here and not in "hwerrorsRxcError" in order to still have the interrupt set (otherwise it will be turned off before we enter here)
		Pac_RxcClearRxBufferOverflowCounter();
#else
		FATAL("Pac_RxcErrorIrq: bufferOverflowCtrNotZero");
#endif // PAC_RXC_NO_ASSERT_FOR_ERROR_5_RXC_ISSUE
	}
#ifdef ENET_INC_ARCH_WAVE600B
	if (rxcErrIrqStatusReg.bitFields.rxfFifoFullDropIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_RXF_FIFO_FULL_DROP_IRQ_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxfFifoFullDropIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxfFifoDecrementLessThanZeroIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_RXF_FIFO_DECREMENT_LESS_THAN_ZERO_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxfFifoDecrementLessThanZeroIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingBfMetricsErrIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_GLOBAL_NEGATION_MISSING_BF_METRICS_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxGloblNegationMissingBfMetricsErrIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingStatusErrIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_GLOBAL_NEGATION_MISSING_STATUS_ERR_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxGloblNegationMissingStatusErrIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingStatusErr2Irq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_GLOBAL_NEGATION_MISSING_STATUS_ERR2_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxGloblNegationMissingStatusErr2Irq");
	}
	if (rxcErrIrqStatusReg.bitFields.rxGloblNegationMissingMetricsErr2Irq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_GLOBAL_NEGATION_MISSING_METRICS_ERR2_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rxGloblNegationMissingMetricsErr2Irq");
	}
	if (rxcErrIrqStatusReg.bitFields.rssiNotMyStaFifoFullDropIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_NOT_MY_STA_FIFO_FULL_DROP_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rssiNotMyStaFifoFullDropIrq");
	}
	if (rxcErrIrqStatusReg.bitFields.rssiNotMyStaFifoDecrementLessThanZeroIrq == TRUE)
	{
		RxcErrorIrqCounter[PAC_RXC_NOT_MY_STA_FIFO_DECREMENT_LESS_THAN_ZERO_OFFSET]++;
		FATAL("Pac_RxcErrorIrq: rssiNotMyStaFifoDecrementLessThanZeroIrq");
	}
#endif
	/* Validate that IRQ status is cleared */
	DEBUG_ASSERT(rxcErrIrqStatusReg.val == 0);
}


/**********************************************************************************
Pac_RxcSetReservedStorageSize


Description:
----------
Configure number of reserved words on the cyclic buffer before the frame header is written.

**********************************************************************************/
void Pac_RxcSetReservedStorageSize(uint8 reservedStorageSize)
{
	RegPacRxcRxControl_u rxControlReg;
	rxControlReg.val = 0;	

	// Storage size is in resolution of 4 bytes. Max value is 15 words.
	ASSERT(reservedStorageSize < 16);

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);
	rxControlReg.bitFields.reservedStorageSize = reservedStorageSize;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);		 
}


/**********************************************************************************
Pac_RxcEnableRssiAbort


Description:
----------
Enable filtering frames based on minimal RSSI value

**********************************************************************************/
void Pac_RxcEnableRssiAbort(void)
{
	RegPacRxcRxControl_u rxControlReg;

	rxControlReg.val = 0;
	
	// Read-Modify-Write
	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);	
	rxControlReg.bitFields.phyRxAbortRssiEn = RXC_ENABLE_PHY_RX_ABORT_RSSI;	
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);
}


/**********************************************************************************
Pac_RxcDisableRssiAbort


Description:
----------
Disable filtering frames based on minimal RSSI value

**********************************************************************************/
void Pac_RxcDisableRssiAbort(void)
{
	RegPacRxcRxControl_u rxControlReg;

	rxControlReg.val = 0;
	
	// Read-Modify-Write
	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);	
	rxControlReg.bitFields.phyRxAbortRssiEn = RXC_DISABLE_PHY_RX_ABORT_RSSI;	
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);
}


/**********************************************************************************
Pac_RxcDisableCcaIndication


Description:
----------
disable CCA  indication on secondary channels , for secondary 20Mhz,  40Mhz and 80Mhz

**********************************************************************************/
void Pac_RxcDisableCcaIndication(void)
{
	RegPacRxcMtForceCca_u forceCcaReg;

	forceCcaReg.val = 0;

	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &forceCcaReg.val);	
	forceCcaReg.bitFields.forceCca20SFree = TRUE; 
	forceCcaReg.bitFields.forceCca40SFree = TRUE;
	forceCcaReg.bitFields.forceCca80SFree = TRUE;	
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, forceCcaReg.val);
}


/**********************************************************************************
Pac_RxcSetHeFragViolationFilter


Description:
----------
Set 2 filters for Fragmentation violation in HE: 1) HE Single MPDU. 2) HE AMPDU

**********************************************************************************/
void Pac_RxcSetHeFragViolationFilter(uint8 supportedHeFragLevel, uint8 vapIndex)
{
	RegPacRxcHeSingleMpduFragmentVapFilterEn_u 	HeSingleMpduFragmentVapFilterEnReg;
	RegPacRxcHeAggMpduFragmentVapFilterEn_u		HeAggMpduFragmentVapFilterEnReg;
	uint32 										singleMpduFilterMask = 0;
	uint32 										ampduFilterMask = 0;	


	HeSingleMpduFragmentVapFilterEnReg.val 	= 0;
	HeAggMpduFragmentVapFilterEnReg.val		= 0;

	// Acording to SAS - Single MPDU Frag rules:
	//1) If VAP HE Fragmentation support level=0, HE Single MPDU Fragment violation indication=1.
	//2) If VAP HE Fragmentation support level<>0, HE Single MPDU Fragment violation indication=0.
	if (supportedHeFragLevel == 0)
	{
		singleMpduFilterMask = (0x1 << vapIndex);
	}

	// Read-Modify(clean+update)-write
	RegAccess_Read(REG_PAC_RXC_HE_SINGLE_MPDU_FRAGMENT_VAP_FILTER_EN, &HeSingleMpduFragmentVapFilterEnReg.val);	
	HeSingleMpduFragmentVapFilterEnReg.bitFields.heSingleMpduFragmentVapFilterEn &= ~(0x1 << vapIndex);	
	HeSingleMpduFragmentVapFilterEnReg.bitFields.heSingleMpduFragmentVapFilterEn |= singleMpduFilterMask;	
	RegAccess_Write(REG_PAC_RXC_HE_SINGLE_MPDU_FRAGMENT_VAP_FILTER_EN, HeSingleMpduFragmentVapFilterEnReg.val);

	// Acording to SAS - AMPDU Frag rules:
	// 1) If VAP HE Fragmentation support level=0 or 1, HE A-MPDU Fragment violation indication=1.
	// 2) If VAP HE Fragmentation support level=2 or 3, HE A-MPDU Fragment violation indication=0.
	if (supportedHeFragLevel <= 1)
	{
		ampduFilterMask = (0x1 << vapIndex);
	}

	// Read-Modify(clean+update)-write
	RegAccess_Read(REG_PAC_RXC_HE_AGG_MPDU_FRAGMENT_VAP_FILTER_EN, &HeAggMpduFragmentVapFilterEnReg.val);	
	HeAggMpduFragmentVapFilterEnReg.bitFields.heAggMpduFragmentVapFilterEn &= ~(0x1 << vapIndex);	
	HeAggMpduFragmentVapFilterEnReg.bitFields.heAggMpduFragmentVapFilterEn |= ampduFilterMask;	
	RegAccess_Write(REG_PAC_RXC_HE_AGG_MPDU_FRAGMENT_VAP_FILTER_EN, HeAggMpduFragmentVapFilterEnReg.val);
}

/**********************************************************************************
Pac_RxcSetStaFrameClass


Description:
----------
Set allowed recieved frame class for the required station (class 1/2/3)

**********************************************************************************/
void Pac_RxcSetStaFrameClass(uint32 staIndex,uint32 allowedFrameClass)
{
	uint32	ramAddress;
	uint32 	frameClassBitOffset;
	uint32	frameClassMask;
	uint32	frameClassWord;

	// Get the correct word address. 16 STAs per word.
	ramAddress = (RXC_RXF_STA_FRAME_CLASS_RAM_START + CONVERT_WORDS_TO_BYTES(staIndex >> RXC_RXF_STA_FRAME_CLASS_STA_IDX_TO_WORD_ADDR_SHIFT));

	// find the location of the STA bits within the word (each STA is 2 bits)
	frameClassBitOffset = ((staIndex & RXC_RXF_STA_FRAME_CLASS_STA_IDX_TO_BIT_OFFSET_MASK) * RXC_RXF_STA_FRAME_CLASS_ENTRY_WIDTH);

	// Get the 2 bits mask in the correct place in the word.
	frameClassMask = (RXC_RXF_STA_FRAME_CLASS_ENTRY_MASK << frameClassBitOffset);

	//Read the word from RAM
	frameClassWord = *(uint32*)ramAddress;
	
	// Clear the old frame class
	frameClassWord &= ~frameClassMask;

	// Set the new frame class
	frameClassWord |=  ((allowedFrameClass << frameClassBitOffset) & frameClassMask);

	// Write the word back to RAM
	*(uint32*)ramAddress = frameClassWord;
}


/**********************************************************************************
Pac_RxcPhyStatusFilterSetGroupIndexForSta


Description:
----------
Assign a Group Index for a specific STA for the purpose of PHY Status filtering.

**********************************************************************************/
void Pac_RxcPhyStatusFilterSetGroupIndexForSta(uint8 groupIndex , StaId staIndex)
{
	uint32	ramAddress;
	uint32 	groupIndexBitOffset;
	uint32	groupIndexMask;
	uint32	groupIndexWord;

	// There are only 8 groups
	ASSERT(groupIndex < RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS);

	// Get the correct word address. 8 STAs per word.
	ramAddress = (RXC_RXF_PHY_STATUS_FILTER_STA_GROUP_INDEX_RAM_START + CONVERT_WORDS_TO_BYTES(staIndex >> RXC_RXF_PHY_STATUS_FILTER_STA_IDX_TO_WORD_ADDR_SHIFT));

	// find the location of the STA bits within the word (each STA has 4 bits for group index. 3 for value and MSB is reserved)
	groupIndexBitOffset = ((staIndex & RXC_RXF_PHY_STATUS_FILTER_STA_IDX_TO_BIT_OFFSET_MASK) * RXC_RXF_PHY_STATUS_FILTER_ENTRY_WIDTH);

	// Get the 4 bits mask in the correct place in the word.
	groupIndexMask = (RXC_RXF_PHY_STATUS_FILTER_ENTRY_MASK << groupIndexBitOffset);

	//Read the word from RAM
	groupIndexWord = *(uint32*)ramAddress;
	
	// Clear the old frame class
	groupIndexWord &= ~groupIndexMask;

	// Set the new Group Index for this STA
	groupIndexWord |=  ((groupIndex << groupIndexBitOffset) & groupIndexMask);

	// Write the word back to RAM
	*(uint32*)ramAddress = groupIndexWord;
}


/**********************************************************************************
Pac_RxcPhyStatusFilterEnableDisable


Description:
----------
Enable / Disable specific type of PHY Status filter (FD RSSI / TD RSSI / RF GAIN / NOISE EST) for a specific group

**********************************************************************************/
void Pac_RxcPhyStatusFilterEnableDisable(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, bool isEnable)
{
	uint32	mask;
	uint32	enableDisable;

	ASSERT(groupIndex < RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS);

	// Enable/Disable the group bit in the relevant filter type (each filter is 8 bits long for 8 groups)
	mask = (0x1 << groupIndex) << (RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS * filterType);
	enableDisable = (isEnable << groupIndex) << (RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS * filterType);
	RegAccess_WriteMasked(REG_PAC_RXC_PHY_STATUS_FIELDS_FILTER_EN, mask, enableDisable);
}


/**********************************************************************************
Pac_RxcPhyStatusFilterSetMinMax


Description:
----------
Decide if to use minimum or maximum between the antennas fields: 0 - Maximum , 1 - Minimum

**********************************************************************************/
void Pac_RxcPhyStatusFilterSetMinMax(RxcPhyStatusFilterType_e filterType, bool isMinimum)
{
	uint32	mask;
	uint32	minMax;

	// Set minimum or maximum for this specific filter type (override last known choice)
	mask = (0x1 << filterType);
	minMax = (isMinimum << filterType);
	RegAccess_WriteMasked(REG_PAC_RXC_PHY_STATUS_FIELDS_FILTER_MIN_MAX_N, mask, minMax);
}


/**********************************************************************************
Pac_RxcPhyStatusFilterSetGreaterOrLessEqual


Description:
----------
Decide if to use "<=" or ">="  for the filter operation .
0: Less or equal signed comparison (<=), 1: Greater or equal signed comparison (>=)

**********************************************************************************/
void Pac_RxcPhyStatusFilterSetGreaterOrLessEqual(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, RxcPhyStatusFilterOpration_e filterOperation)
{
	uint32	mask;
	uint32	GeOrLe;

	// Set LessOrEqual (0x1) or GreaterOrEqual (0x0)
	mask = (0x1 << groupIndex) << (RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS * filterType);
	GeOrLe = (filterOperation << groupIndex) << (RXC_PHY_STATUS_FILTER_NUM_OF_GROUPS * filterType);
	RegAccess_WriteMasked(REG_PAC_RXC_PHY_STATUS_FIELDS_FILTER_GE_LE_N, mask, GeOrLe);
}


/**********************************************************************************
Pac_RxcPhyStatusFilterSetReferenceAndMask


Description:
----------
Assign value to the reference and mask for a specific PHY Status filter type for a specific group

**********************************************************************************/
void Pac_RxcPhyStatusFilterSetReferenceAndMask(uint8 groupIndex ,RxcPhyStatusFilterType_e filterType, uint8 referenceVal, uint8 maskVal)
{
	uint32 referenceRamAddress;
	uint32 MaskRamAddress;
	uint32 filterBitOffset;
	uint32 referenceWord;
	uint32 maskWord;	
	uint32 mask;


	// Get the correct word address. 16 STAs per word.
	referenceRamAddress = RXC_RXF_PHY_STATUS_FILTER_REFERENCE_RAM_START + CONVERT_WORDS_TO_BYTES(groupIndex);
	MaskRamAddress 		= RXC_RXF_PHY_STATUS_FILTER_MASK_RAM_START + CONVERT_WORDS_TO_BYTES(groupIndex);

	// find the location of this filter type within the word (each filter type has 8 bits)
	filterBitOffset = filterType * 8;

	// Get the 8 bits mask in the correct place in the word.
	mask = (0xFF << filterBitOffset);

	//Read the reference word from reference RAM and mask word from mask RAM
	referenceWord 	= *(uint32*)referenceRamAddress;
	maskWord 		= *(uint32*)MaskRamAddress;
	
	// Clear the old reference and mask values of this filter type
	referenceWord 	&= ~mask;
	maskWord 		&= ~mask;	

	// Set the new reference and mask values of this filter type
	referenceWord 	|=  ((referenceVal 	<< filterBitOffset) & mask);
	maskWord		|=  ((maskVal 		<< filterBitOffset) & mask);

	// Write the words back to RAMs
	*(uint32*)referenceRamAddress 	= referenceWord;
	*(uint32*)MaskRamAddress 		= maskWord;
}


/**********************************************************************************
Pac_RxcAutoRespFifoUpdateEnable


Description:
----------
configure the RXC to enable/disable auto resp fifo update

**********************************************************************************/
void Pac_RxcAutoRespFifoUpdateEnable(bool flag)
{
	RegPacRxcRxControl_u rxControlReg;

	rxControlReg.val = 0;
	
	RegAccess_Read(REG_PAC_RXC_RX_CONTROL, &rxControlReg.val);
	rxControlReg.bitFields.autoRespFifoUpdateEn = flag;
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL, rxControlReg.val);
}


/**********************************************************************************
Pac_RxcSnifferModeConfiguration


Description:
----------
configure the RXC to support sniffer mode

**********************************************************************************/
void Pac_RxcSnifferModeConfiguration(uint8 bandId)
{
	RegPacRxcAMpduNoBaAgreementFilterEn_u	aMpduNoBaAgreementFilterEnReg;
	RegPacRxcBssFromToDs00VapFilterEn_u		bssFromToDs00VapFilterEnReg; 
	RegPacRxcBssFromToDs01VapFilterEn_u		bssFromToDs01VapFilterEnReg; 
	RegPacRxcBssFromToDs10VapFilterEn_u		bssFromToDs10VapFilterEnReg; 
	RegPacRxcBssFromToDs11VapFilterEn_u		bssFromToDs11VapFilterEnReg; 
	RegVapAddrRegB0ForceVapIdx0_u			vapAddrRegB0ForceVapIdx0Reg;
	RegVapAddrRegB1ForceVapIdx0_u			vapAddrRegB1ForceVapIdx0Reg;
	
	/****	Disable autoreply	****/
	ILOG2_V("-------- RXC disable autoreply for sniffer--------");
	Pac_RxcAutoRespFifoUpdateEnable(FALSE);
	
	/****	Disable all filters	****/
	/******************************/
	ILOG2_V("-------- RXC diable all filters for Sniffer --------");
	//Disanle the "AMPDU WITH NO BA AGG" FIFO indications
	aMpduNoBaAgreementFilterEnReg.bitFields.aMpduNoBaAgreementFilterEn = RXF_FILTER_STATUS_DISABLE;
	RegAccess_Write(REG_PAC_RXC_A_MPDU_NO_BA_AGREEMENT_FILTER_EN, aMpduNoBaAgreementFilterEnReg.val);	
	//Disable bss_from_to_ds_00_vap_filter_en
	bssFromToDs00VapFilterEnReg.bitFields.bssFromToDs00VapFilterEn = 0x0; 
	RegAccess_Write(REG_PAC_RXC_BSS_FROM_TO_DS_00_VAP_FILTER_EN, bssFromToDs00VapFilterEnReg.val);	
	//Disable bss_from_to_ds_01_vap_filter_en
	bssFromToDs01VapFilterEnReg.bitFields.bssFromToDs01VapFilterEn = 0x0; 
	RegAccess_Write(REG_PAC_RXC_BSS_FROM_TO_DS_01_VAP_FILTER_EN, bssFromToDs01VapFilterEnReg.val);	
	//Disable bss_from_to_ds_10_vap_filter_en
	bssFromToDs10VapFilterEnReg.bitFields.bssFromToDs10VapFilterEn = 0x0; 
	RegAccess_Write(REG_PAC_RXC_BSS_FROM_TO_DS_10_VAP_FILTER_EN, bssFromToDs10VapFilterEnReg.val);	
	//Disable bss_from_to_ds_11_vap_filter_en
	bssFromToDs11VapFilterEnReg.bitFields.bssFromToDs11VapFilterEn = 0x0; 
	RegAccess_Write(REG_PAC_RXC_BSS_FROM_TO_DS_11_VAP_FILTER_EN, bssFromToDs11VapFilterEnReg.val);	

	/****	Force VAP0 			****/
	/*config only band0			  */
	/******************************/
	ILOG2_V("-------- RXC force VAP0 for Sniffer --------");


	if (bandId == CONFIGURATION_MANAGER_BAND_0)
	{
		/* Band0*/
		vapAddrRegB0ForceVapIdx0Reg.bitFields.b0ForceVapIdx0 = ENABLED; 
		RegAccess_Write(REG_VAP_ADDR_REG_B0_FORCE_VAP_IDX0, vapAddrRegB0ForceVapIdx0Reg.val);	
	}
	else
	{
		/* Band1 */
		vapAddrRegB1ForceVapIdx0Reg.bitFields.b1ForceVapIdx0 = ENABLED; 
		RegAccess_Write(REG_VAP_ADDR_REG_B1_FORCE_VAP_IDX0, vapAddrRegB1ForceVapIdx0Reg.val);
		
	}
	
	/****	Disable RSSI abort ****/
	/******************************/
	ILOG2_V("-------- RXC disable RSSI abort for Sniffer --------");
	Pac_RxcDisableRssiAbort(); 
	
}


/**********************************************************************************
Pac_RxcPauseRx


Description:
----------
Pause the RX  - RXC pauses the PHY as well. Interrupt will signal that pause is done.

**********************************************************************************/
void Pac_RxcPauseRx(void)
{
	RegPacRxcRxcHaltRequest_u rxcHaltRequestReg;
	
	rxcHaltRequestReg.val = 0; 
	rxcHaltRequestReg.bitFields.rxcHaltRequest = TRUE;

	RegAccess_Write(REG_PAC_RXC_RXC_HALT_REQUEST, rxcHaltRequestReg.val);	
}


/**********************************************************************************
Pac_RxcResumeRx


Description:
----------
Resume the RX  - RXC resumes the PHY as well. Interrupt will signal that resume is done.

**********************************************************************************/
void Pac_RxcResumeRx(void)
{
	RegPacRxcRxcHaltRequest_u rxcHaltRequestReg;
	
	rxcHaltRequestReg.val = 0; 
	rxcHaltRequestReg.bitFields.rxcHaltRequest = FALSE;

	RegAccess_Write(REG_PAC_RXC_RXC_HALT_REQUEST, rxcHaltRequestReg.val);	
}


/**********************************************************************************
Pac_RxcClearRxHaltIrq


Description:
----------
Clear the RX Halt IRQ (pause or resume)

**********************************************************************************/
void Pac_RxcClearRxHaltIrq(void)
{
	RegPacRxcRxcInHalt_u rxcInHaltReg;
	
	// Clear the interrupt by setting IRQClear bit (read-modify-write)
	RegAccess_Read(REG_PAC_RXC_RXC_IN_HALT, &rxcInHaltReg.val);
	rxcInHaltReg.bitFields.rxcInHaltFwStatusIrqClr = TRUE;
	RegAccess_Write(REG_PAC_RXC_RXC_IN_HALT, rxcInHaltReg.val);	
}



/**********************************************************************************

Pac_RxcInit


Description:
------------
	Initialize RXC HW
Input:
-----

Output:
-------
	None

Returns:
--------
	None
**********************************************************************************/
void Pac_RxcInit(void)
{
	RegPacRxcRxControl_u 					rxControlReg;
	RegPacRxcCcaIfsCompensation_u 			ccaIfsCompReg;
	RegPacRxcRxrdyIfsCompensation_u 		rxReadyIfsCompReg;
	RegPacRxcMinimalRssiThreshold_u 		minimalRssiThresholdReg; 
	RegPacRxcRxfFifoDepthMinusOne_u			rxfFifoDepthMinusOneReg;
	RegPacRxcRxfFifoBaseAddr_u				rxfFifoBaseAddrReg;
	RegPacRxcAMpduNoBaAgreementFilterEn_u	aMpduNoBaAgreementFilterEnReg;
	RegPacRxcMaxMpduLength11AgB_u			maxMpduLength11AgBReg;
	RegPacRxcMaxMpduLength11N_u				maxMpduLength11NReg;
	RegPacRxcMaxMpduLength11AcAx_u			maxMpduLength11AcAx;
	RegPacRxcMpduLengthFilterEn_u			mpduLengthFilterEn;
	RegPacRxcFragmentedAMsduFilterEn_u		fragmentedAMsduFilterEnReg;
	RegPacRxcTypeSubtypeFilter0_u			typeSubtypeFilter0Reg;
	RegPacRxcTypeSubtypeFilter1_u			typeSubtypeFilter1Reg;
	RegPacRxcRxcErrIrqEn_u					rxcErrIrqEnReg;
	uint32 									ccaIfsCompenstation 	= 0; 
	uint32 									rxReadyIfsCompenstation = 0; 
	uint8									vapIndex;
	uint32									filterAddress;
	RegPacRxcErrorIfs1_u					regPacRxcErrorIfs1;
	RegPacRxcErrorIfs2_u					regPacRxcErrorIfs2;
	RegPacRxcErrorIfs3_u					regPacRxcErrorIfs3;
	RegPacRxcErrorIfs4_u					regPacRxcErrorIfs4;
	RegPacRxcErrorIfs5_u					regPacRxcErrorIfs5;
	RegPacRxcSifsPlusPhyRxReadyAirTimeDur_u durSifsPlusPhyRxReadyAirTimeDurReg;
	
	// Init value for all registers
	rxControlReg.val 					= 0;
	ccaIfsCompReg.val 					= 0;
	rxReadyIfsCompReg.val 				= 0;
	minimalRssiThresholdReg.val 		= 0;
	rxfFifoBaseAddrReg.val				= 0;
	rxfFifoDepthMinusOneReg.val			= 0;
	aMpduNoBaAgreementFilterEnReg.val	= 0;
	maxMpduLength11AgBReg.val			= 0;
	maxMpduLength11NReg.val				= 0;
	maxMpduLength11AcAx.val				= 0;
	mpduLengthFilterEn.val				= 0;
	fragmentedAMsduFilterEnReg.val		= 0;
	typeSubtypeFilter0Reg.val			= 0;
	typeSubtypeFilter1Reg.val			= 0;
	rxcErrIrqEnReg.val					= 0;
	regPacRxcErrorIfs1.val				= 0;
	regPacRxcErrorIfs2.val				= 0;
	regPacRxcErrorIfs3.val				= 0;
	regPacRxcErrorIfs4.val				= 0;
	regPacRxcErrorIfs5.val				= 0;
	durSifsPlusPhyRxReadyAirTimeDurReg.val = 0;

	// Set needed bits in control register. The rest will be 'FALSE'
    rxControlReg.bitFields.reservedStorageSize 					= FRAME_PREFIX_RESERVED_LENGTH_IN_WORDS;
	rxControlReg.bitFields.rxBufOutPtrRxDmaIncEn 				= TRUE;
	rxControlReg.bitFields.addr13MulticastBroadcastIndEn 		= TRUE;
	rxControlReg.bitFields.phyRxAbortRssiEn 					= 0;//RXC_ENABLE_PHY_RX_ABORT_RSSI;
	rxControlReg.bitFields.phyMacRxHaltOutPulseEn				= FALSE; // do not Wait for PHY to indicate init done after pause.
	rxControlReg.bitFields.autoRespFifoUpdateEn 				= TRUE;
	rxControlReg.bitFields.txcRxAirIdleIncludeInternalDelayEn	= TRUE;
	rxControlReg.bitFields.rxNavSettingEn						= TRUE;
	rxControlReg.bitFields.errorIfsSettingEn					= TRUE;
	rxControlReg.bitFields.rxNavEifsUnsupportedMcsEn			= TRUE;
	rxControlReg.bitFields.cfEndCmpAddr2ToVapAddrEn				= TRUE;
	rxControlReg.bitFields.bsrcFifoUpdateEn						= TRUE;
	rxControlReg.bitFields.mbfmFifoUpdateEn						= TRUE;
	rxControlReg.bitFields.bfRptMgmtActionNoAckEn				= TRUE;
#ifdef ENET_INC_ARCH_WAVE600B
    rxControlReg.bitFields.mbfmBfeeNdpFifoUpdateEn				= TRUE;
#endif //ENET_INC_ARCH_WAVE600B
#ifdef ENET_INC_ARCH_WAVE600D2
	rxControlReg.bitFields.baaFifoUpdateEn						= TRUE; // It's relevant to STA mode in HE TB STA mode, we need to support Rx of broadcast MBA, and to find our BACK section. This enable to pass the broadcast MBA from RxC to BAA for analyzing.options: TRUE - Enable; FALSE - Disable;According to Yossi enable should work for AP mode and well as STA mode
	rxControlReg.bitFields.sequencerCheckNdpFbResponseEn		= TRUE; //NFRP: if non of the stations responed go to CF-END
#endif //ENET_INC_ARCH_WAVE600D2
	RegAccess_Write(REG_PAC_RXC_RX_CONTROL,rxControlReg.val);

	// Rxc Compensations (Yossi Amar said it's ok to have same values as gen5b)
	ccaIfsCompenstation = Pac_GetHwDelays(PAC_DELAY_RXC_CCA);
	ccaIfsCompReg.bitFields.ccaIfsCompensation = System_ConvertNsToSystemClk(ccaIfsCompenstation) - RXC_HW_DELAY;
	RegAccess_Write(REG_PAC_RXC_CCA_IFS_COMPENSATION,ccaIfsCompReg.val);
	rxReadyIfsCompenstation = Pac_GetHwDelays(PAC_DELAY_RXC_RX_READY);
	rxReadyIfsCompReg.bitFields.rxrdyIfsCompensation = System_ConvertNsToSystemClk(rxReadyIfsCompenstation) - RXC_HW_DELAY;
	RegAccess_Write(REG_PAC_RXC_RXRDY_IFS_COMPENSATION,rxReadyIfsCompReg.val);
	
	// Set Minimal RSSI threshold value. Signed value. Used for setting PHY abort and NAV update.
	minimalRssiThresholdReg.bitFields.minimalRssiThreshold = RXC_MINIMAL_RSSI_THRESHOLD_VALUE;
	RegAccess_Write(REG_PAC_RXC_MINIMAL_RSSI_THRESHOLD, minimalRssiThresholdReg.val);

	// Reset RAM of MPDU TYPE/SUBTYPE reject filter
	memset32((void*)RXC_RXF_MPDU_TYPE_FILTER_RAM_START, 0, (RXC_RXF_MPDU_TYPE_FILTER_NUM_WORDS_PER_VAP * HW_NUM_OF_VAPS)); 

	//Configure the  RAM of MPDU TYPE/SUBTYPE reject filter
	//Frames with the type and subtype specified will be filtered if the filter status is enabled.
	for (vapIndex = 0 ; vapIndex < HW_NUM_OF_VAPS ; vapIndex++)
	{
		filterAddress = RXC_RXF_MPDU_TYPE_FILTER_RAM_START + (vapIndex * CONVERT_WORDS_TO_BYTES(RXC_RXF_MPDU_TYPE_FILTER_NUM_WORDS_PER_VAP));
		memcpy32((void*)filterAddress, RxcMpduTypeSubTypeRejectFilter, RXC_RXF_MPDU_TYPE_FILTER_NUM_WORDS_PER_VAP);
	}

	// configure global TYPE/SUBTYPE filter. This filter will not generate MPDU if matched and will not load NAV (!!). The RAM filter above will generate MPDU with error.
	// In fact - if both filters are configured with the same value - this global filter will win, and there won't be  MPDU descriptor and no NAV load.
	// Currently - we don't use the global filter since we want to load the NAV.
	typeSubtypeFilter0Reg.bitFields.type00SubtypeFilter = 0;
	typeSubtypeFilter0Reg.bitFields.type01SubtypeFilter = 0;
	RegAccess_Write(REG_PAC_RXC_TYPE_SUBTYPE_FILTER0, typeSubtypeFilter0Reg.val);
	typeSubtypeFilter1Reg.bitFields.type10SubtypeFilter = 0;
	typeSubtypeFilter1Reg.bitFields.type11SubtypeFilter = 0;
	RegAccess_Write(REG_PAC_RXC_TYPE_SUBTYPE_FILTER1, typeSubtypeFilter1Reg.val);


	// configure the "AMPDU WITH NO BA AGG" FIFO base address in SHRAM (different per band)
#if defined (ENET_INC_LMAC0)
	rxfFifoBaseAddrReg.bitFields.rxfFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&RxfAmpduNoBaAggrementFifo[0]);
#endif
#if defined (ENET_INC_LMAC1)
	rxfFifoBaseAddrReg.bitFields.rxfFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&RxfAmpduNoBaAggrementFifo_B1[0]);
#endif
	RegAccess_Write(REG_PAC_RXC_RXF_FIFO_BASE_ADDR, rxfFifoBaseAddrReg.val);	

	// configure the "AMPDU WITH NO BA AGG" FIFO depth
	rxfFifoDepthMinusOneReg.bitFields.rxfFifoDepthMinusOne = RXF_FIFO_AMPDU_NO_BA_AGGREMENT_DEPTH - 1;
	RegAccess_Write(REG_PAC_RXC_RXF_FIFO_DEPTH_MINUS_ONE, rxfFifoDepthMinusOneReg.val);	

	//Enable the "AMPDU WITH NO BA AGG" FIFO indications
	aMpduNoBaAgreementFilterEnReg.bitFields.aMpduNoBaAgreementFilterEn = RXF_FILTER_STATUS_ENABLE;
	RegAccess_Write(REG_PAC_RXC_A_MPDU_NO_BA_AGREEMENT_FILTER_EN, aMpduNoBaAgreementFilterEnReg.val);	

	// Set the value of MaxMpduLengthFilter of PHY modes A/B/G
	maxMpduLength11AgBReg.bitFields.maxMpduLength11Ag = RXC_MAX_MPDU_SIZE_PHY_MODE_11_A_G;
	maxMpduLength11AgBReg.bitFields.maxMpduLength11B = RXC_MAX_MPDU_SIZE_PHY_MODE_11_B;
	RegAccess_Write(REG_PAC_RXC_MAX_MPDU_LENGTH_11AG_B, maxMpduLength11AgBReg.val); 	

	// Set the value of MaxMpduLengthFilter of PHY mode N (In agg there are only 12 bits in the delimiter)
	maxMpduLength11NReg.bitFields.maxMpduLength11NAgg = RXC_MAX_MPDU_SIZE_PHY_MODE_11_A_G;
	maxMpduLength11NReg.bitFields.maxMpduLength11NNonAgg = RXC_MAX_MPDU_SIZE_PHY_MODE_11_N;
	RegAccess_Write(REG_PAC_RXC_MAX_MPDU_LENGTH_11N, maxMpduLength11NReg.val); 

	// Set the value of MaxMpduLengthFilter of PHY mode AC/AX
	maxMpduLength11AcAx.bitFields.maxMpduLength11Ac = RXC_MAX_MPDU_SIZE_PHY_MODE_11_AC;
	maxMpduLength11AcAx.bitFields.maxMpduLength11Ax = RXC_MAX_MPDU_SIZE_PHY_MODE_11_AX;
	RegAccess_Write(REG_PAC_RXC_MAX_MPDU_LENGTH_11AC_AX, maxMpduLength11AcAx.val); 	

	// Enable maxMpduLength filter for all phy modes
	mpduLengthFilterEn.bitFields.maxMpduLengthFilterEn = TRUE;
	mpduLengthFilterEn.bitFields.minMpduLengthFilterEn = TRUE;
	RegAccess_Write(REG_PAC_RXC_MPDU_LENGTH_FILTER_EN, mpduLengthFilterEn.val);	

	// Enable fragmented AMSDU filtering
	fragmentedAMsduFilterEnReg.bitFields.fragmentedAMsduFilterEn = TRUE;
	RegAccess_Write(REG_PAC_RXC_FRAGMENTED_A_MSDU_FILTER_EN, fragmentedAMsduFilterEnReg.val); 

	// Enable error IRQs (currently we enable all. We can remove some later if we choose to).
#ifndef ENET_INC_ARCH_WAVE600B
	rxcErrIrqEnReg.bitFields.fifoRxPmClrWhenNotEmptyIrqEn			 	= TRUE;
	rxcErrIrqEnReg.bitFields.rxReadyNegationNoMetricsErrIrqEn		 	= TRUE;
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingMetricsErrIrqEn	 	= TRUE;
	rxcErrIrqEnReg.bitFields.mpduDropDlmInEmptyCtrNotZeroEn 		 	= TRUE;
	rxcErrIrqEnReg.bitFields.phyMacFifoFullDropCtrNotZeroEn 		 	= TRUE;
	rxcErrIrqEnReg.bitFields.bufferOverflowCtrNotZeroEn 			 	= TRUE;
#endif //!ENET_INC_ARCH_WAVE600B
#ifdef ENET_INC_ARCH_WAVE600B
	//Temporary patch untill progmodel is merged from 6.1.0_PF6 to 6.1.0 main:
	rxcErrIrqEnReg.bitFields.fifoRxPmClrWhenNotEmptyIrqEn			 	= FALSE;
	rxcErrIrqEnReg.bitFields.rxReadyNegationNoMetricsErrIrqEn		 	= FALSE;
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingMetricsErrIrqEn	 	= FALSE;
	rxcErrIrqEnReg.bitFields.mpduDropDlmInEmptyCtrNotZeroEn 		 	= TRUE;
	rxcErrIrqEnReg.bitFields.phyMacFifoFullDropCtrNotZeroEn 		 	= FALSE;
	rxcErrIrqEnReg.bitFields.bufferOverflowCtrNotZeroEn 			 	= FALSE;

	rxcErrIrqEnReg.bitFields.rxfFifoFullDropIrqEn 					 	= TRUE;
	rxcErrIrqEnReg.bitFields.rxfFifoDecrementLessThanZeroIrqEn 		 	= TRUE; 
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingBfMetricsErrIrqEn 	= FALSE;
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingStatusErrIrqEn 	 	= FALSE;
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingStatusErr2IrqEn 	 	= FALSE;
	rxcErrIrqEnReg.bitFields.rxGloblNegationMissingMetricsErr2IrqEn  	= FALSE;
	rxcErrIrqEnReg.bitFields.rssiNotMyStaFifoFullDropIrqEn 				= TRUE;
	rxcErrIrqEnReg.bitFields.rssiNotMyStaFifoDecrementLessThanZeroIrqEn = TRUE;
#endif
	RegAccess_Write(REG_PAC_RXC_RXC_ERR_IRQ_EN, rxcErrIrqEnReg.val); 

	// Reset RAM of STA Frame Class
	memset32((void*)RXC_RXF_STA_FRAME_CLASS_RAM_START, 0, RXC_RXF_STA_FRAME_CLASS_RAM_SIZE_IN_WORDS); 


	/*	EIFS (EDCA) = aSIFSTime + AIFSN[AC] * aSlotTime + aSIFSTime + ACKTxTime = 2 * aSIFSTime + AIFSN[AC] * aSlotTime + ACKTxTime

							Sifs Time	Ack Time	Total	Reg Value
		=============================================================
		errorIfs11BShort	10			304 		314 	2512
		errorIfs11BLong		10			152 		162 	1296
		errorIfs11Ag		16			44			60		480
		errorIfs11N			16			44			60		480
		errorIfs11Ac		16			44			60		480

		One SIFS is counted is counted anyway after Rx 
		AIFSN[AC] * aSlotTime is counted by TXH 
		Reg Value = aSIFSTime  + ACKTxTime
		According to SAS  3.17.2.2.1 TXH-TXC configuration:
	*/
	regPacRxcErrorIfs1.bitFields.errorIfs11BShort = 2512;
	regPacRxcErrorIfs1.bitFields.errorIfs11BLong = 1296;
	RegAccess_Write(REG_PAC_RXC_ERROR_IFS1, regPacRxcErrorIfs1.val);

	regPacRxcErrorIfs2.bitFields.errorIfs11Ag = 480;
	regPacRxcErrorIfs2.bitFields.errorIfs11N = 480;
	RegAccess_Write(REG_PAC_RXC_ERROR_IFS2, regPacRxcErrorIfs2.val);

	regPacRxcErrorIfs3.bitFields.errorIfs11Ac = 480;
	RegAccess_Write(REG_PAC_RXC_ERROR_IFS3, regPacRxcErrorIfs3.val);

	regPacRxcErrorIfs4.bitFields.errorIfs11AxSu = 480;
	regPacRxcErrorIfs4.bitFields.errorIfs11AxSuEx = 480;
	RegAccess_Write(REG_PAC_RXC_ERROR_IFS4, regPacRxcErrorIfs4.val);

	regPacRxcErrorIfs5.bitFields.errorIfs11AxMuDl = 480;
	regPacRxcErrorIfs5.bitFields.errorIfs11AxMuTb = 480;
	RegAccess_Write(REG_PAC_RXC_ERROR_IFS5, regPacRxcErrorIfs5.val);

	// 11ag: SIFS = 16uSec	Phy RX Delay = (Legacy) 26uSec	 Total = 42uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11Ag	   = RXC_SIFS_PLUS_PHY_RX_DELAY_11AG; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11a/g/n/ac [microseconds].
	// 11b short preamble: SIFS = 10uSec  Phy RX Delay = (11b short) 98uSec   Total = 108uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11BShort = RXC_SIFS_PLUS_PHY_RX_DELAY_11B_SHORT_PREAMBLE; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11b short preamble [microseconds]
	// 11b long preamble: SIFS = 10uSec  Phy RX Delay = (11b long) 98*2 uSec   Total = 206uSec 
	durSifsPlusPhyRxReadyAirTimeDurReg.bitFields.sifsPlusPhyRxDelayAirTime11BLong  = RXC_SIFS_PLUS_PHY_RX_DELAY_11B_LONG_PREAMBLE; // SIFS + PHY RX Delay when the incoming Rx frame is PHY mode 11b long preamble [microseconds]
	RegAccess_Write(REG_PAC_RXC_SIFS_PLUS_PHY_RX_READY_AIR_TIME_DUR, durSifsPlusPhyRxReadyAirTimeDurReg.val);

}
#endif /* #ifdef ENET_INC_UMAC */


