/***********************************************************************************
 File:		HwErros.c
 Module:		hw Events & Errors
 Purpose:		Handlers for all Hw Errors Events
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Utils_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "ProcessManager_Api.h"
#include "ServicesHandler_Api.h"
#include "ShramStatistics.h"
#include "stringLibApi.h"
#include "BaAnalyzerRegs.h"
#include "Pac_Api.h"
#include "PacManager_api.h"
#include "loggerAPI.h"
#include "BaAnalyzerRegs.h"
#include "TxSequencerRegs.h"
#include "MacWepRegs.h"
#include "TxSelectorRegs.h"
#include "MacHtExtensionsRegs.h"
#include "PacRxcRegs.h"
#include "fast_mem_psd2mips.h"
#include "WlanArmDmaRegs.h"

/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 4

#if defined(ENET_INC_LMAC) 
#define HW_ERRORS_ENABLE_MASK	((1 << HW_ERROR_RXC_RX_BUFFER_OVERFLOW)			| \
								(1 << HW_ERROR_BLOCK_ACK)						| \
								(1 << HW_ERROR_NPU_ARB_BUS)						| \
								(1 << HW_ERROR_SEQUENCER_ERROR)					| \
								(1 << HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED)		| \
								(1 << HW_ERROR_DELIA_AUTO_FILL_COUNTER_LIMIT)	| \
								(1 << HW_ERROR_DELIA_AUTO_FILL_END)				| \
								(1 << HW_ERROR_RX_EOF_ERROR)					| \
								(1 << HW_ERROR_RX_AGGR_ERROR)					| \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)				| \
								(1 << HW_ERROR_ARM_DMA_ABORT)					| \
								(1 << HW_ERROR_TX_SELECTOR)						| \
								(1 << HW_ERROR_BA_ANALYZER)						| \
								(1 << HW_ERROR_WLAN_MAC_XBAR_DMA)				| \
								(1 << HW_ERROR_RX_SECURITY_FAIL)				| \
								(1 << HW_ERROR_RX_CLASSIFIER)					| \
								(1 << HW_ERROR_RXC_RX_FIFO_OVERFLOW)			| \
								(1 << HW_ERROR_DELINEATOR_FIFO_FULL))
#endif /*ENET_INC_LMAC */
#if defined(ENET_INC_UMAC) 
#define HW_ERRORS_ENABLE_MASK	( (1 << HW_ERROR_Q_MANAGER_UM_ERROR))
#endif /*#if defined(ENET_INC_UMAC) */

/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef void (*HwErrorHandler_t)(void);

typedef struct DebugHwCounters
{
	uint32 rxBufferOverflowDroppedMpdus;
	uint32 txSelectorMuPrimaryNotValid;
	uint32 txSelectorMuNoPrimaryInGroup;
} DebugHwCounters_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void hwErrorsDefaultHandler(void);
void hwErrorsRxcRxBufferOverflow(void);
void hwErrorsPacTx(void);
void hwErrorsBlockAck(void);
void hwErrorsNpuArbBus(void);
void hwErrorsSequencerErrorHandling(void);
void hwErrorsSecurityFail(void);
void hwErrorsDeliaLimitTimerExpired(void);
void hwErrorsDeliaAutoFillCounterLimit(void);
void hwErrorsDeliaAutoFillEnd(void);
void hwErrorsRxEof(void);
void hwErrorsRxAggr(void);
void hwErrorsQmanager(void);
void hwErrorsArmDmaAbort(void);
void hwErrorsTxSelector(void);
void hwErrorsBaAccelerator(void);
void hwErrorsWlanMacXbarDma(void);
void hwErrorsRxSecurity(void);
void hwErrorsRxClassifier(void);
void hwErrorsRxFifoOverflow(void);
void hwErrorsBfReportAutoReplyAbort(void);
void hwErrorsDelineatorFifoOverflow(void);
#if defined (ENET_INC_LMAC)
void hwErrorsReadHwRegs(void);
#endif
/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
static const HwErrorHandler_t HwErrorHandlers[HW_ERROR_TOTAL_NUM] = 
{
	hwErrorsRxcRxBufferOverflow, 			/* HW_ERROR_RXC_RX_BUFFER_OVERFLOW */
	hwErrorsPacTx,							/* HW_ERROR_PAC_TX */
	hwErrorsBlockAck,						/* HW_ERROR_BLOCK_ACK */ 
	hwErrorsNpuArbBus,						/* HW_ERROR_NPU_ARB_BUS */
	hwErrorsSequencerErrorHandling,			/* HW_ERROR_SEQUENCER_ERROR */
	hwErrorsSecurityFail,					/* HW_ERROR_SECURITY_FAIL */
	hwErrorsDeliaLimitTimerExpired,			/* HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED */
	hwErrorsDeliaAutoFillCounterLimit,		/* HW_ERROR_DELIA_AUTO_FILL_COUNTER_LIMIT */ 
	hwErrorsDeliaAutoFillEnd,				/* HW_ERROR_DELIA_AUTO_FILL_END */ 
	hwErrorsRxEof,							/* HW_ERROR_RX_EOF_ERROR */ 
	hwErrorsRxAggr,							/* HW_ERROR_RX_AGGR_ERROR */
	hwErrorsQmanager,						/* HW_ERROR_Q_MANAGER_UM_ERROR */
	hwErrorsQmanager,						/* HW_ERROR_Q_MANAGER_LM_ERROR */
	hwErrorsArmDmaAbort,					/* HW_ERROR_ARM_DMA_ABORT */
	hwErrorsTxSelector,						/* HW_ERROR_TX_SELECTOR */
	hwErrorsBaAccelerator,					/* HW_ERROR_BA_ANALYZER */ 
	hwErrorsWlanMacXbarDma,					/* HW_ERROR_WLAN_MAC_XBAR_DMA */
	hwErrorsRxSecurity,						/* HW_ERROR_RX_SECURITY_FAIL */
	hwErrorsRxClassifier,					/* HW_ERROR_RX_CLASSIFIER */ 
	hwErrorsRxFifoOverflow,					/* HW_ERROR_RXC_RX_FIFO_OVERFLOW */
	hwErrorsBfReportAutoReplyAbort,			/* HW_ERROR_BF_REPORT_AUTO_REPLY_ABORT */
	hwErrorsDelineatorFifoOverflow			/* HW_ERROR_DELINEATOR_FIFO_FULL */
};

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/
static uint32 HwErrorsCounters[HW_ERROR_TOTAL_NUM];

#if defined (ENET_INC_LMAC)
static DebugHwCounters_t debugHwCounters;
uint32 WlMacIrq;
uint32 RXstat[8];
uint32 DelineatorState; 
uint32 RxCState; 
uint32 RxAggr;
uint32 DmaAbort0;
uint32 DmaAbort1;
uint32 DmaAbort2;
uint32 DmaAbort3;
uint32 delAggErrCnt;
#endif //ENET_INC_LMAC

RegMacWepTxInterruptsStatus_u	WepTxInterruptsStatus;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

hwErrorsDefaultHandler 

Description:
------------
	Default Handler for Error Events in System, casues ASSERT in FW

**********************************************************************************/
void hwErrorsDefaultHandler(void)
{
#if defined(ENET_INC_UMAC) 
	RegAccess_Write(0xA7140068, 0);
#endif
	ASSERT(0);
}

/**********************************************************************************

hwErrorsRxcRxBufferOverflow

Description:
------------
	

**********************************************************************************/
void hwErrorsRxcRxBufferOverflow(void)
{
 #if defined (ENET_INC_LMAC)
	//read RxC buffer overFlow counter, increment internal counter and clear it in RXC
	debugHwCounters.rxBufferOverflowDroppedMpdus += Pac_RxcReadRxBufferOverflowCounter();
	ILOG0_D("hwErrorsRxcRxBufferOverflow rxBufferOverflowDroppedMpdus = %d",debugHwCounters.rxBufferOverflowDroppedMpdus);
	Pac_RxcClearRxBufferOverflowCounter();
	//unmask event and clear indication
	HwErrors_UnMaskEvent(HW_ERROR_RXC_RX_BUFFER_OVERFLOW);
	Pac_RxcClearRxBufferOverflow();	
	/*Update General Statistics*/
	CyclicBufferOverflowCounter = debugHwCounters.rxBufferOverflowDroppedMpdus;
#endif //ENET_INC_LMAC
}

/**********************************************************************************

hwErrorsPacTx

Description:
------------
	
**********************************************************************************/
void hwErrorsPacTx(void)
{
	/*Assert Was replaced with counter because this error can happen */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_TX_ERROR);
	HwErrors_UnMaskEvent(HW_ERROR_PAC_TX);
}

/**********************************************************************************

hwErrorsBlockAck 

Description:
------------
	Error Handler for the BA geneartor error event	

**********************************************************************************/
void hwErrorsBlockAck(void)
{
	/*Assert Was removed becasue this error can happen (i.e reconnection) */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_BLOCK_ACK_ERROR);
	HwErrors_UnMaskEvent(HW_ERROR_BLOCK_ACK);
}

/**********************************************************************************

hwErrorsNpuArbBus



Description:
------------
	

**********************************************************************************/
void hwErrorsNpuArbBus(void)
{
	ASSERT(0);
}

 /**********************************************************************************

hwErrorsSecurityFail

Description:
------------
	

**********************************************************************************/
void hwErrorsSecurityFail(void)
{
	// Get the error status
	WepTxInterruptsStatus.val = 0;
	RegAccess_Read(REG_MAC_WEP_TX_INTERRUPTS_STATUS, &WepTxInterruptsStatus.val);
	ASSERT(0);
}

/**********************************************************************************

hwErrorsDeliaLimitTimerExpired

Description:
------------

	
**********************************************************************************/
void hwErrorsDeliaLimitTimerExpired(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsDeliaAutoFillCounterLimit

Description:
------------
	
**********************************************************************************/
void hwErrorsDeliaAutoFillCounterLimit(void)
{
	/*Assert Was replaced with counter because this error can happen */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_AUTO_FILL_COUNTER_LIMIT);
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_DEL_1_AUTO_FILL_CTRLIMITIRQ);
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_DEL_2_AUTO_FILL_CTRLIMITIRQ);
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_DEL_3_AUTO_FILL_CTRLIMITIRQ);
	HwErrors_UnMaskEvent(HW_ERROR_DELIA_AUTO_FILL_COUNTER_LIMIT);
}

/**********************************************************************************

hwErrorsDeliaAutoFillEnd

Description:
------------
	
**********************************************************************************/
void hwErrorsDeliaAutoFillEnd(void)
{
	/*Assert Was replaced with counter because this error can happen */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_AUTO_FILL_END);
	HwErrors_UnMaskEvent(HW_ERROR_DELIA_AUTO_FILL_END);
}

/**********************************************************************************

hwErrorsRxEof

Description:
------------
	There are 3 known scenarios that  cause this EOF Error to be asserted:
	
		1)	Number of metrics information does not fit the protocol - we expect two 32-bit words to be delivered to RxC per reception 
			There is a bug in HW  in the  following scenarios:
			-	VHT single frame with bad delimiter
			-	HT frame with bad delimiters (not a single delimiter is found)
			In those cases, the delineator wrongly sets one data strobe (32bit) and two metrics strobes (2x32bit) causes the num of events 
			counted to be 4:
				 - Two from first packet - the bug, and 
				-  Two from the following packet - which causes the error
			RxC completely ignores those strobes therfoe we shoudn't assert the system
		2) After PHY RX error, the aggregate indication from prev. frame is not cleared on the next one (the one that fails). 	
		   This is an HW Bug which full requires recovery sequence 
		3) Number of data words do not fit the indication given at the beginning of a session 
		    for example:when Rx ready is prematurely negated - this can be emulated by resetting the delineator / phy RxBE during packet reception.

**********************************************************************************/
void hwErrorsRxEof(void)
{
#if defined (ENET_INC_LMAC)
	hwErrorsReadHwRegs();
	FATAL("hwErrorsRxEof");
#endif //ENET_INC_LMAC
}

/**********************************************************************************

hwErrorsRxAggr

Description:
------------
	There are 3 known scenarios that  cause this EOF Error to be asserted:
		1) Delineator  get out of sync (For exampel disable RXC without Halt (Gen5))
		2) Hw bug which casues false error in the  following scenario:
			In case the last subframe in aggregation is very short (was found  when the last frame in aggregation was Action NoAck frame), 
			and RxBE de-asserts Rx ready signal right after last data
			(1 cycle after the "ready" signal", very rare ,it would happen only in STBC mode, 80MHz, Viterbi) - 

**********************************************************************************/
void hwErrorsRxAggr(void)
{

#if defined(ENET_INC_LMAC) 
	hwErrorsReadHwRegs();
	FATAL("hwErrorsRxAggr");
#endif //ENET_INC_LMAC
}

/**********************************************************************************

hwErrorsQmanager

Description:
------------
	

**********************************************************************************/
void hwErrorsQmanager(void)
{
	HwQManager_ErrorEvent();
}

/**********************************************************************************

hwErrorsArmDmaAbort

Description:
------------
	

**********************************************************************************/
void hwErrorsArmDmaAbort(void)
{
#if defined(ENET_INC_LMAC) 
	hwErrorsReadHwRegs();
	ASSERT(0);
#endif
}

#if defined (ENET_INC_LMAC)
/**********************************************************************************

hwErrorsReadHwRegs


Description:
------------
	

**********************************************************************************/
void hwErrorsReadHwRegs(void)
{
	uint8 rxstatIndex = 0;

	RegAccess_Read(REG_WLAN_ARM_DMA_FTR0, &DmaAbort0);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR1, &DmaAbort1);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR2, &DmaAbort2);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR3, &DmaAbort3);
	RegAccess_Read(MEM_BE_ERR_COUNTER, &RxAggr);
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ, &WlMacIrq);
	for (rxstatIndex = 0; rxstatIndex < 8 ; rxstatIndex++)
	{
		RegAccess_Read((REG_MAC_HT_EXTENSIONS_RX_STAT0_LO + (4*rxstatIndex)), &RXstat[rxstatIndex]);
	}
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_DEL_DEBUG2, &DelineatorState);
	RegAccess_Read(REG_PAC_RXC_RXC_STATE, &RxCState);
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_DELINEATOR_AGG_ERR_CNT, &delAggErrCnt);
}
#endif



/**********************************************************************************

hwErrorsTxSelector


Description:
------------
	

**********************************************************************************/
void hwErrorsTxSelector(void)
{
#if defined (ENET_INC_LMAC)
	RegTxSelectorTxSelMuErr_u txSelectorMuError;
	RegTxSelectorTxSelErr_u txSelectorSuError;
	RegTxSelectorTxSelMuErrClr_u txSelectorMuErrClr;

	txSelectorMuErrClr.val = 0;
	
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_MU_ERR, &txSelectorMuError.val);
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_ERR, &txSelectorSuError.val);
	
	if((txSelectorSuError.val != 0) || (txSelectorMuError.bitFields.primaryGrpThresErr == TRUE))
	{
		FATAL("hwErrorsTxSelector");
	}
	else
	{
	 	/* The following two errors can't cause FATAL because in case of LOCKED group
	 	   Selector still checks these errors which is incorrect behavior */
		if (txSelectorMuError.bitFields.primaryNotValidErr)
		{
			debugHwCounters.txSelectorMuPrimaryNotValid++;
			txSelectorMuErrClr.bitFields.primaryNotValidErrClr = 1;
		}
		else
		{
			debugHwCounters.txSelectorMuNoPrimaryInGroup++;
			txSelectorMuErrClr.bitFields.noPrimaryInGrpErrClr = 1;
		}
		RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_ERR_CLR, txSelectorMuErrClr.val);
		HwErrors_UnMaskEvent(HW_ERROR_TX_SELECTOR);
	}
#endif //ENET_INC_LMAC
}

/**********************************************************************************

hwErrorsBaAccelerator


Description:
------------
	

**********************************************************************************/
void hwErrorsBaAccelerator(void)
{
	ASSERT(0);
}

/**********************************************************************************

isr_linkAdaptation_SequencerErrorHandling 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void hwErrorsSequencerErrorHandling(void)
{
	ASSERT(0);
}


/**********************************************************************************

hwErrorsWlanMacXbarDma

Description:
------------
	

**********************************************************************************/
void hwErrorsWlanMacXbarDma(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsRxSecurity

Description:
------------
	

**********************************************************************************/
void hwErrorsRxSecurity(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsRxClassifier

Description:
------------
	

**********************************************************************************/
void hwErrorsRxClassifier(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsRxFifoOverflow

Description:
------------
	
**********************************************************************************/
void hwErrorsRxFifoOverflow(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsDelineatorFifoOverflow


Description:
------------
	
**********************************************************************************/
void hwErrorsBfReportAutoReplyAbort(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsDelineatorFifoOverflow


Description:
------------
	
	For VHT, the standard defines that the 1st delimiter should indicate EOF - which is then parsed by the delineator "VHT single frame". 
	After that, we should not get another MPDU. An error test case where two VHT single packets were injected in a single A-MPDUs 
	caused delineator FIFO overflow .(we do not expect this to happen - since this is a synthetic case) 
	
**********************************************************************************/
void hwErrorsDelineatorFifoOverflow(void)
{
#if defined(ENET_INC_LMAC) 
	FATAL("hwErrorsDelineatorFifoOverflow");
#endif //ENET_INC_LMAC	
}
	 	 
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

HwEvents_MaskEvent 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be masked
Output:
-------
	None

**********************************************************************************/
void HwErrors_MaskEvent(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrorsIntMaskedStatus_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = 0;
	maskVal = (1 << hwError);
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

/**********************************************************************************

HwErrors_UnMaskEvent 

Description:
------------
	unmask Specific Hw event interrupt
Input:
-----
	hwEvent - The Hw Event to be unmasked
Output:
-------
	None

**********************************************************************************/
void HwErrors_UnMaskEvent(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrorsIntMaskedStatus_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = (1 << hwError);
	maskVal = (1 << hwError);
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

/**********************************************************************************

isr_HwErrors 

Description:
------------
	General isr for all error events in system
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwErrors(void)
{
	
	RegEmeraldEnvHwErrorsIntStatus_u hwErrorRegister;
	uint32 errorIndex;
	
	hwErrorRegister.val = 0;
	
	RegAccess_Read(REG_EMERALD_ENV_HW_ERRORS_INT_MASKED_STATUS, &hwErrorRegister.val);

	while(hwErrorRegister.val != 0)
	{

		errorIndex = Utils_FindFirstSetAndClear(&hwErrorRegister.val);
		ASSERT(errorIndex < HW_ERROR_TOTAL_NUM);
		//("isr_HwErrors errorIndex %d ",errorIndex);

		HwErrors_MaskEvent((HwErrorId_e)errorIndex);
			
		HwErrorsCounters[errorIndex]++;

		ILOG0_DD("isr_HwErrors errorIndex %d num of errors %d",errorIndex,HwErrorsCounters[errorIndex]);

		HwErrorHandlers[errorIndex]();
	}

}

/**********************************************************************************

HwErrors_Init 

Description:
------------
	Enabling all required Hw Error Events to generated to FW
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif
void HwErrors_Init(void)
{
	memset32(&HwErrorsCounters,0,HW_ERROR_TOTAL_NUM);

#if defined (ENET_INC_LMAC)
	memset(&debugHwCounters,0, sizeof(DebugHwCounters_t));
#endif //ENET_INC_LMAC
	
	RegAccess_Write(REG_EMERALD_ENV_HW_ERRORS_INT_EN, HW_ERRORS_ENABLE_MASK);
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif

