/***********************************************************************************
 File:		HwEvents.c
 Module:		Hw Events And Errors
 Purpose:		Handle all Hw Events
 Description:	This module is the API to the Rx Classifier which is responbile for frame filtering and classification 
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "OSAL_Tasks.h"
#include "OSAL_UpperMacMessages.h"
#include "Utils_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "HostInterface_API.h"
#include "InterruptManager_Api.h"
#include "EventsManager_api.h"
#include "loggerAPI.h"
#include "CpuLoad_Api.h"

#ifdef ENET_INC_UMAC 
#include "Pac_Api.h"
#include "PhyDriver_API.h"
#include "PacketTrafficArbitrator_api.h" 
#endif /* #ifdef ENET_INC_UMAC */

#include "Pauser_Api.h"
#include "HwQManager_API.h"
#include "TxSelector_Api.h"


/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 5

#if defined(ENET_INC_LMAC) 
#define HW_EVENTS_ENABLE_MASK	0  
#endif /*ENET_INC_LMAC */
#if defined(ENET_INC_UMAC) 

#define HW_EVENTS_ENABLE_MASK	((1 << HW_EVENT_RX_CLASSIFIER_FRAME_CLASS_VIOLATION) 	| \
								(1 << HW_EVENT_RX_CLASSIFIER_REKEY) 					| \
								(1 << HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY) 				| \
								(1 << HW_EVENT_PD_THRESHOLD_REACHED) 					| \
								(1 << HW_EVENT_RXPP_FRAGMENT_FIFO_NOT_EMPTY) 			| \
								(1 << HW_EVENT_RAB_NPU_TO_UPI_IRQ) 						| \
								(1 << HW_EVENT_RXF_FIFO_NOT_EMPTY) 						| \
								(1 << HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY) 				| \
								(1 << HW_EVENT_MU_LOCKER_DONE_INDICATION) 				| \
								(1 << HW_EVENT_RX_HALT_INDICATION) 						| \
								(1 << HW_EVENT_AGER_SINGLE_CHECK_DONE)					| \
								(1 << HW_EVENT_EXTERNAL_IRQ))

#endif /*#if defined(ENET_INC_UMAC) */

/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef void (*HwEventHandler_t)(void);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void hwEventsDefaultHandler(void);
static void hwEventsRxClassifierFrameClassViolation(void);
static void hwEventsRxPpFragmentFifoNotEmpty(void);
static void hwEventsRxClassifierRxhAvailablePendingRtd(void);
static void hwEventsRxClassifierRekey(void);
static void hwEventsPsSettingFifoNotEmpty(void);
static void hwEventsPdThresholdReached(void);
static void hwEventsRxfFifoNotEmpty(void);
static void hwEventsHwQueueManagerLowTxReady(void);
static void hwEventsPhy(void);
static void hwEventsPtaGpio0(void);
#ifndef ATOMIC_COUNTER_ENABLER
static void hwEventsRabNpuToUpiIrq(void);
#endif
static void hwEventsRxHaltIndication(void);
static void hwEventsAgerSingleCheckDone(void);
static void hwEventsTxSelectorGroupLockerReqServiced(void);


/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
const HwEventHandler_t HwEventsHandlers[HW_EVENT_TOTAL_NUM] = 
{
	hwEventsRxPpFragmentFifoNotEmpty, 			/* HW_EVENT_RXPP_FRAGMENT_FIFO_NOT_EMPTY */
	hwEventsRxClassifierFrameClassViolation,	/* HW_EVENT_RX_CLASSIFIER_FRAME_CLASS_VIOLATION */
	hwEventsRxClassifierRekey,					/* HW_EVENT_RX_CLASSIFIER_REKEY */
	hwEventsPsSettingFifoNotEmpty,				/* HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY */ 
	hwEventsPdThresholdReached,					/* HW_EVENT_PD_THRESHOLD_REACHED */
	hwEventsDefaultHandler, 					/* HW_EVENT_RD_THRESHOLD_REACHED */
	hwEventsPhy, 								/* HW_EVENT_PHY */ 
	hwEventsPtaGpio0,							/* HW_EVENT_EXTERNAL_IRQ */
#ifdef ATOMIC_COUNTER_ENABLER
	hwEventsDefaultHandler,			 			/* HW_EVENT_RAB_NPU_TO_UPI_IRQ */
#else
	hwEventsRabNpuToUpiIrq, 					/* HW_EVENT_RAB_NPU_TO_UPI_IRQ */
#endif	
	hwEventsDefaultHandler,						/* HW_EVENT_UART */
	hwEventsRxfFifoNotEmpty, 					/* HW_EVENT_RXF_FIFO_NOT_EMPTY */
	hwEventsHwQueueManagerLowTxReady, 			/* HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY */
	hwEventsDefaultHandler, 					/* HW_EVENT_Q_MANAGER_LOW_PRI_RX_READY */
	hwEventsDefaultHandler, 					/* HW_EVENT_Q_MANAGER_DESCRIPTOR_POOL */
	hwEventsDefaultHandler, 					/* HW_EVENT_MIPS_INTERNAL_IRQ */ 
	hwEventsDefaultHandler, 					/* HW_EVENT_TX_SELECTOR */ 
	hwEventsDefaultHandler, 					/* HW_EVENT_TXH_NTD_SW_REQ */
	hwEventsDefaultHandler, 					/* HW_EVENT_RESERVED_1 */
	hwEventsDefaultHandler, 					/* HW_EVENT_RX_SECURITY_COMPLETE */ 
	hwEventsDefaultHandler, 					/* HW_EVENT_RX_SECURITY_DESC_DONE_ACC_NOT_EMPTY */
	hwEventsDefaultHandler, 					/* HW_EVENT_RX_SECURITY_DESC_MON_ADDR_CMP_RESULT */
	hwEventsRxClassifierRxhAvailablePendingRtd, /* HW_EVENT_RX_CLASSIFIER_RXH_AVAILABLE_PENDING_RTD*/
	hwEventsAgerSingleCheckDone,				/* HW_EVENT_AGER_SINGLE_CHECK_DONE */
	hwEventsRxHaltIndication,					/* HW_EVENT_RX_HALT_INDICATION */
	hwEventsDefaultHandler,						/* HW_EVENT_TX_DMA_MON_ACC_NON_EMPTY_IND */
	hwEventsDefaultHandler,						/* HW_EVENT_DMA_WRRAPER_CH1_DONE_ACC_NOT_EMPTY */
	hwEventsDefaultHandler, 					/* HW_EVENT_TX_SECURITY_COMPLETE */
	hwEventsDefaultHandler,						/* HW_EVENT_MAC_PHY_GP_IF_INDICATION */
	hwEventsTxSelectorGroupLockerReqServiced,	/* HW_EVENT_MU_LOCKER_DONE_INDICATION */
	hwEventsDefaultHandler,						/* HW_EVENT_RESERVED_6 */
	hwEventsDefaultHandler,						/* HW_EVENT_RESERVED_7 */
	hwEventsDefaultHandler,						/* HW_EVENT_RESERVED_8 */
};

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/
uint32 HwEventsStat[HW_EVENT_TOTAL_NUM];

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

hwEventsDefaultHandler 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsDefaultHandler(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwEventsRxClassifierRxhAvailablePendingRtd 

Description:
------------
	RX Classifier is out ouf resources
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRxClassifierRxhAvailablePendingRtd(void)
{
#if defined(ENET_INC_UMAC) 
	EventManager_TurnOffEvent(EVENT_ID_RX_CLASSIFIER_RXH_AVAILABLE_PENDING_RTD);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CLASSIFIER_RESOURCE_TIMEOUT, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);	
#endif
}

/**********************************************************************************

hwEventsRxClassifierFrameClassViolation 

Description:
------------
	Frame Class violation event from Rx Classifier
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRxClassifierFrameClassViolation(void)
{
#if defined(ENET_INC_UMAC) 
	// turn off the event
	EventManager_TurnOffEvent(EVENT_ID_CLASS_VIOLATION);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CLASS_VIOLATION_EVENT, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);	
#endif
}

/**********************************************************************************

hwEventsRxClassifierFrameClassViolation 

Description:
------------
	Frame Class violation event from Rx Classifier
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRxPpFragmentFifoNotEmpty(void)
{
#if defined(ENET_INC_UMAC) 

	// turn off the event
	EventManager_TurnOffEvent(EVENT_ID_FRAGMENT_FIFO_NOT_EMPTY);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_FRAGMENTATION_EVENT, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);	
#endif
}


/**********************************************************************************

hwEventsRxClassifierRekey 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRxClassifierRekey(void)
{
	FATAL("hwEventsRxClassifierRekey");
}


/**********************************************************************************

hwEventsPsSettingFifoNotEmpty 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsPsSettingFifoNotEmpty(void)
{
#if defined(ENET_INC_UMAC) 
	// turn off the event
	EventManager_TurnOffEvent(EVENT_ID_PS_SETTING_FIFO_NOT_EMPTY);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_PS_SETTINGS_FIFO_NOT_EMPTY, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);	
#endif	
}

/**********************************************************************************

hwEventsRxfFifoNotEmpty 

Description:
------------
Handles the event that the Rxf FIFO is not empty

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRxfFifoNotEmpty(void)
{	
#ifdef ENET_INC_UMAC
	// turn off the event
	EventManager_TurnOffEvent(EVENT_ID_RXF_FIFO_NOT_EMPTY);
	OSAL_SEND_NO_DATA_MESSAGE(TS_MANAGER_ILLEGAL_PACKET, TASK_TS_MANAGER, VAP_ID_DO_NOT_CARE);	
#endif /* ENET_INC_UMAC */	
}

/**********************************************************************************

hwEventsHwQueueManagerLowTxReady 

Description:
------------
Handles the event that the one of the Tx low priority ready list became not empty

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsHwQueueManagerLowTxReady(void)
{	
#ifdef ENET_INC_UMAC 
	K_MSG *qosMessage = NULL;
	EventManager_TurnOffEvent(EVENT_ID_TX_LIST_READY_PDS_LOW_PRI_NOT_EMPTY);

	qosMessage = OSAL_GET_MESSAGE(K_NO_DATA);	
	OSAL_SEND_MESSAGE(QOS_AGER_LIST_NOT_EMPTY, TASK_QOS, qosMessage, VAP_ID_DO_NOT_CARE);	
#endif /* ENET_INC_UMAC */	
}

/**********************************************************************************

hwEventsPdThresholdReached 

Description:
------------
Handles the event that the number of free PDs has reached to the threshold

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsPdThresholdReached(void)
{	
#ifdef ENET_INC_UMAC 
	K_MSG *qosMessage = NULL;

	EventManager_TurnOffEvent(EVENT_ID_PD_THRESHOLD_REACHED);
	qosMessage = OSAL_GET_MESSAGE(K_NO_DATA);	
	OSAL_SEND_MESSAGE(QOS_NUMBER_OF_FREE_PDS_REACHED_THRESHOLD, TASK_QOS, qosMessage, VAP_ID_DO_NOT_CARE);	
#endif /* ENET_INC_UMAC */	
}

/**********************************************************************************

hwEventsPhy 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsPhy(void)
{
#ifdef ENET_INC_UMAC
    /* No need to turn off event since it is already interrupt context */
	isr_PhyDrv_PhyInterrupt();
#endif /* #ifdef ENET_INC_UMAC */
}

/**********************************************************************************

hwEventsPtaGpio0 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsPtaGpio0(void)
{
#ifdef ENET_INC_UMAC
#ifdef PTA_BUILD_IN_PLAT
	Isr_PtaGpio0();
#endif
#endif 
}


#ifndef ATOMIC_COUNTER_ENABLER
/**********************************************************************************

hwEventsRabNpuToUpiIrq 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsRabNpuToUpiIrq(void)
{
#if defined(ENET_INC_UMAC) 
	vHIM_HostISR();
#endif
}

#endif // ATOMIC_COUNTER_ENABLER
/**********************************************************************************

hwEventsRxHaltIndication

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void hwEventsRxHaltIndication(void)
{
#if defined(ENET_INC_UMAC)
	// this event is in UM for gen5b and in LM for gen6
	isr_TxPauser_RxPauseExecuted();
	HwEvents_UnMaskEvent(HW_EVENT_RX_HALT_INDICATION);
#endif
}

/**********************************************************************************

hwEventsAgerSingleCheckDone 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsAgerSingleCheckDone(void)
{
#ifdef ENET_INC_UMAC
    HwQManagerAger_ClearSingleCheckDoneInterrupt();
	OSAL_SEND_NO_DATA_MESSAGE(QOS_SINGLE_CHECK_DONE, TASK_QOS, VAP_ID_DO_NOT_CARE);
#endif /* #ifdef ENET_INC_UMAC */
}

/**********************************************************************************

hwEventsTxSelectorGroupLockerReqServiced 

Description:
------------
Called after group lock done 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsTxSelectorGroupLockerReqServiced(void)
{
#if defined(ENET_INC_UMAC) 
	isr_TxSelector_LockerGroupRequestServiced();
	HwEvents_UnMaskEvent(HW_EVENT_MU_LOCKER_DONE_INDICATION);
#endif
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

HwEvents_MaskEvent 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent(HwEventId_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEventsIntMaskedStatus_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

isr_HwEvents 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent(HwEventId_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEventsIntMaskedStatus_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

isr_HwEvents 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents(void)
{
	RegEmeraldEnvHwEventsIntStatus_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS_INT_MASKED_STATUS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat[eventIndex]++;	
		HwEventsHandlers[eventIndex]();
	}
}

/**********************************************************************************

HwEvents_Init 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif
void HwEvents_Init(void)
{
	memset32(&HwEventsStat[0],0,HW_EVENT_TOTAL_NUM);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS_INT_EN, HW_EVENTS_ENABLE_MASK);
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif

