/***********************************************************************************
 File:		HwErros.c
 Module:		hw Events & Errors
 Purpose:		Handlers for all Hw Errors Events
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "Utils_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "ProcessManager_Api.h"
#include "ServicesHandler_Api.h"
#include "stringLibApi.h"
#include "BaAnalyzerRegs.h"
#include "BaAnalyzer_Api.h"
#include "Pac_Api.h"
#include "PacManager_api.h"
#include "loggerAPI.h"
#include "TxSequencerRegs.h"
#include "MacWepRegs.h"
#include "TxSelectorRegs.h"
#include "OtfAggregator_Api.h"
#include "MacHtExtensionsRegs.h"
#include "fast_mem_psd2mips.h"
#include "WlanArmDma0Regs.h"
#include "WlanArmDma1Regs.h"
#include "UlPostRx_Api.h"
#include "ShramStatistics.h"
#ifdef ENET_INC_ARCH_WAVE600B
#include "TxPdAccRegs.h"
#include "MacTxHcRegs.h"
#include "TfGeneratorRegs.h"
#include "OtfaRegs.h"
#include "AutoRespRegs.h"
#include "HostIfAccRegs.h"
#include "RxPpRegs.h"
#endif
#if defined (ENET_INC_ARCH_WAVE600B) && defined (ENET_INC_LMAC) 
#include "HwMbfm_Api.h"
#endif //(ENET_INC_ARCH_WAVE600B) && (ENET_INC_LMAC)

/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 19

#ifndef ENET_INC_ARCH_WAVE600B

#if defined(ENET_INC_LMAC) 

#ifdef HW_ERROR_BEAMFORMING_ERROR_INDICATION_WORKAROUND

/* HW_ERROR_BEAMFORMING_ERROR indication is wrong in Gen6 Z0, need to mask it out (https://jira-chd.intel.com/browse/WLANVLSIIP-2934) */
#define HW_ERRORS_ENABLE_MASK	((1 << HW_ERROR_RXC_ERROR)										 | \
								(1 << HW_ERROR_PAC_TX_ERROR)									 | \
								(1 << HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY)				 | \
								(1 << HW_ERROR_SEQUENCER_ERROR)									 | \
								(1 << HW_ERROR_TX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED)						 | \
								(1 << HW_ERROR_BSRC_ERROR)										 | \
								(1 << HW_ERROR_DELIA_AUTO_FILL_END)								 | \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)								 | \
								(1 << HW_ERROR_TX_SELECTOR_ERROR)								 | \
								(1 << HW_ERROR_BA_ANALYZER_ERROR)								 | \
								(1 << HW_ERROR_RX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)								 | \
								(1 << HW_ERROR_BF_REPORT_AUTO_REPLY_ABORT_DUE_TO_PHY_MODE_ERROR) | \
								(1 << HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY)					 | \
								(1 << HW_ERROR_PS_SETTING_ERROR)								 | \
								(1 << HW_ERROR_ULPR_ERROR))

#else

#define HW_ERRORS_ENABLE_MASK	((1 << HW_ERROR_RXC_ERROR)										 | \
								(1 << HW_ERROR_PAC_TX_ERROR)									 | \
								(1 << HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY)				 | \
								(1 << HW_ERROR_SEQUENCER_ERROR)									 | \
								(1 << HW_ERROR_TX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED)						 | \
								(1 << HW_ERROR_BSRC_ERROR)										 | \
								(1 << HW_ERROR_DELIA_AUTO_FILL_END)								 | \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)								 | \
								(1 << HW_ERROR_TX_SELECTOR_ERROR)								 | \
								(1 << HW_ERROR_BA_ANALYZER_ERROR)								 | \
								(1 << HW_ERROR_RX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)								 | \
								(1 << HW_ERROR_BF_REPORT_AUTO_REPLY_ABORT_DUE_TO_PHY_MODE_ERROR) | \
								(1 << HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY)					 | \
								(1 << HW_ERROR_PS_SETTING_ERROR)								 | \
								(1 << HW_ERROR_BEAMFORMING_ERROR)								 | \
								(1 << HW_ERROR_ULPR_ERROR))

#endif //HW_ERROR_BEAMFORMING_ERROR_INDICATION_WORKAROUND

#endif /*ENET_INC_LMAC */
				
#if defined(ENET_INC_UMAC) 
#define HW_ERRORS_ENABLE_MASK	  ((1 << HW_ERROR_Q_MANAGER_UM_ERROR)			| \
								  (1 << HW_ERROR_ARM_DMA0_ABORT_ERROR)		  	| \
								  (1 << HW_ERROR_ARM_DMA1_ABORT_ERROR))

#endif /*#if defined(ENET_INC_UMAC) */

#else // ENET_INC_ARCH_WAVE600B

#if defined(ENET_INC_LMAC) 

#define HW_ERRORS_ENABLE_MASK	((1 << HW_ERROR_RXC_ERROR)										 | \
								(1 << HW_ERROR_PAC_TX_ERROR)									 | \
								(1 << HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY)				 | \
								(1 << HW_ERROR_SEQUENCER_ERROR)									 | \
								(1 << HW_ERROR_TX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED)						 | \
								(1 << HW_ERROR_BSRC_ERROR)										 | \
								(1 << HW_ERROR_DELIA_AUTO_FILL_END)								 | \
								(1 << HW_ERROR_TX_SELECTOR_ERROR)								 | \
								(1 << HW_ERROR_BA_ANALYZER_ERROR)								 | \
								(1 << HW_ERROR_RX_SECURITY_FAIL)								 | \
								(1 << HW_ERROR_Q_MANAGER_LM_ERROR)								 | \
								(1 << HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY)					 | \
								(1 << HW_ERROR_PS_SETTING_ERROR)								 | \
								(1 << HW_ERROR_BEAMFORMING_ERROR)								 | \
								(1 << HW_ERROR_ULPR_ERROR)                                       | \
								(1 << HW_ERROR_TXPD_ACC_ERROR)                                   | \
								(1 << HW_ERROR_TX_HC_ERROR)                                      | \
								(1 << HW_ERROR_TF_GEN_ERROR)                                     | \
								(1 << HW_ERROR_OTFA_TID_ERROR_FIFO_ERROR)                        | \
								(1 << HW_ERROR_AUTO_RESPONSE_ERROR))
#endif /*ENET_INC_LMAC */
				
#if defined(ENET_INC_UMAC) 
#define HW_ERRORS_ENABLE_MASK	  ((1 << HW_ERROR_Q_MANAGER_UM_ERROR)			    | \
								  (1 << HW_ERROR_ARM_DMA0_ABORT_ERROR)		  	    | \
								  (1 << HW_ERROR_ARM_DMA1_ABORT_ERROR)              | \
								  (1 << HW_ERROR_HOST_IF_ACC_ERROR)        		    | \
								  (1 << HW_ERROR_RXPP_SHRAM_FIFO_ERROR))

#endif /*#if defined(ENET_INC_UMAC) */

#endif // ENET_INC_ARCH_WAVE600B
/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef void (*HwErrorHandler_t)(void);

typedef struct DebugHwCounters
{
	uint32 rxBufferOverflowDroppedMpdus;
	uint32 txSelectorMuPrimaryNotValid;
	uint32 txSelectorMuNoPrimaryInGroup;
} DebugHwCounters_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
void hwErrorsRxcError(void);
void hwErrorsUnknown(void);
void hwErrorsPacTx(void);
void hwErrorsBlockAck(void);
void hwErrorsSequencerErrorHandling(void);
void hwErrorsSecurityFail(void);
void hwErrorsDeliaLimitTimerExpired(void);
void hwErrorsBsrcError(void);
void hwErrorsDeliaAutoFillEnd(void);
void hwErrorsQmanager(void);
void hwErrorsArmDmaAbort(void);
void hwErrorsTxSelector(void);
void hwErrorsBaAccelerator(void);
void hwErrorsWlanMacXbarDma(void);
void hwErrorsRxSecurity(void);
void hwErrorsAutoResponseErrorfifo(void);
void hwErrorsOtfAggregator_StatusFifo_NotEmpty(void);
void hwErrorsPsSettingError(void);
void hwErrorsBeamformingError(void);
void hwErrorsReadHwRegs(void);
void hwErrorsUlPostRx(void);
#ifdef ENET_INC_ARCH_WAVE600B
void hwErrorsTxpdAcc(void);                                                
void hwErrorsTxHc(void);                                   
void hwErrorsTfGen(void);                                
void hwErrorsOtfaTidErrorFifo(void);                     
void hwErrorsAutoResponse(void);   
void hwErrorsHostIfAcc(void);
void hwErrorsRxppShramFifo(void);                  
#endif
void hwErrorsDefault(void);
/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
#if defined(ENET_INC_LMAC)
static const HwErrorHandler_t HwErrorHandlers[HW_ERROR_TOTAL_NUM] = 
{
	hwErrorsRxcError,								/* HW_ERROR_RXC_ERROR										  			*/
	hwErrorsPacTx,									/* HW_ERROR_PAC_TX_ERROR,									  			*/
	hwErrorsAutoResponseErrorfifo,					/* HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY,					 		*/
	hwErrorsSequencerErrorHandling,					/* HW_ERROR_SEQUENCER_ERROR, 								  			*/
	hwErrorsSecurityFail, 							/* HW_ERROR_TX_SECURITY_FAIL,								  			*/
	hwErrorsDeliaLimitTimerExpired, 				/* HW_ERROR_DELIA_LIMIT_TIMER_EXPIRED,									*/
	hwErrorsBsrcError, 								/* HW_ERROR_BSRC_ERROR,													*/
	hwErrorsDeliaAutoFillEnd,						/* HW_ERROR_DELIA_AUTO_FILL_END,							  			*/
	hwErrorsTxSelector, 							/* HW_ERROR_TX_SELECTOR_ERROR, 											*/
	hwErrorsBaAccelerator,							/* HW_ERROR_BA_ANALYZER_ERROR,									  		*/
	hwErrorsRxSecurity, 							/* HW_ERROR_RX_SECURITY_FAIL, 						 					*/
	hwErrorsQmanager,								/* HW_ERROR_Q_MANAGER_LM_ERROR, 							 			*/
	hwErrorsDefault, 								/* HW_ERROR_BF_REPORT_AUTO_REPLY_ABORT_DUE_TO_PHY_MODE_ERROR			*/
	hwErrorsOtfAggregator_StatusFifo_NotEmpty, 		/* HW_ERROR_OTF_TID_ERROR_FIFO_NOT_EMPTY								*/
	hwErrorsPsSettingError,							/* HW_ERROR_PS_SETTING_ERROR											*/
	hwErrorsBeamformingError,						/* HW_ERROR_BEAMFORMING_ERROR											*/
	hwErrorsUlPostRx,								/* HW_ERROR_ULPR_ERROR													*/	
#ifdef ENET_INC_ARCH_WAVE600B	
    hwErrorsTxpdAcc,                                /* HW_ERROR_TXPD_ACC_ERROR,  										  	*/
    hwErrorsTxHc,                                   /* HW_ERROR_TX_HC_ERROR,										  		*/
	hwErrorsTfGen,                                  /* HW_ERROR_TF_GEN_ERROR,										  		*/
	hwErrorsOtfaTidErrorFifo,                       /* HW_ERROR_OTFA_TID_ERROR_FIFO_ERROR,									*/
	hwErrorsAutoResponse,                           /* HW_ERROR_AUTO_RESPONSE_ERROR,										*/
#else
	hwErrorsDefault,								/* HW_ERROR_RESERVED_17,												*/
	hwErrorsDefault,								/* HW_ERROR_RESERVED_18,												*/
	hwErrorsDefault,								/* HW_ERROR_RESERVED_19,												*/
	hwErrorsDefault,								/* HW_ERROR_RESERVED_20,												*/
	hwErrorsDefault,								/* HW_ERROR_RESERVED_21,												*/	
#endif
	hwErrorsDefault,	                            /* HW_ERROR_RESERVED_22,											    */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_23,										        */	
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_24,										        */	
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_25,										        */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_26,										        */	
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_27,										        */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_28,										        */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_29,										        */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_30,										        */
	hwErrorsDefault,                                /* HW_ERROR_RESERVED_31,										        */
};

#else // UMAC
static const HwErrorHandler_t HwErrorHandlers_Common[HW_ERROR_COMMON_TOTAL_NUM] = 
{
	hwErrorsArmDmaAbort,					/* HW_ERROR_ARM_DMA0_ABORT_ERROR,						  				*/
	hwErrorsArmDmaAbort,					/* HW_ERROR_ARM_DMA1_ABORT_ERROR,										*/
	hwErrorsQmanager,						/* HW_ERROR_Q_MANAGER_UM_ERROR, 										*/
#ifdef ENET_INC_ARCH_WAVE600B
    hwErrorsHostIfAcc,        	            /* HW_ERROR_HOST_IF_ACC_ERROR,                                          */
	hwErrorsRxppShramFifo,                  /* HW_ERROR_RXPP_SHRAM_FIFO_ERROR,                                      */
#else
    hwErrorsDefault,                        /* HW_ERROR_COMMON_RESERVED_3,                                          */
    hwErrorsDefault,                        /* HW_ERROR_COMMON_RESERVED_4,                                          */
#endif
};

#endif //ENET_INC_LMAC

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/
#if defined(ENET_INC_UMAC) || defined (ENET_INC_LMAC0)
static uint32 HwErrorsCounters_Band0[HW_ERROR_TOTAL_NUM];
#endif
#if defined(ENET_INC_UMAC) || defined (ENET_INC_LMAC1)
static uint32 HwErrorsCounters_Band1[HW_ERROR_TOTAL_NUM];
#endif
#if defined(ENET_INC_UMAC)
static uint32 HwErrorCounters_Common[HW_ERROR_COMMON_TOTAL_NUM];
#endif

#if defined (ENET_INC_LMAC)
static DebugHwCounters_t debugHwCounters;
#endif //ENET_INC_LMAC
RegMacWepTxInterruptsStatus_u	WepTxInterruptsStatus;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

#if defined(ENET_INC_LMAC)
/**********************************************************************************

hwErrorsRxcError


Description:
------------
	

**********************************************************************************/
void hwErrorsRxcError(void)
{
	//read RxC buffer overFlow counter, increment internal counter. It will be cleared later in RXC module. If the error is different, counter will remain the same (+0)
	debugHwCounters.rxBufferOverflowDroppedMpdus += Pac_RxcReadRxBufferOverflowCounter();
#ifdef ENET_INC_LMAC0
	CyclicBufferOverflowCounter[CONFIGURATION_MANAGER_BAND_0] = debugHwCounters.rxBufferOverflowDroppedMpdus;
#else
	CyclicBufferOverflowCounter[CONFIGURATION_MANAGER_BAND_1] = debugHwCounters.rxBufferOverflowDroppedMpdus;
#endif //ENET_INC_LMAC0

	Pac_RxcErrorIrq();

	//unmask event
#if defined (ENET_INC_LMAC0)
	HwErrors_UnMaskEvent0(HW_ERROR_RXC_ERROR);
#elif defined (ENET_INC_LMAC1)
	HwErrors_UnMaskEvent1(HW_ERROR_RXC_ERROR);
#endif

}
#endif

/**********************************************************************************

hwErrorsUnknown


Description:
------------
	
**********************************************************************************/
void hwErrorsUnknown(void)
{
	// This assert shouldn't happen. It means we got an HW ERROR we are not familiar with.
	FATAL("hwErrorsUnknown");
}


/**********************************************************************************

hwErrorsPacTx

Description:
------------
	
**********************************************************************************/
void hwErrorsPacTx(void)
{
	/*Assert Was replaced with counter because this error can happen */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_TX_ERROR);

#if defined (ENET_INC_LMAC0)
		HwErrors_UnMaskEvent0(HW_ERROR_PAC_TX_ERROR);
#elif defined (ENET_INC_LMAC1)
		HwErrors_UnMaskEvent1(HW_ERROR_PAC_TX_ERROR);
#endif

}

/**********************************************************************************

hwErrorsBlockAck 

Description:
------------
	Error Handler for the BA geneartor error event	

**********************************************************************************/
void hwErrorsBlockAck(void)
{
	ASSERT(0);
}

 /**********************************************************************************

hwErrorsSecurityFail

Description:
------------
	

**********************************************************************************/
void hwErrorsSecurityFail(void)
{
	 // Get the error status
	 WepTxInterruptsStatus.val = 0;
	 RegAccess_Read(REG_MAC_WEP_TX_INTERRUPTS_STATUS, &WepTxInterruptsStatus.val);
	 ASSERT(0);
}

/**********************************************************************************

hwErrorsDeliaLimitTimerExpired

Description:
------------

	
**********************************************************************************/
void hwErrorsDeliaLimitTimerExpired(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsBsrcError

Description:
------------
Got bsrc_start pulse from RxC while RxC FIFO was not empty
we should not get here
	
**********************************************************************************/
void hwErrorsBsrcError(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsDeliaAutoFillEnd

Description:
------------
	
**********************************************************************************/
void hwErrorsDeliaAutoFillEnd(void)
{
	/*Assert Was replaced with counter because this error can happen */
	InterruptManager_ClearEdgedTriggeredInterrupt(EDGED_TRIGGERED_INT_AUTO_FILL_END);

#if defined (ENET_INC_LMAC0)
		HwErrors_UnMaskEvent0(HW_ERROR_DELIA_AUTO_FILL_END);
#elif defined (ENET_INC_LMAC1)
		HwErrors_UnMaskEvent1(HW_ERROR_DELIA_AUTO_FILL_END);
#endif
	
}


/**********************************************************************************

hwErrorsQmanager

Description:
------------
	

**********************************************************************************/
void hwErrorsQmanager(void)
{
	HwQManager_ErrorEvent();
}

/**********************************************************************************

hwErrorsArmDmaAbort

Description:
------------
	

**********************************************************************************/
void hwErrorsArmDmaAbort(void)
{
	// We can get here from DMA0 or DMA1. Need to check the source of the HW ERROR.
	hwErrorsReadHwRegs();
	ASSERT(0);
}



/**********************************************************************************

hwErrorsReadHwRegs


Description:
------------
	

**********************************************************************************/
void hwErrorsReadHwRegs(void)
{
#if 0 //TBD - add for debug if necessary
	uint8 rxstatIndex = 0;

	RegAccess_Read(REG_WLAN_ARM_DMA_FTR0, &DmaAbort0);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR1, &DmaAbort1);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR2, &DmaAbort2);
	RegAccess_Read(REG_WLAN_ARM_DMA_FTR3, &DmaAbort3);
	RegAccess_Read(MEM_BE_ERR_COUNTER, &RxAggr);
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_WL_MAC_IRQ, &WlMacIrq);
	for (rxstatIndex = 0; rxstatIndex < 8 ; rxstatIndex++)
	{
		RegAccess_Read((REG_MAC_HT_EXTENSIONS_RX_STAT0_LO + (4*rxstatIndex)), &RXstat[rxstatIndex]);
	}
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_DEL_DEBUG2, &DelineatorState);
	RegAccess_Read(REG_PAC_RXC_RXC_STATE, &RxCState);
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_DELINEATOR_AGG_ERR_CNT, &delAggErrCnt);

#endif
}



/**********************************************************************************

hwErrorsTxSelector

Description:
------------
	

**********************************************************************************/
void hwErrorsTxSelector(void)
{
#if defined (ENET_INC_LMAC)

	RegTxSelectorTxSelMuErr_u txSelectorMuError;
	RegTxSelectorTxSelErr_u txSelectorSuError;
	RegTxSelectorTxSelMuErrClr_u txSelectorMuErrClr;

	txSelectorMuErrClr.val = 0;
	
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_MU_ERR, &txSelectorMuError.val);
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_ERR, &txSelectorSuError.val);
	
	if((txSelectorSuError.val != 0) || (txSelectorMuError.bitFields.primaryGrpThresErr == TRUE))
	{
		FATAL("hwErrorsTxSelector");
	}
	else
	{
	 	/* The following two errors can't cause FATAL because in case of LOCKED group
	 	   Selector still checks these errors which is incorrect behavior */
		if (txSelectorMuError.bitFields.primaryNotValidErr)
		{
			debugHwCounters.txSelectorMuPrimaryNotValid++;
			txSelectorMuErrClr.bitFields.primaryNotValidErrClr = 1;
		}
		else
		{
			debugHwCounters.txSelectorMuNoPrimaryInGroup++;
			txSelectorMuErrClr.bitFields.noPrimaryInGrpErrClr = 1;
		}
		RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_ERR_CLR, txSelectorMuErrClr.val);
#if defined (ENET_INC_LMAC0)
		HwErrors_UnMaskEvent0(HW_ERROR_PAC_TX_ERROR);
#elif defined (ENET_INC_LMAC1)
		HwErrors_UnMaskEvent1(HW_ERROR_PAC_TX_ERROR);
#endif
	}
#else // UMAC
	FATAL("hwErrorsTxSelector");
#endif // ENET_INC_LMAC	
}

/**********************************************************************************

hwErrorsBaAccelerator


Description:
------------
	

**********************************************************************************/
void hwErrorsBaAccelerator(void)
{
	isr_BaAnalyzerInterruptErrorHandler();

#if defined (ENET_INC_LMAC0)
	HwErrors_UnMaskEvent0(HW_ERROR_BA_ANALYZER_ERROR);
#else
	HwErrors_UnMaskEvent1(HW_ERROR_BA_ANALYZER_ERROR);
#endif
}

void hwErrorsUlPostRx(void)
{
	isr_UlPostRxErrorHandler();
	
#if defined (ENET_INC_LMAC0)
	HwErrors_UnMaskEvent0(HW_ERROR_ULPR_ERROR);
#else
	HwErrors_UnMaskEvent1(HW_ERROR_ULPR_ERROR);
#endif
}


/**********************************************************************************

isr_linkAdaptation_SequencerErrorHandling 


Description:
------------


Input: 
-----

Returns:
--------
	void - 
	
**********************************************************************************/

void hwErrorsSequencerErrorHandling(void)
{
	ASSERT(0);
}


/**********************************************************************************

hwErrorsWlanMacXbarDma

Description:
------------
	

**********************************************************************************/
void hwErrorsWlanMacXbarDma(void)
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsRxSecurity

Description:
------------
	

**********************************************************************************/
void hwErrorsRxSecurity(void)
{
	ASSERT(0);
}

/**********************************************************************************
hwErrorsAutoResponseErrorfifo  

Description:
------------
	Handle error interrupt of Auto Response.
	
**********************************************************************************/
void hwErrorsAutoResponseErrorfifo(void)
{

#if defined (ENET_INC_LMAC0)	
	HwErrors_MaskEvent0(HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY);
#elif defined (ENET_INC_LMAC1)
	HwErrors_MaskEvent1(HW_ERROR_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY);
#endif

	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_AUTO_RESPONSE_ERROR_FIFO_NOT_EMPTY, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);		
}


/**********************************************************************************

hwErrorsOtfAggregator_StatusFifo_NotEmpty  


Description:
------------
HW error handling that is invoked when OTF_aggregator status FIFO is not empty


Input:	None
-----
 		
Output:	None
-------
	
**********************************************************************************/
void hwErrorsOtfAggregator_StatusFifo_NotEmpty(void)
{
	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_OTF_AGGR_FIFO_NOT_EMPTY, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************
hwErrorsPsSettingError  

Description:
------------
	Handle error interrupt of Power Save.
	
**********************************************************************************/
void hwErrorsPsSettingError(void) 	 
{
	ASSERT(0);
}

/**********************************************************************************

hwErrorsBeamformingError  


Description:
------------
HW error that happens when MBFM module is triggered by Sender before it finished the previous report


Input:	None
-----
 		
Output:	None
-------
	
**********************************************************************************/

void hwErrorsBeamformingError(void)
{
#if defined (ENET_INC_ARCH_WAVE600B) && defined (ENET_INC_LMAC)
	HwMbfm_isrError();

#if defined (ENET_INC_LMAC0)
	HwErrors_UnMaskEvent0(HW_ERROR_BEAMFORMING_ERROR);
#else
	HwErrors_UnMaskEvent1(HW_ERROR_BEAMFORMING_ERROR);
#endif

#else
	ASSERT(0);
#endif // (ENET_INC_ARCH_WAVE600B) && (ENET_INC_LMAC)
}

#ifdef ENET_INC_ARCH_WAVE600B 
/**********************************************************************************
hwErrorsTxpdAcc 

Description:
------------
	Handle error interrupt of TXPD ACC 
	
**********************************************************************************/
void hwErrorsTxpdAcc(void) 
{
    RegTxPdAccTxpdIntErrorStatus_u txPdAccTxpdIntErrorStatus;

    txPdAccTxpdIntErrorStatus.val = 0;
    RegAccess_Read(REG_TX_PD_ACC_TXPD_INT_ERROR_STATUS, &txPdAccTxpdIntErrorStatus.val);
    SLOG0(0, 0, RegTxPdAccTxpdIntErrorStatus_u, &txPdAccTxpdIntErrorStatus); // In case more than 1 bit is turned on at the same time
    if (txPdAccTxpdIntErrorStatus.bitFields.txpdGoWhileActiveMismatchStatus)
    {
        FATAL("hwErrorsTxpdAcc - txpdGoWhileActiveMismatchStatus");
    }
    if (txPdAccTxpdIntErrorStatus.bitFields.txpdSymbolOverheadOverflowStatus)
    {
        FATAL("hwErrorsTxpdAcc - txpdSymbolOverheadOverflowStatus");
    }
    if (txPdAccTxpdIntErrorStatus.bitFields.txpdLastInStaDlmMismatchStatus)
    {
        FATAL("hwErrorsTxpdAcc - txpdLastInStaDlmMismatchStatus");
    }
    if (txPdAccTxpdIntErrorStatus.bitFields.txpdOccupiedPtdUnderflowStatus)
    {
        FATAL("hwErrorsTxpdAcc - txpdOccupiedPtdUnderflowStatus");
    }
    if (txPdAccTxpdIntErrorStatus.bitFields.txpdSnSsnDiffOorStatus)
    {
        FATAL("hwErrorsTxpdAcc - txpdSnSsnDiffOorStatus");
    }
}

/**********************************************************************************

hwErrorsTxHc 

Description:
------------
	Handle error interrupt of TX HC 
	
**********************************************************************************/
void hwErrorsTxHc(void) 
{
    RegMacTxHcTxHcIntErrorStatus_u txHcIntErrorStatus;

    txHcIntErrorStatus.val = 0;
    RegAccess_Read(REG_MAC_TX_HC_TX_HC_INT_ERROR_STATUS, &txHcIntErrorStatus.val);
    SLOG0(0, 0, RegMacTxHcTxHcIntErrorStatus_u, &txHcIntErrorStatus); // In case more than 1 bit is turned on at the same time
    if (txHcIntErrorStatus.bitFields.txHcDmaLengthMismatchStatus)
    {
        FATAL("hwErrorsTxHc - txHcDmaLengthMismatchStatus");
    }
	if (txHcIntErrorStatus.bitFields.txHcMsduNonDataMismatchStatus)
    {
        FATAL("hwErrorsTxHc - txHcMsduNonDataMismatchStatus");
    }
    if (txHcIntErrorStatus.bitFields.txHcMpduNotFirstMsduMismatchStatus)
    {
        FATAL("hwErrorsTxHc - txHcMpduNotFirstMsduMismatchStatus");
    }
    if (txHcIntErrorStatus.bitFields.txHcJobFifoPushWhileFullStatus)
    {
        FATAL("hwErrorsTxHc - txHcJobFifoPushWhileFullStatus");
    }
}

/**********************************************************************************
hwErrorsTfGen 

Description:
------------
	Handle error interrupt of Trigger frame Gen.
	
**********************************************************************************/
void hwErrorsTfGen(void) 
{
    RegTfGeneratorTfGenIntErrorStatus_u tfGenIntErrorStatus;

    tfGenIntErrorStatus.val = 0;
    RegAccess_Read(REG_TF_GENERATOR_TF_GEN_INT_ERROR_STATUS, &tfGenIntErrorStatus.val);
    SLOG0(0, 0, RegTfGeneratorTfGenIntErrorStatus_u, &tfGenIntErrorStatus); // In case more than 1 bit is turned on at the same time
    if (tfGenIntErrorStatus.bitFields.tfrgGoWhileActiveMismatchStatus)
    {
        FATAL("hwErrorsTfGen - tfrgGoWhileActiveMismatchStatus");
    }
    if (tfGenIntErrorStatus.bitFields.tfrgTfModeMismatchStatus)
    {
        FATAL("hwErrorsTfGen - tfrgTfModeMismatchStatus");
    }
    if (tfGenIntErrorStatus.bitFields.tfrgEmptyTidMismatchStatus)
    {
        FATAL("hwErrorsTfGen - tfrgEmptyTidMismatchStatus");
    }
    if (tfGenIntErrorStatus.bitFields.tfrgRuIndexLutOorStatus)
    {
        FATAL("hwErrorsTfGen - tfrgRuIndexLutOorStatus");
    }
	
}

/**********************************************************************************
hwErrorsOtfaTidErrorFifo 

Description:
------------
	Handle error interrupt of OTFA TID error FIFO
	
**********************************************************************************/
void hwErrorsOtfaTidErrorFifo(void) 
{
    RegOtfaErrorIrqStatus_u otfaErrorIrqStatus;

    otfaErrorIrqStatus.val = 0;
    RegAccess_Read(REG_OTFA_ERROR_IRQ_STATUS, &otfaErrorIrqStatus.val);
    SLOG0(0, 0, RegOtfaErrorIrqStatus_u, &otfaErrorIrqStatus); // In case more than 1 bit is turned on at the same time
    if (otfaErrorIrqStatus.bitFields.tidErrorFifoDecrementLessThanZeroIrq)
    {
	    FATAL("hwErrorsOtfaTidErrorFifo - tidErrorFifoDecrementLessThanZeroIrq");
    }
    if (otfaErrorIrqStatus.bitFields.tidErrorFifoFullDropIrq)
    {
        // indication for FW to enlarge the FIFO
        FATAL("hwErrorsOtfaTidErrorFifo - tidErrorFifoFullDropIrq, need to enlarge the FIFO");
    }

}

/**********************************************************************************
hwErrorsAutoResponse


Description:
------------
	Handle error interrupt of Auto response
	
**********************************************************************************/
void hwErrorsAutoResponse(void)	 
{
    RegAutoRespIntErrStatus_u autoRespIntErrStatus; 

    autoRespIntErrStatus.val = 0;
    RegAccess_Read(REG_AUTO_RESP_INT_ERR_STATUS, &autoRespIntErrStatus.val);
    SLOG0(0, 0, RegAutoRespIntErrStatus_u, &autoRespIntErrStatus); // In case more than 1 bit is turned on at the same time
    if (autoRespIntErrStatus.bitFields.autoRespErrFifoDecrementLessThanZeroIrq)
    {
        FATAL("hwErrorsAutoResponse - autoRespErrFifoDecrementLessThanZeroIrq");
    }
    if (autoRespIntErrStatus.bitFields.autoRespErrFifoFullDropIrq)
    {
        // indication for FW to enlarge the FIFO
        FATAL("hwErrorsAutoResponse - autoRespErrFifoFullDropIrq, need to enlarge the FIFO");
    }
}
 
/**********************************************************************************
hwErrorsHostIfAcc


Description:
------------
	Handle error interrupt of Host IF Acc
	
**********************************************************************************/
void hwErrorsHostIfAcc(void)	 
{
    RegHostIfAccHostIfAccErrInt_u    hostIfAccHostIfAccErrInt;
    RegHostIfAccHostIfAccErrIntClr_u hostIfAccHostIfAccErrIntClr;

    hostIfAccHostIfAccErrInt.val = 0;
    hostIfAccHostIfAccErrIntClr.val = 0;
    RegAccess_Read(REG_HOST_IF_ACC_HOST_IF_ACC_ERR_INT, &hostIfAccHostIfAccErrInt.val);
    SLOG0(0, 0, RegHostIfAccHostIfAccErrInt_u, &hostIfAccHostIfAccErrInt); // In case more than 1 bit is turned on at the same time
    if (hostIfAccHostIfAccErrInt.bitFields.byteCountNegativeWriteErr)
    {
        FATAL("hwErrorsHostIfAcc - byteCountNegativeWriteErr");
    }
    if (hostIfAccHostIfAccErrInt.bitFields.byteCountWraparoundErr)
    {
        FATAL("hwErrorsHostIfAcc - byteCountWraparoundErr");
    }  
    if (hostIfAccHostIfAccErrInt.bitFields.txInOwnBitErrInt)
    {
        hostIfAccHostIfAccErrIntClr.bitFields.txInOwnBitErrIntClr = TRUE; // We don't use own bit and we shouldn't get here
    }
    RegAccess_Write(REG_HOST_IF_ACC_HOST_IF_ACC_ERR_INT_CLR, hostIfAccHostIfAccErrIntClr.val);
	HwErrors_UnMaskEvent2(HW_ERROR_HOST_IF_ACC_ERROR);
    // In wave600-2 there will be 2 more interrupts - busyPdCountOverflowInt & busyRdCountOverflowInt. according to jira WLANVLSIIP-3475
}

/**********************************************************************************
hwErrorsRxppShramFifo


Description:
------------
	Handle error interrupt of RX Post Processing SHRAM FIFO
	
**********************************************************************************/
void hwErrorsRxppShramFifo(void) 
{
	RegRxPpShramFifoDebug_u	rxPpShramFifoDebugReg;
	rxPpShramFifoDebugReg.val = 0;
	
	RegAccess_Read(REG_RX_PP_SHRAM_FIFO_DEBUG, &rxPpShramFifoDebugReg.val);

	SLOG0(0, 0, RegRxPpShramFifoDebug_u, &rxPpShramFifoDebugReg); // In case more than 1 bit is turned on at the same time
    if (rxPpShramFifoDebugReg.bitFields.shramFifoFull)
    {
        FATAL("[hwErrorsRxppShramFifo] - shramFifoFull");
    }
    else if (rxPpShramFifoDebugReg.bitFields.shramFifoDecrementLessThanZero)
    {
        FATAL("[hwErrorsRxppShramFifo] - shramFifoDecrementLessThanZero");
    }
	else if (rxPpShramFifoDebugReg.bitFields.shramFifoOverflow)
    {
        FATAL("[hwErrorsRxppShramFifo] - shramFifoOverflow");
    }
	else if (rxPpShramFifoDebugReg.bitFields.shramFifoFullDropCtr != 0)
    {
        FATAL("[hwErrorsRxppShramFifo] - shramFifoFullDropCtr");
    }
	else
	{
	    FATAL("[hwErrorsRxppShramFifo] - Error without known indication");
	}
}
#endif //ENET_INC_ARCH_WAVE600B

/**********************************************************************************
hwErrorsDefault

Description:
------------
	
**********************************************************************************/
void hwErrorsDefault(void)
{
	// This assert shouldn't happen. 
	FATAL("hwErrorsDefault");
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC0)
/**********************************************************************************

HwErrors_MaskEvent0 

Description:
------------
	mask Hw error 0 interrupt - from Band0

Input:
-----
	hwError - The Hw error to be masked
Output:
-------
	None

**********************************************************************************/
void HwErrors_MaskEvent0(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrors0MaskedInterrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = 0;
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS0_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

/**********************************************************************************

HwErrors_UnMaskEvent0 

Description:
------------
	unmask Hw error 0 interrupt - from Band 0
Input:
-----
	hwError - The Hw error to be unmasked
Output:
-------
	None

**********************************************************************************/
void HwErrors_UnMaskEvent0(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrors0MaskedInterrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = (1 << hwError);
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS0_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

/**********************************************************************************

isr_HwErrors_Band0 

Description:
------------
	General isr for all error events from Band0
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwErrors_Band0(void)
{
	
	RegEmeraldEnvHwErrors0MaskedInterrupts_u hwErrorRegister;
	uint32 errorIndex;
	
	hwErrorRegister.val = 0;
	
	RegAccess_Read(REG_EMERALD_ENV_HW_ERRORS0_MASKED_INTERRUPTS, &hwErrorRegister.val);

	while(hwErrorRegister.val != 0)
	{

		errorIndex = Utils_FindFirstSetAndClear(&hwErrorRegister.val);
		DEBUG_ASSERT(errorIndex < HW_ERROR_TOTAL_NUM);

		//("isr_HwErrors errorIndex %d ",errorIndex);

		HwErrors_MaskEvent0((HwErrorId_e)errorIndex);
			
		HwErrorsCounters_Band0[errorIndex]++;

		ILOG0_DD("isr_HwErrors_Band0 errorIndex %d num of errors %d",errorIndex,HwErrorsCounters_Band0[errorIndex]);

		HwErrorHandlers[errorIndex]();
	}

}

#endif // ENET_INC_LMAC0

#if defined (ENET_INC_LMAC1)
/**********************************************************************************

HwErrors_MaskEvent1 

Description:
------------
	mask Hw error 1 interrupt - from Band1

Input:
-----
	hwError - The Hw error to be masked
Output:
-------
	None

**********************************************************************************/
void HwErrors_MaskEvent1(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrors1MaskedInterrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = 0;
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS1_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}
/**********************************************************************************

HwErrors_UnMaskEvent1 

Description:
------------
	unmask Hw error 1 interrupt - from Band 1
Input:
-----
	hwError - The Hw error to be unmasked
Output:
-------
	None

**********************************************************************************/
void HwErrors_UnMaskEvent1(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrors1MaskedInterrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = (1 << hwError);
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS1_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

/**********************************************************************************

isr_HwErrors_Band1 

Description:
------------
	General isr for all error events from Band1
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwErrors_Band1(void)
{
	
	RegEmeraldEnvHwErrors1MaskedInterrupts_u hwErrorRegister;
	uint32 errorIndex;
	
	hwErrorRegister.val = 0;
	
	RegAccess_Read(REG_EMERALD_ENV_HW_ERRORS1_MASKED_INTERRUPTS, &hwErrorRegister.val);

	while(hwErrorRegister.val != 0)
	{

		errorIndex = Utils_FindFirstSetAndClear(&hwErrorRegister.val);
		DEBUG_ASSERT(errorIndex < HW_ERROR_TOTAL_NUM);

		//("isr_HwErrors errorIndex %d ",errorIndex);

		HwErrors_MaskEvent1((HwErrorId_e)errorIndex);
			
		HwErrorsCounters_Band1[errorIndex]++;

		ILOG0_DD("isr_HwErrors_Band1 errorIndex %d num of errors %d",errorIndex,HwErrorsCounters_Band1[errorIndex]);

		HwErrorHandlers[errorIndex]();
	}

}

#endif // ENET_INC_LMAC1

#if defined (ENET_INC_UMAC)

/**********************************************************************************

isr_HwErrors_Common 

Description:
------------
	General isr for all error events from Errors2 
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwErrors_Common(void)
{
	
	RegEmeraldEnvHwErrors2MaskedInterrupts_u hwErrorRegister;
	uint32 errorIndex;
	
	hwErrorRegister.val = 0;
	
	RegAccess_Read(REG_EMERALD_ENV_HW_ERRORS2_MASKED_INTERRUPTS, &hwErrorRegister.val);

	while(hwErrorRegister.val != 0)
	{

		errorIndex = Utils_FindFirstSetAndClear(&hwErrorRegister.val);
		DEBUG_ASSERT(errorIndex < HW_ERROR_TOTAL_NUM);

		//("isr_HwErrors errorIndex %d ",errorIndex);

		HwErrors_MaskEvent2((HwErrorId_e)errorIndex);
			
		HwErrorCounters_Common[errorIndex]++;

		ILOG0_DD("isr_HwErrors_Common errorIndex %d num of errors %d",errorIndex,HwErrorCounters_Common[errorIndex]);

		HwErrorHandlers_Common[errorIndex]();
	}

}


/**********************************************************************************

HwErrors_MaskEvent2 

Description:
------------
	mask Hw error 2 interrupt - common

Input:
-----
	hwError - The Hw error to be masked
Output:
-------
	None

**********************************************************************************/
void HwErrors_MaskEvent2(HwErrorId_e hwError)
{
	RegEmeraldEnvHwErrors2Interrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = 0;
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS2_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}


/**********************************************************************************

HwErrors_UnMaskEvent2 

Description:
------------
	unmask Hw error 2 interrupt - common
Input:
-----
	hwError - The Hw error to be unmasked
Output:
-------
	None

**********************************************************************************/
void HwErrors_UnMaskEvent2(HwErrorIdCommon_e hwError)
{
	RegEmeraldEnvHwErrors2Interrupts_u hwErrosIntMaskedStatusReg;
	uint32 maskVal;

	hwErrosIntMaskedStatusReg.val = (1 << hwError);
	maskVal = (1 << hwError);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_ERRORS2_INT_EN,maskVal,hwErrosIntMaskedStatusReg.val);
}

#endif // ENET_INC_UMAC

/**********************************************************************************

HwErrors_Init 

Description:
------------
	Enabling all required Hw Error Events to generated to FW
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwErrors_Init(void)
{
#if defined (ENET_INC_LMAC)
		memset(&debugHwCounters,0, sizeof(DebugHwCounters_t));
#endif //ENET_INC_LMAC

#if defined (ENET_INC_LMAC0)
		memset32(&HwErrorsCounters_Band0,0,HW_ERROR_TOTAL_NUM);
		RegAccess_Write(REG_EMERALD_ENV_HW_ERRORS0_INT_EN, HW_ERRORS_ENABLE_MASK);
#elif  defined (ENET_INC_LMAC1)
		memset32(&HwErrorsCounters_Band1,0,HW_ERROR_TOTAL_NUM);
		RegAccess_Write(REG_EMERALD_ENV_HW_ERRORS1_INT_EN, HW_ERRORS_ENABLE_MASK);
#elif defined (ENET_INC_UMAC)
		memset32(&HwErrorsCounters_Band0,0,HW_ERROR_TOTAL_NUM);
		memset32(&HwErrorsCounters_Band1,0,HW_ERROR_TOTAL_NUM);
		memset32(&HwErrorCounters_Common,0,HW_ERROR_COMMON_TOTAL_NUM);
		RegAccess_Write(REG_EMERALD_ENV_HW_ERRORS2_INT_EN, HW_ERRORS_ENABLE_MASK);
#endif
	
}



