/***********************************************************************************
 File:		HwEvents.c
 Module:		Hw Events And Errors
 Purpose:		Handle all Hw Events
 Description:	 
************************************************************************************/
/*---------------------------------------------------------------------------------
/								Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "OSAL_Tasks.h"
#include "OSAL_UpperMacMessages.h"
#include "Utils_Api.h"
#include "HwEventsAndErrors_Api.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "HostInterface_API.h"
#include "HostGenRiscInterface_Api.h"
#include "InterruptManager_Api.h"
#include "EventsManager_api.h"
#include "loggerAPI.h"
#include "CpuLoad_Api.h"
#include "TxManager_Api.h"
#include "shram_interruptSection.h"
#include "Pauser_Api.h"
#include "HwQManager_API.h"
#include "TxSelector_Api.h"
#include "ConfigurationManager_api.h"
#ifdef ENET_INC_LMAC
#include "SenderInterface_Api.h"
#include "RxHandler_Api.h"
#include "PreAggregator_Api.h"
#include "PhyDriver_API.h"
#include "HwMbfm_Api.h"
#else
#include "Pac_Api.h"
#include "TxPacketsClassifier_API.h"
#include "RxManager_Api.h"
#include "PacketTrafficArbitrator_api.h"
#endif /* #ifdef ENET_INC_LMAC */
#ifdef ENET_INC_ARCH_WAVE600B
#include "MacGeneralRegs.h"
#endif



/*---------------------------------------------------------------------------------
/								Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 20

#if defined(ENET_INC_LMAC) 

#ifndef GEN6_REL3_5_HW_BUG_PATCH

#define HW_EVENTS_ENABLE_MASK_A				((1 << HW_EVENT_BA_ANALYZER_STATUS_FIFO)	| \
											(1 << HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY)	| \
											(1 << HW_EVENT_PHY) 						| \
											(1 << HW_EVENT_RXF_FIFO_NOT_EMPTY)			| \
											(1 << HW_EVENT_RX_HALT_INDICATION)			| \
											(1 << HW_EVENT_GENRISC_SENDER)				| \
											(1 << HW_EVENT_TSF_TIMER)					| \
											(1 << HW_EVENT_PRE_AGGREGATOR)				| \
											(1 << HW_EVENT_GENRISC_RX_HANDLER))

#else

//patch for HW bug - phy isr that occurs and is not cleared
#define HW_EVENTS_ENABLE_MASK_A				((1 << HW_EVENT_BA_ANALYZER_STATUS_FIFO)	| \
											(1 << HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY)	| \
											(1 << HW_EVENT_PHY) 						| \
											(1 << HW_EVENT_RXF_FIFO_NOT_EMPTY)			| \
											(1 << HW_EVENT_RX_HALT_INDICATION)			| \
											(1 << HW_EVENT_GENRISC_SENDER)				| \
											(1 << HW_EVENT_TSF_TIMER)					| \
											(1 << HW_EVENT_PRE_AGGREGATOR)				| \
											(1 << HW_EVENT_GENRISC_RX_HANDLER))

#endif //GEN6_REL3_5_HW_BUG_PATCH



#define HW_EVENTS_ENABLE_MASK_B				(1 << HW_EVENT_BAND_REL_B_BEAMFORMING_ERROR_FIFO_NOT_EMPTY)


#define  HW_EVENTS_ENABLE_MASK_COMMON_A 	(0)

#endif /*ENET_INC_LMAC */




#if defined(ENET_INC_UMAC) 

#define HW_EVENTS_ENABLE_MASK_A				(1 << HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION)

#ifndef ENET_INC_ARCH_WAVE600B

#define HW_EVENTS_ENABLE_MASK_COMMON_A		((1 << HW_EVENT_RXPP_FRAGMENT_FIFO_NOT_EMPTY) 			| \
											(1 << HW_EVENT_PD_THRESHOLD_REACHED)					| \
											(1 << HW_EVENT_RAB_NPU_TO_UPI_IRQ)						| \
											(1 << HW_EVENT_Q_MANAGER_DISABLE_LIST_NON_EMPTY_UMAC)	| \
											(1 << HW_EVENT_AGER_SINGLE_CHECK_DONE)					| \
											(1 << HW_EVENT_INT_SHRAM_ADDR)							| \
											(1 << HW_EVENT_HOST_IF_GENRISC)							| \
											(1 << HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY_UMAC)			| \
											(1 << HW_EVENT_HIGH_PRI_RX_PD_READY_UMAC)				| \
											(1 << HW_EVENT_HIGH_PRI_TX_PD_READY_UMAC)				| \
											(1 << HW_EVENT_TX_DONE_LISTS_UMAC)						| \
											(1 << HW_EVENT_EXTERNAL_IRQ_GPIO_10))
#else // ENET_INC_ARCH_WAVE600B

#define HW_EVENTS_ENABLE_MASK_COMMON_A		((1 << HW_EVENT_RXPP_FRAGMENT_FIFO_NOT_EMPTY) 			| \
											(1 << HW_EVENT_PD_THRESHOLD_REACHED)					| \
											(1 << HW_EVENT_RAB_NPU_TO_UPI_IRQ)						| \
											(1 << HW_EVENT_Q_MANAGER_DISABLE_LIST_NON_EMPTY_UMAC)	| \
											(1 << HW_EVENT_AGER_SINGLE_CHECK_DONE)					| \
											(1 << HW_EVENT_INT_SHRAM_ADDR)							| \
											(1 << HW_EVENT_HOST_IF_GENRISC)							| \
											(1 << HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY_UMAC)			| \
											(1 << HW_EVENT_HIGH_PRI_RX_PD_READY_UMAC)				| \
											(1 << HW_EVENT_HIGH_PRI_TX_PD_READY_UMAC)				| \
											(1 << HW_EVENT_TX_DONE_LISTS_UMAC)						| \
											(1 << HW_EVENT_EXTERNAL_IRQ_GPIO_10)                    | \
                                            (1 << HW_EVENT_APB_TIMEOUT))

#endif // ENET_INC_ARCH_WAVE600B

#define HW_EVENTS_ENABLE_MASK_B				((1 << HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC) | \
											(1 << HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC))


#endif /*#if defined(ENET_INC_UMAC) */


/*---------------------------------------------------------------------------------
/								Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef void (*HwEventHandler_t)(void);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void HwEventsDefaultHandler(void);
#if defined(ENET_INC_LMAC)
static void HwEventsBaAnalyzer_StatusFifoNotEmpty(void);
static void HwEventsPsSettingFifoNotEmpty(void);
static void HwEventsRxfFifoNotEmpty(void);
static void HwEventsPhy(void);
static void HwEventsRxHaltIndication(void);
static void HwEventsSender(void);
static void HwEventsRxHandler(void);
static void HwEventsPacTsfTimer(void);
static void HwEventsBsrcFwDoneInterrupt(void);
static void HwEventsBeamFormingErrorFifoNotEmpty(void);
#endif
#if defined(ENET_INC_UMAC)
static void HwEventsRxCoordinatorFrameClassViolation_Band0(void);
static void HwEventsRxCoordinatorFrameClassViolation_Band1(void);
static void HwEventsRxPpFragmentFifoNotEmpty(void);
static void HwEventsPdThresholdReached(void);
static void HwEventsHwQueueManagerLowTxReady(void);
static void HwEventsHwQueueManagerTxDoneList(void);
#ifndef ATOMIC_COUNTER_ENABLER
static void HwEventsRabNpuToUpiIrq(void);
#endif
static void HwEventsAgerSingleCheckDone(void);
static void HwEventsTxSelectorGroupLockerReqServiced_Band0(void);
static void HwEventsTxSelectorGroupLockerReqServiced_Band1(void);
static void HwEventsHostIFGenRisc(void);
static void hwEventsPtaGpio0(void);
static void HwEventsPlanLocker_Band0(void);
static void HwEventsPlanLocker_Band1(void);
#ifdef ENET_INC_ARCH_WAVE600B
static void HwEventsApbTimeout(void);
#endif //#ifdef ENET_INC_ARCH_WAVE600B
#endif

#ifdef ENET_INC_UMAC
#ifndef ATOMIC_COUNTER_ENABLER
static void HwEventsRabNpuToUpiIrq(void);
#endif
#endif











/*---------------------------------------------------------------------------------
/							Static Variables								
/----------------------------------------------------------------------------------*/
	/* The following HweventsHandler is defined for the 3 ARCs 
	   The HW_EVENT_MASK_A for this Handler is different for the 2 LMACs and the UMAC */ 

#if defined(ENET_INC_LMAC)
const HwEventHandler_t HwEventsHandlers_A[HW_EVENT_BAND_REL_A_TOTAL_NUM] = 
{
	HwEventsDefaultHandler,							/* HW_EVENT_DELIA_2_GENRISC_TX_START */
	HwEventsDefaultHandler, 						/* HW_EVENT_SEQUENCER_RA_FILLER */
	HwEventsDefaultHandler,							/* HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION */
	HwEventsBaAnalyzer_StatusFifoNotEmpty, 			/* HW_EVENT_BA_ANALYZER_STATUS_FIFO */
	HwEventsPsSettingFifoNotEmpty,					/* HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY */ 
	HwEventsPhy,									/* HW_EVENT_PHY */ 
	HwEventsRxfFifoNotEmpty,						/* HW_EVENT_RXF_FIFO_NOT_EMPTY */
	HwEventsDefaultHandler, 						/* HW_EVENT_TX_SELECTOR */ 
	HwEventsDefaultHandler, 						/* HW_EVENT_TXH_NTD_SW_REQ */
	HwEventsDefaultHandler, 						/* HW_EVENT_ALPHA_FILTERS_CALC_REQ_DONE_INDICATION */	
	HwEventsDefaultHandler, 						/* HW_EVENT_RX_SECURITY_COMPLETE */	
	HwEventsDefaultHandler, 						/* HW_EVENT_RX_AMSDU_PARSER_MSDU_DESC_NOT_EMPTY */
	HwEventsRxHaltIndication,						/* HW_EVENT_RX_HALT_INDICATION */
	HwEventsDefaultHandler, 						/* HW_EVENT_TX_DMA_MON_ACC_NON_EMPTY_IND */
	HwEventsDefaultHandler, 						/* HW_EVENT_DMA_WRRAPER_CH1_DONE_ACC_NOT_EMPTY */
	HwEventsDefaultHandler, 						/* HW_EVENT_TX_SECURITY_COMPLETE */
	HwEventsDefaultHandler, 						/* HW_EVENT_MAC_PHY_GP_IF_INDICATION */
	HwEventsDefaultHandler,							/* HW_EVENT_BAND_REL_A_RESERVED_17 */
	HwEventsDefaultHandler, 						/* HW_EVENT_DELIA_2_GENRISC_TX_END */
	HwEventsSender,									/* HW_EVENT_GENRISC_SENDER */
	HwEventsPreAggregator_Error, 					/* HW_EVENT_PRE_AGGREGATOR */
	HwEventsPacTsfTimer, 							/* HW_EVENT_TSF_TIMER */
	HwEventsRxHandler, 								/* HW_EVENT_GENRISC_RX_HANDLER */
	HwEventsDefaultHandler, 						/* HW_EVENT_SEQUENCER */
	HwEventsDefaultHandler, 						/* HW_EVENT_PAC_TICK_TIMER */
	HwEventsDefaultHandler, 						/* HW_EVENT_TX_DONE_LISTS */
	HwEventsDefaultHandler, 						/* HW_EVENT_RX_DONE_LISTS */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_LOW_PRI_RX_READY */
	HwEventsDefaultHandler, 						/* HW_EVENT_HIGH_PRI_TX_PD_READY */
	HwEventsDefaultHandler, 						/* HW_EVENT_HIGH_PRI_RX_PD_READY */
	HwEventsDefaultHandler,							/* HW_EVENT_Q_MANAGER_DESCRIPTOR_POOL */
};

/* The following HweventsHandler is defined for the 3 ARCs 
   The HW_EVENT_MASK_B for this Handler is different for the 2 LMACs and the UMAC */ 

const HwEventHandler_t HwEventsHandlers_B[HW_EVENT_BAND_REL_B_TOTAL_NUM] = 
{
	HwEventsBsrcFwDoneInterrupt,							/* HW_EVENT_BAND_REL_B_BSRC 							  */
	HwEventsBeamFormingErrorFifoNotEmpty, 					/* HW_EVENT_BAND_REL_B_BEAMFORMING_ERROR_FIFO_NOT_EMPTY   */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC0 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC1 				  */
	HwEventsDefaultHandler,									/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC0 					  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC1 					  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC  					  */
#ifdef ENET_INC_ARCH_WAVE600B
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RXC_RSSI_NOT_MY_STA_FIFO_NOT_EMPTY */
#else
    HwEventsDefaultHandler,                                 /* HW_EVENT_BAND_REL_B_RESERVED_8 */
#endif
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_9  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_10 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_11 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_12 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_13 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_14 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_15 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_16 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_17 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_18 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_19 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_20 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_21 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_22 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_23 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_24 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_25 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_26 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_27 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_28 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_29 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_30 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_31 */
};
#else // UMAC
const HwEventHandler_t HwEventsHandlers_A_Band0[HW_EVENT_BAND_REL_A_TOTAL_NUM] = 
{
	HwEventsDefaultHandler,									/* HW_EVENT_DELIA_2_GENRISC_TX_START */
	HwEventsDefaultHandler, 								/* HW_EVENT_SEQUENCER_RA_FILLER */
	HwEventsRxCoordinatorFrameClassViolation_Band0,			/* HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION */
	HwEventsDefaultHandler, 								/* HW_EVENT_BA_ANALYZER_STATUS_FIFO */
	HwEventsDefaultHandler,									/* HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY */ 
	HwEventsDefaultHandler,									/* HW_EVENT_PHY */ 
	HwEventsDefaultHandler,									/* HW_EVENT_RXF_FIFO_NOT_EMPTY */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_SELECTOR */ 
	HwEventsDefaultHandler, 								/* HW_EVENT_TXH_NTD_SW_REQ */
	HwEventsDefaultHandler, 								/* HW_EVENT_ALPHA_FILTERS_CALC_REQ_DONE_INDICATION */	
	HwEventsDefaultHandler, 								/* HW_EVENT_RX_SECURITY_COMPLETE */	
	HwEventsDefaultHandler,									/* HW_EVENT_RX_AMSDU_PARSER_MSDU_DESC_NOT_EMPTY */
	HwEventsDefaultHandler,									/* HW_EVENT_RX_HALT_INDICATION */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_DMA_MON_ACC_NON_EMPTY_IND */
	HwEventsDefaultHandler, 								/* HW_EVENT_DMA_WRRAPER_CH1_DONE_ACC_NOT_EMPTY */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_SECURITY_COMPLETE */
	HwEventsDefaultHandler, 								/* HW_EVENT_MAC_PHY_GP_IF_INDICATION */
	HwEventsDefaultHandler,									/* HW_EVENT_BAND_REL_A_RESERVED_17 */
	HwEventsDefaultHandler, 								/* HW_EVENT_DELIA_2_GENRISC_TX_END */
	HwEventsDefaultHandler, 								/* HW_EVENT_GENRISC_SENDER */
	HwEventsDefaultHandler, 								/* HW_EVENT_PRE_AGGREGATOR */
	HwEventsDefaultHandler,									/* HW_EVENT_TSF_TIMER */
	HwEventsDefaultHandler,									/* HW_EVENT_GENRISC_RX_HANDLER */
	HwEventsDefaultHandler, 								/* HW_EVENT_SEQUENCER */
	HwEventsDefaultHandler,									/* HW_EVENT_PAC_TICK_TIMER */
	HwEventsDefaultHandler,									/* HW_EVENT_TX_DONE_LISTS */
	HwEventsDefaultHandler, 								/* HW_EVENT_RX_DONE_LISTS */
	HwEventsDefaultHandler,									/* HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY */
	HwEventsDefaultHandler, 								/* HW_EVENT_Q_MANAGER_LOW_PRI_RX_READY */	
	HwEventsDefaultHandler,									/* HW_EVENT_HIGH_PRI_TX_PD_READY */
	HwEventsDefaultHandler,									/* HW_EVENT_HIGH_PRI_RX_PD_READY */
	HwEventsDefaultHandler, 								/* HW_EVENT_Q_MANAGER_DESCRIPTOR_POOL */
};

const HwEventHandler_t HwEventsHandlers_A_Band1[HW_EVENT_BAND_REL_A_TOTAL_NUM] = 
{
	HwEventsDefaultHandler,									/* HW_EVENT_RESERVED_0 */
	HwEventsDefaultHandler, 								/* HW_EVENT_SEQUENCER_RA_FILLER */
	HwEventsRxCoordinatorFrameClassViolation_Band1,			/* HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION */
	HwEventsDefaultHandler, 								/* HW_EVENT_BA_ANALYZER_STATUS_FIFO */
	HwEventsDefaultHandler, 								/* HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY */ 
	HwEventsDefaultHandler, 								/* HW_EVENT_PHY */ 
	HwEventsDefaultHandler, 								/* HW_EVENT_RXF_FIFO_NOT_EMPTY */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_SELECTOR */ 
	HwEventsDefaultHandler, 								/* HW_EVENT_TXH_NTD_SW_REQ */
	HwEventsDefaultHandler, 								/* HW_EVENT_ALPHA_FILTERS_CALC_REQ_DONE_INDICATION */	
	HwEventsDefaultHandler, 								/* HW_EVENT_RX_SECURITY_COMPLETE */	
	HwEventsDefaultHandler,									/* HW_EVENT_RX_AMSDU_PARSER_MSDU_DESC_NOT_EMPTY */
	HwEventsDefaultHandler, 								/* HW_EVENT_RX_HALT_INDICATION */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_DMA_MON_ACC_NON_EMPTY_IND */
	HwEventsDefaultHandler, 								/* HW_EVENT_DMA_WRRAPER_CH1_DONE_ACC_NOT_EMPTY */
	HwEventsDefaultHandler, 								/* HW_EVENT_TX_SECURITY_COMPLETE */
	HwEventsDefaultHandler, 								/* HW_EVENT_MAC_PHY_GP_IF_INDICATION */
	HwEventsDefaultHandler,									/* HW_EVENT_BAND_REL_A_RESERVED_17 */
	HwEventsDefaultHandler, 								/* HW_EVENT_DELIA_2_GENRISC_TX_END */
	HwEventsDefaultHandler, 								/* HW_EVENT_GENRISC_SENDER */
	HwEventsDefaultHandler, 								/* HW_EVENT_PRE_AGGREGATOR */
	HwEventsDefaultHandler, 								/* HW_EVENT_TSF_TIMER */
	HwEventsDefaultHandler, 								/* HW_EVENT_GENRISC_RX_HANDLER */
	HwEventsDefaultHandler, 								/* HW_EVENT_SEQUENCER */
	HwEventsDefaultHandler,									/* HW_EVENT_PAC_TICK_TIMER */
	HwEventsDefaultHandler,									/* HW_EVENT_TX_DONE_LISTS */
	HwEventsDefaultHandler, 								/* HW_EVENT_RX_DONE_LISTS */
	HwEventsDefaultHandler,									/* HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY */
	HwEventsDefaultHandler, 								/* HW_EVENT_Q_MANAGER_LOW_PRI_RX_READY */		
	HwEventsDefaultHandler, 								/* HW_EVENT_HIGH_PRI_TX_PD_READY */ 
	HwEventsDefaultHandler,									/* HW_EVENT_HIGH_PRI_RX_PD_READY */	
	HwEventsDefaultHandler, 								/* HW_EVENT_Q_MANAGER_DESCRIPTOR_POOL */
};

const HwEventHandler_t HwEventsHandlers_B_Band0[HW_EVENT_BAND_REL_B_TOTAL_NUM] = 
{
	HwEventsDefaultHandler,									/* HW_EVENT_BAND_REL_B_BSRC 							  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_BEAMFORMING_ERROR_FIFO_NOT_EMPTY   */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC0 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC1 				  */
	HwEventsPlanLocker_Band0,								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC0 					  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC1 					  */
	HwEventsTxSelectorGroupLockerReqServiced_Band0, 		/* HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC  					  */
#ifdef ENET_INC_ARCH_WAVE600B
    HwEventsDefaultHandler,                                 /* HW_EVENT_BAND_REL_B_RXC_RSSI_NOT_MY_STA_FIFO_NOT_EMPTY */
#else
    HwEventsDefaultHandler,                                 /* HW_EVENT_BAND_REL_B_RESERVED_8 */
#endif
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_9  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_10 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_11 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_12 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_13 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_14 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_15 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_16 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_17 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_18 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_19 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_20 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_21 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_22 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_23 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_24 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_25 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_26 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_27 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_28 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_29 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_30 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_31 */
};
const HwEventHandler_t HwEventsHandlers_B_Band1[HW_EVENT_BAND_REL_B_TOTAL_NUM] = 
{
	HwEventsDefaultHandler,									/* HW_EVENT_BAND_REL_B_BSRC 							  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_BEAMFORMING_ERROR_FIFO_NOT_EMPTY   */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC0 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_LMAC1 				  */
	HwEventsPlanLocker_Band1, 								/* HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC 				  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC0 					  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_MU_LOCKER_LMAC1 					  */
	HwEventsTxSelectorGroupLockerReqServiced_Band1, 		/* HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC  					  */
#ifdef ENET_INC_ARCH_WAVE600B
    HwEventsDefaultHandler,                                 /* HW_EVENT_BAND_REL_B_RXC_RSSI_NOT_MY_STA_FIFO_NOT_EMPTY */
#else
    HwEventsDefaultHandler,                                 /* HW_EVENT_BAND_REL_B_RESERVED_8 */
#endif
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_9  */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_10 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_11 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_12 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_13 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_14 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_15 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_16 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_17 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_18 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_19 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_20 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_21 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_22 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_23 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_24 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_25 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_26 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_27 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_28 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_29 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_30 */
	HwEventsDefaultHandler, 								/* HW_EVENT_BAND_REL_B_RESERVED_31 */
};

const HwEventHandler_t HwEventsHandlersCommon_A[HW_EVENT_COMMON_TOTAL_NUM] = 
{
	HwEventsDefaultHandler, 						/* HW_EVENT_RESERVED_0 */
	HwEventsRxPpFragmentFifoNotEmpty,				/* HW_EVENT_RXPP_FRAGMENT_FIFO_NOT_EMPTY */
	HwEventsPdThresholdReached,						/* HW_EVENT_PD_THRESHOLD_REACHED */
	HwEventsDefaultHandler, 						/* HW_EVENT_RD_THRESHOLD_REACHED */
	HwEventsDefaultHandler,							/* HW_EVENT_EXTERNAL_IRQ */
#ifdef ATOMIC_COUNTER_ENABLER
	HwEventsDefaultHandler,							/* if atomic counters are available we disalbe this interrupt (HW_EVENT_RAB_NPU_TO_UPI_IRQ) */
#else
	HwEventsRabNpuToUpiIrq, 						/* HW_EVENT_RAB_NPU_TO_UPI_IRQ */
#endif
	isr_TxManager_DisableListNotEmpty, 				/* HW_EVENT_Q_MANAGER_DISABLE_LIST_NON_EMPTY_UMAC */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_DISABLE_LIST_NON_EMPTY_LMAC0 */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_DISABLE_LIST_NON_EMPTY_LMAC1 */
	HwEventsAgerSingleCheckDone,					/* HW_EVENT_AGER_SINGLE_CHECK_DONE */
	SharedRamInt_ISR,								/* HW_EVENT_INT_SHRAM_ADDR */
	HwEventsDefaultHandler, 						/* HW_EVENT_EXTERNAL_IRQ_GPIO_1 */
	hwEventsPtaGpio0, 								/* HW_EVENT_EXTERNAL_IRQ_GPIO_10 */
	HwEventsDefaultHandler, 						/* HW_EVENT_UART_LMAC0 */ 
	HwEventsDefaultHandler, 						/* HW_EVENT_UART_LMAC1 */ 
	HwEventsDefaultHandler, 						/* HW_EVENT_UART_UMAC */
	HwEventsDefaultHandler, 						/* HW_EVENT_PROTECTED_DB_LMAC0 */
	HwEventsDefaultHandler, 						/* HW_EVENT_PROTECTED_DB_LMAC1 */ 
	HwEventsDefaultHandler, 						/* HW_EVENT_PROTECTED_DB_UMAC */
	HwEventsDefaultHandler, 						/* HW_EVENT_EXTERNAL_IRQ_GPIO_11 */
#ifdef ENET_INC_ARCH_WAVE600B
	HwEventsApbTimeout, 							/* HW_EVENT_APB_TIMEOUT */
#else
    HwEventsDefaultHandler,                         /* HW_EVENT_RESERVED_20 */
#endif	
	HwEventsDefaultHandler,							/* HW_EVENT_PCIe_0 */
	HwEventsDefaultHandler,							/* HW_EVENT_PCIe_1 */
	HwEventsHostIFGenRisc,							/* HW_EVENT_HOST_IF_GENRISC */
	HwEventsDefaultHandler, 						/* HW_EVENT_RESERVED_24 */
	HwEventsHwQueueManagerTxDoneList,				/* HW_EVENT_TX_DONE_LISTS_UMAC */
	HwEventsDefaultHandler,							/* HW_EVENT_RX_DONE_LISTS_UMAC */
	HwEventsHwQueueManagerLowTxReady,				/* HW_EVENT_Q_MANAGER_LOW_PRI_TX_READY_UMAC */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_LOW_PRI_RX_READY_UMAC */	
	HwEventsTxPacketsClassifier_ReadyPds,			/* HW_EVENT_HIGH_PRI_TX_PD_READY_UMAC */
	HwEventsRxManager_RdReady, 						/* HW_EVENT_HIGH_PRI_RX_PD_READY_UMAC */
	HwEventsDefaultHandler, 						/* HW_EVENT_Q_MANAGER_DESCRIPTOR_POOL_UMAC */
};

#endif

/*---------------------------------------------------------------------------------
/								Debug								
/----------------------------------------------------------------------------------*/
#if defined(ENET_INC_UMAC) || defined (ENET_INC_LMAC0)
uint32 HwEventsStat_A_Band0[HW_EVENT_BAND_REL_A_TOTAL_NUM];
uint32 HwEventsStat_B_Band0[HW_EVENT_BAND_REL_B_TOTAL_NUM];
#endif
#if defined(ENET_INC_UMAC) || defined (ENET_INC_LMAC1)
uint32 HwEventsStat_A_Band1[HW_EVENT_BAND_REL_A_TOTAL_NUM];
uint32 HwEventsStat_B_Band1[HW_EVENT_BAND_REL_B_TOTAL_NUM];
#endif
#if defined(ENET_INC_UMAC)
uint32 HwEventsStat_Common_A[HW_EVENT_COMMON_TOTAL_NUM];
#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

HwEventsDefaultHandler 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsDefaultHandler(void)
{
	FATAL("HwEventsDefaultHandler");
}

#if defined(ENET_INC_UMAC)
/**********************************************************************************

HwEventsRxCoordinatorFrameClassViolation_Band0 

Description:
------------
	Frame Class violation event from Rx Coordinator
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsRxCoordinatorFrameClassViolation_Band0(void) 
{
	// turn off the event
	HwEvents_MaskEvent_A_Band0(HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CLASS_VIOLATION_EVENT, TASK_RX_MANAGER, ConfigurationManager_GetFirstVapForBand0());	

}
/**********************************************************************************

HwEventsRxCoordinatorFrameClassViolation_Band1 

Description:
------------
	Frame Class violation event from Rx Coordinator
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsRxCoordinatorFrameClassViolation_Band1(void) 
{
	// turn off the event
	HwEvents_MaskEvent_A_Band1(HW_EVENT_RX_COORDINATOR_FRAME_CLASS_VIOLATION);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CLASS_VIOLATION_EVENT, TASK_RX_MANAGER, ConfigurationManager_GetFirstVapForBand1());	
}

/**********************************************************************************

HwEventsRxPpFragmentFifoNotEmpty 

Description:
------------
	Frame Class violation event from Rx Classifier
Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsRxPpFragmentFifoNotEmpty(void)
{
	// turn off the event
	EventManager_TurnOffEvent(EVENT_ID_FRAGMENT_FIFO_NOT_EMPTY);
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_FRAGMENTATION_EVENT, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);	
}
#endif

#if defined(ENET_INC_LMAC)
/**********************************************************************************

HwEventsBaAnalyzer_StatusFifoNotEmpty 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsBaAnalyzer_StatusFifoNotEmpty(void)
{
	// In wave600 this event is in the LM and in wave500 it's in UM.

	
	// turn off the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_MaskEvent_A_Band0(HW_EVENT_BA_ANALYZER_STATUS_FIFO);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_BA_ANALYZER_STATUS_FIFO);
#endif		
			
	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_STATUS_FIFO_NOT_EMPTY, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);



}

/**********************************************************************************

HwEventsPsSettingFifoNotEmpty 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsPsSettingFifoNotEmpty(void)
{
	// In wave600 this event is in the LM and in wave500 it's in UM.
	// So - in wave600 we send a MSG to PAC MANAGER and in wave500 we send a MSG to TX MANAGER
	
	// turn off the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_MaskEvent_A_Band0(HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_PS_SETTING_FIFO_NOT_EMPTY);
#endif

	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_PS_SETTINGS_FIFO_NOT_EMPTY, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

HwEventsSender 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsSender(void)
{	
	// turn off the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_MaskEvent_A_Band0(HW_EVENT_GENRISC_SENDER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_GENRISC_SENDER);
#endif

	isr_Sender();

	// turn on the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_UnMaskEvent_A_Band0(HW_EVENT_GENRISC_SENDER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_UnMaskEvent_A_Band1(HW_EVENT_GENRISC_SENDER);
#endif
}

/**********************************************************************************

HwEventsRxHandler 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsRxHandler(void)
{
	// turn off the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_MaskEvent_A_Band0(HW_EVENT_GENRISC_RX_HANDLER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_GENRISC_RX_HANDLER);
#endif

	isr_RxHandler();

	// turn on the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_UnMaskEvent_A_Band0(HW_EVENT_GENRISC_RX_HANDLER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_UnMaskEvent_A_Band1(HW_EVENT_GENRISC_RX_HANDLER);
#endif
}
#endif // ENET_INC_LMAC

#if defined(ENET_INC_UMAC)
/**********************************************************************************

HwEventsHostIFGenRisc 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsHostIFGenRisc(void)
{	
	// turn off the event
	HwEvents_MaskEvent_Common_A(HW_EVENT_HOST_IF_GENRISC);

	isr_HostGenRisc();

	// turn on the event
	HwEvents_UnMaskEvent_Common_A(HW_EVENT_HOST_IF_GENRISC);
}
#endif // ENET_INC_UMAC

#if defined(ENET_INC_LMAC)
/**********************************************************************************

HwEventsPacTsfTimer 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsPacTsfTimer(void)
{
// turn off the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_MaskEvent_A_Band0(HW_EVENT_TSF_TIMER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_TSF_TIMER);
#endif
	// call interrupt routine
	vLM_pac_tsf_timer_int();

	// turn on the event
#if defined (ENET_INC_LMAC0)	
	HwEvents_UnMaskEvent_A_Band0(HW_EVENT_TSF_TIMER);
#elif defined (ENET_INC_LMAC1)
	HwEvents_UnMaskEvent_A_Band1(HW_EVENT_TSF_TIMER);
#endif
}
#endif // ENET_INC_LMAC


#if defined(ENET_INC_LMAC)
/**********************************************************************************

HwEventsRxfFifoNotEmpty 

Description:
------------
Handles the event that the Rxf FIFO is not empty

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsRxfFifoNotEmpty(void)
{	
	// In wave600 this event is in the LM and in wave500 it's in UM.
	// So - in wave600 we send a MSG to PAC MANAGER and in wave500 we send a MSG to TS MANAGER


#if defined(ENET_INC_LMAC0)
	HwEvents_MaskEvent_A_Band0(HW_EVENT_RXF_FIFO_NOT_EMPTY);
#elif defined (ENET_INC_LMAC1)
	HwEvents_MaskEvent_A_Band1(HW_EVENT_RXF_FIFO_NOT_EMPTY);
#endif

	OSAL_SEND_NO_DATA_MESSAGE(PAC_MANAGER_GET_AGG_WITH_NO_BA_FIFO, TASK_PAC_MANAGER, VAP_ID_DO_NOT_CARE);	



}
#endif // ENET_INC_LMAC

#ifdef ENET_INC_UMAC 
/**********************************************************************************

HwEventsHwQueueManagerLowTxReady

Description:
------------
Handles the event that the one of the Tx low priority ready list became not empty

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsHwQueueManagerLowTxReady(void)
{
	K_MSG *qosMessage = NULL;
	EventManager_TurnOffEvent(EVENT_ID_TX_LIST_READY_PDS_LOW_PRI_NOT_EMPTY);

	//TODO: should pass the BandId in the message when BAND1 will be working 
	qosMessage = OSAL_GET_MESSAGE(K_NO_DATA);	
	OSAL_SEND_MESSAGE(QOS_AGER_LIST_NOT_EMPTY, TASK_QOS, qosMessage, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

HwEventsHwQueueManagerTxDoneList 


Description:
------------
This function handles interrupt from the HW queue manager when one of the done list
became not empty

Input: 
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void HwEventsHwQueueManagerTxDoneList(void)
{ 
	K_MSG *txManagerMessage = NULL;

	/* Mask all not empty interrupts in HW event */
	EventManager_TurnOffEvent(EVENT_ID_TX_DONE_LISTS);

	txManagerMessage = OSAL_GET_MESSAGE(K_NO_DATA);	

	OSAL_SEND_MESSAGE(TX_MANAGER_DONE_LIST_NOT_EMPTY, TASK_TX_MANAGER, txManagerMessage, VAP_ID_DO_NOT_CARE);	
}

/**********************************************************************************

HwEventsPdThresholdReached 

Description:
------------
Handles the event that the number of free PDs has reached to the threshold

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsPdThresholdReached(void)
{	
	K_MSG *qosMessage = NULL;


	EventManager_TurnOffEvent(EVENT_ID_PD_THRESHOLD_REACHED);
	qosMessage = OSAL_GET_MESSAGE(K_NO_DATA);	
	OSAL_SEND_MESSAGE(QOS_NUMBER_OF_FREE_PDS_REACHED_THRESHOLD, TASK_QOS, qosMessage, VAP_ID_DO_NOT_CARE);	
}
#endif /* ENET_INC_UMAC */	


#ifdef ENET_INC_LMAC
/**********************************************************************************

HwEventsPhy 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsPhy(void)
{
    /* No need to turn off event since it is already interrupt context */
	isr_PhyDrv_PhyInterrupt();
}
#endif 


#ifdef ENET_INC_UMAC
/**********************************************************************************

hwEventsPtaGpio0 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void hwEventsPtaGpio0(void)
{
#ifdef PTA_BUILD_IN_PLAT
	Isr_PtaGpio0();
#endif
}

#ifndef ATOMIC_COUNTER_ENABLER
/**********************************************************************************

HwEventsRabNpuToUpiIrq 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsRabNpuToUpiIrq(void)
{
	vHIM_HostISR();
}
#endif // ATOMIC_COUNTER_ENABLER

#ifdef ENET_INC_ARCH_WAVE600B

/**********************************************************************************

HwEventsApbTimeout 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
#define	SIZE_OF_APB_DBG_INFO 10
uint32 apbTimouInfo[SIZE_OF_APB_DBG_INFO]= {0};
static void HwEventsApbTimeout(void)
{
	memcpy32(&apbTimouInfo[0],(uint32 *)REG_MAC_GENERAL_UCPU_TIMEOUT_ADDR_L, SIZE_OF_APB_DBG_INFO);
	
	FATAL("HwEventsApbTimeout");	
}

#endif //#ifdef ENET_INC_ARCH_WAVE600B

#endif 





#if defined (ENET_INC_LMAC)
/**********************************************************************************

HwEventsRxHaltIndication

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsRxHaltIndication(void)
{
	// this event is in UM for gen5b and in LM for gen6
	isr_TxPauser_RxPauseExecuted();

#if defined(ENET_INC_LMAC0)
	HwEvents_UnMaskEvent_A_Band0(HW_EVENT_RX_HALT_INDICATION);
#elif defined (ENET_INC_LMAC1)
	HwEvents_UnMaskEvent_A_Band1(HW_EVENT_RX_HALT_INDICATION);
#endif

}
#endif //ENET_INC_LMAC

#ifdef ENET_INC_UMAC
/**********************************************************************************

HwEventsAgerSingleCheckDone 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
static void HwEventsAgerSingleCheckDone(void)
{
    HwQManagerAger_ClearSingleCheckDoneInterrupt();
	OSAL_SEND_NO_DATA_MESSAGE(QOS_SINGLE_CHECK_DONE, TASK_QOS, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

HwEventsTxSelectorGroupLockerReqServiced_Band0 

Description:
------------
Called after group lock done 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsTxSelectorGroupLockerReqServiced_Band0(void)
{
	HwEvents_MaskEvent_B_Band0(HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC);
	isr_TxSelector_LockerGroupRequestServiced(CONFIGURATION_MANAGER_BAND_0);
	HwEvents_UnMaskEvent_B_Band0(HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC);
}

/**********************************************************************************

HwEventsTxSelectorGroupLockerReqServiced_Band1 

Description:
------------
Called after group lock done 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsTxSelectorGroupLockerReqServiced_Band1(void)
{
	HwEvents_MaskEvent_B_Band1(HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC);
	isr_TxSelector_LockerGroupRequestServiced(CONFIGURATION_MANAGER_BAND_1);
	HwEvents_UnMaskEvent_B_Band1(HW_EVENT_BAND_REL_B_MU_LOCKER_UMAC);

}
#endif // ENET_INC_UMAC

#if defined(ENET_INC_LMAC)
/**********************************************************************************

HwEventsBsrcFwDoneInterrupt 

Description:
------------
Called when FW indirect access has completed
this option is not used in our design so an ASSERT is issued

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsBsrcFwDoneInterrupt(void)
{
	// Shouldn't get here because this option is disabled in BSRC init
	ASSERT(0);
}
#endif


/**********************************************************************************

HwEventsBeamFormingErrorFifoNotEmpty 

Description:
------------
Called after BeamForming Error fifo is not empty 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
#ifdef ENET_INC_LMAC
static void HwEventsBeamFormingErrorFifoNotEmpty(void)
{
	HwMbfm_ErrorFifoHandler();
}
#endif //ENET_INC_LMAC


#if defined(ENET_INC_UMAC) 
/**********************************************************************************

HwEventsPlanLocker_Band0 

Description:
------------
Called after Plan Locker Band0 interrupt was received 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsPlanLocker_Band0(void)
{
	HwEvents_MaskEvent_B_Band0(HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC);
	hwEventsTxSelectorLockerPlanRequestServiced(CONFIGURATION_MANAGER_BAND_0);
	HwEvents_UnMaskEvent_B_Band0(HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC);
}

/**********************************************************************************

HwEventsPlanLocker_Band1 

Description:
------------
Called after Plan Locker Band0 interrupt was received 

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/

static void HwEventsPlanLocker_Band1(void)
{ 
	HwEvents_MaskEvent_B_Band1(HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC);
	hwEventsTxSelectorLockerPlanRequestServiced(CONFIGURATION_MANAGER_BAND_1);
	HwEvents_UnMaskEvent_B_Band1(HW_EVENT_BAND_REL_B_PLAN_LOCKER_UMAC);
}

#endif


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

HwEvents_MaskEvent_A_Band0 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent_A_Band0(HwEventIdBandRelA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents0AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS0A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_MaskEvent_A_Band1 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent_A_Band1(HwEventIdBandRelA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents1AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS1A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


/**********************************************************************************

HwEvents_MaskEvent_B_Band0 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent_B_Band0(HwEventIdBandRelB_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents0BMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS0B_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_MaskEvent_B_Band1 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent_B_Band1(HwEventIdBandRelB_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents1BMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS1B_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_MaskEvent_Common_A 

Description:
------------
	mask Specific Hw event interrupt

Input:
-----
	hwEvent - The Hw Event to be Masked
Output:
-------
	None

**********************************************************************************/
void HwEvents_MaskEvent_Common_A(HwEventIdCommonA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents2AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = 0;
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS2A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_UnMaskEvent_A_Band0

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent_A_Band0(HwEventIdBandRelA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents0AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS0A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_UnMaskEvent_A_Band1 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent_A_Band1(HwEventIdBandRelA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents1AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS1A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


/**********************************************************************************

HwEvents_UnMaskEvent_B_Band0 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent_B_Band0(HwEventIdBandRelB_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents0BMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS0B_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_UnMaskEvent_B_Band1 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent_B_Band1(HwEventIdBandRelB_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents1BMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS1B_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************

HwEvents_UnMaskEvent_Common_A 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_UnMaskEvent_Common_A(HwEventIdCommonA_e hwEvent)
{
	TX_INTERRUPT_SAVE_AREA;
	RegEmeraldEnvHwEvents2AMaskedInterrupts_u hwEventsIntMaskStatusReg;
	uint32 maskVal;

	hwEventsIntMaskStatusReg.val = (1 << hwEvent);
	maskVal = (1 << hwEvent);

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	RegAccess_WriteMasked(REG_EMERALD_ENV_HW_EVENTS2A_INT_EN,maskVal,hwEventsIntMaskStatusReg.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}



/**********************************************************************************

isr_HwEvents_A_Band0 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents_A_Band0(void)
{
#ifndef ENET_INC_LMAC1

	RegEmeraldEnvHwEvents0AMaskedInterrupts_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS0A_MASKED_INTERRUPTS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat_A_Band0[eventIndex]++;
#if defined(ENET_INC_LMAC)
		HwEventsHandlers_A[eventIndex]();
#else
		HwEventsHandlers_A_Band0[eventIndex]();
#endif		
	}
#else
		FATAL("LMAC1 doesn't handle Events of Band0");
#endif // !(ENET_INC_LMAC1)	
	
}

/**********************************************************************************

isr_HwEvents_A_Band1 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents_A_Band1(void)
{
#ifndef ENET_INC_LMAC0
	RegEmeraldEnvHwEvents1AMaskedInterrupts_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS1A_MASKED_INTERRUPTS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat_A_Band1[eventIndex]++;	
#if defined(ENET_INC_LMAC)
		HwEventsHandlers_A[eventIndex]();
#else
		HwEventsHandlers_A_Band1[eventIndex]();
#endif		
	}
#else
	FATAL("LMAC0 doesn't handle Events of Band1");
#endif // !(ENET_INC_LMAC0)	
}

/**********************************************************************************

isr_HwEvents_B_Band0 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents_B_Band0(void)
{
#ifndef ENET_INC_LMAC1

	RegEmeraldEnvHwEvents0BMaskedInterrupts_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS0B_MASKED_INTERRUPTS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat_B_Band0[eventIndex]++;	
#if defined(ENET_INC_LMAC)
		HwEventsHandlers_B[eventIndex]();
#else
		HwEventsHandlers_B_Band0[eventIndex]();
#endif		
	}
#else
			FATAL("LMAC1 doesn't handle Events of Band0");
#endif // !(ENET_INC_LMAC1)	
	
}

/**********************************************************************************

isr_HwEvents_B_Band1 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents_B_Band1(void)
{
#ifndef ENET_INC_LMAC0

	RegEmeraldEnvHwEvents1BMaskedInterrupts_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS1B_MASKED_INTERRUPTS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat_B_Band1[eventIndex]++;	

#if defined(ENET_INC_LMAC)
		HwEventsHandlers_B[eventIndex]();
#else
		HwEventsHandlers_B_Band1[eventIndex]();
#endif		
	}
#else
				FATAL("LMAC0 doesn't handle Events of Band1");
#endif // !(ENET_INC_LMAC1)	
	
}


/**********************************************************************************

isr_HwEvents_Common_A 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
ISR_VOID isr_HwEvents_Common_A(void)
{
#if defined(ENET_INC_UMAC)

	RegEmeraldEnvHwEvents2AMaskedInterrupts_u hwEventInsStatusReg;
	uint32 eventIndex;
	ACCUMULATE_CPU_IDLE_TIME();
	RegAccess_Read(REG_EMERALD_ENV_HW_EVENTS2A_MASKED_INTERRUPTS, &hwEventInsStatusReg.val);

	while(hwEventInsStatusReg.val != 0)
	{
		eventIndex = Utils_FindFirstSetAndClear(&hwEventInsStatusReg.val);

		HwEventsStat_Common_A[eventIndex]++;	
		HwEventsHandlersCommon_A[eventIndex]();
	}
#endif // ENET_INC_UMAC	
}


/**********************************************************************************

HwEvents_Init 

Description:
------------

Input:
-----
	None
Output:
-------
	None

**********************************************************************************/
void HwEvents_Init(void)
{
#ifdef ENET_INC_LMAC0	
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS0A_INT_EN, HW_EVENTS_ENABLE_MASK_A);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS0B_INT_EN, HW_EVENTS_ENABLE_MASK_B);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS2A_INT_EN, HW_EVENTS_ENABLE_MASK_COMMON_A);
	memset32(&HwEventsStat_A_Band0[0],0,HW_EVENT_BAND_REL_A_TOTAL_NUM);	
	memset32(&HwEventsStat_B_Band0[0],0,HW_EVENT_BAND_REL_B_TOTAL_NUM);
#elif  defined (ENET_INC_LMAC1)
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS1A_INT_EN, HW_EVENTS_ENABLE_MASK_A);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS1B_INT_EN, HW_EVENTS_ENABLE_MASK_B);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS2A_INT_EN, HW_EVENTS_ENABLE_MASK_COMMON_A);
	memset32(&HwEventsStat_A_Band1[0],0,HW_EVENT_BAND_REL_A_TOTAL_NUM);	
	memset32(&HwEventsStat_B_Band1[0],0,HW_EVENT_BAND_REL_B_TOTAL_NUM);
#elif defined (ENET_INC_UMAC)
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS0A_INT_EN, HW_EVENTS_ENABLE_MASK_A);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS1A_INT_EN, HW_EVENTS_ENABLE_MASK_A);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS2A_INT_EN, HW_EVENTS_ENABLE_MASK_COMMON_A);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS0B_INT_EN, HW_EVENTS_ENABLE_MASK_B);
	RegAccess_Write(REG_EMERALD_ENV_HW_EVENTS1B_INT_EN, HW_EVENTS_ENABLE_MASK_B);
	memset32(&HwEventsStat_A_Band0[0],0,HW_EVENT_BAND_REL_A_TOTAL_NUM);	
	memset32(&HwEventsStat_A_Band1[0],0,HW_EVENT_BAND_REL_A_TOTAL_NUM);	
	memset32(&HwEventsStat_Common_A[0],0,HW_EVENT_COMMON_TOTAL_NUM);
	memset32(&HwEventsStat_B_Band0[0],0,HW_EVENT_BAND_REL_B_TOTAL_NUM);
	memset32(&HwEventsStat_B_Band1[0],0,HW_EVENT_BAND_REL_B_TOTAL_NUM);
#endif
}


