/***************************************************************
 File:		HwLogger.c
 Module:	  	HwLogger
 Purpose: 	holds all the implementation of the HwLogger API functions
 Description:  This module contains the implementation of the HwLogger driver module 

***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "ErrorHandler_Api.h"
#include "mhi_umi.h"
#include "LoggerRegs.h"
#include "shram_hw_logger.h"
#include "ShramRxDescriptors.h"
#include "HwLoggerAPI.h"
#include "HwLoggerRam.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "OSAL_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "HwLoggerLogsDefinitions.h"
#include "Dut_Api.h"
#include "init_ifmsg.h"
#include "shram_man_msgs.h"
#include "PsManager_Api.h"
#include "HostIfAccRegs.h"
#include "BaAnalyzerRegs.h"
#include "AggBuilderRegs.h"
#include "RxPpRegs.h"
#include "TxHandlerRegs.h"
#include "TxSequencerRegs.h"
#include "RxClassifierRegs.h"
#include "MacBfRegs.h"
#include "BeaconRegs.h"
#include "TxSelectorRegs.h"
#include "MacHtExtensionsRegs.h"
#include "MacWepRegs.h"
#include "PacDelRegs.h"
#include "PacRxdRegs.h"
#include "QManagerUmacRegs.h"
#include "PacDelRegs.h"
#include "HwLoggerPhyGenRiscLogDefs.h"
#include "PsSettingRegs.h"
#include "PhyRxTdRegs.h"
#include "MacWep1Regs.h"
#include "MacWep2Regs.h"
#include "MacWep3Regs.h"
#include "PacDel1Regs.h"
#include "PacDel2Regs.h"
#include "PacDel3Regs.h"
#include "TxPdAccRegs.h"
#include "Pac_Api.h"
#include "ShramHim.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 6


#define MAX_LOGGER_STREAM_NUM 				1 

#define OFFSET_OF_LOGGER_HEADER_EXT_INFO	1
#define NUM_OF_LOGGER_FIFOS					12


#define HW_LOGGER_UM_STREAM_ID				0 
#define HW_LOGGER_LM_STREAM_ID				0

#define BUFFER_THRESH_BYTES_NUM				2
#define STREAM_ENABLE						1

#define LOGGER_UM_FIFO_ID					0
#define LOGGER_LM_FIFO_ID					1
#define LOGGER_PHY_GENRISC_FIFO_ID			5
#define LOGGER_PHY_HW_FIFO_ID				6

#if defined (ENET_INC_UMAC)
#define LOGGER_MIPS_OID 					LOGGER_UM_WLAN_IF
#define LOGGER_FAST_DATA_IN_REG 			REG_LOGGER_UMAC_FAST_MES_DATA_IN

#define LOGGER_FILTER_TYPE_LOG_LEVEL			0x0 
#define LOGGER_FILTER_TYPE_FIFO					0x1
#define LOGGER_DATA_IN_PTR_MAX_LOG_LEVEL		0x7

#define LOGGER_HDR_START_TRIGGER_TYPE 			0x0
#define LOGGER_HDR_STOP_TRIGGER_TYPE 			0x1

#define LOGGER_FIRST_HW_SOURCE_INDEX 			0x7
#define LOGGER_DEFAULT_PRIORITY 				0x7
#define LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD	1000 // 100 milisecond threshold
#define LOGGER_BACKGROUND_BUFFER_TIME_THRESHOLD	1000000 // ~inf

#define LOGGER_NUM_OF_HW_FIFO                   6
#define LOGGER_NUM_OF_OPTIONALS_HW_BLOCKS_PER_FIFO 8



#else 
#define LOGGER_MIPS_OID 					LOGGER_LM_WLAN_IF //The Hw fifo's OID is 0, at the old logger - the LM is zero - need to arange oid allocation 
#define LOGGER_FAST_DATA_IN_REG 			REG_LOGGER_LMAC_FAST_MES_DATA_IN

#endif 
#define ENABLE_GENRISC_LOGS_ADDRESS 0x2000C
#define LOGGER_MAX_WAIT_ON_FIFO_FLUSH (0x1000) //uSec
//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------
#define STREAM_0	0 
#define STREAM_1	1 
#define STREAM_2	2 
#define STREAM_3	3 

#define THREE_LS_BYTES_IN_HALF_WORD	0x0FFF
#define THREE_LS_BYTES_IN_WORD		0x00000FFF



typedef union
{
	uint16 val;
	struct
	{
		uint16 BE0:1; 
		uint16 logVersion:4;
		uint16 priority:4; 
		uint16 BE1:1; 
		uint16 wlanIf:2; 
		uint16 destination:2; 
		uint16	reserved:2;  
	}bitFields;
}HwLoggeFifoMessage_u; 

typedef struct HwLogger_StreamsSharedParameters 
{

	uint32 bufferThreshold:16;
	uint32 assignedFifosId:16; /*12 logger fifos */  
	uint32 logVersion:4;
	uint32 be0:1; 
	uint32 be1:1; 
	uint32 streamEnable:1;
	uint32 streamId:2;  
	uint32 wlanIf:2; 
	uint32 reserved:21;
} HwLogger_StreamsSharedParameters_t;

typedef struct HwLogger_SharedParameters 
{
	uint32 shramBufferSize:16;
	uint32 maxNumberOfStreams:4;
	uint32 numberOfActiveStream:4; 
	uint32 reserved:8;
	
} HwLogger_SharedParameters_t;


void hwLoggerEnableDisableHostAccTxInLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccTxOutLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccRxInLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccRxOutLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableAggBuilderAndBaAnalyzerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxLiberatorLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelineatorLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxPpLogs(bool shortLogEnable , bool longLogEnable);
void  hwLoggerEnableDisableTxHLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxLiberatorLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSequencerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxCLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePsManagerLogs(bool shortLogEnable, bool longLogEnable);

void hwLoggerEnableDisableDeliaLogs(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableDelia1Logs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia2Logs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia3Logs(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableQmAgerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxListsLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxDataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxListsLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxDataLogs(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableTxSelectorOrLockerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSelectorBitmapLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxClassLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableSecurityLogs(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableSecurity1Logs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableSecurity2Logs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableSecurity3Logs(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableBeaconLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableActivityLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeamFormLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableAggBuilderLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBaAnlayzerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableUnknown(bool shortLogEnable , bool longLogEnable);

void hwLoggerEnableDisableTxPdAccLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuLockerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePacExtraLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuBfLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuBitmapSelectLogs(bool shortLogEnable , bool longLogEnable);



typedef void (*hwBlocksLogHandler)(bool , bool);
static const hwBlocksLogHandler hwLogsHandler[LOGGER_NUM_OF_HW_FIFO][LOGGER_NUM_OF_OPTIONALS_HW_BLOCKS_PER_FIFO] = 
{
	{hwLoggerEnableDisableHostAccTxInLogs			  	, hwLoggerEnableDisableDelineatorLogs 			, hwLoggerEnableDisableRxCLogs				, hwLoggerEnableDisableDeliaLogs	, hwLoggerEnableDisableQmAgerLogs			, hwLoggerEnableDisableQmRxListsLogs	, hwLoggerEnableDisableSecurity2Logs , hwLoggerEnableDisableUnknown},
	{hwLoggerEnableDisableHostAccTxOutLogs 		  		, hwLoggerEnableDisableRxPpLogs					, hwLoggerEnableDisableTxSelectorBitmapLogs , hwLoggerEnableDisableSecurityLogs , hwLoggerEnableDisableActivityLogs 		, hwLoggerEnableDisableTxPdAccLogs		, hwLoggerEnableDisableDelia1Logs 	 , hwLoggerEnableDisableUnknown},
	{hwLoggerEnableDisableHostAccRxInLogs			  	, hwLoggerEnableDisableTxHLogs					, hwLoggerEnableDisableRxCLogs				, hwLoggerEnableDisableMuLockerLogs	, hwLoggerEnableDisableRxClassLogs			, hwLoggerEnableDisableAggBuilderLogs	, hwLoggerEnableDisableSecurity1Logs , hwLoggerEnableDisableUnknown},
	{hwLoggerEnableDisableHostAccRxOutLogs 		  		, hwLoggerEnableDisableTxSelectorOrLockerLogs 	, hwLoggerEnableDisableQmTxListsLogs		, hwLoggerEnableDisableBeaconLogs	, hwLoggerEnableDisableBeamFormLogs 		, hwLoggerEnableDisableBaAnlayzerLogs	, hwLoggerEnableDisableSecurity3Logs , hwLoggerEnableDisableUnknown},
	{hwLoggerEnableDisableAggBuilderAndBaAnalyzerLogs 	, hwLoggerEnableDisableRxLiberatorLogs		 	, hwLoggerEnableDisableQmTxDataLogs			, hwLoggerEnableDisablePacExtraLogs	, hwLoggerEnableDisableTxSelectorBitmapLogs , hwLoggerEnableDisableMuBfLogs			, hwLoggerEnableDisableDelia2Logs	 , hwLoggerEnableDisableUnknown},
	{hwLoggerEnableDisableTxLiberatorLogs			  	, hwLoggerEnableDisableTxSequencerLogs			, hwLoggerEnableDisableRxClassLogs			, hwLoggerEnableDisablePsManagerLogs		, hwLoggerEnableDisableActivityLogs 		, hwLoggerEnableDisableQmRxDataLogs		, hwLoggerEnableDisableDelia3Logs	 , hwLoggerEnableDisableMuBitmapSelectLogs},
};


//---------------------------------------------------------------------------------
//						Data structures					
//---------------------------------------------------------------------------------
static HwQueueManagerRequestParams_t HwLoggerHwQueueManagerRequestParams;
static HwLogger_SharedParameters_t HwLoggerSharedParams; 
static HwLogger_StreamsSharedParameters_t HwLoggerStreamSharedParams[MAX_LOGGER_STREAM_NUM];


//---------------------------------------------------------------------------------
//						Private Function 				
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Public Function 			
//---------------------------------------------------------------------------------


/**********************************************************************************
hwLoggerResourceAllocation



Description:
------------
	Allocates the shared ram nuffers and their descriptors. 
	the number of buffers and the buffer size are being calculated according to the logger max stream number 
	Configures the logger's lists ID 
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
uint32 HwLoggerResourceAllocation(uint32 loggerMaxStreamNum)
{
	RegLoggerBufferLenght_u loggerBufferLength; 
	uint32 bufferSize;  
	uint32 numOfBuffers; 
	uint32 i = 0;
	
	ASSERT((0 < loggerMaxStreamNum) && (loggerMaxStreamNum < 5)); 

	/* numOfBuffers =  loggerMaxStreamNum *2 (except 1),  bufferSize = total size/numOfBuffers */
	// 1-> 4,	2-> 4,	 3-> 6,	  4-> 8 
	numOfBuffers = loggerMaxStreamNum<<1; 
	 // in case the loggerMaxStreamNum is one- number of buffers should be multiplied again  
	if (loggerMaxStreamNum == 1)
	{
		numOfBuffers <<= 1;
	}

	bufferSize = (HW_LOGGER_SHARED_RAM_BUFFER_SIZE/numOfBuffers);    // 6K divided to 2,4,6,8 buffers, each one at size 3000,1500,1000,750 
    bufferSize = ((bufferSize + 15) & ~0x0F);   // align to 16 will give us the following sizes: 3008,1504,1008,752

	HwLoggerSharedParams.maxNumberOfStreams = loggerMaxStreamNum; 
	HwLoggerSharedParams.shramBufferSize = bufferSize;

	// set the buffer length to the logger register 
	loggerBufferLength.val = 0; 
	loggerBufferLength.bitFields.bufferLenght = bufferSize; //calculated by hwLoggerResourceAllocation
	RegAccess_Write(REG_LOGGER_BUFFER_LENGHT, loggerBufferLength.val); 
	
	//allocate shared ram buffer desciptors 
	for(i =0; i < numOfBuffers; i++)
	{
		//allocate BDs and make them point to   HwLoggerHwQueueManagerRequestParams +  i*bufferSize
		HwLoggerShramBDs[i].nextBdPtr = SET_NEXT_PD(&HwLoggerShramBDs[i+1]);
		HwLoggerShramBDs[i].currentBufferLength = bufferSize;
		HwLoggerShramBDs[i].shramMemoryAddr = (void*)CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((void*)(((uint32)(&HwLoggerShramBuffers))+(i*bufferSize)));
	}

	HwLoggerShramBDs[i-1].nextBdPtr = NEXT_PD_NULL;
	//Now push to the free BD list 
	HwLoggerHwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM; 
	HwLoggerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_TX_POOL_LIST_LOGGER_FREE_BD;
	HwLoggerHwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwLoggerHwQueueManagerRequestParams.pHeadDesc = HwLoggerShramBDs;
	HwLoggerHwQueueManagerRequestParams.pTailDesc = &HwLoggerShramBDs[i-1];
	HwQManager_PushPacketListToTail(&HwLoggerHwQueueManagerRequestParams); //push all BDs to list 
	/* we might have 4 reserved PDs*/

}


/**********************************************************************************
Add Stream req
	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_AddStreamReq(K_MSG *psMsg)
{	
	uint32 fifoIndex; 
	uint32 streamId;
	uint32 regOffset;
	HwLoggerHeader_t extInfo;
	uint16 secondWord;
	uint32 hwFifoIndex;
	RegLoggerHwUnit1SecondHeaderWord_u fifoMsgHdr;
	RegLoggerFifoEnable_u loggerFifoEnableReg; 
	RegLoggerBufferLenght_u loggerBufferLengh; 
	RegLoggerLoggerHwUnitCfgMux_u loggerHwUnitCfgMux;
	UMI_ADD_STREAM_REQ_t* addStreamStructurePtr;
   	RegLoggerBufferCloseTimeThr_u bufferClosingTimeThr;
   	RegLoggerMode_u	loggerModeReg;
    
	uint8 i;
	uint32 regAddress = 0x0;
	void* udpHeaderAddr = NULL;
	uint16 hwModuleFifo;

	addStreamStructurePtr = (UMI_ADD_STREAM_REQ_t*) pK_MSG_DATA(psMsg);
	
	streamId = addStreamStructurePtr->streamId; 
    
	/*validation*/
	DEBUG_ASSERT(streamId < MAX_LOGGER_STREAM_NUM); 
	DEBUG_ASSERT(HwLoggerStreamSharedParams[streamId].streamEnable == 0); //stream already enabled! 
	HwLoggerSharedParams.numberOfActiveStream++;
	/* set constant values for SW message header	 */ 
	HwLoggerStreamSharedParams[streamId].bufferThreshold = addStreamStructurePtr->bufferThreshold; 
	HwLoggerStreamSharedParams[streamId].streamEnable = STREAM_ENABLE; 
	HwLoggerStreamSharedParams[streamId].assignedFifosId = (addStreamStructurePtr->assignedFifosId)&(THREE_LS_BYTES_IN_HALF_WORD);

	/*fill the message header template*/
	/* fill data on a local message header, later will be assigned to the correspond modules*/

	
	extInfo.word1 = 0;
	extInfo.be0 = addStreamStructurePtr->be0;
	extInfo.logVer = addStreamStructurePtr->logVersion;
	extInfo.pri = LOGGER_DEFAULT_PRIORITY; 
	extInfo.be1 = addStreamStructurePtr->be1; 
	extInfo.wlanIf = addStreamStructurePtr->wlanIf;  
	extInfo.dest = addStreamStructurePtr->streamId;

	secondWord = (extInfo.word1 >> 16); // we need to configure the HW only the 16 MSb
	
	/*configure the udp parameters */
	udpHeaderAddr = (void*)(LOGGER_RAM_FISRT_UDP_HEADER_START_ADD + UDP_HEADER_WORD_SIZE*streamId); 
	memcpy32(udpHeaderAddr,&(addStreamStructurePtr->udpHeader[0]),UDP_HEADER_WORD_SIZE); 

	//configure UDP Header structure in Rx Handler - used for sniffer
	DUT_SnifferSetUdpHeader(addStreamStructurePtr);

	RegAccess_Read(REG_LOGGER_BUFFER_LENGHT, &loggerBufferLengh.val); 
	//the total buffer size was being set at initialization phase, make sure that the register holds the right value 
	ASSERT(loggerBufferLengh.bitFields.bufferLenght == HwLoggerSharedParams.shramBufferSize);
    
    // ** //
    // indicate HIM if logger is background or online
    loggerActiveMode = addStreamStructurePtr->loggerActiveMode;

    // in background mode (also called offline) we dont want to flush buffer at timer, so we set timer high.
    // We only use X buffers, if flush to early(by timer) it will not get full - > less logs in each buffer.
    if (loggerActiveMode == LOGGER_STATE_ACTIVE_BACKGROUND)
    {
        bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_BACKGROUND_BUFFER_TIME_THRESHOLD; 
    }
    else if (loggerActiveMode == LOGGER_STATE_ACTIVE_ONLINE)
    {
       	bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD;    
    }
    
    // write changes to register
    RegAccess_Write(REG_LOGGER_BUFFER_CLOSE_TIME_THR, bufferClosingTimeThr.val);
    
    /* Enable logger clock (to push buffer changing */
	loggerModeReg.bitFields.listConverterBlockEn = TRUE;
	RegAccess_Write(REG_LOGGER_MODE,loggerModeReg.val);
    // ** //
    
	switch(streamId)
    {
	    case STREAM_0:
	    {
			RegLoggerUmacLmacFastMesSecWord_u RegLoggerUmacLmacFastMesSecWord; 
			RegLoggerHostIfHandlerFastMesSecWord_u hostIfhandlerFastMesSecWord;
			RegLoggerSenderFastMesSecWord_u senderPhyGenriscFastMesSecWord;
			RegLoggerSwPreCalculatedChecksumDst0_u swPreCalcCheckSum;
			RegLoggerBuffer01Thr_u loggerBuffer_01_Threshold; 

			RegAccess_Read(REG_LOGGER_BUFFER0_1THR, &loggerBuffer_01_Threshold.val); 
			loggerBuffer_01_Threshold.bitFields.buffer0Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER0_1THR, loggerBuffer_01_Threshold.val);
			
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst0 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST0, swPreCalcCheckSum.val);

			/* Configure the Fast message header (last 16 bits of the second word, both for UM and LM )*/
			RegAccess_Read(REG_LOGGER_UMAC_LMAC_FAST_MES_SEC_WORD, &RegLoggerUmacLmacFastMesSecWord.val);
			RegLoggerUmacLmacFastMesSecWord.bitFields.umacFastMesSecWord = secondWord; 
			RegLoggerUmacLmacFastMesSecWord.bitFields.lmacFastMesSecWord = secondWord; 
			RegAccess_Write(REG_LOGGER_UMAC_LMAC_FAST_MES_SEC_WORD, RegLoggerUmacLmacFastMesSecWord.val);

			hostIfhandlerFastMesSecWord.bitFields.hostIfFastMesSecWord		= secondWord;
			hostIfhandlerFastMesSecWord.bitFields.rxHandlerFastMesSecWord	= secondWord;
			RegAccess_Write(REG_LOGGER_HOST_IF_HANDLER_FAST_MES_SEC_WORD, hostIfhandlerFastMesSecWord.val);

			senderPhyGenriscFastMesSecWord.bitFields.senderFastMesSecWord  = secondWord;
			senderPhyGenriscFastMesSecWord.bitFields.phyRiscFastMesSecWord = secondWord;
			RegAccess_Write(REG_LOGGER_SENDER_FAST_MES_SEC_WORD, senderPhyGenriscFastMesSecWord.val);
					
			break;
	    }
	    case STREAM_1:
		{ 
			RegLoggerSwPreCalculatedChecksumDst1_u swPreCalcCheckSum; 
			RegLoggerBuffer01Thr_u loggerBuffer_01_Threshold;
				
			RegAccess_Read(REG_LOGGER_BUFFER0_1THR, &loggerBuffer_01_Threshold.val); 
			loggerBuffer_01_Threshold.bitFields.buffer1Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER0_1THR, loggerBuffer_01_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst1 =  addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST1, swPreCalcCheckSum.val);
			break;
	    }
	    case STREAM_2:
	    {
			RegLoggerSwPreCalculatedChecksumDst2_u swPreCalcCheckSum;
			RegLoggerBuffer23Thr_u loggerBuffer_23_Threshold;

			RegAccess_Read(REG_LOGGER_BUFFER2_3THR, &loggerBuffer_23_Threshold.val); 
			loggerBuffer_23_Threshold.bitFields.buffer2Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER2_3THR, loggerBuffer_23_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst2 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST2, swPreCalcCheckSum.val);
			break;
		}  
		case STREAM_3: 
		{
			RegLoggerSwPreCalculatedChecksumDst3_u swPreCalcCheckSum; 
			RegLoggerBuffer23Thr_u loggerBuffer_23_Threshold;
			
			RegAccess_Read(REG_LOGGER_BUFFER2_3THR, &loggerBuffer_23_Threshold.val); 
			loggerBuffer_23_Threshold.bitFields.buffer3Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER2_3THR, loggerBuffer_23_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst3 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST3, swPreCalcCheckSum.val);
			break; 
		}
	    default:
	        break;
	}
	/* configure each HW fifo header, that is assigned with that stream*/
	regOffset = 0;

	fifoIndex = LOGGER_FIRST_HW_SOURCE_INDEX;
	for ( i = 0 ; i < (LOGGER_NUM_OF_HW_FIFOS>>1) ; i++)
	{
		
		if(addStreamStructurePtr->assignedFifosId & (0x1 << fifoIndex))
		{
			hwFifoIndex = fifoIndex - LOGGER_FIRST_HW_SOURCE_INDEX;
			hwModuleFifo = addStreamStructurePtr->hwModuleFifo[hwFifoIndex];
			hwLogsHandler[hwFifoIndex][hwModuleFifo](TRUE , FALSE);
			
			regAddress = REG_LOGGER_HW_UNIT1_SECOND_HEADER_WORD + regOffset; 
			RegAccess_Read(regAddress, &fifoMsgHdr.val);
			fifoMsgHdr.bitFields.hwUnit1SecondHeaderWord = secondWord; 
			RegAccess_Write(regAddress, fifoMsgHdr.val);
		}

		fifoIndex++;
		
		if(addStreamStructurePtr->assignedFifosId & (0x1 << fifoIndex))
		{
			
			hwFifoIndex = fifoIndex - LOGGER_FIRST_HW_SOURCE_INDEX;
			hwModuleFifo = addStreamStructurePtr->hwModuleFifo[hwFifoIndex];
			hwLogsHandler[hwFifoIndex][hwModuleFifo](TRUE , FALSE);

			regAddress = REG_LOGGER_HW_UNIT1_SECOND_HEADER_WORD + regOffset;
			RegAccess_Read(regAddress, &fifoMsgHdr.val);
			fifoMsgHdr.bitFields.hwUnit2SecondHeaderWord = secondWord; 
			RegAccess_Write(regAddress, fifoMsgHdr.val);


		}

		fifoIndex++;
		regOffset+=4;
	}
	

	loggerHwUnitCfgMux.bitFields.loggerHwUnit1CfgMux = addStreamStructurePtr->hwModuleFifo[0];
	loggerHwUnitCfgMux.bitFields.loggerHwUnit2CfgMux = addStreamStructurePtr->hwModuleFifo[1];
	loggerHwUnitCfgMux.bitFields.loggerHwUnit3CfgMux = addStreamStructurePtr->hwModuleFifo[2];
	loggerHwUnitCfgMux.bitFields.loggerHwUnit4CfgMux = addStreamStructurePtr->hwModuleFifo[3];
	loggerHwUnitCfgMux.bitFields.loggerHwUnit5CfgMux = addStreamStructurePtr->hwModuleFifo[4];
	loggerHwUnitCfgMux.bitFields.loggerHwUnit6CfgMux = addStreamStructurePtr->hwModuleFifo[5];
	
	RegAccess_Write(REG_LOGGER_LOGGER_HW_UNIT_CFG_MUX, loggerHwUnitCfgMux.val);

	/* Enable the correspond fifo's */
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnableReg.val); 
	loggerFifoEnableReg.val |= addStreamStructurePtr->assignedFifosId; 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val);


	if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_UM_FIFO_ID))
	{
		logFillMsgHdrTemplate(addStreamStructurePtr);
	}

	if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_LM_FIFO_ID))
	{		
		OSAL_SEND_MESSAGE(LINK_ADAPTATION_LOG_FILL_HDR_TEMPLATE, TASK_LINK_ADAPTATION,psMsg, VAP_ID_DO_NOT_CARE);
	}
	else
	{
		OSAL_SEND_MESSAGE(UMI_MC_MAN_ADD_STREAM_CFM, TASK_UM_IF_TASK, psMsg, VAP_ID_DO_NOT_CARE);
	}

	if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_PHY_GENRISC_FIFO_ID))
	{
		RegAccess_Write(PHY_TD_BASE_ADDR + ENABLE_GENRISC_LOGS_ADDRESS, 0x1);
	}
}


/**********************************************************************************
HwLogger_RemoveStreamReq

	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_RemoveStreamReq(uint16 streamId)
{	
	TX_INTERRUPT_SAVE_AREA;
	RegLoggerFifoEnable_u loggerFifoEnableReg; 
	RegLoggerNumMessages1_u numMessagesInfifo1; 
	RegLoggerNumMessages2_u numMessagesInfifo2; 
	RegLoggerNumMessages3_u numMessagesInfifo3; 
	uint32 timeStamp;

	HwLoggerStreamSharedParams[streamId].streamEnable = 0;
	/* disable the stream's  fifos*/
	loggerFifoEnableReg.val = 0; 
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnableReg.val); 
	loggerFifoEnableReg.val &= (THREE_LS_BYTES_IN_WORD); /*As only 12 ls bits are valid, we mask the rest of the word*/
	loggerFifoEnableReg.val ^=  HwLoggerStreamSharedParams[streamId].assignedFifosId;
	ASSERT(loggerFifoEnableReg.val == 0); //currently we use only 1 stream so there should not be any fifos enabled after remove stream request 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val); 


	//currently we use only 1 stream so there should not be any messages left in fifos after remove stream request 
	//wait for all logs to be flushed from all fifos
	timeStamp = GET_TSF_TIMER_LOW();
	do
	{
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_1, &numMessagesInfifo1.val); 
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_2, &numMessagesInfifo2.val);
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_3, &numMessagesInfifo3.val);
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= LOGGER_MAX_WAIT_ON_FIFO_FLUSH);
	}while((numMessagesInfifo1.val != 0) || (numMessagesInfifo2.val != 0) || (numMessagesInfifo3.val != 0));

	OSAL_DISABLE_INTERRUPTS(&interrupt_save); // disable interrupts so we won't get any logs when trying to send the last one of these stream 

	//send a fast log message with the expedite bit on, to the current stream - in order to  flush the buffer 
	HwLogger_SendFastLog(LOG_LOCAL_FID, LOG_LOCAL_GID,  __LINE__, 1); //"STREAM_TERMINATION"  log 

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	
//	HwLogger_EnablePhyLogs();
	if ( HwLoggerStreamSharedParams[streamId].assignedFifosId & (1<<LOGGER_PHY_GENRISC_FIFO_ID))
	{
		RegAccess_Write(PHY_TD_BASE_ADDR + ENABLE_GENRISC_LOGS_ADDRESS, 0x0);
	}
	HwLoggerSharedParams.numberOfActiveStream--;
}


///**********************************************************************************
//HwLogger_EnablePhyLogs
//	
//Description:
//------------
//	
//Input:
//-----
//	Max stream number 
//Output:
//-------
//	None
//Returns:
//--------
//	void
//**********************************************************************************/
//void HwLogger_EnablePhyLogs(void) 
//{
//	uint32 i;
//	uint32 enableLogs = FALSE;
//	RegPhyRxTdPhyRxtdReg7F_u regPhyRxTdPhyRxtdReg7F;
//
//	regPhyRxTdPhyRxtdReg7F.val = 0;
//
//	for (i = 0; i < MAX_LOGGER_STREAM_NUM; i++)
//	{
//		if (HwLoggerStreamSharedParams[i].assignedFifosId & ((1 << LOGGER_PHY_GENRISC_FIFO_ID) | (1 << LOGGER_PHY_HW_FIFO_ID)))
//		{
//			enableLogs = TRUE;
//			break;
//		}
//	}
//
//	/* set PHY logs */
//	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG7F, &regPhyRxTdPhyRxtdReg7F.val); 
//	regPhyRxTdPhyRxtdReg7F.bitFields.loggerTestBusGclkEn = enableLogs; 
//	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG7F, regPhyRxTdPhyRxtdReg7F.val);
//}


/**********************************************************************************
HwLogger_SendFastLog



	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SendFastLog(uint32 FID, uint32 GID, uint32 LID, uint32 exp) 
{
	HwLoggerHeader_t header; 

	header.word0 = 0;
	header.oid= 0; 
	header.fid = FID; 
	header.gid = GID; 
	header.lid = LID; 
	header.exp = exp; 

	RegAccess_Write(LOGGER_FAST_DATA_IN_REG, header.word0); 
}


/**********************************************************************************
HwLogger_AddFifoFilterReq




	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_AddFifoFilterReq(UMI_ADD_FIFO_FILTER_REQ_t* addFifoFilterReq)
{
	if(addFifoFilterReq->FilterType == LOGGER_FILTER_TYPE_LOG_LEVEL)
	{
		RegLoggerStopMaskTriggerHigh_u stopMaskTriggerHigh_u; //this register holds the log level from which logs would be forward
		RegAccess_Read(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, &stopMaskTriggerHigh_u.val); 
		stopMaskTriggerHigh_u.bitFields.logLevelToForward = addFifoFilterReq->logLevel; //set the correspond log level 
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopMaskTriggerHigh_u.val);		
	}
	else // LOGGER_FILTER_TYPE_FIFO (disable specific modlule's fifos )
	{
		RegLoggerFifoEnable_u loggerFifoEnable; 
		uint16 enabledFifoMask =0; 
		
		enabledFifoMask = ~(addFifoFilterReq->fifosBitmap); 
		RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnable.val); 
		loggerFifoEnable.val &= enabledFifoMask; //mask the correspond fifios
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnable.val);
	}
}


/**********************************************************************************
HwLogger_RemoveFifoFilterReq


	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_RemoveFifoFilterReq(UMI_REMOVE_FIFO_FILTER_REQ_t* removeFifoFilterReq)
{
	if(removeFifoFilterReq->FilterType == LOGGER_FILTER_TYPE_LOG_LEVEL)
	{
		RegLoggerStopMaskTriggerHigh_u stopMaskTriggerHigh_u; //this register holds the log level from which logs would be forward
		RegAccess_Read(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, &stopMaskTriggerHigh_u.val); 
		stopMaskTriggerHigh_u.bitFields.logLevelToForward = LOGGER_DATA_IN_PTR_MAX_LOG_LEVEL; //set the maximum log level so no logs would be filtered 
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, stopMaskTriggerHigh_u.val);	
	}
	else // LOGGER_FILTER_TYPE_FIFO (enable specific modlule's fifos )
	{
		RegLoggerFifoEnable_u loggerFifoEnable; 
		uint16 enabledFifoMask =0; 
	
		enabledFifoMask =	removeFifoFilterReq->fifosBitmap; 
		RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnable.val); 
		loggerFifoEnable.val |= enabledFifoMask; //mask the correspond fifios
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnable.val);
	}
}


/**********************************************************************************
HwLogger_SetHdrTriggerReq


	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SetHdrTriggerReq(UMI_SET_TRIGGER_REQ_t* setHdrTriggerReq) 
{
	if(setHdrTriggerReq->u8TriggerType == LOGGER_HDR_START_TRIGGER_TYPE)
	{
		RegLoggerStartDataTriggerLow_u 	startDataTriggerLow;
		RegLoggerStartDataTriggerHigh_u StartDataTriggerHigh; 
		RegLoggerStartMaskTriggerLow_u	startDataTriggerMaskLow;
		RegLoggerStartMaskTriggerHigh_u	startDataTriggerMaskHigh; 
	
		startDataTriggerLow.bitFields.startDataTriggerLow = setHdrTriggerReq->messageHeaderFirst4Bytes; 
		StartDataTriggerHigh.bitFields.startDataTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		startDataTriggerMaskLow.bitFields.startMaskTriggerLow = setHdrTriggerReq->messageHeaderMaskFirst4Bytes; 
		startDataTriggerMaskHigh.bitFields.startMaskTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		/*Write the header to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_DATA_TRIGGER_LOW, startDataTriggerLow.val);
		RegAccess_Write(REG_LOGGER_START_DATA_TRIGGER_HIGH, StartDataTriggerHigh.val);
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startDataTriggerMaskHigh.val);	
	}
	else
	{
		RegLoggerStopDataTriggerLow_u 	stopDataTriggerLow;
		RegLoggerStopDataTriggerHigh_u 	stopDataTriggerHigh; 
		RegLoggerStopMaskTriggerLow_u	stopDataTriggerMaskLow;
		RegLoggerStopMaskTriggerHigh_u	stopDataTriggerMaskHigh; 
		RegLoggerTimeToRecorderAfterStop_u timeToRecordAfterStop; 

		ASSERT(setHdrTriggerReq->u8TriggerType == LOGGER_HDR_STOP_TRIGGER_TYPE);//Make sure the type is valid 
	
		stopDataTriggerLow.bitFields.stopDataTriggerLow = setHdrTriggerReq->messageHeaderFirst4Bytes; 
		stopDataTriggerHigh.bitFields.stopDataTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		stopDataTriggerMaskLow.bitFields.stopMaskTriggerLow = setHdrTriggerReq->messageHeaderMaskFirst4Bytes; 
		stopDataTriggerMaskHigh.bitFields.stopMaskTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes;
		timeToRecordAfterStop.bitFields.timeToRecorderAfterStop = 0x0; // Time to record after sop 
		/*Write the header to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_DATA_TRIGGER_LOW, stopDataTriggerLow.val);
		RegAccess_Write(REG_LOGGER_STOP_DATA_TRIGGER_HIGH, stopDataTriggerHigh.val);
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_LOW, stopDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopDataTriggerMaskHigh.val);
		RegAccess_Write(REG_LOGGER_TIME_TO_RECORDER_AFTER_STOP, timeToRecordAfterStop.val);
	}

}


/**********************************************************************************
HwLogger_ResetHdrTriggerReq



	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_ResetHdrTriggerReq(UMI_RESET_TRIGGER_REQ_t* resetHdrTriggerReq)
{
	if(resetHdrTriggerReq->u8TriggerType == LOGGER_HDR_START_TRIGGER_TYPE)
	{ 
		RegLoggerStartMaskTriggerLow_u	startDataTriggerMaskLow;
		RegLoggerStartMaskTriggerHigh_u	startDataTriggerMaskHigh; 
	
		startDataTriggerMaskLow.bitFields.startMaskTriggerLow = 0x0; //Reset all active bytes 
		startDataTriggerMaskHigh.bitFields.startMaskTriggerHigh = 0x0; //Reset all active bytes 
	
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startDataTriggerMaskHigh.val);	
	}
	else
	{
		RegLoggerStopMaskTriggerLow_u	stopDataTriggerMaskLow;
		RegLoggerStopMaskTriggerHigh_u	stopDataTriggerMaskHigh; 

		ASSERT(resetHdrTriggerReq->u8TriggerType == LOGGER_HDR_STOP_TRIGGER_TYPE);//Make sure the type is valid 

		stopDataTriggerMaskLow.bitFields.stopMaskTriggerLow = 0x0; //Reset all active bytes 
		stopDataTriggerMaskHigh.bitFields.stopMaskTriggerHigh = 0x0; //Reset all active bytes 
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_LOW, stopDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopDataTriggerMaskHigh.val);
	}
}


/**********************************************************************************
HwLogger_ResetHdrTriggerReq



	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SetPhyHeaderReq(UMI_SET_PHY_HDR_REQ_t* resetHdrTriggerReq)
{
	RegLoggerPhyFifoFirstHeaderWord_u	phyFirstHeaderWord;
	RegLoggerPhyFifoSecondHeaderWord_u	phySecondHeaderWord; 
	RegLoggerPhyFifoMesageLenght_u		phyMessageLength; 

	RegLoggerFifoEnable_u				loggerfifoEnable;

	phyFirstHeaderWord.val = 0;
	phySecondHeaderWord.val = 0;
	phyMessageLength.val = 0;
	loggerfifoEnable.val = 0;

	phyFirstHeaderWord.bitFields.phyFifoFirstHeaderWord = resetHdrTriggerReq->phyFifoFirstHeaderWord;
	phySecondHeaderWord.bitFields.phyFifoSecondHeaderWord = resetHdrTriggerReq->phyFifoSecondHeaderWord;
	phyMessageLength.bitFields.phyFifoMessageLenght = resetHdrTriggerReq->phyFifoMessageLength;

	/*Write the header mask to the start trigger filter*/
	RegAccess_Write(REG_LOGGER_PHY_FIFO_FIRST_HEADER_WORD, phyFirstHeaderWord.val);
	RegAccess_Write(REG_LOGGER_PHY_FIFO_SECOND_HEADER_WORD, phySecondHeaderWord.val);
	RegAccess_Write(REG_LOGGER_PHY_FIFO_MESAGE_LENGHT, phyMessageLength.val);

	//Now enable the phy fifo 
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerfifoEnable.val);
	loggerfifoEnable.bitFields.phyFifoEnable = 0x1; 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerfifoEnable.val);
}


/**********************************************************************************
hwLoggerEnableDisableHostAccTxInLogs	
	
Description:
------------
	enable or disable HW TxIn  logs
Input:
-----
	bool shortLogEnable - enable / disable TxIn short  logs
	bool longLogEnable -enable / disable TxIn long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccTxInLogs(bool shortLogEnable , bool longLogEnable)
{

	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable TxIn logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.txInLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.txInLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.txInLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.txInLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.txInLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.txInLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);

}


/**********************************************************************************
hwLoggerEnableDisableHostAccTxOutLogs	
	
Description:
------------
	enable or disable HW TxOut  logs
Input:
-----
	bool shortLogEnable - enable / disable TxOut short  logs
	bool longLogEnable -enable / disable TxOut long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccTxOutLogs(bool shortLogEnable , bool longLogEnable)
{

	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable TxOut logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.txOutLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.txOutLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.txOutLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.txOutLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.txOutLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.txOutLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);

}


/**********************************************************************************
hwLoggerEnableDisableHostAccRxInLogs	
	
Description:
------------
	enable or disable HW RxIn  logs
Input:
-----
	bool shortLogEnable - enable / disable RxIn short  logs
	bool longLogEnable -enable / disable RxIn long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccRxInLogs(bool shortLogEnable , bool longLogEnable)
{

	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable RxIn logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.rxInLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.rxInLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.rxInLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.rxInLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.rxInLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.rxInLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);

}


/**********************************************************************************
hwLoggerEnableDisableHostAccRxOutLogs	
	
Description:
------------
	enable or disable HW RxOut  logs
Input:
-----
	bool shortLogEnable - enable / disable RxOut short  logs
	bool longLogEnable -enable / disable RxOut long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccRxOutLogs(bool shortLogEnable , bool longLogEnable)
{

	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable RxOut logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.rxOutLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.rxOutLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.rxOutLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.rxOutLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.rxOutLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.rxOutLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);

}


/**********************************************************************************
hwLoggerEnableDisableBaAnlayzerLogs	
	
Description:
------------
	enable or disable HW BaAnlayzer  logs
Input:
-----
	bool shortLogEnable - enable / disable BaAnlayzer  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBaAnlayzerLogs(bool shortLogEnable , bool longLogEnable)
{
	RegBaAnalyzerLoggerReg_u baAnalyzerLogger;
	
	// disable BaAnalyzer logs
	RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegEn = FALSE;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG,baAnalyzerLogger.val);

	// need to check that we are not while writing log to fifo
	while (baAnalyzerLogger.bitFields.baAnalyzerLoggerRegActive == 1)
	{
		RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	}
	
	// enable logs
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegEn = shortLogEnable;
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG, baAnalyzerLogger.val);	
}


/**********************************************************************************
hwLoggerEnableDisableTxLiberatorLogs	
	
Description:
------------
	enable or disable HW Tx liberator  logs
Input:
-----
	bool shortLogEnable - enable / disable Tx liberator  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxLiberatorLogs(bool shortLogEnable , bool longLogEnable)
{
	RegBaAnalyzerLoggerReg_u baAnalyzerLogger;
	
	// disable TxLiberator logs
	RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	baAnalyzerLogger.bitFields.liberatorLoggerRegEn = FALSE;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG,baAnalyzerLogger.val);

	// need to check that we are not while writing log to fifo
	while (baAnalyzerLogger.bitFields.liberatorLoggerRegActive == 1)
	{
		RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	}
	
	// enable logs
	baAnalyzerLogger.bitFields.liberatorLoggerRegEn = shortLogEnable;
	baAnalyzerLogger.bitFields.liberatorLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG, baAnalyzerLogger.val);	
}


/**********************************************************************************
hwLoggerEnableDisableRxLiberatorLogs	
	
Description:
------------
	enable or disable HW Rx liberator  logs
Input:
-----
	bool shortLogEnable - enable / disable Rx liberator  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxLiberatorLogs(bool shortLogEnable , bool longLogEnable)
{
	RegBaAnalyzerLoggerReg_u baAnalyzerLogger;
	
	// disable RxLiberator logs
	RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	baAnalyzerLogger.bitFields.rxLiberatorLoggerRegEn = FALSE;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG,baAnalyzerLogger.val);

	// need to check that we are not while writing log to fifo
	while (baAnalyzerLogger.bitFields.rxLiberatorLoggerRegActive == 1)
	{
		RegAccess_Read(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val);
	}
	
	// enable logs
	baAnalyzerLogger.bitFields.rxLiberatorLoggerRegEn = shortLogEnable;
	baAnalyzerLogger.bitFields.rxLiberatorLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_BA_ANALYZER_LOGGER_REG, baAnalyzerLogger.val);	
}


/**********************************************************************************
hwLoggerEnableDisableAggBuilderLogs	
	
Description:
------------
	enable or disable HW Agg builder  logs
Input:
-----
	bool shortLogEnable - enable / disable Agg handler  logs
	bool longLogEnable - enable / disable Agg handler Std logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableAggBuilderLogs(bool shortLogEnable , bool longLogEnable)
{
	RegAggBuilderAggBuilderLogger_u			aggBuilderLogger;
	RegAggBuilderAggBuilderLoggerActive_u   aggBuilderLoggerActive;


	aggBuilderLogger.val = 0;
	RegAccess_Write(REG_AGG_BUILDER_AGG_BUILDER_LOGGER, aggBuilderLogger.val); // disable agg builder logs


	do
	{
		RegAccess_Read(REG_AGG_BUILDER_AGG_BUILDER_LOGGER_ACTIVE, &aggBuilderLoggerActive.val);
	}while (aggBuilderLoggerActive.bitFields.aggBuilderLoggerActive == 1);// need to check that we are not while writing log to fifo

	// enable logs
	aggBuilderLogger.bitFields.aggBuilderLoggerEn = shortLogEnable;
	aggBuilderLogger.bitFields.aggBuilderLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_AGG_BUILDER_AGG_BUILDER_LOGGER, aggBuilderLogger.val);
}


/**********************************************************************************
hwLoggerEnableDisableAggBuilderAndBaAnalyzerLogs	
	
Description:
------------
	enable or disable HW Agg builder and Ba analyzer logs
Input:
-----
	bool shortLogEnable - enable / disable Agg handler and Ba analyzer  logs
	bool longLogEnable - enable / disable Agg handler Std logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableAggBuilderAndBaAnalyzerLogs(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableAggBuilderLogs(shortLogEnable , longLogEnable);
	hwLoggerEnableDisableBaAnlayzerLogs(shortLogEnable , longLogEnable);
}


/**********************************************************************************
hwLoggerEnableDisableRxPpLogs	
	
Description:
------------
	enable or disable HW RxPp logs
Input:
-----
	bool shortLogEnable - enable / disable RxPp  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxPpLogs(bool shortLogEnable , bool longLogEnable)
{
	RegRxPpRxLoggerCfg_u rxPpRxLoggerCfg;
	RegRxPpRxLoggerBusy_u rxPpRxLoggerBusy;
	
	// disable  logs
	rxPpRxLoggerCfg.val = 0;
	RegAccess_Write(REG_RX_PP_RX_LOGGER_CFG,rxPpRxLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_RX_PP_RX_LOGGER_BUSY, &rxPpRxLoggerBusy.val);
	}while (rxPpRxLoggerBusy.bitFields.rxPpLoggerBusy == 1);
	

	// enable logs
	rxPpRxLoggerCfg.bitFields.rxPpLoggerEn = shortLogEnable;
	rxPpRxLoggerCfg.bitFields.rxPpLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_RX_PP_RX_LOGGER_CFG, rxPpRxLoggerCfg.val);
}


/**********************************************************************************
hwLoggerEnableDisableTxHLogs	
	
Description:
------------
	enable or disable HW TxHandler logs
Input:
-----
	bool shortLogEnable - enable / disable TxHandler  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void  hwLoggerEnableDisableTxHLogs(bool shortLogEnable , bool longLogEnable)
{
	RegTxHandlerTxhLogger_u txHandlerTxhLogger;
	RegTxHandlerTxhLoggerActive_u txHandlerTxhLoggerActive;
	
	// disable TxH logs
	txHandlerTxhLogger.val = 0;
	RegAccess_Write(REG_TX_HANDLER_TXH_LOGGER,txHandlerTxhLogger.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_TX_HANDLER_TXH_LOGGER_ACTIVE, &txHandlerTxhLoggerActive.val);
	}while (txHandlerTxhLoggerActive.bitFields.txhLoggerActive == 1);
	

	// enable logs
	txHandlerTxhLogger.bitFields.txhLoggerEn = shortLogEnable;
	txHandlerTxhLogger.bitFields.txhLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_TX_HANDLER_TXH_LOGGER, txHandlerTxhLogger.val);
}


/**********************************************************************************
hwLoggerEnableDisableTxSequencerLogs	
	
Description:
------------
	enable or disable HW TxSequencer logs
Input:
-----
	bool shortLogEnable - enable / disable TxSequencer  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSequencerLogs(bool shortLogEnable , bool longLogEnable)
{
	RegTxSequencerSequencerLogger_u			txSequencerSequencerLogger;
	RegTxSequencerSequencerLoggerActive_u   txSequencerSequencerLoggerActive;


	txSequencerSequencerLogger.val = 0;
	RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_LOGGER, txSequencerSequencerLogger.val); // disable TxSequencer logs


	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_TX_SEQUENCER_SEQUENCER_LOGGER_ACTIVE, &txSequencerSequencerLoggerActive.val);
	}while (txSequencerSequencerLoggerActive.bitFields.sequencerLoggerActive == 1);

	// enable logs
	txSequencerSequencerLogger.bitFields.sequencerLoggerEn = shortLogEnable;
	txSequencerSequencerLogger.bitFields.sequencerLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_LOGGER, txSequencerSequencerLogger.val);
}

/**********************************************************************************
hwLoggerEnableDisablePsManagerLogs 

Description:
------------
	Enable or Disable the Hw logs of the Ps Setting Hw Module
		1) Disable Logs
		2) Wait for log activity to finish
		3) Enable\Disable logs as required
	
Input: 
-----
	shortLogEnable - Enable or Disable fifo log
	longLogEnable - Not relevant for this module

Returns:
--------
	None 
**********************************************************************************/
void hwLoggerEnableDisablePsManagerLogs(bool shortLogEnable, bool longLogEnable)
{
	RegPsSettingPsLogger_u psLoggerReg;
	RegPsSettingPsLoggerActive_u  psLoggerActiveReg;

	// disable	logs
	psLoggerReg.val = 0;
	RegAccess_Write(REG_PS_SETTING_PS_LOGGER,psLoggerReg.val);
	
	// need to check that we are not in the middle writing log to fifo
	do
	{
		RegAccess_Read(REG_PS_SETTING_PS_LOGGER_ACTIVE, &psLoggerActiveReg.val);
	}while (psLoggerActiveReg.bitFields.psLoggerActive == 1);
	
	// enable logs
	psLoggerReg.bitFields.psLoggerEn = shortLogEnable;
	psLoggerReg.bitFields.psLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PS_SETTING_PS_LOGGER, psLoggerReg.val);	
}



/**********************************************************************************
hwLoggerEnableDisableQmTxListsLogs	
	
Description:
------------
	enable or disable HW Tx lists logs
Input:
-----
	bool shortLogEnable - enable / disable Tx lists  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxListsLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerUmacSwMaster3LoggerCfg_u QmTxListsLoggerCfg;
	RegQManagerUmacSwMaster3LoggerBusy_u QmTxListsLoggerBusy;
	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, &QmTxListsLoggerCfg.val);
	QmTxListsLoggerCfg.bitFields.qmTxListsDlmLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG,QmTxListsLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_BUSY, &QmTxListsLoggerBusy.val);
	}while (QmTxListsLoggerBusy.bitFields.qmTxListsDlmLoggerBusy == 1);
	
	// enable logs
	QmTxListsLoggerCfg.bitFields.qmTxListsDlmLoggerEn = shortLogEnable;
	QmTxListsLoggerCfg.bitFields.qmTxListsDlmLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, QmTxListsLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableQmTxDataLogs	
	
Description:
------------
	enable or disable HW Tx data logs
Input:
-----
	bool shortLogEnable - enable / disable Tx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxDataLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerUmacSwMaster3LoggerCfg_u QmTxDataLoggerCfg;
	RegQManagerUmacSwMaster3LoggerBusy_u QmTxDataLoggerBusy;
	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, &QmTxDataLoggerCfg.val);
	QmTxDataLoggerCfg.bitFields.qmTxDataDlmLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG,QmTxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_BUSY, &QmTxDataLoggerBusy.val);
	}while (QmTxDataLoggerBusy.bitFields.qmTxDataDlmLoggerBusy == 1);
	
	// enable logs
	QmTxDataLoggerCfg.bitFields.qmTxDataDlmLoggerEn = shortLogEnable;
	QmTxDataLoggerCfg.bitFields.qmTxDataDlmLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, QmTxDataLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableQmRxDataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxDataLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerUmacSwMaster3LoggerCfg_u QmRxDataLoggerCfg;
	RegQManagerUmacSwMaster3LoggerBusy_u QmRxDataLoggerBusy;
	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.qmRxDataDlmLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_BUSY, &QmRxDataLoggerBusy.val);
	}while (QmRxDataLoggerBusy.bitFields.qmRxDataDlmLoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.qmRxDataDlmLoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.qmRxDataDlmLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableQmRxListsLogs	
	
Description:
------------
	enable or disable HW Rx List logs
Input:
-----
	bool shortLogEnable - enable / disable Rx list  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxListsLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerUmacSwMaster3LoggerCfg_u QmRxListsLoggerCfg;
	RegQManagerUmacSwMaster3LoggerBusy_u QmRxListsLoggerBusy;
	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, &QmRxListsLoggerCfg.val);
	QmRxListsLoggerCfg.bitFields.qmRxListsDlmLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG,QmRxListsLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_BUSY, &QmRxListsLoggerBusy.val);
	}while (QmRxListsLoggerBusy.bitFields.qmRxListsDlmLoggerBusy == 1);
	
	// enable logs
	QmRxListsLoggerCfg.bitFields.qmRxListsDlmLoggerEn = shortLogEnable;
	QmRxListsLoggerCfg.bitFields.qmRxListsDlmLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, QmRxListsLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableQmAgerLogs	
	
Description:
------------
	enable or disable HW Ager logs
Input:
-----
	bool shortLogEnable - enable / disable  Ager  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmAgerLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerUmacSwMaster3LoggerCfg_u QmAgerLoggerCfg;
	RegQManagerUmacSwMaster3LoggerBusy_u QmAgerLoggerBusy;
	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, &QmAgerLoggerCfg.val);
	QmAgerLoggerCfg.bitFields.qmAgerLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG,QmAgerLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_BUSY, &QmAgerLoggerBusy.val);
	}while (QmAgerLoggerBusy.bitFields.qmAgerLoggerBusy == 1);
	
	// enable logs
	QmAgerLoggerCfg.bitFields.qmAgerLoggerEn = shortLogEnable;
	QmAgerLoggerCfg.bitFields.qmAgerLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_UMAC_SW_MASTER3_LOGGER_CFG, QmAgerLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableRxClassLogs	
	
Description:
------------
	enable or disable HW Rx Classifier logs
Input:
-----
	bool shortLogEnable - enable / disable  Rx Classifier  logs
	bool longLogEnable - enable / disable  Rx Classifier Rtd logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxClassLogs(bool shortLogEnable , bool longLogEnable)
{
	RegRxClassifierRxClasLoggerControl_u rxClasLoggerControl;
	RegRxClassifierRxClasLoggerActive_u  rxClasLoggerActive;

	// disable  logs
	rxClasLoggerControl.val = 0;
	RegAccess_Write(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_CONTROL,rxClasLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_ACTIVE, &rxClasLoggerActive.val);
	}while (rxClasLoggerActive.bitFields.rxClasLoggerActive == 1);
	
	// enable logs
	rxClasLoggerControl.bitFields.rxClasLoggerEn = (shortLogEnable || longLogEnable);
	rxClasLoggerControl.bitFields.rxClasLoggerRtdEn = (shortLogEnable || longLogEnable);

	rxClasLoggerControl.bitFields.rxClasLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_CONTROL, rxClasLoggerControl.val);	
}


/**********************************************************************************
hwLoggerEnableDisableBeamFormLogs	
	
Description:
------------
	enable or disable HW BeamForm logs
Input:
-----
	bool shortLogEnable - enable / disable bfImplicit and bfReport  logs
	bool longLogEnable - enable / disable Ndpa and PhyNdp  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBeamFormLogs(bool shortLogEnable , bool longLogEnable)
{
	RegMacBfBfLoggerControl_u bfLoggerControl;
	RegMacBfBfLoggerActive_u  bfLoggerActive;

	// disable  logs
	bfLoggerControl.val = 0;
	RegAccess_Write(REG_MAC_BF_BF_LOGGER_CONTROL,bfLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_BF_BF_LOGGER_ACTIVE, &bfLoggerActive.val);
	}while (bfLoggerActive.bitFields.bfLoggerActive == 1);
	
	// enable logs
	bfLoggerControl.bitFields.bfNdpaLoggerEn = longLogEnable;
	bfLoggerControl.bitFields.bfPhyNdpLoggerEn = longLogEnable;
	bfLoggerControl.bitFields.bfImplicitLoggerEn = shortLogEnable;
	bfLoggerControl.bitFields.bfReportLoggerEn = shortLogEnable;
	bfLoggerControl.bitFields.bfNdpaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	bfLoggerControl.bitFields.bfPhyNdpLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	bfLoggerControl.bitFields.bfImplicitLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	bfLoggerControl.bitFields.bfReportLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_BF_BF_LOGGER_CONTROL, bfLoggerControl.val);	
}


/**********************************************************************************
hwLoggerEnableDisableTxSelectorOrLockerLogs	
	
Description:
------------
	enable or disable HW TX selector bitmap logs
Input:
-----
	bool shortLogEnable - enable / disable  TX selector or Locker  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSelectorOrLockerLogs(bool shortLogEnable , bool longLogEnable)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;

	// disable  logs
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val);
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerEn = FALSE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val);
	}while (txSelLoggerBusy.bitFields.txSelLoggerSelectionLockerBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerSel = 0; // 0x0: Selection logger , 0x1: Locker logger
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableTxSelectorBitmapLogs	
	
Description:
------------
	enable or disable HW TX selector bitmap logs
Input:
-----
	bool shortLogEnable - enable / disable  TX selector bitmap  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSelectorBitmapLogs(bool shortLogEnable , bool longLogEnable)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;

	// disable  logs
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val);
	txSelLoggerCfg.bitFields.txSelLoggerBitmapUpdateEn = FALSE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val);
	}while (txSelLoggerBusy.bitFields.txSelLoggerBitmapUpdateBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerBitmapUpdateEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerBitmapPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableDeliaLogs	
	
Description:
------------
	enable or disable HW Delia logs
Input:
-----
	bool shortLogEnable - enable / disable  Delia  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableDeliaLogs(bool shortLogEnable , bool longLogEnable)
{
	RegPacDelDeliaLoggerControl_u DeliaLoggerControl;
	RegPacDelDeliaLoggerActive_u  DeliaLoggerActive;

	// disable  logs
	DeliaLoggerControl.val = 0;
	RegAccess_Write(REG_PAC_DEL_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_PAC_DEL_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val);
	}while (DeliaLoggerActive.bitFields.deliaLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PAC_DEL_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val);	
}


void hwLoggerEnableDisableDelia1Logs(bool shortLogEnable , bool longLogEnable)
{
	RegPacDel1DeliaLoggerControl_u DeliaLoggerControl;
	RegPacDel1DeliaLoggerActive_u  DeliaLoggerActive;

	// disable  logs
	DeliaLoggerControl.val = 0;
	RegAccess_Write(REG_PAC_DEL1_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_PAC_DEL1_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val);
	}while (DeliaLoggerActive.bitFields.deliaLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PAC_DEL1_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val);	
}


void hwLoggerEnableDisableDelia2Logs(bool shortLogEnable , bool longLogEnable)
{
	RegPacDel2DeliaLoggerControl_u DeliaLoggerControl;
	RegPacDel2DeliaLoggerActive_u  DeliaLoggerActive;

	// disable  logs
	DeliaLoggerControl.val = 0;
	RegAccess_Write(REG_PAC_DEL2_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_PAC_DEL2_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val);
	}while (DeliaLoggerActive.bitFields.deliaLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PAC_DEL2_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val);	
}


void hwLoggerEnableDisableDelia3Logs(bool shortLogEnable , bool longLogEnable)
{
	RegPacDel3DeliaLoggerControl_u DeliaLoggerControl;
	RegPacDel3DeliaLoggerActive_u  DeliaLoggerActive;

	// disable  logs
	DeliaLoggerControl.val = 0;
	RegAccess_Write(REG_PAC_DEL3_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_PAC_DEL3_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val);
	}while (DeliaLoggerActive.bitFields.deliaLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PAC_DEL3_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val);	
}


	
/**********************************************************************************
hwLoggerEnableDisableRxCLogs	
	
Description:
------------
	enable or disable HW RXC logs
Input:
-----
	bool shortLogEnable - enable / disable RXC logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxCLogs(bool shortLogEnable , bool longLogEnable)
{
	RegPacRxdRxdLoggerControl_u rxcLoggerControl;
	RegPacRxdRxdLoggerActive_u  rxcLoggerActive;

	// disable  logs
	rxcLoggerControl.val = 0;
	RegAccess_Write(REG_PAC_RXD_RXD_LOGGER_CONTROL,rxcLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_PAC_RXD_RXD_LOGGER_ACTIVE, &rxcLoggerActive.val);
	}while (rxcLoggerActive.bitFields.rxdLoggerActive == 1);
	
	// enable logs
	rxcLoggerControl.bitFields.rxdLoggerEn = shortLogEnable;
	rxcLoggerControl.bitFields.rxdLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_PAC_RXD_RXD_LOGGER_CONTROL, rxcLoggerControl.val);	
}


/**********************************************************************************
hwLoggerEnableDisableDelineatorLogs	
	
Description:
------------
	enable or disable HW deliniator logs
Input:
-----
	bool shortLogEnable - enable / disable deliniator logs
	bool longLogEnable - report mode : 0 -  Short report (only status and metrics).  1-Elaborated report (status, {ZLD count,delimiter error,MPDU length}, metrics

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableDelineatorLogs(bool shortLogEnable , bool longLogEnable)
{
	RegMacHtExtensionsDelineatorLoggerCtl_u delineatorLoggerCtl;
	RegMacHtExtensionsHwLoggerStatus_u  hwLoggerStatus;

	// disable  logs
	delineatorLoggerCtl.val = 0;
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_DELINEATOR_LOGGER_CTL,delineatorLoggerCtl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_LOGGER_STATUS, &hwLoggerStatus.val);
	}while (hwLoggerStatus.bitFields.delineatorLoggerActive == 1);
	
	// enable logs
	delineatorLoggerCtl.bitFields.delLoggerMode = (shortLogEnable && longLogEnable); 	// "0" = Short report (only status and metrics) , "1" = Elaborated report (status, {ZLD count,delimiter error,MPDU length}, metrics)
	delineatorLoggerCtl.bitFields.delLoggerEna = shortLogEnable;
	delineatorLoggerCtl.bitFields.delLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_HT_EXTENSIONS_DELINEATOR_LOGGER_CTL, delineatorLoggerCtl.val);	
}


/**********************************************************************************
hwLoggerEnableDisableActivityLogs	
	
Description:
------------
	enable or disable HW activity logs
Input:
-----
	bool shortLogEnable - enable / disable activity logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableActivityLogs(bool shortLogEnable , bool longLogEnable)
{
	RegMacHtExtensionsActivityLoggerCtl_u activityLoggerCtl;
	RegMacHtExtensionsHwLoggerStatus_u  hwLoggerStatus;

	// disable  logs
	RegAccess_Read(REG_MAC_HT_EXTENSIONS_ACTIVITY_LOGGER_CTL, &activityLoggerCtl.val);
	activityLoggerCtl.bitFields.actLoggerEna = 0;
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_ACTIVITY_LOGGER_CTL,activityLoggerCtl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_HT_EXTENSIONS_HW_LOGGER_STATUS, &hwLoggerStatus.val);
	}while (hwLoggerStatus.bitFields.actLoggerRunning == 1);
	
	// enable logs
	activityLoggerCtl.bitFields.actLoggerPeriod = 1279; // 4usec - cycle resolution
	activityLoggerCtl.bitFields.actLoggerReportLength = 0x32; 
	activityLoggerCtl.bitFields.actLoggerEna = shortLogEnable;
	activityLoggerCtl.bitFields.actLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_HT_EXTENSIONS_ACTIVITY_LOGGER_CTL, activityLoggerCtl.val);	
}


/**********************************************************************************
hwLoggerEnableDisableSecurityLogs	
	
Description:
------------
	enable or disable HW security logs
Input:
-----
	bool shortLogEnable - enable / disable security logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableSecurityLogs(bool shortLogEnable , bool longLogEnable)
{
	RegMacWepSecLogger_u wepSecLogger;
	RegMacWepSecLoggerActive_u  wepSecLoggerActive;

	// disable  logs
	wepSecLogger.val = 0;
	RegAccess_Write(REG_MAC_WEP_SEC_LOGGER,wepSecLogger.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_WEP_SEC_LOGGER_ACTIVE, &wepSecLoggerActive.val);
	}while (wepSecLoggerActive.bitFields.secLoggerActive == 1);
	
	// enable logs
	wepSecLogger.bitFields.secLoggerEn = shortLogEnable;
	wepSecLogger.bitFields.secLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_WEP_SEC_LOGGER, wepSecLogger.val);	
}

void hwLoggerEnableDisableSecurity1Logs(bool shortLogEnable , bool longLogEnable)
{
	RegMacWep1SecLogger_u wepSecLogger;
	RegMacWep1SecLoggerActive_u  wepSecLoggerActive;

	// disable  logs
	wepSecLogger.val = 0;
	RegAccess_Write(REG_MAC_WEP1_SEC_LOGGER,wepSecLogger.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_WEP1_SEC_LOGGER_ACTIVE, &wepSecLoggerActive.val);
	}while (wepSecLoggerActive.bitFields.secLoggerActive == 1);
	
	// enable logs
	wepSecLogger.bitFields.secLoggerEn = shortLogEnable;
	wepSecLogger.bitFields.secLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_WEP1_SEC_LOGGER, wepSecLogger.val);	
}


void hwLoggerEnableDisableSecurity2Logs(bool shortLogEnable , bool longLogEnable)
{
	RegMacWep2SecLogger_u wepSecLogger;
	RegMacWep2SecLoggerActive_u  wepSecLoggerActive;

	// disable  logs
	wepSecLogger.val = 0;
	RegAccess_Write(REG_MAC_WEP2_SEC_LOGGER,wepSecLogger.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_WEP2_SEC_LOGGER_ACTIVE, &wepSecLoggerActive.val);
	}while (wepSecLoggerActive.bitFields.secLoggerActive == 1);
	
	// enable logs
	wepSecLogger.bitFields.secLoggerEn = shortLogEnable;
	wepSecLogger.bitFields.secLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_WEP2_SEC_LOGGER, wepSecLogger.val);	
}


void hwLoggerEnableDisableSecurity3Logs(bool shortLogEnable , bool longLogEnable)
{
	RegMacWep3SecLogger_u wepSecLogger;
	RegMacWep3SecLoggerActive_u  wepSecLoggerActive;

	// disable  logs
	wepSecLogger.val = 0;
	RegAccess_Write(REG_MAC_WEP3_SEC_LOGGER,wepSecLogger.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_WEP3_SEC_LOGGER_ACTIVE, &wepSecLoggerActive.val);
	}while (wepSecLoggerActive.bitFields.secLoggerActive == 1);
	
	// enable logs
	wepSecLogger.bitFields.secLoggerEn = shortLogEnable;
	wepSecLogger.bitFields.secLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_WEP3_SEC_LOGGER, wepSecLogger.val);	
}


/**********************************************************************************
hwLoggerEnableDisableBeaconLogs	
	
Description:
------------
	enable or disable HW beacon logs
Input:
-----
	bool shortLogEnable - enable / disable beacon logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBeaconLogs(bool shortLogEnable , bool longLogEnable)
{
	RegBeaconBeaconLoggerCfg_u 	 beaconLoggerCfg;
	RegBeaconBeaconLoggerBusy_u  beaconLoggerBusy;

	// disable  logs
	beaconLoggerCfg.val = 0;
	RegAccess_Write(REG_BEACON_BEACON_LOGGER_CFG,beaconLoggerCfg.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_BEACON_BEACON_LOGGER_BUSY, &beaconLoggerBusy.val);
	}while (beaconLoggerBusy.bitFields.beaconTimersLoggerBusy == 1);
	
	// enable logs
	beaconLoggerCfg.bitFields.beaconTimersLoggerEn = shortLogEnable;
	beaconLoggerCfg.bitFields.beaconTimersLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_BEACON_BEACON_LOGGER_CFG, beaconLoggerCfg.val);	
}


/**********************************************************************************
hwLoggerEnableDisableUnknown	
	
Description:
------------
	dummy function for empty entry in HW fifo configuration matrix
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableUnknown(bool shortLogEnable , bool longLogEnable)
{
	DEBUG_FATAL("hwLoggerEnableDisableUnknown");
}



void hwLoggerEnableDisableTxPdAccLogs(bool shortLogEnable , bool longLogEnable)
{
	RegTxPdAccTxPdAccLoggerControl_u txPdAccLoggerControl;
	RegTxPdAccTxPdAccLoggerActive_u  txPdAccLoggerActive;

	// disable  logs
	txPdAccLoggerControl.val = 0;
	RegAccess_Write(REG_TX_PD_ACC_TX_PD_ACC_LOGGER_CONTROL,txPdAccLoggerControl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_TX_PD_ACC_TX_PD_ACC_LOGGER_ACTIVE, &txPdAccLoggerActive.val);
	}while (txPdAccLoggerActive.bitFields.txPdAccLoggerActive == 1);
	
	// enable logs
	txPdAccLoggerControl.bitFields.txPdAccLoggerEn = shortLogEnable;
	txPdAccLoggerControl.bitFields.txPdAccLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_TX_PD_ACC_TX_PD_ACC_LOGGER_CONTROL, txPdAccLoggerControl.val);
}

void hwLoggerEnableDisableMuLockerLogs(bool shortLogEnable , bool longLogEnable)
{
	DEBUG_FATAL("hwLoggerEnableDisableMuLockerLogs");
}

void hwLoggerEnableDisablePacExtraLogs(bool shortLogEnable , bool longLogEnable)
{
	DEBUG_FATAL("hwLoggerEnableDisablePacExtraLogs");
}

void hwLoggerEnableDisableMuBfLogs(bool shortLogEnable , bool longLogEnable)
{
	DEBUG_FATAL("hwLoggerEnableDisableMuBfLogs");
}

void hwLoggerEnableDisableMuBitmapSelectLogs(bool shortLogEnable , bool longLogEnable)
{
	DEBUG_FATAL("hwLoggerEnableDisableMuBitmapSelectLogs");
}


/**********************************************************************************
HwLogger_Init	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void HwLogger_Init(void)
{	 
	RegLoggerListId_u loggerListsIdsReg; 
	RegLoggerFifoEnable_u loggerFifoEnableReg;
	RegLoggerBufferCloseTimeThr_u bufferClosingTimeThr;
	RegLoggerStartMaskTriggerLow_u startMaskTriggerLow; 
	RegLoggerStartMaskTriggerLow_u startMaskTriggerHigh; 
	RegLoggerStopMaskTriggerHigh_u stopMaskTrrigerHigh;
	RegLoggerHwUnit1FirstHeaderWord_u hw1FisrtHeaderwordReg;
	RegLoggerMode_u	loggerModeReg;
	uint32 hwHeaderRegAddress; 
	uint8 fifoIndex;  
	HwLogs_u *hwLogs;
	PHYHwLogs_u	*phHwLogs; 

	UNUSED_PARAM(hwLogs);
	UNUSED_PARAM(phHwLogs);

	/*Clear SHRAM buffers and Buffer descriptors*/
	memset(&HwLoggerShramBuffers, 0, sizeof(HwLoggerShramBuffer_t));
	memset(HwLoggerShramBDs, 0, sizeof(LoggerBd_t)*NUM_OF_HW_LOGGER_SHRAM_BUFFERS);

	/*REG_LOGGER_LIST_ID 0x84 */
	/*Buffer descriptors lists*/
	loggerListsIdsReg.val = 0; 
	loggerListsIdsReg.bitFields.bufferEmtyTxDplIdx = HW_Q_MANAGER_TX_POOL_LIST_LOGGER_FREE_BD;
	loggerListsIdsReg.bitFields.bufferFullTxDplIdx = HW_Q_MANAGER_READY_LIST_LOGGER_FULL_BD;
	/*  RDs lists */
	loggerListsIdsReg.bitFields.loggerRdEmtyRxDplIdx = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST; 
	loggerListsIdsReg.bitFields.loggerRdFullRxDplIdx = HW_Q_MANAGER_RX_DONE_LIST_LOGGER; 
	RegAccess_Write(REG_LOGGER_LIST_ID,loggerListsIdsReg.val);
	loggerFifoEnableReg.val = 0; //All fifo's initialies to disable mode 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val); 	
	/* Configure  Buffer Time Threshold*/
	bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD; 
	RegAccess_Write(REG_LOGGER_BUFFER_CLOSE_TIME_THR, bufferClosingTimeThr.val);

	startMaskTriggerLow.val = 0; 
	RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startMaskTriggerLow.val);	
	startMaskTriggerHigh.val = 0; 
	RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startMaskTriggerHigh.val);
	stopMaskTrrigerHigh.bitFields.logLevelToForward = 0; // forward from log level  0  
	RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopMaskTrrigerHigh.val);

	/* initialize all HW first header word to 0, */
	hwHeaderRegAddress = REG_LOGGER_HW_UNIT1_FIRST_HEADER_WORD; 
	hw1FisrtHeaderwordReg.val = 0;
	for(fifoIndex = LOGGER_FIRST_HW_SOURCE_INDEX; fifoIndex < NUM_OF_LOGGER_FIFOS; fifoIndex++)
	{	
		RegAccess_Write((hwHeaderRegAddress + (fifoIndex-LOGGER_FIRST_HW_SOURCE_INDEX)*4), hw1FisrtHeaderwordReg.val);
	}

	HwLoggerSharedParams.maxNumberOfStreams = 0; 

    /* Enable logger clock */
	loggerModeReg.bitFields.listConverterBlockEn = TRUE;
	RegAccess_Write(REG_LOGGER_MODE,loggerModeReg.val);

	memset(HwLoggerStreamSharedParams, 0, sizeof(HwLoggerStreamSharedParams));
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif







