/***************************************************************
 File:		HwLogger.c
 Module:	  	HwLogger
 Purpose: 	holds all the implementation of the HwLogger API functions
 Description:  This module contains the implementation of the HwLogger driver module 

***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "ErrorHandler_Api.h"
#include "mhi_umi.h"
#include "LoggerRegs.h"
#include "shram_hw_logger.h"
#include "ShramRxDescriptors.h"
#include "HwLoggerAPI.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "OSAL_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "HwLoggerLogsDefinitions.h"
#include "Dut_Api.h"
#include "init_ifmsg.h"
#include "shram_man_msgs.h"
#include "PsManager_Api.h"
#include "HostIfAccRegs.h"
#include "BaAnalyzerRegs.h"
#include "PreAggRegs.h"
#include "RxPpRegs.h"
#include "TxHandlerRegs.h"
#include "TxSequencerRegs.h"
#include "RxClassifierRegs.h"
#include "MacBfmRegs.h"
#include "BeaconRegs.h"
#include "TxSelectorRegs.h"
#include "MacHtExtensionsRegs.h"
#include "MacWepRegs.h"
#include "PacDelRegs.h"
#include "QManagerCpu2Regs.h"
#include "DlmRegs.h"
#include "MacGeneralRegs.h"
#include "PacDelRegs.h"
//#include "HwLoggerPhyGenRiscLogDefs.h"
#include "PsSettingRegs.h"
#include "PhyRxTdRegs.h"
#include "TxPdAccRegs.h"
#include "Pac_Api.h"
#include "AutoRespRegs.h"
#include "TfGeneratorRegs.h"
#include "PacRxcRegs.h"
#include "PacExtrapolatorRegs.h"
#include "BsrcRegsRegs.h"
#include "OtfaRegs.h"
#include "RxCoordinatorRegs.h"
#include "MacTxHcRegs.h"
#include "RxMsduParserRegs.h"
#include "ShramHim.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 7

#define MAX_LOGGER_STREAM_NUM 			1 
#define STREAM_ENABLE					1
#define NUM_OF_LOGGER_FIFOS				28
#define LOGGER_MAX_WAIT_ON_FIFO_FLUSH 	(0x1000) //uSec
#define THREE_LS_BYTES_IN_HALF_WORD		0x0FFF
#define THREE_LS_BYTES_IN_WORD			0x00000FFF
#define STREAM_0						0 
#define STREAM_1						1 
#define STREAM_2						2 
#define STREAM_3						3 

#if defined (ENET_INC_UMAC)
#define LOGGER_FAST_DATA_IN_REG 				REG_LOGGER_UMAC_FAST_MES_DATA_IN
#define LOGGER_FILTER_TYPE_LOG_LEVEL			0x0 
#define LOGGER_DATA_IN_PTR_MAX_LOG_LEVEL		0x7
#define LOGGER_HDR_START_TRIGGER_TYPE 			0x0
#define LOGGER_HDR_STOP_TRIGGER_TYPE 			0x1
#define LOGGER_FIRST_HW_SOURCE_INDEX 			0x7
#define LOGGER_DEFAULT_PRIORITY 				0x7
#define LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD	1000 // 100 milisecond threshold
#define LOGGER_BACKGROUND_BUFFER_TIME_THRESHOLD	1000000 // ~inf
#define LOGGER_NUM_OF_HW_FIFO                   16
#define LOGGER_NUM_OF_OPTIONALS_HW_BLOCKS_PER_FIFO 6
#define LOGGER_NUM_OF_COMMON_AND_BAND0_FIFOS	10

#else 
#define LOGGER_FAST_DATA_IN_REG 				REG_LOGGER_LMAC_FAST_MES_DATA_IN
#endif 
//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

typedef struct HwLogger_StreamsSharedParameters 
{
	uint32 assignedFifosId;
	uint32 bufferThreshold:16;
	uint32 logVersion:4;
	uint32 be0:1; 
	uint32 be1:1; 
	uint32 streamEnable:1;
	uint32 streamId:2;  
	uint32 wlanIf:2; 
	uint32 reserved:5;
} HwLogger_StreamsSharedParameters_t;

typedef struct HwLogger_SharedParameters 
{
	uint32 shramBufferSize:16;
	uint32 maxNumberOfStreams:4;
	uint32 numberOfActiveStream:4; 
	uint32 reserved:8;	
} HwLogger_SharedParameters_t;

void hwLoggerEnableDisableUnknown(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxDataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxListsLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxDataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxListsLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmAgerLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxLiberatorLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxPpLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxLiberatorLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccTxInLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccTxOutLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccRxInLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableHostAccRxOutLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxMpdu0DataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmTxMpdu1DataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxMpdu0DataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmRxMpdu1DataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableQmDmaDataLogs(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxHLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxHLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxHLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableTxSequencerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSequencerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSequencerLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableRxCLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxCLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxCLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisablePsManagerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePsManagerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePsManagerLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableDelia0LogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia0LogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia0LogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableDelia1LogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia1LogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDelia1LogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableTxSelectorOrLockerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSelectorOrLockerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSelectorOrLockerLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableTxSelectorBitmapLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSelectorBitmapLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxSelectorBitmapLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableRxClassLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxClassLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxClassLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableSecurityLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableSecurityLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableSecurityLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableBeaconLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeaconLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeaconLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableActivityLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableActivityLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeamFormLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeamFormLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBeamFormLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableBaAnlayzerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBaAnlayzerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBaAnlayzerLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableTxPdAccLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxPdAccLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxPdAccLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableMuLockerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuLockerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuLockerLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableMuBitmapSelectLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuBitmapSelectLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableMuBitmapSelectLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisablePreAggLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePreAggLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePreAggLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableAutoRespLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableAutoRespLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableAutoRespLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisableTfGenLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTfGenLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTfGenLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId );
void hwLoggerEnableDisablePacExtrLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePacExtrLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePacExtrLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId);
void hwLoggerEnableDisablePlanLockerLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePlanLockerLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisablePlanLockerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);
void hwLoggerEnableDisableBsrcLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBsrcLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableBsrcLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);
void hwLoggerEnableDisableRxHcAndDeaggLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableRxHcAndDeaggLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableDeaggLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);
void hwLoggerEnableDisableOtfaLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableOtfaLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableOtfaLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);
void hwLoggerEnableDisableCoordinatorLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableCoordinatorLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableCoordinatorLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);
void hwLoggerEnableDisableTxHeaderConversionLogsBand0(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxHeaderConversionLogsBand1(bool shortLogEnable , bool longLogEnable);
void hwLoggerEnableDisableTxHeaderConversionLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId);


typedef void (*hwBlocksLogHandler)(bool , bool);
static const hwBlocksLogHandler hwLogsHandler[LOGGER_NUM_OF_HW_FIFO][LOGGER_NUM_OF_OPTIONALS_HW_BLOCKS_PER_FIFO] = 
{
	//Common section 
	{hwLoggerEnableDisableHostAccTxInLogs		, hwLoggerEnableDisableQmRxListsLogs		, hwLoggerEnableDisableTxLiberatorLogs				, hwLoggerEnableDisableRxPpLogs						, hwLoggerEnableDisableActivityLogsBand0 		, hwLoggerEnableDisableQmTxMpdu1DataLogs 	},
	{hwLoggerEnableDisableHostAccTxOutLogs		, hwLoggerEnableDisableQmRxDataLogs			, hwLoggerEnableDisableTxLiberatorLogs				, hwLoggerEnableDisableRxPpLogs						, hwLoggerEnableDisableActivityLogsBand1 		, hwLoggerEnableDisableQmTxMpdu0DataLogs 	},
	{hwLoggerEnableDisableHostAccRxInLogs		, hwLoggerEnableDisableQmTxListsLogs		, hwLoggerEnableDisableRxLiberatorLogs				, hwLoggerEnableDisableQmAgerLogs					, hwLoggerEnableDisableQmDmaDataLogs			, hwLoggerEnableDisableQmRxMpdu1DataLogs 	},
	{hwLoggerEnableDisableHostAccRxOutLogs		, hwLoggerEnableDisableQmTxDataLogs			, hwLoggerEnableDisableRxLiberatorLogs				, hwLoggerEnableDisableQmAgerLogs					, hwLoggerEnableDisableUnknown /*Reserved 	*/	, hwLoggerEnableDisableQmRxMpdu0DataLogs 	},
	//band 0 section
	{hwLoggerEnableDisableUnknown /*Reserved*/	, hwLoggerEnableDisableSecurityLogsBand0	, hwLoggerEnableDisablePreAggLogsBand0				, hwLoggerEnableDisableMuBitmapSelectLogsBand0 		, hwLoggerEnableDisableBsrcLogsBand0 			, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableRxClassLogsBand0		, hwLoggerEnableDisableUnknown /*Reserved*/	, hwLoggerEnableDisableOtfaLogsBand0				, hwLoggerEnableDisableMuLockerLogsBand0 			, hwLoggerEnableDisableUnknown /*Reserved 	*/	, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableRxHcAndDeaggLogsBand0	, hwLoggerEnableDisableBeamFormLogsBand0	, hwLoggerEnableDisableTxHeaderConversionLogsBand0	, hwLoggerEnableDisableTxSelectorOrLockerLogsBand0	, hwLoggerEnableDisableBaAnlayzerLogsBand0 		, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableCoordinatorLogsBand0	, hwLoggerEnableDisablePsManagerLogsBand0	, hwLoggerEnableDisableTxPdAccLogsBand0				, hwLoggerEnableDisableTxSelectorBitmapLogsBand0	, hwLoggerEnableDisableDelia0LogsBand0 			, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableAutoRespLogsBand0		, hwLoggerEnableDisableUnknown /*Reserved*/	, hwLoggerEnableDisableTfGenLogsBand0				, hwLoggerEnableDisableTxSequencerLogsBand0			, hwLoggerEnableDisablePlanLockerLogsBand0 		, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableBsrcLogsBand0			, hwLoggerEnableDisablePacExtrLogsBand0		, hwLoggerEnableDisableDelia1LogsBand0				, hwLoggerEnableDisableTxHLogsBand0					, hwLoggerEnableDisableBeaconLogsBand0 			, hwLoggerEnableDisableRxCLogsBand0 	 	},
	//band 1 section
	{hwLoggerEnableDisableUnknown /*Reserved*/	, hwLoggerEnableDisableSecurityLogsBand1	, hwLoggerEnableDisablePreAggLogsBand1				, hwLoggerEnableDisableMuBitmapSelectLogsBand1		, hwLoggerEnableDisableBsrcLogsBand1			, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableRxClassLogsBand1		, hwLoggerEnableDisableUnknown /*Reserved*/ , hwLoggerEnableDisableOtfaLogsBand1				, hwLoggerEnableDisableMuLockerLogsBand1			, hwLoggerEnableDisableUnknown /*Reserved	*/	, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableRxHcAndDeaggLogsBand1 , hwLoggerEnableDisableBeamFormLogsBand1	, hwLoggerEnableDisableTxHeaderConversionLogsBand1	, hwLoggerEnableDisableTxSelectorOrLockerLogsBand1	, hwLoggerEnableDisableBaAnlayzerLogsBand1		, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableCoordinatorLogsBand1	, hwLoggerEnableDisablePsManagerLogsBand1	, hwLoggerEnableDisableTxPdAccLogsBand1 			, hwLoggerEnableDisableTxSelectorBitmapLogsBand1	, hwLoggerEnableDisableDelia0LogsBand1			, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableAutoRespLogsBand1 	, hwLoggerEnableDisableUnknown /*Reserved*/ , hwLoggerEnableDisableTfGenLogsBand1				, hwLoggerEnableDisableTxSequencerLogsBand1 		, hwLoggerEnableDisablePlanLockerLogsBand1		, hwLoggerEnableDisableUnknown /*Reserved*/	},
	{hwLoggerEnableDisableBsrcLogsBand1 		, hwLoggerEnableDisablePacExtrLogsBand1 	, hwLoggerEnableDisableDelia1LogsBand1				, hwLoggerEnableDisableTxHLogsBand1 				, hwLoggerEnableDisableBeaconLogsBand1			, hwLoggerEnableDisableRxCLogsBand1 		},
};

//---------------------------------------------------------------------------------
//						Data structures					
//---------------------------------------------------------------------------------
static HwQueueManagerRequestParams_t HwLoggerHwQueueManagerRequestParams;
static HwLogger_SharedParameters_t HwLoggerSharedParams; 
static HwLogger_StreamsSharedParameters_t HwLoggerStreamSharedParams[MAX_LOGGER_STREAM_NUM];


//---------------------------------------------------------------------------------
//						Private Function 				
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Public Function 			
//---------------------------------------------------------------------------------


/**********************************************************************************
hwLoggerResourceAllocation



Description:
------------
	Allocates the shared ram nuffers and their descriptors. 
	the number of buffers and the buffer size are being calculated according to the logger max stream number 
	Configures the logger's lists ID 
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
uint32 HwLoggerResourceAllocation(uint32 loggerMaxStreamNum)
{
	RegLoggerBufferLenght_u loggerBufferLength; 
	uint32 bufferSize;  
	uint32 numOfBuffers; 
	uint32 i = 0;
	
	ASSERT((0 < loggerMaxStreamNum) && (loggerMaxStreamNum < 5)); 

	/* numOfBuffers =  loggerMaxStreamNum *2 (except 1),  bufferSize = total size/numOfBuffers */
	// 1-> 4,	2-> 4,	 3-> 6,	  4-> 8 
	numOfBuffers = loggerMaxStreamNum<<1; 
	 // in case the loggerMaxStreamNum is one- number of buffers should be multiplied again  
	if (loggerMaxStreamNum == 1)
	{
		numOfBuffers <<= 1;
	}

	bufferSize = (HW_LOGGER_SHARED_RAM_BUFFER_SIZE/numOfBuffers);    // 6K divided to 2,4,6,8 buffers, each one at size 3000,1500,1000,750 
    bufferSize = ((bufferSize + 15) & ~0x0F);   // align to 16 will give us the following sizes: 3008,1504,1008,752

	HwLoggerSharedParams.maxNumberOfStreams = loggerMaxStreamNum; 
	HwLoggerSharedParams.shramBufferSize = bufferSize;

	// set the buffer length to the logger register 
	loggerBufferLength.val = 0; 
	loggerBufferLength.bitFields.bufferLenght = bufferSize; //calculated by hwLoggerResourceAllocation
	RegAccess_Write(REG_LOGGER_BUFFER_LENGHT, loggerBufferLength.val); 
	
	//allocate shared ram buffer desciptors 
	for(i =0; i < numOfBuffers; i++)
	{
		//allocate BDs and make them point to   HwLoggerHwQueueManagerRequestParams +  i*bufferSize
		HwLoggerShramBDs[i].nextPdPointer = SET_NEXT_PD(&HwLoggerShramBDs[i+1]);
		HwLoggerShramBDs[i].bufferLength = bufferSize;
		HwLoggerShramBDs[i].memPointer = (uint32)(CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((((uint32)(&HwLoggerShramBuffers))+(i*bufferSize))));
	}

	HwLoggerShramBDs[i-1].nextPdPointer = NEXT_PD_NULL;
	//Now push to the free BD list 
	HwLoggerHwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM; 
	HwLoggerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_TX_POOL_LIST_LOGGER_FREE_BD;
	HwLoggerHwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwLoggerHwQueueManagerRequestParams.pHeadDesc = HwLoggerShramBDs;
	HwLoggerHwQueueManagerRequestParams.pTailDesc = &HwLoggerShramBDs[i-1];
	HwQManager_PushPacketListToTail(&HwLoggerHwQueueManagerRequestParams); //push all BDs to list 
	/* we might have 4 reserved PDs*/

	return numOfBuffers;
}
/**********************************************************************************
Add Stream req
	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_AddStreamReq(K_MSG *psMsg)
{	
	uint32 fifoIndex; 
	uint32 streamId;
	uint32 regOffset;
	HwLoggerHeader_t extInfo;
	uint16 secondWord;
	uint32 hwFifoIndex;
	RegLoggerHwUnit1SecondHeaderWord_u fifoMsgHdr;
	RegLoggerFifoEnable_u loggerFifoEnableReg; 
	RegLoggerBufferLenght_u loggerBufferLengh; 
	RegMacHtExtensionsLoggerFifosSrcSel_u macHtExtensionsLoggerFifosSrcSel;
	RegMacGeneralLoggerFifosSrcSel_u		macGeneralLoggerFifosSrcSel;
	UMI_ADD_STREAM_REQ_t* addStreamStructurePtr;
   	RegLoggerBufferCloseTimeThr_u bufferClosingTimeThr;
   	RegLoggerMode_u	loggerModeReg;
    
	uint8 i;
	uint32 regAddress = 0x0;
	void* udpHeaderAddr = NULL;
	uint16 hwModuleFifo;
	K_MSG*	lm0Msg;
	K_MSG*	lm1Msg;
	HwLoggerHeader_t* lm0MsgPayload;
	HwLoggerHeader_t* lm1MsgPayload;

	addStreamStructurePtr = (UMI_ADD_STREAM_REQ_t*) pK_MSG_DATA(psMsg);


	// Patch for CDB: If logs for LM0 are turned on - turn them also on for LM1
	// currently it's turned off. It can be removed once the WEB is implemented correctly, and logs can be turned on for band 1.
/*
	if ((addStreamStructurePtr->assignedFifosId & (1<<LOGGER_LM0_FIFO_ID)) != 0)
	{
		addStreamStructurePtr->assignedFifosId |= (1<<LOGGER_LM1_FIFO_ID);
	}
*/
	streamId = addStreamStructurePtr->streamId; 
	/*validation*/
	DEBUG_ASSERT(streamId < MAX_LOGGER_STREAM_NUM); 
	DEBUG_ASSERT(HwLoggerStreamSharedParams[streamId].streamEnable == 0); //stream already enabled! 
	HwLoggerSharedParams.numberOfActiveStream++;
	/* set constant values for SW message header	 */ 
	HwLoggerStreamSharedParams[streamId].bufferThreshold = addStreamStructurePtr->bufferThreshold; 
	HwLoggerStreamSharedParams[streamId].streamEnable = STREAM_ENABLE; 
	HwLoggerStreamSharedParams[streamId].assignedFifosId = addStreamStructurePtr->assignedFifosId;

	/*fill the message header template*/
	/* fill data on a local message header, later will be assigned to the correspond modules*/
	
	extInfo.word1 = 0;
	extInfo.be0 = addStreamStructurePtr->be0;
	extInfo.logVer = addStreamStructurePtr->logVersion;
	extInfo.pri = LOGGER_DEFAULT_PRIORITY; 
	extInfo.be1 = addStreamStructurePtr->be1; 
	extInfo.wlanIf = addStreamStructurePtr->wlanIf;  
	extInfo.dest = addStreamStructurePtr->streamId;
	
	RegAccess_Read(REG_LOGGER_BUFFER_LENGHT, &loggerBufferLengh.val); 
	//the total buffer size was being set at initialization phase, make sure that the register holds the right value 
	ASSERT(loggerBufferLengh.bitFields.bufferLenght == HwLoggerSharedParams.shramBufferSize);
    
    // ** //
    // indicate HIM if logger is background or online
    loggerActiveMode = addStreamStructurePtr->loggerActiveMode;

    // in background mode (also called offline) we dont want to flush buffer at timer, so we set timer high.
    // We only use X buffers, if flush to early(by timer) it will not get full - > less logs in each buffer.
    if (loggerActiveMode == LOGGER_STATE_ACTIVE_BACKGROUND)
    {
        bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_BACKGROUND_BUFFER_TIME_THRESHOLD; 
    }
    else if (loggerActiveMode == LOGGER_STATE_ACTIVE_ONLINE)
    {
       	bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD;    
    }
    
    // write changes to register
    RegAccess_Write(REG_LOGGER_BUFFER_CLOSE_TIME_THR, bufferClosingTimeThr.val);
    
    /* Enable logger clock (to push buffer changing */
	loggerModeReg.bitFields.listConverterBlockEn = TRUE;
	RegAccess_Write(REG_LOGGER_MODE,loggerModeReg.val);
    // ** //
    
	switch(streamId)
    {
	    case STREAM_0:
	    {
			RegLoggerUmacLmacFastMesSecWord_u RegLoggerUmacLmacFastMesSecWord; 
			RegLoggerHostIfHandlerFastMesSecWord_u hostIfhandlerFastMesSecWord;
			RegLoggerSenderFastMesSecWord_u senderPhyGenriscFastMesSecWord;
			RegLoggerLmacHandler2FastMesSecWord_u lmacHandler2FastMesSecWord;
			RegLoggerSenderPhyRisc2FastMesSecWord_u senderPhyRisc2FastMesSecWord;			
			RegLoggerSwPreCalculatedChecksumDst0_u swPreCalcCheckSum;
			RegLoggerBuffer01Thr_u loggerBuffer_01_Threshold; 

			RegAccess_Read(REG_LOGGER_BUFFER0_1THR, &loggerBuffer_01_Threshold.val); 
			loggerBuffer_01_Threshold.bitFields.buffer0Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER0_1THR, loggerBuffer_01_Threshold.val);
			
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst0 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST0, swPreCalcCheckSum.val);

			extInfo.band = CONFIGURATION_MANAGER_BAND_0;
			secondWord = (extInfo.word1 >> 16); // we need to configure the HW only the 16 MSb

			/* Configure the Fast message header (last 16 bits of the second word, both for UM and LMs )*/
			RegLoggerUmacLmacFastMesSecWord.bitFields.umacFastMesSecWord = secondWord; 
			RegLoggerUmacLmacFastMesSecWord.bitFields.lmac1FastMesSecWord = secondWord; 
			RegAccess_Write(REG_LOGGER_UMAC_LMAC_FAST_MES_SEC_WORD, RegLoggerUmacLmacFastMesSecWord.val);

			hostIfhandlerFastMesSecWord.bitFields.hostIfFastMesSecWord		= secondWord;
			hostIfhandlerFastMesSecWord.bitFields.rxHandler1FastMesSecWord	= secondWord;
			RegAccess_Write(REG_LOGGER_HOST_IF_HANDLER_FAST_MES_SEC_WORD, hostIfhandlerFastMesSecWord.val);

			senderPhyGenriscFastMesSecWord.bitFields.sender1FastMesSecWord  = secondWord;
			senderPhyGenriscFastMesSecWord.bitFields.phyRisc1FastMesSecWord = secondWord;
			RegAccess_Write(REG_LOGGER_SENDER_FAST_MES_SEC_WORD, senderPhyGenriscFastMesSecWord.val);

			extInfo.band = CONFIGURATION_MANAGER_BAND_1;
			secondWord = (extInfo.word1 >> 16); // we need to configure the HW only the 16 MSb

			lmacHandler2FastMesSecWord.bitFields.lmac2FastMesSecWord  = secondWord;
			lmacHandler2FastMesSecWord.bitFields.rxHandler2FastMesSecWord = secondWord;
			RegAccess_Write(REG_LOGGER_LMAC_HANDLER2_FAST_MES_SEC_WORD, lmacHandler2FastMesSecWord.val);

			senderPhyRisc2FastMesSecWord.bitFields.sender2FastMesSecWord  = secondWord;
			senderPhyRisc2FastMesSecWord.bitFields.phyRisc2FastMesSecWord = secondWord;
			RegAccess_Write(REG_LOGGER_SENDER_PHY_RISC2_FAST_MES_SEC_WORD, senderPhyRisc2FastMesSecWord.val);
					
			break;
	    }
#if (MAX_LOGGER_STREAM_NUM > STREAM_1)
	    case STREAM_1:
		{ 
			RegLoggerSwPreCalculatedChecksumDst1_u swPreCalcCheckSum; 
			RegLoggerBuffer01Thr_u loggerBuffer_01_Threshold;
				
			RegAccess_Read(REG_LOGGER_BUFFER0_1THR, &loggerBuffer_01_Threshold.val); 
			loggerBuffer_01_Threshold.bitFields.buffer1Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER0_1THR, loggerBuffer_01_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst1 =  addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST1, swPreCalcCheckSum.val);
			break;
	    }
#endif
#if (MAX_LOGGER_STREAM_NUM > STREAM_2)
	    case STREAM_2:
	    {
			RegLoggerSwPreCalculatedChecksumDst2_u swPreCalcCheckSum;
			RegLoggerBuffer23Thr_u loggerBuffer_23_Threshold;

			RegAccess_Read(REG_LOGGER_BUFFER2_3THR, &loggerBuffer_23_Threshold.val); 
			loggerBuffer_23_Threshold.bitFields.buffer2Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER2_3THR, loggerBuffer_23_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst2 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST2, swPreCalcCheckSum.val);
			break;
		}
#endif
#if (MAX_LOGGER_STREAM_NUM > STREAM_3)
		case STREAM_3: 
		{
			RegLoggerSwPreCalculatedChecksumDst3_u swPreCalcCheckSum; 
			RegLoggerBuffer23Thr_u loggerBuffer_23_Threshold;
			
			RegAccess_Read(REG_LOGGER_BUFFER2_3THR, &loggerBuffer_23_Threshold.val); 
			loggerBuffer_23_Threshold.bitFields.buffer3Thr = addStreamStructurePtr->bufferThreshold;
			RegAccess_Write(REG_LOGGER_BUFFER2_3THR, loggerBuffer_23_Threshold.val);
			swPreCalcCheckSum.val = 0; 
			swPreCalcCheckSum.bitFields.swPreCalculatedChecksumDst3 = addStreamStructurePtr->swPreCalcChecksum; 
			RegAccess_Write(REG_LOGGER_SW_PRE_CALCULATED_CHECKSUM_DST3, swPreCalcCheckSum.val);
			break; 
		}
#endif
	    default:
	        break;
	}
	/* configure each HW fifo header, that is assigned with that stream*/
	regOffset = 0;

	fifoIndex = LOGGER_FIRST_HW_SOURCE_INDEX;
	for ( i = 0 ; i < (LOGGER_NUM_OF_HW_FIFOS>>1) ; i++)
	{

		// The first 4 fifos are common (marked as band0). The next 6 are band0. The last 6 are band1.
		if (i*2 < LOGGER_NUM_OF_COMMON_AND_BAND0_FIFOS)
		{
			extInfo.band = CONFIGURATION_MANAGER_BAND_0;
			secondWord = (extInfo.word1 >> 16); // we need to configure the HW only the 16 MSb
		}
		else
		{
			extInfo.band = CONFIGURATION_MANAGER_BAND_1;
			secondWord = (extInfo.word1 >> 16); // we need to configure the HW only the 16 MSb
		}
		
		if(addStreamStructurePtr->assignedFifosId & (0x1 << fifoIndex))
		{
			hwFifoIndex = fifoIndex - LOGGER_FIRST_HW_SOURCE_INDEX;
			hwModuleFifo = addStreamStructurePtr->hwModuleFifo[hwFifoIndex];
			hwLogsHandler[hwFifoIndex][hwModuleFifo](TRUE , FALSE);
			
			regAddress = REG_LOGGER_HW_UNIT1_SECOND_HEADER_WORD + regOffset; 
			RegAccess_Read(regAddress, &fifoMsgHdr.val);
			fifoMsgHdr.bitFields.hwUnit1SecondHeaderWord = secondWord; 
			RegAccess_Write(regAddress, fifoMsgHdr.val);
		}

		fifoIndex++;
		
		if(addStreamStructurePtr->assignedFifosId & (0x1 << fifoIndex))
		{
			
			hwFifoIndex = fifoIndex - LOGGER_FIRST_HW_SOURCE_INDEX;
			hwModuleFifo = addStreamStructurePtr->hwModuleFifo[hwFifoIndex];
			hwLogsHandler[hwFifoIndex][hwModuleFifo](TRUE , FALSE);

			regAddress = REG_LOGGER_HW_UNIT1_SECOND_HEADER_WORD + regOffset;
			RegAccess_Read(regAddress, &fifoMsgHdr.val);
			fifoMsgHdr.bitFields.hwUnit2SecondHeaderWord = secondWord; 
			RegAccess_Write(regAddress, fifoMsgHdr.val);


		}

		fifoIndex++;
		regOffset+=4;
	}
	
	for(i =0;i <LOGGER_NUM_OF_HW_FIFOS;i++)
	{
			addStreamStructurePtr->hwModuleFifo[i]++;
	}
	
	macGeneralLoggerFifosSrcSel.bitFields.fifo0SrcSel = addStreamStructurePtr->hwModuleFifo[0];
	macGeneralLoggerFifosSrcSel.bitFields.fifo1SrcSel = addStreamStructurePtr->hwModuleFifo[1];
	macGeneralLoggerFifosSrcSel.bitFields.fifo2SrcSel = addStreamStructurePtr->hwModuleFifo[2];
	macGeneralLoggerFifosSrcSel.bitFields.fifo3SrcSel = addStreamStructurePtr->hwModuleFifo[3];

	RegAccess_Write(REG_MAC_GENERAL_LOGGER_FIFOS_SRC_SEL, macGeneralLoggerFifosSrcSel.val);


	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo0SrcSel = addStreamStructurePtr->hwModuleFifo[4];
	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo1SrcSel = addStreamStructurePtr->hwModuleFifo[5];
	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo2SrcSel = addStreamStructurePtr->hwModuleFifo[6];
	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo3SrcSel = addStreamStructurePtr->hwModuleFifo[7];
	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo4SrcSel = addStreamStructurePtr->hwModuleFifo[8];
	macHtExtensionsLoggerFifosSrcSel.bitFields.fifo5SrcSel = addStreamStructurePtr->hwModuleFifo[9];
	
	RegAccess_WritePerBand(REG_MAC_HT_EXTENSIONS_LOGGER_FIFOS_SRC_SEL, macHtExtensionsLoggerFifosSrcSel.val, CONFIGURATION_MANAGER_BAND_0);

#ifdef GEN6_REL3_5_HW_BUG_PATCH
	if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
	{
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo0SrcSel = addStreamStructurePtr->hwModuleFifo[10];
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo1SrcSel = addStreamStructurePtr->hwModuleFifo[11];
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo2SrcSel = addStreamStructurePtr->hwModuleFifo[12];
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo3SrcSel = addStreamStructurePtr->hwModuleFifo[13];
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo4SrcSel = addStreamStructurePtr->hwModuleFifo[14];
		macHtExtensionsLoggerFifosSrcSel.bitFields.fifo5SrcSel = addStreamStructurePtr->hwModuleFifo[15];

		RegAccess_WritePerBand(REG_MAC_HT_EXTENSIONS_LOGGER_FIFOS_SRC_SEL, macHtExtensionsLoggerFifosSrcSel.val, CONFIGURATION_MANAGER_BAND_1);
	}

	/* Enable the correspond fifo's */
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnableReg.val); 
	loggerFifoEnableReg.val |= addStreamStructurePtr->assignedFifosId; 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val);

	if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_UM_FIFO_ID))
	{
		extInfo.band = CONFIGURATION_MANAGER_COMMON_BAND;
		logFillMsgHdrTemplate(&extInfo);
	}

	if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_LM0_FIFO_ID))
	{
		lm0Msg = OSAL_GET_MESSAGE(sizeof(HwLoggerHeader_t));
		lm0MsgPayload = (HwLoggerHeader_t*) pK_MSG_DATA(lm0Msg);
		extInfo.band = CONFIGURATION_MANAGER_BAND_0;
		MEMCPY(lm0MsgPayload, &extInfo, sizeof(HwLoggerHeader_t));
		// Forward the MSG to LM0 to configure the log MSG header.
		OSAL_SEND_MESSAGE(PAC_MANAGER_LOGGER_FILL_MSG_HEADER_TEMPLATE_REQ, TASK_PAC_MANAGER, lm0Msg, ConfigurationManager_GetFirstVapForBand0());
	}
#ifdef GEN6_REL3_5_HW_BUG_PATCH
	if(ConfigurationManager_GetBandConfigurationMode() != CONFIGURATION_MODE_SINGLE_BAND)
#endif
	{
		if (addStreamStructurePtr->assignedFifosId & (1<<LOGGER_LM1_FIFO_ID))
		{
			lm1Msg = OSAL_GET_MESSAGE(sizeof(HwLoggerHeader_t));
			lm1MsgPayload = (HwLoggerHeader_t*) pK_MSG_DATA(lm1Msg);
			extInfo.band = CONFIGURATION_MANAGER_BAND_1;
			MEMCPY(lm1MsgPayload, &extInfo, sizeof(HwLoggerHeader_t));
			// Forward the MSG to LM0 to configure the log MSG header.
			OSAL_SEND_MESSAGE(PAC_MANAGER_LOGGER_FILL_MSG_HEADER_TEMPLATE_REQ, TASK_PAC_MANAGER, lm1Msg, ConfigurationManager_GetFirstVapForBand1());
		}
	}
	/*configure the udp parameters */
	udpHeaderAddr = (void*)(LOGGER_WRAPPER_UDP + UDP_HEADER_WORD_SIZE*streamId); 
	memcpy32(udpHeaderAddr,&(addStreamStructurePtr->udpHeader[0]),UDP_HEADER_WORD_SIZE); 

	//configure UDP Header structure in Rx Handler - used for sniffer
	/* RxHandler is in LM in gen6 */
	OSAL_SEND_MESSAGE(PAC_MANAGER_SET_SNIFFER_UDP_HDR_REQ, TASK_PAC_MANAGER, psMsg, ConfigurationManager_GetFirstVapForBand0());
}


/**********************************************************************************
HwLogger_RemoveStreamReq

	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_RemoveStreamReq(uint16 streamId)
{	
	TX_INTERRUPT_SAVE_AREA;
	RegLoggerFifoEnable_u loggerFifoEnableReg; 
	RegLoggerNumMessages1_u numMessagesInfifo1; 
	RegLoggerNumMessages2_u numMessagesInfifo2; 
	RegLoggerNumMessages3_u numMessagesInfifo3; 
	uint32 timeStamp;

	HwLoggerStreamSharedParams[streamId].streamEnable = 0;
	/* disable the stream's  fifos*/
	loggerFifoEnableReg.val = 0; 
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnableReg.val); 
	loggerFifoEnableReg.val ^=  HwLoggerStreamSharedParams[streamId].assignedFifosId;
	ASSERT(loggerFifoEnableReg.val == 0); //currently we use only 1 stream so there should not be any fifos enabled after remove stream request 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val); 


	//currently we use only 1 stream so there should not be any messages left in fifos after remove stream request 
	RegAccess_Read(REG_LOGGER_NUM_MESSAGES_1, &numMessagesInfifo1.val); 
	RegAccess_Read(REG_LOGGER_NUM_MESSAGES_2, &numMessagesInfifo2.val); 
	RegAccess_Read(REG_LOGGER_NUM_MESSAGES_3, &numMessagesInfifo3.val); 

	//wait for all logs to be flushed from all fifos
	timeStamp = GET_TSF_TIMER_LOW();
	while((numMessagesInfifo1.val != 0) || (numMessagesInfifo2.val != 0) || (numMessagesInfifo3.val != 0))
	{
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_1, &numMessagesInfifo1.val); 
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_2, &numMessagesInfifo2.val);
		RegAccess_Read(REG_LOGGER_NUM_MESSAGES_3, &numMessagesInfifo3.val);
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= LOGGER_MAX_WAIT_ON_FIFO_FLUSH);
	}

	OSAL_DISABLE_INTERRUPTS(&interrupt_save); // disable interrupts so we won't get any logs when trying to send the last one of these stream 

	//send a fast log message with the expedite bit on, to the current stream - in order to  flush the buffer 
	HwLogger_SendFastLog(LOG_LOCAL_FID, LOG_LOCAL_GID,  __LINE__, 1); //"STREAM_TERMINATION"  log 

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
//	HwLogger_EnablePhyLogs();

	HwLoggerSharedParams.numberOfActiveStream--;
}


///**********************************************************************************
//HwLogger_EnablePhyLogs
//	
//Description:
//------------
//	
//Input:
//-----
//	Max stream number 
//Output:
//-------
//	None
//Returns:
//--------
//	void
//**********************************************************************************/
//void HwLogger_EnablePhyLogs(void) 
//{
//	uint32 i;
//	uint32 enableLogs = FALSE;
//	RegPhyRxTdPhyRxtdReg7F_u regPhyRxTdPhyRxtdReg7F;
//
//	regPhyRxTdPhyRxtdReg7F.val = 0;
//
//	for (i = 0; i < MAX_LOGGER_STREAM_NUM; i++)
//	{
//		if (HwLoggerStreamSharedParams[i].assignedFifosId & ((1 << LOGGER_PHY_GENRISC_FIFO_ID) | (1 << LOGGER_PHY_HW_FIFO_ID)))
//		{
//			enableLogs = TRUE;
//			break;
//		}
//	}
//
//	/* set PHY logs */
//	RegAccess_Read(REG_PHY_RX_TD_PHY_RXTD_REG7F, &regPhyRxTdPhyRxtdReg7F.val); 
//	regPhyRxTdPhyRxtdReg7F.bitFields.loggerTestBusGclkEn = enableLogs; 
//	RegAccess_Write(REG_PHY_RX_TD_PHY_RXTD_REG7F, regPhyRxTdPhyRxtdReg7F.val);
//}


/**********************************************************************************
HwLogger_SendFastLog



	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SendFastLog(uint32 FID, uint32 GID, uint32 LID, uint32 exp) 
{
	HwLoggerHeader_t header; 

	header.word0 = 0;
	header.oid= 0; 
	header.fid = FID; 
	header.gid = GID; 
	header.lid = LID; 
	header.exp = exp; 

	RegAccess_Write(LOGGER_FAST_DATA_IN_REG, header.word0); 
}


/**********************************************************************************
HwLogger_AddFifoFilterReq




	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_AddFifoFilterReq(UMI_ADD_FIFO_FILTER_REQ_t* addFifoFilterReq)
{
	if(addFifoFilterReq->FilterType == LOGGER_FILTER_TYPE_LOG_LEVEL)
	{
		RegLoggerStopMaskTriggerHigh_u stopMaskTriggerHigh_u; //this register holds the log level from which logs would be forward
		RegAccess_Read(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, &stopMaskTriggerHigh_u.val); 
		stopMaskTriggerHigh_u.bitFields.logLevelToForward = addFifoFilterReq->logLevel; //set the correspond log level 
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopMaskTriggerHigh_u.val);		
	}
	else // LOGGER_FILTER_TYPE_FIFO (disable specific modlule's fifos )
	{
		RegLoggerFifoEnable_u loggerFifoEnable; 
		uint16 enabledFifoMask =0; 
		
		enabledFifoMask = ~(addFifoFilterReq->fifosBitmap); 
		RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnable.val); 
		loggerFifoEnable.val &= enabledFifoMask; //mask the correspond fifios
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnable.val);
	}
}


/**********************************************************************************
HwLogger_RemoveFifoFilterReq


	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_RemoveFifoFilterReq(UMI_REMOVE_FIFO_FILTER_REQ_t* removeFifoFilterReq)
{
	if(removeFifoFilterReq->FilterType == LOGGER_FILTER_TYPE_LOG_LEVEL)
	{
		RegLoggerStopMaskTriggerHigh_u stopMaskTriggerHigh_u; //this register holds the log level from which logs would be forward
		RegAccess_Read(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, &stopMaskTriggerHigh_u.val); 
		stopMaskTriggerHigh_u.bitFields.logLevelToForward = LOGGER_DATA_IN_PTR_MAX_LOG_LEVEL; //set the maximum log level so no logs would be filtered 
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, stopMaskTriggerHigh_u.val);	
	}
	else // LOGGER_FILTER_TYPE_FIFO (enable specific modlule's fifos )
	{
		RegLoggerFifoEnable_u loggerFifoEnable; 
		uint16 enabledFifoMask =0; 
	
		enabledFifoMask =	removeFifoFilterReq->fifosBitmap; 
		RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerFifoEnable.val); 
		loggerFifoEnable.val |= enabledFifoMask; //mask the correspond fifios
		RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnable.val);
	}
}


/**********************************************************************************
HwLogger_SetHdrTriggerReq


	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SetHdrTriggerReq(UMI_SET_TRIGGER_REQ_t* setHdrTriggerReq) 
{
	if(setHdrTriggerReq->u8TriggerType == LOGGER_HDR_START_TRIGGER_TYPE)
	{
		RegLoggerStartDataTriggerLow_u 	startDataTriggerLow;
		RegLoggerStartDataTriggerHigh_u StartDataTriggerHigh; 
		RegLoggerStartMaskTriggerLow_u	startDataTriggerMaskLow;
		RegLoggerStartMaskTriggerHigh_u	startDataTriggerMaskHigh; 
	
		startDataTriggerLow.bitFields.startDataTriggerLow = setHdrTriggerReq->messageHeaderFirst4Bytes; 
		StartDataTriggerHigh.bitFields.startDataTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		startDataTriggerMaskLow.bitFields.startMaskTriggerLow = setHdrTriggerReq->messageHeaderMaskFirst4Bytes; 
		startDataTriggerMaskHigh.bitFields.startMaskTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		/*Write the header to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_DATA_TRIGGER_LOW, startDataTriggerLow.val);
		RegAccess_Write(REG_LOGGER_START_DATA_TRIGGER_HIGH, StartDataTriggerHigh.val);
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startDataTriggerMaskHigh.val);	
	}
	else
	{
		RegLoggerStopDataTriggerLow_u 	stopDataTriggerLow;
		RegLoggerStopDataTriggerHigh_u 	stopDataTriggerHigh; 
		RegLoggerStopMaskTriggerLow_u	stopDataTriggerMaskLow;
		RegLoggerStopMaskTriggerHigh_u	stopDataTriggerMaskHigh; 
		RegLoggerTimeToRecorderAfterStop_u timeToRecordAfterStop; 

		ASSERT(setHdrTriggerReq->u8TriggerType == LOGGER_HDR_STOP_TRIGGER_TYPE);//Make sure the type is valid 
	
		stopDataTriggerLow.bitFields.stopDataTriggerLow = setHdrTriggerReq->messageHeaderFirst4Bytes; 
		stopDataTriggerHigh.bitFields.stopDataTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes; 
		stopDataTriggerMaskLow.bitFields.stopMaskTriggerLow = setHdrTriggerReq->messageHeaderMaskFirst4Bytes; 
		stopDataTriggerMaskHigh.bitFields.stopMaskTriggerHigh = setHdrTriggerReq->messageHeaderMaskLast2Bytes;
		timeToRecordAfterStop.bitFields.timeToRecorderAfterStop = 0x0; // Time to record after sop 
		/*Write the header to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_DATA_TRIGGER_LOW, stopDataTriggerLow.val);
		RegAccess_Write(REG_LOGGER_STOP_DATA_TRIGGER_HIGH, stopDataTriggerHigh.val);
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_LOW, stopDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopDataTriggerMaskHigh.val);
		RegAccess_Write(REG_LOGGER_TIME_TO_RECORDER_AFTER_STOP, timeToRecordAfterStop.val);
	}

}


/**********************************************************************************
HwLogger_ResetHdrTriggerReq



	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_ResetHdrTriggerReq(UMI_RESET_TRIGGER_REQ_t* resetHdrTriggerReq)
{
	if(resetHdrTriggerReq->u8TriggerType == LOGGER_HDR_START_TRIGGER_TYPE)
	{ 
		RegLoggerStartMaskTriggerLow_u	startDataTriggerMaskLow;
		RegLoggerStartMaskTriggerHigh_u	startDataTriggerMaskHigh; 
	
		startDataTriggerMaskLow.bitFields.startMaskTriggerLow = 0x0; //Reset all active bytes 
		startDataTriggerMaskHigh.bitFields.startMaskTriggerHigh = 0x0; //Reset all active bytes 
	
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startDataTriggerMaskHigh.val);	
	}
	else
	{
		RegLoggerStopMaskTriggerLow_u	stopDataTriggerMaskLow;
		RegLoggerStopMaskTriggerHigh_u	stopDataTriggerMaskHigh; 

		ASSERT(resetHdrTriggerReq->u8TriggerType == LOGGER_HDR_STOP_TRIGGER_TYPE);//Make sure the type is valid 

		stopDataTriggerMaskLow.bitFields.stopMaskTriggerLow = 0x0; //Reset all active bytes 
		stopDataTriggerMaskHigh.bitFields.stopMaskTriggerHigh = 0x0; //Reset all active bytes 
		/*Write the header mask to the start trigger filter*/
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_LOW, stopDataTriggerMaskLow.val);
		RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopDataTriggerMaskHigh.val);
	}
}


/**********************************************************************************
HwLogger_ResetHdrTriggerReq



	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_SetPhyHeaderReq(UMI_SET_PHY_HDR_REQ_t* resetHdrTriggerReq)
{
 	RegLoggerPhyFifo1FirstHeaderWord_u	phyFirstHeaderWord;// fifo 1 stands for band0
	RegLoggerPhyFifo1SecondHeaderWord_u	phySecondHeaderWord; 
	RegLoggerPhyFifo1MesageLenght_u		phyMessageLength; 
	RegLoggerFifoEnable_u				loggerfifoEnable; 

	phyFirstHeaderWord.val = 0;
	phySecondHeaderWord.val = 0;
	phyMessageLength.val = 0;
	loggerfifoEnable.val = 0;

	phyFirstHeaderWord.bitFields.phyFifo1FirstHeaderWord = resetHdrTriggerReq->phyFifoFirstHeaderWord;
	phySecondHeaderWord.bitFields.phyFifo1SecondHeaderWord = resetHdrTriggerReq->phyFifoSecondHeaderWord;
	phyMessageLength.bitFields.phyFifo1MessageLenght = resetHdrTriggerReq->phyFifoMessageLength;
	/*Write the header mask to the start trigger filter*/
	RegAccess_Write(REG_LOGGER_PHY_FIFO1_FIRST_HEADER_WORD, phyFirstHeaderWord.val);
	RegAccess_Write(REG_LOGGER_PHY_FIFO1_SECOND_HEADER_WORD, phySecondHeaderWord.val);
	RegAccess_Write(REG_LOGGER_PHY_FIFO1_MESAGE_LENGHT, phyMessageLength.val);

	//Now enable the phy fifo 
	RegAccess_Read(REG_LOGGER_FIFO_ENABLE, &loggerfifoEnable.val);
	loggerfifoEnable.bitFields.phyFifo1Enable = 0x1; 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerfifoEnable.val);
}


/**********************************************************************************
hwLoggerEnableDisableHostAccTxInLogs	
	
Description:
------------
	enable or disable HW TxIn  logs
Input:
-----
	bool shortLogEnable - enable / disable TxIn short  logs
	bool longLogEnable -enable / disable TxIn long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccTxInLogs(bool shortLogEnable , bool longLogEnable)
{
	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	volatile RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable TxIn logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.txInLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.txInLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.txInLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.txInLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.txInLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.txInLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);
}
/**********************************************************************************
hwLoggerEnableDisableHostAccTxOutLogs	
	
Description:
------------
	enable or disable HW TxOut  logs
Input:
-----
	bool shortLogEnable - enable / disable TxOut short  logs
	bool longLogEnable -enable / disable TxOut long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccTxOutLogs(bool shortLogEnable , bool longLogEnable)
{
	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	volatile RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable TxOut logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.txOutLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.txOutLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.txOutLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.txOutLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.txOutLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.txOutLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);

}
/**********************************************************************************
hwLoggerEnableDisableHostAccRxInLogs	
	
Description:
------------
	enable or disable HW RxIn  logs
Input:
-----
	bool shortLogEnable - enable / disable RxIn short  logs
	bool longLogEnable -enable / disable RxIn long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccRxInLogs(bool shortLogEnable , bool longLogEnable)
{
	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	volatile RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable RxIn logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.rxInLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.rxInLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.rxInLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.rxInLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.rxInLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.rxInLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);
}
/**********************************************************************************
hwLoggerEnableDisableHostAccRxOutLogs	
	
Description:
------------
	enable or disable HW RxOut  logs
Input:
-----
	bool shortLogEnable - enable / disable RxOut short  logs
	bool longLogEnable -enable / disable RxOut long  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableHostAccRxOutLogs(bool shortLogEnable , bool longLogEnable)
{
	RegHostIfAccHostLoggerCfg_u hostLoggerCfg;
	volatile RegHostIfAccHostLoggerBusy_u loggerBusy;
	
	// disable RxOut logs
	RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_CFG, &hostLoggerCfg.val);
	hostLoggerCfg.bitFields.rxOutLoggerShortEn = FALSE;
	hostLoggerCfg.bitFields.rxOutLoggerLongEn = FALSE;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG,hostLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_HOST_IF_ACC_HOST_LOGGER_BUSY, &loggerBusy.val);
	}while (loggerBusy.bitFields.rxOutLoggerBusy == 1);
	

	// enable logs
	hostLoggerCfg.bitFields.rxOutLoggerShortEn = shortLogEnable;
	hostLoggerCfg.bitFields.rxOutLoggerLongEn = longLogEnable;
	hostLoggerCfg.bitFields.rxOutLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_HOST_IF_ACC_HOST_LOGGER_CFG, hostLoggerCfg.val);
}
/**********************************************************************************
hwLoggerEnableDisableTxLiberatorLogs	
	
Description:
------------
	enable or disable HW Tx liberator  logs
Input:
-----
	bool shortLogEnable - enable / disable Tx liberator  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxLiberatorLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegMacGeneralLiberatorLoggerReg_u txLiberatorLogger;
	UNUSED_PARAM(longLogEnable);	
	// disable TxLiberator logs
	RegAccess_Read(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, &txLiberatorLogger.val);
	txLiberatorLogger.bitFields.liberatorLoggerRegEn = FALSE;
	RegAccess_Write(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG,txLiberatorLogger.val);

	// need to check that we are not while writing log to fifo
	while (txLiberatorLogger.bitFields.liberatorLoggerRegActive == 1)
	{
		RegAccess_Read(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, &txLiberatorLogger.val);
	}
	
	// enable logs
	txLiberatorLogger.bitFields.liberatorLoggerRegEn = shortLogEnable;
	txLiberatorLogger.bitFields.liberatorLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, txLiberatorLogger.val);	
}

/**********************************************************************************
hwLoggerEnableDisableRxLiberatorLogs	
	
Description:
------------
	enable or disable HW Rx liberator  logs
Input:
-----
	bool shortLogEnable - enable / disable Rx liberator  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxLiberatorLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegMacGeneralLiberatorLoggerReg_u RxLiberatorLogger;
	UNUSED_PARAM(longLogEnable);	
	// disable RxLiberator logs
	RegAccess_Read(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, &RxLiberatorLogger.val);
	RxLiberatorLogger.bitFields.rxLiberatorLoggerRegEn = FALSE;
	RegAccess_Write(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG,RxLiberatorLogger.val);

	// need to check that we are not while writing log to fifo
	while (RxLiberatorLogger.bitFields.rxLiberatorLoggerRegActive == 1)
	{
		RegAccess_Read(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, &RxLiberatorLogger.val);
	}
	// enable logs
	RxLiberatorLogger.bitFields.rxLiberatorLoggerRegEn = shortLogEnable;
	RxLiberatorLogger.bitFields.rxLiberatorLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_MAC_GENERAL_LIBERATOR_LOGGER_REG, RxLiberatorLogger.val);	
}
/**********************************************************************************
hwLoggerEnableDisableRxPpLogs	
	
Description:
------------
	enable or disable HW RxPp logs
Input:
-----
	bool shortLogEnable - enable / disable RxPp  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxPpLogs(bool shortLogEnable , bool longLogEnable)
{
	RegRxPpRxLoggerCfg_u rxPpRxLoggerCfg;
	volatile RegRxPpRxLoggerBusy_u rxPpRxLoggerBusy;
	UNUSED_PARAM(longLogEnable);	
	// disable  logs
	rxPpRxLoggerCfg.val = 0;
	RegAccess_Write(REG_RX_PP_RX_LOGGER_CFG,rxPpRxLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_RX_PP_RX_LOGGER_BUSY, &rxPpRxLoggerBusy.val);
	}while (rxPpRxLoggerBusy.bitFields.rxPpLoggerBusy == 1);
	

	// enable logs
	rxPpRxLoggerCfg.bitFields.rxPpLoggerEn = shortLogEnable;
	rxPpRxLoggerCfg.bitFields.rxPpLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_RX_PP_RX_LOGGER_CFG, rxPpRxLoggerCfg.val);
}

/**********************************************************************************
hwLoggerEnableDisableQmTxListsLogs	
	
Description:
------------
	enable or disable HW Tx lists logs
Input:
-----
	bool shortLogEnable - enable / disable Tx lists  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxListsLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm1LoggerCfg_u QmTxListsLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM1_LOGGER_CFG, &QmTxListsLoggerCfg.val);
	QmTxListsLoggerCfg.bitFields.dlm1LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM1_LOGGER_CFG,QmTxListsLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM1_LOGGER_CFG, &QmTxListsLoggerCfg.val);
	}while (QmTxListsLoggerCfg.bitFields.dlm1LoggerBusy == 1);
	
	// enable logs
	QmTxListsLoggerCfg.bitFields.dlm1LoggerEn = shortLogEnable;
	QmTxListsLoggerCfg.bitFields.dlm1LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM1_LOGGER_CFG, QmTxListsLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmTxDataLogs	
	
Description:
------------
	enable or disable HW Tx data logs
Input:
-----
	bool shortLogEnable - enable / disable Tx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxDataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm0LoggerCfg_u QmTxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM0_LOGGER_CFG, &QmTxDataLoggerCfg.val);
	QmTxDataLoggerCfg.bitFields.dlm0LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM0_LOGGER_CFG,QmTxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM0_LOGGER_CFG, &QmTxDataLoggerCfg.val);
	}while (QmTxDataLoggerCfg.bitFields.dlm0LoggerBusy == 1);
	
	// enable logs
	QmTxDataLoggerCfg.bitFields.dlm0LoggerEn = shortLogEnable;
	QmTxDataLoggerCfg.bitFields.dlm0LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM0_LOGGER_CFG, QmTxDataLoggerCfg.val);	
}

/**********************************************************************************
hwLoggerEnableDisableQmRxDataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxDataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM2_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM2_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM2_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM2_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmDmaDataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmDmaDataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM4_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM4_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM4_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM4_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmRxMpdu0DataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxMpdu0DataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM5_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM5_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM5_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM5_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmRxMpdu1DataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxMpdu1DataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM6_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM6_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM6_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM6_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmTxMpdu0DataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxMpdu0DataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM7_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM7_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM7_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM7_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmTxMpdu1DataLogs	
	
Description:
------------
	enable or disable HW Rx data logs
Input:
-----
	bool shortLogEnable - enable / disable Rx data  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmTxMpdu1DataLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm2LoggerCfg_u QmRxDataLoggerCfg;
	UNUSED_PARAM(longLogEnable);
	// disable  logs
	RegAccess_Read(REG_DLM_DLM8_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM8_LOGGER_CFG,QmRxDataLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM8_LOGGER_CFG, &QmRxDataLoggerCfg.val);
	}while (QmRxDataLoggerCfg.bitFields.dlm2LoggerBusy == 1);
	
	// enable logs
	QmRxDataLoggerCfg.bitFields.dlm2LoggerEn = shortLogEnable;
	QmRxDataLoggerCfg.bitFields.dlm2LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM8_LOGGER_CFG, QmRxDataLoggerCfg.val);	
}
/**********************************************************************************
hwLoggerEnableDisableQmRxListsLogs	
	
Description:
------------
	enable or disable HW Rx List logs
Input:
-----
	bool shortLogEnable - enable / disable Rx list  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmRxListsLogs(bool shortLogEnable , bool longLogEnable)
{
	volatile RegDlmDlm3LoggerCfg_u QmRxListsLoggerCfg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_DLM_DLM3_LOGGER_CFG, &QmRxListsLoggerCfg.val);
	QmRxListsLoggerCfg.bitFields.dlm3LoggerEn = FALSE;
	RegAccess_Write(REG_DLM_DLM3_LOGGER_CFG,QmRxListsLoggerCfg.val);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_DLM_DLM3_LOGGER_CFG, &QmRxListsLoggerCfg.val);
	}while (QmRxListsLoggerCfg.bitFields.dlm3LoggerBusy == 1);
	
	// enable logs
	QmRxListsLoggerCfg.bitFields.dlm3LoggerEn = shortLogEnable;
	QmRxListsLoggerCfg.bitFields.dlm3LoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_DLM_DLM3_LOGGER_CFG, QmRxListsLoggerCfg.val);	
}

/**********************************************************************************
hwLoggerEnableDisableQmAgerLogs	
	
Description:
------------
	enable or disable HW Ager logs
Input:
-----
	bool shortLogEnable - enable / disable  Ager  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableQmAgerLogs(bool shortLogEnable , bool longLogEnable)
{
	RegQManagerCpu2LoggerAgerCfg_u QmAgerLoggerCfg;
	volatile RegQManagerCpu2LoggerAgerBusy_u regQManagerCpu2LoggerAgerBusy;
	RegQManagerCpu2LoggerAgerPriority_u regQManagerCpu2LoggerAgerPriority;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_Read(REG_Q_MANAGER_CPU2_LOGGER_AGER_CFG, &QmAgerLoggerCfg.val);
	QmAgerLoggerCfg.bitFields.agerLoggerEn = FALSE;
	RegAccess_Write(REG_Q_MANAGER_CPU2_LOGGER_AGER_CFG,QmAgerLoggerCfg.val);

	RegAccess_Read(REG_Q_MANAGER_CPU2_LOGGER_AGER_BUSY, &regQManagerCpu2LoggerAgerBusy.val);
	while (regQManagerCpu2LoggerAgerBusy.bitFields.agerLoggerBusy == 1)
	{
		RegAccess_Read(REG_Q_MANAGER_CPU2_LOGGER_AGER_BUSY, &regQManagerCpu2LoggerAgerBusy.val);
	}

	// enable logs
	QmAgerLoggerCfg.bitFields.agerLoggerEn = shortLogEnable;
	RegAccess_Write(REG_Q_MANAGER_CPU2_LOGGER_AGER_CFG, QmAgerLoggerCfg.val);	

	regQManagerCpu2LoggerAgerPriority.bitFields.agerLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_Q_MANAGER_CPU2_LOGGER_AGER_PRIORITY, regQManagerCpu2LoggerAgerPriority.val);	
}

/**********************************************************************************
hwLoggerEnableDisableBaAnlayzerLogs	
	
Description:
------------
	enable or disable HW BaAnlayzer  logs
Input:
-----
	bool shortLogEnable - enable / disable BaAnlayzer  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBaAnlayzerLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBaAnlayzerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableBaAnlayzerLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBaAnlayzerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableBaAnlayzerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	volatile RegBaAnalyzerLoggerReg_u baAnalyzerLogger;
	UNUSED_PARAM(longLogEnable);
	// disable BaAnalyzer logs
	RegAccess_ReadPerBand(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val, bandId);
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegEn = FALSE;
	RegAccess_WritePerBand(REG_BA_ANALYZER_LOGGER_REG,baAnalyzerLogger.val, bandId);

	// need to check that we are not while writing log to fifo
	while (baAnalyzerLogger.bitFields.baAnalyzerLoggerRegActive == 1)
	{
		RegAccess_ReadPerBand(REG_BA_ANALYZER_LOGGER_REG, &baAnalyzerLogger.val, bandId);
	}
	
	// enable logs
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegEn = shortLogEnable;
	baAnalyzerLogger.bitFields.baAnalyzerLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_BA_ANALYZER_LOGGER_REG, baAnalyzerLogger.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisablePreAggLogsBand0	
	
Description:
------------
	enable or disable HW Agg builder  logs
Input:
-----
	bool shortLogEnable - enable / disable Agg handler  logs
	bool longLogEnable - enable / disable Agg handler Std logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisablePreAggLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePreAggLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisablePreAggLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePreAggLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisablePreAggLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId )
{
	RegPreAggPreAggLogger_u			preAggLogger;
	volatile RegPreAggPreAggLoggerActive_u   preAggLoggerActive;
	UNUSED_PARAM(longLogEnable);

	preAggLogger.val = 0;
	RegAccess_WritePerBand(REG_PRE_AGG_PRE_AGG_LOGGER, preAggLogger.val, bandId); // disable agg builder logs

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PRE_AGG_PRE_AGG_LOGGER_ACTIVE, &preAggLoggerActive.val, bandId);
	}while (preAggLoggerActive.bitFields.preAggLoggerActive == 1);

	// enable logs
	preAggLogger.bitFields.preAggLoggerEn = shortLogEnable;
	preAggLogger.bitFields.preAggLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_PRE_AGG_PRE_AGG_LOGGER, preAggLogger.val, bandId);
}
/**********************************************************************************
hwLoggerEnableDisableAutoRespLogsBand0	
	
Description:
------------

Input:
-----


Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableAutoRespLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableAutoRespLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableAutoRespLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableAutoRespLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableAutoRespLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId )
{
	volatile RegAutoRespAutoRespLogger_u			autoRespLogger;
	UNUSED_PARAM(longLogEnable);

	RegAccess_ReadPerBand(REG_AUTO_RESP_AUTO_RESP_LOGGER, &autoRespLogger.val, bandId);
	autoRespLogger.bitFields.autoRespLoggerEn = 0;

	RegAccess_WritePerBand(REG_AUTO_RESP_AUTO_RESP_LOGGER, autoRespLogger.val, bandId); // disable agg builder logs


	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_AUTO_RESP_AUTO_RESP_LOGGER, &autoRespLogger.val, bandId);
	}while (autoRespLogger.bitFields.loggerActive == 1);

	// enable logs
	autoRespLogger.bitFields.autoRespLoggerEn = shortLogEnable;
	autoRespLogger.bitFields.autoRespLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_AUTO_RESP_AUTO_RESP_LOGGER, autoRespLogger.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisableTfGenLogsBand0	
	
Description:
------------

Input:
-----


Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTfGenLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableAutoRespLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTfGenLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableAutoRespLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableTfGenLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId )
{
	RegTfGeneratorTfGenLoggerControl_u			tfGenLogger;
	volatile RegTfGeneratorTfGenLoggerStatus_u			tfGenLoggerStatus;
	UNUSED_PARAM(longLogEnable);

	tfGenLogger.val = 0;

	RegAccess_WritePerBand(REG_TF_GENERATOR_TF_GEN_LOGGER_CONTROL, tfGenLogger.val, bandId); 

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TF_GENERATOR_TF_GEN_LOGGER_STATUS, &tfGenLoggerStatus.val, bandId);
	}while (tfGenLoggerStatus.bitFields.tfGenLoggerIdle == 0);

	// enable logs
	tfGenLogger.bitFields.tfGenLoggerEn = shortLogEnable;
	tfGenLogger.bitFields.tfGenLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_AUTO_RESP_AUTO_RESP_LOGGER, tfGenLogger.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisableTxHLogs	
	
Description:
------------
	enable or disable HW TxHandler logs
Input:
-----
	bool shortLogEnable - enable / disable TxHandler  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void  hwLoggerEnableDisableTxHLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	 hwLoggerEnableDisableTxHLogsPerBand( shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void  hwLoggerEnableDisableTxHLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	 hwLoggerEnableDisableTxHLogsPerBand( shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void  hwLoggerEnableDisableTxHLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxHandlerTxhLogger_u txHandlerTxhLogger;
	volatile RegTxHandlerTxhLoggerActive_u txHandlerTxhLoggerActive;
	UNUSED_PARAM(longLogEnable);
	// disable TxH logs
	txHandlerTxhLogger.val = 0;
	RegAccess_WritePerBand(REG_TX_HANDLER_TXH_LOGGER,txHandlerTxhLogger.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_HANDLER_TXH_LOGGER_ACTIVE, &txHandlerTxhLoggerActive.val, bandId);
	}while (txHandlerTxhLoggerActive.bitFields.txhLoggerActive == 1);
	

	// enable logs
	txHandlerTxhLogger.bitFields.txhLoggerEn = shortLogEnable;
	txHandlerTxhLogger.bitFields.txhLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_HANDLER_TXH_LOGGER, txHandlerTxhLogger.val, bandId);
}
/**********************************************************************************
hwLoggerEnableDisableTxSequencerLogs	
	
Description:
------------
	enable or disable HW TxSequencer logs
Input:
-----
	bool shortLogEnable - enable / disable TxSequencer  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSequencerLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSequencerLogsPerBand(shortLogEnable,longLogEnable,CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTxSequencerLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSequencerLogsPerBand(shortLogEnable,longLogEnable,CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableTxSequencerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSequencerSequencerLogger_u			txSequencerSequencerLogger;
	volatile RegTxSequencerSequencerLoggerActive_u   txSequencerSequencerLoggerActive;
	UNUSED_PARAM(longLogEnable);

	txSequencerSequencerLogger.val = 0;
	RegAccess_WritePerBand(REG_TX_SEQUENCER_SEQUENCER_LOGGER, txSequencerSequencerLogger.val, bandId); 

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SEQUENCER_SEQUENCER_LOGGER_ACTIVE, &txSequencerSequencerLoggerActive.val, bandId);
	}while (txSequencerSequencerLoggerActive.bitFields.sequencerLoggerActive == 1);

	// enable logs
	txSequencerSequencerLogger.bitFields.sequencerLoggerEn = shortLogEnable;
	txSequencerSequencerLogger.bitFields.sequencerLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SEQUENCER_SEQUENCER_LOGGER, txSequencerSequencerLogger.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisablePsManagerLogs 

Description:
------------
	Enable or Disable the Hw logs of the Ps Setting Hw Module
		1) Disable Logs
		2) Wait for log activity to finish
		3) Enable/Disable logs as required
	
Input: 
-----
	shortLogEnable - Enable or Disable fifo log
	longLogEnable - Not relevant for this module

Returns:
--------
	None 
**********************************************************************************/
void hwLoggerEnableDisablePsManagerLogsBand0(bool shortLogEnable, bool longLogEnable)
{
	hwLoggerEnableDisablePsManagerLogsPerBand(shortLogEnable, longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisablePsManagerLogsBand1(bool shortLogEnable, bool longLogEnable)
{
	hwLoggerEnableDisablePsManagerLogsPerBand(shortLogEnable, longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisablePsManagerLogsPerBand(bool shortLogEnable, bool longLogEnable, uint8 bandId)
{
	RegPsSettingPsLogger_u psLoggerReg;
	volatile RegPsSettingPsLoggerActive_u  psLoggerActiveReg;
	UNUSED_PARAM(longLogEnable);

	// disable	logs
	psLoggerReg.val = 0;
	RegAccess_WritePerBand(REG_PS_SETTING_PS_LOGGER,psLoggerReg.val, bandId);
	
	// need to check that we are not in the middle writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PS_SETTING_PS_LOGGER_ACTIVE, &psLoggerActiveReg.val, bandId);
	}while (psLoggerActiveReg.bitFields.psLoggerActive == 1);
	
	// enable logs
	psLoggerReg.bitFields.psLoggerEn = shortLogEnable;
	psLoggerReg.bitFields.psLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_PS_SETTING_PS_LOGGER, psLoggerReg.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisableRxClassLogs	
	
Description:
------------
	enable or disable HW Rx Classifier logs
Input:
-----
	bool shortLogEnable - enable / disable  Rx Classifier  logs
	bool longLogEnable - enable / disable  Rx Classifier Rtd logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxClassLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableRxClassLogsPerBand(shortLogEnable, longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}

void hwLoggerEnableDisableRxClassLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableRxClassLogsPerBand(shortLogEnable, longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableRxClassLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegRxClassifierRxClasLoggerControl_u rxClasLoggerControl;
	volatile RegRxClassifierRxClasLoggerActive_u  rxClasLoggerActive;

	// disable  logs
	rxClasLoggerControl.val = 0;
	RegAccess_WritePerBand(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_CONTROL,rxClasLoggerControl.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_ACTIVE, &rxClasLoggerActive.val, bandId);
	}while (rxClasLoggerActive.bitFields.rxClasLoggerActive == 1);
	
	// enable logs
	rxClasLoggerControl.bitFields.rxClasLoggerEn = (shortLogEnable || longLogEnable);
	rxClasLoggerControl.bitFields.rxClasLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_RX_CLASSIFIER_RX_CLAS_LOGGER_CONTROL, rxClasLoggerControl.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisableBeamFormLogs	
	
Description:
------------
	enable or disable HW BeamForm logs
Input:
-----
	bool shortLogEnable - enable / disable bfImplicit and bfReport  logs
	bool longLogEnable - enable / disable Ndpa and PhyNdp  logs

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBeamFormLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBeamFormLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableBeamFormLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBeamFormLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableBeamFormLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	UNUSED_PARAM(longLogEnable);

#ifdef ENET_INC_ARCH_WAVE600B
	volatile RegMacBfmMbfmLoggerCtrl1_u bfLoggerControl1;
	RegMacBfmMbfmLoggerCtrl2_u  bfLoggerControl2;

	// disable  logs
	bfLoggerControl1.val = 0;

	RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL1,bfLoggerControl1.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL1, &bfLoggerControl1.val, bandId);
	}while (bfLoggerControl1.bitFields.mbfmLoggerActiveInd == 1);

	// enable logs
	bfLoggerControl1.bitFields.swLoggerEn = shortLogEnable;
	bfLoggerControl1.bitFields.swLoggerLimit = 0x40;
	bfLoggerControl1.bitFields.swLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
#ifdef ENET_INC_ARCH_WAVE600D2
	bfLoggerControl2.bitFields.swLoggerStreamEn = 0xFFFF;
#else
	bfLoggerControl2.bitFields.swLoggerStreamEn = 0xFFFFFF;
#endif
	RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL2, bfLoggerControl2.val, bandId);	
	RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL1, bfLoggerControl1.val, bandId);

#else

	volatile RegMacBfmMbfmLoggerCtrl_u bfLoggerControl;
	//RegMacBfmMbfmLoggerCfg_u bfLoggerCfg;

	// disable	logs
	bfLoggerControl.val = 0;

	RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL,bfLoggerControl.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL, &bfLoggerControl.val, bandId);
	}while (bfLoggerControl.bitFields.mbfmLoggerActiveInd == 1);

	// enable logs
	bfLoggerControl.bitFields.swLoggerEn = shortLogEnable;
	bfLoggerControl.bitFields.swLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	bfLoggerControl.bitFields.swLoggerStreamEn = 0xFFFF;
	// bfLoggerCfg.bitFields.swLoggerCfg = 0x0; //TBD miryam to check

	//RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CFG, bfLoggerCfg.val, bandId);	
	RegAccess_WritePerBand(REG_MAC_BFM_MBFM_LOGGER_CTRL, bfLoggerControl.val, bandId);

#endif //ENET_INC_ARCH_WAVE600B

}

/**********************************************************************************
hwLoggerEnableDisableTxSelectorOrLockerLogs	
	
Description:
------------
	enable or disable HW TX selector bitmap logs
Input:
-----
	bool shortLogEnable - enable / disable  TX selector or Locker  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSelectorOrLockerLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSelectorOrLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTxSelectorOrLockerLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSelectorOrLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableTxSelectorOrLockerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	volatile RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val, bandId);
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerEn = FALSE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val, bandId);
	}while (txSelLoggerBusy.bitFields.txSelLoggerSelectionLockerBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerSel = 0; // 0x0: Selection logger , 0x1: Locker logger
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisableTxSelectorBitmapLogs	
	
Description:
------------
	enable or disable HW TX selector bitmap logs
Input:
-----
	bool shortLogEnable - enable / disable  TX selector bitmap  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxSelectorBitmapLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSelectorBitmapLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTxSelectorBitmapLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxSelectorBitmapLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableTxSelectorBitmapLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	volatile RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val, bandId);
	txSelLoggerCfg.bitFields.txSelLoggerBitmapUpdateEn = FALSE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val, bandId);
	}while (txSelLoggerBusy.bitFields.txSelLoggerBitmapUpdateBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerBitmapUpdateEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerBitmapPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val, bandId);	
}



/**********************************************************************************
hwLoggerEnableDisableDelia0Logs	
	
Description:
------------
	enable or disable HW Delia0 logs
Input:
-----
	bool shortLogEnable - enable / disable  Delia0  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableDelia0LogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDelia0LogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableDelia0LogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDelia0LogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableDelia0LogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegPacDelDeliaLoggerControl_u DeliaLoggerControl;
	volatile RegPacDelDeliaLoggerActive_u  DeliaLoggerActive;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL, &DeliaLoggerControl.val, bandId);

	DeliaLoggerControl.bitFields.deliaLoggerEn = 0;

	RegAccess_WritePerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PAC_DEL_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val, bandId);
	}while (DeliaLoggerActive.bitFields.deliaLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val, bandId);	
}


/**********************************************************************************
hwLoggerEnableDisableDelia1Logs	- DELIA AF (AUTO FILL) LOG
	
Description:
------------
	enable or disable HW Delia1 logs
Input:
-----
	bool shortLogEnable - enable / disable  Delia1  logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableDelia1LogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDelia1LogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableDelia1LogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDelia1LogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableDelia1LogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegPacDelDeliaLoggerControl_u DeliaLoggerControl;
	volatile RegPacDelDeliaLoggerActive_u  DeliaLoggerActive;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL, &DeliaLoggerControl.val, bandId);

	DeliaLoggerControl.bitFields.deliaAfLoggerEn = 0;

	RegAccess_WritePerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL,DeliaLoggerControl.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PAC_DEL_DELIA_LOGGER_ACTIVE, &DeliaLoggerActive.val, bandId);
	}while (DeliaLoggerActive.bitFields.deliaAfLoggerActive == 1);
	
	// enable logs
	DeliaLoggerControl.bitFields.deliaAfLoggerEn = shortLogEnable;
	DeliaLoggerControl.bitFields.deliaAfLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	DeliaLoggerControl.bitFields.deliaAfLoggerCaptureOnlySopEop = 1;

	RegAccess_WritePerBand(REG_PAC_DEL_DELIA_LOGGER_CONTROL, DeliaLoggerControl.val, bandId);	
}

	
/**********************************************************************************
hwLoggerEnableDisableRxCLogs	
	
Description:
------------
	enable or disable HW RXC logs
Input:
-----
	bool shortLogEnable - enable / disable RXC logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxCLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableRxCLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableRxCLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableRxCLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableRxCLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegPacRxcRxcLogger_u rxcLoggerControl;
	volatile RegPacRxcRxcLoggerActive_u  rxcLoggerActive;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	rxcLoggerControl.val = 0;
	RegAccess_WritePerBand(REG_PAC_RXC_RXC_LOGGER,rxcLoggerControl.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PAC_RXC_RXC_LOGGER_ACTIVE, &rxcLoggerActive.val, bandId);
	}while (rxcLoggerActive.bitFields.rxcLoggerActive == 1);
	// enable logs
	rxcLoggerControl.bitFields.rxcLoggerEn = shortLogEnable;
	rxcLoggerControl.bitFields.rxcLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_PAC_RXC_RXC_LOGGER, rxcLoggerControl.val, bandId);	

}
/**********************************************************************************
hwLoggerEnableDisableActivityLogs	
	
Description:
------------
	enable or disable HW activity logs
Input:
-----
	bool shortLogEnable - enable / disable activity logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableActivityLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	RegMacGeneralActivityLoggerCtl_u activityLoggerCtl;
	volatile RegMacGeneralHwLoggerStatus_u  hwLoggerStatus;
	UNUSED_PARAM(longLogEnable);

	// disable	logs 
	RegAccess_Read(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL, &activityLoggerCtl.val);
	activityLoggerCtl.bitFields.actLoggerEnaB0 = 0;
	RegAccess_Write(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL,activityLoggerCtl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_GENERAL_HW_LOGGER_STATUS, &hwLoggerStatus.val);
	}while (hwLoggerStatus.bitFields.actLoggerRunning0 == 1);
	
	// enable logs
	activityLoggerCtl.bitFields.actLoggerPeriod = 1279; // 4usec - cycle resolution
	activityLoggerCtl.bitFields.actLoggerReportLength = 0x32; 
	activityLoggerCtl.bitFields.actLoggerEnaB0 = shortLogEnable;
	activityLoggerCtl.bitFields.actLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL, activityLoggerCtl.val); 
}

void hwLoggerEnableDisableActivityLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	RegMacGeneralActivityLoggerCtl_u activityLoggerCtl;
	volatile RegMacGeneralHwLoggerStatus_u  hwLoggerStatus;
	UNUSED_PARAM(longLogEnable);

	// disable	logs //
	RegAccess_Read(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL, &activityLoggerCtl.val);
	activityLoggerCtl.bitFields.actLoggerEnaB1 = 0;
	RegAccess_Write(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL,activityLoggerCtl.val);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_Read(REG_MAC_GENERAL_HW_LOGGER_STATUS, &hwLoggerStatus.val);
	}while (hwLoggerStatus.bitFields.actLoggerRunning1 == 1);
	
	// enable logs
	activityLoggerCtl.bitFields.actLoggerPeriod = 1279; // 4usec - cycle resolution
	activityLoggerCtl.bitFields.actLoggerReportLength = 0x32; 
	activityLoggerCtl.bitFields.actLoggerEnaB1 = shortLogEnable;
	activityLoggerCtl.bitFields.actLoggerPrio = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_Write(REG_MAC_GENERAL_ACTIVITY_LOGGER_CTL, activityLoggerCtl.val); 
}


/**********************************************************************************
hwLoggerEnableDisableSecurityLogsBand0	
	
Description:
------------
	enable or disable HW security logs
Input:
-----
	bool shortLogEnable - enable / disable security logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableSecurityLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableSecurityLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}

void hwLoggerEnableDisableSecurityLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableSecurityLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableSecurityLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId)
{
	RegMacWepSecLogger_u wepSecLogger;
	volatile RegMacWepSecLoggerActive_u  wepSecLoggerActive;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	wepSecLogger.val = 0;
	RegAccess_WritePerBand(REG_MAC_WEP_SEC_LOGGER,wepSecLogger.val,bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_MAC_WEP_SEC_LOGGER_ACTIVE, &wepSecLoggerActive.val,bandId);
	}while (wepSecLoggerActive.bitFields.secLoggerActive == 1);
	
	// enable logs
	wepSecLogger.bitFields.secLoggerEn = shortLogEnable;
	wepSecLogger.bitFields.secLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_MAC_WEP_SEC_LOGGER, wepSecLogger.val,bandId);	
}

/**********************************************************************************
hwLoggerEnableDisablePacExtrLogs
	
Description:
------------
	enable or disable HW security logs
Input:
-----
	bool shortLogEnable - enable / disable security logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisablePacExtrLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePacExtrLogsPerBand(shortLogEnable, longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}

void hwLoggerEnableDisablePacExtrLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePacExtrLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisablePacExtrLogsPerBand(bool shortLogEnable , bool longLogEnable,uint8 bandId)
{
	RegPacExtrapolatorLoggerPriority_u pacExtrapolatorLoggerPriority;
	RegPacExtrapolatorLoggerEn_u  pacExtrapolatorLoggerEn;
	volatile RegPacExtrapolatorLoggerActive_u pacExtrapolatorLoggerActive;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	pacExtrapolatorLoggerEn.val = 0;
	RegAccess_WritePerBand(REG_PAC_EXTRAPOLATOR_LOGGER_EN,pacExtrapolatorLoggerEn.val,bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_PAC_EXTRAPOLATOR_LOGGER_ACTIVE, &pacExtrapolatorLoggerActive.val,bandId);
	}while (pacExtrapolatorLoggerActive.bitFields.loggerActive == 1);
	
	// enable logs

	pacExtrapolatorLoggerPriority.bitFields.loggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_PAC_EXTRAPOLATOR_LOGGER_PRIORITY, pacExtrapolatorLoggerPriority.val,bandId);	

	pacExtrapolatorLoggerEn.bitFields.hwLoggerEn = shortLogEnable;
	RegAccess_WritePerBand(REG_PAC_EXTRAPOLATOR_LOGGER_EN, pacExtrapolatorLoggerEn.val,bandId);
}

/**********************************************************************************
hwLoggerEnableDisableBeaconLogs	
	
Description:
------------
	enable or disable HW beacon logs
Input:
-----
	bool shortLogEnable - enable / disable beacon logs
	bool longLogEnable - dont care

Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBeaconLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBeaconLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableBeaconLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBeaconLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}

void hwLoggerEnableDisableBeaconLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegBeaconBeaconLoggerCfg_u 	 beaconLoggerCfg;
	volatile RegBeaconBeaconLoggerBusy_u  beaconLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	beaconLoggerCfg.val = 0;
	RegAccess_WritePerBand(REG_BEACON_BEACON_LOGGER_CFG,beaconLoggerCfg.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_BEACON_BEACON_LOGGER_BUSY, &beaconLoggerBusy.val, bandId);
	}while (beaconLoggerBusy.bitFields.beaconTimersLoggerBusy == 1);
	
	// enable logs
	beaconLoggerCfg.bitFields.beaconTimersLoggerEn = shortLogEnable;
	beaconLoggerCfg.bitFields.beaconTimersLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_BEACON_BEACON_LOGGER_CFG, beaconLoggerCfg.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisableTxPdAccLogs	
	
Description:
------------
	dummy function for empty entry in HW fifo configuration matrix
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableTxPdAccLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxPdAccLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTxPdAccLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxPdAccLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableTxPdAccLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxPdAccTxpdLoggerControl_u txPdAccLoggerReg;
	volatile RegTxPdAccTxpdLoggerStatus_u txPdAccLoggerStatusReg;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	txPdAccLoggerReg.val = 0;
	RegAccess_WritePerBand(REG_TX_PD_ACC_TXPD_LOGGER_CONTROL,txPdAccLoggerReg.val, bandId);
	
	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_PD_ACC_TXPD_LOGGER_STATUS, &txPdAccLoggerStatusReg.val, bandId);
	}while (txPdAccLoggerStatusReg.bitFields.txpdLoggerIdle == 0);
	
	// enable logs
	txPdAccLoggerReg.bitFields.txpdLoggerEn = shortLogEnable;
	txPdAccLoggerReg.bitFields.txpdLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;

	RegAccess_WritePerBand(REG_TX_PD_ACC_TXPD_LOGGER_CONTROL, txPdAccLoggerReg.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisableMuLockerLogs

	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

void hwLoggerEnableDisableMuLockerLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableMuLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableMuLockerLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableMuLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableMuLockerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	volatile RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val, bandId);
	txSelLoggerCfg.bitFields.txSelLoggerMuLockerEn = FALSE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val, bandId);
	}while (txSelLoggerBusy.bitFields.txSelLoggerMuLockerBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerMuLockerEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerMuLockerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val, bandId);	
}
/**********************************************************************************
hwLoggerEnableDisablePlanLockerLogs

	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

void hwLoggerEnableDisablePlanLockerLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePlanLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisablePlanLockerLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisablePlanLockerLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisablePlanLockerLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	volatile RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	// disable  logs
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val, bandId);
	txSelLoggerCfg.bitFields.txSelLoggerPlanLockerEn = FALSE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val, bandId);
	}while (txSelLoggerBusy.bitFields.txSelLoggerPlanLockerBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerSelectionLockerSel = 0; // 0x0: Selection logger , 0x1: Locker logger
	txSelLoggerCfg.bitFields.txSelLoggerPlanLockerEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerPlanLockerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val, bandId);	

}
/**********************************************************************************
hwLoggerEnableDisableMuBitmapSelectLogs


	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableMuBitmapSelectLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableMuBitmapSelectLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableMuBitmapSelectLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableMuBitmapSelectLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableMuBitmapSelectLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegTxSelectorTxSelLoggerCfg_u txSelLoggerCfg;
	volatile RegTxSelectorTxSelLoggerBusy_u txSelLoggerBusy;
	UNUSED_PARAM(longLogEnable);

	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, &txSelLoggerCfg.val, bandId);
	txSelLoggerCfg.bitFields.txSelLoggerMuBitmapUpdateEn = FALSE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG,txSelLoggerCfg.val, bandId);

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_BUSY, &txSelLoggerBusy.val, bandId);
	}while (txSelLoggerBusy.bitFields.txSelLoggerMuBitmapUpdateBusy == 1);
	
	// enable logs
	txSelLoggerCfg.bitFields.txSelLoggerMuBitmapUpdateEn = shortLogEnable;
	txSelLoggerCfg.bitFields.txSelLoggerMuBitmapPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOGGER_CFG, txSelLoggerCfg.val, bandId);	
}

/**********************************************************************************
hwLoggerEnableDisableBsrcLogs



	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableBsrcLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBsrcLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableBsrcLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableBsrcLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableBsrcLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	volatile RegBsrcRegsLoggerReg_u			bsrcgLogger;
	UNUSED_PARAM(longLogEnable);

	bsrcgLogger.val = 0;
	RegAccess_WritePerBand(REG_BSRC_REGS_LOGGER_REG, bsrcgLogger.val, bandId); // disable  logs

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_BSRC_REGS_LOGGER_REG, &bsrcgLogger.val, bandId);
	}while (bsrcgLogger.bitFields.bsrcLoggerRegActive == 1);

	// enable logs
	bsrcgLogger.bitFields.bsrcLoggerRegEn = shortLogEnable;
	bsrcgLogger.bitFields.bsrcLoggerRegPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_BSRC_REGS_LOGGER_REG, bsrcgLogger.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisableRxHcAndDeaggLogs
	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableRxHcAndDeaggLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDeaggLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableRxHcAndDeaggLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableDeaggLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableDeaggLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegRxMsduParserRxMsduParserLoggerCfg_u			rxMsduParserLogger;
	volatile RegRxMsduParserMsduParserStatus_u				msduParserStatus;
	UNUSED_PARAM(longLogEnable);
	UNUSED_PARAM(bandId);	
	rxMsduParserLogger.val = 0;
	RegAccess_Write(REG_RX_MSDU_PARSER_RX_MSDU_PARSER_LOGGER_CFG, rxMsduParserLogger.val); // disable agg builder logs

	do
	{
		RegAccess_Read(REG_RX_MSDU_PARSER_MSDU_PARSER_STATUS, &msduParserStatus.val);
	}while (msduParserStatus.bitFields.rxMsduParserLoggerSm != 0);// need to check that we are not while writing log to fifo

	// enable logs
	rxMsduParserLogger.bitFields.rxMsduParserLoggerEn = shortLogEnable;
	rxMsduParserLogger.bitFields.rxMsduParserLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_Write(REG_RX_MSDU_PARSER_RX_MSDU_PARSER_LOGGER_CFG, rxMsduParserLogger.val);
}

/**********************************************************************************
hwLoggerEnableDisableOtfaLogs
	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

void hwLoggerEnableDisableOtfaLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableOtfaLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableOtfaLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableOtfaLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableOtfaLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	volatile RegOtfaLogger_u			otfaLogger;
	UNUSED_PARAM(longLogEnable);

	otfaLogger.val = 0;
	RegAccess_WritePerBand(REG_OTFA_LOGGER, otfaLogger.val, bandId); // disable  logs

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_OTFA_LOGGER, &otfaLogger.val, bandId);
	}while (otfaLogger.bitFields.loggerActive == 1);

	// enable logs
	otfaLogger.bitFields.loggerEnable = shortLogEnable;
	otfaLogger.bitFields.otfaLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_OTFA_LOGGER, otfaLogger.val, bandId);
}
/**********************************************************************************
hwLoggerEnableDisableCoordinatorLogs
	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

void hwLoggerEnableDisableCoordinatorLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableCoordinatorLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableCoordinatorLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableCoordinatorLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableCoordinatorLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	volatile RegRxCoordinatorLoggerCntrl_u			coorLogger;
	UNUSED_PARAM(longLogEnable);

	coorLogger.val = 0;
	RegAccess_WritePerBand(REG_RX_COORDINATOR_LOGGER_CNTRL, coorLogger.val, bandId); // disable  logs

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_RX_COORDINATOR_LOGGER_CNTRL, &coorLogger.val, bandId);
	}while (coorLogger.bitFields.rxCoordinatorLoggerActive == 1);

	// enable logs
	coorLogger.bitFields.swRxCoordinatorLoggerEn = shortLogEnable;
	coorLogger.bitFields.swRxCoordinatorLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_RX_COORDINATOR_LOGGER_CNTRL, coorLogger.val, bandId);
}

/**********************************************************************************
hwLoggerEnableDisableTxHeaderConversionLogs
	
Description:
------------

Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

void hwLoggerEnableDisableTxHeaderConversionLogsBand0(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxHeaderConversionLogsPerBand(shortLogEnable , longLogEnable,  CONFIGURATION_MANAGER_BAND_0);
}
void hwLoggerEnableDisableTxHeaderConversionLogsBand1(bool shortLogEnable , bool longLogEnable)
{
	hwLoggerEnableDisableTxHeaderConversionLogsPerBand(shortLogEnable , longLogEnable, CONFIGURATION_MANAGER_BAND_1);
}
void hwLoggerEnableDisableTxHeaderConversionLogsPerBand(bool shortLogEnable , bool longLogEnable, uint8 bandId)
{
	RegMacTxHcTxHcLoggerControl_u txHcLoggerReg;
	volatile RegMacTxHcTxHcLoggerStatus_u txHcLoggerStatusReg;
	UNUSED_PARAM(longLogEnable);

	txHcLoggerReg.val = 0;
	txHcLoggerStatusReg.val = 0;
	RegAccess_WritePerBand(REG_MAC_TX_HC_TX_HC_LOGGER_CONTROL, txHcLoggerReg.val, bandId); // disable  logs

	// need to check that we are not while writing log to fifo
	do
	{
		RegAccess_ReadPerBand(REG_MAC_TX_HC_TX_HC_LOGGER_STATUS, &txHcLoggerStatusReg.val, bandId);
	}while (txHcLoggerStatusReg.bitFields.txHcLoggerIdle == 0);

	// enable logs
	txHcLoggerReg.bitFields.txHcLoggerEn = shortLogEnable;
	txHcLoggerReg.bitFields.txHcLoggerPriority = LOGGER_HW_DEFAULT_PRIORITY;
	RegAccess_WritePerBand(REG_MAC_TX_HC_TX_HC_LOGGER_CONTROL, txHcLoggerReg.val, bandId);
}
/**********************************************************************************
hwLoggerEnableDisableUnknown	
	
Description:
------------
	dummy function for empty entry in HW fifo configuration matrix
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void hwLoggerEnableDisableUnknown(bool shortLogEnable , bool longLogEnable)
{
	UNUSED_PARAM(shortLogEnable);	
	UNUSED_PARAM(longLogEnable);
	//DEBUG_FATAL("hwLoggerEnableDisableUnknown");
}
/**********************************************************************************
HwLogger_Init	
	
Description:
------------
	
Input:
-----
	Max stream number 
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void HwLogger_Init(void)
{	 
	RegLoggerListId_u loggerListsIdsReg; 
	RegLoggerFifoEnable_u loggerFifoEnableReg;
	RegLoggerBufferCloseTimeThr_u bufferClosingTimeThr;
	RegLoggerStartMaskTriggerLow_u startMaskTriggerLow; 
	RegLoggerStartMaskTriggerLow_u startMaskTriggerHigh; 
	RegLoggerStopMaskTriggerHigh_u stopMaskTrrigerHigh;
	RegLoggerHwUnit1FirstHeaderWord_u hw1FisrtHeaderwordReg;
	RegLoggerMode_u	loggerModeReg;
	uint32 hwHeaderRegAddress; 
	uint8 fifoIndex;  
	HwLogs_u *hwLogs;


	UNUSED_PARAM(hwLogs);


	/*Clear SHRAM buffers and Buffer descriptors*/
	memset(&HwLoggerShramBuffers, 0, sizeof(HwLoggerShramBuffer_t));
	memset(HwLoggerShramBDs, 0, sizeof(LoggerBd_t)*NUM_OF_HW_LOGGER_SHRAM_BUFFERS);

	/*REG_LOGGER_LIST_ID 0x84 */
	/*Buffer descriptors lists*/ 
	loggerListsIdsReg.val = 0; 
	loggerListsIdsReg.bitFields.bufferEmtyTxDplIdx = HW_Q_MANAGER_TX_POOL_LIST_LOGGER_FREE_BD;
	loggerListsIdsReg.bitFields.bufferFullTxDplIdx = HW_Q_MANAGER_READY_LIST_LOGGER_FULL_BD;
	/*  RDs lists */
	loggerListsIdsReg.bitFields.loggerRdEmtyRxDplIdx = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST; 
	loggerListsIdsReg.bitFields.loggerRdFullRxDplIdx = HW_Q_MANAGER_RX_DONE_LIST_LOGGER; 
	RegAccess_Write(REG_LOGGER_LIST_ID,loggerListsIdsReg.val);
	loggerFifoEnableReg.val = 0; //All fifo's initialies to disable mode 
	RegAccess_Write(REG_LOGGER_FIFO_ENABLE, loggerFifoEnableReg.val); 	
	/* Configure  Buffer Time Threshold*/
	bufferClosingTimeThr.bitFields.bufferCloseTimeThr = LOGGER_DEFAULT_BUFFER_TIME_THRESHOLD; 
	RegAccess_Write(REG_LOGGER_BUFFER_CLOSE_TIME_THR, bufferClosingTimeThr.val);

	startMaskTriggerLow.val = 0; 
	RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_LOW, startMaskTriggerLow.val);	
	startMaskTriggerHigh.val = 0; 
	RegAccess_Write(REG_LOGGER_START_MASK_TRIGGER_HIGH, startMaskTriggerHigh.val);
	stopMaskTrrigerHigh.bitFields.logLevelToForward = 0; // forward from log level  0  
	RegAccess_Write(REG_LOGGER_STOP_MASK_TRIGGER_HIGH, stopMaskTrrigerHigh.val);

	/* initialize all HW first header word to 0, */
	hwHeaderRegAddress = REG_LOGGER_HW_UNIT1_FIRST_HEADER_WORD; 
	hw1FisrtHeaderwordReg.val = 0;
	for(fifoIndex = LOGGER_FIRST_HW_SOURCE_INDEX; fifoIndex < NUM_OF_LOGGER_FIFOS; fifoIndex++)
	{	
		RegAccess_Write((hwHeaderRegAddress + (fifoIndex-LOGGER_FIRST_HW_SOURCE_INDEX)*4), hw1FisrtHeaderwordReg.val);
	}

	HwLoggerSharedParams.maxNumberOfStreams = 0; 

    /* Enable logger clock */
	loggerModeReg.bitFields.listConverterBlockEn = TRUE;
	RegAccess_Write(REG_LOGGER_MODE,loggerModeReg.val);

	memset(HwLoggerStreamSharedParams, 0, sizeof(HwLoggerStreamSharedParams));
}







