/***********************************************************************************
 File:		HwMbfm.c
 Module:		Hw MAC Beamforming
 Purpose:		
 Description:	FW driver for Hw MAC Beamforming Module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "ShramBeamforming.h"
#include "loggerAPI.h"
#include "VapDb.h"
#include "ieee80211.h"
#include "MacBfmRegs.h"
#include "HwMbfm_Api.h"
#include "DeliaRecipes_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_HW_DRIVERS
#define LOG_LOCAL_FID 	3

#define HW_BEAMORMING_RU_END_20MHZ				(8)		// value taken from system doc: SAS - Beamforming.doc
#define HW_BEAMORMING_RU_END_40MHZ				(17)	// value taken from system doc: SAS - Beamforming.doc
#define HW_BEAMORMING_RU_END_80MHZ				(36)	// value taken from system doc: SAS - Beamforming.doc
#define HW_BEAMORMING_RU_END_160MHZ				(73)	// value taken from system doc: SAS - Beamforming.doc


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
MacBfmDb_t HwMbfmDb;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************
HwMbfm_CheckFifoError  

Description:
------------
	Check if there were errors on the error fifo
	
**********************************************************************************/
void HwMbfm_CheckFifoError(void)
{
	RegMacBfmMbfmFifoDebug_u			regMbfmFifoDebug;
	RegMacBfmMbfmFifoClearStrb_u		regMbfmfifoClearStrb;

	regMbfmfifoClearStrb.val = 0;
	
	// Read fifo debug register
	RegAccess_Read(REG_MAC_BFM_MBFM_FIFO_DEBUG,&regMbfmFifoDebug.val);

	// Count the number of times the error fifo was full (nothing we can do with it)
	HwMbfmDb.fifoFullDropCounter += regMbfmFifoDebug.bitFields.swMbfmFifoFullDropCtr;
	ASSERT(HwMbfmDb.fifoFullDropCounter == 0); // While we still ramp up gen6b - we want to assert here so we can decide if we want to increase the fifo size. Later we can remove this assert.

	// Clear fifo full drop counter
	regMbfmfifoClearStrb.bitFields.swMbfmFifoClearFullDropCtrStrb = TRUE;
	RegAccess_Write(REG_MAC_BFM_MBFM_FIFO_CLEAR_STRB,regMbfmfifoClearStrb.val);

	
	// check that we didn't decrease the fifo more than we were allowed
	ASSERT(regMbfmFifoDebug.bitFields.swMbfmFifoDecrementLessThanZero == FALSE);
}


/**********************************************************************************

HwMbfm_ErrorFifoHandler

Description:
------------
	read from hw mbfm error fifo upon error
	currently only raise a counter
	
Input:
-----
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwMbfm_ErrorFifoHandler(void)
{
	RegMacBfmMbfmFifoRdEntriesNum_u 	regMacBfmMbfmFifoRdEntriesNum;
	RegMacBfmMbfmFifoNumEntriesCount_u 	regMacBfmMbfmFifoNumEntriesCount;

	regMacBfmMbfmFifoRdEntriesNum.val = 0;
	regMacBfmMbfmFifoNumEntriesCount.val = 0;

    RegAccess_Read(REG_MAC_BFM_MBFM_FIFO_NUM_ENTRIES_COUNT, &regMacBfmMbfmFifoNumEntriesCount.val);	
	
	regMacBfmMbfmFifoRdEntriesNum.bitFields.swMbfmFifoRdEntriesNum = regMacBfmMbfmFifoNumEntriesCount.bitFields.swMbfmFifoNumEntriesCount;
	
	//this must not be a fatal because it depends on how station transmits its bf report
	//currently only count the number of times this error occured
	HwMbfmDb.fifoNotEmptyCounter += regMacBfmMbfmFifoRdEntriesNum.bitFields.swMbfmFifoRdEntriesNum;

	//decrement counter by number of read reports
	RegAccess_Write(REG_MAC_BFM_MBFM_FIFO_RD_ENTRIES_NUM, regMacBfmMbfmFifoRdEntriesNum.val);

	HwMbfm_CheckFifoError();

}


void HwMbfm_isrError(void)
{
	MacBfmIsrCtrl_t	isrCtrlErrorCauses;
	RegMacBfmMbfmIsrCtrlClr_u isrCtrlClr;

	isrCtrlErrorCauses.val = 0;
	isrCtrlClr.val = 0;

    RegAccess_Read(REG_MAC_BFM_MBFM_ISR_CTRL, &isrCtrlErrorCauses.val);

	/*Currently add FATAL for each cause saparately to be able to change each FATAL to diferent implementation as needed during bring-up*/

	if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseFifoOverflow)
	{
		isrCtrlClr.bitFields.mbfmGlobalErrIrq0Clr = TRUE;		
		FATAL("HwMbfm_isrError: FifoOverflow"); //should not occur since this isr is not enabled
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBfeeZeroLength)
	{
		isrCtrlClr.bitFields.mbfmGlobalErrIrq1Clr = TRUE;
		FATAL("HwMbfm_isrError: BFee Zero Length");
	}	
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBferAuthenticationFailed) 
	{				
		ILOG0_V("HwMbfm_isrError: mbfmGlobalErrIrqCauseBferAuthenticationFailed");
		isrCtrlClr.bitFields.mbfmGlobalErrIrq2Clr = TRUE;
		// Authentication error (wrong MAC address in BF report) can be caused by CRC error - no need to assert
		HwMbfmDb.BferAuthenticationFailedErrorcounter++;
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBfeeNtxError) 
	{		
		ILOG0_V("HwMbfm_isrError: mbfmGlobalErrIrqCauseBfeeNtxError");
		isrCtrlClr.bitFields.mbfmGlobalErrIrq3Clr = TRUE;
		/* This error will be raised as well when Phy Mode error occurs */
		HwMbfmDb.BfeeNtxErrorCounter++;
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBfeeTxPhyModeError)       
	{		
		isrCtrlClr.bitFields.mbfmGlobalErrIrq4Clr = TRUE;
		/* Phy mode error: Phy mode in STA DB (for the received NDP BW) is illegal - BF report Tx is aborted.
		e.g: Recevied NDP is 20MHz HE SU, but Phy mode in STA DB for 20MHz is 11b - we shouldn't assert in this case */
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseHtMcsNss32OrDcm)
	{
		isrCtrlClr.bitFields.mbfmGlobalErrIrq5Clr = TRUE;		
		FATAL("HwMbfm_isrError: Ht MCS NSS 320 or DCM");
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBferGeneralError)         
	{
		ILOG0_V("HwMbfm_isrError: mbfmGlobalErrIrqCauseBferGeneralError");
		isrCtrlClr.bitFields.mbfmGlobalErrIrq6Clr = TRUE;
		
		/*this error could happen in case station is not responding correctly to the report and therefore we shouldn't assert
		replace it with dedicated counter
		*/
		HwMbfmDb.globalIsrErrorcounter++;
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBfeeUnexpectedRxcAccess)
	{
		isrCtrlClr.bitFields.mbfmGlobalErrIrq7Clr = TRUE;		
		FATAL("HwMbfm_isrError: Bfee Unexpected Rxc Access");
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqCauseBfeeLostNdp)    
	{
		ILOG0_V("HwMbfm_isrError: mbfmGlobalErrIrqCauseBfeeLostNdp");
		isrCtrlClr.bitFields.mbfmGlobalErrIrq8Clr = TRUE;		
		/*this error happens when AP as BFee waits for NDP, but decides to start sounding sequence as BFer - means that NDP was missed
		*/		
		HwMbfmDb.BfeeLostNdpErrorCounter++;
	}
	else if(isrCtrlErrorCauses.bitFields.mbfmGlobalErrIrqBfeeGeneralError)	  
	{
		isrCtrlClr.bitFields.mbfmGlobalErrIrq9Clr = TRUE;
		FATAL("HwMbfm_isrError: mbfmGlobalErrIrqBfeeError");
	}
	else
	{
		FATAL("HwMbfm_isrError: cause of error is missing"); // Means that interrupt was raised, but there is no dedicated handler to its cause.
	}

	//clear isr causes
	RegAccess_Write(REG_MAC_BFM_MBFM_ISR_CTRL_CLR, isrCtrlClr.val);

}



/**********************************************************************************

HwMbfm_Init 

Description:
------------
	Enable MAC Beamforming module HW
	Configure required base addreses 

Input:
-----
Output:
-------
	
Returns:
--------
	void - 
**********************************************************************************/
void HwMbfm_Init(void)
{
	RegMacBfmMbfmCfgOt_u 			 mbfmCfgOtReg;
	RegMacBfmMbfmFifoBaseAddr_u 	 mbfmMbfmFifoBaseAddr;
	RegMacBfmMbfmFifoDepthMinusOne_u mbfmMbfmFifoDepthMinusOne;
	RegMacBfmMbfmGlobalEn_u			 mbfmMbGlobalEn;
	RegMacBfmMbfmBfeeTxcPointerA_u	 mbfmBfeePointerA;
	RegMacBfmMbfmBfeeTxcPointerB_u	 mbfmBfeePointerB;
	RegMacBfmMbfmBfeeTxcPointerC_u	 mbfmBfeePointerC;
	RegMacBfmMbfmTxRecipeCtrl_u		 mbfmTxRecipeCtrl;
	MacBfmIsrCtrl_t					 mbfmIsrCtrl;

	mbfmMbfmFifoBaseAddr.val = 0;
	mbfmMbGlobalEn.val = 0;
	mbfmMbfmFifoDepthMinusOne.val = 0;
	mbfmBfeePointerA.val = 0;
	mbfmBfeePointerB.val = 0;
	mbfmBfeePointerC.val = 0;
	mbfmTxRecipeCtrl.val = 0;
	mbfmIsrCtrl.val	= 0;

	
	memset(&HwMbfmDb, 0, sizeof(MacBfmDb_t));


	/*All init sequence should be done exactly in the order it is written to be aligned to requrements and what is tested in verification*/
	

	//global enable mbfm block
	mbfmMbGlobalEn.bitFields.swBfGlobalEn = TRUE;
	mbfmMbGlobalEn.bitFields.swBfeeGlobalEn = TRUE;
	RegAccess_Write(REG_MAC_BFM_MBFM_GLOBAL_EN, mbfmMbGlobalEn.val);

	// Init One Time register
	
	// HW asks to read modify write this register in order to keep the default value of the field "sw_limit_pipe_flow_control" & "sw_shift_reg_timeout"
	RegAccess_Read(REG_MAC_BFM_MBFM_CFG_OT, &(mbfmCfgOtReg.val));

	// The following fields are hooks for HW bugs */
	mbfmCfgOtReg.bitFields.swCheckFeedbackSegments 	= TRUE;		// Software configures whether to verify that feedback segments fields are 0 as part of MIMO control validation , 1'b1 - verify the feedback segments fields  , 1'b0 - ignore the feedback segments fields
	mbfmCfgOtReg.bitFields.swCheckHeRuEnd		  	= TRUE;		// Software configures whether to verify the RU END field for HE BFRs as part of MIMO control validation , 1'b1 - verify the RU END field  , 1'b0 - ignore the RU END field 
	mbfmCfgOtReg.bitFields.swCheckHeRuStart		  	= TRUE;		// Software configures whether to verify the RU START field for HE BFRs as part of MIMO control validation , 1'b1 - verify the RU START field is 0 , 1'b0 - ignore the RU START field 
	mbfmCfgOtReg.bitFields.swCheckBwInHt		  	= TRUE;  	// Software configures whether to verify the exact BW in HT mode as part of MIMO control validation , 1'b1 - verify the BW field  , 1'b0 - ignore the BW field
	mbfmCfgOtReg.bitFields.swCheckBw		 		= TRUE;  	// Software configures whether to verify the exact BW for all phy modes as part of MIMO control validation , 1'b1 - verify the BW field  , 1'b0 - ignore the BW field
	mbfmCfgOtReg.bitFields.swCheckSoundingModeHe	= TRUE;  	// whether to verify that each MIMO control in TB response is indeed HE format , 1 - check this match , 0 - Ignore checking and allow also VHT/HT formats
	mbfmCfgOtReg.bitFields.swLengthCheckEn		  	= TRUE;		// whether to check BFR MPDU length calculated by RXC (from delimiter) to the MPDU length calculated by the MBFM according to mimo control fields.	, Software must modify this bit prior to setting sw_clr_expected_bfr_pulse , 1'b1 - check length , 1'b0 - don't check length.
	mbfmCfgOtReg.bitFields.swCheckPhyMode		  	= TRUE;		// Software configures whether to verify the exact phy mode as part of MIMO control validation , 1'b1 - verify the PHY mode field  , 1'b0 - ignore the PHY mode field, and use Action + Category to decide the phy mode
	mbfmCfgOtReg.bitFields.swCheckNgHeMu		  	= TRUE;  	// Software configures whether to verify the exact value of received Ng in HE MU report as part of MIMO control validation , 1'b1 - verify the Ng field  , 1'b0 - ignore the Ng field check
	mbfmCfgOtReg.bitFields.swCheckCodebookHeMu	 	= TRUE;		// Software configures whether to verify the exact value of received codebook in HE MU report as part of MIMO control validation , 1'b1 - verify the codebook field  , 1'b0 - ignore the codebook field check
	mbfmCfgOtReg.bitFields.swCheckFeedbackType	  	= TRUE;		// This field manipulates the feedback type field matching as part of MIMO control check , 1'b1 - validate exact match , 1'b0 - allow MBFM to parse SU report in case of requested MU report. This SU report will only update the SU database and not MU DB.
	mbfmCfgOtReg.bitFields.swMuBfrWithoutDSnr	  	= FALSE;	// 802.11 spec expects Delta SNR after the phases stage. Write 1'b0 to this field to support MU report with no delta SNR.	
	mbfmCfgOtReg.bitFields.swAuthCheckEn		  	= TRUE;		// whether to check ADDR1 & ADDR2 in the MPDU as part of the BFR validity checks. This is a global configuration for all the BFRs processed by the MBFM.  , Software must modify this bit prior to setting sw_clr_expected_bfr_pulse , 1'b1 - check those fields , 1'b0 - don't check those fields.
	mbfmCfgOtReg.bitFields.swHeSuBfrWithDSnr       	= FALSE;	

	RegAccess_Write(REG_MAC_BFM_MBFM_CFG_OT, mbfmCfgOtReg.val);


		// Init base address of the Error FIFO
		
#if defined(ENET_INC_LMAC0) 
		memset(BeamformingErrorFifo,	0, (sizeof(BeamformingErrorFifo)));
		mbfmMbfmFifoBaseAddr.bitFields.swMbfmFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BeamformingErrorFifo[0]);
#endif
			
#if defined(ENET_INC_LMAC1) 
		memset(BeamformingErrorFifo_B1, 0, (sizeof(BeamformingErrorFifo_B1)));
		mbfmMbfmFifoBaseAddr.bitFields.swMbfmFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&BeamformingErrorFifo_B1[0]);
#endif


	RegAccess_Write(REG_MAC_BFM_MBFM_FIFO_BASE_ADDR, mbfmMbfmFifoBaseAddr.val);


	// Init FIFO size of the error FIFO (16 entries of 8 bytes)
	mbfmMbfmFifoDepthMinusOne.bitFields.swMbfmFifoDepthMinusOne = HW_BEAMFORMING_FIFO_DEPTH_MINUS_ONE;
	RegAccess_Write(REG_MAC_BFM_MBFM_FIFO_DEPTH_MINUS_ONE, mbfmMbfmFifoDepthMinusOne.val);


	/*for bfee - configure where in circular buffer the BF Report response (which contains 3 parts)  would be written (currently always to user0 in cyclic buffer*/
	mbfmBfeePointerA.bitFields.swTxcSectionA4BBaseAddr = 0x10;	
	RegAccess_Write(REG_MAC_BFM_MBFM_BFEE_TXC_POINTER_A, mbfmBfeePointerA.val);
	
	mbfmBfeePointerB.bitFields.swTxcSectionB4BBaseAddr = 0x20;	
	RegAccess_Write(REG_MAC_BFM_MBFM_BFEE_TXC_POINTER_B, mbfmBfeePointerB.val);
	
	mbfmBfeePointerC.bitFields.swTxcSectionC4BBaseAddr = 0x40;	
	RegAccess_Write(REG_MAC_BFM_MBFM_BFEE_TXC_POINTER_C, mbfmBfeePointerC.val);

	mbfmTxRecipeCtrl.bitFields.swTxRecipe4BBaseAddr  = ((uint32)pMbfmTcrs - B0_TX_RECIPES_BASE_ADDR) >> 2;	
	RegAccess_Write(REG_MAC_BFM_MBFM_TX_RECIPE_CTRL, mbfmTxRecipeCtrl.val);   

	//enable ISR errors for all causes except fifoOverflow, which is handled as part of errorFifo handler
#ifdef HW_MBFM_WRONG_ZERO_LEN_BFEE_ERROR_WORKAROUND
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeZeroLength				= FALSE;
#else
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeZeroLength				= TRUE;
#endif //HW_MBFM_WRONG_ZERO_LEN_BFEE_ERROR_WORKAROUND
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBferAuthenticationFailed  	= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeNtxError 			 	= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeTxPhyModeError		 	= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseHtMcsNss32OrDcm			= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBferGeneralError 		 	= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeUnexpectedRxcAccess	= TRUE;
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeLostNdp				= TRUE;
#ifdef HW_MBFM_GLOBAL_IRQ_ERROR_FALSE_ALARM_WORKAROUND
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeGeneralError 			= FALSE;
#else
	mbfmIsrCtrl.bitFields.mbfmGlobalErrIrqEnCauseBfeeGeneralError			= TRUE;
#endif //HW_MBFM_GLOBAL_IRQ_ERROR_FALSE_ALARM_WORKAROUND
	RegAccess_Write(REG_MAC_BFM_MBFM_ISR_CTRL, mbfmIsrCtrl.val);   

}





